%feature("docstring") OT::NonLinearLeastSquaresCalibration
"Non-linear least-squares calibration algorithm.

Parameters
----------
model : :class:`~openturns.Function`
    The parametric function to be calibrated.
inputObservations : 2-d sequence of float
    The sample of input observations.
    Can have dimension 0 to specify no observations.
outputObservations : 2-d sequence of float
    The sample of output observations.
startingPoint : sequence of float
    The reference value of the parameter, used as the starting point
    of the optimization.

Notes
-----
NonLinearLeastSquaresCalibration is the minimum variance estimator of the parameter 
of a given model with no assumption on the dependence of the model with respect 
to the parameter.

The prior distribution of the parameter is a noninformative prior
emulated using a flat :class:`~openturns.Normal` centered on the `startingPoint` and with a variance equal to SpecFunc.MaxScalar.

The posterior distribution of the parameter is :class:`~openturns.Normal` and reflects the 
variability of the optimum parameter depending on the observation sample. 
By default, the posterior distribution is evaluated based on a linear approximation 
of the model at the optimum. 
This corresponds to using the :class:`~openturns.LinearLeastSquaresCalibration` at the optimum, 
and is named *Laplace approximation* in the Bayesian context. 
However, if the key `NonLinearLeastSquaresCalibration-BootstrapSize` in the :class:`~openturns.ResourceMap` 
is set to a nonzero positive integer, then a bootstrap resampling of the observations 
is performed and the posterior distribution is based on a :class:`~openturns.KernelSmoothing` 
of the sample of boostrap optimum parameters. 

The resulting distribution of the output error is a :class:`~openturns.Normal` and is 
computed from the residuals. 

If least squares optimization algorithms are enabled, then the algorithm used is the
first found by `Build` of :class:`~openturns.OptimizationAlgorithm`.
Otherwise, the algorithm :class:`~openturns.TNC` is used combined with
a multistart algorithm which makes use of the `NonLinearLeastSquaresCalibration-MultiStartSize`
key in the :class:`~openturns.ResourceMap`.

Please read :ref:`code_calibration` for more details.

See also
--------
GaussianLinearCalibration, LinearLeastSquaresCalibration, GaussianNonLinearCalibration

Examples
--------
Calibrate a nonlinear model using non-linear least-squares:

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> m = 10
>>> x = [[0.5 + i] for i in range(m)]
>>> inVars = ['a', 'b', 'c', 'x']
>>> formulas = ['a + b * exp(c * x)']
>>> model = ot.SymbolicFunction(inVars, formulas)
>>> p_ref = [2.8, 1.2, 0.5]
>>> params = [0, 1, 2]
>>> modelX = ot.ParametricFunction(model, params, p_ref)
>>> y = modelX(x)
>>> y += ot.Normal(0.0, 0.05).getSample(m)
>>> startingPoint = [1.0]*3
>>> algo = ot.NonLinearLeastSquaresCalibration(modelX, x, y, startingPoint)
>>> algo.run()
>>> print(algo.getResult().getParameterMAP())
[2.773...,1.203...,0.499...]"

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::getModelObservations
"Accessor to the model evaluation at the `startingPoint`.

Returns
-------
modelObservation : :class:`~openturns.Sample`
    Evaluation of the model at the `startingPoint`."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::getGradientObservations
"Accessor to the model gradient at the `startingPoint`.

Returns
-------
gradientObservation : :class:`~openturns.Matrix`
    Gradient of the model at the `startingPoint`."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::getStartingPoint
"Accessor to the parameter `startingPoint`.

Returns
-------
startingPoint : :class:`~openturns.Point`
    Parameter startingPoint."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::getOptimizationAlgorithm
"Accessor to the optimization algorithm used for the computation.

Returns
-------
algo : :class:`~openturns.OptimizationAlgorithm`
    Optimization algorithm used for the computation."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::setOptimizationAlgorithm
"Accessor to the optimization algorithm used for the computation.

Parameters
----------
algo : :class:`~openturns.OptimizationAlgorithm`
    Optimization algorithm to use for the computation."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::getBootstrapSize
"Accessor to the bootstrap size used to sample the posterior distribution.

Returns
-------
size : int
    Bootstrap size used to sample the posterior distribution. A value of 0
    means that no bootstrap has been done but a linear approximation has
    been used to get the posterior distribution, using the :class:`~openturns.GaussianLinearCalibration`
    algorithm at the maximum a posteriori estimate."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::setBootstrapSize
"Accessor to the bootstrap size used to sample the posterior distribution.

Parameters
----------
size : int
    Bootstrap size used to sample the posterior distribution. A value of 0
    means that no bootstrap has to be done but a linear approximation has
    been used to get the posterior distribution, using the :class:`~openturns.GaussianLinearCalibration`
    algorithm at the maximum a posteriori estimate."

// ---------------------------------------------------------------------

%feature("docstring") OT::NonLinearLeastSquaresCalibration::BuildResidualFunction
"Build a residual function given a parametric model, input and output observations.

Parameters
----------
model : :class:`~openturns.Function`
    Parametric model.
inputObservations : 2-d sequence of float
    Input observations associated to the output observations.
outputObservations : :class:`~openturns.Function`
    Output observations.

Returns
-------
residual : :class:`~openturns.Function`
Residual function.

Notes
-----
Given a parametric model :math:`F_{\\theta}:\\Rset^n\\rightarrow\\Rset^p` with
parameter :math:`\\theta\\in\\Rset^m`, a sample of input points
:math:`(x_i)_{i=1,\\dots,N}` and the associated output
:math:`(y_i)_{i=1,\\dots,N}`, the residual function :math:`f` is defined by:

.. math::

    \\forall \\theta\\in\\Rset^m, f(\\theta)=\\left(\\begin{array}{c}
    F_{\\theta}(x_1)-y_1 \\\\
    \\vdots \\\\
    F_{\\theta}(x_N)-y_N
    \\end{array}
    \\right)

"
