%feature("docstring") OT::GaussLegendre
"Tensorized integration algorithm of Gauss-Legendre.

Available constructors:
    GaussLegendre(*dimension=1*)

    GaussLegendre(*discretization*)

Parameters
----------
dimension : int, :math:`dimension>0`
    The dimension of the functions to integrate. The default discretization is *GaussLegendre-DefaultMarginalIntegrationPointsNumber* in each dimension, see :class:`~openturns.ResourceMap`.
discretization : sequence of int
    The number of nodes in each dimension. The sequence must be non-empty and must contain only positive values.

Notes
-----
The Gauss-Legendre algorithm enables to approximate the definite integral:

.. math::

    \\int_{\\vect{a}}^\\vect{b} f(\\vect{t})\\di{\\vect{t}}


with :math:`f: \\Rset^d \\mapsto \\Rset^p`, :math:`\\vect{a}, \\vect{b}\\in\\Rset^d` using a fixed tensorized Gauss-Legendre approximation:

.. math::

    \\int_{\\vect{a}}^\\vect{b} f(\\vect{t})\\di{\\vect{t}} = \\sum_{\\vect{n}\\in \\cN}\\left(\\prod_{i=1}^d w^{N_i}_{n_i}\\right)f(\\xi^{N_1}_{n_1},\\dots,\\xi^{N_d}_{n_d})

where :math:`\\xi^{N_i}_{n_i}` is the :math:`n_i` th node of the :math:`N_i` -points Gauss-Legendre 1D integration rule and :math:`w^{N_i}_{n_i}` the associated weight.

Examples
--------
Create a Gauss-Legendre algorithm:

>>> import openturns as ot
>>> algo = ot.GaussLegendre(2)
>>> algo = ot.GaussLegendre([2, 4, 5])"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::integrate
"Evaluation of the integral of :math:`f` on an interval.

Available usages:
    integrate(*f, interval*)

Parameters
----------
f : :class:`~openturns.Function`, :math:`f: \\Rset^d \\mapsto \\Rset^p`
    The integrand function.
interval : :class:`~openturns.Interval`, :math:`interval \\in \\Rset^d` 
    The integration domain. 

Returns
-------
value : :class:`~openturns.Point`
    Approximation of the integral.


Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x'], ['sin(x)'])
>>> a = -2.5
>>> b = 4.5
>>> algoGL = ot.GaussLegendre([10])
>>> value = algoGL.integrate(f, ot.Interval(a, b))[0]
>>> print(value)
-0.590...
"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::integrateWithNodes
"Evaluation of the integral of :math:`f` on an interval.

Parameters
----------
f : :class:`~openturns.Function`, :math:`f: \\Rset^d \\mapsto \\Rset^p`
    The integrand function.
interval : :class:`~openturns.Interval`, :math:`interval \\in \\Rset^d`
    The integration domain.

Returns
-------
value : :class:`~openturns.Point`
    Approximation of the integral.
nodes : :class:`~openturns.Sample`.
    The integration nodes.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x'], ['sin(x)'])
>>> a = -2.5
>>> b = 4.5
>>> algoGL = ot.GaussLegendre([10])
>>> value, nodes = algoGL.integrateWithNodes(f, ot.Interval(a, b))
>>> print(value[0])
-0.590...
>>> print(nodes)
0 : [ -2.40867  ]
1 : [ -2.02772  ]
2 : [ -1.37793  ]
3 : [ -0.516884 ]
4 : [  0.47894  ]
5 : [  1.52106  ]
6 : [  2.51688  ]
7 : [  3.37793  ]
8 : [  4.02772  ]
9 : [  4.40867  ]
"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::getDiscretization
"Accessor to the discretization of the tensorized rule.

Returns
-------
discretization : :class:`~openturns.Indices`
    The number of integration point in each dimension."

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::getNodes
"Accessor to the integration nodes.

Returns
-------
nodes : :class:`~openturns.Sample`
    The tensorized Gauss-Legendre integration nodes on :math:`[0,1]^d` where :math:`d>0` is the dimension of the integration algorithm."

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussLegendre::getWeights
"Accessor to the integration weights.

Returns
-------
weights : :class:`~openturns.Point`
    The tensorized Gauss-Legendre integration weights on :math:`[0,1]^d` where :math:`d>0` is the dimension of the integration algorithm."

