// =============================================================================
// PROJECT CHRONO - http://projectchrono.org
//
// Copyright (c) 2014 projectchrono.org
// All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file at the top level of the distribution and at
// http://projectchrono.org/license-chrono.txt.
//
// =============================================================================
// Authors: Rainer Gericke, Radu Serban
// =============================================================================
//
// Base class for a coil-spring or air-sprin solid axle suspension, typically
// combined with the rotary arm steering.
//
// Derived from ChSuspension, but still an abstract base class.
//
// Steerable solid axle guided by a triangular and two longitudinal links.
// Sprung by coil springs and translational dampers.
//
// The suspension subsystem is modeled with respect to a right-handed frame,
// with X pointing towards the front, Y to the left, and Z up (ISO standard).
// The suspension reference frame is assumed to be always aligned with that of
// the vehicle.  When attached to a chassis, only an offset is provided.
//
// All point locations are assumed to be given for the left half of the
// suspension and will be mirrored (reflecting the y coordinates) to construct
// the right side.
//
// =============================================================================

#ifndef CH_SOLID_BELLCRANK_3LINK_AXLE_H
#define CH_SOLID_BELLCRANK_3LINK_AXLE_H

#include <vector>

#include "chrono_vehicle/ChApiVehicle.h"
#include "chrono_vehicle/wheeled_vehicle/ChSuspension.h"

namespace chrono {
namespace vehicle {

/// @addtogroup vehicle_wheeled_suspension
/// @{

/// Base class for a coil-spring or air-spring solid axle suspension, typically
/// combined with the rotary arm steering.
///
/// The suspension subsystem is modeled with respect to a right-handed frame,
/// with X pointing towards the front, Y to the left, and Z up (ISO standard).
/// The suspension reference frame is assumed to be always aligned with that of
/// the vehicle.  When attached to a chassis, only an offset is provided.
///
/// All point locations are assumed to be given for the left half of the
/// suspension and will be mirrored (reflecting the y coordinates) to construct
/// the right side.
class CH_VEHICLE_API ChSolidBellcrankThreeLinkAxle : public ChSuspension {
  public:
    ChSolidBellcrankThreeLinkAxle(const std::string& name  ///< [in] name of the subsystem
    );

    virtual ~ChSolidBellcrankThreeLinkAxle();

    /// Get the name of the vehicle subsystem template.
    virtual std::string GetTemplateName() const override { return "SolidBellcrankThreeLinkAxle"; }

    /// Specify whether or not this suspension can be steered.
    virtual bool IsSteerable() const final override { return true; }

    /// Specify whether or not this is an independent suspension.
    virtual bool IsIndependent() const final override { return false; }

    /// Initialize this suspension subsystem.
    /// The suspension subsystem is initialized by attaching it to the specified chassis and (if provided) to the
    /// specified subchassis, at the specified location (with respect to and expressed in the reference frame of the
    /// chassis). It is assumed that the suspension reference frame is always aligned with the chassis reference frame.
    /// If a steering subsystem is provided, the suspension tierods are to be attached to the steering's central link
    /// body (steered suspension); otherwise they are to be attached to the chassis (non-steered suspension).
    virtual void Initialize(
        std::shared_ptr<ChChassis> chassis,        ///< [in] associated chassis subsystem
        std::shared_ptr<ChSubchassis> subchassis,  ///< [in] associated subchassis subsystem (may be null)
        std::shared_ptr<ChSteering> steering,      ///< [in] associated steering subsystem (may be null)
        const ChVector3d& location,                ///< [in] location relative to the chassis frame
        double left_ang_vel = 0,                   ///< [in] initial angular velocity of left wheel
        double right_ang_vel = 0                   ///< [in] initial angular velocity of right wheel
        ) override;

    /// Add visualization assets for the suspension subsystem.
    /// This default implementation uses primitives.
    virtual void AddVisualizationAssets(VisualizationType vis) override;

    /// Remove visualization assets for the suspension subsystem.
    virtual void RemoveVisualizationAssets() override;

    /// Get the wheel track for the suspension subsystem.
    virtual double GetTrack() override;

    /// Get a handle to the specified spring element.
    std::shared_ptr<ChLinkTSDA> GetSpring(VehicleSide side) const { return m_spring[side]; }

    /// Get a handle to the specified shock (damper) element.
    std::shared_ptr<ChLinkTSDA> GetShock(VehicleSide side) const { return m_shock[side]; }

    /// Return current suspension TSDA force information on the specified side.
    virtual std::vector<ForceTSDA> ReportSuspensionForce(VehicleSide side) const override;

    /// Get the force in the spring element.
    double GetSpringForce(VehicleSide side) const { return m_spring[side]->GetForce(); }

    /// Get the current length of the spring element
    double GetSpringLength(VehicleSide side) const { return m_spring[side]->GetLength(); }

    /// Get the current deformation of the spring element.
    double GetSpringDeformation(VehicleSide side) const { return m_spring[side]->GetDeformation(); }

    /// Get the force in the shock (damper) element.
    double GetShockForce(VehicleSide side) const { return m_shock[side]->GetForce(); }

    /// Get the current length of the shock (damper) element.
    double GetShockLength(VehicleSide side) const { return m_shock[side]->GetLength(); }

    /// Get the current deformation velocity of the shock (damper) element.
    double GetShockVelocity(VehicleSide side) const { return m_shock[side]->GetVelocity(); }

    /// Log current constraint violations.
    virtual void LogConstraintViolations(VehicleSide side) override;

    void LogHardpointLocations(const ChVector3d& ref, bool inches = false);

  protected:
    /// Identifiers for the various hardpoints.
    enum PointId {
        SHOCK_A,      ///< shock, axle
        SHOCK_C,      ///< shock, chassis
        SPRING_A,     ///< spring, axle
        SPRING_C,     ///< spring, chassis
        SPINDLE,      ///< spindle location
        TRIANGLE_A,   ///< triangle link location on the axle (single point y = 0)
        TRIANGLE_C,   ///< triangle link locationon the chassis (two points)
        LINK_A,       ///< longitudinal link location on the axle (two points)
        LINK_C,       ///< longitudinal link location on the chassis (two points)
        BELLCRANK_A,  ///< location of revolute joint of bellcrank/axle
        BELLCRANK_D,  ///< location of connection point bellcrank/draglink
        BELLCRANK_T,  ///< location of connection point bellcrank/knuckles
        DRAGLINK_S,   ///< location of connection point draglink/steering
        KNUCKLE_L,    ///< lower point kingpin axis
        KNUCKLE_U,    ///< upper point kingin axis
        KNUCKLE_T,    ///< location of knuckle/tierod connection
        KNUCKLE_CM,   ///< center of mass knuckle
        NUM_POINTS
    };

    virtual void InitializeInertiaProperties() override;
    virtual void UpdateInertiaProperties() override;

    /// Return the location of the specified hardpoint.
    /// The returned location must be expressed in the suspension reference frame.
    virtual const ChVector3d getLocation(PointId which) = 0;

    /// Return the camber angle, in radians (default: 0).
    virtual double getCamberAngle() const { return 0; }

    /// Return the toe angle, in radians (default: 0).
    /// A positive value indicates toe-in, a negative value indicates toe-out.
    virtual double getToeAngle() const { return 0; }

    /// Return the center of mass of the axle tube.
    virtual const ChVector3d getAxleTubeCOM() const = 0;

    /// Return the mass of the axle tube body.
    virtual double getAxleTubeMass() const = 0;
    /// Return the mass of the spindle body.
    virtual double getSpindleMass() const = 0;
    /// Return the mass of the bellcrank body.
    virtual double getBellcrankMass() const = 0;
    /// Return the mass of the bellcrank body.
    virtual double getDraglinkMass() const = 0;
    /// Return the mass of the knuckle body.
    virtual double getKnuckleMass() const = 0;
    /// Return the mass of the triangle body.
    virtual double getTriangleMass() const = 0;
    /// Return the mass of the link body.
    virtual double getLinkMass() const = 0;
    /// Return the mass of the tierod body.
    virtual double getTierodMass() const = 0;

    /// Return the radius of the axle tube body (visualization only).
    virtual double getAxleTubeRadius() const = 0;

    /// Return the moments of inertia of the axle tube body.
    virtual const ChVector3d& getAxleTubeInertia() const = 0;
    /// Return the moments of inertia of the spindle body.
    virtual const ChVector3d& getSpindleInertia() const = 0;
    /// Return the moments of inertia of the spindle body.
    virtual const ChVector3d& getBellcrankInertia() const = 0;
    /// Return the moments of inertia of the spindle body.
    virtual const ChVector3d& getDraglinkInertia() const = 0;
    /// Return the moments of inertia of the spindle body.
    virtual const ChVector3d& getKnuckleInertia() const = 0;
    /// Return the moments of inertia of the triangle body.
    virtual const ChVector3d& getTriangleInertia() const = 0;
    /// Return the moments of inertia of the link body.
    virtual const ChVector3d& getLinkInertia() const = 0;
    /// Return the moments of inertia of the tierod body.
    virtual const ChVector3d& getTierodInertia() const = 0;

    /// Return the inertia of the axle shaft.
    virtual double getAxleInertia() const = 0;

    /// Return the free (rest) length of the spring element.
    virtual double getSpringRestLength() const = 0;
    /// Return the free (rest) length of the shock element.
    virtual double getShockRestLength() const { return 0; }
    /// Return the functor object for spring force.
    virtual std::shared_ptr<ChLinkTSDA::ForceFunctor> getSpringForceFunctor() const = 0;
    /// Return the functor object for shock force.
    virtual std::shared_ptr<ChLinkTSDA::ForceFunctor> getShockForceFunctor() const = 0;

    std::shared_ptr<ChBody> m_axleTube;    ///< axle tube body
    std::shared_ptr<ChBody> m_bellcrank;   ///< bellcrank body
    std::shared_ptr<ChBody> m_knuckle[2];  ///< knuckle body
    std::shared_ptr<ChBody> m_draglink;    ///< draglink body

    std::shared_ptr<ChLinkLockRevolute> m_revBellcrank;        ///< rotation about z-axis
    std::shared_ptr<ChLinkLockSpherical> m_sphericalDraglink;  ///< connection draglink/steering
    std::shared_ptr<ChLinkUniversal> m_universalDraglink;      ///< connection draglink/bellcrank
    std::shared_ptr<ChLinkLockRevolute> m_revKingpin[2];       ///< kingpin rotational axis joint

    std::shared_ptr<ChBody> m_triangleBody;              ///< axle guide body with spherical link and rotary link
    std::shared_ptr<ChLinkLockRevolute> m_triangleRev;   ///< triangle to chassis revolute joint
    std::shared_ptr<ChLinkLockSpherical> m_triangleSph;  ///< triangle to axle tube spherical joint

    std::shared_ptr<ChBody> m_linkBody[2];  ///< axle guide body with spherical link and universal link
    std::shared_ptr<ChLinkUniversal> m_linkBodyToChassis[2];
    std::shared_ptr<ChLinkLockSpherical> m_linkBodyToAxleTube[2];

    std::shared_ptr<ChBody> m_tierodBody[2];                          ///< tierod bodies
    std::shared_ptr<ChLinkLockSpherical> m_tierodBodyToKnuckle[2];    ///< tierod-knuckle connection
    std::shared_ptr<ChLinkLockSpherical> m_tierodBodyToBellcrank[2];  ///< tierod-bellcranck connection

    std::shared_ptr<ChLinkTSDA> m_shock[2];   ///< spring links (L/R)
    std::shared_ptr<ChLinkTSDA> m_spring[2];  ///< shock links (L/R)

  private:
    // Hardpoint absolute locations
    std::vector<ChVector3d> m_pointsL;
    std::vector<ChVector3d> m_pointsR;

    // Points for axle tube visualization
    ChVector3d m_axleOuterL;
    ChVector3d m_axleOuterR;

    // Points for triangle visualization
    ChVector3d m_triangle_sph_point;
    ChVector3d m_triangle_left_point;
    ChVector3d m_triangle_right_point;

    // Points for link visualization
    ChVector3d m_link_axleL;
    ChVector3d m_link_axleR;
    ChVector3d m_link_chassisL;
    ChVector3d m_link_chassisR;

    // Points for tierod visualization
    ChVector3d m_tierodOuterL;
    ChVector3d m_tierodOuterR;
    ChVector3d m_tierodInnerL;
    ChVector3d m_tierodInnerR;

    void InitializeSide(VehicleSide side,
                        std::shared_ptr<ChBodyAuxRef> chassis,
                        std::shared_ptr<ChBody> tierod_body,
                        const std::vector<ChVector3d>& points,
                        double ang_vel);

    static void AddVisualizationLink(std::shared_ptr<ChBody> body,
                                     const ChVector3d pt_1,
                                     const ChVector3d pt_2,
                                     double radius,
                                     const ChColor& color);

    virtual void ExportComponentList(rapidjson::Document& jsonDocument) const override;

    virtual void Output(ChVehicleOutput& database) const override;

    static const std::string m_pointNames[NUM_POINTS];
};

/// @} vehicle_wheeled_suspension

}  // end namespace vehicle
}  // end namespace chrono

#endif
