//$ nobt
//$ nocpp

/**
 * @file r8bconf.h
 *
 * @brief The "configuration" inclusion file you can modify.
 *
 * This is the "configuration" inclusion file for the "r8brain-free-src"
 * sample rate converter. You may redefine the macros here as you see fit.
 *
 * r8brain-free-src Copyright (c) 2013-2023 Aleksey Vaneev
 * See the "LICENSE" file for license.
 */

#ifndef R8BCONF_INCLUDED
#define R8BCONF_INCLUDED

#if !defined( R8BASSERT )
	/**
	 * Assertion macro used to check for certain run-time conditions. By
	 * default no action is taken if assertion fails.
	 *
	 * @param e Expression to check.
	 */

	#define R8BASSERT( e )
#endif // !defined( R8BASSERT )

#if !defined( R8BCONSOLE )
	/**
	 * Console output macro, used to output various resampler status strings,
	 * including filter design parameters, convolver parameters.
	 *
	 * @param ... Expression to send to the console, usually consists of a
	 * standard "printf" format string followed by several parameters
	 * (__VA_ARGS__).
	 */

	#define R8BCONSOLE( ... )
#endif // !defined( R8BCONSOLE )

#if !defined( R8B_BASECLASS )
	/**
	 * Macro defines the name of the class from which all classes that are
	 * designed to be created on heap are derived. The default
	 * r8b::CStdClassAllocator class uses "stdlib" memory allocation
	 * functions.
	 *
	 * The classes that are best placed on stack or as class members are not
	 * derived from any class.
	 */

	#define R8B_BASECLASS :: r8b :: CStdClassAllocator
#endif // !defined( R8B_BASECLASS )

#if !defined( R8B_MEMALLOCCLASS )
	/**
	 * Macro defines the name of the class that implements raw memory
	 * allocation functions, see the r8b::CStdMemAllocator class for details.
	 */

	#define R8B_MEMALLOCCLASS :: r8b :: CStdMemAllocator
#endif // !defined( R8B_MEMALLOCCLASS )

#if !defined( R8B_DSPBASECLASS )
	/**
	 * Macro defines the name of the class from which all CDSPProcessor
	 * objects are derived. The default value is R8B_BASECLASS. These objects
	 * are dynamically allocated, but are not cached in the global static
	 * variables.
	 */

	#define R8B_DSPBASECLASS R8B_BASECLASS
#endif // !defined( R8B_DSPBASECLASS )

#if !defined( R8B_FILTER_CACHE_MAX )
	/**
	 * This macro specifies the number of filters kept in the cache at most.
	 * The actual number can be higher if many different filters are in use at
	 * the same time.
	 */

	#define R8B_FILTER_CACHE_MAX 96
#endif // !defined( R8B_FILTER_CACHE_MAX )

#if !defined( R8B_FRACBANK_CACHE_MAX )
	/**
	 * This macro specifies the number of whole-number stepping fractional
	 * delay filter banks kept in the cache at most. The actual number can be
	 * higher if many different filter banks are in use at the same time. As
	 * filter banks are usually big objects, it is advisable to keep this
	 * cache size small.
	 */

	#define R8B_FRACBANK_CACHE_MAX 12
#endif // !defined( R8B_FRACBANK_CACHE_MAX )

#if !defined( R8B_FLTTEST )
	/**
	 * This macro, when equal to 1, enables fractional delay filter bank
	 * testing: in this mode the filter bank becomes a dynamic member of the
	 * CDSPFracInterpolator object instead of being a global static object.
	 */

	#define R8B_FLTTEST 0
#endif // !defined( R8B_FLTTEST )

#if !defined( R8B_FASTTIMING )
	/**
	 * This macro, when equal to 1, enables a fast interpolation sample
	 * timing technique. This technique improves interpolation performance
	 * (by around 10%) at the expense of a minor sample-timing drift which is
	 * on the order of 1e-6 samples per 10 billion output samples. This
	 * setting does not apply to whole-number stepping, if it is in use, as
	 * such stepping provides zero timing error without performance impact.
	 * Also does not apply to the cases when a whole-numbered (2X, 3X, etc.)
	 * resampling is in the actual use.
	 */

	#define R8B_FASTTIMING 0
#endif // !defined( R8B_FASTTIMING )

#if !defined( R8B_EXTFFT )
	/**
	 * This macro, when equal to 1, extends length of low-pass filters' FFT
	 * block by a factor of 2, by zero-padding it. This usually improves the
	 * overall time performance of the resampler at the expense of a higher
	 * overall latency (initial processing delay). If such delay is not an
	 * issue, setting this macro to 1 is preferrable. This macro can only have
	 * a value of 0 or 1.
	 */

	#define R8B_EXTFFT 0
#endif // !defined( R8B_EXTFFT )

#if !defined( R8B_IPP )
	/**
	 * Set the R8B_IPP macro definition to 1 to enable the use of Intel IPP's
	 * fast Fourier transform functions. Also uncomment and correct the IPP
	 * header inclusion macros.
	 *
	 * Do not forget to call the ippInit() function at the start of the
	 * application, before using this library's functions.
	 */

	#define R8B_IPP 0

//	#include <ippcore.h>
//	#include <ipps.h>
#endif // !defined( R8B_IPP )

#if !defined( R8B_PFFFT_DOUBLE )
	/**
	 * When defined as 1, enables PFFFT "double" routines which are fast, and
	 * which provide the highest precision.
	 */

	#define R8B_PFFFT_DOUBLE 0
#endif // !defined( R8B_PFFFT_DOUBLE )

#if !defined( R8B_PFFFT )
	/**
	 * When defined as 1, enables PFFFT routines which are fast, but which
	 * are limited to 24-bit precision. May be a good choice for time-series
	 * interpolation, when stop-band attenuation higher than 120 dB is not
	 * required.
	 */

	#define R8B_PFFFT 0
#else // !defined( R8B_PFFFT )
	/**
	 * Handle the case when both R8B_PFFFT and R8B_PFFFT_DOUBLE were enabled
	 * together by mistake.
	 */

	#if R8B_PFFFT && R8B_PFFFT_DOUBLE
		#error r8brain-free-src: R8B_PFFFT and R8B_PFFFT_DOUBLE collision.
	#endif // R8B_PFFFT && R8B_PFFFT_DOUBLE
#endif // !defined( R8B_PFFFT )

#if R8B_PFFFT
	#define R8B_FLOATFFT 1
#endif // R8B_PFFFT

#if !defined( R8B_FLOATFFT )
	/**
	 * The R8B_FLOATFFT definition enables double-to-float buffer conversions
	 * for FFT operations, for algorithms that work with "float" values. This
	 * macro should not be changed from the default "0" here.
	 */

	#define R8B_FLOATFFT 0
#endif // !defined( R8B_FLOATFFT )

#endif // R8BCONF_INCLUDED
