#!/usr/bin/env python3
#BEGIN_LEGAL
#
#Copyright (c) 2025 Intel Corporation
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
#END_LEGAL
import argparse
import sys
from cffi import FFI
from pathlib import Path
from typing import Optional, List
from dataclasses import dataclass, field

# Import strings containing function and type definitions for XED
sys.path.append(str(Path(__file__).parent)) # Root directory for the XedPy init strings
import xedpy_types  # Assumed to contain cdef_string for types (TBD autogenerated)
import xedpy_funcs  # Assumed to contain cdef_string for functions (TBD autogenerated)

xedEnum = int   # Python type hint for XED C enumerations


def load_xed_shared_library(lib_path: Path):
    """
    This function loads the XED shared library specified by the given path using
    the Foreign Function Interface (FFI). It declares the necessary XED functions,
    structs, and enums, and then loads the shared library.

    Args:
        lib_path (Path): The path to the XED shared library.

    """
    if XedPy.ffi is not None and XedPy.lib is not None:
        return  # Ensure the XED library is only initialized once
    if not lib_path.exists():
        raise FileNotFoundError(f'XED shared library not found at {lib_path}')

    ffi = FFI()
    xed_cffi_def_string = xedpy_types.cdef_string + xedpy_funcs.cdef_string

    try:
        ffi.cdef(xed_cffi_def_string)  # Declare XED functions, structs, and enums
        lib = ffi.dlopen(str(lib_path.resolve()))  # Load shared library
    except Exception as e:
        # Helper print for debugging CFFI cdef string by printing with line numbers
        print('\n=== CFFI Definition Debugging ===')
        for index, line in enumerate(xed_cffi_def_string.split('\n'), start=1):
            print(f'{index:02d}: {line}')
        print('================================\n')
        raise RuntimeError(f'Failed to load XED library: {e}')

    # Initialize XED internal library tables
    lib.xed_tables_init_py()

    # Initialize XedPy class-level attributes
    XedPy.ffi = ffi
    XedPy.lib = lib


@dataclass(frozen=True)
class XedInstructionOperand:
    operand_name: str
    operand_reg: Optional[str]
    operand_length: int
    action: str
    visibility: str
    is_mem: bool
    is_imm: bool
    imm_value: int = field(default=None)
    base_reg: str = field(default=None)
    index_reg: str = field(default=None)
    displacement: int = field(default=None)
    scale: int = field(default=None)

    def __str__(self):
        return (
            f"XedInstructionOperand(\n"
            f"  operand_name='{self.operand_name}',\n"
            f"  operand_reg='{self.operand_reg}',\n"
            f"  operand_length={self.operand_length},\n"
            f"  action='{self.action}',\n"
            f"  visibility={self.visibility},\n"
            f"  is_mem={self.is_mem},\n"
            f"  is_imm={self.is_imm},\n"
            f"  imm_value={self.imm_value},\n"
            f"  base_reg='{self.base_reg}',\n"
            f"  index_reg='{self.index_reg}',\n"
            f"  displacement={self.displacement},\n"
            f"  scale={self.scale}\n"
            f")"
        )


class XedPy:
    BUFFER_SIZE = 700  # Constant for string buffer size
    ffi = None
    lib = None
    MAX_INSTRUCTION_BYTES = 15

    def __init__(
            self, *,
            chip: str = 'ALL',
            m_mode: str = 'XED_MACHINE_MODE_LONG_64',
            addr_width: str = 'XED_ADDRESS_WIDTH_64b'
    ):
        # Create the main XED decoded object
        # `xedd` is used to hold the decoded instruction data, including all its
        # associated metadata (e.g., instruction type, operands, disassembly, etc.).
        # This object is updated with the results of each decoding operation.
        assert XedPy.ffi, f'XedPy.ffi should be initialized first, use load_xed_shared_library()'
        assert XedPy.lib, f'XedPy.lib should be initialized first, use load_xed_shared_library()'

        self.xedd = self.ffi.new('xed_decoded_inst_t*')

        # Set decoder ISA support by a given chip
        self.xed_chip: xedEnum = None
        self.set_xed_chip(chip)

        # Defaults: 64-bit mode, 64-bit address width
        self.xed_dstate = self.ffi.new('xed_state_t*')
        self.set_xed_state(self.get_xed_enum_val(m_mode), self.get_xed_enum_val(addr_width))
        # Default Intel assembly syntax
        self.disas_syntax_default: xedEnum = self.lib.XED_SYNTAX_INTEL

    def cstr_to_str(self, cstr: 'ffi.CData') -> str:
        """Convert a C char to a Python string."""
        return self.ffi.string(cstr).decode('utf-8')

    def str_to_cstr(self, string: str) -> 'ffi.CData':
        """Convert a Python string to a C char."""
        return string.encode('utf-8')

    def set_xed_state(self, machine_mode: xedEnum, address_mode: xedEnum) -> None:
        """Set the decoder state to the specified machine and address modes."""
        self.lib.xed_state_init2_py(self.xed_dstate, machine_mode, address_mode)

    def set_xed_chip(self, chip_name: str) -> None:
        """Set the decoder to use the specified XED chip."""
        chip_name = chip_name.upper() # Ensure the chip name is in uppercase
        chip: xedEnum = self.lib.str2xed_chip_enum_t_py(self.str_to_cstr(chip_name))
        if chip == 0:
            raise ValueError(f'Got an INVALID XED chip for: {chip_name}')
        self.xed_chip = chip

    def get_xed_chip(self) -> str:
        """Get the name of the current XED chip."""
        chip_name: str = self.cstr_to_str(self.lib.xed_chip_enum_t2str_py(self.xed_chip))
        return chip_name

    def get_xed_enum_val(self, enum: str) -> xedEnum:
        """Get the value of a given XED enumeration."""
        return getattr(self.lib, enum, None)

    def init_xedd(self) -> None:
        """Initialize the XED decoded instruction object with the current state and chip."""
        self.lib.xed_decoded_inst_zero_set_mode_py(self.xedd, self.xed_dstate)
        self.lib.xed_decoded_inst_set_input_chip_py(self.xedd, self.xed_chip)

    def decode_instruction(self, instruction_bytes: bytes) -> None:
        """
        Decode a given instruction byte sequence using XED. Raise exception in case decode fails

        Args:
            instruction_bytes (bytes): The raw instruction bytes to be decoded.
        """
        self.init_xedd()
        ilen = min(len(instruction_bytes), self.MAX_INSTRUCTION_BYTES)
        err: xedEnum = self.lib.xed_decode_py(self.xedd, instruction_bytes, ilen)
        if err != self.lib.XED_ERROR_NONE:
            if err is None:
                raise RuntimeError('XED decode_instruction returned None. Aborting.')
            error_string: str = self.cstr_to_str(self.lib.xed_error_enum_t2str_py(err))
            raise RuntimeError(f'XED ERROR: {error_string}')

    def get_iclass(self) -> str:
        iclass: xedEnum = self.lib.xed_decoded_inst_get_iclass_py(self.xedd)
        return self.cstr_to_str(self.lib.xed_iclass_enum_t2str_py(iclass))

    def get_category(self) -> str:
        category: xedEnum = self.lib.xed_decoded_inst_get_category_py(self.xedd)
        return self.cstr_to_str(self.lib.xed_category_enum_t2str_py(category))

    def get_extension(self) -> str:
        extension: xedEnum = self.lib.xed_decoded_inst_get_extension_py(self.xedd)
        return self.cstr_to_str(self.lib.xed_extension_enum_t2str_py(extension))

    def get_iform(self) -> str:
        iform: xedEnum = self.lib.xed_decoded_inst_get_iform_enum_py(self.xedd)
        return self.cstr_to_str(self.lib.xed_iform_enum_t2str_py(iform))

    def get_isa_set(self) -> str:
        isa_set: xedEnum = self.lib.xed_decoded_inst_get_isa_set_py(self.xedd)
        return self.cstr_to_str(self.lib.xed_isa_set_enum_t2str_py(isa_set))

    def print_decoded_instruction_info(self) -> None:
        """Print detailed information about the decoded instruction."""
        PAD = 12

        print(f'{"ICLASS":<{PAD}}: {self.get_iclass()}')
        print(f'{"CATEGORY":<{PAD}}: {self.get_category()}')
        print(f'{"EXTENSION":<{PAD}}: {self.get_extension()}')
        print(f'{"IFORM":<{PAD}}: {self.get_iform()}')
        print(f'{"ISA_SET":<{PAD}}: {self.get_isa_set()}')

        # Allocate a buffer for the instruction format dump
        buffer = self.ffi.new(f'char[{self.BUFFER_SIZE}]')
        self.lib.xed_decoded_inst_dump_xed_format_py(self.xedd, buffer, self.BUFFER_SIZE, 0)
        print('\n==== XED Operands: ====')
        print(self.cstr_to_str(buffer), end='\n\n')

    def disassemble(self, syntax: xedEnum = None) -> str:
        """Return the assembly string of a decoded xedd instruction."""
        buffer = self.ffi.new(f'char[{self.BUFFER_SIZE}]')
        # Init the xed_print_info_t struct
        pi = self.ffi.new('xed_print_info_t*')
        self.lib.xed_init_print_info_py(pi)
        pi.p = self.xedd
        pi.blen = self.BUFFER_SIZE
        pi.buf = buffer
        if not syntax:
            syntax = self.disas_syntax_default
        pi.syntax = syntax

        ok = self.lib.xed_format_generic_py(pi)  # Disassemble the decoded instruction
        if not ok:
            raise RuntimeError('XED disassemble ERROR')

        return self.cstr_to_str(buffer)

    def get_operands(self) -> List[XedInstructionOperand]:
        instr_operands = []

        xi = self.lib.xed_decoded_inst_inst_py(self.xedd)
        noperands = self.lib.xed_inst_noperands_py(xi)

        mem_operand_index = 0
        for op_num in range(noperands):
            xed_operand_t = self.lib.xed_inst_operand_py(self.xedd._inst, op_num)

            op_enum = self.lib.xed_operand_name_py(xed_operand_t)
            op_c_str = self.lib.xed_operand_enum_t2str_py(op_enum)
            op_str = self.cstr_to_str(op_c_str)

            reg_enum = self.lib.xed_decoded_inst_get_reg_py(self.xedd, op_enum)
            reg_c_str = self.lib.xed_reg_enum_t2str_py(reg_enum)
            reg_str = self.cstr_to_str(reg_c_str)

            visibility_enum = self.lib.xed_operand_operand_visibility_py(xed_operand_t)
            visibility_c_str = self.lib.xed_operand_visibility_enum_t2str_py(visibility_enum)
            visibility_str = self.cstr_to_str(visibility_c_str)

            action_enum = self.lib.xed_decoded_inst_operand_action_py(self.xedd, op_num)
            action_c_str = self.lib.xed_operand_action_enum_t2str_py(action_enum)
            action_str = self.cstr_to_str(action_c_str)

            operand_length = self.lib.xed_decoded_inst_operand_length_bits_py(self.xedd, op_num)
            is_mem = op_str in {"AGEN", "MEM0", "MEM1"}  # determine if the operand is a memory operand
            is_imm = op_str in {"IMM0", "IMM1"}  # determine if the operand is an immediate operand

            if is_imm:
                if op_str =="IMM0":
                    if self.lib.xed_decoded_inst_get_immediate_is_signed_py(self.xedd):
                        imm_val = self.lib.xed_decoded_inst_get_signed_immediate_py(self.xedd)
                    else:
                        imm_val = self.lib.xed_decoded_inst_get_unsigned_immediate_py(self.xedd)
                else:
                    imm_val = self.lib.xed_decoded_inst_get_second_immediate_py(self.xedd)

                operand_struct = XedInstructionOperand(
                    operand_name=op_str,
                    operand_reg=None,
                    operand_length=operand_length,
                    action=action_str,
                    visibility=visibility_str,
                    is_mem=is_mem,
                    is_imm=is_imm,
                    imm_value=imm_val
                )
            elif not is_mem:
                operand_struct = XedInstructionOperand(
                    operand_name=op_str,
                    operand_reg=reg_str,
                    operand_length=operand_length,
                    action=action_str,
                    visibility=visibility_str,
                    is_mem=is_mem,
                    is_imm=is_imm,
                )
            else:
                base_reg_enum = self.lib.xed_decoded_inst_get_base_reg_py(self.xedd, mem_operand_index)
                base_reg_c_str = self.lib.xed_reg_enum_t2str_py(base_reg_enum)
                base_reg_str = self.cstr_to_str(base_reg_c_str)
                base_reg_str = base_reg_str if base_reg_str != "INVALID" else None

                index_reg_enum = self.lib.xed_decoded_inst_get_index_reg_py(self.xedd, mem_operand_index)
                index_reg_c_str = self.lib.xed_reg_enum_t2str_py(index_reg_enum)
                index_reg_str = self.cstr_to_str(index_reg_c_str)
                # only first operand can be index operand
                if mem_operand_index == 0 and index_reg_str != "INVALID":
                    # only have a scale if the index exists
                    scale = self.lib.xed_decoded_inst_get_scale_py(self.xedd, mem_operand_index)
                else:
                    index_reg_str = None
                    scale = None

                disp = None
                if self.lib.xed_decoded_inst_get_memory_displacement_width_py(self.xedd, mem_operand_index):
                    disp = self.lib.xed_decoded_inst_get_memory_displacement_py(self.xedd, mem_operand_index)

                operand_struct = XedInstructionOperand(
                    operand_name=op_str,
                    operand_reg=None,
                    operand_length=operand_length,
                    action=action_str,
                    visibility=visibility_str,
                    is_mem=is_mem,
                    is_imm=is_imm,
                    base_reg=base_reg_str,
                    index_reg=index_reg_str,
                    scale=scale,
                    displacement=disp,
                )

                mem_operand_index += 1

            instr_operands.append(operand_struct)

        return instr_operands


def parse_arguments() -> argparse.Namespace:
    """Parse command-line arguments"""
    parser = argparse.ArgumentParser(description='Python example using XED via CFFI')
    parser.add_argument('--xed-lib',
                        dest='lib_path',
                        help='Path to XED shared object library.',
                        type=Path,
                        default=None)
    args = parser.parse_args()

    # Default to the expected library path
    if args.lib_path is None:
        xed_root: Path = Path(__file__).parents[2].resolve()
        args.lib_path = xed_root / 'obj' / 'xed.dll'
        if not args.lib_path.exists():
            args.lib_path = args.lib_path.with_name('libxed.so')
        assert args.lib_path.exists(), f'Could not find XED library {args.lib_path}'

    return args


def example_run() -> None:
    """Example function to handle library loading, instruction decoding, and output."""
    args = parse_arguments()
    load_xed_shared_library(args.lib_path) # load the XED library and init CFFi binding

    xed = XedPy()

    ### Change XedPy default configs:
    # xed.set_xed_state(xed.lib.XED_MACHINE_MODE_LONG_64, xed.lib.XED_ADDRESS_WIDTH_64b)
    # xed.set_xed_chip('FUTURE')
    ###
    print(f'[Decoder Chip] {xed.get_xed_chip()}')
    # Define the raw instruction bytes to be decoded
    instruction_bytes = bytes([213, 17, 255, 16, 195, 0x66, 0xd5, 0x80, 0x12, 0x4C, 0x83, 0x00, 0x00, 0x00, 98, 91,
                               253, 59, 0, 188, 60, 232, 206, 61, 110, 206])
    bytes_len = len(instruction_bytes)
    inst_len = 0
    instruction_view = memoryview(instruction_bytes) # Use memoryview for efficient slicing

    while inst_len < bytes_len:
        inst_p = instruction_view[inst_len : inst_len + xed.MAX_INSTRUCTION_BYTES].tobytes()

        # Decode the instruction
        xed.decode_instruction(inst_p)

        # Get the length of the decoded instruction
        decoded_length = xed.lib.xed_decoded_inst_get_length_py(xed.xedd)
        # Check for zero-length decoded instruction to avoid infinite loops
        if decoded_length == 0:
            raise RuntimeError(f'Decoded instruction has zero length. Aborting. Instruction bytes: {inst_p.hex()}')
        # Update the memoryview starting index
        inst_len += decoded_length

        print(f'\nDecoded Bytes: {inst_p[:decoded_length].hex()}')

        print(f'Disassembly: {xed.disassemble()}')

        print('Operands detailed information:')
        operands = xed.get_operands()
        for op in operands:
            print(op)

    # Print detailed information about the decoded instruction
    print('\n[Decoded Instruction Information]')
    xed.print_decoded_instruction_info()

    print('\n[Disassembly Output]')
    for syntax in ['INTEL', 'ATT']:
        disasm = xed.disassemble(xed.get_xed_enum_val(f'XED_SYNTAX_{syntax}'))
        print(f'Disassembly ({syntax}): {disasm}')

    # Extending functionality beyond xedpy's built-in methods:
    # This demonstrates how to directly access the underlying XED C API via CFFI.
    # Below, we use `xed_classify_apx_py()` to determine whether an instruction
    # belongs to the APX instruction set. This method allows extending features
    # not natively provided by the xedpy wrapper.
    apx: bool = XedPy.lib.xed_classify_apx_py(xed.xedd)
    print(f'\n[Direct XED C API Call] xed_classify_apx_py(): returns {apx}')


if __name__ == '__main__':
    example_run()
