package delete

import (
	"fmt"
	"strings"
	"testing"

	"gitlab.com/gitlab-org/cli/pkg/iostreams"

	"github.com/stretchr/testify/assert"
	gitlab "gitlab.com/gitlab-org/api/client-go"
	"gitlab.com/gitlab-org/cli/api"
	"gitlab.com/gitlab-org/cli/commands/cmdtest"
)

func TestMain(m *testing.M) {
	cmdtest.InitTest(m, "issue_delete_test")
}

func TestNewCmdDelete(t *testing.T) {
	t.Parallel()
	oldDeleteMR := api.DeleteMR

	api.DeleteIssue = func(client *gitlab.Client, projectID any, issueID int) error {
		if projectID == "" || projectID == "NAMESPACE/WRONG_REPO" || projectID == "expected_err" || issueID == 0 {
			return fmt.Errorf("error expected")
		}
		return nil
	}
	api.GetIssue = func(client *gitlab.Client, projectID any, issueID int) (*gitlab.Issue, error) {
		if projectID == "" || projectID == "WRONG_REPO" || projectID == "expected_err" {
			return nil, fmt.Errorf("error expected")
		}
		return &gitlab.Issue{
			IID: issueID,
		}, nil
	}

	tests := []struct {
		name       string
		args       []string
		wantErr    bool
		errMsg     string
		assertFunc func(*testing.T, string, string)
	}{
		{
			name:    "delete",
			args:    []string{"0", "-R", "NAMESPACE/WRONG_REPO"},
			wantErr: true,
		},
		{
			name:    "id exists",
			args:    []string{"1"},
			wantErr: false,
			assertFunc: func(t *testing.T, out string, err string) {
				assert.Contains(t, err, "✓ Issue deleted.\n")
			},
		},
		{
			name:    "delete on different repo",
			args:    []string{"12", "-R", "profclems/glab"},
			wantErr: false,
			assertFunc: func(t *testing.T, out string, stderr string) {
				assert.Contains(t, stderr, "✓ Issue deleted.\n")
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			io, _, stdout, stderr := iostreams.Test()
			io.IsaTTY = true
			io.IsErrTTY = true
			f := cmdtest.StubFactory("", io)

			cmd := NewCmdDelete(f)

			cmd.Flags().StringP("repo", "R", "", "")

			cli := strings.Join(tt.args, " ")
			t.Log(cli)
			_, err := cmdtest.RunCommand(cmd, cli)
			if !tt.wantErr {
				assert.Nil(t, err)
				tt.assertFunc(t, stdout.String(), stderr.String())
			} else {
				assert.NotNil(t, err)
			}
		})
	}

	api.DeleteMR = oldDeleteMR
}
