#.rst:
# FindOracle
# ----------
#
# Configure Oracle libraries
#
# ORACLE_FOUND - system has Oracle OCI
# ORACLE_INCLUDE_DIRS - where to find oci.h
# ORACLE_LIBRARIES - the libraries to link against to use Oracle OCI
#
# Components: CLNTSH, CLNTST, OCCI, XML, OCIEI
# CLNTSH is enabled by default
# for example: FIND_PACKAGE(Oracle REQUIRED CLNTSH OCCI)
#
# This module will set the following variables in your project:
#
# ``ORACLE_FOUND``
#   Oracle installation detected (InstantClient or "thick" client)
# ``ORACLE_INCLUDE_DIRS``
#   Oracle include directory (where oci.h resides)
# ``ORACLE_CLNTSH_LIBRARY``
#   Library libclntsh.so.*
# ``ORACLE_CLNTST_LIBRARY``
#   Library libclntst??.a (not present in InstantClient, must be generated by executing genclntst)
# ``ORACLE_OCCI_LIBRARY``
#   OCCI C++ client library
# ``ORACLE_XML_LIBRARY``
#   Oracle XMLTYPE library libxml?.a (usually static, only present in thick client)
# ``ORACLE_LIBRARIES``
#   All Oracle libraries detected libclntsh.so.*, libociei.so.8 libxml.a ....
# ``ORACLE_OCI_VERSION``
#   This is set to 8i, 9i, 10g, 11g, 12c

# TOra Project: Configure Oracle libraries
#
# Copyright (c) 2007 Petr Vanek <petr@yarpen.cz>
# Copyright (c) 2016 Ivan Brezina <ibre5041@ibrezina.net>
# Redistribution and use is allowed according to the terms of the BSD license.
#

# Port to GDAL
# Copyright (c) 2018 Hiroshi Miura
#

if(DEFINED Oracle_ROOT)
    set(ORACLE_HOME "${Oracle_ROOT}")
elseif(DEFINED ORACLE_ROOT)
    set(ORACLE_HOME "${ORACLE_ROOT}")
else()
    file(TO_CMAKE_PATH "$ENV{ORACLE_HOME}" ORACLE_HOME)
endif()
get_filename_component(XE_HOME "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\KEY_XE;ORACLE_HOME]" ABSOLUTE CACHE)
mark_as_advanced(XE_HOME)

set(ORACLE_INCLUDES_LOCATION
        ${ORACLE_HOME}/rdbms/public
        ${ORACLE_HOME}/include
        # xdk
        ${ORACLE_HOME}/xdk/include
        # sdk
        ${ORACLE_HOME}/sdk/include/
        # xe on windows
        ${ORACLE_HOME}/OCI/include/
        # instant client from rpm
        /usr/include/oracle/11.2/client${LIB_SUFFIX}/
        /usr/include/oracle/12.1/client${LIB_SUFFIX}/
        # instantclient on OS X from macports
        /opt/local/lib/oracle/sdk/include/
        # Oracle XE Windows
        ${XE_HOME}/oci/include/
   )

find_path(Oracle_INCLUDE_DIR oci.h
          PATHS ${ORACLE_INCLUDES_LOCATION}
          HINTS ${ORACLE_PATH_LIB}
          NO_DEFAULT_PATH NO_CMAKE_ENVIRONMENT_PATH NO_CMAKE_PATH NO_SYSTEM_ENVIRONMENT_PATH NO_CMAKE_SYSTEM_PATH
)

# guess OCI version
#   we need it before lib checking because of OS X bugs in instant client (see below)
if (NOT DEFINED ORACLE_OCI_VERSION AND ORACLE_INCLUDES)
  try_run(OCIVER_TYPE OCIVER_COMPILED
          ${PROJECT_BINARY_DIR}
          ${PROJECT_SOURCE_DIR}/cmake/modules/OCIVersion.c
          CMAKE_FLAGS "-DINCLUDE_DIRECTORIES:STRING=${ORACLE_INCLUDES}"
          OUTPUT_VARIABLE OCIVER)

  if(${OCIVER_TYPE} GREATER "120")
    set(ORACLE_OCI_VERSION "12C")
  elseif(${OCIVER_TYPE} GREATER "110")
    set(ORACLE_OCI_VERSION "11G")
  elseif(${OCIVER_TYPE} EQUAL "102")
    set(ORACLE_OCI_VERSION "10G_R2")
  elseif(${OCIVER_TYPE} EQUAL "101")
    set(ORACLE_OCI_VERSION "10G")
  elseif(${OCIVER_TYPE} EQUAL "91")
    set(ORACLE_OCI_VERSION "9")
  elseif(${OCIVER_TYPE} EQUAL "92")
    set(ORACLE_OCI_VERSION "9")
  elseif(NOT ${OCIVER_TYPE} GREATER "90")
    set(ORACLE_OCI_VERSION "8I")
  else(${OCIVER_TYPE} GREATER "110")
    set(ORACLE_OCI_VERSION "10G_R2")
  endif ()

  message(STATUS "Guessed ORACLE_OCI_VERSION value: ${ORACLE_OCI_VERSION} for ${OCIVER_TYPE}")

  if(APPLE)
    if(${OCIVER_TYPE} LESS 110)
        list(FIND Oracle_FIND_COMPONENTS "OCIEI" _nextcomp)
        if(_nextcomp EQUAL -1)
            list(APPEND Oracle_FIND_COMPONENTS OCIEI)
            message(STATUS "    APPLE requires libociei for instantclient < 11, enforcing ociei")
        endif()
    endif()
  endif()
endif()

set(ORACLE_LIB_LOCATION
        # TODO add 32/64bit version here (lib32/lib64 on some platforms)
        ${ORACLE_HOME}/lib
        # xe on windows
        ${ORACLE_HOME}/OCI/lib/MSVC/
        ${ORACLE_HOME}
        # InstantClient on WinXP
        ${ORACLE_HOME}/sdk/lib/msvc/
        # Instant client from rpm
        /usr/lib${LIB_SUFFIX}/
        /usr/lib/oracle/11.2/client${LIB_SUFFIX}/lib/
        /usr/lib/oracle/12.1/client${LIB_SUFFIX}/lib/
        # instant client on OS X from macports
        /opt/local/lib/oracle/
        # Oracle XE Windows
        ${XE_HOME}/oci/lib/MSVC/
    )

# The NO_DEFAULT_PATH is necessary here in the case there is Oracle DB
# *and* instant client installed. The order is given in ORACLE_LIB_LOCATION.
# See above.
find_library(Oracle_LIBRARY
             NAMES oci
             PATHS ${ORACLE_LIB_LOCATION}
             HINTS ${ORACLE_PATH_LIBS}
             NO_DEFAULT_PATH
             )

# Oracle InstantClient 12.1 has no liboci. It seems we just need libclntsh
# for the GDAL drivers.
if( Oracle_CAN_USE_CLNTSH_AS_MAIN_LIBRARY AND "${Oracle_FIND_COMPONENTS}" STREQUAL "")
    set(Oracle_FIND_COMPONENTS CLNTSH)
endif()

foreach(_comp IN LISTS Oracle_FIND_COMPONENTS)
  set(Oracle_${_comp}_FOUND FALSE)
endforeach()

set(Oracle_known_components CLNTSH CLNTST OCCI XML OCIEI)

set(Oracle_CLNTSH_lib libclntsh clntsh)
set(Oracle_CLNTST_lib libclntst libclntst9 libclntst10 libclntst11 libclntst12 clntst clntst9 clntst10 clntst11 clntst12)
set(Oracle_OCCI_lib libocci occi)
set(Oracle_XML_lib
    libxml xml libxml10 xml10 libxml11 xml11 oraxml10 oraxml11)
set(Oracle_XML_header xml.h)
set(Oracle_OCIEI_lib libociei ociei)

foreach(_comp IN LISTS Oracle_known_components)
    list(FIND Oracle_FIND_COMPONENTS "${_comp}" _nextcomp)
    if(_nextcomp GREATER -1)
        find_library(Oracle_${_comp}_LIBRARY
                     NAMES ${Oracle_${_comp}_lib}
                     PATHS ${ORACLE_LIB_LOCATION}
                     NO_DEFAULT_PATH
        )
        mark_as_advanced(Oracle_${_comp}_LIBRARY)
        if(Oracle_${_comp}_header)
            find_path(Oracle_${_comp}_INCLUDE_DIR
                      NAMES ${Oracle_${_comp}_header}
                      PATHS ${ORACLE_INCLUDES_LOCATION}
                      NO_DEFAULT_PATH)
            mark_as_advanced(Oracle_${_comp}_INCLUDE_DIR)
        endif()
    endif()
endforeach()

if( NOT Oracle_LIBRARY AND Oracle_CAN_USE_CLNTSH_AS_MAIN_LIBRARY AND Oracle_CLNTSH_LIBRARY )
    set(Oracle_LIBRARY "${Oracle_CLNTSH_LIBRARY}")
endif()

include(FindPackageHandleStandardArgs)
find_package_handle_standard_args(Oracle
                                  FOUND_VAR Oracle_FOUND
                                  REQUIRED_VARS Oracle_LIBRARY Oracle_INCLUDE_DIR
                                  VERSION_VAR ORACLE_OCI_VERSION
                                  HANDLE_COMPONENTS)
mark_as_advanced(Oracle_INCLUDE_DIR Oracle_LIBRARY)

IF(Oracle_FOUND)
    set(Oracle_INCLUDE_DIRS "${Oracle_INCLUDE_DIR}")
    set(Oracle_LIBRARIES "${Oracle_LIBRARY}")
    if(Oracle_XML_INCLUDE_DIR AND Oracle_XML_LIBRARY)
        list(APPEND Oracle_INCLUDE_DIRS "${Oracle_XML_INCLUDE_DIR}")
    endif()
    if(NOT TARGET Oracle::OCI)
        add_library(Oracle::OCI UNKNOWN IMPORTED)
        set_target_properties(Oracle::OCI PROPERTIES
                              INTERFACE_INCLUDE_DIRECTORIES "${Oracle_INCLUDE_DIR}"
                              IMPORTED_LINK_INTERFACE_LANGUAGES "C"
                              IMPORTED_LOCATION "${Oracle_LIBRARY}"
                              )
        foreach(tgt IN LISTS Oracle_known_components)
            if(Oracle_${tgt}_FOUND)
                add_library(Oracle::${tgt} UNKNOWN IMPORTED)
                set_target_properties(Oracle::${tgt} PROPERTIES
                                      IMPORTED_LINK_INTERFACE_LANGUAGES "C"
                                      IMPORTED_LOCATION "${Oracle_${tgt}_LIBRARY}")
            endif()
        endforeach()
        if(Oracle_XML_INCLUDE_DIR)
            if(TARGET Oracle::XML)
                set_property(Oracle::XML APPEND PROPERTY
                             INTERFACE_INCLUDE_DIRECTORIES "${Oracle_XML_INCLUDE_DIR}")
            endif()
        endif()
    endif()
endif()
