/**
 * FreeRDP: A Remote Desktop Protocol Implementation
 * NCrush (RDP6) Bulk Data Compression
 *
 * Copyright 2014 Marc-Andre Moreau <marcandre.moreau@gmail.com>
 * Copyright 2017 Armin Novak <armin.novak@thincast.com>
 * Copyright 2017 Thincast Technologies GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <freerdp/config.h>

#include <winpr/assert.h>
#include <winpr/cast.h>
#include <winpr/crt.h>
#include <winpr/print.h>
#include <winpr/bitstream.h>

#include <freerdp/log.h>
#include <freerdp/types.h>

#include "ncrush.h"

#define TAG FREERDP_TAG("codec")

struct s_NCRUSH_CONTEXT
{
	ALIGN64 BOOL Compressor;
	ALIGN64 BYTE* HistoryPtr;
	ALIGN64 UINT32 HistoryOffset;
	ALIGN64 UINT32 HistoryEndOffset;
	ALIGN64 UINT32 HistoryBufferSize;
	ALIGN64 BYTE HistoryBuffer[65536];
	ALIGN64 UINT32 HistoryBufferFence;
	ALIGN64 UINT32 OffsetCache[4];
	ALIGN64 UINT16 HashTable[65536];
	ALIGN64 UINT16 MatchTable[65536];
	ALIGN64 BYTE HuffTableCopyOffset[1024];
	ALIGN64 BYTE HuffTableLOM[4096];
};

static const UINT16 HuffTableLEC[8192] = {
	0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA068, 0x5111, 0x7007, 0x6113, 0x90C0,
	0x6108, 0x8018, 0x611B, 0xA0B3, 0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA091,
	0x5121, 0x7080, 0x6115, 0xA03A, 0x610A, 0x9012, 0x611D, 0xA0D7, 0x510B, 0x6122, 0x610E, 0x9035,
	0x6001, 0x7123, 0x6118, 0xA07A, 0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C4,
	0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A1, 0x5121, 0x7102, 0x6116, 0xA056,
	0x610C, 0x901D, 0x611E, 0xA0E8, 0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA071,
	0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BB, 0x510F, 0x7004, 0x6110, 0x9049,
	0x6002, 0x800D, 0x6119, 0xA099, 0x5121, 0x70FF, 0x6115, 0xA04C, 0x610A, 0x9017, 0x611D, 0xA0DF,
	0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA087, 0x5111, 0x700A, 0x6114, 0xA023,
	0x6109, 0x80FE, 0x611C, 0xA0CE, 0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0A9,
	0x5121, 0x7103, 0x6116, 0xA05F, 0x610C, 0x9022, 0x611E, 0xA0F5, 0x510B, 0x611F, 0x610D, 0x9029,
	0x6000, 0x7105, 0x6117, 0xA06C, 0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B7,
	0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA095, 0x5121, 0x7080, 0x6115, 0xA046,
	0x610A, 0x9015, 0x611D, 0xA0DB, 0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA07E,
	0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0C9, 0x510F, 0x7005, 0x6112, 0x907F,
	0x6107, 0x8010, 0x611A, 0xA0A5, 0x5121, 0x7102, 0x6116, 0xA05B, 0x610C, 0x901F, 0x611E, 0xA0EC,
	0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA075, 0x5111, 0x7008, 0x6113, 0x90F1,
	0x6108, 0x8040, 0x611B, 0xA0BF, 0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09D,
	0x5121, 0x70FF, 0x6115, 0xA052, 0x610A, 0x901B, 0x611D, 0xA0E4, 0x510B, 0x6122, 0x610E, 0x903F,
	0x6001, 0x7124, 0x6118, 0xA08C, 0x5111, 0x700A, 0x6114, 0xA02F, 0x6109, 0x8120, 0x611C, 0xA0D3,
	0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AE, 0x5121, 0x7103, 0x6116, 0xA064,
	0x610C, 0x9025, 0x611E, 0xA0FA, 0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06A,
	0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B5, 0x510F, 0x7003, 0x6110, 0x9043,
	0x6002, 0x800B, 0x6119, 0xA093, 0x5121, 0x7080, 0x6115, 0xA03D, 0x610A, 0x9014, 0x611D, 0xA0D9,
	0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07C, 0x5111, 0x7009, 0x6114, 0x90F8,
	0x6109, 0x8060, 0x611C, 0xA0C7, 0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A3,
	0x5121, 0x7102, 0x6116, 0xA058, 0x610C, 0x901E, 0x611E, 0xA0EA, 0x510B, 0x611F, 0x610D, 0x9030,
	0x6000, 0x7106, 0x6117, 0xA073, 0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BD,
	0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09B, 0x5121, 0x70FF, 0x6115, 0xA04E,
	0x610A, 0x901A, 0x611D, 0xA0E2, 0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08A,
	0x5111, 0x700A, 0x6114, 0xA02D, 0x6109, 0x80FE, 0x611C, 0xA0D1, 0x510F, 0x7006, 0x6112, 0x9084,
	0x6107, 0x8011, 0x611A, 0xA0AC, 0x5121, 0x7103, 0x6116, 0xA062, 0x610C, 0x9024, 0x611E, 0xA0F7,
	0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06E, 0x5111, 0x7007, 0x6113, 0x90D0,
	0x6108, 0x8019, 0x611B, 0xA0B9, 0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA097,
	0x5121, 0x7080, 0x6115, 0xA04A, 0x610A, 0x9016, 0x611D, 0xA0DD, 0x510B, 0x6122, 0x610E, 0x9039,
	0x6001, 0x7123, 0x6118, 0xA085, 0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CB,
	0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A7, 0x5121, 0x7102, 0x6116, 0xA05D,
	0x610C, 0x9021, 0x611E, 0xA0EF, 0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA077,
	0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C2, 0x510F, 0x7004, 0x6110, 0x9059,
	0x6002, 0x800E, 0x6119, 0xA09F, 0x5121, 0x70FF, 0x6115, 0xA054, 0x610A, 0x901C, 0x611D, 0xA0E6,
	0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08E, 0x5111, 0x700A, 0x6114, 0xA034,
	0x6109, 0x8120, 0x611C, 0xA0D5, 0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B0,
	0x5121, 0x7103, 0x6116, 0xA066, 0x610C, 0x9026, 0x611E, 0xA104, 0x510B, 0x611F, 0x610D, 0x9027,
	0x6000, 0x7105, 0x6117, 0xA069, 0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B4,
	0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA092, 0x5121, 0x7080, 0x6115, 0xA03B,
	0x610A, 0x9012, 0x611D, 0xA0D8, 0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07B,
	0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C5, 0x510F, 0x7005, 0x6112, 0x9070,
	0x6107, 0x800F, 0x611A, 0xA0A2, 0x5121, 0x7102, 0x6116, 0xA057, 0x610C, 0x901D, 0x611E, 0xA0E9,
	0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA072, 0x5111, 0x7008, 0x6113, 0x90E0,
	0x6108, 0x8020, 0x611B, 0xA0BC, 0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA09A,
	0x5121, 0x70FF, 0x6115, 0xA04D, 0x610A, 0x9017, 0x611D, 0xA0E1, 0x510B, 0x6122, 0x610E, 0x903C,
	0x6001, 0x7124, 0x6118, 0xA089, 0x5111, 0x700A, 0x6114, 0xA02B, 0x6109, 0x80FE, 0x611C, 0xA0CF,
	0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0AA, 0x5121, 0x7103, 0x6116, 0xA061,
	0x610C, 0x9022, 0x611E, 0xA0F6, 0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06D,
	0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B8, 0x510F, 0x7003, 0x6110, 0x9044,
	0x6002, 0x800C, 0x6119, 0xA096, 0x5121, 0x7080, 0x6115, 0xA047, 0x610A, 0x9015, 0x611D, 0xA0DC,
	0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA083, 0x5111, 0x7009, 0x6114, 0x90FC,
	0x6109, 0x80F0, 0x611C, 0xA0CA, 0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A6,
	0x5121, 0x7102, 0x6116, 0xA05C, 0x610C, 0x901F, 0x611E, 0xA0ED, 0x510B, 0x611F, 0x610D, 0x9031,
	0x6000, 0x7106, 0x6117, 0xA076, 0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0C1,
	0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09E, 0x5121, 0x70FF, 0x6115, 0xA053,
	0x610A, 0x901B, 0x611D, 0xA0E5, 0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08D,
	0x5111, 0x700A, 0x6114, 0xA032, 0x6109, 0x8120, 0x611C, 0xA0D4, 0x510F, 0x7006, 0x6112, 0x9088,
	0x6107, 0x8013, 0x611A, 0xA0AF, 0x5121, 0x7103, 0x6116, 0xA065, 0x610C, 0x9025, 0x611E, 0xA0FB,
	0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06B, 0x5111, 0x7007, 0x6113, 0x90C6,
	0x6108, 0x8018, 0x611B, 0xA0B6, 0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA094,
	0x5121, 0x7080, 0x6115, 0xA045, 0x610A, 0x9014, 0x611D, 0xA0DA, 0x510B, 0x6122, 0x610E, 0x9037,
	0x6001, 0x7123, 0x6118, 0xA07D, 0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C8,
	0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A4, 0x5121, 0x7102, 0x6116, 0xA05A,
	0x610C, 0x901E, 0x611E, 0xA0EB, 0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA074,
	0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BE, 0x510F, 0x7004, 0x6110, 0x9050,
	0x6002, 0x800D, 0x6119, 0xA09C, 0x5121, 0x70FF, 0x6115, 0xA04F, 0x610A, 0x901A, 0x611D, 0xA0E3,
	0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08B, 0x5111, 0x700A, 0x6114, 0xA02E,
	0x6109, 0x80FE, 0x611C, 0xA0D2, 0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AD,
	0x5121, 0x7103, 0x6116, 0xA063, 0x610C, 0x9024, 0x611E, 0xA0F9, 0x510B, 0x611F, 0x610D, 0x902A,
	0x6000, 0x7105, 0x6117, 0xA06F, 0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0BA,
	0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA098, 0x5121, 0x7080, 0x6115, 0xA04B,
	0x610A, 0x9016, 0x611D, 0xA0DE, 0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA086,
	0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CD, 0x510F, 0x7005, 0x6112, 0x9081,
	0x6107, 0x8010, 0x611A, 0xA0A8, 0x5121, 0x7102, 0x6116, 0xA05E, 0x610C, 0x9021, 0x611E, 0xA0F3,
	0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA079, 0x5111, 0x7008, 0x6113, 0x90F2,
	0x6108, 0x8040, 0x611B, 0xA0C3, 0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA0A0,
	0x5121, 0x70FF, 0x6115, 0xA055, 0x610A, 0x901C, 0x611D, 0xA0E7, 0x510B, 0x6122, 0x610E, 0x9041,
	0x6001, 0x7124, 0x6118, 0xA08F, 0x5111, 0x700A, 0x6114, 0xA036, 0x6109, 0x8120, 0x611C, 0xA0D6,
	0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B1, 0x5121, 0x7103, 0x6116, 0xA067,
	0x610C, 0x9026, 0x611E, 0xB0B2, 0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA068,
	0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B3, 0x510F, 0x7003, 0x6110, 0x9042,
	0x6002, 0x800B, 0x6119, 0xA091, 0x5121, 0x7080, 0x6115, 0xA03A, 0x610A, 0x9012, 0x611D, 0xA0D7,
	0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07A, 0x5111, 0x7009, 0x6114, 0x90F4,
	0x6109, 0x8060, 0x611C, 0xA0C4, 0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A1,
	0x5121, 0x7102, 0x6116, 0xA056, 0x610C, 0x901D, 0x611E, 0xA0E8, 0x510B, 0x611F, 0x610D, 0x902C,
	0x6000, 0x7106, 0x6117, 0xA071, 0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BB,
	0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA099, 0x5121, 0x70FF, 0x6115, 0xA04C,
	0x610A, 0x9017, 0x611D, 0xA0DF, 0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA087,
	0x5111, 0x700A, 0x6114, 0xA023, 0x6109, 0x80FE, 0x611C, 0xA0CE, 0x510F, 0x7006, 0x6112, 0x9082,
	0x6107, 0x8011, 0x611A, 0xA0A9, 0x5121, 0x7103, 0x6116, 0xA05F, 0x610C, 0x9022, 0x611E, 0xA0F5,
	0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06C, 0x5111, 0x7007, 0x6113, 0x90CC,
	0x6108, 0x8019, 0x611B, 0xA0B7, 0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA095,
	0x5121, 0x7080, 0x6115, 0xA046, 0x610A, 0x9015, 0x611D, 0xA0DB, 0x510B, 0x6122, 0x610E, 0x9038,
	0x6001, 0x7123, 0x6118, 0xA07E, 0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0C9,
	0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A5, 0x5121, 0x7102, 0x6116, 0xA05B,
	0x610C, 0x901F, 0x611E, 0xA0EC, 0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA075,
	0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0BF, 0x510F, 0x7004, 0x6110, 0x9051,
	0x6002, 0x800E, 0x6119, 0xA09D, 0x5121, 0x70FF, 0x6115, 0xA052, 0x610A, 0x901B, 0x611D, 0xA0E4,
	0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08C, 0x5111, 0x700A, 0x6114, 0xA02F,
	0x6109, 0x8120, 0x611C, 0xA0D3, 0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AE,
	0x5121, 0x7103, 0x6116, 0xA064, 0x610C, 0x9025, 0x611E, 0xA0FA, 0x510B, 0x611F, 0x610D, 0x9028,
	0x6000, 0x7105, 0x6117, 0xA06A, 0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B5,
	0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA093, 0x5121, 0x7080, 0x6115, 0xA03D,
	0x610A, 0x9014, 0x611D, 0xA0D9, 0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07C,
	0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C7, 0x510F, 0x7005, 0x6112, 0x9078,
	0x6107, 0x800F, 0x611A, 0xA0A3, 0x5121, 0x7102, 0x6116, 0xA058, 0x610C, 0x901E, 0x611E, 0xA0EA,
	0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA073, 0x5111, 0x7008, 0x6113, 0x90EE,
	0x6108, 0x8020, 0x611B, 0xA0BD, 0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09B,
	0x5121, 0x70FF, 0x6115, 0xA04E, 0x610A, 0x901A, 0x611D, 0xA0E2, 0x510B, 0x6122, 0x610E, 0x903E,
	0x6001, 0x7124, 0x6118, 0xA08A, 0x5111, 0x700A, 0x6114, 0xA02D, 0x6109, 0x80FE, 0x611C, 0xA0D1,
	0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AC, 0x5121, 0x7103, 0x6116, 0xA062,
	0x610C, 0x9024, 0x611E, 0xA0F7, 0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06E,
	0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0B9, 0x510F, 0x7003, 0x6110, 0x9048,
	0x6002, 0x800C, 0x6119, 0xA097, 0x5121, 0x7080, 0x6115, 0xA04A, 0x610A, 0x9016, 0x611D, 0xA0DD,
	0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA085, 0x5111, 0x7009, 0x6114, 0x90FD,
	0x6109, 0x80F0, 0x611C, 0xA0CB, 0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A7,
	0x5121, 0x7102, 0x6116, 0xA05D, 0x610C, 0x9021, 0x611E, 0xA0EF, 0x510B, 0x611F, 0x610D, 0x9033,
	0x6000, 0x7106, 0x6117, 0xA077, 0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C2,
	0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA09F, 0x5121, 0x70FF, 0x6115, 0xA054,
	0x610A, 0x901C, 0x611D, 0xA0E6, 0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08E,
	0x5111, 0x700A, 0x6114, 0xA034, 0x6109, 0x8120, 0x611C, 0xA0D5, 0x510F, 0x7006, 0x6112, 0x9090,
	0x6107, 0x8013, 0x611A, 0xA0B0, 0x5121, 0x7103, 0x6116, 0xA066, 0x610C, 0x9026, 0x611E, 0xA104,
	0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA069, 0x5111, 0x7007, 0x6113, 0x90C0,
	0x6108, 0x8018, 0x611B, 0xA0B4, 0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA092,
	0x5121, 0x7080, 0x6115, 0xA03B, 0x610A, 0x9012, 0x611D, 0xA0D8, 0x510B, 0x6122, 0x610E, 0x9035,
	0x6001, 0x7123, 0x6118, 0xA07B, 0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C5,
	0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A2, 0x5121, 0x7102, 0x6116, 0xA057,
	0x610C, 0x901D, 0x611E, 0xA0E9, 0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA072,
	0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BC, 0x510F, 0x7004, 0x6110, 0x9049,
	0x6002, 0x800D, 0x6119, 0xA09A, 0x5121, 0x70FF, 0x6115, 0xA04D, 0x610A, 0x9017, 0x611D, 0xA0E1,
	0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA089, 0x5111, 0x700A, 0x6114, 0xA02B,
	0x6109, 0x80FE, 0x611C, 0xA0CF, 0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0AA,
	0x5121, 0x7103, 0x6116, 0xA061, 0x610C, 0x9022, 0x611E, 0xA0F6, 0x510B, 0x611F, 0x610D, 0x9029,
	0x6000, 0x7105, 0x6117, 0xA06D, 0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B8,
	0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA096, 0x5121, 0x7080, 0x6115, 0xA047,
	0x610A, 0x9015, 0x611D, 0xA0DC, 0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA083,
	0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0CA, 0x510F, 0x7005, 0x6112, 0x907F,
	0x6107, 0x8010, 0x611A, 0xA0A6, 0x5121, 0x7102, 0x6116, 0xA05C, 0x610C, 0x901F, 0x611E, 0xA0ED,
	0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA076, 0x5111, 0x7008, 0x6113, 0x90F1,
	0x6108, 0x8040, 0x611B, 0xA0C1, 0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09E,
	0x5121, 0x70FF, 0x6115, 0xA053, 0x610A, 0x901B, 0x611D, 0xA0E5, 0x510B, 0x6122, 0x610E, 0x903F,
	0x6001, 0x7124, 0x6118, 0xA08D, 0x5111, 0x700A, 0x6114, 0xA032, 0x6109, 0x8120, 0x611C, 0xA0D4,
	0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AF, 0x5121, 0x7103, 0x6116, 0xA065,
	0x610C, 0x9025, 0x611E, 0xA0FB, 0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06B,
	0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B6, 0x510F, 0x7003, 0x6110, 0x9043,
	0x6002, 0x800B, 0x6119, 0xA094, 0x5121, 0x7080, 0x6115, 0xA045, 0x610A, 0x9014, 0x611D, 0xA0DA,
	0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07D, 0x5111, 0x7009, 0x6114, 0x90F8,
	0x6109, 0x8060, 0x611C, 0xA0C8, 0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A4,
	0x5121, 0x7102, 0x6116, 0xA05A, 0x610C, 0x901E, 0x611E, 0xA0EB, 0x510B, 0x611F, 0x610D, 0x9030,
	0x6000, 0x7106, 0x6117, 0xA074, 0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BE,
	0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09C, 0x5121, 0x70FF, 0x6115, 0xA04F,
	0x610A, 0x901A, 0x611D, 0xA0E3, 0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08B,
	0x5111, 0x700A, 0x6114, 0xA02E, 0x6109, 0x80FE, 0x611C, 0xA0D2, 0x510F, 0x7006, 0x6112, 0x9084,
	0x6107, 0x8011, 0x611A, 0xA0AD, 0x5121, 0x7103, 0x6116, 0xA063, 0x610C, 0x9024, 0x611E, 0xA0F9,
	0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06F, 0x5111, 0x7007, 0x6113, 0x90D0,
	0x6108, 0x8019, 0x611B, 0xA0BA, 0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA098,
	0x5121, 0x7080, 0x6115, 0xA04B, 0x610A, 0x9016, 0x611D, 0xA0DE, 0x510B, 0x6122, 0x610E, 0x9039,
	0x6001, 0x7123, 0x6118, 0xA086, 0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CD,
	0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A8, 0x5121, 0x7102, 0x6116, 0xA05E,
	0x610C, 0x9021, 0x611E, 0xA0F3, 0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA079,
	0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C3, 0x510F, 0x7004, 0x6110, 0x9059,
	0x6002, 0x800E, 0x6119, 0xA0A0, 0x5121, 0x70FF, 0x6115, 0xA055, 0x610A, 0x901C, 0x611D, 0xA0E7,
	0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08F, 0x5111, 0x700A, 0x6114, 0xA036,
	0x6109, 0x8120, 0x611C, 0xA0D6, 0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B1,
	0x5121, 0x7103, 0x6116, 0xA067, 0x610C, 0x9026, 0x611E, 0xD0AB, 0x510B, 0x611F, 0x610D, 0x9027,
	0x6000, 0x7105, 0x6117, 0xA068, 0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B3,
	0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA091, 0x5121, 0x7080, 0x6115, 0xA03A,
	0x610A, 0x9012, 0x611D, 0xA0D7, 0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07A,
	0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C4, 0x510F, 0x7005, 0x6112, 0x9070,
	0x6107, 0x800F, 0x611A, 0xA0A1, 0x5121, 0x7102, 0x6116, 0xA056, 0x610C, 0x901D, 0x611E, 0xA0E8,
	0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA071, 0x5111, 0x7008, 0x6113, 0x90E0,
	0x6108, 0x8020, 0x611B, 0xA0BB, 0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA099,
	0x5121, 0x70FF, 0x6115, 0xA04C, 0x610A, 0x9017, 0x611D, 0xA0DF, 0x510B, 0x6122, 0x610E, 0x903C,
	0x6001, 0x7124, 0x6118, 0xA087, 0x5111, 0x700A, 0x6114, 0xA023, 0x6109, 0x80FE, 0x611C, 0xA0CE,
	0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0A9, 0x5121, 0x7103, 0x6116, 0xA05F,
	0x610C, 0x9022, 0x611E, 0xA0F5, 0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06C,
	0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B7, 0x510F, 0x7003, 0x6110, 0x9044,
	0x6002, 0x800C, 0x6119, 0xA095, 0x5121, 0x7080, 0x6115, 0xA046, 0x610A, 0x9015, 0x611D, 0xA0DB,
	0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA07E, 0x5111, 0x7009, 0x6114, 0x90FC,
	0x6109, 0x80F0, 0x611C, 0xA0C9, 0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A5,
	0x5121, 0x7102, 0x6116, 0xA05B, 0x610C, 0x901F, 0x611E, 0xA0EC, 0x510B, 0x611F, 0x610D, 0x9031,
	0x6000, 0x7106, 0x6117, 0xA075, 0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0BF,
	0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09D, 0x5121, 0x70FF, 0x6115, 0xA052,
	0x610A, 0x901B, 0x611D, 0xA0E4, 0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08C,
	0x5111, 0x700A, 0x6114, 0xA02F, 0x6109, 0x8120, 0x611C, 0xA0D3, 0x510F, 0x7006, 0x6112, 0x9088,
	0x6107, 0x8013, 0x611A, 0xA0AE, 0x5121, 0x7103, 0x6116, 0xA064, 0x610C, 0x9025, 0x611E, 0xA0FA,
	0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06A, 0x5111, 0x7007, 0x6113, 0x90C6,
	0x6108, 0x8018, 0x611B, 0xA0B5, 0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA093,
	0x5121, 0x7080, 0x6115, 0xA03D, 0x610A, 0x9014, 0x611D, 0xA0D9, 0x510B, 0x6122, 0x610E, 0x9037,
	0x6001, 0x7123, 0x6118, 0xA07C, 0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C7,
	0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A3, 0x5121, 0x7102, 0x6116, 0xA058,
	0x610C, 0x901E, 0x611E, 0xA0EA, 0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA073,
	0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BD, 0x510F, 0x7004, 0x6110, 0x9050,
	0x6002, 0x800D, 0x6119, 0xA09B, 0x5121, 0x70FF, 0x6115, 0xA04E, 0x610A, 0x901A, 0x611D, 0xA0E2,
	0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08A, 0x5111, 0x700A, 0x6114, 0xA02D,
	0x6109, 0x80FE, 0x611C, 0xA0D1, 0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AC,
	0x5121, 0x7103, 0x6116, 0xA062, 0x610C, 0x9024, 0x611E, 0xA0F7, 0x510B, 0x611F, 0x610D, 0x902A,
	0x6000, 0x7105, 0x6117, 0xA06E, 0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0B9,
	0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA097, 0x5121, 0x7080, 0x6115, 0xA04A,
	0x610A, 0x9016, 0x611D, 0xA0DD, 0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA085,
	0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CB, 0x510F, 0x7005, 0x6112, 0x9081,
	0x6107, 0x8010, 0x611A, 0xA0A7, 0x5121, 0x7102, 0x6116, 0xA05D, 0x610C, 0x9021, 0x611E, 0xA0EF,
	0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA077, 0x5111, 0x7008, 0x6113, 0x90F2,
	0x6108, 0x8040, 0x611B, 0xA0C2, 0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA09F,
	0x5121, 0x70FF, 0x6115, 0xA054, 0x610A, 0x901C, 0x611D, 0xA0E6, 0x510B, 0x6122, 0x610E, 0x9041,
	0x6001, 0x7124, 0x6118, 0xA08E, 0x5111, 0x700A, 0x6114, 0xA034, 0x6109, 0x8120, 0x611C, 0xA0D5,
	0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B0, 0x5121, 0x7103, 0x6116, 0xA066,
	0x610C, 0x9026, 0x611E, 0xA104, 0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA069,
	0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B4, 0x510F, 0x7003, 0x6110, 0x9042,
	0x6002, 0x800B, 0x6119, 0xA092, 0x5121, 0x7080, 0x6115, 0xA03B, 0x610A, 0x9012, 0x611D, 0xA0D8,
	0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07B, 0x5111, 0x7009, 0x6114, 0x90F4,
	0x6109, 0x8060, 0x611C, 0xA0C5, 0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A2,
	0x5121, 0x7102, 0x6116, 0xA057, 0x610C, 0x901D, 0x611E, 0xA0E9, 0x510B, 0x611F, 0x610D, 0x902C,
	0x6000, 0x7106, 0x6117, 0xA072, 0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BC,
	0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA09A, 0x5121, 0x70FF, 0x6115, 0xA04D,
	0x610A, 0x9017, 0x611D, 0xA0E1, 0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA089,
	0x5111, 0x700A, 0x6114, 0xA02B, 0x6109, 0x80FE, 0x611C, 0xA0CF, 0x510F, 0x7006, 0x6112, 0x9082,
	0x6107, 0x8011, 0x611A, 0xA0AA, 0x5121, 0x7103, 0x6116, 0xA061, 0x610C, 0x9022, 0x611E, 0xA0F6,
	0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06D, 0x5111, 0x7007, 0x6113, 0x90CC,
	0x6108, 0x8019, 0x611B, 0xA0B8, 0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA096,
	0x5121, 0x7080, 0x6115, 0xA047, 0x610A, 0x9015, 0x611D, 0xA0DC, 0x510B, 0x6122, 0x610E, 0x9038,
	0x6001, 0x7123, 0x6118, 0xA083, 0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0CA,
	0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A6, 0x5121, 0x7102, 0x6116, 0xA05C,
	0x610C, 0x901F, 0x611E, 0xA0ED, 0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA076,
	0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0C1, 0x510F, 0x7004, 0x6110, 0x9051,
	0x6002, 0x800E, 0x6119, 0xA09E, 0x5121, 0x70FF, 0x6115, 0xA053, 0x610A, 0x901B, 0x611D, 0xA0E5,
	0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08D, 0x5111, 0x700A, 0x6114, 0xA032,
	0x6109, 0x8120, 0x611C, 0xA0D4, 0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AF,
	0x5121, 0x7103, 0x6116, 0xA065, 0x610C, 0x9025, 0x611E, 0xA0FB, 0x510B, 0x611F, 0x610D, 0x9028,
	0x6000, 0x7105, 0x6117, 0xA06B, 0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B6,
	0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA094, 0x5121, 0x7080, 0x6115, 0xA045,
	0x610A, 0x9014, 0x611D, 0xA0DA, 0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07D,
	0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C8, 0x510F, 0x7005, 0x6112, 0x9078,
	0x6107, 0x800F, 0x611A, 0xA0A4, 0x5121, 0x7102, 0x6116, 0xA05A, 0x610C, 0x901E, 0x611E, 0xA0EB,
	0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA074, 0x5111, 0x7008, 0x6113, 0x90EE,
	0x6108, 0x8020, 0x611B, 0xA0BE, 0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09C,
	0x5121, 0x70FF, 0x6115, 0xA04F, 0x610A, 0x901A, 0x611D, 0xA0E3, 0x510B, 0x6122, 0x610E, 0x903E,
	0x6001, 0x7124, 0x6118, 0xA08B, 0x5111, 0x700A, 0x6114, 0xA02E, 0x6109, 0x80FE, 0x611C, 0xA0D2,
	0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AD, 0x5121, 0x7103, 0x6116, 0xA063,
	0x610C, 0x9024, 0x611E, 0xA0F9, 0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06F,
	0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0BA, 0x510F, 0x7003, 0x6110, 0x9048,
	0x6002, 0x800C, 0x6119, 0xA098, 0x5121, 0x7080, 0x6115, 0xA04B, 0x610A, 0x9016, 0x611D, 0xA0DE,
	0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA086, 0x5111, 0x7009, 0x6114, 0x90FD,
	0x6109, 0x80F0, 0x611C, 0xA0CD, 0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A8,
	0x5121, 0x7102, 0x6116, 0xA05E, 0x610C, 0x9021, 0x611E, 0xA0F3, 0x510B, 0x611F, 0x610D, 0x9033,
	0x6000, 0x7106, 0x6117, 0xA079, 0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C3,
	0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA0A0, 0x5121, 0x70FF, 0x6115, 0xA055,
	0x610A, 0x901C, 0x611D, 0xA0E7, 0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08F,
	0x5111, 0x700A, 0x6114, 0xA036, 0x6109, 0x8120, 0x611C, 0xA0D6, 0x510F, 0x7006, 0x6112, 0x9090,
	0x6107, 0x8013, 0x611A, 0xA0B1, 0x5121, 0x7103, 0x6116, 0xA067, 0x610C, 0x9026, 0x611E, 0xB0B2,
	0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA068, 0x5111, 0x7007, 0x6113, 0x90C0,
	0x6108, 0x8018, 0x611B, 0xA0B3, 0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA091,
	0x5121, 0x7080, 0x6115, 0xA03A, 0x610A, 0x9012, 0x611D, 0xA0D7, 0x510B, 0x6122, 0x610E, 0x9035,
	0x6001, 0x7123, 0x6118, 0xA07A, 0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C4,
	0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A1, 0x5121, 0x7102, 0x6116, 0xA056,
	0x610C, 0x901D, 0x611E, 0xA0E8, 0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA071,
	0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BB, 0x510F, 0x7004, 0x6110, 0x9049,
	0x6002, 0x800D, 0x6119, 0xA099, 0x5121, 0x70FF, 0x6115, 0xA04C, 0x610A, 0x9017, 0x611D, 0xA0DF,
	0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA087, 0x5111, 0x700A, 0x6114, 0xA023,
	0x6109, 0x80FE, 0x611C, 0xA0CE, 0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0A9,
	0x5121, 0x7103, 0x6116, 0xA05F, 0x610C, 0x9022, 0x611E, 0xA0F5, 0x510B, 0x611F, 0x610D, 0x9029,
	0x6000, 0x7105, 0x6117, 0xA06C, 0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B7,
	0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA095, 0x5121, 0x7080, 0x6115, 0xA046,
	0x610A, 0x9015, 0x611D, 0xA0DB, 0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA07E,
	0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0C9, 0x510F, 0x7005, 0x6112, 0x907F,
	0x6107, 0x8010, 0x611A, 0xA0A5, 0x5121, 0x7102, 0x6116, 0xA05B, 0x610C, 0x901F, 0x611E, 0xA0EC,
	0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA075, 0x5111, 0x7008, 0x6113, 0x90F1,
	0x6108, 0x8040, 0x611B, 0xA0BF, 0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09D,
	0x5121, 0x70FF, 0x6115, 0xA052, 0x610A, 0x901B, 0x611D, 0xA0E4, 0x510B, 0x6122, 0x610E, 0x903F,
	0x6001, 0x7124, 0x6118, 0xA08C, 0x5111, 0x700A, 0x6114, 0xA02F, 0x6109, 0x8120, 0x611C, 0xA0D3,
	0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AE, 0x5121, 0x7103, 0x6116, 0xA064,
	0x610C, 0x9025, 0x611E, 0xA0FA, 0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06A,
	0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B5, 0x510F, 0x7003, 0x6110, 0x9043,
	0x6002, 0x800B, 0x6119, 0xA093, 0x5121, 0x7080, 0x6115, 0xA03D, 0x610A, 0x9014, 0x611D, 0xA0D9,
	0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07C, 0x5111, 0x7009, 0x6114, 0x90F8,
	0x6109, 0x8060, 0x611C, 0xA0C7, 0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A3,
	0x5121, 0x7102, 0x6116, 0xA058, 0x610C, 0x901E, 0x611E, 0xA0EA, 0x510B, 0x611F, 0x610D, 0x9030,
	0x6000, 0x7106, 0x6117, 0xA073, 0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BD,
	0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09B, 0x5121, 0x70FF, 0x6115, 0xA04E,
	0x610A, 0x901A, 0x611D, 0xA0E2, 0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08A,
	0x5111, 0x700A, 0x6114, 0xA02D, 0x6109, 0x80FE, 0x611C, 0xA0D1, 0x510F, 0x7006, 0x6112, 0x9084,
	0x6107, 0x8011, 0x611A, 0xA0AC, 0x5121, 0x7103, 0x6116, 0xA062, 0x610C, 0x9024, 0x611E, 0xA0F7,
	0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06E, 0x5111, 0x7007, 0x6113, 0x90D0,
	0x6108, 0x8019, 0x611B, 0xA0B9, 0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA097,
	0x5121, 0x7080, 0x6115, 0xA04A, 0x610A, 0x9016, 0x611D, 0xA0DD, 0x510B, 0x6122, 0x610E, 0x9039,
	0x6001, 0x7123, 0x6118, 0xA085, 0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CB,
	0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A7, 0x5121, 0x7102, 0x6116, 0xA05D,
	0x610C, 0x9021, 0x611E, 0xA0EF, 0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA077,
	0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C2, 0x510F, 0x7004, 0x6110, 0x9059,
	0x6002, 0x800E, 0x6119, 0xA09F, 0x5121, 0x70FF, 0x6115, 0xA054, 0x610A, 0x901C, 0x611D, 0xA0E6,
	0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08E, 0x5111, 0x700A, 0x6114, 0xA034,
	0x6109, 0x8120, 0x611C, 0xA0D5, 0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B0,
	0x5121, 0x7103, 0x6116, 0xA066, 0x610C, 0x9026, 0x611E, 0xA104, 0x510B, 0x611F, 0x610D, 0x9027,
	0x6000, 0x7105, 0x6117, 0xA069, 0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B4,
	0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA092, 0x5121, 0x7080, 0x6115, 0xA03B,
	0x610A, 0x9012, 0x611D, 0xA0D8, 0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07B,
	0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C5, 0x510F, 0x7005, 0x6112, 0x9070,
	0x6107, 0x800F, 0x611A, 0xA0A2, 0x5121, 0x7102, 0x6116, 0xA057, 0x610C, 0x901D, 0x611E, 0xA0E9,
	0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA072, 0x5111, 0x7008, 0x6113, 0x90E0,
	0x6108, 0x8020, 0x611B, 0xA0BC, 0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA09A,
	0x5121, 0x70FF, 0x6115, 0xA04D, 0x610A, 0x9017, 0x611D, 0xA0E1, 0x510B, 0x6122, 0x610E, 0x903C,
	0x6001, 0x7124, 0x6118, 0xA089, 0x5111, 0x700A, 0x6114, 0xA02B, 0x6109, 0x80FE, 0x611C, 0xA0CF,
	0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0AA, 0x5121, 0x7103, 0x6116, 0xA061,
	0x610C, 0x9022, 0x611E, 0xA0F6, 0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06D,
	0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B8, 0x510F, 0x7003, 0x6110, 0x9044,
	0x6002, 0x800C, 0x6119, 0xA096, 0x5121, 0x7080, 0x6115, 0xA047, 0x610A, 0x9015, 0x611D, 0xA0DC,
	0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA083, 0x5111, 0x7009, 0x6114, 0x90FC,
	0x6109, 0x80F0, 0x611C, 0xA0CA, 0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A6,
	0x5121, 0x7102, 0x6116, 0xA05C, 0x610C, 0x901F, 0x611E, 0xA0ED, 0x510B, 0x611F, 0x610D, 0x9031,
	0x6000, 0x7106, 0x6117, 0xA076, 0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0C1,
	0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09E, 0x5121, 0x70FF, 0x6115, 0xA053,
	0x610A, 0x901B, 0x611D, 0xA0E5, 0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08D,
	0x5111, 0x700A, 0x6114, 0xA032, 0x6109, 0x8120, 0x611C, 0xA0D4, 0x510F, 0x7006, 0x6112, 0x9088,
	0x6107, 0x8013, 0x611A, 0xA0AF, 0x5121, 0x7103, 0x6116, 0xA065, 0x610C, 0x9025, 0x611E, 0xA0FB,
	0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06B, 0x5111, 0x7007, 0x6113, 0x90C6,
	0x6108, 0x8018, 0x611B, 0xA0B6, 0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA094,
	0x5121, 0x7080, 0x6115, 0xA045, 0x610A, 0x9014, 0x611D, 0xA0DA, 0x510B, 0x6122, 0x610E, 0x9037,
	0x6001, 0x7123, 0x6118, 0xA07D, 0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C8,
	0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A4, 0x5121, 0x7102, 0x6116, 0xA05A,
	0x610C, 0x901E, 0x611E, 0xA0EB, 0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA074,
	0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BE, 0x510F, 0x7004, 0x6110, 0x9050,
	0x6002, 0x800D, 0x6119, 0xA09C, 0x5121, 0x70FF, 0x6115, 0xA04F, 0x610A, 0x901A, 0x611D, 0xA0E3,
	0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08B, 0x5111, 0x700A, 0x6114, 0xA02E,
	0x6109, 0x80FE, 0x611C, 0xA0D2, 0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AD,
	0x5121, 0x7103, 0x6116, 0xA063, 0x610C, 0x9024, 0x611E, 0xA0F9, 0x510B, 0x611F, 0x610D, 0x902A,
	0x6000, 0x7105, 0x6117, 0xA06F, 0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0BA,
	0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA098, 0x5121, 0x7080, 0x6115, 0xA04B,
	0x610A, 0x9016, 0x611D, 0xA0DE, 0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA086,
	0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CD, 0x510F, 0x7005, 0x6112, 0x9081,
	0x6107, 0x8010, 0x611A, 0xA0A8, 0x5121, 0x7102, 0x6116, 0xA05E, 0x610C, 0x9021, 0x611E, 0xA0F3,
	0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA079, 0x5111, 0x7008, 0x6113, 0x90F2,
	0x6108, 0x8040, 0x611B, 0xA0C3, 0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA0A0,
	0x5121, 0x70FF, 0x6115, 0xA055, 0x610A, 0x901C, 0x611D, 0xA0E7, 0x510B, 0x6122, 0x610E, 0x9041,
	0x6001, 0x7124, 0x6118, 0xA08F, 0x5111, 0x700A, 0x6114, 0xA036, 0x6109, 0x8120, 0x611C, 0xA0D6,
	0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B1, 0x5121, 0x7103, 0x6116, 0xA067,
	0x610C, 0x9026, 0x611E, 0xD101, 0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA068,
	0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B3, 0x510F, 0x7003, 0x6110, 0x9042,
	0x6002, 0x800B, 0x6119, 0xA091, 0x5121, 0x7080, 0x6115, 0xA03A, 0x610A, 0x9012, 0x611D, 0xA0D7,
	0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07A, 0x5111, 0x7009, 0x6114, 0x90F4,
	0x6109, 0x8060, 0x611C, 0xA0C4, 0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A1,
	0x5121, 0x7102, 0x6116, 0xA056, 0x610C, 0x901D, 0x611E, 0xA0E8, 0x510B, 0x611F, 0x610D, 0x902C,
	0x6000, 0x7106, 0x6117, 0xA071, 0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BB,
	0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA099, 0x5121, 0x70FF, 0x6115, 0xA04C,
	0x610A, 0x9017, 0x611D, 0xA0DF, 0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA087,
	0x5111, 0x700A, 0x6114, 0xA023, 0x6109, 0x80FE, 0x611C, 0xA0CE, 0x510F, 0x7006, 0x6112, 0x9082,
	0x6107, 0x8011, 0x611A, 0xA0A9, 0x5121, 0x7103, 0x6116, 0xA05F, 0x610C, 0x9022, 0x611E, 0xA0F5,
	0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06C, 0x5111, 0x7007, 0x6113, 0x90CC,
	0x6108, 0x8019, 0x611B, 0xA0B7, 0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA095,
	0x5121, 0x7080, 0x6115, 0xA046, 0x610A, 0x9015, 0x611D, 0xA0DB, 0x510B, 0x6122, 0x610E, 0x9038,
	0x6001, 0x7123, 0x6118, 0xA07E, 0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0C9,
	0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A5, 0x5121, 0x7102, 0x6116, 0xA05B,
	0x610C, 0x901F, 0x611E, 0xA0EC, 0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA075,
	0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0BF, 0x510F, 0x7004, 0x6110, 0x9051,
	0x6002, 0x800E, 0x6119, 0xA09D, 0x5121, 0x70FF, 0x6115, 0xA052, 0x610A, 0x901B, 0x611D, 0xA0E4,
	0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08C, 0x5111, 0x700A, 0x6114, 0xA02F,
	0x6109, 0x8120, 0x611C, 0xA0D3, 0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AE,
	0x5121, 0x7103, 0x6116, 0xA064, 0x610C, 0x9025, 0x611E, 0xA0FA, 0x510B, 0x611F, 0x610D, 0x9028,
	0x6000, 0x7105, 0x6117, 0xA06A, 0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B5,
	0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA093, 0x5121, 0x7080, 0x6115, 0xA03D,
	0x610A, 0x9014, 0x611D, 0xA0D9, 0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07C,
	0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C7, 0x510F, 0x7005, 0x6112, 0x9078,
	0x6107, 0x800F, 0x611A, 0xA0A3, 0x5121, 0x7102, 0x6116, 0xA058, 0x610C, 0x901E, 0x611E, 0xA0EA,
	0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA073, 0x5111, 0x7008, 0x6113, 0x90EE,
	0x6108, 0x8020, 0x611B, 0xA0BD, 0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09B,
	0x5121, 0x70FF, 0x6115, 0xA04E, 0x610A, 0x901A, 0x611D, 0xA0E2, 0x510B, 0x6122, 0x610E, 0x903E,
	0x6001, 0x7124, 0x6118, 0xA08A, 0x5111, 0x700A, 0x6114, 0xA02D, 0x6109, 0x80FE, 0x611C, 0xA0D1,
	0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AC, 0x5121, 0x7103, 0x6116, 0xA062,
	0x610C, 0x9024, 0x611E, 0xA0F7, 0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06E,
	0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0B9, 0x510F, 0x7003, 0x6110, 0x9048,
	0x6002, 0x800C, 0x6119, 0xA097, 0x5121, 0x7080, 0x6115, 0xA04A, 0x610A, 0x9016, 0x611D, 0xA0DD,
	0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA085, 0x5111, 0x7009, 0x6114, 0x90FD,
	0x6109, 0x80F0, 0x611C, 0xA0CB, 0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A7,
	0x5121, 0x7102, 0x6116, 0xA05D, 0x610C, 0x9021, 0x611E, 0xA0EF, 0x510B, 0x611F, 0x610D, 0x9033,
	0x6000, 0x7106, 0x6117, 0xA077, 0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C2,
	0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA09F, 0x5121, 0x70FF, 0x6115, 0xA054,
	0x610A, 0x901C, 0x611D, 0xA0E6, 0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08E,
	0x5111, 0x700A, 0x6114, 0xA034, 0x6109, 0x8120, 0x611C, 0xA0D5, 0x510F, 0x7006, 0x6112, 0x9090,
	0x6107, 0x8013, 0x611A, 0xA0B0, 0x5121, 0x7103, 0x6116, 0xA066, 0x610C, 0x9026, 0x611E, 0xA104,
	0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA069, 0x5111, 0x7007, 0x6113, 0x90C0,
	0x6108, 0x8018, 0x611B, 0xA0B4, 0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA092,
	0x5121, 0x7080, 0x6115, 0xA03B, 0x610A, 0x9012, 0x611D, 0xA0D8, 0x510B, 0x6122, 0x610E, 0x9035,
	0x6001, 0x7123, 0x6118, 0xA07B, 0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C5,
	0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A2, 0x5121, 0x7102, 0x6116, 0xA057,
	0x610C, 0x901D, 0x611E, 0xA0E9, 0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA072,
	0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BC, 0x510F, 0x7004, 0x6110, 0x9049,
	0x6002, 0x800D, 0x6119, 0xA09A, 0x5121, 0x70FF, 0x6115, 0xA04D, 0x610A, 0x9017, 0x611D, 0xA0E1,
	0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA089, 0x5111, 0x700A, 0x6114, 0xA02B,
	0x6109, 0x80FE, 0x611C, 0xA0CF, 0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0AA,
	0x5121, 0x7103, 0x6116, 0xA061, 0x610C, 0x9022, 0x611E, 0xA0F6, 0x510B, 0x611F, 0x610D, 0x9029,
	0x6000, 0x7105, 0x6117, 0xA06D, 0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B8,
	0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA096, 0x5121, 0x7080, 0x6115, 0xA047,
	0x610A, 0x9015, 0x611D, 0xA0DC, 0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA083,
	0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0CA, 0x510F, 0x7005, 0x6112, 0x907F,
	0x6107, 0x8010, 0x611A, 0xA0A6, 0x5121, 0x7102, 0x6116, 0xA05C, 0x610C, 0x901F, 0x611E, 0xA0ED,
	0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA076, 0x5111, 0x7008, 0x6113, 0x90F1,
	0x6108, 0x8040, 0x611B, 0xA0C1, 0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09E,
	0x5121, 0x70FF, 0x6115, 0xA053, 0x610A, 0x901B, 0x611D, 0xA0E5, 0x510B, 0x6122, 0x610E, 0x903F,
	0x6001, 0x7124, 0x6118, 0xA08D, 0x5111, 0x700A, 0x6114, 0xA032, 0x6109, 0x8120, 0x611C, 0xA0D4,
	0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AF, 0x5121, 0x7103, 0x6116, 0xA065,
	0x610C, 0x9025, 0x611E, 0xA0FB, 0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06B,
	0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B6, 0x510F, 0x7003, 0x6110, 0x9043,
	0x6002, 0x800B, 0x6119, 0xA094, 0x5121, 0x7080, 0x6115, 0xA045, 0x610A, 0x9014, 0x611D, 0xA0DA,
	0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07D, 0x5111, 0x7009, 0x6114, 0x90F8,
	0x6109, 0x8060, 0x611C, 0xA0C8, 0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A4,
	0x5121, 0x7102, 0x6116, 0xA05A, 0x610C, 0x901E, 0x611E, 0xA0EB, 0x510B, 0x611F, 0x610D, 0x9030,
	0x6000, 0x7106, 0x6117, 0xA074, 0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BE,
	0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09C, 0x5121, 0x70FF, 0x6115, 0xA04F,
	0x610A, 0x901A, 0x611D, 0xA0E3, 0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08B,
	0x5111, 0x700A, 0x6114, 0xA02E, 0x6109, 0x80FE, 0x611C, 0xA0D2, 0x510F, 0x7006, 0x6112, 0x9084,
	0x6107, 0x8011, 0x611A, 0xA0AD, 0x5121, 0x7103, 0x6116, 0xA063, 0x610C, 0x9024, 0x611E, 0xA0F9,
	0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06F, 0x5111, 0x7007, 0x6113, 0x90D0,
	0x6108, 0x8019, 0x611B, 0xA0BA, 0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA098,
	0x5121, 0x7080, 0x6115, 0xA04B, 0x610A, 0x9016, 0x611D, 0xA0DE, 0x510B, 0x6122, 0x610E, 0x9039,
	0x6001, 0x7123, 0x6118, 0xA086, 0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CD,
	0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A8, 0x5121, 0x7102, 0x6116, 0xA05E,
	0x610C, 0x9021, 0x611E, 0xA0F3, 0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA079,
	0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C3, 0x510F, 0x7004, 0x6110, 0x9059,
	0x6002, 0x800E, 0x6119, 0xA0A0, 0x5121, 0x70FF, 0x6115, 0xA055, 0x610A, 0x901C, 0x611D, 0xA0E7,
	0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08F, 0x5111, 0x700A, 0x6114, 0xA036,
	0x6109, 0x8120, 0x611C, 0xA0D6, 0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B1,
	0x5121, 0x7103, 0x6116, 0xA067, 0x610C, 0x9026, 0x611E, 0xB0B2, 0x510B, 0x611F, 0x610D, 0x9027,
	0x6000, 0x7105, 0x6117, 0xA068, 0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B3,
	0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA091, 0x5121, 0x7080, 0x6115, 0xA03A,
	0x610A, 0x9012, 0x611D, 0xA0D7, 0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07A,
	0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C4, 0x510F, 0x7005, 0x6112, 0x9070,
	0x6107, 0x800F, 0x611A, 0xA0A1, 0x5121, 0x7102, 0x6116, 0xA056, 0x610C, 0x901D, 0x611E, 0xA0E8,
	0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA071, 0x5111, 0x7008, 0x6113, 0x90E0,
	0x6108, 0x8020, 0x611B, 0xA0BB, 0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA099,
	0x5121, 0x70FF, 0x6115, 0xA04C, 0x610A, 0x9017, 0x611D, 0xA0DF, 0x510B, 0x6122, 0x610E, 0x903C,
	0x6001, 0x7124, 0x6118, 0xA087, 0x5111, 0x700A, 0x6114, 0xA023, 0x6109, 0x80FE, 0x611C, 0xA0CE,
	0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0A9, 0x5121, 0x7103, 0x6116, 0xA05F,
	0x610C, 0x9022, 0x611E, 0xA0F5, 0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06C,
	0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B7, 0x510F, 0x7003, 0x6110, 0x9044,
	0x6002, 0x800C, 0x6119, 0xA095, 0x5121, 0x7080, 0x6115, 0xA046, 0x610A, 0x9015, 0x611D, 0xA0DB,
	0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA07E, 0x5111, 0x7009, 0x6114, 0x90FC,
	0x6109, 0x80F0, 0x611C, 0xA0C9, 0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A5,
	0x5121, 0x7102, 0x6116, 0xA05B, 0x610C, 0x901F, 0x611E, 0xA0EC, 0x510B, 0x611F, 0x610D, 0x9031,
	0x6000, 0x7106, 0x6117, 0xA075, 0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0BF,
	0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09D, 0x5121, 0x70FF, 0x6115, 0xA052,
	0x610A, 0x901B, 0x611D, 0xA0E4, 0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08C,
	0x5111, 0x700A, 0x6114, 0xA02F, 0x6109, 0x8120, 0x611C, 0xA0D3, 0x510F, 0x7006, 0x6112, 0x9088,
	0x6107, 0x8013, 0x611A, 0xA0AE, 0x5121, 0x7103, 0x6116, 0xA064, 0x610C, 0x9025, 0x611E, 0xA0FA,
	0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06A, 0x5111, 0x7007, 0x6113, 0x90C6,
	0x6108, 0x8018, 0x611B, 0xA0B5, 0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA093,
	0x5121, 0x7080, 0x6115, 0xA03D, 0x610A, 0x9014, 0x611D, 0xA0D9, 0x510B, 0x6122, 0x610E, 0x9037,
	0x6001, 0x7123, 0x6118, 0xA07C, 0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C7,
	0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A3, 0x5121, 0x7102, 0x6116, 0xA058,
	0x610C, 0x901E, 0x611E, 0xA0EA, 0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA073,
	0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BD, 0x510F, 0x7004, 0x6110, 0x9050,
	0x6002, 0x800D, 0x6119, 0xA09B, 0x5121, 0x70FF, 0x6115, 0xA04E, 0x610A, 0x901A, 0x611D, 0xA0E2,
	0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08A, 0x5111, 0x700A, 0x6114, 0xA02D,
	0x6109, 0x80FE, 0x611C, 0xA0D1, 0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AC,
	0x5121, 0x7103, 0x6116, 0xA062, 0x610C, 0x9024, 0x611E, 0xA0F7, 0x510B, 0x611F, 0x610D, 0x902A,
	0x6000, 0x7105, 0x6117, 0xA06E, 0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0B9,
	0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA097, 0x5121, 0x7080, 0x6115, 0xA04A,
	0x610A, 0x9016, 0x611D, 0xA0DD, 0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA085,
	0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CB, 0x510F, 0x7005, 0x6112, 0x9081,
	0x6107, 0x8010, 0x611A, 0xA0A7, 0x5121, 0x7102, 0x6116, 0xA05D, 0x610C, 0x9021, 0x611E, 0xA0EF,
	0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA077, 0x5111, 0x7008, 0x6113, 0x90F2,
	0x6108, 0x8040, 0x611B, 0xA0C2, 0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA09F,
	0x5121, 0x70FF, 0x6115, 0xA054, 0x610A, 0x901C, 0x611D, 0xA0E6, 0x510B, 0x6122, 0x610E, 0x9041,
	0x6001, 0x7124, 0x6118, 0xA08E, 0x5111, 0x700A, 0x6114, 0xA034, 0x6109, 0x8120, 0x611C, 0xA0D5,
	0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B0, 0x5121, 0x7103, 0x6116, 0xA066,
	0x610C, 0x9026, 0x611E, 0xA104, 0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA069,
	0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B4, 0x510F, 0x7003, 0x6110, 0x9042,
	0x6002, 0x800B, 0x6119, 0xA092, 0x5121, 0x7080, 0x6115, 0xA03B, 0x610A, 0x9012, 0x611D, 0xA0D8,
	0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07B, 0x5111, 0x7009, 0x6114, 0x90F4,
	0x6109, 0x8060, 0x611C, 0xA0C5, 0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A2,
	0x5121, 0x7102, 0x6116, 0xA057, 0x610C, 0x901D, 0x611E, 0xA0E9, 0x510B, 0x611F, 0x610D, 0x902C,
	0x6000, 0x7106, 0x6117, 0xA072, 0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BC,
	0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA09A, 0x5121, 0x70FF, 0x6115, 0xA04D,
	0x610A, 0x9017, 0x611D, 0xA0E1, 0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA089,
	0x5111, 0x700A, 0x6114, 0xA02B, 0x6109, 0x80FE, 0x611C, 0xA0CF, 0x510F, 0x7006, 0x6112, 0x9082,
	0x6107, 0x8011, 0x611A, 0xA0AA, 0x5121, 0x7103, 0x6116, 0xA061, 0x610C, 0x9022, 0x611E, 0xA0F6,
	0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06D, 0x5111, 0x7007, 0x6113, 0x90CC,
	0x6108, 0x8019, 0x611B, 0xA0B8, 0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA096,
	0x5121, 0x7080, 0x6115, 0xA047, 0x610A, 0x9015, 0x611D, 0xA0DC, 0x510B, 0x6122, 0x610E, 0x9038,
	0x6001, 0x7123, 0x6118, 0xA083, 0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0CA,
	0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A6, 0x5121, 0x7102, 0x6116, 0xA05C,
	0x610C, 0x901F, 0x611E, 0xA0ED, 0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA076,
	0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0C1, 0x510F, 0x7004, 0x6110, 0x9051,
	0x6002, 0x800E, 0x6119, 0xA09E, 0x5121, 0x70FF, 0x6115, 0xA053, 0x610A, 0x901B, 0x611D, 0xA0E5,
	0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08D, 0x5111, 0x700A, 0x6114, 0xA032,
	0x6109, 0x8120, 0x611C, 0xA0D4, 0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AF,
	0x5121, 0x7103, 0x6116, 0xA065, 0x610C, 0x9025, 0x611E, 0xA0FB, 0x510B, 0x611F, 0x610D, 0x9028,
	0x6000, 0x7105, 0x6117, 0xA06B, 0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B6,
	0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA094, 0x5121, 0x7080, 0x6115, 0xA045,
	0x610A, 0x9014, 0x611D, 0xA0DA, 0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07D,
	0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C8, 0x510F, 0x7005, 0x6112, 0x9078,
	0x6107, 0x800F, 0x611A, 0xA0A4, 0x5121, 0x7102, 0x6116, 0xA05A, 0x610C, 0x901E, 0x611E, 0xA0EB,
	0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA074, 0x5111, 0x7008, 0x6113, 0x90EE,
	0x6108, 0x8020, 0x611B, 0xA0BE, 0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09C,
	0x5121, 0x70FF, 0x6115, 0xA04F, 0x610A, 0x901A, 0x611D, 0xA0E3, 0x510B, 0x6122, 0x610E, 0x903E,
	0x6001, 0x7124, 0x6118, 0xA08B, 0x5111, 0x700A, 0x6114, 0xA02E, 0x6109, 0x80FE, 0x611C, 0xA0D2,
	0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AD, 0x5121, 0x7103, 0x6116, 0xA063,
	0x610C, 0x9024, 0x611E, 0xA0F9, 0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06F,
	0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0BA, 0x510F, 0x7003, 0x6110, 0x9048,
	0x6002, 0x800C, 0x6119, 0xA098, 0x5121, 0x7080, 0x6115, 0xA04B, 0x610A, 0x9016, 0x611D, 0xA0DE,
	0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA086, 0x5111, 0x7009, 0x6114, 0x90FD,
	0x6109, 0x80F0, 0x611C, 0xA0CD, 0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A8,
	0x5121, 0x7102, 0x6116, 0xA05E, 0x610C, 0x9021, 0x611E, 0xA0F3, 0x510B, 0x611F, 0x610D, 0x9033,
	0x6000, 0x7106, 0x6117, 0xA079, 0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C3,
	0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA0A0, 0x5121, 0x70FF, 0x6115, 0xA055,
	0x610A, 0x901C, 0x611D, 0xA0E7, 0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08F,
	0x5111, 0x700A, 0x6114, 0xA036, 0x6109, 0x8120, 0x611C, 0xA0D6, 0x510F, 0x7006, 0x6112, 0x9090,
	0x6107, 0x8013, 0x611A, 0xA0B1, 0x5121, 0x7103, 0x6116, 0xA067, 0x610C, 0x9026, 0x611E, 0xD100,
	0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA068, 0x5111, 0x7007, 0x6113, 0x90C0,
	0x6108, 0x8018, 0x611B, 0xA0B3, 0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA091,
	0x5121, 0x7080, 0x6115, 0xA03A, 0x610A, 0x9012, 0x611D, 0xA0D7, 0x510B, 0x6122, 0x610E, 0x9035,
	0x6001, 0x7123, 0x6118, 0xA07A, 0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C4,
	0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A1, 0x5121, 0x7102, 0x6116, 0xA056,
	0x610C, 0x901D, 0x611E, 0xA0E8, 0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA071,
	0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BB, 0x510F, 0x7004, 0x6110, 0x9049,
	0x6002, 0x800D, 0x6119, 0xA099, 0x5121, 0x70FF, 0x6115, 0xA04C, 0x610A, 0x9017, 0x611D, 0xA0DF,
	0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA087, 0x5111, 0x700A, 0x6114, 0xA023,
	0x6109, 0x80FE, 0x611C, 0xA0CE, 0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0A9,
	0x5121, 0x7103, 0x6116, 0xA05F, 0x610C, 0x9022, 0x611E, 0xA0F5, 0x510B, 0x611F, 0x610D, 0x9029,
	0x6000, 0x7105, 0x6117, 0xA06C, 0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B7,
	0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA095, 0x5121, 0x7080, 0x6115, 0xA046,
	0x610A, 0x9015, 0x611D, 0xA0DB, 0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA07E,
	0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0C9, 0x510F, 0x7005, 0x6112, 0x907F,
	0x6107, 0x8010, 0x611A, 0xA0A5, 0x5121, 0x7102, 0x6116, 0xA05B, 0x610C, 0x901F, 0x611E, 0xA0EC,
	0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA075, 0x5111, 0x7008, 0x6113, 0x90F1,
	0x6108, 0x8040, 0x611B, 0xA0BF, 0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09D,
	0x5121, 0x70FF, 0x6115, 0xA052, 0x610A, 0x901B, 0x611D, 0xA0E4, 0x510B, 0x6122, 0x610E, 0x903F,
	0x6001, 0x7124, 0x6118, 0xA08C, 0x5111, 0x700A, 0x6114, 0xA02F, 0x6109, 0x8120, 0x611C, 0xA0D3,
	0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AE, 0x5121, 0x7103, 0x6116, 0xA064,
	0x610C, 0x9025, 0x611E, 0xA0FA, 0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06A,
	0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B5, 0x510F, 0x7003, 0x6110, 0x9043,
	0x6002, 0x800B, 0x6119, 0xA093, 0x5121, 0x7080, 0x6115, 0xA03D, 0x610A, 0x9014, 0x611D, 0xA0D9,
	0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07C, 0x5111, 0x7009, 0x6114, 0x90F8,
	0x6109, 0x8060, 0x611C, 0xA0C7, 0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A3,
	0x5121, 0x7102, 0x6116, 0xA058, 0x610C, 0x901E, 0x611E, 0xA0EA, 0x510B, 0x611F, 0x610D, 0x9030,
	0x6000, 0x7106, 0x6117, 0xA073, 0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BD,
	0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09B, 0x5121, 0x70FF, 0x6115, 0xA04E,
	0x610A, 0x901A, 0x611D, 0xA0E2, 0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08A,
	0x5111, 0x700A, 0x6114, 0xA02D, 0x6109, 0x80FE, 0x611C, 0xA0D1, 0x510F, 0x7006, 0x6112, 0x9084,
	0x6107, 0x8011, 0x611A, 0xA0AC, 0x5121, 0x7103, 0x6116, 0xA062, 0x610C, 0x9024, 0x611E, 0xA0F7,
	0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06E, 0x5111, 0x7007, 0x6113, 0x90D0,
	0x6108, 0x8019, 0x611B, 0xA0B9, 0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA097,
	0x5121, 0x7080, 0x6115, 0xA04A, 0x610A, 0x9016, 0x611D, 0xA0DD, 0x510B, 0x6122, 0x610E, 0x9039,
	0x6001, 0x7123, 0x6118, 0xA085, 0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CB,
	0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A7, 0x5121, 0x7102, 0x6116, 0xA05D,
	0x610C, 0x9021, 0x611E, 0xA0EF, 0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA077,
	0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C2, 0x510F, 0x7004, 0x6110, 0x9059,
	0x6002, 0x800E, 0x6119, 0xA09F, 0x5121, 0x70FF, 0x6115, 0xA054, 0x610A, 0x901C, 0x611D, 0xA0E6,
	0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08E, 0x5111, 0x700A, 0x6114, 0xA034,
	0x6109, 0x8120, 0x611C, 0xA0D5, 0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B0,
	0x5121, 0x7103, 0x6116, 0xA066, 0x610C, 0x9026, 0x611E, 0xA104, 0x510B, 0x611F, 0x610D, 0x9027,
	0x6000, 0x7105, 0x6117, 0xA069, 0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B4,
	0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA092, 0x5121, 0x7080, 0x6115, 0xA03B,
	0x610A, 0x9012, 0x611D, 0xA0D8, 0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07B,
	0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C5, 0x510F, 0x7005, 0x6112, 0x9070,
	0x6107, 0x800F, 0x611A, 0xA0A2, 0x5121, 0x7102, 0x6116, 0xA057, 0x610C, 0x901D, 0x611E, 0xA0E9,
	0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA072, 0x5111, 0x7008, 0x6113, 0x90E0,
	0x6108, 0x8020, 0x611B, 0xA0BC, 0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA09A,
	0x5121, 0x70FF, 0x6115, 0xA04D, 0x610A, 0x9017, 0x611D, 0xA0E1, 0x510B, 0x6122, 0x610E, 0x903C,
	0x6001, 0x7124, 0x6118, 0xA089, 0x5111, 0x700A, 0x6114, 0xA02B, 0x6109, 0x80FE, 0x611C, 0xA0CF,
	0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0AA, 0x5121, 0x7103, 0x6116, 0xA061,
	0x610C, 0x9022, 0x611E, 0xA0F6, 0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06D,
	0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B8, 0x510F, 0x7003, 0x6110, 0x9044,
	0x6002, 0x800C, 0x6119, 0xA096, 0x5121, 0x7080, 0x6115, 0xA047, 0x610A, 0x9015, 0x611D, 0xA0DC,
	0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA083, 0x5111, 0x7009, 0x6114, 0x90FC,
	0x6109, 0x80F0, 0x611C, 0xA0CA, 0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A6,
	0x5121, 0x7102, 0x6116, 0xA05C, 0x610C, 0x901F, 0x611E, 0xA0ED, 0x510B, 0x611F, 0x610D, 0x9031,
	0x6000, 0x7106, 0x6117, 0xA076, 0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0C1,
	0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09E, 0x5121, 0x70FF, 0x6115, 0xA053,
	0x610A, 0x901B, 0x611D, 0xA0E5, 0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08D,
	0x5111, 0x700A, 0x6114, 0xA032, 0x6109, 0x8120, 0x611C, 0xA0D4, 0x510F, 0x7006, 0x6112, 0x9088,
	0x6107, 0x8013, 0x611A, 0xA0AF, 0x5121, 0x7103, 0x6116, 0xA065, 0x610C, 0x9025, 0x611E, 0xA0FB,
	0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06B, 0x5111, 0x7007, 0x6113, 0x90C6,
	0x6108, 0x8018, 0x611B, 0xA0B6, 0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA094,
	0x5121, 0x7080, 0x6115, 0xA045, 0x610A, 0x9014, 0x611D, 0xA0DA, 0x510B, 0x6122, 0x610E, 0x9037,
	0x6001, 0x7123, 0x6118, 0xA07D, 0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C8,
	0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A4, 0x5121, 0x7102, 0x6116, 0xA05A,
	0x610C, 0x901E, 0x611E, 0xA0EB, 0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA074,
	0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BE, 0x510F, 0x7004, 0x6110, 0x9050,
	0x6002, 0x800D, 0x6119, 0xA09C, 0x5121, 0x70FF, 0x6115, 0xA04F, 0x610A, 0x901A, 0x611D, 0xA0E3,
	0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08B, 0x5111, 0x700A, 0x6114, 0xA02E,
	0x6109, 0x80FE, 0x611C, 0xA0D2, 0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AD,
	0x5121, 0x7103, 0x6116, 0xA063, 0x610C, 0x9024, 0x611E, 0xA0F9, 0x510B, 0x611F, 0x610D, 0x902A,
	0x6000, 0x7105, 0x6117, 0xA06F, 0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0BA,
	0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA098, 0x5121, 0x7080, 0x6115, 0xA04B,
	0x610A, 0x9016, 0x611D, 0xA0DE, 0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA086,
	0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CD, 0x510F, 0x7005, 0x6112, 0x9081,
	0x6107, 0x8010, 0x611A, 0xA0A8, 0x5121, 0x7102, 0x6116, 0xA05E, 0x610C, 0x9021, 0x611E, 0xA0F3,
	0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA079, 0x5111, 0x7008, 0x6113, 0x90F2,
	0x6108, 0x8040, 0x611B, 0xA0C3, 0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA0A0,
	0x5121, 0x70FF, 0x6115, 0xA055, 0x610A, 0x901C, 0x611D, 0xA0E7, 0x510B, 0x6122, 0x610E, 0x9041,
	0x6001, 0x7124, 0x6118, 0xA08F, 0x5111, 0x700A, 0x6114, 0xA036, 0x6109, 0x8120, 0x611C, 0xA0D6,
	0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B1, 0x5121, 0x7103, 0x6116, 0xA067,
	0x610C, 0x9026, 0x611E, 0xB0B2, 0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA068,
	0x5111, 0x7007, 0x6113, 0x90C0, 0x6108, 0x8018, 0x611B, 0xA0B3, 0x510F, 0x7003, 0x6110, 0x9042,
	0x6002, 0x800B, 0x6119, 0xA091, 0x5121, 0x7080, 0x6115, 0xA03A, 0x610A, 0x9012, 0x611D, 0xA0D7,
	0x510B, 0x6122, 0x610E, 0x9035, 0x6001, 0x7123, 0x6118, 0xA07A, 0x5111, 0x7009, 0x6114, 0x90F4,
	0x6109, 0x8060, 0x611C, 0xA0C4, 0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A1,
	0x5121, 0x7102, 0x6116, 0xA056, 0x610C, 0x901D, 0x611E, 0xA0E8, 0x510B, 0x611F, 0x610D, 0x902C,
	0x6000, 0x7106, 0x6117, 0xA071, 0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BB,
	0x510F, 0x7004, 0x6110, 0x9049, 0x6002, 0x800D, 0x6119, 0xA099, 0x5121, 0x70FF, 0x6115, 0xA04C,
	0x610A, 0x9017, 0x611D, 0xA0DF, 0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA087,
	0x5111, 0x700A, 0x6114, 0xA023, 0x6109, 0x80FE, 0x611C, 0xA0CE, 0x510F, 0x7006, 0x6112, 0x9082,
	0x6107, 0x8011, 0x611A, 0xA0A9, 0x5121, 0x7103, 0x6116, 0xA05F, 0x610C, 0x9022, 0x611E, 0xA0F5,
	0x510B, 0x611F, 0x610D, 0x9029, 0x6000, 0x7105, 0x6117, 0xA06C, 0x5111, 0x7007, 0x6113, 0x90CC,
	0x6108, 0x8019, 0x611B, 0xA0B7, 0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA095,
	0x5121, 0x7080, 0x6115, 0xA046, 0x610A, 0x9015, 0x611D, 0xA0DB, 0x510B, 0x6122, 0x610E, 0x9038,
	0x6001, 0x7123, 0x6118, 0xA07E, 0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0C9,
	0x510F, 0x7005, 0x6112, 0x907F, 0x6107, 0x8010, 0x611A, 0xA0A5, 0x5121, 0x7102, 0x6116, 0xA05B,
	0x610C, 0x901F, 0x611E, 0xA0EC, 0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA075,
	0x5111, 0x7008, 0x6113, 0x90F1, 0x6108, 0x8040, 0x611B, 0xA0BF, 0x510F, 0x7004, 0x6110, 0x9051,
	0x6002, 0x800E, 0x6119, 0xA09D, 0x5121, 0x70FF, 0x6115, 0xA052, 0x610A, 0x901B, 0x611D, 0xA0E4,
	0x510B, 0x6122, 0x610E, 0x903F, 0x6001, 0x7124, 0x6118, 0xA08C, 0x5111, 0x700A, 0x6114, 0xA02F,
	0x6109, 0x8120, 0x611C, 0xA0D3, 0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AE,
	0x5121, 0x7103, 0x6116, 0xA064, 0x610C, 0x9025, 0x611E, 0xA0FA, 0x510B, 0x611F, 0x610D, 0x9028,
	0x6000, 0x7105, 0x6117, 0xA06A, 0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B5,
	0x510F, 0x7003, 0x6110, 0x9043, 0x6002, 0x800B, 0x6119, 0xA093, 0x5121, 0x7080, 0x6115, 0xA03D,
	0x610A, 0x9014, 0x611D, 0xA0D9, 0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07C,
	0x5111, 0x7009, 0x6114, 0x90F8, 0x6109, 0x8060, 0x611C, 0xA0C7, 0x510F, 0x7005, 0x6112, 0x9078,
	0x6107, 0x800F, 0x611A, 0xA0A3, 0x5121, 0x7102, 0x6116, 0xA058, 0x610C, 0x901E, 0x611E, 0xA0EA,
	0x510B, 0x611F, 0x610D, 0x9030, 0x6000, 0x7106, 0x6117, 0xA073, 0x5111, 0x7008, 0x6113, 0x90EE,
	0x6108, 0x8020, 0x611B, 0xA0BD, 0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09B,
	0x5121, 0x70FF, 0x6115, 0xA04E, 0x610A, 0x901A, 0x611D, 0xA0E2, 0x510B, 0x6122, 0x610E, 0x903E,
	0x6001, 0x7124, 0x6118, 0xA08A, 0x5111, 0x700A, 0x6114, 0xA02D, 0x6109, 0x80FE, 0x611C, 0xA0D1,
	0x510F, 0x7006, 0x6112, 0x9084, 0x6107, 0x8011, 0x611A, 0xA0AC, 0x5121, 0x7103, 0x6116, 0xA062,
	0x610C, 0x9024, 0x611E, 0xA0F7, 0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06E,
	0x5111, 0x7007, 0x6113, 0x90D0, 0x6108, 0x8019, 0x611B, 0xA0B9, 0x510F, 0x7003, 0x6110, 0x9048,
	0x6002, 0x800C, 0x6119, 0xA097, 0x5121, 0x7080, 0x6115, 0xA04A, 0x610A, 0x9016, 0x611D, 0xA0DD,
	0x510B, 0x6122, 0x610E, 0x9039, 0x6001, 0x7123, 0x6118, 0xA085, 0x5111, 0x7009, 0x6114, 0x90FD,
	0x6109, 0x80F0, 0x611C, 0xA0CB, 0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A7,
	0x5121, 0x7102, 0x6116, 0xA05D, 0x610C, 0x9021, 0x611E, 0xA0EF, 0x510B, 0x611F, 0x610D, 0x9033,
	0x6000, 0x7106, 0x6117, 0xA077, 0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C2,
	0x510F, 0x7004, 0x6110, 0x9059, 0x6002, 0x800E, 0x6119, 0xA09F, 0x5121, 0x70FF, 0x6115, 0xA054,
	0x610A, 0x901C, 0x611D, 0xA0E6, 0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08E,
	0x5111, 0x700A, 0x6114, 0xA034, 0x6109, 0x8120, 0x611C, 0xA0D5, 0x510F, 0x7006, 0x6112, 0x9090,
	0x6107, 0x8013, 0x611A, 0xA0B0, 0x5121, 0x7103, 0x6116, 0xA066, 0x610C, 0x9026, 0x611E, 0xA104,
	0x510B, 0x611F, 0x610D, 0x9027, 0x6000, 0x7105, 0x6117, 0xA069, 0x5111, 0x7007, 0x6113, 0x90C0,
	0x6108, 0x8018, 0x611B, 0xA0B4, 0x510F, 0x7003, 0x6110, 0x9042, 0x6002, 0x800B, 0x6119, 0xA092,
	0x5121, 0x7080, 0x6115, 0xA03B, 0x610A, 0x9012, 0x611D, 0xA0D8, 0x510B, 0x6122, 0x610E, 0x9035,
	0x6001, 0x7123, 0x6118, 0xA07B, 0x5111, 0x7009, 0x6114, 0x90F4, 0x6109, 0x8060, 0x611C, 0xA0C5,
	0x510F, 0x7005, 0x6112, 0x9070, 0x6107, 0x800F, 0x611A, 0xA0A2, 0x5121, 0x7102, 0x6116, 0xA057,
	0x610C, 0x901D, 0x611E, 0xA0E9, 0x510B, 0x611F, 0x610D, 0x902C, 0x6000, 0x7106, 0x6117, 0xA072,
	0x5111, 0x7008, 0x6113, 0x90E0, 0x6108, 0x8020, 0x611B, 0xA0BC, 0x510F, 0x7004, 0x6110, 0x9049,
	0x6002, 0x800D, 0x6119, 0xA09A, 0x5121, 0x70FF, 0x6115, 0xA04D, 0x610A, 0x9017, 0x611D, 0xA0E1,
	0x510B, 0x6122, 0x610E, 0x903C, 0x6001, 0x7124, 0x6118, 0xA089, 0x5111, 0x700A, 0x6114, 0xA02B,
	0x6109, 0x80FE, 0x611C, 0xA0CF, 0x510F, 0x7006, 0x6112, 0x9082, 0x6107, 0x8011, 0x611A, 0xA0AA,
	0x5121, 0x7103, 0x6116, 0xA061, 0x610C, 0x9022, 0x611E, 0xA0F6, 0x510B, 0x611F, 0x610D, 0x9029,
	0x6000, 0x7105, 0x6117, 0xA06D, 0x5111, 0x7007, 0x6113, 0x90CC, 0x6108, 0x8019, 0x611B, 0xA0B8,
	0x510F, 0x7003, 0x6110, 0x9044, 0x6002, 0x800C, 0x6119, 0xA096, 0x5121, 0x7080, 0x6115, 0xA047,
	0x610A, 0x9015, 0x611D, 0xA0DC, 0x510B, 0x6122, 0x610E, 0x9038, 0x6001, 0x7123, 0x6118, 0xA083,
	0x5111, 0x7009, 0x6114, 0x90FC, 0x6109, 0x80F0, 0x611C, 0xA0CA, 0x510F, 0x7005, 0x6112, 0x907F,
	0x6107, 0x8010, 0x611A, 0xA0A6, 0x5121, 0x7102, 0x6116, 0xA05C, 0x610C, 0x901F, 0x611E, 0xA0ED,
	0x510B, 0x611F, 0x610D, 0x9031, 0x6000, 0x7106, 0x6117, 0xA076, 0x5111, 0x7008, 0x6113, 0x90F1,
	0x6108, 0x8040, 0x611B, 0xA0C1, 0x510F, 0x7004, 0x6110, 0x9051, 0x6002, 0x800E, 0x6119, 0xA09E,
	0x5121, 0x70FF, 0x6115, 0xA053, 0x610A, 0x901B, 0x611D, 0xA0E5, 0x510B, 0x6122, 0x610E, 0x903F,
	0x6001, 0x7124, 0x6118, 0xA08D, 0x5111, 0x700A, 0x6114, 0xA032, 0x6109, 0x8120, 0x611C, 0xA0D4,
	0x510F, 0x7006, 0x6112, 0x9088, 0x6107, 0x8013, 0x611A, 0xA0AF, 0x5121, 0x7103, 0x6116, 0xA065,
	0x610C, 0x9025, 0x611E, 0xA0FB, 0x510B, 0x611F, 0x610D, 0x9028, 0x6000, 0x7105, 0x6117, 0xA06B,
	0x5111, 0x7007, 0x6113, 0x90C6, 0x6108, 0x8018, 0x611B, 0xA0B6, 0x510F, 0x7003, 0x6110, 0x9043,
	0x6002, 0x800B, 0x6119, 0xA094, 0x5121, 0x7080, 0x6115, 0xA045, 0x610A, 0x9014, 0x611D, 0xA0DA,
	0x510B, 0x6122, 0x610E, 0x9037, 0x6001, 0x7123, 0x6118, 0xA07D, 0x5111, 0x7009, 0x6114, 0x90F8,
	0x6109, 0x8060, 0x611C, 0xA0C8, 0x510F, 0x7005, 0x6112, 0x9078, 0x6107, 0x800F, 0x611A, 0xA0A4,
	0x5121, 0x7102, 0x6116, 0xA05A, 0x610C, 0x901E, 0x611E, 0xA0EB, 0x510B, 0x611F, 0x610D, 0x9030,
	0x6000, 0x7106, 0x6117, 0xA074, 0x5111, 0x7008, 0x6113, 0x90EE, 0x6108, 0x8020, 0x611B, 0xA0BE,
	0x510F, 0x7004, 0x6110, 0x9050, 0x6002, 0x800D, 0x6119, 0xA09C, 0x5121, 0x70FF, 0x6115, 0xA04F,
	0x610A, 0x901A, 0x611D, 0xA0E3, 0x510B, 0x6122, 0x610E, 0x903E, 0x6001, 0x7124, 0x6118, 0xA08B,
	0x5111, 0x700A, 0x6114, 0xA02E, 0x6109, 0x80FE, 0x611C, 0xA0D2, 0x510F, 0x7006, 0x6112, 0x9084,
	0x6107, 0x8011, 0x611A, 0xA0AD, 0x5121, 0x7103, 0x6116, 0xA063, 0x610C, 0x9024, 0x611E, 0xA0F9,
	0x510B, 0x611F, 0x610D, 0x902A, 0x6000, 0x7105, 0x6117, 0xA06F, 0x5111, 0x7007, 0x6113, 0x90D0,
	0x6108, 0x8019, 0x611B, 0xA0BA, 0x510F, 0x7003, 0x6110, 0x9048, 0x6002, 0x800C, 0x6119, 0xA098,
	0x5121, 0x7080, 0x6115, 0xA04B, 0x610A, 0x9016, 0x611D, 0xA0DE, 0x510B, 0x6122, 0x610E, 0x9039,
	0x6001, 0x7123, 0x6118, 0xA086, 0x5111, 0x7009, 0x6114, 0x90FD, 0x6109, 0x80F0, 0x611C, 0xA0CD,
	0x510F, 0x7005, 0x6112, 0x9081, 0x6107, 0x8010, 0x611A, 0xA0A8, 0x5121, 0x7102, 0x6116, 0xA05E,
	0x610C, 0x9021, 0x611E, 0xA0F3, 0x510B, 0x611F, 0x610D, 0x9033, 0x6000, 0x7106, 0x6117, 0xA079,
	0x5111, 0x7008, 0x6113, 0x90F2, 0x6108, 0x8040, 0x611B, 0xA0C3, 0x510F, 0x7004, 0x6110, 0x9059,
	0x6002, 0x800E, 0x6119, 0xA0A0, 0x5121, 0x70FF, 0x6115, 0xA055, 0x610A, 0x901C, 0x611D, 0xA0E7,
	0x510B, 0x6122, 0x610E, 0x9041, 0x6001, 0x7124, 0x6118, 0xA08F, 0x5111, 0x700A, 0x6114, 0xA036,
	0x6109, 0x8120, 0x611C, 0xA0D6, 0x510F, 0x7006, 0x6112, 0x9090, 0x6107, 0x8013, 0x611A, 0xA0B1,
	0x5121, 0x7103, 0x6116, 0xA067, 0x610C, 0x9026, 0x611E, 0xD125
};

static const UINT16 HuffTableLOM[512] = {
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007,
	0x2001, 0x4006, 0x3004, 0x700D, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600B,
	0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x8012, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x7010,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600C, 0x2001, 0x4003, 0x3002, 0x5009,
	0x2001, 0x4006, 0x3004, 0x9018, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A,
	0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x700E, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x600B, 0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x9013,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007,
	0x2001, 0x4006, 0x3004, 0x800F, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600C,
	0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x901C, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x700D,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600B, 0x2001, 0x4003, 0x3002, 0x5009,
	0x2001, 0x4006, 0x3004, 0x8015, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A,
	0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x7010, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x600C, 0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x901A,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007,
	0x2001, 0x4006, 0x3004, 0x700E, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600B,
	0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x9016, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x8011,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600C, 0x2001, 0x4003, 0x3002, 0x5009,
	0x2001, 0x4006, 0x3004, 0x901E, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A,
	0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x700D, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x600B, 0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x8012,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007,
	0x2001, 0x4006, 0x3004, 0x7010, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600C,
	0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x9019, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x700E,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600B, 0x2001, 0x4003, 0x3002, 0x5009,
	0x2001, 0x4006, 0x3004, 0x9014, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A,
	0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x800F, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x600C, 0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x901D,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007,
	0x2001, 0x4006, 0x3004, 0x700D, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600B,
	0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x8015, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x7010,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600C, 0x2001, 0x4003, 0x3002, 0x5009,
	0x2001, 0x4006, 0x3004, 0x901B, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A,
	0x2001, 0x4003, 0x3002, 0x5007, 0x2001, 0x4006, 0x3004, 0x700E, 0x2001, 0x4000, 0x3002, 0x4008,
	0x2001, 0x4005, 0x3004, 0x600B, 0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x9017,
	0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x500A, 0x2001, 0x4003, 0x3002, 0x5007,
	0x2001, 0x4006, 0x3004, 0x8011, 0x2001, 0x4000, 0x3002, 0x4008, 0x2001, 0x4005, 0x3004, 0x600C,
	0x2001, 0x4003, 0x3002, 0x5009, 0x2001, 0x4006, 0x3004, 0x901F
};

static const BYTE HuffTableMask[39] = {
	0x11, /* 0 */
	0x9E, /* 1 */
	0xA1, /* 2 */
	0x00, /* 3 */
	0x00, /* 4 */
	0x01, /* 5 */
	0x00, /* 6 */
	0x03, /* 7 */
	0x00, /* 8 */
	0x07, /* 9 */
	0x00, /* 10 */
	0x0F, /* 11 */
	0x00, /* 12 */
	0x1F, /* 13 */
	0x00, /* 14 */
	0x3F, /* 15 */
	0x00, /* 16 */
	0x7F, /* 17 */
	0x00, /* 18 */
	0xFF, /* 19 */
	0x00, /* 20 */
	0xFF, /* 21 */
	0x01, /* 22 */
	0xFF, /* 23 */
	0x03, /* 24 */
	0xFF, /* 25 */
	0x07, /* 26 */
	0xFF, /* 27 */
	0x0F, /* 28 */
	0xFF, /* 29 */
	0x1F, /* 30 */
	0xFF, /* 31 */
	0x3F, /* 32 */
	0xFF, /* 33 */
	0x7F, /* 34 */
	0xFF, /* 35 */
	0xFF, /* 36 */
	0x00, /* 37 */
	0x00  /* 38 */
};

static const BYTE HuffLengthLEC[294] = {
	6,  /* 0 */
	6,  /* 1 */
	6,  /* 2 */
	7,  /* 3 */
	7,  /* 4 */
	7,  /* 5 */
	7,  /* 6 */
	7,  /* 7 */
	7,  /* 8 */
	7,  /* 9 */
	7,  /* 10 */
	8,  /* 11 */
	8,  /* 12 */
	8,  /* 13 */
	8,  /* 14 */
	8,  /* 15 */
	8,  /* 16 */
	8,  /* 17 */
	9,  /* 18 */
	8,  /* 19 */
	9,  /* 20 */
	9,  /* 21 */
	9,  /* 22 */
	9,  /* 23 */
	8,  /* 24 */
	8,  /* 25 */
	9,  /* 26 */
	9,  /* 27 */
	9,  /* 28 */
	9,  /* 29 */
	9,  /* 30 */
	9,  /* 31 */
	8,  /* 32 */
	9,  /* 33 */
	9,  /* 34 */
	10, /* 35 */
	9,  /* 36 */
	9,  /* 37 */
	9,  /* 38 */
	9,  /* 39 */
	9,  /* 40 */
	9,  /* 41 */
	9,  /* 42 */
	10, /* 43 */
	9,  /* 44 */
	10, /* 45 */
	10, /* 46 */
	10, /* 47 */
	9,  /* 48 */
	9,  /* 49 */
	10, /* 50 */
	9,  /* 51 */
	10, /* 52 */
	9,  /* 53 */
	10, /* 54 */
	9,  /* 55 */
	9,  /* 56 */
	9,  /* 57 */
	10, /* 58 */
	10, /* 59 */
	9,  /* 60 */
	10, /* 61 */
	9,  /* 62 */
	9,  /* 63 */
	8,  /* 64 */
	9,  /* 65 */
	9,  /* 66 */
	9,  /* 67 */
	9,  /* 68 */
	10, /* 69 */
	10, /* 70 */
	10, /* 71 */
	9,  /* 72 */
	9,  /* 73 */
	10, /* 74 */
	10, /* 75 */
	10, /* 76 */
	10, /* 77 */
	10, /* 78 */
	10, /* 79 */
	9,  /* 80 */
	9,  /* 81 */
	10, /* 82 */
	10, /* 83 */
	10, /* 84 */
	10, /* 85 */
	10, /* 86 */
	10, /* 87 */
	10, /* 88 */
	9,  /* 89 */
	10, /* 90 */
	10, /* 91 */
	10, /* 92 */
	10, /* 93 */
	10, /* 94 */
	10, /* 95 */
	8,  /* 96 */
	10, /* 97 */
	10, /* 98 */
	10, /* 99 */
	10, /* 100 */
	10, /* 101 */
	10, /* 102 */
	10, /* 103 */
	10, /* 104 */
	10, /* 105 */
	10, /* 106 */
	10, /* 107 */
	10, /* 108 */
	10, /* 109 */
	10, /* 110 */
	10, /* 111 */
	9,  /* 112 */
	10, /* 113 */
	10, /* 114 */
	10, /* 115 */
	10, /* 116 */
	10, /* 117 */
	10, /* 118 */
	10, /* 119 */
	9,  /* 120 */
	10, /* 121 */
	10, /* 122 */
	10, /* 123 */
	10, /* 124 */
	10, /* 125 */
	10, /* 126 */
	9,  /* 127 */
	7,  /* 128 */
	9,  /* 129 */
	9,  /* 130 */
	10, /* 131 */
	9,  /* 132 */
	10, /* 133 */
	10, /* 134 */
	10, /* 135 */
	9,  /* 136 */
	10, /* 137 */
	10, /* 138 */
	10, /* 139 */
	10, /* 140 */
	10, /* 141 */
	10, /* 142 */
	10, /* 143 */
	9,  /* 144 */
	10, /* 145 */
	10, /* 146 */
	10, /* 147 */
	10, /* 148 */
	10, /* 149 */
	10, /* 150 */
	10, /* 151 */
	10, /* 152 */
	10, /* 153 */
	10, /* 154 */
	10, /* 155 */
	10, /* 156 */
	10, /* 157 */
	10, /* 158 */
	10, /* 159 */
	10, /* 160 */
	10, /* 161 */
	10, /* 162 */
	10, /* 163 */
	10, /* 164 */
	10, /* 165 */
	10, /* 166 */
	10, /* 167 */
	10, /* 168 */
	10, /* 169 */
	10, /* 170 */
	13, /* 171 */
	10, /* 172 */
	10, /* 173 */
	10, /* 174 */
	10, /* 175 */
	10, /* 176 */
	10, /* 177 */
	11, /* 178 */
	10, /* 179 */
	10, /* 180 */
	10, /* 181 */
	10, /* 182 */
	10, /* 183 */
	10, /* 184 */
	10, /* 185 */
	10, /* 186 */
	10, /* 187 */
	10, /* 188 */
	10, /* 189 */
	10, /* 190 */
	10, /* 191 */
	9,  /* 192 */
	10, /* 193 */
	10, /* 194 */
	10, /* 195 */
	10, /* 196 */
	10, /* 197 */
	9,  /* 198 */
	10, /* 199 */
	10, /* 200 */
	10, /* 201 */
	10, /* 202 */
	10, /* 203 */
	9,  /* 204 */
	10, /* 205 */
	10, /* 206 */
	10, /* 207 */
	9,  /* 208 */
	10, /* 209 */
	10, /* 210 */
	10, /* 211 */
	10, /* 212 */
	10, /* 213 */
	10, /* 214 */
	10, /* 215 */
	10, /* 216 */
	10, /* 217 */
	10, /* 218 */
	10, /* 219 */
	10, /* 220 */
	10, /* 221 */
	10, /* 222 */
	10, /* 223 */
	9,  /* 224 */
	10, /* 225 */
	10, /* 226 */
	10, /* 227 */
	10, /* 228 */
	10, /* 229 */
	10, /* 230 */
	10, /* 231 */
	10, /* 232 */
	10, /* 233 */
	10, /* 234 */
	10, /* 235 */
	10, /* 236 */
	10, /* 237 */
	9,  /* 238 */
	10, /* 239 */
	8,  /* 240 */
	9,  /* 241 */
	9,  /* 242 */
	10, /* 243 */
	9,  /* 244 */
	10, /* 245 */
	10, /* 246 */
	10, /* 247 */
	9,  /* 248 */
	10, /* 249 */
	10, /* 250 */
	10, /* 251 */
	9,  /* 252 */
	9,  /* 253 */
	8,  /* 254 */
	7,  /* 255 */
	13, /* 256 */
	13, /* 257 */
	7,  /* 258 */
	7,  /* 259 */
	10, /* 260 */
	7,  /* 261 */
	7,  /* 262 */
	6,  /* 263 */
	6,  /* 264 */
	6,  /* 265 */
	6,  /* 266 */
	5,  /* 267 */
	6,  /* 268 */
	6,  /* 269 */
	6,  /* 270 */
	5,  /* 271 */
	6,  /* 272 */
	5,  /* 273 */
	6,  /* 274 */
	6,  /* 275 */
	6,  /* 276 */
	6,  /* 277 */
	6,  /* 278 */
	6,  /* 279 */
	6,  /* 280 */
	6,  /* 281 */
	6,  /* 282 */
	6,  /* 283 */
	6,  /* 284 */
	6,  /* 285 */
	6,  /* 286 */
	6,  /* 287 */
	8,  /* 288 */
	5,  /* 289 */
	6,  /* 290 */
	7,  /* 291 */
	7,  /* 292 */
	13  /* 293 */
};

static const BYTE HuffCodeLEC[588] = {
	0x04, /* 0 */
	0x00, /* 1 */
	0x24, /* 2 */
	0x00, /* 3 */
	0x14, /* 4 */
	0x00, /* 5 */
	0x11, /* 6 */
	0x00, /* 7 */
	0x51, /* 8 */
	0x00, /* 9 */
	0x31, /* 10 */
	0x00, /* 11 */
	0x71, /* 12 */
	0x00, /* 13 */
	0x09, /* 14 */
	0x00, /* 15 */
	0x49, /* 16 */
	0x00, /* 17 */
	0x29, /* 18 */
	0x00, /* 19 */
	0x69, /* 20 */
	0x00, /* 21 */
	0x15, /* 22 */
	0x00, /* 23 */
	0x95, /* 24 */
	0x00, /* 25 */
	0x55, /* 26 */
	0x00, /* 27 */
	0xD5, /* 28 */
	0x00, /* 29 */
	0x35, /* 30 */
	0x00, /* 31 */
	0xB5, /* 32 */
	0x00, /* 33 */
	0x75, /* 34 */
	0x00, /* 35 */
	0x1D, /* 36 */
	0x00, /* 37 */
	0xF5, /* 38 */
	0x00, /* 39 */
	0x1D, /* 40 */
	0x01, /* 41 */
	0x9D, /* 42 */
	0x00, /* 43 */
	0x9D, /* 44 */
	0x01, /* 45 */
	0x5D, /* 46 */
	0x00, /* 47 */
	0x0D, /* 48 */
	0x00, /* 49 */
	0x8D, /* 50 */
	0x00, /* 51 */
	0x5D, /* 52 */
	0x01, /* 53 */
	0xDD, /* 54 */
	0x00, /* 55 */
	0xDD, /* 56 */
	0x01, /* 57 */
	0x3D, /* 58 */
	0x00, /* 59 */
	0x3D, /* 60 */
	0x01, /* 61 */
	0xBD, /* 62 */
	0x00, /* 63 */
	0x4D, /* 64 */
	0x00, /* 65 */
	0xBD, /* 66 */
	0x01, /* 67 */
	0x7D, /* 68 */
	0x00, /* 69 */
	0x6B, /* 70 */
	0x00, /* 71 */
	0x7D, /* 72 */
	0x01, /* 73 */
	0xFD, /* 74 */
	0x00, /* 75 */
	0xFD, /* 76 */
	0x01, /* 77 */
	0x03, /* 78 */
	0x00, /* 79 */
	0x03, /* 80 */
	0x01, /* 81 */
	0x83, /* 82 */
	0x00, /* 83 */
	0x83, /* 84 */
	0x01, /* 85 */
	0x6B, /* 86 */
	0x02, /* 87 */
	0x43, /* 88 */
	0x00, /* 89 */
	0x6B, /* 90 */
	0x01, /* 91 */
	0x6B, /* 92 */
	0x03, /* 93 */
	0xEB, /* 94 */
	0x00, /* 95 */
	0x43, /* 96 */
	0x01, /* 97 */
	0xC3, /* 98 */
	0x00, /* 99 */
	0xEB, /* 100 */
	0x02, /* 101 */
	0xC3, /* 102 */
	0x01, /* 103 */
	0xEB, /* 104 */
	0x01, /* 105 */
	0x23, /* 106 */
	0x00, /* 107 */
	0xEB, /* 108 */
	0x03, /* 109 */
	0x23, /* 110 */
	0x01, /* 111 */
	0xA3, /* 112 */
	0x00, /* 113 */
	0xA3, /* 114 */
	0x01, /* 115 */
	0x1B, /* 116 */
	0x00, /* 117 */
	0x1B, /* 118 */
	0x02, /* 119 */
	0x63, /* 120 */
	0x00, /* 121 */
	0x1B, /* 122 */
	0x01, /* 123 */
	0x63, /* 124 */
	0x01, /* 125 */
	0xE3, /* 126 */
	0x00, /* 127 */
	0xCD, /* 128 */
	0x00, /* 129 */
	0xE3, /* 130 */
	0x01, /* 131 */
	0x13, /* 132 */
	0x00, /* 133 */
	0x13, /* 134 */
	0x01, /* 135 */
	0x93, /* 136 */
	0x00, /* 137 */
	0x1B, /* 138 */
	0x03, /* 139 */
	0x9B, /* 140 */
	0x00, /* 141 */
	0x9B, /* 142 */
	0x02, /* 143 */
	0x93, /* 144 */
	0x01, /* 145 */
	0x53, /* 146 */
	0x00, /* 147 */
	0x9B, /* 148 */
	0x01, /* 149 */
	0x9B, /* 150 */
	0x03, /* 151 */
	0x5B, /* 152 */
	0x00, /* 153 */
	0x5B, /* 154 */
	0x02, /* 155 */
	0x5B, /* 156 */
	0x01, /* 157 */
	0x5B, /* 158 */
	0x03, /* 159 */
	0x53, /* 160 */
	0x01, /* 161 */
	0xD3, /* 162 */
	0x00, /* 163 */
	0xDB, /* 164 */
	0x00, /* 165 */
	0xDB, /* 166 */
	0x02, /* 167 */
	0xDB, /* 168 */
	0x01, /* 169 */
	0xDB, /* 170 */
	0x03, /* 171 */
	0x3B, /* 172 */
	0x00, /* 173 */
	0x3B, /* 174 */
	0x02, /* 175 */
	0x3B, /* 176 */
	0x01, /* 177 */
	0xD3, /* 178 */
	0x01, /* 179 */
	0x3B, /* 180 */
	0x03, /* 181 */
	0xBB, /* 182 */
	0x00, /* 183 */
	0xBB, /* 184 */
	0x02, /* 185 */
	0xBB, /* 186 */
	0x01, /* 187 */
	0xBB, /* 188 */
	0x03, /* 189 */
	0x7B, /* 190 */
	0x00, /* 191 */
	0x2D, /* 192 */
	0x00, /* 193 */
	0x7B, /* 194 */
	0x02, /* 195 */
	0x7B, /* 196 */
	0x01, /* 197 */
	0x7B, /* 198 */
	0x03, /* 199 */
	0xFB, /* 200 */
	0x00, /* 201 */
	0xFB, /* 202 */
	0x02, /* 203 */
	0xFB, /* 204 */
	0x01, /* 205 */
	0xFB, /* 206 */
	0x03, /* 207 */
	0x07, /* 208 */
	0x00, /* 209 */
	0x07, /* 210 */
	0x02, /* 211 */
	0x07, /* 212 */
	0x01, /* 213 */
	0x07, /* 214 */
	0x03, /* 215 */
	0x87, /* 216 */
	0x00, /* 217 */
	0x87, /* 218 */
	0x02, /* 219 */
	0x87, /* 220 */
	0x01, /* 221 */
	0x87, /* 222 */
	0x03, /* 223 */
	0x33, /* 224 */
	0x00, /* 225 */
	0x47, /* 226 */
	0x00, /* 227 */
	0x47, /* 228 */
	0x02, /* 229 */
	0x47, /* 230 */
	0x01, /* 231 */
	0x47, /* 232 */
	0x03, /* 233 */
	0xC7, /* 234 */
	0x00, /* 235 */
	0xC7, /* 236 */
	0x02, /* 237 */
	0xC7, /* 238 */
	0x01, /* 239 */
	0x33, /* 240 */
	0x01, /* 241 */
	0xC7, /* 242 */
	0x03, /* 243 */
	0x27, /* 244 */
	0x00, /* 245 */
	0x27, /* 246 */
	0x02, /* 247 */
	0x27, /* 248 */
	0x01, /* 249 */
	0x27, /* 250 */
	0x03, /* 251 */
	0xA7, /* 252 */
	0x00, /* 253 */
	0xB3, /* 254 */
	0x00, /* 255 */
	0x19, /* 256 */
	0x00, /* 257 */
	0xB3, /* 258 */
	0x01, /* 259 */
	0x73, /* 260 */
	0x00, /* 261 */
	0xA7, /* 262 */
	0x02, /* 263 */
	0x73, /* 264 */
	0x01, /* 265 */
	0xA7, /* 266 */
	0x01, /* 267 */
	0xA7, /* 268 */
	0x03, /* 269 */
	0x67, /* 270 */
	0x00, /* 271 */
	0xF3, /* 272 */
	0x00, /* 273 */
	0x67, /* 274 */
	0x02, /* 275 */
	0x67, /* 276 */
	0x01, /* 277 */
	0x67, /* 278 */
	0x03, /* 279 */
	0xE7, /* 280 */
	0x00, /* 281 */
	0xE7, /* 282 */
	0x02, /* 283 */
	0xE7, /* 284 */
	0x01, /* 285 */
	0xE7, /* 286 */
	0x03, /* 287 */
	0xF3, /* 288 */
	0x01, /* 289 */
	0x17, /* 290 */
	0x00, /* 291 */
	0x17, /* 292 */
	0x02, /* 293 */
	0x17, /* 294 */
	0x01, /* 295 */
	0x17, /* 296 */
	0x03, /* 297 */
	0x97, /* 298 */
	0x00, /* 299 */
	0x97, /* 300 */
	0x02, /* 301 */
	0x97, /* 302 */
	0x01, /* 303 */
	0x97, /* 304 */
	0x03, /* 305 */
	0x57, /* 306 */
	0x00, /* 307 */
	0x57, /* 308 */
	0x02, /* 309 */
	0x57, /* 310 */
	0x01, /* 311 */
	0x57, /* 312 */
	0x03, /* 313 */
	0xD7, /* 314 */
	0x00, /* 315 */
	0xD7, /* 316 */
	0x02, /* 317 */
	0xD7, /* 318 */
	0x01, /* 319 */
	0xD7, /* 320 */
	0x03, /* 321 */
	0x37, /* 322 */
	0x00, /* 323 */
	0x37, /* 324 */
	0x02, /* 325 */
	0x37, /* 326 */
	0x01, /* 327 */
	0x37, /* 328 */
	0x03, /* 329 */
	0xB7, /* 330 */
	0x00, /* 331 */
	0xB7, /* 332 */
	0x02, /* 333 */
	0xB7, /* 334 */
	0x01, /* 335 */
	0xB7, /* 336 */
	0x03, /* 337 */
	0x77, /* 338 */
	0x00, /* 339 */
	0x77, /* 340 */
	0x02, /* 341 */
	0xFF, /* 342 */
	0x07, /* 343 */
	0x77, /* 344 */
	0x01, /* 345 */
	0x77, /* 346 */
	0x03, /* 347 */
	0xF7, /* 348 */
	0x00, /* 349 */
	0xF7, /* 350 */
	0x02, /* 351 */
	0xF7, /* 352 */
	0x01, /* 353 */
	0xF7, /* 354 */
	0x03, /* 355 */
	0xFF, /* 356 */
	0x03, /* 357 */
	0x0F, /* 358 */
	0x00, /* 359 */
	0x0F, /* 360 */
	0x02, /* 361 */
	0x0F, /* 362 */
	0x01, /* 363 */
	0x0F, /* 364 */
	0x03, /* 365 */
	0x8F, /* 366 */
	0x00, /* 367 */
	0x8F, /* 368 */
	0x02, /* 369 */
	0x8F, /* 370 */
	0x01, /* 371 */
	0x8F, /* 372 */
	0x03, /* 373 */
	0x4F, /* 374 */
	0x00, /* 375 */
	0x4F, /* 376 */
	0x02, /* 377 */
	0x4F, /* 378 */
	0x01, /* 379 */
	0x4F, /* 380 */
	0x03, /* 381 */
	0xCF, /* 382 */
	0x00, /* 383 */
	0x0B, /* 384 */
	0x00, /* 385 */
	0xCF, /* 386 */
	0x02, /* 387 */
	0xCF, /* 388 */
	0x01, /* 389 */
	0xCF, /* 390 */
	0x03, /* 391 */
	0x2F, /* 392 */
	0x00, /* 393 */
	0x2F, /* 394 */
	0x02, /* 395 */
	0x0B, /* 396 */
	0x01, /* 397 */
	0x2F, /* 398 */
	0x01, /* 399 */
	0x2F, /* 400 */
	0x03, /* 401 */
	0xAF, /* 402 */
	0x00, /* 403 */
	0xAF, /* 404 */
	0x02, /* 405 */
	0xAF, /* 406 */
	0x01, /* 407 */
	0x8B, /* 408 */
	0x00, /* 409 */
	0xAF, /* 410 */
	0x03, /* 411 */
	0x6F, /* 412 */
	0x00, /* 413 */
	0x6F, /* 414 */
	0x02, /* 415 */
	0x8B, /* 416 */
	0x01, /* 417 */
	0x6F, /* 418 */
	0x01, /* 419 */
	0x6F, /* 420 */
	0x03, /* 421 */
	0xEF, /* 422 */
	0x00, /* 423 */
	0xEF, /* 424 */
	0x02, /* 425 */
	0xEF, /* 426 */
	0x01, /* 427 */
	0xEF, /* 428 */
	0x03, /* 429 */
	0x1F, /* 430 */
	0x00, /* 431 */
	0x1F, /* 432 */
	0x02, /* 433 */
	0x1F, /* 434 */
	0x01, /* 435 */
	0x1F, /* 436 */
	0x03, /* 437 */
	0x9F, /* 438 */
	0x00, /* 439 */
	0x9F, /* 440 */
	0x02, /* 441 */
	0x9F, /* 442 */
	0x01, /* 443 */
	0x9F, /* 444 */
	0x03, /* 445 */
	0x5F, /* 446 */
	0x00, /* 447 */
	0x4B, /* 448 */
	0x00, /* 449 */
	0x5F, /* 450 */
	0x02, /* 451 */
	0x5F, /* 452 */
	0x01, /* 453 */
	0x5F, /* 454 */
	0x03, /* 455 */
	0xDF, /* 456 */
	0x00, /* 457 */
	0xDF, /* 458 */
	0x02, /* 459 */
	0xDF, /* 460 */
	0x01, /* 461 */
	0xDF, /* 462 */
	0x03, /* 463 */
	0x3F, /* 464 */
	0x00, /* 465 */
	0x3F, /* 466 */
	0x02, /* 467 */
	0x3F, /* 468 */
	0x01, /* 469 */
	0x3F, /* 470 */
	0x03, /* 471 */
	0xBF, /* 472 */
	0x00, /* 473 */
	0xBF, /* 474 */
	0x02, /* 475 */
	0x4B, /* 476 */
	0x01, /* 477 */
	0xBF, /* 478 */
	0x01, /* 479 */
	0xAD, /* 480 */
	0x00, /* 481 */
	0xCB, /* 482 */
	0x00, /* 483 */
	0xCB, /* 484 */
	0x01, /* 485 */
	0xBF, /* 486 */
	0x03, /* 487 */
	0x2B, /* 488 */
	0x00, /* 489 */
	0x7F, /* 490 */
	0x00, /* 491 */
	0x7F, /* 492 */
	0x02, /* 493 */
	0x7F, /* 494 */
	0x01, /* 495 */
	0x2B, /* 496 */
	0x01, /* 497 */
	0x7F, /* 498 */
	0x03, /* 499 */
	0xFF, /* 500 */
	0x00, /* 501 */
	0xFF, /* 502 */
	0x02, /* 503 */
	0xAB, /* 504 */
	0x00, /* 505 */
	0xAB, /* 506 */
	0x01, /* 507 */
	0x6D, /* 508 */
	0x00, /* 509 */
	0x59, /* 510 */
	0x00, /* 511 */
	0xFF, /* 512 */
	0x17, /* 513 */
	0xFF, /* 514 */
	0x0F, /* 515 */
	0x39, /* 516 */
	0x00, /* 517 */
	0x79, /* 518 */
	0x00, /* 519 */
	0xFF, /* 520 */
	0x01, /* 521 */
	0x05, /* 522 */
	0x00, /* 523 */
	0x45, /* 524 */
	0x00, /* 525 */
	0x34, /* 526 */
	0x00, /* 527 */
	0x0C, /* 528 */
	0x00, /* 529 */
	0x2C, /* 530 */
	0x00, /* 531 */
	0x1C, /* 532 */
	0x00, /* 533 */
	0x00, /* 534 */
	0x00, /* 535 */
	0x3C, /* 536 */
	0x00, /* 537 */
	0x02, /* 538 */
	0x00, /* 539 */
	0x22, /* 540 */
	0x00, /* 541 */
	0x10, /* 542 */
	0x00, /* 543 */
	0x12, /* 544 */
	0x00, /* 545 */
	0x08, /* 546 */
	0x00, /* 547 */
	0x32, /* 548 */
	0x00, /* 549 */
	0x0A, /* 550 */
	0x00, /* 551 */
	0x2A, /* 552 */
	0x00, /* 553 */
	0x1A, /* 554 */
	0x00, /* 555 */
	0x3A, /* 556 */
	0x00, /* 557 */
	0x06, /* 558 */
	0x00, /* 559 */
	0x26, /* 560 */
	0x00, /* 561 */
	0x16, /* 562 */
	0x00, /* 563 */
	0x36, /* 564 */
	0x00, /* 565 */
	0x0E, /* 566 */
	0x00, /* 567 */
	0x2E, /* 568 */
	0x00, /* 569 */
	0x1E, /* 570 */
	0x00, /* 571 */
	0x3E, /* 572 */
	0x00, /* 573 */
	0x01, /* 574 */
	0x00, /* 575 */
	0xED, /* 576 */
	0x00, /* 577 */
	0x18, /* 578 */
	0x00, /* 579 */
	0x21, /* 580 */
	0x00, /* 581 */
	0x25, /* 582 */
	0x00, /* 583 */
	0x65, /* 584 */
	0x00, /* 585 */
	0xFF, /* 586 */
	0x1F  /* 587 */
};

static const BYTE HuffLengthLOM[32] = {
	4, /* 0 */
	2, /* 1 */
	3, /* 2 */
	4, /* 3 */
	3, /* 4 */
	4, /* 5 */
	4, /* 6 */
	5, /* 7 */
	4, /* 8 */
	5, /* 9 */
	5, /* 10 */
	6, /* 11 */
	6, /* 12 */
	7, /* 13 */
	7, /* 14 */
	8, /* 15 */
	7, /* 16 */
	8, /* 17 */
	8, /* 18 */
	9, /* 19 */
	9, /* 20 */
	8, /* 21 */
	9, /* 22 */
	9, /* 23 */
	9, /* 24 */
	9, /* 25 */
	9, /* 26 */
	9, /* 27 */
	9, /* 28 */
	9, /* 29 */
	9, /* 30 */
	9  /* 31 */
};

static const UINT16 HuffCodeLOM[32] = {
	0x0001, /* 0 */
	0x0000, /* 1 */
	0x0002, /* 2 */
	0x0009, /* 3 */
	0x0006, /* 4 */
	0x0005, /* 5 */
	0x000D, /* 6 */
	0x000B, /* 7 */
	0x0003, /* 8 */
	0x001B, /* 9 */
	0x0007, /* 10 */
	0x0017, /* 11 */
	0x0037, /* 12 */
	0x000F, /* 13 */
	0x004F, /* 14 */
	0x006F, /* 15 */
	0x002F, /* 16 */
	0x00EF, /* 17 */
	0x001F, /* 18 */
	0x005F, /* 19 */
	0x015F, /* 20 */
	0x009F, /* 21 */
	0x00DF, /* 22 */
	0x01DF, /* 23 */
	0x003F, /* 24 */
	0x013F, /* 25 */
	0x00BF, /* 26 */
	0x01BF, /* 27 */
	0x007F, /* 28 */
	0x017F, /* 29 */
	0x00FF, /* 30 */
	0x01FF  /* 31 */
};

static const UINT32 CopyOffsetBitsLUT[32] = {
	0x0, /* 0 */
	0x0, /* 1 */
	0x0, /* 2 */
	0x0, /* 3 */
	0x1, /* 4 */
	0x1, /* 5 */
	0x2, /* 6 */
	0x2, /* 7 */
	0x3, /* 8 */
	0x3, /* 9 */
	0x4, /* 10 */
	0x4, /* 11 */
	0x5, /* 12 */
	0x5, /* 13 */
	0x6, /* 14 */
	0x6, /* 15 */
	0x7, /* 16 */
	0x7, /* 17 */
	0x8, /* 18 */
	0x8, /* 19 */
	0x9, /* 20 */
	0x9, /* 21 */
	0xA, /* 22 */
	0xA, /* 23 */
	0xB, /* 24 */
	0xB, /* 25 */
	0xC, /* 26 */
	0xC, /* 27 */
	0xD, /* 28 */
	0xD, /* 29 */
	0xE, /* 30 */
	0xE  /* 31 */
};

static const UINT32 CopyOffsetBaseLUT[32] = {
	0x1,    /* 0 */
	0x2,    /* 1 */
	0x3,    /* 2 */
	0x4,    /* 3 */
	0x5,    /* 4 */
	0x7,    /* 5 */
	0x9,    /* 6 */
	0xD,    /* 7 */
	0x11,   /* 8 */
	0x19,   /* 9 */
	0x21,   /* 10 */
	0x31,   /* 11 */
	0x41,   /* 12 */
	0x61,   /* 13 */
	0x81,   /* 14 */
	0xC1,   /* 15 */
	0x101,  /* 16 */
	0x181,  /* 17 */
	0x201,  /* 18 */
	0x301,  /* 19 */
	0x401,  /* 20 */
	0x601,  /* 21 */
	0x801,  /* 22 */
	0xC01,  /* 23 */
	0x1001, /* 24 */
	0x1801, /* 25 */
	0x2001, /* 26 */
	0x3001, /* 27 */
	0x4001, /* 28 */
	0x6001, /* 29 */
	0x8001, /* 30 */
	0xC001  /* 31 */
};

static const UINT32 LOMBitsLUT[30] = {
	0x0, /* 0 */
	0x0, /* 1 */
	0x0, /* 2 */
	0x0, /* 3 */
	0x0, /* 4 */
	0x0, /* 5 */
	0x0, /* 6 */
	0x0, /* 7 */
	0x1, /* 8 */
	0x1, /* 9 */
	0x1, /* 10 */
	0x1, /* 11 */
	0x2, /* 12 */
	0x2, /* 13 */
	0x2, /* 14 */
	0x2, /* 15 */
	0x3, /* 16 */
	0x3, /* 17 */
	0x3, /* 18 */
	0x3, /* 19 */
	0x4, /* 20 */
	0x4, /* 21 */
	0x4, /* 22 */
	0x4, /* 23 */
	0x6, /* 24 */
	0x6, /* 25 */
	0x8, /* 26 */
	0x8, /* 27 */
	0xE, /* 28 */
	0xE  /* 29 */
};

static const UINT32 LOMBaseLUT[30] = {
	0x2,   /* 0 */
	0x3,   /* 1 */
	0x4,   /* 2 */
	0x5,   /* 3 */
	0x6,   /* 4 */
	0x7,   /* 5 */
	0x8,   /* 6 */
	0x9,   /* 7 */
	0xA,   /* 8 */
	0xC,   /* 9 */
	0xE,   /* 10 */
	0x10,  /* 11 */
	0x12,  /* 12 */
	0x16,  /* 13 */
	0x1A,  /* 14 */
	0x1E,  /* 15 */
	0x22,  /* 16 */
	0x2A,  /* 17 */
	0x32,  /* 18 */
	0x3A,  /* 19 */
	0x42,  /* 20 */
	0x52,  /* 21 */
	0x62,  /* 22 */
	0x72,  /* 23 */
	0x82,  /* 24 */
	0xC2,  /* 25 */
	0x102, /* 26 */
	0x202, /* 27 */
	0x2,   /* 28 */
	0x2    /* 29 */
};

static INLINE UINT16 get_word(const BYTE* data)
{
	UINT16 tmp = 0;

	WINPR_ASSERT(data);
	tmp = *data++;
	tmp |= *data << 8;
	return tmp;
}

static INLINE UINT32 get_dword(const BYTE* data)
{
	UINT32 tmp = 0;
	WINPR_ASSERT(data);
	tmp = *data++;
	tmp |= (UINT32)*data++ << 8U;
	tmp |= (UINT32)*data++ << 16U;
	tmp |= (UINT32)*data++ << 24U;
	return tmp;
}

static INLINE BOOL NCrushFetchBits(const BYTE** SrcPtr, const BYTE** SrcEnd, INT32* nbits,
                                   UINT32* bits)
{
	WINPR_ASSERT(SrcPtr);
	WINPR_ASSERT(SrcEnd);
	WINPR_ASSERT(nbits);
	WINPR_ASSERT(bits);

	if (*nbits < 16)
	{
		if ((*SrcPtr + 1) >= *SrcEnd)
		{
			if (*SrcPtr >= *SrcEnd)
			{
				if (*nbits < 0)
					return FALSE;
			}
			else
			{
				const UINT32 src = *(*SrcPtr)++;
				const UINT32 val = (src << *nbits);
				*bits += val;
				*nbits += 8;
			}
		}
		else
		{
			UINT32 tmp = *(*SrcPtr)++;
			tmp |= ((*(*SrcPtr)++) << 8) & 0xFF00;
			*bits += (tmp << *nbits) & UINT32_MAX;
			*nbits += 16;
		}
	}

	return TRUE;
}

static INLINE void NCrushWriteStart(UINT32* bits, UINT32* offset, UINT32* accumulator)
{
	WINPR_ASSERT(bits);
	WINPR_ASSERT(offset);
	WINPR_ASSERT(accumulator);

	*bits = 0;
	*offset = 0;
	*accumulator = 0;
}

static INLINE void NCrushWriteBits(BYTE** DstPtr, UINT32* accumulator, UINT32* offset, UINT32 _bits,
                                   UINT32 _nbits)
{
	WINPR_ASSERT(DstPtr);
	WINPR_ASSERT(accumulator);
	WINPR_ASSERT(offset);

	*accumulator |= _bits << *offset;
	*offset += _nbits;

	if (*offset >= 16)
	{
		*(*DstPtr)++ = (*accumulator & 0xFF);
		*(*DstPtr)++ = ((*accumulator >> 8) & 0xFF);
		*accumulator >>= 16;
		*offset -= 16;
	}
}

static INLINE void NCrushWriteFinish(BYTE** DstPtr, UINT32 accumulator)
{
	WINPR_ASSERT(DstPtr);

	*(*DstPtr)++ = accumulator & 0xFF;
	*(*DstPtr)++ = (accumulator >> 8) & 0xFF;
}

int ncrush_decompress(NCRUSH_CONTEXT* ncrush, const BYTE* pSrcData, UINT32 SrcSize,
                      const BYTE** ppDstData, UINT32* pDstSize, UINT32 flags)
{
	UINT32 index = 0;
	BYTE Literal = 0;
	UINT32 IndexLEC = 0;
	UINT32 BitLength = 0;
	UINT32 CopyOffset = 0;
	UINT32 CopyLength = 0;
	UINT32 OldCopyOffset = 0;
	const BYTE* CopyOffsetPtr = NULL;
	UINT32 LengthOfMatch = 0;
	UINT32 CopyOffsetIndex = 0;
	UINT32 OffsetCacheIndex = 0;
	UINT32 CopyOffsetBits = 0;
	UINT32 CopyOffsetBase = 0;
	UINT32 LengthOfMatchBits = 0;
	UINT32 LengthOfMatchBase = 0;

	WINPR_ASSERT(ncrush);
	WINPR_ASSERT(pSrcData);
	WINPR_ASSERT(ppDstData);
	WINPR_ASSERT(pDstSize);

	if (ncrush->HistoryEndOffset != 65535)
		return -1001;

	BYTE* HistoryBuffer = ncrush->HistoryBuffer;
	const BYTE* HistoryBufferEnd = &HistoryBuffer[ncrush->HistoryEndOffset];

	if (flags & PACKET_AT_FRONT)
	{
		if ((ncrush->HistoryPtr - 32768) <= HistoryBuffer)
			return -1002;

		MoveMemory(HistoryBuffer, (ncrush->HistoryPtr - 32768), 32768);
		ncrush->HistoryPtr = &(HistoryBuffer[32768]);
		ZeroMemory(&HistoryBuffer[32768], 32768);
	}

	if (flags & PACKET_FLUSHED)
	{
		ncrush->HistoryPtr = HistoryBuffer;
		ZeroMemory(HistoryBuffer, sizeof(ncrush->HistoryBuffer));
		ZeroMemory(&(ncrush->OffsetCache), sizeof(ncrush->OffsetCache));
	}

	BYTE* HistoryPtr = ncrush->HistoryPtr;

	if (!(flags & PACKET_COMPRESSED))
	{
		*ppDstData = pSrcData;
		*pDstSize = SrcSize;
		return 1;
	}

	if (SrcSize < 4)
	{
		WLog_ERR(TAG, "Input size short: SrcSize %" PRIu32 " < 4", SrcSize);
		return -1;
	}

	const BYTE* SrcEnd = &pSrcData[SrcSize];
	const BYTE* SrcPtr = pSrcData + 4;

	INT32 nbits = 32;
	UINT32 bits = get_dword(pSrcData);
	while (1)
	{
		while (1)
		{
			const UINT16 Mask = get_word(&HuffTableMask[29]);
			const UINT32 MaskedBits = bits & Mask;
			if (MaskedBits >= ARRAYSIZE(HuffTableLEC))
				return -1;
			IndexLEC = HuffTableLEC[MaskedBits] & 0xFFF;
			BitLength = HuffTableLEC[MaskedBits] >> 12;
			bits >>= BitLength;
			nbits -= WINPR_ASSERTING_INT_CAST(int32_t, BitLength);

			if (!NCrushFetchBits(&SrcPtr, &SrcEnd, &nbits, &bits))
				return -1;

			if (IndexLEC >= 256)
				break;

			if (HistoryPtr >= HistoryBufferEnd)
			{
				WLog_ERR(TAG, "ncrush_decompress error: HistoryPtr (%p) >= HistoryBufferEnd (%p)",
				         (const void*)HistoryPtr, (const void*)HistoryBufferEnd);
				return -1003;
			}

			Literal = (HuffTableLEC[MaskedBits] & 0xFF);
			*HistoryPtr++ = Literal;
		}

		if (IndexLEC == 256)
			break; /* EOS */

		CopyOffsetIndex = IndexLEC - 257;

		if (CopyOffsetIndex >= 32)
		{
			OffsetCacheIndex = IndexLEC - 289;

			if (OffsetCacheIndex >= 4)
				return -1004;

			{
				CopyOffset = ncrush->OffsetCache[OffsetCacheIndex];
				const UINT16 Mask = get_word(&HuffTableMask[21]);
				const UINT32 MaskedBits = bits & Mask;
				if (MaskedBits >= ARRAYSIZE(HuffTableLOM))
					return -1;
				LengthOfMatch = HuffTableLOM[MaskedBits] & 0xFFF;
				BitLength = HuffTableLOM[MaskedBits] >> 12;
				bits >>= BitLength;
				nbits -= WINPR_ASSERTING_INT_CAST(int32_t, BitLength);
			}

			if (!NCrushFetchBits(&SrcPtr, &SrcEnd, &nbits, &bits))
				return -1;

			if (LengthOfMatch >= ARRAYSIZE(LOMBitsLUT))
				return -1;

			LengthOfMatchBits = LOMBitsLUT[LengthOfMatch];

			if (LengthOfMatch >= ARRAYSIZE(LOMBaseLUT))
				return -1;
			LengthOfMatchBase = LOMBaseLUT[LengthOfMatch];

			if (LengthOfMatchBits)
			{
				const size_t idx = (2ull * LengthOfMatchBits) + 3ull;
				if (idx >= ARRAYSIZE(HuffTableMask))
					return -1;

				const UINT16 Mask = get_word(&HuffTableMask[idx]);
				const UINT32 MaskedBits = bits & Mask;
				bits >>= LengthOfMatchBits;
				nbits -= WINPR_ASSERTING_INT_CAST(int32_t, LengthOfMatchBits);
				LengthOfMatchBase += MaskedBits;

				if (!NCrushFetchBits(&SrcPtr, &SrcEnd, &nbits, &bits))
					return -1;
			}

			OldCopyOffset = ncrush->OffsetCache[OffsetCacheIndex];
			ncrush->OffsetCache[OffsetCacheIndex] = ncrush->OffsetCache[0];
			ncrush->OffsetCache[0] = OldCopyOffset;
		}
		else
		{
			if (CopyOffsetIndex >= ARRAYSIZE(CopyOffsetBitsLUT))
				return -1;

			CopyOffsetBits = CopyOffsetBitsLUT[CopyOffsetIndex];

			if (CopyOffsetIndex >= ARRAYSIZE(CopyOffsetBaseLUT))
				return -1;
			CopyOffsetBase = CopyOffsetBaseLUT[CopyOffsetIndex];
			CopyOffset = CopyOffsetBase - 1;

			if (CopyOffsetBits)
			{
				const size_t idx = (2ull * CopyOffsetBits) + 3ull;
				if (idx >= ARRAYSIZE(HuffTableMask))
					return -1;

				{
					const UINT16 Mask = get_word(&HuffTableMask[idx]);
					const UINT32 MaskedBits = bits & Mask;
					const UINT32 tmp = CopyOffsetBase + MaskedBits;
					if (tmp < 1)
						return -1;
					CopyOffset = tmp - 1;
				}
				bits >>= CopyOffsetBits;
				nbits -= WINPR_ASSERTING_INT_CAST(int32_t, CopyOffsetBits);

				if (!NCrushFetchBits(&SrcPtr, &SrcEnd, &nbits, &bits))
					return -1;
			}
			{
				const UINT16 Mask = get_word(&HuffTableMask[21]);
				const UINT32 MaskedBits = bits & Mask;
				if (MaskedBits >= ARRAYSIZE(HuffTableLOM))
					return -1;

				LengthOfMatch = HuffTableLOM[MaskedBits] & 0xFFF;
				BitLength = HuffTableLOM[MaskedBits] >> 12;
				bits >>= BitLength;
				nbits -= WINPR_ASSERTING_INT_CAST(int32_t, BitLength);
			}
			if (!NCrushFetchBits(&SrcPtr, &SrcEnd, &nbits, &bits))
				return -1;

			if (LengthOfMatch >= ARRAYSIZE(LOMBitsLUT))
				return -1;

			LengthOfMatchBits = LOMBitsLUT[LengthOfMatch];

			if (LengthOfMatch >= ARRAYSIZE(LOMBaseLUT))
				return -1;
			LengthOfMatchBase = LOMBaseLUT[LengthOfMatch];

			if (LengthOfMatchBits)
			{
				const size_t idx = (2ull * LengthOfMatchBits) + 3ull;
				if (idx >= ARRAYSIZE(HuffTableMask))
					return -1;

				const UINT16 Mask = get_word(&HuffTableMask[idx]);
				const UINT32 MaskedBits = bits & Mask;
				bits >>= LengthOfMatchBits;
				nbits -= WINPR_ASSERTING_INT_CAST(int32_t, LengthOfMatchBits);
				LengthOfMatchBase += MaskedBits;

				if (!NCrushFetchBits(&SrcPtr, &SrcEnd, &nbits, &bits))
					return -1;
			}

			ncrush->OffsetCache[3] = ncrush->OffsetCache[2];
			ncrush->OffsetCache[2] = ncrush->OffsetCache[1];
			ncrush->OffsetCache[1] = ncrush->OffsetCache[0];
			ncrush->OffsetCache[0] = CopyOffset;
		}

		CopyOffsetPtr = &HistoryBuffer[(HistoryPtr - HistoryBuffer - CopyOffset) & 0xFFFF];
		LengthOfMatch = LengthOfMatchBase;

		if (LengthOfMatch < 2)
			return -1005;

		if ((CopyOffsetPtr >= (HistoryBufferEnd - LengthOfMatch)) ||
		    (HistoryPtr >= (HistoryBufferEnd - LengthOfMatch)))
			return -1006;

		CopyOffsetPtr = HistoryPtr - CopyOffset;
		index = 0;
		CopyLength = (LengthOfMatch > CopyOffset) ? CopyOffset : LengthOfMatch;

		if (CopyOffsetPtr >= HistoryBuffer)
		{
			while (CopyLength > 0)
			{
				*HistoryPtr++ = *CopyOffsetPtr++;
				CopyLength--;
			}

			while (LengthOfMatch > CopyOffset)
			{
				index = ((index >= CopyOffset)) ? 0 : index;
				*HistoryPtr++ = *(CopyOffsetPtr + index++);
				LengthOfMatch--;
			}
		}
		else
		{
			CopyOffsetPtr = HistoryBufferEnd - (CopyOffset - (HistoryPtr - HistoryBuffer));
			CopyOffsetPtr++;

			while (CopyLength && (CopyOffsetPtr <= HistoryBufferEnd))
			{
				*HistoryPtr++ = *CopyOffsetPtr++;
				CopyLength--;
			}

			CopyOffsetPtr = HistoryBuffer;

			while (LengthOfMatch > CopyOffset)
			{
				index = ((index >= CopyOffset)) ? 0 : index;
				*HistoryPtr++ = *(CopyOffsetPtr + index++);
				LengthOfMatch--;
			}
		}

		LengthOfMatch = LengthOfMatchBase;

		if (LengthOfMatch == 2)
			continue;
	}

	if (IndexLEC != 256)
		return -1;

	if (ncrush->HistoryBufferFence != 0xABABABAB)
	{
		WLog_ERR(TAG, "NCrushDecompress: history buffer fence was overwritten, potential buffer "
		              "overflow detected!");
		return -1007;
	}

	const intptr_t hsize = HistoryPtr - ncrush->HistoryPtr;
	WINPR_ASSERT(hsize >= 0);
	WINPR_ASSERT(hsize <= UINT32_MAX);
	*pDstSize = (UINT32)hsize;
	*ppDstData = ncrush->HistoryPtr;
	ncrush->HistoryPtr = HistoryPtr;
	return 1;
}

static int ncrush_hash_table_add(NCRUSH_CONTEXT* ncrush, const BYTE* pSrcData, UINT32 SrcSize,
                                 UINT32 HistoryOffset)
{
	const BYTE* SrcPtr = pSrcData;
	UINT32 Offset = HistoryOffset;
	UINT32 EndOffset = Offset + SrcSize - 8;

	WINPR_ASSERT(ncrush);
	WINPR_ASSERT(pSrcData);
	WINPR_ASSERT(Offset + SrcSize >= 8);

	while (Offset < EndOffset)
	{
		const UINT16 Hash = ncrush->HashTable[get_word(SrcPtr)];
		ncrush->HashTable[get_word(SrcPtr)] = WINPR_ASSERTING_INT_CAST(UINT16, Offset);
		ncrush->MatchTable[Offset] = Hash;
		SrcPtr++;
		Offset++;
	}

	return 1;
}

static intptr_t ncrush_find_match_length(const BYTE* Ptr1, const BYTE* Ptr2, const BYTE* HistoryPtr)
{
	BYTE val1 = 0;
	BYTE val2 = 0;
	const BYTE* Ptr = Ptr1;

	WINPR_ASSERT(Ptr1);
	WINPR_ASSERT(Ptr2);
	WINPR_ASSERT(HistoryPtr);

	do
	{
		if (Ptr1 > HistoryPtr)
			break;

		val1 = *Ptr1++;
		val2 = *Ptr2++;
	} while (val1 == val2);

	return Ptr1 - (Ptr + 1);
}

static int ncrush_find_best_match(NCRUSH_CONTEXT* ncrush, UINT16 HistoryOffset,
                                  UINT32* pMatchOffset)
{
	int Length = 0;

	WINPR_ASSERT(ncrush);
	WINPR_ASSERT(pMatchOffset);

	if (!ncrush->MatchTable[HistoryOffset])
		return -1;

	int MatchLength = 2;
	UINT16 Offset = HistoryOffset;
	BYTE* HistoryBuffer = (BYTE*)ncrush->HistoryBuffer;

	ncrush->MatchTable[0] = HistoryOffset;
	UINT16 MatchOffset = ncrush->MatchTable[HistoryOffset];
	UINT16 NextOffset = ncrush->MatchTable[Offset];
	BYTE* MatchPtr = &HistoryBuffer[MatchLength];

	for (int i = 0; i < 4; i++)
	{
		int j = -1;

		if (j < 0)
		{
			Offset = ncrush->MatchTable[NextOffset];

			if (MatchPtr[NextOffset] == HistoryBuffer[HistoryOffset + MatchLength])
				j = 0;
		}

		if (j < 0)
		{
			NextOffset = ncrush->MatchTable[Offset];

			if (MatchPtr[Offset] == HistoryBuffer[HistoryOffset + MatchLength])
				j = 1;
		}

		if (j < 0)
		{
			Offset = ncrush->MatchTable[NextOffset];

			if (MatchPtr[NextOffset] == HistoryBuffer[HistoryOffset + MatchLength])
				j = 2;
		}

		if (j < 0)
		{
			NextOffset = ncrush->MatchTable[Offset];

			if (MatchPtr[Offset] == HistoryBuffer[HistoryOffset + MatchLength])
				j = 3;
		}

		if (j < 0)
		{
			Offset = ncrush->MatchTable[NextOffset];

			if (MatchPtr[NextOffset] == HistoryBuffer[HistoryOffset + MatchLength])
				j = 4;
		}

		if (j < 0)
		{
			NextOffset = ncrush->MatchTable[Offset];

			if (MatchPtr[Offset] == HistoryBuffer[HistoryOffset + MatchLength])
				j = 5;
		}

		if (j >= 0)
		{
			if ((j % 2) == 0)
				Offset = NextOffset;

			if ((Offset != HistoryOffset) && Offset)
			{
				const intptr_t len =
				    ncrush_find_match_length(&HistoryBuffer[HistoryOffset + 2],
				                             &HistoryBuffer[Offset + 2], ncrush->HistoryPtr);
				WINPR_ASSERT(len < INT_MAX - 2);
				WINPR_ASSERT(len > INT_MIN + 2);

				Length = (int)len + 2;

				if (Length < 2)
					return -1;

				if (Length > 16)
					break;

				if (Length > MatchLength)
				{
					MatchLength = Length;
					MatchOffset = Offset;
				}

				if ((Length <= MatchLength) ||
				    (&HistoryBuffer[HistoryOffset + 2] < ncrush->HistoryPtr))
				{
					NextOffset = ncrush->MatchTable[Offset];
					MatchPtr = &HistoryBuffer[MatchLength];
					continue;
				}
			}

			break;
		}
	}

	ncrush->MatchTable[0] = 0;
	*pMatchOffset = MatchOffset;
	return MatchLength;
}

static int ncrush_move_encoder_windows(NCRUSH_CONTEXT* ncrush, BYTE* HistoryPtr)
{
	WINPR_ASSERT(ncrush);
	WINPR_ASSERT(HistoryPtr);

	const size_t history_half = ARRAYSIZE(ncrush->HistoryBuffer) / 2;
	if (HistoryPtr < &ncrush->HistoryBuffer[history_half])
		return -1;

	if (HistoryPtr > &ncrush->HistoryBuffer[ARRAYSIZE(ncrush->HistoryBuffer)])
		return -1;

	MoveMemory(ncrush->HistoryBuffer, HistoryPtr - history_half, history_half * sizeof(BYTE));
	const intptr_t hsize = HistoryPtr - history_half - ncrush->HistoryBuffer;
	WINPR_ASSERT(hsize <= UINT16_MAX);
	WINPR_ASSERT(hsize >= 0);
	INT32 HistoryOffset = (INT32)hsize;

	for (size_t i = 0; i < ARRAYSIZE(ncrush->HashTable); i++)
	{
		const INT32 NewHash = ncrush->HashTable[i] - HistoryOffset;
		ncrush->HashTable[i] = (NewHash <= 0) ? 0 : WINPR_ASSERTING_INT_CAST(UINT16, NewHash);
	}

	const size_t match_half = ARRAYSIZE(ncrush->MatchTable) / 2;
	for (size_t j = 0; j < match_half; j++)
	{
		if ((size_t)HistoryOffset + j >= ARRAYSIZE(ncrush->MatchTable))
			continue;

		const INT32 NewMatch = ncrush->MatchTable[(size_t)HistoryOffset + j] - HistoryOffset;
		ncrush->MatchTable[j] = (NewMatch <= 0) ? 0 : WINPR_ASSERTING_INT_CAST(UINT16, NewMatch);
	}

	ZeroMemory(&ncrush->MatchTable[match_half], match_half * sizeof(UINT16));
	return 1;
}

int ncrush_compress(NCRUSH_CONTEXT* ncrush, const BYTE* pSrcData, UINT32 SrcSize, BYTE* pDstBuffer,
                    const BYTE** ppDstData, UINT32* pDstSize, UINT32* pFlags)
{
	BYTE Literal = 0;
	const BYTE* SrcPtr = NULL;
	BYTE* DstPtr = NULL;
	UINT32 bits = 0;
	UINT32 offset = 0;
	UINT32 MaskedBits = 0;
	UINT32 accumulator = 0;
	const BYTE* SrcEndPtr = NULL;
	BYTE* DstEndPtr = NULL;
	BYTE* HistoryPtr = NULL;
	BYTE* pDstData = NULL;
	UINT32 DstSize = 0;
	BOOL PacketAtFront = FALSE;
	BOOL PacketFlushed = FALSE;
	UINT32 MatchLength = 0;
	UINT32 IndexLEC = 0;
	UINT32 IndexLOM = 0;
	UINT32 IndexCO = 0;
	UINT32 CodeLEC = 0;
	UINT32 BitLength = 0;
	UINT32 CopyOffset = 0;
	UINT32 MatchOffset = 0;
	UINT32 OldCopyOffset = 0;
	UINT32* OffsetCache = NULL;
	UINT32 OffsetCacheIndex = 0;
	UINT32 HistoryOffset = 0;
	BYTE* HistoryBuffer = NULL;
	UINT32 HistoryBufferSize = 0;
	BYTE* HistoryBufferEndPtr = NULL;
	UINT32 CopyOffsetIndex = 0;
	UINT32 CopyOffsetBits = 0;
	UINT32 CompressionLevel = 2;

	WINPR_ASSERT(ncrush);

	WINPR_ASSERT(ncrush);
	WINPR_ASSERT(pSrcData);
	WINPR_ASSERT(pDstBuffer);
	WINPR_ASSERT(ppDstData);
	WINPR_ASSERT(pDstSize);
	WINPR_ASSERT(pFlags);

	HistoryBuffer = ncrush->HistoryBuffer;
	*pFlags = 0;

	if ((SrcSize + ncrush->HistoryOffset) >= 65529)
	{
		if (ncrush->HistoryOffset == (ncrush->HistoryBufferSize + 1))
		{
			ncrush->HistoryOffset = 0;
			ncrush->HistoryPtr = HistoryBuffer;
			PacketFlushed = TRUE;
		}
		else
		{
			if (ncrush_move_encoder_windows(ncrush, &(HistoryBuffer[ncrush->HistoryOffset])) < 0)
				return -1001;

			ncrush->HistoryPtr = &HistoryBuffer[32768];
			ncrush->HistoryOffset = 32768;
			PacketAtFront = TRUE;
		}
	}
	else
	{
		*pFlags = 0;
	}

	pDstData = pDstBuffer;
	*ppDstData = pDstBuffer;

	if (!pDstData)
		return -1002;

	DstSize = *pDstSize;

	if (DstSize < SrcSize)
		return -1003;

	DstSize = SrcSize;
	NCrushWriteStart(&bits, &offset, &accumulator);
	DstPtr = pDstData;
	SrcPtr = pSrcData;
	SrcEndPtr = &pSrcData[SrcSize];
	DstEndPtr = &pDstData[DstSize - 1];
	OffsetCache = ncrush->OffsetCache;
	HistoryPtr = &HistoryBuffer[ncrush->HistoryOffset];
	HistoryBufferEndPtr = &HistoryBuffer[65536];
	HistoryBufferSize = ncrush->HistoryBufferSize;
	CopyOffset = 0;
	MatchOffset = 0;
	const intptr_t thsize = HistoryPtr - HistoryBuffer;

	ncrush_hash_table_add(ncrush, pSrcData, SrcSize, WINPR_ASSERTING_INT_CAST(UINT32, thsize));
	CopyMemory(HistoryPtr, pSrcData, SrcSize);
	ncrush->HistoryPtr = &HistoryPtr[SrcSize];

	while (SrcPtr < (SrcEndPtr - 2))
	{
		MatchLength = 0;
		const intptr_t hsize = HistoryPtr - HistoryBuffer;
		WINPR_ASSERT(hsize <= UINT32_MAX);
		WINPR_ASSERT(hsize >= 0);
		HistoryOffset = (UINT32)hsize;

		if (ncrush->HistoryPtr && (HistoryPtr > ncrush->HistoryPtr))
			return -1;

		if (HistoryOffset >= 65536)
			return -1004;

		if (ncrush->MatchTable[HistoryOffset])
		{
			int rc = 0;

			MatchOffset = 0;
			rc = ncrush_find_best_match(ncrush, WINPR_ASSERTING_INT_CAST(UINT16, HistoryOffset),
			                            &MatchOffset);

			if (rc < 0)
				return -1005;
			MatchLength = (UINT32)rc;
		}

		if (MatchLength)
			CopyOffset = (HistoryBufferSize - 1) & (HistoryPtr - &HistoryBuffer[MatchOffset]);

		if ((MatchLength == 2) && (CopyOffset >= 64))
			MatchLength = 0;

		if (MatchLength == 0)
		{
			/* Literal */
			Literal = *SrcPtr++;
			HistoryPtr++;

			if ((DstPtr + 2) > DstEndPtr) /* PACKET_FLUSH #1 */
			{
				ncrush_context_reset(ncrush, TRUE);
				*pFlags = PACKET_FLUSHED;
				*pFlags |= CompressionLevel;
				*ppDstData = pSrcData;
				*pDstSize = SrcSize;
				return 1;
			}

			IndexLEC = Literal;
			if (IndexLEC >= ARRAYSIZE(HuffLengthLEC))
				return -1;
			BitLength = HuffLengthLEC[IndexLEC];

			if (IndexLEC * 2ull >= ARRAYSIZE(HuffCodeLEC))
				return -1;
			CodeLEC = get_word(&HuffCodeLEC[2ULL * IndexLEC]);

			if (BitLength > 15)
				return -1006;

			NCrushWriteBits(&DstPtr, &accumulator, &offset, CodeLEC, BitLength);
		}
		else
		{
			HistoryPtr += MatchLength;
			SrcPtr += MatchLength;

			if (!MatchLength)
				return -1007;

			if ((DstPtr + 8) > DstEndPtr) /* PACKET_FLUSH #2 */
			{
				ncrush_context_reset(ncrush, TRUE);
				*pFlags = PACKET_FLUSHED;
				*pFlags |= CompressionLevel;
				*ppDstData = pSrcData;
				*pDstSize = SrcSize;
				return 1;
			}

			OffsetCacheIndex = 5;

			if ((CopyOffset == OffsetCache[0]) || (CopyOffset == OffsetCache[1]) ||
			    (CopyOffset == OffsetCache[2]) || (CopyOffset == OffsetCache[3]))
			{
				if (CopyOffset == OffsetCache[3])
				{
					OldCopyOffset = OffsetCache[3];
					OffsetCache[3] = OffsetCache[0];
					OffsetCache[0] = OldCopyOffset;
					OffsetCacheIndex = 3;
				}
				else
				{
					if (CopyOffset == OffsetCache[2])
					{
						OldCopyOffset = OffsetCache[2];
						OffsetCache[2] = OffsetCache[0];
						OffsetCache[0] = OldCopyOffset;
						OffsetCacheIndex = 2;
					}
					else
					{
						if (CopyOffset == OffsetCache[1])
						{
							OldCopyOffset = OffsetCache[1];
							OffsetCache[1] = OffsetCache[0];
							OffsetCache[0] = OldCopyOffset;
							OffsetCacheIndex = 1;
						}
						else
						{
							if (CopyOffset == OffsetCache[0])
							{
								OffsetCacheIndex = 0;
							}
						}
					}
				}
			}
			else
			{
				OffsetCache[3] = OffsetCache[2];
				OffsetCache[2] = OffsetCache[1];
				OffsetCache[1] = OffsetCache[0];
				OffsetCache[0] = CopyOffset;
			}

			if (OffsetCacheIndex >= 4)
			{
				/* CopyOffset not in OffsetCache */
				if (CopyOffset >= 256)
					bits = (CopyOffset >> 7) + 256;
				else
					bits = CopyOffset;

				CopyOffsetIndex = ncrush->HuffTableCopyOffset[bits + 2];

				if (CopyOffsetIndex >= ARRAYSIZE(CopyOffsetBitsLUT))
					return -1;

				CopyOffsetBits = CopyOffsetBitsLUT[CopyOffsetIndex];
				IndexLEC = 257 + CopyOffsetIndex;
				if (IndexLEC >= ARRAYSIZE(HuffLengthLEC))
					return -1;
				BitLength = HuffLengthLEC[IndexLEC];

				if (IndexLEC * 2ull >= ARRAYSIZE(HuffCodeLEC))
					return -1;
				CodeLEC = get_word(&HuffCodeLEC[2ULL * IndexLEC]);

				if (BitLength > 15)
					return -1008;

				if (CopyOffsetBits > 18)
					return -1009;

				NCrushWriteBits(&DstPtr, &accumulator, &offset, CodeLEC, BitLength);
				UINT16 Mask = WINPR_ASSERTING_INT_CAST(UINT16, (1 << CopyOffsetBits) - 1);
				MaskedBits = CopyOffset & Mask;
				NCrushWriteBits(&DstPtr, &accumulator, &offset, MaskedBits, CopyOffsetBits);

				if ((MatchLength - 2) >= 768)
					IndexCO = 28;
				else
					IndexCO = ncrush->HuffTableLOM[MatchLength];

				if (IndexCO >= ARRAYSIZE(HuffLengthLOM))
					return -1;
				BitLength = HuffLengthLOM[IndexCO];

				if (IndexCO >= ARRAYSIZE(LOMBitsLUT))
					return -1;
				IndexLOM = LOMBitsLUT[IndexCO];

				if (IndexCO >= ARRAYSIZE(HuffCodeLOM))
					return -1;
				NCrushWriteBits(&DstPtr, &accumulator, &offset, HuffCodeLOM[IndexCO], BitLength);
				Mask = WINPR_ASSERTING_INT_CAST(UINT16, (1 << IndexLOM) - 1);
				MaskedBits = (MatchLength - 2) & Mask;
				NCrushWriteBits(&DstPtr, &accumulator, &offset, MaskedBits, IndexLOM);

				if (IndexCO >= ARRAYSIZE(LOMBaseLUT))
					return -1;
				if ((MaskedBits + LOMBaseLUT[IndexCO]) != MatchLength)
					return -1010;
			}
			else
			{
				/* CopyOffset in OffsetCache */
				IndexLEC = 289 + OffsetCacheIndex;
				if (IndexLEC >= ARRAYSIZE(HuffLengthLEC))
					return -1;
				BitLength = HuffLengthLEC[IndexLEC];
				if (IndexLEC * 2ull >= ARRAYSIZE(HuffCodeLEC))
					return -1;
				CodeLEC = get_word(&HuffCodeLEC[2ULL * IndexLEC]);

				if (BitLength >= 15)
					return -1011;

				NCrushWriteBits(&DstPtr, &accumulator, &offset, CodeLEC, BitLength);

				if ((MatchLength - 2) >= 768)
					IndexCO = 28;
				else
					IndexCO = ncrush->HuffTableLOM[MatchLength];

				if (IndexCO >= ARRAYSIZE(HuffLengthLOM))
					return -1;

				BitLength = HuffLengthLOM[IndexCO];

				if (IndexCO >= ARRAYSIZE(LOMBitsLUT))
					return -1;
				IndexLOM = LOMBitsLUT[IndexCO];

				if (IndexCO >= ARRAYSIZE(HuffCodeLOM))
					return -1;
				NCrushWriteBits(&DstPtr, &accumulator, &offset, HuffCodeLOM[IndexCO], BitLength);
				const UINT16 Mask = WINPR_ASSERTING_INT_CAST(UINT16, (1 << IndexLOM) - 1);
				MaskedBits = (MatchLength - 2) & Mask;
				NCrushWriteBits(&DstPtr, &accumulator, &offset, MaskedBits, IndexLOM);

				if (IndexCO >= ARRAYSIZE(LOMBaseLUT))
					return -1;
				if ((MaskedBits + LOMBaseLUT[IndexCO]) != MatchLength)
					return -1012;
			}
		}

		if (HistoryPtr >= HistoryBufferEndPtr)
			return -1013;
	}

	while (SrcPtr < SrcEndPtr)
	{
		if ((DstPtr + 2) > DstEndPtr) /* PACKET_FLUSH #3 */
		{
			ncrush_context_reset(ncrush, TRUE);
			*pFlags = PACKET_FLUSHED;
			*pFlags |= CompressionLevel;
			*ppDstData = pSrcData;
			*pDstSize = SrcSize;
			return 1;
		}

		Literal = *SrcPtr++;
		HistoryPtr++;
		IndexLEC = Literal;
		if (IndexLEC >= ARRAYSIZE(HuffLengthLEC))
			return -1;
		if (IndexLEC * 2ull >= ARRAYSIZE(HuffCodeLEC))
			return -1;
		BitLength = HuffLengthLEC[IndexLEC];
		CodeLEC = get_word(&HuffCodeLEC[2ULL * IndexLEC]);

		if (BitLength > 15)
			return -1014;

		NCrushWriteBits(&DstPtr, &accumulator, &offset, CodeLEC, BitLength);
	}

	if ((DstPtr + 4) >= DstEndPtr) /* PACKET_FLUSH #4 */
	{
		ncrush_context_reset(ncrush, TRUE);
		*pFlags = PACKET_FLUSHED;
		*pFlags |= CompressionLevel;
		*ppDstData = pSrcData;
		*pDstSize = SrcSize;
		return 1;
	}

	IndexLEC = 256;
	BitLength = HuffLengthLEC[IndexLEC];

	if (BitLength > 15)
		return -1015;

	bits = get_word(&HuffCodeLEC[2ULL * IndexLEC]);
	NCrushWriteBits(&DstPtr, &accumulator, &offset, bits, BitLength);
	NCrushWriteFinish(&DstPtr, accumulator);
	const intptr_t dsize = DstPtr - pDstData;
	WINPR_ASSERT(dsize <= UINT32_MAX);
	WINPR_ASSERT(dsize >= 0);
	*pDstSize = (UINT32)dsize;

	if (*pDstSize > SrcSize)
		return -1016;

	*pFlags |= PACKET_COMPRESSED;
	*pFlags |= CompressionLevel;

	if (PacketAtFront)
		*pFlags |= PACKET_AT_FRONT;

	if (PacketFlushed)
		*pFlags |= PACKET_FLUSHED;

	const intptr_t diff = HistoryPtr - HistoryBuffer;
	if (diff > UINT32_MAX)
		return -1;

	ncrush->HistoryOffset = (UINT32)diff;

	if (ncrush->HistoryOffset >= ncrush->HistoryBufferSize)
		return -1;

	return 1;
}

static int ncrush_generate_tables(NCRUSH_CONTEXT* context)
{
	WINPR_ASSERT(context);
	WINPR_ASSERT(28 < ARRAYSIZE(LOMBitsLUT));

	UINT32 cnt = 0;
	for (BYTE i = 0; i < 28; i++)
	{
		for (UINT32 j = 0; j < 1 << LOMBitsLUT[i]; j++)
		{
			size_t l = (cnt++) + 2ull;
			context->HuffTableLOM[l] = i;
		}
	}

	for (UINT32 k = 2; k < 4096; k++)
	{
		size_t i = 28;
		if ((k - 2) < 768)
			i = context->HuffTableLOM[k];

		if (i >= ARRAYSIZE(LOMBitsLUT))
			return -1;
		if (i >= ARRAYSIZE(LOMBaseLUT))
			return -1;

		if (((((1 << LOMBitsLUT[i]) - 1) & (k - 2)) + LOMBaseLUT[i]) != k)
			return -1;
	}

	size_t k = 0;
	for (BYTE i = 0; i < 16; i++)
	{
		for (size_t j = 0; j < 1 << CopyOffsetBitsLUT[i]; j++)
		{
			size_t l = k++ + 2ull;
			context->HuffTableCopyOffset[l] = i;
		}
	}

	k /= 128;

	for (BYTE i = 16; i < 32; i++)
	{
		const UINT32 lut = CopyOffsetBitsLUT[i];
		WINPR_ASSERT(lut >= 7);
		for (size_t j = 0; j < 1 << (lut - 7); j++)
		{
			size_t l = k++ + 2 + 256ull;
			context->HuffTableCopyOffset[l] = i;
		}
	}

	if ((k + 256) > 1024)
		return -1;

	return 1;
}

void ncrush_context_reset(NCRUSH_CONTEXT* ncrush, BOOL flush)
{
	WINPR_ASSERT(ncrush);

	ZeroMemory(&(ncrush->HistoryBuffer), sizeof(ncrush->HistoryBuffer));
	ZeroMemory(&(ncrush->OffsetCache), sizeof(ncrush->OffsetCache));
	ZeroMemory(&(ncrush->MatchTable), sizeof(ncrush->MatchTable));
	ZeroMemory(&(ncrush->HashTable), sizeof(ncrush->HashTable));

	if (flush)
		ncrush->HistoryOffset = ncrush->HistoryBufferSize + 1;
	else
		ncrush->HistoryOffset = 0;

	ncrush->HistoryPtr = &(ncrush->HistoryBuffer[ncrush->HistoryOffset]);
}

NCRUSH_CONTEXT* ncrush_context_new(BOOL Compressor)
{
	NCRUSH_CONTEXT* ncrush = (NCRUSH_CONTEXT*)calloc(1, sizeof(NCRUSH_CONTEXT));

	if (!ncrush)
		goto fail;

	ncrush->Compressor = Compressor;
	ncrush->HistoryBufferSize = 65536;
	ncrush->HistoryEndOffset = ncrush->HistoryBufferSize - 1;
	ncrush->HistoryBufferFence = 0xABABABAB;
	ncrush->HistoryOffset = 0;
	ncrush->HistoryPtr = &(ncrush->HistoryBuffer[ncrush->HistoryOffset]);

	if (ncrush_generate_tables(ncrush) < 0)
	{
		WLog_DBG(TAG, "ncrush_context_new: failed to initialize tables");
		goto fail;
	}

	ncrush_context_reset(ncrush, FALSE);

	return ncrush;
fail:
	ncrush_context_free(ncrush);
	return NULL;
}

void ncrush_context_free(NCRUSH_CONTEXT* ncrush)
{
	free(ncrush);
}
