!--------------------------------------------------------------------------------------------------!
!  DFTB+: general package for performing fast atomistic simulations                                !
!  Copyright (C) 2006 - 2023  DFTB+ developers group                                               !
!                                                                                                  !
!  See the LICENSE file for terms of usage and distribution.                                       !
!--------------------------------------------------------------------------------------------------!

#:include 'common.fypp'

!> Necessary parameters to perform DFT-D3 calculations.
module dftbp_dftb_dftd3param
  use dftbp_common_accuracy, only : dp
  use dftbp_common_constants, only : symbolToNumber
  use dftbp_dftb_dftd4param, only : getSqrtZr4r2
  implicit none

  private
  public :: TDftD3Ref, init
  public :: getC6, initD3References, clearD3References
  public :: numberOfReferences, referenceCN


  !> Number of elements available for DFT-D3
  integer, parameter :: maxElem = 94

  !> Maximum number of references in internal storage
  integer, parameter :: maxRef = 5

  !> Block size to store the scary reference C6 array
  integer, parameter :: blockSize = 19


  !> DFT-D3 reference systems
  type :: TDftD3Ref

    !> Atomic expectation values for extrapolation of C6 coefficients
    real(dp), allocatable :: sqrtZr4r2(:)

    !> Number of reference systems per species
    integer, allocatable :: numberOfReferences(:)

    !> Coordination number per reference system and species
    real(dp), allocatable :: referenceCN(:, :)

    !> Dynamic polarizibility per reference system and species
    real(dp), allocatable :: referenceAlpha(:, :, :)

    !> C6 coefficients for each reference system and species pair
    real(dp), allocatable :: referenceC6(:, :, :, :)

  end type TDftD3Ref


  !> Initialize DFT-D3 reference systems
  interface init
    procedure :: initialize
  end interface


  !> Number of references for all supported elements
  integer, parameter :: numberOfReferences(maxElem) = [ &
    &  2, 1, & ! H,He
    &  2, 3, 5, 5, 4, 3, 2, 1, & ! Li-Ne
    &  2, 3, 4, 5, 4, 3, 2, 1, & ! Na-Ar
    &  2, 3, & ! K,Ca
    &        3, 3, 3, 3, 3, 3, 4, 4, 2, 2, & ! Sc-Zn
    &        4, 5, 4, 3, 2, 1, & ! Ga-Kr
    &  2, 3, & ! Rb,Sr
    &        3, 3, 3, 3, 3, 3, 3, 3, 2, 2, & ! Y-Cd
    &        4, 5, 4, 3, 2, 1, & ! In-Xe
    &  2, 3, & ! Cs,Ba
    &     3, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, & ! La-Eu
    &        2, 3, 3, 3, 3, 3, 3, 3, 2, 2, & ! Lu-Hg
    &        4, 5, 4, 3, 2, 1, & ! Tl-Rn
    &  2, 3, & ! Fr,Ra
    &     2, 2, 2, 2, 2, 2] ! Ac-Pu

  !> Reference coordination numbers for all reference systems
  real(dp), parameter :: referenceCN(maxRef, maxElem) = reshape([ &
    & 0.9118_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! H
    & 0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! He
    & 0.0000_dp,  0.9865_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Li
    & 0.0000_dp,  0.9808_dp,  1.9697_dp,  0.0000_dp,  0.0000_dp, & ! Be
    & 0.0000_dp,  0.9706_dp,  1.9441_dp,  2.9128_dp,  4.5856_dp, & ! B
    & 0.0000_dp,  0.9868_dp,  1.9985_dp,  2.9987_dp,  3.9844_dp, & ! C
    & 0.0000_dp,  0.9944_dp,  2.0143_dp,  2.9903_dp,  0.0000_dp, & ! N
    & 0.0000_dp,  0.9925_dp,  1.9887_dp,  0.0000_dp,  0.0000_dp, & ! O
    & 0.0000_dp,  0.9982_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! F
    & 0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Ne
    & 0.0000_dp,  0.9684_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Na
    & 0.0000_dp,  0.9628_dp,  1.9496_dp,  0.0000_dp,  0.0000_dp, & ! Mg
    & 0.0000_dp,  0.9648_dp,  1.9311_dp,  2.9146_dp,  0.0000_dp, & ! Al
    & 0.0000_dp,  0.9507_dp,  1.9435_dp,  2.9407_dp,  3.8677_dp, & ! Si
    & 0.0000_dp,  0.9947_dp,  2.0102_dp,  2.9859_dp,  0.0000_dp, & ! P
    & 0.0000_dp,  0.9948_dp,  1.9903_dp,  0.0000_dp,  0.0000_dp, & ! S
    & 0.0000_dp,  0.9972_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Cl
    & 0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Ar
    & 0.0000_dp,  0.9767_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! K
    & 0.0000_dp,  0.9831_dp,  1.9349_dp,  0.0000_dp,  0.0000_dp, & ! Ca
    & 0.0000_dp,  1.8627_dp,  2.8999_dp,  0.0000_dp,  0.0000_dp, & ! Sc
    & 0.0000_dp,  1.8299_dp,  3.8675_dp,  0.0000_dp,  0.0000_dp, & ! Ti
    & 0.0000_dp,  1.9138_dp,  2.9110_dp,  0.0000_dp,  0.0000_dp, & ! V
    & 0.0000_dp,  1.8269_dp, 10.6191_dp,  0.0000_dp,  0.0000_dp, & ! Cr
    & 0.0000_dp,  1.6406_dp,  9.8849_dp,  0.0000_dp,  0.0000_dp, & ! Mn
    & 0.0000_dp,  1.6483_dp,  9.1376_dp,  0.0000_dp,  0.0000_dp, & ! Fe
    & 0.0000_dp,  1.7149_dp,  2.9263_dp,  7.7785_dp,  0.0000_dp, & ! Co
    & 0.0000_dp,  1.7937_dp,  6.5458_dp,  6.2918_dp,  0.0000_dp, & ! Ni
    & 0.0000_dp,  0.9576_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Cu
    & 0.0000_dp,  1.9419_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Zn
    & 0.0000_dp,  0.9601_dp,  1.9315_dp,  2.9233_dp,  0.0000_dp, & ! Ga
    & 0.0000_dp,  0.9434_dp,  1.9447_dp,  2.9186_dp,  3.8972_dp, & ! Ge
    & 0.0000_dp,  0.9889_dp,  1.9793_dp,  2.9709_dp,  0.0000_dp, & ! As
    & 0.0000_dp,  0.9901_dp,  1.9812_dp,  0.0000_dp,  0.0000_dp, & ! Se
    & 0.0000_dp,  0.9974_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Br
    & 0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Kr
    & 0.0000_dp,  0.9738_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Rb
    & 0.0000_dp,  0.9801_dp,  1.9143_dp,  0.0000_dp,  0.0000_dp, & ! Sr
    & 0.0000_dp,  1.9153_dp,  2.8903_dp,  0.0000_dp,  0.0000_dp, & ! Y
    & 0.0000_dp,  1.9355_dp,  3.9106_dp,  0.0000_dp,  0.0000_dp, & ! Zr
    & 0.0000_dp,  1.9545_dp,  2.9225_dp,  0.0000_dp,  0.0000_dp, & ! Nb
    & 0.0000_dp,  1.9420_dp, 11.0556_dp,  0.0000_dp,  0.0000_dp, & ! Mo
    & 0.0000_dp,  1.6682_dp,  9.5402_dp,  0.0000_dp,  0.0000_dp, & ! Tc
    & 0.0000_dp,  1.8584_dp,  8.8895_dp,  0.0000_dp,  0.0000_dp, & ! Ru
    & 0.0000_dp,  1.9003_dp,  2.9696_dp,  0.0000_dp,  0.0000_dp, & ! Rh
    & 0.0000_dp,  1.8630_dp,  5.7095_dp,  0.0000_dp,  0.0000_dp, & ! Pd
    & 0.0000_dp,  0.9679_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Ag
    & 0.0000_dp,  1.9539_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Cd
    & 0.0000_dp,  0.9633_dp,  1.9378_dp,  2.9353_dp,  0.0000_dp, & ! In
    & 0.0000_dp,  0.9514_dp,  1.9505_dp,  2.9259_dp,  3.9123_dp, & ! Sn
    & 0.0000_dp,  0.9749_dp,  1.9523_dp,  2.9315_dp,  0.0000_dp, & ! Sb
    & 0.0000_dp,  0.9811_dp,  1.9639_dp,  0.0000_dp,  0.0000_dp, & ! Te
    & 0.0000_dp,  0.9968_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! I
    & 0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Xe
    & 0.0000_dp,  0.9909_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Cs
    & 0.0000_dp,  0.9797_dp,  1.8467_dp,  0.0000_dp,  0.0000_dp, & ! Ba
    & 0.0000_dp,  1.9373_dp,  2.9175_dp,  0.0000_dp,  0.0000_dp, & ! La
    & 2.7991_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Ce
    & 0.0000_dp,  2.9425_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Pr
    & 0.0000_dp,  2.9455_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Nd
    & 0.0000_dp,  2.9413_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Pm
    & 0.0000_dp,  2.9300_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Sm
    & 0.0000_dp,  1.8286_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Eu
    & 0.0000_dp,  2.8732_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Gd
    & 0.0000_dp,  2.9086_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Tb
    & 0.0000_dp,  2.8965_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Dy
    & 0.0000_dp,  2.9242_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Ho
    & 0.0000_dp,  2.9282_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Er
    & 0.0000_dp,  2.9246_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Tm
    & 0.0000_dp,  2.8482_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Yb
    & 0.0000_dp,  2.9219_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Lu
    & 0.0000_dp,  1.9254_dp,  3.8840_dp,  0.0000_dp,  0.0000_dp, & ! Hf
    & 0.0000_dp,  1.9459_dp,  2.8988_dp,  0.0000_dp,  0.0000_dp, & ! Ta
    & 0.0000_dp,  1.9292_dp, 10.9153_dp,  0.0000_dp,  0.0000_dp, & ! W
    & 0.0000_dp,  1.8104_dp,  9.8054_dp,  0.0000_dp,  0.0000_dp, & ! Re
    & 0.0000_dp,  1.8858_dp,  9.1527_dp,  0.0000_dp,  0.0000_dp, & ! Os
    & 0.0000_dp,  1.8648_dp,  2.9424_dp,  0.0000_dp,  0.0000_dp, & ! Ir
    & 0.0000_dp,  1.9188_dp,  6.6669_dp,  0.0000_dp,  0.0000_dp, & ! Pt
    & 0.0000_dp,  0.9846_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Au
    & 0.0000_dp,  1.9896_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Hg
    & 0.0000_dp,  0.9267_dp,  1.9302_dp,  2.9420_dp,  0.0000_dp, & ! Tl
    & 0.0000_dp,  0.9383_dp,  1.9356_dp,  2.9081_dp,  3.9098_dp, & ! Pb
    & 0.0000_dp,  0.9820_dp,  1.9655_dp,  2.9500_dp,  0.0000_dp, & ! Bi
    & 0.0000_dp,  0.9815_dp,  1.9639_dp,  0.0000_dp,  0.0000_dp, & ! Po
    & 0.0000_dp,  0.9954_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! At
    & 0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Rn
    & 0.0000_dp,  0.9705_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Fr
    & 0.0000_dp,  0.9662_dp,  1.8075_dp,  0.0000_dp,  0.0000_dp, & ! Ra
    & 0.0000_dp,  2.9070_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Ac
    & 0.0000_dp,  2.8844_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Th
    & 0.0000_dp,  2.8738_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Pa
    & 0.0000_dp,  2.8878_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! U
    & 0.0000_dp,  2.9095_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp, & ! Np
    & 0.0000_dp,  1.9209_dp,  0.0000_dp,  0.0000_dp,  0.0000_dp],& ! Pu
    & shape(referenceCN))

  !> Storage for reference C6 coefficients
  real(dp), protected, allocatable :: referenceC6(:, :, :)


contains


  !> Initialize coordination number container
  subroutine initialize(this, speciesNames)

    !> Initialised instance at return
    type(TDftD3Ref), intent(out) :: this

    !> Names of species.
    character(*), intent(in) :: speciesNames(:)

    integer :: nSpecies
    integer :: iSp1, iZp1, iRef1, iSp2, iZp2, iRef2
    real(dp) :: c6

    call initD3References()

    nSpecies = size(speciesNames)

    allocate(this%numberOfReferences(nSpecies), this%sqrtZr4r2(nSpecies))
    this%sqrtZr4r2(:) = getSqrtZr4r2(speciesNames)
    this%numberOfReferences(:) = 0
    allocate(this%referenceCN(maxRef, nSpecies), &
        & this%referenceC6(maxRef, maxRef, nSpecies, nSpecies))
    this%referenceCN(:, :) = 0.0_dp
    this%referenceC6(:, :, :, :) = 0.0_dp

    ! load number of reference systems and reference CNs
    do iSp1 = 1, nSpecies
      iZp1 = symbolToNumber(speciesNames(iSp1))
      this%numberOfReferences(iSp1) = numberOfReferences(iZp1)
      do iRef1 = 1, this%numberOfReferences(iSp1)
        this%referenceCN(iRef1,iSp1) = referenceCN(iRef1,iZp1)
      end do
    end do

    ! load C6 coefficients
    do iSp1 = 1, nSpecies
      iZp1 = symbolToNumber(speciesNames(iSp1))
      do iSp2 = 1, iSp1
        iZp2 = symbolToNumber(speciesNames(iSp2))
        do iRef1 = 1, this%numberOfReferences(iSp1)
          do iRef2 = 1, this%numberOfReferences(iSp2)
            c6 = getC6(iRef1, iRef2, iZp1, iZp2)
            this%referenceC6(iRef2,iRef1,iSp2,iSp1) = c6
            this%referenceC6(iRef1,iRef2,iSp1,iSp2) = c6
          end do
        end do
      end do
    end do

  end subroutine initialize


  !> Initialize internal storage from chunks
  subroutine initD3References()

    if (.not.allocated(referenceC6)) call copyC6(referenceC6)

  end subroutine initD3References


  !> Clear internal storage
  subroutine clearD3References()

    if (allocated(referenceC6)) deallocate(referenceC6)

  end subroutine clearD3References


  !> Return C6 coefficients for given reference systems of atom pair
  elemental function getC6(iref, jref, ati, atj) result(c6)

    !> Reference system of first element
    integer, intent(in) :: iref

    !> Reference system of second element
    integer, intent(in) :: jref

    !> Atomic number of first element
    integer, intent(in) :: ati

    !> Atomic number of second element
    integer, intent(in) :: atj

    !> Reference C6 coefficient
    real(dp) :: c6

    integer :: ic

    if (ati > atj) then
      ic = atj + ati*(ati-1)/2
      c6 = referenceC6(iref, jref, ic)
    else
      ic = ati + atj*(atj-1)/2
      c6 = referenceC6(jref, iref, ic)
    endif

  end function getC6


  pure subroutine copyC6(c6ab)
    real(dp), allocatable, target, intent(out) :: c6ab(:, :, :)

    real(dp), pointer :: c6ab1d(:)

    allocate(c6ab(maxRef, maxRef, (maxElem+1)*maxElem/2), source=0.0_dp)
    c6ab1d(1 : maxRef * maxRef * (maxElem + 1) * maxElem / 2) => c6ab

    c6ab1d(0 * maxRef * maxRef + 1 : 19 * maxRef * maxRef) = [ &
      &    3.0267_dp,    4.7379_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    4.7379_dp,    7.5916_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    2.0835_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    3.1287_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    1.5583_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   38.9448_dp,   14.3165_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.9391_dp,   24.0573_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.1508_dp,    8.7773_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1163.4454_dp,  282.2106_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  282.2106_dp,   85.3197_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   24.4415_dp,   18.4656_dp,   12.5931_dp,    0.0000_dp,    0.0000_dp, &
      &   41.3078_dp,   30.7866_dp,   20.4039_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   14.8246_dp,   11.4655_dp,    8.1628_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  494.6190_dp,  352.2160_dp,  194.7036_dp,    0.0000_dp,    0.0000_dp, &
      &  148.1323_dp,  107.5083_dp,   66.2009_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  257.4863_dp,  186.3453_dp,  113.9677_dp,    0.0000_dp,    0.0000_dp, &
      &  186.3453_dp,  135.8445_dp,   84.4000_dp,    0.0000_dp,    0.0000_dp, &
      &  113.9677_dp,   84.4000_dp,   55.1364_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   17.3143_dp,   14.7237_dp,   12.4952_dp,    9.7472_dp,    9.2092_dp, &
      &   28.2767_dp,   23.8412_dp,   20.0597_dp,   15.3531_dp,   14.4541_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   11.0975_dp,    9.5836_dp,    8.2586_dp,    6.6434_dp,    6.3181_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  283.7308_dp,  231.4209_dp,  188.7611_dp,  129.9875_dp,  120.6628_dp, &
      &   93.6729_dp,   77.6633_dp,   64.2553_dp,   47.0338_dp,   43.9746_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  161.5971_dp,  133.7265_dp,  110.4185_dp,   80.4276_dp,   75.1311_dp, &
      &  119.0815_dp,   98.9938_dp,   82.1430_dp,   60.5234_dp,   56.6527_dp, &
      &   76.7230_dp,   64.4301_dp,   53.9919_dp,   40.9358_dp,   38.4745_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  107.1777_dp,   89.7429_dp,   74.9865_dp,   56.4116_dp,   52.9606_dp, &
      &   89.7429_dp,   75.3686_dp,   63.1685_dp,   47.8580_dp,   44.9888_dp, &
      &   74.9865_dp,   63.1685_dp,   53.1128_dp,   40.5192_dp,   38.1398_dp, &
      &   56.4116_dp,   47.8580_dp,   40.5192_dp,   31.4436_dp,   29.6804_dp, &
      &   52.9606_dp,   44.9888_dp,   38.1398_dp,   29.6804_dp,   28.0315_dp, &
      &   12.1402_dp,   11.3932_dp,    9.4203_dp,    8.8210_dp,    7.3662_dp, &
      &   19.2653_dp,   18.0575_dp,   14.7623_dp,   13.7992_dp,   11.3299_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    8.1841_dp,    7.7065_dp,    6.5027_dp,    6.1196_dp,    5.2567_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.9030_dp,  160.4590_dp,  123.7707_dp,  115.9498_dp,   87.8183_dp, &
      &   60.0860_dp,   56.3468_dp,   44.9320_dp,   41.9713_dp,   33.2158_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  102.9560_dp,   96.5431_dp,   76.7552_dp,   71.6818_dp,   56.4662_dp, &
      &   77.0951_dp,   72.3162_dp,   57.8805_dp,   54.0770_dp,   43.0620_dp, &
      &   51.5628_dp,   48.3072_dp,   39.2814_dp,   36.6977_dp,   29.8819_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(19 * maxRef * maxRef + 1 : 38 * maxRef * maxRef) = [ &
      &   71.2794_dp,   66.7986_dp,   54.0854_dp,   50.5303_dp,   40.8962_dp, &
      &   60.2996_dp,   56.5295_dp,   45.9632_dp,   42.9646_dp,   34.9990_dp, &
      &   50.9090_dp,   47.7462_dp,   38.9834_dp,   36.4604_dp,   29.8916_dp, &
      &   39.2449_dp,   36.8166_dp,   30.3536_dp,   28.4121_dp,   23.6210_dp, &
      &   37.0042_dp,   34.7226_dp,   28.6763_dp,   26.8506_dp,   22.3797_dp, &
      &   49.1130_dp,   46.0681_dp,   37.8419_dp,   35.4129_dp,   29.2830_dp, &
      &   46.0681_dp,   43.2452_dp,   35.5219_dp,   33.2540_dp,   27.5206_dp, &
      &   37.8419_dp,   35.5219_dp,   29.3602_dp,   27.5063_dp,   22.9517_dp, &
      &   35.4129_dp,   33.2540_dp,   27.5063_dp,   25.7809_dp,   21.5377_dp, &
      &   29.2830_dp,   27.5206_dp,   22.9517_dp,   21.5377_dp,   18.2067_dp, &
      &    8.7171_dp,    8.1417_dp,    7.6610_dp,    6.7746_dp,    0.0000_dp, &
      &   13.5164_dp,   12.5980_dp,   11.8214_dp,   10.3987_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    6.1380_dp,    5.7601_dp,    5.4559_dp,    4.8836_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  108.4854_dp,  101.2701_dp,   94.0060_dp,   81.9770_dp,    0.0000_dp, &
      &   40.2944_dp,   37.4939_dp,   35.0194_dp,   30.6022_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.6458_dp,   63.8580_dp,   59.5983_dp,   52.0237_dp,    0.0000_dp, &
      &   52.0852_dp,   48.4885_dp,   45.3164_dp,   39.6478_dp,    0.0000_dp, &
      &   35.7902_dp,   33.3274_dp,   31.2407_dp,   27.4225_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.1132_dp,   45.7319_dp,   42.8348_dp,   37.5685_dp,    0.0000_dp, &
      &   41.9015_dp,   39.0433_dp,   36.6063_dp,   32.1644_dp,    0.0000_dp, &
      &   35.6778_dp,   33.2677_dp,   31.2221_dp,   27.4842_dp,    0.0000_dp, &
      &   28.0129_dp,   26.1481_dp,   24.5919_dp,   21.7205_dp,    0.0000_dp, &
      &   26.5083_dp,   24.7525_dp,   23.2900_dp,   20.5886_dp,    0.0000_dp, &
      &   34.8146_dp,   32.4848_dp,   30.5305_dp,   26.9351_dp,    0.0000_dp, &
      &   32.7009_dp,   30.5410_dp,   28.6938_dp,   25.3318_dp,    0.0000_dp, &
      &   27.1704_dp,   25.3827_dp,   23.8965_dp,   21.1488_dp,    0.0000_dp, &
      &   25.4799_dp,   23.8136_dp,   22.4279_dp,   19.8669_dp,    0.0000_dp, &
      &   21.4199_dp,   20.0468_dp,   18.9172_dp,   16.8169_dp,    0.0000_dp, &
      &   25.2685_dp,   23.6295_dp,   22.2794_dp,   19.7707_dp,    0.0000_dp, &
      &   23.6295_dp,   22.1241_dp,   20.8501_dp,   18.5180_dp,    0.0000_dp, &
      &   22.2794_dp,   20.8501_dp,   19.6768_dp,   17.4928_dp,    0.0000_dp, &
      &   19.7707_dp,   18.5180_dp,   17.4928_dp,   15.5817_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    6.7180_dp,    6.0575_dp,    5.3717_dp,    0.0000_dp,    0.0000_dp, &
      &   10.2371_dp,    9.1812_dp,    8.0848_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    4.8949_dp,    4.4593_dp,    4.0179_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.9613_dp,   67.9312_dp,   58.6703_dp,    0.0000_dp,    0.0000_dp, &
      &   29.5575_dp,   26.2708_dp,   22.8930_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.1252_dp,   44.4968_dp,   38.7027_dp,    0.0000_dp,    0.0000_dp, &
      &   38.4132_dp,   34.1983_dp,   29.8523_dp,    0.0000_dp,    0.0000_dp, &
      &   26.8954_dp,   24.0592_dp,   21.1282_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   36.7247_dp,   32.8136_dp,   28.7713_dp,    0.0000_dp,    0.0000_dp, &
      &   31.5354_dp,   28.2327_dp,   24.8185_dp,    0.0000_dp,    0.0000_dp, &
      &   27.0247_dp,   24.2416_dp,   21.3648_dp,    0.0000_dp,    0.0000_dp, &
      &   21.4989_dp,   19.3571_dp,   17.1424_dp,    0.0000_dp,    0.0000_dp, &
      &   20.3998_dp,   18.3831_dp,   16.2987_dp,    0.0000_dp,    0.0000_dp, &
      &   26.5929_dp,   23.9120_dp,   21.1428_dp,    0.0000_dp,    0.0000_dp, &
      &   25.0097_dp,   22.5178_dp,   19.9090_dp,    0.0000_dp,    0.0000_dp, &
      &   20.9597_dp,   18.9034_dp,   16.7855_dp,    0.0000_dp,    0.0000_dp, &
      &   19.6943_dp,   17.7750_dp,   15.8009_dp,    0.0000_dp,    0.0000_dp, &
      &   16.7544_dp,   15.1751_dp,   13.5525_dp,    0.0000_dp,    0.0000_dp, &
      &   19.6546_dp,   17.7698_dp,   15.8364_dp,    0.0000_dp,    0.0000_dp, &
      &   18.4128_dp,   16.6775_dp,   14.8600_dp,    0.0000_dp,    0.0000_dp, &
      &   17.4093_dp,   15.7631_dp,   14.0807_dp,    0.0000_dp,    0.0000_dp, &
      &   15.5249_dp,   14.0793_dp,   12.6077_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.5059_dp,   14.0764_dp,   12.6277_dp,    0.0000_dp,    0.0000_dp, &
      &   14.0764_dp,   12.8161_dp,   11.5009_dp,    0.0000_dp,    0.0000_dp, &
      &   12.6277_dp,   11.5009_dp,   10.3708_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    5.1616_dp,    4.2672_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    7.7441_dp,    6.3000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    3.8825_dp,    3.3077_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(38 * maxRef * maxRef + 1 : 57 * maxRef * maxRef) = [ &
      &   55.0933_dp,   42.0627_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   21.7605_dp,   17.1733_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   36.7453_dp,   28.8615_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   28.4098_dp,   22.5413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.2083_dp,   16.3185_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   27.4821_dp,   22.0931_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   23.7369_dp,   19.2029_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.4592_dp,   16.6531_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   16.4607_dp,   13.5581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.6578_dp,   12.9291_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.2827_dp,   16.6373_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   19.0974_dp,   15.6857_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   16.1312_dp,   13.3529_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.1883_dp,   12.5970_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   13.0546_dp,   10.9391_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.2418_dp,   12.7107_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   14.3015_dp,   11.9470_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   13.5613_dp,   11.3571_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   12.1511_dp,   10.2214_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   12.1834_dp,   10.2867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   11.0994_dp,    9.4048_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   10.0222_dp,    8.5397_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    9.6916_dp,    8.2739_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    8.2739_dp,    7.1341_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    4.0112_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    5.9403_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    3.1025_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   40.4731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   16.3388_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   27.4867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   21.4067_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.4176_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.9022_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   18.1424_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.7127_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   12.7577_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   12.1608_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.6740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   14.7716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   12.5629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   11.8513_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   10.2716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   11.9479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   11.2243_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   10.6740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    9.6050_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    9.6606_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    8.8252_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    8.0175_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    7.7691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    6.6950_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    6.2896_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   46.8232_dp,   20.7567_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   82.5641_dp,   34.9952_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   26.8628_dp,   12.7287_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(57 * maxRef * maxRef + 1 : 76 * maxRef * maxRef) = [ &
      & 1367.3272_dp,  425.7161_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  335.3450_dp,  125.8245_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.4563_dp,  218.6572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  418.6488_dp,  158.3268_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.8060_dp,   96.5125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  338.7212_dp,  136.9536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  276.5671_dp,  113.4016_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  225.7843_dp,   93.7159_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  156.1138_dp,   68.2755_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  145.0020_dp,   63.8123_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.7631_dp,   87.3956_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  192.3729_dp,   82.0144_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  148.7618_dp,   65.2739_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.3753_dp,   61.0174_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.9248_dp,   48.1622_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  130.6563_dp,   58.4853_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  121.9601_dp,   54.4454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  113.3031_dp,   50.8672_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   98.8806_dp,   44.4924_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.0263_dp,   42.8911_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   82.1820_dp,   38.1227_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   71.0999_dp,   33.2636_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   66.8423_dp,   31.6060_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   51.2498_dp,   24.9560_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.2799_dp,   23.7745_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1608.0286_dp,  505.4363_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  505.4363_dp,  186.1052_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   38.3531_dp,   31.9913_dp,   21.8370_dp,    0.0000_dp,    0.0000_dp, &
      &   65.3703_dp,   54.1021_dp,   35.9190_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   23.0320_dp,   19.4648_dp,   13.8496_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  830.8156_dp,  670.0330_dp,  370.8190_dp,    0.0000_dp,    0.0000_dp, &
      &  240.1627_dp,  195.7057_dp,  120.7808_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  418.2164_dp,  340.2602_dp,  208.6769_dp,    0.0000_dp,    0.0000_dp, &
      &  301.3777_dp,  246.2227_dp,  153.1575_dp,    0.0000_dp,    0.0000_dp, &
      &  181.1653_dp,  149.3832_dp,   97.8101_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  258.1303_dp,  212.2180_dp,  136.9932_dp,    0.0000_dp,    0.0000_dp, &
      &  212.9979_dp,  175.5646_dp,  114.4190_dp,    0.0000_dp,    0.0000_dp, &
      &  175.3914_dp,  144.9783_dp,   95.3575_dp,    0.0000_dp,    0.0000_dp, &
      &  126.5425_dp,  105.3054_dp,   71.3023_dp,    0.0000_dp,    0.0000_dp, &
      &  118.0748_dp,   98.3724_dp,   66.8696_dp,    0.0000_dp,    0.0000_dp, &
      &  162.6082_dp,  134.9227_dp,   90.3305_dp,    0.0000_dp,    0.0000_dp, &
      &  152.6130_dp,  126.6657_dp,   84.6178_dp,    0.0000_dp,    0.0000_dp, &
      &  120.7386_dp,  100.5817_dp,   68.2984_dp,    0.0000_dp,    0.0000_dp, &
      &  112.8210_dp,   93.9978_dp,   63.7944_dp,    0.0000_dp,    0.0000_dp, &
      &   88.2620_dp,   74.0028_dp,   51.3525_dp,    0.0000_dp,    0.0000_dp, &
      &  107.6150_dp,   89.9646_dp,   61.8372_dp,    0.0000_dp,    0.0000_dp, &
      &  100.1443_dp,   83.7703_dp,   57.5398_dp,    0.0000_dp,    0.0000_dp, &
      &   93.4130_dp,   78.1760_dp,   53.8803_dp,    0.0000_dp,    0.0000_dp, &
      &   81.5352_dp,   68.3141_dp,   47.2059_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   78.2250_dp,   65.7609_dp,   46.0347_dp,    0.0000_dp,    0.0000_dp, &
      &   69.3675_dp,   58.4303_dp,   41.0605_dp,    0.0000_dp,    0.0000_dp, &
      &   60.2983_dp,   50.8669_dp,   35.9665_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   57.1605_dp,   48.2865_dp,   34.3314_dp,    0.0000_dp,    0.0000_dp, &
      &   44.7141_dp,   37.9884_dp,   27.4841_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.6771_dp,   36.1911_dp,   26.0533_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(76 * maxRef * maxRef + 1 : 95 * maxRef * maxRef) = [ &
      &  985.1697_dp,  794.7597_dp,  442.4019_dp,    0.0000_dp,    0.0000_dp, &
      &  355.9180_dp,  289.6082_dp,  176.8683_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  683.3758_dp,  554.5967_dp,  334.2493_dp,    0.0000_dp,    0.0000_dp, &
      &  554.5967_dp,  451.2708_dp,  274.1590_dp,    0.0000_dp,    0.0000_dp, &
      &  334.2493_dp,  274.1590_dp,  175.5616_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   36.2909_dp,   33.1048_dp,   26.3524_dp,   21.0584_dp,    0.0000_dp, &
      &   60.8558_dp,   55.2636_dp,   43.3661_dp,   34.0700_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.3224_dp,   20.5158_dp,   16.7031_dp,   13.6982_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  705.8254_dp,  627.5309_dp,  456.3613_dp,  322.7925_dp,    0.0000_dp, &
      &  214.7265_dp,  193.0707_dp,  146.4426_dp,  110.2136_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  372.6302_dp,  334.7335_dp,  253.0649_dp,  189.6605_dp,    0.0000_dp, &
      &  270.7461_dp,  243.7612_dp,  185.7310_dp,  140.5632_dp,    0.0000_dp, &
      &  167.2929_dp,  151.5931_dp,  118.1233_dp,   92.0126_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  236.4780_dp,  213.8848_dp,  165.5938_dp,  127.9645_dp,    0.0000_dp, &
      &  196.2014_dp,  177.7254_dp,  138.2838_dp,  107.5158_dp,    0.0000_dp, &
      &  162.4473_dp,  147.3766_dp,  115.2480_dp,   90.1440_dp,    0.0000_dp, &
      &  119.1545_dp,  108.5517_dp,   86.0532_dp,   68.4330_dp,    0.0000_dp, &
      &  111.4377_dp,  101.5889_dp,   80.7005_dp,   64.3332_dp,    0.0000_dp, &
      &  152.0934_dp,  138.3254_dp,  109.0541_dp,   86.1605_dp,    0.0000_dp, &
      &  142.6094_dp,  129.6922_dp,  102.2095_dp,   80.7111_dp,    0.0000_dp, &
      &  113.8560_dp,  103.7907_dp,   82.4303_dp,   65.6968_dp,    0.0000_dp, &
      &  106.3508_dp,   96.9560_dp,   77.0087_dp,   61.3829_dp,    0.0000_dp, &
      &   84.3073_dp,   77.1362_dp,   61.9610_dp,   50.0397_dp,    0.0000_dp, &
      &  102.2000_dp,   93.3542_dp,   74.6117_dp,   59.9074_dp,    0.0000_dp, &
      &   95.0916_dp,   86.8728_dp,   69.4501_dp,   55.7793_dp,    0.0000_dp, &
      &   88.8407_dp,   81.2026_dp,   65.0164_dp,   52.3136_dp,    0.0000_dp, &
      &   77.6520_dp,   71.0231_dp,   56.9713_dp,   45.9399_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   75.0755_dp,   68.7983_dp,   55.5250_dp,   45.0876_dp,    0.0000_dp, &
      &   66.7451_dp,   61.2175_dp,   49.5319_dp,   40.3374_dp,    0.0000_dp, &
      &   58.1951_dp,   53.4365_dp,   43.3788_dp,   35.4602_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   55.3412_dp,   50.8577_dp,   41.3914_dp,   33.9334_dp,    0.0000_dp, &
      &   43.7364_dp,   40.3220_dp,   33.1265_dp,   27.4436_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.5966_dp,   38.3154_dp,   31.3944_dp,   25.9335_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  838.9648_dp,  746.3419_dp,  543.9378_dp,  386.0980_dp,    0.0000_dp, &
      &  316.4588_dp,  284.2251_dp,  214.6796_dp,  160.6594_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  603.4689_dp,  541.0285_dp,  406.0687_dp,  301.3307_dp,    0.0000_dp, &
      &  492.0829_dp,  441.7322_dp,  333.0505_dp,  248.5692_dp,    0.0000_dp, &
      &  304.7337_dp,  275.2609_dp,  212.1971_dp,  163.1100_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  540.5406_dp,  486.2365_dp,  369.3829_dp,  278.5639_dp,    0.0000_dp, &
      &  486.2365_dp,  437.7575_dp,  333.5341_dp,  252.4953_dp,    0.0000_dp, &
      &  369.3829_dp,  333.5341_dp,  256.7497_dp,  196.9411_dp,    0.0000_dp, &
      &  278.5639_dp,  252.4953_dp,  196.9411_dp,  153.5945_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.5947_dp,   27.4989_dp,   25.7021_dp,   24.9833_dp,   21.0064_dp, &
      &   48.5182_dp,   44.9030_dp,   41.8222_dp,   40.6972_dp,   33.7126_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   18.8500_dp,   17.6356_dp,   16.5806_dp,   16.1104_dp,   13.8547_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  495.3449_dp,  450.9583_dp,  412.4323_dp,  408.0802_dp,  308.7953_dp, &
      &  162.0508_dp,  148.7478_dp,  137.4458_dp,  134.4490_dp,  107.2721_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  279.7863_dp,  256.5881_dp,  236.8991_dp,  231.8251_dp,  184.2566_dp, &
      &  205.7544_dp,  189.1090_dp,  174.9315_dp,  171.0398_dp,  137.1783_dp, &
      &  131.8920_dp,  121.8333_dp,  113.2820_dp,  110.3134_dp,   90.6956_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(95 * maxRef * maxRef + 1 : 114 * maxRef * maxRef) = [ &
      &  184.5111_dp,  170.1891_dp,  158.0165_dp,  154.0388_dp,  125.7764_dp, &
      &  154.2901_dp,  142.5123_dp,  132.4831_dp,  129.0898_dp,  105.9826_dp, &
      &  128.7459_dp,  119.0889_dp,  110.8447_dp,  107.9647_dp,   89.1190_dp, &
      &   96.5313_dp,   89.5948_dp,   83.6586_dp,   81.3478_dp,   68.1106_dp, &
      &   90.5740_dp,   84.1158_dp,   78.5835_dp,   76.4038_dp,   64.1083_dp, &
      &  122.1387_dp,  113.2046_dp,  105.5713_dp,  102.7202_dp,   85.5223_dp, &
      &  114.4405_dp,  106.0806_dp,   98.9278_dp,   96.2872_dp,   80.1427_dp, &
      &   92.4931_dp,   85.9068_dp,   80.2619_dp,   78.0574_dp,   65.4879_dp, &
      &   86.3999_dp,   80.2618_dp,   74.9967_dp,   72.9530_dp,   61.2169_dp, &
      &   69.7224_dp,   64.9672_dp,   60.8687_dp,   59.1583_dp,   50.2053_dp, &
      &   83.8498_dp,   78.0190_dp,   73.0073_dp,   70.9761_dp,   59.9319_dp, &
      &   78.0466_dp,   72.6373_dp,   67.9820_dp,   66.1052_dp,   55.8394_dp, &
      &   73.0858_dp,   68.0504_dp,   63.7151_dp,   61.9530_dp,   52.4136_dp, &
      &   64.0576_dp,   59.6887_dp,   55.9197_dp,   54.3857_dp,   46.1031_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   62.5349_dp,   58.3557_dp,   54.7443_dp,   53.2061_dp,   45.3688_dp, &
      &   55.8175_dp,   52.1311_dp,   48.9391_dp,   47.5664_dp,   40.6635_dp, &
      &   48.9066_dp,   45.7273_dp,   42.9679_dp,   41.7687_dp,   35.8247_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   46.6936_dp,   43.6855_dp,   41.0728_dp,   39.9172_dp,   34.3187_dp, &
      &   37.4421_dp,   35.1290_dp,   33.1075_dp,   32.1730_dp,   27.9086_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.4550_dp,   33.2416_dp,   31.3107_dp,   30.4338_dp,   26.3358_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  591.0458_dp,  538.3412_dp,  492.6562_dp,  487.1761_dp,  369.7717_dp, &
      &  237.1326_dp,  217.5049_dp,  200.8038_dp,  196.6526_dp,  156.1545_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  447.6423_dp,  409.9174_dp,  377.8360_dp,  370.3778_dp,  291.8716_dp, &
      &  367.5811_dp,  337.0250_dp,  310.9897_dp,  304.6797_dp,  241.3903_dp, &
      &  236.1388_dp,  217.5471_dp,  201.7758_dp,  196.7841_dp,  159.9115_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  408.9606_dp,  375.5233_dp,  347.1154_dp,  339.4502_dp,  271.3202_dp, &
      &  369.6232_dp,  339.6516_dp,  314.1809_dp,  307.1015_dp,  246.3013_dp, &
      &  285.4825_dp,  262.9770_dp,  243.8407_dp,  237.9434_dp,  193.0534_dp, &
      &  219.9479_dp,  203.2226_dp,  189.0034_dp,  184.0291_dp,  151.4603_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  317.8574_dp,  292.9994_dp,  271.8844_dp,  265.1189_dp,  215.8998_dp, &
      &  292.9994_dp,  270.2650_dp,  250.9376_dp,  244.6288_dp,  199.7530_dp, &
      &  271.8844_dp,  250.9376_dp,  233.1254_dp,  227.1882_dp,  185.9996_dp, &
      &  265.1189_dp,  244.6288_dp,  227.1882_dp,  221.5046_dp,  181.0237_dp, &
      &  215.8998_dp,  199.7530_dp,  185.9996_dp,  181.0237_dp,  149.7734_dp, &
      &   23.7604_dp,   23.0908_dp,   22.7429_dp,   21.2507_dp,    0.0000_dp, &
      &   38.1397_dp,   37.0236_dp,   36.4936_dp,   33.9193_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.6689_dp,   15.2619_dp,   15.0245_dp,   14.1577_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  350.8030_dp,  339.6693_dp,  337.4662_dp,  304.7707_dp,    0.0000_dp, &
      &  121.5067_dp,  117.7472_dp,  116.3941_dp,  106.8466_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  208.7331_dp,  202.2282_dp,  199.9576_dp,  183.3056_dp,    0.0000_dp, &
      &  155.3644_dp,  150.6068_dp,  148.8194_dp,  136.8674_dp,    0.0000_dp, &
      &  102.6176_dp,   99.5656_dp,   98.1859_dp,   91.0206_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  142.3481_dp,  138.0774_dp,  136.2402_dp,  126.0194_dp,    0.0000_dp, &
      &  119.9353_dp,  116.3818_dp,  114.7968_dp,  106.3923_dp,    0.0000_dp, &
      &  100.8432_dp,   97.8947_dp,   96.5324_dp,   89.6391_dp,    0.0000_dp, &
      &   77.0429_dp,   74.8497_dp,   73.7397_dp,   68.8007_dp,    0.0000_dp, &
      &   72.5145_dp,   70.4629_dp,   69.4115_dp,   64.8127_dp,    0.0000_dp, &
      &   96.7503_dp,   93.9663_dp,   92.6080_dp,   86.2427_dp,    0.0000_dp, &
      &   90.6767_dp,   88.0769_dp,   86.8128_dp,   80.8496_dp,    0.0000_dp, &
      &   74.0784_dp,   71.9914_dp,   70.9269_dp,   66.2343_dp,    0.0000_dp, &
      &   69.2515_dp,   67.3088_dp,   66.3194_dp,   61.9433_dp,    0.0000_dp, &
      &   56.7852_dp,   55.2375_dp,   54.3941_dp,   51.0048_dp,    0.0000_dp, &
      &   67.7873_dp,   65.9127_dp,   64.9209_dp,   60.7663_dp,    0.0000_dp, &
      &   63.1669_dp,   61.4286_dp,   60.5085_dp,   56.6508_dp,    0.0000_dp, &
      &   59.2859_dp,   57.6632_dp,   56.7981_dp,   53.2076_dp,    0.0000_dp, &
      &   52.1509_dp,   50.7398_dp,   49.9818_dp,   46.8635_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   51.3098_dp,   49.9374_dp,   49.1729_dp,   46.1927_dp,    0.0000_dp, &
      &   45.9937_dp,   44.7770_dp,   44.0904_dp,   41.4600_dp,    0.0000_dp, &
      &   40.5167_dp,   39.4617_dp,   38.8578_dp,   36.5875_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   38.8084_dp,   37.8035_dp,   37.2205_dp,   35.0733_dp,    0.0000_dp, &
      &   31.5577_dp,   30.7688_dp,   30.2894_dp,   28.6362_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(114 * maxRef * maxRef + 1 : 133 * maxRef * maxRef) = [ &
      &   29.7767_dp,   29.0280_dp,   28.5805_dp,   26.9990_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  420.0064_dp,  406.7150_dp,  403.9802_dp,  365.2110_dp,    0.0000_dp, &
      &  176.9205_dp,  171.4469_dp,  169.5698_dp,  155.4495_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.7801_dp,  320.4176_dp,  317.0824_dp,  289.9090_dp,    0.0000_dp, &
      &  273.5268_dp,  265.0378_dp,  262.1696_dp,  240.1582_dp,    0.0000_dp, &
      &  180.9843_dp,  175.5006_dp,  173.2227_dp,  159.9562_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.2965_dp,  297.8076_dp,  294.3403_dp,  270.3535_dp,    0.0000_dp, &
      &  278.9272_dp,  270.3580_dp,  267.1404_dp,  245.6529_dp,    0.0000_dp, &
      &  218.5302_dp,  211.9205_dp,  209.2061_dp,  193.1132_dp,    0.0000_dp, &
      &  171.3557_dp,  166.2698_dp,  163.9565_dp,  152.0452_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  244.3546_dp,  236.9864_dp,  233.8752_dp,  216.1303_dp,    0.0000_dp, &
      &  226.0657_dp,  219.2863_dp,  216.3693_dp,  200.1422_dp,    0.0000_dp, &
      &  210.4840_dp,  204.1998_dp,  201.4465_dp,  186.5031_dp,    0.0000_dp, &
      &  204.8770_dp,  198.7616_dp,  196.1196_dp,  181.4840_dp,    0.0000_dp, &
      &  169.4356_dp,  164.4669_dp,  162.1289_dp,  150.6308_dp,    0.0000_dp, &
      &  191.6887_dp,  186.0684_dp,  183.4322_dp,  170.4045_dp,    0.0000_dp, &
      &  186.0684_dp,  180.6238_dp,  178.0622_dp,  165.4518_dp,    0.0000_dp, &
      &  183.4322_dp,  178.0622_dp,  175.5534_dp,  163.0764_dp,    0.0000_dp, &
      &  170.4045_dp,  165.4518_dp,  163.0764_dp,  151.6886_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.0948_dp,   19.8667_dp,   19.4729_dp,    0.0000_dp,    0.0000_dp, &
      &   31.7713_dp,   31.4127_dp,   30.7747_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   13.6108_dp,   13.4598_dp,   13.2090_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.7867_dp,  271.3011_dp,  265.5464_dp,    0.0000_dp,    0.0000_dp, &
      &   98.1249_dp,   97.0849_dp,   95.0531_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  167.9513_dp,  166.1807_dp,  162.6875_dp,    0.0000_dp,    0.0000_dp, &
      &  126.0993_dp,  124.7512_dp,  122.1534_dp,    0.0000_dp,    0.0000_dp, &
      &   84.8691_dp,   83.9181_dp,   82.1973_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  117.1121_dp,  115.8168_dp,  113.4324_dp,    0.0000_dp,    0.0000_dp, &
      &   99.2159_dp,   98.1139_dp,   96.1100_dp,    0.0000_dp,    0.0000_dp, &
      &   83.8847_dp,   82.9497_dp,   81.2693_dp,    0.0000_dp,    0.0000_dp, &
      &   64.8918_dp,   64.1573_dp,   62.8783_dp,    0.0000_dp,    0.0000_dp, &
      &   61.2184_dp,   60.5255_dp,   59.3239_dp,    0.0000_dp,    0.0000_dp, &
      &   81.0919_dp,   80.1804_dp,   78.5729_dp,    0.0000_dp,    0.0000_dp, &
      &   76.0552_dp,   75.2039_dp,   73.7006_dp,    0.0000_dp,    0.0000_dp, &
      &   62.5953_dp,   61.8916_dp,   60.6690_dp,    0.0000_dp,    0.0000_dp, &
      &   58.5776_dp,   57.9221_dp,   56.7826_dp,    0.0000_dp,    0.0000_dp, &
      &   48.5681_dp,   48.0221_dp,   47.0943_dp,    0.0000_dp,    0.0000_dp, &
      &   57.6734_dp,   57.0255_dp,   55.9136_dp,    0.0000_dp,    0.0000_dp, &
      &   53.8107_dp,   53.2086_dp,   52.1753_dp,    0.0000_dp,    0.0000_dp, &
      &   50.5946_dp,   50.0298_dp,   49.0629_dp,    0.0000_dp,    0.0000_dp, &
      &   44.6529_dp,   44.1581_dp,   43.3130_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.1470_dp,   43.6548_dp,   42.8245_dp,    0.0000_dp,    0.0000_dp, &
      &   39.7091_dp,   39.2683_dp,   38.5276_dp,    0.0000_dp,    0.0000_dp, &
      &   35.1377_dp,   34.7510_dp,   34.1042_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   33.7264_dp,   33.3550_dp,   32.7365_dp,    0.0000_dp,    0.0000_dp, &
      &   27.7137_dp,   27.4117_dp,   26.9163_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   26.0940_dp,   25.8107_dp,   25.3435_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  328.5990_dp,  325.5969_dp,  318.7120_dp,    0.0000_dp,    0.0000_dp, &
      &  142.5430_dp,  141.0599_dp,  138.1166_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  264.6665_dp,  261.9437_dp,  256.4315_dp,    0.0000_dp,    0.0000_dp, &
      &  219.9489_dp,  217.6629_dp,  213.1036_dp,    0.0000_dp,    0.0000_dp, &
      &  148.2130_dp,  146.5849_dp,  143.5523_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  248.5008_dp,  245.8568_dp,  240.7165_dp,    0.0000_dp,    0.0000_dp, &
      &  226.2156_dp,  223.7942_dp,  219.1285_dp,    0.0000_dp,    0.0000_dp, &
      &  178.8926_dp,  176.9365_dp,  173.2779_dp,    0.0000_dp,    0.0000_dp, &
      &  141.8517_dp,  140.2618_dp,  137.3907_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(133 * maxRef * maxRef + 1 : 152 * maxRef * maxRef) = [ &
      &  200.5374_dp,  198.3266_dp,  194.2314_dp,    0.0000_dp,    0.0000_dp, &
      &  186.0051_dp,  183.9483_dp,  180.1622_dp,    0.0000_dp,    0.0000_dp, &
      &  173.5798_dp,  171.6537_dp,  168.1302_dp,    0.0000_dp,    0.0000_dp, &
      &  168.8225_dp,  166.9597_dp,  163.5352_dp,    0.0000_dp,    0.0000_dp, &
      &  140.9955_dp,  139.4086_dp,  136.5758_dp,    0.0000_dp,    0.0000_dp, &
      &  159.4898_dp,  157.6971_dp,  154.4935_dp,    0.0000_dp,    0.0000_dp, &
      &  154.9229_dp,  153.1823_dp,  150.0748_dp,    0.0000_dp,    0.0000_dp, &
      &  152.6470_dp,  150.9364_dp,  147.8750_dp,    0.0000_dp,    0.0000_dp, &
      &  142.2982_dp,  140.6958_dp,  137.8540_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  134.0066_dp,  132.4922_dp,  129.8413_dp,    0.0000_dp,    0.0000_dp, &
      &  132.4922_dp,  130.9965_dp,  128.3764_dp,    0.0000_dp,    0.0000_dp, &
      &  129.8413_dp,  128.3764_dp,  125.8109_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   16.7052_dp,   16.5273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   26.0170_dp,   25.7414_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   11.6302_dp,   11.5092_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  210.6626_dp,  208.7743_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   77.9865_dp,   77.2023_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  132.9808_dp,  131.6498_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  100.7212_dp,   99.6995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   69.0069_dp,   68.2806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   94.7612_dp,   93.7743_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   80.7272_dp,   79.8833_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.6331_dp,   67.9132_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   53.7343_dp,   53.1638_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.8102_dp,   50.2708_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   66.8407_dp,   66.1354_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   62.7465_dp,   62.0864_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   52.0208_dp,   51.4721_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   48.7448_dp,   48.2328_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   40.8516_dp,   40.4209_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   48.2624_dp,   47.7540_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   45.0952_dp,   44.6217_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.4816_dp,   42.0367_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   37.6295_dp,   37.2381_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   37.3688_dp,   36.9783_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   33.7313_dp,   33.3801_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.9918_dp,   29.6822_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   28.8445_dp,   28.5468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   23.9505_dp,   23.7059_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.5121_dp,   22.2832_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  253.5136_dp,  251.2326_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  113.1025_dp,  111.9829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  208.4978_dp,  206.4521_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  174.1399_dp,  172.4148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  119.3911_dp,  118.1567_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.7594_dp,  195.7647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  180.5430_dp,  178.7127_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.0669_dp,  142.5806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  115.4552_dp,  114.2405_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  161.8686_dp,  160.1879_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  150.5231_dp,  148.9559_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  140.7824_dp,  139.3125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  136.8460_dp,  135.4234_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  115.3542_dp,  114.1358_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  130.4725_dp,  129.0956_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  126.8312_dp,  125.4930_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.9162_dp,  123.6011_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  116.8355_dp,  115.6006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  110.7006_dp,  109.5270_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.4486_dp,  108.2893_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  107.2964_dp,  106.1606_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(152 * maxRef * maxRef + 1 : 171 * maxRef * maxRef) = [ &
      &   92.3460_dp,   91.3667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   91.3667_dp,   90.3985_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   13.8700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   21.3055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    9.9130_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  163.5497_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   62.2205_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.7229_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   80.7174_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.1517_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.7949_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   65.7580_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.1926_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.4613_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.1324_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   55.0898_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   51.7684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   43.2064_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   40.5441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   34.3050_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   40.3435_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   37.7522_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.6334_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   31.6804_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   31.5783_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   28.6000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   25.5514_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   24.6178_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.6394_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   19.3774_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.3440_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   90.1735_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  165.1006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.5190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   96.3813_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  157.9595_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.5798_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  116.2817_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   94.0428_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  130.8927_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  122.0012_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  114.3348_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.1044_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   94.4027_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  106.7698_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  103.8653_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  102.2701_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   95.9336_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   91.4014_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   90.3716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   88.6268_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.9383_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.1256_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.6462_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(171 * maxRef * maxRef + 1 : 190 * maxRef * maxRef) = [ &
      &   76.2376_dp,   29.3000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  134.4441_dp,   48.4499_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.0411_dp,   18.6655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2387.1574_dp,  560.2954_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  555.1516_dp,  168.7412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  972.3197_dp,  292.2224_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  694.8596_dp,  213.4444_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  379.4225_dp,  132.4751_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  554.1980_dp,  186.9566_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  452.5193_dp,  155.7781_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.1535_dp,  129.5963_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.2237_dp,   95.8497_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  236.1802_dp,   89.8569_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  332.2694_dp,  121.9292_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  314.5800_dp,  114.5723_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  242.6366_dp,   92.0240_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  227.6600_dp,   86.1481_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  172.9685_dp,   69.0434_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.2001_dp,   83.2153_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  199.4405_dp,   77.6317_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  185.2181_dp,   72.6818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  162.0160_dp,   63.8686_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  152.1420_dp,   61.9948_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  134.6508_dp,   55.4032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  116.7069_dp,   48.6380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.6439_dp,   46.3520_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.3688_dp,   37.1982_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.0861_dp,   35.2969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2798.6124_dp,  666.0357_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  840.8299_dp,  249.2200_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1642.0587_dp,  473.7661_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1327.3176_dp,  387.3349_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  723.6346_dp,  240.1725_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1387.9361_dp,  424.8926_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1233.2237_dp,  382.6422_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  894.6228_dp,  291.6028_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  628.9537_dp,  220.7264_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  967.6283_dp,  322.8233_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  881.2624_dp,  296.9409_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  805.4924_dp,  274.8056_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  798.8390_dp,  268.9778_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  602.2116_dp,  215.8797_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  684.4968_dp,  244.5729_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  663.1184_dp,  237.2191_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  659.2394_dp,  234.4921_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  595.1033_dp,  215.8467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  535.2384_dp,  199.4672_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.5192_dp,  197.3806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  519.3802_dp,  193.3412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  413.1512_dp,  160.3304_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  409.5105_dp,  158.7330_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  322.1155_dp,  129.4283_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4983.5009_dp, 1112.2277_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1112.2277_dp,  338.0207_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(190 * maxRef * maxRef + 1 : 209 * maxRef * maxRef) = [ &
      &   65.8180_dp,   52.9871_dp,   30.6783_dp,    0.0000_dp,    0.0000_dp, &
      &  113.4857_dp,   90.7927_dp,   50.2368_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.0701_dp,   31.8796_dp,   19.7247_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1614.4719_dp, 1310.0314_dp,  525.3196_dp,    0.0000_dp,    0.0000_dp, &
      &  434.1576_dp,  345.6120_dp,  168.9009_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  757.9070_dp,  602.5661_dp,  291.7021_dp,    0.0000_dp,    0.0000_dp, &
      &  543.5842_dp,  434.0284_dp,  214.3171_dp,    0.0000_dp,    0.0000_dp, &
      &  316.5102_dp,  252.5778_dp,  136.5998_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  454.8439_dp,  362.6862_dp,  191.4135_dp,    0.0000_dp,    0.0000_dp, &
      &  373.8815_dp,  298.7264_dp,  160.0734_dp,    0.0000_dp,    0.0000_dp, &
      &  306.9078_dp,  245.9498_dp,  133.5995_dp,    0.0000_dp,    0.0000_dp, &
      &  217.9457_dp,  175.1434_dp,  100.0610_dp,    0.0000_dp,    0.0000_dp, &
      &  203.0679_dp,  163.3534_dp,   93.9194_dp,    0.0000_dp,    0.0000_dp, &
      &  281.7035_dp,  225.9629_dp,  126.6836_dp,    0.0000_dp,    0.0000_dp, &
      &  265.1206_dp,  212.8957_dp,  118.8255_dp,    0.0000_dp,    0.0000_dp, &
      &  208.0223_dp,  167.4145_dp,   96.0553_dp,    0.0000_dp,    0.0000_dp, &
      &  194.6719_dp,  156.7702_dp,   89.8285_dp,    0.0000_dp,    0.0000_dp, &
      &  150.8203_dp,  122.0707_dp,   72.5524_dp,    0.0000_dp,    0.0000_dp, &
      &  184.5498_dp,  148.9793_dp,   87.1906_dp,    0.0000_dp,    0.0000_dp, &
      &  172.0455_dp,  139.0219_dp,   81.2525_dp,    0.0000_dp,    0.0000_dp, &
      &  160.2750_dp,  129.5841_dp,   76.1365_dp,    0.0000_dp,    0.0000_dp, &
      &  140.0273_dp,  113.4161_dp,   66.8706_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  133.3847_dp,  108.2718_dp,   65.2525_dp,    0.0000_dp,    0.0000_dp, &
      &  118.2411_dp,   96.1584_dp,   58.3423_dp,    0.0000_dp,    0.0000_dp, &
      &  102.7126_dp,   83.7279_dp,   51.2425_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   97.0824_dp,   79.2218_dp,   48.9331_dp,    0.0000_dp,    0.0000_dp, &
      &   75.5850_dp,   62.0580_dp,   39.4048_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.3072_dp,   59.2779_dp,   37.3258_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1907.7081_dp, 1545.7064_dp,  626.5526_dp,    0.0000_dp,    0.0000_dp, &
      &  648.3105_dp,  516.6917_dp,  247.8223_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1252.5932_dp,  997.2486_dp,  468.1527_dp,    0.0000_dp,    0.0000_dp, &
      & 1013.7695_dp,  809.2182_dp,  384.1308_dp,    0.0000_dp,    0.0000_dp, &
      &  590.8737_dp,  470.3059_dp,  245.0983_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1088.8420_dp,  866.9517_dp,  426.0568_dp,    0.0000_dp,    0.0000_dp, &
      &  973.0493_dp,  775.0832_dp,  384.8555_dp,    0.0000_dp,    0.0000_dp, &
      &  721.4089_dp,  575.4721_dp,  296.5572_dp,    0.0000_dp,    0.0000_dp, &
      &  525.8367_dp,  419.7129_dp,  227.8108_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  790.6347_dp,  630.0456_dp,  329.8171_dp,    0.0000_dp,    0.0000_dp, &
      &  722.5111_dp,  576.2962_dp,  303.9725_dp,    0.0000_dp,    0.0000_dp, &
      &  664.0977_dp,  529.8946_dp,  281.9814_dp,    0.0000_dp,    0.0000_dp, &
      &  653.4174_dp,  521.7535_dp,  275.2242_dp,    0.0000_dp,    0.0000_dp, &
      &  507.2703_dp,  405.6765_dp,  223.6162_dp,    0.0000_dp,    0.0000_dp, &
      &  575.4627_dp,  460.2635_dp,  253.1519_dp,    0.0000_dp,    0.0000_dp, &
      &  557.4255_dp,  446.0170_dp,  245.5654_dp,    0.0000_dp,    0.0000_dp, &
      &  552.4889_dp,  442.0702_dp,  242.4454_dp,    0.0000_dp,    0.0000_dp, &
      &  502.9654_dp,  402.8286_dp,  223.9566_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  456.8443_dp,  366.7711_dp,  207.8531_dp,    0.0000_dp,    0.0000_dp, &
      &  452.3745_dp,  363.1951_dp,  205.6006_dp,    0.0000_dp,    0.0000_dp, &
      &  442.8735_dp,  355.6161_dp,  201.3895_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  357.4737_dp,  288.2182_dp,  168.0065_dp,    0.0000_dp,    0.0000_dp, &
      &  354.0935_dp,  285.4905_dp,  166.2878_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.6823_dp,  228.0892_dp,  136.1638_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3240.4393_dp, 2661.0830_dp, 1030.5394_dp,    0.0000_dp,    0.0000_dp, &
      &  860.1771_dp,  688.9911_dp,  337.7795_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(209 * maxRef * maxRef + 1 : 228 * maxRef * maxRef) = [ &
      & 2352.6862_dp, 1886.4221_dp,  832.2379_dp,    0.0000_dp,    0.0000_dp, &
      & 1886.4221_dp, 1525.1891_dp,  664.0620_dp,    0.0000_dp,    0.0000_dp, &
      &  832.2379_dp,  664.0620_dp,  343.3343_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   54.9669_dp,   29.9953_dp,   29.8615_dp,    0.0000_dp,    0.0000_dp, &
      &   94.2083_dp,   48.9753_dp,   48.4929_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   32.8995_dp,   19.3841_dp,   19.4304_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1278.1183_dp,  504.8268_dp,  481.9586_dp,    0.0000_dp,    0.0000_dp, &
      &  354.0317_dp,  163.6446_dp,  159.5715_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  617.3358_dp,  282.4420_dp,  275.0061_dp,    0.0000_dp,    0.0000_dp, &
      &  443.7190_dp,  207.8107_dp,  203.0678_dp,    0.0000_dp,    0.0000_dp, &
      &  261.9443_dp,  132.9868_dp,  131.2980_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  375.0418_dp,  186.1419_dp,  183.2413_dp,    0.0000_dp,    0.0000_dp, &
      &  308.8721_dp,  155.8228_dp,  153.7050_dp,    0.0000_dp,    0.0000_dp, &
      &  253.9534_dp,  130.1725_dp,  128.6716_dp,    0.0000_dp,    0.0000_dp, &
      &  181.6834_dp,   97.7550_dp,   97.1632_dp,    0.0000_dp,    0.0000_dp, &
      &  169.4186_dp,   91.7951_dp,   91.3072_dp,    0.0000_dp,    0.0000_dp, &
      &  234.1950_dp,  123.6379_dp,  122.6115_dp,    0.0000_dp,    0.0000_dp, &
      &  220.1854_dp,  115.9726_dp,  114.9710_dp,    0.0000_dp,    0.0000_dp, &
      &  173.4630_dp,   93.8911_dp,   93.3603_dp,    0.0000_dp,    0.0000_dp, &
      &  162.2609_dp,   87.8153_dp,   87.3048_dp,    0.0000_dp,    0.0000_dp, &
      &  126.3491_dp,   71.0854_dp,   70.9670_dp,    0.0000_dp,    0.0000_dp, &
      &  154.2947_dp,   85.3404_dp,   85.0464_dp,    0.0000_dp,    0.0000_dp, &
      &  143.7658_dp,   79.5383_dp,   79.2586_dp,    0.0000_dp,    0.0000_dp, &
      &  134.0314_dp,   74.5587_dp,   74.3297_dp,    0.0000_dp,    0.0000_dp, &
      &  117.1112_dp,   65.5203_dp,   65.3415_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.9359_dp,   64.0072_dp,   63.9839_dp,    0.0000_dp,    0.0000_dp, &
      &   99.2939_dp,   57.2611_dp,   57.2805_dp,    0.0000_dp,    0.0000_dp, &
      &   86.3365_dp,   50.3373_dp,   50.3920_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.7242_dp,   48.0936_dp,   48.1892_dp,    0.0000_dp,    0.0000_dp, &
      &   63.8564_dp,   38.8097_dp,   38.9899_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   61.0154_dp,   36.7468_dp,   36.8800_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1512.5338_dp,  602.4929_dp,  575.7400_dp,    0.0000_dp,    0.0000_dp, &
      &  527.0917_dp,  239.9714_dp,  233.4674_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1015.5454_dp,  452.7600_dp,  439.2030_dp,    0.0000_dp,    0.0000_dp, &
      &  822.8957_dp,  371.7768_dp,  361.4502_dp,    0.0000_dp,    0.0000_dp, &
      &  486.4653_dp,  238.1540_dp,  234.0068_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  888.6025_dp,  412.9058_dp,  402.9357_dp,    0.0000_dp,    0.0000_dp, &
      &  795.2206_dp,  373.1845_dp,  364.6668_dp,    0.0000_dp,    0.0000_dp, &
      &  592.7050_dp,  288.0834_dp,  282.8628_dp,    0.0000_dp,    0.0000_dp, &
      &  435.4205_dp,  221.8339_dp,  219.1014_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  651.2040_dp,  320.5896_dp,  315.2655_dp,    0.0000_dp,    0.0000_dp, &
      &  595.6725_dp,  295.6101_dp,  291.0037_dp,    0.0000_dp,    0.0000_dp, &
      &  548.2115_dp,  274.3493_dp,  270.3548_dp,    0.0000_dp,    0.0000_dp, &
      &  538.5754_dp,  267.7120_dp,  263.5853_dp,    0.0000_dp,    0.0000_dp, &
      &  420.8895_dp,  217.9628_dp,  215.7027_dp,    0.0000_dp,    0.0000_dp, &
      &  477.2797_dp,  246.7358_dp,  244.1234_dp,    0.0000_dp,    0.0000_dp, &
      &  462.3511_dp,  239.3701_dp,  236.8740_dp,    0.0000_dp,    0.0000_dp, &
      &  457.9608_dp,  236.2965_dp,  233.7282_dp,    0.0000_dp,    0.0000_dp, &
      &  417.7400_dp,  218.4319_dp,  216.4038_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  380.4107_dp,  202.9672_dp,  201.5466_dp,    0.0000_dp,    0.0000_dp, &
      &  376.6142_dp,  200.7622_dp,  199.3322_dp,    0.0000_dp,    0.0000_dp, &
      &  368.7131_dp,  196.6601_dp,  195.2671_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  298.7686_dp,  164.3655_dp,  163.7482_dp,    0.0000_dp,    0.0000_dp, &
      &  295.9028_dp,  162.6806_dp,  162.0542_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  236.1417_dp,  133.4404_dp,  133.2865_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(228 * maxRef * maxRef + 1 : 247 * maxRef * maxRef) = [ &
      & 2549.9412_dp,  988.5881_dp,  943.3962_dp,    0.0000_dp,    0.0000_dp, &
      &  700.7281_dp,  327.7092_dp,  320.0043_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1888.7902_dp,  803.2380_dp,  774.1368_dp,    0.0000_dp,    0.0000_dp, &
      & 1510.1427_dp,  640.7076_dp,  618.1676_dp,    0.0000_dp,    0.0000_dp, &
      &  683.8786_dp,  333.6780_dp,  327.6458_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1522.4676_dp,  660.6780_dp,  638.4869_dp,    0.0000_dp,    0.0000_dp, &
      &  660.6780_dp,  324.5068_dp,  318.8004_dp,    0.0000_dp,    0.0000_dp, &
      &  638.4869_dp,  318.8004_dp,  313.9846_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   53.6875_dp,   29.0009_dp,   30.2593_dp,    0.0000_dp,    0.0000_dp, &
      &   91.3334_dp,   47.2755_dp,   49.0983_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   32.5316_dp,   18.8084_dp,   19.7205_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1192.9128_dp,  488.2931_dp,  486.2098_dp,    0.0000_dp,    0.0000_dp, &
      &  337.3896_dp,  157.6667_dp,  161.3013_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.4898_dp,  272.0201_dp,  277.9371_dp,    0.0000_dp,    0.0000_dp, &
      &  423.6217_dp,  200.3669_dp,  205.3134_dp,    0.0000_dp,    0.0000_dp, &
      &  253.0543_dp,  128.2862_dp,  132.8881_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  361.0865_dp,  179.5077_dp,  185.4074_dp,    0.0000_dp,    0.0000_dp, &
      &  298.0871_dp,  150.3520_dp,  155.5661_dp,    0.0000_dp,    0.0000_dp, &
      &  245.6647_dp,  125.7017_dp,  130.2662_dp,    0.0000_dp,    0.0000_dp, &
      &  177.0660_dp,   94.4758_dp,   98.4363_dp,    0.0000_dp,    0.0000_dp, &
      &  165.2912_dp,   88.7389_dp,   92.5154_dp,    0.0000_dp,    0.0000_dp, &
      &  227.5723_dp,  119.4376_dp,  124.1849_dp,    0.0000_dp,    0.0000_dp, &
      &  213.8682_dp,  112.0523_dp,  116.4471_dp,    0.0000_dp,    0.0000_dp, &
      &  169.2064_dp,   90.7838_dp,   94.6013_dp,    0.0000_dp,    0.0000_dp, &
      &  158.2705_dp,   84.9253_dp,   88.4703_dp,    0.0000_dp,    0.0000_dp, &
      &  124.0013_dp,   68.8297_dp,   71.9595_dp,    0.0000_dp,    0.0000_dp, &
      &  151.0199_dp,   82.5834_dp,   86.2109_dp,    0.0000_dp,    0.0000_dp, &
      &  140.7115_dp,   76.9834_dp,   80.3464_dp,    0.0000_dp,    0.0000_dp, &
      &  131.3015_dp,   72.1828_dp,   75.3609_dp,    0.0000_dp,    0.0000_dp, &
      &  114.8256_dp,   63.4645_dp,   66.2607_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  110.1452_dp,   62.0309_dp,   64.9037_dp,    0.0000_dp,    0.0000_dp, &
      &   97.8345_dp,   55.5160_dp,   58.1127_dp,    0.0000_dp,    0.0000_dp, &
      &   85.2219_dp,   48.8396_dp,   51.1410_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   80.7975_dp,   46.6765_dp,   48.9131_dp,    0.0000_dp,    0.0000_dp, &
      &   63.4748_dp,   37.7222_dp,   39.6015_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.5594_dp,   35.7117_dp,   37.4554_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1413.1570_dp,  582.5270_dp,  580.9067_dp,    0.0000_dp,    0.0000_dp, &
      &  501.2393_dp,  231.2612_dp,  235.9718_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  962.9831_dp,  436.1091_dp,  443.7557_dp,    0.0000_dp,    0.0000_dp, &
      &  781.6993_dp,  358.3602_dp,  365.2736_dp,    0.0000_dp,    0.0000_dp, &
      &  467.4506_dp,  229.5754_dp,  236.7246_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  847.4615_dp,  397.8618_dp,  407.3331_dp,    0.0000_dp,    0.0000_dp, &
      &  759.4532_dp,  359.6497_dp,  368.7014_dp,    0.0000_dp,    0.0000_dp, &
      &  568.9077_dp,  277.8074_dp,  286.1317_dp,    0.0000_dp,    0.0000_dp, &
      &  420.8711_dp,  214.0193_dp,  221.7708_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  626.2353_dp,  309.1081_dp,  318.9550_dp,    0.0000_dp,    0.0000_dp, &
      &  573.4988_dp,  285.0957_dp,  294.4479_dp,    0.0000_dp,    0.0000_dp, &
      &  528.4593_dp,  264.6302_dp,  273.5887_dp,    0.0000_dp,    0.0000_dp, &
      &  518.6333_dp,  258.2554_dp,  266.7243_dp,    0.0000_dp,    0.0000_dp, &
      &  407.8160_dp,  210.3904_dp,  218.3896_dp,    0.0000_dp,    0.0000_dp, &
      &  462.3247_dp,  238.1644_dp,  247.1603_dp,    0.0000_dp,    0.0000_dp, &
      &  447.9590_dp,  231.0773_dp,  239.8294_dp,    0.0000_dp,    0.0000_dp, &
      &  443.4723_dp,  228.1087_dp,  236.6374_dp,    0.0000_dp,    0.0000_dp, &
      &  405.3462_dp,  210.9228_dp,  219.1391_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.2631_dp,  196.1128_dp,  204.1620_dp,    0.0000_dp,    0.0000_dp, &
      &  366.5151_dp,  193.9845_dp,  201.9181_dp,    0.0000_dp,    0.0000_dp, &
      &  358.8554_dp,  190.0290_dp,  197.8037_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(247 * maxRef * maxRef + 1 : 266 * maxRef * maxRef) = [ &
      &  292.1853_dp,  158.9892_dp,  165.9632_dp,    0.0000_dp,    0.0000_dp, &
      &  289.3512_dp,  157.3604_dp,  164.2459_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  231.9208_dp,  129.2180_dp,  135.1586_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2374.6690_dp,  960.3376_dp,  951.6268_dp,    0.0000_dp,    0.0000_dp, &
      &  668.7546_dp,  316.2518_dp,  323.6190_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1779.5162_dp,  774.6863_dp,  781.7679_dp,    0.0000_dp,    0.0000_dp, &
      & 1422.3244_dp,  619.4301_dp,  624.3036_dp,    0.0000_dp,    0.0000_dp, &
      &  656.7474_dp,  321.8441_dp,  331.4851_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1438.2841_dp,  636.8706_dp,  644.9388_dp,    0.0000_dp,    0.0000_dp, &
      &  635.0457_dp,  312.9202_dp,  322.5705_dp,    0.0000_dp,    0.0000_dp, &
      &  615.2494_dp,  307.5913_dp,  317.7589_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1361.9185_dp,  612.2225_dp,  621.6228_dp,    0.0000_dp,    0.0000_dp, &
      &  612.2225_dp,  301.9988_dp,  311.2408_dp,    0.0000_dp,    0.0000_dp, &
      &  621.6228_dp,  311.2408_dp,  321.5890_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.4819_dp,   28.4067_dp,   27.6880_dp,    0.0000_dp,    0.0000_dp, &
      &   83.8313_dp,   46.3089_dp,   44.8842_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   30.1851_dp,   18.4359_dp,   18.1127_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1069.0426_dp,  479.1402_dp,  448.2923_dp,    0.0000_dp,    0.0000_dp, &
      &  306.5482_dp,  154.5762_dp,  147.6410_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.3499_dp,  266.7050_dp,  254.3767_dp,    0.0000_dp,    0.0000_dp, &
      &  385.2754_dp,  196.4027_dp,  187.9531_dp,    0.0000_dp,    0.0000_dp, &
      &  231.8065_dp,  125.6778_dp,  121.4601_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.1023_dp,  175.8909_dp,  169.5221_dp,    0.0000_dp,    0.0000_dp, &
      &  272.8747_dp,  147.3153_dp,  142.2728_dp,    0.0000_dp,    0.0000_dp, &
      &  225.1783_dp,  123.1529_dp,  119.1771_dp,    0.0000_dp,    0.0000_dp, &
      &  162.9990_dp,   92.5451_dp,   90.0574_dp,    0.0000_dp,    0.0000_dp, &
      &  152.2500_dp,   86.9264_dp,   84.6570_dp,    0.0000_dp,    0.0000_dp, &
      &  209.1450_dp,  117.0098_dp,  113.6129_dp,    0.0000_dp,    0.0000_dp, &
      &  196.4885_dp,  109.7797_dp,  106.5711_dp,    0.0000_dp,    0.0000_dp, &
      &  155.8387_dp,   88.9429_dp,   86.6060_dp,    0.0000_dp,    0.0000_dp, &
      &  145.7580_dp,   83.2111_dp,   81.0229_dp,    0.0000_dp,    0.0000_dp, &
      &  114.5870_dp,   67.4383_dp,   65.9473_dp,    0.0000_dp,    0.0000_dp, &
      &  139.3512_dp,   80.9144_dp,   78.9761_dp,    0.0000_dp,    0.0000_dp, &
      &  129.8297_dp,   75.4303_dp,   73.6275_dp,    0.0000_dp,    0.0000_dp, &
      &  121.2119_dp,   70.7347_dp,   69.0809_dp,    0.0000_dp,    0.0000_dp, &
      &  106.0489_dp,   62.2020_dp,   60.7834_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  101.9325_dp,   60.7933_dp,   59.5445_dp,    0.0000_dp,    0.0000_dp, &
      &   90.6007_dp,   54.4109_dp,   53.3400_dp,    0.0000_dp,    0.0000_dp, &
      &   79.0003_dp,   47.8815_dp,   46.9875_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   74.9671_dp,   45.7630_dp,   44.9493_dp,    0.0000_dp,    0.0000_dp, &
      &   59.0665_dp,   36.9962_dp,   36.4513_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.3108_dp,   35.0290_dp,   34.4789_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1267.3139_dp,  571.6455_dp,  535.4483_dp,    0.0000_dp,    0.0000_dp, &
      &  454.8030_dp,  226.7886_dp,  216.1867_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  872.2579_dp,  427.7156_dp,  406.5642_dp,    0.0000_dp,    0.0000_dp, &
      &  708.7684_dp,  351.3928_dp,  334.6783_dp,    0.0000_dp,    0.0000_dp, &
      &  426.8815_dp,  224.9822_dp,  216.4395_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  770.3432_dp,  390.0432_dp,  372.8019_dp,    0.0000_dp,    0.0000_dp, &
      &  690.9156_dp,  352.5575_dp,  337.4128_dp,    0.0000_dp,    0.0000_dp, &
      &  519.1328_dp,  272.2496_dp,  261.7283_dp,    0.0000_dp,    0.0000_dp, &
      &  385.6611_dp,  209.6742_dp,  202.7168_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(266 * maxRef * maxRef + 1 : 285 * maxRef * maxRef) = [ &
      &  572.1129_dp,  302.8985_dp,  291.6318_dp,    0.0000_dp,    0.0000_dp, &
      &  524.2808_dp,  279.3563_dp,  269.2413_dp,    0.0000_dp,    0.0000_dp, &
      &  483.4599_dp,  259.2925_dp,  250.1591_dp,    0.0000_dp,    0.0000_dp, &
      &  474.1585_dp,  253.0661_dp,  243.9654_dp,    0.0000_dp,    0.0000_dp, &
      &  374.2069_dp,  206.1062_dp,  199.6661_dp,    0.0000_dp,    0.0000_dp, &
      &  424.1470_dp,  233.3176_dp,  225.9849_dp,    0.0000_dp,    0.0000_dp, &
      &  411.0121_dp,  226.3767_dp,  219.3033_dp,    0.0000_dp,    0.0000_dp, &
      &  406.7672_dp,  223.4784_dp,  216.4101_dp,    0.0000_dp,    0.0000_dp, &
      &  372.2326_dp,  206.6274_dp,  200.4053_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  340.6020_dp,  192.1113_dp,  186.7665_dp,    0.0000_dp,    0.0000_dp, &
      &  337.1250_dp,  190.0299_dp,  184.7241_dp,    0.0000_dp,    0.0000_dp, &
      &  330.0938_dp,  186.1576_dp,  180.9710_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  269.4863_dp,  155.7507_dp,  151.9416_dp,    0.0000_dp,    0.0000_dp, &
      &  266.8557_dp,  154.1576_dp,  150.3757_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.4002_dp,  126.6027_dp,  123.8635_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2124.1985_dp,  941.9837_dp,  879.7445_dp,    0.0000_dp,    0.0000_dp, &
      &  607.9951_dp,  310.1108_dp,  296.7431_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1604.9178_dp,  760.0718_dp,  718.1630_dp,    0.0000_dp,    0.0000_dp, &
      & 1282.1610_dp,  607.5571_dp,  574.2351_dp,    0.0000_dp,    0.0000_dp, &
      &  599.4521_dp,  315.4686_dp,  303.3970_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1299.5197_dp,  624.7801_dp,  591.8695_dp,    0.0000_dp,    0.0000_dp, &
      &  579.9584_dp,  306.7389_dp,  295.2374_dp,    0.0000_dp,    0.0000_dp, &
      &  562.7180_dp,  301.4420_dp,  290.7309_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1232.3235_dp,  600.5221_dp,  570.2282_dp,    0.0000_dp,    0.0000_dp, &
      &  559.1157_dp,  296.0029_dp,  284.9497_dp,    0.0000_dp,    0.0000_dp, &
      &  568.6316_dp,  305.0201_dp,  294.2417_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1116.0984_dp,  548.3908_dp,  521.4602_dp,    0.0000_dp,    0.0000_dp, &
      &  548.3908_dp,  290.1436_dp,  279.2755_dp,    0.0000_dp,    0.0000_dp, &
      &  521.4602_dp,  279.2755_dp,  269.3349_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.1221_dp,   28.0562_dp,   21.3134_dp,    0.0000_dp,    0.0000_dp, &
      &   65.9837_dp,   45.8173_dp,   33.9427_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   24.1463_dp,   18.1754_dp,   14.4051_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  844.0936_dp,  480.5944_dp,  314.2807_dp,    0.0000_dp,    0.0000_dp, &
      &  240.5033_dp,  153.7539_dp,  107.7527_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  418.1635_dp,  265.4088_dp,  184.9079_dp,    0.0000_dp,    0.0000_dp, &
      &  302.6061_dp,  195.2317_dp,  137.8919_dp,    0.0000_dp,    0.0000_dp, &
      &  182.1558_dp,  124.4645_dp,   91.0985_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  259.3021_dp,  174.3719_dp,  126.3736_dp,    0.0000_dp,    0.0000_dp, &
      &  214.6360_dp,  145.9477_dp,  106.7244_dp,    0.0000_dp,    0.0000_dp, &
      &  177.4000_dp,  121.9340_dp,   89.9610_dp,    0.0000_dp,    0.0000_dp, &
      &  128.7233_dp,   91.4528_dp,   68.9958_dp,    0.0000_dp,    0.0000_dp, &
      &  120.3278_dp,   85.8806_dp,   65.0349_dp,    0.0000_dp,    0.0000_dp, &
      &  164.9882_dp,  115.7182_dp,   86.5464_dp,    0.0000_dp,    0.0000_dp, &
      &  155.1171_dp,  108.5820_dp,   81.2335_dp,    0.0000_dp,    0.0000_dp, &
      &  123.2591_dp,   87.8889_dp,   66.6131_dp,    0.0000_dp,    0.0000_dp, &
      &  115.3697_dp,   82.2331_dp,   62.3963_dp,    0.0000_dp,    0.0000_dp, &
      &   91.0233_dp,   66.5568_dp,   51.4610_dp,    0.0000_dp,    0.0000_dp, &
      &  110.4861_dp,   79.9024_dp,   61.2471_dp,    0.0000_dp,    0.0000_dp, &
      &  103.0189_dp,   74.4890_dp,   57.1766_dp,    0.0000_dp,    0.0000_dp, &
      &   96.2547_dp,   69.8471_dp,   53.7726_dp,    0.0000_dp,    0.0000_dp, &
      &   84.3631_dp,   61.4202_dp,   47.5033_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.2000_dp,   59.9829_dp,   46.8138_dp,    0.0000_dp,    0.0000_dp, &
      &   72.2872_dp,   53.6733_dp,   42.1046_dp,    0.0000_dp,    0.0000_dp, &
      &   63.1867_dp,   47.2297_dp,   37.3024_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.0099_dp,   45.1290_dp,   35.7811_dp,    0.0000_dp,    0.0000_dp, &
      &   47.5318_dp,   36.4594_dp,   29.3923_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(285 * maxRef * maxRef + 1 : 304 * maxRef * maxRef) = [ &
      &   45.2820_dp,   34.5350_dp,   27.7372_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1000.4652_dp,  573.1436_dp,  376.4906_dp,    0.0000_dp,    0.0000_dp, &
      &  357.1508_dp,  225.7837_dp,  157.3830_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  684.4015_dp,  426.2276_dp,  293.6777_dp,    0.0000_dp,    0.0000_dp, &
      &  556.6127_dp,  349.9328_dp,  243.0049_dp,    0.0000_dp,    0.0000_dp, &
      &  335.0118_dp,  223.1855_dp,  160.5554_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  604.4071_dp,  387.8695_dp,  272.5015_dp,    0.0000_dp,    0.0000_dp, &
      &  542.2459_dp,  350.4296_dp,  247.4372_dp,    0.0000_dp,    0.0000_dp, &
      &  407.7288_dp,  270.1623_dp,  193.9815_dp,    0.0000_dp,    0.0000_dp, &
      &  303.1266_dp,  207.6299_dp,  152.2308_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  449.2216_dp,  300.3928_dp,  216.8001_dp,    0.0000_dp,    0.0000_dp, &
      &  411.8879_dp,  276.9512_dp,  200.7344_dp,    0.0000_dp,    0.0000_dp, &
      &  379.9385_dp,  256.9682_dp,  187.0036_dp,    0.0000_dp,    0.0000_dp, &
      &  372.7838_dp,  250.8827_dp,  182.1854_dp,    0.0000_dp,    0.0000_dp, &
      &  294.5037_dp,  203.9633_dp,  150.8344_dp,    0.0000_dp,    0.0000_dp, &
      &  333.8380_dp,  230.9090_dp,  170.6767_dp,    0.0000_dp,    0.0000_dp, &
      &  323.5914_dp,  224.0303_dp,  165.7628_dp,    0.0000_dp,    0.0000_dp, &
      &  320.2734_dp,  221.1986_dp,  163.4733_dp,    0.0000_dp,    0.0000_dp, &
      &  293.2680_dp,  204.4081_dp,  152.0010_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  268.8029_dp,  189.9052_dp,  142.6795_dp,    0.0000_dp,    0.0000_dp, &
      &  266.0762_dp,  187.8570_dp,  141.1092_dp,    0.0000_dp,    0.0000_dp, &
      &  260.5647_dp,  184.0280_dp,  138.2944_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.3668_dp,  153.8057_dp,  117.4258_dp,    0.0000_dp,    0.0000_dp, &
      &  211.2928_dp,  152.2383_dp,  116.2100_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  170.3465_dp,  124.9265_dp,   96.7593_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1683.6701_dp,  945.8700_dp,  617.0566_dp,    0.0000_dp,    0.0000_dp, &
      &  478.9744_dp,  308.3901_dp,  218.9044_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1263.0809_dp,  759.3684_dp,  513.7768_dp,    0.0000_dp,    0.0000_dp, &
      & 1011.5955_dp,  607.0700_dp,  412.0610_dp,    0.0000_dp,    0.0000_dp, &
      &  471.3832_dp,  313.0538_dp,  225.6881_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1021.6871_dp,  623.5332_dp,  425.4745_dp,    0.0000_dp,    0.0000_dp, &
      &  456.2023_dp,  304.3094_dp,  220.1013_dp,    0.0000_dp,    0.0000_dp, &
      &  442.6736_dp,  298.8190_dp,  217.6494_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  968.8565_dp,  598.7981_dp,  412.1387_dp,    0.0000_dp,    0.0000_dp, &
      &  440.0762_dp,  293.6598_dp,  212.6450_dp,    0.0000_dp,    0.0000_dp, &
      &  447.3763_dp,  302.3474_dp,  220.4170_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  877.3855_dp,  546.5244_dp,  378.0660_dp,    0.0000_dp,    0.0000_dp, &
      &  431.6683_dp,  287.8657_dp,  208.4094_dp,    0.0000_dp,    0.0000_dp, &
      &  410.5661_dp,  276.8819_dp,  201.8641_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  690.7425_dp,  430.2353_dp,  298.4894_dp,    0.0000_dp,    0.0000_dp, &
      &  430.2353_dp,  285.6905_dp,  206.3426_dp,    0.0000_dp,    0.0000_dp, &
      &  298.4894_dp,  206.3426_dp,  153.3435_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   43.0028_dp,   26.1810_dp,   18.4694_dp,    0.0000_dp,    0.0000_dp, &
      &   72.4111_dp,   42.3927_dp,   29.1208_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   26.4978_dp,   17.1526_dp,   12.6985_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  891.9756_dp,  416.0836_dp,  256.5689_dp,    0.0000_dp,    0.0000_dp, &
      &  260.9019_dp,  138.7491_dp,   90.4231_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  453.3686_dp,  238.9606_dp,  154.7687_dp,    0.0000_dp,    0.0000_dp, &
      &  328.4012_dp,  176.6817_dp,  116.1032_dp,    0.0000_dp,    0.0000_dp, &
      &  199.6426_dp,  114.6465_dp,   77.7823_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(304 * maxRef * maxRef + 1 : 323 * maxRef * maxRef) = [ &
      &  283.4744_dp,  159.8492_dp,  107.4866_dp,    0.0000_dp,    0.0000_dp, &
      &  234.8000_dp,  134.2136_dp,   91.1091_dp,    0.0000_dp,    0.0000_dp, &
      &  194.1378_dp,  112.4648_dp,   77.0812_dp,    0.0000_dp,    0.0000_dp, &
      &  141.4071_dp,   85.1312_dp,   59.6313_dp,    0.0000_dp,    0.0000_dp, &
      &  132.1984_dp,   80.0375_dp,   56.2921_dp,    0.0000_dp,    0.0000_dp, &
      &  181.0035_dp,  107.3382_dp,   74.5511_dp,    0.0000_dp,    0.0000_dp, &
      &  169.9796_dp,  100.6463_dp,   69.9883_dp,    0.0000_dp,    0.0000_dp, &
      &  135.2994_dp,   81.8731_dp,   57.6893_dp,    0.0000_dp,    0.0000_dp, &
      &  126.5416_dp,   76.5846_dp,   54.0676_dp,    0.0000_dp,    0.0000_dp, &
      &   99.9768_dp,   62.3931_dp,   44.9179_dp,    0.0000_dp,    0.0000_dp, &
      &  121.3233_dp,   74.6998_dp,   53.2828_dp,    0.0000_dp,    0.0000_dp, &
      &  113.0248_dp,   69.6215_dp,   49.7698_dp,    0.0000_dp,    0.0000_dp, &
      &  105.6080_dp,   65.3415_dp,   46.8681_dp,    0.0000_dp,    0.0000_dp, &
      &   92.4636_dp,   57.4915_dp,   41.4862_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   89.1346_dp,   56.3622_dp,   41.0237_dp,    0.0000_dp,    0.0000_dp, &
      &   79.3047_dp,   50.4858_dp,   36.9692_dp,    0.0000_dp,    0.0000_dp, &
      &   69.2598_dp,   44.4888_dp,   32.8495_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   65.8112_dp,   42.5764_dp,   31.5582_dp,    0.0000_dp,    0.0000_dp, &
      &   52.0780_dp,   34.5481_dp,   26.0926_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.5959_dp,   32.6806_dp,   24.5957_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1058.5103_dp,  497.3582_dp,  307.9557_dp,    0.0000_dp,    0.0000_dp, &
      &  386.3426_dp,  202.9676_dp,  131.8257_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  739.0411_dp,  381.2968_dp,  244.7517_dp,    0.0000_dp,    0.0000_dp, &
      &  601.4499_dp,  314.0119_dp,  203.2173_dp,    0.0000_dp,    0.0000_dp, &
      &  366.0024_dp,  204.0446_dp,  136.1619_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  656.0738_dp,  350.4522_dp,  228.9125_dp,    0.0000_dp,    0.0000_dp, &
      &  589.1475_dp,  317.3287_dp,  208.2902_dp,    0.0000_dp,    0.0000_dp, &
      &  444.6283_dp,  246.5588_dp,  164.4045_dp,    0.0000_dp,    0.0000_dp, &
      &  332.3156_dp,  191.3897_dp,  130.0809_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  490.8156_dp,  274.9257_dp,  184.0945_dp,    0.0000_dp,    0.0000_dp, &
      &  450.2234_dp,  253.8800_dp,  170.7515_dp,    0.0000_dp,    0.0000_dp, &
      &  415.6112_dp,  235.9676_dp,  159.3286_dp,    0.0000_dp,    0.0000_dp, &
      &  407.2425_dp,  230.0182_dp,  155.1095_dp,    0.0000_dp,    0.0000_dp, &
      &  323.0948_dp,  188.5834_dp,  129.3324_dp,    0.0000_dp,    0.0000_dp, &
      &  366.1204_dp,  213.4101_dp,  146.3175_dp,    0.0000_dp,    0.0000_dp, &
      &  354.8432_dp,  207.0995_dp,  142.1674_dp,    0.0000_dp,    0.0000_dp, &
      &  351.0224_dp,  204.3319_dp,  140.1443_dp,    0.0000_dp,    0.0000_dp, &
      &  321.7679_dp,  189.3102_dp,  130.6248_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  295.1737_dp,  176.5165_dp,  123.1170_dp,    0.0000_dp,    0.0000_dp, &
      &  292.1257_dp,  174.5775_dp,  121.7541_dp,    0.0000_dp,    0.0000_dp, &
      &  286.0527_dp,  171.0306_dp,  119.3486_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  234.4533_dp,  143.7001_dp,  101.9746_dp,    0.0000_dp,    0.0000_dp, &
      &  232.1447_dp,  142.2151_dp,  100.9143_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  187.1728_dp,  117.2057_dp,   84.5107_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1768.0322_dp,  814.2468_dp,  503.8621_dp,    0.0000_dp,    0.0000_dp, &
      &  518.0583_dp,  278.7259_dp,  184.8170_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1351.4785_dp,  670.8337_dp,  425.2382_dp,    0.0000_dp,    0.0000_dp, &
      & 1079.1532_dp,  535.8149_dp,  341.7674_dp,    0.0000_dp,    0.0000_dp, &
      &  513.6516_dp,  285.8068_dp,  191.6322_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1097.1539_dp,  553.7931_dp,  353.3283_dp,    0.0000_dp,    0.0000_dp, &
      &  497.3474_dp,  278.1965_dp,  187.1364_dp,    0.0000_dp,    0.0000_dp, &
      &  483.5993_dp,  274.1703_dp,  185.5544_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1042.6625_dp,  534.1266_dp,  343.4916_dp,    0.0000_dp,    0.0000_dp, &
      &  479.4969_dp,  268.4657_dp,  180.9105_dp,    0.0000_dp,    0.0000_dp, &
      &  488.7888_dp,  277.5005_dp,  187.9838_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  945.6072_dp,  488.7947_dp,  315.7492_dp,    0.0000_dp,    0.0000_dp, &
      &  470.2529_dp,  263.1127_dp,  177.2993_dp,    0.0000_dp,    0.0000_dp, &
      &  448.0698_dp,  253.9370_dp,  172.1980_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(323 * maxRef * maxRef + 1 : 342 * maxRef * maxRef) = [ &
      &  743.3179_dp,  384.6931_dp,  249.6819_dp,    0.0000_dp,    0.0000_dp, &
      &  468.2961_dp,  260.7773_dp,  175.3867_dp,    0.0000_dp,    0.0000_dp, &
      &  326.3558_dp,  190.5528_dp,  131.8288_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  802.7484_dp,  420.4207_dp,  273.3937_dp,    0.0000_dp,    0.0000_dp, &
      &  420.4207_dp,  239.5554_dp,  162.6844_dp,    0.0000_dp,    0.0000_dp, &
      &  273.3937_dp,  162.6844_dp,  113.8463_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   33.9110_dp,   24.3452_dp,   18.0977_dp,    0.0000_dp,    0.0000_dp, &
      &   56.7993_dp,   39.2742_dp,   28.5665_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   21.1750_dp,   16.0560_dp,   12.4277_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  698.4539_dp,  379.2117_dp,  253.2432_dp,    0.0000_dp,    0.0000_dp, &
      &  203.6149_dp,  127.5492_dp,   88.9538_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  353.5358_dp,  219.4775_dp,  152.2986_dp,    0.0000_dp,    0.0000_dp, &
      &  256.6043_dp,  162.6108_dp,  114.1613_dp,    0.0000_dp,    0.0000_dp, &
      &  156.2845_dp,  106.0286_dp,   76.3494_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  221.7547_dp,  147.6298_dp,  105.5581_dp,    0.0000_dp,    0.0000_dp, &
      &  183.9737_dp,  124.1170_dp,   89.4354_dp,    0.0000_dp,    0.0000_dp, &
      &  152.3887_dp,  104.1437_dp,   75.6327_dp,    0.0000_dp,    0.0000_dp, &
      &  111.3569_dp,   79.0832_dp,   58.4515_dp,    0.0000_dp,    0.0000_dp, &
      &  104.1990_dp,   74.3930_dp,   55.1698_dp,    0.0000_dp,    0.0000_dp, &
      &  142.3467_dp,   99.5880_dp,   73.1079_dp,    0.0000_dp,    0.0000_dp, &
      &  133.7700_dp,   93.3848_dp,   68.6293_dp,    0.0000_dp,    0.0000_dp, &
      &  106.7326_dp,   76.1132_dp,   56.5429_dp,    0.0000_dp,    0.0000_dp, &
      &   99.9017_dp,   71.2110_dp,   52.9928_dp,    0.0000_dp,    0.0000_dp, &
      &   79.2630_dp,   58.1776_dp,   43.9902_dp,    0.0000_dp,    0.0000_dp, &
      &   95.9782_dp,   69.5638_dp,   52.2028_dp,    0.0000_dp,    0.0000_dp, &
      &   89.4868_dp,   64.8468_dp,   48.7558_dp,    0.0000_dp,    0.0000_dp, &
      &   83.6916_dp,   60.8920_dp,   45.9140_dp,    0.0000_dp,    0.0000_dp, &
      &   73.4185_dp,   53.6171_dp,   40.6376_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   70.8942_dp,   52.6351_dp,   40.1709_dp,    0.0000_dp,    0.0000_dp, &
      &   63.1865_dp,   47.1816_dp,   36.1919_dp,    0.0000_dp,    0.0000_dp, &
      &   55.3356_dp,   41.6270_dp,   32.1584_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   52.6310_dp,   39.8628_dp,   30.8925_dp,    0.0000_dp,    0.0000_dp, &
      &   41.8942_dp,   32.4321_dp,   25.5314_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.8659_dp,   30.6648_dp,   24.0741_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  828.8936_dp,  453.5444_dp,  303.9124_dp,    0.0000_dp,    0.0000_dp, &
      &  301.7401_dp,  186.4669_dp,  129.7332_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  576.5121_dp,  349.7116_dp,  240.9962_dp,    0.0000_dp,    0.0000_dp, &
      &  469.6437_dp,  288.3371_dp,  200.0041_dp,    0.0000_dp,    0.0000_dp, &
      &  285.9969_dp,  188.2525_dp,  133.7853_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  512.0823_dp,  322.2781_dp,  225.1663_dp,    0.0000_dp,    0.0000_dp, &
      &  460.0496_dp,  292.0247_dp,  204.8282_dp,    0.0000_dp,    0.0000_dp, &
      &  347.6394_dp,  227.4327_dp,  161.5357_dp,    0.0000_dp,    0.0000_dp, &
      &  260.2229_dp,  177.0542_dp,  127.6833_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.7445_dp,  253.7638_dp,  180.8346_dp,    0.0000_dp,    0.0000_dp, &
      &  352.2408_dp,  234.4843_dp,  167.6917_dp,    0.0000_dp,    0.0000_dp, &
      &  325.3117_dp,  218.0646_dp,  156.4437_dp,    0.0000_dp,    0.0000_dp, &
      &  318.8589_dp,  212.5126_dp,  152.3164_dp,    0.0000_dp,    0.0000_dp, &
      &  253.3965_dp,  174.6758_dp,  126.8932_dp,    0.0000_dp,    0.0000_dp, &
      &  287.1606_dp,  197.6569_dp,  143.5585_dp,    0.0000_dp,    0.0000_dp, &
      &  278.4020_dp,  191.8429_dp,  139.4802_dp,    0.0000_dp,    0.0000_dp, &
      &  275.4138_dp,  189.2497_dp,  137.5043_dp,    0.0000_dp,    0.0000_dp, &
      &  252.6740_dp,  175.4919_dp,  128.1271_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.2625_dp,  163.8794_dp,  120.7060_dp,    0.0000_dp,    0.0000_dp, &
      &  229.8776_dp,  162.0750_dp,  119.3719_dp,    0.0000_dp,    0.0000_dp, &
      &  225.1357_dp,  158.7933_dp,  117.0121_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  185.1823_dp,  133.7324_dp,   99.9119_dp,    0.0000_dp,    0.0000_dp, &
      &  183.3661_dp,  132.3480_dp,   98.8747_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(342 * maxRef * maxRef + 1 : 361 * maxRef * maxRef) = [ &
      &  148.4319_dp,  109.3155_dp,   82.7591_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1388.9522_dp,  742.2520_dp,  497.3105_dp,    0.0000_dp,    0.0000_dp, &
      &  406.0876_dp,  256.7579_dp,  181.7040_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1056.5193_dp,  613.9233_dp,  419.0635_dp,    0.0000_dp,    0.0000_dp, &
      &  845.4686_dp,  490.7481_dp,  336.7038_dp,    0.0000_dp,    0.0000_dp, &
      &  402.1872_dp,  263.7930_dp,  188.2696_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  857.1730_dp,  507.3401_dp,  348.0642_dp,    0.0000_dp,    0.0000_dp, &
      &  389.5974_dp,  256.8889_dp,  183.8257_dp,    0.0000_dp,    0.0000_dp, &
      &  378.9366_dp,  253.4168_dp,  182.2062_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  814.8402_dp,  489.8996_dp,  338.2283_dp,    0.0000_dp,    0.0000_dp, &
      &  375.8242_dp,  247.9636_dp,  177.7007_dp,    0.0000_dp,    0.0000_dp, &
      &  383.0606_dp,  256.5291_dp,  184.5856_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  739.0542_dp,  448.6262_dp,  310.8375_dp,    0.0000_dp,    0.0000_dp, &
      &  368.6136_dp,  243.0148_dp,  174.1594_dp,    0.0000_dp,    0.0000_dp, &
      &  351.3939_dp,  234.7665_dp,  169.0900_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  581.8114_dp,  353.2709_dp,  245.7629_dp,    0.0000_dp,    0.0000_dp, &
      &  367.0785_dp,  240.7834_dp,  172.3036_dp,    0.0000_dp,    0.0000_dp, &
      &  256.8278_dp,  176.6607_dp,  129.3444_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  627.5403_dp,  386.2602_dp,  269.0494_dp,    0.0000_dp,    0.0000_dp, &
      &  329.6273_dp,  221.5340_dp,  159.7402_dp,    0.0000_dp,    0.0000_dp, &
      &  215.5738_dp,  151.0768_dp,  111.6491_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  491.3349_dp,  303.0480_dp,  212.1109_dp,    0.0000_dp,    0.0000_dp, &
      &  303.0480_dp,  204.9925_dp,  148.3171_dp,    0.0000_dp,    0.0000_dp, &
      &  212.1109_dp,  148.3171_dp,  109.5041_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   36.3234_dp,   23.2310_dp,   20.7741_dp,   18.7500_dp,    0.0000_dp, &
      &   60.5746_dp,   37.4496_dp,   33.3307_dp,   29.7680_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.7233_dp,   15.3545_dp,   13.8804_dp,   12.7523_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  701.5324_dp,  362.3746_dp,  319.8448_dp,  270.7598_dp,    0.0000_dp, &
      &  212.8978_dp,  121.6037_dp,  107.5582_dp,   93.8356_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  369.1928_dp,  209.2227_dp,  184.8938_dp,  160.8870_dp,    0.0000_dp, &
      &  268.6284_dp,  155.0554_dp,  137.2998_dp,  120.1880_dp,    0.0000_dp, &
      &  166.2166_dp,  101.0801_dp,   89.7891_dp,   79.7845_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  234.8693_dp,  140.7428_dp,  124.9087_dp,  110.5365_dp,    0.0000_dp, &
      &  195.1731_dp,  118.3530_dp,  105.2019_dp,   93.4541_dp,    0.0000_dp, &
      &  161.8720_dp,   99.3343_dp,   88.4405_dp,   78.8624_dp,    0.0000_dp, &
      &  119.1091_dp,   75.4525_dp,   67.3932_dp,   60.6537_dp,    0.0000_dp, &
      &  111.5063_dp,   70.9871_dp,   63.4543_dp,   57.1987_dp,    0.0000_dp, &
      &  151.8703_dp,   95.0051_dp,   84.7552_dp,   76.0072_dp,    0.0000_dp, &
      &  142.5140_dp,   89.0992_dp,   79.5273_dp,   71.3318_dp,    0.0000_dp, &
      &  114.0913_dp,   72.6443_dp,   64.9856_dp,   58.6057_dp,    0.0000_dp, &
      &  106.6917_dp,   67.9774_dp,   60.8513_dp,   54.9066_dp,    0.0000_dp, &
      &   84.9547_dp,   55.5654_dp,   49.9148_dp,   45.3891_dp,    0.0000_dp, &
      &  102.7503_dp,   66.4218_dp,   59.5628_dp,   53.9695_dp,    0.0000_dp, &
      &   95.7051_dp,   61.9256_dp,   55.5685_dp,   50.3814_dp,    0.0000_dp, &
      &   89.5345_dp,   58.1611_dp,   52.2322_dp,   47.4168_dp,    0.0000_dp, &
      &   78.4696_dp,   51.2318_dp,   46.0841_dp,   41.9193_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   75.9912_dp,   50.3016_dp,   45.3066_dp,   41.3623_dp,    0.0000_dp, &
      &   67.7117_dp,   45.1002_dp,   40.6799_dp,   37.2178_dp,    0.0000_dp, &
      &   59.2688_dp,   39.8131_dp,   35.9892_dp,   33.0215_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.4307_dp,   38.1313_dp,   34.4935_dp,   31.6985_dp,    0.0000_dp, &
      &   44.9407_dp,   31.0516_dp,   28.2122_dp,   26.1031_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.7270_dp,   29.3598_dp,   26.6627_dp,   24.6343_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(361 * maxRef * maxRef + 1 : 380 * maxRef * maxRef) = [ &
      &  834.1826_dp,  433.3736_dp,  382.6821_dp,  324.6241_dp,    0.0000_dp, &
      &  314.1718_dp,  177.8355_dp,  157.3578_dp,  136.9841_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  598.3302_dp,  333.4767_dp,  294.6091_dp,  255.1561_dp,    0.0000_dp, &
      &  488.1715_dp,  274.9840_dp,  243.1809_dp,  211.3388_dp,    0.0000_dp, &
      &  302.4518_dp,  179.4553_dp,  159.0980_dp,  140.3518_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  535.9706_dp,  307.2540_dp,  271.8226_dp,  237.3961_dp,    0.0000_dp, &
      &  482.2990_dp,  278.4173_dp,  246.4539_dp,  215.7066_dp,    0.0000_dp, &
      &  366.7236_dp,  216.8393_dp,  192.2755_dp,  169.4903_dp,    0.0000_dp, &
      &  276.8948_dp,  168.8027_dp,  149.9951_dp,  133.3820_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  405.9907_dp,  241.9159_dp,  214.5726_dp,  189.5488_dp,    0.0000_dp, &
      &  373.0125_dp,  223.5548_dp,  198.4174_dp,  175.5963_dp,    0.0000_dp, &
      &  344.9479_dp,  207.9084_dp,  184.6285_dp,  163.6721_dp,    0.0000_dp, &
      &  337.4624_dp,  202.6388_dp,  179.9690_dp,  159.4063_dp,    0.0000_dp, &
      &  270.0881_dp,  166.5674_dp,  148.2259_dp,  132.2884_dp,    0.0000_dp, &
      &  305.9256_dp,  188.4852_dp,  167.7356_dp,  149.6709_dp,    0.0000_dp, &
      &  296.5789_dp,  182.9511_dp,  162.8541_dp,  145.3801_dp,    0.0000_dp, &
      &  293.1661_dp,  180.4846_dp,  160.6578_dp,  143.3518_dp,    0.0000_dp, &
      &  269.4807_dp,  167.3765_dp,  149.1158_dp,  133.3956_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  248.2116_dp,  156.3423_dp,  139.5430_dp,  125.3725_dp,    0.0000_dp, &
      &  245.5988_dp,  154.6243_dp,  138.0159_dp,  123.9905_dp,    0.0000_dp, &
      &  240.5175_dp,  151.4983_dp,  135.2456_dp,  121.5256_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  198.3570_dp,  127.6513_dp,  114.3091_dp,  103.3931_dp,    0.0000_dp, &
      &  196.3755_dp,  126.3319_dp,  113.1318_dp,  102.3223_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  159.1957_dp,  104.4091_dp,   93.8084_dp,   85.3616_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1382.9771_dp,  710.1818_dp,  628.1238_dp,  531.3044_dp,    0.0000_dp, &
      &  423.3809_dp,  245.0188_dp,  217.5967_dp,  190.9521_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1081.7690_dp,  585.9578_dp,  517.8769_dp,  445.0681_dp,    0.0000_dp, &
      &  862.5036_dp,  468.6899_dp,  414.8338_dp,  357.0721_dp,    0.0000_dp, &
      &  423.9523_dp,  251.5841_dp,  223.4273_dp,  197.3201_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  882.4503_dp,  484.0723_dp,  427.8866_dp,  369.0835_dp,    0.0000_dp, &
      &  411.0572_dp,  245.0035_dp,  217.6909_dp,  192.5221_dp,    0.0000_dp, &
      &  401.1295_dp,  241.6833_dp,  214.8399_dp,  190.5518_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  841.8335_dp,  467.3984_dp,  413.4479_dp,  357.9783_dp,    0.0000_dp, &
      &  396.2748_dp,  236.5325_dp,  210.2464_dp,  186.0369_dp,    0.0000_dp, &
      &  405.5809_dp,  244.6571_dp,  217.5178_dp,  193.0012_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  765.3178_dp,  427.9936_dp,  378.7327_dp,  328.6407_dp,    0.0000_dp, &
      &  388.5729_dp,  231.8180_dp,  206.0757_dp,  182.3353_dp,    0.0000_dp, &
      &  371.5149_dp,  223.9437_dp,  199.2038_dp,  176.7692_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  601.4178_dp,  337.1436_dp,  298.7169_dp,  259.5722_dp,    0.0000_dp, &
      &  386.4486_dp,  229.7022_dp,  204.1750_dp,  180.4774_dp,    0.0000_dp, &
      &  272.6301_dp,  168.5969_dp,  150.4863_dp,  134.6287_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  651.9688_dp,  368.4719_dp,  326.2603_dp,  284.0167_dp,    0.0000_dp, &
      &  349.1941_dp,  211.3025_dp,  187.9564_dp,  166.9417_dp,    0.0000_dp, &
      &  229.5077_dp,  144.2159_dp,  128.9628_dp,  115.9223_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  509.7764_dp,  289.1937_dp,  256.4238_dp,  223.6354_dp,    0.0000_dp, &
      &  321.3448_dp,  195.5430_dp,  174.0536_dp,  154.8608_dp,    0.0000_dp, &
      &  225.7346_dp,  141.5815_dp,  126.5885_dp,  113.7287_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  532.7794_dp,  306.4968_dp,  271.6288_dp,  237.6994_dp,    0.0000_dp, &
      &  306.4968_dp,  186.5462_dp,  166.0911_dp,  147.8057_dp,    0.0000_dp, &
      &  271.6288_dp,  166.0911_dp,  148.0657_dp,  132.0085_dp,    0.0000_dp, &
      &  237.6994_dp,  147.8057_dp,  132.0085_dp,  118.2863_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   37.1596_dp,   22.6822_dp,   19.4648_dp,   20.1853_dp,    0.0000_dp, &
      &   62.2532_dp,   36.5901_dp,   31.0981_dp,   32.5149_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   23.0792_dp,   14.9833_dp,   13.0979_dp,   13.4305_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(380 * maxRef * maxRef + 1 : 399 * maxRef * maxRef) = [ &
      &  738.8580_dp,  355.9606_dp,  291.0488_dp,  320.6585_dp,    0.0000_dp, &
      &  221.1739_dp,  119.0725_dp,   99.3454_dp,  106.1545_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.9140_dp,  204.9086_dp,  170.5936_dp,  182.6955_dp,    0.0000_dp, &
      &  278.7116_dp,  151.7798_dp,  126.9812_dp,  135.2702_dp,    0.0000_dp, &
      &  171.2043_dp,   98.8004_dp,   83.6030_dp,   87.7852_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  242.4174_dp,  137.6254_dp,  116.0923_dp,  122.3915_dp,    0.0000_dp, &
      &  201.1351_dp,  115.7000_dp,   97.9268_dp,  102.9220_dp,    0.0000_dp, &
      &  166.5614_dp,   97.0816_dp,   82.4474_dp,   86.3895_dp,    0.0000_dp, &
      &  122.0144_dp,   73.6865_dp,   63.0740_dp,   65.5618_dp,    0.0000_dp, &
      &  114.1506_dp,   69.3194_dp,   59.4247_dp,   61.6974_dp,    0.0000_dp, &
      &  155.8507_dp,   92.8107_dp,   79.2056_dp,   82.5930_dp,    0.0000_dp, &
      &  146.2711_dp,   87.0432_dp,   74.3149_dp,   77.5124_dp,    0.0000_dp, &
      &  116.8072_dp,   70.9444_dp,   60.8678_dp,   63.2092_dp,    0.0000_dp, &
      &  109.2305_dp,   66.3897_dp,   57.0046_dp,   59.1980_dp,    0.0000_dp, &
      &   86.6601_dp,   54.2397_dp,   46.9071_dp,   48.4149_dp,    0.0000_dp, &
      &  104.9834_dp,   64.8519_dp,   55.8954_dp,   57.8469_dp,    0.0000_dp, &
      &   97.7761_dp,   60.4608_dp,   52.1532_dp,   53.9690_dp,    0.0000_dp, &
      &   91.4295_dp,   56.7868_dp,   49.0511_dp,   50.7198_dp,    0.0000_dp, &
      &   80.0853_dp,   50.0218_dp,   43.3096_dp,   44.7442_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   77.4010_dp,   49.0996_dp,   42.6462_dp,   43.9161_dp,    0.0000_dp, &
      &   68.9086_dp,   44.0173_dp,   38.3194_dp,   39.4125_dp,    0.0000_dp, &
      &   60.2575_dp,   38.8597_dp,   33.9427_dp,   34.8596_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   57.3248_dp,   37.2158_dp,   32.5553_dp,   33.3924_dp,    0.0000_dp, &
      &   45.5155_dp,   30.3004_dp,   26.7000_dp,   27.2705_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   43.3129_dp,   28.6554_dp,   25.2210_dp,   25.7940_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  877.9354_dp,  425.6455_dp,  348.5798_dp,  383.3773_dp,    0.0000_dp, &
      &  326.8271_dp,  174.2001_dp,  145.1854_dp,  155.5786_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  623.6305_dp,  326.7864_dp,  271.2223_dp,  291.9263_dp,    0.0000_dp, &
      &  508.1585_dp,  269.3779_dp,  224.1812_dp,  240.5302_dp,    0.0000_dp, &
      &  312.5862_dp,  175.5343_dp,  147.6912_dp,  156.1327_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  556.5389_dp,  300.8271_dp,  251.1877_dp,  268.1207_dp,    0.0000_dp, &
      &  500.3525_dp,  272.5409_dp,  227.9565_dp,  242.8419_dp,    0.0000_dp, &
      &  379.2280_dp,  212.1194_dp,  178.3984_dp,  188.7699_dp,    0.0000_dp, &
      &  285.1174_dp,  164.9924_dp,  139.7108_dp,  146.6121_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  419.3593_dp,  236.5952_dp,  199.2855_dp,  210.4188_dp,    0.0000_dp, &
      &  385.0048_dp,  218.6066_dp,  184.4175_dp,  194.4184_dp,    0.0000_dp, &
      &  355.7641_dp,  203.2777_dp,  171.7269_dp,  180.7673_dp,    0.0000_dp, &
      &  348.2477_dp,  198.1506_dp,  167.3173_dp,  176.3143_dp,    0.0000_dp, &
      &  277.6794_dp,  162.7632_dp,  138.2635_dp,  144.6644_dp,    0.0000_dp, &
      &  314.5698_dp,  184.1847_dp,  156.4432_dp,  163.7299_dp,    0.0000_dp, &
      &  304.9145_dp,  178.7735_dp,  151.9152_dp,  158.9453_dp,    0.0000_dp, &
      &  301.4974_dp,  176.3750_dp,  149.8328_dp,  156.8549_dp,    0.0000_dp, &
      &  276.7985_dp,  163.5301_dp,  139.2201_dp,  145.4133_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.4683_dp,  152.7036_dp,  130.5102_dp,  135.8470_dp,    0.0000_dp, &
      &  251.8097_dp,  151.0286_dp,  129.0763_dp,  134.3737_dp,    0.0000_dp, &
      &  246.5864_dp,  147.9753_dp,  126.4947_dp,  131.6741_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  202.7705_dp,  124.6317_dp,  107.1976_dp,  111.0248_dp,    0.0000_dp, &
      &  200.7571_dp,  123.3457_dp,  106.0905_dp,  109.8900_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  162.3264_dp,  101.9116_dp,   88.1829_dp,   90.9553_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1458.4501_dp,  697.7650_dp,  570.8659_dp,  630.1412_dp,    0.0000_dp, &
      &  439.2845_dp,  239.8849_dp,  201.3799_dp,  214.4961_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1132.1074_dp,  574.7522_dp,  474.8225_dp,  515.5415_dp,    0.0000_dp, &
      &  902.6014_dp,  459.6812_dp,  380.4201_dp,  412.6142_dp,    0.0000_dp, &
      &  438.2482_dp,  246.1156_dp,  207.4440_dp,  219.3679_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  921.8990_dp,  474.6254_dp,  393.0487_dp,  425.1277_dp,    0.0000_dp, &
      &  424.6693_dp,  239.6563_dp,  202.2392_dp,  213.6129_dp,    0.0000_dp, &
      &  413.7841_dp,  236.3296_dp,  199.8502_dp,  210.4559_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(399 * maxRef * maxRef + 1 : 418 * maxRef * maxRef) = [ &
      &  878.0939_dp,  458.1136_dp,  380.4377_dp,  410.0129_dp,    0.0000_dp, &
      &  409.3555_dp,  231.3631_dp,  195.3516_dp,  206.2651_dp,    0.0000_dp, &
      &  418.3113_dp,  239.2323_dp,  202.3746_dp,  213.0501_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  797.5195_dp,  419.4037_dp,  348.8510_dp,  375.1745_dp,    0.0000_dp, &
      &  401.4329_dp,  226.7601_dp,  191.4715_dp,  202.2050_dp,    0.0000_dp, &
      &  383.2557_dp,  218.9943_dp,  185.3265_dp,  195.1629_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  626.6472_dp,  330.3808_dp,  275.2551_dp,  295.8599_dp,    0.0000_dp, &
      &  399.4530_dp,  224.7176_dp,  189.6203_dp,  200.4598_dp,    0.0000_dp, &
      &  280.2977_dp,  164.7847_dp,  140.4748_dp,  146.9885_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  678.4531_dp,  360.9692_dp,  300.9640_dp,  322.6845_dp,    0.0000_dp, &
      &  360.0172_dp,  206.6126_dp,  174.9529_dp,  184.0478_dp,    0.0000_dp, &
      &  235.4499_dp,  140.9080_dp,  120.6249_dp,  125.7205_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.3221_dp,  283.3011_dp,  236.6842_dp,  253.5365_dp,    0.0000_dp, &
      &  331.0614_dp,  191.1798_dp,  162.1293_dp,  170.3117_dp,    0.0000_dp, &
      &  231.6422_dp,  138.3424_dp,  118.3800_dp,  123.4417_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  553.0834_dp,  300.1055_dp,  251.1837_dp,  267.9482_dp,    0.0000_dp, &
      &  315.7713_dp,  182.3876_dp,  154.7198_dp,  162.5286_dp,    0.0000_dp, &
      &  279.7052_dp,  162.3831_dp,  138.0223_dp,  144.8424_dp,    0.0000_dp, &
      &  244.2053_dp,  144.4518_dp,  123.3171_dp,  128.8641_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  574.7436_dp,  309.2543_dp,  258.3888_dp,  276.2327_dp,    0.0000_dp, &
      &  309.2543_dp,  178.3311_dp,  151.2409_dp,  158.9405_dp,    0.0000_dp, &
      &  258.3888_dp,  151.2409_dp,  128.7820_dp,  134.8933_dp,    0.0000_dp, &
      &  276.2327_dp,  158.9405_dp,  134.8933_dp,  141.8763_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   28.5940_dp,   22.1550_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   47.6499_dp,   35.9871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   18.0216_dp,   14.5147_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  569.4526_dp,  367.4308_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  168.8075_dp,  119.4397_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  292.7955_dp,  205.9313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.9797_dp,  151.8426_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  130.8012_dp,   97.5340_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  185.1741_dp,  136.3705_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  153.8824_dp,  114.3466_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  127.6686_dp,   95.7007_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.7660_dp,   72.1216_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   87.8052_dp,   67.7835_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  119.6350_dp,   91.1025_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  112.3914_dp,   85.4731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   89.9515_dp,   69.4058_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.2028_dp,   64.9641_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   67.0823_dp,   52.7984_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.0855_dp,   63.2709_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   75.5987_dp,   58.9903_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   70.7639_dp,   55.3789_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   62.1299_dp,   48.7630_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.1335_dp,   47.7239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   53.6420_dp,   42.7458_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   47.0585_dp,   37.7083_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.8098_dp,   36.0749_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.8100_dp,   29.2808_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   34.0544_dp,   27.7282_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  676.4773_dp,  438.7198_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.8197_dp,  175.2531_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  476.2426_dp,  330.0018_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  388.4209_dp,  271.2755_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  238.5370_dp,  174.3619_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(418 * maxRef * maxRef + 1 : 437 * maxRef * maxRef) = [ &
      &  424.7723_dp,  301.4650_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  381.9923_dp,  272.6384_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.6824_dp,  210.9019_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  217.8958_dp,  162.8021_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  320.1913_dp,  234.7548_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.1406_dp,  216.6154_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  271.8911_dp,  201.1573_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  266.3120_dp,  196.2996_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.5281_dp,  160.1946_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.7974_dp,  181.3267_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  233.4898_dp,  175.9628_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  230.9091_dp,  173.7006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.1340_dp,  160.7225_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.4075_dp,  149.6421_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.3865_dp,  148.0244_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  189.4122_dp,  145.0250_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  156.3161_dp,  121.6329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  154.7756_dp,  120.3927_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  125.6776_dp,   99.1398_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1129.9518_dp,  722.0002_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  337.1117_dp,  240.1791_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  868.3674_dp,  585.3704_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  694.5058_dp,  467.9156_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  335.3592_dp,  244.6716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  706.0720_dp,  481.6472_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.0850_dp,  238.0128_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  316.7102_dp,  234.0374_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  672.4128_dp,  463.3924_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  313.6026_dp,  229.7226_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  320.2198_dp,  236.8532_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.5700_dp,  423.4193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.5763_dp,  225.2020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.6816_dp,  216.9203_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  480.6938_dp,  333.5014_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  306.1181_dp,  223.4009_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  215.5023_dp,  162.3545_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  519.3078_dp,  363.4171_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  275.7206_dp,  204.4480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.3482_dp,  138.3572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  406.7268_dp,  285.1018_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  253.7038_dp,  188.9445_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  178.3924_dp,  135.9019_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  423.0877_dp,  300.7356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  242.1030_dp,  180.2719_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.8093_dp,  160.4015_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  187.8440_dp,  142.1634_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  439.6365_dp,  310.5133_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  237.1200_dp,  176.3358_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  198.5202_dp,  149.1479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.1434_dp,  157.3396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  337.1808_dp,  238.1207_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  238.1207_dp,  174.9969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.8689_dp,   21.2139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.3469_dp,   34.0312_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   18.9695_dp,   14.1185_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(437 * maxRef * maxRef + 1 : 456 * maxRef * maxRef) = [ &
      &  540.9160_dp,  317.8826_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.4949_dp,  109.0127_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.3106_dp,  187.3012_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.4256_dp,  139.2420_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  134.7933_dp,   91.6227_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  189.6293_dp,  127.2488_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  158.0848_dp,  107.2026_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.5252_dp,   90.1351_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   97.6845_dp,   68.8077_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   91.5728_dp,   64.7798_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.1032_dp,   86.4709_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  116.3987_dp,   81.0686_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.6870_dp,   66.2838_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   87.6124_dp,   62.0228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   70.2820_dp,   50.8856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.7311_dp,   60.7331_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   78.9183_dp,   56.6140_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   73.9210_dp,   53.2070_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.8633_dp,   46.8911_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   63.0774_dp,   46.1355_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.2884_dp,   41.3862_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.3897_dp,   36.5764_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   47.1122_dp,   35.0629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   37.7559_dp,   28.6291_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.8427_dp,   27.0524_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  644.3408_dp,  380.6568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.3925_dp,  159.1228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  472.9143_dp,  297.5440_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  386.8884_dp,  245.8054_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  243.5663_dp,  162.0032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  427.1610_dp,  275.6299_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  385.1449_dp,  250.0730_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.9017_dp,  195.5973_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  224.7197_dp,  153.0776_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.2810_dp,  218.5282_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  301.1709_dp,  202.1306_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  278.9696_dp,  188.1613_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  272.5606_dp,  183.2556_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  219.8870_dp,  151.3141_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  248.9743_dp,  171.1889_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  241.4380_dp,  166.1887_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  238.5063_dp,  163.8884_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  219.8016_dp,  152.2000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.2427_dp,  142.4591_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  201.0704_dp,  140.8804_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  196.9331_dp,  138.0396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  163.3848_dp,  116.6637_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  161.7337_dp,  115.4501_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.8119_dp,   95.6586_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1062.5928_dp,  621.4841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  337.8986_dp,  219.9852_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  846.8527_dp,  519.5182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  675.0299_dp,  415.5638_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  341.1923_dp,  227.0118_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  693.6787_dp,  430.3675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  331.2310_dp,  221.2467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  324.2891_dp,  218.6603_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(456 * maxRef * maxRef + 1 : 475 * maxRef * maxRef) = [ &
      &  664.1009_dp,  416.5328_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  319.3834_dp,  213.6191_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  328.0001_dp,  221.3943_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  605.0623_dp,  381.9629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  313.1286_dp,  209.3428_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  300.3130_dp,  202.5980_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  475.5594_dp,  300.9323_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  311.0603_dp,  207.2926_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  221.9656_dp,  153.1295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  517.0910_dp,  329.5196_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  282.6667_dp,  191.3125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  187.7214_dp,  131.3100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  404.5558_dp,  258.7290_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  260.5341_dp,  177.2034_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  184.5403_dp,  128.8760_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  424.8790_dp,  275.0381_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  248.4855_dp,  169.0461_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  220.4525_dp,  150.5946_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.8570_dp,  134.3706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  440.0900_dp,  282.9656_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  243.1960_dp,  165.2361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  204.3119_dp,  140.4552_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.9447_dp,  147.1597_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  336.6493_dp,  216.9570_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  242.6893_dp,  162.9236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  340.5213_dp,  223.6622_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  223.6622_dp,  153.4528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.1697_dp,   33.2515_dp,   26.9562_dp,   22.5390_dp,    0.0000_dp, &
      &   58.4207_dp,   54.9919_dp,   43.8881_dp,   36.1447_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.0458_dp,   20.9874_dp,   17.4392_dp,   14.9296_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  656.7441_dp,  602.8104_dp,  441.4888_dp,  332.3464_dp,    0.0000_dp, &
      &  202.6900_dp,  188.7622_dp,  145.1043_dp,  115.1284_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  351.1141_dp,  326.6672_dp,  250.1889_dp,  197.7454_dp,    0.0000_dp, &
      &  256.2049_dp,  238.9068_dp,  184.5744_dp,  147.1923_dp,    0.0000_dp, &
      &  159.9273_dp,  150.2148_dp,  118.9619_dp,   97.2346_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  225.3933_dp,  211.2756_dp,  166.1525_dp,  134.8856_dp,    0.0000_dp, &
      &  187.5797_dp,  176.0976_dp,  139.2546_dp,  113.6748_dp,    0.0000_dp, &
      &  155.8080_dp,  146.4890_dp,  116.4830_dp,   95.6039_dp,    0.0000_dp, &
      &  115.1706_dp,  108.7508_dp,   87.7688_dp,   73.0754_dp,    0.0000_dp, &
      &  107.8705_dp,  101.9223_dp,   82.4447_dp,   68.7935_dp,    0.0000_dp, &
      &  146.5544_dp,  138.1497_dp,  110.8384_dp,   91.7661_dp,    0.0000_dp, &
      &  137.4723_dp,  129.5670_dp,  103.9180_dp,   86.0172_dp,    0.0000_dp, &
      &  110.2770_dp,  104.1881_dp,   84.2691_dp,   70.3184_dp,    0.0000_dp, &
      &  103.0738_dp,   97.3838_dp,   78.7812_dp,   65.7584_dp,    0.0000_dp, &
      &   82.3248_dp,   78.0505_dp,   63.9117_dp,   53.9617_dp,    0.0000_dp, &
      &   99.4407_dp,   94.1329_dp,   76.6641_dp,   64.3977_dp,    0.0000_dp, &
      &   92.6025_dp,   87.6645_dp,   71.4234_dp,   60.0229_dp,    0.0000_dp, &
      &   86.6194_dp,   82.0402_dp,   66.9550_dp,   56.3612_dp,    0.0000_dp, &
      &   75.8800_dp,   71.9079_dp,   58.8126_dp,   49.6159_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   73.6253_dp,   69.9039_dp,   57.5425_dp,   48.8310_dp,    0.0000_dp, &
      &   65.6214_dp,   62.3522_dp,   51.4705_dp,   43.7962_dp,    0.0000_dp, &
      &   57.4016_dp,   54.5964_dp,   45.2336_dp,   38.6263_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   54.6782_dp,   52.0476_dp,   43.2385_dp,   37.0129_dp,    0.0000_dp, &
      &   43.5685_dp,   41.5909_dp,   34.9008_dp,   30.1557_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.3681_dp,   39.4587_dp,   33.0228_dp,   28.4654_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(475 * maxRef * maxRef + 1 : 494 * maxRef * maxRef) = [ &
      &  781.3524_dp,  717.7531_dp,  527.0671_dp,  398.0180_dp,    0.0000_dp, &
      &  298.4007_dp,  277.5143_dp,  212.3618_dp,  167.7278_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  567.1638_dp,  526.4236_dp,  399.9263_dp,  313.4658_dp,    0.0000_dp, &
      &  463.5691_dp,  430.8297_dp,  328.9697_dp,  259.2061_dp,    0.0000_dp, &
      &  289.7425_dp,  271.2420_dp,  212.3044_dp,  171.5385_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  510.5854_dp,  475.7318_dp,  366.3371_dp,  291.1704_dp,    0.0000_dp, &
      &  459.9544_dp,  428.9521_dp,  331.4055_dp,  264.2993_dp,    0.0000_dp, &
      &  351.1192_dp,  328.5100_dp,  256.7121_dp,  207.0871_dp,    0.0000_dp, &
      &  266.4534_dp,  250.3494_dp,  198.4813_dp,  162.4065_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.2656_dp,  364.6110_dp,  285.9761_dp,  231.5432_dp,    0.0000_dp, &
      &  357.9411_dp,  335.5206_dp,  263.8740_dp,  214.2270_dp,    0.0000_dp, &
      &  331.2849_dp,  310.7704_dp,  245.0585_dp,  199.4760_dp,    0.0000_dp, &
      &  323.8179_dp,  303.5907_dp,  238.9624_dp,  194.1778_dp,    0.0000_dp, &
      &  260.3091_dp,  244.9446_dp,  195.2461_dp,  160.6091_dp,    0.0000_dp, &
      &  294.7925_dp,  277.3523_dp,  220.9763_dp,  181.6981_dp,    0.0000_dp, &
      &  285.8072_dp,  268.9370_dp,  214.3903_dp,  176.3816_dp,    0.0000_dp, &
      &  282.3871_dp,  265.6407_dp,  211.5571_dp,  173.8931_dp,    0.0000_dp, &
      &  259.9184_dp,  244.7964_dp,  195.7667_dp,  161.5591_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  239.8183_dp,  226.2796_dp,  182.1380_dp,  151.2611_dp,    0.0000_dp, &
      &  237.2563_dp,  223.8453_dp,  180.1361_dp,  149.5674_dp,    0.0000_dp, &
      &  232.3409_dp,  219.2194_dp,  176.4499_dp,  146.5382_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  192.0578_dp,  181.7178_dp,  147.7169_dp,  123.8442_dp,    0.0000_dp, &
      &  190.1139_dp,  179.8683_dp,  146.1873_dp,  122.5429_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  154.3423_dp,  146.3855_dp,  120.0234_dp,  101.4572_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1292.8121_dp, 1184.8327_dp,  865.1066_dp,  648.6879_dp,    0.0000_dp, &
      &  403.0745_dp,  375.8495_dp,  290.5739_dp,  231.9989_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1019.8835_dp,  942.5828_dp,  706.2041_dp,  545.4548_dp,    0.0000_dp, &
      &  813.4943_dp,  751.8144_dp,  564.1147_dp,  436.2844_dp,    0.0000_dp, &
      &  405.6488_dp,  379.6650_dp,  297.1084_dp,  240.0800_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  833.7424_dp,  771.9763_dp,  581.9150_dp,  452.4148_dp,    0.0000_dp, &
      &  393.5300_dp,  368.5417_dp,  288.9924_dp,  234.0165_dp,    0.0000_dp, &
      &  384.8172_dp,  360.9196_dp,  284.4870_dp,  231.5184_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  796.8617_dp,  739.0281_dp,  560.2945_dp,  438.2812_dp,    0.0000_dp, &
      &  379.4109_dp,  355.3498_dp,  278.8293_dp,  225.9136_dp,    0.0000_dp, &
      &  389.1325_dp,  365.0239_dp,  287.8797_dp,  234.4093_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  725.2514_dp,  673.2823_dp,  512.2075_dp,  402.1202_dp,    0.0000_dp, &
      &  371.9427_dp,  348.3293_dp,  273.2440_dp,  221.3382_dp,    0.0000_dp, &
      &  356.2469_dp,  334.1078_dp,  263.3800_dp,  214.3808_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  569.7645_dp,  529.0006_dp,  402.8309_dp,  316.6134_dp,    0.0000_dp, &
      &  369.6302_dp,  345.9796_dp,  270.9168_dp,  219.0704_dp,    0.0000_dp, &
      &  262.1530_dp,  246.6796_dp,  196.7704_dp,  162.0547_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  618.8387_dp,  575.3229_dp,  439.8829_dp,  347.1545_dp,    0.0000_dp, &
      &  335.0615_dp,  314.4244_dp,  248.3153_dp,  202.4740_dp,    0.0000_dp, &
      &  221.1217_dp,  208.5110_dp,  167.5629_dp,  138.9928_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  483.7815_dp,  449.9040_dp,  344.5352_dp,  272.3998_dp,    0.0000_dp, &
      &  308.5529_dp,  289.7588_dp,  229.4318_dp,  187.5580_dp,    0.0000_dp, &
      &  217.4040_dp,  204.9523_dp,  164.5553_dp,  136.3818_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  507.1316_dp,  472.6401_dp,  364.4340_dp,  290.1268_dp,    0.0000_dp, &
      &  294.2408_dp,  276.3130_dp,  218.7964_dp,  178.8774_dp,    0.0000_dp, &
      &  260.7548_dp,  244.9914_dp,  194.3868_dp,  159.2584_dp,    0.0000_dp, &
      &  228.6785_dp,  215.3374_dp,  172.2004_dp,  142.1634_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  525.8162_dp,  489.5499_dp,  376.1112_dp,  298.3141_dp,    0.0000_dp, &
      &  288.0073_dp,  270.4023_dp,  213.9621_dp,  174.8058_dp,    0.0000_dp, &
      &  241.3636_dp,  227.0001_dp,  180.7289_dp,  148.5642_dp,    0.0000_dp, &
      &  256.7960_dp,  241.0016_dp,  190.4799_dp,  155.4813_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  401.9613_dp,  374.2537_dp,  287.7668_dp,  228.4811_dp,    0.0000_dp, &
      &  287.8520_dp,  269.7344_dp,  212.0275_dp,  172.1215_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(494 * maxRef * maxRef + 1 : 513 * maxRef * maxRef) = [ &
      &  405.4565_dp,  378.7329_dp,  294.3129_dp,  236.1661_dp,    0.0000_dp, &
      &  264.4724_dp,  248.7003_dp,  197.8459_dp,  162.4735_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  483.7516_dp,  451.4003_dp,  349.5389_dp,  279.4346_dp,    0.0000_dp, &
      &  451.4003_dp,  421.6497_dp,  327.6772_dp,  262.9134_dp,    0.0000_dp, &
      &  349.5389_dp,  327.6772_dp,  257.8727_dp,  209.5130_dp,    0.0000_dp, &
      &  279.4346_dp,  262.9134_dp,  209.5130_dp,  172.3153_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   31.8170_dp,   29.9527_dp,   28.6359_dp,   25.9255_dp,   23.1784_dp, &
      &   51.9905_dp,   48.7247_dp,   46.4409_dp,   41.7323_dp,   36.9664_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.4318_dp,   19.3761_dp,   18.6186_dp,   17.0614_dp,   15.4801_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  527.9268_dp,  484.0165_dp,  454.1125_dp,  392.4109_dp,  330.5666_dp, &
      &  172.9425_dp,  160.4502_dp,  151.8759_dp,  134.1408_dp,  116.2577_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  298.4220_dp,  276.5717_dp,  261.6034_dp,  230.6287_dp,  199.4034_dp, &
      &  219.7375_dp,  204.1690_dp,  193.4457_dp,  171.2891_dp,  148.9411_dp, &
      &  141.1489_dp,  131.9934_dp,  125.6209_dp,  112.4708_dp,   99.1700_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.3448_dp,  184.2055_dp,  175.0938_dp,  156.2783_dp,  137.2617_dp, &
      &  165.1930_dp,  154.4409_dp,  146.9606_dp,  131.5234_dp,  115.9145_dp, &
      &  137.9956_dp,  129.2227_dp,  123.0969_dp,  110.4653_dp,   97.6880_dp, &
      &  103.6947_dp,   97.4953_dp,   93.1301_dp,   84.1398_dp,   75.0316_dp, &
      &   97.3490_dp,   91.5895_dp,   87.5282_dp,   79.1656_dp,   70.6917_dp, &
      &  131.0954_dp,  123.0574_dp,  117.4198_dp,  105.8004_dp,   94.0357_dp, &
      &  122.8805_dp,  115.3488_dp,  110.0651_dp,   99.1763_dp,   88.1532_dp, &
      &   99.4698_dp,   93.5893_dp,   89.4448_dp,   80.9084_dp,   72.2595_dp, &
      &   92.9621_dp,   87.4781_dp,   83.6131_dp,   75.6514_dp,   67.5854_dp, &
      &   75.2076_dp,   71.0143_dp,   68.0343_dp,   61.9038_dp,   55.6857_dp, &
      &   90.3328_dp,   85.1621_dp,   81.5019_dp,   73.9675_dp,   66.3293_dp, &
      &   84.1276_dp,   79.3278_dp,   75.9287_dp,   68.9323_dp,   61.8399_dp, &
      &   78.8235_dp,   74.3627_dp,   71.2017_dp,   64.6946_dp,   58.0974_dp, &
      &   69.1710_dp,   65.3041_dp,   62.5609_dp,   56.9134_dp,   51.1874_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   67.5932_dp,   63.9245_dp,   61.3099_dp,   55.9314_dp,   50.4741_dp, &
      &   60.4041_dp,   57.1752_dp,   54.8693_dp,   50.1269_dp,   45.3141_dp, &
      &   53.0110_dp,   50.2343_dp,   48.2476_dp,   44.1611_dp,   40.0132_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.6412_dp,   48.0233_dp,   46.1468_dp,   42.2879_dp,   38.3697_dp, &
      &   40.7500_dp,   38.7583_dp,   37.3202_dp,   34.3642_dp,   31.3603_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   38.5746_dp,   36.6615_dp,   35.2842_dp,   32.4515_dp,   29.5740_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  630.1041_dp,  578.0882_dp,  542.6657_dp,  469.5496_dp,  396.2431_dp, &
      &  253.1433_dp,  234.6048_dp,  221.9031_dp,  195.6231_dp,  169.1405_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  477.3744_dp,  441.5239_dp,  417.0307_dp,  366.3319_dp,  315.2720_dp, &
      &  392.2533_dp,  363.3320_dp,  343.5106_dp,  302.5213_dp,  261.2272_dp, &
      &  252.3964_dp,  235.2670_dp,  223.4249_dp,  198.9542_dp,  174.2312_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  436.5060_dp,  405.1611_dp,  383.6305_dp,  339.1034_dp,  294.1970_dp, &
      &  394.6650_dp,  366.6561_dp,  347.3879_dp,  307.5504_dp,  267.3616_dp, &
      &  305.1643_dp,  284.3739_dp,  269.9943_dp,  240.2935_dp,  210.2972_dp, &
      &  235.4382_dp,  220.2317_dp,  209.6430_dp,  187.7933_dp,  165.6908_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  339.8328_dp,  316.9700_dp,  301.1380_dp,  268.4370_dp,  235.3944_dp, &
      &  313.3911_dp,  292.5335_dp,  278.0671_dp,  248.1965_dp,  218.0074_dp, &
      &  290.9092_dp,  271.7438_dp,  258.4338_dp,  230.9563_dp,  203.1783_dp, &
      &  283.6909_dp,  264.8945_dp,  251.8482_dp,  224.9146_dp,  197.6946_dp, &
      &  231.3302_dp,  216.7246_dp,  206.5206_dp,  185.4774_dp,  164.1811_dp, &
      &  261.8235_dp,  245.2683_dp,  233.7044_dp,  209.8561_dp,  185.7226_dp, &
      &  253.9744_dp,  237.9580_dp,  226.7660_dp,  203.6865_dp,  180.3305_dp, &
      &  250.6401_dp,  234.7785_dp,  223.7008_dp,  200.8543_dp,  177.7373_dp, &
      &  231.7564_dp,  217.3350_dp,  207.2390_dp,  186.4258_dp,  165.3571_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  215.3101_dp,  202.2863_dp,  193.1315_dp,  174.2716_dp,  155.1691_dp, &
      &  212.9437_dp,  200.0526_dp,  190.9927_dp,  172.3272_dp,  153.4225_dp, &
      &  208.5685_dp,  195.9558_dp,  187.0906_dp,  168.8263_dp,  150.3278_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  174.1969_dp,  164.1266_dp,  157.0031_dp,  142.3413_dp,  127.4785_dp, &
      &  172.3933_dp,  162.4206_dp,  155.3672_dp,  140.8492_dp,  126.1326_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(513 * maxRef * maxRef + 1 : 532 * maxRef * maxRef) = [ &
      &  141.2063_dp,  133.3794_dp,  127.8113_dp,  116.3589_dp,  104.7412_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1032.6651_dp,  946.5756_dp,  887.7075_dp,  766.5714_dp,  645.2973_dp, &
      &  345.4434_dp,  321.1574_dp,  304.4222_dp,  269.8403_dp,  234.9742_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  843.3644_dp,  777.5361_dp,  732.6433_dp,  639.8000_dp,  546.4784_dp, &
      &  672.6917_dp,  620.6735_dp,  585.0515_dp,  511.5374_dp,  437.6779_dp, &
      &  352.9336_dp,  329.0554_dp,  312.5424_dp,  278.4205_dp,  243.9590_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  694.7024_dp,  641.4064_dp,  605.0343_dp,  529.7799_dp,  454.0771_dp, &
      &  343.1679_dp,  320.1342_dp,  304.1902_dp,  271.2475_dp,  237.9706_dp, &
      &  337.5761_dp,  315.3367_dp,  299.9014_dp,  268.0269_dp,  235.8103_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  668.3786_dp,  618.0401_dp,  583.6237_dp,  512.4274_dp,  440.7662_dp, &
      &  330.9710_dp,  308.8304_dp,  293.4899_dp,  261.8074_dp,  229.8053_dp, &
      &  341.5627_dp,  319.1099_dp,  303.5228_dp,  271.3354_dp,  238.8007_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.7639_dp,  565.2639_dp,  534.1256_dp,  469.7122_dp,  404.8543_dp, &
      &  324.3435_dp,  302.6272_dp,  287.5851_dp,  256.5149_dp,  225.1326_dp, &
      &  312.4130_dp,  291.8713_dp,  277.6090_dp,  248.1606_dp,  218.4003_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  479.9639_dp,  444.4196_dp,  420.0642_dp,  369.7049_dp,  319.0075_dp, &
      &  321.6404_dp,  299.9706_dp,  284.9729_dp,  253.9905_dp,  222.7038_dp, &
      &  232.7992_dp,  218.2207_dp,  208.0381_dp,  187.0285_dp,  165.7746_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  524.1848_dp,  485.7648_dp,  459.4314_dp,  404.9622_dp,  350.0882_dp, &
      &  294.5184_dp,  275.2607_dp,  261.8866_dp,  234.2680_dp,  206.3483_dp, &
      &  197.9394_dp,  185.9246_dp,  177.4973_dp,  160.1207_dp,  142.5295_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  410.2106_dp,  380.3853_dp,  359.9172_dp,  317.5943_dp,  274.9614_dp, &
      &  271.9711_dp,  254.3735_dp,  242.1349_dp,  216.8675_dp,  191.3189_dp, &
      &  194.4157_dp,  182.5691_dp,  174.2654_dp,  157.1410_dp,  139.8068_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  433.8508_dp,  402.9146_dp,  381.6588_dp,  337.6931_dp,  293.3577_dp, &
      &  259.3224_dp,  242.5572_dp,  230.8956_dp,  206.8210_dp,  182.4798_dp, &
      &  230.2133_dp,  215.4752_dp,  205.2122_dp,  184.0272_dp,  162.6068_dp, &
      &  203.6278_dp,  191.0014_dp,  182.1731_dp,  163.9594_dp,  145.5291_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  447.9868_dp,  415.6439_dp,  393.4530_dp,  347.5445_dp,  301.2669_dp, &
      &  253.6134_dp,  237.1737_dp,  225.7435_dp,  202.1442_dp,  178.2858_dp, &
      &  213.9134_dp,  200.4003_dp,  190.9759_dp,  171.5246_dp,  151.8497_dp, &
      &  225.7080_dp,  211.0440_dp,  200.8574_dp,  179.8178_dp,  158.5536_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  342.4261_dp,  317.8624_dp,  300.9874_dp,  266.0920_dp,  230.9258_dp, &
      &  251.5386_dp,  234.8327_dp,  223.2574_dp,  199.3430_dp,  175.1844_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  349.9885_dp,  325.6928_dp,  308.9517_dp,  274.3339_dp,  239.3959_dp, &
      &  234.3407_dp,  219.4508_dp,  209.0707_dp,  187.6481_dp,  165.9764_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  415.9530_dp,  386.7129_dp,  366.5742_dp,  324.9456_dp,  282.9469_dp, &
      &  389.7370_dp,  362.6832_dp,  344.0232_dp,  305.4579_dp,  266.5350_dp, &
      &  306.0934_dp,  285.8049_dp,  271.7272_dp,  242.6615_dp,  213.2887_dp, &
      &  248.1866_dp,  232.5085_dp,  221.5612_dp,  198.9801_dp,  176.1294_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  363.5474_dp,  339.2404_dp,  322.3974_dp,  287.6103_dp,  252.4591_dp, &
      &  339.2404_dp,  316.8521_dp,  301.3102_dp,  269.2215_dp,  236.7867_dp, &
      &  322.3974_dp,  301.3102_dp,  286.6545_dp,  256.4009_dp,  225.8139_dp, &
      &  287.6103_dp,  269.2215_dp,  256.4009_dp,  229.9498_dp,  203.1922_dp, &
      &  252.4591_dp,  236.7867_dp,  225.8139_dp,  203.1922_dp,  180.2912_dp, &
      &   27.7884_dp,   27.3574_dp,   26.7042_dp,   25.8144_dp,    0.0000_dp, &
      &   44.6512_dp,   43.9067_dp,   42.7789_dp,   41.2434_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   18.3284_dp,   18.0816_dp,   17.7051_dp,   17.1900_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  415.1532_dp,  406.3427_dp,  392.4365_dp,  373.0901_dp,    0.0000_dp, &
      &  142.8471_dp,  140.1353_dp,  135.9832_dp,  130.3034_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  245.4831_dp,  240.7559_dp,  233.5151_dp,  223.6099_dp,    0.0000_dp, &
      &  182.5338_dp,  179.1390_dp,  173.9415_dp,  166.8295_dp,    0.0000_dp, &
      &  120.2224_dp,  118.1520_dp,  115.0137_dp,  110.7405_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(532 * maxRef * maxRef + 1 : 551 * maxRef * maxRef) = [ &
      &  166.9060_dp,  163.9655_dp,  159.4969_dp,  153.4055_dp,    0.0000_dp, &
      &  140.5625_dp,  138.1446_dp,  134.4704_dp,  129.4615_dp,    0.0000_dp, &
      &  118.1354_dp,  116.1535_dp,  113.1412_dp,  109.0331_dp,    0.0000_dp, &
      &   90.1373_dp,   88.7108_dp,   86.5486_dp,   83.6036_dp,    0.0000_dp, &
      &   84.8298_dp,   83.5023_dp,   81.4896_dp,   78.7478_dp,    0.0000_dp, &
      &  113.2616_dp,  111.4257_dp,  108.6407_dp,  104.8462_dp,    0.0000_dp, &
      &  106.1686_dp,  104.4532_dp,  101.8469_dp,   98.2930_dp,    0.0000_dp, &
      &   86.6895_dp,   85.3379_dp,   83.2865_dp,   80.4902_dp,    0.0000_dp, &
      &   81.0563_dp,   79.7986_dp,   77.8874_dp,   75.2808_dp,    0.0000_dp, &
      &   66.4147_dp,   65.4414_dp,   63.9621_dp,   61.9438_dp,    0.0000_dp, &
      &   79.3086_dp,   78.1138_dp,   76.2992_dp,   73.8246_dp,    0.0000_dp, &
      &   73.9142_dp,   72.8073_dp,   71.1238_dp,   68.8264_dp,    0.0000_dp, &
      &   69.3770_dp,   68.3477_dp,   66.7818_dp,   64.6446_dp,    0.0000_dp, &
      &   61.0425_dp,   60.1519_dp,   58.7945_dp,   56.9398_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.0289_dp,   59.1764_dp,   57.8784_dp,   56.1057_dp,    0.0000_dp, &
      &   53.8153_dp,   53.0645_dp,   51.9199_dp,   50.3555_dp,    0.0000_dp, &
      &   47.4202_dp,   46.7753_dp,   45.7899_dp,   44.4417_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   45.4180_dp,   44.8079_dp,   43.8762_dp,   42.6013_dp,    0.0000_dp, &
      &   36.9389_dp,   36.4728_dp,   35.7582_dp,   34.7785_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   34.8686_dp,   34.4226_dp,   33.7391_dp,   32.8021_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  496.9338_dp,  486.4638_dp,  469.9596_dp,  447.0179_dp,    0.0000_dp, &
      &  208.1802_dp,  204.1937_dp,  198.0692_dp,  189.6770_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.4927_dp,  381.8396_dp,  370.0638_dp,  353.9178_dp,    0.0000_dp, &
      &  321.8732_dp,  315.6688_dp,  306.1275_dp,  293.0440_dp,    0.0000_dp, &
      &  212.3392_dp,  208.5236_dp,  202.7244_dp,  194.8198_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  361.1913_dp,  354.3706_dp,  343.9343_dp,  329.6624_dp,    0.0000_dp, &
      &  327.7229_dp,  321.6043_dp,  312.2501_dp,  299.4628_dp,    0.0000_dp, &
      &  256.4231_dp,  251.8127_dp,  244.7896_dp,  235.2041_dp,    0.0000_dp, &
      &  200.7514_dp,  197.3088_dp,  192.0916_dp,  184.9886_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  286.5909_dp,  281.4893_dp,  273.7334_dp,  263.1595_dp,    0.0000_dp, &
      &  265.0711_dp,  260.4044_dp,  253.3112_dp,  243.6410_dp,    0.0000_dp, &
      &  246.7379_dp,  242.4359_dp,  235.9012_dp,  226.9949_dp,    0.0000_dp, &
      &  240.2303_dp,  236.0287_dp,  229.6362_dp,  220.9165_dp,    0.0000_dp, &
      &  198.4079_dp,  195.0844_dp,  190.0483_dp,  183.1915_dp,    0.0000_dp, &
      &  224.4774_dp,  220.7146_dp,  215.0103_dp,  207.2419_dp,    0.0000_dp, &
      &  217.8920_dp,  214.2514_dp,  208.7309_dp,  201.2119_dp,    0.0000_dp, &
      &  214.8363_dp,  211.2380_dp,  205.7784_dp,  198.3402_dp,    0.0000_dp, &
      &  199.5009_dp,  196.2122_dp,  191.2265_dp,  184.4364_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  186.6348_dp,  183.6462_dp,  179.1160_dp,  172.9456_dp,    0.0000_dp, &
      &  184.5467_dp,  181.5905_dp,  177.1085_dp,  171.0030_dp,    0.0000_dp, &
      &  180.8007_dp,  177.9088_dp,  173.5234_dp,  167.5491_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  152.6035_dp,  150.2742_dp,  146.7405_dp,  141.9245_dp,    0.0000_dp, &
      &  150.9999_dp,  148.6943_dp,  145.1961_dp,  140.4282_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.8561_dp,  123.0358_dp,  120.2698_dp,  116.4964_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  810.6554_dp,  793.6306_dp,  766.5311_dp,  728.6195_dp,    0.0000_dp, &
      &  287.5535_dp,  282.3070_dp,  274.2325_dp,  263.1547_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  679.0044_dp,  665.3221_dp,  644.0351_dp,  614.6705_dp,    0.0000_dp, &
      &  543.0123_dp,  532.2461_dp,  515.4323_dp,  492.1704_dp,    0.0000_dp, &
      &  297.1426_dp,  291.8601_dp,  283.8013_dp,  272.7959_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  562.6947_dp,  551.4999_dp,  534.1575_dp,  510.2912_dp,    0.0000_dp, &
      &  289.5582_dp,  284.4516_dp,  276.6634_dp,  266.0294_dp,    0.0000_dp, &
      &  286.3045_dp,  281.3331_dp,  273.7684_dp,  263.4496_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  544.6797_dp,  534.0266_dp,  517.5611_dp,  494.9276_dp,    0.0000_dp, &
      &  279.4990_dp,  274.5929_dp,  267.1055_dp,  256.8763_dp,    0.0000_dp, &
      &  289.8560_dp,  284.8345_dp,  277.1939_dp,  266.7717_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  499.4970_dp,  489.8210_dp,  474.8898_dp,  454.3822_dp,    0.0000_dp, &
      &  273.8352_dp,  269.0270_dp,  261.6872_dp,  251.6587_dp,    0.0000_dp, &
      &  265.0778_dp,  260.4970_dp,  253.5169_dp,  243.9879_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(551 * maxRef * maxRef + 1 : 570 * maxRef * maxRef) = [ &
      &  393.1922_dp,  385.6577_dp,  374.0038_dp,  357.9757_dp,    0.0000_dp, &
      &  271.0770_dp,  266.2939_dp,  258.9856_dp,  248.9955_dp,    0.0000_dp, &
      &  200.0486_dp,  196.7631_dp,  191.7598_dp,  184.9307_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  430.9146_dp,  422.6874_dp,  410.0199_dp,  392.6408_dp,    0.0000_dp, &
      &  250.2831_dp,  245.9713_dp,  239.4120_dp,  230.4659_dp,    0.0000_dp, &
      &  171.4077_dp,  168.6780_dp,  164.5248_dp,  158.8578_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  338.0096_dp,  331.6389_dp,  321.8092_dp,  308.3075_dp,    0.0000_dp, &
      &  231.7618_dp,  227.8113_dp,  221.8034_dp,  213.6098_dp,    0.0000_dp, &
      &  168.1967_dp,  165.5084_dp,  161.4178_dp,  155.8359_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  359.7150_dp,  353.0074_dp,  342.7215_dp,  328.6403_dp,    0.0000_dp, &
      &  221.0233_dp,  217.2635_dp,  211.5422_dp,  203.7369_dp,    0.0000_dp, &
      &  196.7046_dp,  193.4021_dp,  188.3702_dp,  181.5005_dp,    0.0000_dp, &
      &  175.4121_dp,  172.5578_dp,  168.2143_dp,  162.2878_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.0359_dp,  363.0566_dp,  352.3403_dp,  337.6607_dp,    0.0000_dp, &
      &  216.0043_dp,  212.3221_dp,  206.7172_dp,  199.0695_dp,    0.0000_dp, &
      &  183.4110_dp,  180.3695_dp,  175.7399_dp,  169.4227_dp,    0.0000_dp, &
      &  192.1103_dp,  188.8450_dp,  183.8623_dp,  177.0557_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  283.3323_dp,  278.0578_dp,  269.9330_dp,  258.7830_dp,    0.0000_dp, &
      &  212.8346_dp,  209.1322_dp,  203.4807_dp,  195.7595_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  292.5030_dp,  287.1778_dp,  279.0375_dp,  267.9108_dp,    0.0000_dp, &
      &  200.6408_dp,  197.2771_dp,  192.1686_dp,  185.2060_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.3508_dp,  339.9625_dp,  330.1881_dp,  316.8201_dp,    0.0000_dp, &
      &  325.7268_dp,  319.7871_dp,  310.7110_dp,  298.3060_dp,    0.0000_dp, &
      &  259.1650_dp,  254.6359_dp,  247.7421_dp,  238.3364_dp,    0.0000_dp, &
      &  212.8299_dp,  209.2695_dp,  203.8708_dp,  196.5179_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.1010_dp,  301.6800_dp,  293.4322_dp,  282.1828_dp,    0.0000_dp, &
      &  287.5847_dp,  282.5717_dp,  274.9497_dp,  264.5566_dp,    0.0000_dp, &
      &  273.9655_dp,  269.2308_dp,  262.0354_dp,  252.2261_dp,    0.0000_dp, &
      &  245.8719_dp,  241.7135_dp,  235.4019_dp,  226.8019_dp,    0.0000_dp, &
      &  217.4476_dp,  213.8707_dp,  208.4500_dp,  201.0683_dp,    0.0000_dp, &
      &  262.9498_dp,  258.5263_dp,  251.8155_dp,  242.6736_dp,    0.0000_dp, &
      &  258.5263_dp,  254.1924_dp,  247.6174_dp,  238.6601_dp,    0.0000_dp, &
      &  251.8155_dp,  247.6174_dp,  241.2494_dp,  232.5744_dp,    0.0000_dp, &
      &  242.6736_dp,  238.6601_dp,  232.5744_dp,  224.2851_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   25.3098_dp,   25.3149_dp,   25.1268_dp,    0.0000_dp,    0.0000_dp, &
      &   40.2010_dp,   40.2117_dp,   39.8967_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   17.0243_dp,   17.0280_dp,   16.9157_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  354.5429_dp,  355.0450_dp,  351.8104_dp,    0.0000_dp,    0.0000_dp, &
      &  125.4768_dp,  125.5564_dp,  124.4868_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  215.0238_dp,  215.1671_dp,  213.3149_dp,    0.0000_dp,    0.0000_dp, &
      &  160.9687_dp,  161.0637_dp,  159.7099_dp,    0.0000_dp,    0.0000_dp, &
      &  107.6371_dp,  107.6713_dp,  106.8086_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  148.8008_dp,  148.8590_dp,  147.6502_dp,    0.0000_dp,    0.0000_dp, &
      &  125.8437_dp,  125.8888_dp,  124.8845_dp,    0.0000_dp,    0.0000_dp, &
      &  106.2140_dp,  106.2491_dp,  105.4167_dp,    0.0000_dp,    0.0000_dp, &
      &   81.8381_dp,   81.8564_dp,   81.2400_dp,    0.0000_dp,    0.0000_dp, &
      &   77.1533_dp,   77.1700_dp,   76.5939_dp,    0.0000_dp,    0.0000_dp, &
      &  102.4346_dp,  102.4620_dp,  101.6786_dp,    0.0000_dp,    0.0000_dp, &
      &   96.0599_dp,   96.0872_dp,   95.3553_dp,    0.0000_dp,    0.0000_dp, &
      &   78.8838_dp,   78.9029_dp,   78.3177_dp,    0.0000_dp,    0.0000_dp, &
      &   73.8064_dp,   73.8257_dp,   73.2814_dp,    0.0000_dp,    0.0000_dp, &
      &   60.9913_dp,   61.0041_dp,   60.5723_dp,    0.0000_dp,    0.0000_dp, &
      &   72.5403_dp,   72.5567_dp,   72.0328_dp,    0.0000_dp,    0.0000_dp, &
      &   67.6636_dp,   67.6796_dp,   67.1936_dp,    0.0000_dp,    0.0000_dp, &
      &   63.5919_dp,   63.6074_dp,   63.1549_dp,    0.0000_dp,    0.0000_dp, &
      &   56.0812_dp,   56.0962_dp,   55.7037_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   55.3627_dp,   55.3752_dp,   54.9943_dp,    0.0000_dp,    0.0000_dp, &
      &   49.7552_dp,   49.7665_dp,   49.4291_dp,    0.0000_dp,    0.0000_dp, &
      &   43.9819_dp,   43.9931_dp,   43.7022_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.1923_dp,   42.2027_dp,   41.9260_dp,    0.0000_dp,    0.0000_dp, &
      &   34.5792_dp,   34.5882_dp,   34.3729_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(570 * maxRef * maxRef + 1 : 589 * maxRef * maxRef) = [ &
      &   32.5843_dp,   32.5935_dp,   32.3892_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.1864_dp,  425.7733_dp,  421.9195_dp,    0.0000_dp,    0.0000_dp, &
      &  182.4744_dp,  182.6062_dp,  181.0487_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  339.5742_dp,  339.8427_dp,  336.8869_dp,    0.0000_dp,    0.0000_dp, &
      &  281.7173_dp,  281.9249_dp,  279.5035_dp,    0.0000_dp,    0.0000_dp, &
      &  188.6214_dp,  188.7025_dp,  187.1504_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  317.6182_dp,  317.8133_dp,  315.1147_dp,    0.0000_dp,    0.0000_dp, &
      &  288.8485_dp,  289.0163_dp,  286.5809_dp,    0.0000_dp,    0.0000_dp, &
      &  227.6936_dp,  227.7982_dp,  225.9245_dp,    0.0000_dp,    0.0000_dp, &
      &  179.8635_dp,  179.9198_dp,  178.4835_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  255.0099_dp,  255.1153_dp,  253.0289_dp,    0.0000_dp,    0.0000_dp, &
      &  236.3323_dp,  236.4253_dp,  234.5066_dp,    0.0000_dp,    0.0000_dp, &
      &  220.3806_dp,  220.4624_dp,  218.6853_dp,    0.0000_dp,    0.0000_dp, &
      &  214.4121_dp,  214.4981_dp,  212.7679_dp,    0.0000_dp,    0.0000_dp, &
      &  178.4799_dp,  178.5302_dp,  177.1287_dp,    0.0000_dp,    0.0000_dp, &
      &  201.9019_dp,  201.9603_dp,  200.3747_dp,    0.0000_dp,    0.0000_dp, &
      &  196.0806_dp,  196.1374_dp,  194.6017_dp,    0.0000_dp,    0.0000_dp, &
      &  193.2410_dp,  193.2996_dp,  191.7848_dp,    0.0000_dp,    0.0000_dp, &
      &  179.9374_dp,  179.9862_dp,  178.5907_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.1308_dp,  169.1712_dp,  167.8868_dp,    0.0000_dp,    0.0000_dp, &
      &  167.2264_dp,  167.2672_dp,  165.9975_dp,    0.0000_dp,    0.0000_dp, &
      &  163.8681_dp,  163.9083_dp,  162.6659_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.3163_dp,  139.3453_dp,  138.3253_dp,    0.0000_dp,    0.0000_dp, &
      &  137.8443_dp,  137.8735_dp,  136.8644_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  114.7449_dp,  114.7677_dp,  113.9577_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  692.9116_dp,  693.9818_dp,  687.7370_dp,    0.0000_dp,    0.0000_dp, &
      &  254.3670_dp,  254.5376_dp,  252.4498_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.9647_dp,  588.5763_dp,  583.4009_dp,    0.0000_dp,    0.0000_dp, &
      &  471.4822_dp,  471.9821_dp,  467.8832_dp,    0.0000_dp,    0.0000_dp, &
      &  264.3790_dp,  264.5072_dp,  262.3613_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  488.8715_dp,  489.3305_dp,  485.0594_dp,    0.0000_dp,    0.0000_dp, &
      &  258.0084_dp,  258.1295_dp,  256.0474_dp,    0.0000_dp,    0.0000_dp, &
      &  255.8702_dp,  255.9746_dp,  253.9272_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  475.0348_dp,  475.4453_dp,  471.3428_dp,    0.0000_dp,    0.0000_dp, &
      &  249.2258_dp,  249.3437_dp,  247.3402_dp,    0.0000_dp,    0.0000_dp, &
      &  259.1489_dp,  259.2539_dp,  257.1841_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  436.5718_dp,  436.9293_dp,  433.1830_dp,    0.0000_dp,    0.0000_dp, &
      &  244.1554_dp,  244.2727_dp,  242.3111_dp,    0.0000_dp,    0.0000_dp, &
      &  237.0577_dp,  237.1592_dp,  235.2729_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  344.2956_dp,  344.5858_dp,  341.6637_dp,    0.0000_dp,    0.0000_dp, &
      &  241.4585_dp,  241.5803_dp,  239.6357_dp,    0.0000_dp,    0.0000_dp, &
      &  180.4705_dp,  180.5365_dp,  179.1559_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  377.8276_dp,  378.1133_dp,  374.9026_dp,    0.0000_dp,    0.0000_dp, &
      &  223.9857_dp,  224.0757_dp,  222.2961_dp,    0.0000_dp,    0.0000_dp, &
      &  155.4154_dp,  155.4652_dp,  154.3023_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  297.0396_dp,  297.2697_dp,  294.7778_dp,    0.0000_dp,    0.0000_dp, &
      &  207.7948_dp,  207.8748_dp,  206.2367_dp,    0.0000_dp,    0.0000_dp, &
      &  152.4128_dp,  152.4629_dp,  151.3202_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  317.0174_dp,  317.2233_dp,  314.5706_dp,    0.0000_dp,    0.0000_dp, &
      &  198.2213_dp,  198.2993_dp,  196.7403_dp,    0.0000_dp,    0.0000_dp, &
      &  176.7844_dp,  176.8558_dp,  175.4826_dp,    0.0000_dp,    0.0000_dp, &
      &  158.4907_dp,  158.5463_dp,  157.3426_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.3419_dp,  325.5667_dp,  322.8238_dp,    0.0000_dp,    0.0000_dp, &
      &  193.6432_dp,  193.7213_dp,  192.1970_dp,    0.0000_dp,    0.0000_dp, &
      &  165.1961_dp,  165.2584_dp,  163.9864_dp,    0.0000_dp,    0.0000_dp, &
      &  172.2718_dp,  172.3493_dp,  171.0031_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(589 * maxRef * maxRef + 1 : 608 * maxRef * maxRef) = [ &
      &  249.6385_dp,  249.8210_dp,  247.7460_dp,    0.0000_dp,    0.0000_dp, &
      &  190.0771_dp,  190.1685_dp,  188.6560_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  259.0459_dp,  259.1916_dp,  257.0583_dp,    0.0000_dp,    0.0000_dp, &
      &  180.4189_dp,  180.4812_dp,  179.0747_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.9501_dp,  306.1275_dp,  303.5764_dp,    0.0000_dp,    0.0000_dp, &
      &  288.3934_dp,  288.5493_dp,  286.1624_dp,    0.0000_dp,    0.0000_dp, &
      &  231.3353_dp,  231.4312_dp,  229.5690_dp,    0.0000_dp,    0.0000_dp, &
      &  191.4849_dp,  191.5424_dp,  190.0442_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.6530_dp,  273.7680_dp,  271.5474_dp,    0.0000_dp,    0.0000_dp, &
      &  256.8517_dp,  256.9523_dp,  254.8857_dp,    0.0000_dp,    0.0000_dp, &
      &  245.0690_dp,  245.1604_dp,  243.2002_dp,    0.0000_dp,    0.0000_dp, &
      &  220.7846_dp,  220.8565_dp,  219.1161_dp,    0.0000_dp,    0.0000_dp, &
      &  196.1962_dp,  196.2489_dp,  194.7306_dp,    0.0000_dp,    0.0000_dp, &
      &  236.3473_dp,  236.4206_dp,  234.5631_dp,    0.0000_dp,    0.0000_dp, &
      &  232.5072_dp,  232.5784_dp,  230.7558_dp,    0.0000_dp,    0.0000_dp, &
      &  226.6845_dp,  226.7519_dp,  224.9818_dp,    0.0000_dp,    0.0000_dp, &
      &  218.7531_dp,  218.8149_dp,  217.1161_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.6738_dp,  213.7298_dp,  212.0915_dp,    0.0000_dp,    0.0000_dp, &
      &  213.7298_dp,  213.7862_dp,  212.1475_dp,    0.0000_dp,    0.0000_dp, &
      &  212.0915_dp,  212.1475_dp,  210.5231_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.4834_dp,   22.6230_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.2740_dp,   35.5123_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   15.4553_dp,   15.5387_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.7897_dp,  297.6835_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  107.3380_dp,  108.2048_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  183.3742_dp,  184.8832_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.2747_dp,  139.3617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.8911_dp,   94.5515_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  129.2495_dp,  130.1881_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.8060_dp,  110.5802_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.1003_dp,   93.7374_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.4613_dp,   72.9221_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.4416_dp,   68.8714_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   90.3424_dp,   90.9343_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.7723_dp,   85.3265_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   70.0352_dp,   70.4744_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   65.5868_dp,   65.9967_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   54.6824_dp,   55.0029_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.7632_dp,   65.1543_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.4728_dp,   60.8359_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.9187_dp,   57.2578_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.3344_dp,   50.6298_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.8795_dp,   50.1632_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.9510_dp,   45.2019_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.8821_dp,   40.1002_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   38.3225_dp,   38.5294_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   31.6699_dp,   31.8313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.7958_dp,   29.9501_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  354.2869_dp,  357.7246_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  155.8202_dp,  157.1005_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  288.2736_dp,  290.7276_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.1565_dp,  242.1476_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  163.2221_dp,  164.4383_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  272.0098_dp,  274.1894_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  247.9692_dp,  249.9248_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  196.9743_dp,  198.4473_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  157.0191_dp,  158.1182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(608 * maxRef * maxRef + 1 : 627 * maxRef * maxRef) = [ &
      &  221.0537_dp,  222.6812_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  205.2976_dp,  206.7880_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  191.7992_dp,  193.1738_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  186.4927_dp,  187.8378_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  156.4781_dp,  157.5423_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  176.9930_dp,  178.1984_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  171.9910_dp,  173.1582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.4313_dp,  170.5858_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  158.2104_dp,  159.2666_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  149.4563_dp,  150.4203_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.7679_dp,  148.7219_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.8387_dp,  145.7724_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.0909_dp,  124.8495_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  122.7763_dp,  123.5274_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  102.9471_dp,  103.5467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  577.2318_dp,  582.8914_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  219.4647_dp,  221.1642_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  495.9754_dp,  500.4152_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  399.0226_dp,  402.5380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  229.3147_dp,  231.0107_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  413.7468_dp,  417.3648_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  224.1375_dp,  225.7784_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  222.9256_dp,  224.5209_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  403.6576_dp,  407.0970_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.6921_dp,  218.2711_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  225.8818_dp,  227.4937_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  371.8071_dp,  374.9283_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.2792_dp,  213.8276_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  206.7277_dp,  208.2022_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.9185_dp,  296.3607_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  209.7393_dp,  211.2809_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  158.8704_dp,  159.9362_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  322.8433_dp,  325.4945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.4536_dp,  196.8395_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  137.5406_dp,  138.4296_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.5306_dp,  256.5939_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.6836_dp,  182.9550_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  134.8051_dp,  135.6806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  272.3053_dp,  274.4620_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  173.3793_dp,  174.5910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  155.0169_dp,  156.0859_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.7541_dp,  140.6812_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  278.7705_dp,  281.0154_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.3133_dp,  170.5004_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  145.1828_dp,  146.1681_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  150.7426_dp,  151.8006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.5059_dp,  216.2140_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  165.5860_dp,  166.7816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  223.6348_dp,  225.3457_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  158.2213_dp,  159.3049_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.3458_dp,  265.3939_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  248.8211_dp,  250.7247_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  201.2706_dp,  202.7225_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  167.9500_dp,  169.0925_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  237.6295_dp,  239.3636_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  223.5744_dp,  225.1792_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.6651_dp,  215.1816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.2591_dp,  194.5928_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  172.5823_dp,  173.7316_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(627 * maxRef * maxRef + 1 : 646 * maxRef * maxRef) = [ &
      &  207.0772_dp,  208.4957_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.8400_dp,  205.2305_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  198.9308_dp,  200.2786_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  192.2421_dp,  193.5315_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  188.3605_dp,  189.5974_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  188.4031_dp,  189.6408_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  186.9994_dp,  188.2263_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  167.1297_dp,  168.1791_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  168.1791_dp,  169.2375_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   19.8182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   30.7261_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   13.9211_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  244.5554_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   91.3333_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  155.5616_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  118.1240_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.3306_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.5371_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   95.1763_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.0518_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   63.6768_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.2544_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.1080_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   74.2865_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   61.7254_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   57.8665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   48.6500_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   57.3895_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   53.6496_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.5746_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.8546_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.6003_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   40.3043_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.8961_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   34.5452_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   28.7800_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   27.0430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.5574_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  132.4348_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  243.5965_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.7473_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  140.3605_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  231.6904_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  211.6949_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.3547_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  136.1238_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  190.3934_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  177.1811_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  165.8230_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  161.1703_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  136.2102_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  154.0575_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  149.7933_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.5193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.1076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.0928_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  129.6123_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  127.0791_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(646 * maxRef * maxRef + 1 : 665 * maxRef * maxRef) = [ &
      &  109.6839_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  108.5226_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   91.6430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  480.2472_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  188.4819_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  416.9777_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  336.6483_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.7645_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  348.8474_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.5868_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.0224_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  341.6273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  187.3231_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.6666_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  315.3251_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  183.5171_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  179.1965_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.9413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.1819_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.9788_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.6419_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.5097_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  120.9219_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  217.2071_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  157.8661_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  118.4578_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.7594_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  150.7190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  135.1147_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  122.4536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  237.7447_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.1417_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  126.8094_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.1600_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  183.5347_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  143.4525_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  192.0484_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  137.8590_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  225.4537_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.4821_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  174.0230_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  146.2910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  205.0598_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.3634_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  185.0751_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  168.0194_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  150.7260_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  180.1818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  177.4716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  173.3582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  167.7505_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  164.8494_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  164.8833_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  163.6906_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.1830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  148.0689_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(665 * maxRef * maxRef + 1 : 684 * maxRef * maxRef) = [ &
      &  130.4017_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   85.9499_dp,   35.1963_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  150.9340_dp,   57.9039_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.0491_dp,   22.6231_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2647.3331_dp,  657.4076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  618.5681_dp,  199.6771_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1082.7052_dp,  345.4378_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  774.9656_dp,  252.9574_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.1704_dp,  157.9479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  620.0878_dp,  222.5081_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  506.9421_dp,  185.7206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  415.2201_dp,  154.7827_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  286.2486_dp,  114.9721_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  266.1018_dp,  107.8668_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  373.5384_dp,  145.9974_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  353.6302_dp,  137.2125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.3757_dp,  110.4856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  256.5198_dp,  103.4559_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.5996_dp,   83.2418_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.6940_dp,  100.1418_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  225.1919_dp,   93.4584_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  209.2378_dp,   87.5447_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  183.1486_dp,   77.0040_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  172.3398_dp,   74.8868_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  152.6757_dp,   67.0045_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  132.4879_dp,   58.9015_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.5879_dp,   56.1779_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   96.2235_dp,   45.2489_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   92.3827_dp,   42.8957_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3104.1016_dp,  781.8527_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  936.3006_dp,  294.6795_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1826.4648_dp,  559.1337_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1477.6556_dp,  457.7831_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  808.8537_dp,  285.4219_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1546.9229_dp,  502.9933_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1375.2558_dp,  453.3636_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  999.7400_dp,  346.4959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  704.9609_dp,  263.2439_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1082.0563_dp,  383.9084_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  986.0438_dp,  353.4142_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  901.7739_dp,  327.3080_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  894.0072_dp,  320.2665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  675.8635_dp,  257.9020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  768.1335_dp,  292.1604_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  744.2475_dp,  283.4368_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  739.7269_dp,  280.1183_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  668.4401_dp,  258.1520_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  602.2221_dp,  239.0572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  596.8763_dp,  236.5467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  584.3784_dp,  231.7265_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  466.1565_dp,  192.7877_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  462.0252_dp,  190.8602_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  364.4067_dp,  156.1010_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(684 * maxRef * maxRef + 1 : 703 * maxRef * maxRef) = [ &
      & 5530.2806_dp, 1305.8840_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1240.8631_dp,  401.0371_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3599.4349_dp, 1013.0706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2958.0410_dp,  812.3489_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1152.0197_dp,  401.6572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2834.0920_dp,  826.1041_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1105.4608_dp,  389.9057_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1056.1444_dp,  381.2151_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2641.2435_dp,  789.4117_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1074.1973_dp,  376.3919_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1065.4841_dp,  385.5830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2363.7175_dp,  718.2125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1053.5834_dp,  369.0649_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  984.9504_dp,  353.5929_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1874.0326_dp,  566.1856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1057.5718_dp,  366.8686_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  692.7359_dp,  261.7869_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1968.7631_dp,  612.6444_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.9433_dp,  332.2215_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  566.7169_dp,  221.5102_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1547.1982_dp,  480.6236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  831.7960_dp,  306.2720_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.2177_dp,  217.7195_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1541.8997_dp,  501.5940_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  795.8859_dp,  292.2992_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  704.2730_dp,  259.8039_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  596.8306_dp,  228.5106_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1625.1034_dp,  519.9826_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.8441_dp,  286.1241_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  640.5782_dp,  240.6629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  705.9328_dp,  255.8512_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1259.4427_dp,  399.3534_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  807.8786_dp,  286.0131_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1186.2631_dp,  401.0548_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  697.2012_dp,  262.7332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1442.3807_dp,  477.9737_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1322.7322_dp,  446.0798_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  968.1682_dp,  345.9921_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  727.9978_dp,  277.1701_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1155.1526_dp,  411.0389_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1059.6044_dp,  382.5012_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  994.2046_dp,  362.8020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  859.6583_dp,  322.1096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.9394_dp,  281.0734_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  909.4257_dp,  343.3978_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  890.4931_dp,  337.2160_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  860.3672_dp,  327.7032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  818.2247_dp,  314.6518_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  778.9204_dp,  304.5312_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  780.0986_dp,  304.7279_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  773.1255_dp,  302.2530_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  650.3606_dp,  263.4592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  656.6574_dp,  265.4647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  542.3160_dp,  226.8590_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 6138.7755_dp, 1457.7817_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1457.7817_dp,  476.2591_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(703 * maxRef * maxRef + 1 : 722 * maxRef * maxRef) = [ &
      &   78.3901_dp,   64.4065_dp,   36.8606_dp,    0.0000_dp,    0.0000_dp, &
      &  135.0664_dp,  110.3275_dp,   60.1892_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   46.6658_dp,   38.8333_dp,   23.8401_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1944.2260_dp, 1618.2465_dp,  628.2399_dp,    0.0000_dp,    0.0000_dp, &
      &  517.8393_dp,  421.4610_dp,  201.6900_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  903.9265_dp,  734.7926_dp,  348.1690_dp,    0.0000_dp,    0.0000_dp, &
      &  648.6422_dp,  529.5494_dp,  256.1295_dp,    0.0000_dp,    0.0000_dp, &
      &  376.6468_dp,  306.9433_dp,  163.4621_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  541.5729_dp,  441.1261_dp,  228.9512_dp,    0.0000_dp,    0.0000_dp, &
      &  445.2608_dp,  363.3607_dp,  191.6429_dp,    0.0000_dp,    0.0000_dp, &
      &  365.6571_dp,  299.2914_dp,  160.1098_dp,    0.0000_dp,    0.0000_dp, &
      &  259.5279_dp,  212.8759_dp,  120.1300_dp,    0.0000_dp,    0.0000_dp, &
      &  241.8504_dp,  198.5660_dp,  112.8093_dp,    0.0000_dp,    0.0000_dp, &
      &  335.4597_dp,  274.7031_dp,  151.9745_dp,    0.0000_dp,    0.0000_dp, &
      &  315.8923_dp,  258.9885_dp,  142.6113_dp,    0.0000_dp,    0.0000_dp, &
      &  247.8285_dp,  203.5706_dp,  115.4107_dp,    0.0000_dp,    0.0000_dp, &
      &  232.0032_dp,  190.6976_dp,  107.9688_dp,    0.0000_dp,    0.0000_dp, &
      &  179.8395_dp,  148.5141_dp,   87.3891_dp,    0.0000_dp,    0.0000_dp, &
      &  219.9608_dp,  181.1973_dp,  104.9004_dp,    0.0000_dp,    0.0000_dp, &
      &  205.1574_dp,  169.1772_dp,   97.8085_dp,    0.0000_dp,    0.0000_dp, &
      &  191.1268_dp,  157.6836_dp,   91.6725_dp,    0.0000_dp,    0.0000_dp, &
      &  167.0865_dp,  138.0927_dp,   80.5868_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  159.1460_dp,  131.7846_dp,   78.6933_dp,    0.0000_dp,    0.0000_dp, &
      &  141.1592_dp,  117.1018_dp,   70.4285_dp,    0.0000_dp,    0.0000_dp, &
      &  122.6928_dp,  102.0116_dp,   61.9159_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  115.9647_dp,   96.5080_dp,   59.1415_dp,    0.0000_dp,    0.0000_dp, &
      &   90.4081_dp,   75.6758_dp,   47.7411_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   86.4627_dp,   72.2703_dp,   45.1941_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2296.1906_dp, 1907.9592_dp,  749.3005_dp,    0.0000_dp,    0.0000_dp, &
      &  773.9854_dp,  630.8304_dp,  296.0005_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1495.8141_dp, 1218.1274_dp,  558.7819_dp,    0.0000_dp,    0.0000_dp, &
      & 1210.9593_dp,  988.7857_dp,  458.8001_dp,    0.0000_dp,    0.0000_dp, &
      &  703.3906_dp,  571.9448_dp,  292.9174_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1298.5356_dp, 1057.0218_dp,  508.8061_dp,    0.0000_dp,    0.0000_dp, &
      & 1160.2539_dp,  944.7535_dp,  459.7370_dp,    0.0000_dp,    0.0000_dp, &
      &  859.5678_dp,  700.6746_dp,  354.5526_dp,    0.0000_dp,    0.0000_dp, &
      &  625.6996_dp,  509.9927_dp,  272.6252_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  941.4121_dp,  766.3847_dp,  394.3455_dp,    0.0000_dp,    0.0000_dp, &
      &  860.3421_dp,  701.0009_dp,  363.5872_dp,    0.0000_dp,    0.0000_dp, &
      &  790.6765_dp,  644.4017_dp,  337.3757_dp,    0.0000_dp,    0.0000_dp, &
      &  778.3539_dp,  634.9125_dp,  329.3386_dp,    0.0000_dp,    0.0000_dp, &
      &  603.7134_dp,  492.9574_dp,  267.8525_dp,    0.0000_dp,    0.0000_dp, &
      &  684.9632_dp,  559.3821_dp,  303.2521_dp,    0.0000_dp,    0.0000_dp, &
      &  663.5656_dp,  542.1262_dp,  294.2134_dp,    0.0000_dp,    0.0000_dp, &
      &  657.7877_dp,  537.4352_dp,  290.4761_dp,    0.0000_dp,    0.0000_dp, &
      &  598.7498_dp,  489.5943_dp,  268.4526_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.9627_dp,  445.7860_dp,  249.4267_dp,    0.0000_dp,    0.0000_dp, &
      &  538.6724_dp,  441.4719_dp,  246.7293_dp,    0.0000_dp,    0.0000_dp, &
      &  527.3838_dp,  432.2783_dp,  241.6949_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.8997_dp,  350.4273_dp,  201.9997_dp,    0.0000_dp,    0.0000_dp, &
      &  421.8867_dp,  347.1255_dp,  199.9353_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  335.8690_dp,  277.4759_dp,  164.0275_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3915.3529_dp, 3302.2504_dp, 1234.9320_dp,    0.0000_dp,    0.0000_dp, &
      & 1027.9671_dp,  841.9983_dp,  404.3537_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(722 * maxRef * maxRef + 1 : 741 * maxRef * maxRef) = [ &
      & 2819.4935_dp, 2314.8420_dp,  994.4676_dp,    0.0000_dp,    0.0000_dp, &
      & 2265.4155_dp, 1876.9417_dp,  794.5479_dp,    0.0000_dp,    0.0000_dp, &
      &  991.9143_dp,  808.6909_dp,  410.7857_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2260.4217_dp, 1849.7319_dp,  816.9275_dp,    0.0000_dp,    0.0000_dp, &
      &  957.2271_dp,  779.9488_dp,  399.3277_dp,    0.0000_dp,    0.0000_dp, &
      &  922.1430_dp,  752.1641_dp,  392.1967_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2128.3951_dp, 1740.7410_dp,  784.6886_dp,    0.0000_dp,    0.0000_dp, &
      &  923.6557_dp,  754.6753_dp,  385.2756_dp,    0.0000_dp,    0.0000_dp, &
      &  931.2172_dp,  759.5978_dp,  396.8216_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1918.6941_dp, 1568.2263_dp,  716.2778_dp,    0.0000_dp,    0.0000_dp, &
      &  906.2364_dp,  740.1763_dp,  377.6508_dp,    0.0000_dp,    0.0000_dp, &
      &  855.9005_dp,  699.1302_dp,  363.3125_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1511.3043_dp, 1238.5036_dp,  563.7114_dp,    0.0000_dp,    0.0000_dp, &
      &  905.6187_dp,  739.8315_dp,  374.7437_dp,    0.0000_dp,    0.0000_dp, &
      &  612.3549_dp,  501.5114_dp,  270.7543_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1614.7190_dp, 1318.8096_dp,  613.8418_dp,    0.0000_dp,    0.0000_dp, &
      &  799.0125_dp,  651.8405_dp,  342.1821_dp,    0.0000_dp,    0.0000_dp, &
      &  506.8036_dp,  415.8340_dp,  230.1231_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1263.2327_dp, 1034.0602_dp,  481.0447_dp,    0.0000_dp,    0.0000_dp, &
      &  731.2330_dp,  596.9767_dp,  315.9355_dp,    0.0000_dp,    0.0000_dp, &
      &  499.4375_dp,  409.6773_dp,  226.0537_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1290.8718_dp, 1052.2336_dp,  506.7272_dp,    0.0000_dp,    0.0000_dp, &
      &  698.0762_dp,  570.3024_dp,  301.3614_dp,    0.0000_dp,    0.0000_dp, &
      &  617.2493_dp,  504.9905_dp,  267.8234_dp,    0.0000_dp,    0.0000_dp, &
      &  530.3589_dp,  434.4503_dp,  236.7608_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1351.3702_dp, 1101.6666_dp,  523.7002_dp,    0.0000_dp,    0.0000_dp, &
      &  684.7715_dp,  559.3883_dp,  294.8004_dp,    0.0000_dp,    0.0000_dp, &
      &  565.7790_dp,  462.8902_dp,  248.7379_dp,    0.0000_dp,    0.0000_dp, &
      &  614.6139_dp,  502.4634_dp,  262.8982_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1037.7284_dp,  848.7952_dp,  401.1714_dp,    0.0000_dp,    0.0000_dp, &
      &  697.8623_dp,  569.9151_dp,  292.9783_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1009.6809_dp,  822.5167_dp,  407.9499_dp,    0.0000_dp,    0.0000_dp, &
      &  618.5928_dp,  505.2608_dp,  271.9818_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1216.5138_dp,  991.9123_dp,  484.8813_dp,    0.0000_dp,    0.0000_dp, &
      & 1123.9167_dp,  916.2312_dp,  453.9182_dp,    0.0000_dp,    0.0000_dp, &
      &  841.3610_dp,  686.5985_dp,  355.5621_dp,    0.0000_dp,    0.0000_dp, &
      &  649.2640_dp,  530.2448_dp,  287.6167_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1004.4824_dp,  818.4887_dp,  422.1813_dp,    0.0000_dp,    0.0000_dp, &
      &  926.0025_dp,  755.0584_dp,  393.7660_dp,    0.0000_dp,    0.0000_dp, &
      &  872.4465_dp,  711.5862_dp,  374.0975_dp,    0.0000_dp,    0.0000_dp, &
      &  761.7364_dp,  621.9263_dp,  333.4648_dp,    0.0000_dp,    0.0000_dp, &
      &  650.4849_dp,  531.8762_dp,  292.4289_dp,    0.0000_dp,    0.0000_dp, &
      &  808.3370_dp,  660.0928_dp,  355.9394_dp,    0.0000_dp,    0.0000_dp, &
      &  792.0816_dp,  647.0220_dp,  349.6598_dp,    0.0000_dp,    0.0000_dp, &
      &  766.7421_dp,  626.5605_dp,  340.0688_dp,    0.0000_dp,    0.0000_dp, &
      &  731.7462_dp,  598.2102_dp,  326.9624_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  700.0532_dp,  573.0850_dp,  317.0938_dp,    0.0000_dp,    0.0000_dp, &
      &  700.8015_dp,  573.7135_dp,  317.2498_dp,    0.0000_dp,    0.0000_dp, &
      &  694.6556_dp,  568.7416_dp,  314.6933_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  590.7326_dp,  485.0650_dp,  275.4476_dp,    0.0000_dp,    0.0000_dp, &
      &  596.0214_dp,  489.3461_dp,  277.4689_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  496.9120_dp,  409.3761_dp,  237.9219_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4349.2577_dp, 3670.9261_dp, 1380.8411_dp,    0.0000_dp,    0.0000_dp, &
      & 1210.8108_dp,  992.7970_dp,  481.0605_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3381.3672_dp, 2782.4577_dp, 1185.8018_dp,    0.0000_dp,    0.0000_dp, &
      & 2782.4577_dp, 2312.6602_dp,  968.0787_dp,    0.0000_dp,    0.0000_dp, &
      & 1185.8018_dp,  968.0787_dp,  491.7448_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(741 * maxRef * maxRef + 1 : 760 * maxRef * maxRef) = [ &
      &   70.0570_dp,   51.2356_dp,   36.0543_dp,    0.0000_dp,    0.0000_dp, &
      &  119.2961_dp,   85.6815_dp,   58.2755_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.4195_dp,   31.9889_dp,   23.6462_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1585.3950_dp, 1074.5875_dp,  567.3202_dp,    0.0000_dp,    0.0000_dp, &
      &  442.9448_dp,  306.8862_dp,  189.8894_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  771.4844_dp,  532.5110_dp,  326.9186_dp,    0.0000_dp,    0.0000_dp, &
      &  556.1503_dp,  387.6564_dp,  242.0080_dp,    0.0000_dp,    0.0000_dp, &
      &  330.7190_dp,  235.5232_dp,  157.4277_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  472.4364_dp,  334.1351_dp,  219.3347_dp,    0.0000_dp,    0.0000_dp, &
      &  389.8823_dp,  277.4050_dp,  184.2847_dp,    0.0000_dp,    0.0000_dp, &
      &  321.2652_dp,  230.1384_dp,  154.5280_dp,    0.0000_dp,    0.0000_dp, &
      &  231.1173_dp,  168.1354_dp,  117.1570_dp,    0.0000_dp,    0.0000_dp, &
      &  215.7263_dp,  157.3598_dp,  110.1737_dp,    0.0000_dp,    0.0000_dp, &
      &  297.2171_dp,  214.7760_dp,  147.6049_dp,    0.0000_dp,    0.0000_dp, &
      &  279.4727_dp,  201.9744_dp,  138.4353_dp,    0.0000_dp,    0.0000_dp, &
      &  220.8830_dp,  161.1115_dp,  112.6680_dp,    0.0000_dp,    0.0000_dp, &
      &  206.6589_dp,  150.7894_dp,  105.3858_dp,    0.0000_dp,    0.0000_dp, &
      &  161.7738_dp,  119.7654_dp,   85.9668_dp,    0.0000_dp,    0.0000_dp, &
      &  197.0611_dp,  144.9207_dp,  102.8479_dp,    0.0000_dp,    0.0000_dp, &
      &  183.6851_dp,  135.1839_dp,   95.8897_dp,    0.0000_dp,    0.0000_dp, &
      &  171.3574_dp,  126.3470_dp,   89.9626_dp,    0.0000_dp,    0.0000_dp, &
      &  149.8906_dp,  110.8238_dp,   79.1550_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  143.6602_dp,  107.0438_dp,   77.6335_dp,    0.0000_dp,    0.0000_dp, &
      &  127.6282_dp,   95.4458_dp,   69.5806_dp,    0.0000_dp,    0.0000_dp, &
      &  111.1606_dp,   83.5108_dp,   61.2796_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.3447_dp,   79.4076_dp,   58.6359_dp,    0.0000_dp,    0.0000_dp, &
      &   82.7337_dp,   63.1889_dp,   47.5914_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   78.9316_dp,   60.0628_dp,   44.9753_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1876.8634_dp, 1272.7028_dp,  678.1726_dp,    0.0000_dp,    0.0000_dp, &
      &  658.7963_dp,  454.9928_dp,  277.5701_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1266.7602_dp,  869.3739_dp,  521.1430_dp,    0.0000_dp,    0.0000_dp, &
      & 1028.1365_dp,  709.6961_dp,  429.5059_dp,    0.0000_dp,    0.0000_dp, &
      &  611.6961_dp,  430.3940_dp,  279.6834_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1112.2980_dp,  771.8606_dp,  479.6884_dp,    0.0000_dp,    0.0000_dp, &
      &  996.3757_dp,  693.4732_dp,  434.5102_dp,    0.0000_dp,    0.0000_dp, &
      &  745.1801_dp,  524.2267_dp,  338.0153_dp,    0.0000_dp,    0.0000_dp, &
      &  549.9242_dp,  392.0660_dp,  262.7634_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  819.5418_dp,  578.0550_dp,  377.0639_dp,    0.0000_dp,    0.0000_dp, &
      &  750.3817_dp,  530.8180_dp,  348.3162_dp,    0.0000_dp,    0.0000_dp, &
      &  691.2029_dp,  490.2154_dp,  323.8305_dp,    0.0000_dp,    0.0000_dp, &
      &  678.7577_dp,  480.7496_dp,  315.6210_dp,    0.0000_dp,    0.0000_dp, &
      &  532.6876_dp,  382.0537_dp,  259.1062_dp,    0.0000_dp,    0.0000_dp, &
      &  603.9916_dp,  433.0171_dp,  293.2324_dp,    0.0000_dp,    0.0000_dp, &
      &  585.2467_dp,  419.8718_dp,  284.5819_dp,    0.0000_dp,    0.0000_dp, &
      &  579.5105_dp,  415.3578_dp,  280.7471_dp,    0.0000_dp,    0.0000_dp, &
      &  529.4213_dp,  381.1414_dp,  260.2254_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  483.3797_dp,  350.5775_dp,  242.8256_dp,    0.0000_dp,    0.0000_dp, &
      &  478.5204_dp,  346.9710_dp,  240.1493_dp,    0.0000_dp,    0.0000_dp, &
      &  468.5269_dp,  339.8082_dp,  235.2717_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  381.2462_dp,  279.7355_dp,  197.8796_dp,    0.0000_dp,    0.0000_dp, &
      &  377.5642_dp,  276.9773_dp,  195.8267_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  302.5143_dp,  224.3126_dp,  161.5056_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3166.0625_dp, 2163.7031_dp, 1110.5426_dp,    0.0000_dp,    0.0000_dp, &
      &  879.0346_dp,  613.8463_dp,  381.7778_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(760 * maxRef * maxRef + 1 : 779 * maxRef * maxRef) = [ &
      & 2350.8018_dp, 1603.6362_dp,  916.0855_dp,    0.0000_dp,    0.0000_dp, &
      & 1882.1338_dp, 1293.5159_dp,  732.1243_dp,    0.0000_dp,    0.0000_dp, &
      &  860.3229_dp,  605.8453_dp,  391.8360_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1896.7809_dp, 1297.2964_dp,  756.5443_dp,    0.0000_dp,    0.0000_dp, &
      &  831.6679_dp,  586.3135_dp,  381.4719_dp,    0.0000_dp,    0.0000_dp, &
      &  805.0465_dp,  570.8618_dp,  376.1623_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1794.3180_dp, 1232.3433_dp,  730.0584_dp,    0.0000_dp,    0.0000_dp, &
      &  802.0324_dp,  566.8670_dp,  368.1481_dp,    0.0000_dp,    0.0000_dp, &
      &  813.3302_dp,  577.0286_dp,  380.7419_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1622.4689_dp, 1116.9531_dp,  668.2729_dp,    0.0000_dp,    0.0000_dp, &
      &  786.7255_dp,  555.7215_dp,  360.7708_dp,    0.0000_dp,    0.0000_dp, &
      &  746.4335_dp,  529.6824_dp,  348.3728_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1276.4352_dp,  880.6069_dp,  526.0324_dp,    0.0000_dp,    0.0000_dp, &
      &  784.7489_dp,  553.4764_dp,  357.4837_dp,    0.0000_dp,    0.0000_dp, &
      &  538.9029_dp,  387.0982_dp,  261.7042_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1371.4492_dp,  947.5834_dp,  575.0060_dp,    0.0000_dp,    0.0000_dp, &
      &  698.6760_dp,  496.3331_dp,  328.6223_dp,    0.0000_dp,    0.0000_dp, &
      &  448.7768_dp,  324.9751_dp,  223.5637_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1072.3265_dp,  742.6641_dp,  450.9120_dp,    0.0000_dp,    0.0000_dp, &
      &  640.6641_dp,  456.3878_dp,  303.9644_dp,    0.0000_dp,    0.0000_dp, &
      &  441.9258_dp,  319.6860_dp,  219.4714_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1105.3016_dp,  768.2107_dp,  477.8977_dp,    0.0000_dp,    0.0000_dp, &
      &  611.3403_dp,  435.6701_dp,  289.9137_dp,    0.0000_dp,    0.0000_dp, &
      &  540.8706_dp,  386.3776_dp,  257.9190_dp,    0.0000_dp,    0.0000_dp, &
      &  467.8612_dp,  336.9173_dp,  229.2527_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1153.5737_dp,  799.3710_dp,  492.5198_dp,    0.0000_dp,    0.0000_dp, &
      &  599.2652_dp,  426.7405_dp,  283.4424_dp,    0.0000_dp,    0.0000_dp, &
      &  497.3996_dp,  356.4737_dp,  240.1326_dp,    0.0000_dp,    0.0000_dp, &
      &  536.6745_dp,  381.7903_dp,  252.4245_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  884.1648_dp,  614.0983_dp,  377.2375_dp,    0.0000_dp,    0.0000_dp, &
      &  606.8585_dp,  429.3743_dp,  280.2569_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  870.6935_dp,  609.0499_dp,  387.0821_dp,    0.0000_dp,    0.0000_dp, &
      &  544.3140_dp,  389.5811_dp,  262.5735_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1045.5505_dp,  729.5813_dp,  458.9033_dp,    0.0000_dp,    0.0000_dp, &
      &  969.0777_dp,  678.2590_dp,  430.7925_dp,    0.0000_dp,    0.0000_dp, &
      &  733.3555_dp,  519.4437_dp,  340.6535_dp,    0.0000_dp,    0.0000_dp, &
      &  572.5452_dp,  410.5273_dp,  278.1125_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  874.7836_dp,  617.9985_dp,  403.9646_dp,    0.0000_dp,    0.0000_dp, &
      &  808.5991_dp,  573.2139_dp,  377.6930_dp,    0.0000_dp,    0.0000_dp, &
      &  763.3410_dp,  542.3655_dp,  359.4274_dp,    0.0000_dp,    0.0000_dp, &
      &  669.7473_dp,  478.7132_dp,  321.7190_dp,    0.0000_dp,    0.0000_dp, &
      &  575.5731_dp,  414.6203_dp,  283.5949_dp,    0.0000_dp,    0.0000_dp, &
      &  711.7506_dp,  509.5503_dp,  343.7975_dp,    0.0000_dp,    0.0000_dp, &
      &  697.8037_dp,  500.0071_dp,  337.9057_dp,    0.0000_dp,    0.0000_dp, &
      &  676.2036_dp,  485.2461_dp,  328.9431_dp,    0.0000_dp,    0.0000_dp, &
      &  646.4774_dp,  464.9254_dp,  316.7189_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  620.3334_dp,  448.1381_dp,  307.9733_dp,    0.0000_dp,    0.0000_dp, &
      &  620.8918_dp,  448.4872_dp,  308.0920_dp,    0.0000_dp,    0.0000_dp, &
      &  615.5317_dp,  444.7322_dp,  305.6508_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  526.8299_dp,  384.2901_dp,  268.9880_dp,    0.0000_dp,    0.0000_dp, &
      &  531.3426_dp,  387.3876_dp,  270.8810_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.7013_dp,  328.1316_dp,  233.4581_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3520.9466_dp, 2411.4577_dp, 1244.1578_dp,    0.0000_dp,    0.0000_dp, &
      & 1037.4434_dp,  726.8613_dp,  455.2460_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2813.7007_dp, 1920.9295_dp, 1091.2408_dp,    0.0000_dp,    0.0000_dp, &
      & 2305.6086_dp, 1586.5402_dp,  890.7440_dp,    0.0000_dp,    0.0000_dp, &
      & 1028.2536_dp,  725.1809_dp,  469.2639_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(779 * maxRef * maxRef + 1 : 798 * maxRef * maxRef) = [ &
      & 2365.8925_dp, 1624.3636_dp,  955.0153_dp,    0.0000_dp,    0.0000_dp, &
      & 1624.3636_dp, 1132.2983_dp,  679.4591_dp,    0.0000_dp,    0.0000_dp, &
      &  955.0153_dp,  679.4591_dp,  451.0845_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   63.7801_dp,   39.6669_dp,   36.5405_dp,    0.0000_dp,    0.0000_dp, &
      &  107.7306_dp,   64.7068_dp,   58.8625_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.1088_dp,   25.7037_dp,   24.0883_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1363.2842_dp,  678.4810_dp,  562.1583_dp,    0.0000_dp,    0.0000_dp, &
      &  391.9179_dp,  216.7170_dp,  190.2283_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  681.5283_dp,  374.0135_dp,  327.2192_dp,    0.0000_dp,    0.0000_dp, &
      &  493.0100_dp,  275.3644_dp,  242.7264_dp,    0.0000_dp,    0.0000_dp, &
      &  297.4750_dp,  175.6564_dp,  158.7442_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  423.2302_dp,  246.0147_dp,  220.8361_dp,    0.0000_dp,    0.0000_dp, &
      &  350.1970_dp,  206.0014_dp,  185.7762_dp,    0.0000_dp,    0.0000_dp, &
      &  289.2869_dp,  172.1810_dp,  155.9706_dp,    0.0000_dp,    0.0000_dp, &
      &  209.9066_dp,  129.2409_dp,  118.6243_dp,    0.0000_dp,    0.0000_dp, &
      &  196.1543_dp,  121.3842_dp,  111.6080_dp,    0.0000_dp,    0.0000_dp, &
      &  269.0490_dp,  163.4548_dp,  149.2658_dp,    0.0000_dp,    0.0000_dp, &
      &  252.8213_dp,  153.4229_dp,  139.9877_dp,    0.0000_dp,    0.0000_dp, &
      &  200.7817_dp,  124.1920_dp,  114.1335_dp,    0.0000_dp,    0.0000_dp, &
      &  187.8203_dp,  116.1973_dp,  106.7629_dp,    0.0000_dp,    0.0000_dp, &
      &  148.0069_dp,   94.1173_dp,   87.3116_dp,    0.0000_dp,    0.0000_dp, &
      &  179.7777_dp,  112.9339_dp,  104.3380_dp,    0.0000_dp,    0.0000_dp, &
      &  167.5501_dp,  105.3151_dp,   97.2881_dp,    0.0000_dp,    0.0000_dp, &
      &  156.4540_dp,   98.7165_dp,   91.3072_dp,    0.0000_dp,    0.0000_dp, &
      &  136.9578_dp,   86.8035_dp,   80.3758_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.7852_dp,   84.7876_dp,   78.9339_dp,    0.0000_dp,    0.0000_dp, &
      &  117.2328_dp,   75.8994_dp,   70.7865_dp,    0.0000_dp,    0.0000_dp, &
      &  102.2836_dp,   66.7498_dp,   62.3893_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   97.0977_dp,   63.7680_dp,   59.7299_dp,    0.0000_dp,    0.0000_dp, &
      &   76.6712_dp,   51.5158_dp,   48.5769_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   73.0315_dp,   48.7600_dp,   45.8824_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1616.2863_dp,  808.9973_dp,  672.4303_dp,    0.0000_dp,    0.0000_dp, &
      &  581.2480_dp,  318.1678_dp,  277.7936_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1113.8339_dp,  600.5511_dp,  520.6868_dp,    0.0000_dp,    0.0000_dp, &
      &  905.7750_dp,  493.3057_dp,  429.6462_dp,    0.0000_dp,    0.0000_dp, &
      &  546.7935_dp,  314.6402_dp,  281.2950_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  985.0494_dp,  546.7176_dp,  480.7313_dp,    0.0000_dp,    0.0000_dp, &
      &  883.8541_dp,  494.0245_dp,  435.7814_dp,    0.0000_dp,    0.0000_dp, &
      &  665.0386_dp,  381.0309_dp,  339.8622_dp,    0.0000_dp,    0.0000_dp, &
      &  494.9507_dp,  292.9765_dp,  265.0243_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  733.1807_dp,  423.6992_dp,  379.4212_dp,    0.0000_dp,    0.0000_dp, &
      &  672.1834_dp,  390.7192_dp,  350.7076_dp,    0.0000_dp,    0.0000_dp, &
      &  620.0792_dp,  362.5672_dp,  326.2431_dp,    0.0000_dp,    0.0000_dp, &
      &  608.0848_dp,  354.0018_dp,  317.8567_dp,    0.0000_dp,    0.0000_dp, &
      &  480.7318_dp,  287.9444_dp,  261.6469_dp,    0.0000_dp,    0.0000_dp, &
      &  544.8882_dp,  326.0087_dp,  296.0806_dp,    0.0000_dp,    0.0000_dp, &
      &  528.0841_dp,  316.3169_dp,  287.3830_dp,    0.0000_dp,    0.0000_dp, &
      &  522.5721_dp,  312.3050_dp,  283.4537_dp,    0.0000_dp,    0.0000_dp, &
      &  478.5196_dp,  288.6637_dp,  262.9675_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  438.3890_dp,  268.3077_dp,  245.7286_dp,    0.0000_dp,    0.0000_dp, &
      &  433.9036_dp,  265.4089_dp,  243.0089_dp,    0.0000_dp,    0.0000_dp, &
      &  424.8748_dp,  259.9991_dp,  238.0843_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  347.5334_dp,  217.4322_dp,  200.6665_dp,    0.0000_dp,    0.0000_dp, &
      &  344.1318_dp,  215.2101_dp,  198.5775_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  276.9908_dp,  176.6690_dp,  164.0761_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(798 * maxRef * maxRef + 1 : 817 * maxRef * maxRef) = [ &
      & 2711.1542_dp, 1335.9899_dp, 1099.8393_dp,    0.0000_dp,    0.0000_dp, &
      &  778.5238_dp,  435.0649_dp,  383.0224_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2048.5251_dp, 1070.9104_dp,  912.5205_dp,    0.0000_dp,    0.0000_dp, &
      & 1637.9667_dp,  856.7131_dp,  729.6211_dp,    0.0000_dp,    0.0000_dp, &
      &  768.2000_dp,  441.4951_dp,  394.1110_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1659.1156_dp,  879.0165_dp,  754.6182_dp,    0.0000_dp,    0.0000_dp, &
      &  743.4318_dp,  429.3019_dp,  383.8645_dp,    0.0000_dp,    0.0000_dp, &
      &  721.7837_dp,  421.5388_dp,  378.9395_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1574.1401_dp,  844.1565_dp,  729.1622_dp,    0.0000_dp,    0.0000_dp, &
      &  716.8609_dp,  414.2030_dp,  370.5158_dp,    0.0000_dp,    0.0000_dp, &
      &  729.4243_dp,  426.5157_dp,  383.5999_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1426.0687_dp,  770.4185_dp,  667.9695_dp,    0.0000_dp,    0.0000_dp, &
      &  703.0784_dp,  406.0223_dp,  363.0711_dp,    0.0000_dp,    0.0000_dp, &
      &  668.9667_dp,  390.6056_dp,  350.9659_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1121.5489_dp,  606.7040_dp,  525.9535_dp,    0.0000_dp,    0.0000_dp, &
      &  700.5432_dp,  402.9200_dp,  359.6267_dp,    0.0000_dp,    0.0000_dp, &
      &  485.9447_dp,  291.2297_dp,  264.3347_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1208.7451_dp,  660.1005_dp,  575.3920_dp,    0.0000_dp,    0.0000_dp, &
      &  627.0790_dp,  367.7937_dp,  331.1873_dp,    0.0000_dp,    0.0000_dp, &
      &  406.3267_dp,  247.5981_dp,  226.1684_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  945.1713_dp,  517.5812_dp,  451.4130_dp,    0.0000_dp,    0.0000_dp, &
      &  575.7752_dp,  339.6278_dp,  306.5124_dp,    0.0000_dp,    0.0000_dp, &
      &  399.9295_dp,  243.2086_dp,  221.9858_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  978.9767_dp,  544.6643_dp,  479.1089_dp,    0.0000_dp,    0.0000_dp, &
      &  549.3294_dp,  323.9921_dp,  292.3514_dp,    0.0000_dp,    0.0000_dp, &
      &  486.3325_dp,  288.0243_dp,  260.2108_dp,    0.0000_dp,    0.0000_dp, &
      &  422.5208_dp,  254.6698_dp,  231.6783_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1019.7589_dp,  562.9344_dp,  493.3692_dp,    0.0000_dp,    0.0000_dp, &
      &  538.2480_dp,  316.9417_dp,  285.7823_dp,    0.0000_dp,    0.0000_dp, &
      &  448.1600_dp,  267.4973_dp,  242.4422_dp,    0.0000_dp,    0.0000_dp, &
      &  481.5000_dp,  282.7021_dp,  254.4567_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.1133_dp,  431.5544_dp,  377.9964_dp,    0.0000_dp,    0.0000_dp, &
      &  542.9499_dp,  314.9908_dp,  282.1724_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  774.5871_dp,  438.4080_dp,  388.7295_dp,    0.0000_dp,    0.0000_dp, &
      &  490.5097_dp,  292.4005_dp,  265.0402_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  928.1982_dp,  521.1802_dp,  460.4959_dp,    0.0000_dp,    0.0000_dp, &
      &  862.0363_dp,  487.8759_dp,  432.6309_dp,    0.0000_dp,    0.0000_dp, &
      &  656.8592_dp,  382.1952_dp,  343.0520_dp,    0.0000_dp,    0.0000_dp, &
      &  516.5656_dp,  309.1964_dp,  280.8241_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  782.9442_dp,  453.6905_dp,  406.6196_dp,    0.0000_dp,    0.0000_dp, &
      &  724.9805_dp,  423.2076_dp,  380.4575_dp,    0.0000_dp,    0.0000_dp, &
      &  685.2716_dp,  402.0877_dp,  362.2407_dp,    0.0000_dp,    0.0000_dp, &
      &  603.1519_dp,  358.4772_dp,  324.6431_dp,    0.0000_dp,    0.0000_dp, &
      &  520.4592_dp,  314.4398_dp,  286.6200_dp,    0.0000_dp,    0.0000_dp, &
      &  641.5564_dp,  382.6458_dp,  347.0371_dp,    0.0000_dp,    0.0000_dp, &
      &  629.2190_dp,  375.9179_dp,  341.1482_dp,    0.0000_dp,    0.0000_dp, &
      &  610.1761_dp,  365.6312_dp,  332.1968_dp,    0.0000_dp,    0.0000_dp, &
      &  584.0160_dp,  351.5641_dp,  319.9916_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  561.5474_dp,  341.0338_dp,  311.4245_dp,    0.0000_dp,    0.0000_dp, &
      &  562.0001_dp,  341.2040_dp,  311.5368_dp,    0.0000_dp,    0.0000_dp, &
      &  557.2071_dp,  338.4610_dp,  309.0838_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  479.0040_dp,  296.3944_dp,  272.4910_dp,    0.0000_dp,    0.0000_dp, &
      &  482.9873_dp,  298.5634_dp,  274.3837_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  406.8727_dp,  256.1501_dp,  236.8855_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3017.6856_dp, 1493.7993_dp, 1232.9252_dp,    0.0000_dp,    0.0000_dp, &
      &  920.1358_dp,  517.6961_dp,  457.0717_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(817 * maxRef * maxRef + 1 : 836 * maxRef * maxRef) = [ &
      & 2449.4723_dp, 1277.5813_dp, 1086.8394_dp,    0.0000_dp,    0.0000_dp, &
      & 2003.7829_dp, 1044.3401_dp,  887.4810_dp,    0.0000_dp,    0.0000_dp, &
      &  918.2378_dp,  528.6574_dp,  472.1081_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2072.6170_dp, 1106.6630_dp,  953.4821_dp,    0.0000_dp,    0.0000_dp, &
      & 1429.3576_dp,  780.8212_dp,  680.2683_dp,    0.0000_dp,    0.0000_dp, &
      &  857.6527_dp,  504.4325_dp,  454.7356_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1822.7181_dp,  987.8257_dp,  857.6011_dp,    0.0000_dp,    0.0000_dp, &
      &  987.8257_dp,  568.6789_dp,  507.5216_dp,    0.0000_dp,    0.0000_dp, &
      &  857.6011_dp,  507.5216_dp,  458.6896_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   58.6768_dp,   40.0304_dp,   35.3152_dp,    0.0000_dp,    0.0000_dp, &
      &   98.5183_dp,   65.3129_dp,   56.8466_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   36.3418_dp,   25.9534_dp,   23.3592_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1209.0304_dp,  684.1040_dp,  549.6114_dp,    0.0000_dp,    0.0000_dp, &
      &  353.5008_dp,  218.9706_dp,  184.1222_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  613.9850_dp,  377.9693_dp,  316.7162_dp,    0.0000_dp,    0.0000_dp, &
      &  445.3736_dp,  278.0881_dp,  234.9568_dp,    0.0000_dp,    0.0000_dp, &
      &  271.2606_dp,  177.3357_dp,  153.2948_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  384.9072_dp,  248.4266_dp,  213.3735_dp,    0.0000_dp,    0.0000_dp, &
      &  319.1151_dp,  207.9943_dp,  179.5291_dp,    0.0000_dp,    0.0000_dp, &
      &  264.1285_dp,  173.7946_dp,  150.7698_dp,    0.0000_dp,    0.0000_dp, &
      &  192.7765_dp,  130.4379_dp,  114.6328_dp,    0.0000_dp,    0.0000_dp, &
      &  180.3042_dp,  122.5075_dp,  107.8711_dp,    0.0000_dp,    0.0000_dp, &
      &  246.5241_dp,  164.9980_dp,  144.2558_dp,    0.0000_dp,    0.0000_dp, &
      &  231.5976_dp,  154.8764_dp,  135.3493_dp,    0.0000_dp,    0.0000_dp, &
      &  184.5477_dp,  125.3625_dp,  110.3639_dp,    0.0000_dp,    0.0000_dp, &
      &  172.6395_dp,  117.3051_dp,  103.2775_dp,    0.0000_dp,    0.0000_dp, &
      &  136.7050_dp,   95.0009_dp,   84.5041_dp,    0.0000_dp,    0.0000_dp, &
      &  165.6943_dp,  113.9989_dp,  100.9460_dp,    0.0000_dp,    0.0000_dp, &
      &  154.4356_dp,  106.3141_dp,   94.1631_dp,    0.0000_dp,    0.0000_dp, &
      &  144.3122_dp,   99.6625_dp,   88.3944_dp,    0.0000_dp,    0.0000_dp, &
      &  126.4307_dp,   87.6504_dp,   77.8681_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  121.9868_dp,   85.6024_dp,   76.4658_dp,    0.0000_dp,    0.0000_dp, &
      &  108.6378_dp,   76.6347_dp,   68.6087_dp,    0.0000_dp,    0.0000_dp, &
      &   94.9281_dp,   67.4128_dp,   60.5217_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   90.2234_dp,   64.4010_dp,   57.9467_dp,    0.0000_dp,    0.0000_dp, &
      &   71.5493_dp,   52.0412_dp,   47.1941_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.0769_dp,   49.2635_dp,   44.5776_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1434.6641_dp,  816.0317_dp,  657.1151_dp,    0.0000_dp,    0.0000_dp, &
      &  523.4058_dp,  321.5378_dp,  269.1778_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1000.5663_dp,  607.0073_dp,  504.6753_dp,    0.0000_dp,    0.0000_dp, &
      &  814.9289_dp,  498.3486_dp,  416.4183_dp,    0.0000_dp,    0.0000_dp, &
      &  496.4807_dp,  317.7923_dp,  271.7876_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  889.0302_dp,  552.3550_dp,  465.2259_dp,    0.0000_dp,    0.0000_dp, &
      &  798.6094_dp,  499.0761_dp,  421.6461_dp,    0.0000_dp,    0.0000_dp, &
      &  603.3575_dp,  384.7678_dp,  328.5727_dp,    0.0000_dp,    0.0000_dp, &
      &  451.5308_dp,  295.7786_dp,  255.9320_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  666.1507_dp,  427.8837_dp,  366.6169_dp,    0.0000_dp,    0.0000_dp, &
      &  611.3147_dp,  394.5403_dp,  338.8785_dp,    0.0000_dp,    0.0000_dp, &
      &  564.4901_dp,  366.0984_dp,  315.2073_dp,    0.0000_dp,    0.0000_dp, &
      &  553.1447_dp,  357.4612_dp,  307.2381_dp,    0.0000_dp,    0.0000_dp, &
      &  439.4281_dp,  290.6641_dp,  252.7061_dp,    0.0000_dp,    0.0000_dp, &
      &  497.9703_dp,  329.0947_dp,  285.9926_dp,    0.0000_dp,    0.0000_dp, &
      &  482.7036_dp,  319.3099_dp,  277.6188_dp,    0.0000_dp,    0.0000_dp, &
      &  477.4772_dp,  315.2763_dp,  273.8649_dp,    0.0000_dp,    0.0000_dp, &
      &  437.9280_dp,  291.3813_dp,  254.0460_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  402.1974_dp,  270.8059_dp,  237.4448_dp,    0.0000_dp,    0.0000_dp, &
      &  398.0419_dp,  267.8858_dp,  234.8321_dp,    0.0000_dp,    0.0000_dp, &
      &  389.7889_dp,  262.4295_dp,  230.0871_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(836 * maxRef * maxRef + 1 : 855 * maxRef * maxRef) = [ &
      &  320.0676_dp,  219.4462_dp,  194.0301_dp,    0.0000_dp,    0.0000_dp, &
      &  316.9110_dp,  217.2078_dp,  192.0192_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  255.9769_dp,  178.3183_dp,  158.7915_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2400.6918_dp, 1343.3714_dp, 1078.8336_dp,    0.0000_dp,    0.0000_dp, &
      &  703.3060_dp,  439.5228_dp,  371.4433_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1830.8523_dp, 1081.9930_dp,  887.6167_dp,    0.0000_dp,    0.0000_dp, &
      & 1463.8890_dp,  864.1411_dp,  710.7573_dp,    0.0000_dp,    0.0000_dp, &
      &  697.3148_dp,  445.9895_dp,  381.2592_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1486.0620_dp,  888.3356_dp,  733.0114_dp,    0.0000_dp,    0.0000_dp, &
      &  675.3314_dp,  433.7865_dp,  371.3244_dp,    0.0000_dp,    0.0000_dp, &
      &  656.9546_dp,  425.7299_dp,  366.3716_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1412.6818_dp,  853.0464_dp,  707.8111_dp,    0.0000_dp,    0.0000_dp, &
      &  651.2847_dp,  418.3068_dp,  358.5232_dp,    0.0000_dp,    0.0000_dp, &
      &  664.0452_dp,  430.7613_dp,  370.8800_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1281.3349_dp,  778.5266_dp,  648.1120_dp,    0.0000_dp,    0.0000_dp, &
      &  638.7014_dp,  410.1033_dp,  351.3471_dp,    0.0000_dp,    0.0000_dp, &
      &  608.8442_dp,  394.4854_dp,  339.4910_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1007.8571_dp,  613.0224_dp,  510.7385_dp,    0.0000_dp,    0.0000_dp, &
      &  635.9639_dp,  406.9894_dp,  348.1021_dp,    0.0000_dp,    0.0000_dp, &
      &  444.2328_dp,  294.1186_dp,  255.7658_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1087.9837_dp,  667.0305_dp,  557.9514_dp,    0.0000_dp,    0.0000_dp, &
      &  571.1862_dp,  371.4691_dp,  320.2256_dp,    0.0000_dp,    0.0000_dp, &
      &  372.5215_dp,  250.0266_dp,  218.8458_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  851.0634_dp,  523.0020_dp,  438.0603_dp,    0.0000_dp,    0.0000_dp, &
      &  524.9588_dp,  343.0012_dp,  296.3749_dp,    0.0000_dp,    0.0000_dp, &
      &  366.5329_dp,  245.6037_dp,  214.8053_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  883.8890_dp,  550.4092_dp,  464.0668_dp,    0.0000_dp,    0.0000_dp, &
      &  500.8355_dp,  327.1995_dp,  282.7396_dp,    0.0000_dp,    0.0000_dp, &
      &  443.7007_dp,  290.8988_dp,  251.7856_dp,    0.0000_dp,    0.0000_dp, &
      &  386.6491_dp,  257.1961_dp,  224.1590_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  919.5354_dp,  568.9088_dp,  478.0372_dp,    0.0000_dp,    0.0000_dp, &
      &  490.5973_dp,  320.0960_dp,  276.4113_dp,    0.0000_dp,    0.0000_dp, &
      &  409.4287_dp,  270.1688_dp,  234.5558_dp,    0.0000_dp,    0.0000_dp, &
      &  438.6502_dp,  285.5945_dp,  246.3077_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  704.3789_dp,  436.1042_dp,  366.6648_dp,    0.0000_dp,    0.0000_dp, &
      &  493.6363_dp,  318.2136_dp,  273.1065_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  701.3549_dp,  442.9886_dp,  376.2399_dp,    0.0000_dp,    0.0000_dp, &
      &  448.0328_dp,  295.2762_dp,  256.2171_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  839.2828_dp,  526.4964_dp,  445.7279_dp,    0.0000_dp,    0.0000_dp, &
      &  780.4805_dp,  492.8316_dp,  418.6209_dp,    0.0000_dp,    0.0000_dp, &
      &  597.4745_dp,  385.9385_dp,  331.6718_dp,    0.0000_dp,    0.0000_dp, &
      &  472.1406_dp,  312.1568_dp,  271.3137_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  711.6823_dp,  458.1920_dp,  393.0366_dp,    0.0000_dp,    0.0000_dp, &
      &  659.8039_dp,  427.3622_dp,  367.7137_dp,    0.0000_dp,    0.0000_dp, &
      &  624.2061_dp,  406.0194_dp,  350.0801_dp,    0.0000_dp,    0.0000_dp, &
      &  550.6011_dp,  361.9295_dp,  313.6909_dp,    0.0000_dp,    0.0000_dp, &
      &  476.4489_dp,  317.4077_dp,  276.8993_dp,    0.0000_dp,    0.0000_dp, &
      &  586.0084_dp,  386.3091_dp,  335.2921_dp,    0.0000_dp,    0.0000_dp, &
      &  574.9009_dp,  379.5097_dp,  329.6139_dp,    0.0000_dp,    0.0000_dp, &
      &  557.7850_dp,  369.1125_dp,  320.9667_dp,    0.0000_dp,    0.0000_dp, &
      &  534.2915_dp,  354.8970_dp,  309.1643_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  514.5077_dp,  344.2425_dp,  300.9258_dp,    0.0000_dp,    0.0000_dp, &
      &  514.8945_dp,  344.4169_dp,  301.0426_dp,    0.0000_dp,    0.0000_dp, &
      &  510.5464_dp,  341.6492_dp,  298.6808_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  440.2999_dp,  299.1535_dp,  263.4034_dp,    0.0000_dp,    0.0000_dp, &
      &  443.8848_dp,  301.3463_dp,  265.2340_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(855 * maxRef * maxRef + 1 : 874 * maxRef * maxRef) = [ &
      &  375.1339_dp,  258.5304_dp,  229.1191_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2673.9344_dp, 1501.7789_dp, 1209.2081_dp,    0.0000_dp,    0.0000_dp, &
      &  832.1368_dp,  522.9449_dp,  443.2682_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2188.2864_dp, 1290.4826_dp, 1057.8748_dp,    0.0000_dp,    0.0000_dp, &
      & 1789.7638_dp, 1052.8629_dp,  865.2462_dp,    0.0000_dp,    0.0000_dp, &
      &  833.7135_dp,  534.0203_dp,  456.8822_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1858.6377_dp, 1118.1402_dp,  926.1439_dp,    0.0000_dp,    0.0000_dp, &
      & 1286.4770_dp,  787.4950_dp,  660.7377_dp,    0.0000_dp,    0.0000_dp, &
      &  781.5377_dp,  509.4181_dp,  439.6548_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1638.4938_dp,  998.1224_dp,  832.1743_dp,    0.0000_dp,    0.0000_dp, &
      &  896.7952_dp,  574.4591_dp,  491.3081_dp,    0.0000_dp,    0.0000_dp, &
      &  782.3088_dp,  512.4995_dp,  443.4182_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1475.2500_dp,  906.0801_dp,  758.7789_dp,    0.0000_dp,    0.0000_dp, &
      &  906.0801_dp,  580.5824_dp,  496.1127_dp,    0.0000_dp,    0.0000_dp, &
      &  758.7789_dp,  496.1127_dp,  428.8962_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   46.0406_dp,   37.5478_dp,   31.6681_dp,    0.0000_dp,    0.0000_dp, &
      &   76.3661_dp,   60.9773_dp,   50.4173_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.1617_dp,   24.5530_dp,   21.3516_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  890.5126_dp,  628.1817_dp,  461.5751_dp,    0.0000_dp,    0.0000_dp, &
      &  267.3761_dp,  202.5173_dp,  159.4404_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  463.3065_dp,  349.1897_dp,  273.5446_dp,    0.0000_dp,    0.0000_dp, &
      &  337.9035_dp,  257.6088_dp,  204.1519_dp,    0.0000_dp,    0.0000_dp, &
      &  209.0959_dp,  165.2066_dp,  135.2428_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  295.3592_dp,  231.0449_dp,  187.4666_dp,    0.0000_dp,    0.0000_dp, &
      &  245.8527_dp,  193.7557_dp,  158.3563_dp,    0.0000_dp,    0.0000_dp, &
      &  204.3092_dp,  162.1983_dp,  133.5090_dp,    0.0000_dp,    0.0000_dp, &
      &  150.7447_dp,  122.1941_dp,  102.4883_dp,    0.0000_dp,    0.0000_dp, &
      &  141.2522_dp,  114.8462_dp,   96.6030_dp,    0.0000_dp,    0.0000_dp, &
      &  191.9528_dp,  154.3288_dp,  128.4978_dp,    0.0000_dp,    0.0000_dp, &
      &  180.3390_dp,  144.8738_dp,  120.5946_dp,    0.0000_dp,    0.0000_dp, &
      &  144.6355_dp,  117.5524_dp,   98.8903_dp,    0.0000_dp,    0.0000_dp, &
      &  135.3716_dp,  110.0255_dp,   92.5983_dp,    0.0000_dp,    0.0000_dp, &
      &  108.2356_dp,   89.4229_dp,   76.3869_dp,    0.0000_dp,    0.0000_dp, &
      &  130.6045_dp,  107.1304_dp,   90.9043_dp,    0.0000_dp,    0.0000_dp, &
      &  121.8066_dp,   99.9381_dp,   84.8582_dp,    0.0000_dp,    0.0000_dp, &
      &  114.0004_dp,   93.7434_dp,   79.7672_dp,    0.0000_dp,    0.0000_dp, &
      &  100.1080_dp,   82.5251_dp,   70.4244_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   97.0675_dp,   80.7360_dp,   69.4181_dp,    0.0000_dp,    0.0000_dp, &
      &   86.6803_dp,   72.3520_dp,   62.4324_dp,    0.0000_dp,    0.0000_dp, &
      &   76.0239_dp,   63.7402_dp,   55.2484_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.4186_dp,   60.9414_dp,   52.9845_dp,    0.0000_dp,    0.0000_dp, &
      &   57.9784_dp,   49.4166_dp,   43.4760_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   55.0487_dp,   46.7498_dp,   41.0035_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1058.3899_dp,  749.4974_dp,  553.0357_dp,    0.0000_dp,    0.0000_dp, &
      &  395.0058_dp,  297.1713_dp,  232.5932_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  751.6628_dp,  559.8856_dp,  433.8278_dp,    0.0000_dp,    0.0000_dp, &
      &  614.0265_dp,  460.4234_dp,  359.1888_dp,    0.0000_dp,    0.0000_dp, &
      &  379.7626_dp,  295.1816_dp,  238.0169_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  673.1193_dp,  511.0810_dp,  403.2914_dp,    0.0000_dp,    0.0000_dp, &
      &  605.9135_dp,  462.1678_dp,  366.3171_dp,    0.0000_dp,    0.0000_dp, &
      &  461.0586_dp,  357.3576_dp,  287.5225_dp,    0.0000_dp,    0.0000_dp, &
      &  348.3397_dp,  275.6435_dp,  225.9636_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(874 * maxRef * maxRef + 1 : 893 * maxRef * maxRef) = [ &
      &  510.2352_dp,  397.6602_dp,  321.5163_dp,    0.0000_dp,    0.0000_dp, &
      &  469.1092_dp,  366.9549_dp,  297.7436_dp,    0.0000_dp,    0.0000_dp, &
      &  433.9678_dp,  340.7358_dp,  277.4312_dp,    0.0000_dp,    0.0000_dp, &
      &  424.8072_dp,  332.5980_dp,  270.1899_dp,    0.0000_dp,    0.0000_dp, &
      &  340.3478_dp,  271.2960_dp,  223.9586_dp,    0.0000_dp,    0.0000_dp, &
      &  385.5895_dp,  307.1384_dp,  253.4123_dp,    0.0000_dp,    0.0000_dp, &
      &  373.9427_dp,  298.0661_dp,  246.1086_dp,    0.0000_dp,    0.0000_dp, &
      &  369.6794_dp,  294.2436_dp,  242.6676_dp,    0.0000_dp,    0.0000_dp, &
      &  340.0592_dp,  272.2421_dp,  225.7030_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  313.8657_dp,  253.4985_dp,  211.9069_dp,    0.0000_dp,    0.0000_dp, &
      &  310.5861_dp,  250.7573_dp,  209.5599_dp,    0.0000_dp,    0.0000_dp, &
      &  304.2109_dp,  245.6719_dp,  205.3695_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  251.7723_dp,  206.0459_dp,  174.3920_dp,    0.0000_dp,    0.0000_dp, &
      &  249.2671_dp,  203.9397_dp,  172.5758_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  202.8572_dp,  167.9014_dp,  143.6389_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1765.8946_dp, 1237.9643_dp,  905.0848_dp,    0.0000_dp,    0.0000_dp, &
      &  534.6303_dp,  407.5906_dp,  323.6381_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1365.4837_dp,  996.0157_dp,  757.0858_dp,    0.0000_dp,    0.0000_dp, &
      & 1092.6733_dp,  797.4773_dp,  607.0919_dp,    0.0000_dp,    0.0000_dp, &
      &  533.7978_dp,  414.4811_dp,  334.3206_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1112.0098_dp,  818.5161_dp,  627.5560_dp,    0.0000_dp,    0.0000_dp, &
      &  517.7340_dp,  403.1540_dp,  326.0787_dp,    0.0000_dp,    0.0000_dp, &
      &  505.2346_dp,  396.3470_dp,  322.6655_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1060.6255_dp,  787.0565_dp,  608.3000_dp,    0.0000_dp,    0.0000_dp, &
      &  499.5011_dp,  389.1787_dp,  314.9983_dp,    0.0000_dp,    0.0000_dp, &
      &  510.8967_dp,  401.0919_dp,  326.7670_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  963.9434_dp,  718.8469_dp,  558.2324_dp,    0.0000_dp,    0.0000_dp, &
      &  489.8160_dp,  381.4804_dp,  308.6801_dp,    0.0000_dp,    0.0000_dp, &
      &  468.4059_dp,  367.3750_dp,  299.1495_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  758.9955_dp,  566.4300_dp,  440.5516_dp,    0.0000_dp,    0.0000_dp, &
      &  487.1911_dp,  378.4474_dp,  305.5273_dp,    0.0000_dp,    0.0000_dp, &
      &  344.8105_dp,  274.8422_dp,  227.2947_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  820.9313_dp,  616.5993_dp,  482.1362_dp,    0.0000_dp,    0.0000_dp, &
      &  439.9326_dp,  346.0423_dp,  282.4334_dp,    0.0000_dp,    0.0000_dp, &
      &  290.7494_dp,  234.1315_dp,  195.4456_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  643.1247_dp,  483.8537_dp,  379.2688_dp,    0.0000_dp,    0.0000_dp, &
      &  405.0817_dp,  319.7637_dp,  261.8630_dp,    0.0000_dp,    0.0000_dp, &
      &  285.8977_dp,  229.9399_dp,  191.7333_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  670.3490_dp,  509.6999_dp,  403.1484_dp,    0.0000_dp,    0.0000_dp, &
      &  386.5245_dp,  305.0887_dp,  249.8695_dp,    0.0000_dp,    0.0000_dp, &
      &  343.0710_dp,  271.4559_dp,  222.9546_dp,    0.0000_dp,    0.0000_dp, &
      &  300.6753_dp,  240.5139_dp,  199.5325_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.8348_dp,  526.3795_dp,  414.3156_dp,    0.0000_dp,    0.0000_dp, &
      &  378.4594_dp,  298.4129_dp,  244.1847_dp,    0.0000_dp,    0.0000_dp, &
      &  317.3472_dp,  252.3281_dp,  208.1570_dp,    0.0000_dp,    0.0000_dp, &
      &  338.3535_dp,  266.2568_dp,  217.6457_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.6535_dp,  403.8400_dp,  318.3239_dp,    0.0000_dp,    0.0000_dp, &
      &  379.2299_dp,  296.1989_dp,  240.3673_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  534.5207_dp,  410.9866_dp,  328.4707_dp,    0.0000_dp,    0.0000_dp, &
      &  346.8387_dp,  275.6055_dp,  227.0405_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  637.9233_dp,  487.9858_dp,  388.0446_dp,    0.0000_dp,    0.0000_dp, &
      &  594.5765_dp,  457.1641_dp,  365.2802_dp,    0.0000_dp,    0.0000_dp, &
      &  458.8906_dp,  359.1580_dp,  291.7241_dp,    0.0000_dp,    0.0000_dp, &
      &  365.7390_dp,  291.3944_dp,  240.5109_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  545.7845_dp,  426.0703_dp,  345.1578_dp,    0.0000_dp,    0.0000_dp, &
      &  507.1503_dp,  397.7592_dp,  323.6293_dp,    0.0000_dp,    0.0000_dp, &
      &  480.5552_dp,  378.1231_dp,  308.5781_dp,    0.0000_dp,    0.0000_dp, &
      &  425.5849_dp,  337.5775_dp,  277.5321_dp,    0.0000_dp,    0.0000_dp, &
      &  370.1677_dp,  296.6250_dp,  246.1182_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(893 * maxRef * maxRef + 1 : 912 * maxRef * maxRef) = [ &
      &  453.4186_dp,  360.4496_dp,  296.9165_dp,    0.0000_dp,    0.0000_dp, &
      &  445.0799_dp,  354.1868_dp,  292.0475_dp,    0.0000_dp,    0.0000_dp, &
      &  432.2496_dp,  344.6133_dp,  284.6411_dp,    0.0000_dp,    0.0000_dp, &
      &  414.6509_dp,  331.5230_dp,  274.5376_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  400.4998_dp,  321.9415_dp,  267.9623_dp,    0.0000_dp,    0.0000_dp, &
      &  400.7722_dp,  322.0978_dp,  268.0526_dp,    0.0000_dp,    0.0000_dp, &
      &  397.4644_dp,  319.5340_dp,  265.9986_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  344.9826_dp,  280.4702_dp,  235.9264_dp,    0.0000_dp,    0.0000_dp, &
      &  347.6800_dp,  282.4933_dp,  237.5019_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  295.7994_dp,  242.9684_dp,  206.3637_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1969.4796_dp, 1385.1228_dp, 1016.3718_dp,    0.0000_dp,    0.0000_dp, &
      &  633.9471_dp,  485.4129_dp,  387.1236_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1631.5424_dp, 1188.1982_dp,  902.1097_dp,    0.0000_dp,    0.0000_dp, &
      & 1335.0700_dp,  972.1233_dp,  738.6401_dp,    0.0000_dp,    0.0000_dp, &
      &  638.7768_dp,  496.5128_dp,  401.0541_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1394.0976_dp, 1031.4745_dp,  795.1764_dp,    0.0000_dp,    0.0000_dp, &
      &  971.5959_dp,  730.1446_dp,  571.8530_dp,    0.0000_dp,    0.0000_dp, &
      &  602.4253_dp,  474.6807_dp,  388.0718_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1233.9095_dp,  922.0660_dp,  717.6632_dp,    0.0000_dp,    0.0000_dp, &
      &  687.1848_dp,  534.0108_dp,  431.3512_dp,    0.0000_dp,    0.0000_dp, &
      &  604.1475_dp,  477.8870_dp,  392.0655_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1114.0924_dp,  837.9849_dp,  656.3774_dp,    0.0000_dp,    0.0000_dp, &
      &  694.2983_dp,  539.3532_dp,  435.5692_dp,    0.0000_dp,    0.0000_dp, &
      &  585.8126_dp,  462.6546_dp,  379.2075_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  845.8972_dp,  643.4721_dp,  509.7424_dp,    0.0000_dp,    0.0000_dp, &
      &  643.4721_dp,  502.0073_dp,  407.0659_dp,    0.0000_dp,    0.0000_dp, &
      &  509.7424_dp,  407.0659_dp,  337.1075_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   51.0527_dp,   41.3385_dp,   28.2627_dp,    0.0000_dp,    0.0000_dp, &
      &   85.0800_dp,   67.8924_dp,   44.5963_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   32.0588_dp,   26.6217_dp,   19.3556_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1011.6099_dp,  763.7814_dp,  391.4255_dp,    0.0000_dp,    0.0000_dp, &
      &  300.6533_dp,  232.9539_dp,  138.3613_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  521.4249_dp,  402.7727_dp,  236.8457_dp,    0.0000_dp,    0.0000_dp, &
      &  379.5254_dp,  295.3816_dp,  177.6837_dp,    0.0000_dp,    0.0000_dp, &
      &  233.4606_dp,  185.0244_dp,  119.1206_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.3257_dp,  260.3796_dp,  164.5777_dp,    0.0000_dp,    0.0000_dp, &
      &  274.5334_dp,  217.4762_dp,  139.4763_dp,    0.0000_dp,    0.0000_dp, &
      &  227.7959_dp,  181.3885_dp,  117.9758_dp,    0.0000_dp,    0.0000_dp, &
      &  167.3783_dp,  134.9705_dp,   91.2518_dp,    0.0000_dp,    0.0000_dp, &
      &  156.7256_dp,  126.6571_dp,   86.1276_dp,    0.0000_dp,    0.0000_dp, &
      &  213.4799_dp,  171.2581_dp,  114.0774_dp,    0.0000_dp,    0.0000_dp, &
      &  200.5464_dp,  160.9223_dp,  107.0906_dp,    0.0000_dp,    0.0000_dp, &
      &  160.4520_dp,  129.7079_dp,   88.2158_dp,    0.0000_dp,    0.0000_dp, &
      &  150.1398_dp,  121.4381_dp,   82.6496_dp,    0.0000_dp,    0.0000_dp, &
      &  119.5995_dp,   97.8417_dp,   68.6237_dp,    0.0000_dp,    0.0000_dp, &
      &  144.5697_dp,  117.6479_dp,   81.4217_dp,    0.0000_dp,    0.0000_dp, &
      &  134.7887_dp,  109.7755_dp,   76.0532_dp,    0.0000_dp,    0.0000_dp, &
      &  126.0801_dp,  102.8506_dp,   71.5723_dp,    0.0000_dp,    0.0000_dp, &
      &  110.6119_dp,   90.4660_dp,   63.3081_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  107.0588_dp,   88.0609_dp,   62.5886_dp,    0.0000_dp,    0.0000_dp, &
      &   95.4966_dp,   78.7920_dp,   56.3962_dp,    0.0000_dp,    0.0000_dp, &
      &   83.6409_dp,   69.2838_dp,   50.0406_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.6121_dp,   66.1093_dp,   48.0540_dp,    0.0000_dp,    0.0000_dp, &
      &   63.5079_dp,   53.2880_dp,   39.6668_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(912 * maxRef * maxRef + 1 : 931 * maxRef * maxRef) = [ &
      &   60.3521_dp,   50.5087_dp,   37.3723_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1201.5234_dp,  908.4025_dp,  469.7696_dp,    0.0000_dp,    0.0000_dp, &
      &  444.5269_dp,  343.5592_dp,  201.5403_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  847.3497_dp,  651.2564_dp,  374.2691_dp,    0.0000_dp,    0.0000_dp, &
      &  691.4497_dp,  533.7085_dp,  310.8083_dp,    0.0000_dp,    0.0000_dp, &
      &  425.2656_dp,  333.8166_dp,  208.4131_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  756.6195_dp,  586.9862_dp,  350.2804_dp,    0.0000_dp,    0.0000_dp, &
      &  680.5427_dp,  529.3043_dp,  318.7388_dp,    0.0000_dp,    0.0000_dp, &
      &  516.4766_dp,  405.2011_dp,  251.6421_dp,    0.0000_dp,    0.0000_dp, &
      &  388.8226_dp,  308.4359_dp,  199.1588_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  571.0501_dp,  449.1191_dp,  281.8480_dp,    0.0000_dp,    0.0000_dp, &
      &  524.6479_dp,  413.5991_dp,  261.4105_dp,    0.0000_dp,    0.0000_dp, &
      &  485.0098_dp,  383.1734_dp,  243.9188_dp,    0.0000_dp,    0.0000_dp, &
      &  474.9418_dp,  374.8284_dp,  237.4146_dp,    0.0000_dp,    0.0000_dp, &
      &  379.3148_dp,  302.3644_dp,  197.9963_dp,    0.0000_dp,    0.0000_dp, &
      &  429.7677_dp,  342.4883_dp,  224.0034_dp,    0.0000_dp,    0.0000_dp, &
      &  416.7080_dp,  332.2783_dp,  217.6347_dp,    0.0000_dp,    0.0000_dp, &
      &  412.0413_dp,  328.3317_dp,  214.5176_dp,    0.0000_dp,    0.0000_dp, &
      &  378.6058_dp,  302.7493_dp,  199.9418_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  348.7744_dp,  280.5613_dp,  188.4044_dp,    0.0000_dp,    0.0000_dp, &
      &  345.1434_dp,  277.6002_dp,  186.3093_dp,    0.0000_dp,    0.0000_dp, &
      &  338.0296_dp,  271.9434_dp,  182.6170_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  278.9123_dp,  226.4833_dp,  155.9405_dp,    0.0000_dp,    0.0000_dp, &
      &  276.1455_dp,  224.2095_dp,  154.3118_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  224.0794_dp,  183.5205_dp,  129.1105_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2007.6137_dp, 1518.8755_dp,  767.9339_dp,    0.0000_dp,    0.0000_dp, &
      &  599.9302_dp,  468.0977_dp,  282.4384_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1543.4258_dp, 1178.3302_dp,  649.5000_dp,    0.0000_dp,    0.0000_dp, &
      & 1234.9316_dp,  945.9943_dp,  521.8458_dp,    0.0000_dp,    0.0000_dp, &
      &  597.4915_dp,  469.5752_dp,  293.1198_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1255.3446_dp,  961.3657_dp,  539.8662_dp,    0.0000_dp,    0.0000_dp, &
      &  579.1623_dp,  455.9109_dp,  286.2241_dp,    0.0000_dp,    0.0000_dp, &
      &  564.5560_dp,  446.1492_dp,  283.8784_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1195.8434_dp,  919.3099_dp,  524.9173_dp,    0.0000_dp,    0.0000_dp, &
      &  558.7211_dp,  440.2561_dp,  276.6567_dp,    0.0000_dp,    0.0000_dp, &
      &  570.7922_dp,  451.2866_dp,  287.5807_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1086.0231_dp,  836.7341_dp,  482.5622_dp,    0.0000_dp,    0.0000_dp, &
      &  547.8906_dp,  431.6391_dp,  271.1001_dp,    0.0000_dp,    0.0000_dp, &
      &  523.3223_dp,  413.8230_dp,  263.3409_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  854.7474_dp,  659.7201_dp,  381.4006_dp,    0.0000_dp,    0.0000_dp, &
      &  545.1724_dp,  428.9662_dp,  268.1325_dp,    0.0000_dp,    0.0000_dp, &
      &  383.9186_dp,  306.7585_dp,  201.4707_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  923.8743_dp,  714.1830_dp,  417.8614_dp,    0.0000_dp,    0.0000_dp, &
      &  491.3210_dp,  388.9066_dp,  248.7895_dp,    0.0000_dp,    0.0000_dp, &
      &  323.0552_dp,  259.8051_dp,  173.9292_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  723.3202_dp,  560.3648_dp,  329.3116_dp,    0.0000_dp,    0.0000_dp, &
      &  452.0831_dp,  358.6562_dp,  231.0081_dp,    0.0000_dp,    0.0000_dp, &
      &  317.7432_dp,  255.3330_dp,  170.5557_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  752.9678_dp,  585.2935_dp,  350.8564_dp,    0.0000_dp,    0.0000_dp, &
      &  431.3478_dp,  342.2979_dp,  220.4802_dp,    0.0000_dp,    0.0000_dp, &
      &  382.5646_dp,  304.2749_dp,  197.0680_dp,    0.0000_dp,    0.0000_dp, &
      &  334.5583_dp,  267.8666_dp,  177.1019_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  782.2483_dp,  606.4740_dp,  359.9017_dp,    0.0000_dp,    0.0000_dp, &
      &  422.4144_dp,  335.0287_dp,  215.4030_dp,    0.0000_dp,    0.0000_dp, &
      &  353.5546_dp,  281.9707_dp,  184.3086_dp,    0.0000_dp,    0.0000_dp, &
      &  377.6302_dp,  299.4461_dp,  192.0591_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(931 * maxRef * maxRef + 1 : 950 * maxRef * maxRef) = [ &
      &  599.6192_dp,  465.8147_dp,  276.9970_dp,    0.0000_dp,    0.0000_dp, &
      &  423.9154_dp,  334.5615_dp,  211.4238_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  599.3141_dp,  468.4529_dp,  286.9927_dp,    0.0000_dp,    0.0000_dp, &
      &  386.6256_dp,  307.8815_dp,  200.7562_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  715.9811_dp,  558.1522_dp,  338.2627_dp,    0.0000_dp,    0.0000_dp, &
      &  666.7601_dp,  521.1441_dp,  318.9968_dp,    0.0000_dp,    0.0000_dp, &
      &  513.0410_dp,  404.9175_dp,  256.3933_dp,    0.0000_dp,    0.0000_dp, &
      &  407.5902_dp,  324.9095_dp,  212.6937_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.5327_dp,  480.8926_dp,  302.9389_dp,    0.0000_dp,    0.0000_dp, &
      &  566.8294_dp,  447.7131_dp,  284.5505_dp,    0.0000_dp,    0.0000_dp, &
      &  536.7795_dp,  424.7791_dp,  271.6474_dp,    0.0000_dp,    0.0000_dp, &
      &  474.6638_dp,  377.4276_dp,  245.0471_dp,    0.0000_dp,    0.0000_dp, &
      &  412.0594_dp,  329.6696_dp,  218.1156_dp,    0.0000_dp,    0.0000_dp, &
      &  505.5091_dp,  402.4540_dp,  262.3480_dp,    0.0000_dp,    0.0000_dp, &
      &  496.1023_dp,  395.2472_dp,  258.1617_dp,    0.0000_dp,    0.0000_dp, &
      &  481.6229_dp,  384.1644_dp,  251.7973_dp,    0.0000_dp,    0.0000_dp, &
      &  461.7577_dp,  368.9607_dp,  243.1165_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.4778_dp,  357.2511_dp,  237.8235_dp,    0.0000_dp,    0.0000_dp, &
      &  445.7916_dp,  357.4737_dp,  237.8953_dp,    0.0000_dp,    0.0000_dp, &
      &  442.0786_dp,  354.5768_dp,  236.1084_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  382.7591_dp,  309.3530_dp,  210.3812_dp,    0.0000_dp,    0.0000_dp, &
      &  385.7986_dp,  311.6901_dp,  211.7410_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.3827_dp,  266.5777_dp,  184.8477_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2237.9665_dp, 1696.1214_dp,  863.7655_dp,    0.0000_dp,    0.0000_dp, &
      &  710.7779_dp,  556.1255_dp,  338.5014_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1844.3774_dp, 1408.3895_dp,  773.9463_dp,    0.0000_dp,    0.0000_dp, &
      & 1509.2981_dp, 1156.2604_dp,  634.8168_dp,    0.0000_dp,    0.0000_dp, &
      &  714.7188_dp,  562.4313_dp,  351.9564_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1572.3759_dp, 1207.9868_dp,  685.7489_dp,    0.0000_dp,    0.0000_dp, &
      & 1093.2424_dp,  848.3727_dp,  496.6399_dp,    0.0000_dp,    0.0000_dp, &
      &  672.5553_dp,  532.9719_dp,  342.0346_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1389.5979_dp, 1072.1935_dp,  621.0438_dp,    0.0000_dp,    0.0000_dp, &
      &  768.8428_dp,  605.3389_dp,  378.6511_dp,    0.0000_dp,    0.0000_dp, &
      &  674.0068_dp,  535.3153_dp,  346.0266_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1253.3465_dp,  970.2070_dp,  569.4159_dp,    0.0000_dp,    0.0000_dp, &
      &  776.7368_dp,  611.2764_dp,  382.3225_dp,    0.0000_dp,    0.0000_dp, &
      &  653.6028_dp,  519.0845_dp,  334.7174_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  949.6490_dp,  739.7915_dp,  444.3471_dp,    0.0000_dp,    0.0000_dp, &
      &  719.3659_dp,  567.7670_dp,  357.9634_dp,    0.0000_dp,    0.0000_dp, &
      &  567.4441_dp,  453.7197_dp,  298.8508_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1067.0169_dp,  829.2408_dp,  493.7430_dp,    0.0000_dp,    0.0000_dp, &
      &  829.2408_dp,  649.7143_dp,  397.0367_dp,    0.0000_dp,    0.0000_dp, &
      &  493.7430_dp,  397.0367_dp,  265.8654_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.5745_dp,   33.1875_dp,   26.8142_dp,    0.0000_dp,    0.0000_dp, &
      &   65.0863_dp,   53.3648_dp,   42.1627_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   25.4622_dp,   22.0616_dp,   18.4864_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  730.5533_dp,  517.3330_dp,  364.4102_dp,    0.0000_dp,    0.0000_dp, &
      &  223.8871_dp,  172.9708_dp,  129.8907_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  387.2681_dp,  297.5186_dp,  222.1489_dp,    0.0000_dp,    0.0000_dp, &
      &  283.5695_dp,  220.6746_dp,  166.9872_dp,    0.0000_dp,    0.0000_dp, &
      &  177.5174_dp,  143.8851_dp,  112.4391_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(950 * maxRef * maxRef + 1 : 969 * maxRef * maxRef) = [ &
      &  249.9382_dp,  200.3299_dp,  155.1628_dp,    0.0000_dp,    0.0000_dp, &
      &  208.6362_dp,  168.5944_dp,  131.6650_dp,    0.0000_dp,    0.0000_dp, &
      &  173.8767_dp,  141.6201_dp,  111.5108_dp,    0.0000_dp,    0.0000_dp, &
      &  129.2726_dp,  107.7128_dp,   86.4986_dp,    0.0000_dp,    0.0000_dp, &
      &  121.2876_dp,  101.3825_dp,   81.6853_dp,    0.0000_dp,    0.0000_dp, &
      &  164.1279_dp,  135.5517_dp,  108.0200_dp,    0.0000_dp,    0.0000_dp, &
      &  154.1938_dp,  127.2125_dp,  101.4184_dp,    0.0000_dp,    0.0000_dp, &
      &  124.2389_dp,  103.7956_dp,   83.6946_dp,    0.0000_dp,    0.0000_dp, &
      &  116.3247_dp,   97.1744_dp,   78.4368_dp,    0.0000_dp,    0.0000_dp, &
      &   93.6269_dp,   79.5750_dp,   65.2905_dp,    0.0000_dp,    0.0000_dp, &
      &  112.6378_dp,   95.0190_dp,   77.3772_dp,    0.0000_dp,    0.0000_dp, &
      &  105.0905_dp,   88.6613_dp,   72.2950_dp,    0.0000_dp,    0.0000_dp, &
      &   98.4728_dp,   83.2744_dp,   68.0720_dp,    0.0000_dp,    0.0000_dp, &
      &   86.6163_dp,   73.4259_dp,   60.2641_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.2725_dp,   72.1173_dp,   59.6460_dp,    0.0000_dp,    0.0000_dp, &
      &   75.3910_dp,   64.7448_dp,   53.7863_dp,    0.0000_dp,    0.0000_dp, &
      &   66.3025_dp,   57.1928_dp,   47.7847_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   63.2599_dp,   54.7781_dp,   45.9133_dp,    0.0000_dp,    0.0000_dp, &
      &   50.9770_dp,   44.7087_dp,   37.9961_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   48.3429_dp,   42.2432_dp,   35.7884_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  869.3774_dp,  618.6669_dp,  437.6491_dp,    0.0000_dp,    0.0000_dp, &
      &  330.2273_dp,  253.0900_dp,  189.1300_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  626.2104_dp,  474.4180_dp,  350.6141_dp,    0.0000_dp,    0.0000_dp, &
      &  512.6759_dp,  391.3606_dp,  291.4916_dp,    0.0000_dp,    0.0000_dp, &
      &  320.6678_dp,  255.2289_dp,  196.3102_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  564.0716_dp,  437.0209_dp,  328.9434_dp,    0.0000_dp,    0.0000_dp, &
      &  508.5289_dp,  396.0634_dp,  299.5256_dp,    0.0000_dp,    0.0000_dp, &
      &  388.9956_dp,  308.5459_dp,  236.9854_dp,    0.0000_dp,    0.0000_dp, &
      &  295.9256_dp,  240.2687_dp,  188.0446_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  431.2132_dp,  344.2113_dp,  265.5808_dp,    0.0000_dp,    0.0000_dp, &
      &  396.9862_dp,  318.1795_dp,  246.4680_dp,    0.0000_dp,    0.0000_dp, &
      &  367.7306_dp,  295.9631_dp,  230.0988_dp,    0.0000_dp,    0.0000_dp, &
      &  359.6929_dp,  288.5406_dp,  223.9239_dp,    0.0000_dp,    0.0000_dp, &
      &  289.9361_dp,  237.2798_dp,  187.1659_dp,    0.0000_dp,    0.0000_dp, &
      &  328.4072_dp,  268.5416_dp,  211.7390_dp,    0.0000_dp,    0.0000_dp, &
      &  318.5912_dp,  260.7012_dp,  205.7533_dp,    0.0000_dp,    0.0000_dp, &
      &  314.8297_dp,  257.2059_dp,  202.7848_dp,    0.0000_dp,    0.0000_dp, &
      &  290.2077_dp,  238.6014_dp,  189.1560_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  268.7824_dp,  223.0883_dp,  178.4935_dp,    0.0000_dp,    0.0000_dp, &
      &  265.9528_dp,  220.6477_dp,  176.5078_dp,    0.0000_dp,    0.0000_dp, &
      &  260.5335_dp,  216.2066_dp,  173.0241_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.8070_dp,  182.4721_dp,  148.0763_dp,    0.0000_dp,    0.0000_dp, &
      &  214.6380_dp,  180.5916_dp,  146.5298_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  175.5876_dp,  149.5222_dp,  122.8625_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1447.3048_dp, 1015.7997_dp,  715.2376_dp,    0.0000_dp,    0.0000_dp, &
      &  449.3030_dp,  349.4753_dp,  265.8033_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1131.1769_dp,  835.2692_dp,  607.2921_dp,    0.0000_dp,    0.0000_dp, &
      &  905.7515_dp,  668.7646_dp,  488.3352_dp,    0.0000_dp,    0.0000_dp, &
      &  450.9597_dp,  358.3732_dp,  276.2842_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  923.5811_dp,  689.6088_dp,  505.2854_dp,    0.0000_dp,    0.0000_dp, &
      &  437.8432_dp,  349.0702_dp,  269.9059_dp,    0.0000_dp,    0.0000_dp, &
      &  428.2629_dp,  344.2948_dp,  267.9088_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  883.1492_dp,  665.8143_dp,  491.8643_dp,    0.0000_dp,    0.0000_dp, &
      &  422.5708_dp,  337.0553_dp,  260.9491_dp,    0.0000_dp,    0.0000_dp, &
      &  433.1914_dp,  348.5490_dp,  271.4387_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  803.8807_dp,  609.6159_dp,  452.4755_dp,    0.0000_dp,    0.0000_dp, &
      &  414.3575_dp,  330.3567_dp,  255.7125_dp,    0.0000_dp,    0.0000_dp, &
      &  397.1558_dp,  319.1584_dp,  248.6021_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(969 * maxRef * maxRef + 1 : 988 * maxRef * maxRef) = [ &
      &  633.4394_dp,  480.6382_dp,  357.8667_dp,    0.0000_dp,    0.0000_dp, &
      &  411.8187_dp,  327.3553_dp,  252.8515_dp,    0.0000_dp,    0.0000_dp, &
      &  294.2258_dp,  240.6605_dp,  190.7170_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  686.1772_dp,  524.7748_dp,  392.1922_dp,    0.0000_dp,    0.0000_dp, &
      &  373.3442_dp,  301.0397_dp,  234.9229_dp,    0.0000_dp,    0.0000_dp, &
      &  249.0737_dp,  206.0020_dp,  164.9015_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  538.1435_dp,  412.2468_dp,  309.3380_dp,    0.0000_dp,    0.0000_dp, &
      &  344.2322_dp,  278.6526_dp,  218.2588_dp,    0.0000_dp,    0.0000_dp, &
      &  244.8184_dp,  202.2130_dp,  161.6818_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  562.4800_dp,  436.4054_dp,  329.8121_dp,    0.0000_dp,    0.0000_dp, &
      &  328.5015_dp,  265.8752_dp,  208.3381_dp,    0.0000_dp,    0.0000_dp, &
      &  291.9635_dp,  236.9046_dp,  186.3559_dp,    0.0000_dp,    0.0000_dp, &
      &  256.9295_dp,  210.9797_dp,  167.7448_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  582.9014_dp,  449.5806_dp,  338.0805_dp,    0.0000_dp,    0.0000_dp, &
      &  321.5506_dp,  259.9472_dp,  203.5228_dp,    0.0000_dp,    0.0000_dp, &
      &  270.5502_dp,  220.7203_dp,  174.4070_dp,    0.0000_dp,    0.0000_dp, &
      &  287.4614_dp,  231.8285_dp,  181.5196_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  447.4259_dp,  345.1043_dp,  260.4199_dp,    0.0000_dp,    0.0000_dp, &
      &  321.2502_dp,  256.9480_dp,  199.5641_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  450.1543_dp,  353.7880_dp,  270.1794_dp,    0.0000_dp,    0.0000_dp, &
      &  295.4307_dp,  240.9124_dp,  189.8458_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  536.1093_dp,  418.8573_dp,  318.1224_dp,    0.0000_dp,    0.0000_dp, &
      &  500.5162_dp,  393.3571_dp,  300.2053_dp,    0.0000_dp,    0.0000_dp, &
      &  388.6043_dp,  311.5991_dp,  241.8646_dp,    0.0000_dp,    0.0000_dp, &
      &  311.6254_dp,  254.8812_dp,  201.1040_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  461.6795_dp,  369.1734_dp,  285.6102_dp,    0.0000_dp,    0.0000_dp, &
      &  429.6996_dp,  345.3872_dp,  268.4538_dp,    0.0000_dp,    0.0000_dp, &
      &  407.6339_dp,  328.8341_dp,  256.3994_dp,    0.0000_dp,    0.0000_dp, &
      &  362.0366_dp,  294.6634_dp,  231.5527_dp,    0.0000_dp,    0.0000_dp, &
      &  316.0442_dp,  260.1176_dp,  206.3914_dp,    0.0000_dp,    0.0000_dp, &
      &  385.9883_dp,  314.9253_dp,  247.9607_dp,    0.0000_dp,    0.0000_dp, &
      &  379.0427_dp,  309.6035_dp,  244.0467_dp,    0.0000_dp,    0.0000_dp, &
      &  368.3695_dp,  301.4924_dp,  238.0966_dp,    0.0000_dp,    0.0000_dp, &
      &  353.7383_dp,  290.4181_dp,  229.9806_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  342.3814_dp,  282.7302_dp,  225.1670_dp,    0.0000_dp,    0.0000_dp, &
      &  342.5970_dp,  282.8460_dp,  225.2330_dp,    0.0000_dp,    0.0000_dp, &
      &  339.8157_dp,  280.6374_dp,  223.5555_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  296.2668_dp,  247.6203_dp,  199.5549_dp,    0.0000_dp,    0.0000_dp, &
      &  298.5174_dp,  249.3404_dp,  200.8289_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  255.1516_dp,  215.5683_dp,  175.6524_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1615.8387_dp, 1138.5265_dp,  805.0065_dp,    0.0000_dp,    0.0000_dp, &
      &  533.6135_dp,  417.0831_dp,  318.8041_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1351.1988_dp,  995.6116_dp,  723.6911_dp,    0.0000_dp,    0.0000_dp, &
      & 1106.1259_dp,  814.1975_dp,  594.0387_dp,    0.0000_dp,    0.0000_dp, &
      &  539.9639_dp,  429.5383_dp,  331.8687_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1159.9071_dp,  871.2600_dp,  642.4279_dp,    0.0000_dp,    0.0000_dp, &
      &  812.6286_dp,  621.3058_dp,  466.5106_dp,    0.0000_dp,    0.0000_dp, &
      &  511.4722_dp,  413.1903_dp,  323.0124_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1029.7665_dp,  782.7166_dp,  582.5544_dp,    0.0000_dp,    0.0000_dp, &
      &  580.9007_dp,  461.9141_dp,  357.0819_dp,    0.0000_dp,    0.0000_dp, &
      &  513.6197_dp,  416.7152_dp,  326.9477_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.7445_dp,  713.6477_dp,  534.6266_dp,    0.0000_dp,    0.0000_dp, &
      &  586.8945_dp,  466.5358_dp,  360.5430_dp,    0.0000_dp,    0.0000_dp, &
      &  497.9286_dp,  403.2032_dp,  316.2990_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  710.2837_dp,  551.1240_dp,  417.9889_dp,    0.0000_dp,    0.0000_dp, &
      &  544.8075_dp,  435.1062_dp,  337.8186_dp,    0.0000_dp,    0.0000_dp, &
      &  435.0888_dp,  356.6338_dp,  282.8832_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(988 * maxRef * maxRef + 1 : 1007 * maxRef * maxRef) = [ &
      &  796.1951_dp,  614.8185_dp,  464.1239_dp,    0.0000_dp,    0.0000_dp, &
      &  623.1386_dp,  488.3264_dp,  374.0345_dp,    0.0000_dp,    0.0000_dp, &
      &  380.5730_dp,  314.9171_dp,  252.0075_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  598.1988_dp,  468.5695_dp,  358.4840_dp,    0.0000_dp,    0.0000_dp, &
      &  468.5695_dp,  379.2655_dp,  297.6676_dp,    0.0000_dp,    0.0000_dp, &
      &  358.4840_dp,  297.6676_dp,  239.0071_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   43.1661_dp,   31.1024_dp,   26.9803_dp,    0.0000_dp,    0.0000_dp, &
      &   71.1985_dp,   49.9090_dp,   42.7714_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   27.5745_dp,   20.7776_dp,   18.4026_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  794.2006_dp,  482.4342_dp,  390.2766_dp,    0.0000_dp,    0.0000_dp, &
      &  245.2561_dp,  161.3499_dp,  134.6931_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  424.4120_dp,  277.4174_dp,  230.8996_dp,    0.0000_dp,    0.0000_dp, &
      &  310.3781_dp,  205.9530_dp,  172.6021_dp,    0.0000_dp,    0.0000_dp, &
      &  194.3984_dp,  134.4576_dp,  114.5587_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.7195_dp,  187.1327_dp,  158.7124_dp,    0.0000_dp,    0.0000_dp, &
      &  228.2822_dp,  157.5949_dp,  134.2505_dp,    0.0000_dp,    0.0000_dp, &
      &  190.0445_dp,  132.4776_dp,  113.3508_dp,    0.0000_dp,    0.0000_dp, &
      &  141.1109_dp,  100.8958_dp,   87.2376_dp,    0.0000_dp,    0.0000_dp, &
      &  132.3281_dp,   94.9987_dp,   82.2887_dp,    0.0000_dp,    0.0000_dp, &
      &  179.2708_dp,  126.9085_dp,  109.2841_dp,    0.0000_dp,    0.0000_dp, &
      &  168.3115_dp,  119.1265_dp,  102.6155_dp,    0.0000_dp,    0.0000_dp, &
      &  135.4760_dp,   97.2961_dp,   84.3298_dp,    0.0000_dp,    0.0000_dp, &
      &  126.7778_dp,   91.1171_dp,   79.0279_dp,    0.0000_dp,    0.0000_dp, &
      &  101.8194_dp,   74.7287_dp,   65.3949_dp,    0.0000_dp,    0.0000_dp, &
      &  122.6426_dp,   89.1659_dp,   77.7074_dp,    0.0000_dp,    0.0000_dp, &
      &  114.3496_dp,   83.2227_dp,   72.5863_dp,    0.0000_dp,    0.0000_dp, &
      &  107.1084_dp,   78.1976_dp,   68.3029_dp,    0.0000_dp,    0.0000_dp, &
      &   94.1007_dp,   69.0005_dp,   60.4149_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   91.4959_dp,   67.8114_dp,   59.6194_dp,    0.0000_dp,    0.0000_dp, &
      &   81.7640_dp,   60.9159_dp,   53.6934_dp,    0.0000_dp,    0.0000_dp, &
      &   71.8071_dp,   53.8670_dp,   47.6422_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.4888_dp,   51.6117_dp,   45.7289_dp,    0.0000_dp,    0.0000_dp, &
      &   55.0240_dp,   42.2084_dp,   37.6967_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   52.2072_dp,   39.8762_dp,   35.5563_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  945.4221_dp,  577.0181_dp,  467.8086_dp,    0.0000_dp,    0.0000_dp, &
      &  361.4287_dp,  236.1444_dp,  196.6926_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  685.6628_dp,  442.3381_dp,  366.3245_dp,    0.0000_dp,    0.0000_dp, &
      &  561.0010_dp,  365.0705_dp,  303.5312_dp,    0.0000_dp,    0.0000_dp, &
      &  351.4302_dp,  238.3187_dp,  201.3988_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  617.9244_dp,  407.6999_dp,  340.7455_dp,    0.0000_dp,    0.0000_dp, &
      &  557.0043_dp,  369.5814_dp,  309.6367_dp,    0.0000_dp,    0.0000_dp, &
      &  425.9837_dp,  288.1291_dp,  243.3210_dp,    0.0000_dp,    0.0000_dp, &
      &  324.0408_dp,  224.5657_dp,  191.4938_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  472.3738_dp,  321.4594_dp,  272.0913_dp,    0.0000_dp,    0.0000_dp, &
      &  434.7251_dp,  297.2339_dp,  252.1101_dp,    0.0000_dp,    0.0000_dp, &
      &  402.6239_dp,  276.5427_dp,  235.0108_dp,    0.0000_dp,    0.0000_dp, &
      &  393.6613_dp,  269.6241_dp,  228.9329_dp,    0.0000_dp,    0.0000_dp, &
      &  317.2064_dp,  221.9100_dp,  190.0213_dp,    0.0000_dp,    0.0000_dp, &
      &  359.2547_dp,  251.1477_dp,  215.0145_dp,    0.0000_dp,    0.0000_dp, &
      &  348.4449_dp,  243.8439_dp,  208.8723_dp,    0.0000_dp,    0.0000_dp, &
      &  344.2987_dp,  240.5751_dp,  205.9678_dp,    0.0000_dp,    0.0000_dp, &
      &  317.2629_dp,  223.2549_dp,  191.6962_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.5214_dp,  208.9067_dp,  180.2700_dp,    0.0000_dp,    0.0000_dp, &
      &  290.4146_dp,  206.6258_dp,  178.2877_dp,    0.0000_dp,    0.0000_dp, &
      &  284.4682_dp,  202.4803_dp,  174.7517_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  236.2750_dp,  171.1187_dp,  148.8148_dp,    0.0000_dp,    0.0000_dp, &
      &  233.9031_dp,  169.3584_dp,  147.2755_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1007 * maxRef * maxRef + 1 : 1026 * maxRef * maxRef) = [ &
      &  190.9352_dp,  140.4273_dp,  122.9822_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1567.4954_dp,  948.5140_dp,  767.4205_dp,    0.0000_dp,    0.0000_dp, &
      &  490.5807_dp,  326.5947_dp,  274.5928_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1234.6013_dp,  779.0547_dp,  640.1431_dp,    0.0000_dp,    0.0000_dp, &
      &  986.3727_dp,  624.2850_dp,  514.1385_dp,    0.0000_dp,    0.0000_dp, &
      &  493.3926_dp,  334.8827_dp,  283.4399_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1009.1933_dp,  643.1893_dp,  530.5111_dp,    0.0000_dp,    0.0000_dp, &
      &  478.9598_dp,  326.2503_dp,  276.5669_dp,    0.0000_dp,    0.0000_dp, &
      &  468.5358_dp,  321.8583_dp,  273.7179_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  965.2308_dp,  621.1807_dp,  514.4744_dp,    0.0000_dp,    0.0000_dp, &
      &  462.0314_dp,  315.0982_dp,  267.2985_dp,    0.0000_dp,    0.0000_dp, &
      &  473.8973_dp,  325.8582_dp,  277.2406_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  878.8027_dp,  568.8348_dp,  472.2401_dp,    0.0000_dp,    0.0000_dp, &
      &  453.0262_dp,  308.8482_dp,  261.9807_dp,    0.0000_dp,    0.0000_dp, &
      &  434.2176_dp,  298.4577_dp,  253.9853_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  691.6314_dp,  448.7527_dp,  373.2232_dp,    0.0000_dp,    0.0000_dp, &
      &  450.1943_dp,  306.0340_dp,  259.3191_dp,    0.0000_dp,    0.0000_dp, &
      &  321.1384_dp,  225.3888_dp,  193.5970_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  750.3251_dp,  489.7933_dp,  408.0411_dp,    0.0000_dp,    0.0000_dp, &
      &  408.3677_dp,  281.5134_dp,  239.7905_dp,    0.0000_dp,    0.0000_dp, &
      &  271.6188_dp,  193.0894_dp,  166.7533_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.7544_dp,  385.0153_dp,  321.4850_dp,    0.0000_dp,    0.0000_dp, &
      &  376.4045_dp,  260.6567_dp,  222.4640_dp,    0.0000_dp,    0.0000_dp, &
      &  267.0048_dp,  189.5285_dp,  163.5813_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  615.4364_dp,  407.4574_dp,  341.3760_dp,    0.0000_dp,    0.0000_dp, &
      &  359.1057_dp,  248.7388_dp,  212.3500_dp,    0.0000_dp,    0.0000_dp, &
      &  318.8732_dp,  221.7615_dp,  189.7380_dp,    0.0000_dp,    0.0000_dp, &
      &  280.3953_dp,  197.6518_dp,  170.0773_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  637.7634_dp,  419.6709_dp,  350.7171_dp,    0.0000_dp,    0.0000_dp, &
      &  351.4986_dp,  243.1909_dp,  207.5282_dp,    0.0000_dp,    0.0000_dp, &
      &  295.4451_dp,  206.6729_dp,  177.2508_dp,    0.0000_dp,    0.0000_dp, &
      &  313.9577_dp,  216.9806_dp,  185.2154_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  488.7501_dp,  322.3988_dp,  269.9886_dp,    0.0000_dp,    0.0000_dp, &
      &  351.1222_dp,  240.3260_dp,  204.2511_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  492.6319_dp,  330.4713_dp,  278.3620_dp,    0.0000_dp,    0.0000_dp, &
      &  322.9831_dp,  225.4398_dp,  193.0221_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  586.7732_dp,  391.0508_dp,  328.3978_dp,    0.0000_dp,    0.0000_dp, &
      &  547.8346_dp,  367.3197_dp,  309.2352_dp,    0.0000_dp,    0.0000_dp, &
      &  425.2069_dp,  291.2261_dp,  247.3262_dp,    0.0000_dp,    0.0000_dp, &
      &  340.8328_dp,  238.4311_dp,  204.2277_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  505.4452_dp,  344.9100_dp,  292.3920_dp,    0.0000_dp,    0.0000_dp, &
      &  470.3078_dp,  322.7816_dp,  274.2990_dp,    0.0000_dp,    0.0000_dp, &
      &  446.0876_dp,  307.3740_dp,  261.6413_dp,    0.0000_dp,    0.0000_dp, &
      &  396.0183_dp,  275.5707_dp,  235.5323_dp,    0.0000_dp,    0.0000_dp, &
      &  345.5051_dp,  243.4182_dp,  209.1152_dp,    0.0000_dp,    0.0000_dp, &
      &  422.1971_dp,  294.5421_dp,  252.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  414.5411_dp,  289.5934_dp,  247.9167_dp,    0.0000_dp,    0.0000_dp, &
      &  402.8017_dp,  282.0464_dp,  241.6956_dp,    0.0000_dp,    0.0000_dp, &
      &  386.7290_dp,  271.7387_dp,  233.2016_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  374.0632_dp,  264.6717_dp,  227.8311_dp,    0.0000_dp,    0.0000_dp, &
      &  374.2900_dp,  264.7815_dp,  227.9132_dp,    0.0000_dp,    0.0000_dp, &
      &  371.2277_dp,  262.7255_dp,  226.1893_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  323.1941_dp,  232.0580_dp,  201.0472_dp,    0.0000_dp,    0.0000_dp, &
      &  325.6650_dp,  233.6608_dp,  202.3758_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  277.8680_dp,  202.2567_dp,  176.3002_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1026 * maxRef * maxRef + 1 : 1045 * maxRef * maxRef) = [ &
      & 1749.7615_dp, 1063.3477_dp,  862.1222_dp,    0.0000_dp,    0.0000_dp, &
      &  582.3742_dp,  389.9183_dp,  328.6889_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1473.5487_dp,  928.8255_dp,  763.1756_dp,    0.0000_dp,    0.0000_dp, &
      & 1203.4301_dp,  760.2437_dp,  625.8949_dp,    0.0000_dp,    0.0000_dp, &
      &  590.3744_dp,  401.5048_dp,  340.2392_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1266.8436_dp,  812.9338_dp,  672.6836_dp,    0.0000_dp,    0.0000_dp, &
      &  886.2750_dp,  580.4145_dp,  484.7257_dp,    0.0000_dp,    0.0000_dp, &
      &  559.3453_dp,  386.3915_dp,  329.4057_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1125.3403_dp,  730.4970_dp,  607.3080_dp,    0.0000_dp,    0.0000_dp, &
      &  634.8024_dp,  431.8288_dp,  366.0539_dp,    0.0000_dp,    0.0000_dp, &
      &  561.6169_dp,  389.7673_dp,  332.9103_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1018.2915_dp,  666.2259_dp,  555.7204_dp,    0.0000_dp,    0.0000_dp, &
      &  641.4164_dp,  436.1440_dp,  369.6744_dp,    0.0000_dp,    0.0000_dp, &
      &  544.0821_dp,  377.2249_dp,  322.2101_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  775.8353_dp,  514.9128_dp,  432.2782_dp,    0.0000_dp,    0.0000_dp, &
      &  595.1503_dp,  406.9338_dp,  345.7600_dp,    0.0000_dp,    0.0000_dp, &
      &  475.0273_dp,  333.9387_dp,  286.9340_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  869.8962_dp,  574.2456_dp,  480.9081_dp,    0.0000_dp,    0.0000_dp, &
      &  680.0524_dp,  456.5618_dp,  385.2595_dp,    0.0000_dp,    0.0000_dp, &
      &  415.1463_dp,  295.1012_dp,  254.7749_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  653.1775_dp,  438.0426_dp,  369.4328_dp,    0.0000_dp,    0.0000_dp, &
      &  511.8315_dp,  354.9282_dp,  303.2934_dp,    0.0000_dp,    0.0000_dp, &
      &  390.8937_dp,  279.0333_dp,  241.3497_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  713.9427_dp,  478.2848_dp,  402.9498_dp,    0.0000_dp,    0.0000_dp, &
      &  478.2848_dp,  332.2425_dp,  284.1821_dp,    0.0000_dp,    0.0000_dp, &
      &  402.9498_dp,  284.1821_dp,  244.6586_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   40.2372_dp,   28.9661_dp,   28.0631_dp,    0.0000_dp,    0.0000_dp, &
      &   66.1211_dp,   46.3489_dp,   44.6498_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   25.8716_dp,   19.4689_dp,   19.0208_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  722.4294_dp,  443.9106_dp,  412.2171_dp,    0.0000_dp,    0.0000_dp, &
      &  225.8042_dp,  149.1209_dp,  141.5639_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  390.4310_dp,  256.2327_dp,  242.8825_dp,    0.0000_dp,    0.0000_dp, &
      &  286.0395_dp,  190.4826_dp,  181.1811_dp,    0.0000_dp,    0.0000_dp, &
      &  180.2178_dp,  124.7151_dp,  119.7960_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  253.3425_dp,  173.4360_dp,  166.1533_dp,    0.0000_dp,    0.0000_dp, &
      &  211.5559_dp,  146.2020_dp,  140.3599_dp,    0.0000_dp,    0.0000_dp, &
      &  176.3381_dp,  123.0227_dp,  118.3487_dp,    0.0000_dp,    0.0000_dp, &
      &  131.4019_dp,   93.8997_dp,   90.8284_dp,    0.0000_dp,    0.0000_dp, &
      &  123.2913_dp,   88.4515_dp,   85.6281_dp,    0.0000_dp,    0.0000_dp, &
      &  166.7111_dp,  118.0137_dp,  113.9126_dp,    0.0000_dp,    0.0000_dp, &
      &  156.5006_dp,  110.7946_dp,  106.9320_dp,    0.0000_dp,    0.0000_dp, &
      &  126.2362_dp,   90.6249_dp,   87.7320_dp,    0.0000_dp,    0.0000_dp, &
      &  118.1427_dp,   84.8954_dp,   82.1930_dp,    0.0000_dp,    0.0000_dp, &
      &   95.1612_dp,   69.7730_dp,   67.8363_dp,    0.0000_dp,    0.0000_dp, &
      &  114.4763_dp,   83.1710_dp,   80.7118_dp,    0.0000_dp,    0.0000_dp, &
      &  106.7438_dp,   77.6479_dp,   75.3636_dp,    0.0000_dp,    0.0000_dp, &
      &  100.0370_dp,   72.9965_dp,   70.8909_dp,    0.0000_dp,    0.0000_dp, &
      &   87.9430_dp,   64.4651_dp,   62.6532_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   85.6427_dp,   63.4125_dp,   61.7619_dp,    0.0000_dp,    0.0000_dp, &
      &   76.5871_dp,   57.0049_dp,   55.5744_dp,    0.0000_dp,    0.0000_dp, &
      &   67.3354_dp,   50.4709_dp,   49.2629_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.2707_dp,   48.3824_dp,   47.2645_dp,    0.0000_dp,    0.0000_dp, &
      &   51.7740_dp,   39.6636_dp,   38.8697_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   49.0990_dp,   37.4647_dp,   36.6837_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1045 * maxRef * maxRef + 1 : 1064 * maxRef * maxRef) = [ &
      &  860.6209_dp,  531.1676_dp,  493.9241_dp,    0.0000_dp,    0.0000_dp, &
      &  332.4392_dp,  218.2333_dp,  206.7940_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  629.5912_dp,  408.3022_dp,  385.7118_dp,    0.0000_dp,    0.0000_dp, &
      &  515.6457_dp,  337.2256_dp,  319.2176_dp,    0.0000_dp,    0.0000_dp, &
      &  324.9551_dp,  220.7375_dp,  211.0831_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  569.1410_dp,  376.8776_dp,  358.0287_dp,    0.0000_dp,    0.0000_dp, &
      &  513.4163_dp,  341.7946_dp,  325.1370_dp,    0.0000_dp,    0.0000_dp, &
      &  393.6776_dp,  266.8484_dp,  254.9948_dp,    0.0000_dp,    0.0000_dp, &
      &  300.4977_dp,  208.3470_dp,  200.2114_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  436.9463_dp,  297.8174_dp,  285.0138_dp,    0.0000_dp,    0.0000_dp, &
      &  402.3662_dp,  275.4925_dp,  263.9234_dp,    0.0000_dp,    0.0000_dp, &
      &  372.8898_dp,  256.4133_dp,  245.8989_dp,    0.0000_dp,    0.0000_dp, &
      &  364.4241_dp,  249.9808_dp,  239.5605_dp,    0.0000_dp,    0.0000_dp, &
      &  294.5220_dp,  206.0675_dp,  198.4222_dp,    0.0000_dp,    0.0000_dp, &
      &  333.5219_dp,  233.2106_dp,  224.5211_dp,    0.0000_dp,    0.0000_dp, &
      &  323.5276_dp,  226.4605_dp,  218.0675_dp,    0.0000_dp,    0.0000_dp, &
      &  319.6077_dp,  223.4125_dp,  215.0564_dp,    0.0000_dp,    0.0000_dp, &
      &  294.7986_dp,  207.4505_dp,  199.9979_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.1568_dp,  194.3373_dp,  187.8007_dp,    0.0000_dp,    0.0000_dp, &
      &  270.2524_dp,  192.2172_dp,  185.7368_dp,    0.0000_dp,    0.0000_dp, &
      &  264.7345_dp,  188.3752_dp,  182.0388_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  220.4099_dp,  159.4914_dp,  154.6693_dp,    0.0000_dp,    0.0000_dp, &
      &  218.1903_dp,  157.8524_dp,  153.0708_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  178.5021_dp,  131.1338_dp,  127.5483_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1424.0271_dp,  873.2984_dp,  809.4255_dp,    0.0000_dp,    0.0000_dp, &
      &  452.2464_dp,  302.4635_dp,  287.8260_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1129.6670_dp,  718.4983_dp,  674.5740_dp,    0.0000_dp,    0.0000_dp, &
      &  902.4405_dp,  576.1374_dp,  541.0773_dp,    0.0000_dp,    0.0000_dp, &
      &  456.2063_dp,  310.3909_dp,  296.8146_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  924.8556_dp,  593.4994_dp,  558.7593_dp,    0.0000_dp,    0.0000_dp, &
      &  443.0816_dp,  302.4898_dp,  289.4965_dp,    0.0000_dp,    0.0000_dp, &
      &  433.9484_dp,  298.5692_dp,  286.3025_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  885.7657_dp,  573.5972_dp,  541.3741_dp,    0.0000_dp,    0.0000_dp, &
      &  427.4577_dp,  292.2135_dp,  279.7105_dp,    0.0000_dp,    0.0000_dp, &
      &  438.9752_dp,  302.3112_dp,  289.9530_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  807.1258_dp,  525.4732_dp,  496.6905_dp,    0.0000_dp,    0.0000_dp, &
      &  419.1134_dp,  286.4278_dp,  274.1490_dp,    0.0000_dp,    0.0000_dp, &
      &  402.1750_dp,  276.9472_dp,  265.5767_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  635.3257_dp,  414.7918_dp,  392.2297_dp,    0.0000_dp,    0.0000_dp, &
      &  416.3230_dp,  283.7785_dp,  271.4250_dp,    0.0000_dp,    0.0000_dp, &
      &  298.3014_dp,  209.5976_dp,  201.9007_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  689.9647_dp,  452.7353_dp,  428.8509_dp,    0.0000_dp,    0.0000_dp, &
      &  378.4295_dp,  261.2635_dp,  250.7187_dp,    0.0000_dp,    0.0000_dp, &
      &  252.7595_dp,  179.7824_dp,  173.6495_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  540.6599_dp,  356.1344_dp,  337.5770_dp,    0.0000_dp,    0.0000_dp, &
      &  349.0258_dp,  242.0147_dp,  232.4750_dp,    0.0000_dp,    0.0000_dp, &
      &  248.4203_dp,  176.4513_dp,  170.3774_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  567.1044_dp,  376.9928_dp,  358.3743_dp,    0.0000_dp,    0.0000_dp, &
      &  332.9896_dp,  230.9788_dp,  221.8767_dp,    0.0000_dp,    0.0000_dp, &
      &  295.8366_dp,  206.0649_dp,  198.1008_dp,    0.0000_dp,    0.0000_dp, &
      &  260.6337_dp,  183.8900_dp,  177.3010_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.1819_dp,  388.1200_dp,  368.4032_dp,    0.0000_dp,    0.0000_dp, &
      &  325.8858_dp,  225.8165_dp,  216.8602_dp,    0.0000_dp,    0.0000_dp, &
      &  274.3366_dp,  192.1416_dp,  184.9554_dp,    0.0000_dp,    0.0000_dp, &
      &  291.0309_dp,  201.5536_dp,  193.4900_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  450.0559_dp,  298.3894_dp,  283.3292_dp,    0.0000_dp,    0.0000_dp, &
      &  325.0466_dp,  223.0154_dp,  213.6379_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1064 * maxRef * maxRef + 1 : 1083 * maxRef * maxRef) = [ &
      &  454.8031_dp,  306.0633_dp,  291.8725_dp,    0.0000_dp,    0.0000_dp, &
      &  299.8317_dp,  209.4546_dp,  201.5508_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  541.1501_dp,  361.8780_dp,  344.5686_dp,    0.0000_dp,    0.0000_dp, &
      &  505.6698_dp,  340.0675_dp,  324.2726_dp,    0.0000_dp,    0.0000_dp, &
      &  393.6370_dp,  270.0591_dp,  258.7879_dp,    0.0000_dp,    0.0000_dp, &
      &  316.4721_dp,  221.4649_dp,  213.2345_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  467.7042_dp,  319.6946_dp,  306.1192_dp,    0.0000_dp,    0.0000_dp, &
      &  435.5268_dp,  299.3280_dp,  286.9870_dp,    0.0000_dp,    0.0000_dp, &
      &  413.3244_dp,  285.1360_dp,  273.6234_dp,    0.0000_dp,    0.0000_dp, &
      &  367.4287_dp,  255.8441_dp,  246.0490_dp,    0.0000_dp,    0.0000_dp, &
      &  321.1117_dp,  226.2274_dp,  218.1530_dp,    0.0000_dp,    0.0000_dp, &
      &  391.8528_dp,  273.5010_dp,  263.1821_dp,    0.0000_dp,    0.0000_dp, &
      &  384.8152_dp,  268.9427_dp,  258.8699_dp,    0.0000_dp,    0.0000_dp, &
      &  374.0353_dp,  261.9897_dp,  252.3030_dp,    0.0000_dp,    0.0000_dp, &
      &  359.2842_dp,  252.4918_dp,  243.3399_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  347.8393_dp,  246.0921_dp,  237.5192_dp,    0.0000_dp,    0.0000_dp, &
      &  348.0402_dp,  246.1937_dp,  237.6065_dp,    0.0000_dp,    0.0000_dp, &
      &  345.2124_dp,  244.2954_dp,  235.7943_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  301.1390_dp,  216.0945_dp,  209.1971_dp,    0.0000_dp,    0.0000_dp, &
      &  303.4107_dp,  217.5738_dp,  210.5970_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  259.3964_dp,  188.6339_dp,  183.1095_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1590.4415_dp,  979.4032_dp,  908.8152_dp,    0.0000_dp,    0.0000_dp, &
      &  537.2565_dp,  361.3047_dp,  344.2670_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1347.9007_dp,  856.7169_dp,  804.0036_dp,    0.0000_dp,    0.0000_dp, &
      & 1100.5370_dp,  701.6550_dp,  658.5196_dp,    0.0000_dp,    0.0000_dp, &
      &  545.9768_dp,  372.2612_dp,  356.1148_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1161.9289_dp,  750.5933_dp,  707.8343_dp,    0.0000_dp,    0.0000_dp, &
      &  814.8402_dp,  536.8566_dp,  508.6839_dp,    0.0000_dp,    0.0000_dp, &
      &  518.4374_dp,  358.6133_dp,  344.3053_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1033.8735_dp,  674.9929_dp,  638.4463_dp,    0.0000_dp,    0.0000_dp, &
      &  587.0202_dp,  400.4209_dp,  383.0425_dp,    0.0000_dp,    0.0000_dp, &
      &  520.8781_dp,  361.8771_dp,  347.7992_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  936.5528_dp,  615.9759_dp,  583.7653_dp,    0.0000_dp,    0.0000_dp, &
      &  593.1414_dp,  404.4300_dp,  386.8574_dp,    0.0000_dp,    0.0000_dp, &
      &  504.5037_dp,  350.2910_dp,  336.5535_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  714.9495_dp,  476.7090_dp,  453.3284_dp,    0.0000_dp,    0.0000_dp, &
      &  550.7557_dp,  377.5535_dp,  361.5781_dp,    0.0000_dp,    0.0000_dp, &
      &  441.3319_dp,  310.5005_dp,  299.2251_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  801.0442_dp,  531.3700_dp,  504.6607_dp,    0.0000_dp,    0.0000_dp, &
      &  627.5810_dp,  423.1279_dp,  403.4188_dp,    0.0000_dp,    0.0000_dp, &
      &  386.2986_dp,  274.6918_dp,  265.3352_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  602.7956_dp,  405.9398_dp,  386.9728_dp,    0.0000_dp,    0.0000_dp, &
      &  474.6369_dp,  329.6826_dp,  316.7499_dp,    0.0000_dp,    0.0000_dp, &
      &  363.9513_dp,  259.8584_dp,  251.2231_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  658.8753_dp,  443.0659_dp,  422.3203_dp,    0.0000_dp,    0.0000_dp, &
      &  443.6280_dp,  308.7059_dp,  296.6934_dp,    0.0000_dp,    0.0000_dp, &
      &  374.5394_dp,  264.4321_dp,  254.9713_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  608.5041_dp,  411.1361_dp,  392.3548_dp,    0.0000_dp,    0.0000_dp, &
      &  411.1361_dp,  286.9519_dp,  275.9575_dp,    0.0000_dp,    0.0000_dp, &
      &  392.3548_dp,  275.9575_dp,  265.8746_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   33.5413_dp,   27.9518_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   54.9658_dp,   44.7835_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   21.7896_dp,   18.7731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1083 * maxRef * maxRef + 1 : 1102 * maxRef * maxRef) = [ &
      &  614.1477_dp,  432.6078_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  188.3374_dp,  144.6563_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.5775_dp,  248.6573_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  238.6962_dp,  184.6530_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  149.7198_dp,  120.5975_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  210.6811_dp,  167.8350_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  176.0603_dp,  141.4060_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  146.9017_dp,  118.9236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.4741_dp,   90.6508_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  102.7753_dp,   85.3773_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.8758_dp,  113.9993_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  130.5144_dp,  107.0286_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.3572_dp,   87.4951_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   98.7032_dp,   81.9715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.6657_dp,   67.3070_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   95.7137_dp,   80.2663_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   89.3419_dp,   74.9354_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   83.7875_dp,   70.4509_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   73.8057_dp,   62.2209_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   71.8926_dp,   61.1748_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.3880_dp,   54.9856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   56.7511_dp,   48.6912_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   54.1903_dp,   46.6726_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   43.8480_dp,   38.2539_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.5780_dp,   36.1486_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  731.0832_dp,  517.5680_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  277.9428_dp,  211.8419_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  526.4776_dp,  396.6125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  431.2807_dp,  327.3509_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  270.1633_dp,  213.7477_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  474.5713_dp,  365.5269_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  427.9962_dp,  331.3832_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.7472_dp,  258.4065_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.6759_dp,  201.4661_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  363.3585_dp,  288.2863_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  334.6673_dp,  266.6008_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  310.1186_dp,  248.0750_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  303.3752_dp,  241.8989_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  244.8685_dp,  199.1553_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  277.3577_dp,  225.3945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  269.1220_dp,  218.8617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  265.9503_dp,  215.9419_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  245.2996_dp,  200.4348_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  227.5024_dp,  187.6597_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  225.1181_dp,  185.6197_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  220.5585_dp,  181.9100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  183.9876_dp,  153.9032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  182.1547_dp,  152.3270_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  149.4269_dp,  126.4825_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1218.4699_dp,  851.0088_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  379.0836_dp,  293.2911_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  951.4935_dp,  698.8618_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  762.6174_dp,  560.1298_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  380.4283_dp,  300.6094_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  776.8914_dp,  576.9636_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  369.4933_dp,  292.9063_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  361.5057_dp,  288.9408_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1102 * maxRef * maxRef + 1 : 1121 * maxRef * maxRef) = [ &
      &  743.2041_dp,  557.2880_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  356.7245_dp,  282.9332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  365.7124_dp,  292.5514_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  676.6553_dp,  510.3644_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  349.8288_dp,  277.3525_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  335.4436_dp,  268.0353_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.6878_dp,  402.8383_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  347.6786_dp,  274.8452_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.1613_dp,  202.6684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  577.8127_dp,  439.5099_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  315.3482_dp,  252.8298_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  211.2390_dp,  173.7381_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  453.6327_dp,  345.6995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  290.9083_dp,  234.1511_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  207.6181_dp,  170.5403_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  473.9244_dp,  365.6925_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  277.6823_dp,  223.4805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  247.0487_dp,  199.3629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  217.7112_dp,  177.7934_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  490.9783_dp,  376.6281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  271.8088_dp,  218.5075_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  229.0537_dp,  185.8430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  243.2009_dp,  195.0917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  377.3531_dp,  289.5580_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  271.4634_dp,  215.9581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  379.5655_dp,  296.6728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.8348_dp,  202.5685_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  451.5887_dp,  350.8010_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  421.7420_dp,  329.5387_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.8852_dp,  261.3654_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.3258_dp,  214.0794_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.3075_dp,  309.4543_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  362.5076_dp,  289.6411_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  344.0065_dp,  275.8484_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.7755_dp,  247.3728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  267.2135_dp,  218.5854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  326.0371_dp,  264.3957_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  320.2231_dp,  259.9706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  311.2799_dp,  253.2180_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  299.0137_dp,  243.9926_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.6469_dp,  237.7217_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.8328_dp,  237.8238_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  287.5025_dp,  235.9871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  251.1189_dp,  208.6044_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  253.0095_dp,  210.0411_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.7336_dp,  182.0016_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1360.7517_dp,  954.1320_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  450.4714_dp,  350.2295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1136.9099_dp,  833.3168_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.6022_dp,  682.1817_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  455.7267_dp,  360.4891_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  976.2025_dp,  729.3316_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  684.9796_dp,  520.8724_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  431.9705_dp,  346.9360_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  867.0056_dp,  655.4329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  490.3817_dp,  387.7336_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  433.9278_dp,  349.9975_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1121 * maxRef * maxRef + 1 : 1140 * maxRef * maxRef) = [ &
      &  784.8188_dp,  597.8528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  495.4800_dp,  391.6637_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  420.8552_dp,  338.8335_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  599.0530_dp,  462.3383_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  460.2367_dp,  365.5261_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  368.2898_dp,  300.1493_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  671.1717_dp,  515.4953_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  526.0977_dp,  410.0667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  322.5821_dp,  265.4033_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  505.0269_dp,  393.5088_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  396.2742_dp,  318.9665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  304.0550_dp,  251.0367_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  551.0802_dp,  429.5143_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.6378_dp,  298.6678_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  313.1143_dp,  255.6479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  508.7742_dp,  398.4662_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.6707_dp,  277.6023_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.8015_dp,  266.8459_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  426.7450_dp,  333.1583_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  333.1583_dp,  268.6130_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.6879_dp,   27.9281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   58.3568_dp,   44.5006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   23.1563_dp,   18.8607_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  621.0536_dp,  408.4429_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.0940_dp,  141.0901_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  340.4122_dp,  242.0950_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.9915_dp,  180.5207_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  158.6937_dp,  119.4623_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  222.6273_dp,  165.6587_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  186.2199_dp,  139.8793_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  155.4778_dp,  117.8843_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  116.3922_dp,   90.4277_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.2887_dp,   85.2274_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.4138_dp,  113.4369_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.3672_dp,  106.4332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.9252_dp,   87.2915_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  104.7692_dp,   81.7500_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.7139_dp,   67.4028_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  101.7377_dp,   80.2492_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   94.8770_dp,   74.8925_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   88.9858_dp,   70.4420_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   78.3016_dp,   62.2131_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.4092_dp,   61.3234_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.3943_dp,   55.1392_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   60.2334_dp,   48.8497_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   57.5500_dp,   46.8692_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   46.5340_dp,   38.4906_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.1062_dp,   36.3401_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  740.5787_dp,  489.5129_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.8474_dp,  205.9647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  547.6721_dp,  384.1728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  449.1567_dp,  317.9002_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  285.2161_dp,  210.5498_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1140 * maxRef * maxRef + 1 : 1159 * maxRef * maxRef) = [ &
      &  497.0337_dp,  356.8654_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  448.8071_dp,  324.0830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  345.2997_dp,  254.2236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  264.7291_dp,  199.6670_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.6790_dp,  284.2139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  353.5976_dp,  263.1438_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.9622_dp,  245.1614_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  320.3429_dp,  238.7691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  259.8815_dp,  197.7904_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.2465_dp,  223.7783_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  285.4802_dp,  217.3199_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.9469_dp,  214.2984_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  260.3892_dp,  199.2714_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  241.7614_dp,  187.0163_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  239.1784_dp,  184.9522_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  234.3154_dp,  181.2582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.7044_dp,  153.8592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  193.7273_dp,  152.2646_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  158.9650_dp,  126.7356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1222.5553_dp,  800.2593_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  395.5135_dp,  286.2735_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  978.4251_dp,  670.2633_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.6600_dp,  537.0889_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  400.4535_dp,  295.6941_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  802.6020_dp,  555.6679_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.1859_dp,  288.3875_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  381.7298_dp,  285.2814_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  770.0303_dp,  538.5502_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  375.5261_dp,  278.5869_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  386.2241_dp,  288.9111_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  702.4213_dp,  494.2237_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  368.1871_dp,  273.0331_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  353.8227_dp,  264.5314_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  553.0919_dp,  389.9953_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  365.5531_dp,  270.2890_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.4511_dp,  200.9381_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  601.4053_dp,  426.8579_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  333.1518_dp,  249.8201_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  223.7644_dp,  172.7698_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  471.5329_dp,  335.7642_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.5209_dp,  231.6170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  219.8750_dp,  169.5246_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  495.6313_dp,  356.9172_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.4074_dp,  221.0248_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  260.8701_dp,  197.2313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  230.4010_dp,  176.4697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  512.6220_dp,  366.8668_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  287.0958_dp,  216.0211_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  242.1824_dp,  184.1433_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  256.3643_dp,  192.6099_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  393.0510_dp,  281.8675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  285.8180_dp,  212.7468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  398.4572_dp,  290.7895_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  264.5716_dp,  200.8224_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  473.4203_dp,  343.2943_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  442.8639_dp,  323.1080_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.0568_dp,  257.9007_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  279.2886_dp,  212.5148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1159 * maxRef * maxRef + 1 : 1178 * maxRef * maxRef) = [ &
      &  410.9058_dp,  305.1468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.0204_dp,  286.0551_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  363.7536_dp,  272.7228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  323.9292_dp,  245.2119_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  283.7237_dp,  217.3760_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  345.6085_dp,  262.2840_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  339.4814_dp,  257.9689_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.1075_dp,  251.4082_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  317.2880_dp,  242.4605_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.5539_dp,  236.5779_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.7214_dp,  236.6604_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.2454_dp,  234.8470_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  266.9700_dp,  208.2113_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  268.9491_dp,  209.6083_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  230.5493_dp,  182.0867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1366.3938_dp,  898.5342_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  470.3119_dp,  342.3410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1167.0471_dp,  798.4528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  952.7533_dp,  653.2782_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  479.3797_dp,  354.6096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1009.4418_dp,  703.7992_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  710.2172_dp,  505.6747_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  456.4875_dp,  342.9987_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  900.1154_dp,  635.1370_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  515.3861_dp,  381.3405_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  459.0164_dp,  346.4792_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  816.5495_dp,  580.8412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  520.7629_dp,  385.1206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  444.4932_dp,  335.1312_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  624.9506_dp,  450.9776_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  484.0286_dp,  359.9059_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.8300_dp,  297.8478_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  699.5382_dp,  502.1168_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  549.6338_dp,  401.2382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  341.9237_dp,  264.0201_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  527.9387_dp,  384.9436_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  418.2579_dp,  315.3365_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  322.4084_dp,  249.9360_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  577.0107_dp,  420.3746_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  391.0675_dp,  295.3036_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  331.0815_dp,  253.6663_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.4134_dp,  390.5770_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  362.6432_dp,  274.6154_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.6055_dp,  264.5941_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.8701_dp,  325.9662_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  351.3712_dp,  265.5450_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  468.1900_dp,  345.0599_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  345.0599_dp,  263.4508_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   44.0417_dp,   42.2827_dp,   34.8931_dp,   30.3193_dp,    0.0000_dp, &
      &   72.9076_dp,   69.6754_dp,   56.5189_dp,   48.4270_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   27.9586_dp,   27.0236_dp,   22.9076_dp,   20.3402_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1178 * maxRef * maxRef + 1 : 1197 * maxRef * maxRef) = [ &
      &  828.9673_dp,  769.2514_dp,  565.2654_dp,  444.6978_dp,    0.0000_dp, &
      &  253.1370_dp,  239.0419_dp,  185.9170_dp,  153.8264_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  438.3396_dp,  413.4965_dp,  320.2807_dp,  264.0439_dp,    0.0000_dp, &
      &  320.0912_dp,  302.6678_dp,  236.6771_dp,  196.7454_dp,    0.0000_dp, &
      &  199.4111_dp,  190.1313_dp,  152.9239_dp,  130.1139_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.1737_dp,  267.4750_dp,  213.4484_dp,  180.4568_dp,    0.0000_dp, &
      &  234.2171_dp,  223.1629_dp,  179.1755_dp,  152.2615_dp,    0.0000_dp, &
      &  194.7649_dp,  185.8576_dp,  150.1267_dp,  128.2191_dp,    0.0000_dp, &
      &  144.1259_dp,  138.1855_dp,  113.4826_dp,   98.2235_dp,    0.0000_dp, &
      &  135.0814_dp,  129.5979_dp,  106.6946_dp,   92.5355_dp,    0.0000_dp, &
      &  183.3368_dp,  175.4585_dp,  143.1602_dp,  123.2705_dp,    0.0000_dp, &
      &  172.1229_dp,  164.6812_dp,  134.3014_dp,  115.6130_dp,    0.0000_dp, &
      &  138.2831_dp,  132.6505_dp,  109.2000_dp,   94.7153_dp,    0.0000_dp, &
      &  129.3853_dp,  124.1085_dp,  102.1877_dp,   88.6572_dp,    0.0000_dp, &
      &  103.6198_dp,   99.7522_dp,   83.2270_dp,   72.9703_dp,    0.0000_dp, &
      &  124.9798_dp,  120.1279_dp,   99.6435_dp,   86.9569_dp,    0.0000_dp, &
      &  116.4959_dp,  111.9707_dp,   92.9086_dp,   81.1110_dp,    0.0000_dp, &
      &  109.0825_dp,  104.8972_dp,   87.2076_dp,   76.2530_dp,    0.0000_dp, &
      &   95.7727_dp,   92.1403_dp,   76.7798_dp,   67.2704_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   92.9968_dp,   89.6520_dp,   75.2351_dp,   66.2741_dp,    0.0000_dp, &
      &   83.0278_dp,   80.0986_dp,   67.4193_dp,   59.5344_dp,    0.0000_dp, &
      &   72.8571_dp,   70.3511_dp,   59.4511_dp,   52.6690_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   69.4558_dp,   67.1232_dp,   56.8922_dp,   50.5156_dp,    0.0000_dp, &
      &   55.6589_dp,   53.9364_dp,   46.2082_dp,   41.3746_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   52.8517_dp,   51.1747_dp,   43.7215_dp,   39.0674_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  986.0999_dp,  915.9413_dp,  675.2268_dp,  532.8538_dp,    0.0000_dp, &
      &  373.3794_dp,  352.0074_dp,  272.3735_dp,  224.4097_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  709.3565_dp,  667.2844_dp,  512.0902_dp,  418.8567_dp,    0.0000_dp, &
      &  579.9234_dp,  546.2762_dp,  421.5573_dp,  346.4940_dp,    0.0000_dp, &
      &  361.3835_dp,  343.3098_dp,  272.5677_dp,  229.4425_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  637.5832_dp,  602.4143_dp,  469.4540_dp,  389.0698_dp,    0.0000_dp, &
      &  574.3420_dp,  543.2188_dp,  424.8967_dp,  353.2654_dp,    0.0000_dp, &
      &  438.2474_dp,  416.0033_dp,  329.5920_dp,  276.9940_dp,    0.0000_dp, &
      &  332.3410_dp,  316.9852_dp,  255.2803_dp,  217.4318_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  485.5392_dp,  461.5093_dp,  367.2028_dp,  309.6779_dp,    0.0000_dp, &
      &  446.5937_dp,  424.8293_dp,  339.0315_dp,  286.6412_dp,    0.0000_dp, &
      &  413.3754_dp,  393.5617_dp,  315.0177_dp,  266.9974_dp,    0.0000_dp, &
      &  404.3294_dp,  384.6752_dp,  307.2515_dp,  259.9858_dp,    0.0000_dp, &
      &  324.9345_dp,  310.4122_dp,  251.4716_dp,  215.2424_dp,    0.0000_dp, &
      &  368.0275_dp,  351.5157_dp,  284.6146_dp,  243.5106_dp,    0.0000_dp, &
      &  356.9069_dp,  340.9396_dp,  276.2154_dp,  236.4461_dp,    0.0000_dp, &
      &  352.7247_dp,  336.8285_dp,  272.5864_dp,  233.1422_dp,    0.0000_dp, &
      &  324.7346_dp,  310.5048_dp,  252.4456_dp,  216.7223_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  299.9853_dp,  287.3922_dp,  235.3301_dp,  203.2097_dp,    0.0000_dp, &
      &  296.8214_dp,  284.3350_dp,  232.7649_dp,  200.9558_dp,    0.0000_dp, &
      &  290.7255_dp,  278.5087_dp,  228.0473_dp,  196.9225_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.9104_dp,  231.4554_dp,  191.5816_dp,  166.8867_dp,    0.0000_dp, &
      &  238.4988_dp,  229.1229_dp,  189.6139_dp,  165.1495_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  194.2646_dp,  187.0916_dp,  156.3179_dp,  137.2010_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1638.8153_dp, 1517.3687_dp, 1110.1907_dp,  869.9598_dp,    0.0000_dp, &
      &  505.6838_dp,  478.0141_dp,  374.0585_dp,  311.3187_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1281.2612_dp, 1198.9287_dp,  905.1186_dp,  729.9594_dp,    0.0000_dp, &
      & 1024.1614_dp,  957.9897_dp,  723.9814_dp,  584.4834_dp,    0.0000_dp, &
      &  507.2605_dp,  481.6729_dp,  382.2720_dp,  321.8065_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1045.8401_dp,  980.8464_dp,  745.8282_dp,  605.3047_dp,    0.0000_dp, &
      &  492.1904_dp,  467.6700_dp,  372.0117_dp,  313.7920_dp,    0.0000_dp, &
      &  481.0066_dp,  457.8184_dp,  366.3017_dp,  310.4193_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1197 * maxRef * maxRef + 1 : 1216 * maxRef * maxRef) = [ &
      &  999.0971_dp,  938.7641_dp,  718.5537_dp,  586.5752_dp,    0.0000_dp, &
      &  474.8112_dp,  451.1658_dp,  359.1057_dp,  303.0566_dp,    0.0000_dp, &
      &  486.4497_dp,  463.0777_dp,  370.7390_dp,  314.3438_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  908.9753_dp,  855.0744_dp,  657.1002_dp,  538.2672_dp,    0.0000_dp, &
      &  465.5647_dp,  442.3420_dp,  351.9807_dp,  296.9854_dp,    0.0000_dp, &
      &  445.7766_dp,  424.2272_dp,  339.4399_dp,  287.7026_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  715.2799_dp,  672.8526_dp,  517.5497_dp,  424.4016_dp,    0.0000_dp, &
      &  462.8304_dp,  439.4728_dp,  348.9930_dp,  293.9840_dp,    0.0000_dp, &
      &  328.7836_dp,  314.0026_dp,  254.5781_dp,  218.1566_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  775.2736_dp,  730.5221_dp,  564.6362_dp,  464.8411_dp,    0.0000_dp, &
      &  419.0707_dp,  399.1079_dp,  320.0431_dp,  271.7273_dp,    0.0000_dp, &
      &  277.6350_dp,  265.7575_dp,  217.2467_dp,  187.4120_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  607.0964_dp,  572.1729_dp,  442.9855_dp,  365.3129_dp,    0.0000_dp, &
      &  386.0628_dp,  367.9592_dp,  295.9209_dp,  251.8512_dp,    0.0000_dp, &
      &  272.9769_dp,  261.2268_dp,  213.3373_dp,  183.8976_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  634.7263_dp,  599.8255_dp,  468.1580_dp,  388.6270_dp,    0.0000_dp, &
      &  368.3224_dp,  351.0300_dp,  282.3130_dp,  240.2860_dp,    0.0000_dp, &
      &  326.8852_dp,  311.6813_dp,  251.2206_dp,  214.2453_dp,    0.0000_dp, &
      &  286.9299_dp,  274.2537_dp,  222.9925_dp,  191.5317_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  658.2526_dp,  621.3305_dp,  482.9577_dp,  399.5106_dp,    0.0000_dp, &
      &  360.5731_dp,  343.5620_dp,  276.0871_dp,  234.8390_dp,    0.0000_dp, &
      &  302.6341_dp,  288.8824_dp,  233.7477_dp,  199.9712_dp,    0.0000_dp, &
      &  322.0771_dp,  306.7120_dp,  246.1570_dp,  209.2104_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  504.3802_dp,  476.0165_dp,  370.2813_dp,  306.5977_dp,    0.0000_dp, &
      &  360.6626_dp,  342.8931_dp,  273.5167_dp,  231.2700_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  507.2452_dp,  480.5972_dp,  378.4634_dp,  316.5382_dp,    0.0000_dp, &
      &  330.9585_dp,  315.9166_dp,  255.4101_dp,  218.3027_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  604.7183_dp,  572.2658_dp,  448.7448_dp,  373.9479_dp,    0.0000_dp, &
      &  564.1564_dp,  534.5127_dp,  420.8557_dp,  351.9154_dp,    0.0000_dp, &
      &  436.7407_dp,  415.4641_dp,  331.7913_dp,  280.7261_dp,    0.0000_dp, &
      &  349.1330_dp,  333.4815_dp,  270.0988_dp,  231.1618_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  519.3501_dp,  493.7984_dp,  393.4823_dp,  332.2932_dp,    0.0000_dp, &
      &  482.9115_dp,  459.6285_dp,  367.6282_dp,  311.4260_dp,    0.0000_dp, &
      &  457.8119_dp,  436.0573_dp,  349.6800_dp,  296.8562_dp,    0.0000_dp, &
      &  405.9303_dp,  387.3404_dp,  312.6183_dp,  266.7944_dp,    0.0000_dp, &
      &  353.6035_dp,  338.1831_dp,  275.1658_dp,  236.3761_dp,    0.0000_dp, &
      &  432.6167_dp,  413.0126_dp,  333.9114_dp,  285.3599_dp,    0.0000_dp, &
      &  424.7008_dp,  405.5453_dp,  328.1518_dp,  280.6350_dp,    0.0000_dp, &
      &  412.5541_dp,  394.1073_dp,  319.3719_dp,  273.4592_dp,    0.0000_dp, &
      &  395.9157_dp,  378.4539_dp,  307.3832_dp,  263.6781_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  382.5968_dp,  366.1519_dp,  298.6933_dp,  257.1422_dp,    0.0000_dp, &
      &  382.8381_dp,  366.3652_dp,  298.8244_dp,  257.2275_dp,    0.0000_dp, &
      &  379.6844_dp,  363.3697_dp,  296.4552_dp,  255.2418_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  329.9250_dp,  316.5231_dp,  260.5925_dp,  226.0224_dp,    0.0000_dp, &
      &  332.4791_dp,  318.9305_dp,  262.4494_dp,  227.5472_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  283.1269_dp,  272.2288_dp,  226.0215_dp,  197.3762_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1828.5696_dp, 1694.2091_dp, 1243.0036_dp,  976.6194_dp,    0.0000_dp, &
      &  599.8987_dp,  567.6002_dp,  445.7605_dp,  372.1527_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1529.6862_dp, 1430.6640_dp, 1078.8447_dp,  869.2745_dp,    0.0000_dp, &
      & 1250.1173_dp, 1168.5036_dp,  881.5739_dp,  710.6790_dp,    0.0000_dp, &
      &  606.8145_dp,  576.2896_dp,  457.8102_dp,  385.7594_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1311.8922_dp, 1231.6979_dp,  940.6363_dp,  766.4059_dp,    0.0000_dp, &
      &  916.1592_dp,  862.8833_dp,  667.6471_dp,  550.2164_dp,    0.0000_dp, &
      &  573.8021_dp,  546.6690_dp,  438.9483_dp,  373.0875_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1163.5963_dp, 1095.0419_dp,  842.9594_dp,  691.5997_dp,    0.0000_dp, &
      &  652.5654_dp,  619.6537_dp,  492.2291_dp,  414.7814_dp,    0.0000_dp, &
      &  575.8020_dp,  549.0572_dp,  442.2260_dp,  376.8173_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1216 * maxRef * maxRef + 1 : 1235 * maxRef * maxRef) = [ &
      & 1051.8917_dp,  991.4013_dp,  767.2243_dp,  632.3580_dp,    0.0000_dp, &
      &  659.2865_dp,  626.0254_dp,  497.1382_dp,  418.8482_dp,    0.0000_dp, &
      &  557.9340_dp,  531.7720_dp,  427.8632_dp,  364.3226_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  799.9994_dp,  755.8918_dp,  590.4050_dp,  490.5720_dp,    0.0000_dp, &
      &  611.3911_dp,  581.0549_dp,  463.0670_dp,  391.2722_dp,    0.0000_dp, &
      &  486.2116_dp,  464.5679_dp,  377.1375_dp,  323.4882_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  897.6451_dp,  847.3557_dp,  659.5747_dp,  546.3930_dp,    0.0000_dp, &
      &  700.4229_dp,  663.0380_dp,  521.6978_dp,  436.2161_dp,    0.0000_dp, &
      &  424.3145_dp,  406.2117_dp,  332.1088_dp,  286.5085_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  672.6516_dp,  636.7480_dp,  500.7277_dp,  418.4851_dp,    0.0000_dp, &
      &  524.8236_dp,  500.1628_dp,  402.3825_dp,  342.6308_dp,    0.0000_dp, &
      &  399.3109_dp,  382.5457_dp,  313.5888_dp,  271.1133_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  735.2525_dp,  696.1321_dp,  547.3300_dp,  457.2780_dp,    0.0000_dp, &
      &  490.3325_dp,  467.3771_dp,  376.3625_dp,  320.7454_dp,    0.0000_dp, &
      &  412.2889_dp,  394.0370_dp,  320.4011_dp,  275.2351_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  678.0996_dp,  642.6456_dp,  506.9866_dp,  424.7783_dp,    0.0000_dp, &
      &  454.0576_dp,  432.9973_dp,  349.3149_dp,  298.1562_dp,    0.0000_dp, &
      &  432.3183_dp,  412.9408_dp,  334.9874_dp,  287.2026_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  567.3303_dp,  537.2058_dp,  423.1021_dp,  354.1221_dp,    0.0000_dp, &
      &  440.2733_dp,  419.6868_dp,  338.1165_dp,  288.2865_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  593.3574_dp,  563.0312_dp,  446.1184_dp,  375.1486_dp,    0.0000_dp, &
      &  430.3607_dp,  411.1576_dp,  333.6408_dp,  286.0771_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  757.7397_dp,  716.7957_dp,  561.9006_dp,  468.2521_dp,    0.0000_dp, &
      &  716.7957_dp,  678.9920_dp,  534.7248_dp,  447.3269_dp,    0.0000_dp, &
      &  561.9006_dp,  534.7248_dp,  427.9140_dp,  362.7514_dp,    0.0000_dp, &
      &  468.2521_dp,  447.3269_dp,  362.7514_dp,  310.8315_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.5334_dp,   39.0500_dp,   37.9122_dp,   34.5503_dp,   31.6020_dp, &
      &   67.9430_dp,   63.5189_dp,   61.4868_dp,   55.5643_dp,   50.3940_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   26.7862_dp,   25.4105_dp,   24.7833_dp,   22.8875_dp,   21.2154_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  708.4597_dp,  642.4190_dp,  611.5612_dp,  527.1851_dp,  455.9199_dp, &
      &  228.0814_dp,  210.3815_dp,  202.2053_dp,  179.0706_dp,  159.1376_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  393.7991_dp,  362.7434_dp,  348.3949_dp,  307.8814_dp,  273.0046_dp, &
      &  289.4713_dp,  267.5036_dp,  257.3551_dp,  228.5584_dp,  203.7227_dp, &
      &  184.6849_dp,  172.1767_dp,  166.4248_dp,  149.7680_dp,  135.2605_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  258.7146_dp,  240.5934_dp,  232.2513_dp,  208.2379_dp,  187.3808_dp, &
      &  216.4276_dp,  201.6758_dp,  194.8895_dp,  175.2745_dp,  158.2137_dp, &
      &  180.6996_dp,  168.7248_dp,  163.2184_dp,  147.2385_dp,  133.3214_dp, &
      &  135.4439_dp,  127.1409_dp,  123.3340_dp,  112.1331_dp,  102.3247_dp, &
      &  127.1521_dp,  119.4559_dp,  115.9286_dp,  105.5295_dp,   96.4179_dp, &
      &  171.4367_dp,  160.5876_dp,  155.6083_dp,  141.0407_dp,  128.3113_dp, &
      &  160.7910_dp,  150.6040_dp,  145.9274_dp,  132.2562_dp,  120.3190_dp, &
      &  130.0766_dp,  122.2030_dp,  118.5942_dp,  107.9636_dp,   98.6540_dp, &
      &  121.6598_dp,  114.3072_dp,  110.9372_dp,  101.0131_dp,   92.3260_dp, &
      &   98.3431_dp,   92.8021_dp,   90.2683_dp,   82.7170_dp,   76.0810_dp, &
      &  118.1509_dp,  111.2768_dp,  108.1299_dp,   98.8025_dp,   90.6184_dp, &
      &  110.0951_dp,  103.7070_dp,  100.7826_dp,   92.1154_dp,   84.5138_dp, &
      &  103.2113_dp,   97.2826_dp,   94.5696_dp,   86.5174_dp,   79.4522_dp, &
      &   90.6876_dp,   85.5479_dp,   83.1968_dp,   76.2100_dp,   70.0800_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   88.5408_dp,   83.7120_dp,   81.5058_dp,   74.9029_dp,   69.0939_dp, &
      &   79.1704_dp,   74.9289_dp,   72.9923_dp,   67.1811_dp,   62.0665_dp, &
      &   69.6046_dp,   65.9635_dp,   64.3022_dp,   59.3029_dp,   54.9005_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   66.4996_dp,   63.0806_dp,   61.5214_dp,   56.8151_dp,   52.6661_dp, &
      &   53.6243_dp,   51.0492_dp,   49.8778_dp,   46.3030_dp,   43.1432_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.8110_dp,   48.3258_dp,   47.1942_dp,   43.7556_dp,   40.7195_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1235 * maxRef * maxRef + 1 : 1254 * maxRef * maxRef) = [ &
      &  845.1348_dp,  767.0955_dp,  730.6587_dp,  630.8687_dp,  546.5096_dp, &
      &  334.7023_dp,  308.2470_dp,  296.0155_dp,  261.5400_dp,  231.9019_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  631.9434_dp,  580.4500_dp,  556.6241_dp,  489.7532_dp,  432.3765_dp, &
      &  518.6518_dp,  477.2848_dp,  458.1425_dp,  404.2763_dp,  358.0089_dp, &
      &  331.2828_dp,  307.5475_dp,  296.6147_dp,  265.2575_dp,  238.0549_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  575.4018_dp,  531.0794_dp,  510.6087_dp,  452.6192_dp,  402.6252_dp, &
      &  519.8424_dp,  480.3735_dp,  462.1521_dp,  410.4176_dp,  365.7711_dp, &
      &  400.7894_dp,  371.8779_dp,  358.5500_dp,  320.3924_dp,  287.3337_dp, &
      &  308.0840_dp,  287.3337_dp,  277.7927_dp,  250.1421_dp,  226.0504_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.7623_dp,  414.1403_dp,  399.5774_dp,  357.7428_dp,  321.4363_dp, &
      &  410.8960_dp,  382.1255_dp,  368.8796_dp,  330.7531_dp,  297.6403_dp, &
      &  381.2230_dp,  354.8687_dp,  342.7407_dp,  307.7570_dp,  277.3459_dp, &
      &  372.0785_dp,  346.1431_dp,  334.2022_dp,  299.8218_dp,  269.9679_dp, &
      &  302.4879_dp,  282.6684_dp,  273.5628_dp,  247.0572_dp,  223.9283_dp, &
      &  342.4169_dp,  319.9307_dp,  309.5992_dp,  279.5392_dp,  253.3166_dp, &
      &  332.1722_dp,  310.4238_dp,  300.4318_dp,  271.3488_dp,  245.9773_dp, &
      &  327.9460_dp,  306.3712_dp,  296.4572_dp,  267.6299_dp,  242.4940_dp, &
      &  303.0033_dp,  283.4898_dp,  274.5293_dp,  248.3782_dp,  225.5418_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.3255_dp,  263.8277_dp,  255.8013_dp,  232.2434_dp,  211.6318_dp, &
      &  278.2787_dp,  260.9497_dp,  253.0003_dp,  229.6756_dp,  209.2713_dp, &
      &  272.5881_dp,  255.6336_dp,  247.8563_dp,  225.0338_dp,  205.0687_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  227.5635_dp,  214.1723_dp,  208.0402_dp,  189.8824_dp,  173.9517_dp, &
      &  225.2381_dp,  211.9709_dp,  205.8954_dp,  187.9096_dp,  172.1318_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  184.5845_dp,  174.2639_dp,  169.5457_dp,  155.4627_dp,  143.0794_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1390.2912_dp, 1259.5964_dp, 1198.2993_dp, 1031.5350_dp,  891.2091_dp, &
      &  456.4613_dp,  422.0055_dp,  406.0845_dp,  360.9675_dp,  322.1273_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1123.0616_dp, 1026.6097_dp,  981.8206_dp,  857.4537_dp,  751.4576_dp, &
      &  896.4155_dp,  819.9926_dp,  784.4219_dp,  685.8003_dp,  601.8691_dp, &
      &  464.1193_dp,  430.8853_dp,  415.5727_dp,  371.7123_dp,  333.7206_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  923.0182_dp,  845.5447_dp,  809.6247_dp,  709.4418_dp,  623.8277_dp, &
      &  451.1340_dp,  419.1432_dp,  404.4096_dp,  362.1394_dp,  325.5023_dp, &
      &  443.0605_dp,  412.3939_dp,  398.2795_dp,  357.6129_dp,  322.2908_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  886.6318_dp,  813.8964_dp,  780.2017_dp,  685.8851_dp,  605.1321_dp, &
      &  435.1948_dp,  404.4378_dp,  390.2655_dp,  349.6106_dp,  314.3804_dp, &
      &  448.2770_dp,  417.3325_dp,  403.0919_dp,  362.0440_dp,  326.3851_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  809.4134_dp,  743.9253_dp,  713.6060_dp,  628.5419_dp,  555.6200_dp, &
      &  426.5933_dp,  396.4079_dp,  382.4997_dp,  342.6120_dp,  308.0520_dp, &
      &  410.3518_dp,  381.9818_dp,  368.9209_dp,  331.3093_dp,  298.6591_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  636.6561_dp,  585.4069_dp,  561.6694_dp,  495.0936_dp,  438.0628_dp, &
      &  423.3208_dp,  393.1215_dp,  379.2021_dp,  339.3449_dp,  304.8385_dp, &
      &  305.5108_dp,  285.6107_dp,  276.4674_dp,  249.8939_dp,  226.7483_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  693.7442_dp,  638.7587_dp,  613.3230_dp,  541.7184_dp,  480.2245_dp, &
      &  386.6184_dp,  360.1114_dp,  347.9151_dp,  312.7295_dp,  282.1503_dp, &
      &  259.5584_dp,  243.2984_dp,  235.8366_dp,  214.0118_dp,  194.9556_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.3629_dp,  500.6304_dp,  480.8588_dp,  425.1904_dp,  377.4064_dp, &
      &  356.9159_dp,  332.7595_dp,  321.6484_dp,  289.5301_dp,  261.5954_dp, &
      &  255.0100_dp,  238.9587_dp,  231.5914_dp,  210.0623_dp,  191.2703_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  572.8169_dp,  528.9965_dp,  508.7598_dp,  451.4360_dp,  402.0434_dp, &
      &  340.4265_dp,  317.3978_dp,  306.8041_dp,  276.1884_dp,  249.5667_dp, &
      &  302.4131_dp,  282.1750_dp,  272.8668_dp,  245.9388_dp,  222.5235_dp, &
      &  267.2082_dp,  250.0264_dp,  242.1351_dp,  219.1541_dp,  199.1182_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  592.0464_dp,  546.0473_dp,  524.7928_dp,  464.7381_dp,  413.0565_dp, &
      &  333.0301_dp,  310.4237_dp,  300.0232_dp,  269.9857_dp,  243.8751_dp, &
      &  280.8455_dp,  262.3696_dp,  253.8779_dp,  229.2387_dp,  207.7855_dp, &
      &  296.8872_dp,  276.6393_dp,  267.3225_dp,  240.4605_dp,  217.1392_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  453.2089_dp,  418.1730_dp,  401.9752_dp,  356.2413_dp,  316.9274_dp, &
      &  331.0063_dp,  307.8194_dp,  297.1406_dp,  266.4758_dp,  239.8928_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1254 * maxRef * maxRef + 1 : 1273 * maxRef * maxRef) = [ &
      &  461.1971_dp,  427.1025_dp,  411.3778_dp,  366.5785_dp,  327.8656_dp, &
      &  307.2635_dp,  286.9464_dp,  277.6083_dp,  250.5077_dp,  226.8951_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.0549_dp,  506.8838_dp,  487.8786_dp,  433.8635_dp,  387.2413_dp, &
      &  513.0315_dp,  475.1014_dp,  457.6031_dp,  407.7353_dp,  364.6366_dp, &
      &  401.7251_dp,  373.7047_dp,  360.8017_dp,  323.6722_dp,  291.4396_dp, &
      &  324.8229_dp,  303.5179_dp,  293.7290_dp,  265.2550_dp,  240.4178_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  477.1034_dp,  443.4892_dp,  428.0103_dp,  383.5189_dp,  344.9065_dp, &
      &  444.9048_dp,  414.0613_dp,  399.8646_dp,  358.9520_dp,  323.4073_dp, &
      &  422.6322_dp,  393.6595_dp,  380.3291_dp,  341.8407_dp,  308.3757_dp, &
      &  376.6108_dp,  351.5201_dp,  339.9859_dp,  306.5241_dp,  277.3717_dp, &
      &  330.1301_dp,  308.9411_dp,  299.2117_dp,  270.8052_dp,  245.9922_dp, &
      &  401.9415_dp,  375.3694_dp,  363.1576_dp,  327.6790_dp,  296.7505_dp, &
      &  394.8165_dp,  368.8196_dp,  356.8734_dp,  322.1450_dp,  291.8644_dp, &
      &  383.9466_dp,  358.8398_dp,  347.3049_dp,  313.7342_dp,  284.4506_dp, &
      &  369.1013_dp,  345.2176_dp,  334.2483_dp,  302.2666_dp,  274.3493_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  357.7871_dp,  335.1182_dp,  324.7135_dp,  294.2742_dp,  267.6724_dp, &
      &  357.9609_dp,  335.2667_dp,  324.8502_dp,  294.3804_dp,  267.7542_dp, &
      &  355.0652_dp,  332.5825_dp,  322.2635_dp,  292.0737_dp,  265.6909_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  310.5093_dp,  291.7179_dp,  283.1051_dp,  257.7196_dp,  235.4797_dp, &
      &  312.7994_dp,  293.8239_dp,  285.1261_dp,  259.5011_dp,  237.0549_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  267.9532_dp,  252.4408_dp,  245.3409_dp,  224.2655_dp,  205.7618_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1554.3698_dp, 1409.5386_dp, 1341.6152_dp, 1156.6447_dp, 1000.9319_dp, &
      &  542.8777_dp,  502.4986_dp,  483.8461_dp,  430.8811_dp,  385.2516_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1338.5665_dp, 1223.3073_dp, 1169.7495_dp, 1021.1992_dp,  894.7056_dp, &
      & 1091.6636_dp,  998.1919_dp,  954.6446_dp,  834.0979_dp,  731.6257_dp, &
      &  555.3233_dp,  515.7606_dp,  497.5316_dp,  445.2988_dp,  400.0650_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1161.3674_dp, 1065.4342_dp, 1020.9627_dp,  896.6822_dp,  790.3938_dp, &
      &  818.4130_dp,  754.1146_dp,  724.2934_dp,  640.4740_dp,  568.6159_dp, &
      &  529.9186_dp,  493.8129_dp,  477.2036_dp,  429.2279_dp,  387.5181_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1037.2812_dp,  953.9322_dp,  915.3455_dp,  806.9968_dp,  714.0953_dp, &
      &  596.8919_dp,  554.4107_dp,  534.8274_dp,  478.7355_dp,  430.1818_dp, &
      &  533.0732_dp,  497.2424_dp,  480.7668_dp,  433.0658_dp,  391.5532_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  941.7792_dp,  867.5475_dp,  833.2050_dp,  736.4802_dp,  653.4187_dp, &
      &  602.9692_dp,  559.9850_dp,  540.1792_dp,  483.4478_dp,  434.3422_dp, &
      &  515.7374_dp,  480.9564_dp,  464.9556_dp,  418.6918_dp,  378.4693_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  721.2906_dp,  666.4197_dp,  641.0643_dp,  569.2766_dp,  507.4919_dp, &
      &  560.5104_dp,  521.1563_dp,  503.0236_dp,  450.9840_dp,  405.9028_dp, &
      &  452.4616_dp,  423.1570_dp,  409.6963_dp,  370.5143_dp,  336.3592_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  807.2358_dp,  744.9993_dp,  716.2239_dp,  634.9209_dp,  565.0031_dp, &
      &  634.7831_dp,  587.9355_dp,  566.2902_dp,  504.7650_dp,  451.7317_dp, &
      &  396.8920_dp,  372.0410_dp,  360.6378_dp,  327.2637_dp,  298.1122_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  609.6152_dp,  564.4589_dp,  543.6104_dp,  484.3486_dp,  433.2544_dp, &
      &  484.8855_dp,  452.1682_dp,  437.1184_dp,  393.6270_dp,  355.8286_dp, &
      &  374.1870_dp,  351.0591_dp,  340.4508_dp,  309.3410_dp,  282.1480_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  666.8954_dp,  617.3791_dp,  594.5277_dp,  529.5371_dp,  473.4591_dp, &
      &  453.1794_dp,  422.7438_dp,  408.7434_dp,  368.2710_dp,  333.0973_dp, &
      &  383.7851_dp,  359.1380_dp,  347.8166_dp,  314.8522_dp,  286.1251_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  616.7522_dp,  571.5602_dp,  550.7159_dp,  491.3041_dp,  439.9855_dp, &
      &  420.1268_dp,  392.1471_dp,  379.2791_dp,  342.0404_dp,  309.6665_dp, &
      &  401.8433_dp,  375.7345_dp,  363.7391_dp,  328.8698_dp,  298.4950_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  514.4477_dp,  476.5966_dp,  459.1219_dp,  409.4318_dp,  366.5943_dp, &
      &  406.9029_dp,  379.6335_dp,  367.0907_dp,  330.8354_dp,  299.3318_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  541.5490_dp,  502.5513_dp,  484.5759_dp,  433.1992_dp,  388.7621_dp, &
      &  400.3474_dp,  374.3430_dp,  362.3966_dp,  327.6519_dp,  297.3658_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1273 * maxRef * maxRef + 1 : 1292 * maxRef * maxRef) = [ &
      &  685.6138_dp,  634.1195_dp,  610.3382_dp,  542.8413_dp,  484.6532_dp, &
      &  651.1403_dp,  603.0965_dp,  580.9260_dp,  517.8029_dp,  463.2993_dp, &
      &  517.1479_dp,  481.4185_dp,  464.9677_dp,  417.6093_dp,  376.5051_dp, &
      &  435.6252_dp,  407.2275_dp,  394.1807_dp,  356.2359_dp,  323.1509_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  627.5677_dp,  582.7693_dp,  562.1276_dp,  502.9891_dp,  451.7578_dp, &
      &  582.7693_dp,  542.0439_dp,  523.2910_dp,  469.3751_dp,  422.5950_dp, &
      &  562.1276_dp,  523.2910_dp,  505.4148_dp,  453.9195_dp,  409.2008_dp, &
      &  502.9891_dp,  469.3751_dp,  453.9195_dp,  409.1453_dp,  370.1674_dp, &
      &  451.7578_dp,  422.5950_dp,  409.2008_dp,  370.1674_dp,  336.1020_dp, &
      &   37.7681_dp,   37.1186_dp,   36.2277_dp,   35.0939_dp,    0.0000_dp, &
      &   60.9470_dp,   59.7796_dp,   58.1962_dp,   56.1946_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   24.8583_dp,   24.5068_dp,   24.0141_dp,   23.3792_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  586.6888_dp,  569.5020_dp,  546.8547_dp,  518.6427_dp,    0.0000_dp, &
      &  197.7698_dp,  193.0801_dp,  186.8353_dp,  179.0274_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  340.3030_dp,  332.0665_dp,  321.1170_dp,  307.4421_dp,    0.0000_dp, &
      &  252.1649_dp,  246.3580_dp,  238.6037_dp,  228.8894_dp,    0.0000_dp, &
      &  164.5348_dp,  161.2234_dp,  156.7541_dp,  151.1215_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  229.0421_dp,  224.2438_dp,  217.7905_dp,  209.6745_dp,    0.0000_dp, &
      &  192.5542_dp,  188.6577_dp,  183.4000_dp,  176.7743_dp,    0.0000_dp, &
      &  161.5570_dp,  158.4037_dp,  154.1335_dp,  148.7395_dp,    0.0000_dp, &
      &  122.6873_dp,  120.5099_dp,  117.5338_dp,  113.7542_dp,    0.0000_dp, &
      &  115.4018_dp,  113.3867_dp,  110.6276_dp,  107.1200_dp,    0.0000_dp, &
      &  154.4851_dp,  151.6334_dp,  147.7519_dp,  142.8348_dp,    0.0000_dp, &
      &  144.8462_dp,  142.1737_dp,  138.5349_dp,  133.9240_dp,    0.0000_dp, &
      &  118.0337_dp,  115.9742_dp,  113.1548_dp,  109.5710_dp,    0.0000_dp, &
      &  110.4089_dp,  108.4883_dp,  105.8584_dp,  102.5148_dp,    0.0000_dp, &
      &   90.1797_dp,   88.7434_dp,   86.7574_dp,   84.2177_dp,    0.0000_dp, &
      &  107.8436_dp,  106.0538_dp,  103.5908_dp,  100.4500_dp,    0.0000_dp, &
      &  100.5182_dp,   98.8583_dp,   96.5724_dp,   93.6562_dp,    0.0000_dp, &
      &   94.3654_dp,   92.8262_dp,   90.7043_dp,   87.9958_dp,    0.0000_dp, &
      &   83.0577_dp,   81.7279_dp,   79.8912_dp,   77.5440_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.5358_dp,   80.2899_dp,   78.5600_dp,   76.3424_dp,    0.0000_dp, &
      &   73.0718_dp,   71.9816_dp,   70.4634_dp,   68.5141_dp,    0.0000_dp, &
      &   64.4291_dp,   63.4972_dp,   62.1954_dp,   60.5209_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   61.6912_dp,   60.8174_dp,   59.5937_dp,   58.0176_dp,    0.0000_dp, &
      &   50.1473_dp,   49.4970_dp,   48.5766_dp,   47.3835_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   47.4073_dp,   46.7775_dp,   45.8894_dp,   44.7408_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  701.6903_dp,  681.3650_dp,  654.5716_dp,  621.1943_dp,    0.0000_dp, &
      &  288.9888_dp,  281.9923_dp,  272.6893_dp,  261.0668_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  541.9826_dp,  528.3545_dp,  510.2824_dp,  487.7418_dp,    0.0000_dp, &
      &  446.9230_dp,  435.9923_dp,  421.4626_dp,  403.3092_dp,    0.0000_dp, &
      &  292.0246_dp,  285.7251_dp,  277.2781_dp,  266.6745_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  499.7292_dp,  487.9847_dp,  472.3352_dp,  452.7605_dp,    0.0000_dp, &
      &  452.8494_dp,  442.3946_dp,  428.4437_dp,  410.9788_dp,    0.0000_dp, &
      &  352.7849_dp,  345.1303_dp,  334.8650_dp,  321.9748_dp,    0.0000_dp, &
      &  274.7333_dp,  269.2400_dp,  261.8224_dp,  252.4713_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  393.6899_dp,  385.3102_dp,  374.0566_dp,  359.9159_dp,    0.0000_dp, &
      &  363.7846_dp,  356.1681_dp,  345.9236_dp,  333.0384_dp,    0.0000_dp, &
      &  338.3189_dp,  331.3449_dp,  321.9519_dp,  310.1284_dp,    0.0000_dp, &
      &  329.6625_dp,  322.8072_dp,  313.5794_dp,  301.9669_dp,    0.0000_dp, &
      &  271.0387_dp,  265.8053_dp,  258.7140_dp,  249.7552_dp,    0.0000_dp, &
      &  306.6912_dp,  300.7558_dp,  292.7145_dp,  282.5564_dp,    0.0000_dp, &
      &  297.6598_dp,  291.9225_dp,  284.1462_dp,  274.3202_dp,    0.0000_dp, &
      &  293.6168_dp,  287.9266_dp,  280.2178_dp,  270.4800_dp,    0.0000_dp, &
      &  272.2813_dp,  267.1401_dp,  260.1578_dp,  251.3243_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.2418_dp,  249.6469_dp,  243.3785_dp,  235.4268_dp,    0.0000_dp, &
      &  251.4344_dp,  246.8844_dp,  240.6781_dp,  232.8057_dp,    0.0000_dp, &
      &  246.3350_dp,  241.8844_dp,  235.8128_dp,  228.1108_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  207.4043_dp,  203.9103_dp,  199.1082_dp,  192.9893_dp,    0.0000_dp, &
      &  205.2512_dp,  201.7898_dp,  197.0330_dp,  190.9724_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1292 * maxRef * maxRef + 1 : 1311 * maxRef * maxRef) = [ &
      &  169.4533_dp,  166.7796_dp,  163.0783_dp,  158.3420_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1147.6700_dp, 1113.9369_dp, 1069.4211_dp, 1013.8323_dp,    0.0000_dp, &
      &  397.8416_dp,  388.7844_dp,  376.6755_dp,  361.4928_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  950.6543_dp,  925.3266_dp,  891.8232_dp,  850.0618_dp,    0.0000_dp, &
      &  759.7988_dp,  739.8380_dp,  713.3777_dp,  680.3184_dp,    0.0000_dp, &
      &  409.0173_dp,  400.2337_dp,  388.4468_dp,  373.6423_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  785.8323_dp,  765.4303_dp,  738.4121_dp,  704.7245_dp,    0.0000_dp, &
      &  398.3152_dp,  389.8645_dp,  378.5124_dp,  364.2456_dp,    0.0000_dp, &
      &  393.0115_dp,  384.9081_dp,  373.9963_dp,  360.2627_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  758.9406_dp,  739.7751_dp,  714.3497_dp,  682.6170_dp,    0.0000_dp, &
      &  384.4487_dp,  376.3336_dp,  365.4257_dp,  351.7089_dp,    0.0000_dp, &
      &  397.8333_dp,  389.6578_dp,  378.6459_dp,  364.7838_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.0683_dp,  677.8014_dp,  654.8718_dp,  626.2392_dp,    0.0000_dp, &
      &  376.7544_dp,  368.7908_dp,  358.0883_dp,  344.6316_dp,    0.0000_dp, &
      &  364.0189_dp,  356.5349_dp,  346.4532_dp,  333.7597_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  547.1981_dp,  533.7253_dp,  515.8152_dp,  493.4310_dp,    0.0000_dp, &
      &  373.2591_dp,  365.2940_dp,  354.5977_dp,  341.1548_dp,    0.0000_dp, &
      &  273.8552_dp,  268.6291_dp,  261.5413_dp,  252.5817_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  598.5168_dp,  584.0079_dp,  564.7110_dp,  540.5946_dp,    0.0000_dp, &
      &  343.5205_dp,  336.5178_dp,  327.0807_dp,  315.1977_dp,    0.0000_dp, &
      &  234.1750_dp,  229.9158_dp,  224.1126_dp,  216.7567_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  469.4569_dp,  458.2135_dp,  443.2412_dp,  424.5124_dp,    0.0000_dp, &
      &  317.8624_dp,  311.4859_dp,  302.8803_dp,  292.0345_dp,    0.0000_dp, &
      &  229.8877_dp,  225.6812_dp,  219.9538_dp,  212.6969_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  498.0472_dp,  486.4645_dp,  471.0186_dp,  451.6893_dp,    0.0000_dp, &
      &  303.1856_dp,  297.1110_dp,  288.9115_dp,  278.5760_dp,    0.0000_dp, &
      &  269.8010_dp,  264.4748_dp,  257.2754_dp,  248.1924_dp,    0.0000_dp, &
      &  240.0396_dp,  235.5276_dp,  229.4010_dp,  221.6512_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  513.0585_dp,  500.9011_dp,  484.7111_dp,  464.4665_dp,    0.0000_dp, &
      &  296.4064_dp,  290.4433_dp,  282.3973_dp,  272.2578_dp,    0.0000_dp, &
      &  251.3164_dp,  246.4555_dp,  239.8735_dp,  231.5614_dp,    0.0000_dp, &
      &  263.9496_dp,  258.6194_dp,  251.4301_dp,  242.3721_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  393.0249_dp,  383.8010_dp,  371.5028_dp,  356.1095_dp,    0.0000_dp, &
      &  292.8589_dp,  286.7436_dp,  278.5192_dp,  268.1748_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  403.8621_dp,  394.8434_dp,  382.7819_dp,  367.6637_dp,    0.0000_dp, &
      &  274.7816_dp,  269.4208_dp,  262.1684_dp,  253.0149_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  478.3993_dp,  467.5119_dp,  452.9662_dp,  434.7417_dp,    0.0000_dp, &
      &  449.3006_dp,  439.2692_dp,  425.8475_dp,  409.0175_dp,    0.0000_dp, &
      &  355.8474_dp,  348.4425_dp,  338.4772_dp,  325.9378_dp,    0.0000_dp, &
      &  290.9600_dp,  285.3363_dp,  277.7186_dp,  268.0968_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  421.8616_dp,  412.9661_dp,  401.0098_dp,  385.9785_dp,    0.0000_dp, &
      &  394.5822_dp,  386.4260_dp,  375.4438_dp,  361.6218_dp,    0.0000_dp, &
      &  375.6047_dp,  367.9463_dp,  357.6221_dp,  344.6189_dp,    0.0000_dp, &
      &  336.4289_dp,  329.8047_dp,  320.8460_dp,  309.5410_dp,    0.0000_dp, &
      &  296.8145_dp,  291.2302_dp,  283.6451_dp,  274.0484_dp,    0.0000_dp, &
      &  359.5564_dp,  352.5427_dp,  343.0487_dp,  331.0618_dp,    0.0000_dp, &
      &  353.4245_dp,  346.5656_dp,  337.2763_dp,  325.5439_dp,    0.0000_dp, &
      &  344.1036_dp,  337.4828_dp,  328.5083_dp,  317.1679_dp,    0.0000_dp, &
      &  331.3946_dp,  325.0999_dp,  316.5569_dp,  305.7538_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  322.3608_dp,  316.3984_dp,  308.2846_dp,  298.0072_dp,    0.0000_dp, &
      &  322.4812_dp,  316.5125_dp,  308.3906_dp,  298.1033_dp,    0.0000_dp, &
      &  319.9354_dp,  314.0234_dp,  305.9773_dp,  295.7851_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.8050_dp,  276.8856_dp,  270.1506_dp,  261.5889_dp,    0.0000_dp, &
      &  283.7761_dp,  278.8076_dp,  272.0076_dp,  263.3649_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  244.7902_dp,  240.7515_dp,  235.1882_dp,  228.0904_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1311 * maxRef * maxRef + 1 : 1330 * maxRef * maxRef) = [ &
      & 1286.1750_dp, 1248.8109_dp, 1199.4646_dp, 1137.8099_dp,    0.0000_dp, &
      &  474.5641_dp,  463.9632_dp,  449.7669_dp,  431.9480_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1132.1713_dp, 1101.9644_dp, 1061.9947_dp, 1012.1518_dp,    0.0000_dp, &
      &  924.1369_dp,  899.7796_dp,  867.4828_dp,  827.1106_dp,    0.0000_dp, &
      &  489.8228_dp,  479.3863_dp,  465.3683_dp,  447.7500_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  992.4001_dp,  967.1672_dp,  933.6956_dp,  891.9140_dp,    0.0000_dp, &
      &  707.0933_dp,  690.2435_dp,  667.7703_dp,  639.6032_dp,    0.0000_dp, &
      &  471.4064_dp,  461.8766_dp,  449.0205_dp,  432.8219_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  892.0877_dp,  870.1334_dp,  840.9533_dp,  804.4937_dp,    0.0000_dp, &
      &  526.5426_dp,  515.3541_dp,  500.3168_dp,  481.4071_dp,    0.0000_dp, &
      &  475.3821_dp,  465.9289_dp,  453.1570_dp,  437.0500_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  813.4345_dp,  793.8771_dp,  767.8415_dp,  735.2812_dp,    0.0000_dp, &
      &  531.7424_dp,  520.4182_dp,  505.2035_dp,  486.0786_dp,    0.0000_dp, &
      &  459.5750_dp,  450.4159_dp,  438.0410_dp,  422.4320_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  627.6849_dp,  613.2470_dp,  593.9624_dp,  569.7978_dp,    0.0000_dp, &
      &  495.6948_dp,  485.3374_dp,  471.3990_dp,  453.8578_dp,    0.0000_dp, &
      &  406.0312_dp,  398.3318_dp,  387.8818_dp,  374.6663_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  700.5163_dp,  684.1292_dp,  662.2699_dp,  634.8991_dp,    0.0000_dp, &
      &  555.7794_dp,  543.4822_dp,  527.0000_dp,  506.2976_dp,    0.0000_dp, &
      &  358.1519_dp,  351.6390_dp,  342.7635_dp,  331.5119_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.3804_dp,  521.5071_dp,  505.6089_dp,  485.6592_dp,    0.0000_dp, &
      &  431.9865_dp,  423.3751_dp,  411.7430_dp,  397.0742_dp,    0.0000_dp, &
      &  338.3541_dp,  332.2995_dp,  324.0357_dp,  313.5500_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  583.3346_dp,  570.2790_dp,  552.8128_dp,  530.9113_dp,    0.0000_dp, &
      &  404.0384_dp,  396.0359_dp,  385.2196_dp,  371.5740_dp,    0.0000_dp, &
      &  344.8082_dp,  338.3472_dp,  329.5690_dp,  318.4602_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  540.9125_dp,  528.9954_dp,  513.0345_dp,  493.0084_dp,    0.0000_dp, &
      &  375.0995_dp,  367.7493_dp,  357.8055_dp,  345.2533_dp,    0.0000_dp, &
      &  360.3420_dp,  353.4857_dp,  344.1857_dp,  332.4291_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  450.6427_dp,  440.7041_dp,  427.3871_dp,  410.6685_dp,    0.0000_dp, &
      &  362.8740_dp,  355.7080_dp,  346.0216_dp,  333.8012_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  476.5783_dp,  466.2941_dp,  452.4998_dp,  435.1777_dp,    0.0000_dp, &
      &  359.0555_dp,  352.2132_dp,  342.9362_dp,  331.2119_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  598.2891_dp,  584.7110_dp,  566.5642_dp,  543.8198_dp,    0.0000_dp, &
      &  570.3015_dp,  557.6272_dp,  540.6624_dp,  519.3820_dp,    0.0000_dp, &
      &  458.7676_dp,  449.3448_dp,  436.6520_dp,  420.6714_dp,    0.0000_dp, &
      &  390.5107_dp,  383.0280_dp,  372.8876_dp,  360.0757_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  553.3592_dp,  541.5209_dp,  525.6310_dp,  505.6697_dp,    0.0000_dp, &
      &  515.9467_dp,  505.1900_dp,  490.7203_dp,  472.5189_dp,    0.0000_dp, &
      &  498.7395_dp,  488.4839_dp,  474.6720_dp,  457.2860_dp,    0.0000_dp, &
      &  448.9769_dp,  440.1085_dp,  428.1222_dp,  413.0024_dp,    0.0000_dp, &
      &  405.6849_dp,  398.0011_dp,  387.5752_dp,  374.3929_dp,    0.0000_dp, &
      &  492.9379_dp,  483.0596_dp,  469.7255_dp,  452.9187_dp,    0.0000_dp, &
      &  483.0596_dp,  473.4703_dp,  460.5150_dp,  444.1770_dp,    0.0000_dp, &
      &  469.7255_dp,  460.5150_dp,  448.0573_dp,  432.3357_dp,    0.0000_dp, &
      &  452.9187_dp,  444.1770_dp,  432.3357_dp,  417.3790_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   35.4841_dp,   35.4861_dp,   35.2461_dp,    0.0000_dp,    0.0000_dp, &
      &   56.7427_dp,   56.7156_dp,   56.2769_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   23.6911_dp,   23.7100_dp,   23.5851_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  521.2511_dp,  519.4518_dp,  512.7164_dp,    0.0000_dp,    0.0000_dp, &
      &  180.3056_dp,  179.9812_dp,  178.1745_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  309.5439_dp,  308.9426_dp,  305.7629_dp,    0.0000_dp,    0.0000_dp, &
      &  230.6309_dp,  230.2641_dp,  228.0340_dp,    0.0000_dp,    0.0000_dp, &
      &  152.4956_dp,  152.3806_dp,  151.1275_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1330 * maxRef * maxRef + 1 : 1349 * maxRef * maxRef) = [ &
      &  211.4898_dp,  211.2799_dp,  209.4559_dp,    0.0000_dp,    0.0000_dp, &
      &  178.3919_dp,  178.2509_dp,  176.7765_dp,    0.0000_dp,    0.0000_dp, &
      &  150.1754_dp,  150.0870_dp,  148.8998_dp,    0.0000_dp,    0.0000_dp, &
      &  114.9752_dp,  114.9640_dp,  114.1553_dp,    0.0000_dp,    0.0000_dp, &
      &  108.2922_dp,  108.2898_dp,  107.5434_dp,    0.0000_dp,    0.0000_dp, &
      &  144.3032_dp,  144.2600_dp,  143.1947_dp,    0.0000_dp,    0.0000_dp, &
      &  135.3122_dp,  135.2720_dp,  134.2742_dp,    0.0000_dp,    0.0000_dp, &
      &  110.7747_dp,  110.7711_dp,  110.0087_dp,    0.0000_dp,    0.0000_dp, &
      &  103.6503_dp,  103.6477_dp,  102.9373_dp,    0.0000_dp,    0.0000_dp, &
      &   85.2343_dp,   85.2657_dp,   84.7425_dp,    0.0000_dp,    0.0000_dp, &
      &  101.6124_dp,  101.6315_dp,  100.9745_dp,    0.0000_dp,    0.0000_dp, &
      &   94.7508_dp,   94.7705_dp,   94.1622_dp,    0.0000_dp,    0.0000_dp, &
      &   89.0359_dp,   89.0582_dp,   88.4953_dp,    0.0000_dp,    0.0000_dp, &
      &   78.4819_dp,   78.5067_dp,   78.0224_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   77.2970_dp,   77.3365_dp,   76.8863_dp,    0.0000_dp,    0.0000_dp, &
      &   69.3901_dp,   69.4317_dp,   69.0399_dp,    0.0000_dp,    0.0000_dp, &
      &   61.3159_dp,   61.3585_dp,   61.0258_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   58.7881_dp,   58.8332_dp,   58.5223_dp,    0.0000_dp,    0.0000_dp, &
      &   48.0535_dp,   48.1037_dp,   47.8771_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   45.3613_dp,   45.4044_dp,   45.1836_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  624.4076_dp,  622.3099_dp,  614.3523_dp,    0.0000_dp,    0.0000_dp, &
      &  262.8846_dp,  262.3697_dp,  259.6698_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  490.8760_dp,  489.7820_dp,  484.5038_dp,    0.0000_dp,    0.0000_dp, &
      &  406.0728_dp,  405.2529_dp,  401.0304_dp,    0.0000_dp,    0.0000_dp, &
      &  268.8595_dp,  268.5418_dp,  266.1377_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  456.0400_dp,  455.2477_dp,  450.7267_dp,    0.0000_dp,    0.0000_dp, &
      &  414.0544_dp,  413.3856_dp,  409.3685_dp,    0.0000_dp,    0.0000_dp, &
      &  324.6278_dp,  324.2349_dp,  321.3129_dp,    0.0000_dp,    0.0000_dp, &
      &  254.7805_dp,  254.5963_dp,  252.5189_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  362.9518_dp,  362.5561_dp,  359.3640_dp,    0.0000_dp,    0.0000_dp, &
      &  335.9240_dp,  335.5914_dp,  332.6960_dp,    0.0000_dp,    0.0000_dp, &
      &  312.8750_dp,  312.5939_dp,  309.9478_dp,    0.0000_dp,    0.0000_dp, &
      &  304.6285_dp,  304.3388_dp,  301.7357_dp,    0.0000_dp,    0.0000_dp, &
      &  252.1604_dp,  252.0272_dp,  250.0579_dp,    0.0000_dp,    0.0000_dp, &
      &  285.2779_dp,  285.1239_dp,  282.8902_dp,    0.0000_dp,    0.0000_dp, &
      &  276.9809_dp,  276.8372_dp,  274.6795_dp,    0.0000_dp,    0.0000_dp, &
      &  273.0920_dp,  272.9417_dp,  270.8005_dp,    0.0000_dp,    0.0000_dp, &
      &  253.8269_dp,  253.7228_dp,  251.7949_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  237.9018_dp,  237.8571_dp,  236.1456_dp,    0.0000_dp,    0.0000_dp, &
      &  235.2519_dp,  235.2059_dp,  233.5109_dp,    0.0000_dp,    0.0000_dp, &
      &  230.5137_dp,  230.4702_dp,  228.8128_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.1861_dp,  195.2131_dp,  193.9269_dp,    0.0000_dp,    0.0000_dp, &
      &  193.1449_dp,  193.1704_dp,  191.8960_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  160.2674_dp,  160.3331_dp,  159.3608_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1019.3121_dp, 1015.7738_dp, 1002.5559_dp,    0.0000_dp,    0.0000_dp, &
      &  364.4090_dp,  363.8482_dp,  360.3799_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  855.1882_dp,  852.9036_dp,  843.0540_dp,    0.0000_dp,    0.0000_dp, &
      &  684.7073_dp,  682.9621_dp,  675.2096_dp,    0.0000_dp,    0.0000_dp, &
      &  376.8093_dp,  376.3706_dp,  373.0245_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  709.1303_dp,  707.3769_dp,  699.4586_dp,    0.0000_dp,    0.0000_dp, &
      &  367.3908_dp,  366.9900_dp,  363.7755_dp,    0.0000_dp,    0.0000_dp, &
      &  363.4777_dp,  363.1446_dp,  360.0716_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  687.1235_dp,  685.5687_dp,  678.1485_dp,    0.0000_dp,    0.0000_dp, &
      &  354.7793_dp,  354.4029_dp,  351.3179_dp,    0.0000_dp,    0.0000_dp, &
      &  368.0557_dp,  367.7252_dp,  364.6262_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  630.4901_dp,  629.1390_dp,  622.4635_dp,    0.0000_dp,    0.0000_dp, &
      &  347.6374_dp,  347.2648_dp,  344.2372_dp,    0.0000_dp,    0.0000_dp, &
      &  336.7719_dp,  336.4671_dp,  333.6308_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1349 * maxRef * maxRef + 1 : 1368 * maxRef * maxRef) = [ &
      &  496.9143_dp,  495.8801_dp,  490.6786_dp,    0.0000_dp,    0.0000_dp, &
      &  344.0982_dp,  343.7087_dp,  340.6774_dp,    0.0000_dp,    0.0000_dp, &
      &  255.1077_dp,  254.9795_dp,  253.0195_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  544.4156_dp,  543.3438_dp,  537.7470_dp,    0.0000_dp,    0.0000_dp, &
      &  318.0485_dp,  317.7750_dp,  315.1229_dp,    0.0000_dp,    0.0000_dp, &
      &  219.0427_dp,  218.9858_dp,  217.3993_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  427.6439_dp,  426.8361_dp,  422.5068_dp,    0.0000_dp,    0.0000_dp, &
      &  294.7348_dp,  294.5079_dp,  292.0979_dp,    0.0000_dp,    0.0000_dp, &
      &  214.9236_dp,  214.8607_dp,  213.2923_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  455.0836_dp,  454.3170_dp,  449.8664_dp,    0.0000_dp,    0.0000_dp, &
      &  281.1633_dp,  280.9482_dp,  278.6529_dp,    0.0000_dp,    0.0000_dp, &
      &  250.5626_dp,  250.3896_dp,  248.3821_dp,    0.0000_dp,    0.0000_dp, &
      &  223.8964_dp,  223.7999_dp,  222.1108_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  467.8493_dp,  467.0009_dp,  462.3208_dp,    0.0000_dp,    0.0000_dp, &
      &  274.7751_dp,  274.5580_dp,  272.3038_dp,    0.0000_dp,    0.0000_dp, &
      &  233.8236_dp,  233.6876_dp,  231.8603_dp,    0.0000_dp,    0.0000_dp, &
      &  244.6310_dp,  244.4292_dp,  242.4149_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  358.8167_dp,  358.1870_dp,  354.6423_dp,    0.0000_dp,    0.0000_dp, &
      &  270.5526_dp,  270.2783_dp,  267.9569_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.5936_dp,  370.0674_dp,  366.6162_dp,    0.0000_dp,    0.0000_dp, &
      &  255.4289_dp,  255.2713_dp,  253.2520_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  438.1102_dp,  437.4417_dp,  433.2665_dp,    0.0000_dp,    0.0000_dp, &
      &  412.2793_dp,  411.7018_dp,  407.8626_dp,    0.0000_dp,    0.0000_dp, &
      &  328.8076_dp,  328.4905_dp,  325.6786_dp,    0.0000_dp,    0.0000_dp, &
      &  270.6774_dp,  270.5297_dp,  268.4136_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.2997_dp,  388.8946_dp,  385.5109_dp,    0.0000_dp,    0.0000_dp, &
      &  364.8243_dp,  364.4882_dp,  361.3931_dp,    0.0000_dp,    0.0000_dp, &
      &  347.7286_dp,  347.4358_dp,  344.5346_dp,    0.0000_dp,    0.0000_dp, &
      &  312.4631_dp,  312.2619_dp,  309.7636_dp,    0.0000_dp,    0.0000_dp, &
      &  276.7786_dp,  276.6685_dp,  274.5751_dp,    0.0000_dp,    0.0000_dp, &
      &  334.2234_dp,  334.0263_dp,  331.3844_dp,    0.0000_dp,    0.0000_dp, &
      &  328.6756_dp,  328.4909_dp,  325.9092_dp,    0.0000_dp,    0.0000_dp, &
      &  320.2529_dp,  320.0878_dp,  317.5986_dp,    0.0000_dp,    0.0000_dp, &
      &  308.7739_dp,  308.6359_dp,  306.2735_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  301.0545_dp,  300.9617_dp,  298.7331_dp,    0.0000_dp,    0.0000_dp, &
      &  301.1506_dp,  301.0567_dp,  298.8255_dp,    0.0000_dp,    0.0000_dp, &
      &  298.8156_dp,  298.7247_dp,  296.5153_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  264.4516_dp,  264.4446_dp,  262.6226_dp,    0.0000_dp,    0.0000_dp, &
      &  266.2386_dp,  266.2278_dp,  264.3868_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  230.7426_dp,  230.7947_dp,  229.3135_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1144.1913_dp, 1140.3397_dp, 1125.7134_dp,    0.0000_dp,    0.0000_dp, &
      &  435.5555_dp,  434.9391_dp,  430.8890_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1018.3413_dp, 1015.6122_dp, 1003.8665_dp,    0.0000_dp,    0.0000_dp, &
      &  832.5060_dp,  830.3661_dp,  820.9055_dp,    0.0000_dp,    0.0000_dp, &
      &  451.6295_dp,  451.1258_dp,  447.1555_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  897.8043_dp,  895.7304_dp,  885.9576_dp,    0.0000_dp,    0.0000_dp, &
      &  644.4494_dp,  643.2697_dp,  636.7834_dp,    0.0000_dp,    0.0000_dp, &
      &  436.8022_dp,  436.4520_dp,  432.8478_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  810.1028_dp,  808.4252_dp,  799.9471_dp,    0.0000_dp,    0.0000_dp, &
      &  485.6176_dp,  485.0856_dp,  480.8308_dp,    0.0000_dp,    0.0000_dp, &
      &  441.1513_dp,  440.8392_dp,  437.2712_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  740.6211_dp,  739.2097_dp,  731.6741_dp,    0.0000_dp,    0.0000_dp, &
      &  490.3206_dp,  489.7750_dp,  485.4685_dp,    0.0000_dp,    0.0000_dp, &
      &  426.4202_dp,  426.1118_dp,  422.6552_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  574.2932_dp,  573.3700_dp,  567.8341_dp,    0.0000_dp,    0.0000_dp, &
      &  457.9302_dp,  457.4717_dp,  453.5417_dp,    0.0000_dp,    0.0000_dp, &
      &  378.4312_dp,  378.2579_dp,  375.3725_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1368 * maxRef * maxRef + 1 : 1387 * maxRef * maxRef) = [ &
      &  639.7465_dp,  638.6462_dp,  632.3502_dp,    0.0000_dp,    0.0000_dp, &
      &  510.5682_dp,  509.8781_dp,  505.1830_dp,    0.0000_dp,    0.0000_dp, &
      &  335.0059_dp,  334.9230_dp,  332.4971_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  489.6956_dp,  489.0100_dp,  484.4740_dp,    0.0000_dp,    0.0000_dp, &
      &  400.8319_dp,  400.5365_dp,  397.2874_dp,    0.0000_dp,    0.0000_dp, &
      &  316.9123_dp,  316.8578_dp,  314.6082_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  535.2189_dp,  534.4511_dp,  529.4579_dp,    0.0000_dp,    0.0000_dp, &
      &  375.1303_dp,  374.8656_dp,  371.8493_dp,    0.0000_dp,    0.0000_dp, &
      &  321.7225_dp,  321.5899_dp,  319.1735_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  497.0959_dp,  496.4313_dp,  491.8812_dp,    0.0000_dp,    0.0000_dp, &
      &  348.6074_dp,  348.3802_dp,  345.6140_dp,    0.0000_dp,    0.0000_dp, &
      &  335.7605_dp,  335.5949_dp,  333.0245_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  414.1512_dp,  413.5913_dp,  409.7995_dp,    0.0000_dp,    0.0000_dp, &
      &  337.0131_dp,  336.7771_dp,  334.0775_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  438.8861_dp,  438.3534_dp,  434.4358_dp,    0.0000_dp,    0.0000_dp, &
      &  334.4917_dp,  334.3241_dp,  331.7567_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.1267_dp,  547.2913_dp,  542.0891_dp,    0.0000_dp,    0.0000_dp, &
      &  523.6131_dp,  522.8863_dp,  518.0408_dp,    0.0000_dp,    0.0000_dp, &
      &  424.4718_dp,  424.0862_dp,  420.5155_dp,    0.0000_dp,    0.0000_dp, &
      &  363.5974_dp,  363.4077_dp,  360.5967_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  509.9698_dp,  509.3849_dp,  504.8763_dp,    0.0000_dp,    0.0000_dp, &
      &  476.6811_dp,  476.2087_dp,  472.1243_dp,    0.0000_dp,    0.0000_dp, &
      &  461.3868_dp,  460.9671_dp,  457.0793_dp,    0.0000_dp,    0.0000_dp, &
      &  416.8945_dp,  416.6107_dp,  413.2653_dp,    0.0000_dp,    0.0000_dp, &
      &  378.0963_dp,  377.9250_dp,  375.0423_dp,    0.0000_dp,    0.0000_dp, &
      &  457.1057_dp,  456.7589_dp,  453.0254_dp,    0.0000_dp,    0.0000_dp, &
      &  448.3333_dp,  448.0173_dp,  444.3975_dp,    0.0000_dp,    0.0000_dp, &
      &  436.4430_dp,  436.1655_dp,  432.6941_dp,    0.0000_dp,    0.0000_dp, &
      &  421.4187_dp,  421.1871_dp,  417.8991_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.5355_dp,  425.3152_dp,  422.0187_dp,    0.0000_dp,    0.0000_dp, &
      &  425.3152_dp,  425.1015_dp,  421.8178_dp,    0.0000_dp,    0.0000_dp, &
      &  422.0187_dp,  421.8178_dp,  418.5789_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   32.5171_dp,   32.8151_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   51.4656_dp,   51.9496_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   22.0813_dp,   22.2717_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  449.8512_dp,  454.3954_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  159.8670_dp,  161.4303_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.7398_dp,  276.4309_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  205.2237_dp,  207.2185_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  137.6192_dp,  138.9273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  190.1094_dp,  191.9273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  160.9679_dp,  162.4939_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  136.0239_dp,  137.3020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.0660_dp,  106.0350_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   99.1109_dp,  100.0210_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.4051_dp,  132.6251_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  123.2538_dp,  124.3958_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  101.4216_dp,  102.3489_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   94.9489_dp,   95.8139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   78.6686_dp,   79.3716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.4562_dp,   94.2991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   87.2028_dp,   87.9869_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   82.0374_dp,   82.7713_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.4531_dp,   73.0954_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   71.6059_dp,   72.2358_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.4141_dp,   64.9764_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   57.0738_dp,   57.5655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   54.7990_dp,   55.2689_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   45.0848_dp,   45.4618_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1387 * maxRef * maxRef + 1 : 1406 * maxRef * maxRef) = [ &
      &   42.4954_dp,   42.8512_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  539.8536_dp,  545.2855_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.6053_dp,  234.8776_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  432.1792_dp,  436.4452_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  358.8052_dp,  362.3258_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.8943_dp,  243.2097_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  404.7631_dp,  408.7149_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  368.2792_dp,  371.8615_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  290.7378_dp,  293.5337_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  230.0798_dp,  232.2617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.6912_dp,  328.8157_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  301.9852_dp,  304.8715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.7256_dp,  284.4096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.1045_dp,  276.7161_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  228.5370_dp,  230.6872_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  258.5121_dp,  260.9446_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  251.1081_dp,  253.4677_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  247.4732_dp,  249.7992_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  230.5860_dp,  232.7426_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  217.0337_dp,  219.0430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.6002_dp,  216.5866_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  210.3176_dp,  212.2628_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  179.2306_dp,  180.8604_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  177.3458_dp,  178.9581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  148.0216_dp,  149.3430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  880.1741_dp,  889.0026_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.1068_dp,  328.2207_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  747.9306_dp,  755.3478_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  600.2698_dp,  606.1859_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  338.0494_dp,  341.2732_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  622.1841_dp,  628.3317_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.0418_dp,  333.1801_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.4340_dp,  330.5365_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  605.0508_dp,  610.9949_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  318.9171_dp,  321.9433_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  331.6800_dp,  334.8197_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  556.3166_dp,  561.7643_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  312.4714_dp,  315.4349_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  303.5459_dp,  306.4123_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  439.1660_dp,  443.4401_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  309.0063_dp,  311.9395_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  231.7484_dp,  233.8940_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  481.8077_dp,  486.5025_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  286.8720_dp,  289.5784_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  199.8924_dp,  201.7229_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  379.2212_dp,  382.8900_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  266.2910_dp,  268.7934_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  196.0288_dp,  197.8251_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  404.6929_dp,  408.6064_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.0877_dp,  256.4721_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  226.8510_dp,  228.9654_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.6855_dp,  205.5633_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  415.1303_dp,  419.1592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  248.2237_dp,  250.5536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.1168_dp,  214.0853_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  221.0250_dp,  223.0896_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1406 * maxRef * maxRef + 1 : 1425 * maxRef * maxRef) = [ &
      &  318.9642_dp,  322.0346_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  243.5475_dp,  245.8435_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  331.0651_dp,  334.2414_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  231.4006_dp,  233.5632_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  390.4542_dp,  394.2289_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  368.2183_dp,  371.7655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  295.8817_dp,  298.6948_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  245.3469_dp,  247.6485_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  349.7760_dp,  353.1156_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  328.4750_dp,  331.5987_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  313.5280_dp,  316.5011_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  282.7195_dp,  285.3820_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  251.5229_dp,  253.8712_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  302.6743_dp,  305.5214_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  297.8064_dp,  300.6043_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  290.4211_dp,  293.1445_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  280.3577_dp,  282.9800_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.0630_dp,  276.6109_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.1377_dp,  276.6862_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  272.0583_dp,  274.5862_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  242.0556_dp,  244.2752_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  243.6296_dp,  245.8647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  212.2820_dp,  214.2004_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  989.8921_dp,  999.7871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.4684_dp,  393.1809_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  890.6723_dp,  899.4930_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  729.7176_dp,  736.9021_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  405.6158_dp,  409.4712_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  790.0257_dp,  797.7901_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  571.8432_dp,  577.3805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  394.3185_dp,  398.0382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  715.7318_dp,  722.7240_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  436.3046_dp,  440.4474_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  398.8895_dp,  402.6413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  656.2276_dp,  662.6068_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  440.4599_dp,  444.6402_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  385.5923_dp,  389.2112_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  511.6772_dp,  516.5932_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  412.2275_dp,  416.1207_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.9096_dp,  347.1000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  568.7909_dp,  574.2805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  456.9707_dp,  461.3223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.6667_dp,  308.4743_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  437.9992_dp,  442.1710_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  362.5374_dp,  365.9282_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.6001_dp,  292.2483_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  478.2481_dp,  482.8217_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  339.5620_dp,  342.7279_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  292.8227_dp,  295.5179_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  444.9662_dp,  449.2061_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  315.9288_dp,  318.8629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.1216_dp,  307.9410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.9344_dp,  374.4467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.2122_dp,  308.0475_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1425 * maxRef * maxRef + 1 : 1444 * maxRef * maxRef) = [ &
      &  393.7655_dp,  397.4979_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  303.8410_dp,  306.6549_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  488.9709_dp,  493.6625_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  468.1680_dp,  472.6441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  382.6809_dp,  386.2858_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.0106_dp,  333.0801_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  457.7156_dp,  462.0716_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  428.9963_dp,  433.0589_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  415.8173_dp,  419.7450_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  377.2255_dp,  380.7615_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.4912_dp,  346.6856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  412.9852_dp,  416.8715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  405.4400_dp,  409.2485_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  395.1626_dp,  398.8660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  382.1376_dp,  385.7084_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  386.0997_dp,  389.7029_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  385.9988_dp,  389.5996_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.1837_dp,  386.7550_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  351.9667_dp,  355.2175_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  355.2175_dp,  358.4995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.6055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   46.3794_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   20.4594_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  386.9969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  140.9818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.7689_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.6260_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  123.4070_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.8578_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.3697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  122.4634_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   95.3973_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   90.1296_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  118.9178_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.5923_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   92.2876_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   86.4581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.1617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   85.4293_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.7773_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   75.1427_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   66.5081_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   65.9403_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   59.4468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   52.8269_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.7900_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.0634_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.5951_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  465.2817_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  204.7986_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  378.6134_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  315.4581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.5479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  357.2960_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.7625_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  258.8687_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  206.4581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1444 * maxRef * maxRef + 1 : 1463 * maxRef * maxRef) = [ &
      &  290.4985_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  269.8372_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  252.1337_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  245.1798_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  205.8095_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.7786_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  226.2211_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  222.8660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  208.1507_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  196.7374_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  194.5237_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  190.6822_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  163.5368_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  161.8119_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  135.8592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  758.3253_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  288.7275_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  651.5392_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  524.3547_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  301.6273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.5631_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.8706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.2741_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.4312_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  285.1325_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  297.1871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  488.6522_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  279.3612_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  272.0795_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  386.5067_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  276.0406_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  209.3622_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  424.4125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  257.2803_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.3837_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  334.8259_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  239.2190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  177.7905_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  358.0954_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  228.3256_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  204.2663_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  184.2665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  366.5597_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  222.9856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  191.3621_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  198.6584_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  282.2888_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  218.0929_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.2138_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  208.4055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.1345_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.0838_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  264.7126_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  221.0120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  312.4265_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.9966_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  281.0046_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.2458_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  227.1324_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1463 * maxRef * maxRef + 1 : 1482 * maxRef * maxRef) = [ &
      &  272.4110_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  268.1700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  261.7361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  252.9678_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  247.9275_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  247.9867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  246.1495_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  220.1581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  221.5367_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  194.0699_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  854.5223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.6791_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  776.0859_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  637.4660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  362.3556_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  692.2386_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  505.2262_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  353.9178_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  629.5230_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.9385_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  358.5675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  578.7876_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  393.6121_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.7063_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  453.7998_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  369.1531_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  310.7442_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  503.3811_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  407.0970_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  277.2771_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.9532_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  326.1157_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.1051_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.2684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  305.7201_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  265.0611_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  396.3468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  284.7964_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  275.7577_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  330.7646_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.9744_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  351.5285_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.4407_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  434.0932_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  416.5053_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.1281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  297.7796_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  408.5952_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.9457_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  372.6480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  339.3505_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  310.1806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  370.9572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  364.5064_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  355.6742_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  344.4458_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1482 * maxRef * maxRef + 1 : 1501 * maxRef * maxRef) = [ &
      &  348.2239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  348.2164_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  345.8283_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  318.8905_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  321.8029_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  290.2223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.0096_dp,   42.3058_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  184.0223_dp,   68.9537_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   61.3184_dp,   27.4695_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3242.2404_dp,  732.9120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  752.3418_dp,  231.4165_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1316.2189_dp,  399.4020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  943.8547_dp,  294.1459_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  517.8726_dp,  187.1318_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  754.9243_dp,  262.2387_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  617.6058_dp,  219.6387_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  506.5054_dp,  183.6568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  349.4860_dp,  137.8005_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  324.9872_dp,  129.4461_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  455.6960_dp,  174.2718_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  431.5172_dp,  163.6715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  333.8130_dp,  132.4662_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  313.2347_dp,  123.9803_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  239.2442_dp,  100.4742_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  294.1483_dp,  120.4982_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  275.2567_dp,  112.4278_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  255.7499_dp,  105.3726_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  223.9076_dp,   92.7050_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  210.8672_dp,   90.5303_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  186.8791_dp,   81.0845_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  162.2001_dp,   71.3269_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  152.5672_dp,   68.1239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  117.9718_dp,   55.0786_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  113.1764_dp,   52.1041_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3798.9439_dp,  873.2665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1138.9256_dp,  340.0624_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2220.9652_dp,  642.1846_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1799.0180_dp,  527.5630_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  983.9584_dp,  335.2430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1881.7164_dp,  583.6850_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1673.1922_dp,  527.3282_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1217.3141_dp,  406.3724_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  858.6799_dp,  312.0540_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1317.0056_dp,  451.6171_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1200.5803_dp,  416.4861_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1098.1239_dp,  386.4327_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1088.8566_dp,  377.4920_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  823.8201_dp,  306.7839_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  936.3069_dp,  347.4030_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  907.2993_dp,  337.1135_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  901.7293_dp,  332.8876_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  815.1518_dp,  307.6574_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  735.0061_dp,  286.0418_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  728.4653_dp,  282.9669_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  713.2229_dp,  277.2121_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1501 * maxRef * maxRef + 1 : 1520 * maxRef * maxRef) = [ &
      &  569.6701_dp,  231.9483_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  564.6006_dp,  229.5837_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.8171_dp,  188.5950_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 6808.3900_dp, 1449.5860_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1511.8898_dp,  465.5182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4387.1669_dp, 1149.8295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3619.2660_dp,  921.7934_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1402.1290_dp,  471.0509_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3450.8369_dp,  942.2745_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1344.5274_dp,  457.9012_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1285.9957_dp,  449.4774_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3215.8544_dp,  904.2225_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1308.6418_dp,  442.0547_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1297.3700_dp,  454.7799_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2877.5279_dp,  824.7780_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1283.1193_dp,  433.3139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1199.7581_dp,  416.6755_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2283.0674_dp,  650.0627_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1287.9616_dp,  430.1040_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  844.4635_dp,  310.7016_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2396.3445_dp,  706.1412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1110.3588_dp,  392.0831_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  691.3086_dp,  264.1180_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1884.2824_dp,  554.0925_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1013.0459_dp,  362.0455_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  682.0579_dp,  259.4193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1875.7209_dp,  581.8458_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  969.5462_dp,  345.4470_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  858.1614_dp,  307.2297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  727.5711_dp,  271.6004_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1976.8609_dp,  601.5808_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  952.3401_dp,  337.9454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  780.5520_dp,  285.2553_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  859.7252_dp,  301.6400_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1533.4063_dp,  461.7019_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  983.5916_dp,  336.0821_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1443.0004_dp,  467.8731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  849.3446_dp,  311.5800_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1755.5105_dp,  556.5016_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1609.8951_dp,  520.7490_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1179.2492_dp,  407.5724_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  887.2092_dp,  329.4410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1406.2206_dp,  483.7117_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1290.3249_dp,  451.1717_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1210.8497_dp,  428.6170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1047.5077_dp,  382.0562_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  883.9799_dp,  335.0524_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1108.3145_dp,  407.7958_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1085.3677_dp,  400.6439_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1048.8144_dp,  389.6837_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  997.6271_dp,  374.6769_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  950.1975_dp,  363.5286_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  951.6305_dp,  363.7202_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  943.1487_dp,  360.8053_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  794.2184_dp,  316.0700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  801.8614_dp,  318.3827_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1520 * maxRef * maxRef + 1 : 1539 * maxRef * maxRef) = [ &
      &  662.9556_dp,  273.2999_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 7560.4361_dp, 1620.7604_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1776.8880_dp,  553.9780_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5305.1627_dp, 1372.8515_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4496.9846_dp, 1124.9345_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1681.3249_dp,  564.3454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4289.6097_dp, 1186.3885_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2950.2475_dp,  837.6135_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1515.3755_dp,  537.9093_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3674.8556_dp, 1057.7660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1819.0406_dp,  607.1951_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1502.0119_dp,  541.1259_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3256.3633_dp,  959.7908_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1826.4461_dp,  613.2010_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1473.7970_dp,  524.1562_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2398.7520_dp,  735.1498_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1687.7490_dp,  570.2621_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1239.1934_dp,  460.1263_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2726.0486_dp,  822.5829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2068.8557_dp,  647.6686_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1053.7818_dp,  403.9077_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1968.3066_dp,  621.1968_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1387.1242_dp,  492.8301_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  982.3016_dp,  380.9010_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2129.9275_dp,  678.2767_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1295.8507_dp,  460.8194_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1051.3610_dp,  390.6048_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1935.8392_dp,  627.0057_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1193.7040_dp,  427.3247_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1107.7988_dp,  408.6188_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1657.7829_dp,  524.5075_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1162.5520_dp,  413.7796_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1663.0447_dp,  550.2565_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1094.9943_dp,  406.5459_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2226.4867_dp,  697.3745_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2062.6273_dp,  661.8423_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1514.1166_dp,  525.0776_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1190.0927_dp,  442.0537_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1891.9207_dp,  636.7612_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1716.1526_dp,  591.2764_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1633.6348_dp,  570.2993_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1409.0375_dp,  510.2857_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1220.0459_dp,  458.3940_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1566.5042_dp,  561.2533_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1521.2386_dp,  549.3044_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1461.4083_dp,  533.2393_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1386.5806_dp,  513.0291_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1394.5709_dp,  517.5807_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1389.9562_dp,  517.0109_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1372.2422_dp,  512.4692_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1207.5672_dp,  465.0703_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1219.6272_dp,  469.4818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1043.3955_dp,  415.7690_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1539 * maxRef * maxRef + 1 : 1558 * maxRef * maxRef) = [ &
      & 9330.7294_dp, 1975.9738_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1975.9738_dp,  649.0924_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   99.4579_dp,   83.0141_dp,   44.8321_dp,    0.0000_dp,    0.0000_dp, &
      &  171.7033_dp,  142.5680_dp,   72.9013_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   59.1141_dp,   49.9232_dp,   29.1864_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2551.5260_dp, 2180.4015_dp,  750.3227_dp,    0.0000_dp,    0.0000_dp, &
      &  664.5719_dp,  551.0792_dp,  242.2872_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1160.4419_dp,  961.1049_dp,  417.8652_dp,    0.0000_dp,    0.0000_dp, &
      &  832.6251_dp,  692.8415_dp,  308.1410_dp,    0.0000_dp,    0.0000_dp, &
      &  479.3661_dp,  397.2194_dp,  197.5839_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  690.6852_dp,  572.3002_dp,  276.3430_dp,    0.0000_dp,    0.0000_dp, &
      &  567.5136_dp,  471.0503_dp,  231.6543_dp,    0.0000_dp,    0.0000_dp, &
      &  465.9766_dp,  388.0268_dp,  193.8381_dp,    0.0000_dp,    0.0000_dp, &
      &  329.4784_dp,  274.5914_dp,  145.9330_dp,    0.0000_dp,    0.0000_dp, &
      &  306.9686_dp,  256.0547_dp,  137.1250_dp,    0.0000_dp,    0.0000_dp, &
      &  426.3667_dp,  354.8466_dp,  184.3462_dp,    0.0000_dp,    0.0000_dp, &
      &  401.8774_dp,  334.9207_dp,  173.0249_dp,    0.0000_dp,    0.0000_dp, &
      &  314.6837_dp,  262.6181_dp,  140.2904_dp,    0.0000_dp,    0.0000_dp, &
      &  294.7168_dp,  246.1256_dp,  131.2676_dp,    0.0000_dp,    0.0000_dp, &
      &  228.0556_dp,  191.2494_dp,  106.5800_dp,    0.0000_dp,    0.0000_dp, &
      &  279.0742_dp,  233.5006_dp,  127.7419_dp,    0.0000_dp,    0.0000_dp, &
      &  260.4481_dp,  218.1596_dp,  119.1434_dp,    0.0000_dp,    0.0000_dp, &
      &  242.5528_dp,  203.2396_dp,  111.7057_dp,    0.0000_dp,    0.0000_dp, &
      &  212.1297_dp,  178.0536_dp,   98.2670_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  201.7411_dp,  169.6030_dp,   96.0950_dp,    0.0000_dp,    0.0000_dp, &
      &  178.9549_dp,  150.7056_dp,   86.0773_dp,    0.0000_dp,    0.0000_dp, &
      &  155.5437_dp,  131.2633_dp,   75.7417_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  146.9174_dp,  124.0802_dp,   72.3852_dp,    0.0000_dp,    0.0000_dp, &
      &  114.4676_dp,   97.1926_dp,   58.5836_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.5000_dp,   92.8494_dp,   55.4092_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3009.7233_dp, 2565.9398_dp,  895.1626_dp,    0.0000_dp,    0.0000_dp, &
      &  995.3660_dp,  826.9565_dp,  355.3439_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1926.3504_dp, 1599.6001_dp,  669.7642_dp,    0.0000_dp,    0.0000_dp, &
      & 1559.5050_dp, 1298.9425_dp,  550.6761_dp,    0.0000_dp,    0.0000_dp, &
      &  897.4029_dp,  742.3673_dp,  353.0531_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1665.7662_dp, 1381.4389_dp,  611.4261_dp,    0.0000_dp,    0.0000_dp, &
      & 1487.2964_dp, 1233.5817_dp,  552.8634_dp,    0.0000_dp,    0.0000_dp, &
      & 1098.7485_dp,  911.8321_dp,  427.3949_dp,    0.0000_dp,    0.0000_dp, &
      &  796.1066_dp,  659.7476_dp,  329.5915_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1201.2021_dp,  994.8926_dp,  475.6506_dp,    0.0000_dp,    0.0000_dp, &
      & 1097.3980_dp,  909.6567_dp,  438.8597_dp,    0.0000_dp,    0.0000_dp, &
      & 1007.8503_dp,  835.4801_dp,  407.4620_dp,    0.0000_dp,    0.0000_dp, &
      &  993.2763_dp,  824.3552_dp,  397.6803_dp,    0.0000_dp,    0.0000_dp, &
      &  767.6095_dp,  637.1516_dp,  324.2955_dp,    0.0000_dp,    0.0000_dp, &
      &  871.1647_dp,  723.2504_dp,  367.1449_dp,    0.0000_dp,    0.0000_dp, &
      &  844.0120_dp,  700.9983_dp,  356.2697_dp,    0.0000_dp,    0.0000_dp, &
      &  836.9970_dp,  695.2639_dp,  351.6844_dp,    0.0000_dp,    0.0000_dp, &
      &  761.1433_dp,  632.6159_dp,  325.3328_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  690.8651_dp,  575.3317_dp,  302.7908_dp,    0.0000_dp,    0.0000_dp, &
      &  684.2353_dp,  569.8509_dp,  299.5059_dp,    0.0000_dp,    0.0000_dp, &
      &  669.9106_dp,  557.9918_dp,  293.4143_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  540.3160_dp,  451.5825_dp,  245.8635_dp,    0.0000_dp,    0.0000_dp, &
      &  535.2678_dp,  447.3680_dp,  243.3421_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.7909_dp,  357.1935_dp,  200.1094_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1558 * maxRef * maxRef + 1 : 1577 * maxRef * maxRef) = [ &
      & 5171.2978_dp, 4495.8369_dp, 1476.8694_dp,    0.0000_dp,    0.0000_dp, &
      & 1322.2619_dp, 1104.1068_dp,  486.9346_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3659.1655_dp, 3070.4624_dp, 1190.5306_dp,    0.0000_dp,    0.0000_dp, &
      & 2951.3362_dp, 2505.6537_dp,  952.5771_dp,    0.0000_dp,    0.0000_dp, &
      & 1267.8490_dp, 1051.9461_dp,  495.4172_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2924.4395_dp, 2443.4277_dp,  978.5884_dp,    0.0000_dp,    0.0000_dp, &
      & 1222.7426_dp, 1013.0298_dp,  481.8238_dp,    0.0000_dp,    0.0000_dp, &
      & 1176.2602_dp,  975.9665_dp,  473.6984_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2748.9797_dp, 2294.6062_dp,  940.9324_dp,    0.0000_dp,    0.0000_dp, &
      & 1180.9424_dp,  982.3638_dp,  465.0211_dp,    0.0000_dp,    0.0000_dp, &
      & 1187.6789_dp,  985.4293_dp,  479.3449_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2475.1427_dp, 2064.0309_dp,  859.3730_dp,    0.0000_dp,    0.0000_dp, &
      & 1158.6662_dp,  963.2725_dp,  455.7896_dp,    0.0000_dp,    0.0000_dp, &
      & 1092.6689_dp,  908.1340_dp,  438.9190_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1952.1126_dp, 1632.7022_dp,  676.8113_dp,    0.0000_dp,    0.0000_dp, &
      & 1158.6672_dp,  963.6534_dp,  452.1352_dp,    0.0000_dp,    0.0000_dp, &
      &  780.0838_dp,  649.5624_dp,  328.0436_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2079.4950_dp, 1732.0375_dp,  737.0898_dp,    0.0000_dp,    0.0000_dp, &
      & 1018.6688_dp,  845.2119_dp,  413.4328_dp,    0.0000_dp,    0.0000_dp, &
      &  644.6600_dp,  537.5804_dp,  279.2953_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1628.4040_dp, 1359.6156_dp,  578.0899_dp,    0.0000_dp,    0.0000_dp, &
      &  931.8610_dp,  773.6698_dp,  381.9591_dp,    0.0000_dp,    0.0000_dp, &
      &  635.3811_dp,  529.7155_dp,  274.2873_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1656.9384_dp, 1375.9674_dp,  609.2865_dp,    0.0000_dp,    0.0000_dp, &
      &  889.9346_dp,  739.5090_dp,  364.3760_dp,    0.0000_dp,    0.0000_dp, &
      &  787.1393_dp,  654.9942_dp,  324.0498_dp,    0.0000_dp,    0.0000_dp, &
      &  675.0933_dp,  562.1481_dp,  286.9784_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1736.3987_dp, 1442.5133_dp,  629.2493_dp,    0.0000_dp,    0.0000_dp, &
      &  873.1622_dp,  725.5203_dp,  356.3892_dp,    0.0000_dp,    0.0000_dp, &
      &  720.7044_dp,  599.5116_dp,  301.1629_dp,    0.0000_dp,    0.0000_dp, &
      &  784.5252_dp,  652.3996_dp,  317.8255_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1335.7319_dp, 1113.8541_dp,  482.4149_dp,    0.0000_dp,    0.0000_dp, &
      &  891.6885_dp,  740.9872_dp,  353.7227_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1292.6630_dp, 1072.0643_dp,  491.2073_dp,    0.0000_dp,    0.0000_dp, &
      &  787.2771_dp,  653.7336_dp,  329.1343_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1559.7012_dp, 1295.4563_dp,  583.5203_dp,    0.0000_dp,    0.0000_dp, &
      & 1439.3610_dp, 1194.8773_dp,  546.6402_dp,    0.0000_dp,    0.0000_dp, &
      & 1073.9458_dp,  891.8106_dp,  429.3221_dp,    0.0000_dp,    0.0000_dp, &
      &  825.7096_dp,  685.5002_dp,  348.1978_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1281.9152_dp, 1062.7315_dp,  509.4520_dp,    0.0000_dp,    0.0000_dp, &
      & 1180.9729_dp,  979.5742_dp,  475.5382_dp,    0.0000_dp,    0.0000_dp, &
      & 1112.0376_dp,  922.5046_dp,  452.0196_dp,    0.0000_dp,    0.0000_dp, &
      &  969.6559_dp,  804.9599_dp,  403.4560_dp,    0.0000_dp,    0.0000_dp, &
      &  826.6644_dp,  686.9957_dp,  354.4023_dp,    0.0000_dp,    0.0000_dp, &
      & 1028.5424_dp,  853.9157_dp,  430.8021_dp,    0.0000_dp,    0.0000_dp, &
      & 1007.7913_dp,  836.9364_dp,  423.2918_dp,    0.0000_dp,    0.0000_dp, &
      &  975.3242_dp,  810.2357_dp,  411.8184_dp,    0.0000_dp,    0.0000_dp, &
      &  930.3838_dp,  773.1293_dp,  396.1357_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  889.6147_dp,  740.1496_dp,  384.5873_dp,    0.0000_dp,    0.0000_dp, &
      &  890.6264_dp,  741.0220_dp,  384.7697_dp,    0.0000_dp,    0.0000_dp, &
      &  882.8069_dp,  734.5865_dp,  381.6942_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  749.8475_dp,  625.5504_dp,  334.8115_dp,    0.0000_dp,    0.0000_dp, &
      &  756.6271_dp,  631.1397_dp,  337.2327_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  630.2436_dp,  527.3521_dp,  289.7981_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5743.5673_dp, 4997.9230_dp, 1652.2832_dp,    0.0000_dp,    0.0000_dp, &
      & 1556.9958_dp, 1301.3999_dp,  579.7980_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1577 * maxRef * maxRef + 1 : 1596 * maxRef * maxRef) = [ &
      & 4394.3607_dp, 3697.6569_dp, 1419.9417_dp,    0.0000_dp,    0.0000_dp, &
      & 3631.4540_dp, 3095.7594_dp, 1160.9040_dp,    0.0000_dp,    0.0000_dp, &
      & 1516.5369_dp, 1260.1619_dp,  593.3572_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3639.4735_dp, 3045.1620_dp, 1233.0006_dp,    0.0000_dp,    0.0000_dp, &
      & 2485.6310_dp, 2101.0372_dp,  872.2882_dp,    0.0000_dp,    0.0000_dp, &
      & 1391.1764_dp, 1154.9423_dp,  567.2445_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3160.1776_dp, 2637.6782_dp, 1102.2638_dp,    0.0000_dp,    0.0000_dp, &
      & 1635.3622_dp, 1360.4000_dp,  638.0792_dp,    0.0000_dp,    0.0000_dp, &
      & 1384.6011_dp, 1149.7163_dp,  571.0295_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2819.5147_dp, 2352.1278_dp, 1001.6660_dp,    0.0000_dp,    0.0000_dp, &
      & 1650.9334_dp, 1369.3825_dp,  644.4515_dp,    0.0000_dp,    0.0000_dp, &
      & 1349.3970_dp, 1122.7672_dp,  552.6718_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2098.4813_dp, 1750.3829_dp,  768.8382_dp,    0.0000_dp,    0.0000_dp, &
      & 1520.1317_dp, 1266.0803_dp,  599.6614_dp,    0.0000_dp,    0.0000_dp, &
      & 1148.5106_dp,  956.0229_dp,  486.0298_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2373.9131_dp, 1980.9508_dp,  859.6392_dp,    0.0000_dp,    0.0000_dp, &
      & 1811.0626_dp, 1516.2134_dp,  678.1232_dp,    0.0000_dp,    0.0000_dp, &
      &  984.1861_dp,  820.4174_dp,  427.1818_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1735.4827_dp, 1447.6221_dp,  650.7128_dp,    0.0000_dp,    0.0000_dp, &
      & 1270.1002_dp, 1056.4422_dp,  519.5738_dp,    0.0000_dp,    0.0000_dp, &
      &  919.9110_dp,  767.3168_dp,  403.0280_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1890.1371_dp, 1572.3023_dp,  711.0892_dp,    0.0000_dp,    0.0000_dp, &
      & 1185.1747_dp,  986.7643_dp,  485.8072_dp,    0.0000_dp,    0.0000_dp, &
      &  972.1452_dp,  810.5895_dp,  412.5321_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1727.2738_dp, 1436.0400_dp,  657.9647_dp,    0.0000_dp,    0.0000_dp, &
      & 1093.0044_dp,  910.5268_dp,  450.6082_dp,    0.0000_dp,    0.0000_dp, &
      & 1024.0901_dp,  852.7680_dp,  431.4907_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1460.5858_dp, 1219.4904_dp,  549.4299_dp,    0.0000_dp,    0.0000_dp, &
      & 1063.3544_dp,  885.4039_dp,  436.2139_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1493.7369_dp, 1241.2980_dp,  578.1096_dp,    0.0000_dp,    0.0000_dp, &
      & 1016.1802_dp,  845.1580_dp,  429.5551_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1964.0913_dp, 1635.6158_dp,  730.5114_dp,    0.0000_dp,    0.0000_dp, &
      & 1834.2595_dp, 1525.9278_dp,  694.2560_dp,    0.0000_dp,    0.0000_dp, &
      & 1377.5185_dp, 1145.3931_dp,  553.0827_dp,    0.0000_dp,    0.0000_dp, &
      & 1105.9339_dp,  919.0928_dp,  467.1608_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1710.8832_dp, 1420.0022_dp,  669.7960_dp,    0.0000_dp,    0.0000_dp, &
      & 1561.8289_dp, 1296.6408_dp,  622.6826_dp,    0.0000_dp,    0.0000_dp, &
      & 1492.4625_dp, 1239.0351_dp,  600.9779_dp,    0.0000_dp,    0.0000_dp, &
      & 1300.6349_dp, 1080.2049_dp,  538.6735_dp,    0.0000_dp,    0.0000_dp, &
      & 1137.6146_dp,  945.5274_dp,  484.6892_dp,    0.0000_dp,    0.0000_dp, &
      & 1442.5135_dp, 1197.3790_dp,  592.2077_dp,    0.0000_dp,    0.0000_dp, &
      & 1403.5743_dp, 1165.3603_dp,  579.8000_dp,    0.0000_dp,    0.0000_dp, &
      & 1352.0527_dp, 1122.9068_dp,  563.1051_dp,    0.0000_dp,    0.0000_dp, &
      & 1287.7738_dp, 1069.7955_dp,  542.0984_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1295.6354_dp, 1076.7503_dp,  546.9393_dp,    0.0000_dp,    0.0000_dp, &
      & 1292.0505_dp, 1073.8698_dp,  546.3872_dp,    0.0000_dp,    0.0000_dp, &
      & 1276.8973_dp, 1061.4185_dp,  541.6776_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1131.7221_dp,  942.2524_dp,  492.1377_dp,    0.0000_dp,    0.0000_dp, &
      & 1142.9378_dp,  951.5410_dp,  496.7990_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  985.1266_dp,  822.0568_dp,  440.4668_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 7016.4917_dp, 6141.9784_dp, 2013.0282_dp,    0.0000_dp,    0.0000_dp, &
      & 1760.0602_dp, 1469.0705_dp,  681.3467_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5726.9887_dp, 4845.1776_dp, 1816.0675_dp,    0.0000_dp,    0.0000_dp, &
      & 4845.1776_dp, 4171.7842_dp, 1511.3430_dp,    0.0000_dp,    0.0000_dp, &
      & 1816.0675_dp, 1511.3430_dp,  716.5475_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1596 * maxRef * maxRef + 1 : 1615 * maxRef * maxRef) = [ &
      &   89.1018_dp,   64.6906_dp,   46.8202_dp,    0.0000_dp,    0.0000_dp, &
      &  152.1106_dp,  108.3059_dp,   75.8394_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   53.7982_dp,   40.3329_dp,   30.5886_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2085.0661_dp, 1384.1283_dp,  748.9052_dp,    0.0000_dp,    0.0000_dp, &
      &  570.4266_dp,  389.9070_dp,  248.4867_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  994.0316_dp,  676.7374_dp,  428.0498_dp,    0.0000_dp,    0.0000_dp, &
      &  716.0758_dp,  492.5824_dp,  316.4815_dp,    0.0000_dp,    0.0000_dp, &
      &  422.2754_dp,  297.9018_dp,  205.0850_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  604.4915_dp,  423.0815_dp,  286.0290_dp,    0.0000_dp,    0.0000_dp, &
      &  498.4534_dp,  351.1166_dp,  240.1361_dp,    0.0000_dp,    0.0000_dp, &
      &  410.5123_dp,  291.2744_dp,  201.2083_dp,    0.0000_dp,    0.0000_dp, &
      &  294.1692_dp,  212.3569_dp,  152.2250_dp,    0.0000_dp,    0.0000_dp, &
      &  274.4932_dp,  198.7197_dp,  143.1068_dp,    0.0000_dp,    0.0000_dp, &
      &  378.7928_dp,  271.4248_dp,  191.9355_dp,    0.0000_dp,    0.0000_dp, &
      &  356.4610_dp,  255.3534_dp,  180.0326_dp,    0.0000_dp,    0.0000_dp, &
      &  281.1504_dp,  203.4829_dp,  146.3330_dp,    0.0000_dp,    0.0000_dp, &
      &  263.1381_dp,  190.4785_dp,  136.8686_dp,    0.0000_dp,    0.0000_dp, &
      &  205.5407_dp,  151.1411_dp,  111.4633_dp,    0.0000_dp,    0.0000_dp, &
      &  250.5577_dp,  182.9391_dp,  133.4410_dp,    0.0000_dp,    0.0000_dp, &
      &  233.6619_dp,  170.6938_dp,  124.4124_dp,    0.0000_dp,    0.0000_dp, &
      &  217.8948_dp,  159.4968_dp,  116.6803_dp,    0.0000_dp,    0.0000_dp, &
      &  190.6370_dp,  139.9128_dp,  102.6251_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  182.4075_dp,  135.0322_dp,  100.5575_dp,    0.0000_dp,    0.0000_dp, &
      &  162.0371_dp,  120.4004_dp,   90.0932_dp,    0.0000_dp,    0.0000_dp, &
      &  141.0933_dp,  105.3236_dp,   79.2877_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  133.6123_dp,  100.1090_dp,   75.8298_dp,    0.0000_dp,    0.0000_dp, &
      &  104.8095_dp,   79.6151_dp,   61.4499_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  100.0277_dp,   75.6791_dp,   58.0795_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2465.5965_dp, 1637.7878_dp,  894.7448_dp,    0.0000_dp,    0.0000_dp, &
      &  850.0768_dp,  578.7007_dp,  363.4612_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1637.1586_dp, 1106.7221_dp,  683.2371_dp,    0.0000_dp,    0.0000_dp, &
      & 1328.2510_dp,  903.4563_dp,  562.6808_dp,    0.0000_dp,    0.0000_dp, &
      &  783.1302_dp,  545.0752_dp,  364.9162_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1431.7496_dp,  980.4376_dp,  627.5348_dp,    0.0000_dp,    0.0000_dp, &
      & 1281.5229_dp,  880.4900_dp,  568.1457_dp,    0.0000_dp,    0.0000_dp, &
      &  955.5488_dp,  664.5904_dp,  441.2004_dp,    0.0000_dp,    0.0000_dp, &
      &  701.9398_dp,  495.8077_dp,  342.2242_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1049.1934_dp,  732.1211_dp,  491.8929_dp,    0.0000_dp,    0.0000_dp, &
      &  960.2332_dp,  672.1591_dp,  454.2176_dp,    0.0000_dp,    0.0000_dp, &
      &  883.8815_dp,  620.5160_dp,  422.1202_dp,    0.0000_dp,    0.0000_dp, &
      &  868.8607_dp,  608.8756_dp,  411.5436_dp,    0.0000_dp,    0.0000_dp, &
      &  679.3502_dp,  482.9601_dp,  337.2225_dp,    0.0000_dp,    0.0000_dp, &
      &  770.4924_dp,  547.4653_dp,  381.6757_dp,    0.0000_dp,    0.0000_dp, &
      &  746.5993_dp,  530.8573_dp,  370.3899_dp,    0.0000_dp,    0.0000_dp, &
      &  739.5674_dp,  525.2515_dp,  365.4498_dp,    0.0000_dp,    0.0000_dp, &
      &  674.9489_dp,  481.7395_dp,  338.5362_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  615.5559_dp,  442.8799_dp,  315.6193_dp,    0.0000_dp,    0.0000_dp, &
      &  609.4420_dp,  438.3496_dp,  312.1500_dp,    0.0000_dp,    0.0000_dp, &
      &  596.7186_dp,  429.3015_dp,  305.7996_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  484.7691_dp,  353.1467_dp,  256.8455_dp,    0.0000_dp,    0.0000_dp, &
      &  480.1244_dp,  349.6764_dp,  254.1851_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  384.2306_dp,  283.0351_dp,  209.3707_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4185.8968_dp, 2800.9569_dp, 1467.2405_dp,    0.0000_dp,    0.0000_dp, &
      & 1133.7749_dp,  780.6806_dp,  499.2067_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1615 * maxRef * maxRef + 1 : 1634 * maxRef * maxRef) = [ &
      & 3059.8941_dp, 2050.4253_dp, 1204.1091_dp,    0.0000_dp,    0.0000_dp, &
      & 2456.7601_dp, 1658.3459_dp,  962.2662_dp,    0.0000_dp,    0.0000_dp, &
      & 1103.1556_dp,  767.9412_dp,  511.3063_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2461.7619_dp, 1655.7011_dp,  993.2805_dp,    0.0000_dp,    0.0000_dp, &
      & 1065.8140_dp,  742.6398_dp,  497.6257_dp,    0.0000_dp,    0.0000_dp, &
      & 1030.0975_dp,  722.8068_dp,  490.3181_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2324.7454_dp, 1571.2363_dp,  957.5700_dp,    0.0000_dp,    0.0000_dp, &
      & 1028.4169_dp,  718.6782_dp,  480.2028_dp,    0.0000_dp,    0.0000_dp, &
      & 1040.5570_dp,  730.5584_dp,  496.2446_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2099.5853_dp, 1423.1173_dp,  876.0050_dp,    0.0000_dp,    0.0000_dp, &
      & 1008.8336_dp,  704.4731_dp,  470.5903_dp,    0.0000_dp,    0.0000_dp, &
      &  955.7169_dp,  670.9208_dp,  454.0832_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1653.4435_dp, 1122.6576_dp,  689.4794_dp,    0.0000_dp,    0.0000_dp, &
      & 1006.9636_dp,  701.8768_dp,  466.4241_dp,    0.0000_dp,    0.0000_dp, &
      &  688.3433_dp,  489.6953_dp,  340.5132_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1771.7653_dp, 1206.1393_dp,  753.0932_dp,    0.0000_dp,    0.0000_dp, &
      &  893.5056_dp,  628.2340_dp,  428.1805_dp,    0.0000_dp,    0.0000_dp, &
      &  572.2667_dp,  410.7620_dp,  290.5610_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1386.3132_dp,  945.6808_dp,  590.4390_dp,    0.0000_dp,    0.0000_dp, &
      &  818.8994_dp,  577.5263_dp,  395.8945_dp,    0.0000_dp,    0.0000_dp, &
      &  563.6245_dp,  404.1026_dp,  285.2677_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1423.4023_dp,  975.9836_dp,  625.0103_dp,    0.0000_dp,    0.0000_dp, &
      &  781.6350_dp,  551.4226_dp,  377.5895_dp,    0.0000_dp,    0.0000_dp, &
      &  691.6368_dp,  489.0580_dp,  335.8153_dp,    0.0000_dp,    0.0000_dp, &
      &  597.1154_dp,  426.0198_dp,  298.1443_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1487.1454_dp, 1016.1745_dp,  644.5028_dp,    0.0000_dp,    0.0000_dp, &
      &  766.3677_dp,  540.1679_dp,  369.1952_dp,    0.0000_dp,    0.0000_dp, &
      &  635.3561_dp,  450.9348_dp,  312.4878_dp,    0.0000_dp,    0.0000_dp, &
      &  686.9902_dp,  483.4812_dp,  328.8388_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1141.4078_dp,  781.2881_dp,  493.6053_dp,    0.0000_dp,    0.0000_dp, &
      &  777.7070_dp,  544.0661_dp,  365.3990_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1118.3973_dp,  772.6650_dp,  505.5837_dp,    0.0000_dp,    0.0000_dp, &
      &  694.7959_dp,  492.6342_dp,  341.7316_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1344.7968_dp,  926.3723_dp,  599.8301_dp,    0.0000_dp,    0.0000_dp, &
      & 1245.0382_dp,  860.6632_dp,  562.7686_dp,    0.0000_dp,    0.0000_dp, &
      &  938.9286_dp,  657.9651_dp,  444.1573_dp,    0.0000_dp,    0.0000_dp, &
      &  730.3259_dp,  518.9832_dp,  361.9337_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1120.0021_dp,  782.7401_dp,  526.8660_dp,    0.0000_dp,    0.0000_dp, &
      & 1034.4900_dp,  725.7474_dp,  492.3604_dp,    0.0000_dp,    0.0000_dp, &
      &  976.0066_dp,  686.4772_dp,  468.3886_dp,    0.0000_dp,    0.0000_dp, &
      &  855.1298_dp,  605.4842_dp,  418.8957_dp,    0.0000_dp,    0.0000_dp, &
      &  733.5714_dp,  523.9575_dp,  368.8682_dp,    0.0000_dp,    0.0000_dp, &
      &  908.3670_dp,  644.3560_dp,  447.5546_dp,    0.0000_dp,    0.0000_dp, &
      &  890.4764_dp,  632.2602_dp,  439.8384_dp,    0.0000_dp,    0.0000_dp, &
      &  862.6780_dp,  613.5158_dp,  428.0906_dp,    0.0000_dp,    0.0000_dp, &
      &  824.3488_dp,  587.6805_dp,  412.0610_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  790.4855_dp,  566.2920_dp,  400.4704_dp,    0.0000_dp,    0.0000_dp, &
      &  791.2471_dp,  566.7513_dp,  400.6320_dp,    0.0000_dp,    0.0000_dp, &
      &  784.4000_dp,  562.0006_dp,  397.4445_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  670.3843_dp,  485.2955_dp,  349.3710_dp,    0.0000_dp,    0.0000_dp, &
      &  676.1930_dp,  489.2291_dp,  351.8502_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  566.5101_dp,  414.1664_dp,  302.8960_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4653.7454_dp, 3121.5559_dp, 1643.0879_dp,    0.0000_dp,    0.0000_dp, &
      & 1337.5010_dp,  924.2203_dp,  594.9926_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3666.6556_dp, 2458.0845_dp, 1434.6583_dp,    0.0000_dp,    0.0000_dp, &
      & 3014.0781_dp, 2036.5362_dp, 1171.1205_dp,    0.0000_dp,    0.0000_dp, &
      & 1319.0312_dp,  919.4490_dp,  612.2994_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1634 * maxRef * maxRef + 1 : 1653 * maxRef * maxRef) = [ &
      & 3069.3037_dp, 2072.7978_dp, 1253.1317_dp,    0.0000_dp,    0.0000_dp, &
      & 2105.7491_dp, 1446.2394_dp,  889.9501_dp,    0.0000_dp,    0.0000_dp, &
      & 1221.2152_dp,  860.0465_dp,  587.7222_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2682.0879_dp, 1821.2145_dp, 1124.0457_dp,    0.0000_dp,    0.0000_dp, &
      & 1420.6192_dp,  990.1059_dp,  658.2098_dp,    0.0000_dp,    0.0000_dp, &
      & 1218.3604_dp,  860.7503_dp,  592.2396_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2401.9326_dp, 1637.9243_dp, 1023.5000_dp,    0.0000_dp,    0.0000_dp, &
      & 1434.8956_dp,  997.9222_dp,  664.7346_dp,    0.0000_dp,    0.0000_dp, &
      & 1184.6610_dp,  836.5504_dp,  572.6815_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1798.1970_dp, 1235.6226_dp,  787.9045_dp,    0.0000_dp,    0.0000_dp, &
      & 1323.2583_dp,  925.8359_dp,  619.0974_dp,    0.0000_dp,    0.0000_dp, &
      & 1015.1154_dp,  723.2437_dp,  504.8956_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2029.6163_dp, 1390.9989_dp,  880.0314_dp,    0.0000_dp,    0.0000_dp, &
      & 1557.0674_dp, 1078.7908_dp,  696.3752_dp,    0.0000_dp,    0.0000_dp, &
      &  874.2449_dp,  627.7003_dp,  444.5757_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1493.8895_dp, 1032.5864_dp,  668.2859_dp,    0.0000_dp,    0.0000_dp, &
      & 1114.5962_dp,  786.5886_dp,  538.1720_dp,    0.0000_dp,    0.0000_dp, &
      &  818.6281_dp,  589.4768_dp,  419.6916_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1629.9126_dp, 1125.4492_dp,  730.8135_dp,    0.0000_dp,    0.0000_dp, &
      & 1040.1066_dp,  734.8918_dp,  503.1910_dp,    0.0000_dp,    0.0000_dp, &
      &  859.0221_dp,  613.1365_dp,  428.4259_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1493.4855_dp, 1034.1628_dp,  677.0271_dp,    0.0000_dp,    0.0000_dp, &
      &  960.1346_dp,  679.6572_dp,  466.8855_dp,    0.0000_dp,    0.0000_dp, &
      &  903.9928_dp,  643.4443_dp,  447.9267_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1257.4297_dp,  870.4102_dp,  564.2490_dp,    0.0000_dp,    0.0000_dp, &
      &  933.1680_dp,  659.4710_dp,  451.7417_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1295.9364_dp,  900.7673_dp,  595.7322_dp,    0.0000_dp,    0.0000_dp, &
      &  898.2092_dp,  639.3832_dp,  446.1765_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1689.4085_dp, 1164.0480_dp,  749.9911_dp,    0.0000_dp,    0.0000_dp, &
      & 1583.8792_dp, 1095.4590_dp,  714.0617_dp,    0.0000_dp,    0.0000_dp, &
      & 1204.5072_dp,  845.7173_dp,  572.0823_dp,    0.0000_dp,    0.0000_dp, &
      &  977.8582_dp,  695.6184_dp,  485.3739_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1489.0440_dp, 1037.3050_dp,  691.3637_dp,    0.0000_dp,    0.0000_dp, &
      & 1364.4290_dp,  955.2258_dp,  643.8177_dp,    0.0000_dp,    0.0000_dp, &
      & 1306.5898_dp,  917.1231_dp,  621.9390_dp,    0.0000_dp,    0.0000_dp, &
      & 1145.4236_dp,  810.2212_dp,  558.8413_dp,    0.0000_dp,    0.0000_dp, &
      & 1007.8507_dp,  718.6770_dp,  504.0290_dp,    0.0000_dp,    0.0000_dp, &
      & 1268.2622_dp,  894.7026_dp,  613.9793_dp,    0.0000_dp,    0.0000_dp, &
      & 1235.5723_dp,  873.2287_dp,  601.4334_dp,    0.0000_dp,    0.0000_dp, &
      & 1192.2239_dp,  844.5925_dp,  584.5218_dp,    0.0000_dp,    0.0000_dp, &
      & 1138.1053_dp,  808.6931_dp,  563.2228_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1145.5595_dp,  814.8114_dp,  568.3523_dp,    0.0000_dp,    0.0000_dp, &
      & 1142.8009_dp,  813.2828_dp,  567.8637_dp,    0.0000_dp,    0.0000_dp, &
      & 1130.1276_dp,  805.0118_dp,  563.1134_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1006.5104_dp,  722.4932_dp,  512.5389_dp,    0.0000_dp,    0.0000_dp, &
      & 1016.4155_dp,  729.4949_dp,  517.3851_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  880.8202_dp,  637.9590_dp,  459.5735_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5675.7434_dp, 3825.0881_dp, 2001.2557_dp,    0.0000_dp,    0.0000_dp, &
      & 1524.9399_dp, 1063.3274_dp,  702.1148_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4754.4826_dp, 3186.2921_dp, 1830.2684_dp,    0.0000_dp,    0.0000_dp, &
      & 3994.2273_dp, 2705.5816_dp, 1519.7933_dp,    0.0000_dp,    0.0000_dp, &
      & 1581.4671_dp, 1105.9643_dp,  739.9160_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3990.6172_dp, 2690.9591_dp, 1603.5452_dp,    0.0000_dp,    0.0000_dp, &
      & 2690.9591_dp, 1850.0003_dp, 1126.9136_dp,    0.0000_dp,    0.0000_dp, &
      & 1603.5452_dp, 1126.9136_dp,  766.0202_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1653 * maxRef * maxRef + 1 : 1672 * maxRef * maxRef) = [ &
      &   44.5104_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   71.8830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   29.3081_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  704.3345_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  234.5271_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  403.7427_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  298.8782_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  194.1728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  270.6187_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  227.4189_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  190.7463_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.6238_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  136.0287_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  182.2164_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  170.9262_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.1811_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  130.2341_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  106.3035_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  127.1355_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  118.5470_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.2827_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   97.9882_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   96.1189_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   86.1665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   75.9910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.7359_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   59.1379_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   55.9048_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  841.8986_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.1382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  644.2220_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.8650_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  345.1382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  592.4047_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  536.5626_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  417.2208_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  324.1533_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  465.2631_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  429.8074_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  399.5853_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.5669_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  319.6854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  361.8015_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  351.1585_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.4675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  321.1378_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  299.7447_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  296.4581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  290.4555_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  244.4468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  241.9223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  199.7138_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1381.0183_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  472.2455_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1672 * maxRef * maxRef + 1 : 1691 * maxRef * maxRef) = [ &
      & 1134.6748_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  907.4012_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  483.9715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  936.4068_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  471.1854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  464.4600_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  903.3306_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  454.8200_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  470.1341_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  826.7047_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.7570_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  430.3341_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  651.1440_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  441.7809_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  323.4660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  711.1423_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  405.8737_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  276.3955_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  558.0177_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  375.4568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  271.3545_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  590.7387_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  358.1677_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  318.8026_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  283.4108_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  608.9242_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  350.2054_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  296.8236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  312.1102_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  466.8110_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.4374_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  478.3350_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  324.3365_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  566.8783_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  532.0683_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  420.5733_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.2601_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  498.6284_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  466.1880_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  443.6388_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  397.0824_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  350.0197_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  424.2906_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  417.0350_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  405.9846_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  390.9033_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  380.1606_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  380.3160_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  377.3149_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  332.1884_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  334.5262_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  288.5010_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1547.0780_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  563.1480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1352.0803_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1104.4325_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  579.7357_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1691 * maxRef * maxRef + 1 : 1710 * maxRef * maxRef) = [ &
      & 1182.0124_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  840.8410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  556.9697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1060.9721_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  623.2841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  561.4468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  966.6201_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  629.4977_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.0506_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  745.1176_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  586.6430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  479.4247_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  831.8360_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  659.2348_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  422.6628_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  632.6639_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  510.5864_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  399.2351_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  691.5646_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  477.5945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  407.2558_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  640.9611_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  443.3629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  425.6143_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  534.5993_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  428.9980_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  564.3910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  423.8910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  709.5009_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  675.7911_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  542.3835_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  460.8857_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  654.6723_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.0042_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  589.4528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.1369_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  478.5981_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  582.1618_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  570.3846_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  554.4976_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  534.4801_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  539.4167_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  538.9780_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  534.5244_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  487.0333_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  491.6150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  437.2834_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1884.4554_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  664.8346_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1724.7415_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1433.0469_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  700.8041_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1512.2504_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1064.5933_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  725.6894_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1710 * maxRef * maxRef + 1 : 1729 * maxRef * maxRef) = [ &
      &  688.0353_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   88.6877_dp,   43.2202_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  152.4502_dp,   69.7341_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   53.1523_dp,   28.5060_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2210.3168_dp,  680.2479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  583.9797_dp,  227.0579_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1018.9540_dp,  390.7983_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  732.1642_dp,  289.4434_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  424.7883_dp,  188.2849_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.7615_dp,  262.3192_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  502.4967_dp,  220.5170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  413.1209_dp,  185.0194_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.4486_dp,  140.3959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.5743_dp,  132.0706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  379.1038_dp,  176.8339_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  357.1399_dp,  165.8805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  280.4649_dp,  135.1380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  262.6644_dp,  126.4572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  204.0039_dp,  103.2933_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  249.2633_dp,  123.4955_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.5783_dp,  115.1605_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.7820_dp,  108.1159_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  189.7084_dp,   95.2178_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  180.8383_dp,   93.4327_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  160.5233_dp,   83.7765_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.7472_dp,   73.9032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  132.1584_dp,   70.7481_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  103.3616_dp,   57.5599_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   98.8220_dp,   54.4061_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2609.0440_dp,  813.2353_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  873.5134_dp,  332.1495_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1687.5419_dp,  623.3223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1367.2714_dp,  513.7956_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  792.9063_dp,  334.4652_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1464.4446_dp,  573.5902_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1308.6041_dp,  519.6165_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  969.7366_dp,  404.2910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  705.7873_dp,  314.3449_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1061.5213_dp,  450.9253_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  970.3957_dp,  416.6272_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  891.9042_dp,  387.3888_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  878.3222_dp,  377.6480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  681.4251_dp,  310.1096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  773.1721_dp,  350.9578_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  749.1438_dp,  340.6470_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  742.6702_dp,  336.0824_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  676.2031_dp,  311.5824_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  614.8652_dp,  290.9369_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  608.9119_dp,  287.7449_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  596.1986_dp,  281.9238_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  482.2509_dp,  237.4072_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  477.7197_dp,  234.9544_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  381.0503_dp,  194.0698_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1729 * maxRef * maxRef + 1 : 1748 * maxRef * maxRef) = [ &
      & 4473.7784_dp, 1333.7958_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1162.2835_dp,  457.4338_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3190.2086_dp, 1097.1659_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2571.5319_dp,  877.5528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1119.4842_dp,  469.0498_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2554.7353_dp,  905.7241_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1079.9647_dp,  456.7087_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1040.8154_dp,  450.3076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2405.0561_dp,  874.0066_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1043.3187_dp,  440.8726_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1051.1001_dp,  455.8234_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2167.6325_dp,  800.0152_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1023.5395_dp,  432.0839_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  966.6619_dp,  417.2395_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1709.1137_dp,  630.1961_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1022.9773_dp,  428.1936_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  692.4667_dp,  313.8469_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1823.7858_dp,  688.3693_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  902.0482_dp,  393.5556_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  573.5714_dp,  268.2902_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1428.1039_dp,  540.2311_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  825.7735_dp,  364.1170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  565.2043_dp,  263.3851_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1457.0358_dp,  572.0738_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  788.5809_dp,  347.3566_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  697.7563_dp,  309.2299_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  599.8783_dp,  275.0230_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1525.4141_dp,  589.5695_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  773.5549_dp,  339.6232_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  639.6214_dp,  287.9649_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  694.6969_dp,  302.6820_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1172.9549_dp,  452.0342_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  788.3761_dp,  335.8606_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1139.4352_dp,  463.4148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  698.7243_dp,  314.6193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1372.8649_dp,  549.0692_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1268.2581_dp,  515.4512_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  949.6978_dp,  407.7143_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  733.0346_dp,  332.9869_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1133.1627_dp,  483.3174_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1044.8624_dp,  451.9571_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  984.5356_dp,  430.1513_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  859.9043_dp,  385.1320_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  734.6883_dp,  339.6203_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  912.5263_dp,  411.5533_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  894.2815_dp,  404.5340_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  865.7939_dp,  393.8448_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  826.4044_dp,  379.2572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  791.0214_dp,  368.9206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  791.8817_dp,  369.0697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  784.9774_dp,  366.1629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  668.3328_dp,  322.5272_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  674.2876_dp,  324.7894_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  563.0156_dp,  280.2438_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4971.0778_dp, 1494.4074_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1369.5245_dp,  545.5917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1748 * maxRef * maxRef + 1 : 1767 * maxRef * maxRef) = [ &
      & 3828.9626_dp, 1307.3665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3162.1198_dp, 1068.0684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1338.9789_dp,  561.9076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3181.4405_dp, 1143.5524_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2178.3229_dp,  814.0637_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1231.9890_dp,  540.0992_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2767.9894_dp, 1026.8226_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1442.9680_dp,  604.1250_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1227.1618_dp,  544.5203_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2472.6905_dp,  935.7467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1456.5113_dp,  610.1366_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1195.2654_dp,  526.6759_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1844.0715_dp,  721.6673_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1342.8116_dp,  568.7148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1019.8084_dp,  465.1818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2084.5975_dp,  805.5111_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1593.6331_dp,  638.7376_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  875.5927_dp,  410.2593_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1527.6056_dp,  612.9682_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1125.1127_dp,  495.2056_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  819.0425_dp,  387.5759_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1664.4640_dp,  669.9917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1050.0763_dp,  463.2412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  863.5344_dp,  395.2161_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1522.4806_dp,  621.0687_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  968.8758_dp,  430.0863_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  909.2919_dp,  412.9759_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1286.0441_dp,  518.0207_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  942.3547_dp,  416.1281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1318.2791_dp,  546.9938_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  902.6294_dp,  411.2941_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1728.2162_dp,  687.2667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1615.9882_dp,  654.7497_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1218.8063_dp,  525.8956_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  982.2388_dp,  447.1551_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1511.0459_dp,  634.5156_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1381.1677_dp,  591.3671_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1320.7683_dp,  571.5166_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1153.3721_dp,  514.1941_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1010.9240_dp,  464.3765_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1278.3406_dp,  564.5753_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1244.3690_dp,  553.2005_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1199.3914_dp,  537.8508_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1143.2636_dp,  518.5062_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1150.4200_dp,  523.3205_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1147.3718_dp,  522.9067_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1134.1681_dp,  518.6075_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1007.0198_dp,  472.7004_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1016.9588_dp,  477.1430_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  878.3798_dp,  424.5915_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 6073.8935_dp, 1820.4201_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1552.0845_dp,  644.3858_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4983.3126_dp, 1667.4685_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4213.9225_dp, 1385.6402_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1604.0906_dp,  679.3561_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1767 * maxRef * maxRef + 1 : 1786 * maxRef * maxRef) = [ &
      & 4150.3203_dp, 1462.8142_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2790.9891_dp, 1030.6163_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1619.6684_dp,  703.6339_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1527.1461_dp,  667.2018_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4342.2386_dp, 1476.7389_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1476.7389_dp,  647.0250_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   85.4080_dp,   42.3343_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  146.5851_dp,   68.2931_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   51.2952_dp,   27.9324_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2091.5068_dp,  665.5329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  558.6972_dp,  222.2883_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  974.6167_dp,  382.5760_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  700.4442_dp,  283.3715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  408.1203_dp,  184.3805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  586.1896_dp,  256.8646_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  482.5077_dp,  215.9438_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  396.7581_dp,  181.1913_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  282.4641_dp,  137.5122_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.3886_dp,  129.3614_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  364.6519_dp,  173.1933_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.4153_dp,  162.4665_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  269.9865_dp,  132.3691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  252.8197_dp,  123.8682_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  196.6165_dp,  101.1917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.1139_dp,  120.9759_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  224.0040_dp,  112.8134_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  208.8357_dp,  105.9150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  182.7560_dp,   93.2841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  174.3678_dp,   91.5407_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  154.8067_dp,   82.0849_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  134.8019_dp,   72.4156_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  127.5308_dp,   69.3261_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   99.8331_dp,   56.4117_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   95.4202_dp,   53.3201_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2470.6020_dp,  795.6873_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  834.9263_dp,  325.1636_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1611.7831_dp,  610.1625_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1305.9000_dp,  502.9655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  760.7352_dp,  327.4991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1401.3191_dp,  561.5498_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1252.6928_dp,  508.7250_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  929.6042_dp,  395.8577_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  678.1931_dp,  307.8310_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1018.5098_dp,  441.5369_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.2917_dp,  407.9629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  856.2817_dp,  379.3429_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  842.8268_dp,  369.7985_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  655.1209_dp,  303.6993_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  743.2407_dp,  343.7019_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  720.1450_dp,  333.6065_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  713.7913_dp,  329.1343_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  650.2639_dp,  305.1528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  591.6752_dp,  284.9529_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  585.9134_dp,  281.8265_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  573.6840_dp,  276.1263_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1786 * maxRef * maxRef + 1 : 1805 * maxRef * maxRef) = [ &
      &  464.5062_dp,  232.5513_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  460.1250_dp,  230.1488_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  367.3163_dp,  190.1224_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4216.3998_dp, 1304.7563_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1111.2638_dp,  447.8636_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3036.3662_dp, 1073.8235_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2442.0495_dp,  858.8383_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1073.3952_dp,  459.2919_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2435.0889_dp,  886.5266_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1036.2055_dp,  447.2220_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  998.9971_dp,  440.9674_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2294.2857_dp,  855.5354_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1000.2155_dp,  431.7097_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1008.9483_dp,  446.3717_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2068.9809_dp,  783.1386_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  981.3457_dp,  423.1055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  927.5254_dp,  408.5880_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1630.6018_dp,  616.9100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  980.4829_dp,  419.2898_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  665.3571_dp,  307.3826_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1742.1942_dp,  673.8862_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  866.0442_dp,  385.4043_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  551.6130_dp,  262.7854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1363.8456_dp,  528.8780_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  793.0239_dp,  356.5850_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.5117_dp,  257.9795_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1394.0822_dp,  560.0917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  757.1711_dp,  340.1719_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  669.9835_dp,  302.8447_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  576.6205_dp,  269.3672_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1458.7399_dp,  577.1994_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  742.6757_dp,  332.5971_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  614.5241_dp,  282.0297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  666.7343_dp,  296.4247_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1120.9931_dp,  442.5579_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  756.1577_dp,  328.8954_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1091.5722_dp,  453.7462_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  671.4732_dp,  308.1251_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1314.2306_dp,  537.5790_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1214.7867_dp,  504.6830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.1851_dp,  399.2448_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  704.6787_dp,  326.1098_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1087.2533_dp,  473.2673_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1002.9138_dp,  442.5728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  945.3065_dp,  421.2295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  826.2471_dp,  377.1647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  706.5967_dp,  332.6173_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  877.0066_dp,  403.0441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  859.5248_dp,  396.1729_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  832.2670_dp,  385.7096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  794.6136_dp,  371.4307_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  760.8949_dp,  361.3222_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  761.6998_dp,  361.4679_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  755.0713_dp,  358.6221_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  643.4305_dp,  315.9132_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  649.1282_dp,  318.1278_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1805 * maxRef * maxRef + 1 : 1824 * maxRef * maxRef) = [ &
      &  542.4383_dp,  274.5224_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4685.0846_dp, 1461.9055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1309.7014_dp,  534.1947_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3642.0492_dp, 1279.5273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2999.9283_dp, 1045.2541_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1283.6521_dp,  550.2241_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3032.9183_dp, 1119.3527_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2074.8807_dp,  796.8844_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1182.9141_dp,  528.9145_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2642.0526_dp, 1005.1742_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1383.2190_dp,  591.5630_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1178.7222_dp,  533.2574_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2361.6974_dp,  916.0648_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1397.0298_dp,  597.4603_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1147.4757_dp,  515.7797_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1763.1139_dp,  706.5573_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1287.2763_dp,  556.9066_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  980.1462_dp,  455.5995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1992.1534_dp,  788.6151_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1523.8714_dp,  625.3966_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  842.1651_dp,  401.8371_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1461.6458_dp,  600.1777_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1080.1628_dp,  484.9704_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  787.9846_dp,  379.6309_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1593.3696_dp,  656.0092_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1008.0521_dp,  453.6734_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  829.8526_dp,  387.0905_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1458.1671_dp,  608.1289_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  930.2250_dp,  421.2135_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  873.7547_dp,  404.4764_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1230.4969_dp,  507.2269_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  904.6688_dp,  407.5420_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1263.3482_dp,  535.6228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  867.5938_dp,  402.8281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1653.5039_dp,  672.8996_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1547.2492_dp,  641.0897_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1169.3656_dp,  514.9974_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  944.1820_dp,  437.9410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1448.8711_dp,  621.3216_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1325.1151_dp,  579.0951_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1267.6087_dp,  559.6696_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1107.9903_dp,  503.5686_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  972.0426_dp,  454.8102_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1227.7455_dp,  552.8934_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1195.3394_dp,  541.7614_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1152.4260_dp,  526.7389_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1098.8818_dp,  507.8063_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1105.8143_dp,  512.5245_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1102.9411_dp,  512.1209_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1090.3536_dp,  507.9139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  968.7947_dp,  462.9825_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  978.3471_dp,  467.3327_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  845.6703_dp,  415.8938_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1824 * maxRef * maxRef + 1 : 1843 * maxRef * maxRef) = [ &
      & 5716.7999_dp, 1780.7379_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1486.4897_dp,  630.9699_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4733.4185_dp, 1631.8559_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3987.3600_dp, 1355.8874_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1537.9273_dp,  665.2422_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3951.9956_dp, 1431.7796_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2655.1171_dp, 1008.8176_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1554.6376_dp,  689.0464_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1465.9695_dp,  653.3875_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4125.9023_dp, 1445.2611_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1417.6855_dp,  633.6333_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3924.4211_dp, 1387.5284_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1387.5284_dp,  620.5206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   83.3310_dp,   41.0356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  142.9620_dp,   66.1439_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   50.0748_dp,   27.1150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2031.2740_dp,  642.1808_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  544.1232_dp,  214.9157_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  949.1255_dp,  369.8142_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  682.1901_dp,  274.0453_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  397.9479_dp,  178.5085_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  571.4113_dp,  248.6071_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  470.4029_dp,  209.0627_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  386.8363_dp,  175.4697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  275.5610_dp,  133.2643_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  256.9654_dp,  125.3807_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  355.6712_dp,  167.7968_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  334.9245_dp,  157.4078_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.3922_dp,  128.3013_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  246.6348_dp,  120.0668_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  191.8733_dp,   98.1467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  234.2913_dp,  117.3026_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  218.5611_dp,  109.3944_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.7729_dp,  102.7150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  178.3241_dp,   90.4807_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  170.1817_dp,   88.8159_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  151.0970_dp,   79.6566_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.5785_dp,   70.2898_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.4946_dp,   67.2999_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   97.4789_dp,   54.7947_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.1635_dp,   51.7860_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2399.8207_dp,  767.8676_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  812.9350_dp,  314.3316_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1568.9885_dp,  589.6141_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1271.2883_dp,  486.1582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  741.4881_dp,  316.8969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1364.8494_dp,  542.9683_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1220.2246_dp,  491.9700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  905.8860_dp,  383.0238_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  661.3178_dp,  298.0452_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1843 * maxRef * maxRef + 1 : 1862 * maxRef * maxRef) = [ &
      &  992.7507_dp,  427.2858_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  907.7970_dp,  394.8501_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  834.7641_dp,  367.1968_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  821.5324_dp,  357.9364_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  638.9063_dp,  294.1261_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.8197_dp,  332.8624_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  702.2959_dp,  323.0966_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  696.0628_dp,  318.7539_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  634.2089_dp,  295.5871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  577.1696_dp,  276.1129_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  571.5397_dp,  273.0819_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.6108_dp,  267.5627_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  453.2313_dp,  225.4562_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  448.9517_dp,  223.1261_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  358.4721_dp,  184.4110_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4092.0575_dp, 1259.0280_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1082.0865_dp,  433.2074_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2952.9794_dp, 1037.1340_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2374.0445_dp,  829.6408_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1046.0388_dp,  444.4629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2369.1244_dp,  856.4434_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1009.8893_dp,  432.8276_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  973.8228_dp,  426.8683_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2232.6503_dp,  826.7257_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  974.7227_dp,  417.8360_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  983.5427_dp,  432.1120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2013.7194_dp,  756.8843_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  956.3321_dp,  409.5058_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  904.0753_dp,  395.5412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1586.8512_dp,  596.2971_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  955.4042_dp,  405.7835_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  648.7727_dp,  297.7522_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1696.0441_dp,  651.4438_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  844.2856_dp,  373.1223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  537.9977_dp,  254.6469_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1327.6064_dp,  511.3384_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  773.1575_dp,  345.2676_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.0826_dp,  249.9799_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1357.7395_dp,  541.6401_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  738.1748_dp,  329.3812_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  653.1705_dp,  293.2805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  562.3122_dp,  260.9610_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1420.5045_dp,  558.0904_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.0207_dp,  322.0375_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  599.1969_dp,  273.1670_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  649.9094_dp,  287.0163_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1091.4214_dp,  427.9630_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  736.9509_dp,  318.3642_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1063.4898_dp,  438.9546_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  654.7776_dp,  298.4106_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1280.1762_dp,  519.9536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1183.4931_dp,  488.2162_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  888.1417_dp,  386.4453_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  687.2223_dp,  315.8374_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1059.7536_dp,  458.0385_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  977.6523_dp,  428.4016_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  921.5753_dp,  407.7871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  805.6691_dp,  365.2295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  689.1790_dp,  322.2035_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1862 * maxRef * maxRef + 1 : 1881 * maxRef * maxRef) = [ &
      &  855.2177_dp,  390.3161_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  838.1837_dp,  383.6775_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  811.6356_dp,  373.5692_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  774.9718_dp,  359.7752_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  742.1649_dp,  350.0554_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  742.9435_dp,  350.1952_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  736.4811_dp,  347.4428_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  627.7342_dp,  306.1959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  633.2835_dp,  308.3361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  529.3070_dp,  266.1893_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4547.0720_dp, 1410.8565_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1275.3968_dp,  516.8026_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3541.4930_dp, 1235.8086_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2915.7949_dp, 1009.7020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1250.8766_dp,  532.4996_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2950.9303_dp, 1081.5963_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2018.9224_dp,  770.4895_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1153.2166_dp,  512.0867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2571.4964_dp,  971.5644_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1347.7804_dp,  572.5226_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1149.2501_dp,  516.3573_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2299.0481_dp,  885.6282_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1361.3178_dp,  578.2213_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1118.6278_dp,  499.4343_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1716.8038_dp,  683.3681_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1254.4039_dp,  539.0640_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  955.7891_dp,  441.3364_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1939.6215_dp,  762.6134_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1483.9817_dp,  605.0829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  821.4043_dp,  389.3842_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1423.5531_dp,  580.6563_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1052.9974_dp,  469.6134_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  768.6159_dp,  367.9133_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1552.0522_dp,  634.6293_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  982.6875_dp,  439.3361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  809.2032_dp,  375.0228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1420.5469_dp,  588.3919_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  906.8511_dp,  407.9420_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  851.9952_dp,  391.8190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1198.4194_dp,  490.7833_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  881.9071_dp,  394.6814_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1230.9598_dp,  518.3356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  846.0616_dp,  390.2136_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1610.4142_dp,  650.8890_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1507.2236_dp,  620.2296_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1139.7842_dp,  498.5669_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  920.7728_dp,  424.1984_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1411.9588_dp,  601.2862_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1291.5707_dp,  560.5406_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1235.6389_dp,  541.7976_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1080.3308_dp,  487.6436_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  948.0215_dp,  440.5691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1197.0157_dp,  535.3426_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1165.4811_dp,  524.6028_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1123.7195_dp,  510.1049_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1071.6137_dp,  491.8297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1881 * maxRef * maxRef + 1 : 1900 * maxRef * maxRef) = [ &
      & 1078.3854_dp,  496.4206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1075.5989_dp,  496.0396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1063.3516_dp,  491.9827_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  944.9814_dp,  448.6020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  954.2967_dp,  452.8135_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  825.0473_dp,  403.1233_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5547.4900_dp, 1718.6565_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1448.1464_dp,  610.6517_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4601.2171_dp, 1575.9396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3873.7284_dp, 1309.6071_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1498.6941_dp,  643.9008_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3844.0367_dp, 1383.3184_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2582.9722_dp,  975.3429_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1515.4611_dp,  667.0629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1429.0690_dp,  632.6069_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4011.3635_dp, 1395.9648_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1382.0320_dp,  613.5022_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3816.1118_dp, 1340.2848_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1352.6449_dp,  600.8101_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3710.9375_dp, 1306.6123_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1306.6123_dp,  581.7435_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.4123_dp,   41.1186_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.6254_dp,   66.4057_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   48.9406_dp,   27.0910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1976.6955_dp,  653.7125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.7909_dp,  216.8695_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  925.8158_dp,  373.3538_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  665.4868_dp,  276.3828_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  388.5925_dp,  179.3900_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  557.8380_dp,  250.0690_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  459.2753_dp,  210.1448_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  377.7092_dp,  176.2659_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  269.1899_dp,  133.6061_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  251.0346_dp,  125.6675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  347.3921_dp,  168.3507_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.0991_dp,  157.9406_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  257.3029_dp,  128.5963_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  240.9233_dp,  120.3408_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  187.4818_dp,   98.2270_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  228.9065_dp,  117.4740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.5277_dp,  109.5523_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  199.0885_dp,  102.8398_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  174.2214_dp,   90.5682_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  166.3004_dp,   88.8332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.6552_dp,   79.6477_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  128.5846_dp,   70.2511_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  121.6727_dp,   67.2415_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   95.2846_dp,   54.6852_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1900 * maxRef * maxRef + 1 : 1919 * maxRef * maxRef) = [ &
      &   91.0614_dp,   51.6977_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2335.6474_dp,  781.2050_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  792.8336_dp,  317.4096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1529.9120_dp,  596.0022_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1239.6749_dp,  491.1385_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  723.8209_dp,  318.9485_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1331.4786_dp,  547.7742_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1190.5003_dp,  496.0959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  884.1322_dp,  385.6361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  645.7939_dp,  299.4728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  969.1026_dp,  429.9494_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  886.2193_dp,  397.1752_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  814.9918_dp,  369.2253_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  801.9751_dp,  360.0151_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  623.9763_dp,  295.3345_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  707.8604_dp,  334.2517_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  685.8620_dp,  324.4246_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  679.7433_dp,  320.1057_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  619.4176_dp,  296.6816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  563.7890_dp,  276.9116_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  558.2816_dp,  273.8804_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  546.6292_dp,  268.3382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  442.8095_dp,  225.8370_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  438.6243_dp,  223.5079_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  350.2812_dp,  184.5332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3979.6309_dp, 1283.6315_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1055.3891_dp,  436.8854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2877.0680_dp, 1050.9848_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2312.2147_dp,  841.0519_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1020.9351_dp,  447.3826_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2308.9962_dp,  866.9228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  985.7264_dp,  435.5172_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  950.6909_dp,  429.2527_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2176.4184_dp,  836.0994_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  951.3257_dp,  420.4671_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  960.1954_dp,  434.4919_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1963.2723_dp,  765.0512_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  933.3738_dp,  412.0874_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  882.5313_dp,  397.7647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1546.9195_dp,  602.7113_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  932.3946_dp,  408.4460_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  633.5005_dp,  298.9583_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1653.8773_dp,  657.9635_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  824.2830_dp,  375.1051_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  525.4406_dp,  255.4370_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1294.4951_dp,  516.3770_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  754.8861_dp,  346.9868_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  517.6989_dp,  250.7830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1324.4770_dp,  546.3363_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  720.7052_dp,  331.0282_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  637.7039_dp,  294.6745_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  549.1278_dp,  261.9299_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1385.5313_dp,  563.2255_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  706.8671_dp,  323.6772_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  585.0856_dp,  274.3346_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  634.4415_dp,  288.5206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1919 * maxRef * maxRef + 1 : 1938 * maxRef * maxRef) = [ &
      & 1064.3798_dp,  431.9033_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  719.3118_dp,  320.2684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1037.7490_dp,  442.2545_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  639.4075_dp,  299.7092_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1248.9915_dp,  524.1676_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1154.8172_dp,  491.9157_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  866.9787_dp,  388.7055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  671.1482_dp,  317.1445_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1034.5024_dp,  460.8179_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  954.4431_dp,  430.8119_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  899.7622_dp,  409.9536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  786.7340_dp,  366.8929_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  673.1287_dp,  323.3676_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  835.1622_dp,  392.0139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  818.5381_dp,  385.3099_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  792.6385_dp,  375.0947_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  756.8785_dp,  361.1491_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.8987_dp,  351.2182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  725.6536_dp,  351.3644_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  719.3436_dp,  348.5929_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  613.2405_dp,  306.8959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  618.6540_dp,  309.0575_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  517.1617_dp,  266.5494_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4422.2696_dp, 1437.9300_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1243.9950_dp,  520.9703_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3449.9906_dp, 1252.6342_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2839.3501_dp, 1024.0101_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1220.8006_dp,  535.9497_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2876.1716_dp, 1094.2695_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1967.8827_dp,  778.5614_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1125.9127_dp,  514.7308_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2507.0805_dp,  981.8553_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1315.2670_dp,  576.2453_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1122.1393_dp,  518.8345_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2241.8085_dp,  894.3917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1328.5415_dp,  581.9305_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1092.1025_dp,  501.9170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1674.4360_dp,  689.3091_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1224.2314_dp,  542.3803_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  933.3543_dp,  443.0547_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1891.5883_dp,  769.5990_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1447.4665_dp,  609.8655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  802.2571_dp,  390.5841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1388.6668_dp,  585.2065_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1028.0158_dp,  471.9441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  750.7432_dp,  368.9368_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1514.2016_dp,  639.5705_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  959.3598_dp,  441.4769_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  790.1791_dp,  376.4231_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1386.0622_dp,  592.7151_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  885.3477_dp,  409.8444_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  831.9541_dp,  393.3803_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1169.0360_dp,  494.5443_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  860.9698_dp,  396.5804_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1938 * maxRef * maxRef + 1 : 1957 * maxRef * maxRef) = [ &
      & 1201.2473_dp,  521.8571_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  826.2261_dp,  391.7491_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1570.9643_dp,  656.2319_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1470.5492_dp,  624.9431_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1112.6065_dp,  501.3687_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  899.2107_dp,  425.8977_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1378.0830_dp,  605.1903_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1260.7608_dp,  563.8386_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1206.2616_dp,  544.8078_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1054.8803_dp,  489.9104_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  925.8883_dp,  442.2254_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1168.7524_dp,  537.9899_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1138.0116_dp,  527.0919_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1097.2997_dp,  512.3917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1046.5046_dp,  493.8687_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1053.1255_dp,  498.4291_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1050.4168_dp,  498.0193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1038.4792_dp,  493.8976_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  923.0194_dp,  449.9920_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  932.1166_dp,  454.2236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  806.0002_dp,  404.0174_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5394.4846_dp, 1752.1410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1412.9908_dp,  614.7789_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4481.0449_dp, 1598.6396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3770.6523_dp, 1329.7509_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1462.6834_dp,  647.8742_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3745.6921_dp, 1400.5068_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2517.2239_dp,  986.0439_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1479.4584_dp,  670.6863_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1395.1471_dp,  635.9157_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3907.1923_dp, 1415.2709_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1349.2524_dp,  616.6572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3717.5407_dp, 1358.1904_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1320.5717_dp,  603.8868_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3615.2067_dp, 1323.9329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1275.6493_dp,  584.6788_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3522.0508_dp, 1292.4475_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1292.4475_dp,  587.7891_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.7138_dp,   39.2564_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  136.6695_dp,   63.7629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   47.9365_dp,   25.6913_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1928.0810_dp,  653.6846_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  518.9498_dp,  211.7400_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  905.1101_dp,  365.1112_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  650.6557_dp,  269.2305_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  380.3011_dp,  172.7866_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1957 * maxRef * maxRef + 1 : 1976 * maxRef * maxRef) = [ &
      &  545.8015_dp,  241.6343_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  449.4102_dp,  202.6148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  369.6195_dp,  169.5824_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  263.5484_dp,  127.7725_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  245.7832_dp,  120.0875_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  340.0580_dp,  161.3933_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  320.1657_dp,  151.4713_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  251.9099_dp,  122.9404_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  235.8641_dp,  115.0733_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  183.5940_dp,   93.5193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  224.1383_dp,  112.0512_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  209.0700_dp,  104.5097_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  194.9397_dp,   98.0628_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  170.5870_dp,   86.3386_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  162.8638_dp,   84.4731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.6075_dp,   75.6908_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  125.9329_dp,   66.7166_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  119.1734_dp,   63.7996_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   93.3410_dp,   51.7557_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   89.1988_dp,   48.9809_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2278.4898_dp,  780.2674_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  774.9719_dp,  310.6864_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1495.1792_dp,  585.2367_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1211.5825_dp,  481.1216_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  708.1495_dp,  308.8219_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1301.8438_dp,  534.3757_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1164.1089_dp,  483.2415_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  864.8322_dp,  373.6925_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  632.0359_dp,  288.3313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  948.1288_dp,  415.9150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  867.0844_dp,  383.7764_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  797.4609_dp,  356.3677_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  784.6310_dp,  347.8087_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.7486_dp,  283.7443_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  692.8338_dp,  321.2120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  671.3009_dp,  311.7158_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  665.2819_dp,  307.7168_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  606.3141_dp,  284.7085_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  551.9390_dp,  265.0869_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  546.5395_dp,  262.2219_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  535.1318_dp,  256.9068_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  433.5831_dp,  215.4603_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  429.4810_dp,  213.2614_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.0307_dp,  175.5908_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3879.4729_dp, 1285.6965_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1031.6716_dp,  425.9370_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2809.5233_dp, 1039.4662_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2257.1993_dp,  831.2750_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  998.6585_dp,  433.5252_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2255.5157_dp,  854.8216_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  964.2855_dp,  421.6969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  930.1750_dp,  414.5893_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2126.4190_dp,  822.1810_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  930.5654_dp,  407.0094_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  939.4890_dp,  419.5622_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1918.4254_dp,  751.0893_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  913.0014_dp,  398.9816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  863.4210_dp,  384.2576_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1976 * maxRef * maxRef + 1 : 1995 * maxRef * maxRef) = [ &
      & 1511.4156_dp,  591.6660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.9735_dp,  395.7987_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  619.9599_dp,  287.5236_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1616.4023_dp,  644.4421_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  806.5432_dp,  362.0493_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  514.3112_dp,  244.9612_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1265.0643_dp,  505.6064_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  738.6836_dp,  334.5615_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  506.7226_dp,  240.5922_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1294.9313_dp,  533.0040_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  705.2129_dp,  319.1993_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  623.9865_dp,  284.0111_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  537.4395_dp,  251.6701_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1354.4595_dp,  550.3952_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  691.6538_dp,  312.2221_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  572.5728_dp,  264.0373_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  620.7184_dp,  278.5957_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1040.3490_dp,  422.1376_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  703.6592_dp,  309.8959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1014.8959_dp,  429.9304_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  625.7820_dp,  288.4004_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1221.3030_dp,  510.2157_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1129.3623_dp,  478.0396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  848.2084_dp,  375.6249_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  656.9043_dp,  304.8174_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1012.1053_dp,  445.6375_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  933.8615_dp,  416.0190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  880.4215_dp,  395.4915_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  769.9512_dp,  353.0881_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  658.9099_dp,  310.2550_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  817.3888_dp,  376.9991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  801.1286_dp,  370.4418_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  775.8051_dp,  360.4255_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  740.8479_dp,  346.7365_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  709.6039_dp,  336.6910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  710.3376_dp,  336.8539_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  704.1627_dp,  334.1734_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  600.4064_dp,  293.3036_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  605.6994_dp,  295.4210_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  506.4098_dp,  254.0859_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4311.0961_dp, 1438.5445_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1216.1018_dp,  507.2285_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3368.5646_dp, 1239.5504_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2771.3234_dp, 1012.8078_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1194.1104_dp,  519.2271_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2809.6885_dp, 1077.0858_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1922.5226_dp,  762.0033_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1101.7008_dp,  496.5039_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2449.8188_dp,  963.2715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1286.4106_dp,  558.2951_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1098.1031_dp,  499.8637_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2190.9387_dp,  875.5931_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1299.4488_dp,  563.9816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1068.5805_dp,  483.8542_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1636.7956_dp,  672.4914_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1197.4564_dp,  524.9085_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  913.4679_dp,  425.7815_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(1995 * maxRef * maxRef + 1 : 2014 * maxRef * maxRef) = [ &
      & 1848.9097_dp,  751.7567_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1415.0358_dp,  593.2386_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  785.2898_dp,  374.4473_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1357.6810_dp,  569.5060_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1005.8586_dp,  455.1389_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  734.9066_dp,  353.3996_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1480.5878_dp,  622.3214_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  938.6688_dp,  425.6722_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  773.3122_dp,  361.7114_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1355.4425_dp,  575.9965_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  866.2748_dp,  394.9658_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  814.1845_dp,  378.3019_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1142.9352_dp,  481.1524_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  842.3966_dp,  382.4279_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1174.8699_dp,  506.3303_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  808.6413_dp,  376.6190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1535.9254_dp,  639.2239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1437.9844_dp,  607.6149_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1088.4965_dp,  484.4102_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  880.0979_dp,  409.4190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1348.0216_dp,  586.3317_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1233.4280_dp,  545.1890_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1180.2039_dp,  526.2392_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1032.3159_dp,  471.8373_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  906.2740_dp,  424.6928_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1143.6919_dp,  518.5975_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1113.6574_dp,  507.7568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1073.8792_dp,  493.1741_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1024.2500_dp,  474.8294_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1030.7380_dp,  479.0649_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1028.0990_dp,  478.5801_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1016.4371_dp,  474.4660_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  903.5630_dp,  431.2267_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  912.4671_dp,  435.2963_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  789.1320_dp,  386.1460_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5258.2068_dp, 1751.8319_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1381.7863_dp,  595.9499_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4374.0826_dp, 1584.5929_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3678.9146_dp, 1317.6067_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1430.7306_dp,  626.9684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3658.2120_dp, 1380.9239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2458.7886_dp,  965.8237_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1447.5288_dp,  647.4730_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1365.0615_dp,  613.6537_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3814.4921_dp, 1400.3880_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1320.1809_dp,  594.9070_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3629.8268_dp, 1342.9247_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1292.1268_dp,  582.5641_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3530.0228_dp, 1308.7424_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1248.1896_dp,  563.9015_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2014 * maxRef * maxRef + 1 : 2033 * maxRef * maxRef) = [ &
      & 3439.1601_dp, 1277.3568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1264.5213_dp,  567.3171_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3358.3122_dp, 1249.5054_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1249.5054_dp,  549.0281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   64.1880_dp,   41.3275_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  108.3662_dp,   66.8235_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   39.5786_dp,   27.1595_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1407.7335_dp,  659.0924_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  396.7223_dp,  218.6195_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  689.9287_dp,  376.4610_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  499.1837_dp,  278.5080_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  299.3233_dp,  180.6115_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  426.4897_dp,  251.8424_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  352.9087_dp,  211.5485_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  291.6491_dp,  177.3613_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  211.2565_dp,  134.3264_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.4553_dp,  126.3205_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  270.9183_dp,  169.3159_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  254.8122_dp,  158.8301_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  202.2978_dp,  129.2461_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  189.3824_dp,  120.9324_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  149.2918_dp,   98.6213_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.2581_dp,  117.9977_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.0556_dp,  110.0265_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  157.9221_dp,  103.2648_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  138.4298_dp,   90.9100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  133.1438_dp,   89.1359_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  118.5388_dp,   79.8949_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  103.6007_dp,   70.4331_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   98.3573_dp,   67.4030_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   77.8742_dp,   54.7600_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   74.1912_dp,   51.7756_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1667.3895_dp,  787.6187_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  589.6821_dp,  319.9572_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1130.8198_dp,  601.0384_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  919.6329_dp,  495.1147_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  551.1193_dp,  321.3077_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  996.7924_dp,  552.1545_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  893.9515_dp,  499.9820_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  671.2900_dp,  388.4558_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  498.0207_dp,  301.4874_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  739.0200_dp,  433.0662_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  677.4806_dp,  399.9816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  624.7321_dp,  371.7819_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  613.2627_dp,  362.5021_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  483.6888_dp,  297.2065_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.3633_dp,  336.3700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  531.5424_dp,  326.4592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  526.1827_dp,  322.1177_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  481.6031_dp,  298.4773_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  441.2303_dp,  278.4551_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  436.7785_dp,  275.4054_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  427.7338_dp,  269.8237_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  350.0369_dp,  226.9109_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  346.6461_dp,  224.5698_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2033 * maxRef * maxRef + 1 : 2052 * maxRef * maxRef) = [ &
      &  279.3515_dp,  185.2621_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2817.6333_dp, 1292.7733_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  790.7854_dp,  439.9765_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2094.5326_dp, 1059.7192_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1680.6338_dp,  847.4486_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  776.0695_dp,  450.5280_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1691.7052_dp,  874.1314_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  750.7848_dp,  438.5550_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  728.1122_dp,  432.1441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1602.8927_dp,  842.8840_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.5886_dp,  423.3069_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  735.7972_dp,  437.4020_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1450.7174_dp,  771.1840_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  710.7237_dp,  414.8741_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  675.5150_dp,  400.3793_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1142.7015_dp,  607.3487_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  708.5769_dp,  411.2230_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  490.5998_dp,  300.6688_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1228.0426_dp,  663.1313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  632.5743_dp,  377.5747_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  410.0863_dp,  256.7746_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  961.5600_dp,  520.2569_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  580.7773_dp,  349.2095_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  403.6739_dp,  252.1075_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  992.0673_dp,  550.5045_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  554.3462_dp,  333.1240_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  491.2047_dp,  296.4553_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  426.4723_dp,  263.3886_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1034.1932_dp,  567.6026_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.2742_dp,  325.7328_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  452.3968_dp,  275.9455_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  486.7180_dp,  290.3039_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  793.8763_dp,  435.0829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.9119_dp,  322.3728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  783.5114_dp,  445.4982_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  494.4243_dp,  301.5583_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  939.3652_dp,  528.1305_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  871.6966_dp,  495.5668_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  662.7701_dp,  391.3632_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  520.0484_dp,  319.1288_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  789.6369_dp,  464.0671_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  730.9227_dp,  433.7675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  690.6823_dp,  412.7144_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  607.5073_dp,  369.2466_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  523.7968_dp,  325.3103_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  645.9849_dp,  394.5033_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  633.5809_dp,  387.7336_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  614.3654_dp,  377.4212_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.9119_dp,  363.3458_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  565.2991_dp,  353.2521_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  565.7914_dp,  353.3989_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  560.9891_dp,  350.6032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  482.3172_dp,  308.4763_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  486.3444_dp,  310.6569_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  409.9803_dp,  267.7464_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2052 * maxRef * maxRef + 1 : 2071 * maxRef * maxRef) = [ &
      & 3135.9242_dp, 1447.9360_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  934.6082_dp,  524.5374_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2507.7589_dp, 1262.8491_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2059.4692_dp, 1031.5808_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  928.2866_dp,  539.6227_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2113.2076_dp, 1103.0742_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1458.1969_dp,  784.0479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  865.0011_dp,  518.0912_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1854.4837_dp,  989.5940_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  999.3615_dp,  580.1629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  864.5896_dp,  522.1512_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1665.4372_dp,  901.2747_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1009.4419_dp,  585.9401_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  840.0098_dp,  505.0620_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1253.0730_dp,  694.2958_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  932.8332_dp,  545.9255_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  723.9342_dp,  445.6125_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1411.6759_dp,  775.2973_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1088.9839_dp,  613.9817_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  626.3824_dp,  392.6667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1045.0617_dp,  589.2413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  790.5176_dp,  474.8590_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.6156_dp,  370.8350_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1140.4525_dp,  644.1313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  738.1317_dp,  444.1417_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  613.3959_dp,  378.4806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1047.1279_dp,  596.8654_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  682.2073_dp,  412.2532_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  644.6362_dp,  395.6090_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  880.5313_dp,  497.8410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  662.6029_dp,  398.9285_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.0796_dp,  525.4121_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  640.7393_dp,  394.0161_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1179.9720_dp,  660.9815_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1109.2250_dp,  629.3894_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  851.5816_dp,  504.6237_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  697.1742_dp,  428.4350_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1048.0976_dp,  609.4014_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  963.2234_dp,  567.6368_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  923.8826_dp,  548.4168_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  813.7102_dp,  492.9882_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  719.4208_dp,  444.8426_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  899.5162_dp,  541.4600_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  877.2337_dp,  530.4464_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  847.6165_dp,  515.5966_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  810.5952_dp,  496.8907_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  816.3047_dp,  501.4441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  814.5718_dp,  501.0205_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  805.9652_dp,  496.8526_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  720.9847_dp,  452.4916_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  728.0053_dp,  456.7532_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  634.2163_dp,  406.0479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3823.8426_dp, 1763.6917_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1071.5717_dp,  618.8508_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2071 * maxRef * maxRef + 1 : 2090 * maxRef * maxRef) = [ &
      & 3243.6561_dp, 1611.3765_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2720.7653_dp, 1339.0290_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1114.5114_dp,  652.1810_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2740.5478_dp, 1411.6458_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1860.7370_dp,  992.8293_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1134.1326_dp,  675.1196_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1070.9597_dp,  639.9970_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2838.1357_dp, 1426.3405_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1036.4403_dp,  620.5881_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2705.6826_dp, 1368.9945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1014.5593_dp,  607.7340_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2632.6657_dp, 1334.4823_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  980.6114_dp,  588.3813_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2566.0607_dp, 1302.7626_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  991.5176_dp,  591.5368_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2506.8352_dp, 1274.6283_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  974.0690_dp,  571.0450_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1891.6719_dp,  999.0617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  999.0617_dp,  595.3628_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   70.1794_dp,   38.9150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  120.4056_dp,   62.7971_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.3679_dp,   25.6695_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1802.3852_dp,  612.8166_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  463.5442_dp,  204.5514_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  808.4866_dp,  352.0753_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  582.1224_dp,  260.7245_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  335.3665_dp,  169.5717_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  482.7977_dp,  236.2659_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  397.4499_dp,  198.6025_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  327.2333_dp,  166.6202_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  232.1028_dp,  126.4184_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.4708_dp,  118.9198_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  299.8469_dp,  159.2422_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  282.8547_dp,  149.3811_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  222.0934_dp,  121.6899_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  208.1604_dp,  113.8754_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  161.9091_dp,   93.0070_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  197.6234_dp,  111.2043_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  184.5904_dp,  103.7034_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  172.0857_dp,   97.3591_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  150.8204_dp,   85.7475_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  143.7754_dp,   84.1363_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  127.7828_dp,   75.4465_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.4277_dp,   66.5559_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  105.3970_dp,   63.7152_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   82.7154_dp,   51.8414_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.0484_dp,   49.0050_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2090 * maxRef * maxRef + 1 : 2109 * maxRef * maxRef) = [ &
      & 2124.0831_dp,  732.6457_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.0211_dp,  299.2468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1343.3651_dp,  561.5991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1089.8214_dp,  462.8827_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  626.4637_dp,  301.2799_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1162.1560_dp,  516.7263_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1038.1047_dp,  468.0868_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  768.1313_dp,  364.1571_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  557.1841_dp,  283.1061_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  839.1776_dp,  406.1522_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  767.3055_dp,  375.2459_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  705.0006_dp,  348.9033_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.1617_dp,  340.1321_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  538.2012_dp,  279.2705_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.8318_dp,  316.0555_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  592.0135_dp,  306.7678_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  587.0972_dp,  302.6610_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  534.4195_dp,  280.5856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  486.2319_dp,  261.9772_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  481.5894_dp,  259.1043_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  471.5858_dp,  253.8632_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  381.9584_dp,  213.7617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  378.3984_dp,  211.5547_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  302.4149_dp,  174.7380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3687.6305_dp, 1201.3307_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  927.3672_dp,  412.0655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2564.1443_dp,  988.5095_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2081.6173_dp,  790.5227_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  887.0421_dp,  422.5156_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2045.4927_dp,  816.0391_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  855.1471_dp,  411.3969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  823.6822_dp,  405.6040_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1922.8302_dp,  787.4378_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  827.7970_dp,  397.1190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  831.7726_dp,  410.5703_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1730.9890_dp,  720.7679_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  811.9905_dp,  389.2080_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  766.0776_dp,  375.8195_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1368.0049_dp,  567.7536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  812.0495_dp,  385.7107_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.8466_dp,  282.6706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1454.1614_dp,  620.1711_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  713.7552_dp,  354.4940_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  454.6351_dp,  241.6291_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1140.8713_dp,  486.6971_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  653.4752_dp,  327.9705_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  447.9970_dp,  237.2164_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1157.9011_dp,  515.3839_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  624.4481_dp,  312.8735_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  553.1437_dp,  278.5319_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  475.3065_dp,  247.7080_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1213.2014_dp,  531.1595_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  612.6329_dp,  305.9112_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  506.6985_dp,  259.3743_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  550.9242_dp,  272.6492_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  935.7544_dp,  407.2414_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  625.2396_dp,  302.5447_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2109 * maxRef * maxRef + 1 : 2128 * maxRef * maxRef) = [ &
      &  903.5991_dp,  417.4763_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  552.5803_dp,  283.3765_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1090.0939_dp,  494.6170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1006.1273_dp,  464.3202_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  752.0406_dp,  367.2307_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  579.2127_dp,  299.8930_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  896.4651_dp,  435.3339_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  826.5104_dp,  407.0718_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  778.5964_dp,  387.4235_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  679.7752_dp,  346.8564_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  580.5610_dp,  305.8457_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  721.2042_dp,  370.6421_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  706.8707_dp,  364.3169_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  684.3805_dp,  354.6851_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  653.1815_dp,  341.5412_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  625.4480_dp,  332.2173_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  626.1705_dp,  332.3519_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  620.7464_dp,  329.7337_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  528.9381_dp,  290.4179_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.6485_dp,  292.4564_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  446.2084_dp,  252.3325_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4098.5678_dp, 1345.9552_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1093.0934_dp,  491.4621_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3083.9899_dp, 1177.8523_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2566.8729_dp,  962.0972_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1062.1534_dp,  506.1454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2548.5911_dp, 1030.2577_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1752.1144_dp,  733.2622_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  974.9708_dp,  486.4650_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2211.4907_dp,  925.0730_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1146.1276_dp,  544.1619_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  970.8551_dp,  490.4332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1973.5648_dp,  843.0014_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1155.3947_dp,  549.5950_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  947.3214_dp,  474.3613_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1470.5265_dp,  650.1095_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1066.7077_dp,  512.2607_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  807.7328_dp,  418.9542_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1663.1347_dp,  725.6560_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1272.7235_dp,  575.3515_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  693.6613_dp,  369.4767_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1217.3511_dp,  552.1793_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  891.8764_dp,  446.0410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  648.9904_dp,  349.0479_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1323.5223_dp,  603.5669_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  832.9670_dp,  417.2520_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  684.9715_dp,  355.9594_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1209.6847_dp,  559.4921_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  768.7904_dp,  387.3909_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  720.7315_dp,  371.9667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1025.6835_dp,  466.6597_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  747.7247_dp,  374.8322_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1046.5789_dp,  492.7625_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  714.6206_dp,  370.4592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2128 * maxRef * maxRef + 1 : 2147 * maxRef * maxRef) = [ &
      & 1375.6067_dp,  619.1366_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1284.5648_dp,  589.8300_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  966.3182_dp,  473.7077_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  777.0365_dp,  402.7499_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1197.5597_dp,  571.5751_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1094.0703_dp,  532.6855_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1045.8286_dp,  514.7943_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  912.5542_dp,  463.1355_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  799.4055_dp,  418.2405_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1011.2975_dp,  508.5171_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  984.3669_dp,  498.2630_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  948.6782_dp,  484.4278_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  904.0756_dp,  466.9936_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  909.9127_dp,  471.3194_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  907.4929_dp,  470.9435_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  897.0258_dp,  467.0673_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  796.6995_dp,  425.6942_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  804.5343_dp,  429.6945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.4350_dp,  382.3429_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 5022.6967_dp, 1639.3995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1235.5701_dp,  580.3841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4030.0910_dp, 1502.1922_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3441.9852_dp, 1248.0182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1273.3916_dp,  611.9003_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3335.7435_dp, 1317.8387_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2250.9654_dp,  928.2705_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1282.3779_dp,  633.7528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1209.6287_dp,  600.9518_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3507.1716_dp, 1330.3810_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1169.6897_dp,  582.7773_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3325.0797_dp, 1277.2241_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1144.6770_dp,  570.7175_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3231.3241_dp, 1245.1109_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1105.6577_dp,  552.5843_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3146.1740_dp, 1215.5853_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1121.8549_dp,  555.4295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3070.3853_dp, 1189.3991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1111.1877_dp,  535.8760_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2276.2532_dp,  933.7228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1130.0310_dp,  558.9762_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2851.6677_dp, 1053.6587_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1053.6587_dp,  524.9195_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   67.9580_dp,   39.3967_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  116.3178_dp,   63.7816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.1622_dp,   25.8658_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2147 * maxRef * maxRef + 1 : 2166 * maxRef * maxRef) = [ &
      & 1720.7353_dp,  638.4850_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  445.0101_dp,  209.6685_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  775.7718_dp,  361.1822_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.2917_dp,  266.9922_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  323.5941_dp,  172.5188_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  465.2439_dp,  240.7836_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  383.2952_dp,  202.1595_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  315.8875_dp,  169.4228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  224.5940_dp,  128.1009_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  209.5360_dp,  120.4474_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.8455_dp,  161.5691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  273.3375_dp,  151.5963_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  214.9399_dp,  123.2619_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  201.4294_dp,  115.3491_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  156.9831_dp,   93.9771_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  191.4550_dp,  112.4836_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  178.8074_dp,  104.8991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  166.7356_dp,   98.4445_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  146.1445_dp,   86.6740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  139.4937_dp,   84.9288_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.0158_dp,   76.1209_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  108.1820_dp,   67.1060_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  102.3811_dp,   64.2056_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   80.4607_dp,   52.1454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.8557_dp,   49.3166_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2027.8568_dp,  762.5439_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  666.6515_dp,  307.1464_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1287.2471_dp,  577.4805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1045.1961_dp,  475.4827_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  603.2928_dp,  307.3477_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1116.0341_dp,  529.4209_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  997.3837_dp,  479.1923_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  739.4538_dp,  371.7482_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  537.7266_dp,  287.9502_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  808.3322_dp,  414.1717_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  739.3975_dp,  382.4286_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  679.6544_dp,  355.3534_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  669.8712_dp,  346.6120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  519.8201_dp,  283.7221_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  589.8941_dp,  321.1363_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  571.7481_dp,  311.6696_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  566.8734_dp,  307.5736_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  516.3864_dp,  284.8700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  470.2977_dp,  265.6124_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  465.7758_dp,  262.7163_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  456.1062_dp,  257.3927_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  369.9833_dp,  216.2940_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  366.5166_dp,  214.0702_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  293.2905_dp,  176.5117_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3526.9705_dp, 1255.3434_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  890.5056_dp,  422.0535_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2451.7734_dp, 1021.3342_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1992.8728_dp,  817.4261_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  853.8199_dp,  431.1749_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1957.4581_dp,  841.3784_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  822.8420_dp,  419.5774_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  793.8541_dp,  413.1795_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2166 * maxRef * maxRef + 1 : 2185 * maxRef * maxRef) = [ &
      & 1841.5883_dp,  810.5712_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  797.2364_dp,  405.0758_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  801.7112_dp,  418.1829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1658.7019_dp,  741.1995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.8272_dp,  397.0180_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  738.2489_dp,  382.8866_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1310.7538_dp,  583.8816_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.6260_dp,  393.6386_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  529.8069_dp,  287.2154_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1394.4853_dp,  636.8428_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  688.1125_dp,  360.9437_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  439.4052_dp,  245.1092_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1094.0016_dp,  499.7029_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  630.2533_dp,  333.7444_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  432.9245_dp,  240.6788_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1111.8174_dp,  527.9435_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  602.2672_dp,  318.4010_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.5343_dp,  283.3520_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  459.0336_dp,  251.5373_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1164.2755_dp,  544.6244_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  590.7694_dp,  311.3689_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  489.0187_dp,  263.6377_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.9799_dp,  277.6170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  897.8373_dp,  417.6349_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  602.1416_dp,  308.4467_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  868.7577_dp,  426.7546_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.3938_dp,  288.0409_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1047.5640_dp,  506.1315_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  967.4193_dp,  474.6781_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.6816_dp,  374.2592_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.3590_dp,  304.6980_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  863.5860_dp,  443.8226_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  796.6299_dp,  414.6873_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  750.7323_dp,  394.4535_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  656.0889_dp,  352.6781_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  561.0491_dp,  310.4631_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  696.2760_dp,  376.7249_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  682.5126_dp,  370.2368_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  660.9421_dp,  360.3420_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  631.0316_dp,  346.8276_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  604.6048_dp,  337.0758_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  605.2828_dp,  337.2239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  600.0555_dp,  334.5522_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  511.9716_dp,  294.1510_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  516.4905_dp,  296.2434_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  432.3893_dp,  255.1809_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3921.6962_dp, 1405.6043_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1050.0925_dp,  503.0095_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2948.5688_dp, 1217.6072_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2457.8402_dp,  995.6332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1022.3708_dp,  516.4699_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2440.1762_dp, 1061.3052_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1683.0549_dp,  753.6872_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  940.1178_dp,  495.1913_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2119.4908_dp,  950.9922_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1102.5614_dp,  555.3166_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  936.5992_dp,  498.9051_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2185 * maxRef * maxRef + 1 : 2204 * maxRef * maxRef) = [ &
      & 1892.7858_dp,  865.5324_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1110.6913_dp,  560.7912_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  913.6348_dp,  482.7380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1411.7363_dp,  666.0895_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1027.1026_dp,  522.4406_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  779.9682_dp,  425.5550_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1596.2074_dp,  744.0872_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1223.2739_dp,  588.6854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  670.5114_dp,  374.7726_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1169.6785_dp,  564.8984_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  859.9729_dp,  453.9296_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  627.5659_dp,  353.8700_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1271.8937_dp,  617.3535_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  803.2511_dp,  424.5778_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  661.4373_dp,  361.4953_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1163.0374_dp,  571.8202_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  741.5149_dp,  394.0563_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.7803_dp,  377.9042_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  985.5697_dp,  477.2912_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  721.0103_dp,  381.3842_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1006.8382_dp,  503.1205_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  690.0423_dp,  376.3121_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1321.5257_dp,  633.7493_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1234.8582_dp,  603.0805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.2135_dp,  482.6219_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  750.3549_dp,  409.1408_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1152.7975_dp,  583.2130_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1053.9748_dp,  542.9404_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1007.9093_dp,  524.3974_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  880.5098_dp,  471.0117_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  772.2715_dp,  424.6821_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  975.4435_dp,  517.4288_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  949.7160_dp,  506.8159_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  915.6032_dp,  492.5152_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  872.9520_dp,  474.5058_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  878.6753_dp,  478.8244_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  876.4054_dp,  478.3961_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  866.4126_dp,  474.3763_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  770.2908_dp,  431.7667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  777.8548_dp,  435.8349_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  673.1394_dp,  387.2187_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4812.3323_dp, 1712.8830_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1188.8012_dp,  592.5973_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3852.2949_dp, 1555.2120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3298.9422_dp, 1294.3501_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1226.1232_dp,  624.0645_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3192.4039_dp, 1359.3802_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2163.4599_dp,  954.9829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1236.0668_dp,  645.4413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1166.1633_dp,  611.8361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3355.8765_dp, 1375.8472_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1127.7999_dp,  593.2406_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2204 * maxRef * maxRef + 1 : 2223 * maxRef * maxRef) = [ &
      & 3179.9629_dp, 1319.7747_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1103.6788_dp,  580.9422_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3090.4072_dp, 1286.3433_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1066.1553_dp,  562.4103_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3009.0778_dp, 1255.6329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1081.6001_dp,  565.5841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2936.7065_dp, 1228.3897_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1069.9921_dp,  546.4196_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2181.0675_dp,  960.8995_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1089.2614_dp,  569.2318_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2729.9739_dp, 1091.3776_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1015.8840_dp,  534.3480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2617.3310_dp, 1051.8609_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1051.8609_dp,  544.4356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   72.2203_dp,   37.0295_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  123.5220_dp,   59.6586_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   43.5711_dp,   24.4806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1704.9679_dp,  575.7499_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  465.4100_dp,  193.4738_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  811.3688_dp,  332.8646_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  583.7067_dp,  246.7464_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  343.2850_dp,  160.9616_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  491.8875_dp,  224.0856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  405.3285_dp,  188.4765_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  333.5597_dp,  158.2170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  238.5984_dp,  120.2375_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  222.5870_dp,  113.1311_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  307.5198_dp,  151.3607_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  289.3920_dp,  141.9797_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  228.0857_dp,  115.7612_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.5146_dp,  108.3262_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  166.5380_dp,   88.5822_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  203.1554_dp,  105.8582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  189.4531_dp,   98.7212_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  176.7026_dp,   92.6939_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  154.6287_dp,   81.6520_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  147.8355_dp,   80.1704_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  131.3004_dp,   71.9104_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  114.3820_dp,   63.4527_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  108.3076_dp,   60.7598_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   84.9478_dp,   49.4793_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.1416_dp,   46.7595_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2016.3392_dp,  688.5910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  694.0742_dp,  282.8651_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1337.4783_dp,  530.3870_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1084.2313_dp,  437.4255_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  637.8198_dp,  285.5962_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2223 * maxRef * maxRef + 1 : 2242 * maxRef * maxRef) = [ &
      & 1167.9358_dp,  488.8472_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1045.0038_dp,  443.0078_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  778.1323_dp,  345.1182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  570.6508_dp,  268.7620_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  854.0740_dp,  385.0952_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.3748_dp,  355.8982_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  719.0323_dp,  331.0143_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  706.9630_dp,  322.6156_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  551.8787_dp,  265.2733_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  625.9394_dp,  300.1974_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  606.4940_dp,  291.3907_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  600.8855_dp,  287.4557_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.0877_dp,  266.6113_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  499.4576_dp,  249.0984_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  494.5285_dp,  246.3593_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  484.2101_dp,  241.3800_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  392.9385_dp,  203.4529_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  389.1985_dp,  201.3478_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  311.2487_dp,  166.4466_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3419.5160_dp, 1127.9480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  924.6197_dp,  389.9518_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2501.4164_dp,  931.7443_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2006.2958_dp,  745.1781_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  898.6515_dp,  400.4754_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2012.0352_dp,  769.8336_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  868.1104_dp,  390.0277_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  838.3710_dp,  384.7744_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1899.2044_dp,  743.4069_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  837.4618_dp,  376.5089_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  846.8568_dp,  389.5084_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1714.8635_dp,  680.7697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  821.6274_dp,  368.9941_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  777.8991_dp,  356.5084_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1350.2932_dp,  536.2731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  820.3071_dp,  365.5972_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.7016_dp,  268.4867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1446.6020_dp,  586.1315_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  727.2378_dp,  336.3582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  464.9730_dp,  229.6866_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1131.7710_dp,  460.0415_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  666.3378_dp,  311.2798_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  458.0409_dp,  225.4697_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1161.4905_dp,  487.6202_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  636.0357_dp,  296.9473_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  562.7864_dp,  264.4013_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  485.4961_dp,  235.3426_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1213.9348_dp,  502.3206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  623.6996_dp,  290.3139_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  516.8454_dp,  246.3103_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.3819_dp,  258.6987_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.6793_dp,  385.1327_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  633.5260_dp,  286.8927_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  912.0082_dp,  395.3654_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  565.0858_dp,  269.0965_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1096.4450_dp,  468.2277_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1014.7443_dp,  439.7399_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  764.1475_dp,  348.3079_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  593.5044_dp,  284.8492_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2242 * maxRef * maxRef + 1 : 2261 * maxRef * maxRef) = [ &
      &  911.7223_dp,  412.8162_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  841.7526_dp,  386.1634_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  793.9592_dp,  367.6206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.1305_dp,  329.3403_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  595.7532_dp,  290.6346_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  738.2129_dp,  351.9875_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  723.6013_dp,  346.0085_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  700.8927_dp,  336.9098_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  669.5829_dp,  324.4969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  641.7452_dp,  315.7701_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  642.3795_dp,  315.8929_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  636.8114_dp,  313.4113_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  543.7162_dp,  276.2768_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  548.4629_dp,  278.2032_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  459.1205_dp,  240.2275_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3800.8304_dp, 1264.1518_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1090.3500_dp,  465.2606_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2997.0212_dp, 1110.0442_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2460.9470_dp,  906.7106_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1074.3220_dp,  479.7811_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2507.3572_dp,  972.3864_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1717.0123_dp,  693.0607_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  993.5143_dp,  461.6481_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2190.0272_dp,  873.8994_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1156.8729_dp,  515.8086_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  990.8440_dp,  465.5596_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1960.5194_dp,  796.8327_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1168.8393_dp,  520.9380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  963.5525_dp,  450.2413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1466.8017_dp,  615.1052_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1077.4205_dp,  485.7214_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  825.0243_dp,  397.9983_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1655.9466_dp,  686.3381_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1268.9174_dp,  544.7906_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  710.0771_dp,  351.2333_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1218.0651_dp,  522.8164_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  906.9214_dp,  423.3418_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  664.7955_dp,  331.8942_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1329.1426_dp,  571.4828_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  846.3103_dp,  396.0475_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  698.3448_dp,  338.1899_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1217.6582_dp,  529.9388_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  781.2040_dp,  367.7646_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  735.1282_dp,  353.3208_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1025.3902_dp,  441.8976_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  759.5084_dp,  355.7893_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1056.3718_dp,  466.9433_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  730.4286_dp,  351.9076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1377.8901_dp,  586.0395_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1291.3513_dp,  558.5785_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  980.6060_dp,  449.3638_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  795.0694_dp,  382.5686_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1213.1043_dp,  541.7910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1111.0005_dp,  505.1927_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1063.6161_dp,  488.3600_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.6850_dp,  439.6941_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  819.1008_dp,  397.3735_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2261 * maxRef * maxRef + 1 : 2280 * maxRef * maxRef) = [ &
      & 1031.8071_dp,  482.6582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1005.0059_dp,  473.0076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  969.4956_dp,  459.9774_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  925.1907_dp,  443.5507_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  931.1239_dp,  447.6960_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  928.8165_dp,  447.3606_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  918.4190_dp,  443.7162_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  817.3220_dp,  404.6813_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  825.3644_dp,  408.4793_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  714.6477_dp,  363.7360_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4632.8428_dp, 1539.8453_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1241.6267_dp,  550.0548_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3885.5093_dp, 1414.9943_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3259.8698_dp, 1175.4332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1287.4672_dp,  580.1915_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3259.9132_dp, 1243.1883_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2193.8636_dp,  877.1478_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1304.7191_dp,  601.2846_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1230.5989_dp,  570.2451_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3391.5411_dp, 1253.7691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1190.2918_dp,  553.0414_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3229.8168_dp, 1203.9727_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1165.0450_dp,  541.6048_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3141.6098_dp, 1173.7842_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1125.5430_dp,  524.4310_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3061.2537_dp, 1146.0194_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1139.6825_dp,  527.0243_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2989.7660_dp, 1121.3978_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1124.6771_dp,  508.1297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2237.9026_dp,  881.7549_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1148.8304_dp,  530.3616_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2724.3602_dp,  992.6267_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1072.3978_dp,  498.1319_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2606.5136_dp,  957.3153_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1106.4740_dp,  506.8895_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2664.1668_dp, 1011.4677_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1011.4677_dp,  472.7950_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   70.7154_dp,   36.0084_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  120.9302_dp,   57.9760_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   42.6658_dp,   23.8289_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2280 * maxRef * maxRef + 1 : 2299 * maxRef * maxRef) = [ &
      & 1664.7700_dp,  557.2454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  455.2827_dp,  187.7050_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  793.6877_dp,  322.8840_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  571.0039_dp,  239.4431_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  336.0507_dp,  156.3709_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  481.4385_dp,  217.6279_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  396.7377_dp,  183.0885_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  326.4963_dp,  153.7302_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  233.6155_dp,  116.9007_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  217.9416_dp,  110.0016_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  301.0692_dp,  147.1244_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  283.3001_dp,  138.0041_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  223.3166_dp,  112.5588_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  209.0416_dp,  105.3302_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  163.0702_dp,   86.1743_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  198.9185_dp,  102.9588_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  185.4931_dp,   96.0193_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  173.0120_dp,   90.1627_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  151.3924_dp,   79.4291_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  144.7587_dp,   78.0079_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  128.5666_dp,   69.9791_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  111.9978_dp,   61.7568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  106.0552_dp,   59.1423_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   83.1827_dp,   48.1805_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   79.4539_dp,   45.5279_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1968.9928_dp,  666.5534_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  678.8486_dp,  274.3732_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1307.9814_dp,  514.2872_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1060.3386_dp,  424.2493_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  624.2488_dp,  277.3103_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1142.5656_dp,  474.3095_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1022.3674_dp,  429.8982_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  761.4623_dp,  335.0783_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  558.6379_dp,  261.1100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  835.8969_dp,  373.9540_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  764.7668_dp,  345.6425_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  703.7881_dp,  321.5131_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  691.9095_dp,  313.3296_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  540.2901_dp,  257.7790_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  612.7809_dp,  291.7100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  593.7403_dp,  283.1587_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  588.2297_dp,  279.3232_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  536.5858_dp,  259.1140_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  489.0107_dp,  242.1588_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  484.1791_dp,  239.4936_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  474.0753_dp,  234.6551_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  384.7496_dp,  197.8642_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  381.0847_dp,  195.8155_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  304.7748_dp,  161.9300_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3337.2227_dp, 1091.5091_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  904.3257_dp,  378.4205_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2444.6758_dp,  902.8445_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1960.2391_dp,  722.1111_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  879.3887_dp,  388.8507_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1966.9050_dp,  746.1810_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  849.5394_dp,  378.7408_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  820.5423_dp,  373.7251_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2299 * maxRef * maxRef + 1 : 2318 * maxRef * maxRef) = [ &
      & 1856.8743_dp,  720.7667_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  819.4973_dp,  365.6231_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  828.8555_dp,  378.3321_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1676.8129_dp,  660.1463_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  803.9989_dp,  358.3213_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  761.3040_dp,  346.2718_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1320.1912_dp,  520.0492_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  802.6609_dp,  354.9945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  547.8503_dp,  260.9096_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1414.7048_dp,  568.5111_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  711.8001_dp,  326.7279_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  455.1812_dp,  223.2746_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1106.7246_dp,  446.2432_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  652.2155_dp,  302.4017_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  448.3896_dp,  219.1675_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1136.1897_dp,  473.1486_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  622.5372_dp,  288.4783_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  550.8249_dp,  256.8824_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  475.2458_dp,  228.7263_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1187.3873_dp,  487.3312_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.4511_dp,  282.0254_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  505.9042_dp,  239.3413_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  547.4456_dp,  251.2997_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.1707_dp,  373.6528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  619.9597_dp,  278.6211_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  892.3332_dp,  383.7696_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  553.1687_dp,  261.4745_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1072.6768_dp,  454.4189_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  992.8398_dp,  426.8411_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  747.8611_dp,  338.2814_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  581.0286_dp,  276.8003_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  892.3017_dp,  400.8971_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  823.8687_dp,  375.0691_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  777.1265_dp,  357.0949_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  680.4664_dp,  319.9904_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  583.2649_dp,  282.4712_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  722.6654_dp,  342.0170_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  708.3653_dp,  336.2183_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  686.1478_dp,  327.3956_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  655.5207_dp,  315.3605_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  628.2934_dp,  306.9300_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  628.9108_dp,  307.0478_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  623.4596_dp,  304.6384_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  532.3641_dp,  268.6347_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  537.0079_dp,  270.5030_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  449.5592_dp,  233.6551_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3709.4297_dp, 1223.4668_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1066.4469_dp,  451.5684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2928.7164_dp, 1075.5659_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2404.1111_dp,  878.5837_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1051.2437_dp,  465.8723_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2451.1775_dp,  942.6854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1678.5781_dp,  672.2671_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  972.4302_dp,  448.4531_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2141.4202_dp,  847.4869_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1131.9421_dp,  500.8568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  969.8725_dp,  452.3068_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2318 * maxRef * maxRef + 1 : 2337 * maxRef * maxRef) = [ &
      & 1917.2203_dp,  772.9187_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1143.6857_dp,  505.8297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  943.0626_dp,  437.4079_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1434.6130_dp,  596.8707_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1054.2484_dp,  471.7011_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  807.5992_dp,  386.7854_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1619.5186_dp,  665.8997_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1241.1151_dp,  528.8006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  695.1436_dp,  341.4299_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1191.4716_dp,  507.4568_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  887.6202_dp,  411.2731_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  650.8338_dp,  322.6630_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1300.2637_dp,  554.6882_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  828.2826_dp,  384.7714_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  683.5612_dp,  328.6834_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1191.2962_dp,  514.4346_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  764.5705_dp,  357.3192_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  719.5693_dp,  343.3582_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1002.9797_dp,  428.9424_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  743.3221_dp,  345.6658_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1033.6009_dp,  453.3608_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  715.0198_dp,  341.9894_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1347.8497_dp,  568.7508_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1263.3503_dp,  542.1999_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  959.6719_dp,  436.4645_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  778.3235_dp,  371.7768_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1187.1040_dp,  526.0806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1087.2901_dp,  490.6416_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1040.9736_dp,  474.3436_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.9836_dp,  427.2006_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  801.8905_dp,  386.1954_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1009.9583_dp,  468.8968_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  983.7511_dp,  459.5521_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  949.0275_dp,  446.9312_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  905.7052_dp,  431.0179_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  911.5137_dp,  435.0603_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  909.2616_dp,  434.7424_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  899.0951_dp,  431.2148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  800.1918_dp,  393.3829_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  808.0654_dp,  397.0728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  699.7231_dp,  353.6841_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4520.9645_dp, 1490.3329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1214.7181_dp,  534.0818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3796.0924_dp, 1370.8120_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3183.6011_dp, 1138.7329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1259.8096_dp,  563.4346_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3186.2327_dp, 1205.0073_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2144.4704_dp,  850.7547_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1276.9567_dp,  584.0468_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1204.4185_dp,  553.9356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3313.8879_dp, 1214.8410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1164.9825_dp,  537.2399_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2337 * maxRef * maxRef + 1 : 2356 * maxRef * maxRef) = [ &
      & 3156.1992_dp, 1166.6903_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1140.2779_dp,  526.1331_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3070.0842_dp, 1137.4645_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1101.6251_dp,  509.4631_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2991.6275_dp, 1110.5819_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1115.3978_dp,  511.9439_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2921.8310_dp, 1086.7437_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1100.5512_dp,  493.4684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2187.7803_dp,  855.0138_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1124.3657_dp,  515.1733_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2661.1461_dp,  961.7148_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1049.5999_dp,  483.8982_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2546.1249_dp,  927.6008_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1082.8259_dp,  492.3380_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2603.9653_dp,  980.3403_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  990.0070_dp,  459.3155_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2545.1713_dp,  959.5536_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  959.5536_dp,  446.2314_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   69.3791_dp,   35.7480_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  118.6070_dp,   57.5937_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.8727_dp,   23.6297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1627.0951_dp,  555.2598_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  446.0030_dp,  186.7210_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  777.4602_dp,  321.2393_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  559.3898_dp,  238.1420_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  329.5355_dp,  155.3867_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  471.9860_dp,  216.3099_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  388.9890_dp,  181.9383_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  320.1437_dp,  152.7300_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  229.1777_dp,  116.0754_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  213.8101_dp,  109.2144_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  295.3014_dp,  146.1173_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  277.8493_dp,  137.0581_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  219.0722_dp,  111.7498_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  205.0594_dp,  104.5694_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  160.0071_dp,   85.5107_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  195.1630_dp,  102.1888_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  181.9823_dp,   95.2979_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  169.7429_dp,   89.4774_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  148.5273_dp,   78.8150_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  142.0478_dp,   77.3874_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  126.1614_dp,   69.4137_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  109.9031_dp,   61.2454_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  104.0800_dp,   58.6465_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   81.6437_dp,   47.7551_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   77.9788_dp,   45.1298_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2356 * maxRef * maxRef + 1 : 2375 * maxRef * maxRef) = [ &
      & 1924.6593_dp,  664.0994_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  664.8580_dp,  272.9668_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1280.7911_dp,  511.8009_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1038.3601_dp,  422.1177_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  611.9392_dp,  275.6811_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1119.3391_dp,  471.7945_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1001.6784_dp,  427.5655_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  746.3210_dp,  333.1239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  547.8252_dp,  259.4546_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  819.4280_dp,  371.7271_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  749.7413_dp,  343.5475_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  690.0184_dp,  319.5322_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  678.2927_dp,  311.4138_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  529.8922_dp,  256.0890_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  600.9701_dp,  289.8009_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  582.2956_dp,  281.2978_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  576.8643_dp,  277.4949_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  526.2842_dp,  257.3786_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  479.6909_dp,  240.4740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  474.9443_dp,  237.8283_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  465.0326_dp,  233.0206_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  377.4873_dp,  196.4062_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  373.8878_dp,  194.3732_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  299.0622_dp,  160.6761_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3259.9368_dp, 1087.6970_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  885.7538_dp,  376.3047_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2391.9954_dp,  898.8634_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1917.4640_dp,  718.8719_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  861.8975_dp,  386.5382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1925.1263_dp,  742.7390_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  832.6961_dp,  376.4552_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  804.4238_dp,  371.4100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1817.7881_dp,  717.2897_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  803.2032_dp,  363.4076_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  812.5860_dp,  375.9796_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1641.7354_dp,  656.8806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  788.0064_dp,  356.1503_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  746.2943_dp,  344.1155_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1292.4384_dp,  517.4322_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  786.6336_dp,  352.8632_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  537.2011_dp,  259.1578_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1385.3712_dp,  565.5959_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  697.8565_dp,  324.6761_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  446.4219_dp,  221.7117_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1083.6957_dp,  443.9051_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  639.4782_dp,  300.4715_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  439.7506_dp,  217.6401_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1113.0259_dp,  470.5820_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  610.3600_dp,  286.6332_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  540.0415_dp,  255.2100_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  466.0487_dp,  227.1685_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1163.0369_dp,  484.7511_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  598.4935_dp,  280.2274_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  496.0618_dp,  237.7527_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  536.6581_dp,  249.6938_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  892.3516_dp,  371.6376_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  607.6633_dp,  276.9038_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2375 * maxRef * maxRef + 1 : 2394 * maxRef * maxRef) = [ &
      &  874.3961_dp,  381.5809_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  542.4516_dp,  259.7613_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1050.9705_dp,  451.9002_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  972.8753_dp,  424.4203_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  733.1234_dp,  336.2103_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  569.8286_dp,  274.9818_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  874.7132_dp,  398.4787_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  807.7031_dp,  372.7595_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  761.9321_dp,  354.8650_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  667.2758_dp,  317.9241_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  572.0834_dp,  280.5719_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  708.6950_dp,  339.7906_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  694.6807_dp,  334.0190_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  672.9150_dp,  325.2371_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  642.9169_dp,  313.2573_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  616.2658_dp,  304.8342_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  616.8666_dp,  304.9520_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  611.5213_dp,  302.5558_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  522.2655_dp,  266.7093_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  526.8147_dp,  268.5684_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  441.0932_dp,  231.9047_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3623.6511_dp, 1219.0706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1044.6045_dp,  448.9840_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2865.2613_dp, 1070.8351_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2351.2803_dp,  874.6707_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1030.2928_dp,  463.0740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2399.2487_dp,  938.1873_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1643.2139_dp,  668.7656_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  953.4036_dp,  445.6186_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2096.6400_dp,  843.2361_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1109.3035_dp,  497.8395_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  950.9799_dp,  449.4035_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1877.4174_dp,  768.9101_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1120.8436_dp,  502.7808_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  924.5841_dp,  434.6007_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1405.1352_dp,  593.5778_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1033.2364_dp,  468.8041_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  791.9602_dp,  384.1823_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1586.1100_dp,  662.3109_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1215.7298_dp,  525.7481_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  681.7920_dp,  339.0464_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1167.1878_dp,  504.5373_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  870.2063_dp,  408.6237_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  638.3682_dp,  320.3788_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1273.9026_dp,  551.5239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  812.0214_dp,  382.2739_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  670.2940_dp,  326.4378_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1167.2707_dp,  511.4434_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  749.5779_dp,  354.9728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  705.5909_dp,  341.0443_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  982.5210_dp,  426.4375_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  728.7203_dp,  343.4086_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1012.8904_dp,  450.6589_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  701.1829_dp,  339.6919_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2394 * maxRef * maxRef + 1 : 2413 * maxRef * maxRef) = [ &
      & 1320.3887_dp,  565.5647_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1237.8116_dp,  539.0868_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  940.7333_dp,  433.7382_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  763.2869_dp,  369.2991_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1163.4988_dp,  522.9356_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1065.8196_dp,  487.6285_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1020.4995_dp,  471.3898_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  894.2426_dp,  424.4358_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  786.4597_dp,  383.6007_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  990.2588_dp,  465.9059_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  964.6052_dp,  456.5948_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  930.6131_dp,  444.0225_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  888.2036_dp,  428.1726_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  893.9080_dp,  432.1732_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  891.7106_dp,  431.8507_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  881.7600_dp,  428.3345_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  784.8841_dp,  390.6589_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  792.6061_dp,  394.3257_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  686.4452_dp,  351.1344_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 4415.9694_dp, 1484.9661_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1190.2611_dp,  530.8706_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3712.8812_dp, 1364.9240_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3112.5536_dp, 1133.8329_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1234.7344_dp,  559.9910_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3118.0001_dp, 1199.3874_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2098.9702_dp,  846.3794_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1251.8739_dp,  580.3947_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1180.7776_dp,  550.4264_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3241.7266_dp, 1209.4663_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1142.1370_dp,  533.8225_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3087.8489_dp, 1161.4494_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1117.9234_dp,  522.7836_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3003.6922_dp, 1132.3365_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1080.0441_dp,  506.2081_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2927.0118_dp, 1105.5604_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1093.4602_dp,  508.7048_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2858.7977_dp, 1081.8160_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1078.6794_dp,  490.4334_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2141.5264_dp,  850.7546_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1102.2614_dp,  511.9281_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2602.3122_dp,  957.4948_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1029.0199_dp,  480.8218_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2489.9643_dp,  923.4785_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1061.4329_dp,  489.2608_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2548.1802_dp,  975.8085_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  970.6528_dp,  456.3728_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2413 * maxRef * maxRef + 1 : 2432 * maxRef * maxRef) = [ &
      & 2490.6989_dp,  955.1104_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  940.8140_dp,  443.3639_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2437.4539_dp,  936.4457_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  936.4457_dp,  440.5228_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   68.4949_dp,   37.3986_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  117.1811_dp,   60.4871_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   41.2866_dp,   24.5435_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1612.7581_dp,  592.3651_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  441.3220_dp,  197.5946_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  769.3983_dp,  340.2385_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  553.4266_dp,  251.7223_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  325.6844_dp,  163.4855_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  466.6099_dp,  227.8802_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  384.4701_dp,  191.4109_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  316.3522_dp,  160.4620_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  226.3056_dp,  121.5617_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  211.1076_dp,  114.3083_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  291.6817_dp,  153.2146_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  274.4500_dp,  143.6930_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  216.3044_dp,  116.9313_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  202.4669_dp,  109.3879_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  157.8890_dp,   89.1915_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  192.6324_dp,  106.7340_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  179.6202_dp,   99.5059_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  167.5247_dp,   93.3808_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  146.5709_dp,   82.1806_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  140.1300_dp,   80.5863_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  124.4402_dp,   72.2182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  108.3826_dp,   63.6396_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  102.6251_dp,   60.9031_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   80.4575_dp,   49.4502_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &   76.8581_dp,   46.7569_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1907.5332_dp,  708.0629_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  657.9922_dp,  289.0123_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1267.8866_dp,  542.7763_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1027.7356_dp,  447.1652_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  605.0843_dp,  290.7410_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1107.5215_dp,  499.1071_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  990.9808_dp,  452.0055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  738.0233_dp,  351.3735_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  541.3978_dp,  272.9068_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  810.1845_dp,  391.8451_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  741.2030_dp,  361.9140_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  682.0827_dp,  336.4239_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  670.5479_dp,  327.9441_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  523.5513_dp,  269.0182_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  593.7911_dp,  304.4460_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  575.3262_dp,  295.4611_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  569.9851_dp,  291.5055_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  519.9103_dp,  270.1359_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  473.7390_dp,  251.9975_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  469.0567_dp,  249.2287_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  459.2634_dp,  244.1704_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  372.6267_dp,  205.2969_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  369.0769_dp,  203.1740_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp]

    c6ab1d(2432 * maxRef * maxRef + 1 : 2451 * maxRef * maxRef) = [ &
      &  295.0836_dp,  167.5526_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 3231.7117_dp, 1160.0767_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  876.2899_dp,  397.2856_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 2369.0958_dp,  955.0787_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1899.1306_dp,  763.2645_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  852.2547_dp,  407.3976_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1906.2802_dp,  788.4325_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  823.3089_dp,  396.5845_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  795.1856_dp,  390.9216_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1799.6377_dp,  760.5639_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  794.1404_dp,  382.7505_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  803.2347_dp,  395.6779_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1625.1430_dp,  696.0633_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  779.1219_dp,  375.1083_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  737.7269_dp,  362.0987_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1279.3493_dp,  547.9588_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  777.8232_dp,  371.7528_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.7581_dp,  271.9114_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1371.1198_dp,  598.7622_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  689.7847_dp,  341.5632_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  440.9182_dp,  232.2310_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1072.4959_dp,  469.5788_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  632.0123_dp,  315.9092_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  434.3472_dp,  228.0072_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1101.2182_dp,  497.4063_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  603.2352_dp,  301.3273_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  533.6923_dp,  268.0922_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  460.4063_dp,  238.2171_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1150.8553_dp,  512.7473_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  591.5255_dp,  294.6262_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  490.1530_dp,  249.5747_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  530.4351_dp,  262.4692_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  882.9919_dp,  392.8083_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  600.7572_dp,  291.4639_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  864.8546_dp,  402.7269_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  536.0077_dp,  272.8496_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      & 1039.6601_dp,  477.3755_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  962.2681_dp,  448.0328_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  724.7596_dp,  354.0246_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  563.0164_dp,  288.8264_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  864.7996_dp,  419.8297_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &
      &  798.4380_dp,  392.4501_dp,    0.0000_dp,    0.0000_dp,    0.0000_dp, &