// AStyleTest_StylesCpp1.cpp
// Copyright (c) 2018 by Jim Pattee <jimp03@email.com>.
// This code is licensed under the MIT License.
// License.md describes the conditions under which this software may be distributed.

//----------------------------------------------------------------------------
// headers
//----------------------------------------------------------------------------

#include "AStyleTest.h"

//----------------------------------------------------------------------------
// anonymous namespace
//----------------------------------------------------------------------------

namespace {
//
//----------------------------------------------------------------------------
// AStyle C++ text for style tests
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// AStyle C++ Default Style
// Additional tests are in the break braces tests
//----------------------------------------------------------------------------

struct StyleDefaultCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleDefaultCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleDefaultCppF, LongOption)
{
	// test default style option
	char options[] = "";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(textIn, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, SpaceIndent)
{
	// test default style with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "      if (isFoo\n"
	    "                  && isBar)\n"
	    "      {\n"
	    "            bar();\n"
	    "      }\n"
	    "      else\n"
	    "            anotherBar();\n"
	    "}\n";
	char options[] = "indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, Tab)
{
	// test default style with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	        && isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, TabIndent)
{
	// test default style with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	            && isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, ForceTab)
{
	// test default style with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, ForceTabIndent)
{
	// test default style with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, MinConditionalIndent1)
{
	// default should use a default setting of MINCOND_TWO
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, MinConditionalIndent2)
{
	// test default style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "        && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedClass)
{
	// test nested classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedClass_IndentClass)
{
	// test nested classes with indented classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "        {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                            {\n"
	    "                                bar();\n"
	    "                            }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedNamespace)
{
	// test nested namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "namespace B\n"
	    "{\n"
	    "namespace C\n"
	    "{\n"
	    "void foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedNamespace_IndentNamespace)
{
	// test nested indented namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    namespace B\n"
	    "    {\n"
	    "        namespace C\n"
	    "        {\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedNamespaceClass)
{
	// test namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "class A\n"
	    "{\n"
	    "public:\n"
	    "    namespace B\n"
	    "    {\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace C\n"
	    "        {\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedNamespaceClass_IndentNamespace)
{
	// test indented namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace B\n"
	    "        {\n"
	    "            class B\n"
	    "            {\n"
	    "            public:\n"
	    "                namespace C\n"
	    "                {\n"
	    "                    class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                            {\n"
	    "                                bar();\n"
	    "                            }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test indented namespaces within an indented class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "            {\n"
	    "                class B\n"
	    "                {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                        {\n"
	    "                            class C\n"
	    "                            {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                    {\n"
	    "                                        if (isFoo)\n"
	    "                                        {\n"
	    "                                            bar();\n"
	    "                                        }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, Pico)
{
	// test default style with pico braces
	// the closing braces should be broken
	// the ENTIRE one-line block should be broken
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2();\n"
	    "    }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2();\n"
	    "    }\n"
	    "}\n";
	char options[] = "";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleDefaultCpp, PicoOneLine)
{
	// test default style with pico braces and one-line blocks
	// the closing braces should be broken
	// the ENTIRE one-line block should be broken
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "    {\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n";
	char options[] = "";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ Allman Style
// Additional tests are in the break braces tests
//----------------------------------------------------------------------------

struct StyleAllmanCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleAllmanCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleAllmanCppF, AllmanOption)
{
	// test allman style option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleAllmanCppF, AnsiOption)
{
	// test allman style ANSI option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=ansi";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleAllmanCppF, BSDOption)
{
	// test allman style BSD option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=bsd";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleAllmanCppF, BreakOption)
{
	// test allman style break option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=break";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleAllmanCppF, ShortOption)
{
	// test allman style short option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A1";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, SpaceIndent)
{
	// test allman style with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "      if (isFoo\n"
	    "                  && isBar)\n"
	    "      {\n"
	    "            bar();\n"
	    "      }\n"
	    "      else\n"
	    "            anotherBar();\n"
	    "}\n";
	char options[] = "style=allman, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, Tab)
{
	// test allman style with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	        && isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=allman, indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, TabIndent)
{
	// test allman style with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	            && isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=allman, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, ForceTab)
{
	// test allman style with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=allman, indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, ForceTabIndent)
{
	// test allman style with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "	{\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=allman, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, Struct)
{
	// test allman style with a struct
	// structs should be broken
	char text[] =
	    "\nstruct FooStruct\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, Enum)
{
	// test allman style with an enum
	// enums should be broken
	char text[] =
	    "\nenum Encoding\n"
	    "{\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "};\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, MinConditionalIndent1)
{
	// allman should use a default setting of MINCOND_TWO
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, MinConditionalIndent2)
{
	// test allman style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "        && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=allman, min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedClass)
{
	// test nested classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedClass_IndentClass)
{
	// test nested classes with indented classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "        {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                            {\n"
	    "                                bar();\n"
	    "                            }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=allman, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedNamespace)
{
	// test nested namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "namespace B\n"
	    "{\n"
	    "namespace C\n"
	    "{\n"
	    "void foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedNamespace_IndentNamespace)
{
	// test nested indented namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    namespace B\n"
	    "    {\n"
	    "        namespace C\n"
	    "        {\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=allman, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedNamespaceClass)
{
	// test namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "class A\n"
	    "{\n"
	    "public:\n"
	    "    namespace B\n"
	    "    {\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace C\n"
	    "        {\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedNamespaceClass_IndentNamespace)
{
	// test indented namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace B\n"
	    "        {\n"
	    "            class B\n"
	    "            {\n"
	    "            public:\n"
	    "                namespace C\n"
	    "                {\n"
	    "                    class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                            {\n"
	    "                                bar();\n"
	    "                            }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=allman, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test indented namespaces within an indented class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "            {\n"
	    "                class B\n"
	    "                {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                        {\n"
	    "                            class C\n"
	    "                            {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                    {\n"
	    "                                        if (isFoo)\n"
	    "                                        {\n"
	    "                                            bar();\n"
	    "                                        }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=allman, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, Pico)
{
	// test allman style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "    }\n"
	    "    else\n"
	    "    {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleAllmanCpp, PicoOneLine)
{
	// test allman style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "    {\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=allman";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ Java Style
// Additional tests are in the attach braces tests
//----------------------------------------------------------------------------

struct StyleJavaCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleJavaCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleJavaCppF, LongOption)
{
	// test java style option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleJavaCppF, AttachOption)
{
	// test java style attach option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=attach";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleJavaCppF, ShortOption)
{
	// test java style short option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, SpaceIndent)
{
	// test java style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "      if (isFoo\n"
	    "                  && isBar) {\n"
	    "            bar();\n"
	    "      } else\n"
	    "            anotherBar();\n"
	    "}\n";
	char options[] = "style=java, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, Tab)
{
	// test java style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "	        && isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=java, indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, TabIndent)
{
	// test java style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "	            && isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=java, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, ForceTab)
{
	// test java style option with force tab
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=java, indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, ForceTabIndent)
{
	// test java style option with force tab
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=java, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, Struct)
{
	// test java style with a struct
	// structs should be attached
	char text[] =
	    "\nstruct FooStruct {\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, Enum)
{
	// test java style with an enum
	// enums should be attached
	char text[] =
	    "\nenum Encoding {\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "};\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, MinConditionalIndent1)
{
	// java should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo\n"
	    "            && isBar) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, MinConditionalIndent2)
{
	// test java style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo\n"
	    "        && isBar) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=java, min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedClass)
{
	// test nested classes
	char text[] =
	    "\nclass A {\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C {\n"
	    "        public:\n"
	    "            void foo(bool isFoo) {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                } else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedClass_IndentClass)
{
	// test nested classes with indented classes
	char text[] =
	    "\nclass A {\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo) {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                            } else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=java, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedNamespace)
{
	// test nested namespaces
	char text[] =
	    "\nnamespace A {\n"
	    "namespace B {\n"
	    "namespace C {\n"
	    "void foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedNamespace_IndentNamespace)
{
	// test nested indented namespaces
	char text[] =
	    "\nnamespace A {\n"
	    "    namespace B {\n"
	    "        namespace C {\n"
	    "            void foo(bool isFoo) {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                } else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=java, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedNamespaceClass)
{
	// test namespaces within a class
	char text[] =
	    "\nnamespace A {\n"
	    "class A {\n"
	    "public:\n"
	    "    namespace B {\n"
	    "    class B {\n"
	    "    public:\n"
	    "        namespace C {\n"
	    "        class C {\n"
	    "        public:\n"
	    "            void foo(bool isFoo) {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                } else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedNamespaceClass_IndentNamespace)
{
	// test indented namespaces within a class
	char text[] =
	    "\nnamespace A {\n"
	    "    class A {\n"
	    "    public:\n"
	    "        namespace B {\n"
	    "            class B {\n"
	    "            public:\n"
	    "                namespace C {\n"
	    "                    class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo) {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                            } else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=java, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test indented namespaces within an indented class
	char text[] =
	    "\nnamespace A {\n"
	    "    class A {\n"
	    "        public:\n"
	    "            namespace B {\n"
	    "                class B {\n"
	    "                    public:\n"
	    "                        namespace C {\n"
	    "                            class C {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo) {\n"
	    "                                        if (isFoo) {\n"
	    "                                            bar();\n"
	    "                                        } else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=java, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, Pico)
{
	// test java style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "    } else {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleJavaCpp, PicoOneLine)
{
	// test java style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else {\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=java";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ K & R Style
// Additional tests are in the linux braces tests
//----------------------------------------------------------------------------

struct StyleKRCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleKRCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleKRCppF, LongOption)
{
	// test k&r style option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleKRCppF, LongOption1)
{
	// test k&r style alternate kr option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=k&r";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleKRCppF, LongOption2)
{
	// test k&r style alternate k/r option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=k/r";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleKRCppF, Short)
{
	// test k&r style short option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A3";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, SpaceIndent)
{
	// test k&r style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "      if (isFoo\n"
	    "                  && isBar) {\n"
	    "            bar();\n"
	    "      } else\n"
	    "            anotherBar();\n"
	    "}\n";
	char options[] = "style=kr, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, Tab)
{
	// test k&r style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	        && isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=kr, indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, TabIndent)
{
	// test k&r style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	            && isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=kr, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, ForceTab)
{
	// test k&r style option with force tab
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=kr, indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, ForceTabIndent)
{
	// test k&r style option with force tab
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "	} else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=kr, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, Struct)
{
	// test k&r style with a struct
	// structs should be attached
	char text[] =
	    "\nstruct FooStruct {\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, Enum)
{
	// test k&r style with an enum
	// enums should be attached
	char text[] =
	    "\nenum Encoding {\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "};\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, MinConditionalIndent1)
{
	// k&r should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, MinConditionalIndent2)
{
	// test k&r style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "        && isBar) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=kr, min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedClass)
{
	// test nested classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                } else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedClass_IndentClass)
{
	// test nested classes with indented classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "        {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                            } else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=kr, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedNamespace)
{
	// test nested namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "namespace B\n"
	    "{\n"
	    "namespace C\n"
	    "{\n"
	    "void foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedNamespace_IndentNamespace)
{
	// test nested indented namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    namespace B\n"
	    "    {\n"
	    "        namespace C\n"
	    "        {\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                } else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=kr, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedNamespaceClass)
{
	// test namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "class A\n"
	    "{\n"
	    "public:\n"
	    "    namespace B\n"
	    "    {\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace C\n"
	    "        {\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                } else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedNamespaceClass_IndentNamespace)
{
	// test indented namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace B\n"
	    "        {\n"
	    "            class B\n"
	    "            {\n"
	    "            public:\n"
	    "                namespace C\n"
	    "                {\n"
	    "                    class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                            } else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=kr, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test indented namespaces within an indented class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "            {\n"
	    "                class B\n"
	    "                {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                        {\n"
	    "                            class C\n"
	    "                            {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                    {\n"
	    "                                        if (isFoo) {\n"
	    "                                            bar();\n"
	    "                                        } else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=kr, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, Pico)
{
	// test k&r style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "    } else {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleKRCpp, PicoOneLine)
{
	// test k&r style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    } else {\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=kr";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ Stroustrup Style
// Additional tests are in the stroustrup braces tests
//----------------------------------------------------------------------------

struct StyleStroustrupCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleStroustrupCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleStroustrupCppF, LongOption)
{
	// test stroustrup style option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "private:\n"
	    "     bool var1;\n"
	    "     void func1();\n"
	    "protected:\n"
	    "     bool var2;\n"
	    "     void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "     if (isFoo) {\n"
	    "          bar();\n"
	    "     }\n"
	    "     else\n"
	    "          anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=stroustrup, indent=spaces=5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleStroustrupCppF, ShortOption)
{
	// test stroustrup style short option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "private:\n"
	    "     bool var1;\n"
	    "     void func1();\n"
	    "protected:\n"
	    "     bool var2;\n"
	    "     void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "     if (isFoo) {\n"
	    "          bar();\n"
	    "     }\n"
	    "     else\n"
	    "          anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A4, indent=spaces=5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, SpaceIndent)
{
	// test stroustrup style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "      if (isFoo\n"
	    "                  && isBar) {\n"
	    "            bar();\n"
	    "      }\n"
	    "      else\n"
	    "            anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, Tab)
{
	// test stroustrup style option with tab indent
	// default indent is 5
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	          && isBar) {\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=tab=5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, TabIndent)
{
	// test stroustrup style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	            && isBar) {\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, ForceTab)
{
	// test stroustrup style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=force-tab=5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, ForceTabIndent)
{
	// test stroustrup style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "	}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, Struct)
{
	// test stroustrup style with a struct
	// structs should be attached
	char text[] =
	    "\nstruct FooStruct {\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n";
	char options[] = "style=stroustrup";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, Enum)
{
	// test stroustrup style with an enum
	// enums should be attached
	char text[] =
	    "\nenum Encoding {\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "};\n";
	char options[] = "style=stroustrup";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, MinConditionalIndent1)
{
	// stroustrup should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "     if (isFoo\n"
	    "               && isBar) {\n"
	    "          bar();\n"
	    "     }\n"
	    "     else\n"
	    "          anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=spaces=5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, MinConditionalIndent2)
{
	// test stroustrup style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "     if (isFoo\n"
	    "         && isBar) {\n"
	    "          bar();\n"
	    "     }\n"
	    "     else\n"
	    "          anotherBar();\n"
	    "}\n";
	char options[] = "style=stroustrup, min-conditional-indent=0, indent=spaces=5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedClass)
{
	// test nested classes
	char text[] =
	    "\nclass A {\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedClass_IndentClass)
{
	// test nested classes with indented classes
	char text[] =
	    "\nclass A {\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                            }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=stroustrup, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedNamespace)
{
	// test nested namespaces
	char text[] =
	    "\nnamespace A {\n"
	    "namespace B {\n"
	    "namespace C {\n"
	    "void foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedNamespace_IndentNamespace)
{
	// test nested indented namespaces
	char text[] =
	    "\nnamespace A {\n"
	    "    namespace B {\n"
	    "        namespace C {\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=stroustrup, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedNamespaceClass)
{
	// test namespaces within a class
	char text[] =
	    "\nnamespace A {\n"
	    "class A {\n"
	    "public:\n"
	    "    namespace B {\n"
	    "    class B {\n"
	    "    public:\n"
	    "        namespace C {\n"
	    "        class C {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "style=stroustrup, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedNamespaceClass_IndentNamespace)
{
	// test indented namespaces within a class
	char text[] =
	    "\nnamespace A {\n"
	    "    class A {\n"
	    "    public:\n"
	    "        namespace B {\n"
	    "            class B {\n"
	    "            public:\n"
	    "                namespace C {\n"
	    "                    class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                            }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=stroustrup, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test indented namespaces within an indented class
	char text[] =
	    "\nnamespace A {\n"
	    "    class A {\n"
	    "        public:\n"
	    "            namespace B {\n"
	    "                class B {\n"
	    "                    public:\n"
	    "                        namespace C {\n"
	    "                            class C {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                    {\n"
	    "                                        if (isFoo) {\n"
	    "                                            bar();\n"
	    "                                        }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=stroustrup, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, Pico)
{
	// test stroustrup style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "    }\n"
	    "    else {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=stroustrup";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleStroustrupCpp, PicoOneLine)
{
	// test stroustrup style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else {\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=stroustrup";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ Whitesmith Style
// There are NO additional tests are in the braces tests
//----------------------------------------------------------------------------

struct StyleWhitesmithCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleWhitesmithCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleWhitesmithCppF, LongOption)
{
	// test whitesmith style option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "    {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleWhitesmithCppF, ShortOption)
{
	// test whitesmith style short option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "    {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A5";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, SpaceIndent)
{
	// test whitesmith style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "      {\n"
	    "      if (isFoo\n"
	    "                  && isBar)\n"
	    "            {\n"
	    "            bar();\n"
	    "            }\n"
	    "      else\n"
	    "            anotherBar();\n"
	    "      }\n";
	char options[] = "style=whitesmith, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, Tab)
{
	// test whitesmith style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "	{\n"
	    "	if (isFoo\n"
	    "	        && isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=whitesmith, indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, TabIndent)
{
	// test whitesmith style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "	{\n"
	    "	if (isFoo\n"
	    "	            && isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=whitesmith, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, ForceTab)
{
	// test whitesmith style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "	{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=whitesmith, indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, ForceTabIndent)
{
	// test whitesmith style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "	{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=whitesmith, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, Struct)
{
	// test whitesmith style with a struct
	// structs should be broken
	char text[] =
	    "\nstruct FooStruct\n"
	    "    {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, Enum)
{
	// test whitesmith style with an enum
	// enums should be broken
	char text[] =
	    "\nenum Encoding\n"
	    "    {\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "    };\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, MinConditionalIndent1)
{
	// whitesmith should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, MinConditionalIndent2)
{
	// test whitesmith style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo\n"
	    "        && isBar)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n";
	char options[] = "style=whitesmith, min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, IndentedSwitchBlock)
{
	// test whitesmith style with automatic switch block indent
	char text[] =
	    "\nvoid Foo(int fooBar)\n"
	    "    {\n"
	    "    switch (fooBar)\n"
	    "        {\n"
	    "        case 1:\n"
	    "            fooBar = 1;\n"
	    "            break;\n"
	    "        case 2:\n"
	    "            {\n"
	    "            fooBar = 2;\n"
	    "            }\n"
	    "        break;\n"
	    "        default:\n"
	    "            break;\n"
	    "        }\n"
	    "    int bar = true;\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedClass)
{
	// test whitesmith style nested classes
	char text[] =
	    "\nclass A\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "            {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                            {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                            }\n"
	    "                    }\n"
	    "            }\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedClass_IndentClass)
{
	// test whitesmith style nested classes with indented classes
	char text[] =
	    "\nclass A\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "            {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                            {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                            }\n"
	    "                    }\n"
	    "            }\n"
	    "    }\n";
	char options[] = "style=whitesmith, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedNamespace)
{
	// test whitesmith style nested namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "namespace B\n"
	    "{\n"
	    "namespace C\n"
	    "{\n"
	    "void foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedNamespace_IndentNamespace)
{
	// test whitesmith style nested indented namespaces
	char text[] =
	    "\nnamespace A\n"
	    "    {\n"
	    "    namespace B\n"
	    "        {\n"
	    "        namespace C\n"
	    "            {\n"
	    "            void foo(bool isFoo)\n"
	    "                {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                    bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=whitesmith, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedNamespaceClass)
{
	// test whitesmith style namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "class A\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace B\n"
	    "        {\n"
	    "        class B\n"
	    "            {\n"
	    "            public:\n"
	    "                namespace C\n"
	    "                {\n"
	    "                class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                            {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                            }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedNamespaceClass_IndentNamespace)
{
	// test whitesmith style indented namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "    {\n"
	    "    class A\n"
	    "        {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "                {\n"
	    "                class B\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                            {\n"
	    "                            class C\n"
	    "                                {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                        {\n"
	    "                                        if (isFoo)\n"
	    "                                            {\n"
	    "                                            bar();\n"
	    "                                            }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                        }\n"
	    "                                }\n"
	    "                            }\n"
	    "                    }\n"
	    "                }\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=whitesmith, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test  whitesmith style with indented namespaces within an indented class
	char text[] =
	    "\nnamespace A\n"
	    "    {\n"
	    "    class A\n"
	    "        {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "                {\n"
	    "                class B\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                            {\n"
	    "                            class C\n"
	    "                                {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                        {\n"
	    "                                        if (isFoo)\n"
	    "                                            {\n"
	    "                                            bar();\n"
	    "                                            }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                        }\n"
	    "                                }\n"
	    "                            }\n"
	    "                    }\n"
	    "                }\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=whitesmith, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, Const)
{
	// initial indent with whitesmith style when a const function is used
	char text[] =
	    "\nint Foo(bool isBar) const\n"
	    "    {\n"
	    "    if (isBar)\n"
	    "        {\n"
	    "        bar();\n"
	    "        return 1;\n"
	    "        }\n"
	    "    else\n"
	    "        return 0;\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, HorstmannComments)
{
	// test whitesmith style with Horstmann comments
	char textIn[] =
	    "\nbool foo()\n"
	    "{   while (confs)\n"
	    "    {   /*Replace all '|' with '_'.\n"
	    "         * This is vital.\n"
	    "         */\n"
	    "        ConfigName = Attribute(name);\n"
	    "    }\n"
	    "}\n";
	char text[] =
	    "\nbool foo()\n"
	    "    {\n"
	    "    while (confs)\n"
	    "        {\n"
	    "        /*Replace all '|' with '_'.\n"
	    "         * This is vital.\n"
	    "         */\n"
	    "        ConfigName = Attribute(name);\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, Arrays)
{
	// test whitesmith style with arrays
	char text[] =
	    "\nconst int cmdLineDesc[] =\n"
	    "    {\n"
	    "        { CMD_LINE_SWITCH1 },\n"
	    "        { CMD_LINE_SWITCH2 },\n"
	    "\n"
	    "        { CMD_LINE_SWITCH3 },\n"
	    "        { CMD_LINE_SWITCH4 },\n"
	    "\n"
	    "        { CMD_LINE_SWITCH5 },\n"
	    "        { CMD_LINE_SWITCH6 },\n"
	    "        { CMD_LINE_NONE }\n"
	    "    };\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, Pico)
{
	// test whitesmith style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "        }\n"
	    "    else\n"
	    "        {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleWhitesmithCpp, PicoOneLine)
{
	// test whitesmith style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "    {\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        {\n"
	    "        anotherBar();\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=whitesmith";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ VTK Style
// There are NO additional tests are in the braces tests
//----------------------------------------------------------------------------

struct StyleVTKCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleVTKCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleVTKCppF, LongOption)
{
	// test vtk style option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleVTKCppF, ShortOption)
{
	// test vtk style short option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A15";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, SpaceIndent)
{
	// test vtk style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "      if (isFoo\n"
	    "                  && isBar)\n"
	    "            {\n"
	    "            bar();\n"
	    "            }\n"
	    "      else\n"
	    "            anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, Tab)
{
	// test vtk style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	        && isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk, indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, TabIndent)
{
	// test vtk style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	            && isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, ForceTab)
{
	// test vtk style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk, indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, ForceTabIndent)
{
	// test vtk style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "		{\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, Struct)
{
	// test vtk style with a struct
	// structs should be broken
	char text[] =
	    "\nstruct FooStruct\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, Enum)
{
	// test vtk style with an enum
	// enums should be broken
	char text[] =
	    "\nenum Encoding\n"
	    "{\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "};\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, MinConditionalIndent1)
{
	// vtk should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, MinConditionalIndent2)
{
	// test vtk style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "        && isBar)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=vtk, min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, IndentedSwitchBlock)
{
	// test vtk style with automatic switch block indent
	char text[] =
	    "\nvoid Foo(int fooBar)\n"
	    "{\n"
	    "    switch (fooBar)\n"
	    "        {\n"
	    "        case 1:\n"
	    "            fooBar = 1;\n"
	    "            break;\n"
	    "        case 2:\n"
	    "            {\n"
	    "            fooBar = 2;\n"
	    "            }\n"
	    "        break;\n"
	    "        default:\n"
	    "            break;\n"
	    "        }\n"
	    "    int bar = true;\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedClass)
{
	// test vtk style nested classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                    bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            };\n"
	    "        };\n"
	    "    };\n"
	    "};\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedClass_IndentClass)
{
	// test vtk style nested classes with indented classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "        {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=vtk, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedNamespace)
{
	// test vtk style nested namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "namespace B\n"
	    "{\n"
	    "namespace C\n"
	    "{\n"
	    "void foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedNamespace_IndentNamespace)
{
	// test vtk style nested indented namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    namespace B\n"
	    "    {\n"
	    "        namespace C\n"
	    "        {\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                    bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=vtk, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedNamespaceClass)
{
	// test vtk style namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "class A\n"
	    "{\n"
	    "public:\n"
	    "    namespace B\n"
	    "    {\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace C\n"
	    "        {\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                    bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedNamespaceClass_IndentNamespace)
{
	// test vtk style indented namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace B\n"
	    "        {\n"
	    "            class B\n"
	    "            {\n"
	    "            public:\n"
	    "                namespace C\n"
	    "                {\n"
	    "                    class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=vtk, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test  vtk style with indented namespaces within an indented class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "            {\n"
	    "                class B\n"
	    "                {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                        {\n"
	    "                            class C\n"
	    "                            {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                    {\n"
	    "                                        if (isFoo)\n"
	    "                                            {\n"
	    "                                            bar();\n"
	    "                                            }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=vtk, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, Const)
{
	// initial indent with vtk style when a const function is used
	char text[] =
	    "\nint Foo(bool isBar) const\n"
	    "{\n"
	    "    if (isBar)\n"
	    "        {\n"
	    "        bar();\n"
	    "        return 1;\n"
	    "        }\n"
	    "    else\n"
	    "        return 0;\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, HorstmannComments)
{
	// test vtk style with Horstmann comments
	char textIn[] =
	    "\nbool foo()\n"
	    "{   while (confs)\n"
	    "    {   /*Replace all '|' with '_'.\n"
	    "         * This is vital.\n"
	    "         */\n"
	    "        ConfigName = Attribute(name);\n"
	    "}\n"
	    "}\n";
	char text[] =
	    "\nbool foo()\n"
	    "{\n"
	    "    while (confs)\n"
	    "        {\n"
	    "        /*Replace all '|' with '_'.\n"
	    "         * This is vital.\n"
	    "         */\n"
	    "        ConfigName = Attribute(name);\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, Arrays)
{
	// test vtk style with arrays
	char text[] =
	    "\nconst int cmdLineDesc[] =\n"
	    "{\n"
	    "    { CMD_LINE_SWITCH1 },\n"
	    "    { CMD_LINE_SWITCH2 },\n"
	    "\n"
	    "    { CMD_LINE_SWITCH3 },\n"
	    "    { CMD_LINE_SWITCH4 },\n"
	    "\n"
	    "    { CMD_LINE_SWITCH5 },\n"
	    "    { CMD_LINE_SWITCH6 },\n"
	    "    { CMD_LINE_NONE }\n"
	    "};\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, Pico)
{
	// test vtk style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "        }\n"
	    "    else\n"
	    "        {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleVTKCpp, PicoOneLine)
{
	// test vtk style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        {\n"
	    "        anotherBar();\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=vtk";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ Banner Style
// There are NO additional tests are in the braces tests
//----------------------------------------------------------------------------

struct StyleBannerCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleBannerCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleBannerCppF, BannerOption)
{
	// test banner style option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleBannerCppF, RatliffOption)
{
	// test ratliff style option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=ratliff";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleBannerCppF, ShortOption)
{
	// test banner style short option
	char text[] =
	    "\nnamespace FooName {\n"
	    "\n"
	    "class FooClass {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, SpaceIndent)
{
	// test banner style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "      if (isFoo\n"
	    "                  && isBar) {\n"
	    "            bar();\n"
	    "            }\n"
	    "      else\n"
	    "            anotherBar();\n"
	    "      }\n";
	char options[] = "style=banner, indent=spaces=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, Tab)
{
	// test banner style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "	        && isBar) {\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=banner, indent=tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, TabIndent)
{
	// test banner style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "	            && isBar) {\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=banner, indent=tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, ForceTab)
{
	// test banner style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=banner, indent=force-tab";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, ForceTabIndent)
{
	// test banner style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "	if (isFoo\n"
	    "			&& isBar) {\n"
	    "		bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "	}\n";
	char options[] = "style=banner, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, Struct)
{
	// test banner style with a struct
	// structs should be attached
	char text[] =
	    "\nstruct FooStruct {\n"
	    "    private:\n"
	    "        bool var1;\n"
	    "        void func1();\n"
	    "    protected:\n"
	    "        bool var2;\n"
	    "        void func2();\n"
	    "    };\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, Enum)
{
	// test banner style with an enum
	// enums should be attached
	char text[] =
	    "\nenum Encoding {\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "    };\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, MinConditionalIndent1)
{
	// banner should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo\n"
	    "            && isBar) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, MinConditionalIndent2)
{
	// test banner style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo\n"
	    "        && isBar) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n";
	char options[] = "style=banner, min-conditional-indent=0";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, IndentedSwitchBlock)
{
	// test banner style with automatic switch block indent
	char text[] =
	    "\nvoid Foo(int fooBar) {\n"
	    "    switch (fooBar) {\n"
	    "        case 1:\n"
	    "            fooBar = 1;\n"
	    "            break;\n"
	    "        case 2: {\n"
	    "            fooBar = 2;\n"
	    "            }\n"
	    "        break;\n"
	    "        default:\n"
	    "            break;\n"
	    "        }\n"
	    "    int bar = true;\n"
	    "    }\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedClass)
{
	// test banner style nested classes
	char text[] =
	    "\nclass A {\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo) {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                            }\n"
	    "                    }\n"
	    "            }\n"
	    "    }\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedClass_IndentClass)
{
	// test banner style nested classes with indented classes
	char text[] =
	    "\nclass A {\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo) {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                            }\n"
	    "                    }\n"
	    "            }\n"
	    "    }\n";
	char options[] = "style=banner, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedNamespace)
{
	// test banner style nested namespaces
	char text[] =
	    "\nnamespace A {\n"
	    "namespace B {\n"
	    "namespace C {\n"
	    "void foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "    }\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedNamespace_IndentNamespace)
{
	// test banner style nested indented namespaces
	char text[] =
	    "\nnamespace A {\n"
	    "    namespace B {\n"
	    "        namespace C {\n"
	    "            void foo(bool isFoo) {\n"
	    "                if (isFoo) {\n"
	    "                    bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=banner, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedNamespaceClass)
{
	// test banner style namespaces within a class
	char text[] =
	    "\nnamespace A {\n"
	    "class A {\n"
	    "    public:\n"
	    "    namespace B {\n"
	    "        class B {\n"
	    "            public:\n"
	    "            namespace C {\n"
	    "                class C {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo) {\n"
	    "                            if (isFoo) {\n"
	    "                                bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                            }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedNamespaceClass_IndentNamespace)
{
	// test banner style indented namespaces within a class
	char text[] =
	    "\nnamespace A {\n"
	    "    class A {\n"
	    "        public:\n"
	    "            namespace B {\n"
	    "                class B {\n"
	    "                    public:\n"
	    "                        namespace C {\n"
	    "                            class C {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo) {\n"
	    "                                        if (isFoo) {\n"
	    "                                            bar();\n"
	    "                                            }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                        }\n"
	    "                                }\n"
	    "                            }\n"
	    "                    }\n"
	    "                }\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=banner, indent-namespaces";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test  banner style with indented namespaces within an indented class
	char text[] =
	    "\nnamespace A {\n"
	    "    class A {\n"
	    "        public:\n"
	    "            namespace B {\n"
	    "                class B {\n"
	    "                    public:\n"
	    "                        namespace C {\n"
	    "                            class C {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo) {\n"
	    "                                        if (isFoo) {\n"
	    "                                            bar();\n"
	    "                                            }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                        }\n"
	    "                                }\n"
	    "                            }\n"
	    "                    }\n"
	    "                }\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=banner, indent-namespaces, indent-classes";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, Const)
{
	// initial indent with banner style when a const function is used
	char text[] =
	    "\nint Foo(bool isBar) const {\n"
	    "    if (isBar) {\n"
	    "        bar();\n"
	    "        return 1;\n"
	    "        }\n"
	    "    else\n"
	    "        return 0;\n"
	    "    }\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, Arrays)
{
	// test banner style with arrays
	char text[] =
	    "\nconst int cmdLineDesc[] = {\n"
	    "        { CMD_LINE_SWITCH1 },\n"
	    "        { CMD_LINE_SWITCH2 },\n"
	    "\n"
	    "        { CMD_LINE_SWITCH3 },\n"
	    "        { CMD_LINE_SWITCH4 },\n"
	    "\n"
	    "        { CMD_LINE_SWITCH5 },\n"
	    "        { CMD_LINE_SWITCH6 },\n"
	    "        { CMD_LINE_NONE }\n"
	    "    };\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, Pico)
{
	// test banner style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "    {   anotherBar1();\n"
	    "        anotherBar2(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar1();\n"
	    "        bar2();\n"
	    "        }\n"
	    "    else {\n"
	    "        anotherBar1();\n"
	    "        anotherBar2();\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleBannerCpp, PicoOneLine)
{
	// test banner style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "    {   anotherBar(); } }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo) {\n"
	    "    if (isFoo) {\n"
	    "        bar();\n"
	    "        }\n"
	    "    else {\n"
	    "        anotherBar();\n"
	    "        }\n"
	    "    }\n";
	char options[] = "style=banner";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------
// AStyle C++ Gnu Style
// There are NO additional tests are in the braces tests
//----------------------------------------------------------------------------

struct StyleGnuCppF : public Test
{
	string textStr;
	const char* textIn;

	StyleGnuCppF()
	{
		textStr =
		    "\nnamespace FooName\n"
		    "{\n"
		    "\n"
		    "class FooClass\n"
		    "{\n"
		    "private:\n"
		    "    bool var1;\n"
		    "    void func1();\n"
		    "protected:\n"
		    "    bool var2;\n"
		    "    void func2();\n"
		    "};\n"
		    "\n"
		    "void FooClass::Foo(bool isFoo)\n"
		    "{\n"
		    "    if (isFoo)\n"
		    "    {\n"
		    "        bar();\n"
		    "    }\n"
		    "    else\n"
		    "        anotherBar();\n"
		    "}\n"
		    "\n"
		    "}   // end FooName\n";
		textIn = textStr.c_str();
	}
};

TEST_F(StyleGnuCppF, LongOption)
{
	// test gnu style option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "  bool var1;\n"
	    "  void func1();\n"
	    "protected:\n"
	    "  bool var2;\n"
	    "  void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "  if (isFoo)\n"
	    "    {\n"
	    "      bar();\n"
	    "    }\n"
	    "  else\n"
	    "    anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "style=gnu, indent=spaces=2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST_F(StyleGnuCppF, ShortOption)
{
	// test gnu style short option
	char text[] =
	    "\nnamespace FooName\n"
	    "{\n"
	    "\n"
	    "class FooClass\n"
	    "{\n"
	    "private:\n"
	    "  bool var1;\n"
	    "  void func1();\n"
	    "protected:\n"
	    "  bool var2;\n"
	    "  void func2();\n"
	    "};\n"
	    "\n"
	    "void FooClass::Foo(bool isFoo)\n"
	    "{\n"
	    "  if (isFoo)\n"
	    "    {\n"
	    "      bar();\n"
	    "    }\n"
	    "  else\n"
	    "    anotherBar();\n"
	    "}\n"
	    "\n"
	    "}   // end FooName\n";
	char options[] = "-A7, indent=spaces=2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, SpaceIndent)
{
	// test gnu style option with space indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "   if (isFoo\n"
	    "         && isBar)\n"
	    "      {\n"
	    "         bar();\n"
	    "      }\n"
	    "   else\n"
	    "      anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=3";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Tab)
{
	// test gnu style option with tab indent
	// default indent is 2
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	    && isBar)\n"
	    "		{\n"
	    "			bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, indent=tab=2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, TabIndent)
{
	// test gnu style option with tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "	      && isBar)\n"
	    "		{\n"
	    "			bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, indent=tab=3";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, ForceTab)
{
	// test gnu style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "		{\n"
	    "			bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, indent=force-tab=2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, ForceTabIndent)
{
	// test gnu style option with force tab indent
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "	if (isFoo\n"
	    "			&& isBar)\n"
	    "		{\n"
	    "			bar();\n"
	    "		}\n"
	    "	else\n"
	    "		anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, indent=force-tab=6";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Struct1)
{
	// test gnu style with a struct
	// structs should be broken
	char text[] =
	    "\nstruct FooStruct\n"
	    "{\n"
	    "private:\n"
	    "    bool var1;\n"
	    "    void func1();\n"
	    "protected:\n"
	    "    bool var2;\n"
	    "    void func2();\n"
	    "};\n";
	char options[] = "style=gnu";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Struct2)
{
	// test gnu style with structs and unions
	// the opening brace should NOT be indented
	char text[] =
	    "\nstruct b_type\n"
	    "{\n"
	    "    int i;\n"
	    "    int j;\n"
	    "};\n"
	    "\n"
	    "union b_type\n"
	    "{\n"
	    "    int i;\n"
	    "    int j;\n"
	    "};\n"
	    "\n"
	    "// struct with inheritance and method\n"
	    "struct b_fooStruct : public bar\n"
	    "{\n"
	    "    Foobar(bool isBar)\n"
	    "    {\n"
	    "        if (isBar)\n"
	    "            {\n"
	    "                Foo();\n"
	    "            }\n"
	    "        else\n"
	    "            bar();\n"
	    "        index = i;\n"
	    "    }\n"
	    "    long index;\n"
	    "};\n";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Enum)
{
	// test gnu style with an enum
	// enums should be broken
	char text[] =
	    "\nenum Encoding\n"
	    "{\n"
	    "    ENCODING_8BIT,\n"
	    "    UTF_16BE,\n"
	    "    UTF_16LE\n"
	    "};\n";
	char options[] = "style=gnu";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, MinConditionalIndent1)
{
	// gnu should use a default setting of MINCOND_TWO
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "  if (isFoo\n"
	    "      && isBar)\n"
	    "    {\n"
	    "      bar();\n"
	    "    }\n"
	    "  else\n"
	    "    anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, MinConditionalIndent2)
{
	// test gnu style option with min conditional indent 0
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo\n"
	    "            && isBar)\n"
	    "    {\n"
	    "        bar();\n"
	    "    }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "  if (isFoo\n"
	    "      && isBar)\n"
	    "    {\n"
	    "      bar();\n"
	    "    }\n"
	    "  else\n"
	    "    anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu, min-conditional-indent=0, indent=spaces=2";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, SwitchBlock)
{
	// test gnu style with switch blocks
	char text[] =
	    "\nvoid Foo(int fooBar)\n"
	    "{\n"
	    "    switch (fooBar)\n"
	    "        {\n"
	    "        case 1:\n"
	    "            fooBar = 1;\n"
	    "            break;\n"
	    "        case 2:\n"
	    "        {\n"
	    "            fooBar = 2;\n"
	    "        }\n"
	    "        break;\n"
	    "        default:\n"
	    "            break;\n"
	    "        }\n"
	    "    int bar = true;\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, IndentedSwitchBlock)
{
	// test gnu style with indented switch blocks
	char text[] =
	    "\nvoid Foo(int fooBar)\n"
	    "{\n"
	    "    switch (fooBar)\n"
	    "        {\n"
	    "            case 1:\n"
	    "                fooBar = 1;\n"
	    "                break;\n"
	    "            case 2:\n"
	    "            {\n"
	    "                fooBar = 2;\n"
	    "            }\n"
	    "            break;\n"
	    "            default:\n"
	    "                break;\n"
	    "        }\n"
	    "    int bar = true;\n"
	    "}\n";
	char options[] = "style=gnu, indent-switches";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedClass)
{
	// test gnu style with nested classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "public:\n"
	    "    int foo1;\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        int foo2;\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                        bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedClass_IndentClass)
{
	// test gnu style with nested classes and indented classes
	char text[] =
	    "\nclass A\n"
	    "{\n"
	    "    public:\n"
	    "        int foo1;\n"
	    "        class B\n"
	    "        {\n"
	    "            public:\n"
	    "                int foo2;\n"
	    "                class C\n"
	    "                {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                    bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                }\n"
	    "        }\n"
	    "}\n";
	char options[] = "style=gnu, indent-classes, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedNamespace)
{
	// test gnu style nested namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "namespace B\n"
	    "{\n"
	    "namespace C\n"
	    "{\n"
	    "void foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "            bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n"
	    "}\n"
	    "}\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedNamespace_IndentNamespace)
{
	// test gnu style with nested indented namespaces
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    namespace B\n"
	    "    {\n"
	    "        namespace C\n"
	    "        {\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                        bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=gnu, indent-namespaces, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedNamespaceClass)
{
	// test gnu style with namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "class A\n"
	    "{\n"
	    "public:\n"
	    "    namespace B\n"
	    "    {\n"
	    "    class B\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace C\n"
	    "        {\n"
	    "        class C\n"
	    "        {\n"
	    "        public:\n"
	    "            void foo(bool isFoo)\n"
	    "            {\n"
	    "                if (isFoo)\n"
	    "                    {\n"
	    "                        bar();\n"
	    "                    }\n"
	    "                else\n"
	    "                    anotherBar();\n"
	    "            }\n"
	    "        }\n"
	    "        }\n"
	    "    }\n"
	    "    }\n"
	    "}\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedNamespaceClass_IndentNamespace)
{
	// test gnu style with indented namespaces within a class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "    public:\n"
	    "        namespace B\n"
	    "        {\n"
	    "            class B\n"
	    "            {\n"
	    "            public:\n"
	    "                namespace C\n"
	    "                {\n"
	    "                    class C\n"
	    "                    {\n"
	    "                    public:\n"
	    "                        void foo(bool isFoo)\n"
	    "                        {\n"
	    "                            if (isFoo)\n"
	    "                                {\n"
	    "                                    bar();\n"
	    "                                }\n"
	    "                            else\n"
	    "                                anotherBar();\n"
	    "                        }\n"
	    "                    }\n"
	    "                }\n"
	    "            }\n"
	    "        }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=gnu, indent-namespaces, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, NestedNamespaceClass_IndentNamespaceClass)
{
	// test gnu style with indented namespaces within an indented class
	char text[] =
	    "\nnamespace A\n"
	    "{\n"
	    "    class A\n"
	    "    {\n"
	    "        public:\n"
	    "            namespace B\n"
	    "            {\n"
	    "                class B\n"
	    "                {\n"
	    "                    public:\n"
	    "                        namespace C\n"
	    "                        {\n"
	    "                            class C\n"
	    "                            {\n"
	    "                                public:\n"
	    "                                    void foo(bool isFoo)\n"
	    "                                    {\n"
	    "                                        if (isFoo)\n"
	    "                                            {\n"
	    "                                                bar();\n"
	    "                                            }\n"
	    "                                        else\n"
	    "                                            anotherBar();\n"
	    "                                    }\n"
	    "                            }\n"
	    "                        }\n"
	    "                }\n"
	    "            }\n"
	    "    }\n"
	    "}\n";
	char options[] = "style=gnu, indent-namespaces, indent-classes, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Const1)
{
	// no initial indent with gnu style when a const function is used
	char text[] =
	    "\nint Foo(bool isBar) const\n"
	    "{\n"
	    "  if (isBar)\n"
	    "    {\n"
	    "      bar();\n"
	    "      return 1;\n"
	    "    }\n"
	    "  else\n"
	    "    return 0;\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=2";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Const2)
{
	// no initial indent with gnu style when a const function is used
	char text[] =
	    "\nint Foo(bool isBar) const\n"
	    "{\n"
	    "  if (isBar)\n"
	    "    {\n"
	    "      bar();\n"
	    "      return 1;\n"
	    "    }\n"
	    "  else\n"
	    "    return 0;\n"
	    "}\n";
	char options[] = "style=gnu, indent=spaces=2";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, ConstVolatile)
{
	// Test gnu style with "const volatile".
	// The keyword "Volatile" used as a  type qualifier caused the data after the "if"
	// statement to receive 1 less indent than needed.
	char text[] =
	    "\nbool MDMsgBatchManager::IsReindexRequired () const volatile\n"
	    "{\n"
	    "    bool result = false;\n"
	    "    if (isReindexed)\n"
	    "        {\n"
	    "            result = true;\n"
	    "        }\n"
	    "    return result;\n"
	    "}";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, VolatileConst)
{
	// Test gnu style with "volatile const".
	// The keyword "Volatile" used as a  type qualifier caused the data after the "if"
	// statement to receive 1 less indent than needed.
	char text[] =
	    "\nbool MDMsgBatchManager::IsReindexRequired () volatile const\n"
	    "{\n"
	    "    bool result = false;\n"
	    "    if (isReindexed)\n"
	    "        {\n"
	    "            result = true;\n"
	    "        }\n"
	    "    return result;\n"
	    "}";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, VolatileOnly)
{
	// Test gnu style with "volatile" only.
	// The keyword "volatile" used as a  type qualifier caused the data after the "if"
	// statement to receive 1 less indent than needed.
	char text[] =
	    "\nbool MDMsgBatchManager::IsReindexRequired () volatile\n"
	    "{\n"
	    "    bool result = false;\n"
	    "    if (isReindexed)\n"
	    "        {\n"
	    "            result = true;\n"
	    "        }\n"
	    "    return result;\n"
	    "}";
	char options[] = "style=gnu, indent=spaces=4";
	char* textOut = AStyleMain(text, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, Pico)
{
	// test gnu style with pico braces
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar1();\n"
	    "        bar2(); }\n"
	    "    else\n"
	    "        anotherBar1(); }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "            bar1();\n"
	    "            bar2();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar1();\n"
	    "}\n";
	char options[] = "style=gnu";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

TEST(StyleGnuCpp, PicoOneLine)
{
	// test gnu style with pico braces and one-line blocks
	char textIn[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{   if (isFoo)\n"
	    "    {   bar(); }\n"
	    "    else\n"
	    "        anotherBar(); }\n";
	char text[] =
	    "\nvoid Foo(bool isFoo)\n"
	    "{\n"
	    "    if (isFoo)\n"
	    "        {\n"
	    "            bar();\n"
	    "        }\n"
	    "    else\n"
	    "        anotherBar();\n"
	    "}\n";
	char options[] = "style=gnu";
	char* textOut = AStyleMain(textIn, options, errorHandler, memoryAlloc);
	EXPECT_STREQ(text, textOut);
	delete[] textOut;
}

//----------------------------------------------------------------------------

}  // namespace
