--
--  Copyright (C) 2022-2023, AdaCore
--
--  SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
--

pragma Ada_2022;
pragma Style_Checks ("M131");

--  This package contains data from index-jis0208.txt file
--
--  # Identifier: cbaa91f3deb7d0841faf5c33041fc15a285da0e87e64ab802c4bf04b7c4da861
--  # Date: 2018-01-06
--
--  see the Encoding Standard
--
--  https://encoding.spec.whatwg.org/

with Interfaces;

private package VSS.Strings.Converters.Decoders.Index_JIS0208 is

   Table : constant
     array (Interfaces.Unsigned_32 range 0 ..  11_103)
       of VSS.Unicode.Code_Point :=
     [0      => 16#3000#,  --  　 (IDEOGRAPHIC SPACE)
      1      => 16#3001#,  --  、 (IDEOGRAPHIC COMMA)
      2      => 16#3002#,  --  。 (IDEOGRAPHIC FULL STOP)
      3      => 16#FF0C#,  --  ， (FULLWIDTH COMMA)
      4      => 16#FF0E#,  --  ． (FULLWIDTH FULL STOP)
      5      => 16#30FB#,  --  ・ (KATAKANA MIDDLE DOT)
      6      => 16#FF1A#,  --  ： (FULLWIDTH COLON)
      7      => 16#FF1B#,  --  ； (FULLWIDTH SEMICOLON)
      8      => 16#FF1F#,  --  ？ (FULLWIDTH QUESTION MARK)
      9      => 16#FF01#,  --  ！ (FULLWIDTH EXCLAMATION MARK)
      10     => 16#309B#,  --  ゛ (KATAKANA-HIRAGANA VOICED SOUND MARK)
      11     => 16#309C#,  --  ゜ (KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK)
      12     => 16#00B4#,  --  ´ (ACUTE ACCENT)
      13     => 16#FF40#,  --  ｀ (FULLWIDTH GRAVE ACCENT)
      14     => 16#00A8#,  --  ¨ (DIAERESIS)
      15     => 16#FF3E#,  --  ＾ (FULLWIDTH CIRCUMFLEX ACCENT)
      16     => 16#FFE3#,  --  ￣ (FULLWIDTH MACRON)
      17     => 16#FF3F#,  --  ＿ (FULLWIDTH LOW LINE)
      18     => 16#30FD#,  --  ヽ (KATAKANA ITERATION MARK)
      19     => 16#30FE#,  --  ヾ (KATAKANA VOICED ITERATION MARK)
      20     => 16#309D#,  --  ゝ (HIRAGANA ITERATION MARK)
      21     => 16#309E#,  --  ゞ (HIRAGANA VOICED ITERATION MARK)
      22     => 16#3003#,  --  〃 (DITTO MARK)
      23     => 16#4EDD#,  --  仝 (<CJK Ideograph>)
      24     => 16#3005#,  --  々 (IDEOGRAPHIC ITERATION MARK)
      25     => 16#3006#,  --  〆 (IDEOGRAPHIC CLOSING MARK)
      26     => 16#3007#,  --  〇 (IDEOGRAPHIC NUMBER ZERO)
      27     => 16#30FC#,  --  ー (KATAKANA-HIRAGANA PROLONGED SOUND MARK)
      28     => 16#2015#,  --  ― (HORIZONTAL BAR)
      29     => 16#2010#,  --  ‐ (HYPHEN)
      30     => 16#FF0F#,  --  ／ (FULLWIDTH SOLIDUS)
      31     => 16#FF3C#,  --  ＼ (FULLWIDTH REVERSE SOLIDUS)
      32     => 16#FF5E#,  --  ～ (FULLWIDTH TILDE)
      33     => 16#2225#,  --  ∥ (PARALLEL TO)
      34     => 16#FF5C#,  --  ｜ (FULLWIDTH VERTICAL LINE)
      35     => 16#2026#,  --  … (HORIZONTAL ELLIPSIS)
      36     => 16#2025#,  --  ‥ (TWO DOT LEADER)
      37     => 16#2018#,  --  ‘ (LEFT SINGLE QUOTATION MARK)
      38     => 16#2019#,  --  ’ (RIGHT SINGLE QUOTATION MARK)
      39     => 16#201C#,  --  “ (LEFT DOUBLE QUOTATION MARK)
      40     => 16#201D#,  --  ” (RIGHT DOUBLE QUOTATION MARK)
      41     => 16#FF08#,  --  （ (FULLWIDTH LEFT PARENTHESIS)
      42     => 16#FF09#,  --  ） (FULLWIDTH RIGHT PARENTHESIS)
      43     => 16#3014#,  --  〔 (LEFT TORTOISE SHELL BRACKET)
      44     => 16#3015#,  --  〕 (RIGHT TORTOISE SHELL BRACKET)
      45     => 16#FF3B#,  --  ［ (FULLWIDTH LEFT SQUARE BRACKET)
      46     => 16#FF3D#,  --  ］ (FULLWIDTH RIGHT SQUARE BRACKET)
      47     => 16#FF5B#,  --  ｛ (FULLWIDTH LEFT CURLY BRACKET)
      48     => 16#FF5D#,  --  ｝ (FULLWIDTH RIGHT CURLY BRACKET)
      49     => 16#3008#,  --  〈 (LEFT ANGLE BRACKET)
      50     => 16#3009#,  --  〉 (RIGHT ANGLE BRACKET)
      51     => 16#300A#,  --  《 (LEFT DOUBLE ANGLE BRACKET)
      52     => 16#300B#,  --  》 (RIGHT DOUBLE ANGLE BRACKET)
      53     => 16#300C#,  --  「 (LEFT CORNER BRACKET)
      54     => 16#300D#,  --  」 (RIGHT CORNER BRACKET)
      55     => 16#300E#,  --  『 (LEFT WHITE CORNER BRACKET)
      56     => 16#300F#,  --  』 (RIGHT WHITE CORNER BRACKET)
      57     => 16#3010#,  --  【 (LEFT BLACK LENTICULAR BRACKET)
      58     => 16#3011#,  --  】 (RIGHT BLACK LENTICULAR BRACKET)
      59     => 16#FF0B#,  --  ＋ (FULLWIDTH PLUS SIGN)
      60     => 16#FF0D#,  --  － (FULLWIDTH HYPHEN-MINUS)
      61     => 16#00B1#,  --  ± (PLUS-MINUS SIGN)
      62     => 16#00D7#,  --  × (MULTIPLICATION SIGN)
      63     => 16#00F7#,  --  ÷ (DIVISION SIGN)
      64     => 16#FF1D#,  --  ＝ (FULLWIDTH EQUALS SIGN)
      65     => 16#2260#,  --  ≠ (NOT EQUAL TO)
      66     => 16#FF1C#,  --  ＜ (FULLWIDTH LESS-THAN SIGN)
      67     => 16#FF1E#,  --  ＞ (FULLWIDTH GREATER-THAN SIGN)
      68     => 16#2266#,  --  ≦ (LESS-THAN OVER EQUAL TO)
      69     => 16#2267#,  --  ≧ (GREATER-THAN OVER EQUAL TO)
      70     => 16#221E#,  --  ∞ (INFINITY)
      71     => 16#2234#,  --  ∴ (THEREFORE)
      72     => 16#2642#,  --  ♂ (MALE SIGN)
      73     => 16#2640#,  --  ♀ (FEMALE SIGN)
      74     => 16#00B0#,  --  ° (DEGREE SIGN)
      75     => 16#2032#,  --  ′ (PRIME)
      76     => 16#2033#,  --  ″ (DOUBLE PRIME)
      77     => 16#2103#,  --  ℃ (DEGREE CELSIUS)
      78     => 16#FFE5#,  --  ￥ (FULLWIDTH YEN SIGN)
      79     => 16#FF04#,  --  ＄ (FULLWIDTH DOLLAR SIGN)
      80     => 16#FFE0#,  --  ￠ (FULLWIDTH CENT SIGN)
      81     => 16#FFE1#,  --  ￡ (FULLWIDTH POUND SIGN)
      82     => 16#FF05#,  --  ％ (FULLWIDTH PERCENT SIGN)
      83     => 16#FF03#,  --  ＃ (FULLWIDTH NUMBER SIGN)
      84     => 16#FF06#,  --  ＆ (FULLWIDTH AMPERSAND)
      85     => 16#FF0A#,  --  ＊ (FULLWIDTH ASTERISK)
      86     => 16#FF20#,  --  ＠ (FULLWIDTH COMMERCIAL AT)
      87     => 16#00A7#,  --  § (SECTION SIGN)
      88     => 16#2606#,  --  ☆ (WHITE STAR)
      89     => 16#2605#,  --  ★ (BLACK STAR)
      90     => 16#25CB#,  --  ○ (WHITE CIRCLE)
      91     => 16#25CF#,  --  ● (BLACK CIRCLE)
      92     => 16#25CE#,  --  ◎ (BULLSEYE)
      93     => 16#25C7#,  --  ◇ (WHITE DIAMOND)
      94     => 16#25C6#,  --  ◆ (BLACK DIAMOND)
      95     => 16#25A1#,  --  □ (WHITE SQUARE)
      96     => 16#25A0#,  --  ■ (BLACK SQUARE)
      97     => 16#25B3#,  --  △ (WHITE UP-POINTING TRIANGLE)
      98     => 16#25B2#,  --  ▲ (BLACK UP-POINTING TRIANGLE)
      99     => 16#25BD#,  --  ▽ (WHITE DOWN-POINTING TRIANGLE)
      100    => 16#25BC#,  --  ▼ (BLACK DOWN-POINTING TRIANGLE)
      101    => 16#203B#,  --  ※ (REFERENCE MARK)
      102    => 16#3012#,  --  〒 (POSTAL MARK)
      103    => 16#2192#,  --  → (RIGHTWARDS ARROW)
      104    => 16#2190#,  --  ← (LEFTWARDS ARROW)
      105    => 16#2191#,  --  ↑ (UPWARDS ARROW)
      106    => 16#2193#,  --  ↓ (DOWNWARDS ARROW)
      107    => 16#3013#,  --  〓 (GETA MARK)
      119    => 16#2208#,  --  ∈ (ELEMENT OF)
      120    => 16#220B#,  --  ∋ (CONTAINS AS MEMBER)
      121    => 16#2286#,  --  ⊆ (SUBSET OF OR EQUAL TO)
      122    => 16#2287#,  --  ⊇ (SUPERSET OF OR EQUAL TO)
      123    => 16#2282#,  --  ⊂ (SUBSET OF)
      124    => 16#2283#,  --  ⊃ (SUPERSET OF)
      125    => 16#222A#,  --  ∪ (UNION)
      126    => 16#2229#,  --  ∩ (INTERSECTION)
      135    => 16#2227#,  --  ∧ (LOGICAL AND)
      136    => 16#2228#,  --  ∨ (LOGICAL OR)
      137    => 16#FFE2#,  --  ￢ (FULLWIDTH NOT SIGN)
      138    => 16#21D2#,  --  ⇒ (RIGHTWARDS DOUBLE ARROW)
      139    => 16#21D4#,  --  ⇔ (LEFT RIGHT DOUBLE ARROW)
      140    => 16#2200#,  --  ∀ (FOR ALL)
      141    => 16#2203#,  --  ∃ (THERE EXISTS)
      153    => 16#2220#,  --  ∠ (ANGLE)
      154    => 16#22A5#,  --  ⊥ (UP TACK)
      155    => 16#2312#,  --  ⌒ (ARC)
      156    => 16#2202#,  --  ∂ (PARTIAL DIFFERENTIAL)
      157    => 16#2207#,  --  ∇ (NABLA)
      158    => 16#2261#,  --  ≡ (IDENTICAL TO)
      159    => 16#2252#,  --  ≒ (APPROXIMATELY EQUAL TO OR THE IMAGE OF)
      160    => 16#226A#,  --  ≪ (MUCH LESS-THAN)
      161    => 16#226B#,  --  ≫ (MUCH GREATER-THAN)
      162    => 16#221A#,  --  √ (SQUARE ROOT)
      163    => 16#223D#,  --  ∽ (REVERSED TILDE)
      164    => 16#221D#,  --  ∝ (PROPORTIONAL TO)
      165    => 16#2235#,  --  ∵ (BECAUSE)
      166    => 16#222B#,  --  ∫ (INTEGRAL)
      167    => 16#222C#,  --  ∬ (DOUBLE INTEGRAL)
      175    => 16#212B#,  --  Å (ANGSTROM SIGN)
      176    => 16#2030#,  --  ‰ (PER MILLE SIGN)
      177    => 16#266F#,  --  ♯ (MUSIC SHARP SIGN)
      178    => 16#266D#,  --  ♭ (MUSIC FLAT SIGN)
      179    => 16#266A#,  --  ♪ (EIGHTH NOTE)
      180    => 16#2020#,  --  † (DAGGER)
      181    => 16#2021#,  --  ‡ (DOUBLE DAGGER)
      182    => 16#00B6#,  --  ¶ (PILCROW SIGN)
      187    => 16#25EF#,  --  ◯ (LARGE CIRCLE)
      203    => 16#FF10#,  --  ０ (FULLWIDTH DIGIT ZERO)
      204    => 16#FF11#,  --  １ (FULLWIDTH DIGIT ONE)
      205    => 16#FF12#,  --  ２ (FULLWIDTH DIGIT TWO)
      206    => 16#FF13#,  --  ３ (FULLWIDTH DIGIT THREE)
      207    => 16#FF14#,  --  ４ (FULLWIDTH DIGIT FOUR)
      208    => 16#FF15#,  --  ５ (FULLWIDTH DIGIT FIVE)
      209    => 16#FF16#,  --  ６ (FULLWIDTH DIGIT SIX)
      210    => 16#FF17#,  --  ７ (FULLWIDTH DIGIT SEVEN)
      211    => 16#FF18#,  --  ８ (FULLWIDTH DIGIT EIGHT)
      212    => 16#FF19#,  --  ９ (FULLWIDTH DIGIT NINE)
      220    => 16#FF21#,  --  Ａ (FULLWIDTH LATIN CAPITAL LETTER A)
      221    => 16#FF22#,  --  Ｂ (FULLWIDTH LATIN CAPITAL LETTER B)
      222    => 16#FF23#,  --  Ｃ (FULLWIDTH LATIN CAPITAL LETTER C)
      223    => 16#FF24#,  --  Ｄ (FULLWIDTH LATIN CAPITAL LETTER D)
      224    => 16#FF25#,  --  Ｅ (FULLWIDTH LATIN CAPITAL LETTER E)
      225    => 16#FF26#,  --  Ｆ (FULLWIDTH LATIN CAPITAL LETTER F)
      226    => 16#FF27#,  --  Ｇ (FULLWIDTH LATIN CAPITAL LETTER G)
      227    => 16#FF28#,  --  Ｈ (FULLWIDTH LATIN CAPITAL LETTER H)
      228    => 16#FF29#,  --  Ｉ (FULLWIDTH LATIN CAPITAL LETTER I)
      229    => 16#FF2A#,  --  Ｊ (FULLWIDTH LATIN CAPITAL LETTER J)
      230    => 16#FF2B#,  --  Ｋ (FULLWIDTH LATIN CAPITAL LETTER K)
      231    => 16#FF2C#,  --  Ｌ (FULLWIDTH LATIN CAPITAL LETTER L)
      232    => 16#FF2D#,  --  Ｍ (FULLWIDTH LATIN CAPITAL LETTER M)
      233    => 16#FF2E#,  --  Ｎ (FULLWIDTH LATIN CAPITAL LETTER N)
      234    => 16#FF2F#,  --  Ｏ (FULLWIDTH LATIN CAPITAL LETTER O)
      235    => 16#FF30#,  --  Ｐ (FULLWIDTH LATIN CAPITAL LETTER P)
      236    => 16#FF31#,  --  Ｑ (FULLWIDTH LATIN CAPITAL LETTER Q)
      237    => 16#FF32#,  --  Ｒ (FULLWIDTH LATIN CAPITAL LETTER R)
      238    => 16#FF33#,  --  Ｓ (FULLWIDTH LATIN CAPITAL LETTER S)
      239    => 16#FF34#,  --  Ｔ (FULLWIDTH LATIN CAPITAL LETTER T)
      240    => 16#FF35#,  --  Ｕ (FULLWIDTH LATIN CAPITAL LETTER U)
      241    => 16#FF36#,  --  Ｖ (FULLWIDTH LATIN CAPITAL LETTER V)
      242    => 16#FF37#,  --  Ｗ (FULLWIDTH LATIN CAPITAL LETTER W)
      243    => 16#FF38#,  --  Ｘ (FULLWIDTH LATIN CAPITAL LETTER X)
      244    => 16#FF39#,  --  Ｙ (FULLWIDTH LATIN CAPITAL LETTER Y)
      245    => 16#FF3A#,  --  Ｚ (FULLWIDTH LATIN CAPITAL LETTER Z)
      252    => 16#FF41#,  --  ａ (FULLWIDTH LATIN SMALL LETTER A)
      253    => 16#FF42#,  --  ｂ (FULLWIDTH LATIN SMALL LETTER B)
      254    => 16#FF43#,  --  ｃ (FULLWIDTH LATIN SMALL LETTER C)
      255    => 16#FF44#,  --  ｄ (FULLWIDTH LATIN SMALL LETTER D)
      256    => 16#FF45#,  --  ｅ (FULLWIDTH LATIN SMALL LETTER E)
      257    => 16#FF46#,  --  ｆ (FULLWIDTH LATIN SMALL LETTER F)
      258    => 16#FF47#,  --  ｇ (FULLWIDTH LATIN SMALL LETTER G)
      259    => 16#FF48#,  --  ｈ (FULLWIDTH LATIN SMALL LETTER H)
      260    => 16#FF49#,  --  ｉ (FULLWIDTH LATIN SMALL LETTER I)
      261    => 16#FF4A#,  --  ｊ (FULLWIDTH LATIN SMALL LETTER J)
      262    => 16#FF4B#,  --  ｋ (FULLWIDTH LATIN SMALL LETTER K)
      263    => 16#FF4C#,  --  ｌ (FULLWIDTH LATIN SMALL LETTER L)
      264    => 16#FF4D#,  --  ｍ (FULLWIDTH LATIN SMALL LETTER M)
      265    => 16#FF4E#,  --  ｎ (FULLWIDTH LATIN SMALL LETTER N)
      266    => 16#FF4F#,  --  ｏ (FULLWIDTH LATIN SMALL LETTER O)
      267    => 16#FF50#,  --  ｐ (FULLWIDTH LATIN SMALL LETTER P)
      268    => 16#FF51#,  --  ｑ (FULLWIDTH LATIN SMALL LETTER Q)
      269    => 16#FF52#,  --  ｒ (FULLWIDTH LATIN SMALL LETTER R)
      270    => 16#FF53#,  --  ｓ (FULLWIDTH LATIN SMALL LETTER S)
      271    => 16#FF54#,  --  ｔ (FULLWIDTH LATIN SMALL LETTER T)
      272    => 16#FF55#,  --  ｕ (FULLWIDTH LATIN SMALL LETTER U)
      273    => 16#FF56#,  --  ｖ (FULLWIDTH LATIN SMALL LETTER V)
      274    => 16#FF57#,  --  ｗ (FULLWIDTH LATIN SMALL LETTER W)
      275    => 16#FF58#,  --  ｘ (FULLWIDTH LATIN SMALL LETTER X)
      276    => 16#FF59#,  --  ｙ (FULLWIDTH LATIN SMALL LETTER Y)
      277    => 16#FF5A#,  --  ｚ (FULLWIDTH LATIN SMALL LETTER Z)
      282    => 16#3041#,  --  ぁ (HIRAGANA LETTER SMALL A)
      283    => 16#3042#,  --  あ (HIRAGANA LETTER A)
      284    => 16#3043#,  --  ぃ (HIRAGANA LETTER SMALL I)
      285    => 16#3044#,  --  い (HIRAGANA LETTER I)
      286    => 16#3045#,  --  ぅ (HIRAGANA LETTER SMALL U)
      287    => 16#3046#,  --  う (HIRAGANA LETTER U)
      288    => 16#3047#,  --  ぇ (HIRAGANA LETTER SMALL E)
      289    => 16#3048#,  --  え (HIRAGANA LETTER E)
      290    => 16#3049#,  --  ぉ (HIRAGANA LETTER SMALL O)
      291    => 16#304A#,  --  お (HIRAGANA LETTER O)
      292    => 16#304B#,  --  か (HIRAGANA LETTER KA)
      293    => 16#304C#,  --  が (HIRAGANA LETTER GA)
      294    => 16#304D#,  --  き (HIRAGANA LETTER KI)
      295    => 16#304E#,  --  ぎ (HIRAGANA LETTER GI)
      296    => 16#304F#,  --  く (HIRAGANA LETTER KU)
      297    => 16#3050#,  --  ぐ (HIRAGANA LETTER GU)
      298    => 16#3051#,  --  け (HIRAGANA LETTER KE)
      299    => 16#3052#,  --  げ (HIRAGANA LETTER GE)
      300    => 16#3053#,  --  こ (HIRAGANA LETTER KO)
      301    => 16#3054#,  --  ご (HIRAGANA LETTER GO)
      302    => 16#3055#,  --  さ (HIRAGANA LETTER SA)
      303    => 16#3056#,  --  ざ (HIRAGANA LETTER ZA)
      304    => 16#3057#,  --  し (HIRAGANA LETTER SI)
      305    => 16#3058#,  --  じ (HIRAGANA LETTER ZI)
      306    => 16#3059#,  --  す (HIRAGANA LETTER SU)
      307    => 16#305A#,  --  ず (HIRAGANA LETTER ZU)
      308    => 16#305B#,  --  せ (HIRAGANA LETTER SE)
      309    => 16#305C#,  --  ぜ (HIRAGANA LETTER ZE)
      310    => 16#305D#,  --  そ (HIRAGANA LETTER SO)
      311    => 16#305E#,  --  ぞ (HIRAGANA LETTER ZO)
      312    => 16#305F#,  --  た (HIRAGANA LETTER TA)
      313    => 16#3060#,  --  だ (HIRAGANA LETTER DA)
      314    => 16#3061#,  --  ち (HIRAGANA LETTER TI)
      315    => 16#3062#,  --  ぢ (HIRAGANA LETTER DI)
      316    => 16#3063#,  --  っ (HIRAGANA LETTER SMALL TU)
      317    => 16#3064#,  --  つ (HIRAGANA LETTER TU)
      318    => 16#3065#,  --  づ (HIRAGANA LETTER DU)
      319    => 16#3066#,  --  て (HIRAGANA LETTER TE)
      320    => 16#3067#,  --  で (HIRAGANA LETTER DE)
      321    => 16#3068#,  --  と (HIRAGANA LETTER TO)
      322    => 16#3069#,  --  ど (HIRAGANA LETTER DO)
      323    => 16#306A#,  --  な (HIRAGANA LETTER NA)
      324    => 16#306B#,  --  に (HIRAGANA LETTER NI)
      325    => 16#306C#,  --  ぬ (HIRAGANA LETTER NU)
      326    => 16#306D#,  --  ね (HIRAGANA LETTER NE)
      327    => 16#306E#,  --  の (HIRAGANA LETTER NO)
      328    => 16#306F#,  --  は (HIRAGANA LETTER HA)
      329    => 16#3070#,  --  ば (HIRAGANA LETTER BA)
      330    => 16#3071#,  --  ぱ (HIRAGANA LETTER PA)
      331    => 16#3072#,  --  ひ (HIRAGANA LETTER HI)
      332    => 16#3073#,  --  び (HIRAGANA LETTER BI)
      333    => 16#3074#,  --  ぴ (HIRAGANA LETTER PI)
      334    => 16#3075#,  --  ふ (HIRAGANA LETTER HU)
      335    => 16#3076#,  --  ぶ (HIRAGANA LETTER BU)
      336    => 16#3077#,  --  ぷ (HIRAGANA LETTER PU)
      337    => 16#3078#,  --  へ (HIRAGANA LETTER HE)
      338    => 16#3079#,  --  べ (HIRAGANA LETTER BE)
      339    => 16#307A#,  --  ぺ (HIRAGANA LETTER PE)
      340    => 16#307B#,  --  ほ (HIRAGANA LETTER HO)
      341    => 16#307C#,  --  ぼ (HIRAGANA LETTER BO)
      342    => 16#307D#,  --  ぽ (HIRAGANA LETTER PO)
      343    => 16#307E#,  --  ま (HIRAGANA LETTER MA)
      344    => 16#307F#,  --  み (HIRAGANA LETTER MI)
      345    => 16#3080#,  --  む (HIRAGANA LETTER MU)
      346    => 16#3081#,  --  め (HIRAGANA LETTER ME)
      347    => 16#3082#,  --  も (HIRAGANA LETTER MO)
      348    => 16#3083#,  --  ゃ (HIRAGANA LETTER SMALL YA)
      349    => 16#3084#,  --  や (HIRAGANA LETTER YA)
      350    => 16#3085#,  --  ゅ (HIRAGANA LETTER SMALL YU)
      351    => 16#3086#,  --  ゆ (HIRAGANA LETTER YU)
      352    => 16#3087#,  --  ょ (HIRAGANA LETTER SMALL YO)
      353    => 16#3088#,  --  よ (HIRAGANA LETTER YO)
      354    => 16#3089#,  --  ら (HIRAGANA LETTER RA)
      355    => 16#308A#,  --  り (HIRAGANA LETTER RI)
      356    => 16#308B#,  --  る (HIRAGANA LETTER RU)
      357    => 16#308C#,  --  れ (HIRAGANA LETTER RE)
      358    => 16#308D#,  --  ろ (HIRAGANA LETTER RO)
      359    => 16#308E#,  --  ゎ (HIRAGANA LETTER SMALL WA)
      360    => 16#308F#,  --  わ (HIRAGANA LETTER WA)
      361    => 16#3090#,  --  ゐ (HIRAGANA LETTER WI)
      362    => 16#3091#,  --  ゑ (HIRAGANA LETTER WE)
      363    => 16#3092#,  --  を (HIRAGANA LETTER WO)
      364    => 16#3093#,  --  ん (HIRAGANA LETTER N)
      376    => 16#30A1#,  --  ァ (KATAKANA LETTER SMALL A)
      377    => 16#30A2#,  --  ア (KATAKANA LETTER A)
      378    => 16#30A3#,  --  ィ (KATAKANA LETTER SMALL I)
      379    => 16#30A4#,  --  イ (KATAKANA LETTER I)
      380    => 16#30A5#,  --  ゥ (KATAKANA LETTER SMALL U)
      381    => 16#30A6#,  --  ウ (KATAKANA LETTER U)
      382    => 16#30A7#,  --  ェ (KATAKANA LETTER SMALL E)
      383    => 16#30A8#,  --  エ (KATAKANA LETTER E)
      384    => 16#30A9#,  --  ォ (KATAKANA LETTER SMALL O)
      385    => 16#30AA#,  --  オ (KATAKANA LETTER O)
      386    => 16#30AB#,  --  カ (KATAKANA LETTER KA)
      387    => 16#30AC#,  --  ガ (KATAKANA LETTER GA)
      388    => 16#30AD#,  --  キ (KATAKANA LETTER KI)
      389    => 16#30AE#,  --  ギ (KATAKANA LETTER GI)
      390    => 16#30AF#,  --  ク (KATAKANA LETTER KU)
      391    => 16#30B0#,  --  グ (KATAKANA LETTER GU)
      392    => 16#30B1#,  --  ケ (KATAKANA LETTER KE)
      393    => 16#30B2#,  --  ゲ (KATAKANA LETTER GE)
      394    => 16#30B3#,  --  コ (KATAKANA LETTER KO)
      395    => 16#30B4#,  --  ゴ (KATAKANA LETTER GO)
      396    => 16#30B5#,  --  サ (KATAKANA LETTER SA)
      397    => 16#30B6#,  --  ザ (KATAKANA LETTER ZA)
      398    => 16#30B7#,  --  シ (KATAKANA LETTER SI)
      399    => 16#30B8#,  --  ジ (KATAKANA LETTER ZI)
      400    => 16#30B9#,  --  ス (KATAKANA LETTER SU)
      401    => 16#30BA#,  --  ズ (KATAKANA LETTER ZU)
      402    => 16#30BB#,  --  セ (KATAKANA LETTER SE)
      403    => 16#30BC#,  --  ゼ (KATAKANA LETTER ZE)
      404    => 16#30BD#,  --  ソ (KATAKANA LETTER SO)
      405    => 16#30BE#,  --  ゾ (KATAKANA LETTER ZO)
      406    => 16#30BF#,  --  タ (KATAKANA LETTER TA)
      407    => 16#30C0#,  --  ダ (KATAKANA LETTER DA)
      408    => 16#30C1#,  --  チ (KATAKANA LETTER TI)
      409    => 16#30C2#,  --  ヂ (KATAKANA LETTER DI)
      410    => 16#30C3#,  --  ッ (KATAKANA LETTER SMALL TU)
      411    => 16#30C4#,  --  ツ (KATAKANA LETTER TU)
      412    => 16#30C5#,  --  ヅ (KATAKANA LETTER DU)
      413    => 16#30C6#,  --  テ (KATAKANA LETTER TE)
      414    => 16#30C7#,  --  デ (KATAKANA LETTER DE)
      415    => 16#30C8#,  --  ト (KATAKANA LETTER TO)
      416    => 16#30C9#,  --  ド (KATAKANA LETTER DO)
      417    => 16#30CA#,  --  ナ (KATAKANA LETTER NA)
      418    => 16#30CB#,  --  ニ (KATAKANA LETTER NI)
      419    => 16#30CC#,  --  ヌ (KATAKANA LETTER NU)
      420    => 16#30CD#,  --  ネ (KATAKANA LETTER NE)
      421    => 16#30CE#,  --  ノ (KATAKANA LETTER NO)
      422    => 16#30CF#,  --  ハ (KATAKANA LETTER HA)
      423    => 16#30D0#,  --  バ (KATAKANA LETTER BA)
      424    => 16#30D1#,  --  パ (KATAKANA LETTER PA)
      425    => 16#30D2#,  --  ヒ (KATAKANA LETTER HI)
      426    => 16#30D3#,  --  ビ (KATAKANA LETTER BI)
      427    => 16#30D4#,  --  ピ (KATAKANA LETTER PI)
      428    => 16#30D5#,  --  フ (KATAKANA LETTER HU)
      429    => 16#30D6#,  --  ブ (KATAKANA LETTER BU)
      430    => 16#30D7#,  --  プ (KATAKANA LETTER PU)
      431    => 16#30D8#,  --  ヘ (KATAKANA LETTER HE)
      432    => 16#30D9#,  --  ベ (KATAKANA LETTER BE)
      433    => 16#30DA#,  --  ペ (KATAKANA LETTER PE)
      434    => 16#30DB#,  --  ホ (KATAKANA LETTER HO)
      435    => 16#30DC#,  --  ボ (KATAKANA LETTER BO)
      436    => 16#30DD#,  --  ポ (KATAKANA LETTER PO)
      437    => 16#30DE#,  --  マ (KATAKANA LETTER MA)
      438    => 16#30DF#,  --  ミ (KATAKANA LETTER MI)
      439    => 16#30E0#,  --  ム (KATAKANA LETTER MU)
      440    => 16#30E1#,  --  メ (KATAKANA LETTER ME)
      441    => 16#30E2#,  --  モ (KATAKANA LETTER MO)
      442    => 16#30E3#,  --  ャ (KATAKANA LETTER SMALL YA)
      443    => 16#30E4#,  --  ヤ (KATAKANA LETTER YA)
      444    => 16#30E5#,  --  ュ (KATAKANA LETTER SMALL YU)
      445    => 16#30E6#,  --  ユ (KATAKANA LETTER YU)
      446    => 16#30E7#,  --  ョ (KATAKANA LETTER SMALL YO)
      447    => 16#30E8#,  --  ヨ (KATAKANA LETTER YO)
      448    => 16#30E9#,  --  ラ (KATAKANA LETTER RA)
      449    => 16#30EA#,  --  リ (KATAKANA LETTER RI)
      450    => 16#30EB#,  --  ル (KATAKANA LETTER RU)
      451    => 16#30EC#,  --  レ (KATAKANA LETTER RE)
      452    => 16#30ED#,  --  ロ (KATAKANA LETTER RO)
      453    => 16#30EE#,  --  ヮ (KATAKANA LETTER SMALL WA)
      454    => 16#30EF#,  --  ワ (KATAKANA LETTER WA)
      455    => 16#30F0#,  --  ヰ (KATAKANA LETTER WI)
      456    => 16#30F1#,  --  ヱ (KATAKANA LETTER WE)
      457    => 16#30F2#,  --  ヲ (KATAKANA LETTER WO)
      458    => 16#30F3#,  --  ン (KATAKANA LETTER N)
      459    => 16#30F4#,  --  ヴ (KATAKANA LETTER VU)
      460    => 16#30F5#,  --  ヵ (KATAKANA LETTER SMALL KA)
      461    => 16#30F6#,  --  ヶ (KATAKANA LETTER SMALL KE)
      470    => 16#0391#,  --  Α (GREEK CAPITAL LETTER ALPHA)
      471    => 16#0392#,  --  Β (GREEK CAPITAL LETTER BETA)
      472    => 16#0393#,  --  Γ (GREEK CAPITAL LETTER GAMMA)
      473    => 16#0394#,  --  Δ (GREEK CAPITAL LETTER DELTA)
      474    => 16#0395#,  --  Ε (GREEK CAPITAL LETTER EPSILON)
      475    => 16#0396#,  --  Ζ (GREEK CAPITAL LETTER ZETA)
      476    => 16#0397#,  --  Η (GREEK CAPITAL LETTER ETA)
      477    => 16#0398#,  --  Θ (GREEK CAPITAL LETTER THETA)
      478    => 16#0399#,  --  Ι (GREEK CAPITAL LETTER IOTA)
      479    => 16#039A#,  --  Κ (GREEK CAPITAL LETTER KAPPA)
      480    => 16#039B#,  --  Λ (GREEK CAPITAL LETTER LAMDA)
      481    => 16#039C#,  --  Μ (GREEK CAPITAL LETTER MU)
      482    => 16#039D#,  --  Ν (GREEK CAPITAL LETTER NU)
      483    => 16#039E#,  --  Ξ (GREEK CAPITAL LETTER XI)
      484    => 16#039F#,  --  Ο (GREEK CAPITAL LETTER OMICRON)
      485    => 16#03A0#,  --  Π (GREEK CAPITAL LETTER PI)
      486    => 16#03A1#,  --  Ρ (GREEK CAPITAL LETTER RHO)
      487    => 16#03A3#,  --  Σ (GREEK CAPITAL LETTER SIGMA)
      488    => 16#03A4#,  --  Τ (GREEK CAPITAL LETTER TAU)
      489    => 16#03A5#,  --  Υ (GREEK CAPITAL LETTER UPSILON)
      490    => 16#03A6#,  --  Φ (GREEK CAPITAL LETTER PHI)
      491    => 16#03A7#,  --  Χ (GREEK CAPITAL LETTER CHI)
      492    => 16#03A8#,  --  Ψ (GREEK CAPITAL LETTER PSI)
      493    => 16#03A9#,  --  Ω (GREEK CAPITAL LETTER OMEGA)
      502    => 16#03B1#,  --  α (GREEK SMALL LETTER ALPHA)
      503    => 16#03B2#,  --  β (GREEK SMALL LETTER BETA)
      504    => 16#03B3#,  --  γ (GREEK SMALL LETTER GAMMA)
      505    => 16#03B4#,  --  δ (GREEK SMALL LETTER DELTA)
      506    => 16#03B5#,  --  ε (GREEK SMALL LETTER EPSILON)
      507    => 16#03B6#,  --  ζ (GREEK SMALL LETTER ZETA)
      508    => 16#03B7#,  --  η (GREEK SMALL LETTER ETA)
      509    => 16#03B8#,  --  θ (GREEK SMALL LETTER THETA)
      510    => 16#03B9#,  --  ι (GREEK SMALL LETTER IOTA)
      511    => 16#03BA#,  --  κ (GREEK SMALL LETTER KAPPA)
      512    => 16#03BB#,  --  λ (GREEK SMALL LETTER LAMDA)
      513    => 16#03BC#,  --  μ (GREEK SMALL LETTER MU)
      514    => 16#03BD#,  --  ν (GREEK SMALL LETTER NU)
      515    => 16#03BE#,  --  ξ (GREEK SMALL LETTER XI)
      516    => 16#03BF#,  --  ο (GREEK SMALL LETTER OMICRON)
      517    => 16#03C0#,  --  π (GREEK SMALL LETTER PI)
      518    => 16#03C1#,  --  ρ (GREEK SMALL LETTER RHO)
      519    => 16#03C3#,  --  σ (GREEK SMALL LETTER SIGMA)
      520    => 16#03C4#,  --  τ (GREEK SMALL LETTER TAU)
      521    => 16#03C5#,  --  υ (GREEK SMALL LETTER UPSILON)
      522    => 16#03C6#,  --  φ (GREEK SMALL LETTER PHI)
      523    => 16#03C7#,  --  χ (GREEK SMALL LETTER CHI)
      524    => 16#03C8#,  --  ψ (GREEK SMALL LETTER PSI)
      525    => 16#03C9#,  --  ω (GREEK SMALL LETTER OMEGA)
      564    => 16#0410#,  --  А (CYRILLIC CAPITAL LETTER A)
      565    => 16#0411#,  --  Б (CYRILLIC CAPITAL LETTER BE)
      566    => 16#0412#,  --  В (CYRILLIC CAPITAL LETTER VE)
      567    => 16#0413#,  --  Г (CYRILLIC CAPITAL LETTER GHE)
      568    => 16#0414#,  --  Д (CYRILLIC CAPITAL LETTER DE)
      569    => 16#0415#,  --  Е (CYRILLIC CAPITAL LETTER IE)
      570    => 16#0401#,  --  Ё (CYRILLIC CAPITAL LETTER IO)
      571    => 16#0416#,  --  Ж (CYRILLIC CAPITAL LETTER ZHE)
      572    => 16#0417#,  --  З (CYRILLIC CAPITAL LETTER ZE)
      573    => 16#0418#,  --  И (CYRILLIC CAPITAL LETTER I)
      574    => 16#0419#,  --  Й (CYRILLIC CAPITAL LETTER SHORT I)
      575    => 16#041A#,  --  К (CYRILLIC CAPITAL LETTER KA)
      576    => 16#041B#,  --  Л (CYRILLIC CAPITAL LETTER EL)
      577    => 16#041C#,  --  М (CYRILLIC CAPITAL LETTER EM)
      578    => 16#041D#,  --  Н (CYRILLIC CAPITAL LETTER EN)
      579    => 16#041E#,  --  О (CYRILLIC CAPITAL LETTER O)
      580    => 16#041F#,  --  П (CYRILLIC CAPITAL LETTER PE)
      581    => 16#0420#,  --  Р (CYRILLIC CAPITAL LETTER ER)
      582    => 16#0421#,  --  С (CYRILLIC CAPITAL LETTER ES)
      583    => 16#0422#,  --  Т (CYRILLIC CAPITAL LETTER TE)
      584    => 16#0423#,  --  У (CYRILLIC CAPITAL LETTER U)
      585    => 16#0424#,  --  Ф (CYRILLIC CAPITAL LETTER EF)
      586    => 16#0425#,  --  Х (CYRILLIC CAPITAL LETTER HA)
      587    => 16#0426#,  --  Ц (CYRILLIC CAPITAL LETTER TSE)
      588    => 16#0427#,  --  Ч (CYRILLIC CAPITAL LETTER CHE)
      589    => 16#0428#,  --  Ш (CYRILLIC CAPITAL LETTER SHA)
      590    => 16#0429#,  --  Щ (CYRILLIC CAPITAL LETTER SHCHA)
      591    => 16#042A#,  --  Ъ (CYRILLIC CAPITAL LETTER HARD SIGN)
      592    => 16#042B#,  --  Ы (CYRILLIC CAPITAL LETTER YERU)
      593    => 16#042C#,  --  Ь (CYRILLIC CAPITAL LETTER SOFT SIGN)
      594    => 16#042D#,  --  Э (CYRILLIC CAPITAL LETTER E)
      595    => 16#042E#,  --  Ю (CYRILLIC CAPITAL LETTER YU)
      596    => 16#042F#,  --  Я (CYRILLIC CAPITAL LETTER YA)
      612    => 16#0430#,  --  а (CYRILLIC SMALL LETTER A)
      613    => 16#0431#,  --  б (CYRILLIC SMALL LETTER BE)
      614    => 16#0432#,  --  в (CYRILLIC SMALL LETTER VE)
      615    => 16#0433#,  --  г (CYRILLIC SMALL LETTER GHE)
      616    => 16#0434#,  --  д (CYRILLIC SMALL LETTER DE)
      617    => 16#0435#,  --  е (CYRILLIC SMALL LETTER IE)
      618    => 16#0451#,  --  ё (CYRILLIC SMALL LETTER IO)
      619    => 16#0436#,  --  ж (CYRILLIC SMALL LETTER ZHE)
      620    => 16#0437#,  --  з (CYRILLIC SMALL LETTER ZE)
      621    => 16#0438#,  --  и (CYRILLIC SMALL LETTER I)
      622    => 16#0439#,  --  й (CYRILLIC SMALL LETTER SHORT I)
      623    => 16#043A#,  --  к (CYRILLIC SMALL LETTER KA)
      624    => 16#043B#,  --  л (CYRILLIC SMALL LETTER EL)
      625    => 16#043C#,  --  м (CYRILLIC SMALL LETTER EM)
      626    => 16#043D#,  --  н (CYRILLIC SMALL LETTER EN)
      627    => 16#043E#,  --  о (CYRILLIC SMALL LETTER O)
      628    => 16#043F#,  --  п (CYRILLIC SMALL LETTER PE)
      629    => 16#0440#,  --  р (CYRILLIC SMALL LETTER ER)
      630    => 16#0441#,  --  с (CYRILLIC SMALL LETTER ES)
      631    => 16#0442#,  --  т (CYRILLIC SMALL LETTER TE)
      632    => 16#0443#,  --  у (CYRILLIC SMALL LETTER U)
      633    => 16#0444#,  --  ф (CYRILLIC SMALL LETTER EF)
      634    => 16#0445#,  --  х (CYRILLIC SMALL LETTER HA)
      635    => 16#0446#,  --  ц (CYRILLIC SMALL LETTER TSE)
      636    => 16#0447#,  --  ч (CYRILLIC SMALL LETTER CHE)
      637    => 16#0448#,  --  ш (CYRILLIC SMALL LETTER SHA)
      638    => 16#0449#,  --  щ (CYRILLIC SMALL LETTER SHCHA)
      639    => 16#044A#,  --  ъ (CYRILLIC SMALL LETTER HARD SIGN)
      640    => 16#044B#,  --  ы (CYRILLIC SMALL LETTER YERU)
      641    => 16#044C#,  --  ь (CYRILLIC SMALL LETTER SOFT SIGN)
      642    => 16#044D#,  --  э (CYRILLIC SMALL LETTER E)
      643    => 16#044E#,  --  ю (CYRILLIC SMALL LETTER YU)
      644    => 16#044F#,  --  я (CYRILLIC SMALL LETTER YA)
      658    => 16#2500#,  --  ─ (BOX DRAWINGS LIGHT HORIZONTAL)
      659    => 16#2502#,  --  │ (BOX DRAWINGS LIGHT VERTICAL)
      660    => 16#250C#,  --  ┌ (BOX DRAWINGS LIGHT DOWN AND RIGHT)
      661    => 16#2510#,  --  ┐ (BOX DRAWINGS LIGHT DOWN AND LEFT)
      662    => 16#2518#,  --  ┘ (BOX DRAWINGS LIGHT UP AND LEFT)
      663    => 16#2514#,  --  └ (BOX DRAWINGS LIGHT UP AND RIGHT)
      664    => 16#251C#,  --  ├ (BOX DRAWINGS LIGHT VERTICAL AND RIGHT)
      665    => 16#252C#,  --  ┬ (BOX DRAWINGS LIGHT DOWN AND HORIZONTAL)
      666    => 16#2524#,  --  ┤ (BOX DRAWINGS LIGHT VERTICAL AND LEFT)
      667    => 16#2534#,  --  ┴ (BOX DRAWINGS LIGHT UP AND HORIZONTAL)
      668    => 16#253C#,  --  ┼ (BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL)
      669    => 16#2501#,  --  ━ (BOX DRAWINGS HEAVY HORIZONTAL)
      670    => 16#2503#,  --  ┃ (BOX DRAWINGS HEAVY VERTICAL)
      671    => 16#250F#,  --  ┏ (BOX DRAWINGS HEAVY DOWN AND RIGHT)
      672    => 16#2513#,  --  ┓ (BOX DRAWINGS HEAVY DOWN AND LEFT)
      673    => 16#251B#,  --  ┛ (BOX DRAWINGS HEAVY UP AND LEFT)
      674    => 16#2517#,  --  ┗ (BOX DRAWINGS HEAVY UP AND RIGHT)
      675    => 16#2523#,  --  ┣ (BOX DRAWINGS HEAVY VERTICAL AND RIGHT)
      676    => 16#2533#,  --  ┳ (BOX DRAWINGS HEAVY DOWN AND HORIZONTAL)
      677    => 16#252B#,  --  ┫ (BOX DRAWINGS HEAVY VERTICAL AND LEFT)
      678    => 16#253B#,  --  ┻ (BOX DRAWINGS HEAVY UP AND HORIZONTAL)
      679    => 16#254B#,  --  ╋ (BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL)
      680    => 16#2520#,  --  ┠ (BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT)
      681    => 16#252F#,  --  ┯ (BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY)
      682    => 16#2528#,  --  ┨ (BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT)
      683    => 16#2537#,  --  ┷ (BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY)
      684    => 16#253F#,  --  ┿ (BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY)
      685    => 16#251D#,  --  ┝ (BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY)
      686    => 16#2530#,  --  ┰ (BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT)
      687    => 16#2525#,  --  ┥ (BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY)
      688    => 16#2538#,  --  ┸ (BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT)
      689    => 16#2542#,  --  ╂ (BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT)
      1128   => 16#2460#,  --  ① (CIRCLED DIGIT ONE)
      1129   => 16#2461#,  --  ② (CIRCLED DIGIT TWO)
      1130   => 16#2462#,  --  ③ (CIRCLED DIGIT THREE)
      1131   => 16#2463#,  --  ④ (CIRCLED DIGIT FOUR)
      1132   => 16#2464#,  --  ⑤ (CIRCLED DIGIT FIVE)
      1133   => 16#2465#,  --  ⑥ (CIRCLED DIGIT SIX)
      1134   => 16#2466#,  --  ⑦ (CIRCLED DIGIT SEVEN)
      1135   => 16#2467#,  --  ⑧ (CIRCLED DIGIT EIGHT)
      1136   => 16#2468#,  --  ⑨ (CIRCLED DIGIT NINE)
      1137   => 16#2469#,  --  ⑩ (CIRCLED NUMBER TEN)
      1138   => 16#246A#,  --  ⑪ (CIRCLED NUMBER ELEVEN)
      1139   => 16#246B#,  --  ⑫ (CIRCLED NUMBER TWELVE)
      1140   => 16#246C#,  --  ⑬ (CIRCLED NUMBER THIRTEEN)
      1141   => 16#246D#,  --  ⑭ (CIRCLED NUMBER FOURTEEN)
      1142   => 16#246E#,  --  ⑮ (CIRCLED NUMBER FIFTEEN)
      1143   => 16#246F#,  --  ⑯ (CIRCLED NUMBER SIXTEEN)
      1144   => 16#2470#,  --  ⑰ (CIRCLED NUMBER SEVENTEEN)
      1145   => 16#2471#,  --  ⑱ (CIRCLED NUMBER EIGHTEEN)
      1146   => 16#2472#,  --  ⑲ (CIRCLED NUMBER NINETEEN)
      1147   => 16#2473#,  --  ⑳ (CIRCLED NUMBER TWENTY)
      1148   => 16#2160#,  --  Ⅰ (ROMAN NUMERAL ONE)
      1149   => 16#2161#,  --  Ⅱ (ROMAN NUMERAL TWO)
      1150   => 16#2162#,  --  Ⅲ (ROMAN NUMERAL THREE)
      1151   => 16#2163#,  --  Ⅳ (ROMAN NUMERAL FOUR)
      1152   => 16#2164#,  --  Ⅴ (ROMAN NUMERAL FIVE)
      1153   => 16#2165#,  --  Ⅵ (ROMAN NUMERAL SIX)
      1154   => 16#2166#,  --  Ⅶ (ROMAN NUMERAL SEVEN)
      1155   => 16#2167#,  --  Ⅷ (ROMAN NUMERAL EIGHT)
      1156   => 16#2168#,  --  Ⅸ (ROMAN NUMERAL NINE)
      1157   => 16#2169#,  --  Ⅹ (ROMAN NUMERAL TEN)
      1159   => 16#3349#,  --  ㍉ (SQUARE MIRI)
      1160   => 16#3314#,  --  ㌔ (SQUARE KIRO)
      1161   => 16#3322#,  --  ㌢ (SQUARE SENTI)
      1162   => 16#334D#,  --  ㍍ (SQUARE MEETORU)
      1163   => 16#3318#,  --  ㌘ (SQUARE GURAMU)
      1164   => 16#3327#,  --  ㌧ (SQUARE TON)
      1165   => 16#3303#,  --  ㌃ (SQUARE AARU)
      1166   => 16#3336#,  --  ㌶ (SQUARE HEKUTAARU)
      1167   => 16#3351#,  --  ㍑ (SQUARE RITTORU)
      1168   => 16#3357#,  --  ㍗ (SQUARE WATTO)
      1169   => 16#330D#,  --  ㌍ (SQUARE KARORII)
      1170   => 16#3326#,  --  ㌦ (SQUARE DORU)
      1171   => 16#3323#,  --  ㌣ (SQUARE SENTO)
      1172   => 16#332B#,  --  ㌫ (SQUARE PAASENTO)
      1173   => 16#334A#,  --  ㍊ (SQUARE MIRIBAARU)
      1174   => 16#333B#,  --  ㌻ (SQUARE PEEZI)
      1175   => 16#339C#,  --  ㎜ (SQUARE MM)
      1176   => 16#339D#,  --  ㎝ (SQUARE CM)
      1177   => 16#339E#,  --  ㎞ (SQUARE KM)
      1178   => 16#338E#,  --  ㎎ (SQUARE MG)
      1179   => 16#338F#,  --  ㎏ (SQUARE KG)
      1180   => 16#33C4#,  --  ㏄ (SQUARE CC)
      1181   => 16#33A1#,  --  ㎡ (SQUARE M SQUARED)
      1190   => 16#337B#,  --  ㍻ (SQUARE ERA NAME HEISEI)
      1191   => 16#301D#,  --  〝 (REVERSED DOUBLE PRIME QUOTATION MARK)
      1192   => 16#301F#,  --  〟 (LOW DOUBLE PRIME QUOTATION MARK)
      1193   => 16#2116#,  --  № (NUMERO SIGN)
      1194   => 16#33CD#,  --  ㏍ (SQUARE KK)
      1195   => 16#2121#,  --  ℡ (TELEPHONE SIGN)
      1196   => 16#32A4#,  --  ㊤ (CIRCLED IDEOGRAPH HIGH)
      1197   => 16#32A5#,  --  ㊥ (CIRCLED IDEOGRAPH CENTRE)
      1198   => 16#32A6#,  --  ㊦ (CIRCLED IDEOGRAPH LOW)
      1199   => 16#32A7#,  --  ㊧ (CIRCLED IDEOGRAPH LEFT)
      1200   => 16#32A8#,  --  ㊨ (CIRCLED IDEOGRAPH RIGHT)
      1201   => 16#3231#,  --  ㈱ (PARENTHESIZED IDEOGRAPH STOCK)
      1202   => 16#3232#,  --  ㈲ (PARENTHESIZED IDEOGRAPH HAVE)
      1203   => 16#3239#,  --  ㈹ (PARENTHESIZED IDEOGRAPH REPRESENT)
      1204   => 16#337E#,  --  ㍾ (SQUARE ERA NAME MEIZI)
      1205   => 16#337D#,  --  ㍽ (SQUARE ERA NAME TAISYOU)
      1206   => 16#337C#,  --  ㍼ (SQUARE ERA NAME SYOUWA)
      1207   => 16#2252#,  --  ≒ (APPROXIMATELY EQUAL TO OR THE IMAGE OF)
      1208   => 16#2261#,  --  ≡ (IDENTICAL TO)
      1209   => 16#222B#,  --  ∫ (INTEGRAL)
      1210   => 16#222E#,  --  ∮ (CONTOUR INTEGRAL)
      1211   => 16#2211#,  --  ∑ (N-ARY SUMMATION)
      1212   => 16#221A#,  --  √ (SQUARE ROOT)
      1213   => 16#22A5#,  --  ⊥ (UP TACK)
      1214   => 16#2220#,  --  ∠ (ANGLE)
      1215   => 16#221F#,  --  ∟ (RIGHT ANGLE)
      1216   => 16#22BF#,  --  ⊿ (RIGHT TRIANGLE)
      1217   => 16#2235#,  --  ∵ (BECAUSE)
      1218   => 16#2229#,  --  ∩ (INTERSECTION)
      1219   => 16#222A#,  --  ∪ (UNION)
      1410   => 16#4E9C#,  --  亜 (<CJK Ideograph>)
      1411   => 16#5516#,  --  唖 (<CJK Ideograph>)
      1412   => 16#5A03#,  --  娃 (<CJK Ideograph>)
      1413   => 16#963F#,  --  阿 (<CJK Ideograph>)
      1414   => 16#54C0#,  --  哀 (<CJK Ideograph>)
      1415   => 16#611B#,  --  愛 (<CJK Ideograph>)
      1416   => 16#6328#,  --  挨 (<CJK Ideograph>)
      1417   => 16#59F6#,  --  姶 (<CJK Ideograph>)
      1418   => 16#9022#,  --  逢 (<CJK Ideograph>)
      1419   => 16#8475#,  --  葵 (<CJK Ideograph>)
      1420   => 16#831C#,  --  茜 (<CJK Ideograph>)
      1421   => 16#7A50#,  --  穐 (<CJK Ideograph>)
      1422   => 16#60AA#,  --  悪 (<CJK Ideograph>)
      1423   => 16#63E1#,  --  握 (<CJK Ideograph>)
      1424   => 16#6E25#,  --  渥 (<CJK Ideograph>)
      1425   => 16#65ED#,  --  旭 (<CJK Ideograph>)
      1426   => 16#8466#,  --  葦 (<CJK Ideograph>)
      1427   => 16#82A6#,  --  芦 (<CJK Ideograph>)
      1428   => 16#9BF5#,  --  鯵 (<CJK Ideograph>)
      1429   => 16#6893#,  --  梓 (<CJK Ideograph>)
      1430   => 16#5727#,  --  圧 (<CJK Ideograph>)
      1431   => 16#65A1#,  --  斡 (<CJK Ideograph>)
      1432   => 16#6271#,  --  扱 (<CJK Ideograph>)
      1433   => 16#5B9B#,  --  宛 (<CJK Ideograph>)
      1434   => 16#59D0#,  --  姐 (<CJK Ideograph>)
      1435   => 16#867B#,  --  虻 (<CJK Ideograph>)
      1436   => 16#98F4#,  --  飴 (<CJK Ideograph>)
      1437   => 16#7D62#,  --  絢 (<CJK Ideograph>)
      1438   => 16#7DBE#,  --  綾 (<CJK Ideograph>)
      1439   => 16#9B8E#,  --  鮎 (<CJK Ideograph>)
      1440   => 16#6216#,  --  或 (<CJK Ideograph>)
      1441   => 16#7C9F#,  --  粟 (<CJK Ideograph>)
      1442   => 16#88B7#,  --  袷 (<CJK Ideograph>)
      1443   => 16#5B89#,  --  安 (<CJK Ideograph>)
      1444   => 16#5EB5#,  --  庵 (<CJK Ideograph>)
      1445   => 16#6309#,  --  按 (<CJK Ideograph>)
      1446   => 16#6697#,  --  暗 (<CJK Ideograph>)
      1447   => 16#6848#,  --  案 (<CJK Ideograph>)
      1448   => 16#95C7#,  --  闇 (<CJK Ideograph>)
      1449   => 16#978D#,  --  鞍 (<CJK Ideograph>)
      1450   => 16#674F#,  --  杏 (<CJK Ideograph>)
      1451   => 16#4EE5#,  --  以 (<CJK Ideograph>)
      1452   => 16#4F0A#,  --  伊 (<CJK Ideograph>)
      1453   => 16#4F4D#,  --  位 (<CJK Ideograph>)
      1454   => 16#4F9D#,  --  依 (<CJK Ideograph>)
      1455   => 16#5049#,  --  偉 (<CJK Ideograph>)
      1456   => 16#56F2#,  --  囲 (<CJK Ideograph>)
      1457   => 16#5937#,  --  夷 (<CJK Ideograph>)
      1458   => 16#59D4#,  --  委 (<CJK Ideograph>)
      1459   => 16#5A01#,  --  威 (<CJK Ideograph>)
      1460   => 16#5C09#,  --  尉 (<CJK Ideograph>)
      1461   => 16#60DF#,  --  惟 (<CJK Ideograph>)
      1462   => 16#610F#,  --  意 (<CJK Ideograph>)
      1463   => 16#6170#,  --  慰 (<CJK Ideograph>)
      1464   => 16#6613#,  --  易 (<CJK Ideograph>)
      1465   => 16#6905#,  --  椅 (<CJK Ideograph>)
      1466   => 16#70BA#,  --  為 (<CJK Ideograph>)
      1467   => 16#754F#,  --  畏 (<CJK Ideograph>)
      1468   => 16#7570#,  --  異 (<CJK Ideograph>)
      1469   => 16#79FB#,  --  移 (<CJK Ideograph>)
      1470   => 16#7DAD#,  --  維 (<CJK Ideograph>)
      1471   => 16#7DEF#,  --  緯 (<CJK Ideograph>)
      1472   => 16#80C3#,  --  胃 (<CJK Ideograph>)
      1473   => 16#840E#,  --  萎 (<CJK Ideograph>)
      1474   => 16#8863#,  --  衣 (<CJK Ideograph>)
      1475   => 16#8B02#,  --  謂 (<CJK Ideograph>)
      1476   => 16#9055#,  --  違 (<CJK Ideograph>)
      1477   => 16#907A#,  --  遺 (<CJK Ideograph>)
      1478   => 16#533B#,  --  医 (<CJK Ideograph>)
      1479   => 16#4E95#,  --  井 (<CJK Ideograph>)
      1480   => 16#4EA5#,  --  亥 (<CJK Ideograph>)
      1481   => 16#57DF#,  --  域 (<CJK Ideograph>)
      1482   => 16#80B2#,  --  育 (<CJK Ideograph>)
      1483   => 16#90C1#,  --  郁 (<CJK Ideograph>)
      1484   => 16#78EF#,  --  磯 (<CJK Ideograph>)
      1485   => 16#4E00#,  --  一 (<CJK Ideograph>)
      1486   => 16#58F1#,  --  壱 (<CJK Ideograph>)
      1487   => 16#6EA2#,  --  溢 (<CJK Ideograph>)
      1488   => 16#9038#,  --  逸 (<CJK Ideograph>)
      1489   => 16#7A32#,  --  稲 (<CJK Ideograph>)
      1490   => 16#8328#,  --  茨 (<CJK Ideograph>)
      1491   => 16#828B#,  --  芋 (<CJK Ideograph>)
      1492   => 16#9C2F#,  --  鰯 (<CJK Ideograph>)
      1493   => 16#5141#,  --  允 (<CJK Ideograph>)
      1494   => 16#5370#,  --  印 (<CJK Ideograph>)
      1495   => 16#54BD#,  --  咽 (<CJK Ideograph>)
      1496   => 16#54E1#,  --  員 (<CJK Ideograph>)
      1497   => 16#56E0#,  --  因 (<CJK Ideograph>)
      1498   => 16#59FB#,  --  姻 (<CJK Ideograph>)
      1499   => 16#5F15#,  --  引 (<CJK Ideograph>)
      1500   => 16#98F2#,  --  飲 (<CJK Ideograph>)
      1501   => 16#6DEB#,  --  淫 (<CJK Ideograph>)
      1502   => 16#80E4#,  --  胤 (<CJK Ideograph>)
      1503   => 16#852D#,  --  蔭 (<CJK Ideograph>)
      1504   => 16#9662#,  --  院 (<CJK Ideograph>)
      1505   => 16#9670#,  --  陰 (<CJK Ideograph>)
      1506   => 16#96A0#,  --  隠 (<CJK Ideograph>)
      1507   => 16#97FB#,  --  韻 (<CJK Ideograph>)
      1508   => 16#540B#,  --  吋 (<CJK Ideograph>)
      1509   => 16#53F3#,  --  右 (<CJK Ideograph>)
      1510   => 16#5B87#,  --  宇 (<CJK Ideograph>)
      1511   => 16#70CF#,  --  烏 (<CJK Ideograph>)
      1512   => 16#7FBD#,  --  羽 (<CJK Ideograph>)
      1513   => 16#8FC2#,  --  迂 (<CJK Ideograph>)
      1514   => 16#96E8#,  --  雨 (<CJK Ideograph>)
      1515   => 16#536F#,  --  卯 (<CJK Ideograph>)
      1516   => 16#9D5C#,  --  鵜 (<CJK Ideograph>)
      1517   => 16#7ABA#,  --  窺 (<CJK Ideograph>)
      1518   => 16#4E11#,  --  丑 (<CJK Ideograph>)
      1519   => 16#7893#,  --  碓 (<CJK Ideograph>)
      1520   => 16#81FC#,  --  臼 (<CJK Ideograph>)
      1521   => 16#6E26#,  --  渦 (<CJK Ideograph>)
      1522   => 16#5618#,  --  嘘 (<CJK Ideograph>)
      1523   => 16#5504#,  --  唄 (<CJK Ideograph>)
      1524   => 16#6B1D#,  --  欝 (<CJK Ideograph>)
      1525   => 16#851A#,  --  蔚 (<CJK Ideograph>)
      1526   => 16#9C3B#,  --  鰻 (<CJK Ideograph>)
      1527   => 16#59E5#,  --  姥 (<CJK Ideograph>)
      1528   => 16#53A9#,  --  厩 (<CJK Ideograph>)
      1529   => 16#6D66#,  --  浦 (<CJK Ideograph>)
      1530   => 16#74DC#,  --  瓜 (<CJK Ideograph>)
      1531   => 16#958F#,  --  閏 (<CJK Ideograph>)
      1532   => 16#5642#,  --  噂 (<CJK Ideograph>)
      1533   => 16#4E91#,  --  云 (<CJK Ideograph>)
      1534   => 16#904B#,  --  運 (<CJK Ideograph>)
      1535   => 16#96F2#,  --  雲 (<CJK Ideograph>)
      1536   => 16#834F#,  --  荏 (<CJK Ideograph>)
      1537   => 16#990C#,  --  餌 (<CJK Ideograph>)
      1538   => 16#53E1#,  --  叡 (<CJK Ideograph>)
      1539   => 16#55B6#,  --  営 (<CJK Ideograph>)
      1540   => 16#5B30#,  --  嬰 (<CJK Ideograph>)
      1541   => 16#5F71#,  --  影 (<CJK Ideograph>)
      1542   => 16#6620#,  --  映 (<CJK Ideograph>)
      1543   => 16#66F3#,  --  曳 (<CJK Ideograph>)
      1544   => 16#6804#,  --  栄 (<CJK Ideograph>)
      1545   => 16#6C38#,  --  永 (<CJK Ideograph>)
      1546   => 16#6CF3#,  --  泳 (<CJK Ideograph>)
      1547   => 16#6D29#,  --  洩 (<CJK Ideograph>)
      1548   => 16#745B#,  --  瑛 (<CJK Ideograph>)
      1549   => 16#76C8#,  --  盈 (<CJK Ideograph>)
      1550   => 16#7A4E#,  --  穎 (<CJK Ideograph>)
      1551   => 16#9834#,  --  頴 (<CJK Ideograph>)
      1552   => 16#82F1#,  --  英 (<CJK Ideograph>)
      1553   => 16#885B#,  --  衛 (<CJK Ideograph>)
      1554   => 16#8A60#,  --  詠 (<CJK Ideograph>)
      1555   => 16#92ED#,  --  鋭 (<CJK Ideograph>)
      1556   => 16#6DB2#,  --  液 (<CJK Ideograph>)
      1557   => 16#75AB#,  --  疫 (<CJK Ideograph>)
      1558   => 16#76CA#,  --  益 (<CJK Ideograph>)
      1559   => 16#99C5#,  --  駅 (<CJK Ideograph>)
      1560   => 16#60A6#,  --  悦 (<CJK Ideograph>)
      1561   => 16#8B01#,  --  謁 (<CJK Ideograph>)
      1562   => 16#8D8A#,  --  越 (<CJK Ideograph>)
      1563   => 16#95B2#,  --  閲 (<CJK Ideograph>)
      1564   => 16#698E#,  --  榎 (<CJK Ideograph>)
      1565   => 16#53AD#,  --  厭 (<CJK Ideograph>)
      1566   => 16#5186#,  --  円 (<CJK Ideograph>)
      1567   => 16#5712#,  --  園 (<CJK Ideograph>)
      1568   => 16#5830#,  --  堰 (<CJK Ideograph>)
      1569   => 16#5944#,  --  奄 (<CJK Ideograph>)
      1570   => 16#5BB4#,  --  宴 (<CJK Ideograph>)
      1571   => 16#5EF6#,  --  延 (<CJK Ideograph>)
      1572   => 16#6028#,  --  怨 (<CJK Ideograph>)
      1573   => 16#63A9#,  --  掩 (<CJK Ideograph>)
      1574   => 16#63F4#,  --  援 (<CJK Ideograph>)
      1575   => 16#6CBF#,  --  沿 (<CJK Ideograph>)
      1576   => 16#6F14#,  --  演 (<CJK Ideograph>)
      1577   => 16#708E#,  --  炎 (<CJK Ideograph>)
      1578   => 16#7114#,  --  焔 (<CJK Ideograph>)
      1579   => 16#7159#,  --  煙 (<CJK Ideograph>)
      1580   => 16#71D5#,  --  燕 (<CJK Ideograph>)
      1581   => 16#733F#,  --  猿 (<CJK Ideograph>)
      1582   => 16#7E01#,  --  縁 (<CJK Ideograph>)
      1583   => 16#8276#,  --  艶 (<CJK Ideograph>)
      1584   => 16#82D1#,  --  苑 (<CJK Ideograph>)
      1585   => 16#8597#,  --  薗 (<CJK Ideograph>)
      1586   => 16#9060#,  --  遠 (<CJK Ideograph>)
      1587   => 16#925B#,  --  鉛 (<CJK Ideograph>)
      1588   => 16#9D1B#,  --  鴛 (<CJK Ideograph>)
      1589   => 16#5869#,  --  塩 (<CJK Ideograph>)
      1590   => 16#65BC#,  --  於 (<CJK Ideograph>)
      1591   => 16#6C5A#,  --  汚 (<CJK Ideograph>)
      1592   => 16#7525#,  --  甥 (<CJK Ideograph>)
      1593   => 16#51F9#,  --  凹 (<CJK Ideograph>)
      1594   => 16#592E#,  --  央 (<CJK Ideograph>)
      1595   => 16#5965#,  --  奥 (<CJK Ideograph>)
      1596   => 16#5F80#,  --  往 (<CJK Ideograph>)
      1597   => 16#5FDC#,  --  応 (<CJK Ideograph>)
      1598   => 16#62BC#,  --  押 (<CJK Ideograph>)
      1599   => 16#65FA#,  --  旺 (<CJK Ideograph>)
      1600   => 16#6A2A#,  --  横 (<CJK Ideograph>)
      1601   => 16#6B27#,  --  欧 (<CJK Ideograph>)
      1602   => 16#6BB4#,  --  殴 (<CJK Ideograph>)
      1603   => 16#738B#,  --  王 (<CJK Ideograph>)
      1604   => 16#7FC1#,  --  翁 (<CJK Ideograph>)
      1605   => 16#8956#,  --  襖 (<CJK Ideograph>)
      1606   => 16#9D2C#,  --  鴬 (<CJK Ideograph>)
      1607   => 16#9D0E#,  --  鴎 (<CJK Ideograph>)
      1608   => 16#9EC4#,  --  黄 (<CJK Ideograph>)
      1609   => 16#5CA1#,  --  岡 (<CJK Ideograph>)
      1610   => 16#6C96#,  --  沖 (<CJK Ideograph>)
      1611   => 16#837B#,  --  荻 (<CJK Ideograph>)
      1612   => 16#5104#,  --  億 (<CJK Ideograph>)
      1613   => 16#5C4B#,  --  屋 (<CJK Ideograph>)
      1614   => 16#61B6#,  --  憶 (<CJK Ideograph>)
      1615   => 16#81C6#,  --  臆 (<CJK Ideograph>)
      1616   => 16#6876#,  --  桶 (<CJK Ideograph>)
      1617   => 16#7261#,  --  牡 (<CJK Ideograph>)
      1618   => 16#4E59#,  --  乙 (<CJK Ideograph>)
      1619   => 16#4FFA#,  --  俺 (<CJK Ideograph>)
      1620   => 16#5378#,  --  卸 (<CJK Ideograph>)
      1621   => 16#6069#,  --  恩 (<CJK Ideograph>)
      1622   => 16#6E29#,  --  温 (<CJK Ideograph>)
      1623   => 16#7A4F#,  --  穏 (<CJK Ideograph>)
      1624   => 16#97F3#,  --  音 (<CJK Ideograph>)
      1625   => 16#4E0B#,  --  下 (<CJK Ideograph>)
      1626   => 16#5316#,  --  化 (<CJK Ideograph>)
      1627   => 16#4EEE#,  --  仮 (<CJK Ideograph>)
      1628   => 16#4F55#,  --  何 (<CJK Ideograph>)
      1629   => 16#4F3D#,  --  伽 (<CJK Ideograph>)
      1630   => 16#4FA1#,  --  価 (<CJK Ideograph>)
      1631   => 16#4F73#,  --  佳 (<CJK Ideograph>)
      1632   => 16#52A0#,  --  加 (<CJK Ideograph>)
      1633   => 16#53EF#,  --  可 (<CJK Ideograph>)
      1634   => 16#5609#,  --  嘉 (<CJK Ideograph>)
      1635   => 16#590F#,  --  夏 (<CJK Ideograph>)
      1636   => 16#5AC1#,  --  嫁 (<CJK Ideograph>)
      1637   => 16#5BB6#,  --  家 (<CJK Ideograph>)
      1638   => 16#5BE1#,  --  寡 (<CJK Ideograph>)
      1639   => 16#79D1#,  --  科 (<CJK Ideograph>)
      1640   => 16#6687#,  --  暇 (<CJK Ideograph>)
      1641   => 16#679C#,  --  果 (<CJK Ideograph>)
      1642   => 16#67B6#,  --  架 (<CJK Ideograph>)
      1643   => 16#6B4C#,  --  歌 (<CJK Ideograph>)
      1644   => 16#6CB3#,  --  河 (<CJK Ideograph>)
      1645   => 16#706B#,  --  火 (<CJK Ideograph>)
      1646   => 16#73C2#,  --  珂 (<CJK Ideograph>)
      1647   => 16#798D#,  --  禍 (<CJK Ideograph>)
      1648   => 16#79BE#,  --  禾 (<CJK Ideograph>)
      1649   => 16#7A3C#,  --  稼 (<CJK Ideograph>)
      1650   => 16#7B87#,  --  箇 (<CJK Ideograph>)
      1651   => 16#82B1#,  --  花 (<CJK Ideograph>)
      1652   => 16#82DB#,  --  苛 (<CJK Ideograph>)
      1653   => 16#8304#,  --  茄 (<CJK Ideograph>)
      1654   => 16#8377#,  --  荷 (<CJK Ideograph>)
      1655   => 16#83EF#,  --  華 (<CJK Ideograph>)
      1656   => 16#83D3#,  --  菓 (<CJK Ideograph>)
      1657   => 16#8766#,  --  蝦 (<CJK Ideograph>)
      1658   => 16#8AB2#,  --  課 (<CJK Ideograph>)
      1659   => 16#5629#,  --  嘩 (<CJK Ideograph>)
      1660   => 16#8CA8#,  --  貨 (<CJK Ideograph>)
      1661   => 16#8FE6#,  --  迦 (<CJK Ideograph>)
      1662   => 16#904E#,  --  過 (<CJK Ideograph>)
      1663   => 16#971E#,  --  霞 (<CJK Ideograph>)
      1664   => 16#868A#,  --  蚊 (<CJK Ideograph>)
      1665   => 16#4FC4#,  --  俄 (<CJK Ideograph>)
      1666   => 16#5CE8#,  --  峨 (<CJK Ideograph>)
      1667   => 16#6211#,  --  我 (<CJK Ideograph>)
      1668   => 16#7259#,  --  牙 (<CJK Ideograph>)
      1669   => 16#753B#,  --  画 (<CJK Ideograph>)
      1670   => 16#81E5#,  --  臥 (<CJK Ideograph>)
      1671   => 16#82BD#,  --  芽 (<CJK Ideograph>)
      1672   => 16#86FE#,  --  蛾 (<CJK Ideograph>)
      1673   => 16#8CC0#,  --  賀 (<CJK Ideograph>)
      1674   => 16#96C5#,  --  雅 (<CJK Ideograph>)
      1675   => 16#9913#,  --  餓 (<CJK Ideograph>)
      1676   => 16#99D5#,  --  駕 (<CJK Ideograph>)
      1677   => 16#4ECB#,  --  介 (<CJK Ideograph>)
      1678   => 16#4F1A#,  --  会 (<CJK Ideograph>)
      1679   => 16#89E3#,  --  解 (<CJK Ideograph>)
      1680   => 16#56DE#,  --  回 (<CJK Ideograph>)
      1681   => 16#584A#,  --  塊 (<CJK Ideograph>)
      1682   => 16#58CA#,  --  壊 (<CJK Ideograph>)
      1683   => 16#5EFB#,  --  廻 (<CJK Ideograph>)
      1684   => 16#5FEB#,  --  快 (<CJK Ideograph>)
      1685   => 16#602A#,  --  怪 (<CJK Ideograph>)
      1686   => 16#6094#,  --  悔 (<CJK Ideograph>)
      1687   => 16#6062#,  --  恢 (<CJK Ideograph>)
      1688   => 16#61D0#,  --  懐 (<CJK Ideograph>)
      1689   => 16#6212#,  --  戒 (<CJK Ideograph>)
      1690   => 16#62D0#,  --  拐 (<CJK Ideograph>)
      1691   => 16#6539#,  --  改 (<CJK Ideograph>)
      1692   => 16#9B41#,  --  魁 (<CJK Ideograph>)
      1693   => 16#6666#,  --  晦 (<CJK Ideograph>)
      1694   => 16#68B0#,  --  械 (<CJK Ideograph>)
      1695   => 16#6D77#,  --  海 (<CJK Ideograph>)
      1696   => 16#7070#,  --  灰 (<CJK Ideograph>)
      1697   => 16#754C#,  --  界 (<CJK Ideograph>)
      1698   => 16#7686#,  --  皆 (<CJK Ideograph>)
      1699   => 16#7D75#,  --  絵 (<CJK Ideograph>)
      1700   => 16#82A5#,  --  芥 (<CJK Ideograph>)
      1701   => 16#87F9#,  --  蟹 (<CJK Ideograph>)
      1702   => 16#958B#,  --  開 (<CJK Ideograph>)
      1703   => 16#968E#,  --  階 (<CJK Ideograph>)
      1704   => 16#8C9D#,  --  貝 (<CJK Ideograph>)
      1705   => 16#51F1#,  --  凱 (<CJK Ideograph>)
      1706   => 16#52BE#,  --  劾 (<CJK Ideograph>)
      1707   => 16#5916#,  --  外 (<CJK Ideograph>)
      1708   => 16#54B3#,  --  咳 (<CJK Ideograph>)
      1709   => 16#5BB3#,  --  害 (<CJK Ideograph>)
      1710   => 16#5D16#,  --  崖 (<CJK Ideograph>)
      1711   => 16#6168#,  --  慨 (<CJK Ideograph>)
      1712   => 16#6982#,  --  概 (<CJK Ideograph>)
      1713   => 16#6DAF#,  --  涯 (<CJK Ideograph>)
      1714   => 16#788D#,  --  碍 (<CJK Ideograph>)
      1715   => 16#84CB#,  --  蓋 (<CJK Ideograph>)
      1716   => 16#8857#,  --  街 (<CJK Ideograph>)
      1717   => 16#8A72#,  --  該 (<CJK Ideograph>)
      1718   => 16#93A7#,  --  鎧 (<CJK Ideograph>)
      1719   => 16#9AB8#,  --  骸 (<CJK Ideograph>)
      1720   => 16#6D6C#,  --  浬 (<CJK Ideograph>)
      1721   => 16#99A8#,  --  馨 (<CJK Ideograph>)
      1722   => 16#86D9#,  --  蛙 (<CJK Ideograph>)
      1723   => 16#57A3#,  --  垣 (<CJK Ideograph>)
      1724   => 16#67FF#,  --  柿 (<CJK Ideograph>)
      1725   => 16#86CE#,  --  蛎 (<CJK Ideograph>)
      1726   => 16#920E#,  --  鈎 (<CJK Ideograph>)
      1727   => 16#5283#,  --  劃 (<CJK Ideograph>)
      1728   => 16#5687#,  --  嚇 (<CJK Ideograph>)
      1729   => 16#5404#,  --  各 (<CJK Ideograph>)
      1730   => 16#5ED3#,  --  廓 (<CJK Ideograph>)
      1731   => 16#62E1#,  --  拡 (<CJK Ideograph>)
      1732   => 16#64B9#,  --  撹 (<CJK Ideograph>)
      1733   => 16#683C#,  --  格 (<CJK Ideograph>)
      1734   => 16#6838#,  --  核 (<CJK Ideograph>)
      1735   => 16#6BBB#,  --  殻 (<CJK Ideograph>)
      1736   => 16#7372#,  --  獲 (<CJK Ideograph>)
      1737   => 16#78BA#,  --  確 (<CJK Ideograph>)
      1738   => 16#7A6B#,  --  穫 (<CJK Ideograph>)
      1739   => 16#899A#,  --  覚 (<CJK Ideograph>)
      1740   => 16#89D2#,  --  角 (<CJK Ideograph>)
      1741   => 16#8D6B#,  --  赫 (<CJK Ideograph>)
      1742   => 16#8F03#,  --  較 (<CJK Ideograph>)
      1743   => 16#90ED#,  --  郭 (<CJK Ideograph>)
      1744   => 16#95A3#,  --  閣 (<CJK Ideograph>)
      1745   => 16#9694#,  --  隔 (<CJK Ideograph>)
      1746   => 16#9769#,  --  革 (<CJK Ideograph>)
      1747   => 16#5B66#,  --  学 (<CJK Ideograph>)
      1748   => 16#5CB3#,  --  岳 (<CJK Ideograph>)
      1749   => 16#697D#,  --  楽 (<CJK Ideograph>)
      1750   => 16#984D#,  --  額 (<CJK Ideograph>)
      1751   => 16#984E#,  --  顎 (<CJK Ideograph>)
      1752   => 16#639B#,  --  掛 (<CJK Ideograph>)
      1753   => 16#7B20#,  --  笠 (<CJK Ideograph>)
      1754   => 16#6A2B#,  --  樫 (<CJK Ideograph>)
      1755   => 16#6A7F#,  --  橿 (<CJK Ideograph>)
      1756   => 16#68B6#,  --  梶 (<CJK Ideograph>)
      1757   => 16#9C0D#,  --  鰍 (<CJK Ideograph>)
      1758   => 16#6F5F#,  --  潟 (<CJK Ideograph>)
      1759   => 16#5272#,  --  割 (<CJK Ideograph>)
      1760   => 16#559D#,  --  喝 (<CJK Ideograph>)
      1761   => 16#6070#,  --  恰 (<CJK Ideograph>)
      1762   => 16#62EC#,  --  括 (<CJK Ideograph>)
      1763   => 16#6D3B#,  --  活 (<CJK Ideograph>)
      1764   => 16#6E07#,  --  渇 (<CJK Ideograph>)
      1765   => 16#6ED1#,  --  滑 (<CJK Ideograph>)
      1766   => 16#845B#,  --  葛 (<CJK Ideograph>)
      1767   => 16#8910#,  --  褐 (<CJK Ideograph>)
      1768   => 16#8F44#,  --  轄 (<CJK Ideograph>)
      1769   => 16#4E14#,  --  且 (<CJK Ideograph>)
      1770   => 16#9C39#,  --  鰹 (<CJK Ideograph>)
      1771   => 16#53F6#,  --  叶 (<CJK Ideograph>)
      1772   => 16#691B#,  --  椛 (<CJK Ideograph>)
      1773   => 16#6A3A#,  --  樺 (<CJK Ideograph>)
      1774   => 16#9784#,  --  鞄 (<CJK Ideograph>)
      1775   => 16#682A#,  --  株 (<CJK Ideograph>)
      1776   => 16#515C#,  --  兜 (<CJK Ideograph>)
      1777   => 16#7AC3#,  --  竃 (<CJK Ideograph>)
      1778   => 16#84B2#,  --  蒲 (<CJK Ideograph>)
      1779   => 16#91DC#,  --  釜 (<CJK Ideograph>)
      1780   => 16#938C#,  --  鎌 (<CJK Ideograph>)
      1781   => 16#565B#,  --  噛 (<CJK Ideograph>)
      1782   => 16#9D28#,  --  鴨 (<CJK Ideograph>)
      1783   => 16#6822#,  --  栢 (<CJK Ideograph>)
      1784   => 16#8305#,  --  茅 (<CJK Ideograph>)
      1785   => 16#8431#,  --  萱 (<CJK Ideograph>)
      1786   => 16#7CA5#,  --  粥 (<CJK Ideograph>)
      1787   => 16#5208#,  --  刈 (<CJK Ideograph>)
      1788   => 16#82C5#,  --  苅 (<CJK Ideograph>)
      1789   => 16#74E6#,  --  瓦 (<CJK Ideograph>)
      1790   => 16#4E7E#,  --  乾 (<CJK Ideograph>)
      1791   => 16#4F83#,  --  侃 (<CJK Ideograph>)
      1792   => 16#51A0#,  --  冠 (<CJK Ideograph>)
      1793   => 16#5BD2#,  --  寒 (<CJK Ideograph>)
      1794   => 16#520A#,  --  刊 (<CJK Ideograph>)
      1795   => 16#52D8#,  --  勘 (<CJK Ideograph>)
      1796   => 16#52E7#,  --  勧 (<CJK Ideograph>)
      1797   => 16#5DFB#,  --  巻 (<CJK Ideograph>)
      1798   => 16#559A#,  --  喚 (<CJK Ideograph>)
      1799   => 16#582A#,  --  堪 (<CJK Ideograph>)
      1800   => 16#59E6#,  --  姦 (<CJK Ideograph>)
      1801   => 16#5B8C#,  --  完 (<CJK Ideograph>)
      1802   => 16#5B98#,  --  官 (<CJK Ideograph>)
      1803   => 16#5BDB#,  --  寛 (<CJK Ideograph>)
      1804   => 16#5E72#,  --  干 (<CJK Ideograph>)
      1805   => 16#5E79#,  --  幹 (<CJK Ideograph>)
      1806   => 16#60A3#,  --  患 (<CJK Ideograph>)
      1807   => 16#611F#,  --  感 (<CJK Ideograph>)
      1808   => 16#6163#,  --  慣 (<CJK Ideograph>)
      1809   => 16#61BE#,  --  憾 (<CJK Ideograph>)
      1810   => 16#63DB#,  --  換 (<CJK Ideograph>)
      1811   => 16#6562#,  --  敢 (<CJK Ideograph>)
      1812   => 16#67D1#,  --  柑 (<CJK Ideograph>)
      1813   => 16#6853#,  --  桓 (<CJK Ideograph>)
      1814   => 16#68FA#,  --  棺 (<CJK Ideograph>)
      1815   => 16#6B3E#,  --  款 (<CJK Ideograph>)
      1816   => 16#6B53#,  --  歓 (<CJK Ideograph>)
      1817   => 16#6C57#,  --  汗 (<CJK Ideograph>)
      1818   => 16#6F22#,  --  漢 (<CJK Ideograph>)
      1819   => 16#6F97#,  --  澗 (<CJK Ideograph>)
      1820   => 16#6F45#,  --  潅 (<CJK Ideograph>)
      1821   => 16#74B0#,  --  環 (<CJK Ideograph>)
      1822   => 16#7518#,  --  甘 (<CJK Ideograph>)
      1823   => 16#76E3#,  --  監 (<CJK Ideograph>)
      1824   => 16#770B#,  --  看 (<CJK Ideograph>)
      1825   => 16#7AFF#,  --  竿 (<CJK Ideograph>)
      1826   => 16#7BA1#,  --  管 (<CJK Ideograph>)
      1827   => 16#7C21#,  --  簡 (<CJK Ideograph>)
      1828   => 16#7DE9#,  --  緩 (<CJK Ideograph>)
      1829   => 16#7F36#,  --  缶 (<CJK Ideograph>)
      1830   => 16#7FF0#,  --  翰 (<CJK Ideograph>)
      1831   => 16#809D#,  --  肝 (<CJK Ideograph>)
      1832   => 16#8266#,  --  艦 (<CJK Ideograph>)
      1833   => 16#839E#,  --  莞 (<CJK Ideograph>)
      1834   => 16#89B3#,  --  観 (<CJK Ideograph>)
      1835   => 16#8ACC#,  --  諌 (<CJK Ideograph>)
      1836   => 16#8CAB#,  --  貫 (<CJK Ideograph>)
      1837   => 16#9084#,  --  還 (<CJK Ideograph>)
      1838   => 16#9451#,  --  鑑 (<CJK Ideograph>)
      1839   => 16#9593#,  --  間 (<CJK Ideograph>)
      1840   => 16#9591#,  --  閑 (<CJK Ideograph>)
      1841   => 16#95A2#,  --  関 (<CJK Ideograph>)
      1842   => 16#9665#,  --  陥 (<CJK Ideograph>)
      1843   => 16#97D3#,  --  韓 (<CJK Ideograph>)
      1844   => 16#9928#,  --  館 (<CJK Ideograph>)
      1845   => 16#8218#,  --  舘 (<CJK Ideograph>)
      1846   => 16#4E38#,  --  丸 (<CJK Ideograph>)
      1847   => 16#542B#,  --  含 (<CJK Ideograph>)
      1848   => 16#5CB8#,  --  岸 (<CJK Ideograph>)
      1849   => 16#5DCC#,  --  巌 (<CJK Ideograph>)
      1850   => 16#73A9#,  --  玩 (<CJK Ideograph>)
      1851   => 16#764C#,  --  癌 (<CJK Ideograph>)
      1852   => 16#773C#,  --  眼 (<CJK Ideograph>)
      1853   => 16#5CA9#,  --  岩 (<CJK Ideograph>)
      1854   => 16#7FEB#,  --  翫 (<CJK Ideograph>)
      1855   => 16#8D0B#,  --  贋 (<CJK Ideograph>)
      1856   => 16#96C1#,  --  雁 (<CJK Ideograph>)
      1857   => 16#9811#,  --  頑 (<CJK Ideograph>)
      1858   => 16#9854#,  --  顔 (<CJK Ideograph>)
      1859   => 16#9858#,  --  願 (<CJK Ideograph>)
      1860   => 16#4F01#,  --  企 (<CJK Ideograph>)
      1861   => 16#4F0E#,  --  伎 (<CJK Ideograph>)
      1862   => 16#5371#,  --  危 (<CJK Ideograph>)
      1863   => 16#559C#,  --  喜 (<CJK Ideograph>)
      1864   => 16#5668#,  --  器 (<CJK Ideograph>)
      1865   => 16#57FA#,  --  基 (<CJK Ideograph>)
      1866   => 16#5947#,  --  奇 (<CJK Ideograph>)
      1867   => 16#5B09#,  --  嬉 (<CJK Ideograph>)
      1868   => 16#5BC4#,  --  寄 (<CJK Ideograph>)
      1869   => 16#5C90#,  --  岐 (<CJK Ideograph>)
      1870   => 16#5E0C#,  --  希 (<CJK Ideograph>)
      1871   => 16#5E7E#,  --  幾 (<CJK Ideograph>)
      1872   => 16#5FCC#,  --  忌 (<CJK Ideograph>)
      1873   => 16#63EE#,  --  揮 (<CJK Ideograph>)
      1874   => 16#673A#,  --  机 (<CJK Ideograph>)
      1875   => 16#65D7#,  --  旗 (<CJK Ideograph>)
      1876   => 16#65E2#,  --  既 (<CJK Ideograph>)
      1877   => 16#671F#,  --  期 (<CJK Ideograph>)
      1878   => 16#68CB#,  --  棋 (<CJK Ideograph>)
      1879   => 16#68C4#,  --  棄 (<CJK Ideograph>)
      1880   => 16#6A5F#,  --  機 (<CJK Ideograph>)
      1881   => 16#5E30#,  --  帰 (<CJK Ideograph>)
      1882   => 16#6BC5#,  --  毅 (<CJK Ideograph>)
      1883   => 16#6C17#,  --  気 (<CJK Ideograph>)
      1884   => 16#6C7D#,  --  汽 (<CJK Ideograph>)
      1885   => 16#757F#,  --  畿 (<CJK Ideograph>)
      1886   => 16#7948#,  --  祈 (<CJK Ideograph>)
      1887   => 16#5B63#,  --  季 (<CJK Ideograph>)
      1888   => 16#7A00#,  --  稀 (<CJK Ideograph>)
      1889   => 16#7D00#,  --  紀 (<CJK Ideograph>)
      1890   => 16#5FBD#,  --  徽 (<CJK Ideograph>)
      1891   => 16#898F#,  --  規 (<CJK Ideograph>)
      1892   => 16#8A18#,  --  記 (<CJK Ideograph>)
      1893   => 16#8CB4#,  --  貴 (<CJK Ideograph>)
      1894   => 16#8D77#,  --  起 (<CJK Ideograph>)
      1895   => 16#8ECC#,  --  軌 (<CJK Ideograph>)
      1896   => 16#8F1D#,  --  輝 (<CJK Ideograph>)
      1897   => 16#98E2#,  --  飢 (<CJK Ideograph>)
      1898   => 16#9A0E#,  --  騎 (<CJK Ideograph>)
      1899   => 16#9B3C#,  --  鬼 (<CJK Ideograph>)
      1900   => 16#4E80#,  --  亀 (<CJK Ideograph>)
      1901   => 16#507D#,  --  偽 (<CJK Ideograph>)
      1902   => 16#5100#,  --  儀 (<CJK Ideograph>)
      1903   => 16#5993#,  --  妓 (<CJK Ideograph>)
      1904   => 16#5B9C#,  --  宜 (<CJK Ideograph>)
      1905   => 16#622F#,  --  戯 (<CJK Ideograph>)
      1906   => 16#6280#,  --  技 (<CJK Ideograph>)
      1907   => 16#64EC#,  --  擬 (<CJK Ideograph>)
      1908   => 16#6B3A#,  --  欺 (<CJK Ideograph>)
      1909   => 16#72A0#,  --  犠 (<CJK Ideograph>)
      1910   => 16#7591#,  --  疑 (<CJK Ideograph>)
      1911   => 16#7947#,  --  祇 (<CJK Ideograph>)
      1912   => 16#7FA9#,  --  義 (<CJK Ideograph>)
      1913   => 16#87FB#,  --  蟻 (<CJK Ideograph>)
      1914   => 16#8ABC#,  --  誼 (<CJK Ideograph>)
      1915   => 16#8B70#,  --  議 (<CJK Ideograph>)
      1916   => 16#63AC#,  --  掬 (<CJK Ideograph>)
      1917   => 16#83CA#,  --  菊 (<CJK Ideograph>)
      1918   => 16#97A0#,  --  鞠 (<CJK Ideograph>)
      1919   => 16#5409#,  --  吉 (<CJK Ideograph>)
      1920   => 16#5403#,  --  吃 (<CJK Ideograph>)
      1921   => 16#55AB#,  --  喫 (<CJK Ideograph>)
      1922   => 16#6854#,  --  桔 (<CJK Ideograph>)
      1923   => 16#6A58#,  --  橘 (<CJK Ideograph>)
      1924   => 16#8A70#,  --  詰 (<CJK Ideograph>)
      1925   => 16#7827#,  --  砧 (<CJK Ideograph>)
      1926   => 16#6775#,  --  杵 (<CJK Ideograph>)
      1927   => 16#9ECD#,  --  黍 (<CJK Ideograph>)
      1928   => 16#5374#,  --  却 (<CJK Ideograph>)
      1929   => 16#5BA2#,  --  客 (<CJK Ideograph>)
      1930   => 16#811A#,  --  脚 (<CJK Ideograph>)
      1931   => 16#8650#,  --  虐 (<CJK Ideograph>)
      1932   => 16#9006#,  --  逆 (<CJK Ideograph>)
      1933   => 16#4E18#,  --  丘 (<CJK Ideograph>)
      1934   => 16#4E45#,  --  久 (<CJK Ideograph>)
      1935   => 16#4EC7#,  --  仇 (<CJK Ideograph>)
      1936   => 16#4F11#,  --  休 (<CJK Ideograph>)
      1937   => 16#53CA#,  --  及 (<CJK Ideograph>)
      1938   => 16#5438#,  --  吸 (<CJK Ideograph>)
      1939   => 16#5BAE#,  --  宮 (<CJK Ideograph>)
      1940   => 16#5F13#,  --  弓 (<CJK Ideograph>)
      1941   => 16#6025#,  --  急 (<CJK Ideograph>)
      1942   => 16#6551#,  --  救 (<CJK Ideograph>)
      1943   => 16#673D#,  --  朽 (<CJK Ideograph>)
      1944   => 16#6C42#,  --  求 (<CJK Ideograph>)
      1945   => 16#6C72#,  --  汲 (<CJK Ideograph>)
      1946   => 16#6CE3#,  --  泣 (<CJK Ideograph>)
      1947   => 16#7078#,  --  灸 (<CJK Ideograph>)
      1948   => 16#7403#,  --  球 (<CJK Ideograph>)
      1949   => 16#7A76#,  --  究 (<CJK Ideograph>)
      1950   => 16#7AAE#,  --  窮 (<CJK Ideograph>)
      1951   => 16#7B08#,  --  笈 (<CJK Ideograph>)
      1952   => 16#7D1A#,  --  級 (<CJK Ideograph>)
      1953   => 16#7CFE#,  --  糾 (<CJK Ideograph>)
      1954   => 16#7D66#,  --  給 (<CJK Ideograph>)
      1955   => 16#65E7#,  --  旧 (<CJK Ideograph>)
      1956   => 16#725B#,  --  牛 (<CJK Ideograph>)
      1957   => 16#53BB#,  --  去 (<CJK Ideograph>)
      1958   => 16#5C45#,  --  居 (<CJK Ideograph>)
      1959   => 16#5DE8#,  --  巨 (<CJK Ideograph>)
      1960   => 16#62D2#,  --  拒 (<CJK Ideograph>)
      1961   => 16#62E0#,  --  拠 (<CJK Ideograph>)
      1962   => 16#6319#,  --  挙 (<CJK Ideograph>)
      1963   => 16#6E20#,  --  渠 (<CJK Ideograph>)
      1964   => 16#865A#,  --  虚 (<CJK Ideograph>)
      1965   => 16#8A31#,  --  許 (<CJK Ideograph>)
      1966   => 16#8DDD#,  --  距 (<CJK Ideograph>)
      1967   => 16#92F8#,  --  鋸 (<CJK Ideograph>)
      1968   => 16#6F01#,  --  漁 (<CJK Ideograph>)
      1969   => 16#79A6#,  --  禦 (<CJK Ideograph>)
      1970   => 16#9B5A#,  --  魚 (<CJK Ideograph>)
      1971   => 16#4EA8#,  --  亨 (<CJK Ideograph>)
      1972   => 16#4EAB#,  --  享 (<CJK Ideograph>)
      1973   => 16#4EAC#,  --  京 (<CJK Ideograph>)
      1974   => 16#4F9B#,  --  供 (<CJK Ideograph>)
      1975   => 16#4FA0#,  --  侠 (<CJK Ideograph>)
      1976   => 16#50D1#,  --  僑 (<CJK Ideograph>)
      1977   => 16#5147#,  --  兇 (<CJK Ideograph>)
      1978   => 16#7AF6#,  --  競 (<CJK Ideograph>)
      1979   => 16#5171#,  --  共 (<CJK Ideograph>)
      1980   => 16#51F6#,  --  凶 (<CJK Ideograph>)
      1981   => 16#5354#,  --  協 (<CJK Ideograph>)
      1982   => 16#5321#,  --  匡 (<CJK Ideograph>)
      1983   => 16#537F#,  --  卿 (<CJK Ideograph>)
      1984   => 16#53EB#,  --  叫 (<CJK Ideograph>)
      1985   => 16#55AC#,  --  喬 (<CJK Ideograph>)
      1986   => 16#5883#,  --  境 (<CJK Ideograph>)
      1987   => 16#5CE1#,  --  峡 (<CJK Ideograph>)
      1988   => 16#5F37#,  --  強 (<CJK Ideograph>)
      1989   => 16#5F4A#,  --  彊 (<CJK Ideograph>)
      1990   => 16#602F#,  --  怯 (<CJK Ideograph>)
      1991   => 16#6050#,  --  恐 (<CJK Ideograph>)
      1992   => 16#606D#,  --  恭 (<CJK Ideograph>)
      1993   => 16#631F#,  --  挟 (<CJK Ideograph>)
      1994   => 16#6559#,  --  教 (<CJK Ideograph>)
      1995   => 16#6A4B#,  --  橋 (<CJK Ideograph>)
      1996   => 16#6CC1#,  --  況 (<CJK Ideograph>)
      1997   => 16#72C2#,  --  狂 (<CJK Ideograph>)
      1998   => 16#72ED#,  --  狭 (<CJK Ideograph>)
      1999   => 16#77EF#,  --  矯 (<CJK Ideograph>)
      2000   => 16#80F8#,  --  胸 (<CJK Ideograph>)
      2001   => 16#8105#,  --  脅 (<CJK Ideograph>)
      2002   => 16#8208#,  --  興 (<CJK Ideograph>)
      2003   => 16#854E#,  --  蕎 (<CJK Ideograph>)
      2004   => 16#90F7#,  --  郷 (<CJK Ideograph>)
      2005   => 16#93E1#,  --  鏡 (<CJK Ideograph>)
      2006   => 16#97FF#,  --  響 (<CJK Ideograph>)
      2007   => 16#9957#,  --  饗 (<CJK Ideograph>)
      2008   => 16#9A5A#,  --  驚 (<CJK Ideograph>)
      2009   => 16#4EF0#,  --  仰 (<CJK Ideograph>)
      2010   => 16#51DD#,  --  凝 (<CJK Ideograph>)
      2011   => 16#5C2D#,  --  尭 (<CJK Ideograph>)
      2012   => 16#6681#,  --  暁 (<CJK Ideograph>)
      2013   => 16#696D#,  --  業 (<CJK Ideograph>)
      2014   => 16#5C40#,  --  局 (<CJK Ideograph>)
      2015   => 16#66F2#,  --  曲 (<CJK Ideograph>)
      2016   => 16#6975#,  --  極 (<CJK Ideograph>)
      2017   => 16#7389#,  --  玉 (<CJK Ideograph>)
      2018   => 16#6850#,  --  桐 (<CJK Ideograph>)
      2019   => 16#7C81#,  --  粁 (<CJK Ideograph>)
      2020   => 16#50C5#,  --  僅 (<CJK Ideograph>)
      2021   => 16#52E4#,  --  勤 (<CJK Ideograph>)
      2022   => 16#5747#,  --  均 (<CJK Ideograph>)
      2023   => 16#5DFE#,  --  巾 (<CJK Ideograph>)
      2024   => 16#9326#,  --  錦 (<CJK Ideograph>)
      2025   => 16#65A4#,  --  斤 (<CJK Ideograph>)
      2026   => 16#6B23#,  --  欣 (<CJK Ideograph>)
      2027   => 16#6B3D#,  --  欽 (<CJK Ideograph>)
      2028   => 16#7434#,  --  琴 (<CJK Ideograph>)
      2029   => 16#7981#,  --  禁 (<CJK Ideograph>)
      2030   => 16#79BD#,  --  禽 (<CJK Ideograph>)
      2031   => 16#7B4B#,  --  筋 (<CJK Ideograph>)
      2032   => 16#7DCA#,  --  緊 (<CJK Ideograph>)
      2033   => 16#82B9#,  --  芹 (<CJK Ideograph>)
      2034   => 16#83CC#,  --  菌 (<CJK Ideograph>)
      2035   => 16#887F#,  --  衿 (<CJK Ideograph>)
      2036   => 16#895F#,  --  襟 (<CJK Ideograph>)
      2037   => 16#8B39#,  --  謹 (<CJK Ideograph>)
      2038   => 16#8FD1#,  --  近 (<CJK Ideograph>)
      2039   => 16#91D1#,  --  金 (<CJK Ideograph>)
      2040   => 16#541F#,  --  吟 (<CJK Ideograph>)
      2041   => 16#9280#,  --  銀 (<CJK Ideograph>)
      2042   => 16#4E5D#,  --  九 (<CJK Ideograph>)
      2043   => 16#5036#,  --  倶 (<CJK Ideograph>)
      2044   => 16#53E5#,  --  句 (<CJK Ideograph>)
      2045   => 16#533A#,  --  区 (<CJK Ideograph>)
      2046   => 16#72D7#,  --  狗 (<CJK Ideograph>)
      2047   => 16#7396#,  --  玖 (<CJK Ideograph>)
      2048   => 16#77E9#,  --  矩 (<CJK Ideograph>)
      2049   => 16#82E6#,  --  苦 (<CJK Ideograph>)
      2050   => 16#8EAF#,  --  躯 (<CJK Ideograph>)
      2051   => 16#99C6#,  --  駆 (<CJK Ideograph>)
      2052   => 16#99C8#,  --  駈 (<CJK Ideograph>)
      2053   => 16#99D2#,  --  駒 (<CJK Ideograph>)
      2054   => 16#5177#,  --  具 (<CJK Ideograph>)
      2055   => 16#611A#,  --  愚 (<CJK Ideograph>)
      2056   => 16#865E#,  --  虞 (<CJK Ideograph>)
      2057   => 16#55B0#,  --  喰 (<CJK Ideograph>)
      2058   => 16#7A7A#,  --  空 (<CJK Ideograph>)
      2059   => 16#5076#,  --  偶 (<CJK Ideograph>)
      2060   => 16#5BD3#,  --  寓 (<CJK Ideograph>)
      2061   => 16#9047#,  --  遇 (<CJK Ideograph>)
      2062   => 16#9685#,  --  隅 (<CJK Ideograph>)
      2063   => 16#4E32#,  --  串 (<CJK Ideograph>)
      2064   => 16#6ADB#,  --  櫛 (<CJK Ideograph>)
      2065   => 16#91E7#,  --  釧 (<CJK Ideograph>)
      2066   => 16#5C51#,  --  屑 (<CJK Ideograph>)
      2067   => 16#5C48#,  --  屈 (<CJK Ideograph>)
      2068   => 16#6398#,  --  掘 (<CJK Ideograph>)
      2069   => 16#7A9F#,  --  窟 (<CJK Ideograph>)
      2070   => 16#6C93#,  --  沓 (<CJK Ideograph>)
      2071   => 16#9774#,  --  靴 (<CJK Ideograph>)
      2072   => 16#8F61#,  --  轡 (<CJK Ideograph>)
      2073   => 16#7AAA#,  --  窪 (<CJK Ideograph>)
      2074   => 16#718A#,  --  熊 (<CJK Ideograph>)
      2075   => 16#9688#,  --  隈 (<CJK Ideograph>)
      2076   => 16#7C82#,  --  粂 (<CJK Ideograph>)
      2077   => 16#6817#,  --  栗 (<CJK Ideograph>)
      2078   => 16#7E70#,  --  繰 (<CJK Ideograph>)
      2079   => 16#6851#,  --  桑 (<CJK Ideograph>)
      2080   => 16#936C#,  --  鍬 (<CJK Ideograph>)
      2081   => 16#52F2#,  --  勲 (<CJK Ideograph>)
      2082   => 16#541B#,  --  君 (<CJK Ideograph>)
      2083   => 16#85AB#,  --  薫 (<CJK Ideograph>)
      2084   => 16#8A13#,  --  訓 (<CJK Ideograph>)
      2085   => 16#7FA4#,  --  群 (<CJK Ideograph>)
      2086   => 16#8ECD#,  --  軍 (<CJK Ideograph>)
      2087   => 16#90E1#,  --  郡 (<CJK Ideograph>)
      2088   => 16#5366#,  --  卦 (<CJK Ideograph>)
      2089   => 16#8888#,  --  袈 (<CJK Ideograph>)
      2090   => 16#7941#,  --  祁 (<CJK Ideograph>)
      2091   => 16#4FC2#,  --  係 (<CJK Ideograph>)
      2092   => 16#50BE#,  --  傾 (<CJK Ideograph>)
      2093   => 16#5211#,  --  刑 (<CJK Ideograph>)
      2094   => 16#5144#,  --  兄 (<CJK Ideograph>)
      2095   => 16#5553#,  --  啓 (<CJK Ideograph>)
      2096   => 16#572D#,  --  圭 (<CJK Ideograph>)
      2097   => 16#73EA#,  --  珪 (<CJK Ideograph>)
      2098   => 16#578B#,  --  型 (<CJK Ideograph>)
      2099   => 16#5951#,  --  契 (<CJK Ideograph>)
      2100   => 16#5F62#,  --  形 (<CJK Ideograph>)
      2101   => 16#5F84#,  --  径 (<CJK Ideograph>)
      2102   => 16#6075#,  --  恵 (<CJK Ideograph>)
      2103   => 16#6176#,  --  慶 (<CJK Ideograph>)
      2104   => 16#6167#,  --  慧 (<CJK Ideograph>)
      2105   => 16#61A9#,  --  憩 (<CJK Ideograph>)
      2106   => 16#63B2#,  --  掲 (<CJK Ideograph>)
      2107   => 16#643A#,  --  携 (<CJK Ideograph>)
      2108   => 16#656C#,  --  敬 (<CJK Ideograph>)
      2109   => 16#666F#,  --  景 (<CJK Ideograph>)
      2110   => 16#6842#,  --  桂 (<CJK Ideograph>)
      2111   => 16#6E13#,  --  渓 (<CJK Ideograph>)
      2112   => 16#7566#,  --  畦 (<CJK Ideograph>)
      2113   => 16#7A3D#,  --  稽 (<CJK Ideograph>)
      2114   => 16#7CFB#,  --  系 (<CJK Ideograph>)
      2115   => 16#7D4C#,  --  経 (<CJK Ideograph>)
      2116   => 16#7D99#,  --  継 (<CJK Ideograph>)
      2117   => 16#7E4B#,  --  繋 (<CJK Ideograph>)
      2118   => 16#7F6B#,  --  罫 (<CJK Ideograph>)
      2119   => 16#830E#,  --  茎 (<CJK Ideograph>)
      2120   => 16#834A#,  --  荊 (<CJK Ideograph>)
      2121   => 16#86CD#,  --  蛍 (<CJK Ideograph>)
      2122   => 16#8A08#,  --  計 (<CJK Ideograph>)
      2123   => 16#8A63#,  --  詣 (<CJK Ideograph>)
      2124   => 16#8B66#,  --  警 (<CJK Ideograph>)
      2125   => 16#8EFD#,  --  軽 (<CJK Ideograph>)
      2126   => 16#981A#,  --  頚 (<CJK Ideograph>)
      2127   => 16#9D8F#,  --  鶏 (<CJK Ideograph>)
      2128   => 16#82B8#,  --  芸 (<CJK Ideograph>)
      2129   => 16#8FCE#,  --  迎 (<CJK Ideograph>)
      2130   => 16#9BE8#,  --  鯨 (<CJK Ideograph>)
      2131   => 16#5287#,  --  劇 (<CJK Ideograph>)
      2132   => 16#621F#,  --  戟 (<CJK Ideograph>)
      2133   => 16#6483#,  --  撃 (<CJK Ideograph>)
      2134   => 16#6FC0#,  --  激 (<CJK Ideograph>)
      2135   => 16#9699#,  --  隙 (<CJK Ideograph>)
      2136   => 16#6841#,  --  桁 (<CJK Ideograph>)
      2137   => 16#5091#,  --  傑 (<CJK Ideograph>)
      2138   => 16#6B20#,  --  欠 (<CJK Ideograph>)
      2139   => 16#6C7A#,  --  決 (<CJK Ideograph>)
      2140   => 16#6F54#,  --  潔 (<CJK Ideograph>)
      2141   => 16#7A74#,  --  穴 (<CJK Ideograph>)
      2142   => 16#7D50#,  --  結 (<CJK Ideograph>)
      2143   => 16#8840#,  --  血 (<CJK Ideograph>)
      2144   => 16#8A23#,  --  訣 (<CJK Ideograph>)
      2145   => 16#6708#,  --  月 (<CJK Ideograph>)
      2146   => 16#4EF6#,  --  件 (<CJK Ideograph>)
      2147   => 16#5039#,  --  倹 (<CJK Ideograph>)
      2148   => 16#5026#,  --  倦 (<CJK Ideograph>)
      2149   => 16#5065#,  --  健 (<CJK Ideograph>)
      2150   => 16#517C#,  --  兼 (<CJK Ideograph>)
      2151   => 16#5238#,  --  券 (<CJK Ideograph>)
      2152   => 16#5263#,  --  剣 (<CJK Ideograph>)
      2153   => 16#55A7#,  --  喧 (<CJK Ideograph>)
      2154   => 16#570F#,  --  圏 (<CJK Ideograph>)
      2155   => 16#5805#,  --  堅 (<CJK Ideograph>)
      2156   => 16#5ACC#,  --  嫌 (<CJK Ideograph>)
      2157   => 16#5EFA#,  --  建 (<CJK Ideograph>)
      2158   => 16#61B2#,  --  憲 (<CJK Ideograph>)
      2159   => 16#61F8#,  --  懸 (<CJK Ideograph>)
      2160   => 16#62F3#,  --  拳 (<CJK Ideograph>)
      2161   => 16#6372#,  --  捲 (<CJK Ideograph>)
      2162   => 16#691C#,  --  検 (<CJK Ideograph>)
      2163   => 16#6A29#,  --  権 (<CJK Ideograph>)
      2164   => 16#727D#,  --  牽 (<CJK Ideograph>)
      2165   => 16#72AC#,  --  犬 (<CJK Ideograph>)
      2166   => 16#732E#,  --  献 (<CJK Ideograph>)
      2167   => 16#7814#,  --  研 (<CJK Ideograph>)
      2168   => 16#786F#,  --  硯 (<CJK Ideograph>)
      2169   => 16#7D79#,  --  絹 (<CJK Ideograph>)
      2170   => 16#770C#,  --  県 (<CJK Ideograph>)
      2171   => 16#80A9#,  --  肩 (<CJK Ideograph>)
      2172   => 16#898B#,  --  見 (<CJK Ideograph>)
      2173   => 16#8B19#,  --  謙 (<CJK Ideograph>)
      2174   => 16#8CE2#,  --  賢 (<CJK Ideograph>)
      2175   => 16#8ED2#,  --  軒 (<CJK Ideograph>)
      2176   => 16#9063#,  --  遣 (<CJK Ideograph>)
      2177   => 16#9375#,  --  鍵 (<CJK Ideograph>)
      2178   => 16#967A#,  --  険 (<CJK Ideograph>)
      2179   => 16#9855#,  --  顕 (<CJK Ideograph>)
      2180   => 16#9A13#,  --  験 (<CJK Ideograph>)
      2181   => 16#9E78#,  --  鹸 (<CJK Ideograph>)
      2182   => 16#5143#,  --  元 (<CJK Ideograph>)
      2183   => 16#539F#,  --  原 (<CJK Ideograph>)
      2184   => 16#53B3#,  --  厳 (<CJK Ideograph>)
      2185   => 16#5E7B#,  --  幻 (<CJK Ideograph>)
      2186   => 16#5F26#,  --  弦 (<CJK Ideograph>)
      2187   => 16#6E1B#,  --  減 (<CJK Ideograph>)
      2188   => 16#6E90#,  --  源 (<CJK Ideograph>)
      2189   => 16#7384#,  --  玄 (<CJK Ideograph>)
      2190   => 16#73FE#,  --  現 (<CJK Ideograph>)
      2191   => 16#7D43#,  --  絃 (<CJK Ideograph>)
      2192   => 16#8237#,  --  舷 (<CJK Ideograph>)
      2193   => 16#8A00#,  --  言 (<CJK Ideograph>)
      2194   => 16#8AFA#,  --  諺 (<CJK Ideograph>)
      2195   => 16#9650#,  --  限 (<CJK Ideograph>)
      2196   => 16#4E4E#,  --  乎 (<CJK Ideograph>)
      2197   => 16#500B#,  --  個 (<CJK Ideograph>)
      2198   => 16#53E4#,  --  古 (<CJK Ideograph>)
      2199   => 16#547C#,  --  呼 (<CJK Ideograph>)
      2200   => 16#56FA#,  --  固 (<CJK Ideograph>)
      2201   => 16#59D1#,  --  姑 (<CJK Ideograph>)
      2202   => 16#5B64#,  --  孤 (<CJK Ideograph>)
      2203   => 16#5DF1#,  --  己 (<CJK Ideograph>)
      2204   => 16#5EAB#,  --  庫 (<CJK Ideograph>)
      2205   => 16#5F27#,  --  弧 (<CJK Ideograph>)
      2206   => 16#6238#,  --  戸 (<CJK Ideograph>)
      2207   => 16#6545#,  --  故 (<CJK Ideograph>)
      2208   => 16#67AF#,  --  枯 (<CJK Ideograph>)
      2209   => 16#6E56#,  --  湖 (<CJK Ideograph>)
      2210   => 16#72D0#,  --  狐 (<CJK Ideograph>)
      2211   => 16#7CCA#,  --  糊 (<CJK Ideograph>)
      2212   => 16#88B4#,  --  袴 (<CJK Ideograph>)
      2213   => 16#80A1#,  --  股 (<CJK Ideograph>)
      2214   => 16#80E1#,  --  胡 (<CJK Ideograph>)
      2215   => 16#83F0#,  --  菰 (<CJK Ideograph>)
      2216   => 16#864E#,  --  虎 (<CJK Ideograph>)
      2217   => 16#8A87#,  --  誇 (<CJK Ideograph>)
      2218   => 16#8DE8#,  --  跨 (<CJK Ideograph>)
      2219   => 16#9237#,  --  鈷 (<CJK Ideograph>)
      2220   => 16#96C7#,  --  雇 (<CJK Ideograph>)
      2221   => 16#9867#,  --  顧 (<CJK Ideograph>)
      2222   => 16#9F13#,  --  鼓 (<CJK Ideograph>)
      2223   => 16#4E94#,  --  五 (<CJK Ideograph>)
      2224   => 16#4E92#,  --  互 (<CJK Ideograph>)
      2225   => 16#4F0D#,  --  伍 (<CJK Ideograph>)
      2226   => 16#5348#,  --  午 (<CJK Ideograph>)
      2227   => 16#5449#,  --  呉 (<CJK Ideograph>)
      2228   => 16#543E#,  --  吾 (<CJK Ideograph>)
      2229   => 16#5A2F#,  --  娯 (<CJK Ideograph>)
      2230   => 16#5F8C#,  --  後 (<CJK Ideograph>)
      2231   => 16#5FA1#,  --  御 (<CJK Ideograph>)
      2232   => 16#609F#,  --  悟 (<CJK Ideograph>)
      2233   => 16#68A7#,  --  梧 (<CJK Ideograph>)
      2234   => 16#6A8E#,  --  檎 (<CJK Ideograph>)
      2235   => 16#745A#,  --  瑚 (<CJK Ideograph>)
      2236   => 16#7881#,  --  碁 (<CJK Ideograph>)
      2237   => 16#8A9E#,  --  語 (<CJK Ideograph>)
      2238   => 16#8AA4#,  --  誤 (<CJK Ideograph>)
      2239   => 16#8B77#,  --  護 (<CJK Ideograph>)
      2240   => 16#9190#,  --  醐 (<CJK Ideograph>)
      2241   => 16#4E5E#,  --  乞 (<CJK Ideograph>)
      2242   => 16#9BC9#,  --  鯉 (<CJK Ideograph>)
      2243   => 16#4EA4#,  --  交 (<CJK Ideograph>)
      2244   => 16#4F7C#,  --  佼 (<CJK Ideograph>)
      2245   => 16#4FAF#,  --  侯 (<CJK Ideograph>)
      2246   => 16#5019#,  --  候 (<CJK Ideograph>)
      2247   => 16#5016#,  --  倖 (<CJK Ideograph>)
      2248   => 16#5149#,  --  光 (<CJK Ideograph>)
      2249   => 16#516C#,  --  公 (<CJK Ideograph>)
      2250   => 16#529F#,  --  功 (<CJK Ideograph>)
      2251   => 16#52B9#,  --  効 (<CJK Ideograph>)
      2252   => 16#52FE#,  --  勾 (<CJK Ideograph>)
      2253   => 16#539A#,  --  厚 (<CJK Ideograph>)
      2254   => 16#53E3#,  --  口 (<CJK Ideograph>)
      2255   => 16#5411#,  --  向 (<CJK Ideograph>)
      2256   => 16#540E#,  --  后 (<CJK Ideograph>)
      2257   => 16#5589#,  --  喉 (<CJK Ideograph>)
      2258   => 16#5751#,  --  坑 (<CJK Ideograph>)
      2259   => 16#57A2#,  --  垢 (<CJK Ideograph>)
      2260   => 16#597D#,  --  好 (<CJK Ideograph>)
      2261   => 16#5B54#,  --  孔 (<CJK Ideograph>)
      2262   => 16#5B5D#,  --  孝 (<CJK Ideograph>)
      2263   => 16#5B8F#,  --  宏 (<CJK Ideograph>)
      2264   => 16#5DE5#,  --  工 (<CJK Ideograph>)
      2265   => 16#5DE7#,  --  巧 (<CJK Ideograph>)
      2266   => 16#5DF7#,  --  巷 (<CJK Ideograph>)
      2267   => 16#5E78#,  --  幸 (<CJK Ideograph>)
      2268   => 16#5E83#,  --  広 (<CJK Ideograph>)
      2269   => 16#5E9A#,  --  庚 (<CJK Ideograph>)
      2270   => 16#5EB7#,  --  康 (<CJK Ideograph>)
      2271   => 16#5F18#,  --  弘 (<CJK Ideograph>)
      2272   => 16#6052#,  --  恒 (<CJK Ideograph>)
      2273   => 16#614C#,  --  慌 (<CJK Ideograph>)
      2274   => 16#6297#,  --  抗 (<CJK Ideograph>)
      2275   => 16#62D8#,  --  拘 (<CJK Ideograph>)
      2276   => 16#63A7#,  --  控 (<CJK Ideograph>)
      2277   => 16#653B#,  --  攻 (<CJK Ideograph>)
      2278   => 16#6602#,  --  昂 (<CJK Ideograph>)
      2279   => 16#6643#,  --  晃 (<CJK Ideograph>)
      2280   => 16#66F4#,  --  更 (<CJK Ideograph>)
      2281   => 16#676D#,  --  杭 (<CJK Ideograph>)
      2282   => 16#6821#,  --  校 (<CJK Ideograph>)
      2283   => 16#6897#,  --  梗 (<CJK Ideograph>)
      2284   => 16#69CB#,  --  構 (<CJK Ideograph>)
      2285   => 16#6C5F#,  --  江 (<CJK Ideograph>)
      2286   => 16#6D2A#,  --  洪 (<CJK Ideograph>)
      2287   => 16#6D69#,  --  浩 (<CJK Ideograph>)
      2288   => 16#6E2F#,  --  港 (<CJK Ideograph>)
      2289   => 16#6E9D#,  --  溝 (<CJK Ideograph>)
      2290   => 16#7532#,  --  甲 (<CJK Ideograph>)
      2291   => 16#7687#,  --  皇 (<CJK Ideograph>)
      2292   => 16#786C#,  --  硬 (<CJK Ideograph>)
      2293   => 16#7A3F#,  --  稿 (<CJK Ideograph>)
      2294   => 16#7CE0#,  --  糠 (<CJK Ideograph>)
      2295   => 16#7D05#,  --  紅 (<CJK Ideograph>)
      2296   => 16#7D18#,  --  紘 (<CJK Ideograph>)
      2297   => 16#7D5E#,  --  絞 (<CJK Ideograph>)
      2298   => 16#7DB1#,  --  綱 (<CJK Ideograph>)
      2299   => 16#8015#,  --  耕 (<CJK Ideograph>)
      2300   => 16#8003#,  --  考 (<CJK Ideograph>)
      2301   => 16#80AF#,  --  肯 (<CJK Ideograph>)
      2302   => 16#80B1#,  --  肱 (<CJK Ideograph>)
      2303   => 16#8154#,  --  腔 (<CJK Ideograph>)
      2304   => 16#818F#,  --  膏 (<CJK Ideograph>)
      2305   => 16#822A#,  --  航 (<CJK Ideograph>)
      2306   => 16#8352#,  --  荒 (<CJK Ideograph>)
      2307   => 16#884C#,  --  行 (<CJK Ideograph>)
      2308   => 16#8861#,  --  衡 (<CJK Ideograph>)
      2309   => 16#8B1B#,  --  講 (<CJK Ideograph>)
      2310   => 16#8CA2#,  --  貢 (<CJK Ideograph>)
      2311   => 16#8CFC#,  --  購 (<CJK Ideograph>)
      2312   => 16#90CA#,  --  郊 (<CJK Ideograph>)
      2313   => 16#9175#,  --  酵 (<CJK Ideograph>)
      2314   => 16#9271#,  --  鉱 (<CJK Ideograph>)
      2315   => 16#783F#,  --  砿 (<CJK Ideograph>)
      2316   => 16#92FC#,  --  鋼 (<CJK Ideograph>)
      2317   => 16#95A4#,  --  閤 (<CJK Ideograph>)
      2318   => 16#964D#,  --  降 (<CJK Ideograph>)
      2319   => 16#9805#,  --  項 (<CJK Ideograph>)
      2320   => 16#9999#,  --  香 (<CJK Ideograph>)
      2321   => 16#9AD8#,  --  高 (<CJK Ideograph>)
      2322   => 16#9D3B#,  --  鴻 (<CJK Ideograph>)
      2323   => 16#525B#,  --  剛 (<CJK Ideograph>)
      2324   => 16#52AB#,  --  劫 (<CJK Ideograph>)
      2325   => 16#53F7#,  --  号 (<CJK Ideograph>)
      2326   => 16#5408#,  --  合 (<CJK Ideograph>)
      2327   => 16#58D5#,  --  壕 (<CJK Ideograph>)
      2328   => 16#62F7#,  --  拷 (<CJK Ideograph>)
      2329   => 16#6FE0#,  --  濠 (<CJK Ideograph>)
      2330   => 16#8C6A#,  --  豪 (<CJK Ideograph>)
      2331   => 16#8F5F#,  --  轟 (<CJK Ideograph>)
      2332   => 16#9EB9#,  --  麹 (<CJK Ideograph>)
      2333   => 16#514B#,  --  克 (<CJK Ideograph>)
      2334   => 16#523B#,  --  刻 (<CJK Ideograph>)
      2335   => 16#544A#,  --  告 (<CJK Ideograph>)
      2336   => 16#56FD#,  --  国 (<CJK Ideograph>)
      2337   => 16#7A40#,  --  穀 (<CJK Ideograph>)
      2338   => 16#9177#,  --  酷 (<CJK Ideograph>)
      2339   => 16#9D60#,  --  鵠 (<CJK Ideograph>)
      2340   => 16#9ED2#,  --  黒 (<CJK Ideograph>)
      2341   => 16#7344#,  --  獄 (<CJK Ideograph>)
      2342   => 16#6F09#,  --  漉 (<CJK Ideograph>)
      2343   => 16#8170#,  --  腰 (<CJK Ideograph>)
      2344   => 16#7511#,  --  甑 (<CJK Ideograph>)
      2345   => 16#5FFD#,  --  忽 (<CJK Ideograph>)
      2346   => 16#60DA#,  --  惚 (<CJK Ideograph>)
      2347   => 16#9AA8#,  --  骨 (<CJK Ideograph>)
      2348   => 16#72DB#,  --  狛 (<CJK Ideograph>)
      2349   => 16#8FBC#,  --  込 (<CJK Ideograph>)
      2350   => 16#6B64#,  --  此 (<CJK Ideograph>)
      2351   => 16#9803#,  --  頃 (<CJK Ideograph>)
      2352   => 16#4ECA#,  --  今 (<CJK Ideograph>)
      2353   => 16#56F0#,  --  困 (<CJK Ideograph>)
      2354   => 16#5764#,  --  坤 (<CJK Ideograph>)
      2355   => 16#58BE#,  --  墾 (<CJK Ideograph>)
      2356   => 16#5A5A#,  --  婚 (<CJK Ideograph>)
      2357   => 16#6068#,  --  恨 (<CJK Ideograph>)
      2358   => 16#61C7#,  --  懇 (<CJK Ideograph>)
      2359   => 16#660F#,  --  昏 (<CJK Ideograph>)
      2360   => 16#6606#,  --  昆 (<CJK Ideograph>)
      2361   => 16#6839#,  --  根 (<CJK Ideograph>)
      2362   => 16#68B1#,  --  梱 (<CJK Ideograph>)
      2363   => 16#6DF7#,  --  混 (<CJK Ideograph>)
      2364   => 16#75D5#,  --  痕 (<CJK Ideograph>)
      2365   => 16#7D3A#,  --  紺 (<CJK Ideograph>)
      2366   => 16#826E#,  --  艮 (<CJK Ideograph>)
      2367   => 16#9B42#,  --  魂 (<CJK Ideograph>)
      2368   => 16#4E9B#,  --  些 (<CJK Ideograph>)
      2369   => 16#4F50#,  --  佐 (<CJK Ideograph>)
      2370   => 16#53C9#,  --  叉 (<CJK Ideograph>)
      2371   => 16#5506#,  --  唆 (<CJK Ideograph>)
      2372   => 16#5D6F#,  --  嵯 (<CJK Ideograph>)
      2373   => 16#5DE6#,  --  左 (<CJK Ideograph>)
      2374   => 16#5DEE#,  --  差 (<CJK Ideograph>)
      2375   => 16#67FB#,  --  査 (<CJK Ideograph>)
      2376   => 16#6C99#,  --  沙 (<CJK Ideograph>)
      2377   => 16#7473#,  --  瑳 (<CJK Ideograph>)
      2378   => 16#7802#,  --  砂 (<CJK Ideograph>)
      2379   => 16#8A50#,  --  詐 (<CJK Ideograph>)
      2380   => 16#9396#,  --  鎖 (<CJK Ideograph>)
      2381   => 16#88DF#,  --  裟 (<CJK Ideograph>)
      2382   => 16#5750#,  --  坐 (<CJK Ideograph>)
      2383   => 16#5EA7#,  --  座 (<CJK Ideograph>)
      2384   => 16#632B#,  --  挫 (<CJK Ideograph>)
      2385   => 16#50B5#,  --  債 (<CJK Ideograph>)
      2386   => 16#50AC#,  --  催 (<CJK Ideograph>)
      2387   => 16#518D#,  --  再 (<CJK Ideograph>)
      2388   => 16#6700#,  --  最 (<CJK Ideograph>)
      2389   => 16#54C9#,  --  哉 (<CJK Ideograph>)
      2390   => 16#585E#,  --  塞 (<CJK Ideograph>)
      2391   => 16#59BB#,  --  妻 (<CJK Ideograph>)
      2392   => 16#5BB0#,  --  宰 (<CJK Ideograph>)
      2393   => 16#5F69#,  --  彩 (<CJK Ideograph>)
      2394   => 16#624D#,  --  才 (<CJK Ideograph>)
      2395   => 16#63A1#,  --  採 (<CJK Ideograph>)
      2396   => 16#683D#,  --  栽 (<CJK Ideograph>)
      2397   => 16#6B73#,  --  歳 (<CJK Ideograph>)
      2398   => 16#6E08#,  --  済 (<CJK Ideograph>)
      2399   => 16#707D#,  --  災 (<CJK Ideograph>)
      2400   => 16#91C7#,  --  采 (<CJK Ideograph>)
      2401   => 16#7280#,  --  犀 (<CJK Ideograph>)
      2402   => 16#7815#,  --  砕 (<CJK Ideograph>)
      2403   => 16#7826#,  --  砦 (<CJK Ideograph>)
      2404   => 16#796D#,  --  祭 (<CJK Ideograph>)
      2405   => 16#658E#,  --  斎 (<CJK Ideograph>)
      2406   => 16#7D30#,  --  細 (<CJK Ideograph>)
      2407   => 16#83DC#,  --  菜 (<CJK Ideograph>)
      2408   => 16#88C1#,  --  裁 (<CJK Ideograph>)
      2409   => 16#8F09#,  --  載 (<CJK Ideograph>)
      2410   => 16#969B#,  --  際 (<CJK Ideograph>)
      2411   => 16#5264#,  --  剤 (<CJK Ideograph>)
      2412   => 16#5728#,  --  在 (<CJK Ideograph>)
      2413   => 16#6750#,  --  材 (<CJK Ideograph>)
      2414   => 16#7F6A#,  --  罪 (<CJK Ideograph>)
      2415   => 16#8CA1#,  --  財 (<CJK Ideograph>)
      2416   => 16#51B4#,  --  冴 (<CJK Ideograph>)
      2417   => 16#5742#,  --  坂 (<CJK Ideograph>)
      2418   => 16#962A#,  --  阪 (<CJK Ideograph>)
      2419   => 16#583A#,  --  堺 (<CJK Ideograph>)
      2420   => 16#698A#,  --  榊 (<CJK Ideograph>)
      2421   => 16#80B4#,  --  肴 (<CJK Ideograph>)
      2422   => 16#54B2#,  --  咲 (<CJK Ideograph>)
      2423   => 16#5D0E#,  --  崎 (<CJK Ideograph>)
      2424   => 16#57FC#,  --  埼 (<CJK Ideograph>)
      2425   => 16#7895#,  --  碕 (<CJK Ideograph>)
      2426   => 16#9DFA#,  --  鷺 (<CJK Ideograph>)
      2427   => 16#4F5C#,  --  作 (<CJK Ideograph>)
      2428   => 16#524A#,  --  削 (<CJK Ideograph>)
      2429   => 16#548B#,  --  咋 (<CJK Ideograph>)
      2430   => 16#643E#,  --  搾 (<CJK Ideograph>)
      2431   => 16#6628#,  --  昨 (<CJK Ideograph>)
      2432   => 16#6714#,  --  朔 (<CJK Ideograph>)
      2433   => 16#67F5#,  --  柵 (<CJK Ideograph>)
      2434   => 16#7A84#,  --  窄 (<CJK Ideograph>)
      2435   => 16#7B56#,  --  策 (<CJK Ideograph>)
      2436   => 16#7D22#,  --  索 (<CJK Ideograph>)
      2437   => 16#932F#,  --  錯 (<CJK Ideograph>)
      2438   => 16#685C#,  --  桜 (<CJK Ideograph>)
      2439   => 16#9BAD#,  --  鮭 (<CJK Ideograph>)
      2440   => 16#7B39#,  --  笹 (<CJK Ideograph>)
      2441   => 16#5319#,  --  匙 (<CJK Ideograph>)
      2442   => 16#518A#,  --  冊 (<CJK Ideograph>)
      2443   => 16#5237#,  --  刷 (<CJK Ideograph>)
      2444   => 16#5BDF#,  --  察 (<CJK Ideograph>)
      2445   => 16#62F6#,  --  拶 (<CJK Ideograph>)
      2446   => 16#64AE#,  --  撮 (<CJK Ideograph>)
      2447   => 16#64E6#,  --  擦 (<CJK Ideograph>)
      2448   => 16#672D#,  --  札 (<CJK Ideograph>)
      2449   => 16#6BBA#,  --  殺 (<CJK Ideograph>)
      2450   => 16#85A9#,  --  薩 (<CJK Ideograph>)
      2451   => 16#96D1#,  --  雑 (<CJK Ideograph>)
      2452   => 16#7690#,  --  皐 (<CJK Ideograph>)
      2453   => 16#9BD6#,  --  鯖 (<CJK Ideograph>)
      2454   => 16#634C#,  --  捌 (<CJK Ideograph>)
      2455   => 16#9306#,  --  錆 (<CJK Ideograph>)
      2456   => 16#9BAB#,  --  鮫 (<CJK Ideograph>)
      2457   => 16#76BF#,  --  皿 (<CJK Ideograph>)
      2458   => 16#6652#,  --  晒 (<CJK Ideograph>)
      2459   => 16#4E09#,  --  三 (<CJK Ideograph>)
      2460   => 16#5098#,  --  傘 (<CJK Ideograph>)
      2461   => 16#53C2#,  --  参 (<CJK Ideograph>)
      2462   => 16#5C71#,  --  山 (<CJK Ideograph>)
      2463   => 16#60E8#,  --  惨 (<CJK Ideograph>)
      2464   => 16#6492#,  --  撒 (<CJK Ideograph>)
      2465   => 16#6563#,  --  散 (<CJK Ideograph>)
      2466   => 16#685F#,  --  桟 (<CJK Ideograph>)
      2467   => 16#71E6#,  --  燦 (<CJK Ideograph>)
      2468   => 16#73CA#,  --  珊 (<CJK Ideograph>)
      2469   => 16#7523#,  --  産 (<CJK Ideograph>)
      2470   => 16#7B97#,  --  算 (<CJK Ideograph>)
      2471   => 16#7E82#,  --  纂 (<CJK Ideograph>)
      2472   => 16#8695#,  --  蚕 (<CJK Ideograph>)
      2473   => 16#8B83#,  --  讃 (<CJK Ideograph>)
      2474   => 16#8CDB#,  --  賛 (<CJK Ideograph>)
      2475   => 16#9178#,  --  酸 (<CJK Ideograph>)
      2476   => 16#9910#,  --  餐 (<CJK Ideograph>)
      2477   => 16#65AC#,  --  斬 (<CJK Ideograph>)
      2478   => 16#66AB#,  --  暫 (<CJK Ideograph>)
      2479   => 16#6B8B#,  --  残 (<CJK Ideograph>)
      2480   => 16#4ED5#,  --  仕 (<CJK Ideograph>)
      2481   => 16#4ED4#,  --  仔 (<CJK Ideograph>)
      2482   => 16#4F3A#,  --  伺 (<CJK Ideograph>)
      2483   => 16#4F7F#,  --  使 (<CJK Ideograph>)
      2484   => 16#523A#,  --  刺 (<CJK Ideograph>)
      2485   => 16#53F8#,  --  司 (<CJK Ideograph>)
      2486   => 16#53F2#,  --  史 (<CJK Ideograph>)
      2487   => 16#55E3#,  --  嗣 (<CJK Ideograph>)
      2488   => 16#56DB#,  --  四 (<CJK Ideograph>)
      2489   => 16#58EB#,  --  士 (<CJK Ideograph>)
      2490   => 16#59CB#,  --  始 (<CJK Ideograph>)
      2491   => 16#59C9#,  --  姉 (<CJK Ideograph>)
      2492   => 16#59FF#,  --  姿 (<CJK Ideograph>)
      2493   => 16#5B50#,  --  子 (<CJK Ideograph>)
      2494   => 16#5C4D#,  --  屍 (<CJK Ideograph>)
      2495   => 16#5E02#,  --  市 (<CJK Ideograph>)
      2496   => 16#5E2B#,  --  師 (<CJK Ideograph>)
      2497   => 16#5FD7#,  --  志 (<CJK Ideograph>)
      2498   => 16#601D#,  --  思 (<CJK Ideograph>)
      2499   => 16#6307#,  --  指 (<CJK Ideograph>)
      2500   => 16#652F#,  --  支 (<CJK Ideograph>)
      2501   => 16#5B5C#,  --  孜 (<CJK Ideograph>)
      2502   => 16#65AF#,  --  斯 (<CJK Ideograph>)
      2503   => 16#65BD#,  --  施 (<CJK Ideograph>)
      2504   => 16#65E8#,  --  旨 (<CJK Ideograph>)
      2505   => 16#679D#,  --  枝 (<CJK Ideograph>)
      2506   => 16#6B62#,  --  止 (<CJK Ideograph>)
      2507   => 16#6B7B#,  --  死 (<CJK Ideograph>)
      2508   => 16#6C0F#,  --  氏 (<CJK Ideograph>)
      2509   => 16#7345#,  --  獅 (<CJK Ideograph>)
      2510   => 16#7949#,  --  祉 (<CJK Ideograph>)
      2511   => 16#79C1#,  --  私 (<CJK Ideograph>)
      2512   => 16#7CF8#,  --  糸 (<CJK Ideograph>)
      2513   => 16#7D19#,  --  紙 (<CJK Ideograph>)
      2514   => 16#7D2B#,  --  紫 (<CJK Ideograph>)
      2515   => 16#80A2#,  --  肢 (<CJK Ideograph>)
      2516   => 16#8102#,  --  脂 (<CJK Ideograph>)
      2517   => 16#81F3#,  --  至 (<CJK Ideograph>)
      2518   => 16#8996#,  --  視 (<CJK Ideograph>)
      2519   => 16#8A5E#,  --  詞 (<CJK Ideograph>)
      2520   => 16#8A69#,  --  詩 (<CJK Ideograph>)
      2521   => 16#8A66#,  --  試 (<CJK Ideograph>)
      2522   => 16#8A8C#,  --  誌 (<CJK Ideograph>)
      2523   => 16#8AEE#,  --  諮 (<CJK Ideograph>)
      2524   => 16#8CC7#,  --  資 (<CJK Ideograph>)
      2525   => 16#8CDC#,  --  賜 (<CJK Ideograph>)
      2526   => 16#96CC#,  --  雌 (<CJK Ideograph>)
      2527   => 16#98FC#,  --  飼 (<CJK Ideograph>)
      2528   => 16#6B6F#,  --  歯 (<CJK Ideograph>)
      2529   => 16#4E8B#,  --  事 (<CJK Ideograph>)
      2530   => 16#4F3C#,  --  似 (<CJK Ideograph>)
      2531   => 16#4F8D#,  --  侍 (<CJK Ideograph>)
      2532   => 16#5150#,  --  児 (<CJK Ideograph>)
      2533   => 16#5B57#,  --  字 (<CJK Ideograph>)
      2534   => 16#5BFA#,  --  寺 (<CJK Ideograph>)
      2535   => 16#6148#,  --  慈 (<CJK Ideograph>)
      2536   => 16#6301#,  --  持 (<CJK Ideograph>)
      2537   => 16#6642#,  --  時 (<CJK Ideograph>)
      2538   => 16#6B21#,  --  次 (<CJK Ideograph>)
      2539   => 16#6ECB#,  --  滋 (<CJK Ideograph>)
      2540   => 16#6CBB#,  --  治 (<CJK Ideograph>)
      2541   => 16#723E#,  --  爾 (<CJK Ideograph>)
      2542   => 16#74BD#,  --  璽 (<CJK Ideograph>)
      2543   => 16#75D4#,  --  痔 (<CJK Ideograph>)
      2544   => 16#78C1#,  --  磁 (<CJK Ideograph>)
      2545   => 16#793A#,  --  示 (<CJK Ideograph>)
      2546   => 16#800C#,  --  而 (<CJK Ideograph>)
      2547   => 16#8033#,  --  耳 (<CJK Ideograph>)
      2548   => 16#81EA#,  --  自 (<CJK Ideograph>)
      2549   => 16#8494#,  --  蒔 (<CJK Ideograph>)
      2550   => 16#8F9E#,  --  辞 (<CJK Ideograph>)
      2551   => 16#6C50#,  --  汐 (<CJK Ideograph>)
      2552   => 16#9E7F#,  --  鹿 (<CJK Ideograph>)
      2553   => 16#5F0F#,  --  式 (<CJK Ideograph>)
      2554   => 16#8B58#,  --  識 (<CJK Ideograph>)
      2555   => 16#9D2B#,  --  鴫 (<CJK Ideograph>)
      2556   => 16#7AFA#,  --  竺 (<CJK Ideograph>)
      2557   => 16#8EF8#,  --  軸 (<CJK Ideograph>)
      2558   => 16#5B8D#,  --  宍 (<CJK Ideograph>)
      2559   => 16#96EB#,  --  雫 (<CJK Ideograph>)
      2560   => 16#4E03#,  --  七 (<CJK Ideograph>)
      2561   => 16#53F1#,  --  叱 (<CJK Ideograph>)
      2562   => 16#57F7#,  --  執 (<CJK Ideograph>)
      2563   => 16#5931#,  --  失 (<CJK Ideograph>)
      2564   => 16#5AC9#,  --  嫉 (<CJK Ideograph>)
      2565   => 16#5BA4#,  --  室 (<CJK Ideograph>)
      2566   => 16#6089#,  --  悉 (<CJK Ideograph>)
      2567   => 16#6E7F#,  --  湿 (<CJK Ideograph>)
      2568   => 16#6F06#,  --  漆 (<CJK Ideograph>)
      2569   => 16#75BE#,  --  疾 (<CJK Ideograph>)
      2570   => 16#8CEA#,  --  質 (<CJK Ideograph>)
      2571   => 16#5B9F#,  --  実 (<CJK Ideograph>)
      2572   => 16#8500#,  --  蔀 (<CJK Ideograph>)
      2573   => 16#7BE0#,  --  篠 (<CJK Ideograph>)
      2574   => 16#5072#,  --  偲 (<CJK Ideograph>)
      2575   => 16#67F4#,  --  柴 (<CJK Ideograph>)
      2576   => 16#829D#,  --  芝 (<CJK Ideograph>)
      2577   => 16#5C61#,  --  屡 (<CJK Ideograph>)
      2578   => 16#854A#,  --  蕊 (<CJK Ideograph>)
      2579   => 16#7E1E#,  --  縞 (<CJK Ideograph>)
      2580   => 16#820E#,  --  舎 (<CJK Ideograph>)
      2581   => 16#5199#,  --  写 (<CJK Ideograph>)
      2582   => 16#5C04#,  --  射 (<CJK Ideograph>)
      2583   => 16#6368#,  --  捨 (<CJK Ideograph>)
      2584   => 16#8D66#,  --  赦 (<CJK Ideograph>)
      2585   => 16#659C#,  --  斜 (<CJK Ideograph>)
      2586   => 16#716E#,  --  煮 (<CJK Ideograph>)
      2587   => 16#793E#,  --  社 (<CJK Ideograph>)
      2588   => 16#7D17#,  --  紗 (<CJK Ideograph>)
      2589   => 16#8005#,  --  者 (<CJK Ideograph>)
      2590   => 16#8B1D#,  --  謝 (<CJK Ideograph>)
      2591   => 16#8ECA#,  --  車 (<CJK Ideograph>)
      2592   => 16#906E#,  --  遮 (<CJK Ideograph>)
      2593   => 16#86C7#,  --  蛇 (<CJK Ideograph>)
      2594   => 16#90AA#,  --  邪 (<CJK Ideograph>)
      2595   => 16#501F#,  --  借 (<CJK Ideograph>)
      2596   => 16#52FA#,  --  勺 (<CJK Ideograph>)
      2597   => 16#5C3A#,  --  尺 (<CJK Ideograph>)
      2598   => 16#6753#,  --  杓 (<CJK Ideograph>)
      2599   => 16#707C#,  --  灼 (<CJK Ideograph>)
      2600   => 16#7235#,  --  爵 (<CJK Ideograph>)
      2601   => 16#914C#,  --  酌 (<CJK Ideograph>)
      2602   => 16#91C8#,  --  釈 (<CJK Ideograph>)
      2603   => 16#932B#,  --  錫 (<CJK Ideograph>)
      2604   => 16#82E5#,  --  若 (<CJK Ideograph>)
      2605   => 16#5BC2#,  --  寂 (<CJK Ideograph>)
      2606   => 16#5F31#,  --  弱 (<CJK Ideograph>)
      2607   => 16#60F9#,  --  惹 (<CJK Ideograph>)
      2608   => 16#4E3B#,  --  主 (<CJK Ideograph>)
      2609   => 16#53D6#,  --  取 (<CJK Ideograph>)
      2610   => 16#5B88#,  --  守 (<CJK Ideograph>)
      2611   => 16#624B#,  --  手 (<CJK Ideograph>)
      2612   => 16#6731#,  --  朱 (<CJK Ideograph>)
      2613   => 16#6B8A#,  --  殊 (<CJK Ideograph>)
      2614   => 16#72E9#,  --  狩 (<CJK Ideograph>)
      2615   => 16#73E0#,  --  珠 (<CJK Ideograph>)
      2616   => 16#7A2E#,  --  種 (<CJK Ideograph>)
      2617   => 16#816B#,  --  腫 (<CJK Ideograph>)
      2618   => 16#8DA3#,  --  趣 (<CJK Ideograph>)
      2619   => 16#9152#,  --  酒 (<CJK Ideograph>)
      2620   => 16#9996#,  --  首 (<CJK Ideograph>)
      2621   => 16#5112#,  --  儒 (<CJK Ideograph>)
      2622   => 16#53D7#,  --  受 (<CJK Ideograph>)
      2623   => 16#546A#,  --  呪 (<CJK Ideograph>)
      2624   => 16#5BFF#,  --  寿 (<CJK Ideograph>)
      2625   => 16#6388#,  --  授 (<CJK Ideograph>)
      2626   => 16#6A39#,  --  樹 (<CJK Ideograph>)
      2627   => 16#7DAC#,  --  綬 (<CJK Ideograph>)
      2628   => 16#9700#,  --  需 (<CJK Ideograph>)
      2629   => 16#56DA#,  --  囚 (<CJK Ideograph>)
      2630   => 16#53CE#,  --  収 (<CJK Ideograph>)
      2631   => 16#5468#,  --  周 (<CJK Ideograph>)
      2632   => 16#5B97#,  --  宗 (<CJK Ideograph>)
      2633   => 16#5C31#,  --  就 (<CJK Ideograph>)
      2634   => 16#5DDE#,  --  州 (<CJK Ideograph>)
      2635   => 16#4FEE#,  --  修 (<CJK Ideograph>)
      2636   => 16#6101#,  --  愁 (<CJK Ideograph>)
      2637   => 16#62FE#,  --  拾 (<CJK Ideograph>)
      2638   => 16#6D32#,  --  洲 (<CJK Ideograph>)
      2639   => 16#79C0#,  --  秀 (<CJK Ideograph>)
      2640   => 16#79CB#,  --  秋 (<CJK Ideograph>)
      2641   => 16#7D42#,  --  終 (<CJK Ideograph>)
      2642   => 16#7E4D#,  --  繍 (<CJK Ideograph>)
      2643   => 16#7FD2#,  --  習 (<CJK Ideograph>)
      2644   => 16#81ED#,  --  臭 (<CJK Ideograph>)
      2645   => 16#821F#,  --  舟 (<CJK Ideograph>)
      2646   => 16#8490#,  --  蒐 (<CJK Ideograph>)
      2647   => 16#8846#,  --  衆 (<CJK Ideograph>)
      2648   => 16#8972#,  --  襲 (<CJK Ideograph>)
      2649   => 16#8B90#,  --  讐 (<CJK Ideograph>)
      2650   => 16#8E74#,  --  蹴 (<CJK Ideograph>)
      2651   => 16#8F2F#,  --  輯 (<CJK Ideograph>)
      2652   => 16#9031#,  --  週 (<CJK Ideograph>)
      2653   => 16#914B#,  --  酋 (<CJK Ideograph>)
      2654   => 16#916C#,  --  酬 (<CJK Ideograph>)
      2655   => 16#96C6#,  --  集 (<CJK Ideograph>)
      2656   => 16#919C#,  --  醜 (<CJK Ideograph>)
      2657   => 16#4EC0#,  --  什 (<CJK Ideograph>)
      2658   => 16#4F4F#,  --  住 (<CJK Ideograph>)
      2659   => 16#5145#,  --  充 (<CJK Ideograph>)
      2660   => 16#5341#,  --  十 (<CJK Ideograph>)
      2661   => 16#5F93#,  --  従 (<CJK Ideograph>)
      2662   => 16#620E#,  --  戎 (<CJK Ideograph>)
      2663   => 16#67D4#,  --  柔 (<CJK Ideograph>)
      2664   => 16#6C41#,  --  汁 (<CJK Ideograph>)
      2665   => 16#6E0B#,  --  渋 (<CJK Ideograph>)
      2666   => 16#7363#,  --  獣 (<CJK Ideograph>)
      2667   => 16#7E26#,  --  縦 (<CJK Ideograph>)
      2668   => 16#91CD#,  --  重 (<CJK Ideograph>)
      2669   => 16#9283#,  --  銃 (<CJK Ideograph>)
      2670   => 16#53D4#,  --  叔 (<CJK Ideograph>)
      2671   => 16#5919#,  --  夙 (<CJK Ideograph>)
      2672   => 16#5BBF#,  --  宿 (<CJK Ideograph>)
      2673   => 16#6DD1#,  --  淑 (<CJK Ideograph>)
      2674   => 16#795D#,  --  祝 (<CJK Ideograph>)
      2675   => 16#7E2E#,  --  縮 (<CJK Ideograph>)
      2676   => 16#7C9B#,  --  粛 (<CJK Ideograph>)
      2677   => 16#587E#,  --  塾 (<CJK Ideograph>)
      2678   => 16#719F#,  --  熟 (<CJK Ideograph>)
      2679   => 16#51FA#,  --  出 (<CJK Ideograph>)
      2680   => 16#8853#,  --  術 (<CJK Ideograph>)
      2681   => 16#8FF0#,  --  述 (<CJK Ideograph>)
      2682   => 16#4FCA#,  --  俊 (<CJK Ideograph>)
      2683   => 16#5CFB#,  --  峻 (<CJK Ideograph>)
      2684   => 16#6625#,  --  春 (<CJK Ideograph>)
      2685   => 16#77AC#,  --  瞬 (<CJK Ideograph>)
      2686   => 16#7AE3#,  --  竣 (<CJK Ideograph>)
      2687   => 16#821C#,  --  舜 (<CJK Ideograph>)
      2688   => 16#99FF#,  --  駿 (<CJK Ideograph>)
      2689   => 16#51C6#,  --  准 (<CJK Ideograph>)
      2690   => 16#5FAA#,  --  循 (<CJK Ideograph>)
      2691   => 16#65EC#,  --  旬 (<CJK Ideograph>)
      2692   => 16#696F#,  --  楯 (<CJK Ideograph>)
      2693   => 16#6B89#,  --  殉 (<CJK Ideograph>)
      2694   => 16#6DF3#,  --  淳 (<CJK Ideograph>)
      2695   => 16#6E96#,  --  準 (<CJK Ideograph>)
      2696   => 16#6F64#,  --  潤 (<CJK Ideograph>)
      2697   => 16#76FE#,  --  盾 (<CJK Ideograph>)
      2698   => 16#7D14#,  --  純 (<CJK Ideograph>)
      2699   => 16#5DE1#,  --  巡 (<CJK Ideograph>)
      2700   => 16#9075#,  --  遵 (<CJK Ideograph>)
      2701   => 16#9187#,  --  醇 (<CJK Ideograph>)
      2702   => 16#9806#,  --  順 (<CJK Ideograph>)
      2703   => 16#51E6#,  --  処 (<CJK Ideograph>)
      2704   => 16#521D#,  --  初 (<CJK Ideograph>)
      2705   => 16#6240#,  --  所 (<CJK Ideograph>)
      2706   => 16#6691#,  --  暑 (<CJK Ideograph>)
      2707   => 16#66D9#,  --  曙 (<CJK Ideograph>)
      2708   => 16#6E1A#,  --  渚 (<CJK Ideograph>)
      2709   => 16#5EB6#,  --  庶 (<CJK Ideograph>)
      2710   => 16#7DD2#,  --  緒 (<CJK Ideograph>)
      2711   => 16#7F72#,  --  署 (<CJK Ideograph>)
      2712   => 16#66F8#,  --  書 (<CJK Ideograph>)
      2713   => 16#85AF#,  --  薯 (<CJK Ideograph>)
      2714   => 16#85F7#,  --  藷 (<CJK Ideograph>)
      2715   => 16#8AF8#,  --  諸 (<CJK Ideograph>)
      2716   => 16#52A9#,  --  助 (<CJK Ideograph>)
      2717   => 16#53D9#,  --  叙 (<CJK Ideograph>)
      2718   => 16#5973#,  --  女 (<CJK Ideograph>)
      2719   => 16#5E8F#,  --  序 (<CJK Ideograph>)
      2720   => 16#5F90#,  --  徐 (<CJK Ideograph>)
      2721   => 16#6055#,  --  恕 (<CJK Ideograph>)
      2722   => 16#92E4#,  --  鋤 (<CJK Ideograph>)
      2723   => 16#9664#,  --  除 (<CJK Ideograph>)
      2724   => 16#50B7#,  --  傷 (<CJK Ideograph>)
      2725   => 16#511F#,  --  償 (<CJK Ideograph>)
      2726   => 16#52DD#,  --  勝 (<CJK Ideograph>)
      2727   => 16#5320#,  --  匠 (<CJK Ideograph>)
      2728   => 16#5347#,  --  升 (<CJK Ideograph>)
      2729   => 16#53EC#,  --  召 (<CJK Ideograph>)
      2730   => 16#54E8#,  --  哨 (<CJK Ideograph>)
      2731   => 16#5546#,  --  商 (<CJK Ideograph>)
      2732   => 16#5531#,  --  唱 (<CJK Ideograph>)
      2733   => 16#5617#,  --  嘗 (<CJK Ideograph>)
      2734   => 16#5968#,  --  奨 (<CJK Ideograph>)
      2735   => 16#59BE#,  --  妾 (<CJK Ideograph>)
      2736   => 16#5A3C#,  --  娼 (<CJK Ideograph>)
      2737   => 16#5BB5#,  --  宵 (<CJK Ideograph>)
      2738   => 16#5C06#,  --  将 (<CJK Ideograph>)
      2739   => 16#5C0F#,  --  小 (<CJK Ideograph>)
      2740   => 16#5C11#,  --  少 (<CJK Ideograph>)
      2741   => 16#5C1A#,  --  尚 (<CJK Ideograph>)
      2742   => 16#5E84#,  --  庄 (<CJK Ideograph>)
      2743   => 16#5E8A#,  --  床 (<CJK Ideograph>)
      2744   => 16#5EE0#,  --  廠 (<CJK Ideograph>)
      2745   => 16#5F70#,  --  彰 (<CJK Ideograph>)
      2746   => 16#627F#,  --  承 (<CJK Ideograph>)
      2747   => 16#6284#,  --  抄 (<CJK Ideograph>)
      2748   => 16#62DB#,  --  招 (<CJK Ideograph>)
      2749   => 16#638C#,  --  掌 (<CJK Ideograph>)
      2750   => 16#6377#,  --  捷 (<CJK Ideograph>)
      2751   => 16#6607#,  --  昇 (<CJK Ideograph>)
      2752   => 16#660C#,  --  昌 (<CJK Ideograph>)
      2753   => 16#662D#,  --  昭 (<CJK Ideograph>)
      2754   => 16#6676#,  --  晶 (<CJK Ideograph>)
      2755   => 16#677E#,  --  松 (<CJK Ideograph>)
      2756   => 16#68A2#,  --  梢 (<CJK Ideograph>)
      2757   => 16#6A1F#,  --  樟 (<CJK Ideograph>)
      2758   => 16#6A35#,  --  樵 (<CJK Ideograph>)
      2759   => 16#6CBC#,  --  沼 (<CJK Ideograph>)
      2760   => 16#6D88#,  --  消 (<CJK Ideograph>)
      2761   => 16#6E09#,  --  渉 (<CJK Ideograph>)
      2762   => 16#6E58#,  --  湘 (<CJK Ideograph>)
      2763   => 16#713C#,  --  焼 (<CJK Ideograph>)
      2764   => 16#7126#,  --  焦 (<CJK Ideograph>)
      2765   => 16#7167#,  --  照 (<CJK Ideograph>)
      2766   => 16#75C7#,  --  症 (<CJK Ideograph>)
      2767   => 16#7701#,  --  省 (<CJK Ideograph>)
      2768   => 16#785D#,  --  硝 (<CJK Ideograph>)
      2769   => 16#7901#,  --  礁 (<CJK Ideograph>)
      2770   => 16#7965#,  --  祥 (<CJK Ideograph>)
      2771   => 16#79F0#,  --  称 (<CJK Ideograph>)
      2772   => 16#7AE0#,  --  章 (<CJK Ideograph>)
      2773   => 16#7B11#,  --  笑 (<CJK Ideograph>)
      2774   => 16#7CA7#,  --  粧 (<CJK Ideograph>)
      2775   => 16#7D39#,  --  紹 (<CJK Ideograph>)
      2776   => 16#8096#,  --  肖 (<CJK Ideograph>)
      2777   => 16#83D6#,  --  菖 (<CJK Ideograph>)
      2778   => 16#848B#,  --  蒋 (<CJK Ideograph>)
      2779   => 16#8549#,  --  蕉 (<CJK Ideograph>)
      2780   => 16#885D#,  --  衝 (<CJK Ideograph>)
      2781   => 16#88F3#,  --  裳 (<CJK Ideograph>)
      2782   => 16#8A1F#,  --  訟 (<CJK Ideograph>)
      2783   => 16#8A3C#,  --  証 (<CJK Ideograph>)
      2784   => 16#8A54#,  --  詔 (<CJK Ideograph>)
      2785   => 16#8A73#,  --  詳 (<CJK Ideograph>)
      2786   => 16#8C61#,  --  象 (<CJK Ideograph>)
      2787   => 16#8CDE#,  --  賞 (<CJK Ideograph>)
      2788   => 16#91A4#,  --  醤 (<CJK Ideograph>)
      2789   => 16#9266#,  --  鉦 (<CJK Ideograph>)
      2790   => 16#937E#,  --  鍾 (<CJK Ideograph>)
      2791   => 16#9418#,  --  鐘 (<CJK Ideograph>)
      2792   => 16#969C#,  --  障 (<CJK Ideograph>)
      2793   => 16#9798#,  --  鞘 (<CJK Ideograph>)
      2794   => 16#4E0A#,  --  上 (<CJK Ideograph>)
      2795   => 16#4E08#,  --  丈 (<CJK Ideograph>)
      2796   => 16#4E1E#,  --  丞 (<CJK Ideograph>)
      2797   => 16#4E57#,  --  乗 (<CJK Ideograph>)
      2798   => 16#5197#,  --  冗 (<CJK Ideograph>)
      2799   => 16#5270#,  --  剰 (<CJK Ideograph>)
      2800   => 16#57CE#,  --  城 (<CJK Ideograph>)
      2801   => 16#5834#,  --  場 (<CJK Ideograph>)
      2802   => 16#58CC#,  --  壌 (<CJK Ideograph>)
      2803   => 16#5B22#,  --  嬢 (<CJK Ideograph>)
      2804   => 16#5E38#,  --  常 (<CJK Ideograph>)
      2805   => 16#60C5#,  --  情 (<CJK Ideograph>)
      2806   => 16#64FE#,  --  擾 (<CJK Ideograph>)
      2807   => 16#6761#,  --  条 (<CJK Ideograph>)
      2808   => 16#6756#,  --  杖 (<CJK Ideograph>)
      2809   => 16#6D44#,  --  浄 (<CJK Ideograph>)
      2810   => 16#72B6#,  --  状 (<CJK Ideograph>)
      2811   => 16#7573#,  --  畳 (<CJK Ideograph>)
      2812   => 16#7A63#,  --  穣 (<CJK Ideograph>)
      2813   => 16#84B8#,  --  蒸 (<CJK Ideograph>)
      2814   => 16#8B72#,  --  譲 (<CJK Ideograph>)
      2815   => 16#91B8#,  --  醸 (<CJK Ideograph>)
      2816   => 16#9320#,  --  錠 (<CJK Ideograph>)
      2817   => 16#5631#,  --  嘱 (<CJK Ideograph>)
      2818   => 16#57F4#,  --  埴 (<CJK Ideograph>)
      2819   => 16#98FE#,  --  飾 (<CJK Ideograph>)
      2820   => 16#62ED#,  --  拭 (<CJK Ideograph>)
      2821   => 16#690D#,  --  植 (<CJK Ideograph>)
      2822   => 16#6B96#,  --  殖 (<CJK Ideograph>)
      2823   => 16#71ED#,  --  燭 (<CJK Ideograph>)
      2824   => 16#7E54#,  --  織 (<CJK Ideograph>)
      2825   => 16#8077#,  --  職 (<CJK Ideograph>)
      2826   => 16#8272#,  --  色 (<CJK Ideograph>)
      2827   => 16#89E6#,  --  触 (<CJK Ideograph>)
      2828   => 16#98DF#,  --  食 (<CJK Ideograph>)
      2829   => 16#8755#,  --  蝕 (<CJK Ideograph>)
      2830   => 16#8FB1#,  --  辱 (<CJK Ideograph>)
      2831   => 16#5C3B#,  --  尻 (<CJK Ideograph>)
      2832   => 16#4F38#,  --  伸 (<CJK Ideograph>)
      2833   => 16#4FE1#,  --  信 (<CJK Ideograph>)
      2834   => 16#4FB5#,  --  侵 (<CJK Ideograph>)
      2835   => 16#5507#,  --  唇 (<CJK Ideograph>)
      2836   => 16#5A20#,  --  娠 (<CJK Ideograph>)
      2837   => 16#5BDD#,  --  寝 (<CJK Ideograph>)
      2838   => 16#5BE9#,  --  審 (<CJK Ideograph>)
      2839   => 16#5FC3#,  --  心 (<CJK Ideograph>)
      2840   => 16#614E#,  --  慎 (<CJK Ideograph>)
      2841   => 16#632F#,  --  振 (<CJK Ideograph>)
      2842   => 16#65B0#,  --  新 (<CJK Ideograph>)
      2843   => 16#664B#,  --  晋 (<CJK Ideograph>)
      2844   => 16#68EE#,  --  森 (<CJK Ideograph>)
      2845   => 16#699B#,  --  榛 (<CJK Ideograph>)
      2846   => 16#6D78#,  --  浸 (<CJK Ideograph>)
      2847   => 16#6DF1#,  --  深 (<CJK Ideograph>)
      2848   => 16#7533#,  --  申 (<CJK Ideograph>)
      2849   => 16#75B9#,  --  疹 (<CJK Ideograph>)
      2850   => 16#771F#,  --  真 (<CJK Ideograph>)
      2851   => 16#795E#,  --  神 (<CJK Ideograph>)
      2852   => 16#79E6#,  --  秦 (<CJK Ideograph>)
      2853   => 16#7D33#,  --  紳 (<CJK Ideograph>)
      2854   => 16#81E3#,  --  臣 (<CJK Ideograph>)
      2855   => 16#82AF#,  --  芯 (<CJK Ideograph>)
      2856   => 16#85AA#,  --  薪 (<CJK Ideograph>)
      2857   => 16#89AA#,  --  親 (<CJK Ideograph>)
      2858   => 16#8A3A#,  --  診 (<CJK Ideograph>)
      2859   => 16#8EAB#,  --  身 (<CJK Ideograph>)
      2860   => 16#8F9B#,  --  辛 (<CJK Ideograph>)
      2861   => 16#9032#,  --  進 (<CJK Ideograph>)
      2862   => 16#91DD#,  --  針 (<CJK Ideograph>)
      2863   => 16#9707#,  --  震 (<CJK Ideograph>)
      2864   => 16#4EBA#,  --  人 (<CJK Ideograph>)
      2865   => 16#4EC1#,  --  仁 (<CJK Ideograph>)
      2866   => 16#5203#,  --  刃 (<CJK Ideograph>)
      2867   => 16#5875#,  --  塵 (<CJK Ideograph>)
      2868   => 16#58EC#,  --  壬 (<CJK Ideograph>)
      2869   => 16#5C0B#,  --  尋 (<CJK Ideograph>)
      2870   => 16#751A#,  --  甚 (<CJK Ideograph>)
      2871   => 16#5C3D#,  --  尽 (<CJK Ideograph>)
      2872   => 16#814E#,  --  腎 (<CJK Ideograph>)
      2873   => 16#8A0A#,  --  訊 (<CJK Ideograph>)
      2874   => 16#8FC5#,  --  迅 (<CJK Ideograph>)
      2875   => 16#9663#,  --  陣 (<CJK Ideograph>)
      2876   => 16#976D#,  --  靭 (<CJK Ideograph>)
      2877   => 16#7B25#,  --  笥 (<CJK Ideograph>)
      2878   => 16#8ACF#,  --  諏 (<CJK Ideograph>)
      2879   => 16#9808#,  --  須 (<CJK Ideograph>)
      2880   => 16#9162#,  --  酢 (<CJK Ideograph>)
      2881   => 16#56F3#,  --  図 (<CJK Ideograph>)
      2882   => 16#53A8#,  --  厨 (<CJK Ideograph>)
      2883   => 16#9017#,  --  逗 (<CJK Ideograph>)
      2884   => 16#5439#,  --  吹 (<CJK Ideograph>)
      2885   => 16#5782#,  --  垂 (<CJK Ideograph>)
      2886   => 16#5E25#,  --  帥 (<CJK Ideograph>)
      2887   => 16#63A8#,  --  推 (<CJK Ideograph>)
      2888   => 16#6C34#,  --  水 (<CJK Ideograph>)
      2889   => 16#708A#,  --  炊 (<CJK Ideograph>)
      2890   => 16#7761#,  --  睡 (<CJK Ideograph>)
      2891   => 16#7C8B#,  --  粋 (<CJK Ideograph>)
      2892   => 16#7FE0#,  --  翠 (<CJK Ideograph>)
      2893   => 16#8870#,  --  衰 (<CJK Ideograph>)
      2894   => 16#9042#,  --  遂 (<CJK Ideograph>)
      2895   => 16#9154#,  --  酔 (<CJK Ideograph>)
      2896   => 16#9310#,  --  錐 (<CJK Ideograph>)
      2897   => 16#9318#,  --  錘 (<CJK Ideograph>)
      2898   => 16#968F#,  --  随 (<CJK Ideograph>)
      2899   => 16#745E#,  --  瑞 (<CJK Ideograph>)
      2900   => 16#9AC4#,  --  髄 (<CJK Ideograph>)
      2901   => 16#5D07#,  --  崇 (<CJK Ideograph>)
      2902   => 16#5D69#,  --  嵩 (<CJK Ideograph>)
      2903   => 16#6570#,  --  数 (<CJK Ideograph>)
      2904   => 16#67A2#,  --  枢 (<CJK Ideograph>)
      2905   => 16#8DA8#,  --  趨 (<CJK Ideograph>)
      2906   => 16#96DB#,  --  雛 (<CJK Ideograph>)
      2907   => 16#636E#,  --  据 (<CJK Ideograph>)
      2908   => 16#6749#,  --  杉 (<CJK Ideograph>)
      2909   => 16#6919#,  --  椙 (<CJK Ideograph>)
      2910   => 16#83C5#,  --  菅 (<CJK Ideograph>)
      2911   => 16#9817#,  --  頗 (<CJK Ideograph>)
      2912   => 16#96C0#,  --  雀 (<CJK Ideograph>)
      2913   => 16#88FE#,  --  裾 (<CJK Ideograph>)
      2914   => 16#6F84#,  --  澄 (<CJK Ideograph>)
      2915   => 16#647A#,  --  摺 (<CJK Ideograph>)
      2916   => 16#5BF8#,  --  寸 (<CJK Ideograph>)
      2917   => 16#4E16#,  --  世 (<CJK Ideograph>)
      2918   => 16#702C#,  --  瀬 (<CJK Ideograph>)
      2919   => 16#755D#,  --  畝 (<CJK Ideograph>)
      2920   => 16#662F#,  --  是 (<CJK Ideograph>)
      2921   => 16#51C4#,  --  凄 (<CJK Ideograph>)
      2922   => 16#5236#,  --  制 (<CJK Ideograph>)
      2923   => 16#52E2#,  --  勢 (<CJK Ideograph>)
      2924   => 16#59D3#,  --  姓 (<CJK Ideograph>)
      2925   => 16#5F81#,  --  征 (<CJK Ideograph>)
      2926   => 16#6027#,  --  性 (<CJK Ideograph>)
      2927   => 16#6210#,  --  成 (<CJK Ideograph>)
      2928   => 16#653F#,  --  政 (<CJK Ideograph>)
      2929   => 16#6574#,  --  整 (<CJK Ideograph>)
      2930   => 16#661F#,  --  星 (<CJK Ideograph>)
      2931   => 16#6674#,  --  晴 (<CJK Ideograph>)
      2932   => 16#68F2#,  --  棲 (<CJK Ideograph>)
      2933   => 16#6816#,  --  栖 (<CJK Ideograph>)
      2934   => 16#6B63#,  --  正 (<CJK Ideograph>)
      2935   => 16#6E05#,  --  清 (<CJK Ideograph>)
      2936   => 16#7272#,  --  牲 (<CJK Ideograph>)
      2937   => 16#751F#,  --  生 (<CJK Ideograph>)
      2938   => 16#76DB#,  --  盛 (<CJK Ideograph>)
      2939   => 16#7CBE#,  --  精 (<CJK Ideograph>)
      2940   => 16#8056#,  --  聖 (<CJK Ideograph>)
      2941   => 16#58F0#,  --  声 (<CJK Ideograph>)
      2942   => 16#88FD#,  --  製 (<CJK Ideograph>)
      2943   => 16#897F#,  --  西 (<CJK Ideograph>)
      2944   => 16#8AA0#,  --  誠 (<CJK Ideograph>)
      2945   => 16#8A93#,  --  誓 (<CJK Ideograph>)
      2946   => 16#8ACB#,  --  請 (<CJK Ideograph>)
      2947   => 16#901D#,  --  逝 (<CJK Ideograph>)
      2948   => 16#9192#,  --  醒 (<CJK Ideograph>)
      2949   => 16#9752#,  --  青 (<CJK Ideograph>)
      2950   => 16#9759#,  --  静 (<CJK Ideograph>)
      2951   => 16#6589#,  --  斉 (<CJK Ideograph>)
      2952   => 16#7A0E#,  --  税 (<CJK Ideograph>)
      2953   => 16#8106#,  --  脆 (<CJK Ideograph>)
      2954   => 16#96BB#,  --  隻 (<CJK Ideograph>)
      2955   => 16#5E2D#,  --  席 (<CJK Ideograph>)
      2956   => 16#60DC#,  --  惜 (<CJK Ideograph>)
      2957   => 16#621A#,  --  戚 (<CJK Ideograph>)
      2958   => 16#65A5#,  --  斥 (<CJK Ideograph>)
      2959   => 16#6614#,  --  昔 (<CJK Ideograph>)
      2960   => 16#6790#,  --  析 (<CJK Ideograph>)
      2961   => 16#77F3#,  --  石 (<CJK Ideograph>)
      2962   => 16#7A4D#,  --  積 (<CJK Ideograph>)
      2963   => 16#7C4D#,  --  籍 (<CJK Ideograph>)
      2964   => 16#7E3E#,  --  績 (<CJK Ideograph>)
      2965   => 16#810A#,  --  脊 (<CJK Ideograph>)
      2966   => 16#8CAC#,  --  責 (<CJK Ideograph>)
      2967   => 16#8D64#,  --  赤 (<CJK Ideograph>)
      2968   => 16#8DE1#,  --  跡 (<CJK Ideograph>)
      2969   => 16#8E5F#,  --  蹟 (<CJK Ideograph>)
      2970   => 16#78A9#,  --  碩 (<CJK Ideograph>)
      2971   => 16#5207#,  --  切 (<CJK Ideograph>)
      2972   => 16#62D9#,  --  拙 (<CJK Ideograph>)
      2973   => 16#63A5#,  --  接 (<CJK Ideograph>)
      2974   => 16#6442#,  --  摂 (<CJK Ideograph>)
      2975   => 16#6298#,  --  折 (<CJK Ideograph>)
      2976   => 16#8A2D#,  --  設 (<CJK Ideograph>)
      2977   => 16#7A83#,  --  窃 (<CJK Ideograph>)
      2978   => 16#7BC0#,  --  節 (<CJK Ideograph>)
      2979   => 16#8AAC#,  --  説 (<CJK Ideograph>)
      2980   => 16#96EA#,  --  雪 (<CJK Ideograph>)
      2981   => 16#7D76#,  --  絶 (<CJK Ideograph>)
      2982   => 16#820C#,  --  舌 (<CJK Ideograph>)
      2983   => 16#8749#,  --  蝉 (<CJK Ideograph>)
      2984   => 16#4ED9#,  --  仙 (<CJK Ideograph>)
      2985   => 16#5148#,  --  先 (<CJK Ideograph>)
      2986   => 16#5343#,  --  千 (<CJK Ideograph>)
      2987   => 16#5360#,  --  占 (<CJK Ideograph>)
      2988   => 16#5BA3#,  --  宣 (<CJK Ideograph>)
      2989   => 16#5C02#,  --  専 (<CJK Ideograph>)
      2990   => 16#5C16#,  --  尖 (<CJK Ideograph>)
      2991   => 16#5DDD#,  --  川 (<CJK Ideograph>)
      2992   => 16#6226#,  --  戦 (<CJK Ideograph>)
      2993   => 16#6247#,  --  扇 (<CJK Ideograph>)
      2994   => 16#64B0#,  --  撰 (<CJK Ideograph>)
      2995   => 16#6813#,  --  栓 (<CJK Ideograph>)
      2996   => 16#6834#,  --  栴 (<CJK Ideograph>)
      2997   => 16#6CC9#,  --  泉 (<CJK Ideograph>)
      2998   => 16#6D45#,  --  浅 (<CJK Ideograph>)
      2999   => 16#6D17#,  --  洗 (<CJK Ideograph>)
      3000   => 16#67D3#,  --  染 (<CJK Ideograph>)
      3001   => 16#6F5C#,  --  潜 (<CJK Ideograph>)
      3002   => 16#714E#,  --  煎 (<CJK Ideograph>)
      3003   => 16#717D#,  --  煽 (<CJK Ideograph>)
      3004   => 16#65CB#,  --  旋 (<CJK Ideograph>)
      3005   => 16#7A7F#,  --  穿 (<CJK Ideograph>)
      3006   => 16#7BAD#,  --  箭 (<CJK Ideograph>)
      3007   => 16#7DDA#,  --  線 (<CJK Ideograph>)
      3008   => 16#7E4A#,  --  繊 (<CJK Ideograph>)
      3009   => 16#7FA8#,  --  羨 (<CJK Ideograph>)
      3010   => 16#817A#,  --  腺 (<CJK Ideograph>)
      3011   => 16#821B#,  --  舛 (<CJK Ideograph>)
      3012   => 16#8239#,  --  船 (<CJK Ideograph>)
      3013   => 16#85A6#,  --  薦 (<CJK Ideograph>)
      3014   => 16#8A6E#,  --  詮 (<CJK Ideograph>)
      3015   => 16#8CCE#,  --  賎 (<CJK Ideograph>)
      3016   => 16#8DF5#,  --  践 (<CJK Ideograph>)
      3017   => 16#9078#,  --  選 (<CJK Ideograph>)
      3018   => 16#9077#,  --  遷 (<CJK Ideograph>)
      3019   => 16#92AD#,  --  銭 (<CJK Ideograph>)
      3020   => 16#9291#,  --  銑 (<CJK Ideograph>)
      3021   => 16#9583#,  --  閃 (<CJK Ideograph>)
      3022   => 16#9BAE#,  --  鮮 (<CJK Ideograph>)
      3023   => 16#524D#,  --  前 (<CJK Ideograph>)
      3024   => 16#5584#,  --  善 (<CJK Ideograph>)
      3025   => 16#6F38#,  --  漸 (<CJK Ideograph>)
      3026   => 16#7136#,  --  然 (<CJK Ideograph>)
      3027   => 16#5168#,  --  全 (<CJK Ideograph>)
      3028   => 16#7985#,  --  禅 (<CJK Ideograph>)
      3029   => 16#7E55#,  --  繕 (<CJK Ideograph>)
      3030   => 16#81B3#,  --  膳 (<CJK Ideograph>)
      3031   => 16#7CCE#,  --  糎 (<CJK Ideograph>)
      3032   => 16#564C#,  --  噌 (<CJK Ideograph>)
      3033   => 16#5851#,  --  塑 (<CJK Ideograph>)
      3034   => 16#5CA8#,  --  岨 (<CJK Ideograph>)
      3035   => 16#63AA#,  --  措 (<CJK Ideograph>)
      3036   => 16#66FE#,  --  曾 (<CJK Ideograph>)
      3037   => 16#66FD#,  --  曽 (<CJK Ideograph>)
      3038   => 16#695A#,  --  楚 (<CJK Ideograph>)
      3039   => 16#72D9#,  --  狙 (<CJK Ideograph>)
      3040   => 16#758F#,  --  疏 (<CJK Ideograph>)
      3041   => 16#758E#,  --  疎 (<CJK Ideograph>)
      3042   => 16#790E#,  --  礎 (<CJK Ideograph>)
      3043   => 16#7956#,  --  祖 (<CJK Ideograph>)
      3044   => 16#79DF#,  --  租 (<CJK Ideograph>)
      3045   => 16#7C97#,  --  粗 (<CJK Ideograph>)
      3046   => 16#7D20#,  --  素 (<CJK Ideograph>)
      3047   => 16#7D44#,  --  組 (<CJK Ideograph>)
      3048   => 16#8607#,  --  蘇 (<CJK Ideograph>)
      3049   => 16#8A34#,  --  訴 (<CJK Ideograph>)
      3050   => 16#963B#,  --  阻 (<CJK Ideograph>)
      3051   => 16#9061#,  --  遡 (<CJK Ideograph>)
      3052   => 16#9F20#,  --  鼠 (<CJK Ideograph>)
      3053   => 16#50E7#,  --  僧 (<CJK Ideograph>)
      3054   => 16#5275#,  --  創 (<CJK Ideograph>)
      3055   => 16#53CC#,  --  双 (<CJK Ideograph>)
      3056   => 16#53E2#,  --  叢 (<CJK Ideograph>)
      3057   => 16#5009#,  --  倉 (<CJK Ideograph>)
      3058   => 16#55AA#,  --  喪 (<CJK Ideograph>)
      3059   => 16#58EE#,  --  壮 (<CJK Ideograph>)
      3060   => 16#594F#,  --  奏 (<CJK Ideograph>)
      3061   => 16#723D#,  --  爽 (<CJK Ideograph>)
      3062   => 16#5B8B#,  --  宋 (<CJK Ideograph>)
      3063   => 16#5C64#,  --  層 (<CJK Ideograph>)
      3064   => 16#531D#,  --  匝 (<CJK Ideograph>)
      3065   => 16#60E3#,  --  惣 (<CJK Ideograph>)
      3066   => 16#60F3#,  --  想 (<CJK Ideograph>)
      3067   => 16#635C#,  --  捜 (<CJK Ideograph>)
      3068   => 16#6383#,  --  掃 (<CJK Ideograph>)
      3069   => 16#633F#,  --  挿 (<CJK Ideograph>)
      3070   => 16#63BB#,  --  掻 (<CJK Ideograph>)
      3071   => 16#64CD#,  --  操 (<CJK Ideograph>)
      3072   => 16#65E9#,  --  早 (<CJK Ideograph>)
      3073   => 16#66F9#,  --  曹 (<CJK Ideograph>)
      3074   => 16#5DE3#,  --  巣 (<CJK Ideograph>)
      3075   => 16#69CD#,  --  槍 (<CJK Ideograph>)
      3076   => 16#69FD#,  --  槽 (<CJK Ideograph>)
      3077   => 16#6F15#,  --  漕 (<CJK Ideograph>)
      3078   => 16#71E5#,  --  燥 (<CJK Ideograph>)
      3079   => 16#4E89#,  --  争 (<CJK Ideograph>)
      3080   => 16#75E9#,  --  痩 (<CJK Ideograph>)
      3081   => 16#76F8#,  --  相 (<CJK Ideograph>)
      3082   => 16#7A93#,  --  窓 (<CJK Ideograph>)
      3083   => 16#7CDF#,  --  糟 (<CJK Ideograph>)
      3084   => 16#7DCF#,  --  総 (<CJK Ideograph>)
      3085   => 16#7D9C#,  --  綜 (<CJK Ideograph>)
      3086   => 16#8061#,  --  聡 (<CJK Ideograph>)
      3087   => 16#8349#,  --  草 (<CJK Ideograph>)
      3088   => 16#8358#,  --  荘 (<CJK Ideograph>)
      3089   => 16#846C#,  --  葬 (<CJK Ideograph>)
      3090   => 16#84BC#,  --  蒼 (<CJK Ideograph>)
      3091   => 16#85FB#,  --  藻 (<CJK Ideograph>)
      3092   => 16#88C5#,  --  装 (<CJK Ideograph>)
      3093   => 16#8D70#,  --  走 (<CJK Ideograph>)
      3094   => 16#9001#,  --  送 (<CJK Ideograph>)
      3095   => 16#906D#,  --  遭 (<CJK Ideograph>)
      3096   => 16#9397#,  --  鎗 (<CJK Ideograph>)
      3097   => 16#971C#,  --  霜 (<CJK Ideograph>)
      3098   => 16#9A12#,  --  騒 (<CJK Ideograph>)
      3099   => 16#50CF#,  --  像 (<CJK Ideograph>)
      3100   => 16#5897#,  --  増 (<CJK Ideograph>)
      3101   => 16#618E#,  --  憎 (<CJK Ideograph>)
      3102   => 16#81D3#,  --  臓 (<CJK Ideograph>)
      3103   => 16#8535#,  --  蔵 (<CJK Ideograph>)
      3104   => 16#8D08#,  --  贈 (<CJK Ideograph>)
      3105   => 16#9020#,  --  造 (<CJK Ideograph>)
      3106   => 16#4FC3#,  --  促 (<CJK Ideograph>)
      3107   => 16#5074#,  --  側 (<CJK Ideograph>)
      3108   => 16#5247#,  --  則 (<CJK Ideograph>)
      3109   => 16#5373#,  --  即 (<CJK Ideograph>)
      3110   => 16#606F#,  --  息 (<CJK Ideograph>)
      3111   => 16#6349#,  --  捉 (<CJK Ideograph>)
      3112   => 16#675F#,  --  束 (<CJK Ideograph>)
      3113   => 16#6E2C#,  --  測 (<CJK Ideograph>)
      3114   => 16#8DB3#,  --  足 (<CJK Ideograph>)
      3115   => 16#901F#,  --  速 (<CJK Ideograph>)
      3116   => 16#4FD7#,  --  俗 (<CJK Ideograph>)
      3117   => 16#5C5E#,  --  属 (<CJK Ideograph>)
      3118   => 16#8CCA#,  --  賊 (<CJK Ideograph>)
      3119   => 16#65CF#,  --  族 (<CJK Ideograph>)
      3120   => 16#7D9A#,  --  続 (<CJK Ideograph>)
      3121   => 16#5352#,  --  卒 (<CJK Ideograph>)
      3122   => 16#8896#,  --  袖 (<CJK Ideograph>)
      3123   => 16#5176#,  --  其 (<CJK Ideograph>)
      3124   => 16#63C3#,  --  揃 (<CJK Ideograph>)
      3125   => 16#5B58#,  --  存 (<CJK Ideograph>)
      3126   => 16#5B6B#,  --  孫 (<CJK Ideograph>)
      3127   => 16#5C0A#,  --  尊 (<CJK Ideograph>)
      3128   => 16#640D#,  --  損 (<CJK Ideograph>)
      3129   => 16#6751#,  --  村 (<CJK Ideograph>)
      3130   => 16#905C#,  --  遜 (<CJK Ideograph>)
      3131   => 16#4ED6#,  --  他 (<CJK Ideograph>)
      3132   => 16#591A#,  --  多 (<CJK Ideograph>)
      3133   => 16#592A#,  --  太 (<CJK Ideograph>)
      3134   => 16#6C70#,  --  汰 (<CJK Ideograph>)
      3135   => 16#8A51#,  --  詑 (<CJK Ideograph>)
      3136   => 16#553E#,  --  唾 (<CJK Ideograph>)
      3137   => 16#5815#,  --  堕 (<CJK Ideograph>)
      3138   => 16#59A5#,  --  妥 (<CJK Ideograph>)
      3139   => 16#60F0#,  --  惰 (<CJK Ideograph>)
      3140   => 16#6253#,  --  打 (<CJK Ideograph>)
      3141   => 16#67C1#,  --  柁 (<CJK Ideograph>)
      3142   => 16#8235#,  --  舵 (<CJK Ideograph>)
      3143   => 16#6955#,  --  楕 (<CJK Ideograph>)
      3144   => 16#9640#,  --  陀 (<CJK Ideograph>)
      3145   => 16#99C4#,  --  駄 (<CJK Ideograph>)
      3146   => 16#9A28#,  --  騨 (<CJK Ideograph>)
      3147   => 16#4F53#,  --  体 (<CJK Ideograph>)
      3148   => 16#5806#,  --  堆 (<CJK Ideograph>)
      3149   => 16#5BFE#,  --  対 (<CJK Ideograph>)
      3150   => 16#8010#,  --  耐 (<CJK Ideograph>)
      3151   => 16#5CB1#,  --  岱 (<CJK Ideograph>)
      3152   => 16#5E2F#,  --  帯 (<CJK Ideograph>)
      3153   => 16#5F85#,  --  待 (<CJK Ideograph>)
      3154   => 16#6020#,  --  怠 (<CJK Ideograph>)
      3155   => 16#614B#,  --  態 (<CJK Ideograph>)
      3156   => 16#6234#,  --  戴 (<CJK Ideograph>)
      3157   => 16#66FF#,  --  替 (<CJK Ideograph>)
      3158   => 16#6CF0#,  --  泰 (<CJK Ideograph>)
      3159   => 16#6EDE#,  --  滞 (<CJK Ideograph>)
      3160   => 16#80CE#,  --  胎 (<CJK Ideograph>)
      3161   => 16#817F#,  --  腿 (<CJK Ideograph>)
      3162   => 16#82D4#,  --  苔 (<CJK Ideograph>)
      3163   => 16#888B#,  --  袋 (<CJK Ideograph>)
      3164   => 16#8CB8#,  --  貸 (<CJK Ideograph>)
      3165   => 16#9000#,  --  退 (<CJK Ideograph>)
      3166   => 16#902E#,  --  逮 (<CJK Ideograph>)
      3167   => 16#968A#,  --  隊 (<CJK Ideograph>)
      3168   => 16#9EDB#,  --  黛 (<CJK Ideograph>)
      3169   => 16#9BDB#,  --  鯛 (<CJK Ideograph>)
      3170   => 16#4EE3#,  --  代 (<CJK Ideograph>)
      3171   => 16#53F0#,  --  台 (<CJK Ideograph>)
      3172   => 16#5927#,  --  大 (<CJK Ideograph>)
      3173   => 16#7B2C#,  --  第 (<CJK Ideograph>)
      3174   => 16#918D#,  --  醍 (<CJK Ideograph>)
      3175   => 16#984C#,  --  題 (<CJK Ideograph>)
      3176   => 16#9DF9#,  --  鷹 (<CJK Ideograph>)
      3177   => 16#6EDD#,  --  滝 (<CJK Ideograph>)
      3178   => 16#7027#,  --  瀧 (<CJK Ideograph>)
      3179   => 16#5353#,  --  卓 (<CJK Ideograph>)
      3180   => 16#5544#,  --  啄 (<CJK Ideograph>)
      3181   => 16#5B85#,  --  宅 (<CJK Ideograph>)
      3182   => 16#6258#,  --  托 (<CJK Ideograph>)
      3183   => 16#629E#,  --  択 (<CJK Ideograph>)
      3184   => 16#62D3#,  --  拓 (<CJK Ideograph>)
      3185   => 16#6CA2#,  --  沢 (<CJK Ideograph>)
      3186   => 16#6FEF#,  --  濯 (<CJK Ideograph>)
      3187   => 16#7422#,  --  琢 (<CJK Ideograph>)
      3188   => 16#8A17#,  --  託 (<CJK Ideograph>)
      3189   => 16#9438#,  --  鐸 (<CJK Ideograph>)
      3190   => 16#6FC1#,  --  濁 (<CJK Ideograph>)
      3191   => 16#8AFE#,  --  諾 (<CJK Ideograph>)
      3192   => 16#8338#,  --  茸 (<CJK Ideograph>)
      3193   => 16#51E7#,  --  凧 (<CJK Ideograph>)
      3194   => 16#86F8#,  --  蛸 (<CJK Ideograph>)
      3195   => 16#53EA#,  --  只 (<CJK Ideograph>)
      3196   => 16#53E9#,  --  叩 (<CJK Ideograph>)
      3197   => 16#4F46#,  --  但 (<CJK Ideograph>)
      3198   => 16#9054#,  --  達 (<CJK Ideograph>)
      3199   => 16#8FB0#,  --  辰 (<CJK Ideograph>)
      3200   => 16#596A#,  --  奪 (<CJK Ideograph>)
      3201   => 16#8131#,  --  脱 (<CJK Ideograph>)
      3202   => 16#5DFD#,  --  巽 (<CJK Ideograph>)
      3203   => 16#7AEA#,  --  竪 (<CJK Ideograph>)
      3204   => 16#8FBF#,  --  辿 (<CJK Ideograph>)
      3205   => 16#68DA#,  --  棚 (<CJK Ideograph>)
      3206   => 16#8C37#,  --  谷 (<CJK Ideograph>)
      3207   => 16#72F8#,  --  狸 (<CJK Ideograph>)
      3208   => 16#9C48#,  --  鱈 (<CJK Ideograph>)
      3209   => 16#6A3D#,  --  樽 (<CJK Ideograph>)
      3210   => 16#8AB0#,  --  誰 (<CJK Ideograph>)
      3211   => 16#4E39#,  --  丹 (<CJK Ideograph>)
      3212   => 16#5358#,  --  単 (<CJK Ideograph>)
      3213   => 16#5606#,  --  嘆 (<CJK Ideograph>)
      3214   => 16#5766#,  --  坦 (<CJK Ideograph>)
      3215   => 16#62C5#,  --  担 (<CJK Ideograph>)
      3216   => 16#63A2#,  --  探 (<CJK Ideograph>)
      3217   => 16#65E6#,  --  旦 (<CJK Ideograph>)
      3218   => 16#6B4E#,  --  歎 (<CJK Ideograph>)
      3219   => 16#6DE1#,  --  淡 (<CJK Ideograph>)
      3220   => 16#6E5B#,  --  湛 (<CJK Ideograph>)
      3221   => 16#70AD#,  --  炭 (<CJK Ideograph>)
      3222   => 16#77ED#,  --  短 (<CJK Ideograph>)
      3223   => 16#7AEF#,  --  端 (<CJK Ideograph>)
      3224   => 16#7BAA#,  --  箪 (<CJK Ideograph>)
      3225   => 16#7DBB#,  --  綻 (<CJK Ideograph>)
      3226   => 16#803D#,  --  耽 (<CJK Ideograph>)
      3227   => 16#80C6#,  --  胆 (<CJK Ideograph>)
      3228   => 16#86CB#,  --  蛋 (<CJK Ideograph>)
      3229   => 16#8A95#,  --  誕 (<CJK Ideograph>)
      3230   => 16#935B#,  --  鍛 (<CJK Ideograph>)
      3231   => 16#56E3#,  --  団 (<CJK Ideograph>)
      3232   => 16#58C7#,  --  壇 (<CJK Ideograph>)
      3233   => 16#5F3E#,  --  弾 (<CJK Ideograph>)
      3234   => 16#65AD#,  --  断 (<CJK Ideograph>)
      3235   => 16#6696#,  --  暖 (<CJK Ideograph>)
      3236   => 16#6A80#,  --  檀 (<CJK Ideograph>)
      3237   => 16#6BB5#,  --  段 (<CJK Ideograph>)
      3238   => 16#7537#,  --  男 (<CJK Ideograph>)
      3239   => 16#8AC7#,  --  談 (<CJK Ideograph>)
      3240   => 16#5024#,  --  値 (<CJK Ideograph>)
      3241   => 16#77E5#,  --  知 (<CJK Ideograph>)
      3242   => 16#5730#,  --  地 (<CJK Ideograph>)
      3243   => 16#5F1B#,  --  弛 (<CJK Ideograph>)
      3244   => 16#6065#,  --  恥 (<CJK Ideograph>)
      3245   => 16#667A#,  --  智 (<CJK Ideograph>)
      3246   => 16#6C60#,  --  池 (<CJK Ideograph>)
      3247   => 16#75F4#,  --  痴 (<CJK Ideograph>)
      3248   => 16#7A1A#,  --  稚 (<CJK Ideograph>)
      3249   => 16#7F6E#,  --  置 (<CJK Ideograph>)
      3250   => 16#81F4#,  --  致 (<CJK Ideograph>)
      3251   => 16#8718#,  --  蜘 (<CJK Ideograph>)
      3252   => 16#9045#,  --  遅 (<CJK Ideograph>)
      3253   => 16#99B3#,  --  馳 (<CJK Ideograph>)
      3254   => 16#7BC9#,  --  築 (<CJK Ideograph>)
      3255   => 16#755C#,  --  畜 (<CJK Ideograph>)
      3256   => 16#7AF9#,  --  竹 (<CJK Ideograph>)
      3257   => 16#7B51#,  --  筑 (<CJK Ideograph>)
      3258   => 16#84C4#,  --  蓄 (<CJK Ideograph>)
      3259   => 16#9010#,  --  逐 (<CJK Ideograph>)
      3260   => 16#79E9#,  --  秩 (<CJK Ideograph>)
      3261   => 16#7A92#,  --  窒 (<CJK Ideograph>)
      3262   => 16#8336#,  --  茶 (<CJK Ideograph>)
      3263   => 16#5AE1#,  --  嫡 (<CJK Ideograph>)
      3264   => 16#7740#,  --  着 (<CJK Ideograph>)
      3265   => 16#4E2D#,  --  中 (<CJK Ideograph>)
      3266   => 16#4EF2#,  --  仲 (<CJK Ideograph>)
      3267   => 16#5B99#,  --  宙 (<CJK Ideograph>)
      3268   => 16#5FE0#,  --  忠 (<CJK Ideograph>)
      3269   => 16#62BD#,  --  抽 (<CJK Ideograph>)
      3270   => 16#663C#,  --  昼 (<CJK Ideograph>)
      3271   => 16#67F1#,  --  柱 (<CJK Ideograph>)
      3272   => 16#6CE8#,  --  注 (<CJK Ideograph>)
      3273   => 16#866B#,  --  虫 (<CJK Ideograph>)
      3274   => 16#8877#,  --  衷 (<CJK Ideograph>)
      3275   => 16#8A3B#,  --  註 (<CJK Ideograph>)
      3276   => 16#914E#,  --  酎 (<CJK Ideograph>)
      3277   => 16#92F3#,  --  鋳 (<CJK Ideograph>)
      3278   => 16#99D0#,  --  駐 (<CJK Ideograph>)
      3279   => 16#6A17#,  --  樗 (<CJK Ideograph>)
      3280   => 16#7026#,  --  瀦 (<CJK Ideograph>)
      3281   => 16#732A#,  --  猪 (<CJK Ideograph>)
      3282   => 16#82E7#,  --  苧 (<CJK Ideograph>)
      3283   => 16#8457#,  --  著 (<CJK Ideograph>)
      3284   => 16#8CAF#,  --  貯 (<CJK Ideograph>)
      3285   => 16#4E01#,  --  丁 (<CJK Ideograph>)
      3286   => 16#5146#,  --  兆 (<CJK Ideograph>)
      3287   => 16#51CB#,  --  凋 (<CJK Ideograph>)
      3288   => 16#558B#,  --  喋 (<CJK Ideograph>)
      3289   => 16#5BF5#,  --  寵 (<CJK Ideograph>)
      3290   => 16#5E16#,  --  帖 (<CJK Ideograph>)
      3291   => 16#5E33#,  --  帳 (<CJK Ideograph>)
      3292   => 16#5E81#,  --  庁 (<CJK Ideograph>)
      3293   => 16#5F14#,  --  弔 (<CJK Ideograph>)
      3294   => 16#5F35#,  --  張 (<CJK Ideograph>)
      3295   => 16#5F6B#,  --  彫 (<CJK Ideograph>)
      3296   => 16#5FB4#,  --  徴 (<CJK Ideograph>)
      3297   => 16#61F2#,  --  懲 (<CJK Ideograph>)
      3298   => 16#6311#,  --  挑 (<CJK Ideograph>)
      3299   => 16#66A2#,  --  暢 (<CJK Ideograph>)
      3300   => 16#671D#,  --  朝 (<CJK Ideograph>)
      3301   => 16#6F6E#,  --  潮 (<CJK Ideograph>)
      3302   => 16#7252#,  --  牒 (<CJK Ideograph>)
      3303   => 16#753A#,  --  町 (<CJK Ideograph>)
      3304   => 16#773A#,  --  眺 (<CJK Ideograph>)
      3305   => 16#8074#,  --  聴 (<CJK Ideograph>)
      3306   => 16#8139#,  --  脹 (<CJK Ideograph>)
      3307   => 16#8178#,  --  腸 (<CJK Ideograph>)
      3308   => 16#8776#,  --  蝶 (<CJK Ideograph>)
      3309   => 16#8ABF#,  --  調 (<CJK Ideograph>)
      3310   => 16#8ADC#,  --  諜 (<CJK Ideograph>)
      3311   => 16#8D85#,  --  超 (<CJK Ideograph>)
      3312   => 16#8DF3#,  --  跳 (<CJK Ideograph>)
      3313   => 16#929A#,  --  銚 (<CJK Ideograph>)
      3314   => 16#9577#,  --  長 (<CJK Ideograph>)
      3315   => 16#9802#,  --  頂 (<CJK Ideograph>)
      3316   => 16#9CE5#,  --  鳥 (<CJK Ideograph>)
      3317   => 16#52C5#,  --  勅 (<CJK Ideograph>)
      3318   => 16#6357#,  --  捗 (<CJK Ideograph>)
      3319   => 16#76F4#,  --  直 (<CJK Ideograph>)
      3320   => 16#6715#,  --  朕 (<CJK Ideograph>)
      3321   => 16#6C88#,  --  沈 (<CJK Ideograph>)
      3322   => 16#73CD#,  --  珍 (<CJK Ideograph>)
      3323   => 16#8CC3#,  --  賃 (<CJK Ideograph>)
      3324   => 16#93AE#,  --  鎮 (<CJK Ideograph>)
      3325   => 16#9673#,  --  陳 (<CJK Ideograph>)
      3326   => 16#6D25#,  --  津 (<CJK Ideograph>)
      3327   => 16#589C#,  --  墜 (<CJK Ideograph>)
      3328   => 16#690E#,  --  椎 (<CJK Ideograph>)
      3329   => 16#69CC#,  --  槌 (<CJK Ideograph>)
      3330   => 16#8FFD#,  --  追 (<CJK Ideograph>)
      3331   => 16#939A#,  --  鎚 (<CJK Ideograph>)
      3332   => 16#75DB#,  --  痛 (<CJK Ideograph>)
      3333   => 16#901A#,  --  通 (<CJK Ideograph>)
      3334   => 16#585A#,  --  塚 (<CJK Ideograph>)
      3335   => 16#6802#,  --  栂 (<CJK Ideograph>)
      3336   => 16#63B4#,  --  掴 (<CJK Ideograph>)
      3337   => 16#69FB#,  --  槻 (<CJK Ideograph>)
      3338   => 16#4F43#,  --  佃 (<CJK Ideograph>)
      3339   => 16#6F2C#,  --  漬 (<CJK Ideograph>)
      3340   => 16#67D8#,  --  柘 (<CJK Ideograph>)
      3341   => 16#8FBB#,  --  辻 (<CJK Ideograph>)
      3342   => 16#8526#,  --  蔦 (<CJK Ideograph>)
      3343   => 16#7DB4#,  --  綴 (<CJK Ideograph>)
      3344   => 16#9354#,  --  鍔 (<CJK Ideograph>)
      3345   => 16#693F#,  --  椿 (<CJK Ideograph>)
      3346   => 16#6F70#,  --  潰 (<CJK Ideograph>)
      3347   => 16#576A#,  --  坪 (<CJK Ideograph>)
      3348   => 16#58F7#,  --  壷 (<CJK Ideograph>)
      3349   => 16#5B2C#,  --  嬬 (<CJK Ideograph>)
      3350   => 16#7D2C#,  --  紬 (<CJK Ideograph>)
      3351   => 16#722A#,  --  爪 (<CJK Ideograph>)
      3352   => 16#540A#,  --  吊 (<CJK Ideograph>)
      3353   => 16#91E3#,  --  釣 (<CJK Ideograph>)
      3354   => 16#9DB4#,  --  鶴 (<CJK Ideograph>)
      3355   => 16#4EAD#,  --  亭 (<CJK Ideograph>)
      3356   => 16#4F4E#,  --  低 (<CJK Ideograph>)
      3357   => 16#505C#,  --  停 (<CJK Ideograph>)
      3358   => 16#5075#,  --  偵 (<CJK Ideograph>)
      3359   => 16#5243#,  --  剃 (<CJK Ideograph>)
      3360   => 16#8C9E#,  --  貞 (<CJK Ideograph>)
      3361   => 16#5448#,  --  呈 (<CJK Ideograph>)
      3362   => 16#5824#,  --  堤 (<CJK Ideograph>)
      3363   => 16#5B9A#,  --  定 (<CJK Ideograph>)
      3364   => 16#5E1D#,  --  帝 (<CJK Ideograph>)
      3365   => 16#5E95#,  --  底 (<CJK Ideograph>)
      3366   => 16#5EAD#,  --  庭 (<CJK Ideograph>)
      3367   => 16#5EF7#,  --  廷 (<CJK Ideograph>)
      3368   => 16#5F1F#,  --  弟 (<CJK Ideograph>)
      3369   => 16#608C#,  --  悌 (<CJK Ideograph>)
      3370   => 16#62B5#,  --  抵 (<CJK Ideograph>)
      3371   => 16#633A#,  --  挺 (<CJK Ideograph>)
      3372   => 16#63D0#,  --  提 (<CJK Ideograph>)
      3373   => 16#68AF#,  --  梯 (<CJK Ideograph>)
      3374   => 16#6C40#,  --  汀 (<CJK Ideograph>)
      3375   => 16#7887#,  --  碇 (<CJK Ideograph>)
      3376   => 16#798E#,  --  禎 (<CJK Ideograph>)
      3377   => 16#7A0B#,  --  程 (<CJK Ideograph>)
      3378   => 16#7DE0#,  --  締 (<CJK Ideograph>)
      3379   => 16#8247#,  --  艇 (<CJK Ideograph>)
      3380   => 16#8A02#,  --  訂 (<CJK Ideograph>)
      3381   => 16#8AE6#,  --  諦 (<CJK Ideograph>)
      3382   => 16#8E44#,  --  蹄 (<CJK Ideograph>)
      3383   => 16#9013#,  --  逓 (<CJK Ideograph>)
      3384   => 16#90B8#,  --  邸 (<CJK Ideograph>)
      3385   => 16#912D#,  --  鄭 (<CJK Ideograph>)
      3386   => 16#91D8#,  --  釘 (<CJK Ideograph>)
      3387   => 16#9F0E#,  --  鼎 (<CJK Ideograph>)
      3388   => 16#6CE5#,  --  泥 (<CJK Ideograph>)
      3389   => 16#6458#,  --  摘 (<CJK Ideograph>)
      3390   => 16#64E2#,  --  擢 (<CJK Ideograph>)
      3391   => 16#6575#,  --  敵 (<CJK Ideograph>)
      3392   => 16#6EF4#,  --  滴 (<CJK Ideograph>)
      3393   => 16#7684#,  --  的 (<CJK Ideograph>)
      3394   => 16#7B1B#,  --  笛 (<CJK Ideograph>)
      3395   => 16#9069#,  --  適 (<CJK Ideograph>)
      3396   => 16#93D1#,  --  鏑 (<CJK Ideograph>)
      3397   => 16#6EBA#,  --  溺 (<CJK Ideograph>)
      3398   => 16#54F2#,  --  哲 (<CJK Ideograph>)
      3399   => 16#5FB9#,  --  徹 (<CJK Ideograph>)
      3400   => 16#64A4#,  --  撤 (<CJK Ideograph>)
      3401   => 16#8F4D#,  --  轍 (<CJK Ideograph>)
      3402   => 16#8FED#,  --  迭 (<CJK Ideograph>)
      3403   => 16#9244#,  --  鉄 (<CJK Ideograph>)
      3404   => 16#5178#,  --  典 (<CJK Ideograph>)
      3405   => 16#586B#,  --  填 (<CJK Ideograph>)
      3406   => 16#5929#,  --  天 (<CJK Ideograph>)
      3407   => 16#5C55#,  --  展 (<CJK Ideograph>)
      3408   => 16#5E97#,  --  店 (<CJK Ideograph>)
      3409   => 16#6DFB#,  --  添 (<CJK Ideograph>)
      3410   => 16#7E8F#,  --  纏 (<CJK Ideograph>)
      3411   => 16#751C#,  --  甜 (<CJK Ideograph>)
      3412   => 16#8CBC#,  --  貼 (<CJK Ideograph>)
      3413   => 16#8EE2#,  --  転 (<CJK Ideograph>)
      3414   => 16#985B#,  --  顛 (<CJK Ideograph>)
      3415   => 16#70B9#,  --  点 (<CJK Ideograph>)
      3416   => 16#4F1D#,  --  伝 (<CJK Ideograph>)
      3417   => 16#6BBF#,  --  殿 (<CJK Ideograph>)
      3418   => 16#6FB1#,  --  澱 (<CJK Ideograph>)
      3419   => 16#7530#,  --  田 (<CJK Ideograph>)
      3420   => 16#96FB#,  --  電 (<CJK Ideograph>)
      3421   => 16#514E#,  --  兎 (<CJK Ideograph>)
      3422   => 16#5410#,  --  吐 (<CJK Ideograph>)
      3423   => 16#5835#,  --  堵 (<CJK Ideograph>)
      3424   => 16#5857#,  --  塗 (<CJK Ideograph>)
      3425   => 16#59AC#,  --  妬 (<CJK Ideograph>)
      3426   => 16#5C60#,  --  屠 (<CJK Ideograph>)
      3427   => 16#5F92#,  --  徒 (<CJK Ideograph>)
      3428   => 16#6597#,  --  斗 (<CJK Ideograph>)
      3429   => 16#675C#,  --  杜 (<CJK Ideograph>)
      3430   => 16#6E21#,  --  渡 (<CJK Ideograph>)
      3431   => 16#767B#,  --  登 (<CJK Ideograph>)
      3432   => 16#83DF#,  --  菟 (<CJK Ideograph>)
      3433   => 16#8CED#,  --  賭 (<CJK Ideograph>)
      3434   => 16#9014#,  --  途 (<CJK Ideograph>)
      3435   => 16#90FD#,  --  都 (<CJK Ideograph>)
      3436   => 16#934D#,  --  鍍 (<CJK Ideograph>)
      3437   => 16#7825#,  --  砥 (<CJK Ideograph>)
      3438   => 16#783A#,  --  砺 (<CJK Ideograph>)
      3439   => 16#52AA#,  --  努 (<CJK Ideograph>)
      3440   => 16#5EA6#,  --  度 (<CJK Ideograph>)
      3441   => 16#571F#,  --  土 (<CJK Ideograph>)
      3442   => 16#5974#,  --  奴 (<CJK Ideograph>)
      3443   => 16#6012#,  --  怒 (<CJK Ideograph>)
      3444   => 16#5012#,  --  倒 (<CJK Ideograph>)
      3445   => 16#515A#,  --  党 (<CJK Ideograph>)
      3446   => 16#51AC#,  --  冬 (<CJK Ideograph>)
      3447   => 16#51CD#,  --  凍 (<CJK Ideograph>)
      3448   => 16#5200#,  --  刀 (<CJK Ideograph>)
      3449   => 16#5510#,  --  唐 (<CJK Ideograph>)
      3450   => 16#5854#,  --  塔 (<CJK Ideograph>)
      3451   => 16#5858#,  --  塘 (<CJK Ideograph>)
      3452   => 16#5957#,  --  套 (<CJK Ideograph>)
      3453   => 16#5B95#,  --  宕 (<CJK Ideograph>)
      3454   => 16#5CF6#,  --  島 (<CJK Ideograph>)
      3455   => 16#5D8B#,  --  嶋 (<CJK Ideograph>)
      3456   => 16#60BC#,  --  悼 (<CJK Ideograph>)
      3457   => 16#6295#,  --  投 (<CJK Ideograph>)
      3458   => 16#642D#,  --  搭 (<CJK Ideograph>)
      3459   => 16#6771#,  --  東 (<CJK Ideograph>)
      3460   => 16#6843#,  --  桃 (<CJK Ideograph>)
      3461   => 16#68BC#,  --  梼 (<CJK Ideograph>)
      3462   => 16#68DF#,  --  棟 (<CJK Ideograph>)
      3463   => 16#76D7#,  --  盗 (<CJK Ideograph>)
      3464   => 16#6DD8#,  --  淘 (<CJK Ideograph>)
      3465   => 16#6E6F#,  --  湯 (<CJK Ideograph>)
      3466   => 16#6D9B#,  --  涛 (<CJK Ideograph>)
      3467   => 16#706F#,  --  灯 (<CJK Ideograph>)
      3468   => 16#71C8#,  --  燈 (<CJK Ideograph>)
      3469   => 16#5F53#,  --  当 (<CJK Ideograph>)
      3470   => 16#75D8#,  --  痘 (<CJK Ideograph>)
      3471   => 16#7977#,  --  祷 (<CJK Ideograph>)
      3472   => 16#7B49#,  --  等 (<CJK Ideograph>)
      3473   => 16#7B54#,  --  答 (<CJK Ideograph>)
      3474   => 16#7B52#,  --  筒 (<CJK Ideograph>)
      3475   => 16#7CD6#,  --  糖 (<CJK Ideograph>)
      3476   => 16#7D71#,  --  統 (<CJK Ideograph>)
      3477   => 16#5230#,  --  到 (<CJK Ideograph>)
      3478   => 16#8463#,  --  董 (<CJK Ideograph>)
      3479   => 16#8569#,  --  蕩 (<CJK Ideograph>)
      3480   => 16#85E4#,  --  藤 (<CJK Ideograph>)
      3481   => 16#8A0E#,  --  討 (<CJK Ideograph>)
      3482   => 16#8B04#,  --  謄 (<CJK Ideograph>)
      3483   => 16#8C46#,  --  豆 (<CJK Ideograph>)
      3484   => 16#8E0F#,  --  踏 (<CJK Ideograph>)
      3485   => 16#9003#,  --  逃 (<CJK Ideograph>)
      3486   => 16#900F#,  --  透 (<CJK Ideograph>)
      3487   => 16#9419#,  --  鐙 (<CJK Ideograph>)
      3488   => 16#9676#,  --  陶 (<CJK Ideograph>)
      3489   => 16#982D#,  --  頭 (<CJK Ideograph>)
      3490   => 16#9A30#,  --  騰 (<CJK Ideograph>)
      3491   => 16#95D8#,  --  闘 (<CJK Ideograph>)
      3492   => 16#50CD#,  --  働 (<CJK Ideograph>)
      3493   => 16#52D5#,  --  動 (<CJK Ideograph>)
      3494   => 16#540C#,  --  同 (<CJK Ideograph>)
      3495   => 16#5802#,  --  堂 (<CJK Ideograph>)
      3496   => 16#5C0E#,  --  導 (<CJK Ideograph>)
      3497   => 16#61A7#,  --  憧 (<CJK Ideograph>)
      3498   => 16#649E#,  --  撞 (<CJK Ideograph>)
      3499   => 16#6D1E#,  --  洞 (<CJK Ideograph>)
      3500   => 16#77B3#,  --  瞳 (<CJK Ideograph>)
      3501   => 16#7AE5#,  --  童 (<CJK Ideograph>)
      3502   => 16#80F4#,  --  胴 (<CJK Ideograph>)
      3503   => 16#8404#,  --  萄 (<CJK Ideograph>)
      3504   => 16#9053#,  --  道 (<CJK Ideograph>)
      3505   => 16#9285#,  --  銅 (<CJK Ideograph>)
      3506   => 16#5CE0#,  --  峠 (<CJK Ideograph>)
      3507   => 16#9D07#,  --  鴇 (<CJK Ideograph>)
      3508   => 16#533F#,  --  匿 (<CJK Ideograph>)
      3509   => 16#5F97#,  --  得 (<CJK Ideograph>)
      3510   => 16#5FB3#,  --  徳 (<CJK Ideograph>)
      3511   => 16#6D9C#,  --  涜 (<CJK Ideograph>)
      3512   => 16#7279#,  --  特 (<CJK Ideograph>)
      3513   => 16#7763#,  --  督 (<CJK Ideograph>)
      3514   => 16#79BF#,  --  禿 (<CJK Ideograph>)
      3515   => 16#7BE4#,  --  篤 (<CJK Ideograph>)
      3516   => 16#6BD2#,  --  毒 (<CJK Ideograph>)
      3517   => 16#72EC#,  --  独 (<CJK Ideograph>)
      3518   => 16#8AAD#,  --  読 (<CJK Ideograph>)
      3519   => 16#6803#,  --  栃 (<CJK Ideograph>)
      3520   => 16#6A61#,  --  橡 (<CJK Ideograph>)
      3521   => 16#51F8#,  --  凸 (<CJK Ideograph>)
      3522   => 16#7A81#,  --  突 (<CJK Ideograph>)
      3523   => 16#6934#,  --  椴 (<CJK Ideograph>)
      3524   => 16#5C4A#,  --  届 (<CJK Ideograph>)
      3525   => 16#9CF6#,  --  鳶 (<CJK Ideograph>)
      3526   => 16#82EB#,  --  苫 (<CJK Ideograph>)
      3527   => 16#5BC5#,  --  寅 (<CJK Ideograph>)
      3528   => 16#9149#,  --  酉 (<CJK Ideograph>)
      3529   => 16#701E#,  --  瀞 (<CJK Ideograph>)
      3530   => 16#5678#,  --  噸 (<CJK Ideograph>)
      3531   => 16#5C6F#,  --  屯 (<CJK Ideograph>)
      3532   => 16#60C7#,  --  惇 (<CJK Ideograph>)
      3533   => 16#6566#,  --  敦 (<CJK Ideograph>)
      3534   => 16#6C8C#,  --  沌 (<CJK Ideograph>)
      3535   => 16#8C5A#,  --  豚 (<CJK Ideograph>)
      3536   => 16#9041#,  --  遁 (<CJK Ideograph>)
      3537   => 16#9813#,  --  頓 (<CJK Ideograph>)
      3538   => 16#5451#,  --  呑 (<CJK Ideograph>)
      3539   => 16#66C7#,  --  曇 (<CJK Ideograph>)
      3540   => 16#920D#,  --  鈍 (<CJK Ideograph>)
      3541   => 16#5948#,  --  奈 (<CJK Ideograph>)
      3542   => 16#90A3#,  --  那 (<CJK Ideograph>)
      3543   => 16#5185#,  --  内 (<CJK Ideograph>)
      3544   => 16#4E4D#,  --  乍 (<CJK Ideograph>)
      3545   => 16#51EA#,  --  凪 (<CJK Ideograph>)
      3546   => 16#8599#,  --  薙 (<CJK Ideograph>)
      3547   => 16#8B0E#,  --  謎 (<CJK Ideograph>)
      3548   => 16#7058#,  --  灘 (<CJK Ideograph>)
      3549   => 16#637A#,  --  捺 (<CJK Ideograph>)
      3550   => 16#934B#,  --  鍋 (<CJK Ideograph>)
      3551   => 16#6962#,  --  楢 (<CJK Ideograph>)
      3552   => 16#99B4#,  --  馴 (<CJK Ideograph>)
      3553   => 16#7E04#,  --  縄 (<CJK Ideograph>)
      3554   => 16#7577#,  --  畷 (<CJK Ideograph>)
      3555   => 16#5357#,  --  南 (<CJK Ideograph>)
      3556   => 16#6960#,  --  楠 (<CJK Ideograph>)
      3557   => 16#8EDF#,  --  軟 (<CJK Ideograph>)
      3558   => 16#96E3#,  --  難 (<CJK Ideograph>)
      3559   => 16#6C5D#,  --  汝 (<CJK Ideograph>)
      3560   => 16#4E8C#,  --  二 (<CJK Ideograph>)
      3561   => 16#5C3C#,  --  尼 (<CJK Ideograph>)
      3562   => 16#5F10#,  --  弐 (<CJK Ideograph>)
      3563   => 16#8FE9#,  --  迩 (<CJK Ideograph>)
      3564   => 16#5302#,  --  匂 (<CJK Ideograph>)
      3565   => 16#8CD1#,  --  賑 (<CJK Ideograph>)
      3566   => 16#8089#,  --  肉 (<CJK Ideograph>)
      3567   => 16#8679#,  --  虹 (<CJK Ideograph>)
      3568   => 16#5EFF#,  --  廿 (<CJK Ideograph>)
      3569   => 16#65E5#,  --  日 (<CJK Ideograph>)
      3570   => 16#4E73#,  --  乳 (<CJK Ideograph>)
      3571   => 16#5165#,  --  入 (<CJK Ideograph>)
      3572   => 16#5982#,  --  如 (<CJK Ideograph>)
      3573   => 16#5C3F#,  --  尿 (<CJK Ideograph>)
      3574   => 16#97EE#,  --  韮 (<CJK Ideograph>)
      3575   => 16#4EFB#,  --  任 (<CJK Ideograph>)
      3576   => 16#598A#,  --  妊 (<CJK Ideograph>)
      3577   => 16#5FCD#,  --  忍 (<CJK Ideograph>)
      3578   => 16#8A8D#,  --  認 (<CJK Ideograph>)
      3579   => 16#6FE1#,  --  濡 (<CJK Ideograph>)
      3580   => 16#79B0#,  --  禰 (<CJK Ideograph>)
      3581   => 16#7962#,  --  祢 (<CJK Ideograph>)
      3582   => 16#5BE7#,  --  寧 (<CJK Ideograph>)
      3583   => 16#8471#,  --  葱 (<CJK Ideograph>)
      3584   => 16#732B#,  --  猫 (<CJK Ideograph>)
      3585   => 16#71B1#,  --  熱 (<CJK Ideograph>)
      3586   => 16#5E74#,  --  年 (<CJK Ideograph>)
      3587   => 16#5FF5#,  --  念 (<CJK Ideograph>)
      3588   => 16#637B#,  --  捻 (<CJK Ideograph>)
      3589   => 16#649A#,  --  撚 (<CJK Ideograph>)
      3590   => 16#71C3#,  --  燃 (<CJK Ideograph>)
      3591   => 16#7C98#,  --  粘 (<CJK Ideograph>)
      3592   => 16#4E43#,  --  乃 (<CJK Ideograph>)
      3593   => 16#5EFC#,  --  廼 (<CJK Ideograph>)
      3594   => 16#4E4B#,  --  之 (<CJK Ideograph>)
      3595   => 16#57DC#,  --  埜 (<CJK Ideograph>)
      3596   => 16#56A2#,  --  嚢 (<CJK Ideograph>)
      3597   => 16#60A9#,  --  悩 (<CJK Ideograph>)
      3598   => 16#6FC3#,  --  濃 (<CJK Ideograph>)
      3599   => 16#7D0D#,  --  納 (<CJK Ideograph>)
      3600   => 16#80FD#,  --  能 (<CJK Ideograph>)
      3601   => 16#8133#,  --  脳 (<CJK Ideograph>)
      3602   => 16#81BF#,  --  膿 (<CJK Ideograph>)
      3603   => 16#8FB2#,  --  農 (<CJK Ideograph>)
      3604   => 16#8997#,  --  覗 (<CJK Ideograph>)
      3605   => 16#86A4#,  --  蚤 (<CJK Ideograph>)
      3606   => 16#5DF4#,  --  巴 (<CJK Ideograph>)
      3607   => 16#628A#,  --  把 (<CJK Ideograph>)
      3608   => 16#64AD#,  --  播 (<CJK Ideograph>)
      3609   => 16#8987#,  --  覇 (<CJK Ideograph>)
      3610   => 16#6777#,  --  杷 (<CJK Ideograph>)
      3611   => 16#6CE2#,  --  波 (<CJK Ideograph>)
      3612   => 16#6D3E#,  --  派 (<CJK Ideograph>)
      3613   => 16#7436#,  --  琶 (<CJK Ideograph>)
      3614   => 16#7834#,  --  破 (<CJK Ideograph>)
      3615   => 16#5A46#,  --  婆 (<CJK Ideograph>)
      3616   => 16#7F75#,  --  罵 (<CJK Ideograph>)
      3617   => 16#82AD#,  --  芭 (<CJK Ideograph>)
      3618   => 16#99AC#,  --  馬 (<CJK Ideograph>)
      3619   => 16#4FF3#,  --  俳 (<CJK Ideograph>)
      3620   => 16#5EC3#,  --  廃 (<CJK Ideograph>)
      3621   => 16#62DD#,  --  拝 (<CJK Ideograph>)
      3622   => 16#6392#,  --  排 (<CJK Ideograph>)
      3623   => 16#6557#,  --  敗 (<CJK Ideograph>)
      3624   => 16#676F#,  --  杯 (<CJK Ideograph>)
      3625   => 16#76C3#,  --  盃 (<CJK Ideograph>)
      3626   => 16#724C#,  --  牌 (<CJK Ideograph>)
      3627   => 16#80CC#,  --  背 (<CJK Ideograph>)
      3628   => 16#80BA#,  --  肺 (<CJK Ideograph>)
      3629   => 16#8F29#,  --  輩 (<CJK Ideograph>)
      3630   => 16#914D#,  --  配 (<CJK Ideograph>)
      3631   => 16#500D#,  --  倍 (<CJK Ideograph>)
      3632   => 16#57F9#,  --  培 (<CJK Ideograph>)
      3633   => 16#5A92#,  --  媒 (<CJK Ideograph>)
      3634   => 16#6885#,  --  梅 (<CJK Ideograph>)
      3635   => 16#6973#,  --  楳 (<CJK Ideograph>)
      3636   => 16#7164#,  --  煤 (<CJK Ideograph>)
      3637   => 16#72FD#,  --  狽 (<CJK Ideograph>)
      3638   => 16#8CB7#,  --  買 (<CJK Ideograph>)
      3639   => 16#58F2#,  --  売 (<CJK Ideograph>)
      3640   => 16#8CE0#,  --  賠 (<CJK Ideograph>)
      3641   => 16#966A#,  --  陪 (<CJK Ideograph>)
      3642   => 16#9019#,  --  這 (<CJK Ideograph>)
      3643   => 16#877F#,  --  蝿 (<CJK Ideograph>)
      3644   => 16#79E4#,  --  秤 (<CJK Ideograph>)
      3645   => 16#77E7#,  --  矧 (<CJK Ideograph>)
      3646   => 16#8429#,  --  萩 (<CJK Ideograph>)
      3647   => 16#4F2F#,  --  伯 (<CJK Ideograph>)
      3648   => 16#5265#,  --  剥 (<CJK Ideograph>)
      3649   => 16#535A#,  --  博 (<CJK Ideograph>)
      3650   => 16#62CD#,  --  拍 (<CJK Ideograph>)
      3651   => 16#67CF#,  --  柏 (<CJK Ideograph>)
      3652   => 16#6CCA#,  --  泊 (<CJK Ideograph>)
      3653   => 16#767D#,  --  白 (<CJK Ideograph>)
      3654   => 16#7B94#,  --  箔 (<CJK Ideograph>)
      3655   => 16#7C95#,  --  粕 (<CJK Ideograph>)
      3656   => 16#8236#,  --  舶 (<CJK Ideograph>)
      3657   => 16#8584#,  --  薄 (<CJK Ideograph>)
      3658   => 16#8FEB#,  --  迫 (<CJK Ideograph>)
      3659   => 16#66DD#,  --  曝 (<CJK Ideograph>)
      3660   => 16#6F20#,  --  漠 (<CJK Ideograph>)
      3661   => 16#7206#,  --  爆 (<CJK Ideograph>)
      3662   => 16#7E1B#,  --  縛 (<CJK Ideograph>)
      3663   => 16#83AB#,  --  莫 (<CJK Ideograph>)
      3664   => 16#99C1#,  --  駁 (<CJK Ideograph>)
      3665   => 16#9EA6#,  --  麦 (<CJK Ideograph>)
      3666   => 16#51FD#,  --  函 (<CJK Ideograph>)
      3667   => 16#7BB1#,  --  箱 (<CJK Ideograph>)
      3668   => 16#7872#,  --  硲 (<CJK Ideograph>)
      3669   => 16#7BB8#,  --  箸 (<CJK Ideograph>)
      3670   => 16#8087#,  --  肇 (<CJK Ideograph>)
      3671   => 16#7B48#,  --  筈 (<CJK Ideograph>)
      3672   => 16#6AE8#,  --  櫨 (<CJK Ideograph>)
      3673   => 16#5E61#,  --  幡 (<CJK Ideograph>)
      3674   => 16#808C#,  --  肌 (<CJK Ideograph>)
      3675   => 16#7551#,  --  畑 (<CJK Ideograph>)
      3676   => 16#7560#,  --  畠 (<CJK Ideograph>)
      3677   => 16#516B#,  --  八 (<CJK Ideograph>)
      3678   => 16#9262#,  --  鉢 (<CJK Ideograph>)
      3679   => 16#6E8C#,  --  溌 (<CJK Ideograph>)
      3680   => 16#767A#,  --  発 (<CJK Ideograph>)
      3681   => 16#9197#,  --  醗 (<CJK Ideograph>)
      3682   => 16#9AEA#,  --  髪 (<CJK Ideograph>)
      3683   => 16#4F10#,  --  伐 (<CJK Ideograph>)
      3684   => 16#7F70#,  --  罰 (<CJK Ideograph>)
      3685   => 16#629C#,  --  抜 (<CJK Ideograph>)
      3686   => 16#7B4F#,  --  筏 (<CJK Ideograph>)
      3687   => 16#95A5#,  --  閥 (<CJK Ideograph>)
      3688   => 16#9CE9#,  --  鳩 (<CJK Ideograph>)
      3689   => 16#567A#,  --  噺 (<CJK Ideograph>)
      3690   => 16#5859#,  --  塙 (<CJK Ideograph>)
      3691   => 16#86E4#,  --  蛤 (<CJK Ideograph>)
      3692   => 16#96BC#,  --  隼 (<CJK Ideograph>)
      3693   => 16#4F34#,  --  伴 (<CJK Ideograph>)
      3694   => 16#5224#,  --  判 (<CJK Ideograph>)
      3695   => 16#534A#,  --  半 (<CJK Ideograph>)
      3696   => 16#53CD#,  --  反 (<CJK Ideograph>)
      3697   => 16#53DB#,  --  叛 (<CJK Ideograph>)
      3698   => 16#5E06#,  --  帆 (<CJK Ideograph>)
      3699   => 16#642C#,  --  搬 (<CJK Ideograph>)
      3700   => 16#6591#,  --  斑 (<CJK Ideograph>)
      3701   => 16#677F#,  --  板 (<CJK Ideograph>)
      3702   => 16#6C3E#,  --  氾 (<CJK Ideograph>)
      3703   => 16#6C4E#,  --  汎 (<CJK Ideograph>)
      3704   => 16#7248#,  --  版 (<CJK Ideograph>)
      3705   => 16#72AF#,  --  犯 (<CJK Ideograph>)
      3706   => 16#73ED#,  --  班 (<CJK Ideograph>)
      3707   => 16#7554#,  --  畔 (<CJK Ideograph>)
      3708   => 16#7E41#,  --  繁 (<CJK Ideograph>)
      3709   => 16#822C#,  --  般 (<CJK Ideograph>)
      3710   => 16#85E9#,  --  藩 (<CJK Ideograph>)
      3711   => 16#8CA9#,  --  販 (<CJK Ideograph>)
      3712   => 16#7BC4#,  --  範 (<CJK Ideograph>)
      3713   => 16#91C6#,  --  釆 (<CJK Ideograph>)
      3714   => 16#7169#,  --  煩 (<CJK Ideograph>)
      3715   => 16#9812#,  --  頒 (<CJK Ideograph>)
      3716   => 16#98EF#,  --  飯 (<CJK Ideograph>)
      3717   => 16#633D#,  --  挽 (<CJK Ideograph>)
      3718   => 16#6669#,  --  晩 (<CJK Ideograph>)
      3719   => 16#756A#,  --  番 (<CJK Ideograph>)
      3720   => 16#76E4#,  --  盤 (<CJK Ideograph>)
      3721   => 16#78D0#,  --  磐 (<CJK Ideograph>)
      3722   => 16#8543#,  --  蕃 (<CJK Ideograph>)
      3723   => 16#86EE#,  --  蛮 (<CJK Ideograph>)
      3724   => 16#532A#,  --  匪 (<CJK Ideograph>)
      3725   => 16#5351#,  --  卑 (<CJK Ideograph>)
      3726   => 16#5426#,  --  否 (<CJK Ideograph>)
      3727   => 16#5983#,  --  妃 (<CJK Ideograph>)
      3728   => 16#5E87#,  --  庇 (<CJK Ideograph>)
      3729   => 16#5F7C#,  --  彼 (<CJK Ideograph>)
      3730   => 16#60B2#,  --  悲 (<CJK Ideograph>)
      3731   => 16#6249#,  --  扉 (<CJK Ideograph>)
      3732   => 16#6279#,  --  批 (<CJK Ideograph>)
      3733   => 16#62AB#,  --  披 (<CJK Ideograph>)
      3734   => 16#6590#,  --  斐 (<CJK Ideograph>)
      3735   => 16#6BD4#,  --  比 (<CJK Ideograph>)
      3736   => 16#6CCC#,  --  泌 (<CJK Ideograph>)
      3737   => 16#75B2#,  --  疲 (<CJK Ideograph>)
      3738   => 16#76AE#,  --  皮 (<CJK Ideograph>)
      3739   => 16#7891#,  --  碑 (<CJK Ideograph>)
      3740   => 16#79D8#,  --  秘 (<CJK Ideograph>)
      3741   => 16#7DCB#,  --  緋 (<CJK Ideograph>)
      3742   => 16#7F77#,  --  罷 (<CJK Ideograph>)
      3743   => 16#80A5#,  --  肥 (<CJK Ideograph>)
      3744   => 16#88AB#,  --  被 (<CJK Ideograph>)
      3745   => 16#8AB9#,  --  誹 (<CJK Ideograph>)
      3746   => 16#8CBB#,  --  費 (<CJK Ideograph>)
      3747   => 16#907F#,  --  避 (<CJK Ideograph>)
      3748   => 16#975E#,  --  非 (<CJK Ideograph>)
      3749   => 16#98DB#,  --  飛 (<CJK Ideograph>)
      3750   => 16#6A0B#,  --  樋 (<CJK Ideograph>)
      3751   => 16#7C38#,  --  簸 (<CJK Ideograph>)
      3752   => 16#5099#,  --  備 (<CJK Ideograph>)
      3753   => 16#5C3E#,  --  尾 (<CJK Ideograph>)
      3754   => 16#5FAE#,  --  微 (<CJK Ideograph>)
      3755   => 16#6787#,  --  枇 (<CJK Ideograph>)
      3756   => 16#6BD8#,  --  毘 (<CJK Ideograph>)
      3757   => 16#7435#,  --  琵 (<CJK Ideograph>)
      3758   => 16#7709#,  --  眉 (<CJK Ideograph>)
      3759   => 16#7F8E#,  --  美 (<CJK Ideograph>)
      3760   => 16#9F3B#,  --  鼻 (<CJK Ideograph>)
      3761   => 16#67CA#,  --  柊 (<CJK Ideograph>)
      3762   => 16#7A17#,  --  稗 (<CJK Ideograph>)
      3763   => 16#5339#,  --  匹 (<CJK Ideograph>)
      3764   => 16#758B#,  --  疋 (<CJK Ideograph>)
      3765   => 16#9AED#,  --  髭 (<CJK Ideograph>)
      3766   => 16#5F66#,  --  彦 (<CJK Ideograph>)
      3767   => 16#819D#,  --  膝 (<CJK Ideograph>)
      3768   => 16#83F1#,  --  菱 (<CJK Ideograph>)
      3769   => 16#8098#,  --  肘 (<CJK Ideograph>)
      3770   => 16#5F3C#,  --  弼 (<CJK Ideograph>)
      3771   => 16#5FC5#,  --  必 (<CJK Ideograph>)
      3772   => 16#7562#,  --  畢 (<CJK Ideograph>)
      3773   => 16#7B46#,  --  筆 (<CJK Ideograph>)
      3774   => 16#903C#,  --  逼 (<CJK Ideograph>)
      3775   => 16#6867#,  --  桧 (<CJK Ideograph>)
      3776   => 16#59EB#,  --  姫 (<CJK Ideograph>)
      3777   => 16#5A9B#,  --  媛 (<CJK Ideograph>)
      3778   => 16#7D10#,  --  紐 (<CJK Ideograph>)
      3779   => 16#767E#,  --  百 (<CJK Ideograph>)
      3780   => 16#8B2C#,  --  謬 (<CJK Ideograph>)
      3781   => 16#4FF5#,  --  俵 (<CJK Ideograph>)
      3782   => 16#5F6A#,  --  彪 (<CJK Ideograph>)
      3783   => 16#6A19#,  --  標 (<CJK Ideograph>)
      3784   => 16#6C37#,  --  氷 (<CJK Ideograph>)
      3785   => 16#6F02#,  --  漂 (<CJK Ideograph>)
      3786   => 16#74E2#,  --  瓢 (<CJK Ideograph>)
      3787   => 16#7968#,  --  票 (<CJK Ideograph>)
      3788   => 16#8868#,  --  表 (<CJK Ideograph>)
      3789   => 16#8A55#,  --  評 (<CJK Ideograph>)
      3790   => 16#8C79#,  --  豹 (<CJK Ideograph>)
      3791   => 16#5EDF#,  --  廟 (<CJK Ideograph>)
      3792   => 16#63CF#,  --  描 (<CJK Ideograph>)
      3793   => 16#75C5#,  --  病 (<CJK Ideograph>)
      3794   => 16#79D2#,  --  秒 (<CJK Ideograph>)
      3795   => 16#82D7#,  --  苗 (<CJK Ideograph>)
      3796   => 16#9328#,  --  錨 (<CJK Ideograph>)
      3797   => 16#92F2#,  --  鋲 (<CJK Ideograph>)
      3798   => 16#849C#,  --  蒜 (<CJK Ideograph>)
      3799   => 16#86ED#,  --  蛭 (<CJK Ideograph>)
      3800   => 16#9C2D#,  --  鰭 (<CJK Ideograph>)
      3801   => 16#54C1#,  --  品 (<CJK Ideograph>)
      3802   => 16#5F6C#,  --  彬 (<CJK Ideograph>)
      3803   => 16#658C#,  --  斌 (<CJK Ideograph>)
      3804   => 16#6D5C#,  --  浜 (<CJK Ideograph>)
      3805   => 16#7015#,  --  瀕 (<CJK Ideograph>)
      3806   => 16#8CA7#,  --  貧 (<CJK Ideograph>)
      3807   => 16#8CD3#,  --  賓 (<CJK Ideograph>)
      3808   => 16#983B#,  --  頻 (<CJK Ideograph>)
      3809   => 16#654F#,  --  敏 (<CJK Ideograph>)
      3810   => 16#74F6#,  --  瓶 (<CJK Ideograph>)
      3811   => 16#4E0D#,  --  不 (<CJK Ideograph>)
      3812   => 16#4ED8#,  --  付 (<CJK Ideograph>)
      3813   => 16#57E0#,  --  埠 (<CJK Ideograph>)
      3814   => 16#592B#,  --  夫 (<CJK Ideograph>)
      3815   => 16#5A66#,  --  婦 (<CJK Ideograph>)
      3816   => 16#5BCC#,  --  富 (<CJK Ideograph>)
      3817   => 16#51A8#,  --  冨 (<CJK Ideograph>)
      3818   => 16#5E03#,  --  布 (<CJK Ideograph>)
      3819   => 16#5E9C#,  --  府 (<CJK Ideograph>)
      3820   => 16#6016#,  --  怖 (<CJK Ideograph>)
      3821   => 16#6276#,  --  扶 (<CJK Ideograph>)
      3822   => 16#6577#,  --  敷 (<CJK Ideograph>)
      3823   => 16#65A7#,  --  斧 (<CJK Ideograph>)
      3824   => 16#666E#,  --  普 (<CJK Ideograph>)
      3825   => 16#6D6E#,  --  浮 (<CJK Ideograph>)
      3826   => 16#7236#,  --  父 (<CJK Ideograph>)
      3827   => 16#7B26#,  --  符 (<CJK Ideograph>)
      3828   => 16#8150#,  --  腐 (<CJK Ideograph>)
      3829   => 16#819A#,  --  膚 (<CJK Ideograph>)
      3830   => 16#8299#,  --  芙 (<CJK Ideograph>)
      3831   => 16#8B5C#,  --  譜 (<CJK Ideograph>)
      3832   => 16#8CA0#,  --  負 (<CJK Ideograph>)
      3833   => 16#8CE6#,  --  賦 (<CJK Ideograph>)
      3834   => 16#8D74#,  --  赴 (<CJK Ideograph>)
      3835   => 16#961C#,  --  阜 (<CJK Ideograph>)
      3836   => 16#9644#,  --  附 (<CJK Ideograph>)
      3837   => 16#4FAE#,  --  侮 (<CJK Ideograph>)
      3838   => 16#64AB#,  --  撫 (<CJK Ideograph>)
      3839   => 16#6B66#,  --  武 (<CJK Ideograph>)
      3840   => 16#821E#,  --  舞 (<CJK Ideograph>)
      3841   => 16#8461#,  --  葡 (<CJK Ideograph>)
      3842   => 16#856A#,  --  蕪 (<CJK Ideograph>)
      3843   => 16#90E8#,  --  部 (<CJK Ideograph>)
      3844   => 16#5C01#,  --  封 (<CJK Ideograph>)
      3845   => 16#6953#,  --  楓 (<CJK Ideograph>)
      3846   => 16#98A8#,  --  風 (<CJK Ideograph>)
      3847   => 16#847A#,  --  葺 (<CJK Ideograph>)
      3848   => 16#8557#,  --  蕗 (<CJK Ideograph>)
      3849   => 16#4F0F#,  --  伏 (<CJK Ideograph>)
      3850   => 16#526F#,  --  副 (<CJK Ideograph>)
      3851   => 16#5FA9#,  --  復 (<CJK Ideograph>)
      3852   => 16#5E45#,  --  幅 (<CJK Ideograph>)
      3853   => 16#670D#,  --  服 (<CJK Ideograph>)
      3854   => 16#798F#,  --  福 (<CJK Ideograph>)
      3855   => 16#8179#,  --  腹 (<CJK Ideograph>)
      3856   => 16#8907#,  --  複 (<CJK Ideograph>)
      3857   => 16#8986#,  --  覆 (<CJK Ideograph>)
      3858   => 16#6DF5#,  --  淵 (<CJK Ideograph>)
      3859   => 16#5F17#,  --  弗 (<CJK Ideograph>)
      3860   => 16#6255#,  --  払 (<CJK Ideograph>)
      3861   => 16#6CB8#,  --  沸 (<CJK Ideograph>)
      3862   => 16#4ECF#,  --  仏 (<CJK Ideograph>)
      3863   => 16#7269#,  --  物 (<CJK Ideograph>)
      3864   => 16#9B92#,  --  鮒 (<CJK Ideograph>)
      3865   => 16#5206#,  --  分 (<CJK Ideograph>)
      3866   => 16#543B#,  --  吻 (<CJK Ideograph>)
      3867   => 16#5674#,  --  噴 (<CJK Ideograph>)
      3868   => 16#58B3#,  --  墳 (<CJK Ideograph>)
      3869   => 16#61A4#,  --  憤 (<CJK Ideograph>)
      3870   => 16#626E#,  --  扮 (<CJK Ideograph>)
      3871   => 16#711A#,  --  焚 (<CJK Ideograph>)
      3872   => 16#596E#,  --  奮 (<CJK Ideograph>)
      3873   => 16#7C89#,  --  粉 (<CJK Ideograph>)
      3874   => 16#7CDE#,  --  糞 (<CJK Ideograph>)
      3875   => 16#7D1B#,  --  紛 (<CJK Ideograph>)
      3876   => 16#96F0#,  --  雰 (<CJK Ideograph>)
      3877   => 16#6587#,  --  文 (<CJK Ideograph>)
      3878   => 16#805E#,  --  聞 (<CJK Ideograph>)
      3879   => 16#4E19#,  --  丙 (<CJK Ideograph>)
      3880   => 16#4F75#,  --  併 (<CJK Ideograph>)
      3881   => 16#5175#,  --  兵 (<CJK Ideograph>)
      3882   => 16#5840#,  --  塀 (<CJK Ideograph>)
      3883   => 16#5E63#,  --  幣 (<CJK Ideograph>)
      3884   => 16#5E73#,  --  平 (<CJK Ideograph>)
      3885   => 16#5F0A#,  --  弊 (<CJK Ideograph>)
      3886   => 16#67C4#,  --  柄 (<CJK Ideograph>)
      3887   => 16#4E26#,  --  並 (<CJK Ideograph>)
      3888   => 16#853D#,  --  蔽 (<CJK Ideograph>)
      3889   => 16#9589#,  --  閉 (<CJK Ideograph>)
      3890   => 16#965B#,  --  陛 (<CJK Ideograph>)
      3891   => 16#7C73#,  --  米 (<CJK Ideograph>)
      3892   => 16#9801#,  --  頁 (<CJK Ideograph>)
      3893   => 16#50FB#,  --  僻 (<CJK Ideograph>)
      3894   => 16#58C1#,  --  壁 (<CJK Ideograph>)
      3895   => 16#7656#,  --  癖 (<CJK Ideograph>)
      3896   => 16#78A7#,  --  碧 (<CJK Ideograph>)
      3897   => 16#5225#,  --  別 (<CJK Ideograph>)
      3898   => 16#77A5#,  --  瞥 (<CJK Ideograph>)
      3899   => 16#8511#,  --  蔑 (<CJK Ideograph>)
      3900   => 16#7B86#,  --  箆 (<CJK Ideograph>)
      3901   => 16#504F#,  --  偏 (<CJK Ideograph>)
      3902   => 16#5909#,  --  変 (<CJK Ideograph>)
      3903   => 16#7247#,  --  片 (<CJK Ideograph>)
      3904   => 16#7BC7#,  --  篇 (<CJK Ideograph>)
      3905   => 16#7DE8#,  --  編 (<CJK Ideograph>)
      3906   => 16#8FBA#,  --  辺 (<CJK Ideograph>)
      3907   => 16#8FD4#,  --  返 (<CJK Ideograph>)
      3908   => 16#904D#,  --  遍 (<CJK Ideograph>)
      3909   => 16#4FBF#,  --  便 (<CJK Ideograph>)
      3910   => 16#52C9#,  --  勉 (<CJK Ideograph>)
      3911   => 16#5A29#,  --  娩 (<CJK Ideograph>)
      3912   => 16#5F01#,  --  弁 (<CJK Ideograph>)
      3913   => 16#97AD#,  --  鞭 (<CJK Ideograph>)
      3914   => 16#4FDD#,  --  保 (<CJK Ideograph>)
      3915   => 16#8217#,  --  舗 (<CJK Ideograph>)
      3916   => 16#92EA#,  --  鋪 (<CJK Ideograph>)
      3917   => 16#5703#,  --  圃 (<CJK Ideograph>)
      3918   => 16#6355#,  --  捕 (<CJK Ideograph>)
      3919   => 16#6B69#,  --  歩 (<CJK Ideograph>)
      3920   => 16#752B#,  --  甫 (<CJK Ideograph>)
      3921   => 16#88DC#,  --  補 (<CJK Ideograph>)
      3922   => 16#8F14#,  --  輔 (<CJK Ideograph>)
      3923   => 16#7A42#,  --  穂 (<CJK Ideograph>)
      3924   => 16#52DF#,  --  募 (<CJK Ideograph>)
      3925   => 16#5893#,  --  墓 (<CJK Ideograph>)
      3926   => 16#6155#,  --  慕 (<CJK Ideograph>)
      3927   => 16#620A#,  --  戊 (<CJK Ideograph>)
      3928   => 16#66AE#,  --  暮 (<CJK Ideograph>)
      3929   => 16#6BCD#,  --  母 (<CJK Ideograph>)
      3930   => 16#7C3F#,  --  簿 (<CJK Ideograph>)
      3931   => 16#83E9#,  --  菩 (<CJK Ideograph>)
      3932   => 16#5023#,  --  倣 (<CJK Ideograph>)
      3933   => 16#4FF8#,  --  俸 (<CJK Ideograph>)
      3934   => 16#5305#,  --  包 (<CJK Ideograph>)
      3935   => 16#5446#,  --  呆 (<CJK Ideograph>)
      3936   => 16#5831#,  --  報 (<CJK Ideograph>)
      3937   => 16#5949#,  --  奉 (<CJK Ideograph>)
      3938   => 16#5B9D#,  --  宝 (<CJK Ideograph>)
      3939   => 16#5CF0#,  --  峰 (<CJK Ideograph>)
      3940   => 16#5CEF#,  --  峯 (<CJK Ideograph>)
      3941   => 16#5D29#,  --  崩 (<CJK Ideograph>)
      3942   => 16#5E96#,  --  庖 (<CJK Ideograph>)
      3943   => 16#62B1#,  --  抱 (<CJK Ideograph>)
      3944   => 16#6367#,  --  捧 (<CJK Ideograph>)
      3945   => 16#653E#,  --  放 (<CJK Ideograph>)
      3946   => 16#65B9#,  --  方 (<CJK Ideograph>)
      3947   => 16#670B#,  --  朋 (<CJK Ideograph>)
      3948   => 16#6CD5#,  --  法 (<CJK Ideograph>)
      3949   => 16#6CE1#,  --  泡 (<CJK Ideograph>)
      3950   => 16#70F9#,  --  烹 (<CJK Ideograph>)
      3951   => 16#7832#,  --  砲 (<CJK Ideograph>)
      3952   => 16#7E2B#,  --  縫 (<CJK Ideograph>)
      3953   => 16#80DE#,  --  胞 (<CJK Ideograph>)
      3954   => 16#82B3#,  --  芳 (<CJK Ideograph>)
      3955   => 16#840C#,  --  萌 (<CJK Ideograph>)
      3956   => 16#84EC#,  --  蓬 (<CJK Ideograph>)
      3957   => 16#8702#,  --  蜂 (<CJK Ideograph>)
      3958   => 16#8912#,  --  褒 (<CJK Ideograph>)
      3959   => 16#8A2A#,  --  訪 (<CJK Ideograph>)
      3960   => 16#8C4A#,  --  豊 (<CJK Ideograph>)
      3961   => 16#90A6#,  --  邦 (<CJK Ideograph>)
      3962   => 16#92D2#,  --  鋒 (<CJK Ideograph>)
      3963   => 16#98FD#,  --  飽 (<CJK Ideograph>)
      3964   => 16#9CF3#,  --  鳳 (<CJK Ideograph>)
      3965   => 16#9D6C#,  --  鵬 (<CJK Ideograph>)
      3966   => 16#4E4F#,  --  乏 (<CJK Ideograph>)
      3967   => 16#4EA1#,  --  亡 (<CJK Ideograph>)
      3968   => 16#508D#,  --  傍 (<CJK Ideograph>)
      3969   => 16#5256#,  --  剖 (<CJK Ideograph>)
      3970   => 16#574A#,  --  坊 (<CJK Ideograph>)
      3971   => 16#59A8#,  --  妨 (<CJK Ideograph>)
      3972   => 16#5E3D#,  --  帽 (<CJK Ideograph>)
      3973   => 16#5FD8#,  --  忘 (<CJK Ideograph>)
      3974   => 16#5FD9#,  --  忙 (<CJK Ideograph>)
      3975   => 16#623F#,  --  房 (<CJK Ideograph>)
      3976   => 16#66B4#,  --  暴 (<CJK Ideograph>)
      3977   => 16#671B#,  --  望 (<CJK Ideograph>)
      3978   => 16#67D0#,  --  某 (<CJK Ideograph>)
      3979   => 16#68D2#,  --  棒 (<CJK Ideograph>)
      3980   => 16#5192#,  --  冒 (<CJK Ideograph>)
      3981   => 16#7D21#,  --  紡 (<CJK Ideograph>)
      3982   => 16#80AA#,  --  肪 (<CJK Ideograph>)
      3983   => 16#81A8#,  --  膨 (<CJK Ideograph>)
      3984   => 16#8B00#,  --  謀 (<CJK Ideograph>)
      3985   => 16#8C8C#,  --  貌 (<CJK Ideograph>)
      3986   => 16#8CBF#,  --  貿 (<CJK Ideograph>)
      3987   => 16#927E#,  --  鉾 (<CJK Ideograph>)
      3988   => 16#9632#,  --  防 (<CJK Ideograph>)
      3989   => 16#5420#,  --  吠 (<CJK Ideograph>)
      3990   => 16#982C#,  --  頬 (<CJK Ideograph>)
      3991   => 16#5317#,  --  北 (<CJK Ideograph>)
      3992   => 16#50D5#,  --  僕 (<CJK Ideograph>)
      3993   => 16#535C#,  --  卜 (<CJK Ideograph>)
      3994   => 16#58A8#,  --  墨 (<CJK Ideograph>)
      3995   => 16#64B2#,  --  撲 (<CJK Ideograph>)
      3996   => 16#6734#,  --  朴 (<CJK Ideograph>)
      3997   => 16#7267#,  --  牧 (<CJK Ideograph>)
      3998   => 16#7766#,  --  睦 (<CJK Ideograph>)
      3999   => 16#7A46#,  --  穆 (<CJK Ideograph>)
      4000   => 16#91E6#,  --  釦 (<CJK Ideograph>)
      4001   => 16#52C3#,  --  勃 (<CJK Ideograph>)
      4002   => 16#6CA1#,  --  没 (<CJK Ideograph>)
      4003   => 16#6B86#,  --  殆 (<CJK Ideograph>)
      4004   => 16#5800#,  --  堀 (<CJK Ideograph>)
      4005   => 16#5E4C#,  --  幌 (<CJK Ideograph>)
      4006   => 16#5954#,  --  奔 (<CJK Ideograph>)
      4007   => 16#672C#,  --  本 (<CJK Ideograph>)
      4008   => 16#7FFB#,  --  翻 (<CJK Ideograph>)
      4009   => 16#51E1#,  --  凡 (<CJK Ideograph>)
      4010   => 16#76C6#,  --  盆 (<CJK Ideograph>)
      4011   => 16#6469#,  --  摩 (<CJK Ideograph>)
      4012   => 16#78E8#,  --  磨 (<CJK Ideograph>)
      4013   => 16#9B54#,  --  魔 (<CJK Ideograph>)
      4014   => 16#9EBB#,  --  麻 (<CJK Ideograph>)
      4015   => 16#57CB#,  --  埋 (<CJK Ideograph>)
      4016   => 16#59B9#,  --  妹 (<CJK Ideograph>)
      4017   => 16#6627#,  --  昧 (<CJK Ideograph>)
      4018   => 16#679A#,  --  枚 (<CJK Ideograph>)
      4019   => 16#6BCE#,  --  毎 (<CJK Ideograph>)
      4020   => 16#54E9#,  --  哩 (<CJK Ideograph>)
      4021   => 16#69D9#,  --  槙 (<CJK Ideograph>)
      4022   => 16#5E55#,  --  幕 (<CJK Ideograph>)
      4023   => 16#819C#,  --  膜 (<CJK Ideograph>)
      4024   => 16#6795#,  --  枕 (<CJK Ideograph>)
      4025   => 16#9BAA#,  --  鮪 (<CJK Ideograph>)
      4026   => 16#67FE#,  --  柾 (<CJK Ideograph>)
      4027   => 16#9C52#,  --  鱒 (<CJK Ideograph>)
      4028   => 16#685D#,  --  桝 (<CJK Ideograph>)
      4029   => 16#4EA6#,  --  亦 (<CJK Ideograph>)
      4030   => 16#4FE3#,  --  俣 (<CJK Ideograph>)
      4031   => 16#53C8#,  --  又 (<CJK Ideograph>)
      4032   => 16#62B9#,  --  抹 (<CJK Ideograph>)
      4033   => 16#672B#,  --  末 (<CJK Ideograph>)
      4034   => 16#6CAB#,  --  沫 (<CJK Ideograph>)
      4035   => 16#8FC4#,  --  迄 (<CJK Ideograph>)
      4036   => 16#4FAD#,  --  侭 (<CJK Ideograph>)
      4037   => 16#7E6D#,  --  繭 (<CJK Ideograph>)
      4038   => 16#9EBF#,  --  麿 (<CJK Ideograph>)
      4039   => 16#4E07#,  --  万 (<CJK Ideograph>)
      4040   => 16#6162#,  --  慢 (<CJK Ideograph>)
      4041   => 16#6E80#,  --  満 (<CJK Ideograph>)
      4042   => 16#6F2B#,  --  漫 (<CJK Ideograph>)
      4043   => 16#8513#,  --  蔓 (<CJK Ideograph>)
      4044   => 16#5473#,  --  味 (<CJK Ideograph>)
      4045   => 16#672A#,  --  未 (<CJK Ideograph>)
      4046   => 16#9B45#,  --  魅 (<CJK Ideograph>)
      4047   => 16#5DF3#,  --  巳 (<CJK Ideograph>)
      4048   => 16#7B95#,  --  箕 (<CJK Ideograph>)
      4049   => 16#5CAC#,  --  岬 (<CJK Ideograph>)
      4050   => 16#5BC6#,  --  密 (<CJK Ideograph>)
      4051   => 16#871C#,  --  蜜 (<CJK Ideograph>)
      4052   => 16#6E4A#,  --  湊 (<CJK Ideograph>)
      4053   => 16#84D1#,  --  蓑 (<CJK Ideograph>)
      4054   => 16#7A14#,  --  稔 (<CJK Ideograph>)
      4055   => 16#8108#,  --  脈 (<CJK Ideograph>)
      4056   => 16#5999#,  --  妙 (<CJK Ideograph>)
      4057   => 16#7C8D#,  --  粍 (<CJK Ideograph>)
      4058   => 16#6C11#,  --  民 (<CJK Ideograph>)
      4059   => 16#7720#,  --  眠 (<CJK Ideograph>)
      4060   => 16#52D9#,  --  務 (<CJK Ideograph>)
      4061   => 16#5922#,  --  夢 (<CJK Ideograph>)
      4062   => 16#7121#,  --  無 (<CJK Ideograph>)
      4063   => 16#725F#,  --  牟 (<CJK Ideograph>)
      4064   => 16#77DB#,  --  矛 (<CJK Ideograph>)
      4065   => 16#9727#,  --  霧 (<CJK Ideograph>)
      4066   => 16#9D61#,  --  鵡 (<CJK Ideograph>)
      4067   => 16#690B#,  --  椋 (<CJK Ideograph>)
      4068   => 16#5A7F#,  --  婿 (<CJK Ideograph>)
      4069   => 16#5A18#,  --  娘 (<CJK Ideograph>)
      4070   => 16#51A5#,  --  冥 (<CJK Ideograph>)
      4071   => 16#540D#,  --  名 (<CJK Ideograph>)
      4072   => 16#547D#,  --  命 (<CJK Ideograph>)
      4073   => 16#660E#,  --  明 (<CJK Ideograph>)
      4074   => 16#76DF#,  --  盟 (<CJK Ideograph>)
      4075   => 16#8FF7#,  --  迷 (<CJK Ideograph>)
      4076   => 16#9298#,  --  銘 (<CJK Ideograph>)
      4077   => 16#9CF4#,  --  鳴 (<CJK Ideograph>)
      4078   => 16#59EA#,  --  姪 (<CJK Ideograph>)
      4079   => 16#725D#,  --  牝 (<CJK Ideograph>)
      4080   => 16#6EC5#,  --  滅 (<CJK Ideograph>)
      4081   => 16#514D#,  --  免 (<CJK Ideograph>)
      4082   => 16#68C9#,  --  棉 (<CJK Ideograph>)
      4083   => 16#7DBF#,  --  綿 (<CJK Ideograph>)
      4084   => 16#7DEC#,  --  緬 (<CJK Ideograph>)
      4085   => 16#9762#,  --  面 (<CJK Ideograph>)
      4086   => 16#9EBA#,  --  麺 (<CJK Ideograph>)
      4087   => 16#6478#,  --  摸 (<CJK Ideograph>)
      4088   => 16#6A21#,  --  模 (<CJK Ideograph>)
      4089   => 16#8302#,  --  茂 (<CJK Ideograph>)
      4090   => 16#5984#,  --  妄 (<CJK Ideograph>)
      4091   => 16#5B5F#,  --  孟 (<CJK Ideograph>)
      4092   => 16#6BDB#,  --  毛 (<CJK Ideograph>)
      4093   => 16#731B#,  --  猛 (<CJK Ideograph>)
      4094   => 16#76F2#,  --  盲 (<CJK Ideograph>)
      4095   => 16#7DB2#,  --  網 (<CJK Ideograph>)
      4096   => 16#8017#,  --  耗 (<CJK Ideograph>)
      4097   => 16#8499#,  --  蒙 (<CJK Ideograph>)
      4098   => 16#5132#,  --  儲 (<CJK Ideograph>)
      4099   => 16#6728#,  --  木 (<CJK Ideograph>)
      4100   => 16#9ED9#,  --  黙 (<CJK Ideograph>)
      4101   => 16#76EE#,  --  目 (<CJK Ideograph>)
      4102   => 16#6762#,  --  杢 (<CJK Ideograph>)
      4103   => 16#52FF#,  --  勿 (<CJK Ideograph>)
      4104   => 16#9905#,  --  餅 (<CJK Ideograph>)
      4105   => 16#5C24#,  --  尤 (<CJK Ideograph>)
      4106   => 16#623B#,  --  戻 (<CJK Ideograph>)
      4107   => 16#7C7E#,  --  籾 (<CJK Ideograph>)
      4108   => 16#8CB0#,  --  貰 (<CJK Ideograph>)
      4109   => 16#554F#,  --  問 (<CJK Ideograph>)
      4110   => 16#60B6#,  --  悶 (<CJK Ideograph>)
      4111   => 16#7D0B#,  --  紋 (<CJK Ideograph>)
      4112   => 16#9580#,  --  門 (<CJK Ideograph>)
      4113   => 16#5301#,  --  匁 (<CJK Ideograph>)
      4114   => 16#4E5F#,  --  也 (<CJK Ideograph>)
      4115   => 16#51B6#,  --  冶 (<CJK Ideograph>)
      4116   => 16#591C#,  --  夜 (<CJK Ideograph>)
      4117   => 16#723A#,  --  爺 (<CJK Ideograph>)
      4118   => 16#8036#,  --  耶 (<CJK Ideograph>)
      4119   => 16#91CE#,  --  野 (<CJK Ideograph>)
      4120   => 16#5F25#,  --  弥 (<CJK Ideograph>)
      4121   => 16#77E2#,  --  矢 (<CJK Ideograph>)
      4122   => 16#5384#,  --  厄 (<CJK Ideograph>)
      4123   => 16#5F79#,  --  役 (<CJK Ideograph>)
      4124   => 16#7D04#,  --  約 (<CJK Ideograph>)
      4125   => 16#85AC#,  --  薬 (<CJK Ideograph>)
      4126   => 16#8A33#,  --  訳 (<CJK Ideograph>)
      4127   => 16#8E8D#,  --  躍 (<CJK Ideograph>)
      4128   => 16#9756#,  --  靖 (<CJK Ideograph>)
      4129   => 16#67F3#,  --  柳 (<CJK Ideograph>)
      4130   => 16#85AE#,  --  薮 (<CJK Ideograph>)
      4131   => 16#9453#,  --  鑓 (<CJK Ideograph>)
      4132   => 16#6109#,  --  愉 (<CJK Ideograph>)
      4133   => 16#6108#,  --  愈 (<CJK Ideograph>)
      4134   => 16#6CB9#,  --  油 (<CJK Ideograph>)
      4135   => 16#7652#,  --  癒 (<CJK Ideograph>)
      4136   => 16#8AED#,  --  諭 (<CJK Ideograph>)
      4137   => 16#8F38#,  --  輸 (<CJK Ideograph>)
      4138   => 16#552F#,  --  唯 (<CJK Ideograph>)
      4139   => 16#4F51#,  --  佑 (<CJK Ideograph>)
      4140   => 16#512A#,  --  優 (<CJK Ideograph>)
      4141   => 16#52C7#,  --  勇 (<CJK Ideograph>)
      4142   => 16#53CB#,  --  友 (<CJK Ideograph>)
      4143   => 16#5BA5#,  --  宥 (<CJK Ideograph>)
      4144   => 16#5E7D#,  --  幽 (<CJK Ideograph>)
      4145   => 16#60A0#,  --  悠 (<CJK Ideograph>)
      4146   => 16#6182#,  --  憂 (<CJK Ideograph>)
      4147   => 16#63D6#,  --  揖 (<CJK Ideograph>)
      4148   => 16#6709#,  --  有 (<CJK Ideograph>)
      4149   => 16#67DA#,  --  柚 (<CJK Ideograph>)
      4150   => 16#6E67#,  --  湧 (<CJK Ideograph>)
      4151   => 16#6D8C#,  --  涌 (<CJK Ideograph>)
      4152   => 16#7336#,  --  猶 (<CJK Ideograph>)
      4153   => 16#7337#,  --  猷 (<CJK Ideograph>)
      4154   => 16#7531#,  --  由 (<CJK Ideograph>)
      4155   => 16#7950#,  --  祐 (<CJK Ideograph>)
      4156   => 16#88D5#,  --  裕 (<CJK Ideograph>)
      4157   => 16#8A98#,  --  誘 (<CJK Ideograph>)
      4158   => 16#904A#,  --  遊 (<CJK Ideograph>)
      4159   => 16#9091#,  --  邑 (<CJK Ideograph>)
      4160   => 16#90F5#,  --  郵 (<CJK Ideograph>)
      4161   => 16#96C4#,  --  雄 (<CJK Ideograph>)
      4162   => 16#878D#,  --  融 (<CJK Ideograph>)
      4163   => 16#5915#,  --  夕 (<CJK Ideograph>)
      4164   => 16#4E88#,  --  予 (<CJK Ideograph>)
      4165   => 16#4F59#,  --  余 (<CJK Ideograph>)
      4166   => 16#4E0E#,  --  与 (<CJK Ideograph>)
      4167   => 16#8A89#,  --  誉 (<CJK Ideograph>)
      4168   => 16#8F3F#,  --  輿 (<CJK Ideograph>)
      4169   => 16#9810#,  --  預 (<CJK Ideograph>)
      4170   => 16#50AD#,  --  傭 (<CJK Ideograph>)
      4171   => 16#5E7C#,  --  幼 (<CJK Ideograph>)
      4172   => 16#5996#,  --  妖 (<CJK Ideograph>)
      4173   => 16#5BB9#,  --  容 (<CJK Ideograph>)
      4174   => 16#5EB8#,  --  庸 (<CJK Ideograph>)
      4175   => 16#63DA#,  --  揚 (<CJK Ideograph>)
      4176   => 16#63FA#,  --  揺 (<CJK Ideograph>)
      4177   => 16#64C1#,  --  擁 (<CJK Ideograph>)
      4178   => 16#66DC#,  --  曜 (<CJK Ideograph>)
      4179   => 16#694A#,  --  楊 (<CJK Ideograph>)
      4180   => 16#69D8#,  --  様 (<CJK Ideograph>)
      4181   => 16#6D0B#,  --  洋 (<CJK Ideograph>)
      4182   => 16#6EB6#,  --  溶 (<CJK Ideograph>)
      4183   => 16#7194#,  --  熔 (<CJK Ideograph>)
      4184   => 16#7528#,  --  用 (<CJK Ideograph>)
      4185   => 16#7AAF#,  --  窯 (<CJK Ideograph>)
      4186   => 16#7F8A#,  --  羊 (<CJK Ideograph>)
      4187   => 16#8000#,  --  耀 (<CJK Ideograph>)
      4188   => 16#8449#,  --  葉 (<CJK Ideograph>)
      4189   => 16#84C9#,  --  蓉 (<CJK Ideograph>)
      4190   => 16#8981#,  --  要 (<CJK Ideograph>)
      4191   => 16#8B21#,  --  謡 (<CJK Ideograph>)
      4192   => 16#8E0A#,  --  踊 (<CJK Ideograph>)
      4193   => 16#9065#,  --  遥 (<CJK Ideograph>)
      4194   => 16#967D#,  --  陽 (<CJK Ideograph>)
      4195   => 16#990A#,  --  養 (<CJK Ideograph>)
      4196   => 16#617E#,  --  慾 (<CJK Ideograph>)
      4197   => 16#6291#,  --  抑 (<CJK Ideograph>)
      4198   => 16#6B32#,  --  欲 (<CJK Ideograph>)
      4199   => 16#6C83#,  --  沃 (<CJK Ideograph>)
      4200   => 16#6D74#,  --  浴 (<CJK Ideograph>)
      4201   => 16#7FCC#,  --  翌 (<CJK Ideograph>)
      4202   => 16#7FFC#,  --  翼 (<CJK Ideograph>)
      4203   => 16#6DC0#,  --  淀 (<CJK Ideograph>)
      4204   => 16#7F85#,  --  羅 (<CJK Ideograph>)
      4205   => 16#87BA#,  --  螺 (<CJK Ideograph>)
      4206   => 16#88F8#,  --  裸 (<CJK Ideograph>)
      4207   => 16#6765#,  --  来 (<CJK Ideograph>)
      4208   => 16#83B1#,  --  莱 (<CJK Ideograph>)
      4209   => 16#983C#,  --  頼 (<CJK Ideograph>)
      4210   => 16#96F7#,  --  雷 (<CJK Ideograph>)
      4211   => 16#6D1B#,  --  洛 (<CJK Ideograph>)
      4212   => 16#7D61#,  --  絡 (<CJK Ideograph>)
      4213   => 16#843D#,  --  落 (<CJK Ideograph>)
      4214   => 16#916A#,  --  酪 (<CJK Ideograph>)
      4215   => 16#4E71#,  --  乱 (<CJK Ideograph>)
      4216   => 16#5375#,  --  卵 (<CJK Ideograph>)
      4217   => 16#5D50#,  --  嵐 (<CJK Ideograph>)
      4218   => 16#6B04#,  --  欄 (<CJK Ideograph>)
      4219   => 16#6FEB#,  --  濫 (<CJK Ideograph>)
      4220   => 16#85CD#,  --  藍 (<CJK Ideograph>)
      4221   => 16#862D#,  --  蘭 (<CJK Ideograph>)
      4222   => 16#89A7#,  --  覧 (<CJK Ideograph>)
      4223   => 16#5229#,  --  利 (<CJK Ideograph>)
      4224   => 16#540F#,  --  吏 (<CJK Ideograph>)
      4225   => 16#5C65#,  --  履 (<CJK Ideograph>)
      4226   => 16#674E#,  --  李 (<CJK Ideograph>)
      4227   => 16#68A8#,  --  梨 (<CJK Ideograph>)
      4228   => 16#7406#,  --  理 (<CJK Ideograph>)
      4229   => 16#7483#,  --  璃 (<CJK Ideograph>)
      4230   => 16#75E2#,  --  痢 (<CJK Ideograph>)
      4231   => 16#88CF#,  --  裏 (<CJK Ideograph>)
      4232   => 16#88E1#,  --  裡 (<CJK Ideograph>)
      4233   => 16#91CC#,  --  里 (<CJK Ideograph>)
      4234   => 16#96E2#,  --  離 (<CJK Ideograph>)
      4235   => 16#9678#,  --  陸 (<CJK Ideograph>)
      4236   => 16#5F8B#,  --  律 (<CJK Ideograph>)
      4237   => 16#7387#,  --  率 (<CJK Ideograph>)
      4238   => 16#7ACB#,  --  立 (<CJK Ideograph>)
      4239   => 16#844E#,  --  葎 (<CJK Ideograph>)
      4240   => 16#63A0#,  --  掠 (<CJK Ideograph>)
      4241   => 16#7565#,  --  略 (<CJK Ideograph>)
      4242   => 16#5289#,  --  劉 (<CJK Ideograph>)
      4243   => 16#6D41#,  --  流 (<CJK Ideograph>)
      4244   => 16#6E9C#,  --  溜 (<CJK Ideograph>)
      4245   => 16#7409#,  --  琉 (<CJK Ideograph>)
      4246   => 16#7559#,  --  留 (<CJK Ideograph>)
      4247   => 16#786B#,  --  硫 (<CJK Ideograph>)
      4248   => 16#7C92#,  --  粒 (<CJK Ideograph>)
      4249   => 16#9686#,  --  隆 (<CJK Ideograph>)
      4250   => 16#7ADC#,  --  竜 (<CJK Ideograph>)
      4251   => 16#9F8D#,  --  龍 (<CJK Ideograph>)
      4252   => 16#4FB6#,  --  侶 (<CJK Ideograph>)
      4253   => 16#616E#,  --  慮 (<CJK Ideograph>)
      4254   => 16#65C5#,  --  旅 (<CJK Ideograph>)
      4255   => 16#865C#,  --  虜 (<CJK Ideograph>)
      4256   => 16#4E86#,  --  了 (<CJK Ideograph>)
      4257   => 16#4EAE#,  --  亮 (<CJK Ideograph>)
      4258   => 16#50DA#,  --  僚 (<CJK Ideograph>)
      4259   => 16#4E21#,  --  両 (<CJK Ideograph>)
      4260   => 16#51CC#,  --  凌 (<CJK Ideograph>)
      4261   => 16#5BEE#,  --  寮 (<CJK Ideograph>)
      4262   => 16#6599#,  --  料 (<CJK Ideograph>)
      4263   => 16#6881#,  --  梁 (<CJK Ideograph>)
      4264   => 16#6DBC#,  --  涼 (<CJK Ideograph>)
      4265   => 16#731F#,  --  猟 (<CJK Ideograph>)
      4266   => 16#7642#,  --  療 (<CJK Ideograph>)
      4267   => 16#77AD#,  --  瞭 (<CJK Ideograph>)
      4268   => 16#7A1C#,  --  稜 (<CJK Ideograph>)
      4269   => 16#7CE7#,  --  糧 (<CJK Ideograph>)
      4270   => 16#826F#,  --  良 (<CJK Ideograph>)
      4271   => 16#8AD2#,  --  諒 (<CJK Ideograph>)
      4272   => 16#907C#,  --  遼 (<CJK Ideograph>)
      4273   => 16#91CF#,  --  量 (<CJK Ideograph>)
      4274   => 16#9675#,  --  陵 (<CJK Ideograph>)
      4275   => 16#9818#,  --  領 (<CJK Ideograph>)
      4276   => 16#529B#,  --  力 (<CJK Ideograph>)
      4277   => 16#7DD1#,  --  緑 (<CJK Ideograph>)
      4278   => 16#502B#,  --  倫 (<CJK Ideograph>)
      4279   => 16#5398#,  --  厘 (<CJK Ideograph>)
      4280   => 16#6797#,  --  林 (<CJK Ideograph>)
      4281   => 16#6DCB#,  --  淋 (<CJK Ideograph>)
      4282   => 16#71D0#,  --  燐 (<CJK Ideograph>)
      4283   => 16#7433#,  --  琳 (<CJK Ideograph>)
      4284   => 16#81E8#,  --  臨 (<CJK Ideograph>)
      4285   => 16#8F2A#,  --  輪 (<CJK Ideograph>)
      4286   => 16#96A3#,  --  隣 (<CJK Ideograph>)
      4287   => 16#9C57#,  --  鱗 (<CJK Ideograph>)
      4288   => 16#9E9F#,  --  麟 (<CJK Ideograph>)
      4289   => 16#7460#,  --  瑠 (<CJK Ideograph>)
      4290   => 16#5841#,  --  塁 (<CJK Ideograph>)
      4291   => 16#6D99#,  --  涙 (<CJK Ideograph>)
      4292   => 16#7D2F#,  --  累 (<CJK Ideograph>)
      4293   => 16#985E#,  --  類 (<CJK Ideograph>)
      4294   => 16#4EE4#,  --  令 (<CJK Ideograph>)
      4295   => 16#4F36#,  --  伶 (<CJK Ideograph>)
      4296   => 16#4F8B#,  --  例 (<CJK Ideograph>)
      4297   => 16#51B7#,  --  冷 (<CJK Ideograph>)
      4298   => 16#52B1#,  --  励 (<CJK Ideograph>)
      4299   => 16#5DBA#,  --  嶺 (<CJK Ideograph>)
      4300   => 16#601C#,  --  怜 (<CJK Ideograph>)
      4301   => 16#73B2#,  --  玲 (<CJK Ideograph>)
      4302   => 16#793C#,  --  礼 (<CJK Ideograph>)
      4303   => 16#82D3#,  --  苓 (<CJK Ideograph>)
      4304   => 16#9234#,  --  鈴 (<CJK Ideograph>)
      4305   => 16#96B7#,  --  隷 (<CJK Ideograph>)
      4306   => 16#96F6#,  --  零 (<CJK Ideograph>)
      4307   => 16#970A#,  --  霊 (<CJK Ideograph>)
      4308   => 16#9E97#,  --  麗 (<CJK Ideograph>)
      4309   => 16#9F62#,  --  齢 (<CJK Ideograph>)
      4310   => 16#66A6#,  --  暦 (<CJK Ideograph>)
      4311   => 16#6B74#,  --  歴 (<CJK Ideograph>)
      4312   => 16#5217#,  --  列 (<CJK Ideograph>)
      4313   => 16#52A3#,  --  劣 (<CJK Ideograph>)
      4314   => 16#70C8#,  --  烈 (<CJK Ideograph>)
      4315   => 16#88C2#,  --  裂 (<CJK Ideograph>)
      4316   => 16#5EC9#,  --  廉 (<CJK Ideograph>)
      4317   => 16#604B#,  --  恋 (<CJK Ideograph>)
      4318   => 16#6190#,  --  憐 (<CJK Ideograph>)
      4319   => 16#6F23#,  --  漣 (<CJK Ideograph>)
      4320   => 16#7149#,  --  煉 (<CJK Ideograph>)
      4321   => 16#7C3E#,  --  簾 (<CJK Ideograph>)
      4322   => 16#7DF4#,  --  練 (<CJK Ideograph>)
      4323   => 16#806F#,  --  聯 (<CJK Ideograph>)
      4324   => 16#84EE#,  --  蓮 (<CJK Ideograph>)
      4325   => 16#9023#,  --  連 (<CJK Ideograph>)
      4326   => 16#932C#,  --  錬 (<CJK Ideograph>)
      4327   => 16#5442#,  --  呂 (<CJK Ideograph>)
      4328   => 16#9B6F#,  --  魯 (<CJK Ideograph>)
      4329   => 16#6AD3#,  --  櫓 (<CJK Ideograph>)
      4330   => 16#7089#,  --  炉 (<CJK Ideograph>)
      4331   => 16#8CC2#,  --  賂 (<CJK Ideograph>)
      4332   => 16#8DEF#,  --  路 (<CJK Ideograph>)
      4333   => 16#9732#,  --  露 (<CJK Ideograph>)
      4334   => 16#52B4#,  --  労 (<CJK Ideograph>)
      4335   => 16#5A41#,  --  婁 (<CJK Ideograph>)
      4336   => 16#5ECA#,  --  廊 (<CJK Ideograph>)
      4337   => 16#5F04#,  --  弄 (<CJK Ideograph>)
      4338   => 16#6717#,  --  朗 (<CJK Ideograph>)
      4339   => 16#697C#,  --  楼 (<CJK Ideograph>)
      4340   => 16#6994#,  --  榔 (<CJK Ideograph>)
      4341   => 16#6D6A#,  --  浪 (<CJK Ideograph>)
      4342   => 16#6F0F#,  --  漏 (<CJK Ideograph>)
      4343   => 16#7262#,  --  牢 (<CJK Ideograph>)
      4344   => 16#72FC#,  --  狼 (<CJK Ideograph>)
      4345   => 16#7BED#,  --  篭 (<CJK Ideograph>)
      4346   => 16#8001#,  --  老 (<CJK Ideograph>)
      4347   => 16#807E#,  --  聾 (<CJK Ideograph>)
      4348   => 16#874B#,  --  蝋 (<CJK Ideograph>)
      4349   => 16#90CE#,  --  郎 (<CJK Ideograph>)
      4350   => 16#516D#,  --  六 (<CJK Ideograph>)
      4351   => 16#9E93#,  --  麓 (<CJK Ideograph>)
      4352   => 16#7984#,  --  禄 (<CJK Ideograph>)
      4353   => 16#808B#,  --  肋 (<CJK Ideograph>)
      4354   => 16#9332#,  --  録 (<CJK Ideograph>)
      4355   => 16#8AD6#,  --  論 (<CJK Ideograph>)
      4356   => 16#502D#,  --  倭 (<CJK Ideograph>)
      4357   => 16#548C#,  --  和 (<CJK Ideograph>)
      4358   => 16#8A71#,  --  話 (<CJK Ideograph>)
      4359   => 16#6B6A#,  --  歪 (<CJK Ideograph>)
      4360   => 16#8CC4#,  --  賄 (<CJK Ideograph>)
      4361   => 16#8107#,  --  脇 (<CJK Ideograph>)
      4362   => 16#60D1#,  --  惑 (<CJK Ideograph>)
      4363   => 16#67A0#,  --  枠 (<CJK Ideograph>)
      4364   => 16#9DF2#,  --  鷲 (<CJK Ideograph>)
      4365   => 16#4E99#,  --  亙 (<CJK Ideograph>)
      4366   => 16#4E98#,  --  亘 (<CJK Ideograph>)
      4367   => 16#9C10#,  --  鰐 (<CJK Ideograph>)
      4368   => 16#8A6B#,  --  詫 (<CJK Ideograph>)
      4369   => 16#85C1#,  --  藁 (<CJK Ideograph>)
      4370   => 16#8568#,  --  蕨 (<CJK Ideograph>)
      4371   => 16#6900#,  --  椀 (<CJK Ideograph>)
      4372   => 16#6E7E#,  --  湾 (<CJK Ideograph>)
      4373   => 16#7897#,  --  碗 (<CJK Ideograph>)
      4374   => 16#8155#,  --  腕 (<CJK Ideograph>)
      4418   => 16#5F0C#,  --  弌 (<CJK Ideograph>)
      4419   => 16#4E10#,  --  丐 (<CJK Ideograph>)
      4420   => 16#4E15#,  --  丕 (<CJK Ideograph>)
      4421   => 16#4E2A#,  --  个 (<CJK Ideograph>)
      4422   => 16#4E31#,  --  丱 (<CJK Ideograph>)
      4423   => 16#4E36#,  --  丶 (<CJK Ideograph>)
      4424   => 16#4E3C#,  --  丼 (<CJK Ideograph>)
      4425   => 16#4E3F#,  --  丿 (<CJK Ideograph>)
      4426   => 16#4E42#,  --  乂 (<CJK Ideograph>)
      4427   => 16#4E56#,  --  乖 (<CJK Ideograph>)
      4428   => 16#4E58#,  --  乘 (<CJK Ideograph>)
      4429   => 16#4E82#,  --  亂 (<CJK Ideograph>)
      4430   => 16#4E85#,  --  亅 (<CJK Ideograph>)
      4431   => 16#8C6B#,  --  豫 (<CJK Ideograph>)
      4432   => 16#4E8A#,  --  亊 (<CJK Ideograph>)
      4433   => 16#8212#,  --  舒 (<CJK Ideograph>)
      4434   => 16#5F0D#,  --  弍 (<CJK Ideograph>)
      4435   => 16#4E8E#,  --  于 (<CJK Ideograph>)
      4436   => 16#4E9E#,  --  亞 (<CJK Ideograph>)
      4437   => 16#4E9F#,  --  亟 (<CJK Ideograph>)
      4438   => 16#4EA0#,  --  亠 (<CJK Ideograph>)
      4439   => 16#4EA2#,  --  亢 (<CJK Ideograph>)
      4440   => 16#4EB0#,  --  亰 (<CJK Ideograph>)
      4441   => 16#4EB3#,  --  亳 (<CJK Ideograph>)
      4442   => 16#4EB6#,  --  亶 (<CJK Ideograph>)
      4443   => 16#4ECE#,  --  从 (<CJK Ideograph>)
      4444   => 16#4ECD#,  --  仍 (<CJK Ideograph>)
      4445   => 16#4EC4#,  --  仄 (<CJK Ideograph>)
      4446   => 16#4EC6#,  --  仆 (<CJK Ideograph>)
      4447   => 16#4EC2#,  --  仂 (<CJK Ideograph>)
      4448   => 16#4ED7#,  --  仗 (<CJK Ideograph>)
      4449   => 16#4EDE#,  --  仞 (<CJK Ideograph>)
      4450   => 16#4EED#,  --  仭 (<CJK Ideograph>)
      4451   => 16#4EDF#,  --  仟 (<CJK Ideograph>)
      4452   => 16#4EF7#,  --  价 (<CJK Ideograph>)
      4453   => 16#4F09#,  --  伉 (<CJK Ideograph>)
      4454   => 16#4F5A#,  --  佚 (<CJK Ideograph>)
      4455   => 16#4F30#,  --  估 (<CJK Ideograph>)
      4456   => 16#4F5B#,  --  佛 (<CJK Ideograph>)
      4457   => 16#4F5D#,  --  佝 (<CJK Ideograph>)
      4458   => 16#4F57#,  --  佗 (<CJK Ideograph>)
      4459   => 16#4F47#,  --  佇 (<CJK Ideograph>)
      4460   => 16#4F76#,  --  佶 (<CJK Ideograph>)
      4461   => 16#4F88#,  --  侈 (<CJK Ideograph>)
      4462   => 16#4F8F#,  --  侏 (<CJK Ideograph>)
      4463   => 16#4F98#,  --  侘 (<CJK Ideograph>)
      4464   => 16#4F7B#,  --  佻 (<CJK Ideograph>)
      4465   => 16#4F69#,  --  佩 (<CJK Ideograph>)
      4466   => 16#4F70#,  --  佰 (<CJK Ideograph>)
      4467   => 16#4F91#,  --  侑 (<CJK Ideograph>)
      4468   => 16#4F6F#,  --  佯 (<CJK Ideograph>)
      4469   => 16#4F86#,  --  來 (<CJK Ideograph>)
      4470   => 16#4F96#,  --  侖 (<CJK Ideograph>)
      4471   => 16#5118#,  --  儘 (<CJK Ideograph>)
      4472   => 16#4FD4#,  --  俔 (<CJK Ideograph>)
      4473   => 16#4FDF#,  --  俟 (<CJK Ideograph>)
      4474   => 16#4FCE#,  --  俎 (<CJK Ideograph>)
      4475   => 16#4FD8#,  --  俘 (<CJK Ideograph>)
      4476   => 16#4FDB#,  --  俛 (<CJK Ideograph>)
      4477   => 16#4FD1#,  --  俑 (<CJK Ideograph>)
      4478   => 16#4FDA#,  --  俚 (<CJK Ideograph>)
      4479   => 16#4FD0#,  --  俐 (<CJK Ideograph>)
      4480   => 16#4FE4#,  --  俤 (<CJK Ideograph>)
      4481   => 16#4FE5#,  --  俥 (<CJK Ideograph>)
      4482   => 16#501A#,  --  倚 (<CJK Ideograph>)
      4483   => 16#5028#,  --  倨 (<CJK Ideograph>)
      4484   => 16#5014#,  --  倔 (<CJK Ideograph>)
      4485   => 16#502A#,  --  倪 (<CJK Ideograph>)
      4486   => 16#5025#,  --  倥 (<CJK Ideograph>)
      4487   => 16#5005#,  --  倅 (<CJK Ideograph>)
      4488   => 16#4F1C#,  --  伜 (<CJK Ideograph>)
      4489   => 16#4FF6#,  --  俶 (<CJK Ideograph>)
      4490   => 16#5021#,  --  倡 (<CJK Ideograph>)
      4491   => 16#5029#,  --  倩 (<CJK Ideograph>)
      4492   => 16#502C#,  --  倬 (<CJK Ideograph>)
      4493   => 16#4FFE#,  --  俾 (<CJK Ideograph>)
      4494   => 16#4FEF#,  --  俯 (<CJK Ideograph>)
      4495   => 16#5011#,  --  們 (<CJK Ideograph>)
      4496   => 16#5006#,  --  倆 (<CJK Ideograph>)
      4497   => 16#5043#,  --  偃 (<CJK Ideograph>)
      4498   => 16#5047#,  --  假 (<CJK Ideograph>)
      4499   => 16#6703#,  --  會 (<CJK Ideograph>)
      4500   => 16#5055#,  --  偕 (<CJK Ideograph>)
      4501   => 16#5050#,  --  偐 (<CJK Ideograph>)
      4502   => 16#5048#,  --  偈 (<CJK Ideograph>)
      4503   => 16#505A#,  --  做 (<CJK Ideograph>)
      4504   => 16#5056#,  --  偖 (<CJK Ideograph>)
      4505   => 16#506C#,  --  偬 (<CJK Ideograph>)
      4506   => 16#5078#,  --  偸 (<CJK Ideograph>)
      4507   => 16#5080#,  --  傀 (<CJK Ideograph>)
      4508   => 16#509A#,  --  傚 (<CJK Ideograph>)
      4509   => 16#5085#,  --  傅 (<CJK Ideograph>)
      4510   => 16#50B4#,  --  傴 (<CJK Ideograph>)
      4511   => 16#50B2#,  --  傲 (<CJK Ideograph>)
      4512   => 16#50C9#,  --  僉 (<CJK Ideograph>)
      4513   => 16#50CA#,  --  僊 (<CJK Ideograph>)
      4514   => 16#50B3#,  --  傳 (<CJK Ideograph>)
      4515   => 16#50C2#,  --  僂 (<CJK Ideograph>)
      4516   => 16#50D6#,  --  僖 (<CJK Ideograph>)
      4517   => 16#50DE#,  --  僞 (<CJK Ideograph>)
      4518   => 16#50E5#,  --  僥 (<CJK Ideograph>)
      4519   => 16#50ED#,  --  僭 (<CJK Ideograph>)
      4520   => 16#50E3#,  --  僣 (<CJK Ideograph>)
      4521   => 16#50EE#,  --  僮 (<CJK Ideograph>)
      4522   => 16#50F9#,  --  價 (<CJK Ideograph>)
      4523   => 16#50F5#,  --  僵 (<CJK Ideograph>)
      4524   => 16#5109#,  --  儉 (<CJK Ideograph>)
      4525   => 16#5101#,  --  儁 (<CJK Ideograph>)
      4526   => 16#5102#,  --  儂 (<CJK Ideograph>)
      4527   => 16#5116#,  --  儖 (<CJK Ideograph>)
      4528   => 16#5115#,  --  儕 (<CJK Ideograph>)
      4529   => 16#5114#,  --  儔 (<CJK Ideograph>)
      4530   => 16#511A#,  --  儚 (<CJK Ideograph>)
      4531   => 16#5121#,  --  儡 (<CJK Ideograph>)
      4532   => 16#513A#,  --  儺 (<CJK Ideograph>)
      4533   => 16#5137#,  --  儷 (<CJK Ideograph>)
      4534   => 16#513C#,  --  儼 (<CJK Ideograph>)
      4535   => 16#513B#,  --  儻 (<CJK Ideograph>)
      4536   => 16#513F#,  --  儿 (<CJK Ideograph>)
      4537   => 16#5140#,  --  兀 (<CJK Ideograph>)
      4538   => 16#5152#,  --  兒 (<CJK Ideograph>)
      4539   => 16#514C#,  --  兌 (<CJK Ideograph>)
      4540   => 16#5154#,  --  兔 (<CJK Ideograph>)
      4541   => 16#5162#,  --  兢 (<CJK Ideograph>)
      4542   => 16#7AF8#,  --  竸 (<CJK Ideograph>)
      4543   => 16#5169#,  --  兩 (<CJK Ideograph>)
      4544   => 16#516A#,  --  兪 (<CJK Ideograph>)
      4545   => 16#516E#,  --  兮 (<CJK Ideograph>)
      4546   => 16#5180#,  --  冀 (<CJK Ideograph>)
      4547   => 16#5182#,  --  冂 (<CJK Ideograph>)
      4548   => 16#56D8#,  --  囘 (<CJK Ideograph>)
      4549   => 16#518C#,  --  册 (<CJK Ideograph>)
      4550   => 16#5189#,  --  冉 (<CJK Ideograph>)
      4551   => 16#518F#,  --  冏 (<CJK Ideograph>)
      4552   => 16#5191#,  --  冑 (<CJK Ideograph>)
      4553   => 16#5193#,  --  冓 (<CJK Ideograph>)
      4554   => 16#5195#,  --  冕 (<CJK Ideograph>)
      4555   => 16#5196#,  --  冖 (<CJK Ideograph>)
      4556   => 16#51A4#,  --  冤 (<CJK Ideograph>)
      4557   => 16#51A6#,  --  冦 (<CJK Ideograph>)
      4558   => 16#51A2#,  --  冢 (<CJK Ideograph>)
      4559   => 16#51A9#,  --  冩 (<CJK Ideograph>)
      4560   => 16#51AA#,  --  冪 (<CJK Ideograph>)
      4561   => 16#51AB#,  --  冫 (<CJK Ideograph>)
      4562   => 16#51B3#,  --  决 (<CJK Ideograph>)
      4563   => 16#51B1#,  --  冱 (<CJK Ideograph>)
      4564   => 16#51B2#,  --  冲 (<CJK Ideograph>)
      4565   => 16#51B0#,  --  冰 (<CJK Ideograph>)
      4566   => 16#51B5#,  --  况 (<CJK Ideograph>)
      4567   => 16#51BD#,  --  冽 (<CJK Ideograph>)
      4568   => 16#51C5#,  --  凅 (<CJK Ideograph>)
      4569   => 16#51C9#,  --  凉 (<CJK Ideograph>)
      4570   => 16#51DB#,  --  凛 (<CJK Ideograph>)
      4571   => 16#51E0#,  --  几 (<CJK Ideograph>)
      4572   => 16#8655#,  --  處 (<CJK Ideograph>)
      4573   => 16#51E9#,  --  凩 (<CJK Ideograph>)
      4574   => 16#51ED#,  --  凭 (<CJK Ideograph>)
      4575   => 16#51F0#,  --  凰 (<CJK Ideograph>)
      4576   => 16#51F5#,  --  凵 (<CJK Ideograph>)
      4577   => 16#51FE#,  --  凾 (<CJK Ideograph>)
      4578   => 16#5204#,  --  刄 (<CJK Ideograph>)
      4579   => 16#520B#,  --  刋 (<CJK Ideograph>)
      4580   => 16#5214#,  --  刔 (<CJK Ideograph>)
      4581   => 16#520E#,  --  刎 (<CJK Ideograph>)
      4582   => 16#5227#,  --  刧 (<CJK Ideograph>)
      4583   => 16#522A#,  --  刪 (<CJK Ideograph>)
      4584   => 16#522E#,  --  刮 (<CJK Ideograph>)
      4585   => 16#5233#,  --  刳 (<CJK Ideograph>)
      4586   => 16#5239#,  --  刹 (<CJK Ideograph>)
      4587   => 16#524F#,  --  剏 (<CJK Ideograph>)
      4588   => 16#5244#,  --  剄 (<CJK Ideograph>)
      4589   => 16#524B#,  --  剋 (<CJK Ideograph>)
      4590   => 16#524C#,  --  剌 (<CJK Ideograph>)
      4591   => 16#525E#,  --  剞 (<CJK Ideograph>)
      4592   => 16#5254#,  --  剔 (<CJK Ideograph>)
      4593   => 16#526A#,  --  剪 (<CJK Ideograph>)
      4594   => 16#5274#,  --  剴 (<CJK Ideograph>)
      4595   => 16#5269#,  --  剩 (<CJK Ideograph>)
      4596   => 16#5273#,  --  剳 (<CJK Ideograph>)
      4597   => 16#527F#,  --  剿 (<CJK Ideograph>)
      4598   => 16#527D#,  --  剽 (<CJK Ideograph>)
      4599   => 16#528D#,  --  劍 (<CJK Ideograph>)
      4600   => 16#5294#,  --  劔 (<CJK Ideograph>)
      4601   => 16#5292#,  --  劒 (<CJK Ideograph>)
      4602   => 16#5271#,  --  剱 (<CJK Ideograph>)
      4603   => 16#5288#,  --  劈 (<CJK Ideograph>)
      4604   => 16#5291#,  --  劑 (<CJK Ideograph>)
      4605   => 16#8FA8#,  --  辨 (<CJK Ideograph>)
      4606   => 16#8FA7#,  --  辧 (<CJK Ideograph>)
      4607   => 16#52AC#,  --  劬 (<CJK Ideograph>)
      4608   => 16#52AD#,  --  劭 (<CJK Ideograph>)
      4609   => 16#52BC#,  --  劼 (<CJK Ideograph>)
      4610   => 16#52B5#,  --  劵 (<CJK Ideograph>)
      4611   => 16#52C1#,  --  勁 (<CJK Ideograph>)
      4612   => 16#52CD#,  --  勍 (<CJK Ideograph>)
      4613   => 16#52D7#,  --  勗 (<CJK Ideograph>)
      4614   => 16#52DE#,  --  勞 (<CJK Ideograph>)
      4615   => 16#52E3#,  --  勣 (<CJK Ideograph>)
      4616   => 16#52E6#,  --  勦 (<CJK Ideograph>)
      4617   => 16#98ED#,  --  飭 (<CJK Ideograph>)
      4618   => 16#52E0#,  --  勠 (<CJK Ideograph>)
      4619   => 16#52F3#,  --  勳 (<CJK Ideograph>)
      4620   => 16#52F5#,  --  勵 (<CJK Ideograph>)
      4621   => 16#52F8#,  --  勸 (<CJK Ideograph>)
      4622   => 16#52F9#,  --  勹 (<CJK Ideograph>)
      4623   => 16#5306#,  --  匆 (<CJK Ideograph>)
      4624   => 16#5308#,  --  匈 (<CJK Ideograph>)
      4625   => 16#7538#,  --  甸 (<CJK Ideograph>)
      4626   => 16#530D#,  --  匍 (<CJK Ideograph>)
      4627   => 16#5310#,  --  匐 (<CJK Ideograph>)
      4628   => 16#530F#,  --  匏 (<CJK Ideograph>)
      4629   => 16#5315#,  --  匕 (<CJK Ideograph>)
      4630   => 16#531A#,  --  匚 (<CJK Ideograph>)
      4631   => 16#5323#,  --  匣 (<CJK Ideograph>)
      4632   => 16#532F#,  --  匯 (<CJK Ideograph>)
      4633   => 16#5331#,  --  匱 (<CJK Ideograph>)
      4634   => 16#5333#,  --  匳 (<CJK Ideograph>)
      4635   => 16#5338#,  --  匸 (<CJK Ideograph>)
      4636   => 16#5340#,  --  區 (<CJK Ideograph>)
      4637   => 16#5346#,  --  卆 (<CJK Ideograph>)
      4638   => 16#5345#,  --  卅 (<CJK Ideograph>)
      4639   => 16#4E17#,  --  丗 (<CJK Ideograph>)
      4640   => 16#5349#,  --  卉 (<CJK Ideograph>)
      4641   => 16#534D#,  --  卍 (<CJK Ideograph>)
      4642   => 16#51D6#,  --  凖 (<CJK Ideograph>)
      4643   => 16#535E#,  --  卞 (<CJK Ideograph>)
      4644   => 16#5369#,  --  卩 (<CJK Ideograph>)
      4645   => 16#536E#,  --  卮 (<CJK Ideograph>)
      4646   => 16#5918#,  --  夘 (<CJK Ideograph>)
      4647   => 16#537B#,  --  卻 (<CJK Ideograph>)
      4648   => 16#5377#,  --  卷 (<CJK Ideograph>)
      4649   => 16#5382#,  --  厂 (<CJK Ideograph>)
      4650   => 16#5396#,  --  厖 (<CJK Ideograph>)
      4651   => 16#53A0#,  --  厠 (<CJK Ideograph>)
      4652   => 16#53A6#,  --  厦 (<CJK Ideograph>)
      4653   => 16#53A5#,  --  厥 (<CJK Ideograph>)
      4654   => 16#53AE#,  --  厮 (<CJK Ideograph>)
      4655   => 16#53B0#,  --  厰 (<CJK Ideograph>)
      4656   => 16#53B6#,  --  厶 (<CJK Ideograph>)
      4657   => 16#53C3#,  --  參 (<CJK Ideograph>)
      4658   => 16#7C12#,  --  簒 (<CJK Ideograph>)
      4659   => 16#96D9#,  --  雙 (<CJK Ideograph>)
      4660   => 16#53DF#,  --  叟 (<CJK Ideograph>)
      4661   => 16#66FC#,  --  曼 (<CJK Ideograph>)
      4662   => 16#71EE#,  --  燮 (<CJK Ideograph>)
      4663   => 16#53EE#,  --  叮 (<CJK Ideograph>)
      4664   => 16#53E8#,  --  叨 (<CJK Ideograph>)
      4665   => 16#53ED#,  --  叭 (<CJK Ideograph>)
      4666   => 16#53FA#,  --  叺 (<CJK Ideograph>)
      4667   => 16#5401#,  --  吁 (<CJK Ideograph>)
      4668   => 16#543D#,  --  吽 (<CJK Ideograph>)
      4669   => 16#5440#,  --  呀 (<CJK Ideograph>)
      4670   => 16#542C#,  --  听 (<CJK Ideograph>)
      4671   => 16#542D#,  --  吭 (<CJK Ideograph>)
      4672   => 16#543C#,  --  吼 (<CJK Ideograph>)
      4673   => 16#542E#,  --  吮 (<CJK Ideograph>)
      4674   => 16#5436#,  --  吶 (<CJK Ideograph>)
      4675   => 16#5429#,  --  吩 (<CJK Ideograph>)
      4676   => 16#541D#,  --  吝 (<CJK Ideograph>)
      4677   => 16#544E#,  --  呎 (<CJK Ideograph>)
      4678   => 16#548F#,  --  咏 (<CJK Ideograph>)
      4679   => 16#5475#,  --  呵 (<CJK Ideograph>)
      4680   => 16#548E#,  --  咎 (<CJK Ideograph>)
      4681   => 16#545F#,  --  呟 (<CJK Ideograph>)
      4682   => 16#5471#,  --  呱 (<CJK Ideograph>)
      4683   => 16#5477#,  --  呷 (<CJK Ideograph>)
      4684   => 16#5470#,  --  呰 (<CJK Ideograph>)
      4685   => 16#5492#,  --  咒 (<CJK Ideograph>)
      4686   => 16#547B#,  --  呻 (<CJK Ideograph>)
      4687   => 16#5480#,  --  咀 (<CJK Ideograph>)
      4688   => 16#5476#,  --  呶 (<CJK Ideograph>)
      4689   => 16#5484#,  --  咄 (<CJK Ideograph>)
      4690   => 16#5490#,  --  咐 (<CJK Ideograph>)
      4691   => 16#5486#,  --  咆 (<CJK Ideograph>)
      4692   => 16#54C7#,  --  哇 (<CJK Ideograph>)
      4693   => 16#54A2#,  --  咢 (<CJK Ideograph>)
      4694   => 16#54B8#,  --  咸 (<CJK Ideograph>)
      4695   => 16#54A5#,  --  咥 (<CJK Ideograph>)
      4696   => 16#54AC#,  --  咬 (<CJK Ideograph>)
      4697   => 16#54C4#,  --  哄 (<CJK Ideograph>)
      4698   => 16#54C8#,  --  哈 (<CJK Ideograph>)
      4699   => 16#54A8#,  --  咨 (<CJK Ideograph>)
      4700   => 16#54AB#,  --  咫 (<CJK Ideograph>)
      4701   => 16#54C2#,  --  哂 (<CJK Ideograph>)
      4702   => 16#54A4#,  --  咤 (<CJK Ideograph>)
      4703   => 16#54BE#,  --  咾 (<CJK Ideograph>)
      4704   => 16#54BC#,  --  咼 (<CJK Ideograph>)
      4705   => 16#54D8#,  --  哘 (<CJK Ideograph>)
      4706   => 16#54E5#,  --  哥 (<CJK Ideograph>)
      4707   => 16#54E6#,  --  哦 (<CJK Ideograph>)
      4708   => 16#550F#,  --  唏 (<CJK Ideograph>)
      4709   => 16#5514#,  --  唔 (<CJK Ideograph>)
      4710   => 16#54FD#,  --  哽 (<CJK Ideograph>)
      4711   => 16#54EE#,  --  哮 (<CJK Ideograph>)
      4712   => 16#54ED#,  --  哭 (<CJK Ideograph>)
      4713   => 16#54FA#,  --  哺 (<CJK Ideograph>)
      4714   => 16#54E2#,  --  哢 (<CJK Ideograph>)
      4715   => 16#5539#,  --  唹 (<CJK Ideograph>)
      4716   => 16#5540#,  --  啀 (<CJK Ideograph>)
      4717   => 16#5563#,  --  啣 (<CJK Ideograph>)
      4718   => 16#554C#,  --  啌 (<CJK Ideograph>)
      4719   => 16#552E#,  --  售 (<CJK Ideograph>)
      4720   => 16#555C#,  --  啜 (<CJK Ideograph>)
      4721   => 16#5545#,  --  啅 (<CJK Ideograph>)
      4722   => 16#5556#,  --  啖 (<CJK Ideograph>)
      4723   => 16#5557#,  --  啗 (<CJK Ideograph>)
      4724   => 16#5538#,  --  唸 (<CJK Ideograph>)
      4725   => 16#5533#,  --  唳 (<CJK Ideograph>)
      4726   => 16#555D#,  --  啝 (<CJK Ideograph>)
      4727   => 16#5599#,  --  喙 (<CJK Ideograph>)
      4728   => 16#5580#,  --  喀 (<CJK Ideograph>)
      4729   => 16#54AF#,  --  咯 (<CJK Ideograph>)
      4730   => 16#558A#,  --  喊 (<CJK Ideograph>)
      4731   => 16#559F#,  --  喟 (<CJK Ideograph>)
      4732   => 16#557B#,  --  啻 (<CJK Ideograph>)
      4733   => 16#557E#,  --  啾 (<CJK Ideograph>)
      4734   => 16#5598#,  --  喘 (<CJK Ideograph>)
      4735   => 16#559E#,  --  喞 (<CJK Ideograph>)
      4736   => 16#55AE#,  --  單 (<CJK Ideograph>)
      4737   => 16#557C#,  --  啼 (<CJK Ideograph>)
      4738   => 16#5583#,  --  喃 (<CJK Ideograph>)
      4739   => 16#55A9#,  --  喩 (<CJK Ideograph>)
      4740   => 16#5587#,  --  喇 (<CJK Ideograph>)
      4741   => 16#55A8#,  --  喨 (<CJK Ideograph>)
      4742   => 16#55DA#,  --  嗚 (<CJK Ideograph>)
      4743   => 16#55C5#,  --  嗅 (<CJK Ideograph>)
      4744   => 16#55DF#,  --  嗟 (<CJK Ideograph>)
      4745   => 16#55C4#,  --  嗄 (<CJK Ideograph>)
      4746   => 16#55DC#,  --  嗜 (<CJK Ideograph>)
      4747   => 16#55E4#,  --  嗤 (<CJK Ideograph>)
      4748   => 16#55D4#,  --  嗔 (<CJK Ideograph>)
      4749   => 16#5614#,  --  嘔 (<CJK Ideograph>)
      4750   => 16#55F7#,  --  嗷 (<CJK Ideograph>)
      4751   => 16#5616#,  --  嘖 (<CJK Ideograph>)
      4752   => 16#55FE#,  --  嗾 (<CJK Ideograph>)
      4753   => 16#55FD#,  --  嗽 (<CJK Ideograph>)
      4754   => 16#561B#,  --  嘛 (<CJK Ideograph>)
      4755   => 16#55F9#,  --  嗹 (<CJK Ideograph>)
      4756   => 16#564E#,  --  噎 (<CJK Ideograph>)
      4757   => 16#5650#,  --  噐 (<CJK Ideograph>)
      4758   => 16#71DF#,  --  營 (<CJK Ideograph>)
      4759   => 16#5634#,  --  嘴 (<CJK Ideograph>)
      4760   => 16#5636#,  --  嘶 (<CJK Ideograph>)
      4761   => 16#5632#,  --  嘲 (<CJK Ideograph>)
      4762   => 16#5638#,  --  嘸 (<CJK Ideograph>)
      4763   => 16#566B#,  --  噫 (<CJK Ideograph>)
      4764   => 16#5664#,  --  噤 (<CJK Ideograph>)
      4765   => 16#562F#,  --  嘯 (<CJK Ideograph>)
      4766   => 16#566C#,  --  噬 (<CJK Ideograph>)
      4767   => 16#566A#,  --  噪 (<CJK Ideograph>)
      4768   => 16#5686#,  --  嚆 (<CJK Ideograph>)
      4769   => 16#5680#,  --  嚀 (<CJK Ideograph>)
      4770   => 16#568A#,  --  嚊 (<CJK Ideograph>)
      4771   => 16#56A0#,  --  嚠 (<CJK Ideograph>)
      4772   => 16#5694#,  --  嚔 (<CJK Ideograph>)
      4773   => 16#568F#,  --  嚏 (<CJK Ideograph>)
      4774   => 16#56A5#,  --  嚥 (<CJK Ideograph>)
      4775   => 16#56AE#,  --  嚮 (<CJK Ideograph>)
      4776   => 16#56B6#,  --  嚶 (<CJK Ideograph>)
      4777   => 16#56B4#,  --  嚴 (<CJK Ideograph>)
      4778   => 16#56C2#,  --  囂 (<CJK Ideograph>)
      4779   => 16#56BC#,  --  嚼 (<CJK Ideograph>)
      4780   => 16#56C1#,  --  囁 (<CJK Ideograph>)
      4781   => 16#56C3#,  --  囃 (<CJK Ideograph>)
      4782   => 16#56C0#,  --  囀 (<CJK Ideograph>)
      4783   => 16#56C8#,  --  囈 (<CJK Ideograph>)
      4784   => 16#56CE#,  --  囎 (<CJK Ideograph>)
      4785   => 16#56D1#,  --  囑 (<CJK Ideograph>)
      4786   => 16#56D3#,  --  囓 (<CJK Ideograph>)
      4787   => 16#56D7#,  --  囗 (<CJK Ideograph>)
      4788   => 16#56EE#,  --  囮 (<CJK Ideograph>)
      4789   => 16#56F9#,  --  囹 (<CJK Ideograph>)
      4790   => 16#5700#,  --  圀 (<CJK Ideograph>)
      4791   => 16#56FF#,  --  囿 (<CJK Ideograph>)
      4792   => 16#5704#,  --  圄 (<CJK Ideograph>)
      4793   => 16#5709#,  --  圉 (<CJK Ideograph>)
      4794   => 16#5708#,  --  圈 (<CJK Ideograph>)
      4795   => 16#570B#,  --  國 (<CJK Ideograph>)
      4796   => 16#570D#,  --  圍 (<CJK Ideograph>)
      4797   => 16#5713#,  --  圓 (<CJK Ideograph>)
      4798   => 16#5718#,  --  團 (<CJK Ideograph>)
      4799   => 16#5716#,  --  圖 (<CJK Ideograph>)
      4800   => 16#55C7#,  --  嗇 (<CJK Ideograph>)
      4801   => 16#571C#,  --  圜 (<CJK Ideograph>)
      4802   => 16#5726#,  --  圦 (<CJK Ideograph>)
      4803   => 16#5737#,  --  圷 (<CJK Ideograph>)
      4804   => 16#5738#,  --  圸 (<CJK Ideograph>)
      4805   => 16#574E#,  --  坎 (<CJK Ideograph>)
      4806   => 16#573B#,  --  圻 (<CJK Ideograph>)
      4807   => 16#5740#,  --  址 (<CJK Ideograph>)
      4808   => 16#574F#,  --  坏 (<CJK Ideograph>)
      4809   => 16#5769#,  --  坩 (<CJK Ideograph>)
      4810   => 16#57C0#,  --  埀 (<CJK Ideograph>)
      4811   => 16#5788#,  --  垈 (<CJK Ideograph>)
      4812   => 16#5761#,  --  坡 (<CJK Ideograph>)
      4813   => 16#577F#,  --  坿 (<CJK Ideograph>)
      4814   => 16#5789#,  --  垉 (<CJK Ideograph>)
      4815   => 16#5793#,  --  垓 (<CJK Ideograph>)
      4816   => 16#57A0#,  --  垠 (<CJK Ideograph>)
      4817   => 16#57B3#,  --  垳 (<CJK Ideograph>)
      4818   => 16#57A4#,  --  垤 (<CJK Ideograph>)
      4819   => 16#57AA#,  --  垪 (<CJK Ideograph>)
      4820   => 16#57B0#,  --  垰 (<CJK Ideograph>)
      4821   => 16#57C3#,  --  埃 (<CJK Ideograph>)
      4822   => 16#57C6#,  --  埆 (<CJK Ideograph>)
      4823   => 16#57D4#,  --  埔 (<CJK Ideograph>)
      4824   => 16#57D2#,  --  埒 (<CJK Ideograph>)
      4825   => 16#57D3#,  --  埓 (<CJK Ideograph>)
      4826   => 16#580A#,  --  堊 (<CJK Ideograph>)
      4827   => 16#57D6#,  --  埖 (<CJK Ideograph>)
      4828   => 16#57E3#,  --  埣 (<CJK Ideograph>)
      4829   => 16#580B#,  --  堋 (<CJK Ideograph>)
      4830   => 16#5819#,  --  堙 (<CJK Ideograph>)
      4831   => 16#581D#,  --  堝 (<CJK Ideograph>)
      4832   => 16#5872#,  --  塲 (<CJK Ideograph>)
      4833   => 16#5821#,  --  堡 (<CJK Ideograph>)
      4834   => 16#5862#,  --  塢 (<CJK Ideograph>)
      4835   => 16#584B#,  --  塋 (<CJK Ideograph>)
      4836   => 16#5870#,  --  塰 (<CJK Ideograph>)
      4837   => 16#6BC0#,  --  毀 (<CJK Ideograph>)
      4838   => 16#5852#,  --  塒 (<CJK Ideograph>)
      4839   => 16#583D#,  --  堽 (<CJK Ideograph>)
      4840   => 16#5879#,  --  塹 (<CJK Ideograph>)
      4841   => 16#5885#,  --  墅 (<CJK Ideograph>)
      4842   => 16#58B9#,  --  墹 (<CJK Ideograph>)
      4843   => 16#589F#,  --  墟 (<CJK Ideograph>)
      4844   => 16#58AB#,  --  墫 (<CJK Ideograph>)
      4845   => 16#58BA#,  --  墺 (<CJK Ideograph>)
      4846   => 16#58DE#,  --  壞 (<CJK Ideograph>)
      4847   => 16#58BB#,  --  墻 (<CJK Ideograph>)
      4848   => 16#58B8#,  --  墸 (<CJK Ideograph>)
      4849   => 16#58AE#,  --  墮 (<CJK Ideograph>)
      4850   => 16#58C5#,  --  壅 (<CJK Ideograph>)
      4851   => 16#58D3#,  --  壓 (<CJK Ideograph>)
      4852   => 16#58D1#,  --  壑 (<CJK Ideograph>)
      4853   => 16#58D7#,  --  壗 (<CJK Ideograph>)
      4854   => 16#58D9#,  --  壙 (<CJK Ideograph>)
      4855   => 16#58D8#,  --  壘 (<CJK Ideograph>)
      4856   => 16#58E5#,  --  壥 (<CJK Ideograph>)
      4857   => 16#58DC#,  --  壜 (<CJK Ideograph>)
      4858   => 16#58E4#,  --  壤 (<CJK Ideograph>)
      4859   => 16#58DF#,  --  壟 (<CJK Ideograph>)
      4860   => 16#58EF#,  --  壯 (<CJK Ideograph>)
      4861   => 16#58FA#,  --  壺 (<CJK Ideograph>)
      4862   => 16#58F9#,  --  壹 (<CJK Ideograph>)
      4863   => 16#58FB#,  --  壻 (<CJK Ideograph>)
      4864   => 16#58FC#,  --  壼 (<CJK Ideograph>)
      4865   => 16#58FD#,  --  壽 (<CJK Ideograph>)
      4866   => 16#5902#,  --  夂 (<CJK Ideograph>)
      4867   => 16#590A#,  --  夊 (<CJK Ideograph>)
      4868   => 16#5910#,  --  夐 (<CJK Ideograph>)
      4869   => 16#591B#,  --  夛 (<CJK Ideograph>)
      4870   => 16#68A6#,  --  梦 (<CJK Ideograph>)
      4871   => 16#5925#,  --  夥 (<CJK Ideograph>)
      4872   => 16#592C#,  --  夬 (<CJK Ideograph>)
      4873   => 16#592D#,  --  夭 (<CJK Ideograph>)
      4874   => 16#5932#,  --  夲 (<CJK Ideograph>)
      4875   => 16#5938#,  --  夸 (<CJK Ideograph>)
      4876   => 16#593E#,  --  夾 (<CJK Ideograph>)
      4877   => 16#7AD2#,  --  竒 (<CJK Ideograph>)
      4878   => 16#5955#,  --  奕 (<CJK Ideograph>)
      4879   => 16#5950#,  --  奐 (<CJK Ideograph>)
      4880   => 16#594E#,  --  奎 (<CJK Ideograph>)
      4881   => 16#595A#,  --  奚 (<CJK Ideograph>)
      4882   => 16#5958#,  --  奘 (<CJK Ideograph>)
      4883   => 16#5962#,  --  奢 (<CJK Ideograph>)
      4884   => 16#5960#,  --  奠 (<CJK Ideograph>)
      4885   => 16#5967#,  --  奧 (<CJK Ideograph>)
      4886   => 16#596C#,  --  奬 (<CJK Ideograph>)
      4887   => 16#5969#,  --  奩 (<CJK Ideograph>)
      4888   => 16#5978#,  --  奸 (<CJK Ideograph>)
      4889   => 16#5981#,  --  妁 (<CJK Ideograph>)
      4890   => 16#599D#,  --  妝 (<CJK Ideograph>)
      4891   => 16#4F5E#,  --  佞 (<CJK Ideograph>)
      4892   => 16#4FAB#,  --  侫 (<CJK Ideograph>)
      4893   => 16#59A3#,  --  妣 (<CJK Ideograph>)
      4894   => 16#59B2#,  --  妲 (<CJK Ideograph>)
      4895   => 16#59C6#,  --  姆 (<CJK Ideograph>)
      4896   => 16#59E8#,  --  姨 (<CJK Ideograph>)
      4897   => 16#59DC#,  --  姜 (<CJK Ideograph>)
      4898   => 16#598D#,  --  妍 (<CJK Ideograph>)
      4899   => 16#59D9#,  --  姙 (<CJK Ideograph>)
      4900   => 16#59DA#,  --  姚 (<CJK Ideograph>)
      4901   => 16#5A25#,  --  娥 (<CJK Ideograph>)
      4902   => 16#5A1F#,  --  娟 (<CJK Ideograph>)
      4903   => 16#5A11#,  --  娑 (<CJK Ideograph>)
      4904   => 16#5A1C#,  --  娜 (<CJK Ideograph>)
      4905   => 16#5A09#,  --  娉 (<CJK Ideograph>)
      4906   => 16#5A1A#,  --  娚 (<CJK Ideograph>)
      4907   => 16#5A40#,  --  婀 (<CJK Ideograph>)
      4908   => 16#5A6C#,  --  婬 (<CJK Ideograph>)
      4909   => 16#5A49#,  --  婉 (<CJK Ideograph>)
      4910   => 16#5A35#,  --  娵 (<CJK Ideograph>)
      4911   => 16#5A36#,  --  娶 (<CJK Ideograph>)
      4912   => 16#5A62#,  --  婢 (<CJK Ideograph>)
      4913   => 16#5A6A#,  --  婪 (<CJK Ideograph>)
      4914   => 16#5A9A#,  --  媚 (<CJK Ideograph>)
      4915   => 16#5ABC#,  --  媼 (<CJK Ideograph>)
      4916   => 16#5ABE#,  --  媾 (<CJK Ideograph>)
      4917   => 16#5ACB#,  --  嫋 (<CJK Ideograph>)
      4918   => 16#5AC2#,  --  嫂 (<CJK Ideograph>)
      4919   => 16#5ABD#,  --  媽 (<CJK Ideograph>)
      4920   => 16#5AE3#,  --  嫣 (<CJK Ideograph>)
      4921   => 16#5AD7#,  --  嫗 (<CJK Ideograph>)
      4922   => 16#5AE6#,  --  嫦 (<CJK Ideograph>)
      4923   => 16#5AE9#,  --  嫩 (<CJK Ideograph>)
      4924   => 16#5AD6#,  --  嫖 (<CJK Ideograph>)
      4925   => 16#5AFA#,  --  嫺 (<CJK Ideograph>)
      4926   => 16#5AFB#,  --  嫻 (<CJK Ideograph>)
      4927   => 16#5B0C#,  --  嬌 (<CJK Ideograph>)
      4928   => 16#5B0B#,  --  嬋 (<CJK Ideograph>)
      4929   => 16#5B16#,  --  嬖 (<CJK Ideograph>)
      4930   => 16#5B32#,  --  嬲 (<CJK Ideograph>)
      4931   => 16#5AD0#,  --  嫐 (<CJK Ideograph>)
      4932   => 16#5B2A#,  --  嬪 (<CJK Ideograph>)
      4933   => 16#5B36#,  --  嬶 (<CJK Ideograph>)
      4934   => 16#5B3E#,  --  嬾 (<CJK Ideograph>)
      4935   => 16#5B43#,  --  孃 (<CJK Ideograph>)
      4936   => 16#5B45#,  --  孅 (<CJK Ideograph>)
      4937   => 16#5B40#,  --  孀 (<CJK Ideograph>)
      4938   => 16#5B51#,  --  孑 (<CJK Ideograph>)
      4939   => 16#5B55#,  --  孕 (<CJK Ideograph>)
      4940   => 16#5B5A#,  --  孚 (<CJK Ideograph>)
      4941   => 16#5B5B#,  --  孛 (<CJK Ideograph>)
      4942   => 16#5B65#,  --  孥 (<CJK Ideograph>)
      4943   => 16#5B69#,  --  孩 (<CJK Ideograph>)
      4944   => 16#5B70#,  --  孰 (<CJK Ideograph>)
      4945   => 16#5B73#,  --  孳 (<CJK Ideograph>)
      4946   => 16#5B75#,  --  孵 (<CJK Ideograph>)
      4947   => 16#5B78#,  --  學 (<CJK Ideograph>)
      4948   => 16#6588#,  --  斈 (<CJK Ideograph>)
      4949   => 16#5B7A#,  --  孺 (<CJK Ideograph>)
      4950   => 16#5B80#,  --  宀 (<CJK Ideograph>)
      4951   => 16#5B83#,  --  它 (<CJK Ideograph>)
      4952   => 16#5BA6#,  --  宦 (<CJK Ideograph>)
      4953   => 16#5BB8#,  --  宸 (<CJK Ideograph>)
      4954   => 16#5BC3#,  --  寃 (<CJK Ideograph>)
      4955   => 16#5BC7#,  --  寇 (<CJK Ideograph>)
      4956   => 16#5BC9#,  --  寉 (<CJK Ideograph>)
      4957   => 16#5BD4#,  --  寔 (<CJK Ideograph>)
      4958   => 16#5BD0#,  --  寐 (<CJK Ideograph>)
      4959   => 16#5BE4#,  --  寤 (<CJK Ideograph>)
      4960   => 16#5BE6#,  --  實 (<CJK Ideograph>)
      4961   => 16#5BE2#,  --  寢 (<CJK Ideograph>)
      4962   => 16#5BDE#,  --  寞 (<CJK Ideograph>)
      4963   => 16#5BE5#,  --  寥 (<CJK Ideograph>)
      4964   => 16#5BEB#,  --  寫 (<CJK Ideograph>)
      4965   => 16#5BF0#,  --  寰 (<CJK Ideograph>)
      4966   => 16#5BF6#,  --  寶 (<CJK Ideograph>)
      4967   => 16#5BF3#,  --  寳 (<CJK Ideograph>)
      4968   => 16#5C05#,  --  尅 (<CJK Ideograph>)
      4969   => 16#5C07#,  --  將 (<CJK Ideograph>)
      4970   => 16#5C08#,  --  專 (<CJK Ideograph>)
      4971   => 16#5C0D#,  --  對 (<CJK Ideograph>)
      4972   => 16#5C13#,  --  尓 (<CJK Ideograph>)
      4973   => 16#5C20#,  --  尠 (<CJK Ideograph>)
      4974   => 16#5C22#,  --  尢 (<CJK Ideograph>)
      4975   => 16#5C28#,  --  尨 (<CJK Ideograph>)
      4976   => 16#5C38#,  --  尸 (<CJK Ideograph>)
      4977   => 16#5C39#,  --  尹 (<CJK Ideograph>)
      4978   => 16#5C41#,  --  屁 (<CJK Ideograph>)
      4979   => 16#5C46#,  --  屆 (<CJK Ideograph>)
      4980   => 16#5C4E#,  --  屎 (<CJK Ideograph>)
      4981   => 16#5C53#,  --  屓 (<CJK Ideograph>)
      4982   => 16#5C50#,  --  屐 (<CJK Ideograph>)
      4983   => 16#5C4F#,  --  屏 (<CJK Ideograph>)
      4984   => 16#5B71#,  --  孱 (<CJK Ideograph>)
      4985   => 16#5C6C#,  --  屬 (<CJK Ideograph>)
      4986   => 16#5C6E#,  --  屮 (<CJK Ideograph>)
      4987   => 16#4E62#,  --  乢 (<CJK Ideograph>)
      4988   => 16#5C76#,  --  屶 (<CJK Ideograph>)
      4989   => 16#5C79#,  --  屹 (<CJK Ideograph>)
      4990   => 16#5C8C#,  --  岌 (<CJK Ideograph>)
      4991   => 16#5C91#,  --  岑 (<CJK Ideograph>)
      4992   => 16#5C94#,  --  岔 (<CJK Ideograph>)
      4993   => 16#599B#,  --  妛 (<CJK Ideograph>)
      4994   => 16#5CAB#,  --  岫 (<CJK Ideograph>)
      4995   => 16#5CBB#,  --  岻 (<CJK Ideograph>)
      4996   => 16#5CB6#,  --  岶 (<CJK Ideograph>)
      4997   => 16#5CBC#,  --  岼 (<CJK Ideograph>)
      4998   => 16#5CB7#,  --  岷 (<CJK Ideograph>)
      4999   => 16#5CC5#,  --  峅 (<CJK Ideograph>)
      5000   => 16#5CBE#,  --  岾 (<CJK Ideograph>)
      5001   => 16#5CC7#,  --  峇 (<CJK Ideograph>)
      5002   => 16#5CD9#,  --  峙 (<CJK Ideograph>)
      5003   => 16#5CE9#,  --  峩 (<CJK Ideograph>)
      5004   => 16#5CFD#,  --  峽 (<CJK Ideograph>)
      5005   => 16#5CFA#,  --  峺 (<CJK Ideograph>)
      5006   => 16#5CED#,  --  峭 (<CJK Ideograph>)
      5007   => 16#5D8C#,  --  嶌 (<CJK Ideograph>)
      5008   => 16#5CEA#,  --  峪 (<CJK Ideograph>)
      5009   => 16#5D0B#,  --  崋 (<CJK Ideograph>)
      5010   => 16#5D15#,  --  崕 (<CJK Ideograph>)
      5011   => 16#5D17#,  --  崗 (<CJK Ideograph>)
      5012   => 16#5D5C#,  --  嵜 (<CJK Ideograph>)
      5013   => 16#5D1F#,  --  崟 (<CJK Ideograph>)
      5014   => 16#5D1B#,  --  崛 (<CJK Ideograph>)
      5015   => 16#5D11#,  --  崑 (<CJK Ideograph>)
      5016   => 16#5D14#,  --  崔 (<CJK Ideograph>)
      5017   => 16#5D22#,  --  崢 (<CJK Ideograph>)
      5018   => 16#5D1A#,  --  崚 (<CJK Ideograph>)
      5019   => 16#5D19#,  --  崙 (<CJK Ideograph>)
      5020   => 16#5D18#,  --  崘 (<CJK Ideograph>)
      5021   => 16#5D4C#,  --  嵌 (<CJK Ideograph>)
      5022   => 16#5D52#,  --  嵒 (<CJK Ideograph>)
      5023   => 16#5D4E#,  --  嵎 (<CJK Ideograph>)
      5024   => 16#5D4B#,  --  嵋 (<CJK Ideograph>)
      5025   => 16#5D6C#,  --  嵬 (<CJK Ideograph>)
      5026   => 16#5D73#,  --  嵳 (<CJK Ideograph>)
      5027   => 16#5D76#,  --  嵶 (<CJK Ideograph>)
      5028   => 16#5D87#,  --  嶇 (<CJK Ideograph>)
      5029   => 16#5D84#,  --  嶄 (<CJK Ideograph>)
      5030   => 16#5D82#,  --  嶂 (<CJK Ideograph>)
      5031   => 16#5DA2#,  --  嶢 (<CJK Ideograph>)
      5032   => 16#5D9D#,  --  嶝 (<CJK Ideograph>)
      5033   => 16#5DAC#,  --  嶬 (<CJK Ideograph>)
      5034   => 16#5DAE#,  --  嶮 (<CJK Ideograph>)
      5035   => 16#5DBD#,  --  嶽 (<CJK Ideograph>)
      5036   => 16#5D90#,  --  嶐 (<CJK Ideograph>)
      5037   => 16#5DB7#,  --  嶷 (<CJK Ideograph>)
      5038   => 16#5DBC#,  --  嶼 (<CJK Ideograph>)
      5039   => 16#5DC9#,  --  巉 (<CJK Ideograph>)
      5040   => 16#5DCD#,  --  巍 (<CJK Ideograph>)
      5041   => 16#5DD3#,  --  巓 (<CJK Ideograph>)
      5042   => 16#5DD2#,  --  巒 (<CJK Ideograph>)
      5043   => 16#5DD6#,  --  巖 (<CJK Ideograph>)
      5044   => 16#5DDB#,  --  巛 (<CJK Ideograph>)
      5045   => 16#5DEB#,  --  巫 (<CJK Ideograph>)
      5046   => 16#5DF2#,  --  已 (<CJK Ideograph>)
      5047   => 16#5DF5#,  --  巵 (<CJK Ideograph>)
      5048   => 16#5E0B#,  --  帋 (<CJK Ideograph>)
      5049   => 16#5E1A#,  --  帚 (<CJK Ideograph>)
      5050   => 16#5E19#,  --  帙 (<CJK Ideograph>)
      5051   => 16#5E11#,  --  帑 (<CJK Ideograph>)
      5052   => 16#5E1B#,  --  帛 (<CJK Ideograph>)
      5053   => 16#5E36#,  --  帶 (<CJK Ideograph>)
      5054   => 16#5E37#,  --  帷 (<CJK Ideograph>)
      5055   => 16#5E44#,  --  幄 (<CJK Ideograph>)
      5056   => 16#5E43#,  --  幃 (<CJK Ideograph>)
      5057   => 16#5E40#,  --  幀 (<CJK Ideograph>)
      5058   => 16#5E4E#,  --  幎 (<CJK Ideograph>)
      5059   => 16#5E57#,  --  幗 (<CJK Ideograph>)
      5060   => 16#5E54#,  --  幔 (<CJK Ideograph>)
      5061   => 16#5E5F#,  --  幟 (<CJK Ideograph>)
      5062   => 16#5E62#,  --  幢 (<CJK Ideograph>)
      5063   => 16#5E64#,  --  幤 (<CJK Ideograph>)
      5064   => 16#5E47#,  --  幇 (<CJK Ideograph>)
      5065   => 16#5E75#,  --  幵 (<CJK Ideograph>)
      5066   => 16#5E76#,  --  并 (<CJK Ideograph>)
      5067   => 16#5E7A#,  --  幺 (<CJK Ideograph>)
      5068   => 16#9EBC#,  --  麼 (<CJK Ideograph>)
      5069   => 16#5E7F#,  --  广 (<CJK Ideograph>)
      5070   => 16#5EA0#,  --  庠 (<CJK Ideograph>)
      5071   => 16#5EC1#,  --  廁 (<CJK Ideograph>)
      5072   => 16#5EC2#,  --  廂 (<CJK Ideograph>)
      5073   => 16#5EC8#,  --  廈 (<CJK Ideograph>)
      5074   => 16#5ED0#,  --  廐 (<CJK Ideograph>)
      5075   => 16#5ECF#,  --  廏 (<CJK Ideograph>)
      5076   => 16#5ED6#,  --  廖 (<CJK Ideograph>)
      5077   => 16#5EE3#,  --  廣 (<CJK Ideograph>)
      5078   => 16#5EDD#,  --  廝 (<CJK Ideograph>)
      5079   => 16#5EDA#,  --  廚 (<CJK Ideograph>)
      5080   => 16#5EDB#,  --  廛 (<CJK Ideograph>)
      5081   => 16#5EE2#,  --  廢 (<CJK Ideograph>)
      5082   => 16#5EE1#,  --  廡 (<CJK Ideograph>)
      5083   => 16#5EE8#,  --  廨 (<CJK Ideograph>)
      5084   => 16#5EE9#,  --  廩 (<CJK Ideograph>)
      5085   => 16#5EEC#,  --  廬 (<CJK Ideograph>)
      5086   => 16#5EF1#,  --  廱 (<CJK Ideograph>)
      5087   => 16#5EF3#,  --  廳 (<CJK Ideograph>)
      5088   => 16#5EF0#,  --  廰 (<CJK Ideograph>)
      5089   => 16#5EF4#,  --  廴 (<CJK Ideograph>)
      5090   => 16#5EF8#,  --  廸 (<CJK Ideograph>)
      5091   => 16#5EFE#,  --  廾 (<CJK Ideograph>)
      5092   => 16#5F03#,  --  弃 (<CJK Ideograph>)
      5093   => 16#5F09#,  --  弉 (<CJK Ideograph>)
      5094   => 16#5F5D#,  --  彝 (<CJK Ideograph>)
      5095   => 16#5F5C#,  --  彜 (<CJK Ideograph>)
      5096   => 16#5F0B#,  --  弋 (<CJK Ideograph>)
      5097   => 16#5F11#,  --  弑 (<CJK Ideograph>)
      5098   => 16#5F16#,  --  弖 (<CJK Ideograph>)
      5099   => 16#5F29#,  --  弩 (<CJK Ideograph>)
      5100   => 16#5F2D#,  --  弭 (<CJK Ideograph>)
      5101   => 16#5F38#,  --  弸 (<CJK Ideograph>)
      5102   => 16#5F41#,  --  彁 (<CJK Ideograph>)
      5103   => 16#5F48#,  --  彈 (<CJK Ideograph>)
      5104   => 16#5F4C#,  --  彌 (<CJK Ideograph>)
      5105   => 16#5F4E#,  --  彎 (<CJK Ideograph>)
      5106   => 16#5F2F#,  --  弯 (<CJK Ideograph>)
      5107   => 16#5F51#,  --  彑 (<CJK Ideograph>)
      5108   => 16#5F56#,  --  彖 (<CJK Ideograph>)
      5109   => 16#5F57#,  --  彗 (<CJK Ideograph>)
      5110   => 16#5F59#,  --  彙 (<CJK Ideograph>)
      5111   => 16#5F61#,  --  彡 (<CJK Ideograph>)
      5112   => 16#5F6D#,  --  彭 (<CJK Ideograph>)
      5113   => 16#5F73#,  --  彳 (<CJK Ideograph>)
      5114   => 16#5F77#,  --  彷 (<CJK Ideograph>)
      5115   => 16#5F83#,  --  徃 (<CJK Ideograph>)
      5116   => 16#5F82#,  --  徂 (<CJK Ideograph>)
      5117   => 16#5F7F#,  --  彿 (<CJK Ideograph>)
      5118   => 16#5F8A#,  --  徊 (<CJK Ideograph>)
      5119   => 16#5F88#,  --  很 (<CJK Ideograph>)
      5120   => 16#5F91#,  --  徑 (<CJK Ideograph>)
      5121   => 16#5F87#,  --  徇 (<CJK Ideograph>)
      5122   => 16#5F9E#,  --  從 (<CJK Ideograph>)
      5123   => 16#5F99#,  --  徙 (<CJK Ideograph>)
      5124   => 16#5F98#,  --  徘 (<CJK Ideograph>)
      5125   => 16#5FA0#,  --  徠 (<CJK Ideograph>)
      5126   => 16#5FA8#,  --  徨 (<CJK Ideograph>)
      5127   => 16#5FAD#,  --  徭 (<CJK Ideograph>)
      5128   => 16#5FBC#,  --  徼 (<CJK Ideograph>)
      5129   => 16#5FD6#,  --  忖 (<CJK Ideograph>)
      5130   => 16#5FFB#,  --  忻 (<CJK Ideograph>)
      5131   => 16#5FE4#,  --  忤 (<CJK Ideograph>)
      5132   => 16#5FF8#,  --  忸 (<CJK Ideograph>)
      5133   => 16#5FF1#,  --  忱 (<CJK Ideograph>)
      5134   => 16#5FDD#,  --  忝 (<CJK Ideograph>)
      5135   => 16#60B3#,  --  悳 (<CJK Ideograph>)
      5136   => 16#5FFF#,  --  忿 (<CJK Ideograph>)
      5137   => 16#6021#,  --  怡 (<CJK Ideograph>)
      5138   => 16#6060#,  --  恠 (<CJK Ideograph>)
      5139   => 16#6019#,  --  怙 (<CJK Ideograph>)
      5140   => 16#6010#,  --  怐 (<CJK Ideograph>)
      5141   => 16#6029#,  --  怩 (<CJK Ideograph>)
      5142   => 16#600E#,  --  怎 (<CJK Ideograph>)
      5143   => 16#6031#,  --  怱 (<CJK Ideograph>)
      5144   => 16#601B#,  --  怛 (<CJK Ideograph>)
      5145   => 16#6015#,  --  怕 (<CJK Ideograph>)
      5146   => 16#602B#,  --  怫 (<CJK Ideograph>)
      5147   => 16#6026#,  --  怦 (<CJK Ideograph>)
      5148   => 16#600F#,  --  怏 (<CJK Ideograph>)
      5149   => 16#603A#,  --  怺 (<CJK Ideograph>)
      5150   => 16#605A#,  --  恚 (<CJK Ideograph>)
      5151   => 16#6041#,  --  恁 (<CJK Ideograph>)
      5152   => 16#606A#,  --  恪 (<CJK Ideograph>)
      5153   => 16#6077#,  --  恷 (<CJK Ideograph>)
      5154   => 16#605F#,  --  恟 (<CJK Ideograph>)
      5155   => 16#604A#,  --  恊 (<CJK Ideograph>)
      5156   => 16#6046#,  --  恆 (<CJK Ideograph>)
      5157   => 16#604D#,  --  恍 (<CJK Ideograph>)
      5158   => 16#6063#,  --  恣 (<CJK Ideograph>)
      5159   => 16#6043#,  --  恃 (<CJK Ideograph>)
      5160   => 16#6064#,  --  恤 (<CJK Ideograph>)
      5161   => 16#6042#,  --  恂 (<CJK Ideograph>)
      5162   => 16#606C#,  --  恬 (<CJK Ideograph>)
      5163   => 16#606B#,  --  恫 (<CJK Ideograph>)
      5164   => 16#6059#,  --  恙 (<CJK Ideograph>)
      5165   => 16#6081#,  --  悁 (<CJK Ideograph>)
      5166   => 16#608D#,  --  悍 (<CJK Ideograph>)
      5167   => 16#60E7#,  --  惧 (<CJK Ideograph>)
      5168   => 16#6083#,  --  悃 (<CJK Ideograph>)
      5169   => 16#609A#,  --  悚 (<CJK Ideograph>)
      5170   => 16#6084#,  --  悄 (<CJK Ideograph>)
      5171   => 16#609B#,  --  悛 (<CJK Ideograph>)
      5172   => 16#6096#,  --  悖 (<CJK Ideograph>)
      5173   => 16#6097#,  --  悗 (<CJK Ideograph>)
      5174   => 16#6092#,  --  悒 (<CJK Ideograph>)
      5175   => 16#60A7#,  --  悧 (<CJK Ideograph>)
      5176   => 16#608B#,  --  悋 (<CJK Ideograph>)
      5177   => 16#60E1#,  --  惡 (<CJK Ideograph>)
      5178   => 16#60B8#,  --  悸 (<CJK Ideograph>)
      5179   => 16#60E0#,  --  惠 (<CJK Ideograph>)
      5180   => 16#60D3#,  --  惓 (<CJK Ideograph>)
      5181   => 16#60B4#,  --  悴 (<CJK Ideograph>)
      5182   => 16#5FF0#,  --  忰 (<CJK Ideograph>)
      5183   => 16#60BD#,  --  悽 (<CJK Ideograph>)
      5184   => 16#60C6#,  --  惆 (<CJK Ideograph>)
      5185   => 16#60B5#,  --  悵 (<CJK Ideograph>)
      5186   => 16#60D8#,  --  惘 (<CJK Ideograph>)
      5187   => 16#614D#,  --  慍 (<CJK Ideograph>)
      5188   => 16#6115#,  --  愕 (<CJK Ideograph>)
      5189   => 16#6106#,  --  愆 (<CJK Ideograph>)
      5190   => 16#60F6#,  --  惶 (<CJK Ideograph>)
      5191   => 16#60F7#,  --  惷 (<CJK Ideograph>)
      5192   => 16#6100#,  --  愀 (<CJK Ideograph>)
      5193   => 16#60F4#,  --  惴 (<CJK Ideograph>)
      5194   => 16#60FA#,  --  惺 (<CJK Ideograph>)
      5195   => 16#6103#,  --  愃 (<CJK Ideograph>)
      5196   => 16#6121#,  --  愡 (<CJK Ideograph>)
      5197   => 16#60FB#,  --  惻 (<CJK Ideograph>)
      5198   => 16#60F1#,  --  惱 (<CJK Ideograph>)
      5199   => 16#610D#,  --  愍 (<CJK Ideograph>)
      5200   => 16#610E#,  --  愎 (<CJK Ideograph>)
      5201   => 16#6147#,  --  慇 (<CJK Ideograph>)
      5202   => 16#613E#,  --  愾 (<CJK Ideograph>)
      5203   => 16#6128#,  --  愨 (<CJK Ideograph>)
      5204   => 16#6127#,  --  愧 (<CJK Ideograph>)
      5205   => 16#614A#,  --  慊 (<CJK Ideograph>)
      5206   => 16#613F#,  --  愿 (<CJK Ideograph>)
      5207   => 16#613C#,  --  愼 (<CJK Ideograph>)
      5208   => 16#612C#,  --  愬 (<CJK Ideograph>)
      5209   => 16#6134#,  --  愴 (<CJK Ideograph>)
      5210   => 16#613D#,  --  愽 (<CJK Ideograph>)
      5211   => 16#6142#,  --  慂 (<CJK Ideograph>)
      5212   => 16#6144#,  --  慄 (<CJK Ideograph>)
      5213   => 16#6173#,  --  慳 (<CJK Ideograph>)
      5214   => 16#6177#,  --  慷 (<CJK Ideograph>)
      5215   => 16#6158#,  --  慘 (<CJK Ideograph>)
      5216   => 16#6159#,  --  慙 (<CJK Ideograph>)
      5217   => 16#615A#,  --  慚 (<CJK Ideograph>)
      5218   => 16#616B#,  --  慫 (<CJK Ideograph>)
      5219   => 16#6174#,  --  慴 (<CJK Ideograph>)
      5220   => 16#616F#,  --  慯 (<CJK Ideograph>)
      5221   => 16#6165#,  --  慥 (<CJK Ideograph>)
      5222   => 16#6171#,  --  慱 (<CJK Ideograph>)
      5223   => 16#615F#,  --  慟 (<CJK Ideograph>)
      5224   => 16#615D#,  --  慝 (<CJK Ideograph>)
      5225   => 16#6153#,  --  慓 (<CJK Ideograph>)
      5226   => 16#6175#,  --  慵 (<CJK Ideograph>)
      5227   => 16#6199#,  --  憙 (<CJK Ideograph>)
      5228   => 16#6196#,  --  憖 (<CJK Ideograph>)
      5229   => 16#6187#,  --  憇 (<CJK Ideograph>)
      5230   => 16#61AC#,  --  憬 (<CJK Ideograph>)
      5231   => 16#6194#,  --  憔 (<CJK Ideograph>)
      5232   => 16#619A#,  --  憚 (<CJK Ideograph>)
      5233   => 16#618A#,  --  憊 (<CJK Ideograph>)
      5234   => 16#6191#,  --  憑 (<CJK Ideograph>)
      5235   => 16#61AB#,  --  憫 (<CJK Ideograph>)
      5236   => 16#61AE#,  --  憮 (<CJK Ideograph>)
      5237   => 16#61CC#,  --  懌 (<CJK Ideograph>)
      5238   => 16#61CA#,  --  懊 (<CJK Ideograph>)
      5239   => 16#61C9#,  --  應 (<CJK Ideograph>)
      5240   => 16#61F7#,  --  懷 (<CJK Ideograph>)
      5241   => 16#61C8#,  --  懈 (<CJK Ideograph>)
      5242   => 16#61C3#,  --  懃 (<CJK Ideograph>)
      5243   => 16#61C6#,  --  懆 (<CJK Ideograph>)
      5244   => 16#61BA#,  --  憺 (<CJK Ideograph>)
      5245   => 16#61CB#,  --  懋 (<CJK Ideograph>)
      5246   => 16#7F79#,  --  罹 (<CJK Ideograph>)
      5247   => 16#61CD#,  --  懍 (<CJK Ideograph>)
      5248   => 16#61E6#,  --  懦 (<CJK Ideograph>)
      5249   => 16#61E3#,  --  懣 (<CJK Ideograph>)
      5250   => 16#61F6#,  --  懶 (<CJK Ideograph>)
      5251   => 16#61FA#,  --  懺 (<CJK Ideograph>)
      5252   => 16#61F4#,  --  懴 (<CJK Ideograph>)
      5253   => 16#61FF#,  --  懿 (<CJK Ideograph>)
      5254   => 16#61FD#,  --  懽 (<CJK Ideograph>)
      5255   => 16#61FC#,  --  懼 (<CJK Ideograph>)
      5256   => 16#61FE#,  --  懾 (<CJK Ideograph>)
      5257   => 16#6200#,  --  戀 (<CJK Ideograph>)
      5258   => 16#6208#,  --  戈 (<CJK Ideograph>)
      5259   => 16#6209#,  --  戉 (<CJK Ideograph>)
      5260   => 16#620D#,  --  戍 (<CJK Ideograph>)
      5261   => 16#620C#,  --  戌 (<CJK Ideograph>)
      5262   => 16#6214#,  --  戔 (<CJK Ideograph>)
      5263   => 16#621B#,  --  戛 (<CJK Ideograph>)
      5264   => 16#621E#,  --  戞 (<CJK Ideograph>)
      5265   => 16#6221#,  --  戡 (<CJK Ideograph>)
      5266   => 16#622A#,  --  截 (<CJK Ideograph>)
      5267   => 16#622E#,  --  戮 (<CJK Ideograph>)
      5268   => 16#6230#,  --  戰 (<CJK Ideograph>)
      5269   => 16#6232#,  --  戲 (<CJK Ideograph>)
      5270   => 16#6233#,  --  戳 (<CJK Ideograph>)
      5271   => 16#6241#,  --  扁 (<CJK Ideograph>)
      5272   => 16#624E#,  --  扎 (<CJK Ideograph>)
      5273   => 16#625E#,  --  扞 (<CJK Ideograph>)
      5274   => 16#6263#,  --  扣 (<CJK Ideograph>)
      5275   => 16#625B#,  --  扛 (<CJK Ideograph>)
      5276   => 16#6260#,  --  扠 (<CJK Ideograph>)
      5277   => 16#6268#,  --  扨 (<CJK Ideograph>)
      5278   => 16#627C#,  --  扼 (<CJK Ideograph>)
      5279   => 16#6282#,  --  抂 (<CJK Ideograph>)
      5280   => 16#6289#,  --  抉 (<CJK Ideograph>)
      5281   => 16#627E#,  --  找 (<CJK Ideograph>)
      5282   => 16#6292#,  --  抒 (<CJK Ideograph>)
      5283   => 16#6293#,  --  抓 (<CJK Ideograph>)
      5284   => 16#6296#,  --  抖 (<CJK Ideograph>)
      5285   => 16#62D4#,  --  拔 (<CJK Ideograph>)
      5286   => 16#6283#,  --  抃 (<CJK Ideograph>)
      5287   => 16#6294#,  --  抔 (<CJK Ideograph>)
      5288   => 16#62D7#,  --  拗 (<CJK Ideograph>)
      5289   => 16#62D1#,  --  拑 (<CJK Ideograph>)
      5290   => 16#62BB#,  --  抻 (<CJK Ideograph>)
      5291   => 16#62CF#,  --  拏 (<CJK Ideograph>)
      5292   => 16#62FF#,  --  拿 (<CJK Ideograph>)
      5293   => 16#62C6#,  --  拆 (<CJK Ideograph>)
      5294   => 16#64D4#,  --  擔 (<CJK Ideograph>)
      5295   => 16#62C8#,  --  拈 (<CJK Ideograph>)
      5296   => 16#62DC#,  --  拜 (<CJK Ideograph>)
      5297   => 16#62CC#,  --  拌 (<CJK Ideograph>)
      5298   => 16#62CA#,  --  拊 (<CJK Ideograph>)
      5299   => 16#62C2#,  --  拂 (<CJK Ideograph>)
      5300   => 16#62C7#,  --  拇 (<CJK Ideograph>)
      5301   => 16#629B#,  --  抛 (<CJK Ideograph>)
      5302   => 16#62C9#,  --  拉 (<CJK Ideograph>)
      5303   => 16#630C#,  --  挌 (<CJK Ideograph>)
      5304   => 16#62EE#,  --  拮 (<CJK Ideograph>)
      5305   => 16#62F1#,  --  拱 (<CJK Ideograph>)
      5306   => 16#6327#,  --  挧 (<CJK Ideograph>)
      5307   => 16#6302#,  --  挂 (<CJK Ideograph>)
      5308   => 16#6308#,  --  挈 (<CJK Ideograph>)
      5309   => 16#62EF#,  --  拯 (<CJK Ideograph>)
      5310   => 16#62F5#,  --  拵 (<CJK Ideograph>)
      5311   => 16#6350#,  --  捐 (<CJK Ideograph>)
      5312   => 16#633E#,  --  挾 (<CJK Ideograph>)
      5313   => 16#634D#,  --  捍 (<CJK Ideograph>)
      5314   => 16#641C#,  --  搜 (<CJK Ideograph>)
      5315   => 16#634F#,  --  捏 (<CJK Ideograph>)
      5316   => 16#6396#,  --  掖 (<CJK Ideograph>)
      5317   => 16#638E#,  --  掎 (<CJK Ideograph>)
      5318   => 16#6380#,  --  掀 (<CJK Ideograph>)
      5319   => 16#63AB#,  --  掫 (<CJK Ideograph>)
      5320   => 16#6376#,  --  捶 (<CJK Ideograph>)
      5321   => 16#63A3#,  --  掣 (<CJK Ideograph>)
      5322   => 16#638F#,  --  掏 (<CJK Ideograph>)
      5323   => 16#6389#,  --  掉 (<CJK Ideograph>)
      5324   => 16#639F#,  --  掟 (<CJK Ideograph>)
      5325   => 16#63B5#,  --  掵 (<CJK Ideograph>)
      5326   => 16#636B#,  --  捫 (<CJK Ideograph>)
      5327   => 16#6369#,  --  捩 (<CJK Ideograph>)
      5328   => 16#63BE#,  --  掾 (<CJK Ideograph>)
      5329   => 16#63E9#,  --  揩 (<CJK Ideograph>)
      5330   => 16#63C0#,  --  揀 (<CJK Ideograph>)
      5331   => 16#63C6#,  --  揆 (<CJK Ideograph>)
      5332   => 16#63E3#,  --  揣 (<CJK Ideograph>)
      5333   => 16#63C9#,  --  揉 (<CJK Ideograph>)
      5334   => 16#63D2#,  --  插 (<CJK Ideograph>)
      5335   => 16#63F6#,  --  揶 (<CJK Ideograph>)
      5336   => 16#63C4#,  --  揄 (<CJK Ideograph>)
      5337   => 16#6416#,  --  搖 (<CJK Ideograph>)
      5338   => 16#6434#,  --  搴 (<CJK Ideograph>)
      5339   => 16#6406#,  --  搆 (<CJK Ideograph>)
      5340   => 16#6413#,  --  搓 (<CJK Ideograph>)
      5341   => 16#6426#,  --  搦 (<CJK Ideograph>)
      5342   => 16#6436#,  --  搶 (<CJK Ideograph>)
      5343   => 16#651D#,  --  攝 (<CJK Ideograph>)
      5344   => 16#6417#,  --  搗 (<CJK Ideograph>)
      5345   => 16#6428#,  --  搨 (<CJK Ideograph>)
      5346   => 16#640F#,  --  搏 (<CJK Ideograph>)
      5347   => 16#6467#,  --  摧 (<CJK Ideograph>)
      5348   => 16#646F#,  --  摯 (<CJK Ideograph>)
      5349   => 16#6476#,  --  摶 (<CJK Ideograph>)
      5350   => 16#644E#,  --  摎 (<CJK Ideograph>)
      5351   => 16#652A#,  --  攪 (<CJK Ideograph>)
      5352   => 16#6495#,  --  撕 (<CJK Ideograph>)
      5353   => 16#6493#,  --  撓 (<CJK Ideograph>)
      5354   => 16#64A5#,  --  撥 (<CJK Ideograph>)
      5355   => 16#64A9#,  --  撩 (<CJK Ideograph>)
      5356   => 16#6488#,  --  撈 (<CJK Ideograph>)
      5357   => 16#64BC#,  --  撼 (<CJK Ideograph>)
      5358   => 16#64DA#,  --  據 (<CJK Ideograph>)
      5359   => 16#64D2#,  --  擒 (<CJK Ideograph>)
      5360   => 16#64C5#,  --  擅 (<CJK Ideograph>)
      5361   => 16#64C7#,  --  擇 (<CJK Ideograph>)
      5362   => 16#64BB#,  --  撻 (<CJK Ideograph>)
      5363   => 16#64D8#,  --  擘 (<CJK Ideograph>)
      5364   => 16#64C2#,  --  擂 (<CJK Ideograph>)
      5365   => 16#64F1#,  --  擱 (<CJK Ideograph>)
      5366   => 16#64E7#,  --  擧 (<CJK Ideograph>)
      5367   => 16#8209#,  --  舉 (<CJK Ideograph>)
      5368   => 16#64E0#,  --  擠 (<CJK Ideograph>)
      5369   => 16#64E1#,  --  擡 (<CJK Ideograph>)
      5370   => 16#62AC#,  --  抬 (<CJK Ideograph>)
      5371   => 16#64E3#,  --  擣 (<CJK Ideograph>)
      5372   => 16#64EF#,  --  擯 (<CJK Ideograph>)
      5373   => 16#652C#,  --  攬 (<CJK Ideograph>)
      5374   => 16#64F6#,  --  擶 (<CJK Ideograph>)
      5375   => 16#64F4#,  --  擴 (<CJK Ideograph>)
      5376   => 16#64F2#,  --  擲 (<CJK Ideograph>)
      5377   => 16#64FA#,  --  擺 (<CJK Ideograph>)
      5378   => 16#6500#,  --  攀 (<CJK Ideograph>)
      5379   => 16#64FD#,  --  擽 (<CJK Ideograph>)
      5380   => 16#6518#,  --  攘 (<CJK Ideograph>)
      5381   => 16#651C#,  --  攜 (<CJK Ideograph>)
      5382   => 16#6505#,  --  攅 (<CJK Ideograph>)
      5383   => 16#6524#,  --  攤 (<CJK Ideograph>)
      5384   => 16#6523#,  --  攣 (<CJK Ideograph>)
      5385   => 16#652B#,  --  攫 (<CJK Ideograph>)
      5386   => 16#6534#,  --  攴 (<CJK Ideograph>)
      5387   => 16#6535#,  --  攵 (<CJK Ideograph>)
      5388   => 16#6537#,  --  攷 (<CJK Ideograph>)
      5389   => 16#6536#,  --  收 (<CJK Ideograph>)
      5390   => 16#6538#,  --  攸 (<CJK Ideograph>)
      5391   => 16#754B#,  --  畋 (<CJK Ideograph>)
      5392   => 16#6548#,  --  效 (<CJK Ideograph>)
      5393   => 16#6556#,  --  敖 (<CJK Ideograph>)
      5394   => 16#6555#,  --  敕 (<CJK Ideograph>)
      5395   => 16#654D#,  --  敍 (<CJK Ideograph>)
      5396   => 16#6558#,  --  敘 (<CJK Ideograph>)
      5397   => 16#655E#,  --  敞 (<CJK Ideograph>)
      5398   => 16#655D#,  --  敝 (<CJK Ideograph>)
      5399   => 16#6572#,  --  敲 (<CJK Ideograph>)
      5400   => 16#6578#,  --  數 (<CJK Ideograph>)
      5401   => 16#6582#,  --  斂 (<CJK Ideograph>)
      5402   => 16#6583#,  --  斃 (<CJK Ideograph>)
      5403   => 16#8B8A#,  --  變 (<CJK Ideograph>)
      5404   => 16#659B#,  --  斛 (<CJK Ideograph>)
      5405   => 16#659F#,  --  斟 (<CJK Ideograph>)
      5406   => 16#65AB#,  --  斫 (<CJK Ideograph>)
      5407   => 16#65B7#,  --  斷 (<CJK Ideograph>)
      5408   => 16#65C3#,  --  旃 (<CJK Ideograph>)
      5409   => 16#65C6#,  --  旆 (<CJK Ideograph>)
      5410   => 16#65C1#,  --  旁 (<CJK Ideograph>)
      5411   => 16#65C4#,  --  旄 (<CJK Ideograph>)
      5412   => 16#65CC#,  --  旌 (<CJK Ideograph>)
      5413   => 16#65D2#,  --  旒 (<CJK Ideograph>)
      5414   => 16#65DB#,  --  旛 (<CJK Ideograph>)
      5415   => 16#65D9#,  --  旙 (<CJK Ideograph>)
      5416   => 16#65E0#,  --  无 (<CJK Ideograph>)
      5417   => 16#65E1#,  --  旡 (<CJK Ideograph>)
      5418   => 16#65F1#,  --  旱 (<CJK Ideograph>)
      5419   => 16#6772#,  --  杲 (<CJK Ideograph>)
      5420   => 16#660A#,  --  昊 (<CJK Ideograph>)
      5421   => 16#6603#,  --  昃 (<CJK Ideograph>)
      5422   => 16#65FB#,  --  旻 (<CJK Ideograph>)
      5423   => 16#6773#,  --  杳 (<CJK Ideograph>)
      5424   => 16#6635#,  --  昵 (<CJK Ideograph>)
      5425   => 16#6636#,  --  昶 (<CJK Ideograph>)
      5426   => 16#6634#,  --  昴 (<CJK Ideograph>)
      5427   => 16#661C#,  --  昜 (<CJK Ideograph>)
      5428   => 16#664F#,  --  晏 (<CJK Ideograph>)
      5429   => 16#6644#,  --  晄 (<CJK Ideograph>)
      5430   => 16#6649#,  --  晉 (<CJK Ideograph>)
      5431   => 16#6641#,  --  晁 (<CJK Ideograph>)
      5432   => 16#665E#,  --  晞 (<CJK Ideograph>)
      5433   => 16#665D#,  --  晝 (<CJK Ideograph>)
      5434   => 16#6664#,  --  晤 (<CJK Ideograph>)
      5435   => 16#6667#,  --  晧 (<CJK Ideograph>)
      5436   => 16#6668#,  --  晨 (<CJK Ideograph>)
      5437   => 16#665F#,  --  晟 (<CJK Ideograph>)
      5438   => 16#6662#,  --  晢 (<CJK Ideograph>)
      5439   => 16#6670#,  --  晰 (<CJK Ideograph>)
      5440   => 16#6683#,  --  暃 (<CJK Ideograph>)
      5441   => 16#6688#,  --  暈 (<CJK Ideograph>)
      5442   => 16#668E#,  --  暎 (<CJK Ideograph>)
      5443   => 16#6689#,  --  暉 (<CJK Ideograph>)
      5444   => 16#6684#,  --  暄 (<CJK Ideograph>)
      5445   => 16#6698#,  --  暘 (<CJK Ideograph>)
      5446   => 16#669D#,  --  暝 (<CJK Ideograph>)
      5447   => 16#66C1#,  --  曁 (<CJK Ideograph>)
      5448   => 16#66B9#,  --  暹 (<CJK Ideograph>)
      5449   => 16#66C9#,  --  曉 (<CJK Ideograph>)
      5450   => 16#66BE#,  --  暾 (<CJK Ideograph>)
      5451   => 16#66BC#,  --  暼 (<CJK Ideograph>)
      5452   => 16#66C4#,  --  曄 (<CJK Ideograph>)
      5453   => 16#66B8#,  --  暸 (<CJK Ideograph>)
      5454   => 16#66D6#,  --  曖 (<CJK Ideograph>)
      5455   => 16#66DA#,  --  曚 (<CJK Ideograph>)
      5456   => 16#66E0#,  --  曠 (<CJK Ideograph>)
      5457   => 16#663F#,  --  昿 (<CJK Ideograph>)
      5458   => 16#66E6#,  --  曦 (<CJK Ideograph>)
      5459   => 16#66E9#,  --  曩 (<CJK Ideograph>)
      5460   => 16#66F0#,  --  曰 (<CJK Ideograph>)
      5461   => 16#66F5#,  --  曵 (<CJK Ideograph>)
      5462   => 16#66F7#,  --  曷 (<CJK Ideograph>)
      5463   => 16#670F#,  --  朏 (<CJK Ideograph>)
      5464   => 16#6716#,  --  朖 (<CJK Ideograph>)
      5465   => 16#671E#,  --  朞 (<CJK Ideograph>)
      5466   => 16#6726#,  --  朦 (<CJK Ideograph>)
      5467   => 16#6727#,  --  朧 (<CJK Ideograph>)
      5468   => 16#9738#,  --  霸 (<CJK Ideograph>)
      5469   => 16#672E#,  --  朮 (<CJK Ideograph>)
      5470   => 16#673F#,  --  朿 (<CJK Ideograph>)
      5471   => 16#6736#,  --  朶 (<CJK Ideograph>)
      5472   => 16#6741#,  --  杁 (<CJK Ideograph>)
      5473   => 16#6738#,  --  朸 (<CJK Ideograph>)
      5474   => 16#6737#,  --  朷 (<CJK Ideograph>)
      5475   => 16#6746#,  --  杆 (<CJK Ideograph>)
      5476   => 16#675E#,  --  杞 (<CJK Ideograph>)
      5477   => 16#6760#,  --  杠 (<CJK Ideograph>)
      5478   => 16#6759#,  --  杙 (<CJK Ideograph>)
      5479   => 16#6763#,  --  杣 (<CJK Ideograph>)
      5480   => 16#6764#,  --  杤 (<CJK Ideograph>)
      5481   => 16#6789#,  --  枉 (<CJK Ideograph>)
      5482   => 16#6770#,  --  杰 (<CJK Ideograph>)
      5483   => 16#67A9#,  --  枩 (<CJK Ideograph>)
      5484   => 16#677C#,  --  杼 (<CJK Ideograph>)
      5485   => 16#676A#,  --  杪 (<CJK Ideograph>)
      5486   => 16#678C#,  --  枌 (<CJK Ideograph>)
      5487   => 16#678B#,  --  枋 (<CJK Ideograph>)
      5488   => 16#67A6#,  --  枦 (<CJK Ideograph>)
      5489   => 16#67A1#,  --  枡 (<CJK Ideograph>)
      5490   => 16#6785#,  --  枅 (<CJK Ideograph>)
      5491   => 16#67B7#,  --  枷 (<CJK Ideograph>)
      5492   => 16#67EF#,  --  柯 (<CJK Ideograph>)
      5493   => 16#67B4#,  --  枴 (<CJK Ideograph>)
      5494   => 16#67EC#,  --  柬 (<CJK Ideograph>)
      5495   => 16#67B3#,  --  枳 (<CJK Ideograph>)
      5496   => 16#67E9#,  --  柩 (<CJK Ideograph>)
      5497   => 16#67B8#,  --  枸 (<CJK Ideograph>)
      5498   => 16#67E4#,  --  柤 (<CJK Ideograph>)
      5499   => 16#67DE#,  --  柞 (<CJK Ideograph>)
      5500   => 16#67DD#,  --  柝 (<CJK Ideograph>)
      5501   => 16#67E2#,  --  柢 (<CJK Ideograph>)
      5502   => 16#67EE#,  --  柮 (<CJK Ideograph>)
      5503   => 16#67B9#,  --  枹 (<CJK Ideograph>)
      5504   => 16#67CE#,  --  柎 (<CJK Ideograph>)
      5505   => 16#67C6#,  --  柆 (<CJK Ideograph>)
      5506   => 16#67E7#,  --  柧 (<CJK Ideograph>)
      5507   => 16#6A9C#,  --  檜 (<CJK Ideograph>)
      5508   => 16#681E#,  --  栞 (<CJK Ideograph>)
      5509   => 16#6846#,  --  框 (<CJK Ideograph>)
      5510   => 16#6829#,  --  栩 (<CJK Ideograph>)
      5511   => 16#6840#,  --  桀 (<CJK Ideograph>)
      5512   => 16#684D#,  --  桍 (<CJK Ideograph>)
      5513   => 16#6832#,  --  栲 (<CJK Ideograph>)
      5514   => 16#684E#,  --  桎 (<CJK Ideograph>)
      5515   => 16#68B3#,  --  梳 (<CJK Ideograph>)
      5516   => 16#682B#,  --  栫 (<CJK Ideograph>)
      5517   => 16#6859#,  --  桙 (<CJK Ideograph>)
      5518   => 16#6863#,  --  档 (<CJK Ideograph>)
      5519   => 16#6877#,  --  桷 (<CJK Ideograph>)
      5520   => 16#687F#,  --  桿 (<CJK Ideograph>)
      5521   => 16#689F#,  --  梟 (<CJK Ideograph>)
      5522   => 16#688F#,  --  梏 (<CJK Ideograph>)
      5523   => 16#68AD#,  --  梭 (<CJK Ideograph>)
      5524   => 16#6894#,  --  梔 (<CJK Ideograph>)
      5525   => 16#689D#,  --  條 (<CJK Ideograph>)
      5526   => 16#689B#,  --  梛 (<CJK Ideograph>)
      5527   => 16#6883#,  --  梃 (<CJK Ideograph>)
      5528   => 16#6AAE#,  --  檮 (<CJK Ideograph>)
      5529   => 16#68B9#,  --  梹 (<CJK Ideograph>)
      5530   => 16#6874#,  --  桴 (<CJK Ideograph>)
      5531   => 16#68B5#,  --  梵 (<CJK Ideograph>)
      5532   => 16#68A0#,  --  梠 (<CJK Ideograph>)
      5533   => 16#68BA#,  --  梺 (<CJK Ideograph>)
      5534   => 16#690F#,  --  椏 (<CJK Ideograph>)
      5535   => 16#688D#,  --  梍 (<CJK Ideograph>)
      5536   => 16#687E#,  --  桾 (<CJK Ideograph>)
      5537   => 16#6901#,  --  椁 (<CJK Ideograph>)
      5538   => 16#68CA#,  --  棊 (<CJK Ideograph>)
      5539   => 16#6908#,  --  椈 (<CJK Ideograph>)
      5540   => 16#68D8#,  --  棘 (<CJK Ideograph>)
      5541   => 16#6922#,  --  椢 (<CJK Ideograph>)
      5542   => 16#6926#,  --  椦 (<CJK Ideograph>)
      5543   => 16#68E1#,  --  棡 (<CJK Ideograph>)
      5544   => 16#690C#,  --  椌 (<CJK Ideograph>)
      5545   => 16#68CD#,  --  棍 (<CJK Ideograph>)
      5546   => 16#68D4#,  --  棔 (<CJK Ideograph>)
      5547   => 16#68E7#,  --  棧 (<CJK Ideograph>)
      5548   => 16#68D5#,  --  棕 (<CJK Ideograph>)
      5549   => 16#6936#,  --  椶 (<CJK Ideograph>)
      5550   => 16#6912#,  --  椒 (<CJK Ideograph>)
      5551   => 16#6904#,  --  椄 (<CJK Ideograph>)
      5552   => 16#68D7#,  --  棗 (<CJK Ideograph>)
      5553   => 16#68E3#,  --  棣 (<CJK Ideograph>)
      5554   => 16#6925#,  --  椥 (<CJK Ideograph>)
      5555   => 16#68F9#,  --  棹 (<CJK Ideograph>)
      5556   => 16#68E0#,  --  棠 (<CJK Ideograph>)
      5557   => 16#68EF#,  --  棯 (<CJK Ideograph>)
      5558   => 16#6928#,  --  椨 (<CJK Ideograph>)
      5559   => 16#692A#,  --  椪 (<CJK Ideograph>)
      5560   => 16#691A#,  --  椚 (<CJK Ideograph>)
      5561   => 16#6923#,  --  椣 (<CJK Ideograph>)
      5562   => 16#6921#,  --  椡 (<CJK Ideograph>)
      5563   => 16#68C6#,  --  棆 (<CJK Ideograph>)
      5564   => 16#6979#,  --  楹 (<CJK Ideograph>)
      5565   => 16#6977#,  --  楷 (<CJK Ideograph>)
      5566   => 16#695C#,  --  楜 (<CJK Ideograph>)
      5567   => 16#6978#,  --  楸 (<CJK Ideograph>)
      5568   => 16#696B#,  --  楫 (<CJK Ideograph>)
      5569   => 16#6954#,  --  楔 (<CJK Ideograph>)
      5570   => 16#697E#,  --  楾 (<CJK Ideograph>)
      5571   => 16#696E#,  --  楮 (<CJK Ideograph>)
      5572   => 16#6939#,  --  椹 (<CJK Ideograph>)
      5573   => 16#6974#,  --  楴 (<CJK Ideograph>)
      5574   => 16#693D#,  --  椽 (<CJK Ideograph>)
      5575   => 16#6959#,  --  楙 (<CJK Ideograph>)
      5576   => 16#6930#,  --  椰 (<CJK Ideograph>)
      5577   => 16#6961#,  --  楡 (<CJK Ideograph>)
      5578   => 16#695E#,  --  楞 (<CJK Ideograph>)
      5579   => 16#695D#,  --  楝 (<CJK Ideograph>)
      5580   => 16#6981#,  --  榁 (<CJK Ideograph>)
      5581   => 16#696A#,  --  楪 (<CJK Ideograph>)
      5582   => 16#69B2#,  --  榲 (<CJK Ideograph>)
      5583   => 16#69AE#,  --  榮 (<CJK Ideograph>)
      5584   => 16#69D0#,  --  槐 (<CJK Ideograph>)
      5585   => 16#69BF#,  --  榿 (<CJK Ideograph>)
      5586   => 16#69C1#,  --  槁 (<CJK Ideograph>)
      5587   => 16#69D3#,  --  槓 (<CJK Ideograph>)
      5588   => 16#69BE#,  --  榾 (<CJK Ideograph>)
      5589   => 16#69CE#,  --  槎 (<CJK Ideograph>)
      5590   => 16#5BE8#,  --  寨 (<CJK Ideograph>)
      5591   => 16#69CA#,  --  槊 (<CJK Ideograph>)
      5592   => 16#69DD#,  --  槝 (<CJK Ideograph>)
      5593   => 16#69BB#,  --  榻 (<CJK Ideograph>)
      5594   => 16#69C3#,  --  槃 (<CJK Ideograph>)
      5595   => 16#69A7#,  --  榧 (<CJK Ideograph>)
      5596   => 16#6A2E#,  --  樮 (<CJK Ideograph>)
      5597   => 16#6991#,  --  榑 (<CJK Ideograph>)
      5598   => 16#69A0#,  --  榠 (<CJK Ideograph>)
      5599   => 16#699C#,  --  榜 (<CJK Ideograph>)
      5600   => 16#6995#,  --  榕 (<CJK Ideograph>)
      5601   => 16#69B4#,  --  榴 (<CJK Ideograph>)
      5602   => 16#69DE#,  --  槞 (<CJK Ideograph>)
      5603   => 16#69E8#,  --  槨 (<CJK Ideograph>)
      5604   => 16#6A02#,  --  樂 (<CJK Ideograph>)
      5605   => 16#6A1B#,  --  樛 (<CJK Ideograph>)
      5606   => 16#69FF#,  --  槿 (<CJK Ideograph>)
      5607   => 16#6B0A#,  --  權 (<CJK Ideograph>)
      5608   => 16#69F9#,  --  槹 (<CJK Ideograph>)
      5609   => 16#69F2#,  --  槲 (<CJK Ideograph>)
      5610   => 16#69E7#,  --  槧 (<CJK Ideograph>)
      5611   => 16#6A05#,  --  樅 (<CJK Ideograph>)
      5612   => 16#69B1#,  --  榱 (<CJK Ideograph>)
      5613   => 16#6A1E#,  --  樞 (<CJK Ideograph>)
      5614   => 16#69ED#,  --  槭 (<CJK Ideograph>)
      5615   => 16#6A14#,  --  樔 (<CJK Ideograph>)
      5616   => 16#69EB#,  --  槫 (<CJK Ideograph>)
      5617   => 16#6A0A#,  --  樊 (<CJK Ideograph>)
      5618   => 16#6A12#,  --  樒 (<CJK Ideograph>)
      5619   => 16#6AC1#,  --  櫁 (<CJK Ideograph>)
      5620   => 16#6A23#,  --  樣 (<CJK Ideograph>)
      5621   => 16#6A13#,  --  樓 (<CJK Ideograph>)
      5622   => 16#6A44#,  --  橄 (<CJK Ideograph>)
      5623   => 16#6A0C#,  --  樌 (<CJK Ideograph>)
      5624   => 16#6A72#,  --  橲 (<CJK Ideograph>)
      5625   => 16#6A36#,  --  樶 (<CJK Ideograph>)
      5626   => 16#6A78#,  --  橸 (<CJK Ideograph>)
      5627   => 16#6A47#,  --  橇 (<CJK Ideograph>)
      5628   => 16#6A62#,  --  橢 (<CJK Ideograph>)
      5629   => 16#6A59#,  --  橙 (<CJK Ideograph>)
      5630   => 16#6A66#,  --  橦 (<CJK Ideograph>)
      5631   => 16#6A48#,  --  橈 (<CJK Ideograph>)
      5632   => 16#6A38#,  --  樸 (<CJK Ideograph>)
      5633   => 16#6A22#,  --  樢 (<CJK Ideograph>)
      5634   => 16#6A90#,  --  檐 (<CJK Ideograph>)
      5635   => 16#6A8D#,  --  檍 (<CJK Ideograph>)
      5636   => 16#6AA0#,  --  檠 (<CJK Ideograph>)
      5637   => 16#6A84#,  --  檄 (<CJK Ideograph>)
      5638   => 16#6AA2#,  --  檢 (<CJK Ideograph>)
      5639   => 16#6AA3#,  --  檣 (<CJK Ideograph>)
      5640   => 16#6A97#,  --  檗 (<CJK Ideograph>)
      5641   => 16#8617#,  --  蘗 (<CJK Ideograph>)
      5642   => 16#6ABB#,  --  檻 (<CJK Ideograph>)
      5643   => 16#6AC3#,  --  櫃 (<CJK Ideograph>)
      5644   => 16#6AC2#,  --  櫂 (<CJK Ideograph>)
      5645   => 16#6AB8#,  --  檸 (<CJK Ideograph>)
      5646   => 16#6AB3#,  --  檳 (<CJK Ideograph>)
      5647   => 16#6AAC#,  --  檬 (<CJK Ideograph>)
      5648   => 16#6ADE#,  --  櫞 (<CJK Ideograph>)
      5649   => 16#6AD1#,  --  櫑 (<CJK Ideograph>)
      5650   => 16#6ADF#,  --  櫟 (<CJK Ideograph>)
      5651   => 16#6AAA#,  --  檪 (<CJK Ideograph>)
      5652   => 16#6ADA#,  --  櫚 (<CJK Ideograph>)
      5653   => 16#6AEA#,  --  櫪 (<CJK Ideograph>)
      5654   => 16#6AFB#,  --  櫻 (<CJK Ideograph>)
      5655   => 16#6B05#,  --  欅 (<CJK Ideograph>)
      5656   => 16#8616#,  --  蘖 (<CJK Ideograph>)
      5657   => 16#6AFA#,  --  櫺 (<CJK Ideograph>)
      5658   => 16#6B12#,  --  欒 (<CJK Ideograph>)
      5659   => 16#6B16#,  --  欖 (<CJK Ideograph>)
      5660   => 16#9B31#,  --  鬱 (<CJK Ideograph>)
      5661   => 16#6B1F#,  --  欟 (<CJK Ideograph>)
      5662   => 16#6B38#,  --  欸 (<CJK Ideograph>)
      5663   => 16#6B37#,  --  欷 (<CJK Ideograph>)
      5664   => 16#76DC#,  --  盜 (<CJK Ideograph>)
      5665   => 16#6B39#,  --  欹 (<CJK Ideograph>)
      5666   => 16#98EE#,  --  飮 (<CJK Ideograph>)
      5667   => 16#6B47#,  --  歇 (<CJK Ideograph>)
      5668   => 16#6B43#,  --  歃 (<CJK Ideograph>)
      5669   => 16#6B49#,  --  歉 (<CJK Ideograph>)
      5670   => 16#6B50#,  --  歐 (<CJK Ideograph>)
      5671   => 16#6B59#,  --  歙 (<CJK Ideograph>)
      5672   => 16#6B54#,  --  歔 (<CJK Ideograph>)
      5673   => 16#6B5B#,  --  歛 (<CJK Ideograph>)
      5674   => 16#6B5F#,  --  歟 (<CJK Ideograph>)
      5675   => 16#6B61#,  --  歡 (<CJK Ideograph>)
      5676   => 16#6B78#,  --  歸 (<CJK Ideograph>)
      5677   => 16#6B79#,  --  歹 (<CJK Ideograph>)
      5678   => 16#6B7F#,  --  歿 (<CJK Ideograph>)
      5679   => 16#6B80#,  --  殀 (<CJK Ideograph>)
      5680   => 16#6B84#,  --  殄 (<CJK Ideograph>)
      5681   => 16#6B83#,  --  殃 (<CJK Ideograph>)
      5682   => 16#6B8D#,  --  殍 (<CJK Ideograph>)
      5683   => 16#6B98#,  --  殘 (<CJK Ideograph>)
      5684   => 16#6B95#,  --  殕 (<CJK Ideograph>)
      5685   => 16#6B9E#,  --  殞 (<CJK Ideograph>)
      5686   => 16#6BA4#,  --  殤 (<CJK Ideograph>)
      5687   => 16#6BAA#,  --  殪 (<CJK Ideograph>)
      5688   => 16#6BAB#,  --  殫 (<CJK Ideograph>)
      5689   => 16#6BAF#,  --  殯 (<CJK Ideograph>)
      5690   => 16#6BB2#,  --  殲 (<CJK Ideograph>)
      5691   => 16#6BB1#,  --  殱 (<CJK Ideograph>)
      5692   => 16#6BB3#,  --  殳 (<CJK Ideograph>)
      5693   => 16#6BB7#,  --  殷 (<CJK Ideograph>)
      5694   => 16#6BBC#,  --  殼 (<CJK Ideograph>)
      5695   => 16#6BC6#,  --  毆 (<CJK Ideograph>)
      5696   => 16#6BCB#,  --  毋 (<CJK Ideograph>)
      5697   => 16#6BD3#,  --  毓 (<CJK Ideograph>)
      5698   => 16#6BDF#,  --  毟 (<CJK Ideograph>)
      5699   => 16#6BEC#,  --  毬 (<CJK Ideograph>)
      5700   => 16#6BEB#,  --  毫 (<CJK Ideograph>)
      5701   => 16#6BF3#,  --  毳 (<CJK Ideograph>)
      5702   => 16#6BEF#,  --  毯 (<CJK Ideograph>)
      5703   => 16#9EBE#,  --  麾 (<CJK Ideograph>)
      5704   => 16#6C08#,  --  氈 (<CJK Ideograph>)
      5705   => 16#6C13#,  --  氓 (<CJK Ideograph>)
      5706   => 16#6C14#,  --  气 (<CJK Ideograph>)
      5707   => 16#6C1B#,  --  氛 (<CJK Ideograph>)
      5708   => 16#6C24#,  --  氤 (<CJK Ideograph>)
      5709   => 16#6C23#,  --  氣 (<CJK Ideograph>)
      5710   => 16#6C5E#,  --  汞 (<CJK Ideograph>)
      5711   => 16#6C55#,  --  汕 (<CJK Ideograph>)
      5712   => 16#6C62#,  --  汢 (<CJK Ideograph>)
      5713   => 16#6C6A#,  --  汪 (<CJK Ideograph>)
      5714   => 16#6C82#,  --  沂 (<CJK Ideograph>)
      5715   => 16#6C8D#,  --  沍 (<CJK Ideograph>)
      5716   => 16#6C9A#,  --  沚 (<CJK Ideograph>)
      5717   => 16#6C81#,  --  沁 (<CJK Ideograph>)
      5718   => 16#6C9B#,  --  沛 (<CJK Ideograph>)
      5719   => 16#6C7E#,  --  汾 (<CJK Ideograph>)
      5720   => 16#6C68#,  --  汨 (<CJK Ideograph>)
      5721   => 16#6C73#,  --  汳 (<CJK Ideograph>)
      5722   => 16#6C92#,  --  沒 (<CJK Ideograph>)
      5723   => 16#6C90#,  --  沐 (<CJK Ideograph>)
      5724   => 16#6CC4#,  --  泄 (<CJK Ideograph>)
      5725   => 16#6CF1#,  --  泱 (<CJK Ideograph>)
      5726   => 16#6CD3#,  --  泓 (<CJK Ideograph>)
      5727   => 16#6CBD#,  --  沽 (<CJK Ideograph>)
      5728   => 16#6CD7#,  --  泗 (<CJK Ideograph>)
      5729   => 16#6CC5#,  --  泅 (<CJK Ideograph>)
      5730   => 16#6CDD#,  --  泝 (<CJK Ideograph>)
      5731   => 16#6CAE#,  --  沮 (<CJK Ideograph>)
      5732   => 16#6CB1#,  --  沱 (<CJK Ideograph>)
      5733   => 16#6CBE#,  --  沾 (<CJK Ideograph>)
      5734   => 16#6CBA#,  --  沺 (<CJK Ideograph>)
      5735   => 16#6CDB#,  --  泛 (<CJK Ideograph>)
      5736   => 16#6CEF#,  --  泯 (<CJK Ideograph>)
      5737   => 16#6CD9#,  --  泙 (<CJK Ideograph>)
      5738   => 16#6CEA#,  --  泪 (<CJK Ideograph>)
      5739   => 16#6D1F#,  --  洟 (<CJK Ideograph>)
      5740   => 16#884D#,  --  衍 (<CJK Ideograph>)
      5741   => 16#6D36#,  --  洶 (<CJK Ideograph>)
      5742   => 16#6D2B#,  --  洫 (<CJK Ideograph>)
      5743   => 16#6D3D#,  --  洽 (<CJK Ideograph>)
      5744   => 16#6D38#,  --  洸 (<CJK Ideograph>)
      5745   => 16#6D19#,  --  洙 (<CJK Ideograph>)
      5746   => 16#6D35#,  --  洵 (<CJK Ideograph>)
      5747   => 16#6D33#,  --  洳 (<CJK Ideograph>)
      5748   => 16#6D12#,  --  洒 (<CJK Ideograph>)
      5749   => 16#6D0C#,  --  洌 (<CJK Ideograph>)
      5750   => 16#6D63#,  --  浣 (<CJK Ideograph>)
      5751   => 16#6D93#,  --  涓 (<CJK Ideograph>)
      5752   => 16#6D64#,  --  浤 (<CJK Ideograph>)
      5753   => 16#6D5A#,  --  浚 (<CJK Ideograph>)
      5754   => 16#6D79#,  --  浹 (<CJK Ideograph>)
      5755   => 16#6D59#,  --  浙 (<CJK Ideograph>)
      5756   => 16#6D8E#,  --  涎 (<CJK Ideograph>)
      5757   => 16#6D95#,  --  涕 (<CJK Ideograph>)
      5758   => 16#6FE4#,  --  濤 (<CJK Ideograph>)
      5759   => 16#6D85#,  --  涅 (<CJK Ideograph>)
      5760   => 16#6DF9#,  --  淹 (<CJK Ideograph>)
      5761   => 16#6E15#,  --  渕 (<CJK Ideograph>)
      5762   => 16#6E0A#,  --  渊 (<CJK Ideograph>)
      5763   => 16#6DB5#,  --  涵 (<CJK Ideograph>)
      5764   => 16#6DC7#,  --  淇 (<CJK Ideograph>)
      5765   => 16#6DE6#,  --  淦 (<CJK Ideograph>)
      5766   => 16#6DB8#,  --  涸 (<CJK Ideograph>)
      5767   => 16#6DC6#,  --  淆 (<CJK Ideograph>)
      5768   => 16#6DEC#,  --  淬 (<CJK Ideograph>)
      5769   => 16#6DDE#,  --  淞 (<CJK Ideograph>)
      5770   => 16#6DCC#,  --  淌 (<CJK Ideograph>)
      5771   => 16#6DE8#,  --  淨 (<CJK Ideograph>)
      5772   => 16#6DD2#,  --  淒 (<CJK Ideograph>)
      5773   => 16#6DC5#,  --  淅 (<CJK Ideograph>)
      5774   => 16#6DFA#,  --  淺 (<CJK Ideograph>)
      5775   => 16#6DD9#,  --  淙 (<CJK Ideograph>)
      5776   => 16#6DE4#,  --  淤 (<CJK Ideograph>)
      5777   => 16#6DD5#,  --  淕 (<CJK Ideograph>)
      5778   => 16#6DEA#,  --  淪 (<CJK Ideograph>)
      5779   => 16#6DEE#,  --  淮 (<CJK Ideograph>)
      5780   => 16#6E2D#,  --  渭 (<CJK Ideograph>)
      5781   => 16#6E6E#,  --  湮 (<CJK Ideograph>)
      5782   => 16#6E2E#,  --  渮 (<CJK Ideograph>)
      5783   => 16#6E19#,  --  渙 (<CJK Ideograph>)
      5784   => 16#6E72#,  --  湲 (<CJK Ideograph>)
      5785   => 16#6E5F#,  --  湟 (<CJK Ideograph>)
      5786   => 16#6E3E#,  --  渾 (<CJK Ideograph>)
      5787   => 16#6E23#,  --  渣 (<CJK Ideograph>)
      5788   => 16#6E6B#,  --  湫 (<CJK Ideograph>)
      5789   => 16#6E2B#,  --  渫 (<CJK Ideograph>)
      5790   => 16#6E76#,  --  湶 (<CJK Ideograph>)
      5791   => 16#6E4D#,  --  湍 (<CJK Ideograph>)
      5792   => 16#6E1F#,  --  渟 (<CJK Ideograph>)
      5793   => 16#6E43#,  --  湃 (<CJK Ideograph>)
      5794   => 16#6E3A#,  --  渺 (<CJK Ideograph>)
      5795   => 16#6E4E#,  --  湎 (<CJK Ideograph>)
      5796   => 16#6E24#,  --  渤 (<CJK Ideograph>)
      5797   => 16#6EFF#,  --  滿 (<CJK Ideograph>)
      5798   => 16#6E1D#,  --  渝 (<CJK Ideograph>)
      5799   => 16#6E38#,  --  游 (<CJK Ideograph>)
      5800   => 16#6E82#,  --  溂 (<CJK Ideograph>)
      5801   => 16#6EAA#,  --  溪 (<CJK Ideograph>)
      5802   => 16#6E98#,  --  溘 (<CJK Ideograph>)
      5803   => 16#6EC9#,  --  滉 (<CJK Ideograph>)
      5804   => 16#6EB7#,  --  溷 (<CJK Ideograph>)
      5805   => 16#6ED3#,  --  滓 (<CJK Ideograph>)
      5806   => 16#6EBD#,  --  溽 (<CJK Ideograph>)
      5807   => 16#6EAF#,  --  溯 (<CJK Ideograph>)
      5808   => 16#6EC4#,  --  滄 (<CJK Ideograph>)
      5809   => 16#6EB2#,  --  溲 (<CJK Ideograph>)
      5810   => 16#6ED4#,  --  滔 (<CJK Ideograph>)
      5811   => 16#6ED5#,  --  滕 (<CJK Ideograph>)
      5812   => 16#6E8F#,  --  溏 (<CJK Ideograph>)
      5813   => 16#6EA5#,  --  溥 (<CJK Ideograph>)
      5814   => 16#6EC2#,  --  滂 (<CJK Ideograph>)
      5815   => 16#6E9F#,  --  溟 (<CJK Ideograph>)
      5816   => 16#6F41#,  --  潁 (<CJK Ideograph>)
      5817   => 16#6F11#,  --  漑 (<CJK Ideograph>)
      5818   => 16#704C#,  --  灌 (<CJK Ideograph>)
      5819   => 16#6EEC#,  --  滬 (<CJK Ideograph>)
      5820   => 16#6EF8#,  --  滸 (<CJK Ideograph>)
      5821   => 16#6EFE#,  --  滾 (<CJK Ideograph>)
      5822   => 16#6F3F#,  --  漿 (<CJK Ideograph>)
      5823   => 16#6EF2#,  --  滲 (<CJK Ideograph>)
      5824   => 16#6F31#,  --  漱 (<CJK Ideograph>)
      5825   => 16#6EEF#,  --  滯 (<CJK Ideograph>)
      5826   => 16#6F32#,  --  漲 (<CJK Ideograph>)
      5827   => 16#6ECC#,  --  滌 (<CJK Ideograph>)
      5828   => 16#6F3E#,  --  漾 (<CJK Ideograph>)
      5829   => 16#6F13#,  --  漓 (<CJK Ideograph>)
      5830   => 16#6EF7#,  --  滷 (<CJK Ideograph>)
      5831   => 16#6F86#,  --  澆 (<CJK Ideograph>)
      5832   => 16#6F7A#,  --  潺 (<CJK Ideograph>)
      5833   => 16#6F78#,  --  潸 (<CJK Ideograph>)
      5834   => 16#6F81#,  --  澁 (<CJK Ideograph>)
      5835   => 16#6F80#,  --  澀 (<CJK Ideograph>)
      5836   => 16#6F6F#,  --  潯 (<CJK Ideograph>)
      5837   => 16#6F5B#,  --  潛 (<CJK Ideograph>)
      5838   => 16#6FF3#,  --  濳 (<CJK Ideograph>)
      5839   => 16#6F6D#,  --  潭 (<CJK Ideograph>)
      5840   => 16#6F82#,  --  澂 (<CJK Ideograph>)
      5841   => 16#6F7C#,  --  潼 (<CJK Ideograph>)
      5842   => 16#6F58#,  --  潘 (<CJK Ideograph>)
      5843   => 16#6F8E#,  --  澎 (<CJK Ideograph>)
      5844   => 16#6F91#,  --  澑 (<CJK Ideograph>)
      5845   => 16#6FC2#,  --  濂 (<CJK Ideograph>)
      5846   => 16#6F66#,  --  潦 (<CJK Ideograph>)
      5847   => 16#6FB3#,  --  澳 (<CJK Ideograph>)
      5848   => 16#6FA3#,  --  澣 (<CJK Ideograph>)
      5849   => 16#6FA1#,  --  澡 (<CJK Ideograph>)
      5850   => 16#6FA4#,  --  澤 (<CJK Ideograph>)
      5851   => 16#6FB9#,  --  澹 (<CJK Ideograph>)
      5852   => 16#6FC6#,  --  濆 (<CJK Ideograph>)
      5853   => 16#6FAA#,  --  澪 (<CJK Ideograph>)
      5854   => 16#6FDF#,  --  濟 (<CJK Ideograph>)
      5855   => 16#6FD5#,  --  濕 (<CJK Ideograph>)
      5856   => 16#6FEC#,  --  濬 (<CJK Ideograph>)
      5857   => 16#6FD4#,  --  濔 (<CJK Ideograph>)
      5858   => 16#6FD8#,  --  濘 (<CJK Ideograph>)
      5859   => 16#6FF1#,  --  濱 (<CJK Ideograph>)
      5860   => 16#6FEE#,  --  濮 (<CJK Ideograph>)
      5861   => 16#6FDB#,  --  濛 (<CJK Ideograph>)
      5862   => 16#7009#,  --  瀉 (<CJK Ideograph>)
      5863   => 16#700B#,  --  瀋 (<CJK Ideograph>)
      5864   => 16#6FFA#,  --  濺 (<CJK Ideograph>)
      5865   => 16#7011#,  --  瀑 (<CJK Ideograph>)
      5866   => 16#7001#,  --  瀁 (<CJK Ideograph>)
      5867   => 16#700F#,  --  瀏 (<CJK Ideograph>)
      5868   => 16#6FFE#,  --  濾 (<CJK Ideograph>)
      5869   => 16#701B#,  --  瀛 (<CJK Ideograph>)
      5870   => 16#701A#,  --  瀚 (<CJK Ideograph>)
      5871   => 16#6F74#,  --  潴 (<CJK Ideograph>)
      5872   => 16#701D#,  --  瀝 (<CJK Ideograph>)
      5873   => 16#7018#,  --  瀘 (<CJK Ideograph>)
      5874   => 16#701F#,  --  瀟 (<CJK Ideograph>)
      5875   => 16#7030#,  --  瀰 (<CJK Ideograph>)
      5876   => 16#703E#,  --  瀾 (<CJK Ideograph>)
      5877   => 16#7032#,  --  瀲 (<CJK Ideograph>)
      5878   => 16#7051#,  --  灑 (<CJK Ideograph>)
      5879   => 16#7063#,  --  灣 (<CJK Ideograph>)
      5880   => 16#7099#,  --  炙 (<CJK Ideograph>)
      5881   => 16#7092#,  --  炒 (<CJK Ideograph>)
      5882   => 16#70AF#,  --  炯 (<CJK Ideograph>)
      5883   => 16#70F1#,  --  烱 (<CJK Ideograph>)
      5884   => 16#70AC#,  --  炬 (<CJK Ideograph>)
      5885   => 16#70B8#,  --  炸 (<CJK Ideograph>)
      5886   => 16#70B3#,  --  炳 (<CJK Ideograph>)
      5887   => 16#70AE#,  --  炮 (<CJK Ideograph>)
      5888   => 16#70DF#,  --  烟 (<CJK Ideograph>)
      5889   => 16#70CB#,  --  烋 (<CJK Ideograph>)
      5890   => 16#70DD#,  --  烝 (<CJK Ideograph>)
      5891   => 16#70D9#,  --  烙 (<CJK Ideograph>)
      5892   => 16#7109#,  --  焉 (<CJK Ideograph>)
      5893   => 16#70FD#,  --  烽 (<CJK Ideograph>)
      5894   => 16#711C#,  --  焜 (<CJK Ideograph>)
      5895   => 16#7119#,  --  焙 (<CJK Ideograph>)
      5896   => 16#7165#,  --  煥 (<CJK Ideograph>)
      5897   => 16#7155#,  --  煕 (<CJK Ideograph>)
      5898   => 16#7188#,  --  熈 (<CJK Ideograph>)
      5899   => 16#7166#,  --  煦 (<CJK Ideograph>)
      5900   => 16#7162#,  --  煢 (<CJK Ideograph>)
      5901   => 16#714C#,  --  煌 (<CJK Ideograph>)
      5902   => 16#7156#,  --  煖 (<CJK Ideograph>)
      5903   => 16#716C#,  --  煬 (<CJK Ideograph>)
      5904   => 16#718F#,  --  熏 (<CJK Ideograph>)
      5905   => 16#71FB#,  --  燻 (<CJK Ideograph>)
      5906   => 16#7184#,  --  熄 (<CJK Ideograph>)
      5907   => 16#7195#,  --  熕 (<CJK Ideograph>)
      5908   => 16#71A8#,  --  熨 (<CJK Ideograph>)
      5909   => 16#71AC#,  --  熬 (<CJK Ideograph>)
      5910   => 16#71D7#,  --  燗 (<CJK Ideograph>)
      5911   => 16#71B9#,  --  熹 (<CJK Ideograph>)
      5912   => 16#71BE#,  --  熾 (<CJK Ideograph>)
      5913   => 16#71D2#,  --  燒 (<CJK Ideograph>)
      5914   => 16#71C9#,  --  燉 (<CJK Ideograph>)
      5915   => 16#71D4#,  --  燔 (<CJK Ideograph>)
      5916   => 16#71CE#,  --  燎 (<CJK Ideograph>)
      5917   => 16#71E0#,  --  燠 (<CJK Ideograph>)
      5918   => 16#71EC#,  --  燬 (<CJK Ideograph>)
      5919   => 16#71E7#,  --  燧 (<CJK Ideograph>)
      5920   => 16#71F5#,  --  燵 (<CJK Ideograph>)
      5921   => 16#71FC#,  --  燼 (<CJK Ideograph>)
      5922   => 16#71F9#,  --  燹 (<CJK Ideograph>)
      5923   => 16#71FF#,  --  燿 (<CJK Ideograph>)
      5924   => 16#720D#,  --  爍 (<CJK Ideograph>)
      5925   => 16#7210#,  --  爐 (<CJK Ideograph>)
      5926   => 16#721B#,  --  爛 (<CJK Ideograph>)
      5927   => 16#7228#,  --  爨 (<CJK Ideograph>)
      5928   => 16#722D#,  --  爭 (<CJK Ideograph>)
      5929   => 16#722C#,  --  爬 (<CJK Ideograph>)
      5930   => 16#7230#,  --  爰 (<CJK Ideograph>)
      5931   => 16#7232#,  --  爲 (<CJK Ideograph>)
      5932   => 16#723B#,  --  爻 (<CJK Ideograph>)
      5933   => 16#723C#,  --  爼 (<CJK Ideograph>)
      5934   => 16#723F#,  --  爿 (<CJK Ideograph>)
      5935   => 16#7240#,  --  牀 (<CJK Ideograph>)
      5936   => 16#7246#,  --  牆 (<CJK Ideograph>)
      5937   => 16#724B#,  --  牋 (<CJK Ideograph>)
      5938   => 16#7258#,  --  牘 (<CJK Ideograph>)
      5939   => 16#7274#,  --  牴 (<CJK Ideograph>)
      5940   => 16#727E#,  --  牾 (<CJK Ideograph>)
      5941   => 16#7282#,  --  犂 (<CJK Ideograph>)
      5942   => 16#7281#,  --  犁 (<CJK Ideograph>)
      5943   => 16#7287#,  --  犇 (<CJK Ideograph>)
      5944   => 16#7292#,  --  犒 (<CJK Ideograph>)
      5945   => 16#7296#,  --  犖 (<CJK Ideograph>)
      5946   => 16#72A2#,  --  犢 (<CJK Ideograph>)
      5947   => 16#72A7#,  --  犧 (<CJK Ideograph>)
      5948   => 16#72B9#,  --  犹 (<CJK Ideograph>)
      5949   => 16#72B2#,  --  犲 (<CJK Ideograph>)
      5950   => 16#72C3#,  --  狃 (<CJK Ideograph>)
      5951   => 16#72C6#,  --  狆 (<CJK Ideograph>)
      5952   => 16#72C4#,  --  狄 (<CJK Ideograph>)
      5953   => 16#72CE#,  --  狎 (<CJK Ideograph>)
      5954   => 16#72D2#,  --  狒 (<CJK Ideograph>)
      5955   => 16#72E2#,  --  狢 (<CJK Ideograph>)
      5956   => 16#72E0#,  --  狠 (<CJK Ideograph>)
      5957   => 16#72E1#,  --  狡 (<CJK Ideograph>)
      5958   => 16#72F9#,  --  狹 (<CJK Ideograph>)
      5959   => 16#72F7#,  --  狷 (<CJK Ideograph>)
      5960   => 16#500F#,  --  倏 (<CJK Ideograph>)
      5961   => 16#7317#,  --  猗 (<CJK Ideograph>)
      5962   => 16#730A#,  --  猊 (<CJK Ideograph>)
      5963   => 16#731C#,  --  猜 (<CJK Ideograph>)
      5964   => 16#7316#,  --  猖 (<CJK Ideograph>)
      5965   => 16#731D#,  --  猝 (<CJK Ideograph>)
      5966   => 16#7334#,  --  猴 (<CJK Ideograph>)
      5967   => 16#732F#,  --  猯 (<CJK Ideograph>)
      5968   => 16#7329#,  --  猩 (<CJK Ideograph>)
      5969   => 16#7325#,  --  猥 (<CJK Ideograph>)
      5970   => 16#733E#,  --  猾 (<CJK Ideograph>)
      5971   => 16#734E#,  --  獎 (<CJK Ideograph>)
      5972   => 16#734F#,  --  獏 (<CJK Ideograph>)
      5973   => 16#9ED8#,  --  默 (<CJK Ideograph>)
      5974   => 16#7357#,  --  獗 (<CJK Ideograph>)
      5975   => 16#736A#,  --  獪 (<CJK Ideograph>)
      5976   => 16#7368#,  --  獨 (<CJK Ideograph>)
      5977   => 16#7370#,  --  獰 (<CJK Ideograph>)
      5978   => 16#7378#,  --  獸 (<CJK Ideograph>)
      5979   => 16#7375#,  --  獵 (<CJK Ideograph>)
      5980   => 16#737B#,  --  獻 (<CJK Ideograph>)
      5981   => 16#737A#,  --  獺 (<CJK Ideograph>)
      5982   => 16#73C8#,  --  珈 (<CJK Ideograph>)
      5983   => 16#73B3#,  --  玳 (<CJK Ideograph>)
      5984   => 16#73CE#,  --  珎 (<CJK Ideograph>)
      5985   => 16#73BB#,  --  玻 (<CJK Ideograph>)
      5986   => 16#73C0#,  --  珀 (<CJK Ideograph>)
      5987   => 16#73E5#,  --  珥 (<CJK Ideograph>)
      5988   => 16#73EE#,  --  珮 (<CJK Ideograph>)
      5989   => 16#73DE#,  --  珞 (<CJK Ideograph>)
      5990   => 16#74A2#,  --  璢 (<CJK Ideograph>)
      5991   => 16#7405#,  --  琅 (<CJK Ideograph>)
      5992   => 16#746F#,  --  瑯 (<CJK Ideograph>)
      5993   => 16#7425#,  --  琥 (<CJK Ideograph>)
      5994   => 16#73F8#,  --  珸 (<CJK Ideograph>)
      5995   => 16#7432#,  --  琲 (<CJK Ideograph>)
      5996   => 16#743A#,  --  琺 (<CJK Ideograph>)
      5997   => 16#7455#,  --  瑕 (<CJK Ideograph>)
      5998   => 16#743F#,  --  琿 (<CJK Ideograph>)
      5999   => 16#745F#,  --  瑟 (<CJK Ideograph>)
      6000   => 16#7459#,  --  瑙 (<CJK Ideograph>)
      6001   => 16#7441#,  --  瑁 (<CJK Ideograph>)
      6002   => 16#745C#,  --  瑜 (<CJK Ideograph>)
      6003   => 16#7469#,  --  瑩 (<CJK Ideograph>)
      6004   => 16#7470#,  --  瑰 (<CJK Ideograph>)
      6005   => 16#7463#,  --  瑣 (<CJK Ideograph>)
      6006   => 16#746A#,  --  瑪 (<CJK Ideograph>)
      6007   => 16#7476#,  --  瑶 (<CJK Ideograph>)
      6008   => 16#747E#,  --  瑾 (<CJK Ideograph>)
      6009   => 16#748B#,  --  璋 (<CJK Ideograph>)
      6010   => 16#749E#,  --  璞 (<CJK Ideograph>)
      6011   => 16#74A7#,  --  璧 (<CJK Ideograph>)
      6012   => 16#74CA#,  --  瓊 (<CJK Ideograph>)
      6013   => 16#74CF#,  --  瓏 (<CJK Ideograph>)
      6014   => 16#74D4#,  --  瓔 (<CJK Ideograph>)
      6015   => 16#73F1#,  --  珱 (<CJK Ideograph>)
      6016   => 16#74E0#,  --  瓠 (<CJK Ideograph>)
      6017   => 16#74E3#,  --  瓣 (<CJK Ideograph>)
      6018   => 16#74E7#,  --  瓧 (<CJK Ideograph>)
      6019   => 16#74E9#,  --  瓩 (<CJK Ideograph>)
      6020   => 16#74EE#,  --  瓮 (<CJK Ideograph>)
      6021   => 16#74F2#,  --  瓲 (<CJK Ideograph>)
      6022   => 16#74F0#,  --  瓰 (<CJK Ideograph>)
      6023   => 16#74F1#,  --  瓱 (<CJK Ideograph>)
      6024   => 16#74F8#,  --  瓸 (<CJK Ideograph>)
      6025   => 16#74F7#,  --  瓷 (<CJK Ideograph>)
      6026   => 16#7504#,  --  甄 (<CJK Ideograph>)
      6027   => 16#7503#,  --  甃 (<CJK Ideograph>)
      6028   => 16#7505#,  --  甅 (<CJK Ideograph>)
      6029   => 16#750C#,  --  甌 (<CJK Ideograph>)
      6030   => 16#750E#,  --  甎 (<CJK Ideograph>)
      6031   => 16#750D#,  --  甍 (<CJK Ideograph>)
      6032   => 16#7515#,  --  甕 (<CJK Ideograph>)
      6033   => 16#7513#,  --  甓 (<CJK Ideograph>)
      6034   => 16#751E#,  --  甞 (<CJK Ideograph>)
      6035   => 16#7526#,  --  甦 (<CJK Ideograph>)
      6036   => 16#752C#,  --  甬 (<CJK Ideograph>)
      6037   => 16#753C#,  --  甼 (<CJK Ideograph>)
      6038   => 16#7544#,  --  畄 (<CJK Ideograph>)
      6039   => 16#754D#,  --  畍 (<CJK Ideograph>)
      6040   => 16#754A#,  --  畊 (<CJK Ideograph>)
      6041   => 16#7549#,  --  畉 (<CJK Ideograph>)
      6042   => 16#755B#,  --  畛 (<CJK Ideograph>)
      6043   => 16#7546#,  --  畆 (<CJK Ideograph>)
      6044   => 16#755A#,  --  畚 (<CJK Ideograph>)
      6045   => 16#7569#,  --  畩 (<CJK Ideograph>)
      6046   => 16#7564#,  --  畤 (<CJK Ideograph>)
      6047   => 16#7567#,  --  畧 (<CJK Ideograph>)
      6048   => 16#756B#,  --  畫 (<CJK Ideograph>)
      6049   => 16#756D#,  --  畭 (<CJK Ideograph>)
      6050   => 16#7578#,  --  畸 (<CJK Ideograph>)
      6051   => 16#7576#,  --  當 (<CJK Ideograph>)
      6052   => 16#7586#,  --  疆 (<CJK Ideograph>)
      6053   => 16#7587#,  --  疇 (<CJK Ideograph>)
      6054   => 16#7574#,  --  畴 (<CJK Ideograph>)
      6055   => 16#758A#,  --  疊 (<CJK Ideograph>)
      6056   => 16#7589#,  --  疉 (<CJK Ideograph>)
      6057   => 16#7582#,  --  疂 (<CJK Ideograph>)
      6058   => 16#7594#,  --  疔 (<CJK Ideograph>)
      6059   => 16#759A#,  --  疚 (<CJK Ideograph>)
      6060   => 16#759D#,  --  疝 (<CJK Ideograph>)
      6061   => 16#75A5#,  --  疥 (<CJK Ideograph>)
      6062   => 16#75A3#,  --  疣 (<CJK Ideograph>)
      6063   => 16#75C2#,  --  痂 (<CJK Ideograph>)
      6064   => 16#75B3#,  --  疳 (<CJK Ideograph>)
      6065   => 16#75C3#,  --  痃 (<CJK Ideograph>)
      6066   => 16#75B5#,  --  疵 (<CJK Ideograph>)
      6067   => 16#75BD#,  --  疽 (<CJK Ideograph>)
      6068   => 16#75B8#,  --  疸 (<CJK Ideograph>)
      6069   => 16#75BC#,  --  疼 (<CJK Ideograph>)
      6070   => 16#75B1#,  --  疱 (<CJK Ideograph>)
      6071   => 16#75CD#,  --  痍 (<CJK Ideograph>)
      6072   => 16#75CA#,  --  痊 (<CJK Ideograph>)
      6073   => 16#75D2#,  --  痒 (<CJK Ideograph>)
      6074   => 16#75D9#,  --  痙 (<CJK Ideograph>)
      6075   => 16#75E3#,  --  痣 (<CJK Ideograph>)
      6076   => 16#75DE#,  --  痞 (<CJK Ideograph>)
      6077   => 16#75FE#,  --  痾 (<CJK Ideograph>)
      6078   => 16#75FF#,  --  痿 (<CJK Ideograph>)
      6079   => 16#75FC#,  --  痼 (<CJK Ideograph>)
      6080   => 16#7601#,  --  瘁 (<CJK Ideograph>)
      6081   => 16#75F0#,  --  痰 (<CJK Ideograph>)
      6082   => 16#75FA#,  --  痺 (<CJK Ideograph>)
      6083   => 16#75F2#,  --  痲 (<CJK Ideograph>)
      6084   => 16#75F3#,  --  痳 (<CJK Ideograph>)
      6085   => 16#760B#,  --  瘋 (<CJK Ideograph>)
      6086   => 16#760D#,  --  瘍 (<CJK Ideograph>)
      6087   => 16#7609#,  --  瘉 (<CJK Ideograph>)
      6088   => 16#761F#,  --  瘟 (<CJK Ideograph>)
      6089   => 16#7627#,  --  瘧 (<CJK Ideograph>)
      6090   => 16#7620#,  --  瘠 (<CJK Ideograph>)
      6091   => 16#7621#,  --  瘡 (<CJK Ideograph>)
      6092   => 16#7622#,  --  瘢 (<CJK Ideograph>)
      6093   => 16#7624#,  --  瘤 (<CJK Ideograph>)
      6094   => 16#7634#,  --  瘴 (<CJK Ideograph>)
      6095   => 16#7630#,  --  瘰 (<CJK Ideograph>)
      6096   => 16#763B#,  --  瘻 (<CJK Ideograph>)
      6097   => 16#7647#,  --  癇 (<CJK Ideograph>)
      6098   => 16#7648#,  --  癈 (<CJK Ideograph>)
      6099   => 16#7646#,  --  癆 (<CJK Ideograph>)
      6100   => 16#765C#,  --  癜 (<CJK Ideograph>)
      6101   => 16#7658#,  --  癘 (<CJK Ideograph>)
      6102   => 16#7661#,  --  癡 (<CJK Ideograph>)
      6103   => 16#7662#,  --  癢 (<CJK Ideograph>)
      6104   => 16#7668#,  --  癨 (<CJK Ideograph>)
      6105   => 16#7669#,  --  癩 (<CJK Ideograph>)
      6106   => 16#766A#,  --  癪 (<CJK Ideograph>)
      6107   => 16#7667#,  --  癧 (<CJK Ideograph>)
      6108   => 16#766C#,  --  癬 (<CJK Ideograph>)
      6109   => 16#7670#,  --  癰 (<CJK Ideograph>)
      6110   => 16#7672#,  --  癲 (<CJK Ideograph>)
      6111   => 16#7676#,  --  癶 (<CJK Ideograph>)
      6112   => 16#7678#,  --  癸 (<CJK Ideograph>)
      6113   => 16#767C#,  --  發 (<CJK Ideograph>)
      6114   => 16#7680#,  --  皀 (<CJK Ideograph>)
      6115   => 16#7683#,  --  皃 (<CJK Ideograph>)
      6116   => 16#7688#,  --  皈 (<CJK Ideograph>)
      6117   => 16#768B#,  --  皋 (<CJK Ideograph>)
      6118   => 16#768E#,  --  皎 (<CJK Ideograph>)
      6119   => 16#7696#,  --  皖 (<CJK Ideograph>)
      6120   => 16#7693#,  --  皓 (<CJK Ideograph>)
      6121   => 16#7699#,  --  皙 (<CJK Ideograph>)
      6122   => 16#769A#,  --  皚 (<CJK Ideograph>)
      6123   => 16#76B0#,  --  皰 (<CJK Ideograph>)
      6124   => 16#76B4#,  --  皴 (<CJK Ideograph>)
      6125   => 16#76B8#,  --  皸 (<CJK Ideograph>)
      6126   => 16#76B9#,  --  皹 (<CJK Ideograph>)
      6127   => 16#76BA#,  --  皺 (<CJK Ideograph>)
      6128   => 16#76C2#,  --  盂 (<CJK Ideograph>)
      6129   => 16#76CD#,  --  盍 (<CJK Ideograph>)
      6130   => 16#76D6#,  --  盖 (<CJK Ideograph>)
      6131   => 16#76D2#,  --  盒 (<CJK Ideograph>)
      6132   => 16#76DE#,  --  盞 (<CJK Ideograph>)
      6133   => 16#76E1#,  --  盡 (<CJK Ideograph>)
      6134   => 16#76E5#,  --  盥 (<CJK Ideograph>)
      6135   => 16#76E7#,  --  盧 (<CJK Ideograph>)
      6136   => 16#76EA#,  --  盪 (<CJK Ideograph>)
      6137   => 16#862F#,  --  蘯 (<CJK Ideograph>)
      6138   => 16#76FB#,  --  盻 (<CJK Ideograph>)
      6139   => 16#7708#,  --  眈 (<CJK Ideograph>)
      6140   => 16#7707#,  --  眇 (<CJK Ideograph>)
      6141   => 16#7704#,  --  眄 (<CJK Ideograph>)
      6142   => 16#7729#,  --  眩 (<CJK Ideograph>)
      6143   => 16#7724#,  --  眤 (<CJK Ideograph>)
      6144   => 16#771E#,  --  眞 (<CJK Ideograph>)
      6145   => 16#7725#,  --  眥 (<CJK Ideograph>)
      6146   => 16#7726#,  --  眦 (<CJK Ideograph>)
      6147   => 16#771B#,  --  眛 (<CJK Ideograph>)
      6148   => 16#7737#,  --  眷 (<CJK Ideograph>)
      6149   => 16#7738#,  --  眸 (<CJK Ideograph>)
      6150   => 16#7747#,  --  睇 (<CJK Ideograph>)
      6151   => 16#775A#,  --  睚 (<CJK Ideograph>)
      6152   => 16#7768#,  --  睨 (<CJK Ideograph>)
      6153   => 16#776B#,  --  睫 (<CJK Ideograph>)
      6154   => 16#775B#,  --  睛 (<CJK Ideograph>)
      6155   => 16#7765#,  --  睥 (<CJK Ideograph>)
      6156   => 16#777F#,  --  睿 (<CJK Ideograph>)
      6157   => 16#777E#,  --  睾 (<CJK Ideograph>)
      6158   => 16#7779#,  --  睹 (<CJK Ideograph>)
      6159   => 16#778E#,  --  瞎 (<CJK Ideograph>)
      6160   => 16#778B#,  --  瞋 (<CJK Ideograph>)
      6161   => 16#7791#,  --  瞑 (<CJK Ideograph>)
      6162   => 16#77A0#,  --  瞠 (<CJK Ideograph>)
      6163   => 16#779E#,  --  瞞 (<CJK Ideograph>)
      6164   => 16#77B0#,  --  瞰 (<CJK Ideograph>)
      6165   => 16#77B6#,  --  瞶 (<CJK Ideograph>)
      6166   => 16#77B9#,  --  瞹 (<CJK Ideograph>)
      6167   => 16#77BF#,  --  瞿 (<CJK Ideograph>)
      6168   => 16#77BC#,  --  瞼 (<CJK Ideograph>)
      6169   => 16#77BD#,  --  瞽 (<CJK Ideograph>)
      6170   => 16#77BB#,  --  瞻 (<CJK Ideograph>)
      6171   => 16#77C7#,  --  矇 (<CJK Ideograph>)
      6172   => 16#77CD#,  --  矍 (<CJK Ideograph>)
      6173   => 16#77D7#,  --  矗 (<CJK Ideograph>)
      6174   => 16#77DA#,  --  矚 (<CJK Ideograph>)
      6175   => 16#77DC#,  --  矜 (<CJK Ideograph>)
      6176   => 16#77E3#,  --  矣 (<CJK Ideograph>)
      6177   => 16#77EE#,  --  矮 (<CJK Ideograph>)
      6178   => 16#77FC#,  --  矼 (<CJK Ideograph>)
      6179   => 16#780C#,  --  砌 (<CJK Ideograph>)
      6180   => 16#7812#,  --  砒 (<CJK Ideograph>)
      6181   => 16#7926#,  --  礦 (<CJK Ideograph>)
      6182   => 16#7820#,  --  砠 (<CJK Ideograph>)
      6183   => 16#792A#,  --  礪 (<CJK Ideograph>)
      6184   => 16#7845#,  --  硅 (<CJK Ideograph>)
      6185   => 16#788E#,  --  碎 (<CJK Ideograph>)
      6186   => 16#7874#,  --  硴 (<CJK Ideograph>)
      6187   => 16#7886#,  --  碆 (<CJK Ideograph>)
      6188   => 16#787C#,  --  硼 (<CJK Ideograph>)
      6189   => 16#789A#,  --  碚 (<CJK Ideograph>)
      6190   => 16#788C#,  --  碌 (<CJK Ideograph>)
      6191   => 16#78A3#,  --  碣 (<CJK Ideograph>)
      6192   => 16#78B5#,  --  碵 (<CJK Ideograph>)
      6193   => 16#78AA#,  --  碪 (<CJK Ideograph>)
      6194   => 16#78AF#,  --  碯 (<CJK Ideograph>)
      6195   => 16#78D1#,  --  磑 (<CJK Ideograph>)
      6196   => 16#78C6#,  --  磆 (<CJK Ideograph>)
      6197   => 16#78CB#,  --  磋 (<CJK Ideograph>)
      6198   => 16#78D4#,  --  磔 (<CJK Ideograph>)
      6199   => 16#78BE#,  --  碾 (<CJK Ideograph>)
      6200   => 16#78BC#,  --  碼 (<CJK Ideograph>)
      6201   => 16#78C5#,  --  磅 (<CJK Ideograph>)
      6202   => 16#78CA#,  --  磊 (<CJK Ideograph>)
      6203   => 16#78EC#,  --  磬 (<CJK Ideograph>)
      6204   => 16#78E7#,  --  磧 (<CJK Ideograph>)
      6205   => 16#78DA#,  --  磚 (<CJK Ideograph>)
      6206   => 16#78FD#,  --  磽 (<CJK Ideograph>)
      6207   => 16#78F4#,  --  磴 (<CJK Ideograph>)
      6208   => 16#7907#,  --  礇 (<CJK Ideograph>)
      6209   => 16#7912#,  --  礒 (<CJK Ideograph>)
      6210   => 16#7911#,  --  礑 (<CJK Ideograph>)
      6211   => 16#7919#,  --  礙 (<CJK Ideograph>)
      6212   => 16#792C#,  --  礬 (<CJK Ideograph>)
      6213   => 16#792B#,  --  礫 (<CJK Ideograph>)
      6214   => 16#7940#,  --  祀 (<CJK Ideograph>)
      6215   => 16#7960#,  --  祠 (<CJK Ideograph>)
      6216   => 16#7957#,  --  祗 (<CJK Ideograph>)
      6217   => 16#795F#,  --  祟 (<CJK Ideograph>)
      6218   => 16#795A#,  --  祚 (<CJK Ideograph>)
      6219   => 16#7955#,  --  祕 (<CJK Ideograph>)
      6220   => 16#7953#,  --  祓 (<CJK Ideograph>)
      6221   => 16#797A#,  --  祺 (<CJK Ideograph>)
      6222   => 16#797F#,  --  祿 (<CJK Ideograph>)
      6223   => 16#798A#,  --  禊 (<CJK Ideograph>)
      6224   => 16#799D#,  --  禝 (<CJK Ideograph>)
      6225   => 16#79A7#,  --  禧 (<CJK Ideograph>)
      6226   => 16#9F4B#,  --  齋 (<CJK Ideograph>)
      6227   => 16#79AA#,  --  禪 (<CJK Ideograph>)
      6228   => 16#79AE#,  --  禮 (<CJK Ideograph>)
      6229   => 16#79B3#,  --  禳 (<CJK Ideograph>)
      6230   => 16#79B9#,  --  禹 (<CJK Ideograph>)
      6231   => 16#79BA#,  --  禺 (<CJK Ideograph>)
      6232   => 16#79C9#,  --  秉 (<CJK Ideograph>)
      6233   => 16#79D5#,  --  秕 (<CJK Ideograph>)
      6234   => 16#79E7#,  --  秧 (<CJK Ideograph>)
      6235   => 16#79EC#,  --  秬 (<CJK Ideograph>)
      6236   => 16#79E1#,  --  秡 (<CJK Ideograph>)
      6237   => 16#79E3#,  --  秣 (<CJK Ideograph>)
      6238   => 16#7A08#,  --  稈 (<CJK Ideograph>)
      6239   => 16#7A0D#,  --  稍 (<CJK Ideograph>)
      6240   => 16#7A18#,  --  稘 (<CJK Ideograph>)
      6241   => 16#7A19#,  --  稙 (<CJK Ideograph>)
      6242   => 16#7A20#,  --  稠 (<CJK Ideograph>)
      6243   => 16#7A1F#,  --  稟 (<CJK Ideograph>)
      6244   => 16#7980#,  --  禀 (<CJK Ideograph>)
      6245   => 16#7A31#,  --  稱 (<CJK Ideograph>)
      6246   => 16#7A3B#,  --  稻 (<CJK Ideograph>)
      6247   => 16#7A3E#,  --  稾 (<CJK Ideograph>)
      6248   => 16#7A37#,  --  稷 (<CJK Ideograph>)
      6249   => 16#7A43#,  --  穃 (<CJK Ideograph>)
      6250   => 16#7A57#,  --  穗 (<CJK Ideograph>)
      6251   => 16#7A49#,  --  穉 (<CJK Ideograph>)
      6252   => 16#7A61#,  --  穡 (<CJK Ideograph>)
      6253   => 16#7A62#,  --  穢 (<CJK Ideograph>)
      6254   => 16#7A69#,  --  穩 (<CJK Ideograph>)
      6255   => 16#9F9D#,  --  龝 (<CJK Ideograph>)
      6256   => 16#7A70#,  --  穰 (<CJK Ideograph>)
      6257   => 16#7A79#,  --  穹 (<CJK Ideograph>)
      6258   => 16#7A7D#,  --  穽 (<CJK Ideograph>)
      6259   => 16#7A88#,  --  窈 (<CJK Ideograph>)
      6260   => 16#7A97#,  --  窗 (<CJK Ideograph>)
      6261   => 16#7A95#,  --  窕 (<CJK Ideograph>)
      6262   => 16#7A98#,  --  窘 (<CJK Ideograph>)
      6263   => 16#7A96#,  --  窖 (<CJK Ideograph>)
      6264   => 16#7AA9#,  --  窩 (<CJK Ideograph>)
      6265   => 16#7AC8#,  --  竈 (<CJK Ideograph>)
      6266   => 16#7AB0#,  --  窰 (<CJK Ideograph>)
      6267   => 16#7AB6#,  --  窶 (<CJK Ideograph>)
      6268   => 16#7AC5#,  --  竅 (<CJK Ideograph>)
      6269   => 16#7AC4#,  --  竄 (<CJK Ideograph>)
      6270   => 16#7ABF#,  --  窿 (<CJK Ideograph>)
      6271   => 16#9083#,  --  邃 (<CJK Ideograph>)
      6272   => 16#7AC7#,  --  竇 (<CJK Ideograph>)
      6273   => 16#7ACA#,  --  竊 (<CJK Ideograph>)
      6274   => 16#7ACD#,  --  竍 (<CJK Ideograph>)
      6275   => 16#7ACF#,  --  竏 (<CJK Ideograph>)
      6276   => 16#7AD5#,  --  竕 (<CJK Ideograph>)
      6277   => 16#7AD3#,  --  竓 (<CJK Ideograph>)
      6278   => 16#7AD9#,  --  站 (<CJK Ideograph>)
      6279   => 16#7ADA#,  --  竚 (<CJK Ideograph>)
      6280   => 16#7ADD#,  --  竝 (<CJK Ideograph>)
      6281   => 16#7AE1#,  --  竡 (<CJK Ideograph>)
      6282   => 16#7AE2#,  --  竢 (<CJK Ideograph>)
      6283   => 16#7AE6#,  --  竦 (<CJK Ideograph>)
      6284   => 16#7AED#,  --  竭 (<CJK Ideograph>)
      6285   => 16#7AF0#,  --  竰 (<CJK Ideograph>)
      6286   => 16#7B02#,  --  笂 (<CJK Ideograph>)
      6287   => 16#7B0F#,  --  笏 (<CJK Ideograph>)
      6288   => 16#7B0A#,  --  笊 (<CJK Ideograph>)
      6289   => 16#7B06#,  --  笆 (<CJK Ideograph>)
      6290   => 16#7B33#,  --  笳 (<CJK Ideograph>)
      6291   => 16#7B18#,  --  笘 (<CJK Ideograph>)
      6292   => 16#7B19#,  --  笙 (<CJK Ideograph>)
      6293   => 16#7B1E#,  --  笞 (<CJK Ideograph>)
      6294   => 16#7B35#,  --  笵 (<CJK Ideograph>)
      6295   => 16#7B28#,  --  笨 (<CJK Ideograph>)
      6296   => 16#7B36#,  --  笶 (<CJK Ideograph>)
      6297   => 16#7B50#,  --  筐 (<CJK Ideograph>)
      6298   => 16#7B7A#,  --  筺 (<CJK Ideograph>)
      6299   => 16#7B04#,  --  笄 (<CJK Ideograph>)
      6300   => 16#7B4D#,  --  筍 (<CJK Ideograph>)
      6301   => 16#7B0B#,  --  笋 (<CJK Ideograph>)
      6302   => 16#7B4C#,  --  筌 (<CJK Ideograph>)
      6303   => 16#7B45#,  --  筅 (<CJK Ideograph>)
      6304   => 16#7B75#,  --  筵 (<CJK Ideograph>)
      6305   => 16#7B65#,  --  筥 (<CJK Ideograph>)
      6306   => 16#7B74#,  --  筴 (<CJK Ideograph>)
      6307   => 16#7B67#,  --  筧 (<CJK Ideograph>)
      6308   => 16#7B70#,  --  筰 (<CJK Ideograph>)
      6309   => 16#7B71#,  --  筱 (<CJK Ideograph>)
      6310   => 16#7B6C#,  --  筬 (<CJK Ideograph>)
      6311   => 16#7B6E#,  --  筮 (<CJK Ideograph>)
      6312   => 16#7B9D#,  --  箝 (<CJK Ideograph>)
      6313   => 16#7B98#,  --  箘 (<CJK Ideograph>)
      6314   => 16#7B9F#,  --  箟 (<CJK Ideograph>)
      6315   => 16#7B8D#,  --  箍 (<CJK Ideograph>)
      6316   => 16#7B9C#,  --  箜 (<CJK Ideograph>)
      6317   => 16#7B9A#,  --  箚 (<CJK Ideograph>)
      6318   => 16#7B8B#,  --  箋 (<CJK Ideograph>)
      6319   => 16#7B92#,  --  箒 (<CJK Ideograph>)
      6320   => 16#7B8F#,  --  箏 (<CJK Ideograph>)
      6321   => 16#7B5D#,  --  筝 (<CJK Ideograph>)
      6322   => 16#7B99#,  --  箙 (<CJK Ideograph>)
      6323   => 16#7BCB#,  --  篋 (<CJK Ideograph>)
      6324   => 16#7BC1#,  --  篁 (<CJK Ideograph>)
      6325   => 16#7BCC#,  --  篌 (<CJK Ideograph>)
      6326   => 16#7BCF#,  --  篏 (<CJK Ideograph>)
      6327   => 16#7BB4#,  --  箴 (<CJK Ideograph>)
      6328   => 16#7BC6#,  --  篆 (<CJK Ideograph>)
      6329   => 16#7BDD#,  --  篝 (<CJK Ideograph>)
      6330   => 16#7BE9#,  --  篩 (<CJK Ideograph>)
      6331   => 16#7C11#,  --  簑 (<CJK Ideograph>)
      6332   => 16#7C14#,  --  簔 (<CJK Ideograph>)
      6333   => 16#7BE6#,  --  篦 (<CJK Ideograph>)
      6334   => 16#7BE5#,  --  篥 (<CJK Ideograph>)
      6335   => 16#7C60#,  --  籠 (<CJK Ideograph>)
      6336   => 16#7C00#,  --  簀 (<CJK Ideograph>)
      6337   => 16#7C07#,  --  簇 (<CJK Ideograph>)
      6338   => 16#7C13#,  --  簓 (<CJK Ideograph>)
      6339   => 16#7BF3#,  --  篳 (<CJK Ideograph>)
      6340   => 16#7BF7#,  --  篷 (<CJK Ideograph>)
      6341   => 16#7C17#,  --  簗 (<CJK Ideograph>)
      6342   => 16#7C0D#,  --  簍 (<CJK Ideograph>)
      6343   => 16#7BF6#,  --  篶 (<CJK Ideograph>)
      6344   => 16#7C23#,  --  簣 (<CJK Ideograph>)
      6345   => 16#7C27#,  --  簧 (<CJK Ideograph>)
      6346   => 16#7C2A#,  --  簪 (<CJK Ideograph>)
      6347   => 16#7C1F#,  --  簟 (<CJK Ideograph>)
      6348   => 16#7C37#,  --  簷 (<CJK Ideograph>)
      6349   => 16#7C2B#,  --  簫 (<CJK Ideograph>)
      6350   => 16#7C3D#,  --  簽 (<CJK Ideograph>)
      6351   => 16#7C4C#,  --  籌 (<CJK Ideograph>)
      6352   => 16#7C43#,  --  籃 (<CJK Ideograph>)
      6353   => 16#7C54#,  --  籔 (<CJK Ideograph>)
      6354   => 16#7C4F#,  --  籏 (<CJK Ideograph>)
      6355   => 16#7C40#,  --  籀 (<CJK Ideograph>)
      6356   => 16#7C50#,  --  籐 (<CJK Ideograph>)
      6357   => 16#7C58#,  --  籘 (<CJK Ideograph>)
      6358   => 16#7C5F#,  --  籟 (<CJK Ideograph>)
      6359   => 16#7C64#,  --  籤 (<CJK Ideograph>)
      6360   => 16#7C56#,  --  籖 (<CJK Ideograph>)
      6361   => 16#7C65#,  --  籥 (<CJK Ideograph>)
      6362   => 16#7C6C#,  --  籬 (<CJK Ideograph>)
      6363   => 16#7C75#,  --  籵 (<CJK Ideograph>)
      6364   => 16#7C83#,  --  粃 (<CJK Ideograph>)
      6365   => 16#7C90#,  --  粐 (<CJK Ideograph>)
      6366   => 16#7CA4#,  --  粤 (<CJK Ideograph>)
      6367   => 16#7CAD#,  --  粭 (<CJK Ideograph>)
      6368   => 16#7CA2#,  --  粢 (<CJK Ideograph>)
      6369   => 16#7CAB#,  --  粫 (<CJK Ideograph>)
      6370   => 16#7CA1#,  --  粡 (<CJK Ideograph>)
      6371   => 16#7CA8#,  --  粨 (<CJK Ideograph>)
      6372   => 16#7CB3#,  --  粳 (<CJK Ideograph>)
      6373   => 16#7CB2#,  --  粲 (<CJK Ideograph>)
      6374   => 16#7CB1#,  --  粱 (<CJK Ideograph>)
      6375   => 16#7CAE#,  --  粮 (<CJK Ideograph>)
      6376   => 16#7CB9#,  --  粹 (<CJK Ideograph>)
      6377   => 16#7CBD#,  --  粽 (<CJK Ideograph>)
      6378   => 16#7CC0#,  --  糀 (<CJK Ideograph>)
      6379   => 16#7CC5#,  --  糅 (<CJK Ideograph>)
      6380   => 16#7CC2#,  --  糂 (<CJK Ideograph>)
      6381   => 16#7CD8#,  --  糘 (<CJK Ideograph>)
      6382   => 16#7CD2#,  --  糒 (<CJK Ideograph>)
      6383   => 16#7CDC#,  --  糜 (<CJK Ideograph>)
      6384   => 16#7CE2#,  --  糢 (<CJK Ideograph>)
      6385   => 16#9B3B#,  --  鬻 (<CJK Ideograph>)
      6386   => 16#7CEF#,  --  糯 (<CJK Ideograph>)
      6387   => 16#7CF2#,  --  糲 (<CJK Ideograph>)
      6388   => 16#7CF4#,  --  糴 (<CJK Ideograph>)
      6389   => 16#7CF6#,  --  糶 (<CJK Ideograph>)
      6390   => 16#7CFA#,  --  糺 (<CJK Ideograph>)
      6391   => 16#7D06#,  --  紆 (<CJK Ideograph>)
      6392   => 16#7D02#,  --  紂 (<CJK Ideograph>)
      6393   => 16#7D1C#,  --  紜 (<CJK Ideograph>)
      6394   => 16#7D15#,  --  紕 (<CJK Ideograph>)
      6395   => 16#7D0A#,  --  紊 (<CJK Ideograph>)
      6396   => 16#7D45#,  --  絅 (<CJK Ideograph>)
      6397   => 16#7D4B#,  --  絋 (<CJK Ideograph>)
      6398   => 16#7D2E#,  --  紮 (<CJK Ideograph>)
      6399   => 16#7D32#,  --  紲 (<CJK Ideograph>)
      6400   => 16#7D3F#,  --  紿 (<CJK Ideograph>)
      6401   => 16#7D35#,  --  紵 (<CJK Ideograph>)
      6402   => 16#7D46#,  --  絆 (<CJK Ideograph>)
      6403   => 16#7D73#,  --  絳 (<CJK Ideograph>)
      6404   => 16#7D56#,  --  絖 (<CJK Ideograph>)
      6405   => 16#7D4E#,  --  絎 (<CJK Ideograph>)
      6406   => 16#7D72#,  --  絲 (<CJK Ideograph>)
      6407   => 16#7D68#,  --  絨 (<CJK Ideograph>)
      6408   => 16#7D6E#,  --  絮 (<CJK Ideograph>)
      6409   => 16#7D4F#,  --  絏 (<CJK Ideograph>)
      6410   => 16#7D63#,  --  絣 (<CJK Ideograph>)
      6411   => 16#7D93#,  --  經 (<CJK Ideograph>)
      6412   => 16#7D89#,  --  綉 (<CJK Ideograph>)
      6413   => 16#7D5B#,  --  絛 (<CJK Ideograph>)
      6414   => 16#7D8F#,  --  綏 (<CJK Ideograph>)
      6415   => 16#7D7D#,  --  絽 (<CJK Ideograph>)
      6416   => 16#7D9B#,  --  綛 (<CJK Ideograph>)
      6417   => 16#7DBA#,  --  綺 (<CJK Ideograph>)
      6418   => 16#7DAE#,  --  綮 (<CJK Ideograph>)
      6419   => 16#7DA3#,  --  綣 (<CJK Ideograph>)
      6420   => 16#7DB5#,  --  綵 (<CJK Ideograph>)
      6421   => 16#7DC7#,  --  緇 (<CJK Ideograph>)
      6422   => 16#7DBD#,  --  綽 (<CJK Ideograph>)
      6423   => 16#7DAB#,  --  綫 (<CJK Ideograph>)
      6424   => 16#7E3D#,  --  總 (<CJK Ideograph>)
      6425   => 16#7DA2#,  --  綢 (<CJK Ideograph>)
      6426   => 16#7DAF#,  --  綯 (<CJK Ideograph>)
      6427   => 16#7DDC#,  --  緜 (<CJK Ideograph>)
      6428   => 16#7DB8#,  --  綸 (<CJK Ideograph>)
      6429   => 16#7D9F#,  --  綟 (<CJK Ideograph>)
      6430   => 16#7DB0#,  --  綰 (<CJK Ideograph>)
      6431   => 16#7DD8#,  --  緘 (<CJK Ideograph>)
      6432   => 16#7DDD#,  --  緝 (<CJK Ideograph>)
      6433   => 16#7DE4#,  --  緤 (<CJK Ideograph>)
      6434   => 16#7DDE#,  --  緞 (<CJK Ideograph>)
      6435   => 16#7DFB#,  --  緻 (<CJK Ideograph>)
      6436   => 16#7DF2#,  --  緲 (<CJK Ideograph>)
      6437   => 16#7DE1#,  --  緡 (<CJK Ideograph>)
      6438   => 16#7E05#,  --  縅 (<CJK Ideograph>)
      6439   => 16#7E0A#,  --  縊 (<CJK Ideograph>)
      6440   => 16#7E23#,  --  縣 (<CJK Ideograph>)
      6441   => 16#7E21#,  --  縡 (<CJK Ideograph>)
      6442   => 16#7E12#,  --  縒 (<CJK Ideograph>)
      6443   => 16#7E31#,  --  縱 (<CJK Ideograph>)
      6444   => 16#7E1F#,  --  縟 (<CJK Ideograph>)
      6445   => 16#7E09#,  --  縉 (<CJK Ideograph>)
      6446   => 16#7E0B#,  --  縋 (<CJK Ideograph>)
      6447   => 16#7E22#,  --  縢 (<CJK Ideograph>)
      6448   => 16#7E46#,  --  繆 (<CJK Ideograph>)
      6449   => 16#7E66#,  --  繦 (<CJK Ideograph>)
      6450   => 16#7E3B#,  --  縻 (<CJK Ideograph>)
      6451   => 16#7E35#,  --  縵 (<CJK Ideograph>)
      6452   => 16#7E39#,  --  縹 (<CJK Ideograph>)
      6453   => 16#7E43#,  --  繃 (<CJK Ideograph>)
      6454   => 16#7E37#,  --  縷 (<CJK Ideograph>)
      6455   => 16#7E32#,  --  縲 (<CJK Ideograph>)
      6456   => 16#7E3A#,  --  縺 (<CJK Ideograph>)
      6457   => 16#7E67#,  --  繧 (<CJK Ideograph>)
      6458   => 16#7E5D#,  --  繝 (<CJK Ideograph>)
      6459   => 16#7E56#,  --  繖 (<CJK Ideograph>)
      6460   => 16#7E5E#,  --  繞 (<CJK Ideograph>)
      6461   => 16#7E59#,  --  繙 (<CJK Ideograph>)
      6462   => 16#7E5A#,  --  繚 (<CJK Ideograph>)
      6463   => 16#7E79#,  --  繹 (<CJK Ideograph>)
      6464   => 16#7E6A#,  --  繪 (<CJK Ideograph>)
      6465   => 16#7E69#,  --  繩 (<CJK Ideograph>)
      6466   => 16#7E7C#,  --  繼 (<CJK Ideograph>)
      6467   => 16#7E7B#,  --  繻 (<CJK Ideograph>)
      6468   => 16#7E83#,  --  纃 (<CJK Ideograph>)
      6469   => 16#7DD5#,  --  緕 (<CJK Ideograph>)
      6470   => 16#7E7D#,  --  繽 (<CJK Ideograph>)
      6471   => 16#8FAE#,  --  辮 (<CJK Ideograph>)
      6472   => 16#7E7F#,  --  繿 (<CJK Ideograph>)
      6473   => 16#7E88#,  --  纈 (<CJK Ideograph>)
      6474   => 16#7E89#,  --  纉 (<CJK Ideograph>)
      6475   => 16#7E8C#,  --  續 (<CJK Ideograph>)
      6476   => 16#7E92#,  --  纒 (<CJK Ideograph>)
      6477   => 16#7E90#,  --  纐 (<CJK Ideograph>)
      6478   => 16#7E93#,  --  纓 (<CJK Ideograph>)
      6479   => 16#7E94#,  --  纔 (<CJK Ideograph>)
      6480   => 16#7E96#,  --  纖 (<CJK Ideograph>)
      6481   => 16#7E8E#,  --  纎 (<CJK Ideograph>)
      6482   => 16#7E9B#,  --  纛 (<CJK Ideograph>)
      6483   => 16#7E9C#,  --  纜 (<CJK Ideograph>)
      6484   => 16#7F38#,  --  缸 (<CJK Ideograph>)
      6485   => 16#7F3A#,  --  缺 (<CJK Ideograph>)
      6486   => 16#7F45#,  --  罅 (<CJK Ideograph>)
      6487   => 16#7F4C#,  --  罌 (<CJK Ideograph>)
      6488   => 16#7F4D#,  --  罍 (<CJK Ideograph>)
      6489   => 16#7F4E#,  --  罎 (<CJK Ideograph>)
      6490   => 16#7F50#,  --  罐 (<CJK Ideograph>)
      6491   => 16#7F51#,  --  网 (<CJK Ideograph>)
      6492   => 16#7F55#,  --  罕 (<CJK Ideograph>)
      6493   => 16#7F54#,  --  罔 (<CJK Ideograph>)
      6494   => 16#7F58#,  --  罘 (<CJK Ideograph>)
      6495   => 16#7F5F#,  --  罟 (<CJK Ideograph>)
      6496   => 16#7F60#,  --  罠 (<CJK Ideograph>)
      6497   => 16#7F68#,  --  罨 (<CJK Ideograph>)
      6498   => 16#7F69#,  --  罩 (<CJK Ideograph>)
      6499   => 16#7F67#,  --  罧 (<CJK Ideograph>)
      6500   => 16#7F78#,  --  罸 (<CJK Ideograph>)
      6501   => 16#7F82#,  --  羂 (<CJK Ideograph>)
      6502   => 16#7F86#,  --  羆 (<CJK Ideograph>)
      6503   => 16#7F83#,  --  羃 (<CJK Ideograph>)
      6504   => 16#7F88#,  --  羈 (<CJK Ideograph>)
      6505   => 16#7F87#,  --  羇 (<CJK Ideograph>)
      6506   => 16#7F8C#,  --  羌 (<CJK Ideograph>)
      6507   => 16#7F94#,  --  羔 (<CJK Ideograph>)
      6508   => 16#7F9E#,  --  羞 (<CJK Ideograph>)
      6509   => 16#7F9D#,  --  羝 (<CJK Ideograph>)
      6510   => 16#7F9A#,  --  羚 (<CJK Ideograph>)
      6511   => 16#7FA3#,  --  羣 (<CJK Ideograph>)
      6512   => 16#7FAF#,  --  羯 (<CJK Ideograph>)
      6513   => 16#7FB2#,  --  羲 (<CJK Ideograph>)
      6514   => 16#7FB9#,  --  羹 (<CJK Ideograph>)
      6515   => 16#7FAE#,  --  羮 (<CJK Ideograph>)
      6516   => 16#7FB6#,  --  羶 (<CJK Ideograph>)
      6517   => 16#7FB8#,  --  羸 (<CJK Ideograph>)
      6518   => 16#8B71#,  --  譱 (<CJK Ideograph>)
      6519   => 16#7FC5#,  --  翅 (<CJK Ideograph>)
      6520   => 16#7FC6#,  --  翆 (<CJK Ideograph>)
      6521   => 16#7FCA#,  --  翊 (<CJK Ideograph>)
      6522   => 16#7FD5#,  --  翕 (<CJK Ideograph>)
      6523   => 16#7FD4#,  --  翔 (<CJK Ideograph>)
      6524   => 16#7FE1#,  --  翡 (<CJK Ideograph>)
      6525   => 16#7FE6#,  --  翦 (<CJK Ideograph>)
      6526   => 16#7FE9#,  --  翩 (<CJK Ideograph>)
      6527   => 16#7FF3#,  --  翳 (<CJK Ideograph>)
      6528   => 16#7FF9#,  --  翹 (<CJK Ideograph>)
      6529   => 16#98DC#,  --  飜 (<CJK Ideograph>)
      6530   => 16#8006#,  --  耆 (<CJK Ideograph>)
      6531   => 16#8004#,  --  耄 (<CJK Ideograph>)
      6532   => 16#800B#,  --  耋 (<CJK Ideograph>)
      6533   => 16#8012#,  --  耒 (<CJK Ideograph>)
      6534   => 16#8018#,  --  耘 (<CJK Ideograph>)
      6535   => 16#8019#,  --  耙 (<CJK Ideograph>)
      6536   => 16#801C#,  --  耜 (<CJK Ideograph>)
      6537   => 16#8021#,  --  耡 (<CJK Ideograph>)
      6538   => 16#8028#,  --  耨 (<CJK Ideograph>)
      6539   => 16#803F#,  --  耿 (<CJK Ideograph>)
      6540   => 16#803B#,  --  耻 (<CJK Ideograph>)
      6541   => 16#804A#,  --  聊 (<CJK Ideograph>)
      6542   => 16#8046#,  --  聆 (<CJK Ideograph>)
      6543   => 16#8052#,  --  聒 (<CJK Ideograph>)
      6544   => 16#8058#,  --  聘 (<CJK Ideograph>)
      6545   => 16#805A#,  --  聚 (<CJK Ideograph>)
      6546   => 16#805F#,  --  聟 (<CJK Ideograph>)
      6547   => 16#8062#,  --  聢 (<CJK Ideograph>)
      6548   => 16#8068#,  --  聨 (<CJK Ideograph>)
      6549   => 16#8073#,  --  聳 (<CJK Ideograph>)
      6550   => 16#8072#,  --  聲 (<CJK Ideograph>)
      6551   => 16#8070#,  --  聰 (<CJK Ideograph>)
      6552   => 16#8076#,  --  聶 (<CJK Ideograph>)
      6553   => 16#8079#,  --  聹 (<CJK Ideograph>)
      6554   => 16#807D#,  --  聽 (<CJK Ideograph>)
      6555   => 16#807F#,  --  聿 (<CJK Ideograph>)
      6556   => 16#8084#,  --  肄 (<CJK Ideograph>)
      6557   => 16#8086#,  --  肆 (<CJK Ideograph>)
      6558   => 16#8085#,  --  肅 (<CJK Ideograph>)
      6559   => 16#809B#,  --  肛 (<CJK Ideograph>)
      6560   => 16#8093#,  --  肓 (<CJK Ideograph>)
      6561   => 16#809A#,  --  肚 (<CJK Ideograph>)
      6562   => 16#80AD#,  --  肭 (<CJK Ideograph>)
      6563   => 16#5190#,  --  冐 (<CJK Ideograph>)
      6564   => 16#80AC#,  --  肬 (<CJK Ideograph>)
      6565   => 16#80DB#,  --  胛 (<CJK Ideograph>)
      6566   => 16#80E5#,  --  胥 (<CJK Ideograph>)
      6567   => 16#80D9#,  --  胙 (<CJK Ideograph>)
      6568   => 16#80DD#,  --  胝 (<CJK Ideograph>)
      6569   => 16#80C4#,  --  胄 (<CJK Ideograph>)
      6570   => 16#80DA#,  --  胚 (<CJK Ideograph>)
      6571   => 16#80D6#,  --  胖 (<CJK Ideograph>)
      6572   => 16#8109#,  --  脉 (<CJK Ideograph>)
      6573   => 16#80EF#,  --  胯 (<CJK Ideograph>)
      6574   => 16#80F1#,  --  胱 (<CJK Ideograph>)
      6575   => 16#811B#,  --  脛 (<CJK Ideograph>)
      6576   => 16#8129#,  --  脩 (<CJK Ideograph>)
      6577   => 16#8123#,  --  脣 (<CJK Ideograph>)
      6578   => 16#812F#,  --  脯 (<CJK Ideograph>)
      6579   => 16#814B#,  --  腋 (<CJK Ideograph>)
      6580   => 16#968B#,  --  隋 (<CJK Ideograph>)
      6581   => 16#8146#,  --  腆 (<CJK Ideograph>)
      6582   => 16#813E#,  --  脾 (<CJK Ideograph>)
      6583   => 16#8153#,  --  腓 (<CJK Ideograph>)
      6584   => 16#8151#,  --  腑 (<CJK Ideograph>)
      6585   => 16#80FC#,  --  胼 (<CJK Ideograph>)
      6586   => 16#8171#,  --  腱 (<CJK Ideograph>)
      6587   => 16#816E#,  --  腮 (<CJK Ideograph>)
      6588   => 16#8165#,  --  腥 (<CJK Ideograph>)
      6589   => 16#8166#,  --  腦 (<CJK Ideograph>)
      6590   => 16#8174#,  --  腴 (<CJK Ideograph>)
      6591   => 16#8183#,  --  膃 (<CJK Ideograph>)
      6592   => 16#8188#,  --  膈 (<CJK Ideograph>)
      6593   => 16#818A#,  --  膊 (<CJK Ideograph>)
      6594   => 16#8180#,  --  膀 (<CJK Ideograph>)
      6595   => 16#8182#,  --  膂 (<CJK Ideograph>)
      6596   => 16#81A0#,  --  膠 (<CJK Ideograph>)
      6597   => 16#8195#,  --  膕 (<CJK Ideograph>)
      6598   => 16#81A4#,  --  膤 (<CJK Ideograph>)
      6599   => 16#81A3#,  --  膣 (<CJK Ideograph>)
      6600   => 16#815F#,  --  腟 (<CJK Ideograph>)
      6601   => 16#8193#,  --  膓 (<CJK Ideograph>)
      6602   => 16#81A9#,  --  膩 (<CJK Ideograph>)
      6603   => 16#81B0#,  --  膰 (<CJK Ideograph>)
      6604   => 16#81B5#,  --  膵 (<CJK Ideograph>)
      6605   => 16#81BE#,  --  膾 (<CJK Ideograph>)
      6606   => 16#81B8#,  --  膸 (<CJK Ideograph>)
      6607   => 16#81BD#,  --  膽 (<CJK Ideograph>)
      6608   => 16#81C0#,  --  臀 (<CJK Ideograph>)
      6609   => 16#81C2#,  --  臂 (<CJK Ideograph>)
      6610   => 16#81BA#,  --  膺 (<CJK Ideograph>)
      6611   => 16#81C9#,  --  臉 (<CJK Ideograph>)
      6612   => 16#81CD#,  --  臍 (<CJK Ideograph>)
      6613   => 16#81D1#,  --  臑 (<CJK Ideograph>)
      6614   => 16#81D9#,  --  臙 (<CJK Ideograph>)
      6615   => 16#81D8#,  --  臘 (<CJK Ideograph>)
      6616   => 16#81C8#,  --  臈 (<CJK Ideograph>)
      6617   => 16#81DA#,  --  臚 (<CJK Ideograph>)
      6618   => 16#81DF#,  --  臟 (<CJK Ideograph>)
      6619   => 16#81E0#,  --  臠 (<CJK Ideograph>)
      6620   => 16#81E7#,  --  臧 (<CJK Ideograph>)
      6621   => 16#81FA#,  --  臺 (<CJK Ideograph>)
      6622   => 16#81FB#,  --  臻 (<CJK Ideograph>)
      6623   => 16#81FE#,  --  臾 (<CJK Ideograph>)
      6624   => 16#8201#,  --  舁 (<CJK Ideograph>)
      6625   => 16#8202#,  --  舂 (<CJK Ideograph>)
      6626   => 16#8205#,  --  舅 (<CJK Ideograph>)
      6627   => 16#8207#,  --  與 (<CJK Ideograph>)
      6628   => 16#820A#,  --  舊 (<CJK Ideograph>)
      6629   => 16#820D#,  --  舍 (<CJK Ideograph>)
      6630   => 16#8210#,  --  舐 (<CJK Ideograph>)
      6631   => 16#8216#,  --  舖 (<CJK Ideograph>)
      6632   => 16#8229#,  --  舩 (<CJK Ideograph>)
      6633   => 16#822B#,  --  舫 (<CJK Ideograph>)
      6634   => 16#8238#,  --  舸 (<CJK Ideograph>)
      6635   => 16#8233#,  --  舳 (<CJK Ideograph>)
      6636   => 16#8240#,  --  艀 (<CJK Ideograph>)
      6637   => 16#8259#,  --  艙 (<CJK Ideograph>)
      6638   => 16#8258#,  --  艘 (<CJK Ideograph>)
      6639   => 16#825D#,  --  艝 (<CJK Ideograph>)
      6640   => 16#825A#,  --  艚 (<CJK Ideograph>)
      6641   => 16#825F#,  --  艟 (<CJK Ideograph>)
      6642   => 16#8264#,  --  艤 (<CJK Ideograph>)
      6643   => 16#8262#,  --  艢 (<CJK Ideograph>)
      6644   => 16#8268#,  --  艨 (<CJK Ideograph>)
      6645   => 16#826A#,  --  艪 (<CJK Ideograph>)
      6646   => 16#826B#,  --  艫 (<CJK Ideograph>)
      6647   => 16#822E#,  --  舮 (<CJK Ideograph>)
      6648   => 16#8271#,  --  艱 (<CJK Ideograph>)
      6649   => 16#8277#,  --  艷 (<CJK Ideograph>)
      6650   => 16#8278#,  --  艸 (<CJK Ideograph>)
      6651   => 16#827E#,  --  艾 (<CJK Ideograph>)
      6652   => 16#828D#,  --  芍 (<CJK Ideograph>)
      6653   => 16#8292#,  --  芒 (<CJK Ideograph>)
      6654   => 16#82AB#,  --  芫 (<CJK Ideograph>)
      6655   => 16#829F#,  --  芟 (<CJK Ideograph>)
      6656   => 16#82BB#,  --  芻 (<CJK Ideograph>)
      6657   => 16#82AC#,  --  芬 (<CJK Ideograph>)
      6658   => 16#82E1#,  --  苡 (<CJK Ideograph>)
      6659   => 16#82E3#,  --  苣 (<CJK Ideograph>)
      6660   => 16#82DF#,  --  苟 (<CJK Ideograph>)
      6661   => 16#82D2#,  --  苒 (<CJK Ideograph>)
      6662   => 16#82F4#,  --  苴 (<CJK Ideograph>)
      6663   => 16#82F3#,  --  苳 (<CJK Ideograph>)
      6664   => 16#82FA#,  --  苺 (<CJK Ideograph>)
      6665   => 16#8393#,  --  莓 (<CJK Ideograph>)
      6666   => 16#8303#,  --  范 (<CJK Ideograph>)
      6667   => 16#82FB#,  --  苻 (<CJK Ideograph>)
      6668   => 16#82F9#,  --  苹 (<CJK Ideograph>)
      6669   => 16#82DE#,  --  苞 (<CJK Ideograph>)
      6670   => 16#8306#,  --  茆 (<CJK Ideograph>)
      6671   => 16#82DC#,  --  苜 (<CJK Ideograph>)
      6672   => 16#8309#,  --  茉 (<CJK Ideograph>)
      6673   => 16#82D9#,  --  苙 (<CJK Ideograph>)
      6674   => 16#8335#,  --  茵 (<CJK Ideograph>)
      6675   => 16#8334#,  --  茴 (<CJK Ideograph>)
      6676   => 16#8316#,  --  茖 (<CJK Ideograph>)
      6677   => 16#8332#,  --  茲 (<CJK Ideograph>)
      6678   => 16#8331#,  --  茱 (<CJK Ideograph>)
      6679   => 16#8340#,  --  荀 (<CJK Ideograph>)
      6680   => 16#8339#,  --  茹 (<CJK Ideograph>)
      6681   => 16#8350#,  --  荐 (<CJK Ideograph>)
      6682   => 16#8345#,  --  荅 (<CJK Ideograph>)
      6683   => 16#832F#,  --  茯 (<CJK Ideograph>)
      6684   => 16#832B#,  --  茫 (<CJK Ideograph>)
      6685   => 16#8317#,  --  茗 (<CJK Ideograph>)
      6686   => 16#8318#,  --  茘 (<CJK Ideograph>)
      6687   => 16#8385#,  --  莅 (<CJK Ideograph>)
      6688   => 16#839A#,  --  莚 (<CJK Ideograph>)
      6689   => 16#83AA#,  --  莪 (<CJK Ideograph>)
      6690   => 16#839F#,  --  莟 (<CJK Ideograph>)
      6691   => 16#83A2#,  --  莢 (<CJK Ideograph>)
      6692   => 16#8396#,  --  莖 (<CJK Ideograph>)
      6693   => 16#8323#,  --  茣 (<CJK Ideograph>)
      6694   => 16#838E#,  --  莎 (<CJK Ideograph>)
      6695   => 16#8387#,  --  莇 (<CJK Ideograph>)
      6696   => 16#838A#,  --  莊 (<CJK Ideograph>)
      6697   => 16#837C#,  --  荼 (<CJK Ideograph>)
      6698   => 16#83B5#,  --  莵 (<CJK Ideograph>)
      6699   => 16#8373#,  --  荳 (<CJK Ideograph>)
      6700   => 16#8375#,  --  荵 (<CJK Ideograph>)
      6701   => 16#83A0#,  --  莠 (<CJK Ideograph>)
      6702   => 16#8389#,  --  莉 (<CJK Ideograph>)
      6703   => 16#83A8#,  --  莨 (<CJK Ideograph>)
      6704   => 16#83F4#,  --  菴 (<CJK Ideograph>)
      6705   => 16#8413#,  --  萓 (<CJK Ideograph>)
      6706   => 16#83EB#,  --  菫 (<CJK Ideograph>)
      6707   => 16#83CE#,  --  菎 (<CJK Ideograph>)
      6708   => 16#83FD#,  --  菽 (<CJK Ideograph>)
      6709   => 16#8403#,  --  萃 (<CJK Ideograph>)
      6710   => 16#83D8#,  --  菘 (<CJK Ideograph>)
      6711   => 16#840B#,  --  萋 (<CJK Ideograph>)
      6712   => 16#83C1#,  --  菁 (<CJK Ideograph>)
      6713   => 16#83F7#,  --  菷 (<CJK Ideograph>)
      6714   => 16#8407#,  --  萇 (<CJK Ideograph>)
      6715   => 16#83E0#,  --  菠 (<CJK Ideograph>)
      6716   => 16#83F2#,  --  菲 (<CJK Ideograph>)
      6717   => 16#840D#,  --  萍 (<CJK Ideograph>)
      6718   => 16#8422#,  --  萢 (<CJK Ideograph>)
      6719   => 16#8420#,  --  萠 (<CJK Ideograph>)
      6720   => 16#83BD#,  --  莽 (<CJK Ideograph>)
      6721   => 16#8438#,  --  萸 (<CJK Ideograph>)
      6722   => 16#8506#,  --  蔆 (<CJK Ideograph>)
      6723   => 16#83FB#,  --  菻 (<CJK Ideograph>)
      6724   => 16#846D#,  --  葭 (<CJK Ideograph>)
      6725   => 16#842A#,  --  萪 (<CJK Ideograph>)
      6726   => 16#843C#,  --  萼 (<CJK Ideograph>)
      6727   => 16#855A#,  --  蕚 (<CJK Ideograph>)
      6728   => 16#8484#,  --  蒄 (<CJK Ideograph>)
      6729   => 16#8477#,  --  葷 (<CJK Ideograph>)
      6730   => 16#846B#,  --  葫 (<CJK Ideograph>)
      6731   => 16#84AD#,  --  蒭 (<CJK Ideograph>)
      6732   => 16#846E#,  --  葮 (<CJK Ideograph>)
      6733   => 16#8482#,  --  蒂 (<CJK Ideograph>)
      6734   => 16#8469#,  --  葩 (<CJK Ideograph>)
      6735   => 16#8446#,  --  葆 (<CJK Ideograph>)
      6736   => 16#842C#,  --  萬 (<CJK Ideograph>)
      6737   => 16#846F#,  --  葯 (<CJK Ideograph>)
      6738   => 16#8479#,  --  葹 (<CJK Ideograph>)
      6739   => 16#8435#,  --  萵 (<CJK Ideograph>)
      6740   => 16#84CA#,  --  蓊 (<CJK Ideograph>)
      6741   => 16#8462#,  --  葢 (<CJK Ideograph>)
      6742   => 16#84B9#,  --  蒹 (<CJK Ideograph>)
      6743   => 16#84BF#,  --  蒿 (<CJK Ideograph>)
      6744   => 16#849F#,  --  蒟 (<CJK Ideograph>)
      6745   => 16#84D9#,  --  蓙 (<CJK Ideograph>)
      6746   => 16#84CD#,  --  蓍 (<CJK Ideograph>)
      6747   => 16#84BB#,  --  蒻 (<CJK Ideograph>)
      6748   => 16#84DA#,  --  蓚 (<CJK Ideograph>)
      6749   => 16#84D0#,  --  蓐 (<CJK Ideograph>)
      6750   => 16#84C1#,  --  蓁 (<CJK Ideograph>)
      6751   => 16#84C6#,  --  蓆 (<CJK Ideograph>)
      6752   => 16#84D6#,  --  蓖 (<CJK Ideograph>)
      6753   => 16#84A1#,  --  蒡 (<CJK Ideograph>)
      6754   => 16#8521#,  --  蔡 (<CJK Ideograph>)
      6755   => 16#84FF#,  --  蓿 (<CJK Ideograph>)
      6756   => 16#84F4#,  --  蓴 (<CJK Ideograph>)
      6757   => 16#8517#,  --  蔗 (<CJK Ideograph>)
      6758   => 16#8518#,  --  蔘 (<CJK Ideograph>)
      6759   => 16#852C#,  --  蔬 (<CJK Ideograph>)
      6760   => 16#851F#,  --  蔟 (<CJK Ideograph>)
      6761   => 16#8515#,  --  蔕 (<CJK Ideograph>)
      6762   => 16#8514#,  --  蔔 (<CJK Ideograph>)
      6763   => 16#84FC#,  --  蓼 (<CJK Ideograph>)
      6764   => 16#8540#,  --  蕀 (<CJK Ideograph>)
      6765   => 16#8563#,  --  蕣 (<CJK Ideograph>)
      6766   => 16#8558#,  --  蕘 (<CJK Ideograph>)
      6767   => 16#8548#,  --  蕈 (<CJK Ideograph>)
      6768   => 16#8541#,  --  蕁 (<CJK Ideograph>)
      6769   => 16#8602#,  --  蘂 (<CJK Ideograph>)
      6770   => 16#854B#,  --  蕋 (<CJK Ideograph>)
      6771   => 16#8555#,  --  蕕 (<CJK Ideograph>)
      6772   => 16#8580#,  --  薀 (<CJK Ideograph>)
      6773   => 16#85A4#,  --  薤 (<CJK Ideograph>)
      6774   => 16#8588#,  --  薈 (<CJK Ideograph>)
      6775   => 16#8591#,  --  薑 (<CJK Ideograph>)
      6776   => 16#858A#,  --  薊 (<CJK Ideograph>)
      6777   => 16#85A8#,  --  薨 (<CJK Ideograph>)
      6778   => 16#856D#,  --  蕭 (<CJK Ideograph>)
      6779   => 16#8594#,  --  薔 (<CJK Ideograph>)
      6780   => 16#859B#,  --  薛 (<CJK Ideograph>)
      6781   => 16#85EA#,  --  藪 (<CJK Ideograph>)
      6782   => 16#8587#,  --  薇 (<CJK Ideograph>)
      6783   => 16#859C#,  --  薜 (<CJK Ideograph>)
      6784   => 16#8577#,  --  蕷 (<CJK Ideograph>)
      6785   => 16#857E#,  --  蕾 (<CJK Ideograph>)
      6786   => 16#8590#,  --  薐 (<CJK Ideograph>)
      6787   => 16#85C9#,  --  藉 (<CJK Ideograph>)
      6788   => 16#85BA#,  --  薺 (<CJK Ideograph>)
      6789   => 16#85CF#,  --  藏 (<CJK Ideograph>)
      6790   => 16#85B9#,  --  薹 (<CJK Ideograph>)
      6791   => 16#85D0#,  --  藐 (<CJK Ideograph>)
      6792   => 16#85D5#,  --  藕 (<CJK Ideograph>)
      6793   => 16#85DD#,  --  藝 (<CJK Ideograph>)
      6794   => 16#85E5#,  --  藥 (<CJK Ideograph>)
      6795   => 16#85DC#,  --  藜 (<CJK Ideograph>)
      6796   => 16#85F9#,  --  藹 (<CJK Ideograph>)
      6797   => 16#860A#,  --  蘊 (<CJK Ideograph>)
      6798   => 16#8613#,  --  蘓 (<CJK Ideograph>)
      6799   => 16#860B#,  --  蘋 (<CJK Ideograph>)
      6800   => 16#85FE#,  --  藾 (<CJK Ideograph>)
      6801   => 16#85FA#,  --  藺 (<CJK Ideograph>)
      6802   => 16#8606#,  --  蘆 (<CJK Ideograph>)
      6803   => 16#8622#,  --  蘢 (<CJK Ideograph>)
      6804   => 16#861A#,  --  蘚 (<CJK Ideograph>)
      6805   => 16#8630#,  --  蘰 (<CJK Ideograph>)
      6806   => 16#863F#,  --  蘿 (<CJK Ideograph>)
      6807   => 16#864D#,  --  虍 (<CJK Ideograph>)
      6808   => 16#4E55#,  --  乕 (<CJK Ideograph>)
      6809   => 16#8654#,  --  虔 (<CJK Ideograph>)
      6810   => 16#865F#,  --  號 (<CJK Ideograph>)
      6811   => 16#8667#,  --  虧 (<CJK Ideograph>)
      6812   => 16#8671#,  --  虱 (<CJK Ideograph>)
      6813   => 16#8693#,  --  蚓 (<CJK Ideograph>)
      6814   => 16#86A3#,  --  蚣 (<CJK Ideograph>)
      6815   => 16#86A9#,  --  蚩 (<CJK Ideograph>)
      6816   => 16#86AA#,  --  蚪 (<CJK Ideograph>)
      6817   => 16#868B#,  --  蚋 (<CJK Ideograph>)
      6818   => 16#868C#,  --  蚌 (<CJK Ideograph>)
      6819   => 16#86B6#,  --  蚶 (<CJK Ideograph>)
      6820   => 16#86AF#,  --  蚯 (<CJK Ideograph>)
      6821   => 16#86C4#,  --  蛄 (<CJK Ideograph>)
      6822   => 16#86C6#,  --  蛆 (<CJK Ideograph>)
      6823   => 16#86B0#,  --  蚰 (<CJK Ideograph>)
      6824   => 16#86C9#,  --  蛉 (<CJK Ideograph>)
      6825   => 16#8823#,  --  蠣 (<CJK Ideograph>)
      6826   => 16#86AB#,  --  蚫 (<CJK Ideograph>)
      6827   => 16#86D4#,  --  蛔 (<CJK Ideograph>)
      6828   => 16#86DE#,  --  蛞 (<CJK Ideograph>)
      6829   => 16#86E9#,  --  蛩 (<CJK Ideograph>)
      6830   => 16#86EC#,  --  蛬 (<CJK Ideograph>)
      6831   => 16#86DF#,  --  蛟 (<CJK Ideograph>)
      6832   => 16#86DB#,  --  蛛 (<CJK Ideograph>)
      6833   => 16#86EF#,  --  蛯 (<CJK Ideograph>)
      6834   => 16#8712#,  --  蜒 (<CJK Ideograph>)
      6835   => 16#8706#,  --  蜆 (<CJK Ideograph>)
      6836   => 16#8708#,  --  蜈 (<CJK Ideograph>)
      6837   => 16#8700#,  --  蜀 (<CJK Ideograph>)
      6838   => 16#8703#,  --  蜃 (<CJK Ideograph>)
      6839   => 16#86FB#,  --  蛻 (<CJK Ideograph>)
      6840   => 16#8711#,  --  蜑 (<CJK Ideograph>)
      6841   => 16#8709#,  --  蜉 (<CJK Ideograph>)
      6842   => 16#870D#,  --  蜍 (<CJK Ideograph>)
      6843   => 16#86F9#,  --  蛹 (<CJK Ideograph>)
      6844   => 16#870A#,  --  蜊 (<CJK Ideograph>)
      6845   => 16#8734#,  --  蜴 (<CJK Ideograph>)
      6846   => 16#873F#,  --  蜿 (<CJK Ideograph>)
      6847   => 16#8737#,  --  蜷 (<CJK Ideograph>)
      6848   => 16#873B#,  --  蜻 (<CJK Ideograph>)
      6849   => 16#8725#,  --  蜥 (<CJK Ideograph>)
      6850   => 16#8729#,  --  蜩 (<CJK Ideograph>)
      6851   => 16#871A#,  --  蜚 (<CJK Ideograph>)
      6852   => 16#8760#,  --  蝠 (<CJK Ideograph>)
      6853   => 16#875F#,  --  蝟 (<CJK Ideograph>)
      6854   => 16#8778#,  --  蝸 (<CJK Ideograph>)
      6855   => 16#874C#,  --  蝌 (<CJK Ideograph>)
      6856   => 16#874E#,  --  蝎 (<CJK Ideograph>)
      6857   => 16#8774#,  --  蝴 (<CJK Ideograph>)
      6858   => 16#8757#,  --  蝗 (<CJK Ideograph>)
      6859   => 16#8768#,  --  蝨 (<CJK Ideograph>)
      6860   => 16#876E#,  --  蝮 (<CJK Ideograph>)
      6861   => 16#8759#,  --  蝙 (<CJK Ideograph>)
      6862   => 16#8753#,  --  蝓 (<CJK Ideograph>)
      6863   => 16#8763#,  --  蝣 (<CJK Ideograph>)
      6864   => 16#876A#,  --  蝪 (<CJK Ideograph>)
      6865   => 16#8805#,  --  蠅 (<CJK Ideograph>)
      6866   => 16#87A2#,  --  螢 (<CJK Ideograph>)
      6867   => 16#879F#,  --  螟 (<CJK Ideograph>)
      6868   => 16#8782#,  --  螂 (<CJK Ideograph>)
      6869   => 16#87AF#,  --  螯 (<CJK Ideograph>)
      6870   => 16#87CB#,  --  蟋 (<CJK Ideograph>)
      6871   => 16#87BD#,  --  螽 (<CJK Ideograph>)
      6872   => 16#87C0#,  --  蟀 (<CJK Ideograph>)
      6873   => 16#87D0#,  --  蟐 (<CJK Ideograph>)
      6874   => 16#96D6#,  --  雖 (<CJK Ideograph>)
      6875   => 16#87AB#,  --  螫 (<CJK Ideograph>)
      6876   => 16#87C4#,  --  蟄 (<CJK Ideograph>)
      6877   => 16#87B3#,  --  螳 (<CJK Ideograph>)
      6878   => 16#87C7#,  --  蟇 (<CJK Ideograph>)
      6879   => 16#87C6#,  --  蟆 (<CJK Ideograph>)
      6880   => 16#87BB#,  --  螻 (<CJK Ideograph>)
      6881   => 16#87EF#,  --  蟯 (<CJK Ideograph>)
      6882   => 16#87F2#,  --  蟲 (<CJK Ideograph>)
      6883   => 16#87E0#,  --  蟠 (<CJK Ideograph>)
      6884   => 16#880F#,  --  蠏 (<CJK Ideograph>)
      6885   => 16#880D#,  --  蠍 (<CJK Ideograph>)
      6886   => 16#87FE#,  --  蟾 (<CJK Ideograph>)
      6887   => 16#87F6#,  --  蟶 (<CJK Ideograph>)
      6888   => 16#87F7#,  --  蟷 (<CJK Ideograph>)
      6889   => 16#880E#,  --  蠎 (<CJK Ideograph>)
      6890   => 16#87D2#,  --  蟒 (<CJK Ideograph>)
      6891   => 16#8811#,  --  蠑 (<CJK Ideograph>)
      6892   => 16#8816#,  --  蠖 (<CJK Ideograph>)
      6893   => 16#8815#,  --  蠕 (<CJK Ideograph>)
      6894   => 16#8822#,  --  蠢 (<CJK Ideograph>)
      6895   => 16#8821#,  --  蠡 (<CJK Ideograph>)
      6896   => 16#8831#,  --  蠱 (<CJK Ideograph>)
      6897   => 16#8836#,  --  蠶 (<CJK Ideograph>)
      6898   => 16#8839#,  --  蠹 (<CJK Ideograph>)
      6899   => 16#8827#,  --  蠧 (<CJK Ideograph>)
      6900   => 16#883B#,  --  蠻 (<CJK Ideograph>)
      6901   => 16#8844#,  --  衄 (<CJK Ideograph>)
      6902   => 16#8842#,  --  衂 (<CJK Ideograph>)
      6903   => 16#8852#,  --  衒 (<CJK Ideograph>)
      6904   => 16#8859#,  --  衙 (<CJK Ideograph>)
      6905   => 16#885E#,  --  衞 (<CJK Ideograph>)
      6906   => 16#8862#,  --  衢 (<CJK Ideograph>)
      6907   => 16#886B#,  --  衫 (<CJK Ideograph>)
      6908   => 16#8881#,  --  袁 (<CJK Ideograph>)
      6909   => 16#887E#,  --  衾 (<CJK Ideograph>)
      6910   => 16#889E#,  --  袞 (<CJK Ideograph>)
      6911   => 16#8875#,  --  衵 (<CJK Ideograph>)
      6912   => 16#887D#,  --  衽 (<CJK Ideograph>)
      6913   => 16#88B5#,  --  袵 (<CJK Ideograph>)
      6914   => 16#8872#,  --  衲 (<CJK Ideograph>)
      6915   => 16#8882#,  --  袂 (<CJK Ideograph>)
      6916   => 16#8897#,  --  袗 (<CJK Ideograph>)
      6917   => 16#8892#,  --  袒 (<CJK Ideograph>)
      6918   => 16#88AE#,  --  袮 (<CJK Ideograph>)
      6919   => 16#8899#,  --  袙 (<CJK Ideograph>)
      6920   => 16#88A2#,  --  袢 (<CJK Ideograph>)
      6921   => 16#888D#,  --  袍 (<CJK Ideograph>)
      6922   => 16#88A4#,  --  袤 (<CJK Ideograph>)
      6923   => 16#88B0#,  --  袰 (<CJK Ideograph>)
      6924   => 16#88BF#,  --  袿 (<CJK Ideograph>)
      6925   => 16#88B1#,  --  袱 (<CJK Ideograph>)
      6926   => 16#88C3#,  --  裃 (<CJK Ideograph>)
      6927   => 16#88C4#,  --  裄 (<CJK Ideograph>)
      6928   => 16#88D4#,  --  裔 (<CJK Ideograph>)
      6929   => 16#88D8#,  --  裘 (<CJK Ideograph>)
      6930   => 16#88D9#,  --  裙 (<CJK Ideograph>)
      6931   => 16#88DD#,  --  裝 (<CJK Ideograph>)
      6932   => 16#88F9#,  --  裹 (<CJK Ideograph>)
      6933   => 16#8902#,  --  褂 (<CJK Ideograph>)
      6934   => 16#88FC#,  --  裼 (<CJK Ideograph>)
      6935   => 16#88F4#,  --  裴 (<CJK Ideograph>)
      6936   => 16#88E8#,  --  裨 (<CJK Ideograph>)
      6937   => 16#88F2#,  --  裲 (<CJK Ideograph>)
      6938   => 16#8904#,  --  褄 (<CJK Ideograph>)
      6939   => 16#890C#,  --  褌 (<CJK Ideograph>)
      6940   => 16#890A#,  --  褊 (<CJK Ideograph>)
      6941   => 16#8913#,  --  褓 (<CJK Ideograph>)
      6942   => 16#8943#,  --  襃 (<CJK Ideograph>)
      6943   => 16#891E#,  --  褞 (<CJK Ideograph>)
      6944   => 16#8925#,  --  褥 (<CJK Ideograph>)
      6945   => 16#892A#,  --  褪 (<CJK Ideograph>)
      6946   => 16#892B#,  --  褫 (<CJK Ideograph>)
      6947   => 16#8941#,  --  襁 (<CJK Ideograph>)
      6948   => 16#8944#,  --  襄 (<CJK Ideograph>)
      6949   => 16#893B#,  --  褻 (<CJK Ideograph>)
      6950   => 16#8936#,  --  褶 (<CJK Ideograph>)
      6951   => 16#8938#,  --  褸 (<CJK Ideograph>)
      6952   => 16#894C#,  --  襌 (<CJK Ideograph>)
      6953   => 16#891D#,  --  褝 (<CJK Ideograph>)
      6954   => 16#8960#,  --  襠 (<CJK Ideograph>)
      6955   => 16#895E#,  --  襞 (<CJK Ideograph>)
      6956   => 16#8966#,  --  襦 (<CJK Ideograph>)
      6957   => 16#8964#,  --  襤 (<CJK Ideograph>)
      6958   => 16#896D#,  --  襭 (<CJK Ideograph>)
      6959   => 16#896A#,  --  襪 (<CJK Ideograph>)
      6960   => 16#896F#,  --  襯 (<CJK Ideograph>)
      6961   => 16#8974#,  --  襴 (<CJK Ideograph>)
      6962   => 16#8977#,  --  襷 (<CJK Ideograph>)
      6963   => 16#897E#,  --  襾 (<CJK Ideograph>)
      6964   => 16#8983#,  --  覃 (<CJK Ideograph>)
      6965   => 16#8988#,  --  覈 (<CJK Ideograph>)
      6966   => 16#898A#,  --  覊 (<CJK Ideograph>)
      6967   => 16#8993#,  --  覓 (<CJK Ideograph>)
      6968   => 16#8998#,  --  覘 (<CJK Ideograph>)
      6969   => 16#89A1#,  --  覡 (<CJK Ideograph>)
      6970   => 16#89A9#,  --  覩 (<CJK Ideograph>)
      6971   => 16#89A6#,  --  覦 (<CJK Ideograph>)
      6972   => 16#89AC#,  --  覬 (<CJK Ideograph>)
      6973   => 16#89AF#,  --  覯 (<CJK Ideograph>)
      6974   => 16#89B2#,  --  覲 (<CJK Ideograph>)
      6975   => 16#89BA#,  --  覺 (<CJK Ideograph>)
      6976   => 16#89BD#,  --  覽 (<CJK Ideograph>)
      6977   => 16#89BF#,  --  覿 (<CJK Ideograph>)
      6978   => 16#89C0#,  --  觀 (<CJK Ideograph>)
      6979   => 16#89DA#,  --  觚 (<CJK Ideograph>)
      6980   => 16#89DC#,  --  觜 (<CJK Ideograph>)
      6981   => 16#89DD#,  --  觝 (<CJK Ideograph>)
      6982   => 16#89E7#,  --  觧 (<CJK Ideograph>)
      6983   => 16#89F4#,  --  觴 (<CJK Ideograph>)
      6984   => 16#89F8#,  --  觸 (<CJK Ideograph>)
      6985   => 16#8A03#,  --  訃 (<CJK Ideograph>)
      6986   => 16#8A16#,  --  訖 (<CJK Ideograph>)
      6987   => 16#8A10#,  --  訐 (<CJK Ideograph>)
      6988   => 16#8A0C#,  --  訌 (<CJK Ideograph>)
      6989   => 16#8A1B#,  --  訛 (<CJK Ideograph>)
      6990   => 16#8A1D#,  --  訝 (<CJK Ideograph>)
      6991   => 16#8A25#,  --  訥 (<CJK Ideograph>)
      6992   => 16#8A36#,  --  訶 (<CJK Ideograph>)
      6993   => 16#8A41#,  --  詁 (<CJK Ideograph>)
      6994   => 16#8A5B#,  --  詛 (<CJK Ideograph>)
      6995   => 16#8A52#,  --  詒 (<CJK Ideograph>)
      6996   => 16#8A46#,  --  詆 (<CJK Ideograph>)
      6997   => 16#8A48#,  --  詈 (<CJK Ideograph>)
      6998   => 16#8A7C#,  --  詼 (<CJK Ideograph>)
      6999   => 16#8A6D#,  --  詭 (<CJK Ideograph>)
      7000   => 16#8A6C#,  --  詬 (<CJK Ideograph>)
      7001   => 16#8A62#,  --  詢 (<CJK Ideograph>)
      7002   => 16#8A85#,  --  誅 (<CJK Ideograph>)
      7003   => 16#8A82#,  --  誂 (<CJK Ideograph>)
      7004   => 16#8A84#,  --  誄 (<CJK Ideograph>)
      7005   => 16#8AA8#,  --  誨 (<CJK Ideograph>)
      7006   => 16#8AA1#,  --  誡 (<CJK Ideograph>)
      7007   => 16#8A91#,  --  誑 (<CJK Ideograph>)
      7008   => 16#8AA5#,  --  誥 (<CJK Ideograph>)
      7009   => 16#8AA6#,  --  誦 (<CJK Ideograph>)
      7010   => 16#8A9A#,  --  誚 (<CJK Ideograph>)
      7011   => 16#8AA3#,  --  誣 (<CJK Ideograph>)
      7012   => 16#8AC4#,  --  諄 (<CJK Ideograph>)
      7013   => 16#8ACD#,  --  諍 (<CJK Ideograph>)
      7014   => 16#8AC2#,  --  諂 (<CJK Ideograph>)
      7015   => 16#8ADA#,  --  諚 (<CJK Ideograph>)
      7016   => 16#8AEB#,  --  諫 (<CJK Ideograph>)
      7017   => 16#8AF3#,  --  諳 (<CJK Ideograph>)
      7018   => 16#8AE7#,  --  諧 (<CJK Ideograph>)
      7019   => 16#8AE4#,  --  諤 (<CJK Ideograph>)
      7020   => 16#8AF1#,  --  諱 (<CJK Ideograph>)
      7021   => 16#8B14#,  --  謔 (<CJK Ideograph>)
      7022   => 16#8AE0#,  --  諠 (<CJK Ideograph>)
      7023   => 16#8AE2#,  --  諢 (<CJK Ideograph>)
      7024   => 16#8AF7#,  --  諷 (<CJK Ideograph>)
      7025   => 16#8ADE#,  --  諞 (<CJK Ideograph>)
      7026   => 16#8ADB#,  --  諛 (<CJK Ideograph>)
      7027   => 16#8B0C#,  --  謌 (<CJK Ideograph>)
      7028   => 16#8B07#,  --  謇 (<CJK Ideograph>)
      7029   => 16#8B1A#,  --  謚 (<CJK Ideograph>)
      7030   => 16#8AE1#,  --  諡 (<CJK Ideograph>)
      7031   => 16#8B16#,  --  謖 (<CJK Ideograph>)
      7032   => 16#8B10#,  --  謐 (<CJK Ideograph>)
      7033   => 16#8B17#,  --  謗 (<CJK Ideograph>)
      7034   => 16#8B20#,  --  謠 (<CJK Ideograph>)
      7035   => 16#8B33#,  --  謳 (<CJK Ideograph>)
      7036   => 16#97AB#,  --  鞫 (<CJK Ideograph>)
      7037   => 16#8B26#,  --  謦 (<CJK Ideograph>)
      7038   => 16#8B2B#,  --  謫 (<CJK Ideograph>)
      7039   => 16#8B3E#,  --  謾 (<CJK Ideograph>)
      7040   => 16#8B28#,  --  謨 (<CJK Ideograph>)
      7041   => 16#8B41#,  --  譁 (<CJK Ideograph>)
      7042   => 16#8B4C#,  --  譌 (<CJK Ideograph>)
      7043   => 16#8B4F#,  --  譏 (<CJK Ideograph>)
      7044   => 16#8B4E#,  --  譎 (<CJK Ideograph>)
      7045   => 16#8B49#,  --  證 (<CJK Ideograph>)
      7046   => 16#8B56#,  --  譖 (<CJK Ideograph>)
      7047   => 16#8B5B#,  --  譛 (<CJK Ideograph>)
      7048   => 16#8B5A#,  --  譚 (<CJK Ideograph>)
      7049   => 16#8B6B#,  --  譫 (<CJK Ideograph>)
      7050   => 16#8B5F#,  --  譟 (<CJK Ideograph>)
      7051   => 16#8B6C#,  --  譬 (<CJK Ideograph>)
      7052   => 16#8B6F#,  --  譯 (<CJK Ideograph>)
      7053   => 16#8B74#,  --  譴 (<CJK Ideograph>)
      7054   => 16#8B7D#,  --  譽 (<CJK Ideograph>)
      7055   => 16#8B80#,  --  讀 (<CJK Ideograph>)
      7056   => 16#8B8C#,  --  讌 (<CJK Ideograph>)
      7057   => 16#8B8E#,  --  讎 (<CJK Ideograph>)
      7058   => 16#8B92#,  --  讒 (<CJK Ideograph>)
      7059   => 16#8B93#,  --  讓 (<CJK Ideograph>)
      7060   => 16#8B96#,  --  讖 (<CJK Ideograph>)
      7061   => 16#8B99#,  --  讙 (<CJK Ideograph>)
      7062   => 16#8B9A#,  --  讚 (<CJK Ideograph>)
      7063   => 16#8C3A#,  --  谺 (<CJK Ideograph>)
      7064   => 16#8C41#,  --  豁 (<CJK Ideograph>)
      7065   => 16#8C3F#,  --  谿 (<CJK Ideograph>)
      7066   => 16#8C48#,  --  豈 (<CJK Ideograph>)
      7067   => 16#8C4C#,  --  豌 (<CJK Ideograph>)
      7068   => 16#8C4E#,  --  豎 (<CJK Ideograph>)
      7069   => 16#8C50#,  --  豐 (<CJK Ideograph>)
      7070   => 16#8C55#,  --  豕 (<CJK Ideograph>)
      7071   => 16#8C62#,  --  豢 (<CJK Ideograph>)
      7072   => 16#8C6C#,  --  豬 (<CJK Ideograph>)
      7073   => 16#8C78#,  --  豸 (<CJK Ideograph>)
      7074   => 16#8C7A#,  --  豺 (<CJK Ideograph>)
      7075   => 16#8C82#,  --  貂 (<CJK Ideograph>)
      7076   => 16#8C89#,  --  貉 (<CJK Ideograph>)
      7077   => 16#8C85#,  --  貅 (<CJK Ideograph>)
      7078   => 16#8C8A#,  --  貊 (<CJK Ideograph>)
      7079   => 16#8C8D#,  --  貍 (<CJK Ideograph>)
      7080   => 16#8C8E#,  --  貎 (<CJK Ideograph>)
      7081   => 16#8C94#,  --  貔 (<CJK Ideograph>)
      7082   => 16#8C7C#,  --  豼 (<CJK Ideograph>)
      7083   => 16#8C98#,  --  貘 (<CJK Ideograph>)
      7084   => 16#621D#,  --  戝 (<CJK Ideograph>)
      7085   => 16#8CAD#,  --  貭 (<CJK Ideograph>)
      7086   => 16#8CAA#,  --  貪 (<CJK Ideograph>)
      7087   => 16#8CBD#,  --  貽 (<CJK Ideograph>)
      7088   => 16#8CB2#,  --  貲 (<CJK Ideograph>)
      7089   => 16#8CB3#,  --  貳 (<CJK Ideograph>)
      7090   => 16#8CAE#,  --  貮 (<CJK Ideograph>)
      7091   => 16#8CB6#,  --  貶 (<CJK Ideograph>)
      7092   => 16#8CC8#,  --  賈 (<CJK Ideograph>)
      7093   => 16#8CC1#,  --  賁 (<CJK Ideograph>)
      7094   => 16#8CE4#,  --  賤 (<CJK Ideograph>)
      7095   => 16#8CE3#,  --  賣 (<CJK Ideograph>)
      7096   => 16#8CDA#,  --  賚 (<CJK Ideograph>)
      7097   => 16#8CFD#,  --  賽 (<CJK Ideograph>)
      7098   => 16#8CFA#,  --  賺 (<CJK Ideograph>)
      7099   => 16#8CFB#,  --  賻 (<CJK Ideograph>)
      7100   => 16#8D04#,  --  贄 (<CJK Ideograph>)
      7101   => 16#8D05#,  --  贅 (<CJK Ideograph>)
      7102   => 16#8D0A#,  --  贊 (<CJK Ideograph>)
      7103   => 16#8D07#,  --  贇 (<CJK Ideograph>)
      7104   => 16#8D0F#,  --  贏 (<CJK Ideograph>)
      7105   => 16#8D0D#,  --  贍 (<CJK Ideograph>)
      7106   => 16#8D10#,  --  贐 (<CJK Ideograph>)
      7107   => 16#9F4E#,  --  齎 (<CJK Ideograph>)
      7108   => 16#8D13#,  --  贓 (<CJK Ideograph>)
      7109   => 16#8CCD#,  --  賍 (<CJK Ideograph>)
      7110   => 16#8D14#,  --  贔 (<CJK Ideograph>)
      7111   => 16#8D16#,  --  贖 (<CJK Ideograph>)
      7112   => 16#8D67#,  --  赧 (<CJK Ideograph>)
      7113   => 16#8D6D#,  --  赭 (<CJK Ideograph>)
      7114   => 16#8D71#,  --  赱 (<CJK Ideograph>)
      7115   => 16#8D73#,  --  赳 (<CJK Ideograph>)
      7116   => 16#8D81#,  --  趁 (<CJK Ideograph>)
      7117   => 16#8D99#,  --  趙 (<CJK Ideograph>)
      7118   => 16#8DC2#,  --  跂 (<CJK Ideograph>)
      7119   => 16#8DBE#,  --  趾 (<CJK Ideograph>)
      7120   => 16#8DBA#,  --  趺 (<CJK Ideograph>)
      7121   => 16#8DCF#,  --  跏 (<CJK Ideograph>)
      7122   => 16#8DDA#,  --  跚 (<CJK Ideograph>)
      7123   => 16#8DD6#,  --  跖 (<CJK Ideograph>)
      7124   => 16#8DCC#,  --  跌 (<CJK Ideograph>)
      7125   => 16#8DDB#,  --  跛 (<CJK Ideograph>)
      7126   => 16#8DCB#,  --  跋 (<CJK Ideograph>)
      7127   => 16#8DEA#,  --  跪 (<CJK Ideograph>)
      7128   => 16#8DEB#,  --  跫 (<CJK Ideograph>)
      7129   => 16#8DDF#,  --  跟 (<CJK Ideograph>)
      7130   => 16#8DE3#,  --  跣 (<CJK Ideograph>)
      7131   => 16#8DFC#,  --  跼 (<CJK Ideograph>)
      7132   => 16#8E08#,  --  踈 (<CJK Ideograph>)
      7133   => 16#8E09#,  --  踉 (<CJK Ideograph>)
      7134   => 16#8DFF#,  --  跿 (<CJK Ideograph>)
      7135   => 16#8E1D#,  --  踝 (<CJK Ideograph>)
      7136   => 16#8E1E#,  --  踞 (<CJK Ideograph>)
      7137   => 16#8E10#,  --  踐 (<CJK Ideograph>)
      7138   => 16#8E1F#,  --  踟 (<CJK Ideograph>)
      7139   => 16#8E42#,  --  蹂 (<CJK Ideograph>)
      7140   => 16#8E35#,  --  踵 (<CJK Ideograph>)
      7141   => 16#8E30#,  --  踰 (<CJK Ideograph>)
      7142   => 16#8E34#,  --  踴 (<CJK Ideograph>)
      7143   => 16#8E4A#,  --  蹊 (<CJK Ideograph>)
      7144   => 16#8E47#,  --  蹇 (<CJK Ideograph>)
      7145   => 16#8E49#,  --  蹉 (<CJK Ideograph>)
      7146   => 16#8E4C#,  --  蹌 (<CJK Ideograph>)
      7147   => 16#8E50#,  --  蹐 (<CJK Ideograph>)
      7148   => 16#8E48#,  --  蹈 (<CJK Ideograph>)
      7149   => 16#8E59#,  --  蹙 (<CJK Ideograph>)
      7150   => 16#8E64#,  --  蹤 (<CJK Ideograph>)
      7151   => 16#8E60#,  --  蹠 (<CJK Ideograph>)
      7152   => 16#8E2A#,  --  踪 (<CJK Ideograph>)
      7153   => 16#8E63#,  --  蹣 (<CJK Ideograph>)
      7154   => 16#8E55#,  --  蹕 (<CJK Ideograph>)
      7155   => 16#8E76#,  --  蹶 (<CJK Ideograph>)
      7156   => 16#8E72#,  --  蹲 (<CJK Ideograph>)
      7157   => 16#8E7C#,  --  蹼 (<CJK Ideograph>)
      7158   => 16#8E81#,  --  躁 (<CJK Ideograph>)
      7159   => 16#8E87#,  --  躇 (<CJK Ideograph>)
      7160   => 16#8E85#,  --  躅 (<CJK Ideograph>)
      7161   => 16#8E84#,  --  躄 (<CJK Ideograph>)
      7162   => 16#8E8B#,  --  躋 (<CJK Ideograph>)
      7163   => 16#8E8A#,  --  躊 (<CJK Ideograph>)
      7164   => 16#8E93#,  --  躓 (<CJK Ideograph>)
      7165   => 16#8E91#,  --  躑 (<CJK Ideograph>)
      7166   => 16#8E94#,  --  躔 (<CJK Ideograph>)
      7167   => 16#8E99#,  --  躙 (<CJK Ideograph>)
      7168   => 16#8EAA#,  --  躪 (<CJK Ideograph>)
      7169   => 16#8EA1#,  --  躡 (<CJK Ideograph>)
      7170   => 16#8EAC#,  --  躬 (<CJK Ideograph>)
      7171   => 16#8EB0#,  --  躰 (<CJK Ideograph>)
      7172   => 16#8EC6#,  --  軆 (<CJK Ideograph>)
      7173   => 16#8EB1#,  --  躱 (<CJK Ideograph>)
      7174   => 16#8EBE#,  --  躾 (<CJK Ideograph>)
      7175   => 16#8EC5#,  --  軅 (<CJK Ideograph>)
      7176   => 16#8EC8#,  --  軈 (<CJK Ideograph>)
      7177   => 16#8ECB#,  --  軋 (<CJK Ideograph>)
      7178   => 16#8EDB#,  --  軛 (<CJK Ideograph>)
      7179   => 16#8EE3#,  --  軣 (<CJK Ideograph>)
      7180   => 16#8EFC#,  --  軼 (<CJK Ideograph>)
      7181   => 16#8EFB#,  --  軻 (<CJK Ideograph>)
      7182   => 16#8EEB#,  --  軫 (<CJK Ideograph>)
      7183   => 16#8EFE#,  --  軾 (<CJK Ideograph>)
      7184   => 16#8F0A#,  --  輊 (<CJK Ideograph>)
      7185   => 16#8F05#,  --  輅 (<CJK Ideograph>)
      7186   => 16#8F15#,  --  輕 (<CJK Ideograph>)
      7187   => 16#8F12#,  --  輒 (<CJK Ideograph>)
      7188   => 16#8F19#,  --  輙 (<CJK Ideograph>)
      7189   => 16#8F13#,  --  輓 (<CJK Ideograph>)
      7190   => 16#8F1C#,  --  輜 (<CJK Ideograph>)
      7191   => 16#8F1F#,  --  輟 (<CJK Ideograph>)
      7192   => 16#8F1B#,  --  輛 (<CJK Ideograph>)
      7193   => 16#8F0C#,  --  輌 (<CJK Ideograph>)
      7194   => 16#8F26#,  --  輦 (<CJK Ideograph>)
      7195   => 16#8F33#,  --  輳 (<CJK Ideograph>)
      7196   => 16#8F3B#,  --  輻 (<CJK Ideograph>)
      7197   => 16#8F39#,  --  輹 (<CJK Ideograph>)
      7198   => 16#8F45#,  --  轅 (<CJK Ideograph>)
      7199   => 16#8F42#,  --  轂 (<CJK Ideograph>)
      7200   => 16#8F3E#,  --  輾 (<CJK Ideograph>)
      7201   => 16#8F4C#,  --  轌 (<CJK Ideograph>)
      7202   => 16#8F49#,  --  轉 (<CJK Ideograph>)
      7203   => 16#8F46#,  --  轆 (<CJK Ideograph>)
      7204   => 16#8F4E#,  --  轎 (<CJK Ideograph>)
      7205   => 16#8F57#,  --  轗 (<CJK Ideograph>)
      7206   => 16#8F5C#,  --  轜 (<CJK Ideograph>)
      7207   => 16#8F62#,  --  轢 (<CJK Ideograph>)
      7208   => 16#8F63#,  --  轣 (<CJK Ideograph>)
      7209   => 16#8F64#,  --  轤 (<CJK Ideograph>)
      7210   => 16#8F9C#,  --  辜 (<CJK Ideograph>)
      7211   => 16#8F9F#,  --  辟 (<CJK Ideograph>)
      7212   => 16#8FA3#,  --  辣 (<CJK Ideograph>)
      7213   => 16#8FAD#,  --  辭 (<CJK Ideograph>)
      7214   => 16#8FAF#,  --  辯 (<CJK Ideograph>)
      7215   => 16#8FB7#,  --  辷 (<CJK Ideograph>)
      7216   => 16#8FDA#,  --  迚 (<CJK Ideograph>)
      7217   => 16#8FE5#,  --  迥 (<CJK Ideograph>)
      7218   => 16#8FE2#,  --  迢 (<CJK Ideograph>)
      7219   => 16#8FEA#,  --  迪 (<CJK Ideograph>)
      7220   => 16#8FEF#,  --  迯 (<CJK Ideograph>)
      7221   => 16#9087#,  --  邇 (<CJK Ideograph>)
      7222   => 16#8FF4#,  --  迴 (<CJK Ideograph>)
      7223   => 16#9005#,  --  逅 (<CJK Ideograph>)
      7224   => 16#8FF9#,  --  迹 (<CJK Ideograph>)
      7225   => 16#8FFA#,  --  迺 (<CJK Ideograph>)
      7226   => 16#9011#,  --  逑 (<CJK Ideograph>)
      7227   => 16#9015#,  --  逕 (<CJK Ideograph>)
      7228   => 16#9021#,  --  逡 (<CJK Ideograph>)
      7229   => 16#900D#,  --  逍 (<CJK Ideograph>)
      7230   => 16#901E#,  --  逞 (<CJK Ideograph>)
      7231   => 16#9016#,  --  逖 (<CJK Ideograph>)
      7232   => 16#900B#,  --  逋 (<CJK Ideograph>)
      7233   => 16#9027#,  --  逧 (<CJK Ideograph>)
      7234   => 16#9036#,  --  逶 (<CJK Ideograph>)
      7235   => 16#9035#,  --  逵 (<CJK Ideograph>)
      7236   => 16#9039#,  --  逹 (<CJK Ideograph>)
      7237   => 16#8FF8#,  --  迸 (<CJK Ideograph>)
      7238   => 16#904F#,  --  遏 (<CJK Ideograph>)
      7239   => 16#9050#,  --  遐 (<CJK Ideograph>)
      7240   => 16#9051#,  --  遑 (<CJK Ideograph>)
      7241   => 16#9052#,  --  遒 (<CJK Ideograph>)
      7242   => 16#900E#,  --  逎 (<CJK Ideograph>)
      7243   => 16#9049#,  --  遉 (<CJK Ideograph>)
      7244   => 16#903E#,  --  逾 (<CJK Ideograph>)
      7245   => 16#9056#,  --  遖 (<CJK Ideograph>)
      7246   => 16#9058#,  --  遘 (<CJK Ideograph>)
      7247   => 16#905E#,  --  遞 (<CJK Ideograph>)
      7248   => 16#9068#,  --  遨 (<CJK Ideograph>)
      7249   => 16#906F#,  --  遯 (<CJK Ideograph>)
      7250   => 16#9076#,  --  遶 (<CJK Ideograph>)
      7251   => 16#96A8#,  --  隨 (<CJK Ideograph>)
      7252   => 16#9072#,  --  遲 (<CJK Ideograph>)
      7253   => 16#9082#,  --  邂 (<CJK Ideograph>)
      7254   => 16#907D#,  --  遽 (<CJK Ideograph>)
      7255   => 16#9081#,  --  邁 (<CJK Ideograph>)
      7256   => 16#9080#,  --  邀 (<CJK Ideograph>)
      7257   => 16#908A#,  --  邊 (<CJK Ideograph>)
      7258   => 16#9089#,  --  邉 (<CJK Ideograph>)
      7259   => 16#908F#,  --  邏 (<CJK Ideograph>)
      7260   => 16#90A8#,  --  邨 (<CJK Ideograph>)
      7261   => 16#90AF#,  --  邯 (<CJK Ideograph>)
      7262   => 16#90B1#,  --  邱 (<CJK Ideograph>)
      7263   => 16#90B5#,  --  邵 (<CJK Ideograph>)
      7264   => 16#90E2#,  --  郢 (<CJK Ideograph>)
      7265   => 16#90E4#,  --  郤 (<CJK Ideograph>)
      7266   => 16#6248#,  --  扈 (<CJK Ideograph>)
      7267   => 16#90DB#,  --  郛 (<CJK Ideograph>)
      7268   => 16#9102#,  --  鄂 (<CJK Ideograph>)
      7269   => 16#9112#,  --  鄒 (<CJK Ideograph>)
      7270   => 16#9119#,  --  鄙 (<CJK Ideograph>)
      7271   => 16#9132#,  --  鄲 (<CJK Ideograph>)
      7272   => 16#9130#,  --  鄰 (<CJK Ideograph>)
      7273   => 16#914A#,  --  酊 (<CJK Ideograph>)
      7274   => 16#9156#,  --  酖 (<CJK Ideograph>)
      7275   => 16#9158#,  --  酘 (<CJK Ideograph>)
      7276   => 16#9163#,  --  酣 (<CJK Ideograph>)
      7277   => 16#9165#,  --  酥 (<CJK Ideograph>)
      7278   => 16#9169#,  --  酩 (<CJK Ideograph>)
      7279   => 16#9173#,  --  酳 (<CJK Ideograph>)
      7280   => 16#9172#,  --  酲 (<CJK Ideograph>)
      7281   => 16#918B#,  --  醋 (<CJK Ideograph>)
      7282   => 16#9189#,  --  醉 (<CJK Ideograph>)
      7283   => 16#9182#,  --  醂 (<CJK Ideograph>)
      7284   => 16#91A2#,  --  醢 (<CJK Ideograph>)
      7285   => 16#91AB#,  --  醫 (<CJK Ideograph>)
      7286   => 16#91AF#,  --  醯 (<CJK Ideograph>)
      7287   => 16#91AA#,  --  醪 (<CJK Ideograph>)
      7288   => 16#91B5#,  --  醵 (<CJK Ideograph>)
      7289   => 16#91B4#,  --  醴 (<CJK Ideograph>)
      7290   => 16#91BA#,  --  醺 (<CJK Ideograph>)
      7291   => 16#91C0#,  --  釀 (<CJK Ideograph>)
      7292   => 16#91C1#,  --  釁 (<CJK Ideograph>)
      7293   => 16#91C9#,  --  釉 (<CJK Ideograph>)
      7294   => 16#91CB#,  --  釋 (<CJK Ideograph>)
      7295   => 16#91D0#,  --  釐 (<CJK Ideograph>)
      7296   => 16#91D6#,  --  釖 (<CJK Ideograph>)
      7297   => 16#91DF#,  --  釟 (<CJK Ideograph>)
      7298   => 16#91E1#,  --  釡 (<CJK Ideograph>)
      7299   => 16#91DB#,  --  釛 (<CJK Ideograph>)
      7300   => 16#91FC#,  --  釼 (<CJK Ideograph>)
      7301   => 16#91F5#,  --  釵 (<CJK Ideograph>)
      7302   => 16#91F6#,  --  釶 (<CJK Ideograph>)
      7303   => 16#921E#,  --  鈞 (<CJK Ideograph>)
      7304   => 16#91FF#,  --  釿 (<CJK Ideograph>)
      7305   => 16#9214#,  --  鈔 (<CJK Ideograph>)
      7306   => 16#922C#,  --  鈬 (<CJK Ideograph>)
      7307   => 16#9215#,  --  鈕 (<CJK Ideograph>)
      7308   => 16#9211#,  --  鈑 (<CJK Ideograph>)
      7309   => 16#925E#,  --  鉞 (<CJK Ideograph>)
      7310   => 16#9257#,  --  鉗 (<CJK Ideograph>)
      7311   => 16#9245#,  --  鉅 (<CJK Ideograph>)
      7312   => 16#9249#,  --  鉉 (<CJK Ideograph>)
      7313   => 16#9264#,  --  鉤 (<CJK Ideograph>)
      7314   => 16#9248#,  --  鉈 (<CJK Ideograph>)
      7315   => 16#9295#,  --  銕 (<CJK Ideograph>)
      7316   => 16#923F#,  --  鈿 (<CJK Ideograph>)
      7317   => 16#924B#,  --  鉋 (<CJK Ideograph>)
      7318   => 16#9250#,  --  鉐 (<CJK Ideograph>)
      7319   => 16#929C#,  --  銜 (<CJK Ideograph>)
      7320   => 16#9296#,  --  銖 (<CJK Ideograph>)
      7321   => 16#9293#,  --  銓 (<CJK Ideograph>)
      7322   => 16#929B#,  --  銛 (<CJK Ideograph>)
      7323   => 16#925A#,  --  鉚 (<CJK Ideograph>)
      7324   => 16#92CF#,  --  鋏 (<CJK Ideograph>)
      7325   => 16#92B9#,  --  銹 (<CJK Ideograph>)
      7326   => 16#92B7#,  --  銷 (<CJK Ideograph>)
      7327   => 16#92E9#,  --  鋩 (<CJK Ideograph>)
      7328   => 16#930F#,  --  錏 (<CJK Ideograph>)
      7329   => 16#92FA#,  --  鋺 (<CJK Ideograph>)
      7330   => 16#9344#,  --  鍄 (<CJK Ideograph>)
      7331   => 16#932E#,  --  錮 (<CJK Ideograph>)
      7332   => 16#9319#,  --  錙 (<CJK Ideograph>)
      7333   => 16#9322#,  --  錢 (<CJK Ideograph>)
      7334   => 16#931A#,  --  錚 (<CJK Ideograph>)
      7335   => 16#9323#,  --  錣 (<CJK Ideograph>)
      7336   => 16#933A#,  --  錺 (<CJK Ideograph>)
      7337   => 16#9335#,  --  錵 (<CJK Ideograph>)
      7338   => 16#933B#,  --  錻 (<CJK Ideograph>)
      7339   => 16#935C#,  --  鍜 (<CJK Ideograph>)
      7340   => 16#9360#,  --  鍠 (<CJK Ideograph>)
      7341   => 16#937C#,  --  鍼 (<CJK Ideograph>)
      7342   => 16#936E#,  --  鍮 (<CJK Ideograph>)
      7343   => 16#9356#,  --  鍖 (<CJK Ideograph>)
      7344   => 16#93B0#,  --  鎰 (<CJK Ideograph>)
      7345   => 16#93AC#,  --  鎬 (<CJK Ideograph>)
      7346   => 16#93AD#,  --  鎭 (<CJK Ideograph>)
      7347   => 16#9394#,  --  鎔 (<CJK Ideograph>)
      7348   => 16#93B9#,  --  鎹 (<CJK Ideograph>)
      7349   => 16#93D6#,  --  鏖 (<CJK Ideograph>)
      7350   => 16#93D7#,  --  鏗 (<CJK Ideograph>)
      7351   => 16#93E8#,  --  鏨 (<CJK Ideograph>)
      7352   => 16#93E5#,  --  鏥 (<CJK Ideograph>)
      7353   => 16#93D8#,  --  鏘 (<CJK Ideograph>)
      7354   => 16#93C3#,  --  鏃 (<CJK Ideograph>)
      7355   => 16#93DD#,  --  鏝 (<CJK Ideograph>)
      7356   => 16#93D0#,  --  鏐 (<CJK Ideograph>)
      7357   => 16#93C8#,  --  鏈 (<CJK Ideograph>)
      7358   => 16#93E4#,  --  鏤 (<CJK Ideograph>)
      7359   => 16#941A#,  --  鐚 (<CJK Ideograph>)
      7360   => 16#9414#,  --  鐔 (<CJK Ideograph>)
      7361   => 16#9413#,  --  鐓 (<CJK Ideograph>)
      7362   => 16#9403#,  --  鐃 (<CJK Ideograph>)
      7363   => 16#9407#,  --  鐇 (<CJK Ideograph>)
      7364   => 16#9410#,  --  鐐 (<CJK Ideograph>)
      7365   => 16#9436#,  --  鐶 (<CJK Ideograph>)
      7366   => 16#942B#,  --  鐫 (<CJK Ideograph>)
      7367   => 16#9435#,  --  鐵 (<CJK Ideograph>)
      7368   => 16#9421#,  --  鐡 (<CJK Ideograph>)
      7369   => 16#943A#,  --  鐺 (<CJK Ideograph>)
      7370   => 16#9441#,  --  鑁 (<CJK Ideograph>)
      7371   => 16#9452#,  --  鑒 (<CJK Ideograph>)
      7372   => 16#9444#,  --  鑄 (<CJK Ideograph>)
      7373   => 16#945B#,  --  鑛 (<CJK Ideograph>)
      7374   => 16#9460#,  --  鑠 (<CJK Ideograph>)
      7375   => 16#9462#,  --  鑢 (<CJK Ideograph>)
      7376   => 16#945E#,  --  鑞 (<CJK Ideograph>)
      7377   => 16#946A#,  --  鑪 (<CJK Ideograph>)
      7378   => 16#9229#,  --  鈩 (<CJK Ideograph>)
      7379   => 16#9470#,  --  鑰 (<CJK Ideograph>)
      7380   => 16#9475#,  --  鑵 (<CJK Ideograph>)
      7381   => 16#9477#,  --  鑷 (<CJK Ideograph>)
      7382   => 16#947D#,  --  鑽 (<CJK Ideograph>)
      7383   => 16#945A#,  --  鑚 (<CJK Ideograph>)
      7384   => 16#947C#,  --  鑼 (<CJK Ideograph>)
      7385   => 16#947E#,  --  鑾 (<CJK Ideograph>)
      7386   => 16#9481#,  --  钁 (<CJK Ideograph>)
      7387   => 16#947F#,  --  鑿 (<CJK Ideograph>)
      7388   => 16#9582#,  --  閂 (<CJK Ideograph>)
      7389   => 16#9587#,  --  閇 (<CJK Ideograph>)
      7390   => 16#958A#,  --  閊 (<CJK Ideograph>)
      7391   => 16#9594#,  --  閔 (<CJK Ideograph>)
      7392   => 16#9596#,  --  閖 (<CJK Ideograph>)
      7393   => 16#9598#,  --  閘 (<CJK Ideograph>)
      7394   => 16#9599#,  --  閙 (<CJK Ideograph>)
      7395   => 16#95A0#,  --  閠 (<CJK Ideograph>)
      7396   => 16#95A8#,  --  閨 (<CJK Ideograph>)
      7397   => 16#95A7#,  --  閧 (<CJK Ideograph>)
      7398   => 16#95AD#,  --  閭 (<CJK Ideograph>)
      7399   => 16#95BC#,  --  閼 (<CJK Ideograph>)
      7400   => 16#95BB#,  --  閻 (<CJK Ideograph>)
      7401   => 16#95B9#,  --  閹 (<CJK Ideograph>)
      7402   => 16#95BE#,  --  閾 (<CJK Ideograph>)
      7403   => 16#95CA#,  --  闊 (<CJK Ideograph>)
      7404   => 16#6FF6#,  --  濶 (<CJK Ideograph>)
      7405   => 16#95C3#,  --  闃 (<CJK Ideograph>)
      7406   => 16#95CD#,  --  闍 (<CJK Ideograph>)
      7407   => 16#95CC#,  --  闌 (<CJK Ideograph>)
      7408   => 16#95D5#,  --  闕 (<CJK Ideograph>)
      7409   => 16#95D4#,  --  闔 (<CJK Ideograph>)
      7410   => 16#95D6#,  --  闖 (<CJK Ideograph>)
      7411   => 16#95DC#,  --  關 (<CJK Ideograph>)
      7412   => 16#95E1#,  --  闡 (<CJK Ideograph>)
      7413   => 16#95E5#,  --  闥 (<CJK Ideograph>)
      7414   => 16#95E2#,  --  闢 (<CJK Ideograph>)
      7415   => 16#9621#,  --  阡 (<CJK Ideograph>)
      7416   => 16#9628#,  --  阨 (<CJK Ideograph>)
      7417   => 16#962E#,  --  阮 (<CJK Ideograph>)
      7418   => 16#962F#,  --  阯 (<CJK Ideograph>)
      7419   => 16#9642#,  --  陂 (<CJK Ideograph>)
      7420   => 16#964C#,  --  陌 (<CJK Ideograph>)
      7421   => 16#964F#,  --  陏 (<CJK Ideograph>)
      7422   => 16#964B#,  --  陋 (<CJK Ideograph>)
      7423   => 16#9677#,  --  陷 (<CJK Ideograph>)
      7424   => 16#965C#,  --  陜 (<CJK Ideograph>)
      7425   => 16#965E#,  --  陞 (<CJK Ideograph>)
      7426   => 16#965D#,  --  陝 (<CJK Ideograph>)
      7427   => 16#965F#,  --  陟 (<CJK Ideograph>)
      7428   => 16#9666#,  --  陦 (<CJK Ideograph>)
      7429   => 16#9672#,  --  陲 (<CJK Ideograph>)
      7430   => 16#966C#,  --  陬 (<CJK Ideograph>)
      7431   => 16#968D#,  --  隍 (<CJK Ideograph>)
      7432   => 16#9698#,  --  隘 (<CJK Ideograph>)
      7433   => 16#9695#,  --  隕 (<CJK Ideograph>)
      7434   => 16#9697#,  --  隗 (<CJK Ideograph>)
      7435   => 16#96AA#,  --  險 (<CJK Ideograph>)
      7436   => 16#96A7#,  --  隧 (<CJK Ideograph>)
      7437   => 16#96B1#,  --  隱 (<CJK Ideograph>)
      7438   => 16#96B2#,  --  隲 (<CJK Ideograph>)
      7439   => 16#96B0#,  --  隰 (<CJK Ideograph>)
      7440   => 16#96B4#,  --  隴 (<CJK Ideograph>)
      7441   => 16#96B6#,  --  隶 (<CJK Ideograph>)
      7442   => 16#96B8#,  --  隸 (<CJK Ideograph>)
      7443   => 16#96B9#,  --  隹 (<CJK Ideograph>)
      7444   => 16#96CE#,  --  雎 (<CJK Ideograph>)
      7445   => 16#96CB#,  --  雋 (<CJK Ideograph>)
      7446   => 16#96C9#,  --  雉 (<CJK Ideograph>)
      7447   => 16#96CD#,  --  雍 (<CJK Ideograph>)
      7448   => 16#894D#,  --  襍 (<CJK Ideograph>)
      7449   => 16#96DC#,  --  雜 (<CJK Ideograph>)
      7450   => 16#970D#,  --  霍 (<CJK Ideograph>)
      7451   => 16#96D5#,  --  雕 (<CJK Ideograph>)
      7452   => 16#96F9#,  --  雹 (<CJK Ideograph>)
      7453   => 16#9704#,  --  霄 (<CJK Ideograph>)
      7454   => 16#9706#,  --  霆 (<CJK Ideograph>)
      7455   => 16#9708#,  --  霈 (<CJK Ideograph>)
      7456   => 16#9713#,  --  霓 (<CJK Ideograph>)
      7457   => 16#970E#,  --  霎 (<CJK Ideograph>)
      7458   => 16#9711#,  --  霑 (<CJK Ideograph>)
      7459   => 16#970F#,  --  霏 (<CJK Ideograph>)
      7460   => 16#9716#,  --  霖 (<CJK Ideograph>)
      7461   => 16#9719#,  --  霙 (<CJK Ideograph>)
      7462   => 16#9724#,  --  霤 (<CJK Ideograph>)
      7463   => 16#972A#,  --  霪 (<CJK Ideograph>)
      7464   => 16#9730#,  --  霰 (<CJK Ideograph>)
      7465   => 16#9739#,  --  霹 (<CJK Ideograph>)
      7466   => 16#973D#,  --  霽 (<CJK Ideograph>)
      7467   => 16#973E#,  --  霾 (<CJK Ideograph>)
      7468   => 16#9744#,  --  靄 (<CJK Ideograph>)
      7469   => 16#9746#,  --  靆 (<CJK Ideograph>)
      7470   => 16#9748#,  --  靈 (<CJK Ideograph>)
      7471   => 16#9742#,  --  靂 (<CJK Ideograph>)
      7472   => 16#9749#,  --  靉 (<CJK Ideograph>)
      7473   => 16#975C#,  --  靜 (<CJK Ideograph>)
      7474   => 16#9760#,  --  靠 (<CJK Ideograph>)
      7475   => 16#9764#,  --  靤 (<CJK Ideograph>)
      7476   => 16#9766#,  --  靦 (<CJK Ideograph>)
      7477   => 16#9768#,  --  靨 (<CJK Ideograph>)
      7478   => 16#52D2#,  --  勒 (<CJK Ideograph>)
      7479   => 16#976B#,  --  靫 (<CJK Ideograph>)
      7480   => 16#9771#,  --  靱 (<CJK Ideograph>)
      7481   => 16#9779#,  --  靹 (<CJK Ideograph>)
      7482   => 16#9785#,  --  鞅 (<CJK Ideograph>)
      7483   => 16#977C#,  --  靼 (<CJK Ideograph>)
      7484   => 16#9781#,  --  鞁 (<CJK Ideograph>)
      7485   => 16#977A#,  --  靺 (<CJK Ideograph>)
      7486   => 16#9786#,  --  鞆 (<CJK Ideograph>)
      7487   => 16#978B#,  --  鞋 (<CJK Ideograph>)
      7488   => 16#978F#,  --  鞏 (<CJK Ideograph>)
      7489   => 16#9790#,  --  鞐 (<CJK Ideograph>)
      7490   => 16#979C#,  --  鞜 (<CJK Ideograph>)
      7491   => 16#97A8#,  --  鞨 (<CJK Ideograph>)
      7492   => 16#97A6#,  --  鞦 (<CJK Ideograph>)
      7493   => 16#97A3#,  --  鞣 (<CJK Ideograph>)
      7494   => 16#97B3#,  --  鞳 (<CJK Ideograph>)
      7495   => 16#97B4#,  --  鞴 (<CJK Ideograph>)
      7496   => 16#97C3#,  --  韃 (<CJK Ideograph>)
      7497   => 16#97C6#,  --  韆 (<CJK Ideograph>)
      7498   => 16#97C8#,  --  韈 (<CJK Ideograph>)
      7499   => 16#97CB#,  --  韋 (<CJK Ideograph>)
      7500   => 16#97DC#,  --  韜 (<CJK Ideograph>)
      7501   => 16#97ED#,  --  韭 (<CJK Ideograph>)
      7502   => 16#9F4F#,  --  齏 (<CJK Ideograph>)
      7503   => 16#97F2#,  --  韲 (<CJK Ideograph>)
      7504   => 16#7ADF#,  --  竟 (<CJK Ideograph>)
      7505   => 16#97F6#,  --  韶 (<CJK Ideograph>)
      7506   => 16#97F5#,  --  韵 (<CJK Ideograph>)
      7507   => 16#980F#,  --  頏 (<CJK Ideograph>)
      7508   => 16#980C#,  --  頌 (<CJK Ideograph>)
      7509   => 16#9838#,  --  頸 (<CJK Ideograph>)
      7510   => 16#9824#,  --  頤 (<CJK Ideograph>)
      7511   => 16#9821#,  --  頡 (<CJK Ideograph>)
      7512   => 16#9837#,  --  頷 (<CJK Ideograph>)
      7513   => 16#983D#,  --  頽 (<CJK Ideograph>)
      7514   => 16#9846#,  --  顆 (<CJK Ideograph>)
      7515   => 16#984F#,  --  顏 (<CJK Ideograph>)
      7516   => 16#984B#,  --  顋 (<CJK Ideograph>)
      7517   => 16#986B#,  --  顫 (<CJK Ideograph>)
      7518   => 16#986F#,  --  顯 (<CJK Ideograph>)
      7519   => 16#9870#,  --  顰 (<CJK Ideograph>)
      7520   => 16#9871#,  --  顱 (<CJK Ideograph>)
      7521   => 16#9874#,  --  顴 (<CJK Ideograph>)
      7522   => 16#9873#,  --  顳 (<CJK Ideograph>)
      7523   => 16#98AA#,  --  颪 (<CJK Ideograph>)
      7524   => 16#98AF#,  --  颯 (<CJK Ideograph>)
      7525   => 16#98B1#,  --  颱 (<CJK Ideograph>)
      7526   => 16#98B6#,  --  颶 (<CJK Ideograph>)
      7527   => 16#98C4#,  --  飄 (<CJK Ideograph>)
      7528   => 16#98C3#,  --  飃 (<CJK Ideograph>)
      7529   => 16#98C6#,  --  飆 (<CJK Ideograph>)
      7530   => 16#98E9#,  --  飩 (<CJK Ideograph>)
      7531   => 16#98EB#,  --  飫 (<CJK Ideograph>)
      7532   => 16#9903#,  --  餃 (<CJK Ideograph>)
      7533   => 16#9909#,  --  餉 (<CJK Ideograph>)
      7534   => 16#9912#,  --  餒 (<CJK Ideograph>)
      7535   => 16#9914#,  --  餔 (<CJK Ideograph>)
      7536   => 16#9918#,  --  餘 (<CJK Ideograph>)
      7537   => 16#9921#,  --  餡 (<CJK Ideograph>)
      7538   => 16#991D#,  --  餝 (<CJK Ideograph>)
      7539   => 16#991E#,  --  餞 (<CJK Ideograph>)
      7540   => 16#9924#,  --  餤 (<CJK Ideograph>)
      7541   => 16#9920#,  --  餠 (<CJK Ideograph>)
      7542   => 16#992C#,  --  餬 (<CJK Ideograph>)
      7543   => 16#992E#,  --  餮 (<CJK Ideograph>)
      7544   => 16#993D#,  --  餽 (<CJK Ideograph>)
      7545   => 16#993E#,  --  餾 (<CJK Ideograph>)
      7546   => 16#9942#,  --  饂 (<CJK Ideograph>)
      7547   => 16#9949#,  --  饉 (<CJK Ideograph>)
      7548   => 16#9945#,  --  饅 (<CJK Ideograph>)
      7549   => 16#9950#,  --  饐 (<CJK Ideograph>)
      7550   => 16#994B#,  --  饋 (<CJK Ideograph>)
      7551   => 16#9951#,  --  饑 (<CJK Ideograph>)
      7552   => 16#9952#,  --  饒 (<CJK Ideograph>)
      7553   => 16#994C#,  --  饌 (<CJK Ideograph>)
      7554   => 16#9955#,  --  饕 (<CJK Ideograph>)
      7555   => 16#9997#,  --  馗 (<CJK Ideograph>)
      7556   => 16#9998#,  --  馘 (<CJK Ideograph>)
      7557   => 16#99A5#,  --  馥 (<CJK Ideograph>)
      7558   => 16#99AD#,  --  馭 (<CJK Ideograph>)
      7559   => 16#99AE#,  --  馮 (<CJK Ideograph>)
      7560   => 16#99BC#,  --  馼 (<CJK Ideograph>)
      7561   => 16#99DF#,  --  駟 (<CJK Ideograph>)
      7562   => 16#99DB#,  --  駛 (<CJK Ideograph>)
      7563   => 16#99DD#,  --  駝 (<CJK Ideograph>)
      7564   => 16#99D8#,  --  駘 (<CJK Ideograph>)
      7565   => 16#99D1#,  --  駑 (<CJK Ideograph>)
      7566   => 16#99ED#,  --  駭 (<CJK Ideograph>)
      7567   => 16#99EE#,  --  駮 (<CJK Ideograph>)
      7568   => 16#99F1#,  --  駱 (<CJK Ideograph>)
      7569   => 16#99F2#,  --  駲 (<CJK Ideograph>)
      7570   => 16#99FB#,  --  駻 (<CJK Ideograph>)
      7571   => 16#99F8#,  --  駸 (<CJK Ideograph>)
      7572   => 16#9A01#,  --  騁 (<CJK Ideograph>)
      7573   => 16#9A0F#,  --  騏 (<CJK Ideograph>)
      7574   => 16#9A05#,  --  騅 (<CJK Ideograph>)
      7575   => 16#99E2#,  --  駢 (<CJK Ideograph>)
      7576   => 16#9A19#,  --  騙 (<CJK Ideograph>)
      7577   => 16#9A2B#,  --  騫 (<CJK Ideograph>)
      7578   => 16#9A37#,  --  騷 (<CJK Ideograph>)
      7579   => 16#9A45#,  --  驅 (<CJK Ideograph>)
      7580   => 16#9A42#,  --  驂 (<CJK Ideograph>)
      7581   => 16#9A40#,  --  驀 (<CJK Ideograph>)
      7582   => 16#9A43#,  --  驃 (<CJK Ideograph>)
      7583   => 16#9A3E#,  --  騾 (<CJK Ideograph>)
      7584   => 16#9A55#,  --  驕 (<CJK Ideograph>)
      7585   => 16#9A4D#,  --  驍 (<CJK Ideograph>)
      7586   => 16#9A5B#,  --  驛 (<CJK Ideograph>)
      7587   => 16#9A57#,  --  驗 (<CJK Ideograph>)
      7588   => 16#9A5F#,  --  驟 (<CJK Ideograph>)
      7589   => 16#9A62#,  --  驢 (<CJK Ideograph>)
      7590   => 16#9A65#,  --  驥 (<CJK Ideograph>)
      7591   => 16#9A64#,  --  驤 (<CJK Ideograph>)
      7592   => 16#9A69#,  --  驩 (<CJK Ideograph>)
      7593   => 16#9A6B#,  --  驫 (<CJK Ideograph>)
      7594   => 16#9A6A#,  --  驪 (<CJK Ideograph>)
      7595   => 16#9AAD#,  --  骭 (<CJK Ideograph>)
      7596   => 16#9AB0#,  --  骰 (<CJK Ideograph>)
      7597   => 16#9ABC#,  --  骼 (<CJK Ideograph>)
      7598   => 16#9AC0#,  --  髀 (<CJK Ideograph>)
      7599   => 16#9ACF#,  --  髏 (<CJK Ideograph>)
      7600   => 16#9AD1#,  --  髑 (<CJK Ideograph>)
      7601   => 16#9AD3#,  --  髓 (<CJK Ideograph>)
      7602   => 16#9AD4#,  --  體 (<CJK Ideograph>)
      7603   => 16#9ADE#,  --  髞 (<CJK Ideograph>)
      7604   => 16#9ADF#,  --  髟 (<CJK Ideograph>)
      7605   => 16#9AE2#,  --  髢 (<CJK Ideograph>)
      7606   => 16#9AE3#,  --  髣 (<CJK Ideograph>)
      7607   => 16#9AE6#,  --  髦 (<CJK Ideograph>)
      7608   => 16#9AEF#,  --  髯 (<CJK Ideograph>)
      7609   => 16#9AEB#,  --  髫 (<CJK Ideograph>)
      7610   => 16#9AEE#,  --  髮 (<CJK Ideograph>)
      7611   => 16#9AF4#,  --  髴 (<CJK Ideograph>)
      7612   => 16#9AF1#,  --  髱 (<CJK Ideograph>)
      7613   => 16#9AF7#,  --  髷 (<CJK Ideograph>)
      7614   => 16#9AFB#,  --  髻 (<CJK Ideograph>)
      7615   => 16#9B06#,  --  鬆 (<CJK Ideograph>)
      7616   => 16#9B18#,  --  鬘 (<CJK Ideograph>)
      7617   => 16#9B1A#,  --  鬚 (<CJK Ideograph>)
      7618   => 16#9B1F#,  --  鬟 (<CJK Ideograph>)
      7619   => 16#9B22#,  --  鬢 (<CJK Ideograph>)
      7620   => 16#9B23#,  --  鬣 (<CJK Ideograph>)
      7621   => 16#9B25#,  --  鬥 (<CJK Ideograph>)
      7622   => 16#9B27#,  --  鬧 (<CJK Ideograph>)
      7623   => 16#9B28#,  --  鬨 (<CJK Ideograph>)
      7624   => 16#9B29#,  --  鬩 (<CJK Ideograph>)
      7625   => 16#9B2A#,  --  鬪 (<CJK Ideograph>)
      7626   => 16#9B2E#,  --  鬮 (<CJK Ideograph>)
      7627   => 16#9B2F#,  --  鬯 (<CJK Ideograph>)
      7628   => 16#9B32#,  --  鬲 (<CJK Ideograph>)
      7629   => 16#9B44#,  --  魄 (<CJK Ideograph>)
      7630   => 16#9B43#,  --  魃 (<CJK Ideograph>)
      7631   => 16#9B4F#,  --  魏 (<CJK Ideograph>)
      7632   => 16#9B4D#,  --  魍 (<CJK Ideograph>)
      7633   => 16#9B4E#,  --  魎 (<CJK Ideograph>)
      7634   => 16#9B51#,  --  魑 (<CJK Ideograph>)
      7635   => 16#9B58#,  --  魘 (<CJK Ideograph>)
      7636   => 16#9B74#,  --  魴 (<CJK Ideograph>)
      7637   => 16#9B93#,  --  鮓 (<CJK Ideograph>)
      7638   => 16#9B83#,  --  鮃 (<CJK Ideograph>)
      7639   => 16#9B91#,  --  鮑 (<CJK Ideograph>)
      7640   => 16#9B96#,  --  鮖 (<CJK Ideograph>)
      7641   => 16#9B97#,  --  鮗 (<CJK Ideograph>)
      7642   => 16#9B9F#,  --  鮟 (<CJK Ideograph>)
      7643   => 16#9BA0#,  --  鮠 (<CJK Ideograph>)
      7644   => 16#9BA8#,  --  鮨 (<CJK Ideograph>)
      7645   => 16#9BB4#,  --  鮴 (<CJK Ideograph>)
      7646   => 16#9BC0#,  --  鯀 (<CJK Ideograph>)
      7647   => 16#9BCA#,  --  鯊 (<CJK Ideograph>)
      7648   => 16#9BB9#,  --  鮹 (<CJK Ideograph>)
      7649   => 16#9BC6#,  --  鯆 (<CJK Ideograph>)
      7650   => 16#9BCF#,  --  鯏 (<CJK Ideograph>)
      7651   => 16#9BD1#,  --  鯑 (<CJK Ideograph>)
      7652   => 16#9BD2#,  --  鯒 (<CJK Ideograph>)
      7653   => 16#9BE3#,  --  鯣 (<CJK Ideograph>)
      7654   => 16#9BE2#,  --  鯢 (<CJK Ideograph>)
      7655   => 16#9BE4#,  --  鯤 (<CJK Ideograph>)
      7656   => 16#9BD4#,  --  鯔 (<CJK Ideograph>)
      7657   => 16#9BE1#,  --  鯡 (<CJK Ideograph>)
      7658   => 16#9C3A#,  --  鰺 (<CJK Ideograph>)
      7659   => 16#9BF2#,  --  鯲 (<CJK Ideograph>)
      7660   => 16#9BF1#,  --  鯱 (<CJK Ideograph>)
      7661   => 16#9BF0#,  --  鯰 (<CJK Ideograph>)
      7662   => 16#9C15#,  --  鰕 (<CJK Ideograph>)
      7663   => 16#9C14#,  --  鰔 (<CJK Ideograph>)
      7664   => 16#9C09#,  --  鰉 (<CJK Ideograph>)
      7665   => 16#9C13#,  --  鰓 (<CJK Ideograph>)
      7666   => 16#9C0C#,  --  鰌 (<CJK Ideograph>)
      7667   => 16#9C06#,  --  鰆 (<CJK Ideograph>)
      7668   => 16#9C08#,  --  鰈 (<CJK Ideograph>)
      7669   => 16#9C12#,  --  鰒 (<CJK Ideograph>)
      7670   => 16#9C0A#,  --  鰊 (<CJK Ideograph>)
      7671   => 16#9C04#,  --  鰄 (<CJK Ideograph>)
      7672   => 16#9C2E#,  --  鰮 (<CJK Ideograph>)
      7673   => 16#9C1B#,  --  鰛 (<CJK Ideograph>)
      7674   => 16#9C25#,  --  鰥 (<CJK Ideograph>)
      7675   => 16#9C24#,  --  鰤 (<CJK Ideograph>)
      7676   => 16#9C21#,  --  鰡 (<CJK Ideograph>)
      7677   => 16#9C30#,  --  鰰 (<CJK Ideograph>)
      7678   => 16#9C47#,  --  鱇 (<CJK Ideograph>)
      7679   => 16#9C32#,  --  鰲 (<CJK Ideograph>)
      7680   => 16#9C46#,  --  鱆 (<CJK Ideograph>)
      7681   => 16#9C3E#,  --  鰾 (<CJK Ideograph>)
      7682   => 16#9C5A#,  --  鱚 (<CJK Ideograph>)
      7683   => 16#9C60#,  --  鱠 (<CJK Ideograph>)
      7684   => 16#9C67#,  --  鱧 (<CJK Ideograph>)
      7685   => 16#9C76#,  --  鱶 (<CJK Ideograph>)
      7686   => 16#9C78#,  --  鱸 (<CJK Ideograph>)
      7687   => 16#9CE7#,  --  鳧 (<CJK Ideograph>)
      7688   => 16#9CEC#,  --  鳬 (<CJK Ideograph>)
      7689   => 16#9CF0#,  --  鳰 (<CJK Ideograph>)
      7690   => 16#9D09#,  --  鴉 (<CJK Ideograph>)
      7691   => 16#9D08#,  --  鴈 (<CJK Ideograph>)
      7692   => 16#9CEB#,  --  鳫 (<CJK Ideograph>)
      7693   => 16#9D03#,  --  鴃 (<CJK Ideograph>)
      7694   => 16#9D06#,  --  鴆 (<CJK Ideograph>)
      7695   => 16#9D2A#,  --  鴪 (<CJK Ideograph>)
      7696   => 16#9D26#,  --  鴦 (<CJK Ideograph>)
      7697   => 16#9DAF#,  --  鶯 (<CJK Ideograph>)
      7698   => 16#9D23#,  --  鴣 (<CJK Ideograph>)
      7699   => 16#9D1F#,  --  鴟 (<CJK Ideograph>)
      7700   => 16#9D44#,  --  鵄 (<CJK Ideograph>)
      7701   => 16#9D15#,  --  鴕 (<CJK Ideograph>)
      7702   => 16#9D12#,  --  鴒 (<CJK Ideograph>)
      7703   => 16#9D41#,  --  鵁 (<CJK Ideograph>)
      7704   => 16#9D3F#,  --  鴿 (<CJK Ideograph>)
      7705   => 16#9D3E#,  --  鴾 (<CJK Ideograph>)
      7706   => 16#9D46#,  --  鵆 (<CJK Ideograph>)
      7707   => 16#9D48#,  --  鵈 (<CJK Ideograph>)
      7708   => 16#9D5D#,  --  鵝 (<CJK Ideograph>)
      7709   => 16#9D5E#,  --  鵞 (<CJK Ideograph>)
      7710   => 16#9D64#,  --  鵤 (<CJK Ideograph>)
      7711   => 16#9D51#,  --  鵑 (<CJK Ideograph>)
      7712   => 16#9D50#,  --  鵐 (<CJK Ideograph>)
      7713   => 16#9D59#,  --  鵙 (<CJK Ideograph>)
      7714   => 16#9D72#,  --  鵲 (<CJK Ideograph>)
      7715   => 16#9D89#,  --  鶉 (<CJK Ideograph>)
      7716   => 16#9D87#,  --  鶇 (<CJK Ideograph>)
      7717   => 16#9DAB#,  --  鶫 (<CJK Ideograph>)
      7718   => 16#9D6F#,  --  鵯 (<CJK Ideograph>)
      7719   => 16#9D7A#,  --  鵺 (<CJK Ideograph>)
      7720   => 16#9D9A#,  --  鶚 (<CJK Ideograph>)
      7721   => 16#9DA4#,  --  鶤 (<CJK Ideograph>)
      7722   => 16#9DA9#,  --  鶩 (<CJK Ideograph>)
      7723   => 16#9DB2#,  --  鶲 (<CJK Ideograph>)
      7724   => 16#9DC4#,  --  鷄 (<CJK Ideograph>)
      7725   => 16#9DC1#,  --  鷁 (<CJK Ideograph>)
      7726   => 16#9DBB#,  --  鶻 (<CJK Ideograph>)
      7727   => 16#9DB8#,  --  鶸 (<CJK Ideograph>)
      7728   => 16#9DBA#,  --  鶺 (<CJK Ideograph>)
      7729   => 16#9DC6#,  --  鷆 (<CJK Ideograph>)
      7730   => 16#9DCF#,  --  鷏 (<CJK Ideograph>)
      7731   => 16#9DC2#,  --  鷂 (<CJK Ideograph>)
      7732   => 16#9DD9#,  --  鷙 (<CJK Ideograph>)
      7733   => 16#9DD3#,  --  鷓 (<CJK Ideograph>)
      7734   => 16#9DF8#,  --  鷸 (<CJK Ideograph>)
      7735   => 16#9DE6#,  --  鷦 (<CJK Ideograph>)
      7736   => 16#9DED#,  --  鷭 (<CJK Ideograph>)
      7737   => 16#9DEF#,  --  鷯 (<CJK Ideograph>)
      7738   => 16#9DFD#,  --  鷽 (<CJK Ideograph>)
      7739   => 16#9E1A#,  --  鸚 (<CJK Ideograph>)
      7740   => 16#9E1B#,  --  鸛 (<CJK Ideograph>)
      7741   => 16#9E1E#,  --  鸞 (<CJK Ideograph>)
      7742   => 16#9E75#,  --  鹵 (<CJK Ideograph>)
      7743   => 16#9E79#,  --  鹹 (<CJK Ideograph>)
      7744   => 16#9E7D#,  --  鹽 (<CJK Ideograph>)
      7745   => 16#9E81#,  --  麁 (<CJK Ideograph>)
      7746   => 16#9E88#,  --  麈 (<CJK Ideograph>)
      7747   => 16#9E8B#,  --  麋 (<CJK Ideograph>)
      7748   => 16#9E8C#,  --  麌 (<CJK Ideograph>)
      7749   => 16#9E92#,  --  麒 (<CJK Ideograph>)
      7750   => 16#9E95#,  --  麕 (<CJK Ideograph>)
      7751   => 16#9E91#,  --  麑 (<CJK Ideograph>)
      7752   => 16#9E9D#,  --  麝 (<CJK Ideograph>)
      7753   => 16#9EA5#,  --  麥 (<CJK Ideograph>)
      7754   => 16#9EA9#,  --  麩 (<CJK Ideograph>)
      7755   => 16#9EB8#,  --  麸 (<CJK Ideograph>)
      7756   => 16#9EAA#,  --  麪 (<CJK Ideograph>)
      7757   => 16#9EAD#,  --  麭 (<CJK Ideograph>)
      7758   => 16#9761#,  --  靡 (<CJK Ideograph>)
      7759   => 16#9ECC#,  --  黌 (<CJK Ideograph>)
      7760   => 16#9ECE#,  --  黎 (<CJK Ideograph>)
      7761   => 16#9ECF#,  --  黏 (<CJK Ideograph>)
      7762   => 16#9ED0#,  --  黐 (<CJK Ideograph>)
      7763   => 16#9ED4#,  --  黔 (<CJK Ideograph>)
      7764   => 16#9EDC#,  --  黜 (<CJK Ideograph>)
      7765   => 16#9EDE#,  --  點 (<CJK Ideograph>)
      7766   => 16#9EDD#,  --  黝 (<CJK Ideograph>)
      7767   => 16#9EE0#,  --  黠 (<CJK Ideograph>)
      7768   => 16#9EE5#,  --  黥 (<CJK Ideograph>)
      7769   => 16#9EE8#,  --  黨 (<CJK Ideograph>)
      7770   => 16#9EEF#,  --  黯 (<CJK Ideograph>)
      7771   => 16#9EF4#,  --  黴 (<CJK Ideograph>)
      7772   => 16#9EF6#,  --  黶 (<CJK Ideograph>)
      7773   => 16#9EF7#,  --  黷 (<CJK Ideograph>)
      7774   => 16#9EF9#,  --  黹 (<CJK Ideograph>)
      7775   => 16#9EFB#,  --  黻 (<CJK Ideograph>)
      7776   => 16#9EFC#,  --  黼 (<CJK Ideograph>)
      7777   => 16#9EFD#,  --  黽 (<CJK Ideograph>)
      7778   => 16#9F07#,  --  鼇 (<CJK Ideograph>)
      7779   => 16#9F08#,  --  鼈 (<CJK Ideograph>)
      7780   => 16#76B7#,  --  皷 (<CJK Ideograph>)
      7781   => 16#9F15#,  --  鼕 (<CJK Ideograph>)
      7782   => 16#9F21#,  --  鼡 (<CJK Ideograph>)
      7783   => 16#9F2C#,  --  鼬 (<CJK Ideograph>)
      7784   => 16#9F3E#,  --  鼾 (<CJK Ideograph>)
      7785   => 16#9F4A#,  --  齊 (<CJK Ideograph>)
      7786   => 16#9F52#,  --  齒 (<CJK Ideograph>)
      7787   => 16#9F54#,  --  齔 (<CJK Ideograph>)
      7788   => 16#9F63#,  --  齣 (<CJK Ideograph>)
      7789   => 16#9F5F#,  --  齟 (<CJK Ideograph>)
      7790   => 16#9F60#,  --  齠 (<CJK Ideograph>)
      7791   => 16#9F61#,  --  齡 (<CJK Ideograph>)
      7792   => 16#9F66#,  --  齦 (<CJK Ideograph>)
      7793   => 16#9F67#,  --  齧 (<CJK Ideograph>)
      7794   => 16#9F6C#,  --  齬 (<CJK Ideograph>)
      7795   => 16#9F6A#,  --  齪 (<CJK Ideograph>)
      7796   => 16#9F77#,  --  齷 (<CJK Ideograph>)
      7797   => 16#9F72#,  --  齲 (<CJK Ideograph>)
      7798   => 16#9F76#,  --  齶 (<CJK Ideograph>)
      7799   => 16#9F95#,  --  龕 (<CJK Ideograph>)
      7800   => 16#9F9C#,  --  龜 (<CJK Ideograph>)
      7801   => 16#9FA0#,  --  龠 (<CJK Ideograph>)
      7802   => 16#582F#,  --  堯 (<CJK Ideograph>)
      7803   => 16#69C7#,  --  槇 (<CJK Ideograph>)
      7804   => 16#9059#,  --  遙 (<CJK Ideograph>)
      7805   => 16#7464#,  --  瑤 (<CJK Ideograph>)
      7806   => 16#51DC#,  --  凜 (<CJK Ideograph>)
      7807   => 16#7199#,  --  熙 (<CJK Ideograph>)
      8272   => 16#7E8A#,  --  纊 (<CJK Ideograph>)
      8273   => 16#891C#,  --  褜 (<CJK Ideograph>)
      8274   => 16#9348#,  --  鍈 (<CJK Ideograph>)
      8275   => 16#9288#,  --  銈 (<CJK Ideograph>)
      8276   => 16#84DC#,  --  蓜 (<CJK Ideograph>)
      8277   => 16#4FC9#,  --  俉 (<CJK Ideograph>)
      8278   => 16#70BB#,  --  炻 (<CJK Ideograph>)
      8279   => 16#6631#,  --  昱 (<CJK Ideograph>)
      8280   => 16#68C8#,  --  棈 (<CJK Ideograph>)
      8281   => 16#92F9#,  --  鋹 (<CJK Ideograph>)
      8282   => 16#66FB#,  --  曻 (<CJK Ideograph>)
      8283   => 16#5F45#,  --  彅 (<CJK Ideograph>)
      8284   => 16#4E28#,  --  丨 (<CJK Ideograph>)
      8285   => 16#4EE1#,  --  仡 (<CJK Ideograph>)
      8286   => 16#4EFC#,  --  仼 (<CJK Ideograph>)
      8287   => 16#4F00#,  --  伀 (<CJK Ideograph>)
      8288   => 16#4F03#,  --  伃 (<CJK Ideograph>)
      8289   => 16#4F39#,  --  伹 (<CJK Ideograph>)
      8290   => 16#4F56#,  --  佖 (<CJK Ideograph>)
      8291   => 16#4F92#,  --  侒 (<CJK Ideograph>)
      8292   => 16#4F8A#,  --  侊 (<CJK Ideograph>)
      8293   => 16#4F9A#,  --  侚 (<CJK Ideograph>)
      8294   => 16#4F94#,  --  侔 (<CJK Ideograph>)
      8295   => 16#4FCD#,  --  俍 (<CJK Ideograph>)
      8296   => 16#5040#,  --  偀 (<CJK Ideograph>)
      8297   => 16#5022#,  --  倢 (<CJK Ideograph>)
      8298   => 16#4FFF#,  --  俿 (<CJK Ideograph>)
      8299   => 16#501E#,  --  倞 (<CJK Ideograph>)
      8300   => 16#5046#,  --  偆 (<CJK Ideograph>)
      8301   => 16#5070#,  --  偰 (<CJK Ideograph>)
      8302   => 16#5042#,  --  偂 (<CJK Ideograph>)
      8303   => 16#5094#,  --  傔 (<CJK Ideograph>)
      8304   => 16#50F4#,  --  僴 (<CJK Ideograph>)
      8305   => 16#50D8#,  --  僘 (<CJK Ideograph>)
      8306   => 16#514A#,  --  兊 (<CJK Ideograph>)
      8307   => 16#5164#,  --  兤 (<CJK Ideograph>)
      8308   => 16#519D#,  --  冝 (<CJK Ideograph>)
      8309   => 16#51BE#,  --  冾 (<CJK Ideograph>)
      8310   => 16#51EC#,  --  凬 (<CJK Ideograph>)
      8311   => 16#5215#,  --  刕 (<CJK Ideograph>)
      8312   => 16#529C#,  --  劜 (<CJK Ideograph>)
      8313   => 16#52A6#,  --  劦 (<CJK Ideograph>)
      8314   => 16#52C0#,  --  勀 (<CJK Ideograph>)
      8315   => 16#52DB#,  --  勛 (<CJK Ideograph>)
      8316   => 16#5300#,  --  匀 (<CJK Ideograph>)
      8317   => 16#5307#,  --  匇 (<CJK Ideograph>)
      8318   => 16#5324#,  --  匤 (<CJK Ideograph>)
      8319   => 16#5372#,  --  卲 (<CJK Ideograph>)
      8320   => 16#5393#,  --  厓 (<CJK Ideograph>)
      8321   => 16#53B2#,  --  厲 (<CJK Ideograph>)
      8322   => 16#53DD#,  --  叝 (<CJK Ideograph>)
      8323   => 16#FA0E#,  --  﨎 (CJK COMPATIBILITY IDEOGRAPH-FA0E)
      8324   => 16#549C#,  --  咜 (<CJK Ideograph>)
      8325   => 16#548A#,  --  咊 (<CJK Ideograph>)
      8326   => 16#54A9#,  --  咩 (<CJK Ideograph>)
      8327   => 16#54FF#,  --  哿 (<CJK Ideograph>)
      8328   => 16#5586#,  --  喆 (<CJK Ideograph>)
      8329   => 16#5759#,  --  坙 (<CJK Ideograph>)
      8330   => 16#5765#,  --  坥 (<CJK Ideograph>)
      8331   => 16#57AC#,  --  垬 (<CJK Ideograph>)
      8332   => 16#57C8#,  --  埈 (<CJK Ideograph>)
      8333   => 16#57C7#,  --  埇 (<CJK Ideograph>)
      8334   => 16#FA0F#,  --  﨏 (CJK COMPATIBILITY IDEOGRAPH-FA0F)
      8335   => 16#FA10#,  --  塚 (CJK COMPATIBILITY IDEOGRAPH-FA10)
      8336   => 16#589E#,  --  增 (<CJK Ideograph>)
      8337   => 16#58B2#,  --  墲 (<CJK Ideograph>)
      8338   => 16#590B#,  --  夋 (<CJK Ideograph>)
      8339   => 16#5953#,  --  奓 (<CJK Ideograph>)
      8340   => 16#595B#,  --  奛 (<CJK Ideograph>)
      8341   => 16#595D#,  --  奝 (<CJK Ideograph>)
      8342   => 16#5963#,  --  奣 (<CJK Ideograph>)
      8343   => 16#59A4#,  --  妤 (<CJK Ideograph>)
      8344   => 16#59BA#,  --  妺 (<CJK Ideograph>)
      8345   => 16#5B56#,  --  孖 (<CJK Ideograph>)
      8346   => 16#5BC0#,  --  寀 (<CJK Ideograph>)
      8347   => 16#752F#,  --  甯 (<CJK Ideograph>)
      8348   => 16#5BD8#,  --  寘 (<CJK Ideograph>)
      8349   => 16#5BEC#,  --  寬 (<CJK Ideograph>)
      8350   => 16#5C1E#,  --  尞 (<CJK Ideograph>)
      8351   => 16#5CA6#,  --  岦 (<CJK Ideograph>)
      8352   => 16#5CBA#,  --  岺 (<CJK Ideograph>)
      8353   => 16#5CF5#,  --  峵 (<CJK Ideograph>)
      8354   => 16#5D27#,  --  崧 (<CJK Ideograph>)
      8355   => 16#5D53#,  --  嵓 (<CJK Ideograph>)
      8356   => 16#FA11#,  --  﨑 (CJK COMPATIBILITY IDEOGRAPH-FA11)
      8357   => 16#5D42#,  --  嵂 (<CJK Ideograph>)
      8358   => 16#5D6D#,  --  嵭 (<CJK Ideograph>)
      8359   => 16#5DB8#,  --  嶸 (<CJK Ideograph>)
      8360   => 16#5DB9#,  --  嶹 (<CJK Ideograph>)
      8361   => 16#5DD0#,  --  巐 (<CJK Ideograph>)
      8362   => 16#5F21#,  --  弡 (<CJK Ideograph>)
      8363   => 16#5F34#,  --  弴 (<CJK Ideograph>)
      8364   => 16#5F67#,  --  彧 (<CJK Ideograph>)
      8365   => 16#5FB7#,  --  德 (<CJK Ideograph>)
      8366   => 16#5FDE#,  --  忞 (<CJK Ideograph>)
      8367   => 16#605D#,  --  恝 (<CJK Ideograph>)
      8368   => 16#6085#,  --  悅 (<CJK Ideograph>)
      8369   => 16#608A#,  --  悊 (<CJK Ideograph>)
      8370   => 16#60DE#,  --  惞 (<CJK Ideograph>)
      8371   => 16#60D5#,  --  惕 (<CJK Ideograph>)
      8372   => 16#6120#,  --  愠 (<CJK Ideograph>)
      8373   => 16#60F2#,  --  惲 (<CJK Ideograph>)
      8374   => 16#6111#,  --  愑 (<CJK Ideograph>)
      8375   => 16#6137#,  --  愷 (<CJK Ideograph>)
      8376   => 16#6130#,  --  愰 (<CJK Ideograph>)
      8377   => 16#6198#,  --  憘 (<CJK Ideograph>)
      8378   => 16#6213#,  --  戓 (<CJK Ideograph>)
      8379   => 16#62A6#,  --  抦 (<CJK Ideograph>)
      8380   => 16#63F5#,  --  揵 (<CJK Ideograph>)
      8381   => 16#6460#,  --  摠 (<CJK Ideograph>)
      8382   => 16#649D#,  --  撝 (<CJK Ideograph>)
      8383   => 16#64CE#,  --  擎 (<CJK Ideograph>)
      8384   => 16#654E#,  --  敎 (<CJK Ideograph>)
      8385   => 16#6600#,  --  昀 (<CJK Ideograph>)
      8386   => 16#6615#,  --  昕 (<CJK Ideograph>)
      8387   => 16#663B#,  --  昻 (<CJK Ideograph>)
      8388   => 16#6609#,  --  昉 (<CJK Ideograph>)
      8389   => 16#662E#,  --  昮 (<CJK Ideograph>)
      8390   => 16#661E#,  --  昞 (<CJK Ideograph>)
      8391   => 16#6624#,  --  昤 (<CJK Ideograph>)
      8392   => 16#6665#,  --  晥 (<CJK Ideograph>)
      8393   => 16#6657#,  --  晗 (<CJK Ideograph>)
      8394   => 16#6659#,  --  晙 (<CJK Ideograph>)
      8395   => 16#FA12#,  --  晴 (CJK COMPATIBILITY IDEOGRAPH-FA12)
      8396   => 16#6673#,  --  晳 (<CJK Ideograph>)
      8397   => 16#6699#,  --  暙 (<CJK Ideograph>)
      8398   => 16#66A0#,  --  暠 (<CJK Ideograph>)
      8399   => 16#66B2#,  --  暲 (<CJK Ideograph>)
      8400   => 16#66BF#,  --  暿 (<CJK Ideograph>)
      8401   => 16#66FA#,  --  曺 (<CJK Ideograph>)
      8402   => 16#670E#,  --  朎 (<CJK Ideograph>)
      8403   => 16#F929#,  --  朗 (CJK COMPATIBILITY IDEOGRAPH-F929)
      8404   => 16#6766#,  --  杦 (<CJK Ideograph>)
      8405   => 16#67BB#,  --  枻 (<CJK Ideograph>)
      8406   => 16#6852#,  --  桒 (<CJK Ideograph>)
      8407   => 16#67C0#,  --  柀 (<CJK Ideograph>)
      8408   => 16#6801#,  --  栁 (<CJK Ideograph>)
      8409   => 16#6844#,  --  桄 (<CJK Ideograph>)
      8410   => 16#68CF#,  --  棏 (<CJK Ideograph>)
      8411   => 16#FA13#,  --  﨓 (CJK COMPATIBILITY IDEOGRAPH-FA13)
      8412   => 16#6968#,  --  楨 (<CJK Ideograph>)
      8413   => 16#FA14#,  --  﨔 (CJK COMPATIBILITY IDEOGRAPH-FA14)
      8414   => 16#6998#,  --  榘 (<CJK Ideograph>)
      8415   => 16#69E2#,  --  槢 (<CJK Ideograph>)
      8416   => 16#6A30#,  --  樰 (<CJK Ideograph>)
      8417   => 16#6A6B#,  --  橫 (<CJK Ideograph>)
      8418   => 16#6A46#,  --  橆 (<CJK Ideograph>)
      8419   => 16#6A73#,  --  橳 (<CJK Ideograph>)
      8420   => 16#6A7E#,  --  橾 (<CJK Ideograph>)
      8421   => 16#6AE2#,  --  櫢 (<CJK Ideograph>)
      8422   => 16#6AE4#,  --  櫤 (<CJK Ideograph>)
      8423   => 16#6BD6#,  --  毖 (<CJK Ideograph>)
      8424   => 16#6C3F#,  --  氿 (<CJK Ideograph>)
      8425   => 16#6C5C#,  --  汜 (<CJK Ideograph>)
      8426   => 16#6C86#,  --  沆 (<CJK Ideograph>)
      8427   => 16#6C6F#,  --  汯 (<CJK Ideograph>)
      8428   => 16#6CDA#,  --  泚 (<CJK Ideograph>)
      8429   => 16#6D04#,  --  洄 (<CJK Ideograph>)
      8430   => 16#6D87#,  --  涇 (<CJK Ideograph>)
      8431   => 16#6D6F#,  --  浯 (<CJK Ideograph>)
      8432   => 16#6D96#,  --  涖 (<CJK Ideograph>)
      8433   => 16#6DAC#,  --  涬 (<CJK Ideograph>)
      8434   => 16#6DCF#,  --  淏 (<CJK Ideograph>)
      8435   => 16#6DF8#,  --  淸 (<CJK Ideograph>)
      8436   => 16#6DF2#,  --  淲 (<CJK Ideograph>)
      8437   => 16#6DFC#,  --  淼 (<CJK Ideograph>)
      8438   => 16#6E39#,  --  渹 (<CJK Ideograph>)
      8439   => 16#6E5C#,  --  湜 (<CJK Ideograph>)
      8440   => 16#6E27#,  --  渧 (<CJK Ideograph>)
      8441   => 16#6E3C#,  --  渼 (<CJK Ideograph>)
      8442   => 16#6EBF#,  --  溿 (<CJK Ideograph>)
      8443   => 16#6F88#,  --  澈 (<CJK Ideograph>)
      8444   => 16#6FB5#,  --  澵 (<CJK Ideograph>)
      8445   => 16#6FF5#,  --  濵 (<CJK Ideograph>)
      8446   => 16#7005#,  --  瀅 (<CJK Ideograph>)
      8447   => 16#7007#,  --  瀇 (<CJK Ideograph>)
      8448   => 16#7028#,  --  瀨 (<CJK Ideograph>)
      8449   => 16#7085#,  --  炅 (<CJK Ideograph>)
      8450   => 16#70AB#,  --  炫 (<CJK Ideograph>)
      8451   => 16#710F#,  --  焏 (<CJK Ideograph>)
      8452   => 16#7104#,  --  焄 (<CJK Ideograph>)
      8453   => 16#715C#,  --  煜 (<CJK Ideograph>)
      8454   => 16#7146#,  --  煆 (<CJK Ideograph>)
      8455   => 16#7147#,  --  煇 (<CJK Ideograph>)
      8456   => 16#FA15#,  --  凞 (CJK COMPATIBILITY IDEOGRAPH-FA15)
      8457   => 16#71C1#,  --  燁 (<CJK Ideograph>)
      8458   => 16#71FE#,  --  燾 (<CJK Ideograph>)
      8459   => 16#72B1#,  --  犱 (<CJK Ideograph>)
      8460   => 16#72BE#,  --  犾 (<CJK Ideograph>)
      8461   => 16#7324#,  --  猤 (<CJK Ideograph>)
      8462   => 16#FA16#,  --  猪 (CJK COMPATIBILITY IDEOGRAPH-FA16)
      8463   => 16#7377#,  --  獷 (<CJK Ideograph>)
      8464   => 16#73BD#,  --  玽 (<CJK Ideograph>)
      8465   => 16#73C9#,  --  珉 (<CJK Ideograph>)
      8466   => 16#73D6#,  --  珖 (<CJK Ideograph>)
      8467   => 16#73E3#,  --  珣 (<CJK Ideograph>)
      8468   => 16#73D2#,  --  珒 (<CJK Ideograph>)
      8469   => 16#7407#,  --  琇 (<CJK Ideograph>)
      8470   => 16#73F5#,  --  珵 (<CJK Ideograph>)
      8471   => 16#7426#,  --  琦 (<CJK Ideograph>)
      8472   => 16#742A#,  --  琪 (<CJK Ideograph>)
      8473   => 16#7429#,  --  琩 (<CJK Ideograph>)
      8474   => 16#742E#,  --  琮 (<CJK Ideograph>)
      8475   => 16#7462#,  --  瑢 (<CJK Ideograph>)
      8476   => 16#7489#,  --  璉 (<CJK Ideograph>)
      8477   => 16#749F#,  --  璟 (<CJK Ideograph>)
      8478   => 16#7501#,  --  甁 (<CJK Ideograph>)
      8479   => 16#756F#,  --  畯 (<CJK Ideograph>)
      8480   => 16#7682#,  --  皂 (<CJK Ideograph>)
      8481   => 16#769C#,  --  皜 (<CJK Ideograph>)
      8482   => 16#769E#,  --  皞 (<CJK Ideograph>)
      8483   => 16#769B#,  --  皛 (<CJK Ideograph>)
      8484   => 16#76A6#,  --  皦 (<CJK Ideograph>)
      8485   => 16#FA17#,  --  益 (CJK COMPATIBILITY IDEOGRAPH-FA17)
      8486   => 16#7746#,  --  睆 (<CJK Ideograph>)
      8487   => 16#52AF#,  --  劯 (<CJK Ideograph>)
      8488   => 16#7821#,  --  砡 (<CJK Ideograph>)
      8489   => 16#784E#,  --  硎 (<CJK Ideograph>)
      8490   => 16#7864#,  --  硤 (<CJK Ideograph>)
      8491   => 16#787A#,  --  硺 (<CJK Ideograph>)
      8492   => 16#7930#,  --  礰 (<CJK Ideograph>)
      8493   => 16#FA18#,  --  礼 (CJK COMPATIBILITY IDEOGRAPH-FA18)
      8494   => 16#FA19#,  --  神 (CJK COMPATIBILITY IDEOGRAPH-FA19)
      8495   => 16#FA1A#,  --  祥 (CJK COMPATIBILITY IDEOGRAPH-FA1A)
      8496   => 16#7994#,  --  禔 (<CJK Ideograph>)
      8497   => 16#FA1B#,  --  福 (CJK COMPATIBILITY IDEOGRAPH-FA1B)
      8498   => 16#799B#,  --  禛 (<CJK Ideograph>)
      8499   => 16#7AD1#,  --  竑 (<CJK Ideograph>)
      8500   => 16#7AE7#,  --  竧 (<CJK Ideograph>)
      8501   => 16#FA1C#,  --  靖 (CJK COMPATIBILITY IDEOGRAPH-FA1C)
      8502   => 16#7AEB#,  --  竫 (<CJK Ideograph>)
      8503   => 16#7B9E#,  --  箞 (<CJK Ideograph>)
      8504   => 16#FA1D#,  --  精 (CJK COMPATIBILITY IDEOGRAPH-FA1D)
      8505   => 16#7D48#,  --  絈 (<CJK Ideograph>)
      8506   => 16#7D5C#,  --  絜 (<CJK Ideograph>)
      8507   => 16#7DB7#,  --  綷 (<CJK Ideograph>)
      8508   => 16#7DA0#,  --  綠 (<CJK Ideograph>)
      8509   => 16#7DD6#,  --  緖 (<CJK Ideograph>)
      8510   => 16#7E52#,  --  繒 (<CJK Ideograph>)
      8511   => 16#7F47#,  --  罇 (<CJK Ideograph>)
      8512   => 16#7FA1#,  --  羡 (<CJK Ideograph>)
      8513   => 16#FA1E#,  --  羽 (CJK COMPATIBILITY IDEOGRAPH-FA1E)
      8514   => 16#8301#,  --  茁 (<CJK Ideograph>)
      8515   => 16#8362#,  --  荢 (<CJK Ideograph>)
      8516   => 16#837F#,  --  荿 (<CJK Ideograph>)
      8517   => 16#83C7#,  --  菇 (<CJK Ideograph>)
      8518   => 16#83F6#,  --  菶 (<CJK Ideograph>)
      8519   => 16#8448#,  --  葈 (<CJK Ideograph>)
      8520   => 16#84B4#,  --  蒴 (<CJK Ideograph>)
      8521   => 16#8553#,  --  蕓 (<CJK Ideograph>)
      8522   => 16#8559#,  --  蕙 (<CJK Ideograph>)
      8523   => 16#856B#,  --  蕫 (<CJK Ideograph>)
      8524   => 16#FA1F#,  --  﨟 (CJK COMPATIBILITY IDEOGRAPH-FA1F)
      8525   => 16#85B0#,  --  薰 (<CJK Ideograph>)
      8526   => 16#FA20#,  --  蘒 (CJK COMPATIBILITY IDEOGRAPH-FA20)
      8527   => 16#FA21#,  --  﨡 (CJK COMPATIBILITY IDEOGRAPH-FA21)
      8528   => 16#8807#,  --  蠇 (<CJK Ideograph>)
      8529   => 16#88F5#,  --  裵 (<CJK Ideograph>)
      8530   => 16#8A12#,  --  訒 (<CJK Ideograph>)
      8531   => 16#8A37#,  --  訷 (<CJK Ideograph>)
      8532   => 16#8A79#,  --  詹 (<CJK Ideograph>)
      8533   => 16#8AA7#,  --  誧 (<CJK Ideograph>)
      8534   => 16#8ABE#,  --  誾 (<CJK Ideograph>)
      8535   => 16#8ADF#,  --  諟 (<CJK Ideograph>)
      8536   => 16#FA22#,  --  諸 (CJK COMPATIBILITY IDEOGRAPH-FA22)
      8537   => 16#8AF6#,  --  諶 (<CJK Ideograph>)
      8538   => 16#8B53#,  --  譓 (<CJK Ideograph>)
      8539   => 16#8B7F#,  --  譿 (<CJK Ideograph>)
      8540   => 16#8CF0#,  --  賰 (<CJK Ideograph>)
      8541   => 16#8CF4#,  --  賴 (<CJK Ideograph>)
      8542   => 16#8D12#,  --  贒 (<CJK Ideograph>)
      8543   => 16#8D76#,  --  赶 (<CJK Ideograph>)
      8544   => 16#FA23#,  --  﨣 (CJK COMPATIBILITY IDEOGRAPH-FA23)
      8545   => 16#8ECF#,  --  軏 (<CJK Ideograph>)
      8546   => 16#FA24#,  --  﨤 (CJK COMPATIBILITY IDEOGRAPH-FA24)
      8547   => 16#FA25#,  --  逸 (CJK COMPATIBILITY IDEOGRAPH-FA25)
      8548   => 16#9067#,  --  遧 (<CJK Ideograph>)
      8549   => 16#90DE#,  --  郞 (<CJK Ideograph>)
      8550   => 16#FA26#,  --  都 (CJK COMPATIBILITY IDEOGRAPH-FA26)
      8551   => 16#9115#,  --  鄕 (<CJK Ideograph>)
      8552   => 16#9127#,  --  鄧 (<CJK Ideograph>)
      8553   => 16#91DA#,  --  釚 (<CJK Ideograph>)
      8554   => 16#91D7#,  --  釗 (<CJK Ideograph>)
      8555   => 16#91DE#,  --  釞 (<CJK Ideograph>)
      8556   => 16#91ED#,  --  釭 (<CJK Ideograph>)
      8557   => 16#91EE#,  --  釮 (<CJK Ideograph>)
      8558   => 16#91E4#,  --  釤 (<CJK Ideograph>)
      8559   => 16#91E5#,  --  釥 (<CJK Ideograph>)
      8560   => 16#9206#,  --  鈆 (<CJK Ideograph>)
      8561   => 16#9210#,  --  鈐 (<CJK Ideograph>)
      8562   => 16#920A#,  --  鈊 (<CJK Ideograph>)
      8563   => 16#923A#,  --  鈺 (<CJK Ideograph>)
      8564   => 16#9240#,  --  鉀 (<CJK Ideograph>)
      8565   => 16#923C#,  --  鈼 (<CJK Ideograph>)
      8566   => 16#924E#,  --  鉎 (<CJK Ideograph>)
      8567   => 16#9259#,  --  鉙 (<CJK Ideograph>)
      8568   => 16#9251#,  --  鉑 (<CJK Ideograph>)
      8569   => 16#9239#,  --  鈹 (<CJK Ideograph>)
      8570   => 16#9267#,  --  鉧 (<CJK Ideograph>)
      8571   => 16#92A7#,  --  銧 (<CJK Ideograph>)
      8572   => 16#9277#,  --  鉷 (<CJK Ideograph>)
      8573   => 16#9278#,  --  鉸 (<CJK Ideograph>)
      8574   => 16#92E7#,  --  鋧 (<CJK Ideograph>)
      8575   => 16#92D7#,  --  鋗 (<CJK Ideograph>)
      8576   => 16#92D9#,  --  鋙 (<CJK Ideograph>)
      8577   => 16#92D0#,  --  鋐 (<CJK Ideograph>)
      8578   => 16#FA27#,  --  﨧 (CJK COMPATIBILITY IDEOGRAPH-FA27)
      8579   => 16#92D5#,  --  鋕 (<CJK Ideograph>)
      8580   => 16#92E0#,  --  鋠 (<CJK Ideograph>)
      8581   => 16#92D3#,  --  鋓 (<CJK Ideograph>)
      8582   => 16#9325#,  --  錥 (<CJK Ideograph>)
      8583   => 16#9321#,  --  錡 (<CJK Ideograph>)
      8584   => 16#92FB#,  --  鋻 (<CJK Ideograph>)
      8585   => 16#FA28#,  --  﨨 (CJK COMPATIBILITY IDEOGRAPH-FA28)
      8586   => 16#931E#,  --  錞 (<CJK Ideograph>)
      8587   => 16#92FF#,  --  鋿 (<CJK Ideograph>)
      8588   => 16#931D#,  --  錝 (<CJK Ideograph>)
      8589   => 16#9302#,  --  錂 (<CJK Ideograph>)
      8590   => 16#9370#,  --  鍰 (<CJK Ideograph>)
      8591   => 16#9357#,  --  鍗 (<CJK Ideograph>)
      8592   => 16#93A4#,  --  鎤 (<CJK Ideograph>)
      8593   => 16#93C6#,  --  鏆 (<CJK Ideograph>)
      8594   => 16#93DE#,  --  鏞 (<CJK Ideograph>)
      8595   => 16#93F8#,  --  鏸 (<CJK Ideograph>)
      8596   => 16#9431#,  --  鐱 (<CJK Ideograph>)
      8597   => 16#9445#,  --  鑅 (<CJK Ideograph>)
      8598   => 16#9448#,  --  鑈 (<CJK Ideograph>)
      8599   => 16#9592#,  --  閒 (<CJK Ideograph>)
      8600   => 16#F9DC#,  --  隆 (CJK COMPATIBILITY IDEOGRAPH-F9DC)
      8601   => 16#FA29#,  --  﨩 (CJK COMPATIBILITY IDEOGRAPH-FA29)
      8602   => 16#969D#,  --  隝 (<CJK Ideograph>)
      8603   => 16#96AF#,  --  隯 (<CJK Ideograph>)
      8604   => 16#9733#,  --  霳 (<CJK Ideograph>)
      8605   => 16#973B#,  --  霻 (<CJK Ideograph>)
      8606   => 16#9743#,  --  靃 (<CJK Ideograph>)
      8607   => 16#974D#,  --  靍 (<CJK Ideograph>)
      8608   => 16#974F#,  --  靏 (<CJK Ideograph>)
      8609   => 16#9751#,  --  靑 (<CJK Ideograph>)
      8610   => 16#9755#,  --  靕 (<CJK Ideograph>)
      8611   => 16#9857#,  --  顗 (<CJK Ideograph>)
      8612   => 16#9865#,  --  顥 (<CJK Ideograph>)
      8613   => 16#FA2A#,  --  飯 (CJK COMPATIBILITY IDEOGRAPH-FA2A)
      8614   => 16#FA2B#,  --  飼 (CJK COMPATIBILITY IDEOGRAPH-FA2B)
      8615   => 16#9927#,  --  餧 (<CJK Ideograph>)
      8616   => 16#FA2C#,  --  館 (CJK COMPATIBILITY IDEOGRAPH-FA2C)
      8617   => 16#999E#,  --  馞 (<CJK Ideograph>)
      8618   => 16#9A4E#,  --  驎 (<CJK Ideograph>)
      8619   => 16#9AD9#,  --  髙 (<CJK Ideograph>)
      8620   => 16#9ADC#,  --  髜 (<CJK Ideograph>)
      8621   => 16#9B75#,  --  魵 (<CJK Ideograph>)
      8622   => 16#9B72#,  --  魲 (<CJK Ideograph>)
      8623   => 16#9B8F#,  --  鮏 (<CJK Ideograph>)
      8624   => 16#9BB1#,  --  鮱 (<CJK Ideograph>)
      8625   => 16#9BBB#,  --  鮻 (<CJK Ideograph>)
      8626   => 16#9C00#,  --  鰀 (<CJK Ideograph>)
      8627   => 16#9D70#,  --  鵰 (<CJK Ideograph>)
      8628   => 16#9D6B#,  --  鵫 (<CJK Ideograph>)
      8629   => 16#FA2D#,  --  鶴 (CJK COMPATIBILITY IDEOGRAPH-FA2D)
      8630   => 16#9E19#,  --  鸙 (<CJK Ideograph>)
      8631   => 16#9ED1#,  --  黑 (<CJK Ideograph>)
      8634   => 16#2170#,  --  ⅰ (SMALL ROMAN NUMERAL ONE)
      8635   => 16#2171#,  --  ⅱ (SMALL ROMAN NUMERAL TWO)
      8636   => 16#2172#,  --  ⅲ (SMALL ROMAN NUMERAL THREE)
      8637   => 16#2173#,  --  ⅳ (SMALL ROMAN NUMERAL FOUR)
      8638   => 16#2174#,  --  ⅴ (SMALL ROMAN NUMERAL FIVE)
      8639   => 16#2175#,  --  ⅵ (SMALL ROMAN NUMERAL SIX)
      8640   => 16#2176#,  --  ⅶ (SMALL ROMAN NUMERAL SEVEN)
      8641   => 16#2177#,  --  ⅷ (SMALL ROMAN NUMERAL EIGHT)
      8642   => 16#2178#,  --  ⅸ (SMALL ROMAN NUMERAL NINE)
      8643   => 16#2179#,  --  ⅹ (SMALL ROMAN NUMERAL TEN)
      8644   => 16#FFE2#,  --  ￢ (FULLWIDTH NOT SIGN)
      8645   => 16#FFE4#,  --  ￤ (FULLWIDTH BROKEN BAR)
      8646   => 16#FF07#,  --  ＇ (FULLWIDTH APOSTROPHE)
      8647   => 16#FF02#,  --  ＂ (FULLWIDTH QUOTATION MARK)
      10716  => 16#2170#,  --  ⅰ (SMALL ROMAN NUMERAL ONE)
      10717  => 16#2171#,  --  ⅱ (SMALL ROMAN NUMERAL TWO)
      10718  => 16#2172#,  --  ⅲ (SMALL ROMAN NUMERAL THREE)
      10719  => 16#2173#,  --  ⅳ (SMALL ROMAN NUMERAL FOUR)
      10720  => 16#2174#,  --  ⅴ (SMALL ROMAN NUMERAL FIVE)
      10721  => 16#2175#,  --  ⅵ (SMALL ROMAN NUMERAL SIX)
      10722  => 16#2176#,  --  ⅶ (SMALL ROMAN NUMERAL SEVEN)
      10723  => 16#2177#,  --  ⅷ (SMALL ROMAN NUMERAL EIGHT)
      10724  => 16#2178#,  --  ⅸ (SMALL ROMAN NUMERAL NINE)
      10725  => 16#2179#,  --  ⅹ (SMALL ROMAN NUMERAL TEN)
      10726  => 16#2160#,  --  Ⅰ (ROMAN NUMERAL ONE)
      10727  => 16#2161#,  --  Ⅱ (ROMAN NUMERAL TWO)
      10728  => 16#2162#,  --  Ⅲ (ROMAN NUMERAL THREE)
      10729  => 16#2163#,  --  Ⅳ (ROMAN NUMERAL FOUR)
      10730  => 16#2164#,  --  Ⅴ (ROMAN NUMERAL FIVE)
      10731  => 16#2165#,  --  Ⅵ (ROMAN NUMERAL SIX)
      10732  => 16#2166#,  --  Ⅶ (ROMAN NUMERAL SEVEN)
      10733  => 16#2167#,  --  Ⅷ (ROMAN NUMERAL EIGHT)
      10734  => 16#2168#,  --  Ⅸ (ROMAN NUMERAL NINE)
      10735  => 16#2169#,  --  Ⅹ (ROMAN NUMERAL TEN)
      10736  => 16#FFE2#,  --  ￢ (FULLWIDTH NOT SIGN)
      10737  => 16#FFE4#,  --  ￤ (FULLWIDTH BROKEN BAR)
      10738  => 16#FF07#,  --  ＇ (FULLWIDTH APOSTROPHE)
      10739  => 16#FF02#,  --  ＂ (FULLWIDTH QUOTATION MARK)
      10740  => 16#3231#,  --  ㈱ (PARENTHESIZED IDEOGRAPH STOCK)
      10741  => 16#2116#,  --  № (NUMERO SIGN)
      10742  => 16#2121#,  --  ℡ (TELEPHONE SIGN)
      10743  => 16#2235#,  --  ∵ (BECAUSE)
      10744  => 16#7E8A#,  --  纊 (<CJK Ideograph>)
      10745  => 16#891C#,  --  褜 (<CJK Ideograph>)
      10746  => 16#9348#,  --  鍈 (<CJK Ideograph>)
      10747  => 16#9288#,  --  銈 (<CJK Ideograph>)
      10748  => 16#84DC#,  --  蓜 (<CJK Ideograph>)
      10749  => 16#4FC9#,  --  俉 (<CJK Ideograph>)
      10750  => 16#70BB#,  --  炻 (<CJK Ideograph>)
      10751  => 16#6631#,  --  昱 (<CJK Ideograph>)
      10752  => 16#68C8#,  --  棈 (<CJK Ideograph>)
      10753  => 16#92F9#,  --  鋹 (<CJK Ideograph>)
      10754  => 16#66FB#,  --  曻 (<CJK Ideograph>)
      10755  => 16#5F45#,  --  彅 (<CJK Ideograph>)
      10756  => 16#4E28#,  --  丨 (<CJK Ideograph>)
      10757  => 16#4EE1#,  --  仡 (<CJK Ideograph>)
      10758  => 16#4EFC#,  --  仼 (<CJK Ideograph>)
      10759  => 16#4F00#,  --  伀 (<CJK Ideograph>)
      10760  => 16#4F03#,  --  伃 (<CJK Ideograph>)
      10761  => 16#4F39#,  --  伹 (<CJK Ideograph>)
      10762  => 16#4F56#,  --  佖 (<CJK Ideograph>)
      10763  => 16#4F92#,  --  侒 (<CJK Ideograph>)
      10764  => 16#4F8A#,  --  侊 (<CJK Ideograph>)
      10765  => 16#4F9A#,  --  侚 (<CJK Ideograph>)
      10766  => 16#4F94#,  --  侔 (<CJK Ideograph>)
      10767  => 16#4FCD#,  --  俍 (<CJK Ideograph>)
      10768  => 16#5040#,  --  偀 (<CJK Ideograph>)
      10769  => 16#5022#,  --  倢 (<CJK Ideograph>)
      10770  => 16#4FFF#,  --  俿 (<CJK Ideograph>)
      10771  => 16#501E#,  --  倞 (<CJK Ideograph>)
      10772  => 16#5046#,  --  偆 (<CJK Ideograph>)
      10773  => 16#5070#,  --  偰 (<CJK Ideograph>)
      10774  => 16#5042#,  --  偂 (<CJK Ideograph>)
      10775  => 16#5094#,  --  傔 (<CJK Ideograph>)
      10776  => 16#50F4#,  --  僴 (<CJK Ideograph>)
      10777  => 16#50D8#,  --  僘 (<CJK Ideograph>)
      10778  => 16#514A#,  --  兊 (<CJK Ideograph>)
      10779  => 16#5164#,  --  兤 (<CJK Ideograph>)
      10780  => 16#519D#,  --  冝 (<CJK Ideograph>)
      10781  => 16#51BE#,  --  冾 (<CJK Ideograph>)
      10782  => 16#51EC#,  --  凬 (<CJK Ideograph>)
      10783  => 16#5215#,  --  刕 (<CJK Ideograph>)
      10784  => 16#529C#,  --  劜 (<CJK Ideograph>)
      10785  => 16#52A6#,  --  劦 (<CJK Ideograph>)
      10786  => 16#52C0#,  --  勀 (<CJK Ideograph>)
      10787  => 16#52DB#,  --  勛 (<CJK Ideograph>)
      10788  => 16#5300#,  --  匀 (<CJK Ideograph>)
      10789  => 16#5307#,  --  匇 (<CJK Ideograph>)
      10790  => 16#5324#,  --  匤 (<CJK Ideograph>)
      10791  => 16#5372#,  --  卲 (<CJK Ideograph>)
      10792  => 16#5393#,  --  厓 (<CJK Ideograph>)
      10793  => 16#53B2#,  --  厲 (<CJK Ideograph>)
      10794  => 16#53DD#,  --  叝 (<CJK Ideograph>)
      10795  => 16#FA0E#,  --  﨎 (CJK COMPATIBILITY IDEOGRAPH-FA0E)
      10796  => 16#549C#,  --  咜 (<CJK Ideograph>)
      10797  => 16#548A#,  --  咊 (<CJK Ideograph>)
      10798  => 16#54A9#,  --  咩 (<CJK Ideograph>)
      10799  => 16#54FF#,  --  哿 (<CJK Ideograph>)
      10800  => 16#5586#,  --  喆 (<CJK Ideograph>)
      10801  => 16#5759#,  --  坙 (<CJK Ideograph>)
      10802  => 16#5765#,  --  坥 (<CJK Ideograph>)
      10803  => 16#57AC#,  --  垬 (<CJK Ideograph>)
      10804  => 16#57C8#,  --  埈 (<CJK Ideograph>)
      10805  => 16#57C7#,  --  埇 (<CJK Ideograph>)
      10806  => 16#FA0F#,  --  﨏 (CJK COMPATIBILITY IDEOGRAPH-FA0F)
      10807  => 16#FA10#,  --  塚 (CJK COMPATIBILITY IDEOGRAPH-FA10)
      10808  => 16#589E#,  --  增 (<CJK Ideograph>)
      10809  => 16#58B2#,  --  墲 (<CJK Ideograph>)
      10810  => 16#590B#,  --  夋 (<CJK Ideograph>)
      10811  => 16#5953#,  --  奓 (<CJK Ideograph>)
      10812  => 16#595B#,  --  奛 (<CJK Ideograph>)
      10813  => 16#595D#,  --  奝 (<CJK Ideograph>)
      10814  => 16#5963#,  --  奣 (<CJK Ideograph>)
      10815  => 16#59A4#,  --  妤 (<CJK Ideograph>)
      10816  => 16#59BA#,  --  妺 (<CJK Ideograph>)
      10817  => 16#5B56#,  --  孖 (<CJK Ideograph>)
      10818  => 16#5BC0#,  --  寀 (<CJK Ideograph>)
      10819  => 16#752F#,  --  甯 (<CJK Ideograph>)
      10820  => 16#5BD8#,  --  寘 (<CJK Ideograph>)
      10821  => 16#5BEC#,  --  寬 (<CJK Ideograph>)
      10822  => 16#5C1E#,  --  尞 (<CJK Ideograph>)
      10823  => 16#5CA6#,  --  岦 (<CJK Ideograph>)
      10824  => 16#5CBA#,  --  岺 (<CJK Ideograph>)
      10825  => 16#5CF5#,  --  峵 (<CJK Ideograph>)
      10826  => 16#5D27#,  --  崧 (<CJK Ideograph>)
      10827  => 16#5D53#,  --  嵓 (<CJK Ideograph>)
      10828  => 16#FA11#,  --  﨑 (CJK COMPATIBILITY IDEOGRAPH-FA11)
      10829  => 16#5D42#,  --  嵂 (<CJK Ideograph>)
      10830  => 16#5D6D#,  --  嵭 (<CJK Ideograph>)
      10831  => 16#5DB8#,  --  嶸 (<CJK Ideograph>)
      10832  => 16#5DB9#,  --  嶹 (<CJK Ideograph>)
      10833  => 16#5DD0#,  --  巐 (<CJK Ideograph>)
      10834  => 16#5F21#,  --  弡 (<CJK Ideograph>)
      10835  => 16#5F34#,  --  弴 (<CJK Ideograph>)
      10836  => 16#5F67#,  --  彧 (<CJK Ideograph>)
      10837  => 16#5FB7#,  --  德 (<CJK Ideograph>)
      10838  => 16#5FDE#,  --  忞 (<CJK Ideograph>)
      10839  => 16#605D#,  --  恝 (<CJK Ideograph>)
      10840  => 16#6085#,  --  悅 (<CJK Ideograph>)
      10841  => 16#608A#,  --  悊 (<CJK Ideograph>)
      10842  => 16#60DE#,  --  惞 (<CJK Ideograph>)
      10843  => 16#60D5#,  --  惕 (<CJK Ideograph>)
      10844  => 16#6120#,  --  愠 (<CJK Ideograph>)
      10845  => 16#60F2#,  --  惲 (<CJK Ideograph>)
      10846  => 16#6111#,  --  愑 (<CJK Ideograph>)
      10847  => 16#6137#,  --  愷 (<CJK Ideograph>)
      10848  => 16#6130#,  --  愰 (<CJK Ideograph>)
      10849  => 16#6198#,  --  憘 (<CJK Ideograph>)
      10850  => 16#6213#,  --  戓 (<CJK Ideograph>)
      10851  => 16#62A6#,  --  抦 (<CJK Ideograph>)
      10852  => 16#63F5#,  --  揵 (<CJK Ideograph>)
      10853  => 16#6460#,  --  摠 (<CJK Ideograph>)
      10854  => 16#649D#,  --  撝 (<CJK Ideograph>)
      10855  => 16#64CE#,  --  擎 (<CJK Ideograph>)
      10856  => 16#654E#,  --  敎 (<CJK Ideograph>)
      10857  => 16#6600#,  --  昀 (<CJK Ideograph>)
      10858  => 16#6615#,  --  昕 (<CJK Ideograph>)
      10859  => 16#663B#,  --  昻 (<CJK Ideograph>)
      10860  => 16#6609#,  --  昉 (<CJK Ideograph>)
      10861  => 16#662E#,  --  昮 (<CJK Ideograph>)
      10862  => 16#661E#,  --  昞 (<CJK Ideograph>)
      10863  => 16#6624#,  --  昤 (<CJK Ideograph>)
      10864  => 16#6665#,  --  晥 (<CJK Ideograph>)
      10865  => 16#6657#,  --  晗 (<CJK Ideograph>)
      10866  => 16#6659#,  --  晙 (<CJK Ideograph>)
      10867  => 16#FA12#,  --  晴 (CJK COMPATIBILITY IDEOGRAPH-FA12)
      10868  => 16#6673#,  --  晳 (<CJK Ideograph>)
      10869  => 16#6699#,  --  暙 (<CJK Ideograph>)
      10870  => 16#66A0#,  --  暠 (<CJK Ideograph>)
      10871  => 16#66B2#,  --  暲 (<CJK Ideograph>)
      10872  => 16#66BF#,  --  暿 (<CJK Ideograph>)
      10873  => 16#66FA#,  --  曺 (<CJK Ideograph>)
      10874  => 16#670E#,  --  朎 (<CJK Ideograph>)
      10875  => 16#F929#,  --  朗 (CJK COMPATIBILITY IDEOGRAPH-F929)
      10876  => 16#6766#,  --  杦 (<CJK Ideograph>)
      10877  => 16#67BB#,  --  枻 (<CJK Ideograph>)
      10878  => 16#6852#,  --  桒 (<CJK Ideograph>)
      10879  => 16#67C0#,  --  柀 (<CJK Ideograph>)
      10880  => 16#6801#,  --  栁 (<CJK Ideograph>)
      10881  => 16#6844#,  --  桄 (<CJK Ideograph>)
      10882  => 16#68CF#,  --  棏 (<CJK Ideograph>)
      10883  => 16#FA13#,  --  﨓 (CJK COMPATIBILITY IDEOGRAPH-FA13)
      10884  => 16#6968#,  --  楨 (<CJK Ideograph>)
      10885  => 16#FA14#,  --  﨔 (CJK COMPATIBILITY IDEOGRAPH-FA14)
      10886  => 16#6998#,  --  榘 (<CJK Ideograph>)
      10887  => 16#69E2#,  --  槢 (<CJK Ideograph>)
      10888  => 16#6A30#,  --  樰 (<CJK Ideograph>)
      10889  => 16#6A6B#,  --  橫 (<CJK Ideograph>)
      10890  => 16#6A46#,  --  橆 (<CJK Ideograph>)
      10891  => 16#6A73#,  --  橳 (<CJK Ideograph>)
      10892  => 16#6A7E#,  --  橾 (<CJK Ideograph>)
      10893  => 16#6AE2#,  --  櫢 (<CJK Ideograph>)
      10894  => 16#6AE4#,  --  櫤 (<CJK Ideograph>)
      10895  => 16#6BD6#,  --  毖 (<CJK Ideograph>)
      10896  => 16#6C3F#,  --  氿 (<CJK Ideograph>)
      10897  => 16#6C5C#,  --  汜 (<CJK Ideograph>)
      10898  => 16#6C86#,  --  沆 (<CJK Ideograph>)
      10899  => 16#6C6F#,  --  汯 (<CJK Ideograph>)
      10900  => 16#6CDA#,  --  泚 (<CJK Ideograph>)
      10901  => 16#6D04#,  --  洄 (<CJK Ideograph>)
      10902  => 16#6D87#,  --  涇 (<CJK Ideograph>)
      10903  => 16#6D6F#,  --  浯 (<CJK Ideograph>)
      10904  => 16#6D96#,  --  涖 (<CJK Ideograph>)
      10905  => 16#6DAC#,  --  涬 (<CJK Ideograph>)
      10906  => 16#6DCF#,  --  淏 (<CJK Ideograph>)
      10907  => 16#6DF8#,  --  淸 (<CJK Ideograph>)
      10908  => 16#6DF2#,  --  淲 (<CJK Ideograph>)
      10909  => 16#6DFC#,  --  淼 (<CJK Ideograph>)
      10910  => 16#6E39#,  --  渹 (<CJK Ideograph>)
      10911  => 16#6E5C#,  --  湜 (<CJK Ideograph>)
      10912  => 16#6E27#,  --  渧 (<CJK Ideograph>)
      10913  => 16#6E3C#,  --  渼 (<CJK Ideograph>)
      10914  => 16#6EBF#,  --  溿 (<CJK Ideograph>)
      10915  => 16#6F88#,  --  澈 (<CJK Ideograph>)
      10916  => 16#6FB5#,  --  澵 (<CJK Ideograph>)
      10917  => 16#6FF5#,  --  濵 (<CJK Ideograph>)
      10918  => 16#7005#,  --  瀅 (<CJK Ideograph>)
      10919  => 16#7007#,  --  瀇 (<CJK Ideograph>)
      10920  => 16#7028#,  --  瀨 (<CJK Ideograph>)
      10921  => 16#7085#,  --  炅 (<CJK Ideograph>)
      10922  => 16#70AB#,  --  炫 (<CJK Ideograph>)
      10923  => 16#710F#,  --  焏 (<CJK Ideograph>)
      10924  => 16#7104#,  --  焄 (<CJK Ideograph>)
      10925  => 16#715C#,  --  煜 (<CJK Ideograph>)
      10926  => 16#7146#,  --  煆 (<CJK Ideograph>)
      10927  => 16#7147#,  --  煇 (<CJK Ideograph>)
      10928  => 16#FA15#,  --  凞 (CJK COMPATIBILITY IDEOGRAPH-FA15)
      10929  => 16#71C1#,  --  燁 (<CJK Ideograph>)
      10930  => 16#71FE#,  --  燾 (<CJK Ideograph>)
      10931  => 16#72B1#,  --  犱 (<CJK Ideograph>)
      10932  => 16#72BE#,  --  犾 (<CJK Ideograph>)
      10933  => 16#7324#,  --  猤 (<CJK Ideograph>)
      10934  => 16#FA16#,  --  猪 (CJK COMPATIBILITY IDEOGRAPH-FA16)
      10935  => 16#7377#,  --  獷 (<CJK Ideograph>)
      10936  => 16#73BD#,  --  玽 (<CJK Ideograph>)
      10937  => 16#73C9#,  --  珉 (<CJK Ideograph>)
      10938  => 16#73D6#,  --  珖 (<CJK Ideograph>)
      10939  => 16#73E3#,  --  珣 (<CJK Ideograph>)
      10940  => 16#73D2#,  --  珒 (<CJK Ideograph>)
      10941  => 16#7407#,  --  琇 (<CJK Ideograph>)
      10942  => 16#73F5#,  --  珵 (<CJK Ideograph>)
      10943  => 16#7426#,  --  琦 (<CJK Ideograph>)
      10944  => 16#742A#,  --  琪 (<CJK Ideograph>)
      10945  => 16#7429#,  --  琩 (<CJK Ideograph>)
      10946  => 16#742E#,  --  琮 (<CJK Ideograph>)
      10947  => 16#7462#,  --  瑢 (<CJK Ideograph>)
      10948  => 16#7489#,  --  璉 (<CJK Ideograph>)
      10949  => 16#749F#,  --  璟 (<CJK Ideograph>)
      10950  => 16#7501#,  --  甁 (<CJK Ideograph>)
      10951  => 16#756F#,  --  畯 (<CJK Ideograph>)
      10952  => 16#7682#,  --  皂 (<CJK Ideograph>)
      10953  => 16#769C#,  --  皜 (<CJK Ideograph>)
      10954  => 16#769E#,  --  皞 (<CJK Ideograph>)
      10955  => 16#769B#,  --  皛 (<CJK Ideograph>)
      10956  => 16#76A6#,  --  皦 (<CJK Ideograph>)
      10957  => 16#FA17#,  --  益 (CJK COMPATIBILITY IDEOGRAPH-FA17)
      10958  => 16#7746#,  --  睆 (<CJK Ideograph>)
      10959  => 16#52AF#,  --  劯 (<CJK Ideograph>)
      10960  => 16#7821#,  --  砡 (<CJK Ideograph>)
      10961  => 16#784E#,  --  硎 (<CJK Ideograph>)
      10962  => 16#7864#,  --  硤 (<CJK Ideograph>)
      10963  => 16#787A#,  --  硺 (<CJK Ideograph>)
      10964  => 16#7930#,  --  礰 (<CJK Ideograph>)
      10965  => 16#FA18#,  --  礼 (CJK COMPATIBILITY IDEOGRAPH-FA18)
      10966  => 16#FA19#,  --  神 (CJK COMPATIBILITY IDEOGRAPH-FA19)
      10967  => 16#FA1A#,  --  祥 (CJK COMPATIBILITY IDEOGRAPH-FA1A)
      10968  => 16#7994#,  --  禔 (<CJK Ideograph>)
      10969  => 16#FA1B#,  --  福 (CJK COMPATIBILITY IDEOGRAPH-FA1B)
      10970  => 16#799B#,  --  禛 (<CJK Ideograph>)
      10971  => 16#7AD1#,  --  竑 (<CJK Ideograph>)
      10972  => 16#7AE7#,  --  竧 (<CJK Ideograph>)
      10973  => 16#FA1C#,  --  靖 (CJK COMPATIBILITY IDEOGRAPH-FA1C)
      10974  => 16#7AEB#,  --  竫 (<CJK Ideograph>)
      10975  => 16#7B9E#,  --  箞 (<CJK Ideograph>)
      10976  => 16#FA1D#,  --  精 (CJK COMPATIBILITY IDEOGRAPH-FA1D)
      10977  => 16#7D48#,  --  絈 (<CJK Ideograph>)
      10978  => 16#7D5C#,  --  絜 (<CJK Ideograph>)
      10979  => 16#7DB7#,  --  綷 (<CJK Ideograph>)
      10980  => 16#7DA0#,  --  綠 (<CJK Ideograph>)
      10981  => 16#7DD6#,  --  緖 (<CJK Ideograph>)
      10982  => 16#7E52#,  --  繒 (<CJK Ideograph>)
      10983  => 16#7F47#,  --  罇 (<CJK Ideograph>)
      10984  => 16#7FA1#,  --  羡 (<CJK Ideograph>)
      10985  => 16#FA1E#,  --  羽 (CJK COMPATIBILITY IDEOGRAPH-FA1E)
      10986  => 16#8301#,  --  茁 (<CJK Ideograph>)
      10987  => 16#8362#,  --  荢 (<CJK Ideograph>)
      10988  => 16#837F#,  --  荿 (<CJK Ideograph>)
      10989  => 16#83C7#,  --  菇 (<CJK Ideograph>)
      10990  => 16#83F6#,  --  菶 (<CJK Ideograph>)
      10991  => 16#8448#,  --  葈 (<CJK Ideograph>)
      10992  => 16#84B4#,  --  蒴 (<CJK Ideograph>)
      10993  => 16#8553#,  --  蕓 (<CJK Ideograph>)
      10994  => 16#8559#,  --  蕙 (<CJK Ideograph>)
      10995  => 16#856B#,  --  蕫 (<CJK Ideograph>)
      10996  => 16#FA1F#,  --  﨟 (CJK COMPATIBILITY IDEOGRAPH-FA1F)
      10997  => 16#85B0#,  --  薰 (<CJK Ideograph>)
      10998  => 16#FA20#,  --  蘒 (CJK COMPATIBILITY IDEOGRAPH-FA20)
      10999  => 16#FA21#,  --  﨡 (CJK COMPATIBILITY IDEOGRAPH-FA21)
      11000  => 16#8807#,  --  蠇 (<CJK Ideograph>)
      11001  => 16#88F5#,  --  裵 (<CJK Ideograph>)
      11002  => 16#8A12#,  --  訒 (<CJK Ideograph>)
      11003  => 16#8A37#,  --  訷 (<CJK Ideograph>)
      11004  => 16#8A79#,  --  詹 (<CJK Ideograph>)
      11005  => 16#8AA7#,  --  誧 (<CJK Ideograph>)
      11006  => 16#8ABE#,  --  誾 (<CJK Ideograph>)
      11007  => 16#8ADF#,  --  諟 (<CJK Ideograph>)
      11008  => 16#FA22#,  --  諸 (CJK COMPATIBILITY IDEOGRAPH-FA22)
      11009  => 16#8AF6#,  --  諶 (<CJK Ideograph>)
      11010  => 16#8B53#,  --  譓 (<CJK Ideograph>)
      11011  => 16#8B7F#,  --  譿 (<CJK Ideograph>)
      11012  => 16#8CF0#,  --  賰 (<CJK Ideograph>)
      11013  => 16#8CF4#,  --  賴 (<CJK Ideograph>)
      11014  => 16#8D12#,  --  贒 (<CJK Ideograph>)
      11015  => 16#8D76#,  --  赶 (<CJK Ideograph>)
      11016  => 16#FA23#,  --  﨣 (CJK COMPATIBILITY IDEOGRAPH-FA23)
      11017  => 16#8ECF#,  --  軏 (<CJK Ideograph>)
      11018  => 16#FA24#,  --  﨤 (CJK COMPATIBILITY IDEOGRAPH-FA24)
      11019  => 16#FA25#,  --  逸 (CJK COMPATIBILITY IDEOGRAPH-FA25)
      11020  => 16#9067#,  --  遧 (<CJK Ideograph>)
      11021  => 16#90DE#,  --  郞 (<CJK Ideograph>)
      11022  => 16#FA26#,  --  都 (CJK COMPATIBILITY IDEOGRAPH-FA26)
      11023  => 16#9115#,  --  鄕 (<CJK Ideograph>)
      11024  => 16#9127#,  --  鄧 (<CJK Ideograph>)
      11025  => 16#91DA#,  --  釚 (<CJK Ideograph>)
      11026  => 16#91D7#,  --  釗 (<CJK Ideograph>)
      11027  => 16#91DE#,  --  釞 (<CJK Ideograph>)
      11028  => 16#91ED#,  --  釭 (<CJK Ideograph>)
      11029  => 16#91EE#,  --  釮 (<CJK Ideograph>)
      11030  => 16#91E4#,  --  釤 (<CJK Ideograph>)
      11031  => 16#91E5#,  --  釥 (<CJK Ideograph>)
      11032  => 16#9206#,  --  鈆 (<CJK Ideograph>)
      11033  => 16#9210#,  --  鈐 (<CJK Ideograph>)
      11034  => 16#920A#,  --  鈊 (<CJK Ideograph>)
      11035  => 16#923A#,  --  鈺 (<CJK Ideograph>)
      11036  => 16#9240#,  --  鉀 (<CJK Ideograph>)
      11037  => 16#923C#,  --  鈼 (<CJK Ideograph>)
      11038  => 16#924E#,  --  鉎 (<CJK Ideograph>)
      11039  => 16#9259#,  --  鉙 (<CJK Ideograph>)
      11040  => 16#9251#,  --  鉑 (<CJK Ideograph>)
      11041  => 16#9239#,  --  鈹 (<CJK Ideograph>)
      11042  => 16#9267#,  --  鉧 (<CJK Ideograph>)
      11043  => 16#92A7#,  --  銧 (<CJK Ideograph>)
      11044  => 16#9277#,  --  鉷 (<CJK Ideograph>)
      11045  => 16#9278#,  --  鉸 (<CJK Ideograph>)
      11046  => 16#92E7#,  --  鋧 (<CJK Ideograph>)
      11047  => 16#92D7#,  --  鋗 (<CJK Ideograph>)
      11048  => 16#92D9#,  --  鋙 (<CJK Ideograph>)
      11049  => 16#92D0#,  --  鋐 (<CJK Ideograph>)
      11050  => 16#FA27#,  --  﨧 (CJK COMPATIBILITY IDEOGRAPH-FA27)
      11051  => 16#92D5#,  --  鋕 (<CJK Ideograph>)
      11052  => 16#92E0#,  --  鋠 (<CJK Ideograph>)
      11053  => 16#92D3#,  --  鋓 (<CJK Ideograph>)
      11054  => 16#9325#,  --  錥 (<CJK Ideograph>)
      11055  => 16#9321#,  --  錡 (<CJK Ideograph>)
      11056  => 16#92FB#,  --  鋻 (<CJK Ideograph>)
      11057  => 16#FA28#,  --  﨨 (CJK COMPATIBILITY IDEOGRAPH-FA28)
      11058  => 16#931E#,  --  錞 (<CJK Ideograph>)
      11059  => 16#92FF#,  --  鋿 (<CJK Ideograph>)
      11060  => 16#931D#,  --  錝 (<CJK Ideograph>)
      11061  => 16#9302#,  --  錂 (<CJK Ideograph>)
      11062  => 16#9370#,  --  鍰 (<CJK Ideograph>)
      11063  => 16#9357#,  --  鍗 (<CJK Ideograph>)
      11064  => 16#93A4#,  --  鎤 (<CJK Ideograph>)
      11065  => 16#93C6#,  --  鏆 (<CJK Ideograph>)
      11066  => 16#93DE#,  --  鏞 (<CJK Ideograph>)
      11067  => 16#93F8#,  --  鏸 (<CJK Ideograph>)
      11068  => 16#9431#,  --  鐱 (<CJK Ideograph>)
      11069  => 16#9445#,  --  鑅 (<CJK Ideograph>)
      11070  => 16#9448#,  --  鑈 (<CJK Ideograph>)
      11071  => 16#9592#,  --  閒 (<CJK Ideograph>)
      11072  => 16#F9DC#,  --  隆 (CJK COMPATIBILITY IDEOGRAPH-F9DC)
      11073  => 16#FA29#,  --  﨩 (CJK COMPATIBILITY IDEOGRAPH-FA29)
      11074  => 16#969D#,  --  隝 (<CJK Ideograph>)
      11075  => 16#96AF#,  --  隯 (<CJK Ideograph>)
      11076  => 16#9733#,  --  霳 (<CJK Ideograph>)
      11077  => 16#973B#,  --  霻 (<CJK Ideograph>)
      11078  => 16#9743#,  --  靃 (<CJK Ideograph>)
      11079  => 16#974D#,  --  靍 (<CJK Ideograph>)
      11080  => 16#974F#,  --  靏 (<CJK Ideograph>)
      11081  => 16#9751#,  --  靑 (<CJK Ideograph>)
      11082  => 16#9755#,  --  靕 (<CJK Ideograph>)
      11083  => 16#9857#,  --  顗 (<CJK Ideograph>)
      11084  => 16#9865#,  --  顥 (<CJK Ideograph>)
      11085  => 16#FA2A#,  --  飯 (CJK COMPATIBILITY IDEOGRAPH-FA2A)
      11086  => 16#FA2B#,  --  飼 (CJK COMPATIBILITY IDEOGRAPH-FA2B)
      11087  => 16#9927#,  --  餧 (<CJK Ideograph>)
      11088  => 16#FA2C#,  --  館 (CJK COMPATIBILITY IDEOGRAPH-FA2C)
      11089  => 16#999E#,  --  馞 (<CJK Ideograph>)
      11090  => 16#9A4E#,  --  驎 (<CJK Ideograph>)
      11091  => 16#9AD9#,  --  髙 (<CJK Ideograph>)
      11092  => 16#9ADC#,  --  髜 (<CJK Ideograph>)
      11093  => 16#9B75#,  --  魵 (<CJK Ideograph>)
      11094  => 16#9B72#,  --  魲 (<CJK Ideograph>)
      11095  => 16#9B8F#,  --  鮏 (<CJK Ideograph>)
      11096  => 16#9BB1#,  --  鮱 (<CJK Ideograph>)
      11097  => 16#9BBB#,  --  鮻 (<CJK Ideograph>)
      11098  => 16#9C00#,  --  鰀 (<CJK Ideograph>)
      11099  => 16#9D70#,  --  鵰 (<CJK Ideograph>)
      11100  => 16#9D6B#,  --  鵫 (<CJK Ideograph>)
      11101  => 16#FA2D#,  --  鶴 (CJK COMPATIBILITY IDEOGRAPH-FA2D)
      11102  => 16#9E19#,  --  鸙 (<CJK Ideograph>)
      11103  => 16#9ED1#,  --  黑 (<CJK Ideograph>)

      others => 0];

end VSS.Strings.Converters.Decoders.Index_JIS0208;
