--
--  Copyright (C) 2022-2023, AdaCore
--
--  SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
--

pragma Ada_2022;
pragma Style_Checks ("M131");

--  This package contains data from index-jis0212.txt file
--
--  # Identifier: 83bf90dd1c591a4355730d8c4567efc499d74da7490531019ef22a879991cfb7
--  # Date: 2018-01-06
--
--  see the Encoding Standard
--
--  https://encoding.spec.whatwg.org/

with Interfaces;

private package VSS.Strings.Converters.Decoders.EUCJP.JIS0212 is

   Table : constant
     array (Interfaces.Unsigned_32 range 108 .. 7_210)
       of VSS.Unicode.Code_Point :=
     [108    => 16#02D8#,  --  ˘ (BREVE)
      109    => 16#02C7#,  --  ˇ (CARON)
      110    => 16#00B8#,  --  ¸ (CEDILLA)
      111    => 16#02D9#,  --  ˙ (DOT ABOVE)
      112    => 16#02DD#,  --  ˝ (DOUBLE ACUTE ACCENT)
      113    => 16#00AF#,  --  ¯ (MACRON)
      114    => 16#02DB#,  --  ˛ (OGONEK)
      115    => 16#02DA#,  --  ˚ (RING ABOVE)
      116    => 16#FF5E#,  --  ～ (FULLWIDTH TILDE)
      117    => 16#0384#,  --  ΄ (GREEK TONOS)
      118    => 16#0385#,  --  ΅ (GREEK DIALYTIKA TONOS)
      127    => 16#00A1#,  --  ¡ (INVERTED EXCLAMATION MARK)
      128    => 16#00A6#,  --  ¦ (BROKEN BAR)
      129    => 16#00BF#,  --  ¿ (INVERTED QUESTION MARK)
      168    => 16#00BA#,  --  º (MASCULINE ORDINAL INDICATOR)
      169    => 16#00AA#,  --  ª (FEMININE ORDINAL INDICATOR)
      170    => 16#00A9#,  --  © (COPYRIGHT SIGN)
      171    => 16#00AE#,  --  ® (REGISTERED SIGN)
      172    => 16#2122#,  --  ™ (TRADE MARK SIGN)
      173    => 16#00A4#,  --  ¤ (CURRENCY SIGN)
      174    => 16#2116#,  --  № (NUMERO SIGN)
      534    => 16#0386#,  --  Ά (GREEK CAPITAL LETTER ALPHA WITH TONOS)
      535    => 16#0388#,  --  Έ (GREEK CAPITAL LETTER EPSILON WITH TONOS)
      536    => 16#0389#,  --  Ή (GREEK CAPITAL LETTER ETA WITH TONOS)
      537    => 16#038A#,  --  Ί (GREEK CAPITAL LETTER IOTA WITH TONOS)
      538    => 16#03AA#,  --  Ϊ (GREEK CAPITAL LETTER IOTA WITH DIALYTIKA)
      540    => 16#038C#,  --  Ό (GREEK CAPITAL LETTER OMICRON WITH TONOS)
      542    => 16#038E#,  --  Ύ (GREEK CAPITAL LETTER UPSILON WITH TONOS)
      543    => 16#03AB#,  --  Ϋ (GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA)
      545    => 16#038F#,  --  Ώ (GREEK CAPITAL LETTER OMEGA WITH TONOS)
      550    => 16#03AC#,  --  ά (GREEK SMALL LETTER ALPHA WITH TONOS)
      551    => 16#03AD#,  --  έ (GREEK SMALL LETTER EPSILON WITH TONOS)
      552    => 16#03AE#,  --  ή (GREEK SMALL LETTER ETA WITH TONOS)
      553    => 16#03AF#,  --  ί (GREEK SMALL LETTER IOTA WITH TONOS)
      554    => 16#03CA#,  --  ϊ (GREEK SMALL LETTER IOTA WITH DIALYTIKA)
      555    => 16#0390#,  --  ΐ (GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS)
      556    => 16#03CC#,  --  ό (GREEK SMALL LETTER OMICRON WITH TONOS)
      557    => 16#03C2#,  --  ς (GREEK SMALL LETTER FINAL SIGMA)
      558    => 16#03CD#,  --  ύ (GREEK SMALL LETTER UPSILON WITH TONOS)
      559    => 16#03CB#,  --  ϋ (GREEK SMALL LETTER UPSILON WITH DIALYTIKA)
      560    => 16#03B0#,  --  ΰ (GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS)
      561    => 16#03CE#,  --  ώ (GREEK SMALL LETTER OMEGA WITH TONOS)
      597    => 16#0402#,  --  Ђ (CYRILLIC CAPITAL LETTER DJE)
      598    => 16#0403#,  --  Ѓ (CYRILLIC CAPITAL LETTER GJE)
      599    => 16#0404#,  --  Є (CYRILLIC CAPITAL LETTER UKRAINIAN IE)
      600    => 16#0405#,  --  Ѕ (CYRILLIC CAPITAL LETTER DZE)
      601    => 16#0406#,  --  І (CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I)
      602    => 16#0407#,  --  Ї (CYRILLIC CAPITAL LETTER YI)
      603    => 16#0408#,  --  Ј (CYRILLIC CAPITAL LETTER JE)
      604    => 16#0409#,  --  Љ (CYRILLIC CAPITAL LETTER LJE)
      605    => 16#040A#,  --  Њ (CYRILLIC CAPITAL LETTER NJE)
      606    => 16#040B#,  --  Ћ (CYRILLIC CAPITAL LETTER TSHE)
      607    => 16#040C#,  --  Ќ (CYRILLIC CAPITAL LETTER KJE)
      608    => 16#040E#,  --  Ў (CYRILLIC CAPITAL LETTER SHORT U)
      609    => 16#040F#,  --  Џ (CYRILLIC CAPITAL LETTER DZHE)
      645    => 16#0452#,  --  ђ (CYRILLIC SMALL LETTER DJE)
      646    => 16#0453#,  --  ѓ (CYRILLIC SMALL LETTER GJE)
      647    => 16#0454#,  --  є (CYRILLIC SMALL LETTER UKRAINIAN IE)
      648    => 16#0455#,  --  ѕ (CYRILLIC SMALL LETTER DZE)
      649    => 16#0456#,  --  і (CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I)
      650    => 16#0457#,  --  ї (CYRILLIC SMALL LETTER YI)
      651    => 16#0458#,  --  ј (CYRILLIC SMALL LETTER JE)
      652    => 16#0459#,  --  љ (CYRILLIC SMALL LETTER LJE)
      653    => 16#045A#,  --  њ (CYRILLIC SMALL LETTER NJE)
      654    => 16#045B#,  --  ћ (CYRILLIC SMALL LETTER TSHE)
      655    => 16#045C#,  --  ќ (CYRILLIC SMALL LETTER KJE)
      656    => 16#045E#,  --  ў (CYRILLIC SMALL LETTER SHORT U)
      657    => 16#045F#,  --  џ (CYRILLIC SMALL LETTER DZHE)
      752    => 16#00C6#,  --  Æ (LATIN CAPITAL LETTER AE)
      753    => 16#0110#,  --  Đ (LATIN CAPITAL LETTER D WITH STROKE)
      755    => 16#0126#,  --  Ħ (LATIN CAPITAL LETTER H WITH STROKE)
      757    => 16#0132#,  --  Ĳ (LATIN CAPITAL LIGATURE IJ)
      759    => 16#0141#,  --  Ł (LATIN CAPITAL LETTER L WITH STROKE)
      760    => 16#013F#,  --  Ŀ (LATIN CAPITAL LETTER L WITH MIDDLE DOT)
      762    => 16#014A#,  --  Ŋ (LATIN CAPITAL LETTER ENG)
      763    => 16#00D8#,  --  Ø (LATIN CAPITAL LETTER O WITH STROKE)
      764    => 16#0152#,  --  Œ (LATIN CAPITAL LIGATURE OE)
      766    => 16#0166#,  --  Ŧ (LATIN CAPITAL LETTER T WITH STROKE)
      767    => 16#00DE#,  --  Þ (LATIN CAPITAL LETTER THORN)
      784    => 16#00E6#,  --  æ (LATIN SMALL LETTER AE)
      785    => 16#0111#,  --  đ (LATIN SMALL LETTER D WITH STROKE)
      786    => 16#00F0#,  --  ð (LATIN SMALL LETTER ETH)
      787    => 16#0127#,  --  ħ (LATIN SMALL LETTER H WITH STROKE)
      788    => 16#0131#,  --  ı (LATIN SMALL LETTER DOTLESS I)
      789    => 16#0133#,  --  ĳ (LATIN SMALL LIGATURE IJ)
      790    => 16#0138#,  --  ĸ (LATIN SMALL LETTER KRA)
      791    => 16#0142#,  --  ł (LATIN SMALL LETTER L WITH STROKE)
      792    => 16#0140#,  --  ŀ (LATIN SMALL LETTER L WITH MIDDLE DOT)
      793    => 16#0149#,  --  ŉ (LATIN SMALL LETTER N PRECEDED BY APOSTROPHE)
      794    => 16#014B#,  --  ŋ (LATIN SMALL LETTER ENG)
      795    => 16#00F8#,  --  ø (LATIN SMALL LETTER O WITH STROKE)
      796    => 16#0153#,  --  œ (LATIN SMALL LIGATURE OE)
      797    => 16#00DF#,  --  ß (LATIN SMALL LETTER SHARP S)
      798    => 16#0167#,  --  ŧ (LATIN SMALL LETTER T WITH STROKE)
      799    => 16#00FE#,  --  þ (LATIN SMALL LETTER THORN)
      846    => 16#00C1#,  --  Á (LATIN CAPITAL LETTER A WITH ACUTE)
      847    => 16#00C0#,  --  À (LATIN CAPITAL LETTER A WITH GRAVE)
      848    => 16#00C4#,  --  Ä (LATIN CAPITAL LETTER A WITH DIAERESIS)
      849    => 16#00C2#,  --  Â (LATIN CAPITAL LETTER A WITH CIRCUMFLEX)
      850    => 16#0102#,  --  Ă (LATIN CAPITAL LETTER A WITH BREVE)
      851    => 16#01CD#,  --  Ǎ (LATIN CAPITAL LETTER A WITH CARON)
      852    => 16#0100#,  --  Ā (LATIN CAPITAL LETTER A WITH MACRON)
      853    => 16#0104#,  --  Ą (LATIN CAPITAL LETTER A WITH OGONEK)
      854    => 16#00C5#,  --  Å (LATIN CAPITAL LETTER A WITH RING ABOVE)
      855    => 16#00C3#,  --  Ã (LATIN CAPITAL LETTER A WITH TILDE)
      856    => 16#0106#,  --  Ć (LATIN CAPITAL LETTER C WITH ACUTE)
      857    => 16#0108#,  --  Ĉ (LATIN CAPITAL LETTER C WITH CIRCUMFLEX)
      858    => 16#010C#,  --  Č (LATIN CAPITAL LETTER C WITH CARON)
      859    => 16#00C7#,  --  Ç (LATIN CAPITAL LETTER C WITH CEDILLA)
      860    => 16#010A#,  --  Ċ (LATIN CAPITAL LETTER C WITH DOT ABOVE)
      861    => 16#010E#,  --  Ď (LATIN CAPITAL LETTER D WITH CARON)
      862    => 16#00C9#,  --  É (LATIN CAPITAL LETTER E WITH ACUTE)
      863    => 16#00C8#,  --  È (LATIN CAPITAL LETTER E WITH GRAVE)
      864    => 16#00CB#,  --  Ë (LATIN CAPITAL LETTER E WITH DIAERESIS)
      865    => 16#00CA#,  --  Ê (LATIN CAPITAL LETTER E WITH CIRCUMFLEX)
      866    => 16#011A#,  --  Ě (LATIN CAPITAL LETTER E WITH CARON)
      867    => 16#0116#,  --  Ė (LATIN CAPITAL LETTER E WITH DOT ABOVE)
      868    => 16#0112#,  --  Ē (LATIN CAPITAL LETTER E WITH MACRON)
      869    => 16#0118#,  --  Ę (LATIN CAPITAL LETTER E WITH OGONEK)
      871    => 16#011C#,  --  Ĝ (LATIN CAPITAL LETTER G WITH CIRCUMFLEX)
      872    => 16#011E#,  --  Ğ (LATIN CAPITAL LETTER G WITH BREVE)
      873    => 16#0122#,  --  Ģ (LATIN CAPITAL LETTER G WITH CEDILLA)
      874    => 16#0120#,  --  Ġ (LATIN CAPITAL LETTER G WITH DOT ABOVE)
      875    => 16#0124#,  --  Ĥ (LATIN CAPITAL LETTER H WITH CIRCUMFLEX)
      876    => 16#00CD#,  --  Í (LATIN CAPITAL LETTER I WITH ACUTE)
      877    => 16#00CC#,  --  Ì (LATIN CAPITAL LETTER I WITH GRAVE)
      878    => 16#00CF#,  --  Ï (LATIN CAPITAL LETTER I WITH DIAERESIS)
      879    => 16#00CE#,  --  Î (LATIN CAPITAL LETTER I WITH CIRCUMFLEX)
      880    => 16#01CF#,  --  Ǐ (LATIN CAPITAL LETTER I WITH CARON)
      881    => 16#0130#,  --  İ (LATIN CAPITAL LETTER I WITH DOT ABOVE)
      882    => 16#012A#,  --  Ī (LATIN CAPITAL LETTER I WITH MACRON)
      883    => 16#012E#,  --  Į (LATIN CAPITAL LETTER I WITH OGONEK)
      884    => 16#0128#,  --  Ĩ (LATIN CAPITAL LETTER I WITH TILDE)
      885    => 16#0134#,  --  Ĵ (LATIN CAPITAL LETTER J WITH CIRCUMFLEX)
      886    => 16#0136#,  --  Ķ (LATIN CAPITAL LETTER K WITH CEDILLA)
      887    => 16#0139#,  --  Ĺ (LATIN CAPITAL LETTER L WITH ACUTE)
      888    => 16#013D#,  --  Ľ (LATIN CAPITAL LETTER L WITH CARON)
      889    => 16#013B#,  --  Ļ (LATIN CAPITAL LETTER L WITH CEDILLA)
      890    => 16#0143#,  --  Ń (LATIN CAPITAL LETTER N WITH ACUTE)
      891    => 16#0147#,  --  Ň (LATIN CAPITAL LETTER N WITH CARON)
      892    => 16#0145#,  --  Ņ (LATIN CAPITAL LETTER N WITH CEDILLA)
      893    => 16#00D1#,  --  Ñ (LATIN CAPITAL LETTER N WITH TILDE)
      894    => 16#00D3#,  --  Ó (LATIN CAPITAL LETTER O WITH ACUTE)
      895    => 16#00D2#,  --  Ò (LATIN CAPITAL LETTER O WITH GRAVE)
      896    => 16#00D6#,  --  Ö (LATIN CAPITAL LETTER O WITH DIAERESIS)
      897    => 16#00D4#,  --  Ô (LATIN CAPITAL LETTER O WITH CIRCUMFLEX)
      898    => 16#01D1#,  --  Ǒ (LATIN CAPITAL LETTER O WITH CARON)
      899    => 16#0150#,  --  Ő (LATIN CAPITAL LETTER O WITH DOUBLE ACUTE)
      900    => 16#014C#,  --  Ō (LATIN CAPITAL LETTER O WITH MACRON)
      901    => 16#00D5#,  --  Õ (LATIN CAPITAL LETTER O WITH TILDE)
      902    => 16#0154#,  --  Ŕ (LATIN CAPITAL LETTER R WITH ACUTE)
      903    => 16#0158#,  --  Ř (LATIN CAPITAL LETTER R WITH CARON)
      904    => 16#0156#,  --  Ŗ (LATIN CAPITAL LETTER R WITH CEDILLA)
      905    => 16#015A#,  --  Ś (LATIN CAPITAL LETTER S WITH ACUTE)
      906    => 16#015C#,  --  Ŝ (LATIN CAPITAL LETTER S WITH CIRCUMFLEX)
      907    => 16#0160#,  --  Š (LATIN CAPITAL LETTER S WITH CARON)
      908    => 16#015E#,  --  Ş (LATIN CAPITAL LETTER S WITH CEDILLA)
      909    => 16#0164#,  --  Ť (LATIN CAPITAL LETTER T WITH CARON)
      910    => 16#0162#,  --  Ţ (LATIN CAPITAL LETTER T WITH CEDILLA)
      911    => 16#00DA#,  --  Ú (LATIN CAPITAL LETTER U WITH ACUTE)
      912    => 16#00D9#,  --  Ù (LATIN CAPITAL LETTER U WITH GRAVE)
      913    => 16#00DC#,  --  Ü (LATIN CAPITAL LETTER U WITH DIAERESIS)
      914    => 16#00DB#,  --  Û (LATIN CAPITAL LETTER U WITH CIRCUMFLEX)
      915    => 16#016C#,  --  Ŭ (LATIN CAPITAL LETTER U WITH BREVE)
      916    => 16#01D3#,  --  Ǔ (LATIN CAPITAL LETTER U WITH CARON)
      917    => 16#0170#,  --  Ű (LATIN CAPITAL LETTER U WITH DOUBLE ACUTE)
      918    => 16#016A#,  --  Ū (LATIN CAPITAL LETTER U WITH MACRON)
      919    => 16#0172#,  --  Ų (LATIN CAPITAL LETTER U WITH OGONEK)
      920    => 16#016E#,  --  Ů (LATIN CAPITAL LETTER U WITH RING ABOVE)
      921    => 16#0168#,  --  Ũ (LATIN CAPITAL LETTER U WITH TILDE)
      922    => 16#01D7#,  --  Ǘ (LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE)
      923    => 16#01DB#,  --  Ǜ (LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE)
      924    => 16#01D9#,  --  Ǚ (LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON)
      925    => 16#01D5#,  --  Ǖ (LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON)
      926    => 16#0174#,  --  Ŵ (LATIN CAPITAL LETTER W WITH CIRCUMFLEX)
      927    => 16#00DD#,  --  Ý (LATIN CAPITAL LETTER Y WITH ACUTE)
      928    => 16#0178#,  --  Ÿ (LATIN CAPITAL LETTER Y WITH DIAERESIS)
      929    => 16#0176#,  --  Ŷ (LATIN CAPITAL LETTER Y WITH CIRCUMFLEX)
      930    => 16#0179#,  --  Ź (LATIN CAPITAL LETTER Z WITH ACUTE)
      931    => 16#017D#,  --  Ž (LATIN CAPITAL LETTER Z WITH CARON)
      932    => 16#017B#,  --  Ż (LATIN CAPITAL LETTER Z WITH DOT ABOVE)
      940    => 16#00E1#,  --  á (LATIN SMALL LETTER A WITH ACUTE)
      941    => 16#00E0#,  --  à (LATIN SMALL LETTER A WITH GRAVE)
      942    => 16#00E4#,  --  ä (LATIN SMALL LETTER A WITH DIAERESIS)
      943    => 16#00E2#,  --  â (LATIN SMALL LETTER A WITH CIRCUMFLEX)
      944    => 16#0103#,  --  ă (LATIN SMALL LETTER A WITH BREVE)
      945    => 16#01CE#,  --  ǎ (LATIN SMALL LETTER A WITH CARON)
      946    => 16#0101#,  --  ā (LATIN SMALL LETTER A WITH MACRON)
      947    => 16#0105#,  --  ą (LATIN SMALL LETTER A WITH OGONEK)
      948    => 16#00E5#,  --  å (LATIN SMALL LETTER A WITH RING ABOVE)
      949    => 16#00E3#,  --  ã (LATIN SMALL LETTER A WITH TILDE)
      950    => 16#0107#,  --  ć (LATIN SMALL LETTER C WITH ACUTE)
      951    => 16#0109#,  --  ĉ (LATIN SMALL LETTER C WITH CIRCUMFLEX)
      952    => 16#010D#,  --  č (LATIN SMALL LETTER C WITH CARON)
      953    => 16#00E7#,  --  ç (LATIN SMALL LETTER C WITH CEDILLA)
      954    => 16#010B#,  --  ċ (LATIN SMALL LETTER C WITH DOT ABOVE)
      955    => 16#010F#,  --  ď (LATIN SMALL LETTER D WITH CARON)
      956    => 16#00E9#,  --  é (LATIN SMALL LETTER E WITH ACUTE)
      957    => 16#00E8#,  --  è (LATIN SMALL LETTER E WITH GRAVE)
      958    => 16#00EB#,  --  ë (LATIN SMALL LETTER E WITH DIAERESIS)
      959    => 16#00EA#,  --  ê (LATIN SMALL LETTER E WITH CIRCUMFLEX)
      960    => 16#011B#,  --  ě (LATIN SMALL LETTER E WITH CARON)
      961    => 16#0117#,  --  ė (LATIN SMALL LETTER E WITH DOT ABOVE)
      962    => 16#0113#,  --  ē (LATIN SMALL LETTER E WITH MACRON)
      963    => 16#0119#,  --  ę (LATIN SMALL LETTER E WITH OGONEK)
      964    => 16#01F5#,  --  ǵ (LATIN SMALL LETTER G WITH ACUTE)
      965    => 16#011D#,  --  ĝ (LATIN SMALL LETTER G WITH CIRCUMFLEX)
      966    => 16#011F#,  --  ğ (LATIN SMALL LETTER G WITH BREVE)
      968    => 16#0121#,  --  ġ (LATIN SMALL LETTER G WITH DOT ABOVE)
      969    => 16#0125#,  --  ĥ (LATIN SMALL LETTER H WITH CIRCUMFLEX)
      970    => 16#00ED#,  --  í (LATIN SMALL LETTER I WITH ACUTE)
      971    => 16#00EC#,  --  ì (LATIN SMALL LETTER I WITH GRAVE)
      972    => 16#00EF#,  --  ï (LATIN SMALL LETTER I WITH DIAERESIS)
      973    => 16#00EE#,  --  î (LATIN SMALL LETTER I WITH CIRCUMFLEX)
      974    => 16#01D0#,  --  ǐ (LATIN SMALL LETTER I WITH CARON)
      976    => 16#012B#,  --  ī (LATIN SMALL LETTER I WITH MACRON)
      977    => 16#012F#,  --  į (LATIN SMALL LETTER I WITH OGONEK)
      978    => 16#0129#,  --  ĩ (LATIN SMALL LETTER I WITH TILDE)
      979    => 16#0135#,  --  ĵ (LATIN SMALL LETTER J WITH CIRCUMFLEX)
      980    => 16#0137#,  --  ķ (LATIN SMALL LETTER K WITH CEDILLA)
      981    => 16#013A#,  --  ĺ (LATIN SMALL LETTER L WITH ACUTE)
      982    => 16#013E#,  --  ľ (LATIN SMALL LETTER L WITH CARON)
      983    => 16#013C#,  --  ļ (LATIN SMALL LETTER L WITH CEDILLA)
      984    => 16#0144#,  --  ń (LATIN SMALL LETTER N WITH ACUTE)
      985    => 16#0148#,  --  ň (LATIN SMALL LETTER N WITH CARON)
      986    => 16#0146#,  --  ņ (LATIN SMALL LETTER N WITH CEDILLA)
      987    => 16#00F1#,  --  ñ (LATIN SMALL LETTER N WITH TILDE)
      988    => 16#00F3#,  --  ó (LATIN SMALL LETTER O WITH ACUTE)
      989    => 16#00F2#,  --  ò (LATIN SMALL LETTER O WITH GRAVE)
      990    => 16#00F6#,  --  ö (LATIN SMALL LETTER O WITH DIAERESIS)
      991    => 16#00F4#,  --  ô (LATIN SMALL LETTER O WITH CIRCUMFLEX)
      992    => 16#01D2#,  --  ǒ (LATIN SMALL LETTER O WITH CARON)
      993    => 16#0151#,  --  ő (LATIN SMALL LETTER O WITH DOUBLE ACUTE)
      994    => 16#014D#,  --  ō (LATIN SMALL LETTER O WITH MACRON)
      995    => 16#00F5#,  --  õ (LATIN SMALL LETTER O WITH TILDE)
      996    => 16#0155#,  --  ŕ (LATIN SMALL LETTER R WITH ACUTE)
      997    => 16#0159#,  --  ř (LATIN SMALL LETTER R WITH CARON)
      998    => 16#0157#,  --  ŗ (LATIN SMALL LETTER R WITH CEDILLA)
      999    => 16#015B#,  --  ś (LATIN SMALL LETTER S WITH ACUTE)
      1000   => 16#015D#,  --  ŝ (LATIN SMALL LETTER S WITH CIRCUMFLEX)
      1001   => 16#0161#,  --  š (LATIN SMALL LETTER S WITH CARON)
      1002   => 16#015F#,  --  ş (LATIN SMALL LETTER S WITH CEDILLA)
      1003   => 16#0165#,  --  ť (LATIN SMALL LETTER T WITH CARON)
      1004   => 16#0163#,  --  ţ (LATIN SMALL LETTER T WITH CEDILLA)
      1005   => 16#00FA#,  --  ú (LATIN SMALL LETTER U WITH ACUTE)
      1006   => 16#00F9#,  --  ù (LATIN SMALL LETTER U WITH GRAVE)
      1007   => 16#00FC#,  --  ü (LATIN SMALL LETTER U WITH DIAERESIS)
      1008   => 16#00FB#,  --  û (LATIN SMALL LETTER U WITH CIRCUMFLEX)
      1009   => 16#016D#,  --  ŭ (LATIN SMALL LETTER U WITH BREVE)
      1010   => 16#01D4#,  --  ǔ (LATIN SMALL LETTER U WITH CARON)
      1011   => 16#0171#,  --  ű (LATIN SMALL LETTER U WITH DOUBLE ACUTE)
      1012   => 16#016B#,  --  ū (LATIN SMALL LETTER U WITH MACRON)
      1013   => 16#0173#,  --  ų (LATIN SMALL LETTER U WITH OGONEK)
      1014   => 16#016F#,  --  ů (LATIN SMALL LETTER U WITH RING ABOVE)
      1015   => 16#0169#,  --  ũ (LATIN SMALL LETTER U WITH TILDE)
      1016   => 16#01D8#,  --  ǘ (LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE)
      1017   => 16#01DC#,  --  ǜ (LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE)
      1018   => 16#01DA#,  --  ǚ (LATIN SMALL LETTER U WITH DIAERESIS AND CARON)
      1019   => 16#01D6#,  --  ǖ (LATIN SMALL LETTER U WITH DIAERESIS AND MACRON)
      1020   => 16#0175#,  --  ŵ (LATIN SMALL LETTER W WITH CIRCUMFLEX)
      1021   => 16#00FD#,  --  ý (LATIN SMALL LETTER Y WITH ACUTE)
      1022   => 16#00FF#,  --  ÿ (LATIN SMALL LETTER Y WITH DIAERESIS)
      1023   => 16#0177#,  --  ŷ (LATIN SMALL LETTER Y WITH CIRCUMFLEX)
      1024   => 16#017A#,  --  ź (LATIN SMALL LETTER Z WITH ACUTE)
      1025   => 16#017E#,  --  ž (LATIN SMALL LETTER Z WITH CARON)
      1026   => 16#017C#,  --  ż (LATIN SMALL LETTER Z WITH DOT ABOVE)
      1410   => 16#4E02#,  --  丂 (<CJK Ideograph>)
      1411   => 16#4E04#,  --  丄 (<CJK Ideograph>)
      1412   => 16#4E05#,  --  丅 (<CJK Ideograph>)
      1413   => 16#4E0C#,  --  丌 (<CJK Ideograph>)
      1414   => 16#4E12#,  --  丒 (<CJK Ideograph>)
      1415   => 16#4E1F#,  --  丟 (<CJK Ideograph>)
      1416   => 16#4E23#,  --  丣 (<CJK Ideograph>)
      1417   => 16#4E24#,  --  两 (<CJK Ideograph>)
      1418   => 16#4E28#,  --  丨 (<CJK Ideograph>)
      1419   => 16#4E2B#,  --  丫 (<CJK Ideograph>)
      1420   => 16#4E2E#,  --  丮 (<CJK Ideograph>)
      1421   => 16#4E2F#,  --  丯 (<CJK Ideograph>)
      1422   => 16#4E30#,  --  丰 (<CJK Ideograph>)
      1423   => 16#4E35#,  --  丵 (<CJK Ideograph>)
      1424   => 16#4E40#,  --  乀 (<CJK Ideograph>)
      1425   => 16#4E41#,  --  乁 (<CJK Ideograph>)
      1426   => 16#4E44#,  --  乄 (<CJK Ideograph>)
      1427   => 16#4E47#,  --  乇 (<CJK Ideograph>)
      1428   => 16#4E51#,  --  乑 (<CJK Ideograph>)
      1429   => 16#4E5A#,  --  乚 (<CJK Ideograph>)
      1430   => 16#4E5C#,  --  乜 (<CJK Ideograph>)
      1431   => 16#4E63#,  --  乣 (<CJK Ideograph>)
      1432   => 16#4E68#,  --  乨 (<CJK Ideograph>)
      1433   => 16#4E69#,  --  乩 (<CJK Ideograph>)
      1434   => 16#4E74#,  --  乴 (<CJK Ideograph>)
      1435   => 16#4E75#,  --  乵 (<CJK Ideograph>)
      1436   => 16#4E79#,  --  乹 (<CJK Ideograph>)
      1437   => 16#4E7F#,  --  乿 (<CJK Ideograph>)
      1438   => 16#4E8D#,  --  亍 (<CJK Ideograph>)
      1439   => 16#4E96#,  --  亖 (<CJK Ideograph>)
      1440   => 16#4E97#,  --  亗 (<CJK Ideograph>)
      1441   => 16#4E9D#,  --  亝 (<CJK Ideograph>)
      1442   => 16#4EAF#,  --  亯 (<CJK Ideograph>)
      1443   => 16#4EB9#,  --  亹 (<CJK Ideograph>)
      1444   => 16#4EC3#,  --  仃 (<CJK Ideograph>)
      1445   => 16#4ED0#,  --  仐 (<CJK Ideograph>)
      1446   => 16#4EDA#,  --  仚 (<CJK Ideograph>)
      1447   => 16#4EDB#,  --  仛 (<CJK Ideograph>)
      1448   => 16#4EE0#,  --  仠 (<CJK Ideograph>)
      1449   => 16#4EE1#,  --  仡 (<CJK Ideograph>)
      1450   => 16#4EE2#,  --  仢 (<CJK Ideograph>)
      1451   => 16#4EE8#,  --  仨 (<CJK Ideograph>)
      1452   => 16#4EEF#,  --  仯 (<CJK Ideograph>)
      1453   => 16#4EF1#,  --  仱 (<CJK Ideograph>)
      1454   => 16#4EF3#,  --  仳 (<CJK Ideograph>)
      1455   => 16#4EF5#,  --  仵 (<CJK Ideograph>)
      1456   => 16#4EFD#,  --  份 (<CJK Ideograph>)
      1457   => 16#4EFE#,  --  仾 (<CJK Ideograph>)
      1458   => 16#4EFF#,  --  仿 (<CJK Ideograph>)
      1459   => 16#4F00#,  --  伀 (<CJK Ideograph>)
      1460   => 16#4F02#,  --  伂 (<CJK Ideograph>)
      1461   => 16#4F03#,  --  伃 (<CJK Ideograph>)
      1462   => 16#4F08#,  --  伈 (<CJK Ideograph>)
      1463   => 16#4F0B#,  --  伋 (<CJK Ideograph>)
      1464   => 16#4F0C#,  --  伌 (<CJK Ideograph>)
      1465   => 16#4F12#,  --  伒 (<CJK Ideograph>)
      1466   => 16#4F15#,  --  伕 (<CJK Ideograph>)
      1467   => 16#4F16#,  --  伖 (<CJK Ideograph>)
      1468   => 16#4F17#,  --  众 (<CJK Ideograph>)
      1469   => 16#4F19#,  --  伙 (<CJK Ideograph>)
      1470   => 16#4F2E#,  --  伮 (<CJK Ideograph>)
      1471   => 16#4F31#,  --  伱 (<CJK Ideograph>)
      1472   => 16#4F60#,  --  你 (<CJK Ideograph>)
      1473   => 16#4F33#,  --  伳 (<CJK Ideograph>)
      1474   => 16#4F35#,  --  伵 (<CJK Ideograph>)
      1475   => 16#4F37#,  --  伷 (<CJK Ideograph>)
      1476   => 16#4F39#,  --  伹 (<CJK Ideograph>)
      1477   => 16#4F3B#,  --  伻 (<CJK Ideograph>)
      1478   => 16#4F3E#,  --  伾 (<CJK Ideograph>)
      1479   => 16#4F40#,  --  佀 (<CJK Ideograph>)
      1480   => 16#4F42#,  --  佂 (<CJK Ideograph>)
      1481   => 16#4F48#,  --  佈 (<CJK Ideograph>)
      1482   => 16#4F49#,  --  佉 (<CJK Ideograph>)
      1483   => 16#4F4B#,  --  佋 (<CJK Ideograph>)
      1484   => 16#4F4C#,  --  佌 (<CJK Ideograph>)
      1485   => 16#4F52#,  --  佒 (<CJK Ideograph>)
      1486   => 16#4F54#,  --  佔 (<CJK Ideograph>)
      1487   => 16#4F56#,  --  佖 (<CJK Ideograph>)
      1488   => 16#4F58#,  --  佘 (<CJK Ideograph>)
      1489   => 16#4F5F#,  --  佟 (<CJK Ideograph>)
      1490   => 16#4F63#,  --  佣 (<CJK Ideograph>)
      1491   => 16#4F6A#,  --  佪 (<CJK Ideograph>)
      1492   => 16#4F6C#,  --  佬 (<CJK Ideograph>)
      1493   => 16#4F6E#,  --  佮 (<CJK Ideograph>)
      1494   => 16#4F71#,  --  佱 (<CJK Ideograph>)
      1495   => 16#4F77#,  --  佷 (<CJK Ideograph>)
      1496   => 16#4F78#,  --  佸 (<CJK Ideograph>)
      1497   => 16#4F79#,  --  佹 (<CJK Ideograph>)
      1498   => 16#4F7A#,  --  佺 (<CJK Ideograph>)
      1499   => 16#4F7D#,  --  佽 (<CJK Ideograph>)
      1500   => 16#4F7E#,  --  佾 (<CJK Ideograph>)
      1501   => 16#4F81#,  --  侁 (<CJK Ideograph>)
      1502   => 16#4F82#,  --  侂 (<CJK Ideograph>)
      1503   => 16#4F84#,  --  侄 (<CJK Ideograph>)
      1504   => 16#4F85#,  --  侅 (<CJK Ideograph>)
      1505   => 16#4F89#,  --  侉 (<CJK Ideograph>)
      1506   => 16#4F8A#,  --  侊 (<CJK Ideograph>)
      1507   => 16#4F8C#,  --  侌 (<CJK Ideograph>)
      1508   => 16#4F8E#,  --  侎 (<CJK Ideograph>)
      1509   => 16#4F90#,  --  侐 (<CJK Ideograph>)
      1510   => 16#4F92#,  --  侒 (<CJK Ideograph>)
      1511   => 16#4F93#,  --  侓 (<CJK Ideograph>)
      1512   => 16#4F94#,  --  侔 (<CJK Ideograph>)
      1513   => 16#4F97#,  --  侗 (<CJK Ideograph>)
      1514   => 16#4F99#,  --  侙 (<CJK Ideograph>)
      1515   => 16#4F9A#,  --  侚 (<CJK Ideograph>)
      1516   => 16#4F9E#,  --  侞 (<CJK Ideograph>)
      1517   => 16#4F9F#,  --  侟 (<CJK Ideograph>)
      1518   => 16#4FB2#,  --  侲 (<CJK Ideograph>)
      1519   => 16#4FB7#,  --  侷 (<CJK Ideograph>)
      1520   => 16#4FB9#,  --  侹 (<CJK Ideograph>)
      1521   => 16#4FBB#,  --  侻 (<CJK Ideograph>)
      1522   => 16#4FBC#,  --  侼 (<CJK Ideograph>)
      1523   => 16#4FBD#,  --  侽 (<CJK Ideograph>)
      1524   => 16#4FBE#,  --  侾 (<CJK Ideograph>)
      1525   => 16#4FC0#,  --  俀 (<CJK Ideograph>)
      1526   => 16#4FC1#,  --  俁 (<CJK Ideograph>)
      1527   => 16#4FC5#,  --  俅 (<CJK Ideograph>)
      1528   => 16#4FC6#,  --  俆 (<CJK Ideograph>)
      1529   => 16#4FC8#,  --  俈 (<CJK Ideograph>)
      1530   => 16#4FC9#,  --  俉 (<CJK Ideograph>)
      1531   => 16#4FCB#,  --  俋 (<CJK Ideograph>)
      1532   => 16#4FCC#,  --  俌 (<CJK Ideograph>)
      1533   => 16#4FCD#,  --  俍 (<CJK Ideograph>)
      1534   => 16#4FCF#,  --  俏 (<CJK Ideograph>)
      1535   => 16#4FD2#,  --  俒 (<CJK Ideograph>)
      1536   => 16#4FDC#,  --  俜 (<CJK Ideograph>)
      1537   => 16#4FE0#,  --  俠 (<CJK Ideograph>)
      1538   => 16#4FE2#,  --  俢 (<CJK Ideograph>)
      1539   => 16#4FF0#,  --  俰 (<CJK Ideograph>)
      1540   => 16#4FF2#,  --  俲 (<CJK Ideograph>)
      1541   => 16#4FFC#,  --  俼 (<CJK Ideograph>)
      1542   => 16#4FFD#,  --  俽 (<CJK Ideograph>)
      1543   => 16#4FFF#,  --  俿 (<CJK Ideograph>)
      1544   => 16#5000#,  --  倀 (<CJK Ideograph>)
      1545   => 16#5001#,  --  倁 (<CJK Ideograph>)
      1546   => 16#5004#,  --  倄 (<CJK Ideograph>)
      1547   => 16#5007#,  --  倇 (<CJK Ideograph>)
      1548   => 16#500A#,  --  倊 (<CJK Ideograph>)
      1549   => 16#500C#,  --  倌 (<CJK Ideograph>)
      1550   => 16#500E#,  --  倎 (<CJK Ideograph>)
      1551   => 16#5010#,  --  倐 (<CJK Ideograph>)
      1552   => 16#5013#,  --  倓 (<CJK Ideograph>)
      1553   => 16#5017#,  --  倗 (<CJK Ideograph>)
      1554   => 16#5018#,  --  倘 (<CJK Ideograph>)
      1555   => 16#501B#,  --  倛 (<CJK Ideograph>)
      1556   => 16#501C#,  --  倜 (<CJK Ideograph>)
      1557   => 16#501D#,  --  倝 (<CJK Ideograph>)
      1558   => 16#501E#,  --  倞 (<CJK Ideograph>)
      1559   => 16#5022#,  --  倢 (<CJK Ideograph>)
      1560   => 16#5027#,  --  倧 (<CJK Ideograph>)
      1561   => 16#502E#,  --  倮 (<CJK Ideograph>)
      1562   => 16#5030#,  --  倰 (<CJK Ideograph>)
      1563   => 16#5032#,  --  倲 (<CJK Ideograph>)
      1564   => 16#5033#,  --  倳 (<CJK Ideograph>)
      1565   => 16#5035#,  --  倵 (<CJK Ideograph>)
      1566   => 16#5040#,  --  偀 (<CJK Ideograph>)
      1567   => 16#5041#,  --  偁 (<CJK Ideograph>)
      1568   => 16#5042#,  --  偂 (<CJK Ideograph>)
      1569   => 16#5045#,  --  偅 (<CJK Ideograph>)
      1570   => 16#5046#,  --  偆 (<CJK Ideograph>)
      1571   => 16#504A#,  --  偊 (<CJK Ideograph>)
      1572   => 16#504C#,  --  偌 (<CJK Ideograph>)
      1573   => 16#504E#,  --  偎 (<CJK Ideograph>)
      1574   => 16#5051#,  --  偑 (<CJK Ideograph>)
      1575   => 16#5052#,  --  偒 (<CJK Ideograph>)
      1576   => 16#5053#,  --  偓 (<CJK Ideograph>)
      1577   => 16#5057#,  --  偗 (<CJK Ideograph>)
      1578   => 16#5059#,  --  偙 (<CJK Ideograph>)
      1579   => 16#505F#,  --  偟 (<CJK Ideograph>)
      1580   => 16#5060#,  --  偠 (<CJK Ideograph>)
      1581   => 16#5062#,  --  偢 (<CJK Ideograph>)
      1582   => 16#5063#,  --  偣 (<CJK Ideograph>)
      1583   => 16#5066#,  --  偦 (<CJK Ideograph>)
      1584   => 16#5067#,  --  偧 (<CJK Ideograph>)
      1585   => 16#506A#,  --  偪 (<CJK Ideograph>)
      1586   => 16#506D#,  --  偭 (<CJK Ideograph>)
      1587   => 16#5070#,  --  偰 (<CJK Ideograph>)
      1588   => 16#5071#,  --  偱 (<CJK Ideograph>)
      1589   => 16#503B#,  --  倻 (<CJK Ideograph>)
      1590   => 16#5081#,  --  傁 (<CJK Ideograph>)
      1591   => 16#5083#,  --  傃 (<CJK Ideograph>)
      1592   => 16#5084#,  --  傄 (<CJK Ideograph>)
      1593   => 16#5086#,  --  傆 (<CJK Ideograph>)
      1594   => 16#508A#,  --  傊 (<CJK Ideograph>)
      1595   => 16#508E#,  --  傎 (<CJK Ideograph>)
      1596   => 16#508F#,  --  傏 (<CJK Ideograph>)
      1597   => 16#5090#,  --  傐 (<CJK Ideograph>)
      1598   => 16#5092#,  --  傒 (<CJK Ideograph>)
      1599   => 16#5093#,  --  傓 (<CJK Ideograph>)
      1600   => 16#5094#,  --  傔 (<CJK Ideograph>)
      1601   => 16#5096#,  --  傖 (<CJK Ideograph>)
      1602   => 16#509B#,  --  傛 (<CJK Ideograph>)
      1603   => 16#509C#,  --  傜 (<CJK Ideograph>)
      1604   => 16#509E#,  --  傞 (<CJK Ideograph>)
      1605   => 16#509F#,  --  傟 (<CJK Ideograph>)
      1606   => 16#50A0#,  --  傠 (<CJK Ideograph>)
      1607   => 16#50A1#,  --  傡 (<CJK Ideograph>)
      1608   => 16#50A2#,  --  傢 (<CJK Ideograph>)
      1609   => 16#50AA#,  --  傪 (<CJK Ideograph>)
      1610   => 16#50AF#,  --  傯 (<CJK Ideograph>)
      1611   => 16#50B0#,  --  傰 (<CJK Ideograph>)
      1612   => 16#50B9#,  --  傹 (<CJK Ideograph>)
      1613   => 16#50BA#,  --  傺 (<CJK Ideograph>)
      1614   => 16#50BD#,  --  傽 (<CJK Ideograph>)
      1615   => 16#50C0#,  --  僀 (<CJK Ideograph>)
      1616   => 16#50C3#,  --  僃 (<CJK Ideograph>)
      1617   => 16#50C4#,  --  僄 (<CJK Ideograph>)
      1618   => 16#50C7#,  --  僇 (<CJK Ideograph>)
      1619   => 16#50CC#,  --  僌 (<CJK Ideograph>)
      1620   => 16#50CE#,  --  僎 (<CJK Ideograph>)
      1621   => 16#50D0#,  --  僐 (<CJK Ideograph>)
      1622   => 16#50D3#,  --  僓 (<CJK Ideograph>)
      1623   => 16#50D4#,  --  僔 (<CJK Ideograph>)
      1624   => 16#50D8#,  --  僘 (<CJK Ideograph>)
      1625   => 16#50DC#,  --  僜 (<CJK Ideograph>)
      1626   => 16#50DD#,  --  僝 (<CJK Ideograph>)
      1627   => 16#50DF#,  --  僟 (<CJK Ideograph>)
      1628   => 16#50E2#,  --  僢 (<CJK Ideograph>)
      1629   => 16#50E4#,  --  僤 (<CJK Ideograph>)
      1630   => 16#50E6#,  --  僦 (<CJK Ideograph>)
      1631   => 16#50E8#,  --  僨 (<CJK Ideograph>)
      1632   => 16#50E9#,  --  僩 (<CJK Ideograph>)
      1633   => 16#50EF#,  --  僯 (<CJK Ideograph>)
      1634   => 16#50F1#,  --  僱 (<CJK Ideograph>)
      1635   => 16#50F6#,  --  僶 (<CJK Ideograph>)
      1636   => 16#50FA#,  --  僺 (<CJK Ideograph>)
      1637   => 16#50FE#,  --  僾 (<CJK Ideograph>)
      1638   => 16#5103#,  --  儃 (<CJK Ideograph>)
      1639   => 16#5106#,  --  儆 (<CJK Ideograph>)
      1640   => 16#5107#,  --  儇 (<CJK Ideograph>)
      1641   => 16#5108#,  --  儈 (<CJK Ideograph>)
      1642   => 16#510B#,  --  儋 (<CJK Ideograph>)
      1643   => 16#510C#,  --  儌 (<CJK Ideograph>)
      1644   => 16#510D#,  --  儍 (<CJK Ideograph>)
      1645   => 16#510E#,  --  儎 (<CJK Ideograph>)
      1646   => 16#50F2#,  --  僲 (<CJK Ideograph>)
      1647   => 16#5110#,  --  儐 (<CJK Ideograph>)
      1648   => 16#5117#,  --  儗 (<CJK Ideograph>)
      1649   => 16#5119#,  --  儙 (<CJK Ideograph>)
      1650   => 16#511B#,  --  儛 (<CJK Ideograph>)
      1651   => 16#511C#,  --  儜 (<CJK Ideograph>)
      1652   => 16#511D#,  --  儝 (<CJK Ideograph>)
      1653   => 16#511E#,  --  儞 (<CJK Ideograph>)
      1654   => 16#5123#,  --  儣 (<CJK Ideograph>)
      1655   => 16#5127#,  --  儧 (<CJK Ideograph>)
      1656   => 16#5128#,  --  儨 (<CJK Ideograph>)
      1657   => 16#512C#,  --  儬 (<CJK Ideograph>)
      1658   => 16#512D#,  --  儭 (<CJK Ideograph>)
      1659   => 16#512F#,  --  儯 (<CJK Ideograph>)
      1660   => 16#5131#,  --  儱 (<CJK Ideograph>)
      1661   => 16#5133#,  --  儳 (<CJK Ideograph>)
      1662   => 16#5134#,  --  儴 (<CJK Ideograph>)
      1663   => 16#5135#,  --  儵 (<CJK Ideograph>)
      1664   => 16#5138#,  --  儸 (<CJK Ideograph>)
      1665   => 16#5139#,  --  儹 (<CJK Ideograph>)
      1666   => 16#5142#,  --  兂 (<CJK Ideograph>)
      1667   => 16#514A#,  --  兊 (<CJK Ideograph>)
      1668   => 16#514F#,  --  兏 (<CJK Ideograph>)
      1669   => 16#5153#,  --  兓 (<CJK Ideograph>)
      1670   => 16#5155#,  --  兕 (<CJK Ideograph>)
      1671   => 16#5157#,  --  兗 (<CJK Ideograph>)
      1672   => 16#5158#,  --  兘 (<CJK Ideograph>)
      1673   => 16#515F#,  --  兟 (<CJK Ideograph>)
      1674   => 16#5164#,  --  兤 (<CJK Ideograph>)
      1675   => 16#5166#,  --  兦 (<CJK Ideograph>)
      1676   => 16#517E#,  --  兾 (<CJK Ideograph>)
      1677   => 16#5183#,  --  冃 (<CJK Ideograph>)
      1678   => 16#5184#,  --  冄 (<CJK Ideograph>)
      1679   => 16#518B#,  --  冋 (<CJK Ideograph>)
      1680   => 16#518E#,  --  冎 (<CJK Ideograph>)
      1681   => 16#5198#,  --  冘 (<CJK Ideograph>)
      1682   => 16#519D#,  --  冝 (<CJK Ideograph>)
      1683   => 16#51A1#,  --  冡 (<CJK Ideograph>)
      1684   => 16#51A3#,  --  冣 (<CJK Ideograph>)
      1685   => 16#51AD#,  --  冭 (<CJK Ideograph>)
      1686   => 16#51B8#,  --  冸 (<CJK Ideograph>)
      1687   => 16#51BA#,  --  冺 (<CJK Ideograph>)
      1688   => 16#51BC#,  --  冼 (<CJK Ideograph>)
      1689   => 16#51BE#,  --  冾 (<CJK Ideograph>)
      1690   => 16#51BF#,  --  冿 (<CJK Ideograph>)
      1691   => 16#51C2#,  --  凂 (<CJK Ideograph>)
      1692   => 16#51C8#,  --  凈 (<CJK Ideograph>)
      1693   => 16#51CF#,  --  减 (<CJK Ideograph>)
      1694   => 16#51D1#,  --  凑 (<CJK Ideograph>)
      1695   => 16#51D2#,  --  凒 (<CJK Ideograph>)
      1696   => 16#51D3#,  --  凓 (<CJK Ideograph>)
      1697   => 16#51D5#,  --  凕 (<CJK Ideograph>)
      1698   => 16#51D8#,  --  凘 (<CJK Ideograph>)
      1699   => 16#51DE#,  --  凞 (<CJK Ideograph>)
      1700   => 16#51E2#,  --  凢 (<CJK Ideograph>)
      1701   => 16#51E5#,  --  凥 (<CJK Ideograph>)
      1702   => 16#51EE#,  --  凮 (<CJK Ideograph>)
      1703   => 16#51F2#,  --  凲 (<CJK Ideograph>)
      1704   => 16#51F3#,  --  凳 (<CJK Ideograph>)
      1705   => 16#51F4#,  --  凴 (<CJK Ideograph>)
      1706   => 16#51F7#,  --  凷 (<CJK Ideograph>)
      1707   => 16#5201#,  --  刁 (<CJK Ideograph>)
      1708   => 16#5202#,  --  刂 (<CJK Ideograph>)
      1709   => 16#5205#,  --  刅 (<CJK Ideograph>)
      1710   => 16#5212#,  --  划 (<CJK Ideograph>)
      1711   => 16#5213#,  --  刓 (<CJK Ideograph>)
      1712   => 16#5215#,  --  刕 (<CJK Ideograph>)
      1713   => 16#5216#,  --  刖 (<CJK Ideograph>)
      1714   => 16#5218#,  --  刘 (<CJK Ideograph>)
      1715   => 16#5222#,  --  刢 (<CJK Ideograph>)
      1716   => 16#5228#,  --  刨 (<CJK Ideograph>)
      1717   => 16#5231#,  --  刱 (<CJK Ideograph>)
      1718   => 16#5232#,  --  刲 (<CJK Ideograph>)
      1719   => 16#5235#,  --  刵 (<CJK Ideograph>)
      1720   => 16#523C#,  --  刼 (<CJK Ideograph>)
      1721   => 16#5245#,  --  剅 (<CJK Ideograph>)
      1722   => 16#5249#,  --  剉 (<CJK Ideograph>)
      1723   => 16#5255#,  --  剕 (<CJK Ideograph>)
      1724   => 16#5257#,  --  剗 (<CJK Ideograph>)
      1725   => 16#5258#,  --  剘 (<CJK Ideograph>)
      1726   => 16#525A#,  --  剚 (<CJK Ideograph>)
      1727   => 16#525C#,  --  剜 (<CJK Ideograph>)
      1728   => 16#525F#,  --  剟 (<CJK Ideograph>)
      1729   => 16#5260#,  --  剠 (<CJK Ideograph>)
      1730   => 16#5261#,  --  剡 (<CJK Ideograph>)
      1731   => 16#5266#,  --  剦 (<CJK Ideograph>)
      1732   => 16#526E#,  --  剮 (<CJK Ideograph>)
      1733   => 16#5277#,  --  剷 (<CJK Ideograph>)
      1734   => 16#5278#,  --  剸 (<CJK Ideograph>)
      1735   => 16#5279#,  --  剹 (<CJK Ideograph>)
      1736   => 16#5280#,  --  劀 (<CJK Ideograph>)
      1737   => 16#5282#,  --  劂 (<CJK Ideograph>)
      1738   => 16#5285#,  --  劅 (<CJK Ideograph>)
      1739   => 16#528A#,  --  劊 (<CJK Ideograph>)
      1740   => 16#528C#,  --  劌 (<CJK Ideograph>)
      1741   => 16#5293#,  --  劓 (<CJK Ideograph>)
      1742   => 16#5295#,  --  劕 (<CJK Ideograph>)
      1743   => 16#5296#,  --  劖 (<CJK Ideograph>)
      1744   => 16#5297#,  --  劗 (<CJK Ideograph>)
      1745   => 16#5298#,  --  劘 (<CJK Ideograph>)
      1746   => 16#529A#,  --  劚 (<CJK Ideograph>)
      1747   => 16#529C#,  --  劜 (<CJK Ideograph>)
      1748   => 16#52A4#,  --  劤 (<CJK Ideograph>)
      1749   => 16#52A5#,  --  劥 (<CJK Ideograph>)
      1750   => 16#52A6#,  --  劦 (<CJK Ideograph>)
      1751   => 16#52A7#,  --  劧 (<CJK Ideograph>)
      1752   => 16#52AF#,  --  劯 (<CJK Ideograph>)
      1753   => 16#52B0#,  --  劰 (<CJK Ideograph>)
      1754   => 16#52B6#,  --  劶 (<CJK Ideograph>)
      1755   => 16#52B7#,  --  劷 (<CJK Ideograph>)
      1756   => 16#52B8#,  --  劸 (<CJK Ideograph>)
      1757   => 16#52BA#,  --  劺 (<CJK Ideograph>)
      1758   => 16#52BB#,  --  劻 (<CJK Ideograph>)
      1759   => 16#52BD#,  --  劽 (<CJK Ideograph>)
      1760   => 16#52C0#,  --  勀 (<CJK Ideograph>)
      1761   => 16#52C4#,  --  勄 (<CJK Ideograph>)
      1762   => 16#52C6#,  --  勆 (<CJK Ideograph>)
      1763   => 16#52C8#,  --  勈 (<CJK Ideograph>)
      1764   => 16#52CC#,  --  勌 (<CJK Ideograph>)
      1765   => 16#52CF#,  --  勏 (<CJK Ideograph>)
      1766   => 16#52D1#,  --  勑 (<CJK Ideograph>)
      1767   => 16#52D4#,  --  勔 (<CJK Ideograph>)
      1768   => 16#52D6#,  --  勖 (<CJK Ideograph>)
      1769   => 16#52DB#,  --  勛 (<CJK Ideograph>)
      1770   => 16#52DC#,  --  勜 (<CJK Ideograph>)
      1771   => 16#52E1#,  --  勡 (<CJK Ideograph>)
      1772   => 16#52E5#,  --  勥 (<CJK Ideograph>)
      1773   => 16#52E8#,  --  勨 (<CJK Ideograph>)
      1774   => 16#52E9#,  --  勩 (<CJK Ideograph>)
      1775   => 16#52EA#,  --  勪 (<CJK Ideograph>)
      1776   => 16#52EC#,  --  勬 (<CJK Ideograph>)
      1777   => 16#52F0#,  --  勰 (<CJK Ideograph>)
      1778   => 16#52F1#,  --  勱 (<CJK Ideograph>)
      1779   => 16#52F4#,  --  勴 (<CJK Ideograph>)
      1780   => 16#52F6#,  --  勶 (<CJK Ideograph>)
      1781   => 16#52F7#,  --  勷 (<CJK Ideograph>)
      1782   => 16#5300#,  --  匀 (<CJK Ideograph>)
      1783   => 16#5303#,  --  匃 (<CJK Ideograph>)
      1784   => 16#530A#,  --  匊 (<CJK Ideograph>)
      1785   => 16#530B#,  --  匋 (<CJK Ideograph>)
      1786   => 16#530C#,  --  匌 (<CJK Ideograph>)
      1787   => 16#5311#,  --  匑 (<CJK Ideograph>)
      1788   => 16#5313#,  --  匓 (<CJK Ideograph>)
      1789   => 16#5318#,  --  匘 (<CJK Ideograph>)
      1790   => 16#531B#,  --  匛 (<CJK Ideograph>)
      1791   => 16#531C#,  --  匜 (<CJK Ideograph>)
      1792   => 16#531E#,  --  匞 (<CJK Ideograph>)
      1793   => 16#531F#,  --  匟 (<CJK Ideograph>)
      1794   => 16#5325#,  --  匥 (<CJK Ideograph>)
      1795   => 16#5327#,  --  匧 (<CJK Ideograph>)
      1796   => 16#5328#,  --  匨 (<CJK Ideograph>)
      1797   => 16#5329#,  --  匩 (<CJK Ideograph>)
      1798   => 16#532B#,  --  匫 (<CJK Ideograph>)
      1799   => 16#532C#,  --  匬 (<CJK Ideograph>)
      1800   => 16#532D#,  --  匭 (<CJK Ideograph>)
      1801   => 16#5330#,  --  匰 (<CJK Ideograph>)
      1802   => 16#5332#,  --  匲 (<CJK Ideograph>)
      1803   => 16#5335#,  --  匵 (<CJK Ideograph>)
      1804   => 16#533C#,  --  匼 (<CJK Ideograph>)
      1805   => 16#533D#,  --  匽 (<CJK Ideograph>)
      1806   => 16#533E#,  --  匾 (<CJK Ideograph>)
      1807   => 16#5342#,  --  卂 (<CJK Ideograph>)
      1808   => 16#534C#,  --  卌 (<CJK Ideograph>)
      1809   => 16#534B#,  --  卋 (<CJK Ideograph>)
      1810   => 16#5359#,  --  卙 (<CJK Ideograph>)
      1811   => 16#535B#,  --  卛 (<CJK Ideograph>)
      1812   => 16#5361#,  --  卡 (<CJK Ideograph>)
      1813   => 16#5363#,  --  卣 (<CJK Ideograph>)
      1814   => 16#5365#,  --  卥 (<CJK Ideograph>)
      1815   => 16#536C#,  --  卬 (<CJK Ideograph>)
      1816   => 16#536D#,  --  卭 (<CJK Ideograph>)
      1817   => 16#5372#,  --  卲 (<CJK Ideograph>)
      1818   => 16#5379#,  --  卹 (<CJK Ideograph>)
      1819   => 16#537E#,  --  卾 (<CJK Ideograph>)
      1820   => 16#5383#,  --  厃 (<CJK Ideograph>)
      1821   => 16#5387#,  --  厇 (<CJK Ideograph>)
      1822   => 16#5388#,  --  厈 (<CJK Ideograph>)
      1823   => 16#538E#,  --  厎 (<CJK Ideograph>)
      1824   => 16#5393#,  --  厓 (<CJK Ideograph>)
      1825   => 16#5394#,  --  厔 (<CJK Ideograph>)
      1826   => 16#5399#,  --  厙 (<CJK Ideograph>)
      1827   => 16#539D#,  --  厝 (<CJK Ideograph>)
      1828   => 16#53A1#,  --  厡 (<CJK Ideograph>)
      1829   => 16#53A4#,  --  厤 (<CJK Ideograph>)
      1830   => 16#53AA#,  --  厪 (<CJK Ideograph>)
      1831   => 16#53AB#,  --  厫 (<CJK Ideograph>)
      1832   => 16#53AF#,  --  厯 (<CJK Ideograph>)
      1833   => 16#53B2#,  --  厲 (<CJK Ideograph>)
      1834   => 16#53B4#,  --  厴 (<CJK Ideograph>)
      1835   => 16#53B5#,  --  厵 (<CJK Ideograph>)
      1836   => 16#53B7#,  --  厷 (<CJK Ideograph>)
      1837   => 16#53B8#,  --  厸 (<CJK Ideograph>)
      1838   => 16#53BA#,  --  厺 (<CJK Ideograph>)
      1839   => 16#53BD#,  --  厽 (<CJK Ideograph>)
      1840   => 16#53C0#,  --  叀 (<CJK Ideograph>)
      1841   => 16#53C5#,  --  叅 (<CJK Ideograph>)
      1842   => 16#53CF#,  --  叏 (<CJK Ideograph>)
      1843   => 16#53D2#,  --  叒 (<CJK Ideograph>)
      1844   => 16#53D3#,  --  叓 (<CJK Ideograph>)
      1845   => 16#53D5#,  --  叕 (<CJK Ideograph>)
      1846   => 16#53DA#,  --  叚 (<CJK Ideograph>)
      1847   => 16#53DD#,  --  叝 (<CJK Ideograph>)
      1848   => 16#53DE#,  --  叞 (<CJK Ideograph>)
      1849   => 16#53E0#,  --  叠 (<CJK Ideograph>)
      1850   => 16#53E6#,  --  另 (<CJK Ideograph>)
      1851   => 16#53E7#,  --  叧 (<CJK Ideograph>)
      1852   => 16#53F5#,  --  叵 (<CJK Ideograph>)
      1853   => 16#5402#,  --  吂 (<CJK Ideograph>)
      1854   => 16#5413#,  --  吓 (<CJK Ideograph>)
      1855   => 16#541A#,  --  吚 (<CJK Ideograph>)
      1856   => 16#5421#,  --  吡 (<CJK Ideograph>)
      1857   => 16#5427#,  --  吧 (<CJK Ideograph>)
      1858   => 16#5428#,  --  吨 (<CJK Ideograph>)
      1859   => 16#542A#,  --  吪 (<CJK Ideograph>)
      1860   => 16#542F#,  --  启 (<CJK Ideograph>)
      1861   => 16#5431#,  --  吱 (<CJK Ideograph>)
      1862   => 16#5434#,  --  吴 (<CJK Ideograph>)
      1863   => 16#5435#,  --  吵 (<CJK Ideograph>)
      1864   => 16#5443#,  --  呃 (<CJK Ideograph>)
      1865   => 16#5444#,  --  呄 (<CJK Ideograph>)
      1866   => 16#5447#,  --  呇 (<CJK Ideograph>)
      1867   => 16#544D#,  --  呍 (<CJK Ideograph>)
      1868   => 16#544F#,  --  呏 (<CJK Ideograph>)
      1869   => 16#545E#,  --  呞 (<CJK Ideograph>)
      1870   => 16#5462#,  --  呢 (<CJK Ideograph>)
      1871   => 16#5464#,  --  呤 (<CJK Ideograph>)
      1872   => 16#5466#,  --  呦 (<CJK Ideograph>)
      1873   => 16#5467#,  --  呧 (<CJK Ideograph>)
      1874   => 16#5469#,  --  呩 (<CJK Ideograph>)
      1875   => 16#546B#,  --  呫 (<CJK Ideograph>)
      1876   => 16#546D#,  --  呭 (<CJK Ideograph>)
      1877   => 16#546E#,  --  呮 (<CJK Ideograph>)
      1878   => 16#5474#,  --  呴 (<CJK Ideograph>)
      1879   => 16#547F#,  --  呿 (<CJK Ideograph>)
      1880   => 16#5481#,  --  咁 (<CJK Ideograph>)
      1881   => 16#5483#,  --  咃 (<CJK Ideograph>)
      1882   => 16#5485#,  --  咅 (<CJK Ideograph>)
      1883   => 16#5488#,  --  咈 (<CJK Ideograph>)
      1884   => 16#5489#,  --  咉 (<CJK Ideograph>)
      1885   => 16#548D#,  --  咍 (<CJK Ideograph>)
      1886   => 16#5491#,  --  咑 (<CJK Ideograph>)
      1887   => 16#5495#,  --  咕 (<CJK Ideograph>)
      1888   => 16#5496#,  --  咖 (<CJK Ideograph>)
      1889   => 16#549C#,  --  咜 (<CJK Ideograph>)
      1890   => 16#549F#,  --  咟 (<CJK Ideograph>)
      1891   => 16#54A1#,  --  咡 (<CJK Ideograph>)
      1892   => 16#54A6#,  --  咦 (<CJK Ideograph>)
      1893   => 16#54A7#,  --  咧 (<CJK Ideograph>)
      1894   => 16#54A9#,  --  咩 (<CJK Ideograph>)
      1895   => 16#54AA#,  --  咪 (<CJK Ideograph>)
      1896   => 16#54AD#,  --  咭 (<CJK Ideograph>)
      1897   => 16#54AE#,  --  咮 (<CJK Ideograph>)
      1898   => 16#54B1#,  --  咱 (<CJK Ideograph>)
      1899   => 16#54B7#,  --  咷 (<CJK Ideograph>)
      1900   => 16#54B9#,  --  咹 (<CJK Ideograph>)
      1901   => 16#54BA#,  --  咺 (<CJK Ideograph>)
      1902   => 16#54BB#,  --  咻 (<CJK Ideograph>)
      1903   => 16#54BF#,  --  咿 (<CJK Ideograph>)
      1904   => 16#54C6#,  --  哆 (<CJK Ideograph>)
      1905   => 16#54CA#,  --  哊 (<CJK Ideograph>)
      1906   => 16#54CD#,  --  响 (<CJK Ideograph>)
      1907   => 16#54CE#,  --  哎 (<CJK Ideograph>)
      1908   => 16#54E0#,  --  哠 (<CJK Ideograph>)
      1909   => 16#54EA#,  --  哪 (<CJK Ideograph>)
      1910   => 16#54EC#,  --  哬 (<CJK Ideograph>)
      1911   => 16#54EF#,  --  哯 (<CJK Ideograph>)
      1912   => 16#54F6#,  --  哶 (<CJK Ideograph>)
      1913   => 16#54FC#,  --  哼 (<CJK Ideograph>)
      1914   => 16#54FE#,  --  哾 (<CJK Ideograph>)
      1915   => 16#54FF#,  --  哿 (<CJK Ideograph>)
      1916   => 16#5500#,  --  唀 (<CJK Ideograph>)
      1917   => 16#5501#,  --  唁 (<CJK Ideograph>)
      1918   => 16#5505#,  --  唅 (<CJK Ideograph>)
      1919   => 16#5508#,  --  唈 (<CJK Ideograph>)
      1920   => 16#5509#,  --  唉 (<CJK Ideograph>)
      1921   => 16#550C#,  --  唌 (<CJK Ideograph>)
      1922   => 16#550D#,  --  唍 (<CJK Ideograph>)
      1923   => 16#550E#,  --  唎 (<CJK Ideograph>)
      1924   => 16#5515#,  --  唕 (<CJK Ideograph>)
      1925   => 16#552A#,  --  唪 (<CJK Ideograph>)
      1926   => 16#552B#,  --  唫 (<CJK Ideograph>)
      1927   => 16#5532#,  --  唲 (<CJK Ideograph>)
      1928   => 16#5535#,  --  唵 (<CJK Ideograph>)
      1929   => 16#5536#,  --  唶 (<CJK Ideograph>)
      1930   => 16#553B#,  --  唻 (<CJK Ideograph>)
      1931   => 16#553C#,  --  唼 (<CJK Ideograph>)
      1932   => 16#553D#,  --  唽 (<CJK Ideograph>)
      1933   => 16#5541#,  --  啁 (<CJK Ideograph>)
      1934   => 16#5547#,  --  啇 (<CJK Ideograph>)
      1935   => 16#5549#,  --  啉 (<CJK Ideograph>)
      1936   => 16#554A#,  --  啊 (<CJK Ideograph>)
      1937   => 16#554D#,  --  啍 (<CJK Ideograph>)
      1938   => 16#5550#,  --  啐 (<CJK Ideograph>)
      1939   => 16#5551#,  --  啑 (<CJK Ideograph>)
      1940   => 16#5558#,  --  啘 (<CJK Ideograph>)
      1941   => 16#555A#,  --  啚 (<CJK Ideograph>)
      1942   => 16#555B#,  --  啛 (<CJK Ideograph>)
      1943   => 16#555E#,  --  啞 (<CJK Ideograph>)
      1944   => 16#5560#,  --  啠 (<CJK Ideograph>)
      1945   => 16#5561#,  --  啡 (<CJK Ideograph>)
      1946   => 16#5564#,  --  啤 (<CJK Ideograph>)
      1947   => 16#5566#,  --  啦 (<CJK Ideograph>)
      1948   => 16#557F#,  --  啿 (<CJK Ideograph>)
      1949   => 16#5581#,  --  喁 (<CJK Ideograph>)
      1950   => 16#5582#,  --  喂 (<CJK Ideograph>)
      1951   => 16#5586#,  --  喆 (<CJK Ideograph>)
      1952   => 16#5588#,  --  喈 (<CJK Ideograph>)
      1953   => 16#558E#,  --  喎 (<CJK Ideograph>)
      1954   => 16#558F#,  --  喏 (<CJK Ideograph>)
      1955   => 16#5591#,  --  喑 (<CJK Ideograph>)
      1956   => 16#5592#,  --  喒 (<CJK Ideograph>)
      1957   => 16#5593#,  --  喓 (<CJK Ideograph>)
      1958   => 16#5594#,  --  喔 (<CJK Ideograph>)
      1959   => 16#5597#,  --  喗 (<CJK Ideograph>)
      1960   => 16#55A3#,  --  喣 (<CJK Ideograph>)
      1961   => 16#55A4#,  --  喤 (<CJK Ideograph>)
      1962   => 16#55AD#,  --  喭 (<CJK Ideograph>)
      1963   => 16#55B2#,  --  喲 (<CJK Ideograph>)
      1964   => 16#55BF#,  --  喿 (<CJK Ideograph>)
      1965   => 16#55C1#,  --  嗁 (<CJK Ideograph>)
      1966   => 16#55C3#,  --  嗃 (<CJK Ideograph>)
      1967   => 16#55C6#,  --  嗆 (<CJK Ideograph>)
      1968   => 16#55C9#,  --  嗉 (<CJK Ideograph>)
      1969   => 16#55CB#,  --  嗋 (<CJK Ideograph>)
      1970   => 16#55CC#,  --  嗌 (<CJK Ideograph>)
      1971   => 16#55CE#,  --  嗎 (<CJK Ideograph>)
      1972   => 16#55D1#,  --  嗑 (<CJK Ideograph>)
      1973   => 16#55D2#,  --  嗒 (<CJK Ideograph>)
      1974   => 16#55D3#,  --  嗓 (<CJK Ideograph>)
      1975   => 16#55D7#,  --  嗗 (<CJK Ideograph>)
      1976   => 16#55D8#,  --  嗘 (<CJK Ideograph>)
      1977   => 16#55DB#,  --  嗛 (<CJK Ideograph>)
      1978   => 16#55DE#,  --  嗞 (<CJK Ideograph>)
      1979   => 16#55E2#,  --  嗢 (<CJK Ideograph>)
      1980   => 16#55E9#,  --  嗩 (<CJK Ideograph>)
      1981   => 16#55F6#,  --  嗶 (<CJK Ideograph>)
      1982   => 16#55FF#,  --  嗿 (<CJK Ideograph>)
      1983   => 16#5605#,  --  嘅 (<CJK Ideograph>)
      1984   => 16#5608#,  --  嘈 (<CJK Ideograph>)
      1985   => 16#560A#,  --  嘊 (<CJK Ideograph>)
      1986   => 16#560D#,  --  嘍 (<CJK Ideograph>)
      1987   => 16#560E#,  --  嘎 (<CJK Ideograph>)
      1988   => 16#560F#,  --  嘏 (<CJK Ideograph>)
      1989   => 16#5610#,  --  嘐 (<CJK Ideograph>)
      1990   => 16#5611#,  --  嘑 (<CJK Ideograph>)
      1991   => 16#5612#,  --  嘒 (<CJK Ideograph>)
      1992   => 16#5619#,  --  嘙 (<CJK Ideograph>)
      1993   => 16#562C#,  --  嘬 (<CJK Ideograph>)
      1994   => 16#5630#,  --  嘰 (<CJK Ideograph>)
      1995   => 16#5633#,  --  嘳 (<CJK Ideograph>)
      1996   => 16#5635#,  --  嘵 (<CJK Ideograph>)
      1997   => 16#5637#,  --  嘷 (<CJK Ideograph>)
      1998   => 16#5639#,  --  嘹 (<CJK Ideograph>)
      1999   => 16#563B#,  --  嘻 (<CJK Ideograph>)
      2000   => 16#563C#,  --  嘼 (<CJK Ideograph>)
      2001   => 16#563D#,  --  嘽 (<CJK Ideograph>)
      2002   => 16#563F#,  --  嘿 (<CJK Ideograph>)
      2003   => 16#5640#,  --  噀 (<CJK Ideograph>)
      2004   => 16#5641#,  --  噁 (<CJK Ideograph>)
      2005   => 16#5643#,  --  噃 (<CJK Ideograph>)
      2006   => 16#5644#,  --  噄 (<CJK Ideograph>)
      2007   => 16#5646#,  --  噆 (<CJK Ideograph>)
      2008   => 16#5649#,  --  噉 (<CJK Ideograph>)
      2009   => 16#564B#,  --  噋 (<CJK Ideograph>)
      2010   => 16#564D#,  --  噍 (<CJK Ideograph>)
      2011   => 16#564F#,  --  噏 (<CJK Ideograph>)
      2012   => 16#5654#,  --  噔 (<CJK Ideograph>)
      2013   => 16#565E#,  --  噞 (<CJK Ideograph>)
      2014   => 16#5660#,  --  噠 (<CJK Ideograph>)
      2015   => 16#5661#,  --  噡 (<CJK Ideograph>)
      2016   => 16#5662#,  --  噢 (<CJK Ideograph>)
      2017   => 16#5663#,  --  噣 (<CJK Ideograph>)
      2018   => 16#5666#,  --  噦 (<CJK Ideograph>)
      2019   => 16#5669#,  --  噩 (<CJK Ideograph>)
      2020   => 16#566D#,  --  噭 (<CJK Ideograph>)
      2021   => 16#566F#,  --  噯 (<CJK Ideograph>)
      2022   => 16#5671#,  --  噱 (<CJK Ideograph>)
      2023   => 16#5672#,  --  噲 (<CJK Ideograph>)
      2024   => 16#5675#,  --  噵 (<CJK Ideograph>)
      2025   => 16#5684#,  --  嚄 (<CJK Ideograph>)
      2026   => 16#5685#,  --  嚅 (<CJK Ideograph>)
      2027   => 16#5688#,  --  嚈 (<CJK Ideograph>)
      2028   => 16#568B#,  --  嚋 (<CJK Ideograph>)
      2029   => 16#568C#,  --  嚌 (<CJK Ideograph>)
      2030   => 16#5695#,  --  嚕 (<CJK Ideograph>)
      2031   => 16#5699#,  --  嚙 (<CJK Ideograph>)
      2032   => 16#569A#,  --  嚚 (<CJK Ideograph>)
      2033   => 16#569D#,  --  嚝 (<CJK Ideograph>)
      2034   => 16#569E#,  --  嚞 (<CJK Ideograph>)
      2035   => 16#569F#,  --  嚟 (<CJK Ideograph>)
      2036   => 16#56A6#,  --  嚦 (<CJK Ideograph>)
      2037   => 16#56A7#,  --  嚧 (<CJK Ideograph>)
      2038   => 16#56A8#,  --  嚨 (<CJK Ideograph>)
      2039   => 16#56A9#,  --  嚩 (<CJK Ideograph>)
      2040   => 16#56AB#,  --  嚫 (<CJK Ideograph>)
      2041   => 16#56AC#,  --  嚬 (<CJK Ideograph>)
      2042   => 16#56AD#,  --  嚭 (<CJK Ideograph>)
      2043   => 16#56B1#,  --  嚱 (<CJK Ideograph>)
      2044   => 16#56B3#,  --  嚳 (<CJK Ideograph>)
      2045   => 16#56B7#,  --  嚷 (<CJK Ideograph>)
      2046   => 16#56BE#,  --  嚾 (<CJK Ideograph>)
      2047   => 16#56C5#,  --  囅 (<CJK Ideograph>)
      2048   => 16#56C9#,  --  囉 (<CJK Ideograph>)
      2049   => 16#56CA#,  --  囊 (<CJK Ideograph>)
      2050   => 16#56CB#,  --  囋 (<CJK Ideograph>)
      2051   => 16#56CF#,  --  囏 (<CJK Ideograph>)
      2052   => 16#56D0#,  --  囐 (<CJK Ideograph>)
      2053   => 16#56CC#,  --  囌 (<CJK Ideograph>)
      2054   => 16#56CD#,  --  囍 (<CJK Ideograph>)
      2055   => 16#56D9#,  --  囙 (<CJK Ideograph>)
      2056   => 16#56DC#,  --  囜 (<CJK Ideograph>)
      2057   => 16#56DD#,  --  囝 (<CJK Ideograph>)
      2058   => 16#56DF#,  --  囟 (<CJK Ideograph>)
      2059   => 16#56E1#,  --  囡 (<CJK Ideograph>)
      2060   => 16#56E4#,  --  囤 (<CJK Ideograph>)
      2061   => 16#56E5#,  --  囥 (<CJK Ideograph>)
      2062   => 16#56E6#,  --  囦 (<CJK Ideograph>)
      2063   => 16#56E7#,  --  囧 (<CJK Ideograph>)
      2064   => 16#56E8#,  --  囨 (<CJK Ideograph>)
      2065   => 16#56F1#,  --  囱 (<CJK Ideograph>)
      2066   => 16#56EB#,  --  囫 (<CJK Ideograph>)
      2067   => 16#56ED#,  --  园 (<CJK Ideograph>)
      2068   => 16#56F6#,  --  囶 (<CJK Ideograph>)
      2069   => 16#56F7#,  --  囷 (<CJK Ideograph>)
      2070   => 16#5701#,  --  圁 (<CJK Ideograph>)
      2071   => 16#5702#,  --  圂 (<CJK Ideograph>)
      2072   => 16#5707#,  --  圇 (<CJK Ideograph>)
      2073   => 16#570A#,  --  圊 (<CJK Ideograph>)
      2074   => 16#570C#,  --  圌 (<CJK Ideograph>)
      2075   => 16#5711#,  --  圑 (<CJK Ideograph>)
      2076   => 16#5715#,  --  圕 (<CJK Ideograph>)
      2077   => 16#571A#,  --  圚 (<CJK Ideograph>)
      2078   => 16#571B#,  --  圛 (<CJK Ideograph>)
      2079   => 16#571D#,  --  圝 (<CJK Ideograph>)
      2080   => 16#5720#,  --  圠 (<CJK Ideograph>)
      2081   => 16#5722#,  --  圢 (<CJK Ideograph>)
      2082   => 16#5723#,  --  圣 (<CJK Ideograph>)
      2083   => 16#5724#,  --  圤 (<CJK Ideograph>)
      2084   => 16#5725#,  --  圥 (<CJK Ideograph>)
      2085   => 16#5729#,  --  圩 (<CJK Ideograph>)
      2086   => 16#572A#,  --  圪 (<CJK Ideograph>)
      2087   => 16#572C#,  --  圬 (<CJK Ideograph>)
      2088   => 16#572E#,  --  圮 (<CJK Ideograph>)
      2089   => 16#572F#,  --  圯 (<CJK Ideograph>)
      2090   => 16#5733#,  --  圳 (<CJK Ideograph>)
      2091   => 16#5734#,  --  圴 (<CJK Ideograph>)
      2092   => 16#573D#,  --  圽 (<CJK Ideograph>)
      2093   => 16#573E#,  --  圾 (<CJK Ideograph>)
      2094   => 16#573F#,  --  圿 (<CJK Ideograph>)
      2095   => 16#5745#,  --  坅 (<CJK Ideograph>)
      2096   => 16#5746#,  --  坆 (<CJK Ideograph>)
      2097   => 16#574C#,  --  坌 (<CJK Ideograph>)
      2098   => 16#574D#,  --  坍 (<CJK Ideograph>)
      2099   => 16#5752#,  --  坒 (<CJK Ideograph>)
      2100   => 16#5762#,  --  坢 (<CJK Ideograph>)
      2101   => 16#5765#,  --  坥 (<CJK Ideograph>)
      2102   => 16#5767#,  --  坧 (<CJK Ideograph>)
      2103   => 16#5768#,  --  坨 (<CJK Ideograph>)
      2104   => 16#576B#,  --  坫 (<CJK Ideograph>)
      2105   => 16#576D#,  --  坭 (<CJK Ideograph>)
      2106   => 16#576E#,  --  坮 (<CJK Ideograph>)
      2107   => 16#576F#,  --  坯 (<CJK Ideograph>)
      2108   => 16#5770#,  --  坰 (<CJK Ideograph>)
      2109   => 16#5771#,  --  坱 (<CJK Ideograph>)
      2110   => 16#5773#,  --  坳 (<CJK Ideograph>)
      2111   => 16#5774#,  --  坴 (<CJK Ideograph>)
      2112   => 16#5775#,  --  坵 (<CJK Ideograph>)
      2113   => 16#5777#,  --  坷 (<CJK Ideograph>)
      2114   => 16#5779#,  --  坹 (<CJK Ideograph>)
      2115   => 16#577A#,  --  坺 (<CJK Ideograph>)
      2116   => 16#577B#,  --  坻 (<CJK Ideograph>)
      2117   => 16#577C#,  --  坼 (<CJK Ideograph>)
      2118   => 16#577E#,  --  坾 (<CJK Ideograph>)
      2119   => 16#5781#,  --  垁 (<CJK Ideograph>)
      2120   => 16#5783#,  --  垃 (<CJK Ideograph>)
      2121   => 16#578C#,  --  垌 (<CJK Ideograph>)
      2122   => 16#5794#,  --  垔 (<CJK Ideograph>)
      2123   => 16#5797#,  --  垗 (<CJK Ideograph>)
      2124   => 16#5799#,  --  垙 (<CJK Ideograph>)
      2125   => 16#579A#,  --  垚 (<CJK Ideograph>)
      2126   => 16#579C#,  --  垜 (<CJK Ideograph>)
      2127   => 16#579D#,  --  垝 (<CJK Ideograph>)
      2128   => 16#579E#,  --  垞 (<CJK Ideograph>)
      2129   => 16#579F#,  --  垟 (<CJK Ideograph>)
      2130   => 16#57A1#,  --  垡 (<CJK Ideograph>)
      2131   => 16#5795#,  --  垕 (<CJK Ideograph>)
      2132   => 16#57A7#,  --  垧 (<CJK Ideograph>)
      2133   => 16#57A8#,  --  垨 (<CJK Ideograph>)
      2134   => 16#57A9#,  --  垩 (<CJK Ideograph>)
      2135   => 16#57AC#,  --  垬 (<CJK Ideograph>)
      2136   => 16#57B8#,  --  垸 (<CJK Ideograph>)
      2137   => 16#57BD#,  --  垽 (<CJK Ideograph>)
      2138   => 16#57C7#,  --  埇 (<CJK Ideograph>)
      2139   => 16#57C8#,  --  埈 (<CJK Ideograph>)
      2140   => 16#57CC#,  --  埌 (<CJK Ideograph>)
      2141   => 16#57CF#,  --  埏 (<CJK Ideograph>)
      2142   => 16#57D5#,  --  埕 (<CJK Ideograph>)
      2143   => 16#57DD#,  --  埝 (<CJK Ideograph>)
      2144   => 16#57DE#,  --  埞 (<CJK Ideograph>)
      2145   => 16#57E4#,  --  埤 (<CJK Ideograph>)
      2146   => 16#57E6#,  --  埦 (<CJK Ideograph>)
      2147   => 16#57E7#,  --  埧 (<CJK Ideograph>)
      2148   => 16#57E9#,  --  埩 (<CJK Ideograph>)
      2149   => 16#57ED#,  --  埭 (<CJK Ideograph>)
      2150   => 16#57F0#,  --  埰 (<CJK Ideograph>)
      2151   => 16#57F5#,  --  埵 (<CJK Ideograph>)
      2152   => 16#57F6#,  --  埶 (<CJK Ideograph>)
      2153   => 16#57F8#,  --  埸 (<CJK Ideograph>)
      2154   => 16#57FD#,  --  埽 (<CJK Ideograph>)
      2155   => 16#57FE#,  --  埾 (<CJK Ideograph>)
      2156   => 16#57FF#,  --  埿 (<CJK Ideograph>)
      2157   => 16#5803#,  --  堃 (<CJK Ideograph>)
      2158   => 16#5804#,  --  堄 (<CJK Ideograph>)
      2159   => 16#5808#,  --  堈 (<CJK Ideograph>)
      2160   => 16#5809#,  --  堉 (<CJK Ideograph>)
      2161   => 16#57E1#,  --  埡 (<CJK Ideograph>)
      2162   => 16#580C#,  --  堌 (<CJK Ideograph>)
      2163   => 16#580D#,  --  堍 (<CJK Ideograph>)
      2164   => 16#581B#,  --  堛 (<CJK Ideograph>)
      2165   => 16#581E#,  --  堞 (<CJK Ideograph>)
      2166   => 16#581F#,  --  堟 (<CJK Ideograph>)
      2167   => 16#5820#,  --  堠 (<CJK Ideograph>)
      2168   => 16#5826#,  --  堦 (<CJK Ideograph>)
      2169   => 16#5827#,  --  堧 (<CJK Ideograph>)
      2170   => 16#582D#,  --  堭 (<CJK Ideograph>)
      2171   => 16#5832#,  --  堲 (<CJK Ideograph>)
      2172   => 16#5839#,  --  堹 (<CJK Ideograph>)
      2173   => 16#583F#,  --  堿 (<CJK Ideograph>)
      2174   => 16#5849#,  --  塉 (<CJK Ideograph>)
      2175   => 16#584C#,  --  塌 (<CJK Ideograph>)
      2176   => 16#584D#,  --  塍 (<CJK Ideograph>)
      2177   => 16#584F#,  --  塏 (<CJK Ideograph>)
      2178   => 16#5850#,  --  塐 (<CJK Ideograph>)
      2179   => 16#5855#,  --  塕 (<CJK Ideograph>)
      2180   => 16#585F#,  --  塟 (<CJK Ideograph>)
      2181   => 16#5861#,  --  塡 (<CJK Ideograph>)
      2182   => 16#5864#,  --  塤 (<CJK Ideograph>)
      2183   => 16#5867#,  --  塧 (<CJK Ideograph>)
      2184   => 16#5868#,  --  塨 (<CJK Ideograph>)
      2185   => 16#5878#,  --  塸 (<CJK Ideograph>)
      2186   => 16#587C#,  --  塼 (<CJK Ideograph>)
      2187   => 16#587F#,  --  塿 (<CJK Ideograph>)
      2188   => 16#5880#,  --  墀 (<CJK Ideograph>)
      2189   => 16#5881#,  --  墁 (<CJK Ideograph>)
      2190   => 16#5887#,  --  墇 (<CJK Ideograph>)
      2191   => 16#5888#,  --  墈 (<CJK Ideograph>)
      2192   => 16#5889#,  --  墉 (<CJK Ideograph>)
      2193   => 16#588A#,  --  墊 (<CJK Ideograph>)
      2194   => 16#588C#,  --  墌 (<CJK Ideograph>)
      2195   => 16#588D#,  --  墍 (<CJK Ideograph>)
      2196   => 16#588F#,  --  墏 (<CJK Ideograph>)
      2197   => 16#5890#,  --  墐 (<CJK Ideograph>)
      2198   => 16#5894#,  --  墔 (<CJK Ideograph>)
      2199   => 16#5896#,  --  墖 (<CJK Ideograph>)
      2200   => 16#589D#,  --  墝 (<CJK Ideograph>)
      2201   => 16#58A0#,  --  墠 (<CJK Ideograph>)
      2202   => 16#58A1#,  --  墡 (<CJK Ideograph>)
      2203   => 16#58A2#,  --  墢 (<CJK Ideograph>)
      2204   => 16#58A6#,  --  墦 (<CJK Ideograph>)
      2205   => 16#58A9#,  --  墩 (<CJK Ideograph>)
      2206   => 16#58B1#,  --  墱 (<CJK Ideograph>)
      2207   => 16#58B2#,  --  墲 (<CJK Ideograph>)
      2208   => 16#58C4#,  --  壄 (<CJK Ideograph>)
      2209   => 16#58BC#,  --  墼 (<CJK Ideograph>)
      2210   => 16#58C2#,  --  壂 (<CJK Ideograph>)
      2211   => 16#58C8#,  --  壈 (<CJK Ideograph>)
      2212   => 16#58CD#,  --  壍 (<CJK Ideograph>)
      2213   => 16#58CE#,  --  壎 (<CJK Ideograph>)
      2214   => 16#58D0#,  --  壐 (<CJK Ideograph>)
      2215   => 16#58D2#,  --  壒 (<CJK Ideograph>)
      2216   => 16#58D4#,  --  壔 (<CJK Ideograph>)
      2217   => 16#58D6#,  --  壖 (<CJK Ideograph>)
      2218   => 16#58DA#,  --  壚 (<CJK Ideograph>)
      2219   => 16#58DD#,  --  壝 (<CJK Ideograph>)
      2220   => 16#58E1#,  --  壡 (<CJK Ideograph>)
      2221   => 16#58E2#,  --  壢 (<CJK Ideograph>)
      2222   => 16#58E9#,  --  壩 (<CJK Ideograph>)
      2223   => 16#58F3#,  --  壳 (<CJK Ideograph>)
      2224   => 16#5905#,  --  夅 (<CJK Ideograph>)
      2225   => 16#5906#,  --  夆 (<CJK Ideograph>)
      2226   => 16#590B#,  --  夋 (<CJK Ideograph>)
      2227   => 16#590C#,  --  夌 (<CJK Ideograph>)
      2228   => 16#5912#,  --  夒 (<CJK Ideograph>)
      2229   => 16#5913#,  --  夓 (<CJK Ideograph>)
      2230   => 16#5914#,  --  夔 (<CJK Ideograph>)
      2231   => 16#8641#,  --  虁 (<CJK Ideograph>)
      2232   => 16#591D#,  --  夝 (<CJK Ideograph>)
      2233   => 16#5921#,  --  夡 (<CJK Ideograph>)
      2234   => 16#5923#,  --  夣 (<CJK Ideograph>)
      2235   => 16#5924#,  --  夤 (<CJK Ideograph>)
      2236   => 16#5928#,  --  夨 (<CJK Ideograph>)
      2237   => 16#592F#,  --  夯 (<CJK Ideograph>)
      2238   => 16#5930#,  --  夰 (<CJK Ideograph>)
      2239   => 16#5933#,  --  夳 (<CJK Ideograph>)
      2240   => 16#5935#,  --  夵 (<CJK Ideograph>)
      2241   => 16#5936#,  --  夶 (<CJK Ideograph>)
      2242   => 16#593F#,  --  夿 (<CJK Ideograph>)
      2243   => 16#5943#,  --  奃 (<CJK Ideograph>)
      2244   => 16#5946#,  --  奆 (<CJK Ideograph>)
      2245   => 16#5952#,  --  奒 (<CJK Ideograph>)
      2246   => 16#5953#,  --  奓 (<CJK Ideograph>)
      2247   => 16#5959#,  --  奙 (<CJK Ideograph>)
      2248   => 16#595B#,  --  奛 (<CJK Ideograph>)
      2249   => 16#595D#,  --  奝 (<CJK Ideograph>)
      2250   => 16#595E#,  --  奞 (<CJK Ideograph>)
      2251   => 16#595F#,  --  奟 (<CJK Ideograph>)
      2252   => 16#5961#,  --  奡 (<CJK Ideograph>)
      2253   => 16#5963#,  --  奣 (<CJK Ideograph>)
      2254   => 16#596B#,  --  奫 (<CJK Ideograph>)
      2255   => 16#596D#,  --  奭 (<CJK Ideograph>)
      2256   => 16#596F#,  --  奯 (<CJK Ideograph>)
      2257   => 16#5972#,  --  奲 (<CJK Ideograph>)
      2258   => 16#5975#,  --  奵 (<CJK Ideograph>)
      2259   => 16#5976#,  --  奶 (<CJK Ideograph>)
      2260   => 16#5979#,  --  她 (<CJK Ideograph>)
      2261   => 16#597B#,  --  奻 (<CJK Ideograph>)
      2262   => 16#597C#,  --  奼 (<CJK Ideograph>)
      2263   => 16#598B#,  --  妋 (<CJK Ideograph>)
      2264   => 16#598C#,  --  妌 (<CJK Ideograph>)
      2265   => 16#598E#,  --  妎 (<CJK Ideograph>)
      2266   => 16#5992#,  --  妒 (<CJK Ideograph>)
      2267   => 16#5995#,  --  妕 (<CJK Ideograph>)
      2268   => 16#5997#,  --  妗 (<CJK Ideograph>)
      2269   => 16#599F#,  --  妟 (<CJK Ideograph>)
      2270   => 16#59A4#,  --  妤 (<CJK Ideograph>)
      2271   => 16#59A7#,  --  妧 (<CJK Ideograph>)
      2272   => 16#59AD#,  --  妭 (<CJK Ideograph>)
      2273   => 16#59AE#,  --  妮 (<CJK Ideograph>)
      2274   => 16#59AF#,  --  妯 (<CJK Ideograph>)
      2275   => 16#59B0#,  --  妰 (<CJK Ideograph>)
      2276   => 16#59B3#,  --  妳 (<CJK Ideograph>)
      2277   => 16#59B7#,  --  妷 (<CJK Ideograph>)
      2278   => 16#59BA#,  --  妺 (<CJK Ideograph>)
      2279   => 16#59BC#,  --  妼 (<CJK Ideograph>)
      2280   => 16#59C1#,  --  姁 (<CJK Ideograph>)
      2281   => 16#59C3#,  --  姃 (<CJK Ideograph>)
      2282   => 16#59C4#,  --  姄 (<CJK Ideograph>)
      2283   => 16#59C8#,  --  姈 (<CJK Ideograph>)
      2284   => 16#59CA#,  --  姊 (<CJK Ideograph>)
      2285   => 16#59CD#,  --  姍 (<CJK Ideograph>)
      2286   => 16#59D2#,  --  姒 (<CJK Ideograph>)
      2287   => 16#59DD#,  --  姝 (<CJK Ideograph>)
      2288   => 16#59DE#,  --  姞 (<CJK Ideograph>)
      2289   => 16#59DF#,  --  姟 (<CJK Ideograph>)
      2290   => 16#59E3#,  --  姣 (<CJK Ideograph>)
      2291   => 16#59E4#,  --  姤 (<CJK Ideograph>)
      2292   => 16#59E7#,  --  姧 (<CJK Ideograph>)
      2293   => 16#59EE#,  --  姮 (<CJK Ideograph>)
      2294   => 16#59EF#,  --  姯 (<CJK Ideograph>)
      2295   => 16#59F1#,  --  姱 (<CJK Ideograph>)
      2296   => 16#59F2#,  --  姲 (<CJK Ideograph>)
      2297   => 16#59F4#,  --  姴 (<CJK Ideograph>)
      2298   => 16#59F7#,  --  姷 (<CJK Ideograph>)
      2299   => 16#5A00#,  --  娀 (<CJK Ideograph>)
      2300   => 16#5A04#,  --  娄 (<CJK Ideograph>)
      2301   => 16#5A0C#,  --  娌 (<CJK Ideograph>)
      2302   => 16#5A0D#,  --  娍 (<CJK Ideograph>)
      2303   => 16#5A0E#,  --  娎 (<CJK Ideograph>)
      2304   => 16#5A12#,  --  娒 (<CJK Ideograph>)
      2305   => 16#5A13#,  --  娓 (<CJK Ideograph>)
      2306   => 16#5A1E#,  --  娞 (<CJK Ideograph>)
      2307   => 16#5A23#,  --  娣 (<CJK Ideograph>)
      2308   => 16#5A24#,  --  娤 (<CJK Ideograph>)
      2309   => 16#5A27#,  --  娧 (<CJK Ideograph>)
      2310   => 16#5A28#,  --  娨 (<CJK Ideograph>)
      2311   => 16#5A2A#,  --  娪 (<CJK Ideograph>)
      2312   => 16#5A2D#,  --  娭 (<CJK Ideograph>)
      2313   => 16#5A30#,  --  娰 (<CJK Ideograph>)
      2314   => 16#5A44#,  --  婄 (<CJK Ideograph>)
      2315   => 16#5A45#,  --  婅 (<CJK Ideograph>)
      2316   => 16#5A47#,  --  婇 (<CJK Ideograph>)
      2317   => 16#5A48#,  --  婈 (<CJK Ideograph>)
      2318   => 16#5A4C#,  --  婌 (<CJK Ideograph>)
      2319   => 16#5A50#,  --  婐 (<CJK Ideograph>)
      2320   => 16#5A55#,  --  婕 (<CJK Ideograph>)
      2321   => 16#5A5E#,  --  婞 (<CJK Ideograph>)
      2322   => 16#5A63#,  --  婣 (<CJK Ideograph>)
      2323   => 16#5A65#,  --  婥 (<CJK Ideograph>)
      2324   => 16#5A67#,  --  婧 (<CJK Ideograph>)
      2325   => 16#5A6D#,  --  婭 (<CJK Ideograph>)
      2326   => 16#5A77#,  --  婷 (<CJK Ideograph>)
      2327   => 16#5A7A#,  --  婺 (<CJK Ideograph>)
      2328   => 16#5A7B#,  --  婻 (<CJK Ideograph>)
      2329   => 16#5A7E#,  --  婾 (<CJK Ideograph>)
      2330   => 16#5A8B#,  --  媋 (<CJK Ideograph>)
      2331   => 16#5A90#,  --  媐 (<CJK Ideograph>)
      2332   => 16#5A93#,  --  媓 (<CJK Ideograph>)
      2333   => 16#5A96#,  --  媖 (<CJK Ideograph>)
      2334   => 16#5A99#,  --  媙 (<CJK Ideograph>)
      2335   => 16#5A9C#,  --  媜 (<CJK Ideograph>)
      2336   => 16#5A9E#,  --  媞 (<CJK Ideograph>)
      2337   => 16#5A9F#,  --  媟 (<CJK Ideograph>)
      2338   => 16#5AA0#,  --  媠 (<CJK Ideograph>)
      2339   => 16#5AA2#,  --  媢 (<CJK Ideograph>)
      2340   => 16#5AA7#,  --  媧 (<CJK Ideograph>)
      2341   => 16#5AAC#,  --  媬 (<CJK Ideograph>)
      2342   => 16#5AB1#,  --  媱 (<CJK Ideograph>)
      2343   => 16#5AB2#,  --  媲 (<CJK Ideograph>)
      2344   => 16#5AB3#,  --  媳 (<CJK Ideograph>)
      2345   => 16#5AB5#,  --  媵 (<CJK Ideograph>)
      2346   => 16#5AB8#,  --  媸 (<CJK Ideograph>)
      2347   => 16#5ABA#,  --  媺 (<CJK Ideograph>)
      2348   => 16#5ABB#,  --  媻 (<CJK Ideograph>)
      2349   => 16#5ABF#,  --  媿 (<CJK Ideograph>)
      2350   => 16#5AC4#,  --  嫄 (<CJK Ideograph>)
      2351   => 16#5AC6#,  --  嫆 (<CJK Ideograph>)
      2352   => 16#5AC8#,  --  嫈 (<CJK Ideograph>)
      2353   => 16#5ACF#,  --  嫏 (<CJK Ideograph>)
      2354   => 16#5ADA#,  --  嫚 (<CJK Ideograph>)
      2355   => 16#5ADC#,  --  嫜 (<CJK Ideograph>)
      2356   => 16#5AE0#,  --  嫠 (<CJK Ideograph>)
      2357   => 16#5AE5#,  --  嫥 (<CJK Ideograph>)
      2358   => 16#5AEA#,  --  嫪 (<CJK Ideograph>)
      2359   => 16#5AEE#,  --  嫮 (<CJK Ideograph>)
      2360   => 16#5AF5#,  --  嫵 (<CJK Ideograph>)
      2361   => 16#5AF6#,  --  嫶 (<CJK Ideograph>)
      2362   => 16#5AFD#,  --  嫽 (<CJK Ideograph>)
      2363   => 16#5B00#,  --  嬀 (<CJK Ideograph>)
      2364   => 16#5B01#,  --  嬁 (<CJK Ideograph>)
      2365   => 16#5B08#,  --  嬈 (<CJK Ideograph>)
      2366   => 16#5B17#,  --  嬗 (<CJK Ideograph>)
      2367   => 16#5B34#,  --  嬴 (<CJK Ideograph>)
      2368   => 16#5B19#,  --  嬙 (<CJK Ideograph>)
      2369   => 16#5B1B#,  --  嬛 (<CJK Ideograph>)
      2370   => 16#5B1D#,  --  嬝 (<CJK Ideograph>)
      2371   => 16#5B21#,  --  嬡 (<CJK Ideograph>)
      2372   => 16#5B25#,  --  嬥 (<CJK Ideograph>)
      2373   => 16#5B2D#,  --  嬭 (<CJK Ideograph>)
      2374   => 16#5B38#,  --  嬸 (<CJK Ideograph>)
      2375   => 16#5B41#,  --  孁 (<CJK Ideograph>)
      2376   => 16#5B4B#,  --  孋 (<CJK Ideograph>)
      2377   => 16#5B4C#,  --  孌 (<CJK Ideograph>)
      2378   => 16#5B52#,  --  孒 (<CJK Ideograph>)
      2379   => 16#5B56#,  --  孖 (<CJK Ideograph>)
      2380   => 16#5B5E#,  --  孞 (<CJK Ideograph>)
      2381   => 16#5B68#,  --  孨 (<CJK Ideograph>)
      2382   => 16#5B6E#,  --  孮 (<CJK Ideograph>)
      2383   => 16#5B6F#,  --  孯 (<CJK Ideograph>)
      2384   => 16#5B7C#,  --  孼 (<CJK Ideograph>)
      2385   => 16#5B7D#,  --  孽 (<CJK Ideograph>)
      2386   => 16#5B7E#,  --  孾 (<CJK Ideograph>)
      2387   => 16#5B7F#,  --  孿 (<CJK Ideograph>)
      2388   => 16#5B81#,  --  宁 (<CJK Ideograph>)
      2389   => 16#5B84#,  --  宄 (<CJK Ideograph>)
      2390   => 16#5B86#,  --  宆 (<CJK Ideograph>)
      2391   => 16#5B8A#,  --  宊 (<CJK Ideograph>)
      2392   => 16#5B8E#,  --  宎 (<CJK Ideograph>)
      2393   => 16#5B90#,  --  宐 (<CJK Ideograph>)
      2394   => 16#5B91#,  --  宑 (<CJK Ideograph>)
      2395   => 16#5B93#,  --  宓 (<CJK Ideograph>)
      2396   => 16#5B94#,  --  宔 (<CJK Ideograph>)
      2397   => 16#5B96#,  --  宖 (<CJK Ideograph>)
      2398   => 16#5BA8#,  --  宨 (<CJK Ideograph>)
      2399   => 16#5BA9#,  --  宩 (<CJK Ideograph>)
      2400   => 16#5BAC#,  --  宬 (<CJK Ideograph>)
      2401   => 16#5BAD#,  --  宭 (<CJK Ideograph>)
      2402   => 16#5BAF#,  --  宯 (<CJK Ideograph>)
      2403   => 16#5BB1#,  --  宱 (<CJK Ideograph>)
      2404   => 16#5BB2#,  --  宲 (<CJK Ideograph>)
      2405   => 16#5BB7#,  --  宷 (<CJK Ideograph>)
      2406   => 16#5BBA#,  --  宺 (<CJK Ideograph>)
      2407   => 16#5BBC#,  --  宼 (<CJK Ideograph>)
      2408   => 16#5BC0#,  --  寀 (<CJK Ideograph>)
      2409   => 16#5BC1#,  --  寁 (<CJK Ideograph>)
      2410   => 16#5BCD#,  --  寍 (<CJK Ideograph>)
      2411   => 16#5BCF#,  --  寏 (<CJK Ideograph>)
      2412   => 16#5BD6#,  --  寖 (<CJK Ideograph>)
      2413   => 16#5BD7#,  --  寗 (<CJK Ideograph>)
      2414   => 16#5BD8#,  --  寘 (<CJK Ideograph>)
      2415   => 16#5BD9#,  --  寙 (<CJK Ideograph>)
      2416   => 16#5BDA#,  --  寚 (<CJK Ideograph>)
      2417   => 16#5BE0#,  --  寠 (<CJK Ideograph>)
      2418   => 16#5BEF#,  --  寯 (<CJK Ideograph>)
      2419   => 16#5BF1#,  --  寱 (<CJK Ideograph>)
      2420   => 16#5BF4#,  --  寴 (<CJK Ideograph>)
      2421   => 16#5BFD#,  --  寽 (<CJK Ideograph>)
      2422   => 16#5C0C#,  --  尌 (<CJK Ideograph>)
      2423   => 16#5C17#,  --  尗 (<CJK Ideograph>)
      2424   => 16#5C1E#,  --  尞 (<CJK Ideograph>)
      2425   => 16#5C1F#,  --  尟 (<CJK Ideograph>)
      2426   => 16#5C23#,  --  尣 (<CJK Ideograph>)
      2427   => 16#5C26#,  --  尦 (<CJK Ideograph>)
      2428   => 16#5C29#,  --  尩 (<CJK Ideograph>)
      2429   => 16#5C2B#,  --  尫 (<CJK Ideograph>)
      2430   => 16#5C2C#,  --  尬 (<CJK Ideograph>)
      2431   => 16#5C2E#,  --  尮 (<CJK Ideograph>)
      2432   => 16#5C30#,  --  尰 (<CJK Ideograph>)
      2433   => 16#5C32#,  --  尲 (<CJK Ideograph>)
      2434   => 16#5C35#,  --  尵 (<CJK Ideograph>)
      2435   => 16#5C36#,  --  尶 (<CJK Ideograph>)
      2436   => 16#5C59#,  --  屙 (<CJK Ideograph>)
      2437   => 16#5C5A#,  --  屚 (<CJK Ideograph>)
      2438   => 16#5C5C#,  --  屜 (<CJK Ideograph>)
      2439   => 16#5C62#,  --  屢 (<CJK Ideograph>)
      2440   => 16#5C63#,  --  屣 (<CJK Ideograph>)
      2441   => 16#5C67#,  --  屧 (<CJK Ideograph>)
      2442   => 16#5C68#,  --  屨 (<CJK Ideograph>)
      2443   => 16#5C69#,  --  屩 (<CJK Ideograph>)
      2444   => 16#5C6D#,  --  屭 (<CJK Ideograph>)
      2445   => 16#5C70#,  --  屰 (<CJK Ideograph>)
      2446   => 16#5C74#,  --  屴 (<CJK Ideograph>)
      2447   => 16#5C75#,  --  屵 (<CJK Ideograph>)
      2448   => 16#5C7A#,  --  屺 (<CJK Ideograph>)
      2449   => 16#5C7B#,  --  屻 (<CJK Ideograph>)
      2450   => 16#5C7C#,  --  屼 (<CJK Ideograph>)
      2451   => 16#5C7D#,  --  屽 (<CJK Ideograph>)
      2452   => 16#5C87#,  --  岇 (<CJK Ideograph>)
      2453   => 16#5C88#,  --  岈 (<CJK Ideograph>)
      2454   => 16#5C8A#,  --  岊 (<CJK Ideograph>)
      2455   => 16#5C8F#,  --  岏 (<CJK Ideograph>)
      2456   => 16#5C92#,  --  岒 (<CJK Ideograph>)
      2457   => 16#5C9D#,  --  岝 (<CJK Ideograph>)
      2458   => 16#5C9F#,  --  岟 (<CJK Ideograph>)
      2459   => 16#5CA0#,  --  岠 (<CJK Ideograph>)
      2460   => 16#5CA2#,  --  岢 (<CJK Ideograph>)
      2461   => 16#5CA3#,  --  岣 (<CJK Ideograph>)
      2462   => 16#5CA6#,  --  岦 (<CJK Ideograph>)
      2463   => 16#5CAA#,  --  岪 (<CJK Ideograph>)
      2464   => 16#5CB2#,  --  岲 (<CJK Ideograph>)
      2465   => 16#5CB4#,  --  岴 (<CJK Ideograph>)
      2466   => 16#5CB5#,  --  岵 (<CJK Ideograph>)
      2467   => 16#5CBA#,  --  岺 (<CJK Ideograph>)
      2468   => 16#5CC9#,  --  峉 (<CJK Ideograph>)
      2469   => 16#5CCB#,  --  峋 (<CJK Ideograph>)
      2470   => 16#5CD2#,  --  峒 (<CJK Ideograph>)
      2471   => 16#5CDD#,  --  峝 (<CJK Ideograph>)
      2472   => 16#5CD7#,  --  峗 (<CJK Ideograph>)
      2473   => 16#5CEE#,  --  峮 (<CJK Ideograph>)
      2474   => 16#5CF1#,  --  峱 (<CJK Ideograph>)
      2475   => 16#5CF2#,  --  峲 (<CJK Ideograph>)
      2476   => 16#5CF4#,  --  峴 (<CJK Ideograph>)
      2477   => 16#5D01#,  --  崁 (<CJK Ideograph>)
      2478   => 16#5D06#,  --  崆 (<CJK Ideograph>)
      2479   => 16#5D0D#,  --  崍 (<CJK Ideograph>)
      2480   => 16#5D12#,  --  崒 (<CJK Ideograph>)
      2481   => 16#5D2B#,  --  崫 (<CJK Ideograph>)
      2482   => 16#5D23#,  --  崣 (<CJK Ideograph>)
      2483   => 16#5D24#,  --  崤 (<CJK Ideograph>)
      2484   => 16#5D26#,  --  崦 (<CJK Ideograph>)
      2485   => 16#5D27#,  --  崧 (<CJK Ideograph>)
      2486   => 16#5D31#,  --  崱 (<CJK Ideograph>)
      2487   => 16#5D34#,  --  崴 (<CJK Ideograph>)
      2488   => 16#5D39#,  --  崹 (<CJK Ideograph>)
      2489   => 16#5D3D#,  --  崽 (<CJK Ideograph>)
      2490   => 16#5D3F#,  --  崿 (<CJK Ideograph>)
      2491   => 16#5D42#,  --  嵂 (<CJK Ideograph>)
      2492   => 16#5D43#,  --  嵃 (<CJK Ideograph>)
      2493   => 16#5D46#,  --  嵆 (<CJK Ideograph>)
      2494   => 16#5D48#,  --  嵈 (<CJK Ideograph>)
      2495   => 16#5D55#,  --  嵕 (<CJK Ideograph>)
      2496   => 16#5D51#,  --  嵑 (<CJK Ideograph>)
      2497   => 16#5D59#,  --  嵙 (<CJK Ideograph>)
      2498   => 16#5D4A#,  --  嵊 (<CJK Ideograph>)
      2499   => 16#5D5F#,  --  嵟 (<CJK Ideograph>)
      2500   => 16#5D60#,  --  嵠 (<CJK Ideograph>)
      2501   => 16#5D61#,  --  嵡 (<CJK Ideograph>)
      2502   => 16#5D62#,  --  嵢 (<CJK Ideograph>)
      2503   => 16#5D64#,  --  嵤 (<CJK Ideograph>)
      2504   => 16#5D6A#,  --  嵪 (<CJK Ideograph>)
      2505   => 16#5D6D#,  --  嵭 (<CJK Ideograph>)
      2506   => 16#5D70#,  --  嵰 (<CJK Ideograph>)
      2507   => 16#5D79#,  --  嵹 (<CJK Ideograph>)
      2508   => 16#5D7A#,  --  嵺 (<CJK Ideograph>)
      2509   => 16#5D7E#,  --  嵾 (<CJK Ideograph>)
      2510   => 16#5D7F#,  --  嵿 (<CJK Ideograph>)
      2511   => 16#5D81#,  --  嶁 (<CJK Ideograph>)
      2512   => 16#5D83#,  --  嶃 (<CJK Ideograph>)
      2513   => 16#5D88#,  --  嶈 (<CJK Ideograph>)
      2514   => 16#5D8A#,  --  嶊 (<CJK Ideograph>)
      2515   => 16#5D92#,  --  嶒 (<CJK Ideograph>)
      2516   => 16#5D93#,  --  嶓 (<CJK Ideograph>)
      2517   => 16#5D94#,  --  嶔 (<CJK Ideograph>)
      2518   => 16#5D95#,  --  嶕 (<CJK Ideograph>)
      2519   => 16#5D99#,  --  嶙 (<CJK Ideograph>)
      2520   => 16#5D9B#,  --  嶛 (<CJK Ideograph>)
      2521   => 16#5D9F#,  --  嶟 (<CJK Ideograph>)
      2522   => 16#5DA0#,  --  嶠 (<CJK Ideograph>)
      2523   => 16#5DA7#,  --  嶧 (<CJK Ideograph>)
      2524   => 16#5DAB#,  --  嶫 (<CJK Ideograph>)
      2525   => 16#5DB0#,  --  嶰 (<CJK Ideograph>)
      2526   => 16#5DB4#,  --  嶴 (<CJK Ideograph>)
      2527   => 16#5DB8#,  --  嶸 (<CJK Ideograph>)
      2528   => 16#5DB9#,  --  嶹 (<CJK Ideograph>)
      2529   => 16#5DC3#,  --  巃 (<CJK Ideograph>)
      2530   => 16#5DC7#,  --  巇 (<CJK Ideograph>)
      2531   => 16#5DCB#,  --  巋 (<CJK Ideograph>)
      2532   => 16#5DD0#,  --  巐 (<CJK Ideograph>)
      2533   => 16#5DCE#,  --  巎 (<CJK Ideograph>)
      2534   => 16#5DD8#,  --  巘 (<CJK Ideograph>)
      2535   => 16#5DD9#,  --  巙 (<CJK Ideograph>)
      2536   => 16#5DE0#,  --  巠 (<CJK Ideograph>)
      2537   => 16#5DE4#,  --  巤 (<CJK Ideograph>)
      2538   => 16#5DE9#,  --  巩 (<CJK Ideograph>)
      2539   => 16#5DF8#,  --  巸 (<CJK Ideograph>)
      2540   => 16#5DF9#,  --  巹 (<CJK Ideograph>)
      2541   => 16#5E00#,  --  帀 (<CJK Ideograph>)
      2542   => 16#5E07#,  --  帇 (<CJK Ideograph>)
      2543   => 16#5E0D#,  --  帍 (<CJK Ideograph>)
      2544   => 16#5E12#,  --  帒 (<CJK Ideograph>)
      2545   => 16#5E14#,  --  帔 (<CJK Ideograph>)
      2546   => 16#5E15#,  --  帕 (<CJK Ideograph>)
      2547   => 16#5E18#,  --  帘 (<CJK Ideograph>)
      2548   => 16#5E1F#,  --  帟 (<CJK Ideograph>)
      2549   => 16#5E20#,  --  帠 (<CJK Ideograph>)
      2550   => 16#5E2E#,  --  帮 (<CJK Ideograph>)
      2551   => 16#5E28#,  --  帨 (<CJK Ideograph>)
      2552   => 16#5E32#,  --  帲 (<CJK Ideograph>)
      2553   => 16#5E35#,  --  帵 (<CJK Ideograph>)
      2554   => 16#5E3E#,  --  帾 (<CJK Ideograph>)
      2555   => 16#5E4B#,  --  幋 (<CJK Ideograph>)
      2556   => 16#5E50#,  --  幐 (<CJK Ideograph>)
      2557   => 16#5E49#,  --  幉 (<CJK Ideograph>)
      2558   => 16#5E51#,  --  幑 (<CJK Ideograph>)
      2559   => 16#5E56#,  --  幖 (<CJK Ideograph>)
      2560   => 16#5E58#,  --  幘 (<CJK Ideograph>)
      2561   => 16#5E5B#,  --  幛 (<CJK Ideograph>)
      2562   => 16#5E5C#,  --  幜 (<CJK Ideograph>)
      2563   => 16#5E5E#,  --  幞 (<CJK Ideograph>)
      2564   => 16#5E68#,  --  幨 (<CJK Ideograph>)
      2565   => 16#5E6A#,  --  幪 (<CJK Ideograph>)
      2566   => 16#5E6B#,  --  幫 (<CJK Ideograph>)
      2567   => 16#5E6C#,  --  幬 (<CJK Ideograph>)
      2568   => 16#5E6D#,  --  幭 (<CJK Ideograph>)
      2569   => 16#5E6E#,  --  幮 (<CJK Ideograph>)
      2570   => 16#5E70#,  --  幰 (<CJK Ideograph>)
      2571   => 16#5E80#,  --  庀 (<CJK Ideograph>)
      2572   => 16#5E8B#,  --  庋 (<CJK Ideograph>)
      2573   => 16#5E8E#,  --  庎 (<CJK Ideograph>)
      2574   => 16#5EA2#,  --  庢 (<CJK Ideograph>)
      2575   => 16#5EA4#,  --  庤 (<CJK Ideograph>)
      2576   => 16#5EA5#,  --  庥 (<CJK Ideograph>)
      2577   => 16#5EA8#,  --  庨 (<CJK Ideograph>)
      2578   => 16#5EAA#,  --  庪 (<CJK Ideograph>)
      2579   => 16#5EAC#,  --  庬 (<CJK Ideograph>)
      2580   => 16#5EB1#,  --  庱 (<CJK Ideograph>)
      2581   => 16#5EB3#,  --  庳 (<CJK Ideograph>)
      2582   => 16#5EBD#,  --  庽 (<CJK Ideograph>)
      2583   => 16#5EBE#,  --  庾 (<CJK Ideograph>)
      2584   => 16#5EBF#,  --  庿 (<CJK Ideograph>)
      2585   => 16#5EC6#,  --  廆 (<CJK Ideograph>)
      2586   => 16#5ECC#,  --  廌 (<CJK Ideograph>)
      2587   => 16#5ECB#,  --  廋 (<CJK Ideograph>)
      2588   => 16#5ECE#,  --  廎 (<CJK Ideograph>)
      2589   => 16#5ED1#,  --  廑 (<CJK Ideograph>)
      2590   => 16#5ED2#,  --  廒 (<CJK Ideograph>)
      2591   => 16#5ED4#,  --  廔 (<CJK Ideograph>)
      2592   => 16#5ED5#,  --  廕 (<CJK Ideograph>)
      2593   => 16#5EDC#,  --  廜 (<CJK Ideograph>)
      2594   => 16#5EDE#,  --  廞 (<CJK Ideograph>)
      2595   => 16#5EE5#,  --  廥 (<CJK Ideograph>)
      2596   => 16#5EEB#,  --  廫 (<CJK Ideograph>)
      2597   => 16#5F02#,  --  异 (<CJK Ideograph>)
      2598   => 16#5F06#,  --  弆 (<CJK Ideograph>)
      2599   => 16#5F07#,  --  弇 (<CJK Ideograph>)
      2600   => 16#5F08#,  --  弈 (<CJK Ideograph>)
      2601   => 16#5F0E#,  --  弎 (<CJK Ideograph>)
      2602   => 16#5F19#,  --  弙 (<CJK Ideograph>)
      2603   => 16#5F1C#,  --  弜 (<CJK Ideograph>)
      2604   => 16#5F1D#,  --  弝 (<CJK Ideograph>)
      2605   => 16#5F21#,  --  弡 (<CJK Ideograph>)
      2606   => 16#5F22#,  --  弢 (<CJK Ideograph>)
      2607   => 16#5F23#,  --  弣 (<CJK Ideograph>)
      2608   => 16#5F24#,  --  弤 (<CJK Ideograph>)
      2609   => 16#5F28#,  --  弨 (<CJK Ideograph>)
      2610   => 16#5F2B#,  --  弫 (<CJK Ideograph>)
      2611   => 16#5F2C#,  --  弬 (<CJK Ideograph>)
      2612   => 16#5F2E#,  --  弮 (<CJK Ideograph>)
      2613   => 16#5F30#,  --  弰 (<CJK Ideograph>)
      2614   => 16#5F34#,  --  弴 (<CJK Ideograph>)
      2615   => 16#5F36#,  --  弶 (<CJK Ideograph>)
      2616   => 16#5F3B#,  --  弻 (<CJK Ideograph>)
      2617   => 16#5F3D#,  --  弽 (<CJK Ideograph>)
      2618   => 16#5F3F#,  --  弿 (<CJK Ideograph>)
      2619   => 16#5F40#,  --  彀 (<CJK Ideograph>)
      2620   => 16#5F44#,  --  彄 (<CJK Ideograph>)
      2621   => 16#5F45#,  --  彅 (<CJK Ideograph>)
      2622   => 16#5F47#,  --  彇 (<CJK Ideograph>)
      2623   => 16#5F4D#,  --  彍 (<CJK Ideograph>)
      2624   => 16#5F50#,  --  彐 (<CJK Ideograph>)
      2625   => 16#5F54#,  --  彔 (<CJK Ideograph>)
      2626   => 16#5F58#,  --  彘 (<CJK Ideograph>)
      2627   => 16#5F5B#,  --  彛 (<CJK Ideograph>)
      2628   => 16#5F60#,  --  彠 (<CJK Ideograph>)
      2629   => 16#5F63#,  --  彣 (<CJK Ideograph>)
      2630   => 16#5F64#,  --  彤 (<CJK Ideograph>)
      2631   => 16#5F67#,  --  彧 (<CJK Ideograph>)
      2632   => 16#5F6F#,  --  彯 (<CJK Ideograph>)
      2633   => 16#5F72#,  --  彲 (<CJK Ideograph>)
      2634   => 16#5F74#,  --  彴 (<CJK Ideograph>)
      2635   => 16#5F75#,  --  彵 (<CJK Ideograph>)
      2636   => 16#5F78#,  --  彸 (<CJK Ideograph>)
      2637   => 16#5F7A#,  --  彺 (<CJK Ideograph>)
      2638   => 16#5F7D#,  --  彽 (<CJK Ideograph>)
      2639   => 16#5F7E#,  --  彾 (<CJK Ideograph>)
      2640   => 16#5F89#,  --  徉 (<CJK Ideograph>)
      2641   => 16#5F8D#,  --  徍 (<CJK Ideograph>)
      2642   => 16#5F8F#,  --  徏 (<CJK Ideograph>)
      2643   => 16#5F96#,  --  徖 (<CJK Ideograph>)
      2644   => 16#5F9C#,  --  徜 (<CJK Ideograph>)
      2645   => 16#5F9D#,  --  徝 (<CJK Ideograph>)
      2646   => 16#5FA2#,  --  徢 (<CJK Ideograph>)
      2647   => 16#5FA7#,  --  徧 (<CJK Ideograph>)
      2648   => 16#5FAB#,  --  徫 (<CJK Ideograph>)
      2649   => 16#5FA4#,  --  徤 (<CJK Ideograph>)
      2650   => 16#5FAC#,  --  徬 (<CJK Ideograph>)
      2651   => 16#5FAF#,  --  徯 (<CJK Ideograph>)
      2652   => 16#5FB0#,  --  徰 (<CJK Ideograph>)
      2653   => 16#5FB1#,  --  徱 (<CJK Ideograph>)
      2654   => 16#5FB8#,  --  徸 (<CJK Ideograph>)
      2655   => 16#5FC4#,  --  忄 (<CJK Ideograph>)
      2656   => 16#5FC7#,  --  忇 (<CJK Ideograph>)
      2657   => 16#5FC8#,  --  忈 (<CJK Ideograph>)
      2658   => 16#5FC9#,  --  忉 (<CJK Ideograph>)
      2659   => 16#5FCB#,  --  忋 (<CJK Ideograph>)
      2660   => 16#5FD0#,  --  忐 (<CJK Ideograph>)
      2661   => 16#5FD1#,  --  忑 (<CJK Ideograph>)
      2662   => 16#5FD2#,  --  忒 (<CJK Ideograph>)
      2663   => 16#5FD3#,  --  忓 (<CJK Ideograph>)
      2664   => 16#5FD4#,  --  忔 (<CJK Ideograph>)
      2665   => 16#5FDE#,  --  忞 (<CJK Ideograph>)
      2666   => 16#5FE1#,  --  忡 (<CJK Ideograph>)
      2667   => 16#5FE2#,  --  忢 (<CJK Ideograph>)
      2668   => 16#5FE8#,  --  忨 (<CJK Ideograph>)
      2669   => 16#5FE9#,  --  忩 (<CJK Ideograph>)
      2670   => 16#5FEA#,  --  忪 (<CJK Ideograph>)
      2671   => 16#5FEC#,  --  忬 (<CJK Ideograph>)
      2672   => 16#5FED#,  --  忭 (<CJK Ideograph>)
      2673   => 16#5FEE#,  --  忮 (<CJK Ideograph>)
      2674   => 16#5FEF#,  --  忯 (<CJK Ideograph>)
      2675   => 16#5FF2#,  --  忲 (<CJK Ideograph>)
      2676   => 16#5FF3#,  --  忳 (<CJK Ideograph>)
      2677   => 16#5FF6#,  --  忶 (<CJK Ideograph>)
      2678   => 16#5FFA#,  --  忺 (<CJK Ideograph>)
      2679   => 16#5FFC#,  --  忼 (<CJK Ideograph>)
      2680   => 16#6007#,  --  怇 (<CJK Ideograph>)
      2681   => 16#600A#,  --  怊 (<CJK Ideograph>)
      2682   => 16#600D#,  --  怍 (<CJK Ideograph>)
      2683   => 16#6013#,  --  怓 (<CJK Ideograph>)
      2684   => 16#6014#,  --  怔 (<CJK Ideograph>)
      2685   => 16#6017#,  --  怗 (<CJK Ideograph>)
      2686   => 16#6018#,  --  怘 (<CJK Ideograph>)
      2687   => 16#601A#,  --  怚 (<CJK Ideograph>)
      2688   => 16#601F#,  --  怟 (<CJK Ideograph>)
      2689   => 16#6024#,  --  怤 (<CJK Ideograph>)
      2690   => 16#602D#,  --  怭 (<CJK Ideograph>)
      2691   => 16#6033#,  --  怳 (<CJK Ideograph>)
      2692   => 16#6035#,  --  怵 (<CJK Ideograph>)
      2693   => 16#6040#,  --  恀 (<CJK Ideograph>)
      2694   => 16#6047#,  --  恇 (<CJK Ideograph>)
      2695   => 16#6048#,  --  恈 (<CJK Ideograph>)
      2696   => 16#6049#,  --  恉 (<CJK Ideograph>)
      2697   => 16#604C#,  --  恌 (<CJK Ideograph>)
      2698   => 16#6051#,  --  恑 (<CJK Ideograph>)
      2699   => 16#6054#,  --  恔 (<CJK Ideograph>)
      2700   => 16#6056#,  --  恖 (<CJK Ideograph>)
      2701   => 16#6057#,  --  恗 (<CJK Ideograph>)
      2702   => 16#605D#,  --  恝 (<CJK Ideograph>)
      2703   => 16#6061#,  --  恡 (<CJK Ideograph>)
      2704   => 16#6067#,  --  恧 (<CJK Ideograph>)
      2705   => 16#6071#,  --  恱 (<CJK Ideograph>)
      2706   => 16#607E#,  --  恾 (<CJK Ideograph>)
      2707   => 16#607F#,  --  恿 (<CJK Ideograph>)
      2708   => 16#6082#,  --  悂 (<CJK Ideograph>)
      2709   => 16#6086#,  --  悆 (<CJK Ideograph>)
      2710   => 16#6088#,  --  悈 (<CJK Ideograph>)
      2711   => 16#608A#,  --  悊 (<CJK Ideograph>)
      2712   => 16#608E#,  --  悎 (<CJK Ideograph>)
      2713   => 16#6091#,  --  悑 (<CJK Ideograph>)
      2714   => 16#6093#,  --  悓 (<CJK Ideograph>)
      2715   => 16#6095#,  --  悕 (<CJK Ideograph>)
      2716   => 16#6098#,  --  悘 (<CJK Ideograph>)
      2717   => 16#609D#,  --  悝 (<CJK Ideograph>)
      2718   => 16#609E#,  --  悞 (<CJK Ideograph>)
      2719   => 16#60A2#,  --  悢 (<CJK Ideograph>)
      2720   => 16#60A4#,  --  悤 (<CJK Ideograph>)
      2721   => 16#60A5#,  --  悥 (<CJK Ideograph>)
      2722   => 16#60A8#,  --  您 (<CJK Ideograph>)
      2723   => 16#60B0#,  --  悰 (<CJK Ideograph>)
      2724   => 16#60B1#,  --  悱 (<CJK Ideograph>)
      2725   => 16#60B7#,  --  悷 (<CJK Ideograph>)
      2726   => 16#60BB#,  --  悻 (<CJK Ideograph>)
      2727   => 16#60BE#,  --  悾 (<CJK Ideograph>)
      2728   => 16#60C2#,  --  惂 (<CJK Ideograph>)
      2729   => 16#60C4#,  --  惄 (<CJK Ideograph>)
      2730   => 16#60C8#,  --  惈 (<CJK Ideograph>)
      2731   => 16#60C9#,  --  惉 (<CJK Ideograph>)
      2732   => 16#60CA#,  --  惊 (<CJK Ideograph>)
      2733   => 16#60CB#,  --  惋 (<CJK Ideograph>)
      2734   => 16#60CE#,  --  惎 (<CJK Ideograph>)
      2735   => 16#60CF#,  --  惏 (<CJK Ideograph>)
      2736   => 16#60D4#,  --  惔 (<CJK Ideograph>)
      2737   => 16#60D5#,  --  惕 (<CJK Ideograph>)
      2738   => 16#60D9#,  --  惙 (<CJK Ideograph>)
      2739   => 16#60DB#,  --  惛 (<CJK Ideograph>)
      2740   => 16#60DD#,  --  惝 (<CJK Ideograph>)
      2741   => 16#60DE#,  --  惞 (<CJK Ideograph>)
      2742   => 16#60E2#,  --  惢 (<CJK Ideograph>)
      2743   => 16#60E5#,  --  惥 (<CJK Ideograph>)
      2744   => 16#60F2#,  --  惲 (<CJK Ideograph>)
      2745   => 16#60F5#,  --  惵 (<CJK Ideograph>)
      2746   => 16#60F8#,  --  惸 (<CJK Ideograph>)
      2747   => 16#60FC#,  --  惼 (<CJK Ideograph>)
      2748   => 16#60FD#,  --  惽 (<CJK Ideograph>)
      2749   => 16#6102#,  --  愂 (<CJK Ideograph>)
      2750   => 16#6107#,  --  愇 (<CJK Ideograph>)
      2751   => 16#610A#,  --  愊 (<CJK Ideograph>)
      2752   => 16#610C#,  --  愌 (<CJK Ideograph>)
      2753   => 16#6110#,  --  愐 (<CJK Ideograph>)
      2754   => 16#6111#,  --  愑 (<CJK Ideograph>)
      2755   => 16#6112#,  --  愒 (<CJK Ideograph>)
      2756   => 16#6113#,  --  愓 (<CJK Ideograph>)
      2757   => 16#6114#,  --  愔 (<CJK Ideograph>)
      2758   => 16#6116#,  --  愖 (<CJK Ideograph>)
      2759   => 16#6117#,  --  愗 (<CJK Ideograph>)
      2760   => 16#6119#,  --  愙 (<CJK Ideograph>)
      2761   => 16#611C#,  --  愜 (<CJK Ideograph>)
      2762   => 16#611E#,  --  愞 (<CJK Ideograph>)
      2763   => 16#6122#,  --  愢 (<CJK Ideograph>)
      2764   => 16#612A#,  --  愪 (<CJK Ideograph>)
      2765   => 16#612B#,  --  愫 (<CJK Ideograph>)
      2766   => 16#6130#,  --  愰 (<CJK Ideograph>)
      2767   => 16#6131#,  --  愱 (<CJK Ideograph>)
      2768   => 16#6135#,  --  愵 (<CJK Ideograph>)
      2769   => 16#6136#,  --  愶 (<CJK Ideograph>)
      2770   => 16#6137#,  --  愷 (<CJK Ideograph>)
      2771   => 16#6139#,  --  愹 (<CJK Ideograph>)
      2772   => 16#6141#,  --  慁 (<CJK Ideograph>)
      2773   => 16#6145#,  --  慅 (<CJK Ideograph>)
      2774   => 16#6146#,  --  慆 (<CJK Ideograph>)
      2775   => 16#6149#,  --  慉 (<CJK Ideograph>)
      2776   => 16#615E#,  --  慞 (<CJK Ideograph>)
      2777   => 16#6160#,  --  慠 (<CJK Ideograph>)
      2778   => 16#616C#,  --  慬 (<CJK Ideograph>)
      2779   => 16#6172#,  --  慲 (<CJK Ideograph>)
      2780   => 16#6178#,  --  慸 (<CJK Ideograph>)
      2781   => 16#617B#,  --  慻 (<CJK Ideograph>)
      2782   => 16#617C#,  --  慼 (<CJK Ideograph>)
      2783   => 16#617F#,  --  慿 (<CJK Ideograph>)
      2784   => 16#6180#,  --  憀 (<CJK Ideograph>)
      2785   => 16#6181#,  --  憁 (<CJK Ideograph>)
      2786   => 16#6183#,  --  憃 (<CJK Ideograph>)
      2787   => 16#6184#,  --  憄 (<CJK Ideograph>)
      2788   => 16#618B#,  --  憋 (<CJK Ideograph>)
      2789   => 16#618D#,  --  憍 (<CJK Ideograph>)
      2790   => 16#6192#,  --  憒 (<CJK Ideograph>)
      2791   => 16#6193#,  --  憓 (<CJK Ideograph>)
      2792   => 16#6197#,  --  憗 (<CJK Ideograph>)
      2793   => 16#6198#,  --  憘 (<CJK Ideograph>)
      2794   => 16#619C#,  --  憜 (<CJK Ideograph>)
      2795   => 16#619D#,  --  憝 (<CJK Ideograph>)
      2796   => 16#619F#,  --  憟 (<CJK Ideograph>)
      2797   => 16#61A0#,  --  憠 (<CJK Ideograph>)
      2798   => 16#61A5#,  --  憥 (<CJK Ideograph>)
      2799   => 16#61A8#,  --  憨 (<CJK Ideograph>)
      2800   => 16#61AA#,  --  憪 (<CJK Ideograph>)
      2801   => 16#61AD#,  --  憭 (<CJK Ideograph>)
      2802   => 16#61B8#,  --  憸 (<CJK Ideograph>)
      2803   => 16#61B9#,  --  憹 (<CJK Ideograph>)
      2804   => 16#61BC#,  --  憼 (<CJK Ideograph>)
      2805   => 16#61C0#,  --  懀 (<CJK Ideograph>)
      2806   => 16#61C1#,  --  懁 (<CJK Ideograph>)
      2807   => 16#61C2#,  --  懂 (<CJK Ideograph>)
      2808   => 16#61CE#,  --  懎 (<CJK Ideograph>)
      2809   => 16#61CF#,  --  懏 (<CJK Ideograph>)
      2810   => 16#61D5#,  --  懕 (<CJK Ideograph>)
      2811   => 16#61DC#,  --  懜 (<CJK Ideograph>)
      2812   => 16#61DD#,  --  懝 (<CJK Ideograph>)
      2813   => 16#61DE#,  --  懞 (<CJK Ideograph>)
      2814   => 16#61DF#,  --  懟 (<CJK Ideograph>)
      2815   => 16#61E1#,  --  懡 (<CJK Ideograph>)
      2816   => 16#61E2#,  --  懢 (<CJK Ideograph>)
      2817   => 16#61E7#,  --  懧 (<CJK Ideograph>)
      2818   => 16#61E9#,  --  懩 (<CJK Ideograph>)
      2819   => 16#61E5#,  --  懥 (<CJK Ideograph>)
      2820   => 16#61EC#,  --  懬 (<CJK Ideograph>)
      2821   => 16#61ED#,  --  懭 (<CJK Ideograph>)
      2822   => 16#61EF#,  --  懯 (<CJK Ideograph>)
      2823   => 16#6201#,  --  戁 (<CJK Ideograph>)
      2824   => 16#6203#,  --  戃 (<CJK Ideograph>)
      2825   => 16#6204#,  --  戄 (<CJK Ideograph>)
      2826   => 16#6207#,  --  戇 (<CJK Ideograph>)
      2827   => 16#6213#,  --  戓 (<CJK Ideograph>)
      2828   => 16#6215#,  --  戕 (<CJK Ideograph>)
      2829   => 16#621C#,  --  戜 (<CJK Ideograph>)
      2830   => 16#6220#,  --  戠 (<CJK Ideograph>)
      2831   => 16#6222#,  --  戢 (<CJK Ideograph>)
      2832   => 16#6223#,  --  戣 (<CJK Ideograph>)
      2833   => 16#6227#,  --  戧 (<CJK Ideograph>)
      2834   => 16#6229#,  --  戩 (<CJK Ideograph>)
      2835   => 16#622B#,  --  戫 (<CJK Ideograph>)
      2836   => 16#6239#,  --  戹 (<CJK Ideograph>)
      2837   => 16#623D#,  --  戽 (<CJK Ideograph>)
      2838   => 16#6242#,  --  扂 (<CJK Ideograph>)
      2839   => 16#6243#,  --  扃 (<CJK Ideograph>)
      2840   => 16#6244#,  --  扄 (<CJK Ideograph>)
      2841   => 16#6246#,  --  扆 (<CJK Ideograph>)
      2842   => 16#624C#,  --  扌 (<CJK Ideograph>)
      2843   => 16#6250#,  --  扐 (<CJK Ideograph>)
      2844   => 16#6251#,  --  扑 (<CJK Ideograph>)
      2845   => 16#6252#,  --  扒 (<CJK Ideograph>)
      2846   => 16#6254#,  --  扔 (<CJK Ideograph>)
      2847   => 16#6256#,  --  扖 (<CJK Ideograph>)
      2848   => 16#625A#,  --  扚 (<CJK Ideograph>)
      2849   => 16#625C#,  --  扜 (<CJK Ideograph>)
      2850   => 16#6264#,  --  扤 (<CJK Ideograph>)
      2851   => 16#626D#,  --  扭 (<CJK Ideograph>)
      2852   => 16#626F#,  --  扯 (<CJK Ideograph>)
      2853   => 16#6273#,  --  扳 (<CJK Ideograph>)
      2854   => 16#627A#,  --  扺 (<CJK Ideograph>)
      2855   => 16#627D#,  --  扽 (<CJK Ideograph>)
      2856   => 16#628D#,  --  抍 (<CJK Ideograph>)
      2857   => 16#628E#,  --  抎 (<CJK Ideograph>)
      2858   => 16#628F#,  --  抏 (<CJK Ideograph>)
      2859   => 16#6290#,  --  抐 (<CJK Ideograph>)
      2860   => 16#62A6#,  --  抦 (<CJK Ideograph>)
      2861   => 16#62A8#,  --  抨 (<CJK Ideograph>)
      2862   => 16#62B3#,  --  抳 (<CJK Ideograph>)
      2863   => 16#62B6#,  --  抶 (<CJK Ideograph>)
      2864   => 16#62B7#,  --  抷 (<CJK Ideograph>)
      2865   => 16#62BA#,  --  抺 (<CJK Ideograph>)
      2866   => 16#62BE#,  --  抾 (<CJK Ideograph>)
      2867   => 16#62BF#,  --  抿 (<CJK Ideograph>)
      2868   => 16#62C4#,  --  拄 (<CJK Ideograph>)
      2869   => 16#62CE#,  --  拎 (<CJK Ideograph>)
      2870   => 16#62D5#,  --  拕 (<CJK Ideograph>)
      2871   => 16#62D6#,  --  拖 (<CJK Ideograph>)
      2872   => 16#62DA#,  --  拚 (<CJK Ideograph>)
      2873   => 16#62EA#,  --  拪 (<CJK Ideograph>)
      2874   => 16#62F2#,  --  拲 (<CJK Ideograph>)
      2875   => 16#62F4#,  --  拴 (<CJK Ideograph>)
      2876   => 16#62FC#,  --  拼 (<CJK Ideograph>)
      2877   => 16#62FD#,  --  拽 (<CJK Ideograph>)
      2878   => 16#6303#,  --  挃 (<CJK Ideograph>)
      2879   => 16#6304#,  --  挄 (<CJK Ideograph>)
      2880   => 16#630A#,  --  挊 (<CJK Ideograph>)
      2881   => 16#630B#,  --  挋 (<CJK Ideograph>)
      2882   => 16#630D#,  --  挍 (<CJK Ideograph>)
      2883   => 16#6310#,  --  挐 (<CJK Ideograph>)
      2884   => 16#6313#,  --  挓 (<CJK Ideograph>)
      2885   => 16#6316#,  --  挖 (<CJK Ideograph>)
      2886   => 16#6318#,  --  挘 (<CJK Ideograph>)
      2887   => 16#6329#,  --  挩 (<CJK Ideograph>)
      2888   => 16#632A#,  --  挪 (<CJK Ideograph>)
      2889   => 16#632D#,  --  挭 (<CJK Ideograph>)
      2890   => 16#6335#,  --  挵 (<CJK Ideograph>)
      2891   => 16#6336#,  --  挶 (<CJK Ideograph>)
      2892   => 16#6339#,  --  挹 (<CJK Ideograph>)
      2893   => 16#633C#,  --  挼 (<CJK Ideograph>)
      2894   => 16#6341#,  --  捁 (<CJK Ideograph>)
      2895   => 16#6342#,  --  捂 (<CJK Ideograph>)
      2896   => 16#6343#,  --  捃 (<CJK Ideograph>)
      2897   => 16#6344#,  --  捄 (<CJK Ideograph>)
      2898   => 16#6346#,  --  捆 (<CJK Ideograph>)
      2899   => 16#634A#,  --  捊 (<CJK Ideograph>)
      2900   => 16#634B#,  --  捋 (<CJK Ideograph>)
      2901   => 16#634E#,  --  捎 (<CJK Ideograph>)
      2902   => 16#6352#,  --  捒 (<CJK Ideograph>)
      2903   => 16#6353#,  --  捓 (<CJK Ideograph>)
      2904   => 16#6354#,  --  捔 (<CJK Ideograph>)
      2905   => 16#6358#,  --  捘 (<CJK Ideograph>)
      2906   => 16#635B#,  --  捛 (<CJK Ideograph>)
      2907   => 16#6365#,  --  捥 (<CJK Ideograph>)
      2908   => 16#6366#,  --  捦 (<CJK Ideograph>)
      2909   => 16#636C#,  --  捬 (<CJK Ideograph>)
      2910   => 16#636D#,  --  捭 (<CJK Ideograph>)
      2911   => 16#6371#,  --  捱 (<CJK Ideograph>)
      2912   => 16#6374#,  --  捴 (<CJK Ideograph>)
      2913   => 16#6375#,  --  捵 (<CJK Ideograph>)
      2914   => 16#6378#,  --  捸 (<CJK Ideograph>)
      2915   => 16#637C#,  --  捼 (<CJK Ideograph>)
      2916   => 16#637D#,  --  捽 (<CJK Ideograph>)
      2917   => 16#637F#,  --  捿 (<CJK Ideograph>)
      2918   => 16#6382#,  --  掂 (<CJK Ideograph>)
      2919   => 16#6384#,  --  掄 (<CJK Ideograph>)
      2920   => 16#6387#,  --  掇 (<CJK Ideograph>)
      2921   => 16#638A#,  --  掊 (<CJK Ideograph>)
      2922   => 16#6390#,  --  掐 (<CJK Ideograph>)
      2923   => 16#6394#,  --  掔 (<CJK Ideograph>)
      2924   => 16#6395#,  --  掕 (<CJK Ideograph>)
      2925   => 16#6399#,  --  掙 (<CJK Ideograph>)
      2926   => 16#639A#,  --  掚 (<CJK Ideograph>)
      2927   => 16#639E#,  --  掞 (<CJK Ideograph>)
      2928   => 16#63A4#,  --  掤 (<CJK Ideograph>)
      2929   => 16#63A6#,  --  掦 (<CJK Ideograph>)
      2930   => 16#63AD#,  --  掭 (<CJK Ideograph>)
      2931   => 16#63AE#,  --  掮 (<CJK Ideograph>)
      2932   => 16#63AF#,  --  掯 (<CJK Ideograph>)
      2933   => 16#63BD#,  --  掽 (<CJK Ideograph>)
      2934   => 16#63C1#,  --  揁 (<CJK Ideograph>)
      2935   => 16#63C5#,  --  揅 (<CJK Ideograph>)
      2936   => 16#63C8#,  --  揈 (<CJK Ideograph>)
      2937   => 16#63CE#,  --  揎 (<CJK Ideograph>)
      2938   => 16#63D1#,  --  揑 (<CJK Ideograph>)
      2939   => 16#63D3#,  --  揓 (<CJK Ideograph>)
      2940   => 16#63D4#,  --  揔 (<CJK Ideograph>)
      2941   => 16#63D5#,  --  揕 (<CJK Ideograph>)
      2942   => 16#63DC#,  --  揜 (<CJK Ideograph>)
      2943   => 16#63E0#,  --  揠 (<CJK Ideograph>)
      2944   => 16#63E5#,  --  揥 (<CJK Ideograph>)
      2945   => 16#63EA#,  --  揪 (<CJK Ideograph>)
      2946   => 16#63EC#,  --  揬 (<CJK Ideograph>)
      2947   => 16#63F2#,  --  揲 (<CJK Ideograph>)
      2948   => 16#63F3#,  --  揳 (<CJK Ideograph>)
      2949   => 16#63F5#,  --  揵 (<CJK Ideograph>)
      2950   => 16#63F8#,  --  揸 (<CJK Ideograph>)
      2951   => 16#63F9#,  --  揹 (<CJK Ideograph>)
      2952   => 16#6409#,  --  搉 (<CJK Ideograph>)
      2953   => 16#640A#,  --  搊 (<CJK Ideograph>)
      2954   => 16#6410#,  --  搐 (<CJK Ideograph>)
      2955   => 16#6412#,  --  搒 (<CJK Ideograph>)
      2956   => 16#6414#,  --  搔 (<CJK Ideograph>)
      2957   => 16#6418#,  --  搘 (<CJK Ideograph>)
      2958   => 16#641E#,  --  搞 (<CJK Ideograph>)
      2959   => 16#6420#,  --  搠 (<CJK Ideograph>)
      2960   => 16#6422#,  --  搢 (<CJK Ideograph>)
      2961   => 16#6424#,  --  搤 (<CJK Ideograph>)
      2962   => 16#6425#,  --  搥 (<CJK Ideograph>)
      2963   => 16#6429#,  --  搩 (<CJK Ideograph>)
      2964   => 16#642A#,  --  搪 (<CJK Ideograph>)
      2965   => 16#642F#,  --  搯 (<CJK Ideograph>)
      2966   => 16#6430#,  --  搰 (<CJK Ideograph>)
      2967   => 16#6435#,  --  搵 (<CJK Ideograph>)
      2968   => 16#643D#,  --  搽 (<CJK Ideograph>)
      2969   => 16#643F#,  --  搿 (<CJK Ideograph>)
      2970   => 16#644B#,  --  摋 (<CJK Ideograph>)
      2971   => 16#644F#,  --  摏 (<CJK Ideograph>)
      2972   => 16#6451#,  --  摑 (<CJK Ideograph>)
      2973   => 16#6452#,  --  摒 (<CJK Ideograph>)
      2974   => 16#6453#,  --  摓 (<CJK Ideograph>)
      2975   => 16#6454#,  --  摔 (<CJK Ideograph>)
      2976   => 16#645A#,  --  摚 (<CJK Ideograph>)
      2977   => 16#645B#,  --  摛 (<CJK Ideograph>)
      2978   => 16#645C#,  --  摜 (<CJK Ideograph>)
      2979   => 16#645D#,  --  摝 (<CJK Ideograph>)
      2980   => 16#645F#,  --  摟 (<CJK Ideograph>)
      2981   => 16#6460#,  --  摠 (<CJK Ideograph>)
      2982   => 16#6461#,  --  摡 (<CJK Ideograph>)
      2983   => 16#6463#,  --  摣 (<CJK Ideograph>)
      2984   => 16#646D#,  --  摭 (<CJK Ideograph>)
      2985   => 16#6473#,  --  摳 (<CJK Ideograph>)
      2986   => 16#6474#,  --  摴 (<CJK Ideograph>)
      2987   => 16#647B#,  --  摻 (<CJK Ideograph>)
      2988   => 16#647D#,  --  摽 (<CJK Ideograph>)
      2989   => 16#6485#,  --  撅 (<CJK Ideograph>)
      2990   => 16#6487#,  --  撇 (<CJK Ideograph>)
      2991   => 16#648F#,  --  撏 (<CJK Ideograph>)
      2992   => 16#6490#,  --  撐 (<CJK Ideograph>)
      2993   => 16#6491#,  --  撑 (<CJK Ideograph>)
      2994   => 16#6498#,  --  撘 (<CJK Ideograph>)
      2995   => 16#6499#,  --  撙 (<CJK Ideograph>)
      2996   => 16#649B#,  --  撛 (<CJK Ideograph>)
      2997   => 16#649D#,  --  撝 (<CJK Ideograph>)
      2998   => 16#649F#,  --  撟 (<CJK Ideograph>)
      2999   => 16#64A1#,  --  撡 (<CJK Ideograph>)
      3000   => 16#64A3#,  --  撣 (<CJK Ideograph>)
      3001   => 16#64A6#,  --  撦 (<CJK Ideograph>)
      3002   => 16#64A8#,  --  撨 (<CJK Ideograph>)
      3003   => 16#64AC#,  --  撬 (<CJK Ideograph>)
      3004   => 16#64B3#,  --  撳 (<CJK Ideograph>)
      3005   => 16#64BD#,  --  撽 (<CJK Ideograph>)
      3006   => 16#64BE#,  --  撾 (<CJK Ideograph>)
      3007   => 16#64BF#,  --  撿 (<CJK Ideograph>)
      3008   => 16#64C4#,  --  擄 (<CJK Ideograph>)
      3009   => 16#64C9#,  --  擉 (<CJK Ideograph>)
      3010   => 16#64CA#,  --  擊 (<CJK Ideograph>)
      3011   => 16#64CB#,  --  擋 (<CJK Ideograph>)
      3012   => 16#64CC#,  --  擌 (<CJK Ideograph>)
      3013   => 16#64CE#,  --  擎 (<CJK Ideograph>)
      3014   => 16#64D0#,  --  擐 (<CJK Ideograph>)
      3015   => 16#64D1#,  --  擑 (<CJK Ideograph>)
      3016   => 16#64D5#,  --  擕 (<CJK Ideograph>)
      3017   => 16#64D7#,  --  擗 (<CJK Ideograph>)
      3018   => 16#64E4#,  --  擤 (<CJK Ideograph>)
      3019   => 16#64E5#,  --  擥 (<CJK Ideograph>)
      3020   => 16#64E9#,  --  擩 (<CJK Ideograph>)
      3021   => 16#64EA#,  --  擪 (<CJK Ideograph>)
      3022   => 16#64ED#,  --  擭 (<CJK Ideograph>)
      3023   => 16#64F0#,  --  擰 (<CJK Ideograph>)
      3024   => 16#64F5#,  --  擵 (<CJK Ideograph>)
      3025   => 16#64F7#,  --  擷 (<CJK Ideograph>)
      3026   => 16#64FB#,  --  擻 (<CJK Ideograph>)
      3027   => 16#64FF#,  --  擿 (<CJK Ideograph>)
      3028   => 16#6501#,  --  攁 (<CJK Ideograph>)
      3029   => 16#6504#,  --  攄 (<CJK Ideograph>)
      3030   => 16#6508#,  --  攈 (<CJK Ideograph>)
      3031   => 16#6509#,  --  攉 (<CJK Ideograph>)
      3032   => 16#650A#,  --  攊 (<CJK Ideograph>)
      3033   => 16#650F#,  --  攏 (<CJK Ideograph>)
      3034   => 16#6513#,  --  攓 (<CJK Ideograph>)
      3035   => 16#6514#,  --  攔 (<CJK Ideograph>)
      3036   => 16#6516#,  --  攖 (<CJK Ideograph>)
      3037   => 16#6519#,  --  攙 (<CJK Ideograph>)
      3038   => 16#651B#,  --  攛 (<CJK Ideograph>)
      3039   => 16#651E#,  --  攞 (<CJK Ideograph>)
      3040   => 16#651F#,  --  攟 (<CJK Ideograph>)
      3041   => 16#6522#,  --  攢 (<CJK Ideograph>)
      3042   => 16#6526#,  --  攦 (<CJK Ideograph>)
      3043   => 16#6529#,  --  攩 (<CJK Ideograph>)
      3044   => 16#652E#,  --  攮 (<CJK Ideograph>)
      3045   => 16#6531#,  --  攱 (<CJK Ideograph>)
      3046   => 16#653A#,  --  攺 (<CJK Ideograph>)
      3047   => 16#653C#,  --  攼 (<CJK Ideograph>)
      3048   => 16#653D#,  --  攽 (<CJK Ideograph>)
      3049   => 16#6543#,  --  敃 (<CJK Ideograph>)
      3050   => 16#6547#,  --  敇 (<CJK Ideograph>)
      3051   => 16#6549#,  --  敉 (<CJK Ideograph>)
      3052   => 16#6550#,  --  敐 (<CJK Ideograph>)
      3053   => 16#6552#,  --  敒 (<CJK Ideograph>)
      3054   => 16#6554#,  --  敔 (<CJK Ideograph>)
      3055   => 16#655F#,  --  敟 (<CJK Ideograph>)
      3056   => 16#6560#,  --  敠 (<CJK Ideograph>)
      3057   => 16#6567#,  --  敧 (<CJK Ideograph>)
      3058   => 16#656B#,  --  敫 (<CJK Ideograph>)
      3059   => 16#657A#,  --  敺 (<CJK Ideograph>)
      3060   => 16#657D#,  --  敽 (<CJK Ideograph>)
      3061   => 16#6581#,  --  斁 (<CJK Ideograph>)
      3062   => 16#6585#,  --  斅 (<CJK Ideograph>)
      3063   => 16#658A#,  --  斊 (<CJK Ideograph>)
      3064   => 16#6592#,  --  斒 (<CJK Ideograph>)
      3065   => 16#6595#,  --  斕 (<CJK Ideograph>)
      3066   => 16#6598#,  --  斘 (<CJK Ideograph>)
      3067   => 16#659D#,  --  斝 (<CJK Ideograph>)
      3068   => 16#65A0#,  --  斠 (<CJK Ideograph>)
      3069   => 16#65A3#,  --  斣 (<CJK Ideograph>)
      3070   => 16#65A6#,  --  斦 (<CJK Ideograph>)
      3071   => 16#65AE#,  --  斮 (<CJK Ideograph>)
      3072   => 16#65B2#,  --  斲 (<CJK Ideograph>)
      3073   => 16#65B3#,  --  斳 (<CJK Ideograph>)
      3074   => 16#65B4#,  --  斴 (<CJK Ideograph>)
      3075   => 16#65BF#,  --  斿 (<CJK Ideograph>)
      3076   => 16#65C2#,  --  旂 (<CJK Ideograph>)
      3077   => 16#65C8#,  --  旈 (<CJK Ideograph>)
      3078   => 16#65C9#,  --  旉 (<CJK Ideograph>)
      3079   => 16#65CE#,  --  旎 (<CJK Ideograph>)
      3080   => 16#65D0#,  --  旐 (<CJK Ideograph>)
      3081   => 16#65D4#,  --  旔 (<CJK Ideograph>)
      3082   => 16#65D6#,  --  旖 (<CJK Ideograph>)
      3083   => 16#65D8#,  --  旘 (<CJK Ideograph>)
      3084   => 16#65DF#,  --  旟 (<CJK Ideograph>)
      3085   => 16#65F0#,  --  旰 (<CJK Ideograph>)
      3086   => 16#65F2#,  --  旲 (<CJK Ideograph>)
      3087   => 16#65F4#,  --  旴 (<CJK Ideograph>)
      3088   => 16#65F5#,  --  旵 (<CJK Ideograph>)
      3089   => 16#65F9#,  --  旹 (<CJK Ideograph>)
      3090   => 16#65FE#,  --  旾 (<CJK Ideograph>)
      3091   => 16#65FF#,  --  旿 (<CJK Ideograph>)
      3092   => 16#6600#,  --  昀 (<CJK Ideograph>)
      3093   => 16#6604#,  --  昄 (<CJK Ideograph>)
      3094   => 16#6608#,  --  昈 (<CJK Ideograph>)
      3095   => 16#6609#,  --  昉 (<CJK Ideograph>)
      3096   => 16#660D#,  --  昍 (<CJK Ideograph>)
      3097   => 16#6611#,  --  昑 (<CJK Ideograph>)
      3098   => 16#6612#,  --  昒 (<CJK Ideograph>)
      3099   => 16#6615#,  --  昕 (<CJK Ideograph>)
      3100   => 16#6616#,  --  昖 (<CJK Ideograph>)
      3101   => 16#661D#,  --  昝 (<CJK Ideograph>)
      3102   => 16#661E#,  --  昞 (<CJK Ideograph>)
      3103   => 16#6621#,  --  昡 (<CJK Ideograph>)
      3104   => 16#6622#,  --  昢 (<CJK Ideograph>)
      3105   => 16#6623#,  --  昣 (<CJK Ideograph>)
      3106   => 16#6624#,  --  昤 (<CJK Ideograph>)
      3107   => 16#6626#,  --  昦 (<CJK Ideograph>)
      3108   => 16#6629#,  --  昩 (<CJK Ideograph>)
      3109   => 16#662A#,  --  昪 (<CJK Ideograph>)
      3110   => 16#662B#,  --  昫 (<CJK Ideograph>)
      3111   => 16#662C#,  --  昬 (<CJK Ideograph>)
      3112   => 16#662E#,  --  昮 (<CJK Ideograph>)
      3113   => 16#6630#,  --  昰 (<CJK Ideograph>)
      3114   => 16#6631#,  --  昱 (<CJK Ideograph>)
      3115   => 16#6633#,  --  昳 (<CJK Ideograph>)
      3116   => 16#6639#,  --  昹 (<CJK Ideograph>)
      3117   => 16#6637#,  --  昷 (<CJK Ideograph>)
      3118   => 16#6640#,  --  晀 (<CJK Ideograph>)
      3119   => 16#6645#,  --  晅 (<CJK Ideograph>)
      3120   => 16#6646#,  --  晆 (<CJK Ideograph>)
      3121   => 16#664A#,  --  晊 (<CJK Ideograph>)
      3122   => 16#664C#,  --  晌 (<CJK Ideograph>)
      3123   => 16#6651#,  --  晑 (<CJK Ideograph>)
      3124   => 16#664E#,  --  晎 (<CJK Ideograph>)
      3125   => 16#6657#,  --  晗 (<CJK Ideograph>)
      3126   => 16#6658#,  --  晘 (<CJK Ideograph>)
      3127   => 16#6659#,  --  晙 (<CJK Ideograph>)
      3128   => 16#665B#,  --  晛 (<CJK Ideograph>)
      3129   => 16#665C#,  --  晜 (<CJK Ideograph>)
      3130   => 16#6660#,  --  晠 (<CJK Ideograph>)
      3131   => 16#6661#,  --  晡 (<CJK Ideograph>)
      3132   => 16#66FB#,  --  曻 (<CJK Ideograph>)
      3133   => 16#666A#,  --  晪 (<CJK Ideograph>)
      3134   => 16#666B#,  --  晫 (<CJK Ideograph>)
      3135   => 16#666C#,  --  晬 (<CJK Ideograph>)
      3136   => 16#667E#,  --  晾 (<CJK Ideograph>)
      3137   => 16#6673#,  --  晳 (<CJK Ideograph>)
      3138   => 16#6675#,  --  晵 (<CJK Ideograph>)
      3139   => 16#667F#,  --  晿 (<CJK Ideograph>)
      3140   => 16#6677#,  --  晷 (<CJK Ideograph>)
      3141   => 16#6678#,  --  晸 (<CJK Ideograph>)
      3142   => 16#6679#,  --  晹 (<CJK Ideograph>)
      3143   => 16#667B#,  --  晻 (<CJK Ideograph>)
      3144   => 16#6680#,  --  暀 (<CJK Ideograph>)
      3145   => 16#667C#,  --  晼 (<CJK Ideograph>)
      3146   => 16#668B#,  --  暋 (<CJK Ideograph>)
      3147   => 16#668C#,  --  暌 (<CJK Ideograph>)
      3148   => 16#668D#,  --  暍 (<CJK Ideograph>)
      3149   => 16#6690#,  --  暐 (<CJK Ideograph>)
      3150   => 16#6692#,  --  暒 (<CJK Ideograph>)
      3151   => 16#6699#,  --  暙 (<CJK Ideograph>)
      3152   => 16#669A#,  --  暚 (<CJK Ideograph>)
      3153   => 16#669B#,  --  暛 (<CJK Ideograph>)
      3154   => 16#669C#,  --  暜 (<CJK Ideograph>)
      3155   => 16#669F#,  --  暟 (<CJK Ideograph>)
      3156   => 16#66A0#,  --  暠 (<CJK Ideograph>)
      3157   => 16#66A4#,  --  暤 (<CJK Ideograph>)
      3158   => 16#66AD#,  --  暭 (<CJK Ideograph>)
      3159   => 16#66B1#,  --  暱 (<CJK Ideograph>)
      3160   => 16#66B2#,  --  暲 (<CJK Ideograph>)
      3161   => 16#66B5#,  --  暵 (<CJK Ideograph>)
      3162   => 16#66BB#,  --  暻 (<CJK Ideograph>)
      3163   => 16#66BF#,  --  暿 (<CJK Ideograph>)
      3164   => 16#66C0#,  --  曀 (<CJK Ideograph>)
      3165   => 16#66C2#,  --  曂 (<CJK Ideograph>)
      3166   => 16#66C3#,  --  曃 (<CJK Ideograph>)
      3167   => 16#66C8#,  --  曈 (<CJK Ideograph>)
      3168   => 16#66CC#,  --  曌 (<CJK Ideograph>)
      3169   => 16#66CE#,  --  曎 (<CJK Ideograph>)
      3170   => 16#66CF#,  --  曏 (<CJK Ideograph>)
      3171   => 16#66D4#,  --  曔 (<CJK Ideograph>)
      3172   => 16#66DB#,  --  曛 (<CJK Ideograph>)
      3173   => 16#66DF#,  --  曟 (<CJK Ideograph>)
      3174   => 16#66E8#,  --  曨 (<CJK Ideograph>)
      3175   => 16#66EB#,  --  曫 (<CJK Ideograph>)
      3176   => 16#66EC#,  --  曬 (<CJK Ideograph>)
      3177   => 16#66EE#,  --  曮 (<CJK Ideograph>)
      3178   => 16#66FA#,  --  曺 (<CJK Ideograph>)
      3179   => 16#6705#,  --  朅 (<CJK Ideograph>)
      3180   => 16#6707#,  --  朇 (<CJK Ideograph>)
      3181   => 16#670E#,  --  朎 (<CJK Ideograph>)
      3182   => 16#6713#,  --  朓 (<CJK Ideograph>)
      3183   => 16#6719#,  --  朙 (<CJK Ideograph>)
      3184   => 16#671C#,  --  朜 (<CJK Ideograph>)
      3185   => 16#6720#,  --  朠 (<CJK Ideograph>)
      3186   => 16#6722#,  --  朢 (<CJK Ideograph>)
      3187   => 16#6733#,  --  朳 (<CJK Ideograph>)
      3188   => 16#673E#,  --  朾 (<CJK Ideograph>)
      3189   => 16#6745#,  --  杅 (<CJK Ideograph>)
      3190   => 16#6747#,  --  杇 (<CJK Ideograph>)
      3191   => 16#6748#,  --  杈 (<CJK Ideograph>)
      3192   => 16#674C#,  --  杌 (<CJK Ideograph>)
      3193   => 16#6754#,  --  杔 (<CJK Ideograph>)
      3194   => 16#6755#,  --  杕 (<CJK Ideograph>)
      3195   => 16#675D#,  --  杝 (<CJK Ideograph>)
      3196   => 16#6766#,  --  杦 (<CJK Ideograph>)
      3197   => 16#676C#,  --  杬 (<CJK Ideograph>)
      3198   => 16#676E#,  --  杮 (<CJK Ideograph>)
      3199   => 16#6774#,  --  杴 (<CJK Ideograph>)
      3200   => 16#6776#,  --  杶 (<CJK Ideograph>)
      3201   => 16#677B#,  --  杻 (<CJK Ideograph>)
      3202   => 16#6781#,  --  极 (<CJK Ideograph>)
      3203   => 16#6784#,  --  构 (<CJK Ideograph>)
      3204   => 16#678E#,  --  枎 (<CJK Ideograph>)
      3205   => 16#678F#,  --  枏 (<CJK Ideograph>)
      3206   => 16#6791#,  --  枑 (<CJK Ideograph>)
      3207   => 16#6793#,  --  枓 (<CJK Ideograph>)
      3208   => 16#6796#,  --  枖 (<CJK Ideograph>)
      3209   => 16#6798#,  --  枘 (<CJK Ideograph>)
      3210   => 16#6799#,  --  枙 (<CJK Ideograph>)
      3211   => 16#679B#,  --  枛 (<CJK Ideograph>)
      3212   => 16#67B0#,  --  枰 (<CJK Ideograph>)
      3213   => 16#67B1#,  --  枱 (<CJK Ideograph>)
      3214   => 16#67B2#,  --  枲 (<CJK Ideograph>)
      3215   => 16#67B5#,  --  枵 (<CJK Ideograph>)
      3216   => 16#67BB#,  --  枻 (<CJK Ideograph>)
      3217   => 16#67BC#,  --  枼 (<CJK Ideograph>)
      3218   => 16#67BD#,  --  枽 (<CJK Ideograph>)
      3219   => 16#67F9#,  --  柹 (<CJK Ideograph>)
      3220   => 16#67C0#,  --  柀 (<CJK Ideograph>)
      3221   => 16#67C2#,  --  柂 (<CJK Ideograph>)
      3222   => 16#67C3#,  --  柃 (<CJK Ideograph>)
      3223   => 16#67C5#,  --  柅 (<CJK Ideograph>)
      3224   => 16#67C8#,  --  柈 (<CJK Ideograph>)
      3225   => 16#67C9#,  --  柉 (<CJK Ideograph>)
      3226   => 16#67D2#,  --  柒 (<CJK Ideograph>)
      3227   => 16#67D7#,  --  柗 (<CJK Ideograph>)
      3228   => 16#67D9#,  --  柙 (<CJK Ideograph>)
      3229   => 16#67DC#,  --  柜 (<CJK Ideograph>)
      3230   => 16#67E1#,  --  柡 (<CJK Ideograph>)
      3231   => 16#67E6#,  --  柦 (<CJK Ideograph>)
      3232   => 16#67F0#,  --  柰 (<CJK Ideograph>)
      3233   => 16#67F2#,  --  柲 (<CJK Ideograph>)
      3234   => 16#67F6#,  --  柶 (<CJK Ideograph>)
      3235   => 16#67F7#,  --  柷 (<CJK Ideograph>)
      3236   => 16#6852#,  --  桒 (<CJK Ideograph>)
      3237   => 16#6814#,  --  栔 (<CJK Ideograph>)
      3238   => 16#6819#,  --  栙 (<CJK Ideograph>)
      3239   => 16#681D#,  --  栝 (<CJK Ideograph>)
      3240   => 16#681F#,  --  栟 (<CJK Ideograph>)
      3241   => 16#6828#,  --  栨 (<CJK Ideograph>)
      3242   => 16#6827#,  --  栧 (<CJK Ideograph>)
      3243   => 16#682C#,  --  栬 (<CJK Ideograph>)
      3244   => 16#682D#,  --  栭 (<CJK Ideograph>)
      3245   => 16#682F#,  --  栯 (<CJK Ideograph>)
      3246   => 16#6830#,  --  栰 (<CJK Ideograph>)
      3247   => 16#6831#,  --  栱 (<CJK Ideograph>)
      3248   => 16#6833#,  --  栳 (<CJK Ideograph>)
      3249   => 16#683B#,  --  栻 (<CJK Ideograph>)
      3250   => 16#683F#,  --  栿 (<CJK Ideograph>)
      3251   => 16#6844#,  --  桄 (<CJK Ideograph>)
      3252   => 16#6845#,  --  桅 (<CJK Ideograph>)
      3253   => 16#684A#,  --  桊 (<CJK Ideograph>)
      3254   => 16#684C#,  --  桌 (<CJK Ideograph>)
      3255   => 16#6855#,  --  桕 (<CJK Ideograph>)
      3256   => 16#6857#,  --  桗 (<CJK Ideograph>)
      3257   => 16#6858#,  --  桘 (<CJK Ideograph>)
      3258   => 16#685B#,  --  桛 (<CJK Ideograph>)
      3259   => 16#686B#,  --  桫 (<CJK Ideograph>)
      3260   => 16#686E#,  --  桮 (<CJK Ideograph>)
      3261   => 16#686F#,  --  桯 (<CJK Ideograph>)
      3262   => 16#6870#,  --  桰 (<CJK Ideograph>)
      3263   => 16#6871#,  --  桱 (<CJK Ideograph>)
      3264   => 16#6872#,  --  桲 (<CJK Ideograph>)
      3265   => 16#6875#,  --  桵 (<CJK Ideograph>)
      3266   => 16#6879#,  --  桹 (<CJK Ideograph>)
      3267   => 16#687A#,  --  桺 (<CJK Ideograph>)
      3268   => 16#687B#,  --  桻 (<CJK Ideograph>)
      3269   => 16#687C#,  --  桼 (<CJK Ideograph>)
      3270   => 16#6882#,  --  梂 (<CJK Ideograph>)
      3271   => 16#6884#,  --  梄 (<CJK Ideograph>)
      3272   => 16#6886#,  --  梆 (<CJK Ideograph>)
      3273   => 16#6888#,  --  梈 (<CJK Ideograph>)
      3274   => 16#6896#,  --  梖 (<CJK Ideograph>)
      3275   => 16#6898#,  --  梘 (<CJK Ideograph>)
      3276   => 16#689A#,  --  梚 (<CJK Ideograph>)
      3277   => 16#689C#,  --  梜 (<CJK Ideograph>)
      3278   => 16#68A1#,  --  梡 (<CJK Ideograph>)
      3279   => 16#68A3#,  --  梣 (<CJK Ideograph>)
      3280   => 16#68A5#,  --  梥 (<CJK Ideograph>)
      3281   => 16#68A9#,  --  梩 (<CJK Ideograph>)
      3282   => 16#68AA#,  --  梪 (<CJK Ideograph>)
      3283   => 16#68AE#,  --  梮 (<CJK Ideograph>)
      3284   => 16#68B2#,  --  梲 (<CJK Ideograph>)
      3285   => 16#68BB#,  --  梻 (<CJK Ideograph>)
      3286   => 16#68C5#,  --  棅 (<CJK Ideograph>)
      3287   => 16#68C8#,  --  棈 (<CJK Ideograph>)
      3288   => 16#68CC#,  --  棌 (<CJK Ideograph>)
      3289   => 16#68CF#,  --  棏 (<CJK Ideograph>)
      3290   => 16#68D0#,  --  棐 (<CJK Ideograph>)
      3291   => 16#68D1#,  --  棑 (<CJK Ideograph>)
      3292   => 16#68D3#,  --  棓 (<CJK Ideograph>)
      3293   => 16#68D6#,  --  棖 (<CJK Ideograph>)
      3294   => 16#68D9#,  --  棙 (<CJK Ideograph>)
      3295   => 16#68DC#,  --  棜 (<CJK Ideograph>)
      3296   => 16#68DD#,  --  棝 (<CJK Ideograph>)
      3297   => 16#68E5#,  --  棥 (<CJK Ideograph>)
      3298   => 16#68E8#,  --  棨 (<CJK Ideograph>)
      3299   => 16#68EA#,  --  棪 (<CJK Ideograph>)
      3300   => 16#68EB#,  --  棫 (<CJK Ideograph>)
      3301   => 16#68EC#,  --  棬 (<CJK Ideograph>)
      3302   => 16#68ED#,  --  棭 (<CJK Ideograph>)
      3303   => 16#68F0#,  --  棰 (<CJK Ideograph>)
      3304   => 16#68F1#,  --  棱 (<CJK Ideograph>)
      3305   => 16#68F5#,  --  棵 (<CJK Ideograph>)
      3306   => 16#68F6#,  --  棶 (<CJK Ideograph>)
      3307   => 16#68FB#,  --  棻 (<CJK Ideograph>)
      3308   => 16#68FC#,  --  棼 (<CJK Ideograph>)
      3309   => 16#68FD#,  --  棽 (<CJK Ideograph>)
      3310   => 16#6906#,  --  椆 (<CJK Ideograph>)
      3311   => 16#6909#,  --  椉 (<CJK Ideograph>)
      3312   => 16#690A#,  --  椊 (<CJK Ideograph>)
      3313   => 16#6910#,  --  椐 (<CJK Ideograph>)
      3314   => 16#6911#,  --  椑 (<CJK Ideograph>)
      3315   => 16#6913#,  --  椓 (<CJK Ideograph>)
      3316   => 16#6916#,  --  椖 (<CJK Ideograph>)
      3317   => 16#6917#,  --  椗 (<CJK Ideograph>)
      3318   => 16#6931#,  --  椱 (<CJK Ideograph>)
      3319   => 16#6933#,  --  椳 (<CJK Ideograph>)
      3320   => 16#6935#,  --  椵 (<CJK Ideograph>)
      3321   => 16#6938#,  --  椸 (<CJK Ideograph>)
      3322   => 16#693B#,  --  椻 (<CJK Ideograph>)
      3323   => 16#6942#,  --  楂 (<CJK Ideograph>)
      3324   => 16#6945#,  --  楅 (<CJK Ideograph>)
      3325   => 16#6949#,  --  楉 (<CJK Ideograph>)
      3326   => 16#694E#,  --  楎 (<CJK Ideograph>)
      3327   => 16#6957#,  --  楗 (<CJK Ideograph>)
      3328   => 16#695B#,  --  楛 (<CJK Ideograph>)
      3329   => 16#6963#,  --  楣 (<CJK Ideograph>)
      3330   => 16#6964#,  --  楤 (<CJK Ideograph>)
      3331   => 16#6965#,  --  楥 (<CJK Ideograph>)
      3332   => 16#6966#,  --  楦 (<CJK Ideograph>)
      3333   => 16#6968#,  --  楨 (<CJK Ideograph>)
      3334   => 16#6969#,  --  楩 (<CJK Ideograph>)
      3335   => 16#696C#,  --  楬 (<CJK Ideograph>)
      3336   => 16#6970#,  --  楰 (<CJK Ideograph>)
      3337   => 16#6971#,  --  楱 (<CJK Ideograph>)
      3338   => 16#6972#,  --  楲 (<CJK Ideograph>)
      3339   => 16#697A#,  --  楺 (<CJK Ideograph>)
      3340   => 16#697B#,  --  楻 (<CJK Ideograph>)
      3341   => 16#697F#,  --  楿 (<CJK Ideograph>)
      3342   => 16#6980#,  --  榀 (<CJK Ideograph>)
      3343   => 16#698D#,  --  榍 (<CJK Ideograph>)
      3344   => 16#6992#,  --  榒 (<CJK Ideograph>)
      3345   => 16#6996#,  --  榖 (<CJK Ideograph>)
      3346   => 16#6998#,  --  榘 (<CJK Ideograph>)
      3347   => 16#69A1#,  --  榡 (<CJK Ideograph>)
      3348   => 16#69A5#,  --  榥 (<CJK Ideograph>)
      3349   => 16#69A6#,  --  榦 (<CJK Ideograph>)
      3350   => 16#69A8#,  --  榨 (<CJK Ideograph>)
      3351   => 16#69AB#,  --  榫 (<CJK Ideograph>)
      3352   => 16#69AD#,  --  榭 (<CJK Ideograph>)
      3353   => 16#69AF#,  --  榯 (<CJK Ideograph>)
      3354   => 16#69B7#,  --  榷 (<CJK Ideograph>)
      3355   => 16#69B8#,  --  榸 (<CJK Ideograph>)
      3356   => 16#69BA#,  --  榺 (<CJK Ideograph>)
      3357   => 16#69BC#,  --  榼 (<CJK Ideograph>)
      3358   => 16#69C5#,  --  槅 (<CJK Ideograph>)
      3359   => 16#69C8#,  --  槈 (<CJK Ideograph>)
      3360   => 16#69D1#,  --  槑 (<CJK Ideograph>)
      3361   => 16#69D6#,  --  槖 (<CJK Ideograph>)
      3362   => 16#69D7#,  --  槗 (<CJK Ideograph>)
      3363   => 16#69E2#,  --  槢 (<CJK Ideograph>)
      3364   => 16#69E5#,  --  槥 (<CJK Ideograph>)
      3365   => 16#69EE#,  --  槮 (<CJK Ideograph>)
      3366   => 16#69EF#,  --  槯 (<CJK Ideograph>)
      3367   => 16#69F1#,  --  槱 (<CJK Ideograph>)
      3368   => 16#69F3#,  --  槳 (<CJK Ideograph>)
      3369   => 16#69F5#,  --  槵 (<CJK Ideograph>)
      3370   => 16#69FE#,  --  槾 (<CJK Ideograph>)
      3371   => 16#6A00#,  --  樀 (<CJK Ideograph>)
      3372   => 16#6A01#,  --  樁 (<CJK Ideograph>)
      3373   => 16#6A03#,  --  樃 (<CJK Ideograph>)
      3374   => 16#6A0F#,  --  樏 (<CJK Ideograph>)
      3375   => 16#6A11#,  --  樑 (<CJK Ideograph>)
      3376   => 16#6A15#,  --  樕 (<CJK Ideograph>)
      3377   => 16#6A1A#,  --  樚 (<CJK Ideograph>)
      3378   => 16#6A1D#,  --  樝 (<CJK Ideograph>)
      3379   => 16#6A20#,  --  樠 (<CJK Ideograph>)
      3380   => 16#6A24#,  --  樤 (<CJK Ideograph>)
      3381   => 16#6A28#,  --  樨 (<CJK Ideograph>)
      3382   => 16#6A30#,  --  樰 (<CJK Ideograph>)
      3383   => 16#6A32#,  --  樲 (<CJK Ideograph>)
      3384   => 16#6A34#,  --  樴 (<CJK Ideograph>)
      3385   => 16#6A37#,  --  樷 (<CJK Ideograph>)
      3386   => 16#6A3B#,  --  樻 (<CJK Ideograph>)
      3387   => 16#6A3E#,  --  樾 (<CJK Ideograph>)
      3388   => 16#6A3F#,  --  樿 (<CJK Ideograph>)
      3389   => 16#6A45#,  --  橅 (<CJK Ideograph>)
      3390   => 16#6A46#,  --  橆 (<CJK Ideograph>)
      3391   => 16#6A49#,  --  橉 (<CJK Ideograph>)
      3392   => 16#6A4A#,  --  橊 (<CJK Ideograph>)
      3393   => 16#6A4E#,  --  橎 (<CJK Ideograph>)
      3394   => 16#6A50#,  --  橐 (<CJK Ideograph>)
      3395   => 16#6A51#,  --  橑 (<CJK Ideograph>)
      3396   => 16#6A52#,  --  橒 (<CJK Ideograph>)
      3397   => 16#6A55#,  --  橕 (<CJK Ideograph>)
      3398   => 16#6A56#,  --  橖 (<CJK Ideograph>)
      3399   => 16#6A5B#,  --  橛 (<CJK Ideograph>)
      3400   => 16#6A64#,  --  橤 (<CJK Ideograph>)
      3401   => 16#6A67#,  --  橧 (<CJK Ideograph>)
      3402   => 16#6A6A#,  --  橪 (<CJK Ideograph>)
      3403   => 16#6A71#,  --  橱 (<CJK Ideograph>)
      3404   => 16#6A73#,  --  橳 (<CJK Ideograph>)
      3405   => 16#6A7E#,  --  橾 (<CJK Ideograph>)
      3406   => 16#6A81#,  --  檁 (<CJK Ideograph>)
      3407   => 16#6A83#,  --  檃 (<CJK Ideograph>)
      3408   => 16#6A86#,  --  檆 (<CJK Ideograph>)
      3409   => 16#6A87#,  --  檇 (<CJK Ideograph>)
      3410   => 16#6A89#,  --  檉 (<CJK Ideograph>)
      3411   => 16#6A8B#,  --  檋 (<CJK Ideograph>)
      3412   => 16#6A91#,  --  檑 (<CJK Ideograph>)
      3413   => 16#6A9B#,  --  檛 (<CJK Ideograph>)
      3414   => 16#6A9D#,  --  檝 (<CJK Ideograph>)
      3415   => 16#6A9E#,  --  檞 (<CJK Ideograph>)
      3416   => 16#6A9F#,  --  檟 (<CJK Ideograph>)
      3417   => 16#6AA5#,  --  檥 (<CJK Ideograph>)
      3418   => 16#6AAB#,  --  檫 (<CJK Ideograph>)
      3419   => 16#6AAF#,  --  檯 (<CJK Ideograph>)
      3420   => 16#6AB0#,  --  檰 (<CJK Ideograph>)
      3421   => 16#6AB1#,  --  檱 (<CJK Ideograph>)
      3422   => 16#6AB4#,  --  檴 (<CJK Ideograph>)
      3423   => 16#6ABD#,  --  檽 (<CJK Ideograph>)
      3424   => 16#6ABE#,  --  檾 (<CJK Ideograph>)
      3425   => 16#6ABF#,  --  檿 (<CJK Ideograph>)
      3426   => 16#6AC6#,  --  櫆 (<CJK Ideograph>)
      3427   => 16#6AC9#,  --  櫉 (<CJK Ideograph>)
      3428   => 16#6AC8#,  --  櫈 (<CJK Ideograph>)
      3429   => 16#6ACC#,  --  櫌 (<CJK Ideograph>)
      3430   => 16#6AD0#,  --  櫐 (<CJK Ideograph>)
      3431   => 16#6AD4#,  --  櫔 (<CJK Ideograph>)
      3432   => 16#6AD5#,  --  櫕 (<CJK Ideograph>)
      3433   => 16#6AD6#,  --  櫖 (<CJK Ideograph>)
      3434   => 16#6ADC#,  --  櫜 (<CJK Ideograph>)
      3435   => 16#6ADD#,  --  櫝 (<CJK Ideograph>)
      3436   => 16#6AE4#,  --  櫤 (<CJK Ideograph>)
      3437   => 16#6AE7#,  --  櫧 (<CJK Ideograph>)
      3438   => 16#6AEC#,  --  櫬 (<CJK Ideograph>)
      3439   => 16#6AF0#,  --  櫰 (<CJK Ideograph>)
      3440   => 16#6AF1#,  --  櫱 (<CJK Ideograph>)
      3441   => 16#6AF2#,  --  櫲 (<CJK Ideograph>)
      3442   => 16#6AFC#,  --  櫼 (<CJK Ideograph>)
      3443   => 16#6AFD#,  --  櫽 (<CJK Ideograph>)
      3444   => 16#6B02#,  --  欂 (<CJK Ideograph>)
      3445   => 16#6B03#,  --  欃 (<CJK Ideograph>)
      3446   => 16#6B06#,  --  欆 (<CJK Ideograph>)
      3447   => 16#6B07#,  --  欇 (<CJK Ideograph>)
      3448   => 16#6B09#,  --  欉 (<CJK Ideograph>)
      3449   => 16#6B0F#,  --  欏 (<CJK Ideograph>)
      3450   => 16#6B10#,  --  欐 (<CJK Ideograph>)
      3451   => 16#6B11#,  --  欑 (<CJK Ideograph>)
      3452   => 16#6B17#,  --  欗 (<CJK Ideograph>)
      3453   => 16#6B1B#,  --  欛 (<CJK Ideograph>)
      3454   => 16#6B1E#,  --  欞 (<CJK Ideograph>)
      3455   => 16#6B24#,  --  欤 (<CJK Ideograph>)
      3456   => 16#6B28#,  --  欨 (<CJK Ideograph>)
      3457   => 16#6B2B#,  --  欫 (<CJK Ideograph>)
      3458   => 16#6B2C#,  --  欬 (<CJK Ideograph>)
      3459   => 16#6B2F#,  --  欯 (<CJK Ideograph>)
      3460   => 16#6B35#,  --  欵 (<CJK Ideograph>)
      3461   => 16#6B36#,  --  欶 (<CJK Ideograph>)
      3462   => 16#6B3B#,  --  欻 (<CJK Ideograph>)
      3463   => 16#6B3F#,  --  欿 (<CJK Ideograph>)
      3464   => 16#6B46#,  --  歆 (<CJK Ideograph>)
      3465   => 16#6B4A#,  --  歊 (<CJK Ideograph>)
      3466   => 16#6B4D#,  --  歍 (<CJK Ideograph>)
      3467   => 16#6B52#,  --  歒 (<CJK Ideograph>)
      3468   => 16#6B56#,  --  歖 (<CJK Ideograph>)
      3469   => 16#6B58#,  --  歘 (<CJK Ideograph>)
      3470   => 16#6B5D#,  --  歝 (<CJK Ideograph>)
      3471   => 16#6B60#,  --  歠 (<CJK Ideograph>)
      3472   => 16#6B67#,  --  歧 (<CJK Ideograph>)
      3473   => 16#6B6B#,  --  歫 (<CJK Ideograph>)
      3474   => 16#6B6E#,  --  歮 (<CJK Ideograph>)
      3475   => 16#6B70#,  --  歰 (<CJK Ideograph>)
      3476   => 16#6B75#,  --  歵 (<CJK Ideograph>)
      3477   => 16#6B7D#,  --  歽 (<CJK Ideograph>)
      3478   => 16#6B7E#,  --  歾 (<CJK Ideograph>)
      3479   => 16#6B82#,  --  殂 (<CJK Ideograph>)
      3480   => 16#6B85#,  --  殅 (<CJK Ideograph>)
      3481   => 16#6B97#,  --  殗 (<CJK Ideograph>)
      3482   => 16#6B9B#,  --  殛 (<CJK Ideograph>)
      3483   => 16#6B9F#,  --  殟 (<CJK Ideograph>)
      3484   => 16#6BA0#,  --  殠 (<CJK Ideograph>)
      3485   => 16#6BA2#,  --  殢 (<CJK Ideograph>)
      3486   => 16#6BA3#,  --  殣 (<CJK Ideograph>)
      3487   => 16#6BA8#,  --  殨 (<CJK Ideograph>)
      3488   => 16#6BA9#,  --  殩 (<CJK Ideograph>)
      3489   => 16#6BAC#,  --  殬 (<CJK Ideograph>)
      3490   => 16#6BAD#,  --  殭 (<CJK Ideograph>)
      3491   => 16#6BAE#,  --  殮 (<CJK Ideograph>)
      3492   => 16#6BB0#,  --  殰 (<CJK Ideograph>)
      3493   => 16#6BB8#,  --  殸 (<CJK Ideograph>)
      3494   => 16#6BB9#,  --  殹 (<CJK Ideograph>)
      3495   => 16#6BBD#,  --  殽 (<CJK Ideograph>)
      3496   => 16#6BBE#,  --  殾 (<CJK Ideograph>)
      3497   => 16#6BC3#,  --  毃 (<CJK Ideograph>)
      3498   => 16#6BC4#,  --  毄 (<CJK Ideograph>)
      3499   => 16#6BC9#,  --  毉 (<CJK Ideograph>)
      3500   => 16#6BCC#,  --  毌 (<CJK Ideograph>)
      3501   => 16#6BD6#,  --  毖 (<CJK Ideograph>)
      3502   => 16#6BDA#,  --  毚 (<CJK Ideograph>)
      3503   => 16#6BE1#,  --  毡 (<CJK Ideograph>)
      3504   => 16#6BE3#,  --  毣 (<CJK Ideograph>)
      3505   => 16#6BE6#,  --  毦 (<CJK Ideograph>)
      3506   => 16#6BE7#,  --  毧 (<CJK Ideograph>)
      3507   => 16#6BEE#,  --  毮 (<CJK Ideograph>)
      3508   => 16#6BF1#,  --  毱 (<CJK Ideograph>)
      3509   => 16#6BF7#,  --  毷 (<CJK Ideograph>)
      3510   => 16#6BF9#,  --  毹 (<CJK Ideograph>)
      3511   => 16#6BFF#,  --  毿 (<CJK Ideograph>)
      3512   => 16#6C02#,  --  氂 (<CJK Ideograph>)
      3513   => 16#6C04#,  --  氄 (<CJK Ideograph>)
      3514   => 16#6C05#,  --  氅 (<CJK Ideograph>)
      3515   => 16#6C09#,  --  氉 (<CJK Ideograph>)
      3516   => 16#6C0D#,  --  氍 (<CJK Ideograph>)
      3517   => 16#6C0E#,  --  氎 (<CJK Ideograph>)
      3518   => 16#6C10#,  --  氐 (<CJK Ideograph>)
      3519   => 16#6C12#,  --  氒 (<CJK Ideograph>)
      3520   => 16#6C19#,  --  氙 (<CJK Ideograph>)
      3521   => 16#6C1F#,  --  氟 (<CJK Ideograph>)
      3522   => 16#6C26#,  --  氦 (<CJK Ideograph>)
      3523   => 16#6C27#,  --  氧 (<CJK Ideograph>)
      3524   => 16#6C28#,  --  氨 (<CJK Ideograph>)
      3525   => 16#6C2C#,  --  氬 (<CJK Ideograph>)
      3526   => 16#6C2E#,  --  氮 (<CJK Ideograph>)
      3527   => 16#6C33#,  --  氳 (<CJK Ideograph>)
      3528   => 16#6C35#,  --  氵 (<CJK Ideograph>)
      3529   => 16#6C36#,  --  氶 (<CJK Ideograph>)
      3530   => 16#6C3A#,  --  氺 (<CJK Ideograph>)
      3531   => 16#6C3B#,  --  氻 (<CJK Ideograph>)
      3532   => 16#6C3F#,  --  氿 (<CJK Ideograph>)
      3533   => 16#6C4A#,  --  汊 (<CJK Ideograph>)
      3534   => 16#6C4B#,  --  汋 (<CJK Ideograph>)
      3535   => 16#6C4D#,  --  汍 (<CJK Ideograph>)
      3536   => 16#6C4F#,  --  汏 (<CJK Ideograph>)
      3537   => 16#6C52#,  --  汒 (<CJK Ideograph>)
      3538   => 16#6C54#,  --  汔 (<CJK Ideograph>)
      3539   => 16#6C59#,  --  汙 (<CJK Ideograph>)
      3540   => 16#6C5B#,  --  汛 (<CJK Ideograph>)
      3541   => 16#6C5C#,  --  汜 (<CJK Ideograph>)
      3542   => 16#6C6B#,  --  汫 (<CJK Ideograph>)
      3543   => 16#6C6D#,  --  汭 (<CJK Ideograph>)
      3544   => 16#6C6F#,  --  汯 (<CJK Ideograph>)
      3545   => 16#6C74#,  --  汴 (<CJK Ideograph>)
      3546   => 16#6C76#,  --  汶 (<CJK Ideograph>)
      3547   => 16#6C78#,  --  汸 (<CJK Ideograph>)
      3548   => 16#6C79#,  --  汹 (<CJK Ideograph>)
      3549   => 16#6C7B#,  --  汻 (<CJK Ideograph>)
      3550   => 16#6C85#,  --  沅 (<CJK Ideograph>)
      3551   => 16#6C86#,  --  沆 (<CJK Ideograph>)
      3552   => 16#6C87#,  --  沇 (<CJK Ideograph>)
      3553   => 16#6C89#,  --  沉 (<CJK Ideograph>)
      3554   => 16#6C94#,  --  沔 (<CJK Ideograph>)
      3555   => 16#6C95#,  --  沕 (<CJK Ideograph>)
      3556   => 16#6C97#,  --  沗 (<CJK Ideograph>)
      3557   => 16#6C98#,  --  沘 (<CJK Ideograph>)
      3558   => 16#6C9C#,  --  沜 (<CJK Ideograph>)
      3559   => 16#6C9F#,  --  沟 (<CJK Ideograph>)
      3560   => 16#6CB0#,  --  沰 (<CJK Ideograph>)
      3561   => 16#6CB2#,  --  沲 (<CJK Ideograph>)
      3562   => 16#6CB4#,  --  沴 (<CJK Ideograph>)
      3563   => 16#6CC2#,  --  泂 (<CJK Ideograph>)
      3564   => 16#6CC6#,  --  泆 (<CJK Ideograph>)
      3565   => 16#6CCD#,  --  泍 (<CJK Ideograph>)
      3566   => 16#6CCF#,  --  泏 (<CJK Ideograph>)
      3567   => 16#6CD0#,  --  泐 (<CJK Ideograph>)
      3568   => 16#6CD1#,  --  泑 (<CJK Ideograph>)
      3569   => 16#6CD2#,  --  泒 (<CJK Ideograph>)
      3570   => 16#6CD4#,  --  泔 (<CJK Ideograph>)
      3571   => 16#6CD6#,  --  泖 (<CJK Ideograph>)
      3572   => 16#6CDA#,  --  泚 (<CJK Ideograph>)
      3573   => 16#6CDC#,  --  泜 (<CJK Ideograph>)
      3574   => 16#6CE0#,  --  泠 (<CJK Ideograph>)
      3575   => 16#6CE7#,  --  泧 (<CJK Ideograph>)
      3576   => 16#6CE9#,  --  泩 (<CJK Ideograph>)
      3577   => 16#6CEB#,  --  泫 (<CJK Ideograph>)
      3578   => 16#6CEC#,  --  泬 (<CJK Ideograph>)
      3579   => 16#6CEE#,  --  泮 (<CJK Ideograph>)
      3580   => 16#6CF2#,  --  泲 (<CJK Ideograph>)
      3581   => 16#6CF4#,  --  泴 (<CJK Ideograph>)
      3582   => 16#6D04#,  --  洄 (<CJK Ideograph>)
      3583   => 16#6D07#,  --  洇 (<CJK Ideograph>)
      3584   => 16#6D0A#,  --  洊 (<CJK Ideograph>)
      3585   => 16#6D0E#,  --  洎 (<CJK Ideograph>)
      3586   => 16#6D0F#,  --  洏 (<CJK Ideograph>)
      3587   => 16#6D11#,  --  洑 (<CJK Ideograph>)
      3588   => 16#6D13#,  --  洓 (<CJK Ideograph>)
      3589   => 16#6D1A#,  --  洚 (<CJK Ideograph>)
      3590   => 16#6D26#,  --  洦 (<CJK Ideograph>)
      3591   => 16#6D27#,  --  洧 (<CJK Ideograph>)
      3592   => 16#6D28#,  --  洨 (<CJK Ideograph>)
      3593   => 16#6C67#,  --  汧 (<CJK Ideograph>)
      3594   => 16#6D2E#,  --  洮 (<CJK Ideograph>)
      3595   => 16#6D2F#,  --  洯 (<CJK Ideograph>)
      3596   => 16#6D31#,  --  洱 (<CJK Ideograph>)
      3597   => 16#6D39#,  --  洹 (<CJK Ideograph>)
      3598   => 16#6D3C#,  --  洼 (<CJK Ideograph>)
      3599   => 16#6D3F#,  --  洿 (<CJK Ideograph>)
      3600   => 16#6D57#,  --  浗 (<CJK Ideograph>)
      3601   => 16#6D5E#,  --  浞 (<CJK Ideograph>)
      3602   => 16#6D5F#,  --  浟 (<CJK Ideograph>)
      3603   => 16#6D61#,  --  浡 (<CJK Ideograph>)
      3604   => 16#6D65#,  --  浥 (<CJK Ideograph>)
      3605   => 16#6D67#,  --  浧 (<CJK Ideograph>)
      3606   => 16#6D6F#,  --  浯 (<CJK Ideograph>)
      3607   => 16#6D70#,  --  浰 (<CJK Ideograph>)
      3608   => 16#6D7C#,  --  浼 (<CJK Ideograph>)
      3609   => 16#6D82#,  --  涂 (<CJK Ideograph>)
      3610   => 16#6D87#,  --  涇 (<CJK Ideograph>)
      3611   => 16#6D91#,  --  涑 (<CJK Ideograph>)
      3612   => 16#6D92#,  --  涒 (<CJK Ideograph>)
      3613   => 16#6D94#,  --  涔 (<CJK Ideograph>)
      3614   => 16#6D96#,  --  涖 (<CJK Ideograph>)
      3615   => 16#6D97#,  --  涗 (<CJK Ideograph>)
      3616   => 16#6D98#,  --  涘 (<CJK Ideograph>)
      3617   => 16#6DAA#,  --  涪 (<CJK Ideograph>)
      3618   => 16#6DAC#,  --  涬 (<CJK Ideograph>)
      3619   => 16#6DB4#,  --  涴 (<CJK Ideograph>)
      3620   => 16#6DB7#,  --  涷 (<CJK Ideograph>)
      3621   => 16#6DB9#,  --  涹 (<CJK Ideograph>)
      3622   => 16#6DBD#,  --  涽 (<CJK Ideograph>)
      3623   => 16#6DBF#,  --  涿 (<CJK Ideograph>)
      3624   => 16#6DC4#,  --  淄 (<CJK Ideograph>)
      3625   => 16#6DC8#,  --  淈 (<CJK Ideograph>)
      3626   => 16#6DCA#,  --  淊 (<CJK Ideograph>)
      3627   => 16#6DCE#,  --  淎 (<CJK Ideograph>)
      3628   => 16#6DCF#,  --  淏 (<CJK Ideograph>)
      3629   => 16#6DD6#,  --  淖 (<CJK Ideograph>)
      3630   => 16#6DDB#,  --  淛 (<CJK Ideograph>)
      3631   => 16#6DDD#,  --  淝 (<CJK Ideograph>)
      3632   => 16#6DDF#,  --  淟 (<CJK Ideograph>)
      3633   => 16#6DE0#,  --  淠 (<CJK Ideograph>)
      3634   => 16#6DE2#,  --  淢 (<CJK Ideograph>)
      3635   => 16#6DE5#,  --  淥 (<CJK Ideograph>)
      3636   => 16#6DE9#,  --  淩 (<CJK Ideograph>)
      3637   => 16#6DEF#,  --  淯 (<CJK Ideograph>)
      3638   => 16#6DF0#,  --  淰 (<CJK Ideograph>)
      3639   => 16#6DF4#,  --  淴 (<CJK Ideograph>)
      3640   => 16#6DF6#,  --  淶 (<CJK Ideograph>)
      3641   => 16#6DFC#,  --  淼 (<CJK Ideograph>)
      3642   => 16#6E00#,  --  渀 (<CJK Ideograph>)
      3643   => 16#6E04#,  --  渄 (<CJK Ideograph>)
      3644   => 16#6E1E#,  --  渞 (<CJK Ideograph>)
      3645   => 16#6E22#,  --  渢 (<CJK Ideograph>)
      3646   => 16#6E27#,  --  渧 (<CJK Ideograph>)
      3647   => 16#6E32#,  --  渲 (<CJK Ideograph>)
      3648   => 16#6E36#,  --  渶 (<CJK Ideograph>)
      3649   => 16#6E39#,  --  渹 (<CJK Ideograph>)
      3650   => 16#6E3B#,  --  渻 (<CJK Ideograph>)
      3651   => 16#6E3C#,  --  渼 (<CJK Ideograph>)
      3652   => 16#6E44#,  --  湄 (<CJK Ideograph>)
      3653   => 16#6E45#,  --  湅 (<CJK Ideograph>)
      3654   => 16#6E48#,  --  湈 (<CJK Ideograph>)
      3655   => 16#6E49#,  --  湉 (<CJK Ideograph>)
      3656   => 16#6E4B#,  --  湋 (<CJK Ideograph>)
      3657   => 16#6E4F#,  --  湏 (<CJK Ideograph>)
      3658   => 16#6E51#,  --  湑 (<CJK Ideograph>)
      3659   => 16#6E52#,  --  湒 (<CJK Ideograph>)
      3660   => 16#6E53#,  --  湓 (<CJK Ideograph>)
      3661   => 16#6E54#,  --  湔 (<CJK Ideograph>)
      3662   => 16#6E57#,  --  湗 (<CJK Ideograph>)
      3663   => 16#6E5C#,  --  湜 (<CJK Ideograph>)
      3664   => 16#6E5D#,  --  湝 (<CJK Ideograph>)
      3665   => 16#6E5E#,  --  湞 (<CJK Ideograph>)
      3666   => 16#6E62#,  --  湢 (<CJK Ideograph>)
      3667   => 16#6E63#,  --  湣 (<CJK Ideograph>)
      3668   => 16#6E68#,  --  湨 (<CJK Ideograph>)
      3669   => 16#6E73#,  --  湳 (<CJK Ideograph>)
      3670   => 16#6E7B#,  --  湻 (<CJK Ideograph>)
      3671   => 16#6E7D#,  --  湽 (<CJK Ideograph>)
      3672   => 16#6E8D#,  --  溍 (<CJK Ideograph>)
      3673   => 16#6E93#,  --  溓 (<CJK Ideograph>)
      3674   => 16#6E99#,  --  溙 (<CJK Ideograph>)
      3675   => 16#6EA0#,  --  溠 (<CJK Ideograph>)
      3676   => 16#6EA7#,  --  溧 (<CJK Ideograph>)
      3677   => 16#6EAD#,  --  溭 (<CJK Ideograph>)
      3678   => 16#6EAE#,  --  溮 (<CJK Ideograph>)
      3679   => 16#6EB1#,  --  溱 (<CJK Ideograph>)
      3680   => 16#6EB3#,  --  溳 (<CJK Ideograph>)
      3681   => 16#6EBB#,  --  溻 (<CJK Ideograph>)
      3682   => 16#6EBF#,  --  溿 (<CJK Ideograph>)
      3683   => 16#6EC0#,  --  滀 (<CJK Ideograph>)
      3684   => 16#6EC1#,  --  滁 (<CJK Ideograph>)
      3685   => 16#6EC3#,  --  滃 (<CJK Ideograph>)
      3686   => 16#6EC7#,  --  滇 (<CJK Ideograph>)
      3687   => 16#6EC8#,  --  滈 (<CJK Ideograph>)
      3688   => 16#6ECA#,  --  滊 (<CJK Ideograph>)
      3689   => 16#6ECD#,  --  滍 (<CJK Ideograph>)
      3690   => 16#6ECE#,  --  滎 (<CJK Ideograph>)
      3691   => 16#6ECF#,  --  滏 (<CJK Ideograph>)
      3692   => 16#6EEB#,  --  滫 (<CJK Ideograph>)
      3693   => 16#6EED#,  --  滭 (<CJK Ideograph>)
      3694   => 16#6EEE#,  --  滮 (<CJK Ideograph>)
      3695   => 16#6EF9#,  --  滹 (<CJK Ideograph>)
      3696   => 16#6EFB#,  --  滻 (<CJK Ideograph>)
      3697   => 16#6EFD#,  --  滽 (<CJK Ideograph>)
      3698   => 16#6F04#,  --  漄 (<CJK Ideograph>)
      3699   => 16#6F08#,  --  漈 (<CJK Ideograph>)
      3700   => 16#6F0A#,  --  漊 (<CJK Ideograph>)
      3701   => 16#6F0C#,  --  漌 (<CJK Ideograph>)
      3702   => 16#6F0D#,  --  漍 (<CJK Ideograph>)
      3703   => 16#6F16#,  --  漖 (<CJK Ideograph>)
      3704   => 16#6F18#,  --  漘 (<CJK Ideograph>)
      3705   => 16#6F1A#,  --  漚 (<CJK Ideograph>)
      3706   => 16#6F1B#,  --  漛 (<CJK Ideograph>)
      3707   => 16#6F26#,  --  漦 (<CJK Ideograph>)
      3708   => 16#6F29#,  --  漩 (<CJK Ideograph>)
      3709   => 16#6F2A#,  --  漪 (<CJK Ideograph>)
      3710   => 16#6F2F#,  --  漯 (<CJK Ideograph>)
      3711   => 16#6F30#,  --  漰 (<CJK Ideograph>)
      3712   => 16#6F33#,  --  漳 (<CJK Ideograph>)
      3713   => 16#6F36#,  --  漶 (<CJK Ideograph>)
      3714   => 16#6F3B#,  --  漻 (<CJK Ideograph>)
      3715   => 16#6F3C#,  --  漼 (<CJK Ideograph>)
      3716   => 16#6F2D#,  --  漭 (<CJK Ideograph>)
      3717   => 16#6F4F#,  --  潏 (<CJK Ideograph>)
      3718   => 16#6F51#,  --  潑 (<CJK Ideograph>)
      3719   => 16#6F52#,  --  潒 (<CJK Ideograph>)
      3720   => 16#6F53#,  --  潓 (<CJK Ideograph>)
      3721   => 16#6F57#,  --  潗 (<CJK Ideograph>)
      3722   => 16#6F59#,  --  潙 (<CJK Ideograph>)
      3723   => 16#6F5A#,  --  潚 (<CJK Ideograph>)
      3724   => 16#6F5D#,  --  潝 (<CJK Ideograph>)
      3725   => 16#6F5E#,  --  潞 (<CJK Ideograph>)
      3726   => 16#6F61#,  --  潡 (<CJK Ideograph>)
      3727   => 16#6F62#,  --  潢 (<CJK Ideograph>)
      3728   => 16#6F68#,  --  潨 (<CJK Ideograph>)
      3729   => 16#6F6C#,  --  潬 (<CJK Ideograph>)
      3730   => 16#6F7D#,  --  潽 (<CJK Ideograph>)
      3731   => 16#6F7E#,  --  潾 (<CJK Ideograph>)
      3732   => 16#6F83#,  --  澃 (<CJK Ideograph>)
      3733   => 16#6F87#,  --  澇 (<CJK Ideograph>)
      3734   => 16#6F88#,  --  澈 (<CJK Ideograph>)
      3735   => 16#6F8B#,  --  澋 (<CJK Ideograph>)
      3736   => 16#6F8C#,  --  澌 (<CJK Ideograph>)
      3737   => 16#6F8D#,  --  澍 (<CJK Ideograph>)
      3738   => 16#6F90#,  --  澐 (<CJK Ideograph>)
      3739   => 16#6F92#,  --  澒 (<CJK Ideograph>)
      3740   => 16#6F93#,  --  澓 (<CJK Ideograph>)
      3741   => 16#6F94#,  --  澔 (<CJK Ideograph>)
      3742   => 16#6F96#,  --  澖 (<CJK Ideograph>)
      3743   => 16#6F9A#,  --  澚 (<CJK Ideograph>)
      3744   => 16#6F9F#,  --  澟 (<CJK Ideograph>)
      3745   => 16#6FA0#,  --  澠 (<CJK Ideograph>)
      3746   => 16#6FA5#,  --  澥 (<CJK Ideograph>)
      3747   => 16#6FA6#,  --  澦 (<CJK Ideograph>)
      3748   => 16#6FA7#,  --  澧 (<CJK Ideograph>)
      3749   => 16#6FA8#,  --  澨 (<CJK Ideograph>)
      3750   => 16#6FAE#,  --  澮 (<CJK Ideograph>)
      3751   => 16#6FAF#,  --  澯 (<CJK Ideograph>)
      3752   => 16#6FB0#,  --  澰 (<CJK Ideograph>)
      3753   => 16#6FB5#,  --  澵 (<CJK Ideograph>)
      3754   => 16#6FB6#,  --  澶 (<CJK Ideograph>)
      3755   => 16#6FBC#,  --  澼 (<CJK Ideograph>)
      3756   => 16#6FC5#,  --  濅 (<CJK Ideograph>)
      3757   => 16#6FC7#,  --  濇 (<CJK Ideograph>)
      3758   => 16#6FC8#,  --  濈 (<CJK Ideograph>)
      3759   => 16#6FCA#,  --  濊 (<CJK Ideograph>)
      3760   => 16#6FDA#,  --  濚 (<CJK Ideograph>)
      3761   => 16#6FDE#,  --  濞 (<CJK Ideograph>)
      3762   => 16#6FE8#,  --  濨 (<CJK Ideograph>)
      3763   => 16#6FE9#,  --  濩 (<CJK Ideograph>)
      3764   => 16#6FF0#,  --  濰 (<CJK Ideograph>)
      3765   => 16#6FF5#,  --  濵 (<CJK Ideograph>)
      3766   => 16#6FF9#,  --  濹 (<CJK Ideograph>)
      3767   => 16#6FFC#,  --  濼 (<CJK Ideograph>)
      3768   => 16#6FFD#,  --  濽 (<CJK Ideograph>)
      3769   => 16#7000#,  --  瀀 (<CJK Ideograph>)
      3770   => 16#7005#,  --  瀅 (<CJK Ideograph>)
      3771   => 16#7006#,  --  瀆 (<CJK Ideograph>)
      3772   => 16#7007#,  --  瀇 (<CJK Ideograph>)
      3773   => 16#700D#,  --  瀍 (<CJK Ideograph>)
      3774   => 16#7017#,  --  瀗 (<CJK Ideograph>)
      3775   => 16#7020#,  --  瀠 (<CJK Ideograph>)
      3776   => 16#7023#,  --  瀣 (<CJK Ideograph>)
      3777   => 16#702F#,  --  瀯 (<CJK Ideograph>)
      3778   => 16#7034#,  --  瀴 (<CJK Ideograph>)
      3779   => 16#7037#,  --  瀷 (<CJK Ideograph>)
      3780   => 16#7039#,  --  瀹 (<CJK Ideograph>)
      3781   => 16#703C#,  --  瀼 (<CJK Ideograph>)
      3782   => 16#7043#,  --  灃 (<CJK Ideograph>)
      3783   => 16#7044#,  --  灄 (<CJK Ideograph>)
      3784   => 16#7048#,  --  灈 (<CJK Ideograph>)
      3785   => 16#7049#,  --  灉 (<CJK Ideograph>)
      3786   => 16#704A#,  --  灊 (<CJK Ideograph>)
      3787   => 16#704B#,  --  灋 (<CJK Ideograph>)
      3788   => 16#7054#,  --  灔 (<CJK Ideograph>)
      3789   => 16#7055#,  --  灕 (<CJK Ideograph>)
      3790   => 16#705D#,  --  灝 (<CJK Ideograph>)
      3791   => 16#705E#,  --  灞 (<CJK Ideograph>)
      3792   => 16#704E#,  --  灎 (<CJK Ideograph>)
      3793   => 16#7064#,  --  灤 (<CJK Ideograph>)
      3794   => 16#7065#,  --  灥 (<CJK Ideograph>)
      3795   => 16#706C#,  --  灬 (<CJK Ideograph>)
      3796   => 16#706E#,  --  灮 (<CJK Ideograph>)
      3797   => 16#7075#,  --  灵 (<CJK Ideograph>)
      3798   => 16#7076#,  --  灶 (<CJK Ideograph>)
      3799   => 16#707E#,  --  灾 (<CJK Ideograph>)
      3800   => 16#7081#,  --  炁 (<CJK Ideograph>)
      3801   => 16#7085#,  --  炅 (<CJK Ideograph>)
      3802   => 16#7086#,  --  炆 (<CJK Ideograph>)
      3803   => 16#7094#,  --  炔 (<CJK Ideograph>)
      3804   => 16#7095#,  --  炕 (<CJK Ideograph>)
      3805   => 16#7096#,  --  炖 (<CJK Ideograph>)
      3806   => 16#7097#,  --  炗 (<CJK Ideograph>)
      3807   => 16#7098#,  --  炘 (<CJK Ideograph>)
      3808   => 16#709B#,  --  炛 (<CJK Ideograph>)
      3809   => 16#70A4#,  --  炤 (<CJK Ideograph>)
      3810   => 16#70AB#,  --  炫 (<CJK Ideograph>)
      3811   => 16#70B0#,  --  炰 (<CJK Ideograph>)
      3812   => 16#70B1#,  --  炱 (<CJK Ideograph>)
      3813   => 16#70B4#,  --  炴 (<CJK Ideograph>)
      3814   => 16#70B7#,  --  炷 (<CJK Ideograph>)
      3815   => 16#70CA#,  --  烊 (<CJK Ideograph>)
      3816   => 16#70D1#,  --  烑 (<CJK Ideograph>)
      3817   => 16#70D3#,  --  烓 (<CJK Ideograph>)
      3818   => 16#70D4#,  --  烔 (<CJK Ideograph>)
      3819   => 16#70D5#,  --  烕 (<CJK Ideograph>)
      3820   => 16#70D6#,  --  烖 (<CJK Ideograph>)
      3821   => 16#70D8#,  --  烘 (<CJK Ideograph>)
      3822   => 16#70DC#,  --  烜 (<CJK Ideograph>)
      3823   => 16#70E4#,  --  烤 (<CJK Ideograph>)
      3824   => 16#70FA#,  --  烺 (<CJK Ideograph>)
      3825   => 16#7103#,  --  焃 (<CJK Ideograph>)
      3826   => 16#7104#,  --  焄 (<CJK Ideograph>)
      3827   => 16#7105#,  --  焅 (<CJK Ideograph>)
      3828   => 16#7106#,  --  焆 (<CJK Ideograph>)
      3829   => 16#7107#,  --  焇 (<CJK Ideograph>)
      3830   => 16#710B#,  --  焋 (<CJK Ideograph>)
      3831   => 16#710C#,  --  焌 (<CJK Ideograph>)
      3832   => 16#710F#,  --  焏 (<CJK Ideograph>)
      3833   => 16#711E#,  --  焞 (<CJK Ideograph>)
      3834   => 16#7120#,  --  焠 (<CJK Ideograph>)
      3835   => 16#712B#,  --  焫 (<CJK Ideograph>)
      3836   => 16#712D#,  --  焭 (<CJK Ideograph>)
      3837   => 16#712F#,  --  焯 (<CJK Ideograph>)
      3838   => 16#7130#,  --  焰 (<CJK Ideograph>)
      3839   => 16#7131#,  --  焱 (<CJK Ideograph>)
      3840   => 16#7138#,  --  焸 (<CJK Ideograph>)
      3841   => 16#7141#,  --  煁 (<CJK Ideograph>)
      3842   => 16#7145#,  --  煅 (<CJK Ideograph>)
      3843   => 16#7146#,  --  煆 (<CJK Ideograph>)
      3844   => 16#7147#,  --  煇 (<CJK Ideograph>)
      3845   => 16#714A#,  --  煊 (<CJK Ideograph>)
      3846   => 16#714B#,  --  煋 (<CJK Ideograph>)
      3847   => 16#7150#,  --  煐 (<CJK Ideograph>)
      3848   => 16#7152#,  --  煒 (<CJK Ideograph>)
      3849   => 16#7157#,  --  煗 (<CJK Ideograph>)
      3850   => 16#715A#,  --  煚 (<CJK Ideograph>)
      3851   => 16#715C#,  --  煜 (<CJK Ideograph>)
      3852   => 16#715E#,  --  煞 (<CJK Ideograph>)
      3853   => 16#7160#,  --  煠 (<CJK Ideograph>)
      3854   => 16#7168#,  --  煨 (<CJK Ideograph>)
      3855   => 16#7179#,  --  煹 (<CJK Ideograph>)
      3856   => 16#7180#,  --  熀 (<CJK Ideograph>)
      3857   => 16#7185#,  --  熅 (<CJK Ideograph>)
      3858   => 16#7187#,  --  熇 (<CJK Ideograph>)
      3859   => 16#718C#,  --  熌 (<CJK Ideograph>)
      3860   => 16#7192#,  --  熒 (<CJK Ideograph>)
      3861   => 16#719A#,  --  熚 (<CJK Ideograph>)
      3862   => 16#719B#,  --  熛 (<CJK Ideograph>)
      3863   => 16#71A0#,  --  熠 (<CJK Ideograph>)
      3864   => 16#71A2#,  --  熢 (<CJK Ideograph>)
      3865   => 16#71AF#,  --  熯 (<CJK Ideograph>)
      3866   => 16#71B0#,  --  熰 (<CJK Ideograph>)
      3867   => 16#71B2#,  --  熲 (<CJK Ideograph>)
      3868   => 16#71B3#,  --  熳 (<CJK Ideograph>)
      3869   => 16#71BA#,  --  熺 (<CJK Ideograph>)
      3870   => 16#71BF#,  --  熿 (<CJK Ideograph>)
      3871   => 16#71C0#,  --  燀 (<CJK Ideograph>)
      3872   => 16#71C1#,  --  燁 (<CJK Ideograph>)
      3873   => 16#71C4#,  --  燄 (<CJK Ideograph>)
      3874   => 16#71CB#,  --  燋 (<CJK Ideograph>)
      3875   => 16#71CC#,  --  燌 (<CJK Ideograph>)
      3876   => 16#71D3#,  --  燓 (<CJK Ideograph>)
      3877   => 16#71D6#,  --  燖 (<CJK Ideograph>)
      3878   => 16#71D9#,  --  燙 (<CJK Ideograph>)
      3879   => 16#71DA#,  --  燚 (<CJK Ideograph>)
      3880   => 16#71DC#,  --  燜 (<CJK Ideograph>)
      3881   => 16#71F8#,  --  燸 (<CJK Ideograph>)
      3882   => 16#71FE#,  --  燾 (<CJK Ideograph>)
      3883   => 16#7200#,  --  爀 (<CJK Ideograph>)
      3884   => 16#7207#,  --  爇 (<CJK Ideograph>)
      3885   => 16#7208#,  --  爈 (<CJK Ideograph>)
      3886   => 16#7209#,  --  爉 (<CJK Ideograph>)
      3887   => 16#7213#,  --  爓 (<CJK Ideograph>)
      3888   => 16#7217#,  --  爗 (<CJK Ideograph>)
      3889   => 16#721A#,  --  爚 (<CJK Ideograph>)
      3890   => 16#721D#,  --  爝 (<CJK Ideograph>)
      3891   => 16#721F#,  --  爟 (<CJK Ideograph>)
      3892   => 16#7224#,  --  爤 (<CJK Ideograph>)
      3893   => 16#722B#,  --  爫 (<CJK Ideograph>)
      3894   => 16#722F#,  --  爯 (<CJK Ideograph>)
      3895   => 16#7234#,  --  爴 (<CJK Ideograph>)
      3896   => 16#7238#,  --  爸 (<CJK Ideograph>)
      3897   => 16#7239#,  --  爹 (<CJK Ideograph>)
      3898   => 16#7241#,  --  牁 (<CJK Ideograph>)
      3899   => 16#7242#,  --  牂 (<CJK Ideograph>)
      3900   => 16#7243#,  --  牃 (<CJK Ideograph>)
      3901   => 16#7245#,  --  牅 (<CJK Ideograph>)
      3902   => 16#724E#,  --  牎 (<CJK Ideograph>)
      3903   => 16#724F#,  --  牏 (<CJK Ideograph>)
      3904   => 16#7250#,  --  牐 (<CJK Ideograph>)
      3905   => 16#7253#,  --  牓 (<CJK Ideograph>)
      3906   => 16#7255#,  --  牕 (<CJK Ideograph>)
      3907   => 16#7256#,  --  牖 (<CJK Ideograph>)
      3908   => 16#725A#,  --  牚 (<CJK Ideograph>)
      3909   => 16#725C#,  --  牜 (<CJK Ideograph>)
      3910   => 16#725E#,  --  牞 (<CJK Ideograph>)
      3911   => 16#7260#,  --  牠 (<CJK Ideograph>)
      3912   => 16#7263#,  --  牣 (<CJK Ideograph>)
      3913   => 16#7268#,  --  牨 (<CJK Ideograph>)
      3914   => 16#726B#,  --  牫 (<CJK Ideograph>)
      3915   => 16#726E#,  --  牮 (<CJK Ideograph>)
      3916   => 16#726F#,  --  牯 (<CJK Ideograph>)
      3917   => 16#7271#,  --  牱 (<CJK Ideograph>)
      3918   => 16#7277#,  --  牷 (<CJK Ideograph>)
      3919   => 16#7278#,  --  牸 (<CJK Ideograph>)
      3920   => 16#727B#,  --  牻 (<CJK Ideograph>)
      3921   => 16#727C#,  --  牼 (<CJK Ideograph>)
      3922   => 16#727F#,  --  牿 (<CJK Ideograph>)
      3923   => 16#7284#,  --  犄 (<CJK Ideograph>)
      3924   => 16#7289#,  --  犉 (<CJK Ideograph>)
      3925   => 16#728D#,  --  犍 (<CJK Ideograph>)
      3926   => 16#728E#,  --  犎 (<CJK Ideograph>)
      3927   => 16#7293#,  --  犓 (<CJK Ideograph>)
      3928   => 16#729B#,  --  犛 (<CJK Ideograph>)
      3929   => 16#72A8#,  --  犨 (<CJK Ideograph>)
      3930   => 16#72AD#,  --  犭 (<CJK Ideograph>)
      3931   => 16#72AE#,  --  犮 (<CJK Ideograph>)
      3932   => 16#72B1#,  --  犱 (<CJK Ideograph>)
      3933   => 16#72B4#,  --  犴 (<CJK Ideograph>)
      3934   => 16#72BE#,  --  犾 (<CJK Ideograph>)
      3935   => 16#72C1#,  --  狁 (<CJK Ideograph>)
      3936   => 16#72C7#,  --  狇 (<CJK Ideograph>)
      3937   => 16#72C9#,  --  狉 (<CJK Ideograph>)
      3938   => 16#72CC#,  --  狌 (<CJK Ideograph>)
      3939   => 16#72D5#,  --  狕 (<CJK Ideograph>)
      3940   => 16#72D6#,  --  狖 (<CJK Ideograph>)
      3941   => 16#72D8#,  --  狘 (<CJK Ideograph>)
      3942   => 16#72DF#,  --  狟 (<CJK Ideograph>)
      3943   => 16#72E5#,  --  狥 (<CJK Ideograph>)
      3944   => 16#72F3#,  --  狳 (<CJK Ideograph>)
      3945   => 16#72F4#,  --  狴 (<CJK Ideograph>)
      3946   => 16#72FA#,  --  狺 (<CJK Ideograph>)
      3947   => 16#72FB#,  --  狻 (<CJK Ideograph>)
      3948   => 16#72FE#,  --  狾 (<CJK Ideograph>)
      3949   => 16#7302#,  --  猂 (<CJK Ideograph>)
      3950   => 16#7304#,  --  猄 (<CJK Ideograph>)
      3951   => 16#7305#,  --  猅 (<CJK Ideograph>)
      3952   => 16#7307#,  --  猇 (<CJK Ideograph>)
      3953   => 16#730B#,  --  猋 (<CJK Ideograph>)
      3954   => 16#730D#,  --  猍 (<CJK Ideograph>)
      3955   => 16#7312#,  --  猒 (<CJK Ideograph>)
      3956   => 16#7313#,  --  猓 (<CJK Ideograph>)
      3957   => 16#7318#,  --  猘 (<CJK Ideograph>)
      3958   => 16#7319#,  --  猙 (<CJK Ideograph>)
      3959   => 16#731E#,  --  猞 (<CJK Ideograph>)
      3960   => 16#7322#,  --  猢 (<CJK Ideograph>)
      3961   => 16#7324#,  --  猤 (<CJK Ideograph>)
      3962   => 16#7327#,  --  猧 (<CJK Ideograph>)
      3963   => 16#7328#,  --  猨 (<CJK Ideograph>)
      3964   => 16#732C#,  --  猬 (<CJK Ideograph>)
      3965   => 16#7331#,  --  猱 (<CJK Ideograph>)
      3966   => 16#7332#,  --  猲 (<CJK Ideograph>)
      3967   => 16#7335#,  --  猵 (<CJK Ideograph>)
      3968   => 16#733A#,  --  猺 (<CJK Ideograph>)
      3969   => 16#733B#,  --  猻 (<CJK Ideograph>)
      3970   => 16#733D#,  --  猽 (<CJK Ideograph>)
      3971   => 16#7343#,  --  獃 (<CJK Ideograph>)
      3972   => 16#734D#,  --  獍 (<CJK Ideograph>)
      3973   => 16#7350#,  --  獐 (<CJK Ideograph>)
      3974   => 16#7352#,  --  獒 (<CJK Ideograph>)
      3975   => 16#7356#,  --  獖 (<CJK Ideograph>)
      3976   => 16#7358#,  --  獘 (<CJK Ideograph>)
      3977   => 16#735D#,  --  獝 (<CJK Ideograph>)
      3978   => 16#735E#,  --  獞 (<CJK Ideograph>)
      3979   => 16#735F#,  --  獟 (<CJK Ideograph>)
      3980   => 16#7360#,  --  獠 (<CJK Ideograph>)
      3981   => 16#7366#,  --  獦 (<CJK Ideograph>)
      3982   => 16#7367#,  --  獧 (<CJK Ideograph>)
      3983   => 16#7369#,  --  獩 (<CJK Ideograph>)
      3984   => 16#736B#,  --  獫 (<CJK Ideograph>)
      3985   => 16#736C#,  --  獬 (<CJK Ideograph>)
      3986   => 16#736E#,  --  獮 (<CJK Ideograph>)
      3987   => 16#736F#,  --  獯 (<CJK Ideograph>)
      3988   => 16#7371#,  --  獱 (<CJK Ideograph>)
      3989   => 16#7377#,  --  獷 (<CJK Ideograph>)
      3990   => 16#7379#,  --  獹 (<CJK Ideograph>)
      3991   => 16#737C#,  --  獼 (<CJK Ideograph>)
      3992   => 16#7380#,  --  玀 (<CJK Ideograph>)
      3993   => 16#7381#,  --  玁 (<CJK Ideograph>)
      3994   => 16#7383#,  --  玃 (<CJK Ideograph>)
      3995   => 16#7385#,  --  玅 (<CJK Ideograph>)
      3996   => 16#7386#,  --  玆 (<CJK Ideograph>)
      3997   => 16#738E#,  --  玎 (<CJK Ideograph>)
      3998   => 16#7390#,  --  玐 (<CJK Ideograph>)
      3999   => 16#7393#,  --  玓 (<CJK Ideograph>)
      4000   => 16#7395#,  --  玕 (<CJK Ideograph>)
      4001   => 16#7397#,  --  玗 (<CJK Ideograph>)
      4002   => 16#7398#,  --  玘 (<CJK Ideograph>)
      4003   => 16#739C#,  --  玜 (<CJK Ideograph>)
      4004   => 16#739E#,  --  玞 (<CJK Ideograph>)
      4005   => 16#739F#,  --  玟 (<CJK Ideograph>)
      4006   => 16#73A0#,  --  玠 (<CJK Ideograph>)
      4007   => 16#73A2#,  --  玢 (<CJK Ideograph>)
      4008   => 16#73A5#,  --  玥 (<CJK Ideograph>)
      4009   => 16#73A6#,  --  玦 (<CJK Ideograph>)
      4010   => 16#73AA#,  --  玪 (<CJK Ideograph>)
      4011   => 16#73AB#,  --  玫 (<CJK Ideograph>)
      4012   => 16#73AD#,  --  玭 (<CJK Ideograph>)
      4013   => 16#73B5#,  --  玵 (<CJK Ideograph>)
      4014   => 16#73B7#,  --  玷 (<CJK Ideograph>)
      4015   => 16#73B9#,  --  玹 (<CJK Ideograph>)
      4016   => 16#73BC#,  --  玼 (<CJK Ideograph>)
      4017   => 16#73BD#,  --  玽 (<CJK Ideograph>)
      4018   => 16#73BF#,  --  玿 (<CJK Ideograph>)
      4019   => 16#73C5#,  --  珅 (<CJK Ideograph>)
      4020   => 16#73C6#,  --  珆 (<CJK Ideograph>)
      4021   => 16#73C9#,  --  珉 (<CJK Ideograph>)
      4022   => 16#73CB#,  --  珋 (<CJK Ideograph>)
      4023   => 16#73CC#,  --  珌 (<CJK Ideograph>)
      4024   => 16#73CF#,  --  珏 (<CJK Ideograph>)
      4025   => 16#73D2#,  --  珒 (<CJK Ideograph>)
      4026   => 16#73D3#,  --  珓 (<CJK Ideograph>)
      4027   => 16#73D6#,  --  珖 (<CJK Ideograph>)
      4028   => 16#73D9#,  --  珙 (<CJK Ideograph>)
      4029   => 16#73DD#,  --  珝 (<CJK Ideograph>)
      4030   => 16#73E1#,  --  珡 (<CJK Ideograph>)
      4031   => 16#73E3#,  --  珣 (<CJK Ideograph>)
      4032   => 16#73E6#,  --  珦 (<CJK Ideograph>)
      4033   => 16#73E7#,  --  珧 (<CJK Ideograph>)
      4034   => 16#73E9#,  --  珩 (<CJK Ideograph>)
      4035   => 16#73F4#,  --  珴 (<CJK Ideograph>)
      4036   => 16#73F5#,  --  珵 (<CJK Ideograph>)
      4037   => 16#73F7#,  --  珷 (<CJK Ideograph>)
      4038   => 16#73F9#,  --  珹 (<CJK Ideograph>)
      4039   => 16#73FA#,  --  珺 (<CJK Ideograph>)
      4040   => 16#73FB#,  --  珻 (<CJK Ideograph>)
      4041   => 16#73FD#,  --  珽 (<CJK Ideograph>)
      4042   => 16#73FF#,  --  珿 (<CJK Ideograph>)
      4043   => 16#7400#,  --  琀 (<CJK Ideograph>)
      4044   => 16#7401#,  --  琁 (<CJK Ideograph>)
      4045   => 16#7404#,  --  琄 (<CJK Ideograph>)
      4046   => 16#7407#,  --  琇 (<CJK Ideograph>)
      4047   => 16#740A#,  --  琊 (<CJK Ideograph>)
      4048   => 16#7411#,  --  琑 (<CJK Ideograph>)
      4049   => 16#741A#,  --  琚 (<CJK Ideograph>)
      4050   => 16#741B#,  --  琛 (<CJK Ideograph>)
      4051   => 16#7424#,  --  琤 (<CJK Ideograph>)
      4052   => 16#7426#,  --  琦 (<CJK Ideograph>)
      4053   => 16#7428#,  --  琨 (<CJK Ideograph>)
      4054   => 16#7429#,  --  琩 (<CJK Ideograph>)
      4055   => 16#742A#,  --  琪 (<CJK Ideograph>)
      4056   => 16#742B#,  --  琫 (<CJK Ideograph>)
      4057   => 16#742C#,  --  琬 (<CJK Ideograph>)
      4058   => 16#742D#,  --  琭 (<CJK Ideograph>)
      4059   => 16#742E#,  --  琮 (<CJK Ideograph>)
      4060   => 16#742F#,  --  琯 (<CJK Ideograph>)
      4061   => 16#7430#,  --  琰 (<CJK Ideograph>)
      4062   => 16#7431#,  --  琱 (<CJK Ideograph>)
      4063   => 16#7439#,  --  琹 (<CJK Ideograph>)
      4064   => 16#7440#,  --  瑀 (<CJK Ideograph>)
      4065   => 16#7443#,  --  瑃 (<CJK Ideograph>)
      4066   => 16#7444#,  --  瑄 (<CJK Ideograph>)
      4067   => 16#7446#,  --  瑆 (<CJK Ideograph>)
      4068   => 16#7447#,  --  瑇 (<CJK Ideograph>)
      4069   => 16#744B#,  --  瑋 (<CJK Ideograph>)
      4070   => 16#744D#,  --  瑍 (<CJK Ideograph>)
      4071   => 16#7451#,  --  瑑 (<CJK Ideograph>)
      4072   => 16#7452#,  --  瑒 (<CJK Ideograph>)
      4073   => 16#7457#,  --  瑗 (<CJK Ideograph>)
      4074   => 16#745D#,  --  瑝 (<CJK Ideograph>)
      4075   => 16#7462#,  --  瑢 (<CJK Ideograph>)
      4076   => 16#7466#,  --  瑦 (<CJK Ideograph>)
      4077   => 16#7467#,  --  瑧 (<CJK Ideograph>)
      4078   => 16#7468#,  --  瑨 (<CJK Ideograph>)
      4079   => 16#746B#,  --  瑫 (<CJK Ideograph>)
      4080   => 16#746D#,  --  瑭 (<CJK Ideograph>)
      4081   => 16#746E#,  --  瑮 (<CJK Ideograph>)
      4082   => 16#7471#,  --  瑱 (<CJK Ideograph>)
      4083   => 16#7472#,  --  瑲 (<CJK Ideograph>)
      4084   => 16#7480#,  --  璀 (<CJK Ideograph>)
      4085   => 16#7481#,  --  璁 (<CJK Ideograph>)
      4086   => 16#7485#,  --  璅 (<CJK Ideograph>)
      4087   => 16#7486#,  --  璆 (<CJK Ideograph>)
      4088   => 16#7487#,  --  璇 (<CJK Ideograph>)
      4089   => 16#7489#,  --  璉 (<CJK Ideograph>)
      4090   => 16#748F#,  --  璏 (<CJK Ideograph>)
      4091   => 16#7490#,  --  璐 (<CJK Ideograph>)
      4092   => 16#7491#,  --  璑 (<CJK Ideograph>)
      4093   => 16#7492#,  --  璒 (<CJK Ideograph>)
      4094   => 16#7498#,  --  璘 (<CJK Ideograph>)
      4095   => 16#7499#,  --  璙 (<CJK Ideograph>)
      4096   => 16#749A#,  --  璚 (<CJK Ideograph>)
      4097   => 16#749C#,  --  璜 (<CJK Ideograph>)
      4098   => 16#749F#,  --  璟 (<CJK Ideograph>)
      4099   => 16#74A0#,  --  璠 (<CJK Ideograph>)
      4100   => 16#74A1#,  --  璡 (<CJK Ideograph>)
      4101   => 16#74A3#,  --  璣 (<CJK Ideograph>)
      4102   => 16#74A6#,  --  璦 (<CJK Ideograph>)
      4103   => 16#74A8#,  --  璨 (<CJK Ideograph>)
      4104   => 16#74A9#,  --  璩 (<CJK Ideograph>)
      4105   => 16#74AA#,  --  璪 (<CJK Ideograph>)
      4106   => 16#74AB#,  --  璫 (<CJK Ideograph>)
      4107   => 16#74AE#,  --  璮 (<CJK Ideograph>)
      4108   => 16#74AF#,  --  璯 (<CJK Ideograph>)
      4109   => 16#74B1#,  --  璱 (<CJK Ideograph>)
      4110   => 16#74B2#,  --  璲 (<CJK Ideograph>)
      4111   => 16#74B5#,  --  璵 (<CJK Ideograph>)
      4112   => 16#74B9#,  --  璹 (<CJK Ideograph>)
      4113   => 16#74BB#,  --  璻 (<CJK Ideograph>)
      4114   => 16#74BF#,  --  璿 (<CJK Ideograph>)
      4115   => 16#74C8#,  --  瓈 (<CJK Ideograph>)
      4116   => 16#74C9#,  --  瓉 (<CJK Ideograph>)
      4117   => 16#74CC#,  --  瓌 (<CJK Ideograph>)
      4118   => 16#74D0#,  --  瓐 (<CJK Ideograph>)
      4119   => 16#74D3#,  --  瓓 (<CJK Ideograph>)
      4120   => 16#74D8#,  --  瓘 (<CJK Ideograph>)
      4121   => 16#74DA#,  --  瓚 (<CJK Ideograph>)
      4122   => 16#74DB#,  --  瓛 (<CJK Ideograph>)
      4123   => 16#74DE#,  --  瓞 (<CJK Ideograph>)
      4124   => 16#74DF#,  --  瓟 (<CJK Ideograph>)
      4125   => 16#74E4#,  --  瓤 (<CJK Ideograph>)
      4126   => 16#74E8#,  --  瓨 (<CJK Ideograph>)
      4127   => 16#74EA#,  --  瓪 (<CJK Ideograph>)
      4128   => 16#74EB#,  --  瓫 (<CJK Ideograph>)
      4129   => 16#74EF#,  --  瓯 (<CJK Ideograph>)
      4130   => 16#74F4#,  --  瓴 (<CJK Ideograph>)
      4131   => 16#74FA#,  --  瓺 (<CJK Ideograph>)
      4132   => 16#74FB#,  --  瓻 (<CJK Ideograph>)
      4133   => 16#74FC#,  --  瓼 (<CJK Ideograph>)
      4134   => 16#74FF#,  --  瓿 (<CJK Ideograph>)
      4135   => 16#7506#,  --  甆 (<CJK Ideograph>)
      4136   => 16#7512#,  --  甒 (<CJK Ideograph>)
      4137   => 16#7516#,  --  甖 (<CJK Ideograph>)
      4138   => 16#7517#,  --  甗 (<CJK Ideograph>)
      4139   => 16#7520#,  --  甠 (<CJK Ideograph>)
      4140   => 16#7521#,  --  甡 (<CJK Ideograph>)
      4141   => 16#7524#,  --  甤 (<CJK Ideograph>)
      4142   => 16#7527#,  --  甧 (<CJK Ideograph>)
      4143   => 16#7529#,  --  甩 (<CJK Ideograph>)
      4144   => 16#752A#,  --  甪 (<CJK Ideograph>)
      4145   => 16#752F#,  --  甯 (<CJK Ideograph>)
      4146   => 16#7536#,  --  甶 (<CJK Ideograph>)
      4147   => 16#7539#,  --  甹 (<CJK Ideograph>)
      4148   => 16#753D#,  --  甽 (<CJK Ideograph>)
      4149   => 16#753E#,  --  甾 (<CJK Ideograph>)
      4150   => 16#753F#,  --  甿 (<CJK Ideograph>)
      4151   => 16#7540#,  --  畀 (<CJK Ideograph>)
      4152   => 16#7543#,  --  畃 (<CJK Ideograph>)
      4153   => 16#7547#,  --  畇 (<CJK Ideograph>)
      4154   => 16#7548#,  --  畈 (<CJK Ideograph>)
      4155   => 16#754E#,  --  畎 (<CJK Ideograph>)
      4156   => 16#7550#,  --  畐 (<CJK Ideograph>)
      4157   => 16#7552#,  --  畒 (<CJK Ideograph>)
      4158   => 16#7557#,  --  畗 (<CJK Ideograph>)
      4159   => 16#755E#,  --  畞 (<CJK Ideograph>)
      4160   => 16#755F#,  --  畟 (<CJK Ideograph>)
      4161   => 16#7561#,  --  畡 (<CJK Ideograph>)
      4162   => 16#756F#,  --  畯 (<CJK Ideograph>)
      4163   => 16#7571#,  --  畱 (<CJK Ideograph>)
      4164   => 16#7579#,  --  畹 (<CJK Ideograph>)
      4165   => 16#757A#,  --  畺 (<CJK Ideograph>)
      4166   => 16#757B#,  --  畻 (<CJK Ideograph>)
      4167   => 16#757C#,  --  畼 (<CJK Ideograph>)
      4168   => 16#757D#,  --  畽 (<CJK Ideograph>)
      4169   => 16#757E#,  --  畾 (<CJK Ideograph>)
      4170   => 16#7581#,  --  疁 (<CJK Ideograph>)
      4171   => 16#7585#,  --  疅 (<CJK Ideograph>)
      4172   => 16#7590#,  --  疐 (<CJK Ideograph>)
      4173   => 16#7592#,  --  疒 (<CJK Ideograph>)
      4174   => 16#7593#,  --  疓 (<CJK Ideograph>)
      4175   => 16#7595#,  --  疕 (<CJK Ideograph>)
      4176   => 16#7599#,  --  疙 (<CJK Ideograph>)
      4177   => 16#759C#,  --  疜 (<CJK Ideograph>)
      4178   => 16#75A2#,  --  疢 (<CJK Ideograph>)
      4179   => 16#75A4#,  --  疤 (<CJK Ideograph>)
      4180   => 16#75B4#,  --  疴 (<CJK Ideograph>)
      4181   => 16#75BA#,  --  疺 (<CJK Ideograph>)
      4182   => 16#75BF#,  --  疿 (<CJK Ideograph>)
      4183   => 16#75C0#,  --  痀 (<CJK Ideograph>)
      4184   => 16#75C1#,  --  痁 (<CJK Ideograph>)
      4185   => 16#75C4#,  --  痄 (<CJK Ideograph>)
      4186   => 16#75C6#,  --  痆 (<CJK Ideograph>)
      4187   => 16#75CC#,  --  痌 (<CJK Ideograph>)
      4188   => 16#75CE#,  --  痎 (<CJK Ideograph>)
      4189   => 16#75CF#,  --  痏 (<CJK Ideograph>)
      4190   => 16#75D7#,  --  痗 (<CJK Ideograph>)
      4191   => 16#75DC#,  --  痜 (<CJK Ideograph>)
      4192   => 16#75DF#,  --  痟 (<CJK Ideograph>)
      4193   => 16#75E0#,  --  痠 (<CJK Ideograph>)
      4194   => 16#75E1#,  --  痡 (<CJK Ideograph>)
      4195   => 16#75E4#,  --  痤 (<CJK Ideograph>)
      4196   => 16#75E7#,  --  痧 (<CJK Ideograph>)
      4197   => 16#75EC#,  --  痬 (<CJK Ideograph>)
      4198   => 16#75EE#,  --  痮 (<CJK Ideograph>)
      4199   => 16#75EF#,  --  痯 (<CJK Ideograph>)
      4200   => 16#75F1#,  --  痱 (<CJK Ideograph>)
      4201   => 16#75F9#,  --  痹 (<CJK Ideograph>)
      4202   => 16#7600#,  --  瘀 (<CJK Ideograph>)
      4203   => 16#7602#,  --  瘂 (<CJK Ideograph>)
      4204   => 16#7603#,  --  瘃 (<CJK Ideograph>)
      4205   => 16#7604#,  --  瘄 (<CJK Ideograph>)
      4206   => 16#7607#,  --  瘇 (<CJK Ideograph>)
      4207   => 16#7608#,  --  瘈 (<CJK Ideograph>)
      4208   => 16#760A#,  --  瘊 (<CJK Ideograph>)
      4209   => 16#760C#,  --  瘌 (<CJK Ideograph>)
      4210   => 16#760F#,  --  瘏 (<CJK Ideograph>)
      4211   => 16#7612#,  --  瘒 (<CJK Ideograph>)
      4212   => 16#7613#,  --  瘓 (<CJK Ideograph>)
      4213   => 16#7615#,  --  瘕 (<CJK Ideograph>)
      4214   => 16#7616#,  --  瘖 (<CJK Ideograph>)
      4215   => 16#7619#,  --  瘙 (<CJK Ideograph>)
      4216   => 16#761B#,  --  瘛 (<CJK Ideograph>)
      4217   => 16#761C#,  --  瘜 (<CJK Ideograph>)
      4218   => 16#761D#,  --  瘝 (<CJK Ideograph>)
      4219   => 16#761E#,  --  瘞 (<CJK Ideograph>)
      4220   => 16#7623#,  --  瘣 (<CJK Ideograph>)
      4221   => 16#7625#,  --  瘥 (<CJK Ideograph>)
      4222   => 16#7626#,  --  瘦 (<CJK Ideograph>)
      4223   => 16#7629#,  --  瘩 (<CJK Ideograph>)
      4224   => 16#762D#,  --  瘭 (<CJK Ideograph>)
      4225   => 16#7632#,  --  瘲 (<CJK Ideograph>)
      4226   => 16#7633#,  --  瘳 (<CJK Ideograph>)
      4227   => 16#7635#,  --  瘵 (<CJK Ideograph>)
      4228   => 16#7638#,  --  瘸 (<CJK Ideograph>)
      4229   => 16#7639#,  --  瘹 (<CJK Ideograph>)
      4230   => 16#763A#,  --  瘺 (<CJK Ideograph>)
      4231   => 16#763C#,  --  瘼 (<CJK Ideograph>)
      4232   => 16#764A#,  --  癊 (<CJK Ideograph>)
      4233   => 16#7640#,  --  癀 (<CJK Ideograph>)
      4234   => 16#7641#,  --  癁 (<CJK Ideograph>)
      4235   => 16#7643#,  --  癃 (<CJK Ideograph>)
      4236   => 16#7644#,  --  癄 (<CJK Ideograph>)
      4237   => 16#7645#,  --  癅 (<CJK Ideograph>)
      4238   => 16#7649#,  --  癉 (<CJK Ideograph>)
      4239   => 16#764B#,  --  癋 (<CJK Ideograph>)
      4240   => 16#7655#,  --  癕 (<CJK Ideograph>)
      4241   => 16#7659#,  --  癙 (<CJK Ideograph>)
      4242   => 16#765F#,  --  癟 (<CJK Ideograph>)
      4243   => 16#7664#,  --  癤 (<CJK Ideograph>)
      4244   => 16#7665#,  --  癥 (<CJK Ideograph>)
      4245   => 16#766D#,  --  癭 (<CJK Ideograph>)
      4246   => 16#766E#,  --  癮 (<CJK Ideograph>)
      4247   => 16#766F#,  --  癯 (<CJK Ideograph>)
      4248   => 16#7671#,  --  癱 (<CJK Ideograph>)
      4249   => 16#7674#,  --  癴 (<CJK Ideograph>)
      4250   => 16#7681#,  --  皁 (<CJK Ideograph>)
      4251   => 16#7685#,  --  皅 (<CJK Ideograph>)
      4252   => 16#768C#,  --  皌 (<CJK Ideograph>)
      4253   => 16#768D#,  --  皍 (<CJK Ideograph>)
      4254   => 16#7695#,  --  皕 (<CJK Ideograph>)
      4255   => 16#769B#,  --  皛 (<CJK Ideograph>)
      4256   => 16#769C#,  --  皜 (<CJK Ideograph>)
      4257   => 16#769D#,  --  皝 (<CJK Ideograph>)
      4258   => 16#769F#,  --  皟 (<CJK Ideograph>)
      4259   => 16#76A0#,  --  皠 (<CJK Ideograph>)
      4260   => 16#76A2#,  --  皢 (<CJK Ideograph>)
      4261   => 16#76A3#,  --  皣 (<CJK Ideograph>)
      4262   => 16#76A4#,  --  皤 (<CJK Ideograph>)
      4263   => 16#76A5#,  --  皥 (<CJK Ideograph>)
      4264   => 16#76A6#,  --  皦 (<CJK Ideograph>)
      4265   => 16#76A7#,  --  皧 (<CJK Ideograph>)
      4266   => 16#76A8#,  --  皨 (<CJK Ideograph>)
      4267   => 16#76AA#,  --  皪 (<CJK Ideograph>)
      4268   => 16#76AD#,  --  皭 (<CJK Ideograph>)
      4269   => 16#76BD#,  --  皽 (<CJK Ideograph>)
      4270   => 16#76C1#,  --  盁 (<CJK Ideograph>)
      4271   => 16#76C5#,  --  盅 (<CJK Ideograph>)
      4272   => 16#76C9#,  --  盉 (<CJK Ideograph>)
      4273   => 16#76CB#,  --  盋 (<CJK Ideograph>)
      4274   => 16#76CC#,  --  盌 (<CJK Ideograph>)
      4275   => 16#76CE#,  --  盎 (<CJK Ideograph>)
      4276   => 16#76D4#,  --  盔 (<CJK Ideograph>)
      4277   => 16#76D9#,  --  盙 (<CJK Ideograph>)
      4278   => 16#76E0#,  --  盠 (<CJK Ideograph>)
      4279   => 16#76E6#,  --  盦 (<CJK Ideograph>)
      4280   => 16#76E8#,  --  盨 (<CJK Ideograph>)
      4281   => 16#76EC#,  --  盬 (<CJK Ideograph>)
      4282   => 16#76F0#,  --  盰 (<CJK Ideograph>)
      4283   => 16#76F1#,  --  盱 (<CJK Ideograph>)
      4284   => 16#76F6#,  --  盶 (<CJK Ideograph>)
      4285   => 16#76F9#,  --  盹 (<CJK Ideograph>)
      4286   => 16#76FC#,  --  盼 (<CJK Ideograph>)
      4287   => 16#7700#,  --  眀 (<CJK Ideograph>)
      4288   => 16#7706#,  --  眆 (<CJK Ideograph>)
      4289   => 16#770A#,  --  眊 (<CJK Ideograph>)
      4290   => 16#770E#,  --  眎 (<CJK Ideograph>)
      4291   => 16#7712#,  --  眒 (<CJK Ideograph>)
      4292   => 16#7714#,  --  眔 (<CJK Ideograph>)
      4293   => 16#7715#,  --  眕 (<CJK Ideograph>)
      4294   => 16#7717#,  --  眗 (<CJK Ideograph>)
      4295   => 16#7719#,  --  眙 (<CJK Ideograph>)
      4296   => 16#771A#,  --  眚 (<CJK Ideograph>)
      4297   => 16#771C#,  --  眜 (<CJK Ideograph>)
      4298   => 16#7722#,  --  眢 (<CJK Ideograph>)
      4299   => 16#7728#,  --  眨 (<CJK Ideograph>)
      4300   => 16#772D#,  --  眭 (<CJK Ideograph>)
      4301   => 16#772E#,  --  眮 (<CJK Ideograph>)
      4302   => 16#772F#,  --  眯 (<CJK Ideograph>)
      4303   => 16#7734#,  --  眴 (<CJK Ideograph>)
      4304   => 16#7735#,  --  眵 (<CJK Ideograph>)
      4305   => 16#7736#,  --  眶 (<CJK Ideograph>)
      4306   => 16#7739#,  --  眹 (<CJK Ideograph>)
      4307   => 16#773D#,  --  眽 (<CJK Ideograph>)
      4308   => 16#773E#,  --  眾 (<CJK Ideograph>)
      4309   => 16#7742#,  --  睂 (<CJK Ideograph>)
      4310   => 16#7745#,  --  睅 (<CJK Ideograph>)
      4311   => 16#7746#,  --  睆 (<CJK Ideograph>)
      4312   => 16#774A#,  --  睊 (<CJK Ideograph>)
      4313   => 16#774D#,  --  睍 (<CJK Ideograph>)
      4314   => 16#774E#,  --  睎 (<CJK Ideograph>)
      4315   => 16#774F#,  --  睏 (<CJK Ideograph>)
      4316   => 16#7752#,  --  睒 (<CJK Ideograph>)
      4317   => 16#7756#,  --  睖 (<CJK Ideograph>)
      4318   => 16#7757#,  --  睗 (<CJK Ideograph>)
      4319   => 16#775C#,  --  睜 (<CJK Ideograph>)
      4320   => 16#775E#,  --  睞 (<CJK Ideograph>)
      4321   => 16#775F#,  --  睟 (<CJK Ideograph>)
      4322   => 16#7760#,  --  睠 (<CJK Ideograph>)
      4323   => 16#7762#,  --  睢 (<CJK Ideograph>)
      4324   => 16#7764#,  --  睤 (<CJK Ideograph>)
      4325   => 16#7767#,  --  睧 (<CJK Ideograph>)
      4326   => 16#776A#,  --  睪 (<CJK Ideograph>)
      4327   => 16#776C#,  --  睬 (<CJK Ideograph>)
      4328   => 16#7770#,  --  睰 (<CJK Ideograph>)
      4329   => 16#7772#,  --  睲 (<CJK Ideograph>)
      4330   => 16#7773#,  --  睳 (<CJK Ideograph>)
      4331   => 16#7774#,  --  睴 (<CJK Ideograph>)
      4332   => 16#777A#,  --  睺 (<CJK Ideograph>)
      4333   => 16#777D#,  --  睽 (<CJK Ideograph>)
      4334   => 16#7780#,  --  瞀 (<CJK Ideograph>)
      4335   => 16#7784#,  --  瞄 (<CJK Ideograph>)
      4336   => 16#778C#,  --  瞌 (<CJK Ideograph>)
      4337   => 16#778D#,  --  瞍 (<CJK Ideograph>)
      4338   => 16#7794#,  --  瞔 (<CJK Ideograph>)
      4339   => 16#7795#,  --  瞕 (<CJK Ideograph>)
      4340   => 16#7796#,  --  瞖 (<CJK Ideograph>)
      4341   => 16#779A#,  --  瞚 (<CJK Ideograph>)
      4342   => 16#779F#,  --  瞟 (<CJK Ideograph>)
      4343   => 16#77A2#,  --  瞢 (<CJK Ideograph>)
      4344   => 16#77A7#,  --  瞧 (<CJK Ideograph>)
      4345   => 16#77AA#,  --  瞪 (<CJK Ideograph>)
      4346   => 16#77AE#,  --  瞮 (<CJK Ideograph>)
      4347   => 16#77AF#,  --  瞯 (<CJK Ideograph>)
      4348   => 16#77B1#,  --  瞱 (<CJK Ideograph>)
      4349   => 16#77B5#,  --  瞵 (<CJK Ideograph>)
      4350   => 16#77BE#,  --  瞾 (<CJK Ideograph>)
      4351   => 16#77C3#,  --  矃 (<CJK Ideograph>)
      4352   => 16#77C9#,  --  矉 (<CJK Ideograph>)
      4353   => 16#77D1#,  --  矑 (<CJK Ideograph>)
      4354   => 16#77D2#,  --  矒 (<CJK Ideograph>)
      4355   => 16#77D5#,  --  矕 (<CJK Ideograph>)
      4356   => 16#77D9#,  --  矙 (<CJK Ideograph>)
      4357   => 16#77DE#,  --  矞 (<CJK Ideograph>)
      4358   => 16#77DF#,  --  矟 (<CJK Ideograph>)
      4359   => 16#77E0#,  --  矠 (<CJK Ideograph>)
      4360   => 16#77E4#,  --  矤 (<CJK Ideograph>)
      4361   => 16#77E6#,  --  矦 (<CJK Ideograph>)
      4362   => 16#77EA#,  --  矪 (<CJK Ideograph>)
      4363   => 16#77EC#,  --  矬 (<CJK Ideograph>)
      4364   => 16#77F0#,  --  矰 (<CJK Ideograph>)
      4365   => 16#77F1#,  --  矱 (<CJK Ideograph>)
      4366   => 16#77F4#,  --  矴 (<CJK Ideograph>)
      4367   => 16#77F8#,  --  矸 (<CJK Ideograph>)
      4368   => 16#77FB#,  --  矻 (<CJK Ideograph>)
      4369   => 16#7805#,  --  砅 (<CJK Ideograph>)
      4370   => 16#7806#,  --  砆 (<CJK Ideograph>)
      4371   => 16#7809#,  --  砉 (<CJK Ideograph>)
      4372   => 16#780D#,  --  砍 (<CJK Ideograph>)
      4373   => 16#780E#,  --  砎 (<CJK Ideograph>)
      4374   => 16#7811#,  --  砑 (<CJK Ideograph>)
      4375   => 16#781D#,  --  砝 (<CJK Ideograph>)
      4376   => 16#7821#,  --  砡 (<CJK Ideograph>)
      4377   => 16#7822#,  --  砢 (<CJK Ideograph>)
      4378   => 16#7823#,  --  砣 (<CJK Ideograph>)
      4379   => 16#782D#,  --  砭 (<CJK Ideograph>)
      4380   => 16#782E#,  --  砮 (<CJK Ideograph>)
      4381   => 16#7830#,  --  砰 (<CJK Ideograph>)
      4382   => 16#7835#,  --  砵 (<CJK Ideograph>)
      4383   => 16#7837#,  --  砷 (<CJK Ideograph>)
      4384   => 16#7843#,  --  硃 (<CJK Ideograph>)
      4385   => 16#7844#,  --  硄 (<CJK Ideograph>)
      4386   => 16#7847#,  --  硇 (<CJK Ideograph>)
      4387   => 16#7848#,  --  硈 (<CJK Ideograph>)
      4388   => 16#784C#,  --  硌 (<CJK Ideograph>)
      4389   => 16#784E#,  --  硎 (<CJK Ideograph>)
      4390   => 16#7852#,  --  硒 (<CJK Ideograph>)
      4391   => 16#785C#,  --  硜 (<CJK Ideograph>)
      4392   => 16#785E#,  --  硞 (<CJK Ideograph>)
      4393   => 16#7860#,  --  硠 (<CJK Ideograph>)
      4394   => 16#7861#,  --  硡 (<CJK Ideograph>)
      4395   => 16#7863#,  --  硣 (<CJK Ideograph>)
      4396   => 16#7864#,  --  硤 (<CJK Ideograph>)
      4397   => 16#7868#,  --  硨 (<CJK Ideograph>)
      4398   => 16#786A#,  --  硪 (<CJK Ideograph>)
      4399   => 16#786E#,  --  确 (<CJK Ideograph>)
      4400   => 16#787A#,  --  硺 (<CJK Ideograph>)
      4401   => 16#787E#,  --  硾 (<CJK Ideograph>)
      4402   => 16#788A#,  --  碊 (<CJK Ideograph>)
      4403   => 16#788F#,  --  碏 (<CJK Ideograph>)
      4404   => 16#7894#,  --  碔 (<CJK Ideograph>)
      4405   => 16#7898#,  --  碘 (<CJK Ideograph>)
      4406   => 16#78A1#,  --  碡 (<CJK Ideograph>)
      4407   => 16#789D#,  --  碝 (<CJK Ideograph>)
      4408   => 16#789E#,  --  碞 (<CJK Ideograph>)
      4409   => 16#789F#,  --  碟 (<CJK Ideograph>)
      4410   => 16#78A4#,  --  碤 (<CJK Ideograph>)
      4411   => 16#78A8#,  --  碨 (<CJK Ideograph>)
      4412   => 16#78AC#,  --  碬 (<CJK Ideograph>)
      4413   => 16#78AD#,  --  碭 (<CJK Ideograph>)
      4414   => 16#78B0#,  --  碰 (<CJK Ideograph>)
      4415   => 16#78B1#,  --  碱 (<CJK Ideograph>)
      4416   => 16#78B2#,  --  碲 (<CJK Ideograph>)
      4417   => 16#78B3#,  --  碳 (<CJK Ideograph>)
      4418   => 16#78BB#,  --  碻 (<CJK Ideograph>)
      4419   => 16#78BD#,  --  碽 (<CJK Ideograph>)
      4420   => 16#78BF#,  --  碿 (<CJK Ideograph>)
      4421   => 16#78C7#,  --  磇 (<CJK Ideograph>)
      4422   => 16#78C8#,  --  磈 (<CJK Ideograph>)
      4423   => 16#78C9#,  --  磉 (<CJK Ideograph>)
      4424   => 16#78CC#,  --  磌 (<CJK Ideograph>)
      4425   => 16#78CE#,  --  磎 (<CJK Ideograph>)
      4426   => 16#78D2#,  --  磒 (<CJK Ideograph>)
      4427   => 16#78D3#,  --  磓 (<CJK Ideograph>)
      4428   => 16#78D5#,  --  磕 (<CJK Ideograph>)
      4429   => 16#78D6#,  --  磖 (<CJK Ideograph>)
      4430   => 16#78E4#,  --  磤 (<CJK Ideograph>)
      4431   => 16#78DB#,  --  磛 (<CJK Ideograph>)
      4432   => 16#78DF#,  --  磟 (<CJK Ideograph>)
      4433   => 16#78E0#,  --  磠 (<CJK Ideograph>)
      4434   => 16#78E1#,  --  磡 (<CJK Ideograph>)
      4435   => 16#78E6#,  --  磦 (<CJK Ideograph>)
      4436   => 16#78EA#,  --  磪 (<CJK Ideograph>)
      4437   => 16#78F2#,  --  磲 (<CJK Ideograph>)
      4438   => 16#78F3#,  --  磳 (<CJK Ideograph>)
      4439   => 16#7900#,  --  礀 (<CJK Ideograph>)
      4440   => 16#78F6#,  --  磶 (<CJK Ideograph>)
      4441   => 16#78F7#,  --  磷 (<CJK Ideograph>)
      4442   => 16#78FA#,  --  磺 (<CJK Ideograph>)
      4443   => 16#78FB#,  --  磻 (<CJK Ideograph>)
      4444   => 16#78FF#,  --  磿 (<CJK Ideograph>)
      4445   => 16#7906#,  --  礆 (<CJK Ideograph>)
      4446   => 16#790C#,  --  礌 (<CJK Ideograph>)
      4447   => 16#7910#,  --  礐 (<CJK Ideograph>)
      4448   => 16#791A#,  --  礚 (<CJK Ideograph>)
      4449   => 16#791C#,  --  礜 (<CJK Ideograph>)
      4450   => 16#791E#,  --  礞 (<CJK Ideograph>)
      4451   => 16#791F#,  --  礟 (<CJK Ideograph>)
      4452   => 16#7920#,  --  礠 (<CJK Ideograph>)
      4453   => 16#7925#,  --  礥 (<CJK Ideograph>)
      4454   => 16#7927#,  --  礧 (<CJK Ideograph>)
      4455   => 16#7929#,  --  礩 (<CJK Ideograph>)
      4456   => 16#792D#,  --  礭 (<CJK Ideograph>)
      4457   => 16#7931#,  --  礱 (<CJK Ideograph>)
      4458   => 16#7934#,  --  礴 (<CJK Ideograph>)
      4459   => 16#7935#,  --  礵 (<CJK Ideograph>)
      4460   => 16#793B#,  --  礻 (<CJK Ideograph>)
      4461   => 16#793D#,  --  礽 (<CJK Ideograph>)
      4462   => 16#793F#,  --  礿 (<CJK Ideograph>)
      4463   => 16#7944#,  --  祄 (<CJK Ideograph>)
      4464   => 16#7945#,  --  祅 (<CJK Ideograph>)
      4465   => 16#7946#,  --  祆 (<CJK Ideograph>)
      4466   => 16#794A#,  --  祊 (<CJK Ideograph>)
      4467   => 16#794B#,  --  祋 (<CJK Ideograph>)
      4468   => 16#794F#,  --  祏 (<CJK Ideograph>)
      4469   => 16#7951#,  --  祑 (<CJK Ideograph>)
      4470   => 16#7954#,  --  祔 (<CJK Ideograph>)
      4471   => 16#7958#,  --  祘 (<CJK Ideograph>)
      4472   => 16#795B#,  --  祛 (<CJK Ideograph>)
      4473   => 16#795C#,  --  祜 (<CJK Ideograph>)
      4474   => 16#7967#,  --  祧 (<CJK Ideograph>)
      4475   => 16#7969#,  --  祩 (<CJK Ideograph>)
      4476   => 16#796B#,  --  祫 (<CJK Ideograph>)
      4477   => 16#7972#,  --  祲 (<CJK Ideograph>)
      4478   => 16#7979#,  --  祹 (<CJK Ideograph>)
      4479   => 16#797B#,  --  祻 (<CJK Ideograph>)
      4480   => 16#797C#,  --  祼 (<CJK Ideograph>)
      4481   => 16#797E#,  --  祾 (<CJK Ideograph>)
      4482   => 16#798B#,  --  禋 (<CJK Ideograph>)
      4483   => 16#798C#,  --  禌 (<CJK Ideograph>)
      4484   => 16#7991#,  --  禑 (<CJK Ideograph>)
      4485   => 16#7993#,  --  禓 (<CJK Ideograph>)
      4486   => 16#7994#,  --  禔 (<CJK Ideograph>)
      4487   => 16#7995#,  --  禕 (<CJK Ideograph>)
      4488   => 16#7996#,  --  禖 (<CJK Ideograph>)
      4489   => 16#7998#,  --  禘 (<CJK Ideograph>)
      4490   => 16#799B#,  --  禛 (<CJK Ideograph>)
      4491   => 16#799C#,  --  禜 (<CJK Ideograph>)
      4492   => 16#79A1#,  --  禡 (<CJK Ideograph>)
      4493   => 16#79A8#,  --  禨 (<CJK Ideograph>)
      4494   => 16#79A9#,  --  禩 (<CJK Ideograph>)
      4495   => 16#79AB#,  --  禫 (<CJK Ideograph>)
      4496   => 16#79AF#,  --  禯 (<CJK Ideograph>)
      4497   => 16#79B1#,  --  禱 (<CJK Ideograph>)
      4498   => 16#79B4#,  --  禴 (<CJK Ideograph>)
      4499   => 16#79B8#,  --  禸 (<CJK Ideograph>)
      4500   => 16#79BB#,  --  离 (<CJK Ideograph>)
      4501   => 16#79C2#,  --  秂 (<CJK Ideograph>)
      4502   => 16#79C4#,  --  秄 (<CJK Ideograph>)
      4503   => 16#79C7#,  --  秇 (<CJK Ideograph>)
      4504   => 16#79C8#,  --  秈 (<CJK Ideograph>)
      4505   => 16#79CA#,  --  秊 (<CJK Ideograph>)
      4506   => 16#79CF#,  --  秏 (<CJK Ideograph>)
      4507   => 16#79D4#,  --  秔 (<CJK Ideograph>)
      4508   => 16#79D6#,  --  秖 (<CJK Ideograph>)
      4509   => 16#79DA#,  --  秚 (<CJK Ideograph>)
      4510   => 16#79DD#,  --  秝 (<CJK Ideograph>)
      4511   => 16#79DE#,  --  秞 (<CJK Ideograph>)
      4512   => 16#79E0#,  --  秠 (<CJK Ideograph>)
      4513   => 16#79E2#,  --  秢 (<CJK Ideograph>)
      4514   => 16#79E5#,  --  秥 (<CJK Ideograph>)
      4515   => 16#79EA#,  --  秪 (<CJK Ideograph>)
      4516   => 16#79EB#,  --  秫 (<CJK Ideograph>)
      4517   => 16#79ED#,  --  秭 (<CJK Ideograph>)
      4518   => 16#79F1#,  --  秱 (<CJK Ideograph>)
      4519   => 16#79F8#,  --  秸 (<CJK Ideograph>)
      4520   => 16#79FC#,  --  秼 (<CJK Ideograph>)
      4521   => 16#7A02#,  --  稂 (<CJK Ideograph>)
      4522   => 16#7A03#,  --  稃 (<CJK Ideograph>)
      4523   => 16#7A07#,  --  稇 (<CJK Ideograph>)
      4524   => 16#7A09#,  --  稉 (<CJK Ideograph>)
      4525   => 16#7A0A#,  --  稊 (<CJK Ideograph>)
      4526   => 16#7A0C#,  --  稌 (<CJK Ideograph>)
      4527   => 16#7A11#,  --  稑 (<CJK Ideograph>)
      4528   => 16#7A15#,  --  稕 (<CJK Ideograph>)
      4529   => 16#7A1B#,  --  稛 (<CJK Ideograph>)
      4530   => 16#7A1E#,  --  稞 (<CJK Ideograph>)
      4531   => 16#7A21#,  --  稡 (<CJK Ideograph>)
      4532   => 16#7A27#,  --  稧 (<CJK Ideograph>)
      4533   => 16#7A2B#,  --  稫 (<CJK Ideograph>)
      4534   => 16#7A2D#,  --  稭 (<CJK Ideograph>)
      4535   => 16#7A2F#,  --  稯 (<CJK Ideograph>)
      4536   => 16#7A30#,  --  稰 (<CJK Ideograph>)
      4537   => 16#7A34#,  --  稴 (<CJK Ideograph>)
      4538   => 16#7A35#,  --  稵 (<CJK Ideograph>)
      4539   => 16#7A38#,  --  稸 (<CJK Ideograph>)
      4540   => 16#7A39#,  --  稹 (<CJK Ideograph>)
      4541   => 16#7A3A#,  --  稺 (<CJK Ideograph>)
      4542   => 16#7A44#,  --  穄 (<CJK Ideograph>)
      4543   => 16#7A45#,  --  穅 (<CJK Ideograph>)
      4544   => 16#7A47#,  --  穇 (<CJK Ideograph>)
      4545   => 16#7A48#,  --  穈 (<CJK Ideograph>)
      4546   => 16#7A4C#,  --  穌 (<CJK Ideograph>)
      4547   => 16#7A55#,  --  穕 (<CJK Ideograph>)
      4548   => 16#7A56#,  --  穖 (<CJK Ideograph>)
      4549   => 16#7A59#,  --  穙 (<CJK Ideograph>)
      4550   => 16#7A5C#,  --  穜 (<CJK Ideograph>)
      4551   => 16#7A5D#,  --  穝 (<CJK Ideograph>)
      4552   => 16#7A5F#,  --  穟 (<CJK Ideograph>)
      4553   => 16#7A60#,  --  穠 (<CJK Ideograph>)
      4554   => 16#7A65#,  --  穥 (<CJK Ideograph>)
      4555   => 16#7A67#,  --  穧 (<CJK Ideograph>)
      4556   => 16#7A6A#,  --  穪 (<CJK Ideograph>)
      4557   => 16#7A6D#,  --  穭 (<CJK Ideograph>)
      4558   => 16#7A75#,  --  穵 (<CJK Ideograph>)
      4559   => 16#7A78#,  --  穸 (<CJK Ideograph>)
      4560   => 16#7A7E#,  --  穾 (<CJK Ideograph>)
      4561   => 16#7A80#,  --  窀 (<CJK Ideograph>)
      4562   => 16#7A82#,  --  窂 (<CJK Ideograph>)
      4563   => 16#7A85#,  --  窅 (<CJK Ideograph>)
      4564   => 16#7A86#,  --  窆 (<CJK Ideograph>)
      4565   => 16#7A8A#,  --  窊 (<CJK Ideograph>)
      4566   => 16#7A8B#,  --  窋 (<CJK Ideograph>)
      4567   => 16#7A90#,  --  窐 (<CJK Ideograph>)
      4568   => 16#7A91#,  --  窑 (<CJK Ideograph>)
      4569   => 16#7A94#,  --  窔 (<CJK Ideograph>)
      4570   => 16#7A9E#,  --  窞 (<CJK Ideograph>)
      4571   => 16#7AA0#,  --  窠 (<CJK Ideograph>)
      4572   => 16#7AA3#,  --  窣 (<CJK Ideograph>)
      4573   => 16#7AAC#,  --  窬 (<CJK Ideograph>)
      4574   => 16#7AB3#,  --  窳 (<CJK Ideograph>)
      4575   => 16#7AB5#,  --  窵 (<CJK Ideograph>)
      4576   => 16#7AB9#,  --  窹 (<CJK Ideograph>)
      4577   => 16#7ABB#,  --  窻 (<CJK Ideograph>)
      4578   => 16#7ABC#,  --  窼 (<CJK Ideograph>)
      4579   => 16#7AC6#,  --  竆 (<CJK Ideograph>)
      4580   => 16#7AC9#,  --  竉 (<CJK Ideograph>)
      4581   => 16#7ACC#,  --  竌 (<CJK Ideograph>)
      4582   => 16#7ACE#,  --  竎 (<CJK Ideograph>)
      4583   => 16#7AD1#,  --  竑 (<CJK Ideograph>)
      4584   => 16#7ADB#,  --  竛 (<CJK Ideograph>)
      4585   => 16#7AE8#,  --  竨 (<CJK Ideograph>)
      4586   => 16#7AE9#,  --  竩 (<CJK Ideograph>)
      4587   => 16#7AEB#,  --  竫 (<CJK Ideograph>)
      4588   => 16#7AEC#,  --  竬 (<CJK Ideograph>)
      4589   => 16#7AF1#,  --  竱 (<CJK Ideograph>)
      4590   => 16#7AF4#,  --  竴 (<CJK Ideograph>)
      4591   => 16#7AFB#,  --  竻 (<CJK Ideograph>)
      4592   => 16#7AFD#,  --  竽 (<CJK Ideograph>)
      4593   => 16#7AFE#,  --  竾 (<CJK Ideograph>)
      4594   => 16#7B07#,  --  笇 (<CJK Ideograph>)
      4595   => 16#7B14#,  --  笔 (<CJK Ideograph>)
      4596   => 16#7B1F#,  --  笟 (<CJK Ideograph>)
      4597   => 16#7B23#,  --  笣 (<CJK Ideograph>)
      4598   => 16#7B27#,  --  笧 (<CJK Ideograph>)
      4599   => 16#7B29#,  --  笩 (<CJK Ideograph>)
      4600   => 16#7B2A#,  --  笪 (<CJK Ideograph>)
      4601   => 16#7B2B#,  --  笫 (<CJK Ideograph>)
      4602   => 16#7B2D#,  --  笭 (<CJK Ideograph>)
      4603   => 16#7B2E#,  --  笮 (<CJK Ideograph>)
      4604   => 16#7B2F#,  --  笯 (<CJK Ideograph>)
      4605   => 16#7B30#,  --  笰 (<CJK Ideograph>)
      4606   => 16#7B31#,  --  笱 (<CJK Ideograph>)
      4607   => 16#7B34#,  --  笴 (<CJK Ideograph>)
      4608   => 16#7B3D#,  --  笽 (<CJK Ideograph>)
      4609   => 16#7B3F#,  --  笿 (<CJK Ideograph>)
      4610   => 16#7B40#,  --  筀 (<CJK Ideograph>)
      4611   => 16#7B41#,  --  筁 (<CJK Ideograph>)
      4612   => 16#7B47#,  --  筇 (<CJK Ideograph>)
      4613   => 16#7B4E#,  --  筎 (<CJK Ideograph>)
      4614   => 16#7B55#,  --  筕 (<CJK Ideograph>)
      4615   => 16#7B60#,  --  筠 (<CJK Ideograph>)
      4616   => 16#7B64#,  --  筤 (<CJK Ideograph>)
      4617   => 16#7B66#,  --  筦 (<CJK Ideograph>)
      4618   => 16#7B69#,  --  筩 (<CJK Ideograph>)
      4619   => 16#7B6A#,  --  筪 (<CJK Ideograph>)
      4620   => 16#7B6D#,  --  筭 (<CJK Ideograph>)
      4621   => 16#7B6F#,  --  筯 (<CJK Ideograph>)
      4622   => 16#7B72#,  --  筲 (<CJK Ideograph>)
      4623   => 16#7B73#,  --  筳 (<CJK Ideograph>)
      4624   => 16#7B77#,  --  筷 (<CJK Ideograph>)
      4625   => 16#7B84#,  --  箄 (<CJK Ideograph>)
      4626   => 16#7B89#,  --  箉 (<CJK Ideograph>)
      4627   => 16#7B8E#,  --  箎 (<CJK Ideograph>)
      4628   => 16#7B90#,  --  箐 (<CJK Ideograph>)
      4629   => 16#7B91#,  --  箑 (<CJK Ideograph>)
      4630   => 16#7B96#,  --  箖 (<CJK Ideograph>)
      4631   => 16#7B9B#,  --  箛 (<CJK Ideograph>)
      4632   => 16#7B9E#,  --  箞 (<CJK Ideograph>)
      4633   => 16#7BA0#,  --  箠 (<CJK Ideograph>)
      4634   => 16#7BA5#,  --  箥 (<CJK Ideograph>)
      4635   => 16#7BAC#,  --  箬 (<CJK Ideograph>)
      4636   => 16#7BAF#,  --  箯 (<CJK Ideograph>)
      4637   => 16#7BB0#,  --  箰 (<CJK Ideograph>)
      4638   => 16#7BB2#,  --  箲 (<CJK Ideograph>)
      4639   => 16#7BB5#,  --  箵 (<CJK Ideograph>)
      4640   => 16#7BB6#,  --  箶 (<CJK Ideograph>)
      4641   => 16#7BBA#,  --  箺 (<CJK Ideograph>)
      4642   => 16#7BBB#,  --  箻 (<CJK Ideograph>)
      4643   => 16#7BBC#,  --  箼 (<CJK Ideograph>)
      4644   => 16#7BBD#,  --  箽 (<CJK Ideograph>)
      4645   => 16#7BC2#,  --  篂 (<CJK Ideograph>)
      4646   => 16#7BC5#,  --  篅 (<CJK Ideograph>)
      4647   => 16#7BC8#,  --  篈 (<CJK Ideograph>)
      4648   => 16#7BCA#,  --  篊 (<CJK Ideograph>)
      4649   => 16#7BD4#,  --  篔 (<CJK Ideograph>)
      4650   => 16#7BD6#,  --  篖 (<CJK Ideograph>)
      4651   => 16#7BD7#,  --  篗 (<CJK Ideograph>)
      4652   => 16#7BD9#,  --  篙 (<CJK Ideograph>)
      4653   => 16#7BDA#,  --  篚 (<CJK Ideograph>)
      4654   => 16#7BDB#,  --  篛 (<CJK Ideograph>)
      4655   => 16#7BE8#,  --  篨 (<CJK Ideograph>)
      4656   => 16#7BEA#,  --  篪 (<CJK Ideograph>)
      4657   => 16#7BF2#,  --  篲 (<CJK Ideograph>)
      4658   => 16#7BF4#,  --  篴 (<CJK Ideograph>)
      4659   => 16#7BF5#,  --  篵 (<CJK Ideograph>)
      4660   => 16#7BF8#,  --  篸 (<CJK Ideograph>)
      4661   => 16#7BF9#,  --  篹 (<CJK Ideograph>)
      4662   => 16#7BFA#,  --  篺 (<CJK Ideograph>)
      4663   => 16#7BFC#,  --  篼 (<CJK Ideograph>)
      4664   => 16#7BFE#,  --  篾 (<CJK Ideograph>)
      4665   => 16#7C01#,  --  簁 (<CJK Ideograph>)
      4666   => 16#7C02#,  --  簂 (<CJK Ideograph>)
      4667   => 16#7C03#,  --  簃 (<CJK Ideograph>)
      4668   => 16#7C04#,  --  簄 (<CJK Ideograph>)
      4669   => 16#7C06#,  --  簆 (<CJK Ideograph>)
      4670   => 16#7C09#,  --  簉 (<CJK Ideograph>)
      4671   => 16#7C0B#,  --  簋 (<CJK Ideograph>)
      4672   => 16#7C0C#,  --  簌 (<CJK Ideograph>)
      4673   => 16#7C0E#,  --  簎 (<CJK Ideograph>)
      4674   => 16#7C0F#,  --  簏 (<CJK Ideograph>)
      4675   => 16#7C19#,  --  簙 (<CJK Ideograph>)
      4676   => 16#7C1B#,  --  簛 (<CJK Ideograph>)
      4677   => 16#7C20#,  --  簠 (<CJK Ideograph>)
      4678   => 16#7C25#,  --  簥 (<CJK Ideograph>)
      4679   => 16#7C26#,  --  簦 (<CJK Ideograph>)
      4680   => 16#7C28#,  --  簨 (<CJK Ideograph>)
      4681   => 16#7C2C#,  --  簬 (<CJK Ideograph>)
      4682   => 16#7C31#,  --  簱 (<CJK Ideograph>)
      4683   => 16#7C33#,  --  簳 (<CJK Ideograph>)
      4684   => 16#7C34#,  --  簴 (<CJK Ideograph>)
      4685   => 16#7C36#,  --  簶 (<CJK Ideograph>)
      4686   => 16#7C39#,  --  簹 (<CJK Ideograph>)
      4687   => 16#7C3A#,  --  簺 (<CJK Ideograph>)
      4688   => 16#7C46#,  --  籆 (<CJK Ideograph>)
      4689   => 16#7C4A#,  --  籊 (<CJK Ideograph>)
      4690   => 16#7C55#,  --  籕 (<CJK Ideograph>)
      4691   => 16#7C51#,  --  籑 (<CJK Ideograph>)
      4692   => 16#7C52#,  --  籒 (<CJK Ideograph>)
      4693   => 16#7C53#,  --  籓 (<CJK Ideograph>)
      4694   => 16#7C59#,  --  籙 (<CJK Ideograph>)
      4695   => 16#7C5A#,  --  籚 (<CJK Ideograph>)
      4696   => 16#7C5B#,  --  籛 (<CJK Ideograph>)
      4697   => 16#7C5C#,  --  籜 (<CJK Ideograph>)
      4698   => 16#7C5D#,  --  籝 (<CJK Ideograph>)
      4699   => 16#7C5E#,  --  籞 (<CJK Ideograph>)
      4700   => 16#7C61#,  --  籡 (<CJK Ideograph>)
      4701   => 16#7C63#,  --  籣 (<CJK Ideograph>)
      4702   => 16#7C67#,  --  籧 (<CJK Ideograph>)
      4703   => 16#7C69#,  --  籩 (<CJK Ideograph>)
      4704   => 16#7C6D#,  --  籭 (<CJK Ideograph>)
      4705   => 16#7C6E#,  --  籮 (<CJK Ideograph>)
      4706   => 16#7C70#,  --  籰 (<CJK Ideograph>)
      4707   => 16#7C72#,  --  籲 (<CJK Ideograph>)
      4708   => 16#7C79#,  --  籹 (<CJK Ideograph>)
      4709   => 16#7C7C#,  --  籼 (<CJK Ideograph>)
      4710   => 16#7C7D#,  --  籽 (<CJK Ideograph>)
      4711   => 16#7C86#,  --  粆 (<CJK Ideograph>)
      4712   => 16#7C87#,  --  粇 (<CJK Ideograph>)
      4713   => 16#7C8F#,  --  粏 (<CJK Ideograph>)
      4714   => 16#7C94#,  --  粔 (<CJK Ideograph>)
      4715   => 16#7C9E#,  --  粞 (<CJK Ideograph>)
      4716   => 16#7CA0#,  --  粠 (<CJK Ideograph>)
      4717   => 16#7CA6#,  --  粦 (<CJK Ideograph>)
      4718   => 16#7CB0#,  --  粰 (<CJK Ideograph>)
      4719   => 16#7CB6#,  --  粶 (<CJK Ideograph>)
      4720   => 16#7CB7#,  --  粷 (<CJK Ideograph>)
      4721   => 16#7CBA#,  --  粺 (<CJK Ideograph>)
      4722   => 16#7CBB#,  --  粻 (<CJK Ideograph>)
      4723   => 16#7CBC#,  --  粼 (<CJK Ideograph>)
      4724   => 16#7CBF#,  --  粿 (<CJK Ideograph>)
      4725   => 16#7CC4#,  --  糄 (<CJK Ideograph>)
      4726   => 16#7CC7#,  --  糇 (<CJK Ideograph>)
      4727   => 16#7CC8#,  --  糈 (<CJK Ideograph>)
      4728   => 16#7CC9#,  --  糉 (<CJK Ideograph>)
      4729   => 16#7CCD#,  --  糍 (<CJK Ideograph>)
      4730   => 16#7CCF#,  --  糏 (<CJK Ideograph>)
      4731   => 16#7CD3#,  --  糓 (<CJK Ideograph>)
      4732   => 16#7CD4#,  --  糔 (<CJK Ideograph>)
      4733   => 16#7CD5#,  --  糕 (<CJK Ideograph>)
      4734   => 16#7CD7#,  --  糗 (<CJK Ideograph>)
      4735   => 16#7CD9#,  --  糙 (<CJK Ideograph>)
      4736   => 16#7CDA#,  --  糚 (<CJK Ideograph>)
      4737   => 16#7CDD#,  --  糝 (<CJK Ideograph>)
      4738   => 16#7CE6#,  --  糦 (<CJK Ideograph>)
      4739   => 16#7CE9#,  --  糩 (<CJK Ideograph>)
      4740   => 16#7CEB#,  --  糫 (<CJK Ideograph>)
      4741   => 16#7CF5#,  --  糵 (<CJK Ideograph>)
      4742   => 16#7D03#,  --  紃 (<CJK Ideograph>)
      4743   => 16#7D07#,  --  紇 (<CJK Ideograph>)
      4744   => 16#7D08#,  --  紈 (<CJK Ideograph>)
      4745   => 16#7D09#,  --  紉 (<CJK Ideograph>)
      4746   => 16#7D0F#,  --  紏 (<CJK Ideograph>)
      4747   => 16#7D11#,  --  紑 (<CJK Ideograph>)
      4748   => 16#7D12#,  --  紒 (<CJK Ideograph>)
      4749   => 16#7D13#,  --  紓 (<CJK Ideograph>)
      4750   => 16#7D16#,  --  紖 (<CJK Ideograph>)
      4751   => 16#7D1D#,  --  紝 (<CJK Ideograph>)
      4752   => 16#7D1E#,  --  紞 (<CJK Ideograph>)
      4753   => 16#7D23#,  --  紣 (<CJK Ideograph>)
      4754   => 16#7D26#,  --  紦 (<CJK Ideograph>)
      4755   => 16#7D2A#,  --  紪 (<CJK Ideograph>)
      4756   => 16#7D2D#,  --  紭 (<CJK Ideograph>)
      4757   => 16#7D31#,  --  紱 (<CJK Ideograph>)
      4758   => 16#7D3C#,  --  紼 (<CJK Ideograph>)
      4759   => 16#7D3D#,  --  紽 (<CJK Ideograph>)
      4760   => 16#7D3E#,  --  紾 (<CJK Ideograph>)
      4761   => 16#7D40#,  --  絀 (<CJK Ideograph>)
      4762   => 16#7D41#,  --  絁 (<CJK Ideograph>)
      4763   => 16#7D47#,  --  絇 (<CJK Ideograph>)
      4764   => 16#7D48#,  --  絈 (<CJK Ideograph>)
      4765   => 16#7D4D#,  --  絍 (<CJK Ideograph>)
      4766   => 16#7D51#,  --  絑 (<CJK Ideograph>)
      4767   => 16#7D53#,  --  絓 (<CJK Ideograph>)
      4768   => 16#7D57#,  --  絗 (<CJK Ideograph>)
      4769   => 16#7D59#,  --  絙 (<CJK Ideograph>)
      4770   => 16#7D5A#,  --  絚 (<CJK Ideograph>)
      4771   => 16#7D5C#,  --  絜 (<CJK Ideograph>)
      4772   => 16#7D5D#,  --  絝 (<CJK Ideograph>)
      4773   => 16#7D65#,  --  絥 (<CJK Ideograph>)
      4774   => 16#7D67#,  --  絧 (<CJK Ideograph>)
      4775   => 16#7D6A#,  --  絪 (<CJK Ideograph>)
      4776   => 16#7D70#,  --  絰 (<CJK Ideograph>)
      4777   => 16#7D78#,  --  絸 (<CJK Ideograph>)
      4778   => 16#7D7A#,  --  絺 (<CJK Ideograph>)
      4779   => 16#7D7B#,  --  絻 (<CJK Ideograph>)
      4780   => 16#7D7F#,  --  絿 (<CJK Ideograph>)
      4781   => 16#7D81#,  --  綁 (<CJK Ideograph>)
      4782   => 16#7D82#,  --  綂 (<CJK Ideograph>)
      4783   => 16#7D83#,  --  綃 (<CJK Ideograph>)
      4784   => 16#7D85#,  --  綅 (<CJK Ideograph>)
      4785   => 16#7D86#,  --  綆 (<CJK Ideograph>)
      4786   => 16#7D88#,  --  綈 (<CJK Ideograph>)
      4787   => 16#7D8B#,  --  綋 (<CJK Ideograph>)
      4788   => 16#7D8C#,  --  綌 (<CJK Ideograph>)
      4789   => 16#7D8D#,  --  綍 (<CJK Ideograph>)
      4790   => 16#7D91#,  --  綑 (<CJK Ideograph>)
      4791   => 16#7D96#,  --  綖 (<CJK Ideograph>)
      4792   => 16#7D97#,  --  綗 (<CJK Ideograph>)
      4793   => 16#7D9D#,  --  綝 (<CJK Ideograph>)
      4794   => 16#7D9E#,  --  綞 (<CJK Ideograph>)
      4795   => 16#7DA6#,  --  綦 (<CJK Ideograph>)
      4796   => 16#7DA7#,  --  綧 (<CJK Ideograph>)
      4797   => 16#7DAA#,  --  綪 (<CJK Ideograph>)
      4798   => 16#7DB3#,  --  綳 (<CJK Ideograph>)
      4799   => 16#7DB6#,  --  綶 (<CJK Ideograph>)
      4800   => 16#7DB7#,  --  綷 (<CJK Ideograph>)
      4801   => 16#7DB9#,  --  綹 (<CJK Ideograph>)
      4802   => 16#7DC2#,  --  緂 (<CJK Ideograph>)
      4803   => 16#7DC3#,  --  緃 (<CJK Ideograph>)
      4804   => 16#7DC4#,  --  緄 (<CJK Ideograph>)
      4805   => 16#7DC5#,  --  緅 (<CJK Ideograph>)
      4806   => 16#7DC6#,  --  緆 (<CJK Ideograph>)
      4807   => 16#7DCC#,  --  緌 (<CJK Ideograph>)
      4808   => 16#7DCD#,  --  緍 (<CJK Ideograph>)
      4809   => 16#7DCE#,  --  緎 (<CJK Ideograph>)
      4810   => 16#7DD7#,  --  緗 (<CJK Ideograph>)
      4811   => 16#7DD9#,  --  緙 (<CJK Ideograph>)
      4812   => 16#7E00#,  --  縀 (<CJK Ideograph>)
      4813   => 16#7DE2#,  --  緢 (<CJK Ideograph>)
      4814   => 16#7DE5#,  --  緥 (<CJK Ideograph>)
      4815   => 16#7DE6#,  --  緦 (<CJK Ideograph>)
      4816   => 16#7DEA#,  --  緪 (<CJK Ideograph>)
      4817   => 16#7DEB#,  --  緫 (<CJK Ideograph>)
      4818   => 16#7DED#,  --  緭 (<CJK Ideograph>)
      4819   => 16#7DF1#,  --  緱 (<CJK Ideograph>)
      4820   => 16#7DF5#,  --  緵 (<CJK Ideograph>)
      4821   => 16#7DF6#,  --  緶 (<CJK Ideograph>)
      4822   => 16#7DF9#,  --  緹 (<CJK Ideograph>)
      4823   => 16#7DFA#,  --  緺 (<CJK Ideograph>)
      4824   => 16#7E08#,  --  縈 (<CJK Ideograph>)
      4825   => 16#7E10#,  --  縐 (<CJK Ideograph>)
      4826   => 16#7E11#,  --  縑 (<CJK Ideograph>)
      4827   => 16#7E15#,  --  縕 (<CJK Ideograph>)
      4828   => 16#7E17#,  --  縗 (<CJK Ideograph>)
      4829   => 16#7E1C#,  --  縜 (<CJK Ideograph>)
      4830   => 16#7E1D#,  --  縝 (<CJK Ideograph>)
      4831   => 16#7E20#,  --  縠 (<CJK Ideograph>)
      4832   => 16#7E27#,  --  縧 (<CJK Ideograph>)
      4833   => 16#7E28#,  --  縨 (<CJK Ideograph>)
      4834   => 16#7E2C#,  --  縬 (<CJK Ideograph>)
      4835   => 16#7E2D#,  --  縭 (<CJK Ideograph>)
      4836   => 16#7E2F#,  --  縯 (<CJK Ideograph>)
      4837   => 16#7E33#,  --  縳 (<CJK Ideograph>)
      4838   => 16#7E36#,  --  縶 (<CJK Ideograph>)
      4839   => 16#7E3F#,  --  縿 (<CJK Ideograph>)
      4840   => 16#7E44#,  --  繄 (<CJK Ideograph>)
      4841   => 16#7E45#,  --  繅 (<CJK Ideograph>)
      4842   => 16#7E47#,  --  繇 (<CJK Ideograph>)
      4843   => 16#7E4E#,  --  繎 (<CJK Ideograph>)
      4844   => 16#7E50#,  --  繐 (<CJK Ideograph>)
      4845   => 16#7E52#,  --  繒 (<CJK Ideograph>)
      4846   => 16#7E58#,  --  繘 (<CJK Ideograph>)
      4847   => 16#7E5F#,  --  繟 (<CJK Ideograph>)
      4848   => 16#7E61#,  --  繡 (<CJK Ideograph>)
      4849   => 16#7E62#,  --  繢 (<CJK Ideograph>)
      4850   => 16#7E65#,  --  繥 (<CJK Ideograph>)
      4851   => 16#7E6B#,  --  繫 (<CJK Ideograph>)
      4852   => 16#7E6E#,  --  繮 (<CJK Ideograph>)
      4853   => 16#7E6F#,  --  繯 (<CJK Ideograph>)
      4854   => 16#7E73#,  --  繳 (<CJK Ideograph>)
      4855   => 16#7E78#,  --  繸 (<CJK Ideograph>)
      4856   => 16#7E7E#,  --  繾 (<CJK Ideograph>)
      4857   => 16#7E81#,  --  纁 (<CJK Ideograph>)
      4858   => 16#7E86#,  --  纆 (<CJK Ideograph>)
      4859   => 16#7E87#,  --  纇 (<CJK Ideograph>)
      4860   => 16#7E8A#,  --  纊 (<CJK Ideograph>)
      4861   => 16#7E8D#,  --  纍 (<CJK Ideograph>)
      4862   => 16#7E91#,  --  纑 (<CJK Ideograph>)
      4863   => 16#7E95#,  --  纕 (<CJK Ideograph>)
      4864   => 16#7E98#,  --  纘 (<CJK Ideograph>)
      4865   => 16#7E9A#,  --  纚 (<CJK Ideograph>)
      4866   => 16#7E9D#,  --  纝 (<CJK Ideograph>)
      4867   => 16#7E9E#,  --  纞 (<CJK Ideograph>)
      4868   => 16#7F3C#,  --  缼 (<CJK Ideograph>)
      4869   => 16#7F3B#,  --  缻 (<CJK Ideograph>)
      4870   => 16#7F3D#,  --  缽 (<CJK Ideograph>)
      4871   => 16#7F3E#,  --  缾 (<CJK Ideograph>)
      4872   => 16#7F3F#,  --  缿 (<CJK Ideograph>)
      4873   => 16#7F43#,  --  罃 (<CJK Ideograph>)
      4874   => 16#7F44#,  --  罄 (<CJK Ideograph>)
      4875   => 16#7F47#,  --  罇 (<CJK Ideograph>)
      4876   => 16#7F4F#,  --  罏 (<CJK Ideograph>)
      4877   => 16#7F52#,  --  罒 (<CJK Ideograph>)
      4878   => 16#7F53#,  --  罓 (<CJK Ideograph>)
      4879   => 16#7F5B#,  --  罛 (<CJK Ideograph>)
      4880   => 16#7F5C#,  --  罜 (<CJK Ideograph>)
      4881   => 16#7F5D#,  --  罝 (<CJK Ideograph>)
      4882   => 16#7F61#,  --  罡 (<CJK Ideograph>)
      4883   => 16#7F63#,  --  罣 (<CJK Ideograph>)
      4884   => 16#7F64#,  --  罤 (<CJK Ideograph>)
      4885   => 16#7F65#,  --  罥 (<CJK Ideograph>)
      4886   => 16#7F66#,  --  罦 (<CJK Ideograph>)
      4887   => 16#7F6D#,  --  罭 (<CJK Ideograph>)
      4888   => 16#7F71#,  --  罱 (<CJK Ideograph>)
      4889   => 16#7F7D#,  --  罽 (<CJK Ideograph>)
      4890   => 16#7F7E#,  --  罾 (<CJK Ideograph>)
      4891   => 16#7F7F#,  --  罿 (<CJK Ideograph>)
      4892   => 16#7F80#,  --  羀 (<CJK Ideograph>)
      4893   => 16#7F8B#,  --  羋 (<CJK Ideograph>)
      4894   => 16#7F8D#,  --  羍 (<CJK Ideograph>)
      4895   => 16#7F8F#,  --  羏 (<CJK Ideograph>)
      4896   => 16#7F90#,  --  羐 (<CJK Ideograph>)
      4897   => 16#7F91#,  --  羑 (<CJK Ideograph>)
      4898   => 16#7F96#,  --  羖 (<CJK Ideograph>)
      4899   => 16#7F97#,  --  羗 (<CJK Ideograph>)
      4900   => 16#7F9C#,  --  羜 (<CJK Ideograph>)
      4901   => 16#7FA1#,  --  羡 (<CJK Ideograph>)
      4902   => 16#7FA2#,  --  羢 (<CJK Ideograph>)
      4903   => 16#7FA6#,  --  羦 (<CJK Ideograph>)
      4904   => 16#7FAA#,  --  羪 (<CJK Ideograph>)
      4905   => 16#7FAD#,  --  羭 (<CJK Ideograph>)
      4906   => 16#7FB4#,  --  羴 (<CJK Ideograph>)
      4907   => 16#7FBC#,  --  羼 (<CJK Ideograph>)
      4908   => 16#7FBF#,  --  羿 (<CJK Ideograph>)
      4909   => 16#7FC0#,  --  翀 (<CJK Ideograph>)
      4910   => 16#7FC3#,  --  翃 (<CJK Ideograph>)
      4911   => 16#7FC8#,  --  翈 (<CJK Ideograph>)
      4912   => 16#7FCE#,  --  翎 (<CJK Ideograph>)
      4913   => 16#7FCF#,  --  翏 (<CJK Ideograph>)
      4914   => 16#7FDB#,  --  翛 (<CJK Ideograph>)
      4915   => 16#7FDF#,  --  翟 (<CJK Ideograph>)
      4916   => 16#7FE3#,  --  翣 (<CJK Ideograph>)
      4917   => 16#7FE5#,  --  翥 (<CJK Ideograph>)
      4918   => 16#7FE8#,  --  翨 (<CJK Ideograph>)
      4919   => 16#7FEC#,  --  翬 (<CJK Ideograph>)
      4920   => 16#7FEE#,  --  翮 (<CJK Ideograph>)
      4921   => 16#7FEF#,  --  翯 (<CJK Ideograph>)
      4922   => 16#7FF2#,  --  翲 (<CJK Ideograph>)
      4923   => 16#7FFA#,  --  翺 (<CJK Ideograph>)
      4924   => 16#7FFD#,  --  翽 (<CJK Ideograph>)
      4925   => 16#7FFE#,  --  翾 (<CJK Ideograph>)
      4926   => 16#7FFF#,  --  翿 (<CJK Ideograph>)
      4927   => 16#8007#,  --  耇 (<CJK Ideograph>)
      4928   => 16#8008#,  --  耈 (<CJK Ideograph>)
      4929   => 16#800A#,  --  耊 (<CJK Ideograph>)
      4930   => 16#800D#,  --  耍 (<CJK Ideograph>)
      4931   => 16#800E#,  --  耎 (<CJK Ideograph>)
      4932   => 16#800F#,  --  耏 (<CJK Ideograph>)
      4933   => 16#8011#,  --  耑 (<CJK Ideograph>)
      4934   => 16#8013#,  --  耓 (<CJK Ideograph>)
      4935   => 16#8014#,  --  耔 (<CJK Ideograph>)
      4936   => 16#8016#,  --  耖 (<CJK Ideograph>)
      4937   => 16#801D#,  --  耝 (<CJK Ideograph>)
      4938   => 16#801E#,  --  耞 (<CJK Ideograph>)
      4939   => 16#801F#,  --  耟 (<CJK Ideograph>)
      4940   => 16#8020#,  --  耠 (<CJK Ideograph>)
      4941   => 16#8024#,  --  耤 (<CJK Ideograph>)
      4942   => 16#8026#,  --  耦 (<CJK Ideograph>)
      4943   => 16#802C#,  --  耬 (<CJK Ideograph>)
      4944   => 16#802E#,  --  耮 (<CJK Ideograph>)
      4945   => 16#8030#,  --  耰 (<CJK Ideograph>)
      4946   => 16#8034#,  --  耴 (<CJK Ideograph>)
      4947   => 16#8035#,  --  耵 (<CJK Ideograph>)
      4948   => 16#8037#,  --  耷 (<CJK Ideograph>)
      4949   => 16#8039#,  --  耹 (<CJK Ideograph>)
      4950   => 16#803A#,  --  耺 (<CJK Ideograph>)
      4951   => 16#803C#,  --  耼 (<CJK Ideograph>)
      4952   => 16#803E#,  --  耾 (<CJK Ideograph>)
      4953   => 16#8040#,  --  聀 (<CJK Ideograph>)
      4954   => 16#8044#,  --  聄 (<CJK Ideograph>)
      4955   => 16#8060#,  --  聠 (<CJK Ideograph>)
      4956   => 16#8064#,  --  聤 (<CJK Ideograph>)
      4957   => 16#8066#,  --  聦 (<CJK Ideograph>)
      4958   => 16#806D#,  --  聭 (<CJK Ideograph>)
      4959   => 16#8071#,  --  聱 (<CJK Ideograph>)
      4960   => 16#8075#,  --  聵 (<CJK Ideograph>)
      4961   => 16#8081#,  --  肁 (<CJK Ideograph>)
      4962   => 16#8088#,  --  肈 (<CJK Ideograph>)
      4963   => 16#808E#,  --  肎 (<CJK Ideograph>)
      4964   => 16#809C#,  --  肜 (<CJK Ideograph>)
      4965   => 16#809E#,  --  肞 (<CJK Ideograph>)
      4966   => 16#80A6#,  --  肦 (<CJK Ideograph>)
      4967   => 16#80A7#,  --  肧 (<CJK Ideograph>)
      4968   => 16#80AB#,  --  肫 (<CJK Ideograph>)
      4969   => 16#80B8#,  --  肸 (<CJK Ideograph>)
      4970   => 16#80B9#,  --  肹 (<CJK Ideograph>)
      4971   => 16#80C8#,  --  胈 (<CJK Ideograph>)
      4972   => 16#80CD#,  --  胍 (<CJK Ideograph>)
      4973   => 16#80CF#,  --  胏 (<CJK Ideograph>)
      4974   => 16#80D2#,  --  胒 (<CJK Ideograph>)
      4975   => 16#80D4#,  --  胔 (<CJK Ideograph>)
      4976   => 16#80D5#,  --  胕 (<CJK Ideograph>)
      4977   => 16#80D7#,  --  胗 (<CJK Ideograph>)
      4978   => 16#80D8#,  --  胘 (<CJK Ideograph>)
      4979   => 16#80E0#,  --  胠 (<CJK Ideograph>)
      4980   => 16#80ED#,  --  胭 (<CJK Ideograph>)
      4981   => 16#80EE#,  --  胮 (<CJK Ideograph>)
      4982   => 16#80F0#,  --  胰 (<CJK Ideograph>)
      4983   => 16#80F2#,  --  胲 (<CJK Ideograph>)
      4984   => 16#80F3#,  --  胳 (<CJK Ideograph>)
      4985   => 16#80F6#,  --  胶 (<CJK Ideograph>)
      4986   => 16#80F9#,  --  胹 (<CJK Ideograph>)
      4987   => 16#80FA#,  --  胺 (<CJK Ideograph>)
      4988   => 16#80FE#,  --  胾 (<CJK Ideograph>)
      4989   => 16#8103#,  --  脃 (<CJK Ideograph>)
      4990   => 16#810B#,  --  脋 (<CJK Ideograph>)
      4991   => 16#8116#,  --  脖 (<CJK Ideograph>)
      4992   => 16#8117#,  --  脗 (<CJK Ideograph>)
      4993   => 16#8118#,  --  脘 (<CJK Ideograph>)
      4994   => 16#811C#,  --  脜 (<CJK Ideograph>)
      4995   => 16#811E#,  --  脞 (<CJK Ideograph>)
      4996   => 16#8120#,  --  脠 (<CJK Ideograph>)
      4997   => 16#8124#,  --  脤 (<CJK Ideograph>)
      4998   => 16#8127#,  --  脧 (<CJK Ideograph>)
      4999   => 16#812C#,  --  脬 (<CJK Ideograph>)
      5000   => 16#8130#,  --  脰 (<CJK Ideograph>)
      5001   => 16#8135#,  --  脵 (<CJK Ideograph>)
      5002   => 16#813A#,  --  脺 (<CJK Ideograph>)
      5003   => 16#813C#,  --  脼 (<CJK Ideograph>)
      5004   => 16#8145#,  --  腅 (<CJK Ideograph>)
      5005   => 16#8147#,  --  腇 (<CJK Ideograph>)
      5006   => 16#814A#,  --  腊 (<CJK Ideograph>)
      5007   => 16#814C#,  --  腌 (<CJK Ideograph>)
      5008   => 16#8152#,  --  腒 (<CJK Ideograph>)
      5009   => 16#8157#,  --  腗 (<CJK Ideograph>)
      5010   => 16#8160#,  --  腠 (<CJK Ideograph>)
      5011   => 16#8161#,  --  腡 (<CJK Ideograph>)
      5012   => 16#8167#,  --  腧 (<CJK Ideograph>)
      5013   => 16#8168#,  --  腨 (<CJK Ideograph>)
      5014   => 16#8169#,  --  腩 (<CJK Ideograph>)
      5015   => 16#816D#,  --  腭 (<CJK Ideograph>)
      5016   => 16#816F#,  --  腯 (<CJK Ideograph>)
      5017   => 16#8177#,  --  腷 (<CJK Ideograph>)
      5018   => 16#8181#,  --  膁 (<CJK Ideograph>)
      5019   => 16#8190#,  --  膐 (<CJK Ideograph>)
      5020   => 16#8184#,  --  膄 (<CJK Ideograph>)
      5021   => 16#8185#,  --  膅 (<CJK Ideograph>)
      5022   => 16#8186#,  --  膆 (<CJK Ideograph>)
      5023   => 16#818B#,  --  膋 (<CJK Ideograph>)
      5024   => 16#818E#,  --  膎 (<CJK Ideograph>)
      5025   => 16#8196#,  --  膖 (<CJK Ideograph>)
      5026   => 16#8198#,  --  膘 (<CJK Ideograph>)
      5027   => 16#819B#,  --  膛 (<CJK Ideograph>)
      5028   => 16#819E#,  --  膞 (<CJK Ideograph>)
      5029   => 16#81A2#,  --  膢 (<CJK Ideograph>)
      5030   => 16#81AE#,  --  膮 (<CJK Ideograph>)
      5031   => 16#81B2#,  --  膲 (<CJK Ideograph>)
      5032   => 16#81B4#,  --  膴 (<CJK Ideograph>)
      5033   => 16#81BB#,  --  膻 (<CJK Ideograph>)
      5034   => 16#81CB#,  --  臋 (<CJK Ideograph>)
      5035   => 16#81C3#,  --  臃 (<CJK Ideograph>)
      5036   => 16#81C5#,  --  臅 (<CJK Ideograph>)
      5037   => 16#81CA#,  --  臊 (<CJK Ideograph>)
      5038   => 16#81CE#,  --  臎 (<CJK Ideograph>)
      5039   => 16#81CF#,  --  臏 (<CJK Ideograph>)
      5040   => 16#81D5#,  --  臕 (<CJK Ideograph>)
      5041   => 16#81D7#,  --  臗 (<CJK Ideograph>)
      5042   => 16#81DB#,  --  臛 (<CJK Ideograph>)
      5043   => 16#81DD#,  --  臝 (<CJK Ideograph>)
      5044   => 16#81DE#,  --  臞 (<CJK Ideograph>)
      5045   => 16#81E1#,  --  臡 (<CJK Ideograph>)
      5046   => 16#81E4#,  --  臤 (<CJK Ideograph>)
      5047   => 16#81EB#,  --  臫 (<CJK Ideograph>)
      5048   => 16#81EC#,  --  臬 (<CJK Ideograph>)
      5049   => 16#81F0#,  --  臰 (<CJK Ideograph>)
      5050   => 16#81F1#,  --  臱 (<CJK Ideograph>)
      5051   => 16#81F2#,  --  臲 (<CJK Ideograph>)
      5052   => 16#81F5#,  --  臵 (<CJK Ideograph>)
      5053   => 16#81F6#,  --  臶 (<CJK Ideograph>)
      5054   => 16#81F8#,  --  臸 (<CJK Ideograph>)
      5055   => 16#81F9#,  --  臹 (<CJK Ideograph>)
      5056   => 16#81FD#,  --  臽 (<CJK Ideograph>)
      5057   => 16#81FF#,  --  臿 (<CJK Ideograph>)
      5058   => 16#8200#,  --  舀 (<CJK Ideograph>)
      5059   => 16#8203#,  --  舃 (<CJK Ideograph>)
      5060   => 16#820F#,  --  舏 (<CJK Ideograph>)
      5061   => 16#8213#,  --  舓 (<CJK Ideograph>)
      5062   => 16#8214#,  --  舔 (<CJK Ideograph>)
      5063   => 16#8219#,  --  舙 (<CJK Ideograph>)
      5064   => 16#821A#,  --  舚 (<CJK Ideograph>)
      5065   => 16#821D#,  --  舝 (<CJK Ideograph>)
      5066   => 16#8221#,  --  舡 (<CJK Ideograph>)
      5067   => 16#8222#,  --  舢 (<CJK Ideograph>)
      5068   => 16#8228#,  --  舨 (<CJK Ideograph>)
      5069   => 16#8232#,  --  舲 (<CJK Ideograph>)
      5070   => 16#8234#,  --  舴 (<CJK Ideograph>)
      5071   => 16#823A#,  --  舺 (<CJK Ideograph>)
      5072   => 16#8243#,  --  艃 (<CJK Ideograph>)
      5073   => 16#8244#,  --  艄 (<CJK Ideograph>)
      5074   => 16#8245#,  --  艅 (<CJK Ideograph>)
      5075   => 16#8246#,  --  艆 (<CJK Ideograph>)
      5076   => 16#824B#,  --  艋 (<CJK Ideograph>)
      5077   => 16#824E#,  --  艎 (<CJK Ideograph>)
      5078   => 16#824F#,  --  艏 (<CJK Ideograph>)
      5079   => 16#8251#,  --  艑 (<CJK Ideograph>)
      5080   => 16#8256#,  --  艖 (<CJK Ideograph>)
      5081   => 16#825C#,  --  艜 (<CJK Ideograph>)
      5082   => 16#8260#,  --  艠 (<CJK Ideograph>)
      5083   => 16#8263#,  --  艣 (<CJK Ideograph>)
      5084   => 16#8267#,  --  艧 (<CJK Ideograph>)
      5085   => 16#826D#,  --  艭 (<CJK Ideograph>)
      5086   => 16#8274#,  --  艴 (<CJK Ideograph>)
      5087   => 16#827B#,  --  艻 (<CJK Ideograph>)
      5088   => 16#827D#,  --  艽 (<CJK Ideograph>)
      5089   => 16#827F#,  --  艿 (<CJK Ideograph>)
      5090   => 16#8280#,  --  芀 (<CJK Ideograph>)
      5091   => 16#8281#,  --  芁 (<CJK Ideograph>)
      5092   => 16#8283#,  --  芃 (<CJK Ideograph>)
      5093   => 16#8284#,  --  芄 (<CJK Ideograph>)
      5094   => 16#8287#,  --  芇 (<CJK Ideograph>)
      5095   => 16#8289#,  --  芉 (<CJK Ideograph>)
      5096   => 16#828A#,  --  芊 (<CJK Ideograph>)
      5097   => 16#828E#,  --  芎 (<CJK Ideograph>)
      5098   => 16#8291#,  --  芑 (<CJK Ideograph>)
      5099   => 16#8294#,  --  芔 (<CJK Ideograph>)
      5100   => 16#8296#,  --  芖 (<CJK Ideograph>)
      5101   => 16#8298#,  --  芘 (<CJK Ideograph>)
      5102   => 16#829A#,  --  芚 (<CJK Ideograph>)
      5103   => 16#829B#,  --  芛 (<CJK Ideograph>)
      5104   => 16#82A0#,  --  芠 (<CJK Ideograph>)
      5105   => 16#82A1#,  --  芡 (<CJK Ideograph>)
      5106   => 16#82A3#,  --  芣 (<CJK Ideograph>)
      5107   => 16#82A4#,  --  芤 (<CJK Ideograph>)
      5108   => 16#82A7#,  --  芧 (<CJK Ideograph>)
      5109   => 16#82A8#,  --  芨 (<CJK Ideograph>)
      5110   => 16#82A9#,  --  芩 (<CJK Ideograph>)
      5111   => 16#82AA#,  --  芪 (<CJK Ideograph>)
      5112   => 16#82AE#,  --  芮 (<CJK Ideograph>)
      5113   => 16#82B0#,  --  芰 (<CJK Ideograph>)
      5114   => 16#82B2#,  --  芲 (<CJK Ideograph>)
      5115   => 16#82B4#,  --  芴 (<CJK Ideograph>)
      5116   => 16#82B7#,  --  芷 (<CJK Ideograph>)
      5117   => 16#82BA#,  --  芺 (<CJK Ideograph>)
      5118   => 16#82BC#,  --  芼 (<CJK Ideograph>)
      5119   => 16#82BE#,  --  芾 (<CJK Ideograph>)
      5120   => 16#82BF#,  --  芿 (<CJK Ideograph>)
      5121   => 16#82C6#,  --  苆 (<CJK Ideograph>)
      5122   => 16#82D0#,  --  苐 (<CJK Ideograph>)
      5123   => 16#82D5#,  --  苕 (<CJK Ideograph>)
      5124   => 16#82DA#,  --  苚 (<CJK Ideograph>)
      5125   => 16#82E0#,  --  苠 (<CJK Ideograph>)
      5126   => 16#82E2#,  --  苢 (<CJK Ideograph>)
      5127   => 16#82E4#,  --  苤 (<CJK Ideograph>)
      5128   => 16#82E8#,  --  苨 (<CJK Ideograph>)
      5129   => 16#82EA#,  --  苪 (<CJK Ideograph>)
      5130   => 16#82ED#,  --  苭 (<CJK Ideograph>)
      5131   => 16#82EF#,  --  苯 (<CJK Ideograph>)
      5132   => 16#82F6#,  --  苶 (<CJK Ideograph>)
      5133   => 16#82F7#,  --  苷 (<CJK Ideograph>)
      5134   => 16#82FD#,  --  苽 (<CJK Ideograph>)
      5135   => 16#82FE#,  --  苾 (<CJK Ideograph>)
      5136   => 16#8300#,  --  茀 (<CJK Ideograph>)
      5137   => 16#8301#,  --  茁 (<CJK Ideograph>)
      5138   => 16#8307#,  --  茇 (<CJK Ideograph>)
      5139   => 16#8308#,  --  茈 (<CJK Ideograph>)
      5140   => 16#830A#,  --  茊 (<CJK Ideograph>)
      5141   => 16#830B#,  --  茋 (<CJK Ideograph>)
      5142   => 16#8354#,  --  荔 (<CJK Ideograph>)
      5143   => 16#831B#,  --  茛 (<CJK Ideograph>)
      5144   => 16#831D#,  --  茝 (<CJK Ideograph>)
      5145   => 16#831E#,  --  茞 (<CJK Ideograph>)
      5146   => 16#831F#,  --  茟 (<CJK Ideograph>)
      5147   => 16#8321#,  --  茡 (<CJK Ideograph>)
      5148   => 16#8322#,  --  茢 (<CJK Ideograph>)
      5149   => 16#832C#,  --  茬 (<CJK Ideograph>)
      5150   => 16#832D#,  --  茭 (<CJK Ideograph>)
      5151   => 16#832E#,  --  茮 (<CJK Ideograph>)
      5152   => 16#8330#,  --  茰 (<CJK Ideograph>)
      5153   => 16#8333#,  --  茳 (<CJK Ideograph>)
      5154   => 16#8337#,  --  茷 (<CJK Ideograph>)
      5155   => 16#833A#,  --  茺 (<CJK Ideograph>)
      5156   => 16#833C#,  --  茼 (<CJK Ideograph>)
      5157   => 16#833D#,  --  茽 (<CJK Ideograph>)
      5158   => 16#8342#,  --  荂 (<CJK Ideograph>)
      5159   => 16#8343#,  --  荃 (<CJK Ideograph>)
      5160   => 16#8344#,  --  荄 (<CJK Ideograph>)
      5161   => 16#8347#,  --  荇 (<CJK Ideograph>)
      5162   => 16#834D#,  --  荍 (<CJK Ideograph>)
      5163   => 16#834E#,  --  荎 (<CJK Ideograph>)
      5164   => 16#8351#,  --  荑 (<CJK Ideograph>)
      5165   => 16#8355#,  --  荕 (<CJK Ideograph>)
      5166   => 16#8356#,  --  荖 (<CJK Ideograph>)
      5167   => 16#8357#,  --  荗 (<CJK Ideograph>)
      5168   => 16#8370#,  --  荰 (<CJK Ideograph>)
      5169   => 16#8378#,  --  荸 (<CJK Ideograph>)
      5170   => 16#837D#,  --  荽 (<CJK Ideograph>)
      5171   => 16#837F#,  --  荿 (<CJK Ideograph>)
      5172   => 16#8380#,  --  莀 (<CJK Ideograph>)
      5173   => 16#8382#,  --  莂 (<CJK Ideograph>)
      5174   => 16#8384#,  --  莄 (<CJK Ideograph>)
      5175   => 16#8386#,  --  莆 (<CJK Ideograph>)
      5176   => 16#838D#,  --  莍 (<CJK Ideograph>)
      5177   => 16#8392#,  --  莒 (<CJK Ideograph>)
      5178   => 16#8394#,  --  莔 (<CJK Ideograph>)
      5179   => 16#8395#,  --  莕 (<CJK Ideograph>)
      5180   => 16#8398#,  --  莘 (<CJK Ideograph>)
      5181   => 16#8399#,  --  莙 (<CJK Ideograph>)
      5182   => 16#839B#,  --  莛 (<CJK Ideograph>)
      5183   => 16#839C#,  --  莜 (<CJK Ideograph>)
      5184   => 16#839D#,  --  莝 (<CJK Ideograph>)
      5185   => 16#83A6#,  --  莦 (<CJK Ideograph>)
      5186   => 16#83A7#,  --  莧 (<CJK Ideograph>)
      5187   => 16#83A9#,  --  莩 (<CJK Ideograph>)
      5188   => 16#83AC#,  --  莬 (<CJK Ideograph>)
      5189   => 16#83BE#,  --  莾 (<CJK Ideograph>)
      5190   => 16#83BF#,  --  莿 (<CJK Ideograph>)
      5191   => 16#83C0#,  --  菀 (<CJK Ideograph>)
      5192   => 16#83C7#,  --  菇 (<CJK Ideograph>)
      5193   => 16#83C9#,  --  菉 (<CJK Ideograph>)
      5194   => 16#83CF#,  --  菏 (<CJK Ideograph>)
      5195   => 16#83D0#,  --  菐 (<CJK Ideograph>)
      5196   => 16#83D1#,  --  菑 (<CJK Ideograph>)
      5197   => 16#83D4#,  --  菔 (<CJK Ideograph>)
      5198   => 16#83DD#,  --  菝 (<CJK Ideograph>)
      5199   => 16#8353#,  --  荓 (<CJK Ideograph>)
      5200   => 16#83E8#,  --  菨 (<CJK Ideograph>)
      5201   => 16#83EA#,  --  菪 (<CJK Ideograph>)
      5202   => 16#83F6#,  --  菶 (<CJK Ideograph>)
      5203   => 16#83F8#,  --  菸 (<CJK Ideograph>)
      5204   => 16#83F9#,  --  菹 (<CJK Ideograph>)
      5205   => 16#83FC#,  --  菼 (<CJK Ideograph>)
      5206   => 16#8401#,  --  萁 (<CJK Ideograph>)
      5207   => 16#8406#,  --  萆 (<CJK Ideograph>)
      5208   => 16#840A#,  --  萊 (<CJK Ideograph>)
      5209   => 16#840F#,  --  萏 (<CJK Ideograph>)
      5210   => 16#8411#,  --  萑 (<CJK Ideograph>)
      5211   => 16#8415#,  --  萕 (<CJK Ideograph>)
      5212   => 16#8419#,  --  萙 (<CJK Ideograph>)
      5213   => 16#83AD#,  --  莭 (<CJK Ideograph>)
      5214   => 16#842F#,  --  萯 (<CJK Ideograph>)
      5215   => 16#8439#,  --  萹 (<CJK Ideograph>)
      5216   => 16#8445#,  --  葅 (<CJK Ideograph>)
      5217   => 16#8447#,  --  葇 (<CJK Ideograph>)
      5218   => 16#8448#,  --  葈 (<CJK Ideograph>)
      5219   => 16#844A#,  --  葊 (<CJK Ideograph>)
      5220   => 16#844D#,  --  葍 (<CJK Ideograph>)
      5221   => 16#844F#,  --  葏 (<CJK Ideograph>)
      5222   => 16#8451#,  --  葑 (<CJK Ideograph>)
      5223   => 16#8452#,  --  葒 (<CJK Ideograph>)
      5224   => 16#8456#,  --  葖 (<CJK Ideograph>)
      5225   => 16#8458#,  --  葘 (<CJK Ideograph>)
      5226   => 16#8459#,  --  葙 (<CJK Ideograph>)
      5227   => 16#845A#,  --  葚 (<CJK Ideograph>)
      5228   => 16#845C#,  --  葜 (<CJK Ideograph>)
      5229   => 16#8460#,  --  葠 (<CJK Ideograph>)
      5230   => 16#8464#,  --  葤 (<CJK Ideograph>)
      5231   => 16#8465#,  --  葥 (<CJK Ideograph>)
      5232   => 16#8467#,  --  葧 (<CJK Ideograph>)
      5233   => 16#846A#,  --  葪 (<CJK Ideograph>)
      5234   => 16#8470#,  --  葰 (<CJK Ideograph>)
      5235   => 16#8473#,  --  葳 (<CJK Ideograph>)
      5236   => 16#8474#,  --  葴 (<CJK Ideograph>)
      5237   => 16#8476#,  --  葶 (<CJK Ideograph>)
      5238   => 16#8478#,  --  葸 (<CJK Ideograph>)
      5239   => 16#847C#,  --  葼 (<CJK Ideograph>)
      5240   => 16#847D#,  --  葽 (<CJK Ideograph>)
      5241   => 16#8481#,  --  蒁 (<CJK Ideograph>)
      5242   => 16#8485#,  --  蒅 (<CJK Ideograph>)
      5243   => 16#8492#,  --  蒒 (<CJK Ideograph>)
      5244   => 16#8493#,  --  蒓 (<CJK Ideograph>)
      5245   => 16#8495#,  --  蒕 (<CJK Ideograph>)
      5246   => 16#849E#,  --  蒞 (<CJK Ideograph>)
      5247   => 16#84A6#,  --  蒦 (<CJK Ideograph>)
      5248   => 16#84A8#,  --  蒨 (<CJK Ideograph>)
      5249   => 16#84A9#,  --  蒩 (<CJK Ideograph>)
      5250   => 16#84AA#,  --  蒪 (<CJK Ideograph>)
      5251   => 16#84AF#,  --  蒯 (<CJK Ideograph>)
      5252   => 16#84B1#,  --  蒱 (<CJK Ideograph>)
      5253   => 16#84B4#,  --  蒴 (<CJK Ideograph>)
      5254   => 16#84BA#,  --  蒺 (<CJK Ideograph>)
      5255   => 16#84BD#,  --  蒽 (<CJK Ideograph>)
      5256   => 16#84BE#,  --  蒾 (<CJK Ideograph>)
      5257   => 16#84C0#,  --  蓀 (<CJK Ideograph>)
      5258   => 16#84C2#,  --  蓂 (<CJK Ideograph>)
      5259   => 16#84C7#,  --  蓇 (<CJK Ideograph>)
      5260   => 16#84C8#,  --  蓈 (<CJK Ideograph>)
      5261   => 16#84CC#,  --  蓌 (<CJK Ideograph>)
      5262   => 16#84CF#,  --  蓏 (<CJK Ideograph>)
      5263   => 16#84D3#,  --  蓓 (<CJK Ideograph>)
      5264   => 16#84DC#,  --  蓜 (<CJK Ideograph>)
      5265   => 16#84E7#,  --  蓧 (<CJK Ideograph>)
      5266   => 16#84EA#,  --  蓪 (<CJK Ideograph>)
      5267   => 16#84EF#,  --  蓯 (<CJK Ideograph>)
      5268   => 16#84F0#,  --  蓰 (<CJK Ideograph>)
      5269   => 16#84F1#,  --  蓱 (<CJK Ideograph>)
      5270   => 16#84F2#,  --  蓲 (<CJK Ideograph>)
      5271   => 16#84F7#,  --  蓷 (<CJK Ideograph>)
      5272   => 16#8532#,  --  蔲 (<CJK Ideograph>)
      5273   => 16#84FA#,  --  蓺 (<CJK Ideograph>)
      5274   => 16#84FB#,  --  蓻 (<CJK Ideograph>)
      5275   => 16#84FD#,  --  蓽 (<CJK Ideograph>)
      5276   => 16#8502#,  --  蔂 (<CJK Ideograph>)
      5277   => 16#8503#,  --  蔃 (<CJK Ideograph>)
      5278   => 16#8507#,  --  蔇 (<CJK Ideograph>)
      5279   => 16#850C#,  --  蔌 (<CJK Ideograph>)
      5280   => 16#850E#,  --  蔎 (<CJK Ideograph>)
      5281   => 16#8510#,  --  蔐 (<CJK Ideograph>)
      5282   => 16#851C#,  --  蔜 (<CJK Ideograph>)
      5283   => 16#851E#,  --  蔞 (<CJK Ideograph>)
      5284   => 16#8522#,  --  蔢 (<CJK Ideograph>)
      5285   => 16#8523#,  --  蔣 (<CJK Ideograph>)
      5286   => 16#8524#,  --  蔤 (<CJK Ideograph>)
      5287   => 16#8525#,  --  蔥 (<CJK Ideograph>)
      5288   => 16#8527#,  --  蔧 (<CJK Ideograph>)
      5289   => 16#852A#,  --  蔪 (<CJK Ideograph>)
      5290   => 16#852B#,  --  蔫 (<CJK Ideograph>)
      5291   => 16#852F#,  --  蔯 (<CJK Ideograph>)
      5292   => 16#8533#,  --  蔳 (<CJK Ideograph>)
      5293   => 16#8534#,  --  蔴 (<CJK Ideograph>)
      5294   => 16#8536#,  --  蔶 (<CJK Ideograph>)
      5295   => 16#853F#,  --  蔿 (<CJK Ideograph>)
      5296   => 16#8546#,  --  蕆 (<CJK Ideograph>)
      5297   => 16#854F#,  --  蕏 (<CJK Ideograph>)
      5298   => 16#8550#,  --  蕐 (<CJK Ideograph>)
      5299   => 16#8551#,  --  蕑 (<CJK Ideograph>)
      5300   => 16#8552#,  --  蕒 (<CJK Ideograph>)
      5301   => 16#8553#,  --  蕓 (<CJK Ideograph>)
      5302   => 16#8556#,  --  蕖 (<CJK Ideograph>)
      5303   => 16#8559#,  --  蕙 (<CJK Ideograph>)
      5304   => 16#855C#,  --  蕜 (<CJK Ideograph>)
      5305   => 16#855D#,  --  蕝 (<CJK Ideograph>)
      5306   => 16#855E#,  --  蕞 (<CJK Ideograph>)
      5307   => 16#855F#,  --  蕟 (<CJK Ideograph>)
      5308   => 16#8560#,  --  蕠 (<CJK Ideograph>)
      5309   => 16#8561#,  --  蕡 (<CJK Ideograph>)
      5310   => 16#8562#,  --  蕢 (<CJK Ideograph>)
      5311   => 16#8564#,  --  蕤 (<CJK Ideograph>)
      5312   => 16#856B#,  --  蕫 (<CJK Ideograph>)
      5313   => 16#856F#,  --  蕯 (<CJK Ideograph>)
      5314   => 16#8579#,  --  蕹 (<CJK Ideograph>)
      5315   => 16#857A#,  --  蕺 (<CJK Ideograph>)
      5316   => 16#857B#,  --  蕻 (<CJK Ideograph>)
      5317   => 16#857D#,  --  蕽 (<CJK Ideograph>)
      5318   => 16#857F#,  --  蕿 (<CJK Ideograph>)
      5319   => 16#8581#,  --  薁 (<CJK Ideograph>)
      5320   => 16#8585#,  --  薅 (<CJK Ideograph>)
      5321   => 16#8586#,  --  薆 (<CJK Ideograph>)
      5322   => 16#8589#,  --  薉 (<CJK Ideograph>)
      5323   => 16#858B#,  --  薋 (<CJK Ideograph>)
      5324   => 16#858C#,  --  薌 (<CJK Ideograph>)
      5325   => 16#858F#,  --  薏 (<CJK Ideograph>)
      5326   => 16#8593#,  --  薓 (<CJK Ideograph>)
      5327   => 16#8598#,  --  薘 (<CJK Ideograph>)
      5328   => 16#859D#,  --  薝 (<CJK Ideograph>)
      5329   => 16#859F#,  --  薟 (<CJK Ideograph>)
      5330   => 16#85A0#,  --  薠 (<CJK Ideograph>)
      5331   => 16#85A2#,  --  薢 (<CJK Ideograph>)
      5332   => 16#85A5#,  --  薥 (<CJK Ideograph>)
      5333   => 16#85A7#,  --  薧 (<CJK Ideograph>)
      5334   => 16#85B4#,  --  薴 (<CJK Ideograph>)
      5335   => 16#85B6#,  --  薶 (<CJK Ideograph>)
      5336   => 16#85B7#,  --  薷 (<CJK Ideograph>)
      5337   => 16#85B8#,  --  薸 (<CJK Ideograph>)
      5338   => 16#85BC#,  --  薼 (<CJK Ideograph>)
      5339   => 16#85BD#,  --  薽 (<CJK Ideograph>)
      5340   => 16#85BE#,  --  薾 (<CJK Ideograph>)
      5341   => 16#85BF#,  --  薿 (<CJK Ideograph>)
      5342   => 16#85C2#,  --  藂 (<CJK Ideograph>)
      5343   => 16#85C7#,  --  藇 (<CJK Ideograph>)
      5344   => 16#85CA#,  --  藊 (<CJK Ideograph>)
      5345   => 16#85CB#,  --  藋 (<CJK Ideograph>)
      5346   => 16#85CE#,  --  藎 (<CJK Ideograph>)
      5347   => 16#85AD#,  --  薭 (<CJK Ideograph>)
      5348   => 16#85D8#,  --  藘 (<CJK Ideograph>)
      5349   => 16#85DA#,  --  藚 (<CJK Ideograph>)
      5350   => 16#85DF#,  --  藟 (<CJK Ideograph>)
      5351   => 16#85E0#,  --  藠 (<CJK Ideograph>)
      5352   => 16#85E6#,  --  藦 (<CJK Ideograph>)
      5353   => 16#85E8#,  --  藨 (<CJK Ideograph>)
      5354   => 16#85ED#,  --  藭 (<CJK Ideograph>)
      5355   => 16#85F3#,  --  藳 (<CJK Ideograph>)
      5356   => 16#85F6#,  --  藶 (<CJK Ideograph>)
      5357   => 16#85FC#,  --  藼 (<CJK Ideograph>)
      5358   => 16#85FF#,  --  藿 (<CJK Ideograph>)
      5359   => 16#8600#,  --  蘀 (<CJK Ideograph>)
      5360   => 16#8604#,  --  蘄 (<CJK Ideograph>)
      5361   => 16#8605#,  --  蘅 (<CJK Ideograph>)
      5362   => 16#860D#,  --  蘍 (<CJK Ideograph>)
      5363   => 16#860E#,  --  蘎 (<CJK Ideograph>)
      5364   => 16#8610#,  --  蘐 (<CJK Ideograph>)
      5365   => 16#8611#,  --  蘑 (<CJK Ideograph>)
      5366   => 16#8612#,  --  蘒 (<CJK Ideograph>)
      5367   => 16#8618#,  --  蘘 (<CJK Ideograph>)
      5368   => 16#8619#,  --  蘙 (<CJK Ideograph>)
      5369   => 16#861B#,  --  蘛 (<CJK Ideograph>)
      5370   => 16#861E#,  --  蘞 (<CJK Ideograph>)
      5371   => 16#8621#,  --  蘡 (<CJK Ideograph>)
      5372   => 16#8627#,  --  蘧 (<CJK Ideograph>)
      5373   => 16#8629#,  --  蘩 (<CJK Ideograph>)
      5374   => 16#8636#,  --  蘶 (<CJK Ideograph>)
      5375   => 16#8638#,  --  蘸 (<CJK Ideograph>)
      5376   => 16#863A#,  --  蘺 (<CJK Ideograph>)
      5377   => 16#863C#,  --  蘼 (<CJK Ideograph>)
      5378   => 16#863D#,  --  蘽 (<CJK Ideograph>)
      5379   => 16#8640#,  --  虀 (<CJK Ideograph>)
      5380   => 16#8642#,  --  虂 (<CJK Ideograph>)
      5381   => 16#8646#,  --  虆 (<CJK Ideograph>)
      5382   => 16#8652#,  --  虒 (<CJK Ideograph>)
      5383   => 16#8653#,  --  虓 (<CJK Ideograph>)
      5384   => 16#8656#,  --  虖 (<CJK Ideograph>)
      5385   => 16#8657#,  --  虗 (<CJK Ideograph>)
      5386   => 16#8658#,  --  虘 (<CJK Ideograph>)
      5387   => 16#8659#,  --  虙 (<CJK Ideograph>)
      5388   => 16#865D#,  --  虝 (<CJK Ideograph>)
      5389   => 16#8660#,  --  虠 (<CJK Ideograph>)
      5390   => 16#8661#,  --  虡 (<CJK Ideograph>)
      5391   => 16#8662#,  --  虢 (<CJK Ideograph>)
      5392   => 16#8663#,  --  虣 (<CJK Ideograph>)
      5393   => 16#8664#,  --  虤 (<CJK Ideograph>)
      5394   => 16#8669#,  --  虩 (<CJK Ideograph>)
      5395   => 16#866C#,  --  虬 (<CJK Ideograph>)
      5396   => 16#866F#,  --  虯 (<CJK Ideograph>)
      5397   => 16#8675#,  --  虵 (<CJK Ideograph>)
      5398   => 16#8676#,  --  虶 (<CJK Ideograph>)
      5399   => 16#8677#,  --  虷 (<CJK Ideograph>)
      5400   => 16#867A#,  --  虺 (<CJK Ideograph>)
      5401   => 16#868D#,  --  蚍 (<CJK Ideograph>)
      5402   => 16#8691#,  --  蚑 (<CJK Ideograph>)
      5403   => 16#8696#,  --  蚖 (<CJK Ideograph>)
      5404   => 16#8698#,  --  蚘 (<CJK Ideograph>)
      5405   => 16#869A#,  --  蚚 (<CJK Ideograph>)
      5406   => 16#869C#,  --  蚜 (<CJK Ideograph>)
      5407   => 16#86A1#,  --  蚡 (<CJK Ideograph>)
      5408   => 16#86A6#,  --  蚦 (<CJK Ideograph>)
      5409   => 16#86A7#,  --  蚧 (<CJK Ideograph>)
      5410   => 16#86A8#,  --  蚨 (<CJK Ideograph>)
      5411   => 16#86AD#,  --  蚭 (<CJK Ideograph>)
      5412   => 16#86B1#,  --  蚱 (<CJK Ideograph>)
      5413   => 16#86B3#,  --  蚳 (<CJK Ideograph>)
      5414   => 16#86B4#,  --  蚴 (<CJK Ideograph>)
      5415   => 16#86B5#,  --  蚵 (<CJK Ideograph>)
      5416   => 16#86B7#,  --  蚷 (<CJK Ideograph>)
      5417   => 16#86B8#,  --  蚸 (<CJK Ideograph>)
      5418   => 16#86B9#,  --  蚹 (<CJK Ideograph>)
      5419   => 16#86BF#,  --  蚿 (<CJK Ideograph>)
      5420   => 16#86C0#,  --  蛀 (<CJK Ideograph>)
      5421   => 16#86C1#,  --  蛁 (<CJK Ideograph>)
      5422   => 16#86C3#,  --  蛃 (<CJK Ideograph>)
      5423   => 16#86C5#,  --  蛅 (<CJK Ideograph>)
      5424   => 16#86D1#,  --  蛑 (<CJK Ideograph>)
      5425   => 16#86D2#,  --  蛒 (<CJK Ideograph>)
      5426   => 16#86D5#,  --  蛕 (<CJK Ideograph>)
      5427   => 16#86D7#,  --  蛗 (<CJK Ideograph>)
      5428   => 16#86DA#,  --  蛚 (<CJK Ideograph>)
      5429   => 16#86DC#,  --  蛜 (<CJK Ideograph>)
      5430   => 16#86E0#,  --  蛠 (<CJK Ideograph>)
      5431   => 16#86E3#,  --  蛣 (<CJK Ideograph>)
      5432   => 16#86E5#,  --  蛥 (<CJK Ideograph>)
      5433   => 16#86E7#,  --  蛧 (<CJK Ideograph>)
      5434   => 16#8688#,  --  蚈 (<CJK Ideograph>)
      5435   => 16#86FA#,  --  蛺 (<CJK Ideograph>)
      5436   => 16#86FC#,  --  蛼 (<CJK Ideograph>)
      5437   => 16#86FD#,  --  蛽 (<CJK Ideograph>)
      5438   => 16#8704#,  --  蜄 (<CJK Ideograph>)
      5439   => 16#8705#,  --  蜅 (<CJK Ideograph>)
      5440   => 16#8707#,  --  蜇 (<CJK Ideograph>)
      5441   => 16#870B#,  --  蜋 (<CJK Ideograph>)
      5442   => 16#870E#,  --  蜎 (<CJK Ideograph>)
      5443   => 16#870F#,  --  蜏 (<CJK Ideograph>)
      5444   => 16#8710#,  --  蜐 (<CJK Ideograph>)
      5445   => 16#8713#,  --  蜓 (<CJK Ideograph>)
      5446   => 16#8714#,  --  蜔 (<CJK Ideograph>)
      5447   => 16#8719#,  --  蜙 (<CJK Ideograph>)
      5448   => 16#871E#,  --  蜞 (<CJK Ideograph>)
      5449   => 16#871F#,  --  蜟 (<CJK Ideograph>)
      5450   => 16#8721#,  --  蜡 (<CJK Ideograph>)
      5451   => 16#8723#,  --  蜣 (<CJK Ideograph>)
      5452   => 16#8728#,  --  蜨 (<CJK Ideograph>)
      5453   => 16#872E#,  --  蜮 (<CJK Ideograph>)
      5454   => 16#872F#,  --  蜯 (<CJK Ideograph>)
      5455   => 16#8731#,  --  蜱 (<CJK Ideograph>)
      5456   => 16#8732#,  --  蜲 (<CJK Ideograph>)
      5457   => 16#8739#,  --  蜹 (<CJK Ideograph>)
      5458   => 16#873A#,  --  蜺 (<CJK Ideograph>)
      5459   => 16#873C#,  --  蜼 (<CJK Ideograph>)
      5460   => 16#873D#,  --  蜽 (<CJK Ideograph>)
      5461   => 16#873E#,  --  蜾 (<CJK Ideograph>)
      5462   => 16#8740#,  --  蝀 (<CJK Ideograph>)
      5463   => 16#8743#,  --  蝃 (<CJK Ideograph>)
      5464   => 16#8745#,  --  蝅 (<CJK Ideograph>)
      5465   => 16#874D#,  --  蝍 (<CJK Ideograph>)
      5466   => 16#8758#,  --  蝘 (<CJK Ideograph>)
      5467   => 16#875D#,  --  蝝 (<CJK Ideograph>)
      5468   => 16#8761#,  --  蝡 (<CJK Ideograph>)
      5469   => 16#8764#,  --  蝤 (<CJK Ideograph>)
      5470   => 16#8765#,  --  蝥 (<CJK Ideograph>)
      5471   => 16#876F#,  --  蝯 (<CJK Ideograph>)
      5472   => 16#8771#,  --  蝱 (<CJK Ideograph>)
      5473   => 16#8772#,  --  蝲 (<CJK Ideograph>)
      5474   => 16#877B#,  --  蝻 (<CJK Ideograph>)
      5475   => 16#8783#,  --  螃 (<CJK Ideograph>)
      5476   => 16#8784#,  --  螄 (<CJK Ideograph>)
      5477   => 16#8785#,  --  螅 (<CJK Ideograph>)
      5478   => 16#8786#,  --  螆 (<CJK Ideograph>)
      5479   => 16#8787#,  --  螇 (<CJK Ideograph>)
      5480   => 16#8788#,  --  螈 (<CJK Ideograph>)
      5481   => 16#8789#,  --  螉 (<CJK Ideograph>)
      5482   => 16#878B#,  --  螋 (<CJK Ideograph>)
      5483   => 16#878C#,  --  螌 (<CJK Ideograph>)
      5484   => 16#8790#,  --  螐 (<CJK Ideograph>)
      5485   => 16#8793#,  --  螓 (<CJK Ideograph>)
      5486   => 16#8795#,  --  螕 (<CJK Ideograph>)
      5487   => 16#8797#,  --  螗 (<CJK Ideograph>)
      5488   => 16#8798#,  --  螘 (<CJK Ideograph>)
      5489   => 16#8799#,  --  螙 (<CJK Ideograph>)
      5490   => 16#879E#,  --  螞 (<CJK Ideograph>)
      5491   => 16#87A0#,  --  螠 (<CJK Ideograph>)
      5492   => 16#87A3#,  --  螣 (<CJK Ideograph>)
      5493   => 16#87A7#,  --  螧 (<CJK Ideograph>)
      5494   => 16#87AC#,  --  螬 (<CJK Ideograph>)
      5495   => 16#87AD#,  --  螭 (<CJK Ideograph>)
      5496   => 16#87AE#,  --  螮 (<CJK Ideograph>)
      5497   => 16#87B1#,  --  螱 (<CJK Ideograph>)
      5498   => 16#87B5#,  --  螵 (<CJK Ideograph>)
      5499   => 16#87BE#,  --  螾 (<CJK Ideograph>)
      5500   => 16#87BF#,  --  螿 (<CJK Ideograph>)
      5501   => 16#87C1#,  --  蟁 (<CJK Ideograph>)
      5502   => 16#87C8#,  --  蟈 (<CJK Ideograph>)
      5503   => 16#87C9#,  --  蟉 (<CJK Ideograph>)
      5504   => 16#87CA#,  --  蟊 (<CJK Ideograph>)
      5505   => 16#87CE#,  --  蟎 (<CJK Ideograph>)
      5506   => 16#87D5#,  --  蟕 (<CJK Ideograph>)
      5507   => 16#87D6#,  --  蟖 (<CJK Ideograph>)
      5508   => 16#87D9#,  --  蟙 (<CJK Ideograph>)
      5509   => 16#87DA#,  --  蟚 (<CJK Ideograph>)
      5510   => 16#87DC#,  --  蟜 (<CJK Ideograph>)
      5511   => 16#87DF#,  --  蟟 (<CJK Ideograph>)
      5512   => 16#87E2#,  --  蟢 (<CJK Ideograph>)
      5513   => 16#87E3#,  --  蟣 (<CJK Ideograph>)
      5514   => 16#87E4#,  --  蟤 (<CJK Ideograph>)
      5515   => 16#87EA#,  --  蟪 (<CJK Ideograph>)
      5516   => 16#87EB#,  --  蟫 (<CJK Ideograph>)
      5517   => 16#87ED#,  --  蟭 (<CJK Ideograph>)
      5518   => 16#87F1#,  --  蟱 (<CJK Ideograph>)
      5519   => 16#87F3#,  --  蟳 (<CJK Ideograph>)
      5520   => 16#87F8#,  --  蟸 (<CJK Ideograph>)
      5521   => 16#87FA#,  --  蟺 (<CJK Ideograph>)
      5522   => 16#87FF#,  --  蟿 (<CJK Ideograph>)
      5523   => 16#8801#,  --  蠁 (<CJK Ideograph>)
      5524   => 16#8803#,  --  蠃 (<CJK Ideograph>)
      5525   => 16#8806#,  --  蠆 (<CJK Ideograph>)
      5526   => 16#8809#,  --  蠉 (<CJK Ideograph>)
      5527   => 16#880A#,  --  蠊 (<CJK Ideograph>)
      5528   => 16#880B#,  --  蠋 (<CJK Ideograph>)
      5529   => 16#8810#,  --  蠐 (<CJK Ideograph>)
      5530   => 16#8819#,  --  蠙 (<CJK Ideograph>)
      5531   => 16#8812#,  --  蠒 (<CJK Ideograph>)
      5532   => 16#8813#,  --  蠓 (<CJK Ideograph>)
      5533   => 16#8814#,  --  蠔 (<CJK Ideograph>)
      5534   => 16#8818#,  --  蠘 (<CJK Ideograph>)
      5535   => 16#881A#,  --  蠚 (<CJK Ideograph>)
      5536   => 16#881B#,  --  蠛 (<CJK Ideograph>)
      5537   => 16#881C#,  --  蠜 (<CJK Ideograph>)
      5538   => 16#881E#,  --  蠞 (<CJK Ideograph>)
      5539   => 16#881F#,  --  蠟 (<CJK Ideograph>)
      5540   => 16#8828#,  --  蠨 (<CJK Ideograph>)
      5541   => 16#882D#,  --  蠭 (<CJK Ideograph>)
      5542   => 16#882E#,  --  蠮 (<CJK Ideograph>)
      5543   => 16#8830#,  --  蠰 (<CJK Ideograph>)
      5544   => 16#8832#,  --  蠲 (<CJK Ideograph>)
      5545   => 16#8835#,  --  蠵 (<CJK Ideograph>)
      5546   => 16#883A#,  --  蠺 (<CJK Ideograph>)
      5547   => 16#883C#,  --  蠼 (<CJK Ideograph>)
      5548   => 16#8841#,  --  衁 (<CJK Ideograph>)
      5549   => 16#8843#,  --  衃 (<CJK Ideograph>)
      5550   => 16#8845#,  --  衅 (<CJK Ideograph>)
      5551   => 16#8848#,  --  衈 (<CJK Ideograph>)
      5552   => 16#8849#,  --  衉 (<CJK Ideograph>)
      5553   => 16#884A#,  --  衊 (<CJK Ideograph>)
      5554   => 16#884B#,  --  衋 (<CJK Ideograph>)
      5555   => 16#884E#,  --  衎 (<CJK Ideograph>)
      5556   => 16#8851#,  --  衑 (<CJK Ideograph>)
      5557   => 16#8855#,  --  衕 (<CJK Ideograph>)
      5558   => 16#8856#,  --  衖 (<CJK Ideograph>)
      5559   => 16#8858#,  --  衘 (<CJK Ideograph>)
      5560   => 16#885A#,  --  衚 (<CJK Ideograph>)
      5561   => 16#885C#,  --  衜 (<CJK Ideograph>)
      5562   => 16#885F#,  --  衟 (<CJK Ideograph>)
      5563   => 16#8860#,  --  衠 (<CJK Ideograph>)
      5564   => 16#8864#,  --  衤 (<CJK Ideograph>)
      5565   => 16#8869#,  --  衩 (<CJK Ideograph>)
      5566   => 16#8871#,  --  衱 (<CJK Ideograph>)
      5567   => 16#8879#,  --  衹 (<CJK Ideograph>)
      5568   => 16#887B#,  --  衻 (<CJK Ideograph>)
      5569   => 16#8880#,  --  袀 (<CJK Ideograph>)
      5570   => 16#8898#,  --  袘 (<CJK Ideograph>)
      5571   => 16#889A#,  --  袚 (<CJK Ideograph>)
      5572   => 16#889B#,  --  袛 (<CJK Ideograph>)
      5573   => 16#889C#,  --  袜 (<CJK Ideograph>)
      5574   => 16#889F#,  --  袟 (<CJK Ideograph>)
      5575   => 16#88A0#,  --  袠 (<CJK Ideograph>)
      5576   => 16#88A8#,  --  袨 (<CJK Ideograph>)
      5577   => 16#88AA#,  --  袪 (<CJK Ideograph>)
      5578   => 16#88BA#,  --  袺 (<CJK Ideograph>)
      5579   => 16#88BD#,  --  袽 (<CJK Ideograph>)
      5580   => 16#88BE#,  --  袾 (<CJK Ideograph>)
      5581   => 16#88C0#,  --  裀 (<CJK Ideograph>)
      5582   => 16#88CA#,  --  裊 (<CJK Ideograph>)
      5583   => 16#88CB#,  --  裋 (<CJK Ideograph>)
      5584   => 16#88CC#,  --  裌 (<CJK Ideograph>)
      5585   => 16#88CD#,  --  裍 (<CJK Ideograph>)
      5586   => 16#88CE#,  --  裎 (<CJK Ideograph>)
      5587   => 16#88D1#,  --  裑 (<CJK Ideograph>)
      5588   => 16#88D2#,  --  裒 (<CJK Ideograph>)
      5589   => 16#88D3#,  --  裓 (<CJK Ideograph>)
      5590   => 16#88DB#,  --  裛 (<CJK Ideograph>)
      5591   => 16#88DE#,  --  裞 (<CJK Ideograph>)
      5592   => 16#88E7#,  --  裧 (<CJK Ideograph>)
      5593   => 16#88EF#,  --  裯 (<CJK Ideograph>)
      5594   => 16#88F0#,  --  裰 (<CJK Ideograph>)
      5595   => 16#88F1#,  --  裱 (<CJK Ideograph>)
      5596   => 16#88F5#,  --  裵 (<CJK Ideograph>)
      5597   => 16#88F7#,  --  裷 (<CJK Ideograph>)
      5598   => 16#8901#,  --  褁 (<CJK Ideograph>)
      5599   => 16#8906#,  --  褆 (<CJK Ideograph>)
      5600   => 16#890D#,  --  褍 (<CJK Ideograph>)
      5601   => 16#890E#,  --  褎 (<CJK Ideograph>)
      5602   => 16#890F#,  --  褏 (<CJK Ideograph>)
      5603   => 16#8915#,  --  褕 (<CJK Ideograph>)
      5604   => 16#8916#,  --  褖 (<CJK Ideograph>)
      5605   => 16#8918#,  --  褘 (<CJK Ideograph>)
      5606   => 16#8919#,  --  褙 (<CJK Ideograph>)
      5607   => 16#891A#,  --  褚 (<CJK Ideograph>)
      5608   => 16#891C#,  --  褜 (<CJK Ideograph>)
      5609   => 16#8920#,  --  褠 (<CJK Ideograph>)
      5610   => 16#8926#,  --  褦 (<CJK Ideograph>)
      5611   => 16#8927#,  --  褧 (<CJK Ideograph>)
      5612   => 16#8928#,  --  褨 (<CJK Ideograph>)
      5613   => 16#8930#,  --  褰 (<CJK Ideograph>)
      5614   => 16#8931#,  --  褱 (<CJK Ideograph>)
      5615   => 16#8932#,  --  褲 (<CJK Ideograph>)
      5616   => 16#8935#,  --  褵 (<CJK Ideograph>)
      5617   => 16#8939#,  --  褹 (<CJK Ideograph>)
      5618   => 16#893A#,  --  褺 (<CJK Ideograph>)
      5619   => 16#893E#,  --  褾 (<CJK Ideograph>)
      5620   => 16#8940#,  --  襀 (<CJK Ideograph>)
      5621   => 16#8942#,  --  襂 (<CJK Ideograph>)
      5622   => 16#8945#,  --  襅 (<CJK Ideograph>)
      5623   => 16#8946#,  --  襆 (<CJK Ideograph>)
      5624   => 16#8949#,  --  襉 (<CJK Ideograph>)
      5625   => 16#894F#,  --  襏 (<CJK Ideograph>)
      5626   => 16#8952#,  --  襒 (<CJK Ideograph>)
      5627   => 16#8957#,  --  襗 (<CJK Ideograph>)
      5628   => 16#895A#,  --  襚 (<CJK Ideograph>)
      5629   => 16#895B#,  --  襛 (<CJK Ideograph>)
      5630   => 16#895C#,  --  襜 (<CJK Ideograph>)
      5631   => 16#8961#,  --  襡 (<CJK Ideograph>)
      5632   => 16#8962#,  --  襢 (<CJK Ideograph>)
      5633   => 16#8963#,  --  襣 (<CJK Ideograph>)
      5634   => 16#896B#,  --  襫 (<CJK Ideograph>)
      5635   => 16#896E#,  --  襮 (<CJK Ideograph>)
      5636   => 16#8970#,  --  襰 (<CJK Ideograph>)
      5637   => 16#8973#,  --  襳 (<CJK Ideograph>)
      5638   => 16#8975#,  --  襵 (<CJK Ideograph>)
      5639   => 16#897A#,  --  襺 (<CJK Ideograph>)
      5640   => 16#897B#,  --  襻 (<CJK Ideograph>)
      5641   => 16#897C#,  --  襼 (<CJK Ideograph>)
      5642   => 16#897D#,  --  襽 (<CJK Ideograph>)
      5643   => 16#8989#,  --  覉 (<CJK Ideograph>)
      5644   => 16#898D#,  --  覍 (<CJK Ideograph>)
      5645   => 16#8990#,  --  覐 (<CJK Ideograph>)
      5646   => 16#8994#,  --  覔 (<CJK Ideograph>)
      5647   => 16#8995#,  --  覕 (<CJK Ideograph>)
      5648   => 16#899B#,  --  覛 (<CJK Ideograph>)
      5649   => 16#899C#,  --  覜 (<CJK Ideograph>)
      5650   => 16#899F#,  --  覟 (<CJK Ideograph>)
      5651   => 16#89A0#,  --  覠 (<CJK Ideograph>)
      5652   => 16#89A5#,  --  覥 (<CJK Ideograph>)
      5653   => 16#89B0#,  --  覰 (<CJK Ideograph>)
      5654   => 16#89B4#,  --  覴 (<CJK Ideograph>)
      5655   => 16#89B5#,  --  覵 (<CJK Ideograph>)
      5656   => 16#89B6#,  --  覶 (<CJK Ideograph>)
      5657   => 16#89B7#,  --  覷 (<CJK Ideograph>)
      5658   => 16#89BC#,  --  覼 (<CJK Ideograph>)
      5659   => 16#89D4#,  --  觔 (<CJK Ideograph>)
      5660   => 16#89D5#,  --  觕 (<CJK Ideograph>)
      5661   => 16#89D6#,  --  觖 (<CJK Ideograph>)
      5662   => 16#89D7#,  --  觗 (<CJK Ideograph>)
      5663   => 16#89D8#,  --  觘 (<CJK Ideograph>)
      5664   => 16#89E5#,  --  觥 (<CJK Ideograph>)
      5665   => 16#89E9#,  --  觩 (<CJK Ideograph>)
      5666   => 16#89EB#,  --  觫 (<CJK Ideograph>)
      5667   => 16#89ED#,  --  觭 (<CJK Ideograph>)
      5668   => 16#89F1#,  --  觱 (<CJK Ideograph>)
      5669   => 16#89F3#,  --  觳 (<CJK Ideograph>)
      5670   => 16#89F6#,  --  觶 (<CJK Ideograph>)
      5671   => 16#89F9#,  --  觹 (<CJK Ideograph>)
      5672   => 16#89FD#,  --  觽 (<CJK Ideograph>)
      5673   => 16#89FF#,  --  觿 (<CJK Ideograph>)
      5674   => 16#8A04#,  --  訄 (<CJK Ideograph>)
      5675   => 16#8A05#,  --  訅 (<CJK Ideograph>)
      5676   => 16#8A07#,  --  訇 (<CJK Ideograph>)
      5677   => 16#8A0F#,  --  訏 (<CJK Ideograph>)
      5678   => 16#8A11#,  --  訑 (<CJK Ideograph>)
      5679   => 16#8A12#,  --  訒 (<CJK Ideograph>)
      5680   => 16#8A14#,  --  訔 (<CJK Ideograph>)
      5681   => 16#8A15#,  --  訕 (<CJK Ideograph>)
      5682   => 16#8A1E#,  --  訞 (<CJK Ideograph>)
      5683   => 16#8A20#,  --  訠 (<CJK Ideograph>)
      5684   => 16#8A22#,  --  訢 (<CJK Ideograph>)
      5685   => 16#8A24#,  --  訤 (<CJK Ideograph>)
      5686   => 16#8A26#,  --  訦 (<CJK Ideograph>)
      5687   => 16#8A2B#,  --  訫 (<CJK Ideograph>)
      5688   => 16#8A2C#,  --  訬 (<CJK Ideograph>)
      5689   => 16#8A2F#,  --  訯 (<CJK Ideograph>)
      5690   => 16#8A35#,  --  訵 (<CJK Ideograph>)
      5691   => 16#8A37#,  --  訷 (<CJK Ideograph>)
      5692   => 16#8A3D#,  --  訽 (<CJK Ideograph>)
      5693   => 16#8A3E#,  --  訾 (<CJK Ideograph>)
      5694   => 16#8A40#,  --  詀 (<CJK Ideograph>)
      5695   => 16#8A43#,  --  詃 (<CJK Ideograph>)
      5696   => 16#8A45#,  --  詅 (<CJK Ideograph>)
      5697   => 16#8A47#,  --  詇 (<CJK Ideograph>)
      5698   => 16#8A49#,  --  詉 (<CJK Ideograph>)
      5699   => 16#8A4D#,  --  詍 (<CJK Ideograph>)
      5700   => 16#8A4E#,  --  詎 (<CJK Ideograph>)
      5701   => 16#8A53#,  --  詓 (<CJK Ideograph>)
      5702   => 16#8A56#,  --  詖 (<CJK Ideograph>)
      5703   => 16#8A57#,  --  詗 (<CJK Ideograph>)
      5704   => 16#8A58#,  --  詘 (<CJK Ideograph>)
      5705   => 16#8A5C#,  --  詜 (<CJK Ideograph>)
      5706   => 16#8A5D#,  --  詝 (<CJK Ideograph>)
      5707   => 16#8A61#,  --  詡 (<CJK Ideograph>)
      5708   => 16#8A65#,  --  詥 (<CJK Ideograph>)
      5709   => 16#8A67#,  --  詧 (<CJK Ideograph>)
      5710   => 16#8A75#,  --  詵 (<CJK Ideograph>)
      5711   => 16#8A76#,  --  詶 (<CJK Ideograph>)
      5712   => 16#8A77#,  --  詷 (<CJK Ideograph>)
      5713   => 16#8A79#,  --  詹 (<CJK Ideograph>)
      5714   => 16#8A7A#,  --  詺 (<CJK Ideograph>)
      5715   => 16#8A7B#,  --  詻 (<CJK Ideograph>)
      5716   => 16#8A7E#,  --  詾 (<CJK Ideograph>)
      5717   => 16#8A7F#,  --  詿 (<CJK Ideograph>)
      5718   => 16#8A80#,  --  誀 (<CJK Ideograph>)
      5719   => 16#8A83#,  --  誃 (<CJK Ideograph>)
      5720   => 16#8A86#,  --  誆 (<CJK Ideograph>)
      5721   => 16#8A8B#,  --  誋 (<CJK Ideograph>)
      5722   => 16#8A8F#,  --  誏 (<CJK Ideograph>)
      5723   => 16#8A90#,  --  誐 (<CJK Ideograph>)
      5724   => 16#8A92#,  --  誒 (<CJK Ideograph>)
      5725   => 16#8A96#,  --  誖 (<CJK Ideograph>)
      5726   => 16#8A97#,  --  誗 (<CJK Ideograph>)
      5727   => 16#8A99#,  --  誙 (<CJK Ideograph>)
      5728   => 16#8A9F#,  --  誟 (<CJK Ideograph>)
      5729   => 16#8AA7#,  --  誧 (<CJK Ideograph>)
      5730   => 16#8AA9#,  --  誩 (<CJK Ideograph>)
      5731   => 16#8AAE#,  --  誮 (<CJK Ideograph>)
      5732   => 16#8AAF#,  --  誯 (<CJK Ideograph>)
      5733   => 16#8AB3#,  --  誳 (<CJK Ideograph>)
      5734   => 16#8AB6#,  --  誶 (<CJK Ideograph>)
      5735   => 16#8AB7#,  --  誷 (<CJK Ideograph>)
      5736   => 16#8ABB#,  --  誻 (<CJK Ideograph>)
      5737   => 16#8ABE#,  --  誾 (<CJK Ideograph>)
      5738   => 16#8AC3#,  --  諃 (<CJK Ideograph>)
      5739   => 16#8AC6#,  --  諆 (<CJK Ideograph>)
      5740   => 16#8AC8#,  --  諈 (<CJK Ideograph>)
      5741   => 16#8AC9#,  --  諉 (<CJK Ideograph>)
      5742   => 16#8ACA#,  --  諊 (<CJK Ideograph>)
      5743   => 16#8AD1#,  --  諑 (<CJK Ideograph>)
      5744   => 16#8AD3#,  --  諓 (<CJK Ideograph>)
      5745   => 16#8AD4#,  --  諔 (<CJK Ideograph>)
      5746   => 16#8AD5#,  --  諕 (<CJK Ideograph>)
      5747   => 16#8AD7#,  --  諗 (<CJK Ideograph>)
      5748   => 16#8ADD#,  --  諝 (<CJK Ideograph>)
      5749   => 16#8ADF#,  --  諟 (<CJK Ideograph>)
      5750   => 16#8AEC#,  --  諬 (<CJK Ideograph>)
      5751   => 16#8AF0#,  --  諰 (<CJK Ideograph>)
      5752   => 16#8AF4#,  --  諴 (<CJK Ideograph>)
      5753   => 16#8AF5#,  --  諵 (<CJK Ideograph>)
      5754   => 16#8AF6#,  --  諶 (<CJK Ideograph>)
      5755   => 16#8AFC#,  --  諼 (<CJK Ideograph>)
      5756   => 16#8AFF#,  --  諿 (<CJK Ideograph>)
      5757   => 16#8B05#,  --  謅 (<CJK Ideograph>)
      5758   => 16#8B06#,  --  謆 (<CJK Ideograph>)
      5759   => 16#8B0B#,  --  謋 (<CJK Ideograph>)
      5760   => 16#8B11#,  --  謑 (<CJK Ideograph>)
      5761   => 16#8B1C#,  --  謜 (<CJK Ideograph>)
      5762   => 16#8B1E#,  --  謞 (<CJK Ideograph>)
      5763   => 16#8B1F#,  --  謟 (<CJK Ideograph>)
      5764   => 16#8B0A#,  --  謊 (<CJK Ideograph>)
      5765   => 16#8B2D#,  --  謭 (<CJK Ideograph>)
      5766   => 16#8B30#,  --  謰 (<CJK Ideograph>)
      5767   => 16#8B37#,  --  謷 (<CJK Ideograph>)
      5768   => 16#8B3C#,  --  謼 (<CJK Ideograph>)
      5769   => 16#8B42#,  --  譂 (<CJK Ideograph>)
      5770   => 16#8B43#,  --  譃 (<CJK Ideograph>)
      5771   => 16#8B44#,  --  譄 (<CJK Ideograph>)
      5772   => 16#8B45#,  --  譅 (<CJK Ideograph>)
      5773   => 16#8B46#,  --  譆 (<CJK Ideograph>)
      5774   => 16#8B48#,  --  譈 (<CJK Ideograph>)
      5775   => 16#8B52#,  --  譒 (<CJK Ideograph>)
      5776   => 16#8B53#,  --  譓 (<CJK Ideograph>)
      5777   => 16#8B54#,  --  譔 (<CJK Ideograph>)
      5778   => 16#8B59#,  --  譙 (<CJK Ideograph>)
      5779   => 16#8B4D#,  --  譍 (<CJK Ideograph>)
      5780   => 16#8B5E#,  --  譞 (<CJK Ideograph>)
      5781   => 16#8B63#,  --  譣 (<CJK Ideograph>)
      5782   => 16#8B6D#,  --  譭 (<CJK Ideograph>)
      5783   => 16#8B76#,  --  譶 (<CJK Ideograph>)
      5784   => 16#8B78#,  --  譸 (<CJK Ideograph>)
      5785   => 16#8B79#,  --  譹 (<CJK Ideograph>)
      5786   => 16#8B7C#,  --  譼 (<CJK Ideograph>)
      5787   => 16#8B7E#,  --  譾 (<CJK Ideograph>)
      5788   => 16#8B81#,  --  讁 (<CJK Ideograph>)
      5789   => 16#8B84#,  --  讄 (<CJK Ideograph>)
      5790   => 16#8B85#,  --  讅 (<CJK Ideograph>)
      5791   => 16#8B8B#,  --  讋 (<CJK Ideograph>)
      5792   => 16#8B8D#,  --  讍 (<CJK Ideograph>)
      5793   => 16#8B8F#,  --  讏 (<CJK Ideograph>)
      5794   => 16#8B94#,  --  讔 (<CJK Ideograph>)
      5795   => 16#8B95#,  --  讕 (<CJK Ideograph>)
      5796   => 16#8B9C#,  --  讜 (<CJK Ideograph>)
      5797   => 16#8B9E#,  --  讞 (<CJK Ideograph>)
      5798   => 16#8B9F#,  --  讟 (<CJK Ideograph>)
      5799   => 16#8C38#,  --  谸 (<CJK Ideograph>)
      5800   => 16#8C39#,  --  谹 (<CJK Ideograph>)
      5801   => 16#8C3D#,  --  谽 (<CJK Ideograph>)
      5802   => 16#8C3E#,  --  谾 (<CJK Ideograph>)
      5803   => 16#8C45#,  --  豅 (<CJK Ideograph>)
      5804   => 16#8C47#,  --  豇 (<CJK Ideograph>)
      5805   => 16#8C49#,  --  豉 (<CJK Ideograph>)
      5806   => 16#8C4B#,  --  豋 (<CJK Ideograph>)
      5807   => 16#8C4F#,  --  豏 (<CJK Ideograph>)
      5808   => 16#8C51#,  --  豑 (<CJK Ideograph>)
      5809   => 16#8C53#,  --  豓 (<CJK Ideograph>)
      5810   => 16#8C54#,  --  豔 (<CJK Ideograph>)
      5811   => 16#8C57#,  --  豗 (<CJK Ideograph>)
      5812   => 16#8C58#,  --  豘 (<CJK Ideograph>)
      5813   => 16#8C5B#,  --  豛 (<CJK Ideograph>)
      5814   => 16#8C5D#,  --  豝 (<CJK Ideograph>)
      5815   => 16#8C59#,  --  豙 (<CJK Ideograph>)
      5816   => 16#8C63#,  --  豣 (<CJK Ideograph>)
      5817   => 16#8C64#,  --  豤 (<CJK Ideograph>)
      5818   => 16#8C66#,  --  豦 (<CJK Ideograph>)
      5819   => 16#8C68#,  --  豨 (<CJK Ideograph>)
      5820   => 16#8C69#,  --  豩 (<CJK Ideograph>)
      5821   => 16#8C6D#,  --  豭 (<CJK Ideograph>)
      5822   => 16#8C73#,  --  豳 (<CJK Ideograph>)
      5823   => 16#8C75#,  --  豵 (<CJK Ideograph>)
      5824   => 16#8C76#,  --  豶 (<CJK Ideograph>)
      5825   => 16#8C7B#,  --  豻 (<CJK Ideograph>)
      5826   => 16#8C7E#,  --  豾 (<CJK Ideograph>)
      5827   => 16#8C86#,  --  貆 (<CJK Ideograph>)
      5828   => 16#8C87#,  --  貇 (<CJK Ideograph>)
      5829   => 16#8C8B#,  --  貋 (<CJK Ideograph>)
      5830   => 16#8C90#,  --  貐 (<CJK Ideograph>)
      5831   => 16#8C92#,  --  貒 (<CJK Ideograph>)
      5832   => 16#8C93#,  --  貓 (<CJK Ideograph>)
      5833   => 16#8C99#,  --  貙 (<CJK Ideograph>)
      5834   => 16#8C9B#,  --  貛 (<CJK Ideograph>)
      5835   => 16#8C9C#,  --  貜 (<CJK Ideograph>)
      5836   => 16#8CA4#,  --  貤 (<CJK Ideograph>)
      5837   => 16#8CB9#,  --  貹 (<CJK Ideograph>)
      5838   => 16#8CBA#,  --  貺 (<CJK Ideograph>)
      5839   => 16#8CC5#,  --  賅 (<CJK Ideograph>)
      5840   => 16#8CC6#,  --  賆 (<CJK Ideograph>)
      5841   => 16#8CC9#,  --  賉 (<CJK Ideograph>)
      5842   => 16#8CCB#,  --  賋 (<CJK Ideograph>)
      5843   => 16#8CCF#,  --  賏 (<CJK Ideograph>)
      5844   => 16#8CD6#,  --  賖 (<CJK Ideograph>)
      5845   => 16#8CD5#,  --  賕 (<CJK Ideograph>)
      5846   => 16#8CD9#,  --  賙 (<CJK Ideograph>)
      5847   => 16#8CDD#,  --  賝 (<CJK Ideograph>)
      5848   => 16#8CE1#,  --  賡 (<CJK Ideograph>)
      5849   => 16#8CE8#,  --  賨 (<CJK Ideograph>)
      5850   => 16#8CEC#,  --  賬 (<CJK Ideograph>)
      5851   => 16#8CEF#,  --  賯 (<CJK Ideograph>)
      5852   => 16#8CF0#,  --  賰 (<CJK Ideograph>)
      5853   => 16#8CF2#,  --  賲 (<CJK Ideograph>)
      5854   => 16#8CF5#,  --  賵 (<CJK Ideograph>)
      5855   => 16#8CF7#,  --  賷 (<CJK Ideograph>)
      5856   => 16#8CF8#,  --  賸 (<CJK Ideograph>)
      5857   => 16#8CFE#,  --  賾 (<CJK Ideograph>)
      5858   => 16#8CFF#,  --  賿 (<CJK Ideograph>)
      5859   => 16#8D01#,  --  贁 (<CJK Ideograph>)
      5860   => 16#8D03#,  --  贃 (<CJK Ideograph>)
      5861   => 16#8D09#,  --  贉 (<CJK Ideograph>)
      5862   => 16#8D12#,  --  贒 (<CJK Ideograph>)
      5863   => 16#8D17#,  --  贗 (<CJK Ideograph>)
      5864   => 16#8D1B#,  --  贛 (<CJK Ideograph>)
      5865   => 16#8D65#,  --  赥 (<CJK Ideograph>)
      5866   => 16#8D69#,  --  赩 (<CJK Ideograph>)
      5867   => 16#8D6C#,  --  赬 (<CJK Ideograph>)
      5868   => 16#8D6E#,  --  赮 (<CJK Ideograph>)
      5869   => 16#8D7F#,  --  赿 (<CJK Ideograph>)
      5870   => 16#8D82#,  --  趂 (<CJK Ideograph>)
      5871   => 16#8D84#,  --  趄 (<CJK Ideograph>)
      5872   => 16#8D88#,  --  趈 (<CJK Ideograph>)
      5873   => 16#8D8D#,  --  趍 (<CJK Ideograph>)
      5874   => 16#8D90#,  --  趐 (<CJK Ideograph>)
      5875   => 16#8D91#,  --  趑 (<CJK Ideograph>)
      5876   => 16#8D95#,  --  趕 (<CJK Ideograph>)
      5877   => 16#8D9E#,  --  趞 (<CJK Ideograph>)
      5878   => 16#8D9F#,  --  趟 (<CJK Ideograph>)
      5879   => 16#8DA0#,  --  趠 (<CJK Ideograph>)
      5880   => 16#8DA6#,  --  趦 (<CJK Ideograph>)
      5881   => 16#8DAB#,  --  趫 (<CJK Ideograph>)
      5882   => 16#8DAC#,  --  趬 (<CJK Ideograph>)
      5883   => 16#8DAF#,  --  趯 (<CJK Ideograph>)
      5884   => 16#8DB2#,  --  趲 (<CJK Ideograph>)
      5885   => 16#8DB5#,  --  趵 (<CJK Ideograph>)
      5886   => 16#8DB7#,  --  趷 (<CJK Ideograph>)
      5887   => 16#8DB9#,  --  趹 (<CJK Ideograph>)
      5888   => 16#8DBB#,  --  趻 (<CJK Ideograph>)
      5889   => 16#8DC0#,  --  跀 (<CJK Ideograph>)
      5890   => 16#8DC5#,  --  跅 (<CJK Ideograph>)
      5891   => 16#8DC6#,  --  跆 (<CJK Ideograph>)
      5892   => 16#8DC7#,  --  跇 (<CJK Ideograph>)
      5893   => 16#8DC8#,  --  跈 (<CJK Ideograph>)
      5894   => 16#8DCA#,  --  跊 (<CJK Ideograph>)
      5895   => 16#8DCE#,  --  跎 (<CJK Ideograph>)
      5896   => 16#8DD1#,  --  跑 (<CJK Ideograph>)
      5897   => 16#8DD4#,  --  跔 (<CJK Ideograph>)
      5898   => 16#8DD5#,  --  跕 (<CJK Ideograph>)
      5899   => 16#8DD7#,  --  跗 (<CJK Ideograph>)
      5900   => 16#8DD9#,  --  跙 (<CJK Ideograph>)
      5901   => 16#8DE4#,  --  跤 (<CJK Ideograph>)
      5902   => 16#8DE5#,  --  跥 (<CJK Ideograph>)
      5903   => 16#8DE7#,  --  跧 (<CJK Ideograph>)
      5904   => 16#8DEC#,  --  跬 (<CJK Ideograph>)
      5905   => 16#8DF0#,  --  跰 (<CJK Ideograph>)
      5906   => 16#8DBC#,  --  趼 (<CJK Ideograph>)
      5907   => 16#8DF1#,  --  跱 (<CJK Ideograph>)
      5908   => 16#8DF2#,  --  跲 (<CJK Ideograph>)
      5909   => 16#8DF4#,  --  跴 (<CJK Ideograph>)
      5910   => 16#8DFD#,  --  跽 (<CJK Ideograph>)
      5911   => 16#8E01#,  --  踁 (<CJK Ideograph>)
      5912   => 16#8E04#,  --  踄 (<CJK Ideograph>)
      5913   => 16#8E05#,  --  踅 (<CJK Ideograph>)
      5914   => 16#8E06#,  --  踆 (<CJK Ideograph>)
      5915   => 16#8E0B#,  --  踋 (<CJK Ideograph>)
      5916   => 16#8E11#,  --  踑 (<CJK Ideograph>)
      5917   => 16#8E14#,  --  踔 (<CJK Ideograph>)
      5918   => 16#8E16#,  --  踖 (<CJK Ideograph>)
      5919   => 16#8E20#,  --  踠 (<CJK Ideograph>)
      5920   => 16#8E21#,  --  踡 (<CJK Ideograph>)
      5921   => 16#8E22#,  --  踢 (<CJK Ideograph>)
      5922   => 16#8E23#,  --  踣 (<CJK Ideograph>)
      5923   => 16#8E26#,  --  踦 (<CJK Ideograph>)
      5924   => 16#8E27#,  --  踧 (<CJK Ideograph>)
      5925   => 16#8E31#,  --  踱 (<CJK Ideograph>)
      5926   => 16#8E33#,  --  踳 (<CJK Ideograph>)
      5927   => 16#8E36#,  --  踶 (<CJK Ideograph>)
      5928   => 16#8E37#,  --  踷 (<CJK Ideograph>)
      5929   => 16#8E38#,  --  踸 (<CJK Ideograph>)
      5930   => 16#8E39#,  --  踹 (<CJK Ideograph>)
      5931   => 16#8E3D#,  --  踽 (<CJK Ideograph>)
      5932   => 16#8E40#,  --  蹀 (<CJK Ideograph>)
      5933   => 16#8E41#,  --  蹁 (<CJK Ideograph>)
      5934   => 16#8E4B#,  --  蹋 (<CJK Ideograph>)
      5935   => 16#8E4D#,  --  蹍 (<CJK Ideograph>)
      5936   => 16#8E4E#,  --  蹎 (<CJK Ideograph>)
      5937   => 16#8E4F#,  --  蹏 (<CJK Ideograph>)
      5938   => 16#8E54#,  --  蹔 (<CJK Ideograph>)
      5939   => 16#8E5B#,  --  蹛 (<CJK Ideograph>)
      5940   => 16#8E5C#,  --  蹜 (<CJK Ideograph>)
      5941   => 16#8E5D#,  --  蹝 (<CJK Ideograph>)
      5942   => 16#8E5E#,  --  蹞 (<CJK Ideograph>)
      5943   => 16#8E61#,  --  蹡 (<CJK Ideograph>)
      5944   => 16#8E62#,  --  蹢 (<CJK Ideograph>)
      5945   => 16#8E69#,  --  蹩 (<CJK Ideograph>)
      5946   => 16#8E6C#,  --  蹬 (<CJK Ideograph>)
      5947   => 16#8E6D#,  --  蹭 (<CJK Ideograph>)
      5948   => 16#8E6F#,  --  蹯 (<CJK Ideograph>)
      5949   => 16#8E70#,  --  蹰 (<CJK Ideograph>)
      5950   => 16#8E71#,  --  蹱 (<CJK Ideograph>)
      5951   => 16#8E79#,  --  蹹 (<CJK Ideograph>)
      5952   => 16#8E7A#,  --  蹺 (<CJK Ideograph>)
      5953   => 16#8E7B#,  --  蹻 (<CJK Ideograph>)
      5954   => 16#8E82#,  --  躂 (<CJK Ideograph>)
      5955   => 16#8E83#,  --  躃 (<CJK Ideograph>)
      5956   => 16#8E89#,  --  躉 (<CJK Ideograph>)
      5957   => 16#8E90#,  --  躐 (<CJK Ideograph>)
      5958   => 16#8E92#,  --  躒 (<CJK Ideograph>)
      5959   => 16#8E95#,  --  躕 (<CJK Ideograph>)
      5960   => 16#8E9A#,  --  躚 (<CJK Ideograph>)
      5961   => 16#8E9B#,  --  躛 (<CJK Ideograph>)
      5962   => 16#8E9D#,  --  躝 (<CJK Ideograph>)
      5963   => 16#8E9E#,  --  躞 (<CJK Ideograph>)
      5964   => 16#8EA2#,  --  躢 (<CJK Ideograph>)
      5965   => 16#8EA7#,  --  躧 (<CJK Ideograph>)
      5966   => 16#8EA9#,  --  躩 (<CJK Ideograph>)
      5967   => 16#8EAD#,  --  躭 (<CJK Ideograph>)
      5968   => 16#8EAE#,  --  躮 (<CJK Ideograph>)
      5969   => 16#8EB3#,  --  躳 (<CJK Ideograph>)
      5970   => 16#8EB5#,  --  躵 (<CJK Ideograph>)
      5971   => 16#8EBA#,  --  躺 (<CJK Ideograph>)
      5972   => 16#8EBB#,  --  躻 (<CJK Ideograph>)
      5973   => 16#8EC0#,  --  軀 (<CJK Ideograph>)
      5974   => 16#8EC1#,  --  軁 (<CJK Ideograph>)
      5975   => 16#8EC3#,  --  軃 (<CJK Ideograph>)
      5976   => 16#8EC4#,  --  軄 (<CJK Ideograph>)
      5977   => 16#8EC7#,  --  軇 (<CJK Ideograph>)
      5978   => 16#8ECF#,  --  軏 (<CJK Ideograph>)
      5979   => 16#8ED1#,  --  軑 (<CJK Ideograph>)
      5980   => 16#8ED4#,  --  軔 (<CJK Ideograph>)
      5981   => 16#8EDC#,  --  軜 (<CJK Ideograph>)
      5982   => 16#8EE8#,  --  軨 (<CJK Ideograph>)
      5983   => 16#8EEE#,  --  軮 (<CJK Ideograph>)
      5984   => 16#8EF0#,  --  軰 (<CJK Ideograph>)
      5985   => 16#8EF1#,  --  軱 (<CJK Ideograph>)
      5986   => 16#8EF7#,  --  軷 (<CJK Ideograph>)
      5987   => 16#8EF9#,  --  軹 (<CJK Ideograph>)
      5988   => 16#8EFA#,  --  軺 (<CJK Ideograph>)
      5989   => 16#8EED#,  --  軭 (<CJK Ideograph>)
      5990   => 16#8F00#,  --  輀 (<CJK Ideograph>)
      5991   => 16#8F02#,  --  輂 (<CJK Ideograph>)
      5992   => 16#8F07#,  --  輇 (<CJK Ideograph>)
      5993   => 16#8F08#,  --  輈 (<CJK Ideograph>)
      5994   => 16#8F0F#,  --  輏 (<CJK Ideograph>)
      5995   => 16#8F10#,  --  輐 (<CJK Ideograph>)
      5996   => 16#8F16#,  --  輖 (<CJK Ideograph>)
      5997   => 16#8F17#,  --  輗 (<CJK Ideograph>)
      5998   => 16#8F18#,  --  輘 (<CJK Ideograph>)
      5999   => 16#8F1E#,  --  輞 (<CJK Ideograph>)
      6000   => 16#8F20#,  --  輠 (<CJK Ideograph>)
      6001   => 16#8F21#,  --  輡 (<CJK Ideograph>)
      6002   => 16#8F23#,  --  輣 (<CJK Ideograph>)
      6003   => 16#8F25#,  --  輥 (<CJK Ideograph>)
      6004   => 16#8F27#,  --  輧 (<CJK Ideograph>)
      6005   => 16#8F28#,  --  輨 (<CJK Ideograph>)
      6006   => 16#8F2C#,  --  輬 (<CJK Ideograph>)
      6007   => 16#8F2D#,  --  輭 (<CJK Ideograph>)
      6008   => 16#8F2E#,  --  輮 (<CJK Ideograph>)
      6009   => 16#8F34#,  --  輴 (<CJK Ideograph>)
      6010   => 16#8F35#,  --  輵 (<CJK Ideograph>)
      6011   => 16#8F36#,  --  輶 (<CJK Ideograph>)
      6012   => 16#8F37#,  --  輷 (<CJK Ideograph>)
      6013   => 16#8F3A#,  --  輺 (<CJK Ideograph>)
      6014   => 16#8F40#,  --  轀 (<CJK Ideograph>)
      6015   => 16#8F41#,  --  轁 (<CJK Ideograph>)
      6016   => 16#8F43#,  --  轃 (<CJK Ideograph>)
      6017   => 16#8F47#,  --  轇 (<CJK Ideograph>)
      6018   => 16#8F4F#,  --  轏 (<CJK Ideograph>)
      6019   => 16#8F51#,  --  轑 (<CJK Ideograph>)
      6020   => 16#8F52#,  --  轒 (<CJK Ideograph>)
      6021   => 16#8F53#,  --  轓 (<CJK Ideograph>)
      6022   => 16#8F54#,  --  轔 (<CJK Ideograph>)
      6023   => 16#8F55#,  --  轕 (<CJK Ideograph>)
      6024   => 16#8F58#,  --  轘 (<CJK Ideograph>)
      6025   => 16#8F5D#,  --  轝 (<CJK Ideograph>)
      6026   => 16#8F5E#,  --  轞 (<CJK Ideograph>)
      6027   => 16#8F65#,  --  轥 (<CJK Ideograph>)
      6028   => 16#8F9D#,  --  辝 (<CJK Ideograph>)
      6029   => 16#8FA0#,  --  辠 (<CJK Ideograph>)
      6030   => 16#8FA1#,  --  辡 (<CJK Ideograph>)
      6031   => 16#8FA4#,  --  辤 (<CJK Ideograph>)
      6032   => 16#8FA5#,  --  辥 (<CJK Ideograph>)
      6033   => 16#8FA6#,  --  辦 (<CJK Ideograph>)
      6034   => 16#8FB5#,  --  辵 (<CJK Ideograph>)
      6035   => 16#8FB6#,  --  辶 (<CJK Ideograph>)
      6036   => 16#8FB8#,  --  辸 (<CJK Ideograph>)
      6037   => 16#8FBE#,  --  达 (<CJK Ideograph>)
      6038   => 16#8FC0#,  --  迀 (<CJK Ideograph>)
      6039   => 16#8FC1#,  --  迁 (<CJK Ideograph>)
      6040   => 16#8FC6#,  --  迆 (<CJK Ideograph>)
      6041   => 16#8FCA#,  --  迊 (<CJK Ideograph>)
      6042   => 16#8FCB#,  --  迋 (<CJK Ideograph>)
      6043   => 16#8FCD#,  --  迍 (<CJK Ideograph>)
      6044   => 16#8FD0#,  --  运 (<CJK Ideograph>)
      6045   => 16#8FD2#,  --  迒 (<CJK Ideograph>)
      6046   => 16#8FD3#,  --  迓 (<CJK Ideograph>)
      6047   => 16#8FD5#,  --  迕 (<CJK Ideograph>)
      6048   => 16#8FE0#,  --  迠 (<CJK Ideograph>)
      6049   => 16#8FE3#,  --  迣 (<CJK Ideograph>)
      6050   => 16#8FE4#,  --  迤 (<CJK Ideograph>)
      6051   => 16#8FE8#,  --  迨 (<CJK Ideograph>)
      6052   => 16#8FEE#,  --  迮 (<CJK Ideograph>)
      6053   => 16#8FF1#,  --  迱 (<CJK Ideograph>)
      6054   => 16#8FF5#,  --  迵 (<CJK Ideograph>)
      6055   => 16#8FF6#,  --  迶 (<CJK Ideograph>)
      6056   => 16#8FFB#,  --  迻 (<CJK Ideograph>)
      6057   => 16#8FFE#,  --  迾 (<CJK Ideograph>)
      6058   => 16#9002#,  --  适 (<CJK Ideograph>)
      6059   => 16#9004#,  --  逄 (<CJK Ideograph>)
      6060   => 16#9008#,  --  逈 (<CJK Ideograph>)
      6061   => 16#900C#,  --  逌 (<CJK Ideograph>)
      6062   => 16#9018#,  --  逘 (<CJK Ideograph>)
      6063   => 16#901B#,  --  逛 (<CJK Ideograph>)
      6064   => 16#9028#,  --  逨 (<CJK Ideograph>)
      6065   => 16#9029#,  --  逩 (<CJK Ideograph>)
      6066   => 16#902F#,  --  逯 (<CJK Ideograph>)
      6067   => 16#902A#,  --  逪 (<CJK Ideograph>)
      6068   => 16#902C#,  --  逬 (<CJK Ideograph>)
      6069   => 16#902D#,  --  逭 (<CJK Ideograph>)
      6070   => 16#9033#,  --  逳 (<CJK Ideograph>)
      6071   => 16#9034#,  --  逴 (<CJK Ideograph>)
      6072   => 16#9037#,  --  逷 (<CJK Ideograph>)
      6073   => 16#903F#,  --  逿 (<CJK Ideograph>)
      6074   => 16#9043#,  --  遃 (<CJK Ideograph>)
      6075   => 16#9044#,  --  遄 (<CJK Ideograph>)
      6076   => 16#904C#,  --  遌 (<CJK Ideograph>)
      6077   => 16#905B#,  --  遛 (<CJK Ideograph>)
      6078   => 16#905D#,  --  遝 (<CJK Ideograph>)
      6079   => 16#9062#,  --  遢 (<CJK Ideograph>)
      6080   => 16#9066#,  --  遦 (<CJK Ideograph>)
      6081   => 16#9067#,  --  遧 (<CJK Ideograph>)
      6082   => 16#906C#,  --  遬 (<CJK Ideograph>)
      6083   => 16#9070#,  --  遰 (<CJK Ideograph>)
      6084   => 16#9074#,  --  遴 (<CJK Ideograph>)
      6085   => 16#9079#,  --  遹 (<CJK Ideograph>)
      6086   => 16#9085#,  --  邅 (<CJK Ideograph>)
      6087   => 16#9088#,  --  邈 (<CJK Ideograph>)
      6088   => 16#908B#,  --  邋 (<CJK Ideograph>)
      6089   => 16#908C#,  --  邌 (<CJK Ideograph>)
      6090   => 16#908E#,  --  邎 (<CJK Ideograph>)
      6091   => 16#9090#,  --  邐 (<CJK Ideograph>)
      6092   => 16#9095#,  --  邕 (<CJK Ideograph>)
      6093   => 16#9097#,  --  邗 (<CJK Ideograph>)
      6094   => 16#9098#,  --  邘 (<CJK Ideograph>)
      6095   => 16#9099#,  --  邙 (<CJK Ideograph>)
      6096   => 16#909B#,  --  邛 (<CJK Ideograph>)
      6097   => 16#90A0#,  --  邠 (<CJK Ideograph>)
      6098   => 16#90A1#,  --  邡 (<CJK Ideograph>)
      6099   => 16#90A2#,  --  邢 (<CJK Ideograph>)
      6100   => 16#90A5#,  --  邥 (<CJK Ideograph>)
      6101   => 16#90B0#,  --  邰 (<CJK Ideograph>)
      6102   => 16#90B2#,  --  邲 (<CJK Ideograph>)
      6103   => 16#90B3#,  --  邳 (<CJK Ideograph>)
      6104   => 16#90B4#,  --  邴 (<CJK Ideograph>)
      6105   => 16#90B6#,  --  邶 (<CJK Ideograph>)
      6106   => 16#90BD#,  --  邽 (<CJK Ideograph>)
      6107   => 16#90CC#,  --  郌 (<CJK Ideograph>)
      6108   => 16#90BE#,  --  邾 (<CJK Ideograph>)
      6109   => 16#90C3#,  --  郃 (<CJK Ideograph>)
      6110   => 16#90C4#,  --  郄 (<CJK Ideograph>)
      6111   => 16#90C5#,  --  郅 (<CJK Ideograph>)
      6112   => 16#90C7#,  --  郇 (<CJK Ideograph>)
      6113   => 16#90C8#,  --  郈 (<CJK Ideograph>)
      6114   => 16#90D5#,  --  郕 (<CJK Ideograph>)
      6115   => 16#90D7#,  --  郗 (<CJK Ideograph>)
      6116   => 16#90D8#,  --  郘 (<CJK Ideograph>)
      6117   => 16#90D9#,  --  郙 (<CJK Ideograph>)
      6118   => 16#90DC#,  --  郜 (<CJK Ideograph>)
      6119   => 16#90DD#,  --  郝 (<CJK Ideograph>)
      6120   => 16#90DF#,  --  郟 (<CJK Ideograph>)
      6121   => 16#90E5#,  --  郥 (<CJK Ideograph>)
      6122   => 16#90D2#,  --  郒 (<CJK Ideograph>)
      6123   => 16#90F6#,  --  郶 (<CJK Ideograph>)
      6124   => 16#90EB#,  --  郫 (<CJK Ideograph>)
      6125   => 16#90EF#,  --  郯 (<CJK Ideograph>)
      6126   => 16#90F0#,  --  郰 (<CJK Ideograph>)
      6127   => 16#90F4#,  --  郴 (<CJK Ideograph>)
      6128   => 16#90FE#,  --  郾 (<CJK Ideograph>)
      6129   => 16#90FF#,  --  郿 (<CJK Ideograph>)
      6130   => 16#9100#,  --  鄀 (<CJK Ideograph>)
      6131   => 16#9104#,  --  鄄 (<CJK Ideograph>)
      6132   => 16#9105#,  --  鄅 (<CJK Ideograph>)
      6133   => 16#9106#,  --  鄆 (<CJK Ideograph>)
      6134   => 16#9108#,  --  鄈 (<CJK Ideograph>)
      6135   => 16#910D#,  --  鄍 (<CJK Ideograph>)
      6136   => 16#9110#,  --  鄐 (<CJK Ideograph>)
      6137   => 16#9114#,  --  鄔 (<CJK Ideograph>)
      6138   => 16#9116#,  --  鄖 (<CJK Ideograph>)
      6139   => 16#9117#,  --  鄗 (<CJK Ideograph>)
      6140   => 16#9118#,  --  鄘 (<CJK Ideograph>)
      6141   => 16#911A#,  --  鄚 (<CJK Ideograph>)
      6142   => 16#911C#,  --  鄜 (<CJK Ideograph>)
      6143   => 16#911E#,  --  鄞 (<CJK Ideograph>)
      6144   => 16#9120#,  --  鄠 (<CJK Ideograph>)
      6145   => 16#9125#,  --  鄥 (<CJK Ideograph>)
      6146   => 16#9122#,  --  鄢 (<CJK Ideograph>)
      6147   => 16#9123#,  --  鄣 (<CJK Ideograph>)
      6148   => 16#9127#,  --  鄧 (<CJK Ideograph>)
      6149   => 16#9129#,  --  鄩 (<CJK Ideograph>)
      6150   => 16#912E#,  --  鄮 (<CJK Ideograph>)
      6151   => 16#912F#,  --  鄯 (<CJK Ideograph>)
      6152   => 16#9131#,  --  鄱 (<CJK Ideograph>)
      6153   => 16#9134#,  --  鄴 (<CJK Ideograph>)
      6154   => 16#9136#,  --  鄶 (<CJK Ideograph>)
      6155   => 16#9137#,  --  鄷 (<CJK Ideograph>)
      6156   => 16#9139#,  --  鄹 (<CJK Ideograph>)
      6157   => 16#913A#,  --  鄺 (<CJK Ideograph>)
      6158   => 16#913C#,  --  鄼 (<CJK Ideograph>)
      6159   => 16#913D#,  --  鄽 (<CJK Ideograph>)
      6160   => 16#9143#,  --  酃 (<CJK Ideograph>)
      6161   => 16#9147#,  --  酇 (<CJK Ideograph>)
      6162   => 16#9148#,  --  酈 (<CJK Ideograph>)
      6163   => 16#914F#,  --  酏 (<CJK Ideograph>)
      6164   => 16#9153#,  --  酓 (<CJK Ideograph>)
      6165   => 16#9157#,  --  酗 (<CJK Ideograph>)
      6166   => 16#9159#,  --  酙 (<CJK Ideograph>)
      6167   => 16#915A#,  --  酚 (<CJK Ideograph>)
      6168   => 16#915B#,  --  酛 (<CJK Ideograph>)
      6169   => 16#9161#,  --  酡 (<CJK Ideograph>)
      6170   => 16#9164#,  --  酤 (<CJK Ideograph>)
      6171   => 16#9167#,  --  酧 (<CJK Ideograph>)
      6172   => 16#916D#,  --  酭 (<CJK Ideograph>)
      6173   => 16#9174#,  --  酴 (<CJK Ideograph>)
      6174   => 16#9179#,  --  酹 (<CJK Ideograph>)
      6175   => 16#917A#,  --  酺 (<CJK Ideograph>)
      6176   => 16#917B#,  --  酻 (<CJK Ideograph>)
      6177   => 16#9181#,  --  醁 (<CJK Ideograph>)
      6178   => 16#9183#,  --  醃 (<CJK Ideograph>)
      6179   => 16#9185#,  --  醅 (<CJK Ideograph>)
      6180   => 16#9186#,  --  醆 (<CJK Ideograph>)
      6181   => 16#918A#,  --  醊 (<CJK Ideograph>)
      6182   => 16#918E#,  --  醎 (<CJK Ideograph>)
      6183   => 16#9191#,  --  醑 (<CJK Ideograph>)
      6184   => 16#9193#,  --  醓 (<CJK Ideograph>)
      6185   => 16#9194#,  --  醔 (<CJK Ideograph>)
      6186   => 16#9195#,  --  醕 (<CJK Ideograph>)
      6187   => 16#9198#,  --  醘 (<CJK Ideograph>)
      6188   => 16#919E#,  --  醞 (<CJK Ideograph>)
      6189   => 16#91A1#,  --  醡 (<CJK Ideograph>)
      6190   => 16#91A6#,  --  醦 (<CJK Ideograph>)
      6191   => 16#91A8#,  --  醨 (<CJK Ideograph>)
      6192   => 16#91AC#,  --  醬 (<CJK Ideograph>)
      6193   => 16#91AD#,  --  醭 (<CJK Ideograph>)
      6194   => 16#91AE#,  --  醮 (<CJK Ideograph>)
      6195   => 16#91B0#,  --  醰 (<CJK Ideograph>)
      6196   => 16#91B1#,  --  醱 (<CJK Ideograph>)
      6197   => 16#91B2#,  --  醲 (<CJK Ideograph>)
      6198   => 16#91B3#,  --  醳 (<CJK Ideograph>)
      6199   => 16#91B6#,  --  醶 (<CJK Ideograph>)
      6200   => 16#91BB#,  --  醻 (<CJK Ideograph>)
      6201   => 16#91BC#,  --  醼 (<CJK Ideograph>)
      6202   => 16#91BD#,  --  醽 (<CJK Ideograph>)
      6203   => 16#91BF#,  --  醿 (<CJK Ideograph>)
      6204   => 16#91C2#,  --  釂 (<CJK Ideograph>)
      6205   => 16#91C3#,  --  釃 (<CJK Ideograph>)
      6206   => 16#91C5#,  --  釅 (<CJK Ideograph>)
      6207   => 16#91D3#,  --  釓 (<CJK Ideograph>)
      6208   => 16#91D4#,  --  釔 (<CJK Ideograph>)
      6209   => 16#91D7#,  --  釗 (<CJK Ideograph>)
      6210   => 16#91D9#,  --  釙 (<CJK Ideograph>)
      6211   => 16#91DA#,  --  釚 (<CJK Ideograph>)
      6212   => 16#91DE#,  --  釞 (<CJK Ideograph>)
      6213   => 16#91E4#,  --  釤 (<CJK Ideograph>)
      6214   => 16#91E5#,  --  釥 (<CJK Ideograph>)
      6215   => 16#91E9#,  --  釩 (<CJK Ideograph>)
      6216   => 16#91EA#,  --  釪 (<CJK Ideograph>)
      6217   => 16#91EC#,  --  釬 (<CJK Ideograph>)
      6218   => 16#91ED#,  --  釭 (<CJK Ideograph>)
      6219   => 16#91EE#,  --  釮 (<CJK Ideograph>)
      6220   => 16#91EF#,  --  釯 (<CJK Ideograph>)
      6221   => 16#91F0#,  --  釰 (<CJK Ideograph>)
      6222   => 16#91F1#,  --  釱 (<CJK Ideograph>)
      6223   => 16#91F7#,  --  釷 (<CJK Ideograph>)
      6224   => 16#91F9#,  --  釹 (<CJK Ideograph>)
      6225   => 16#91FB#,  --  釻 (<CJK Ideograph>)
      6226   => 16#91FD#,  --  釽 (<CJK Ideograph>)
      6227   => 16#9200#,  --  鈀 (<CJK Ideograph>)
      6228   => 16#9201#,  --  鈁 (<CJK Ideograph>)
      6229   => 16#9204#,  --  鈄 (<CJK Ideograph>)
      6230   => 16#9205#,  --  鈅 (<CJK Ideograph>)
      6231   => 16#9206#,  --  鈆 (<CJK Ideograph>)
      6232   => 16#9207#,  --  鈇 (<CJK Ideograph>)
      6233   => 16#9209#,  --  鈉 (<CJK Ideograph>)
      6234   => 16#920A#,  --  鈊 (<CJK Ideograph>)
      6235   => 16#920C#,  --  鈌 (<CJK Ideograph>)
      6236   => 16#9210#,  --  鈐 (<CJK Ideograph>)
      6237   => 16#9212#,  --  鈒 (<CJK Ideograph>)
      6238   => 16#9213#,  --  鈓 (<CJK Ideograph>)
      6239   => 16#9216#,  --  鈖 (<CJK Ideograph>)
      6240   => 16#9218#,  --  鈘 (<CJK Ideograph>)
      6241   => 16#921C#,  --  鈜 (<CJK Ideograph>)
      6242   => 16#921D#,  --  鈝 (<CJK Ideograph>)
      6243   => 16#9223#,  --  鈣 (<CJK Ideograph>)
      6244   => 16#9224#,  --  鈤 (<CJK Ideograph>)
      6245   => 16#9225#,  --  鈥 (<CJK Ideograph>)
      6246   => 16#9226#,  --  鈦 (<CJK Ideograph>)
      6247   => 16#9228#,  --  鈨 (<CJK Ideograph>)
      6248   => 16#922E#,  --  鈮 (<CJK Ideograph>)
      6249   => 16#922F#,  --  鈯 (<CJK Ideograph>)
      6250   => 16#9230#,  --  鈰 (<CJK Ideograph>)
      6251   => 16#9233#,  --  鈳 (<CJK Ideograph>)
      6252   => 16#9235#,  --  鈵 (<CJK Ideograph>)
      6253   => 16#9236#,  --  鈶 (<CJK Ideograph>)
      6254   => 16#9238#,  --  鈸 (<CJK Ideograph>)
      6255   => 16#9239#,  --  鈹 (<CJK Ideograph>)
      6256   => 16#923A#,  --  鈺 (<CJK Ideograph>)
      6257   => 16#923C#,  --  鈼 (<CJK Ideograph>)
      6258   => 16#923E#,  --  鈾 (<CJK Ideograph>)
      6259   => 16#9240#,  --  鉀 (<CJK Ideograph>)
      6260   => 16#9242#,  --  鉂 (<CJK Ideograph>)
      6261   => 16#9243#,  --  鉃 (<CJK Ideograph>)
      6262   => 16#9246#,  --  鉆 (<CJK Ideograph>)
      6263   => 16#9247#,  --  鉇 (<CJK Ideograph>)
      6264   => 16#924A#,  --  鉊 (<CJK Ideograph>)
      6265   => 16#924D#,  --  鉍 (<CJK Ideograph>)
      6266   => 16#924E#,  --  鉎 (<CJK Ideograph>)
      6267   => 16#924F#,  --  鉏 (<CJK Ideograph>)
      6268   => 16#9251#,  --  鉑 (<CJK Ideograph>)
      6269   => 16#9258#,  --  鉘 (<CJK Ideograph>)
      6270   => 16#9259#,  --  鉙 (<CJK Ideograph>)
      6271   => 16#925C#,  --  鉜 (<CJK Ideograph>)
      6272   => 16#925D#,  --  鉝 (<CJK Ideograph>)
      6273   => 16#9260#,  --  鉠 (<CJK Ideograph>)
      6274   => 16#9261#,  --  鉡 (<CJK Ideograph>)
      6275   => 16#9265#,  --  鉥 (<CJK Ideograph>)
      6276   => 16#9267#,  --  鉧 (<CJK Ideograph>)
      6277   => 16#9268#,  --  鉨 (<CJK Ideograph>)
      6278   => 16#9269#,  --  鉩 (<CJK Ideograph>)
      6279   => 16#926E#,  --  鉮 (<CJK Ideograph>)
      6280   => 16#926F#,  --  鉯 (<CJK Ideograph>)
      6281   => 16#9270#,  --  鉰 (<CJK Ideograph>)
      6282   => 16#9275#,  --  鉵 (<CJK Ideograph>)
      6283   => 16#9276#,  --  鉶 (<CJK Ideograph>)
      6284   => 16#9277#,  --  鉷 (<CJK Ideograph>)
      6285   => 16#9278#,  --  鉸 (<CJK Ideograph>)
      6286   => 16#9279#,  --  鉹 (<CJK Ideograph>)
      6287   => 16#927B#,  --  鉻 (<CJK Ideograph>)
      6288   => 16#927C#,  --  鉼 (<CJK Ideograph>)
      6289   => 16#927D#,  --  鉽 (<CJK Ideograph>)
      6290   => 16#927F#,  --  鉿 (<CJK Ideograph>)
      6291   => 16#9288#,  --  銈 (<CJK Ideograph>)
      6292   => 16#9289#,  --  銉 (<CJK Ideograph>)
      6293   => 16#928A#,  --  銊 (<CJK Ideograph>)
      6294   => 16#928D#,  --  銍 (<CJK Ideograph>)
      6295   => 16#928E#,  --  銎 (<CJK Ideograph>)
      6296   => 16#9292#,  --  銒 (<CJK Ideograph>)
      6297   => 16#9297#,  --  銗 (<CJK Ideograph>)
      6298   => 16#9299#,  --  銙 (<CJK Ideograph>)
      6299   => 16#929F#,  --  銟 (<CJK Ideograph>)
      6300   => 16#92A0#,  --  銠 (<CJK Ideograph>)
      6301   => 16#92A4#,  --  銤 (<CJK Ideograph>)
      6302   => 16#92A5#,  --  銥 (<CJK Ideograph>)
      6303   => 16#92A7#,  --  銧 (<CJK Ideograph>)
      6304   => 16#92A8#,  --  銨 (<CJK Ideograph>)
      6305   => 16#92AB#,  --  銫 (<CJK Ideograph>)
      6306   => 16#92AF#,  --  銯 (<CJK Ideograph>)
      6307   => 16#92B2#,  --  銲 (<CJK Ideograph>)
      6308   => 16#92B6#,  --  銶 (<CJK Ideograph>)
      6309   => 16#92B8#,  --  銸 (<CJK Ideograph>)
      6310   => 16#92BA#,  --  銺 (<CJK Ideograph>)
      6311   => 16#92BB#,  --  銻 (<CJK Ideograph>)
      6312   => 16#92BC#,  --  銼 (<CJK Ideograph>)
      6313   => 16#92BD#,  --  銽 (<CJK Ideograph>)
      6314   => 16#92BF#,  --  銿 (<CJK Ideograph>)
      6315   => 16#92C0#,  --  鋀 (<CJK Ideograph>)
      6316   => 16#92C1#,  --  鋁 (<CJK Ideograph>)
      6317   => 16#92C2#,  --  鋂 (<CJK Ideograph>)
      6318   => 16#92C3#,  --  鋃 (<CJK Ideograph>)
      6319   => 16#92C5#,  --  鋅 (<CJK Ideograph>)
      6320   => 16#92C6#,  --  鋆 (<CJK Ideograph>)
      6321   => 16#92C7#,  --  鋇 (<CJK Ideograph>)
      6322   => 16#92C8#,  --  鋈 (<CJK Ideograph>)
      6323   => 16#92CB#,  --  鋋 (<CJK Ideograph>)
      6324   => 16#92CC#,  --  鋌 (<CJK Ideograph>)
      6325   => 16#92CD#,  --  鋍 (<CJK Ideograph>)
      6326   => 16#92CE#,  --  鋎 (<CJK Ideograph>)
      6327   => 16#92D0#,  --  鋐 (<CJK Ideograph>)
      6328   => 16#92D3#,  --  鋓 (<CJK Ideograph>)
      6329   => 16#92D5#,  --  鋕 (<CJK Ideograph>)
      6330   => 16#92D7#,  --  鋗 (<CJK Ideograph>)
      6331   => 16#92D8#,  --  鋘 (<CJK Ideograph>)
      6332   => 16#92D9#,  --  鋙 (<CJK Ideograph>)
      6333   => 16#92DC#,  --  鋜 (<CJK Ideograph>)
      6334   => 16#92DD#,  --  鋝 (<CJK Ideograph>)
      6335   => 16#92DF#,  --  鋟 (<CJK Ideograph>)
      6336   => 16#92E0#,  --  鋠 (<CJK Ideograph>)
      6337   => 16#92E1#,  --  鋡 (<CJK Ideograph>)
      6338   => 16#92E3#,  --  鋣 (<CJK Ideograph>)
      6339   => 16#92E5#,  --  鋥 (<CJK Ideograph>)
      6340   => 16#92E7#,  --  鋧 (<CJK Ideograph>)
      6341   => 16#92E8#,  --  鋨 (<CJK Ideograph>)
      6342   => 16#92EC#,  --  鋬 (<CJK Ideograph>)
      6343   => 16#92EE#,  --  鋮 (<CJK Ideograph>)
      6344   => 16#92F0#,  --  鋰 (<CJK Ideograph>)
      6345   => 16#92F9#,  --  鋹 (<CJK Ideograph>)
      6346   => 16#92FB#,  --  鋻 (<CJK Ideograph>)
      6347   => 16#92FF#,  --  鋿 (<CJK Ideograph>)
      6348   => 16#9300#,  --  錀 (<CJK Ideograph>)
      6349   => 16#9302#,  --  錂 (<CJK Ideograph>)
      6350   => 16#9308#,  --  錈 (<CJK Ideograph>)
      6351   => 16#930D#,  --  錍 (<CJK Ideograph>)
      6352   => 16#9311#,  --  錑 (<CJK Ideograph>)
      6353   => 16#9314#,  --  錔 (<CJK Ideograph>)
      6354   => 16#9315#,  --  錕 (<CJK Ideograph>)
      6355   => 16#931C#,  --  錜 (<CJK Ideograph>)
      6356   => 16#931D#,  --  錝 (<CJK Ideograph>)
      6357   => 16#931E#,  --  錞 (<CJK Ideograph>)
      6358   => 16#931F#,  --  錟 (<CJK Ideograph>)
      6359   => 16#9321#,  --  錡 (<CJK Ideograph>)
      6360   => 16#9324#,  --  錤 (<CJK Ideograph>)
      6361   => 16#9325#,  --  錥 (<CJK Ideograph>)
      6362   => 16#9327#,  --  錧 (<CJK Ideograph>)
      6363   => 16#9329#,  --  錩 (<CJK Ideograph>)
      6364   => 16#932A#,  --  錪 (<CJK Ideograph>)
      6365   => 16#9333#,  --  錳 (<CJK Ideograph>)
      6366   => 16#9334#,  --  錴 (<CJK Ideograph>)
      6367   => 16#9336#,  --  錶 (<CJK Ideograph>)
      6368   => 16#9337#,  --  錷 (<CJK Ideograph>)
      6369   => 16#9347#,  --  鍇 (<CJK Ideograph>)
      6370   => 16#9348#,  --  鍈 (<CJK Ideograph>)
      6371   => 16#9349#,  --  鍉 (<CJK Ideograph>)
      6372   => 16#9350#,  --  鍐 (<CJK Ideograph>)
      6373   => 16#9351#,  --  鍑 (<CJK Ideograph>)
      6374   => 16#9352#,  --  鍒 (<CJK Ideograph>)
      6375   => 16#9355#,  --  鍕 (<CJK Ideograph>)
      6376   => 16#9357#,  --  鍗 (<CJK Ideograph>)
      6377   => 16#9358#,  --  鍘 (<CJK Ideograph>)
      6378   => 16#935A#,  --  鍚 (<CJK Ideograph>)
      6379   => 16#935E#,  --  鍞 (<CJK Ideograph>)
      6380   => 16#9364#,  --  鍤 (<CJK Ideograph>)
      6381   => 16#9365#,  --  鍥 (<CJK Ideograph>)
      6382   => 16#9367#,  --  鍧 (<CJK Ideograph>)
      6383   => 16#9369#,  --  鍩 (<CJK Ideograph>)
      6384   => 16#936A#,  --  鍪 (<CJK Ideograph>)
      6385   => 16#936D#,  --  鍭 (<CJK Ideograph>)
      6386   => 16#936F#,  --  鍯 (<CJK Ideograph>)
      6387   => 16#9370#,  --  鍰 (<CJK Ideograph>)
      6388   => 16#9371#,  --  鍱 (<CJK Ideograph>)
      6389   => 16#9373#,  --  鍳 (<CJK Ideograph>)
      6390   => 16#9374#,  --  鍴 (<CJK Ideograph>)
      6391   => 16#9376#,  --  鍶 (<CJK Ideograph>)
      6392   => 16#937A#,  --  鍺 (<CJK Ideograph>)
      6393   => 16#937D#,  --  鍽 (<CJK Ideograph>)
      6394   => 16#937F#,  --  鍿 (<CJK Ideograph>)
      6395   => 16#9380#,  --  鎀 (<CJK Ideograph>)
      6396   => 16#9381#,  --  鎁 (<CJK Ideograph>)
      6397   => 16#9382#,  --  鎂 (<CJK Ideograph>)
      6398   => 16#9388#,  --  鎈 (<CJK Ideograph>)
      6399   => 16#938A#,  --  鎊 (<CJK Ideograph>)
      6400   => 16#938B#,  --  鎋 (<CJK Ideograph>)
      6401   => 16#938D#,  --  鎍 (<CJK Ideograph>)
      6402   => 16#938F#,  --  鎏 (<CJK Ideograph>)
      6403   => 16#9392#,  --  鎒 (<CJK Ideograph>)
      6404   => 16#9395#,  --  鎕 (<CJK Ideograph>)
      6405   => 16#9398#,  --  鎘 (<CJK Ideograph>)
      6406   => 16#939B#,  --  鎛 (<CJK Ideograph>)
      6407   => 16#939E#,  --  鎞 (<CJK Ideograph>)
      6408   => 16#93A1#,  --  鎡 (<CJK Ideograph>)
      6409   => 16#93A3#,  --  鎣 (<CJK Ideograph>)
      6410   => 16#93A4#,  --  鎤 (<CJK Ideograph>)
      6411   => 16#93A6#,  --  鎦 (<CJK Ideograph>)
      6412   => 16#93A8#,  --  鎨 (<CJK Ideograph>)
      6413   => 16#93AB#,  --  鎫 (<CJK Ideograph>)
      6414   => 16#93B4#,  --  鎴 (<CJK Ideograph>)
      6415   => 16#93B5#,  --  鎵 (<CJK Ideograph>)
      6416   => 16#93B6#,  --  鎶 (<CJK Ideograph>)
      6417   => 16#93BA#,  --  鎺 (<CJK Ideograph>)
      6418   => 16#93A9#,  --  鎩 (<CJK Ideograph>)
      6419   => 16#93C1#,  --  鏁 (<CJK Ideograph>)
      6420   => 16#93C4#,  --  鏄 (<CJK Ideograph>)
      6421   => 16#93C5#,  --  鏅 (<CJK Ideograph>)
      6422   => 16#93C6#,  --  鏆 (<CJK Ideograph>)
      6423   => 16#93C7#,  --  鏇 (<CJK Ideograph>)
      6424   => 16#93C9#,  --  鏉 (<CJK Ideograph>)
      6425   => 16#93CA#,  --  鏊 (<CJK Ideograph>)
      6426   => 16#93CB#,  --  鏋 (<CJK Ideograph>)
      6427   => 16#93CC#,  --  鏌 (<CJK Ideograph>)
      6428   => 16#93CD#,  --  鏍 (<CJK Ideograph>)
      6429   => 16#93D3#,  --  鏓 (<CJK Ideograph>)
      6430   => 16#93D9#,  --  鏙 (<CJK Ideograph>)
      6431   => 16#93DC#,  --  鏜 (<CJK Ideograph>)
      6432   => 16#93DE#,  --  鏞 (<CJK Ideograph>)
      6433   => 16#93DF#,  --  鏟 (<CJK Ideograph>)
      6434   => 16#93E2#,  --  鏢 (<CJK Ideograph>)
      6435   => 16#93E6#,  --  鏦 (<CJK Ideograph>)
      6436   => 16#93E7#,  --  鏧 (<CJK Ideograph>)
      6437   => 16#93F9#,  --  鏹 (<CJK Ideograph>)
      6438   => 16#93F7#,  --  鏷 (<CJK Ideograph>)
      6439   => 16#93F8#,  --  鏸 (<CJK Ideograph>)
      6440   => 16#93FA#,  --  鏺 (<CJK Ideograph>)
      6441   => 16#93FB#,  --  鏻 (<CJK Ideograph>)
      6442   => 16#93FD#,  --  鏽 (<CJK Ideograph>)
      6443   => 16#9401#,  --  鐁 (<CJK Ideograph>)
      6444   => 16#9402#,  --  鐂 (<CJK Ideograph>)
      6445   => 16#9404#,  --  鐄 (<CJK Ideograph>)
      6446   => 16#9408#,  --  鐈 (<CJK Ideograph>)
      6447   => 16#9409#,  --  鐉 (<CJK Ideograph>)
      6448   => 16#940D#,  --  鐍 (<CJK Ideograph>)
      6449   => 16#940E#,  --  鐎 (<CJK Ideograph>)
      6450   => 16#940F#,  --  鐏 (<CJK Ideograph>)
      6451   => 16#9415#,  --  鐕 (<CJK Ideograph>)
      6452   => 16#9416#,  --  鐖 (<CJK Ideograph>)
      6453   => 16#9417#,  --  鐗 (<CJK Ideograph>)
      6454   => 16#941F#,  --  鐟 (<CJK Ideograph>)
      6455   => 16#942E#,  --  鐮 (<CJK Ideograph>)
      6456   => 16#942F#,  --  鐯 (<CJK Ideograph>)
      6457   => 16#9431#,  --  鐱 (<CJK Ideograph>)
      6458   => 16#9432#,  --  鐲 (<CJK Ideograph>)
      6459   => 16#9433#,  --  鐳 (<CJK Ideograph>)
      6460   => 16#9434#,  --  鐴 (<CJK Ideograph>)
      6461   => 16#943B#,  --  鐻 (<CJK Ideograph>)
      6462   => 16#943F#,  --  鐿 (<CJK Ideograph>)
      6463   => 16#943D#,  --  鐽 (<CJK Ideograph>)
      6464   => 16#9443#,  --  鑃 (<CJK Ideograph>)
      6465   => 16#9445#,  --  鑅 (<CJK Ideograph>)
      6466   => 16#9448#,  --  鑈 (<CJK Ideograph>)
      6467   => 16#944A#,  --  鑊 (<CJK Ideograph>)
      6468   => 16#944C#,  --  鑌 (<CJK Ideograph>)
      6469   => 16#9455#,  --  鑕 (<CJK Ideograph>)
      6470   => 16#9459#,  --  鑙 (<CJK Ideograph>)
      6471   => 16#945C#,  --  鑜 (<CJK Ideograph>)
      6472   => 16#945F#,  --  鑟 (<CJK Ideograph>)
      6473   => 16#9461#,  --  鑡 (<CJK Ideograph>)
      6474   => 16#9463#,  --  鑣 (<CJK Ideograph>)
      6475   => 16#9468#,  --  鑨 (<CJK Ideograph>)
      6476   => 16#946B#,  --  鑫 (<CJK Ideograph>)
      6477   => 16#946D#,  --  鑭 (<CJK Ideograph>)
      6478   => 16#946E#,  --  鑮 (<CJK Ideograph>)
      6479   => 16#946F#,  --  鑯 (<CJK Ideograph>)
      6480   => 16#9471#,  --  鑱 (<CJK Ideograph>)
      6481   => 16#9472#,  --  鑲 (<CJK Ideograph>)
      6482   => 16#9484#,  --  钄 (<CJK Ideograph>)
      6483   => 16#9483#,  --  钃 (<CJK Ideograph>)
      6484   => 16#9578#,  --  镸 (<CJK Ideograph>)
      6485   => 16#9579#,  --  镹 (<CJK Ideograph>)
      6486   => 16#957E#,  --  镾 (<CJK Ideograph>)
      6487   => 16#9584#,  --  閄 (<CJK Ideograph>)
      6488   => 16#9588#,  --  閈 (<CJK Ideograph>)
      6489   => 16#958C#,  --  閌 (<CJK Ideograph>)
      6490   => 16#958D#,  --  閍 (<CJK Ideograph>)
      6491   => 16#958E#,  --  閎 (<CJK Ideograph>)
      6492   => 16#959D#,  --  閝 (<CJK Ideograph>)
      6493   => 16#959E#,  --  閞 (<CJK Ideograph>)
      6494   => 16#959F#,  --  閟 (<CJK Ideograph>)
      6495   => 16#95A1#,  --  閡 (<CJK Ideograph>)
      6496   => 16#95A6#,  --  閦 (<CJK Ideograph>)
      6497   => 16#95A9#,  --  閩 (<CJK Ideograph>)
      6498   => 16#95AB#,  --  閫 (<CJK Ideograph>)
      6499   => 16#95AC#,  --  閬 (<CJK Ideograph>)
      6500   => 16#95B4#,  --  閴 (<CJK Ideograph>)
      6501   => 16#95B6#,  --  閶 (<CJK Ideograph>)
      6502   => 16#95BA#,  --  閺 (<CJK Ideograph>)
      6503   => 16#95BD#,  --  閽 (<CJK Ideograph>)
      6504   => 16#95BF#,  --  閿 (<CJK Ideograph>)
      6505   => 16#95C6#,  --  闆 (<CJK Ideograph>)
      6506   => 16#95C8#,  --  闈 (<CJK Ideograph>)
      6507   => 16#95C9#,  --  闉 (<CJK Ideograph>)
      6508   => 16#95CB#,  --  闋 (<CJK Ideograph>)
      6509   => 16#95D0#,  --  闐 (<CJK Ideograph>)
      6510   => 16#95D1#,  --  闑 (<CJK Ideograph>)
      6511   => 16#95D2#,  --  闒 (<CJK Ideograph>)
      6512   => 16#95D3#,  --  闓 (<CJK Ideograph>)
      6513   => 16#95D9#,  --  闙 (<CJK Ideograph>)
      6514   => 16#95DA#,  --  闚 (<CJK Ideograph>)
      6515   => 16#95DD#,  --  闝 (<CJK Ideograph>)
      6516   => 16#95DE#,  --  闞 (<CJK Ideograph>)
      6517   => 16#95DF#,  --  闟 (<CJK Ideograph>)
      6518   => 16#95E0#,  --  闠 (<CJK Ideograph>)
      6519   => 16#95E4#,  --  闤 (<CJK Ideograph>)
      6520   => 16#95E6#,  --  闦 (<CJK Ideograph>)
      6521   => 16#961D#,  --  阝 (<CJK Ideograph>)
      6522   => 16#961E#,  --  阞 (<CJK Ideograph>)
      6523   => 16#9622#,  --  阢 (<CJK Ideograph>)
      6524   => 16#9624#,  --  阤 (<CJK Ideograph>)
      6525   => 16#9625#,  --  阥 (<CJK Ideograph>)
      6526   => 16#9626#,  --  阦 (<CJK Ideograph>)
      6527   => 16#962C#,  --  阬 (<CJK Ideograph>)
      6528   => 16#9631#,  --  阱 (<CJK Ideograph>)
      6529   => 16#9633#,  --  阳 (<CJK Ideograph>)
      6530   => 16#9637#,  --  阷 (<CJK Ideograph>)
      6531   => 16#9638#,  --  阸 (<CJK Ideograph>)
      6532   => 16#9639#,  --  阹 (<CJK Ideograph>)
      6533   => 16#963A#,  --  阺 (<CJK Ideograph>)
      6534   => 16#963C#,  --  阼 (<CJK Ideograph>)
      6535   => 16#963D#,  --  阽 (<CJK Ideograph>)
      6536   => 16#9641#,  --  陁 (<CJK Ideograph>)
      6537   => 16#9652#,  --  陒 (<CJK Ideograph>)
      6538   => 16#9654#,  --  陔 (<CJK Ideograph>)
      6539   => 16#9656#,  --  陖 (<CJK Ideograph>)
      6540   => 16#9657#,  --  陗 (<CJK Ideograph>)
      6541   => 16#9658#,  --  陘 (<CJK Ideograph>)
      6542   => 16#9661#,  --  陡 (<CJK Ideograph>)
      6543   => 16#966E#,  --  陮 (<CJK Ideograph>)
      6544   => 16#9674#,  --  陴 (<CJK Ideograph>)
      6545   => 16#967B#,  --  陻 (<CJK Ideograph>)
      6546   => 16#967C#,  --  陼 (<CJK Ideograph>)
      6547   => 16#967E#,  --  陾 (<CJK Ideograph>)
      6548   => 16#967F#,  --  陿 (<CJK Ideograph>)
      6549   => 16#9681#,  --  隁 (<CJK Ideograph>)
      6550   => 16#9682#,  --  隂 (<CJK Ideograph>)
      6551   => 16#9683#,  --  隃 (<CJK Ideograph>)
      6552   => 16#9684#,  --  隄 (<CJK Ideograph>)
      6553   => 16#9689#,  --  隉 (<CJK Ideograph>)
      6554   => 16#9691#,  --  隑 (<CJK Ideograph>)
      6555   => 16#9696#,  --  隖 (<CJK Ideograph>)
      6556   => 16#969A#,  --  隚 (<CJK Ideograph>)
      6557   => 16#969D#,  --  隝 (<CJK Ideograph>)
      6558   => 16#969F#,  --  隟 (<CJK Ideograph>)
      6559   => 16#96A4#,  --  隤 (<CJK Ideograph>)
      6560   => 16#96A5#,  --  隥 (<CJK Ideograph>)
      6561   => 16#96A6#,  --  隦 (<CJK Ideograph>)
      6562   => 16#96A9#,  --  隩 (<CJK Ideograph>)
      6563   => 16#96AE#,  --  隮 (<CJK Ideograph>)
      6564   => 16#96AF#,  --  隯 (<CJK Ideograph>)
      6565   => 16#96B3#,  --  隳 (<CJK Ideograph>)
      6566   => 16#96BA#,  --  隺 (<CJK Ideograph>)
      6567   => 16#96CA#,  --  雊 (<CJK Ideograph>)
      6568   => 16#96D2#,  --  雒 (<CJK Ideograph>)
      6569   => 16#5DB2#,  --  嶲 (<CJK Ideograph>)
      6570   => 16#96D8#,  --  雘 (<CJK Ideograph>)
      6571   => 16#96DA#,  --  雚 (<CJK Ideograph>)
      6572   => 16#96DD#,  --  雝 (<CJK Ideograph>)
      6573   => 16#96DE#,  --  雞 (<CJK Ideograph>)
      6574   => 16#96DF#,  --  雟 (<CJK Ideograph>)
      6575   => 16#96E9#,  --  雩 (<CJK Ideograph>)
      6576   => 16#96EF#,  --  雯 (<CJK Ideograph>)
      6577   => 16#96F1#,  --  雱 (<CJK Ideograph>)
      6578   => 16#96FA#,  --  雺 (<CJK Ideograph>)
      6579   => 16#9702#,  --  霂 (<CJK Ideograph>)
      6580   => 16#9703#,  --  霃 (<CJK Ideograph>)
      6581   => 16#9705#,  --  霅 (<CJK Ideograph>)
      6582   => 16#9709#,  --  霉 (<CJK Ideograph>)
      6583   => 16#971A#,  --  霚 (<CJK Ideograph>)
      6584   => 16#971B#,  --  霛 (<CJK Ideograph>)
      6585   => 16#971D#,  --  霝 (<CJK Ideograph>)
      6586   => 16#9721#,  --  霡 (<CJK Ideograph>)
      6587   => 16#9722#,  --  霢 (<CJK Ideograph>)
      6588   => 16#9723#,  --  霣 (<CJK Ideograph>)
      6589   => 16#9728#,  --  霨 (<CJK Ideograph>)
      6590   => 16#9731#,  --  霱 (<CJK Ideograph>)
      6591   => 16#9733#,  --  霳 (<CJK Ideograph>)
      6592   => 16#9741#,  --  靁 (<CJK Ideograph>)
      6593   => 16#9743#,  --  靃 (<CJK Ideograph>)
      6594   => 16#974A#,  --  靊 (<CJK Ideograph>)
      6595   => 16#974E#,  --  靎 (<CJK Ideograph>)
      6596   => 16#974F#,  --  靏 (<CJK Ideograph>)
      6597   => 16#9755#,  --  靕 (<CJK Ideograph>)
      6598   => 16#9757#,  --  靗 (<CJK Ideograph>)
      6599   => 16#9758#,  --  靘 (<CJK Ideograph>)
      6600   => 16#975A#,  --  靚 (<CJK Ideograph>)
      6601   => 16#975B#,  --  靛 (<CJK Ideograph>)
      6602   => 16#9763#,  --  靣 (<CJK Ideograph>)
      6603   => 16#9767#,  --  靧 (<CJK Ideograph>)
      6604   => 16#976A#,  --  靪 (<CJK Ideograph>)
      6605   => 16#976E#,  --  靮 (<CJK Ideograph>)
      6606   => 16#9773#,  --  靳 (<CJK Ideograph>)
      6607   => 16#9776#,  --  靶 (<CJK Ideograph>)
      6608   => 16#9777#,  --  靷 (<CJK Ideograph>)
      6609   => 16#9778#,  --  靸 (<CJK Ideograph>)
      6610   => 16#977B#,  --  靻 (<CJK Ideograph>)
      6611   => 16#977D#,  --  靽 (<CJK Ideograph>)
      6612   => 16#977F#,  --  靿 (<CJK Ideograph>)
      6613   => 16#9780#,  --  鞀 (<CJK Ideograph>)
      6614   => 16#9789#,  --  鞉 (<CJK Ideograph>)
      6615   => 16#9795#,  --  鞕 (<CJK Ideograph>)
      6616   => 16#9796#,  --  鞖 (<CJK Ideograph>)
      6617   => 16#9797#,  --  鞗 (<CJK Ideograph>)
      6618   => 16#9799#,  --  鞙 (<CJK Ideograph>)
      6619   => 16#979A#,  --  鞚 (<CJK Ideograph>)
      6620   => 16#979E#,  --  鞞 (<CJK Ideograph>)
      6621   => 16#979F#,  --  鞟 (<CJK Ideograph>)
      6622   => 16#97A2#,  --  鞢 (<CJK Ideograph>)
      6623   => 16#97AC#,  --  鞬 (<CJK Ideograph>)
      6624   => 16#97AE#,  --  鞮 (<CJK Ideograph>)
      6625   => 16#97B1#,  --  鞱 (<CJK Ideograph>)
      6626   => 16#97B2#,  --  鞲 (<CJK Ideograph>)
      6627   => 16#97B5#,  --  鞵 (<CJK Ideograph>)
      6628   => 16#97B6#,  --  鞶 (<CJK Ideograph>)
      6629   => 16#97B8#,  --  鞸 (<CJK Ideograph>)
      6630   => 16#97B9#,  --  鞹 (<CJK Ideograph>)
      6631   => 16#97BA#,  --  鞺 (<CJK Ideograph>)
      6632   => 16#97BC#,  --  鞼 (<CJK Ideograph>)
      6633   => 16#97BE#,  --  鞾 (<CJK Ideograph>)
      6634   => 16#97BF#,  --  鞿 (<CJK Ideograph>)
      6635   => 16#97C1#,  --  韁 (<CJK Ideograph>)
      6636   => 16#97C4#,  --  韄 (<CJK Ideograph>)
      6637   => 16#97C5#,  --  韅 (<CJK Ideograph>)
      6638   => 16#97C7#,  --  韇 (<CJK Ideograph>)
      6639   => 16#97C9#,  --  韉 (<CJK Ideograph>)
      6640   => 16#97CA#,  --  韊 (<CJK Ideograph>)
      6641   => 16#97CC#,  --  韌 (<CJK Ideograph>)
      6642   => 16#97CD#,  --  韍 (<CJK Ideograph>)
      6643   => 16#97CE#,  --  韎 (<CJK Ideograph>)
      6644   => 16#97D0#,  --  韐 (<CJK Ideograph>)
      6645   => 16#97D1#,  --  韑 (<CJK Ideograph>)
      6646   => 16#97D4#,  --  韔 (<CJK Ideograph>)
      6647   => 16#97D7#,  --  韗 (<CJK Ideograph>)
      6648   => 16#97D8#,  --  韘 (<CJK Ideograph>)
      6649   => 16#97D9#,  --  韙 (<CJK Ideograph>)
      6650   => 16#97DD#,  --  韝 (<CJK Ideograph>)
      6651   => 16#97DE#,  --  韞 (<CJK Ideograph>)
      6652   => 16#97E0#,  --  韠 (<CJK Ideograph>)
      6653   => 16#97DB#,  --  韛 (<CJK Ideograph>)
      6654   => 16#97E1#,  --  韡 (<CJK Ideograph>)
      6655   => 16#97E4#,  --  韤 (<CJK Ideograph>)
      6656   => 16#97EF#,  --  韯 (<CJK Ideograph>)
      6657   => 16#97F1#,  --  韱 (<CJK Ideograph>)
      6658   => 16#97F4#,  --  韴 (<CJK Ideograph>)
      6659   => 16#97F7#,  --  韷 (<CJK Ideograph>)
      6660   => 16#97F8#,  --  韸 (<CJK Ideograph>)
      6661   => 16#97FA#,  --  韺 (<CJK Ideograph>)
      6662   => 16#9807#,  --  頇 (<CJK Ideograph>)
      6663   => 16#980A#,  --  頊 (<CJK Ideograph>)
      6664   => 16#9819#,  --  頙 (<CJK Ideograph>)
      6665   => 16#980D#,  --  頍 (<CJK Ideograph>)
      6666   => 16#980E#,  --  頎 (<CJK Ideograph>)
      6667   => 16#9814#,  --  頔 (<CJK Ideograph>)
      6668   => 16#9816#,  --  頖 (<CJK Ideograph>)
      6669   => 16#981C#,  --  頜 (<CJK Ideograph>)
      6670   => 16#981E#,  --  頞 (<CJK Ideograph>)
      6671   => 16#9820#,  --  頠 (<CJK Ideograph>)
      6672   => 16#9823#,  --  頣 (<CJK Ideograph>)
      6673   => 16#9826#,  --  頦 (<CJK Ideograph>)
      6674   => 16#982B#,  --  頫 (<CJK Ideograph>)
      6675   => 16#982E#,  --  頮 (<CJK Ideograph>)
      6676   => 16#982F#,  --  頯 (<CJK Ideograph>)
      6677   => 16#9830#,  --  頰 (<CJK Ideograph>)
      6678   => 16#9832#,  --  頲 (<CJK Ideograph>)
      6679   => 16#9833#,  --  頳 (<CJK Ideograph>)
      6680   => 16#9835#,  --  頵 (<CJK Ideograph>)
      6681   => 16#9825#,  --  頥 (<CJK Ideograph>)
      6682   => 16#983E#,  --  頾 (<CJK Ideograph>)
      6683   => 16#9844#,  --  顄 (<CJK Ideograph>)
      6684   => 16#9847#,  --  顇 (<CJK Ideograph>)
      6685   => 16#984A#,  --  顊 (<CJK Ideograph>)
      6686   => 16#9851#,  --  顑 (<CJK Ideograph>)
      6687   => 16#9852#,  --  顒 (<CJK Ideograph>)
      6688   => 16#9853#,  --  顓 (<CJK Ideograph>)
      6689   => 16#9856#,  --  顖 (<CJK Ideograph>)
      6690   => 16#9857#,  --  顗 (<CJK Ideograph>)
      6691   => 16#9859#,  --  顙 (<CJK Ideograph>)
      6692   => 16#985A#,  --  顚 (<CJK Ideograph>)
      6693   => 16#9862#,  --  顢 (<CJK Ideograph>)
      6694   => 16#9863#,  --  顣 (<CJK Ideograph>)
      6695   => 16#9865#,  --  顥 (<CJK Ideograph>)
      6696   => 16#9866#,  --  顦 (<CJK Ideograph>)
      6697   => 16#986A#,  --  顪 (<CJK Ideograph>)
      6698   => 16#986C#,  --  顬 (<CJK Ideograph>)
      6699   => 16#98AB#,  --  颫 (<CJK Ideograph>)
      6700   => 16#98AD#,  --  颭 (<CJK Ideograph>)
      6701   => 16#98AE#,  --  颮 (<CJK Ideograph>)
      6702   => 16#98B0#,  --  颰 (<CJK Ideograph>)
      6703   => 16#98B4#,  --  颴 (<CJK Ideograph>)
      6704   => 16#98B7#,  --  颷 (<CJK Ideograph>)
      6705   => 16#98B8#,  --  颸 (<CJK Ideograph>)
      6706   => 16#98BA#,  --  颺 (<CJK Ideograph>)
      6707   => 16#98BB#,  --  颻 (<CJK Ideograph>)
      6708   => 16#98BF#,  --  颿 (<CJK Ideograph>)
      6709   => 16#98C2#,  --  飂 (<CJK Ideograph>)
      6710   => 16#98C5#,  --  飅 (<CJK Ideograph>)
      6711   => 16#98C8#,  --  飈 (<CJK Ideograph>)
      6712   => 16#98CC#,  --  飌 (<CJK Ideograph>)
      6713   => 16#98E1#,  --  飡 (<CJK Ideograph>)
      6714   => 16#98E3#,  --  飣 (<CJK Ideograph>)
      6715   => 16#98E5#,  --  飥 (<CJK Ideograph>)
      6716   => 16#98E6#,  --  飦 (<CJK Ideograph>)
      6717   => 16#98E7#,  --  飧 (<CJK Ideograph>)
      6718   => 16#98EA#,  --  飪 (<CJK Ideograph>)
      6719   => 16#98F3#,  --  飳 (<CJK Ideograph>)
      6720   => 16#98F6#,  --  飶 (<CJK Ideograph>)
      6721   => 16#9902#,  --  餂 (<CJK Ideograph>)
      6722   => 16#9907#,  --  餇 (<CJK Ideograph>)
      6723   => 16#9908#,  --  餈 (<CJK Ideograph>)
      6724   => 16#9911#,  --  餑 (<CJK Ideograph>)
      6725   => 16#9915#,  --  餕 (<CJK Ideograph>)
      6726   => 16#9916#,  --  餖 (<CJK Ideograph>)
      6727   => 16#9917#,  --  餗 (<CJK Ideograph>)
      6728   => 16#991A#,  --  餚 (<CJK Ideograph>)
      6729   => 16#991B#,  --  餛 (<CJK Ideograph>)
      6730   => 16#991C#,  --  餜 (<CJK Ideograph>)
      6731   => 16#991F#,  --  餟 (<CJK Ideograph>)
      6732   => 16#9922#,  --  餢 (<CJK Ideograph>)
      6733   => 16#9926#,  --  餦 (<CJK Ideograph>)
      6734   => 16#9927#,  --  餧 (<CJK Ideograph>)
      6735   => 16#992B#,  --  餫 (<CJK Ideograph>)
      6736   => 16#9931#,  --  餱 (<CJK Ideograph>)
      6737   => 16#9932#,  --  餲 (<CJK Ideograph>)
      6738   => 16#9933#,  --  餳 (<CJK Ideograph>)
      6739   => 16#9934#,  --  餴 (<CJK Ideograph>)
      6740   => 16#9935#,  --  餵 (<CJK Ideograph>)
      6741   => 16#9939#,  --  餹 (<CJK Ideograph>)
      6742   => 16#993A#,  --  餺 (<CJK Ideograph>)
      6743   => 16#993B#,  --  餻 (<CJK Ideograph>)
      6744   => 16#993C#,  --  餼 (<CJK Ideograph>)
      6745   => 16#9940#,  --  饀 (<CJK Ideograph>)
      6746   => 16#9941#,  --  饁 (<CJK Ideograph>)
      6747   => 16#9946#,  --  饆 (<CJK Ideograph>)
      6748   => 16#9947#,  --  饇 (<CJK Ideograph>)
      6749   => 16#9948#,  --  饈 (<CJK Ideograph>)
      6750   => 16#994D#,  --  饍 (<CJK Ideograph>)
      6751   => 16#994E#,  --  饎 (<CJK Ideograph>)
      6752   => 16#9954#,  --  饔 (<CJK Ideograph>)
      6753   => 16#9958#,  --  饘 (<CJK Ideograph>)
      6754   => 16#9959#,  --  饙 (<CJK Ideograph>)
      6755   => 16#995B#,  --  饛 (<CJK Ideograph>)
      6756   => 16#995C#,  --  饜 (<CJK Ideograph>)
      6757   => 16#995E#,  --  饞 (<CJK Ideograph>)
      6758   => 16#995F#,  --  饟 (<CJK Ideograph>)
      6759   => 16#9960#,  --  饠 (<CJK Ideograph>)
      6760   => 16#999B#,  --  馛 (<CJK Ideograph>)
      6761   => 16#999D#,  --  馝 (<CJK Ideograph>)
      6762   => 16#999F#,  --  馟 (<CJK Ideograph>)
      6763   => 16#99A6#,  --  馦 (<CJK Ideograph>)
      6764   => 16#99B0#,  --  馰 (<CJK Ideograph>)
      6765   => 16#99B1#,  --  馱 (<CJK Ideograph>)
      6766   => 16#99B2#,  --  馲 (<CJK Ideograph>)
      6767   => 16#99B5#,  --  馵 (<CJK Ideograph>)
      6768   => 16#99B9#,  --  馹 (<CJK Ideograph>)
      6769   => 16#99BA#,  --  馺 (<CJK Ideograph>)
      6770   => 16#99BD#,  --  馽 (<CJK Ideograph>)
      6771   => 16#99BF#,  --  馿 (<CJK Ideograph>)
      6772   => 16#99C3#,  --  駃 (<CJK Ideograph>)
      6773   => 16#99C9#,  --  駉 (<CJK Ideograph>)
      6774   => 16#99D3#,  --  駓 (<CJK Ideograph>)
      6775   => 16#99D4#,  --  駔 (<CJK Ideograph>)
      6776   => 16#99D9#,  --  駙 (<CJK Ideograph>)
      6777   => 16#99DA#,  --  駚 (<CJK Ideograph>)
      6778   => 16#99DC#,  --  駜 (<CJK Ideograph>)
      6779   => 16#99DE#,  --  駞 (<CJK Ideograph>)
      6780   => 16#99E7#,  --  駧 (<CJK Ideograph>)
      6781   => 16#99EA#,  --  駪 (<CJK Ideograph>)
      6782   => 16#99EB#,  --  駫 (<CJK Ideograph>)
      6783   => 16#99EC#,  --  駬 (<CJK Ideograph>)
      6784   => 16#99F0#,  --  駰 (<CJK Ideograph>)
      6785   => 16#99F4#,  --  駴 (<CJK Ideograph>)
      6786   => 16#99F5#,  --  駵 (<CJK Ideograph>)
      6787   => 16#99F9#,  --  駹 (<CJK Ideograph>)
      6788   => 16#99FD#,  --  駽 (<CJK Ideograph>)
      6789   => 16#99FE#,  --  駾 (<CJK Ideograph>)
      6790   => 16#9A02#,  --  騂 (<CJK Ideograph>)
      6791   => 16#9A03#,  --  騃 (<CJK Ideograph>)
      6792   => 16#9A04#,  --  騄 (<CJK Ideograph>)
      6793   => 16#9A0B#,  --  騋 (<CJK Ideograph>)
      6794   => 16#9A0C#,  --  騌 (<CJK Ideograph>)
      6795   => 16#9A10#,  --  騐 (<CJK Ideograph>)
      6796   => 16#9A11#,  --  騑 (<CJK Ideograph>)
      6797   => 16#9A16#,  --  騖 (<CJK Ideograph>)
      6798   => 16#9A1E#,  --  騞 (<CJK Ideograph>)
      6799   => 16#9A20#,  --  騠 (<CJK Ideograph>)
      6800   => 16#9A22#,  --  騢 (<CJK Ideograph>)
      6801   => 16#9A23#,  --  騣 (<CJK Ideograph>)
      6802   => 16#9A24#,  --  騤 (<CJK Ideograph>)
      6803   => 16#9A27#,  --  騧 (<CJK Ideograph>)
      6804   => 16#9A2D#,  --  騭 (<CJK Ideograph>)
      6805   => 16#9A2E#,  --  騮 (<CJK Ideograph>)
      6806   => 16#9A33#,  --  騳 (<CJK Ideograph>)
      6807   => 16#9A35#,  --  騵 (<CJK Ideograph>)
      6808   => 16#9A36#,  --  騶 (<CJK Ideograph>)
      6809   => 16#9A38#,  --  騸 (<CJK Ideograph>)
      6810   => 16#9A47#,  --  驇 (<CJK Ideograph>)
      6811   => 16#9A41#,  --  驁 (<CJK Ideograph>)
      6812   => 16#9A44#,  --  驄 (<CJK Ideograph>)
      6813   => 16#9A4A#,  --  驊 (<CJK Ideograph>)
      6814   => 16#9A4B#,  --  驋 (<CJK Ideograph>)
      6815   => 16#9A4C#,  --  驌 (<CJK Ideograph>)
      6816   => 16#9A4E#,  --  驎 (<CJK Ideograph>)
      6817   => 16#9A51#,  --  驑 (<CJK Ideograph>)
      6818   => 16#9A54#,  --  驔 (<CJK Ideograph>)
      6819   => 16#9A56#,  --  驖 (<CJK Ideograph>)
      6820   => 16#9A5D#,  --  驝 (<CJK Ideograph>)
      6821   => 16#9AAA#,  --  骪 (<CJK Ideograph>)
      6822   => 16#9AAC#,  --  骬 (<CJK Ideograph>)
      6823   => 16#9AAE#,  --  骮 (<CJK Ideograph>)
      6824   => 16#9AAF#,  --  骯 (<CJK Ideograph>)
      6825   => 16#9AB2#,  --  骲 (<CJK Ideograph>)
      6826   => 16#9AB4#,  --  骴 (<CJK Ideograph>)
      6827   => 16#9AB5#,  --  骵 (<CJK Ideograph>)
      6828   => 16#9AB6#,  --  骶 (<CJK Ideograph>)
      6829   => 16#9AB9#,  --  骹 (<CJK Ideograph>)
      6830   => 16#9ABB#,  --  骻 (<CJK Ideograph>)
      6831   => 16#9ABE#,  --  骾 (<CJK Ideograph>)
      6832   => 16#9ABF#,  --  骿 (<CJK Ideograph>)
      6833   => 16#9AC1#,  --  髁 (<CJK Ideograph>)
      6834   => 16#9AC3#,  --  髃 (<CJK Ideograph>)
      6835   => 16#9AC6#,  --  髆 (<CJK Ideograph>)
      6836   => 16#9AC8#,  --  髈 (<CJK Ideograph>)
      6837   => 16#9ACE#,  --  髎 (<CJK Ideograph>)
      6838   => 16#9AD0#,  --  髐 (<CJK Ideograph>)
      6839   => 16#9AD2#,  --  髒 (<CJK Ideograph>)
      6840   => 16#9AD5#,  --  髕 (<CJK Ideograph>)
      6841   => 16#9AD6#,  --  髖 (<CJK Ideograph>)
      6842   => 16#9AD7#,  --  髗 (<CJK Ideograph>)
      6843   => 16#9ADB#,  --  髛 (<CJK Ideograph>)
      6844   => 16#9ADC#,  --  髜 (<CJK Ideograph>)
      6845   => 16#9AE0#,  --  髠 (<CJK Ideograph>)
      6846   => 16#9AE4#,  --  髤 (<CJK Ideograph>)
      6847   => 16#9AE5#,  --  髥 (<CJK Ideograph>)
      6848   => 16#9AE7#,  --  髧 (<CJK Ideograph>)
      6849   => 16#9AE9#,  --  髩 (<CJK Ideograph>)
      6850   => 16#9AEC#,  --  髬 (<CJK Ideograph>)
      6851   => 16#9AF2#,  --  髲 (<CJK Ideograph>)
      6852   => 16#9AF3#,  --  髳 (<CJK Ideograph>)
      6853   => 16#9AF5#,  --  髵 (<CJK Ideograph>)
      6854   => 16#9AF9#,  --  髹 (<CJK Ideograph>)
      6855   => 16#9AFA#,  --  髺 (<CJK Ideograph>)
      6856   => 16#9AFD#,  --  髽 (<CJK Ideograph>)
      6857   => 16#9AFF#,  --  髿 (<CJK Ideograph>)
      6858   => 16#9B00#,  --  鬀 (<CJK Ideograph>)
      6859   => 16#9B01#,  --  鬁 (<CJK Ideograph>)
      6860   => 16#9B02#,  --  鬂 (<CJK Ideograph>)
      6861   => 16#9B03#,  --  鬃 (<CJK Ideograph>)
      6862   => 16#9B04#,  --  鬄 (<CJK Ideograph>)
      6863   => 16#9B05#,  --  鬅 (<CJK Ideograph>)
      6864   => 16#9B08#,  --  鬈 (<CJK Ideograph>)
      6865   => 16#9B09#,  --  鬉 (<CJK Ideograph>)
      6866   => 16#9B0B#,  --  鬋 (<CJK Ideograph>)
      6867   => 16#9B0C#,  --  鬌 (<CJK Ideograph>)
      6868   => 16#9B0D#,  --  鬍 (<CJK Ideograph>)
      6869   => 16#9B0E#,  --  鬎 (<CJK Ideograph>)
      6870   => 16#9B10#,  --  鬐 (<CJK Ideograph>)
      6871   => 16#9B12#,  --  鬒 (<CJK Ideograph>)
      6872   => 16#9B16#,  --  鬖 (<CJK Ideograph>)
      6873   => 16#9B19#,  --  鬙 (<CJK Ideograph>)
      6874   => 16#9B1B#,  --  鬛 (<CJK Ideograph>)
      6875   => 16#9B1C#,  --  鬜 (<CJK Ideograph>)
      6876   => 16#9B20#,  --  鬠 (<CJK Ideograph>)
      6877   => 16#9B26#,  --  鬦 (<CJK Ideograph>)
      6878   => 16#9B2B#,  --  鬫 (<CJK Ideograph>)
      6879   => 16#9B2D#,  --  鬭 (<CJK Ideograph>)
      6880   => 16#9B33#,  --  鬳 (<CJK Ideograph>)
      6881   => 16#9B34#,  --  鬴 (<CJK Ideograph>)
      6882   => 16#9B35#,  --  鬵 (<CJK Ideograph>)
      6883   => 16#9B37#,  --  鬷 (<CJK Ideograph>)
      6884   => 16#9B39#,  --  鬹 (<CJK Ideograph>)
      6885   => 16#9B3A#,  --  鬺 (<CJK Ideograph>)
      6886   => 16#9B3D#,  --  鬽 (<CJK Ideograph>)
      6887   => 16#9B48#,  --  魈 (<CJK Ideograph>)
      6888   => 16#9B4B#,  --  魋 (<CJK Ideograph>)
      6889   => 16#9B4C#,  --  魌 (<CJK Ideograph>)
      6890   => 16#9B55#,  --  魕 (<CJK Ideograph>)
      6891   => 16#9B56#,  --  魖 (<CJK Ideograph>)
      6892   => 16#9B57#,  --  魗 (<CJK Ideograph>)
      6893   => 16#9B5B#,  --  魛 (<CJK Ideograph>)
      6894   => 16#9B5E#,  --  魞 (<CJK Ideograph>)
      6895   => 16#9B61#,  --  魡 (<CJK Ideograph>)
      6896   => 16#9B63#,  --  魣 (<CJK Ideograph>)
      6897   => 16#9B65#,  --  魥 (<CJK Ideograph>)
      6898   => 16#9B66#,  --  魦 (<CJK Ideograph>)
      6899   => 16#9B68#,  --  魨 (<CJK Ideograph>)
      6900   => 16#9B6A#,  --  魪 (<CJK Ideograph>)
      6901   => 16#9B6B#,  --  魫 (<CJK Ideograph>)
      6902   => 16#9B6C#,  --  魬 (<CJK Ideograph>)
      6903   => 16#9B6D#,  --  魭 (<CJK Ideograph>)
      6904   => 16#9B6E#,  --  魮 (<CJK Ideograph>)
      6905   => 16#9B73#,  --  魳 (<CJK Ideograph>)
      6906   => 16#9B75#,  --  魵 (<CJK Ideograph>)
      6907   => 16#9B77#,  --  魷 (<CJK Ideograph>)
      6908   => 16#9B78#,  --  魸 (<CJK Ideograph>)
      6909   => 16#9B79#,  --  魹 (<CJK Ideograph>)
      6910   => 16#9B7F#,  --  魿 (<CJK Ideograph>)
      6911   => 16#9B80#,  --  鮀 (<CJK Ideograph>)
      6912   => 16#9B84#,  --  鮄 (<CJK Ideograph>)
      6913   => 16#9B85#,  --  鮅 (<CJK Ideograph>)
      6914   => 16#9B86#,  --  鮆 (<CJK Ideograph>)
      6915   => 16#9B87#,  --  鮇 (<CJK Ideograph>)
      6916   => 16#9B89#,  --  鮉 (<CJK Ideograph>)
      6917   => 16#9B8A#,  --  鮊 (<CJK Ideograph>)
      6918   => 16#9B8B#,  --  鮋 (<CJK Ideograph>)
      6919   => 16#9B8D#,  --  鮍 (<CJK Ideograph>)
      6920   => 16#9B8F#,  --  鮏 (<CJK Ideograph>)
      6921   => 16#9B90#,  --  鮐 (<CJK Ideograph>)
      6922   => 16#9B94#,  --  鮔 (<CJK Ideograph>)
      6923   => 16#9B9A#,  --  鮚 (<CJK Ideograph>)
      6924   => 16#9B9D#,  --  鮝 (<CJK Ideograph>)
      6925   => 16#9B9E#,  --  鮞 (<CJK Ideograph>)
      6926   => 16#9BA6#,  --  鮦 (<CJK Ideograph>)
      6927   => 16#9BA7#,  --  鮧 (<CJK Ideograph>)
      6928   => 16#9BA9#,  --  鮩 (<CJK Ideograph>)
      6929   => 16#9BAC#,  --  鮬 (<CJK Ideograph>)
      6930   => 16#9BB0#,  --  鮰 (<CJK Ideograph>)
      6931   => 16#9BB1#,  --  鮱 (<CJK Ideograph>)
      6932   => 16#9BB2#,  --  鮲 (<CJK Ideograph>)
      6933   => 16#9BB7#,  --  鮷 (<CJK Ideograph>)
      6934   => 16#9BB8#,  --  鮸 (<CJK Ideograph>)
      6935   => 16#9BBB#,  --  鮻 (<CJK Ideograph>)
      6936   => 16#9BBC#,  --  鮼 (<CJK Ideograph>)
      6937   => 16#9BBE#,  --  鮾 (<CJK Ideograph>)
      6938   => 16#9BBF#,  --  鮿 (<CJK Ideograph>)
      6939   => 16#9BC1#,  --  鯁 (<CJK Ideograph>)
      6940   => 16#9BC7#,  --  鯇 (<CJK Ideograph>)
      6941   => 16#9BC8#,  --  鯈 (<CJK Ideograph>)
      6942   => 16#9BCE#,  --  鯎 (<CJK Ideograph>)
      6943   => 16#9BD0#,  --  鯐 (<CJK Ideograph>)
      6944   => 16#9BD7#,  --  鯗 (<CJK Ideograph>)
      6945   => 16#9BD8#,  --  鯘 (<CJK Ideograph>)
      6946   => 16#9BDD#,  --  鯝 (<CJK Ideograph>)
      6947   => 16#9BDF#,  --  鯟 (<CJK Ideograph>)
      6948   => 16#9BE5#,  --  鯥 (<CJK Ideograph>)
      6949   => 16#9BE7#,  --  鯧 (<CJK Ideograph>)
      6950   => 16#9BEA#,  --  鯪 (<CJK Ideograph>)
      6951   => 16#9BEB#,  --  鯫 (<CJK Ideograph>)
      6952   => 16#9BEF#,  --  鯯 (<CJK Ideograph>)
      6953   => 16#9BF3#,  --  鯳 (<CJK Ideograph>)
      6954   => 16#9BF7#,  --  鯷 (<CJK Ideograph>)
      6955   => 16#9BF8#,  --  鯸 (<CJK Ideograph>)
      6956   => 16#9BF9#,  --  鯹 (<CJK Ideograph>)
      6957   => 16#9BFA#,  --  鯺 (<CJK Ideograph>)
      6958   => 16#9BFD#,  --  鯽 (<CJK Ideograph>)
      6959   => 16#9BFF#,  --  鯿 (<CJK Ideograph>)
      6960   => 16#9C00#,  --  鰀 (<CJK Ideograph>)
      6961   => 16#9C02#,  --  鰂 (<CJK Ideograph>)
      6962   => 16#9C0B#,  --  鰋 (<CJK Ideograph>)
      6963   => 16#9C0F#,  --  鰏 (<CJK Ideograph>)
      6964   => 16#9C11#,  --  鰑 (<CJK Ideograph>)
      6965   => 16#9C16#,  --  鰖 (<CJK Ideograph>)
      6966   => 16#9C18#,  --  鰘 (<CJK Ideograph>)
      6967   => 16#9C19#,  --  鰙 (<CJK Ideograph>)
      6968   => 16#9C1A#,  --  鰚 (<CJK Ideograph>)
      6969   => 16#9C1C#,  --  鰜 (<CJK Ideograph>)
      6970   => 16#9C1E#,  --  鰞 (<CJK Ideograph>)
      6971   => 16#9C22#,  --  鰢 (<CJK Ideograph>)
      6972   => 16#9C23#,  --  鰣 (<CJK Ideograph>)
      6973   => 16#9C26#,  --  鰦 (<CJK Ideograph>)
      6974   => 16#9C27#,  --  鰧 (<CJK Ideograph>)
      6975   => 16#9C28#,  --  鰨 (<CJK Ideograph>)
      6976   => 16#9C29#,  --  鰩 (<CJK Ideograph>)
      6977   => 16#9C2A#,  --  鰪 (<CJK Ideograph>)
      6978   => 16#9C31#,  --  鰱 (<CJK Ideograph>)
      6979   => 16#9C35#,  --  鰵 (<CJK Ideograph>)
      6980   => 16#9C36#,  --  鰶 (<CJK Ideograph>)
      6981   => 16#9C37#,  --  鰷 (<CJK Ideograph>)
      6982   => 16#9C3D#,  --  鰽 (<CJK Ideograph>)
      6983   => 16#9C41#,  --  鱁 (<CJK Ideograph>)
      6984   => 16#9C43#,  --  鱃 (<CJK Ideograph>)
      6985   => 16#9C44#,  --  鱄 (<CJK Ideograph>)
      6986   => 16#9C45#,  --  鱅 (<CJK Ideograph>)
      6987   => 16#9C49#,  --  鱉 (<CJK Ideograph>)
      6988   => 16#9C4A#,  --  鱊 (<CJK Ideograph>)
      6989   => 16#9C4E#,  --  鱎 (<CJK Ideograph>)
      6990   => 16#9C4F#,  --  鱏 (<CJK Ideograph>)
      6991   => 16#9C50#,  --  鱐 (<CJK Ideograph>)
      6992   => 16#9C53#,  --  鱓 (<CJK Ideograph>)
      6993   => 16#9C54#,  --  鱔 (<CJK Ideograph>)
      6994   => 16#9C56#,  --  鱖 (<CJK Ideograph>)
      6995   => 16#9C58#,  --  鱘 (<CJK Ideograph>)
      6996   => 16#9C5B#,  --  鱛 (<CJK Ideograph>)
      6997   => 16#9C5D#,  --  鱝 (<CJK Ideograph>)
      6998   => 16#9C5E#,  --  鱞 (<CJK Ideograph>)
      6999   => 16#9C5F#,  --  鱟 (<CJK Ideograph>)
      7000   => 16#9C63#,  --  鱣 (<CJK Ideograph>)
      7001   => 16#9C69#,  --  鱩 (<CJK Ideograph>)
      7002   => 16#9C6A#,  --  鱪 (<CJK Ideograph>)
      7003   => 16#9C5C#,  --  鱜 (<CJK Ideograph>)
      7004   => 16#9C6B#,  --  鱫 (<CJK Ideograph>)
      7005   => 16#9C68#,  --  鱨 (<CJK Ideograph>)
      7006   => 16#9C6E#,  --  鱮 (<CJK Ideograph>)
      7007   => 16#9C70#,  --  鱰 (<CJK Ideograph>)
      7008   => 16#9C72#,  --  鱲 (<CJK Ideograph>)
      7009   => 16#9C75#,  --  鱵 (<CJK Ideograph>)
      7010   => 16#9C77#,  --  鱷 (<CJK Ideograph>)
      7011   => 16#9C7B#,  --  鱻 (<CJK Ideograph>)
      7012   => 16#9CE6#,  --  鳦 (<CJK Ideograph>)
      7013   => 16#9CF2#,  --  鳲 (<CJK Ideograph>)
      7014   => 16#9CF7#,  --  鳷 (<CJK Ideograph>)
      7015   => 16#9CF9#,  --  鳹 (<CJK Ideograph>)
      7016   => 16#9D0B#,  --  鴋 (<CJK Ideograph>)
      7017   => 16#9D02#,  --  鴂 (<CJK Ideograph>)
      7018   => 16#9D11#,  --  鴑 (<CJK Ideograph>)
      7019   => 16#9D17#,  --  鴗 (<CJK Ideograph>)
      7020   => 16#9D18#,  --  鴘 (<CJK Ideograph>)
      7021   => 16#9D1C#,  --  鴜 (<CJK Ideograph>)
      7022   => 16#9D1D#,  --  鴝 (<CJK Ideograph>)
      7023   => 16#9D1E#,  --  鴞 (<CJK Ideograph>)
      7024   => 16#9D2F#,  --  鴯 (<CJK Ideograph>)
      7025   => 16#9D30#,  --  鴰 (<CJK Ideograph>)
      7026   => 16#9D32#,  --  鴲 (<CJK Ideograph>)
      7027   => 16#9D33#,  --  鴳 (<CJK Ideograph>)
      7028   => 16#9D34#,  --  鴴 (<CJK Ideograph>)
      7029   => 16#9D3A#,  --  鴺 (<CJK Ideograph>)
      7030   => 16#9D3C#,  --  鴼 (<CJK Ideograph>)
      7031   => 16#9D45#,  --  鵅 (<CJK Ideograph>)
      7032   => 16#9D3D#,  --  鴽 (<CJK Ideograph>)
      7033   => 16#9D42#,  --  鵂 (<CJK Ideograph>)
      7034   => 16#9D43#,  --  鵃 (<CJK Ideograph>)
      7035   => 16#9D47#,  --  鵇 (<CJK Ideograph>)
      7036   => 16#9D4A#,  --  鵊 (<CJK Ideograph>)
      7037   => 16#9D53#,  --  鵓 (<CJK Ideograph>)
      7038   => 16#9D54#,  --  鵔 (<CJK Ideograph>)
      7039   => 16#9D5F#,  --  鵟 (<CJK Ideograph>)
      7040   => 16#9D63#,  --  鵣 (<CJK Ideograph>)
      7041   => 16#9D62#,  --  鵢 (<CJK Ideograph>)
      7042   => 16#9D65#,  --  鵥 (<CJK Ideograph>)
      7043   => 16#9D69#,  --  鵩 (<CJK Ideograph>)
      7044   => 16#9D6A#,  --  鵪 (<CJK Ideograph>)
      7045   => 16#9D6B#,  --  鵫 (<CJK Ideograph>)
      7046   => 16#9D70#,  --  鵰 (<CJK Ideograph>)
      7047   => 16#9D76#,  --  鵶 (<CJK Ideograph>)
      7048   => 16#9D77#,  --  鵷 (<CJK Ideograph>)
      7049   => 16#9D7B#,  --  鵻 (<CJK Ideograph>)
      7050   => 16#9D7C#,  --  鵼 (<CJK Ideograph>)
      7051   => 16#9D7E#,  --  鵾 (<CJK Ideograph>)
      7052   => 16#9D83#,  --  鶃 (<CJK Ideograph>)
      7053   => 16#9D84#,  --  鶄 (<CJK Ideograph>)
      7054   => 16#9D86#,  --  鶆 (<CJK Ideograph>)
      7055   => 16#9D8A#,  --  鶊 (<CJK Ideograph>)
      7056   => 16#9D8D#,  --  鶍 (<CJK Ideograph>)
      7057   => 16#9D8E#,  --  鶎 (<CJK Ideograph>)
      7058   => 16#9D92#,  --  鶒 (<CJK Ideograph>)
      7059   => 16#9D93#,  --  鶓 (<CJK Ideograph>)
      7060   => 16#9D95#,  --  鶕 (<CJK Ideograph>)
      7061   => 16#9D96#,  --  鶖 (<CJK Ideograph>)
      7062   => 16#9D97#,  --  鶗 (<CJK Ideograph>)
      7063   => 16#9D98#,  --  鶘 (<CJK Ideograph>)
      7064   => 16#9DA1#,  --  鶡 (<CJK Ideograph>)
      7065   => 16#9DAA#,  --  鶪 (<CJK Ideograph>)
      7066   => 16#9DAC#,  --  鶬 (<CJK Ideograph>)
      7067   => 16#9DAE#,  --  鶮 (<CJK Ideograph>)
      7068   => 16#9DB1#,  --  鶱 (<CJK Ideograph>)
      7069   => 16#9DB5#,  --  鶵 (<CJK Ideograph>)
      7070   => 16#9DB9#,  --  鶹 (<CJK Ideograph>)
      7071   => 16#9DBC#,  --  鶼 (<CJK Ideograph>)
      7072   => 16#9DBF#,  --  鶿 (<CJK Ideograph>)
      7073   => 16#9DC3#,  --  鷃 (<CJK Ideograph>)
      7074   => 16#9DC7#,  --  鷇 (<CJK Ideograph>)
      7075   => 16#9DC9#,  --  鷉 (<CJK Ideograph>)
      7076   => 16#9DCA#,  --  鷊 (<CJK Ideograph>)
      7077   => 16#9DD4#,  --  鷔 (<CJK Ideograph>)
      7078   => 16#9DD5#,  --  鷕 (<CJK Ideograph>)
      7079   => 16#9DD6#,  --  鷖 (<CJK Ideograph>)
      7080   => 16#9DD7#,  --  鷗 (<CJK Ideograph>)
      7081   => 16#9DDA#,  --  鷚 (<CJK Ideograph>)
      7082   => 16#9DDE#,  --  鷞 (<CJK Ideograph>)
      7083   => 16#9DDF#,  --  鷟 (<CJK Ideograph>)
      7084   => 16#9DE0#,  --  鷠 (<CJK Ideograph>)
      7085   => 16#9DE5#,  --  鷥 (<CJK Ideograph>)
      7086   => 16#9DE7#,  --  鷧 (<CJK Ideograph>)
      7087   => 16#9DE9#,  --  鷩 (<CJK Ideograph>)
      7088   => 16#9DEB#,  --  鷫 (<CJK Ideograph>)
      7089   => 16#9DEE#,  --  鷮 (<CJK Ideograph>)
      7090   => 16#9DF0#,  --  鷰 (<CJK Ideograph>)
      7091   => 16#9DF3#,  --  鷳 (<CJK Ideograph>)
      7092   => 16#9DF4#,  --  鷴 (<CJK Ideograph>)
      7093   => 16#9DFE#,  --  鷾 (<CJK Ideograph>)
      7094   => 16#9E0A#,  --  鸊 (<CJK Ideograph>)
      7095   => 16#9E02#,  --  鸂 (<CJK Ideograph>)
      7096   => 16#9E07#,  --  鸇 (<CJK Ideograph>)
      7097   => 16#9E0E#,  --  鸎 (<CJK Ideograph>)
      7098   => 16#9E10#,  --  鸐 (<CJK Ideograph>)
      7099   => 16#9E11#,  --  鸑 (<CJK Ideograph>)
      7100   => 16#9E12#,  --  鸒 (<CJK Ideograph>)
      7101   => 16#9E15#,  --  鸕 (<CJK Ideograph>)
      7102   => 16#9E16#,  --  鸖 (<CJK Ideograph>)
      7103   => 16#9E19#,  --  鸙 (<CJK Ideograph>)
      7104   => 16#9E1C#,  --  鸜 (<CJK Ideograph>)
      7105   => 16#9E1D#,  --  鸝 (<CJK Ideograph>)
      7106   => 16#9E7A#,  --  鹺 (<CJK Ideograph>)
      7107   => 16#9E7B#,  --  鹻 (<CJK Ideograph>)
      7108   => 16#9E7C#,  --  鹼 (<CJK Ideograph>)
      7109   => 16#9E80#,  --  麀 (<CJK Ideograph>)
      7110   => 16#9E82#,  --  麂 (<CJK Ideograph>)
      7111   => 16#9E83#,  --  麃 (<CJK Ideograph>)
      7112   => 16#9E84#,  --  麄 (<CJK Ideograph>)
      7113   => 16#9E85#,  --  麅 (<CJK Ideograph>)
      7114   => 16#9E87#,  --  麇 (<CJK Ideograph>)
      7115   => 16#9E8E#,  --  麎 (<CJK Ideograph>)
      7116   => 16#9E8F#,  --  麏 (<CJK Ideograph>)
      7117   => 16#9E96#,  --  麖 (<CJK Ideograph>)
      7118   => 16#9E98#,  --  麘 (<CJK Ideograph>)
      7119   => 16#9E9B#,  --  麛 (<CJK Ideograph>)
      7120   => 16#9E9E#,  --  麞 (<CJK Ideograph>)
      7121   => 16#9EA4#,  --  麤 (<CJK Ideograph>)
      7122   => 16#9EA8#,  --  麨 (<CJK Ideograph>)
      7123   => 16#9EAC#,  --  麬 (<CJK Ideograph>)
      7124   => 16#9EAE#,  --  麮 (<CJK Ideograph>)
      7125   => 16#9EAF#,  --  麯 (<CJK Ideograph>)
      7126   => 16#9EB0#,  --  麰 (<CJK Ideograph>)
      7127   => 16#9EB3#,  --  麳 (<CJK Ideograph>)
      7128   => 16#9EB4#,  --  麴 (<CJK Ideograph>)
      7129   => 16#9EB5#,  --  麵 (<CJK Ideograph>)
      7130   => 16#9EC6#,  --  黆 (<CJK Ideograph>)
      7131   => 16#9EC8#,  --  黈 (<CJK Ideograph>)
      7132   => 16#9ECB#,  --  黋 (<CJK Ideograph>)
      7133   => 16#9ED5#,  --  黕 (<CJK Ideograph>)
      7134   => 16#9EDF#,  --  黟 (<CJK Ideograph>)
      7135   => 16#9EE4#,  --  黤 (<CJK Ideograph>)
      7136   => 16#9EE7#,  --  黧 (<CJK Ideograph>)
      7137   => 16#9EEC#,  --  黬 (<CJK Ideograph>)
      7138   => 16#9EED#,  --  黭 (<CJK Ideograph>)
      7139   => 16#9EEE#,  --  黮 (<CJK Ideograph>)
      7140   => 16#9EF0#,  --  黰 (<CJK Ideograph>)
      7141   => 16#9EF1#,  --  黱 (<CJK Ideograph>)
      7142   => 16#9EF2#,  --  黲 (<CJK Ideograph>)
      7143   => 16#9EF5#,  --  黵 (<CJK Ideograph>)
      7144   => 16#9EF8#,  --  黸 (<CJK Ideograph>)
      7145   => 16#9EFF#,  --  黿 (<CJK Ideograph>)
      7146   => 16#9F02#,  --  鼂 (<CJK Ideograph>)
      7147   => 16#9F03#,  --  鼃 (<CJK Ideograph>)
      7148   => 16#9F09#,  --  鼉 (<CJK Ideograph>)
      7149   => 16#9F0F#,  --  鼏 (<CJK Ideograph>)
      7150   => 16#9F10#,  --  鼐 (<CJK Ideograph>)
      7151   => 16#9F11#,  --  鼑 (<CJK Ideograph>)
      7152   => 16#9F12#,  --  鼒 (<CJK Ideograph>)
      7153   => 16#9F14#,  --  鼔 (<CJK Ideograph>)
      7154   => 16#9F16#,  --  鼖 (<CJK Ideograph>)
      7155   => 16#9F17#,  --  鼗 (<CJK Ideograph>)
      7156   => 16#9F19#,  --  鼙 (<CJK Ideograph>)
      7157   => 16#9F1A#,  --  鼚 (<CJK Ideograph>)
      7158   => 16#9F1B#,  --  鼛 (<CJK Ideograph>)
      7159   => 16#9F1F#,  --  鼟 (<CJK Ideograph>)
      7160   => 16#9F22#,  --  鼢 (<CJK Ideograph>)
      7161   => 16#9F26#,  --  鼦 (<CJK Ideograph>)
      7162   => 16#9F2A#,  --  鼪 (<CJK Ideograph>)
      7163   => 16#9F2B#,  --  鼫 (<CJK Ideograph>)
      7164   => 16#9F2F#,  --  鼯 (<CJK Ideograph>)
      7165   => 16#9F31#,  --  鼱 (<CJK Ideograph>)
      7166   => 16#9F32#,  --  鼲 (<CJK Ideograph>)
      7167   => 16#9F34#,  --  鼴 (<CJK Ideograph>)
      7168   => 16#9F37#,  --  鼷 (<CJK Ideograph>)
      7169   => 16#9F39#,  --  鼹 (<CJK Ideograph>)
      7170   => 16#9F3A#,  --  鼺 (<CJK Ideograph>)
      7171   => 16#9F3C#,  --  鼼 (<CJK Ideograph>)
      7172   => 16#9F3D#,  --  鼽 (<CJK Ideograph>)
      7173   => 16#9F3F#,  --  鼿 (<CJK Ideograph>)
      7174   => 16#9F41#,  --  齁 (<CJK Ideograph>)
      7175   => 16#9F43#,  --  齃 (<CJK Ideograph>)
      7176   => 16#9F44#,  --  齄 (<CJK Ideograph>)
      7177   => 16#9F45#,  --  齅 (<CJK Ideograph>)
      7178   => 16#9F46#,  --  齆 (<CJK Ideograph>)
      7179   => 16#9F47#,  --  齇 (<CJK Ideograph>)
      7180   => 16#9F53#,  --  齓 (<CJK Ideograph>)
      7181   => 16#9F55#,  --  齕 (<CJK Ideograph>)
      7182   => 16#9F56#,  --  齖 (<CJK Ideograph>)
      7183   => 16#9F57#,  --  齗 (<CJK Ideograph>)
      7184   => 16#9F58#,  --  齘 (<CJK Ideograph>)
      7185   => 16#9F5A#,  --  齚 (<CJK Ideograph>)
      7186   => 16#9F5D#,  --  齝 (<CJK Ideograph>)
      7187   => 16#9F5E#,  --  齞 (<CJK Ideograph>)
      7188   => 16#9F68#,  --  齨 (<CJK Ideograph>)
      7189   => 16#9F69#,  --  齩 (<CJK Ideograph>)
      7190   => 16#9F6D#,  --  齭 (<CJK Ideograph>)
      7191   => 16#9F6E#,  --  齮 (<CJK Ideograph>)
      7192   => 16#9F6F#,  --  齯 (<CJK Ideograph>)
      7193   => 16#9F70#,  --  齰 (<CJK Ideograph>)
      7194   => 16#9F71#,  --  齱 (<CJK Ideograph>)
      7195   => 16#9F73#,  --  齳 (<CJK Ideograph>)
      7196   => 16#9F75#,  --  齵 (<CJK Ideograph>)
      7197   => 16#9F7A#,  --  齺 (<CJK Ideograph>)
      7198   => 16#9F7D#,  --  齽 (<CJK Ideograph>)
      7199   => 16#9F8F#,  --  龏 (<CJK Ideograph>)
      7200   => 16#9F90#,  --  龐 (<CJK Ideograph>)
      7201   => 16#9F91#,  --  龑 (<CJK Ideograph>)
      7202   => 16#9F92#,  --  龒 (<CJK Ideograph>)
      7203   => 16#9F94#,  --  龔 (<CJK Ideograph>)
      7204   => 16#9F96#,  --  龖 (<CJK Ideograph>)
      7205   => 16#9F97#,  --  龗 (<CJK Ideograph>)
      7206   => 16#9F9E#,  --  龞 (<CJK Ideograph>)
      7207   => 16#9FA1#,  --  龡 (<CJK Ideograph>)
      7208   => 16#9FA2#,  --  龢 (<CJK Ideograph>)
      7209   => 16#9FA3#,  --  龣 (<CJK Ideograph>)
      7210   => 16#9FA5#,  --  龥 (<CJK Ideograph>)

      others => 0];

end VSS.Strings.Converters.Decoders.EUCJP.JIS0212;
