%feature("docstring") OT::QuantileMatchingFactory
"Estimation by matching quantiles.

Implements generic estimation by matching quantiles.

Parameters
----------
distribution : :class:`~openturns.Distribution`
    The distribution defining the parametric model to be adjusted to data.
    Its parameters define the starting point of the algorithm.
probabilities : increasing sequence of float in [0, 1]
    The probabilities :math:`p_i \\in [0,1]` corresponding to the quantiles.
    At most :math:`K` values can be given.
    It is possible to provide less than :math:`K` values if :meth:`setKnownParameter` is used
bounds : :class:`~openturns.Interval`, optional
    Parameter bounds. The default bounds is an empty interval, which implies that the optimization problem is unbounded.

Notes
-----
We consider a parametric model whose cumulative distribution function :math:`F_{\\vect{\\theta}}`
is parametrized by :math:`\\vect{\\theta} \\in \\Rset^K`.

Given a set of probabilities :math:`p_1, ..., p_K \\in [0, 1]`
and a set of scalars :math:`q_1, ..., q_K \\in \\Rset`,
we want to determine :math:`\\vect{\\theta}` such that :math:`q_i` be the quantile of
order :math:`p_i` of :math:`F_{\\vect{\\theta}}`:

.. math::

    F_{\\vect{\\theta}}(q_i) = p_i, \\quad 1 \\leq i \\leq K

This is useful when some expert is able to specify such quantiles.

When the quantiles :math:`q_i` are not known, they can be deduced from a given sample. We denote
:math:`\\hat{q}_1, ..., \\hat{q}_K` the empirical quantiles obtained. In that case, we want to determine
:math:`\\vect{\\theta}` such that :math:`\\hat{q}_i` be the quantile of
order :math:`p_i` of :math:`F_{\\vect{\\theta}}`:

.. math::

    F_{\\vect{\\theta}}(\\hat{q}_i) = p_i, \\quad 1 \\leq i \\leq K

The optimal :math:`\\vect{\\theta}` is searched as the solution of the optimization problem:

.. math::

    \\vect{\\theta}_{opt} = \\argmin_{\\vect{\\theta} \\in \\Rset^K} \\sum_{i=1}^K (F^{-1}_{\\vect{\\theta}}(p_i) - \\hat{q}_i)^2

See also
--------
DistributionFactory

Examples
--------
Fit a distribution with 2 parameters. Hence, two quantiles are used to estimate the parameters.
Here, both quantiles are estimated from a given sample.

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(0.9, 1.7).getSample(10)
>>> factory = ot.QuantileMatchingFactory(ot.Normal(), [0.01, 0.99])
>>> inf_distribution = factory.build(sample)
>>> print(inf_distribution)
Normal(mu = 0.267484, sigma = 1.32218)
>>> print(factory.getProbabilities())
[0.01,0.99]

We see that the default value of the :math:`\\epsilon` parameter is so that we consider the 1% and 99% percentile ranks.

Use 5% and 95% percentile ranks:

>>> probabilities = [0.05, 0.95]
>>> factory = ot.QuantileMatchingFactory(ot.Normal(), probabilities)
>>> inf_distribution = factory.build(sample)

We can also make the optimization problem easier by specifying bounds to the parameters:

>>> bounds = ot.Interval([-0.1, 0.0], [1.0, 2.0])
>>> factory = ot.QuantileMatchingFactory(ot.Normal(), probabilities, bounds)
>>> inf_distribution = factory.build(sample)

An example with 4 parameters:

>>> # A distribution with 4 parameters
>>> distribution = ot.Beta(2.0, 3.0, 4.0, 5.0)
>>> sample = distribution.getSample(10)
>>> distribution = ot.Beta()
>>> factory = ot.QuantileMatchingFactory(distribution, [0.01, 1/3, 2/3, 0.99])
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::setOptimizationAlgorithm
"Accessor to the solver.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::getOptimizationAlgorithm
"Accessor to the solver.

Returns
-------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::setOptimizationBounds
"Accessor to the optimization bounds.

Parameters
----------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::getOptimizationBounds
"Accessor to the optimization bounds.

Returns
-------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::setKnownParameter
"Accessor to the known parameters.

Notes
-----
Using this method may cause the class to ignore some values of the given probabilities (or the default ones)
as the underlying least squares problem is solved in dimension :math:`K` minus
the number of known parameters.

Parameters
----------
values : sequence of float
    Values of fixed parameters.
indices : sequence of int
    Indices of fixed parameters.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> sample = distribution.getSample(10)
>>> factory = ot.QuantileMatchingFactory(ot.Beta(), [0.01, 0.99])
>>> # set (a,b) out of (r, t, a, b)
>>> factory.setKnownParameter([-1.0, 1.0], [2, 3])
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::getKnownParameterValues
"Accessor to the known parameters indices.

Returns
-------
values : :class:`~openturns.Point`
    The values of the fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::getKnownParameterIndices
"Accessor to the known parameters indices.

Returns
-------
indices : :class:`~openturns.Indices`
    The indices of the fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::buildFromQuantiles
"Build from given quantiles.

Parameters
----------
quantiles : increasing sequence of float
    The quantiles of the distribution, matching the probabilities provided to the constructor.

Returns
-------
dist : :class:`~openturns.Distribution`
    The estimated distribution.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::setProbabilities
"Accessor to the fractiles.

Parameters
----------
probabilities : sequence of float
    The probabilities :math:`p_i`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileMatchingFactory::getProbabilities
"Accessor to the probabilities.

Returns
-------
probabilities : :class:`~openturns.Point`
    The probabilities :math:`p_i`."
