%feature("docstring") OT::Matrix
"Real rectangular matrix.

Parameters
----------
n_rows : int, :math:`n_r > 0`, optional
    Number of rows.
    Default is 1.
n_columns : int, :math:`n_c > 0`, optional
    Number of columns.
    Default is 1.
values : sequence of float with size :math:`n_r \\times n_c`, optional
    Values. **column-major** ordering is used (like Fortran) for
    reshaping the flat list of values.
    Default creates a zero matrix.

Examples
--------
Create a matrix

>>> import openturns as ot
>>> M = ot.Matrix(2, 2, range(2 * 2))
>>> print(M)
[[ 0 2 ]
 [ 1 3 ]]

Get or set terms

>>> print(M[0, 0])
0.0
>>> M[0, 0] = 1.
>>> print(M[0, 0])
1.0
>>> print(M[:, 0])
[[ 1 ]
 [ 1 ]]

Create an openturns matrix from a numpy 2d-array (or matrix, or 2d-list)...

>>> import numpy as np
>>> np_2d_array = np.array([[1.0, 2.0, 3.0], [4.0, 5.0, 6.0]])
>>> ot_matrix = ot.Matrix(np_2d_array)

and back

>>> np_matrix = np.matrix(ot_matrix)

Basic linear algebra operations (provided the dimensions are compatible)

>>> A = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> B = ot.Matrix(np.eye(2))
>>> C = ot.Matrix(3, 2, [1.] * 3 * 2)
>>> print(A * B - C)
[[ 0 1 ]
 [ 2 3 ]
 [ 4 5 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::clean
"Set elements smaller than a threshold to zero.

Parameters
----------
threshold : float
    Threshold for zeroing elements.

Returns
-------
cleaned_matrix : :class:`~openturns.Matrix`
    Input matrix with elements smaller than the threshold set to zero."

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeQR
"Compute the QR factorization.

By default, it is the *economic* decomposition which is computed.
The economic QR factorization of a rectangular matrix :math:`\\mat{M}` with
:math:`n_r \\geq n_c` (more rows than columns) is defined as follows:

.. math::

    \\mat{M} = \\mat{Q} \\mat{R}
            = \\mat{Q} \\begin{bmatrix} \\mat{R}_1 \\\\ \\mat{0} \\end{bmatrix}
            = \\begin{bmatrix} \\mat{Q}_1, \\mat{Q}_2 \\end{bmatrix}
              \\begin{bmatrix} \\mat{R}_1 \\\\ \\mat{0} \\end{bmatrix}
            = \\mat{Q}_1 \\mat{R}_1

where :math:`\\mat{R}_1` is an :math:`n_c \\times n_c` upper triangular matrix,
:math:`\\mat{Q}_1` is :math:`n_r \\times n_c`, :math:`\\mat{Q}_2` is
:math:`n_r \\times (n_r - n_c)`, and :math:`\\mat{Q}_1` and :math:`\\mat{Q}_2`
both have orthogonal columns.

Parameters
----------
full_qr : bool, optional
    A flag telling whether `Q`, `R` or `Q1`, `R1` are returned.
    Default is *False* and returns `Q1`, `R1`.

Returns
-------
Q1 : :class:`~openturns.Matrix`
    The orthogonal matrix of the economic QR factorization.
R1 : :class:`~openturns.TriangularMatrix`
    The right (upper) triangular matrix of the economic QR factorization.
Q : :class:`~openturns.Matrix`
    The orthogonal matrix of the full QR factorization.
R : :class:`~openturns.TriangularMatrix`
    The right (upper) triangular matrix of the full QR factorization.

Notes
-----
The economic QR factorization is often used for solving overdetermined linear
systems (where the operator :math:`\\mat{M}` has :math:`n_r \\geq n_c`) in the
least-square sense because it implies solving a (simple) triangular system:

.. math::

    \\vect{\\hat{x}} = \\arg\\min\\limits_{\\vect{x} \\in \\Rset^{n_r}} \\|\\mat{M} \\vect{x} - \\vect{b}\\|
                   = \\mat{R}_1^{-1} (\\Tr{\\mat{Q}_1} \\vect{b})

This uses LAPACK's `DGEQRF <http://www.netlib.org/lapack/lapack-3.1.1/html/dgeqrf.f.html>`_
and `DORGQR <http://www.netlib.org/lapack/lapack-3.1.1/html/dorgqr.f.html>`_.

Examples
--------
>>> import openturns as ot
>>> import numpy as np
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> Q1, R1 = M.computeQR()
>>> np.testing.assert_array_almost_equal(Q1 * R1, M)"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeSingularValues
"Compute the singular values.

Parameters
----------
fullSVD : bool, optional
    Whether the null parts of the orthogonal factors are explicitly stored
    or not.
    Default is *False* and computes a reduced SVD.

Returns
-------
singular_values : :class:`~openturns.Point`
    The vector of singular values with size :math:`n = \\min(n_r, n_c)` that
    form the diagonal of the :math:`n_r \\times n_c` matrix
    :math:`\\mat{\\Sigma}` of the SVD decomposition.

See also
--------
computeSVD

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> print(M.computeSingularValues())
[9.52552,0.514301]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeQRInPlace
"Compute the QR factorization in place.

Similar to :meth:`computeQR`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeSingularValuesInPlace
"Compute the singular values in place.

Similar to :meth:`computeSingularValues` but the matrix is modified in place to avoid copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeSVD
"Compute the singular values decomposition (SVD).

The singular values decomposition of a rectangular matrix :math:`\\mat{M}` with
size :math:`n_r > n_c` reads:

.. math::

    \\mat{M} = \\mat{U} \\mat{\\Sigma} \\Tr{\\mat{V}}

where :math:`\\mat{U}` is an :math:`n_r \\times n_r` orthogonal matrix,
:math:`\\mat{\\Sigma}` is an :math:`n_r \\times n_c` diagonal matrix and
:math:`\\mat{V}` is an :math:`n_c \\times n_c` orthogonal matrix.

Parameters
----------
fullSVD : bool, optional
    Whether the null parts of the orthogonal factors are explicitly stored
    or not.
    Default is *False* and computes a reduced SVD.

Returns
-------
singular_values : :class:`~openturns.Point`
    The vector of singular values with size :math:`n = \\min(n_r, n_c)` that
    form the diagonal of the :math:`n_r \\times n_c` matrix
    :math:`\\mat{\\Sigma}` of the SVD.
U : :class:`~openturns.SquareMatrix`
    The left orthogonal matrix of the SVD.
VT : :class:`~openturns.SquareMatrix`
    The transposed right orthogonal matrix of the SVD.

Notes
-----
This uses LAPACK's `DGESDD <http://www.netlib.org/lapack/lapack-3.1.1/html/dgesdd.f.html>`_.

Examples
--------
>>> import openturns as ot
>>> import numpy as np
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> singular_values, U, VT = M.computeSVD(True)
>>> Sigma = ot.Matrix(M.getNbRows(), M.getNbColumns())
>>> for i in range(singular_values.getSize()):
...     Sigma[i, i] = singular_values[i]
>>> np.testing.assert_array_almost_equal(U * Sigma * VT, M)"


// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeSVDInPlace
"Compute the singular values decomposition (SVD).

Unlike computeSVD, this modifies the matrix in place and avoids a copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::getNbColumns
"Accessor to the number of columns.

Returns
-------
n_columns : int"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::getNbRows
"Accessor to the number of rows.

Returns
-------
n_rows : int"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::solveLinearSystem
"Solve a rectangular linear system whose the present matrix is the operator.

Parameters
----------
rhs : :class:`~openturns.Point` or :class:`~openturns.Matrix` with :math:`n_r` values or rows, respectively
    The right hand side member of the linear system.

Returns
-------
solution : :class:`~openturns.Point` or :class:`~openturns.Matrix`
    The solution of the rectangular linear system.

Notes
-----

This will handle both matrices and vectors, as well as underdetermined, square
or overdetermined linear systems although you'd better type explicitly your
matrix if it has some properties that could simplify the resolution (see
:class:`~openturns.TriangularMatrix`, :class:`~openturns.SquareMatrix`).

This uses LAPACK's `DGELSY <http://www.netlib.org/lapack/lapack-3.1.1/html/dgelsy.f.html>`_.
The `RCOND` parameter of this routine can be changed through the `MatrixImplementation-DefaultSmallPivot`
key of the :class:`~openturns.ResourceMap`.

Examples
--------
>>> import openturns as ot
>>> import numpy as np
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> b = ot.Point([1.0] * 3)
>>> x = M.solveLinearSystem(b)
>>> np.testing.assert_array_almost_equal(M * x, b)"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::solveLinearSystemInPlace
"Solve a rectangular linear system whose the present matrix is the operator.

Similar to :meth:`solveLinearSystem` except the matrix is modified in-place
during the resolution avoiding the need to allocate an extra copy if the
original copy is not re-used.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::reshape
"Reshape the matrix.

Parameters
----------
newRowDim : int
    The row dimension of the reshaped matrix.
newColDim : int
    The column dimension of the reshaped matrix.

Returns
-------
MT : :class:`~openturns.Matrix`
    The reshaped matrix.

Notes
-----
If the size of the reshaped matrix is smaller than the size of the matrix to be
reshaped, only the :math:`newRowDim\\times newColDim` first elements are kept (in
a column-major storage sense). If the size is greater, the new elements are set
to zero.
    
Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> print(M)
[[ 1 2 ]
 [ 3 4 ]
 [ 5 6 ]]
>>> print(M.reshape(1, 6))
1x6
[[ 1 3 5 2 4 6 ]]
>>> print(M.reshape(2, 2))
[[ 1 5 ]
 [ 3 2 ]]
>>> print(M.reshape(2, 6))
2x6
[[ 1 5 4 0 0 0 ]
 [ 3 2 6 0 0 0 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::reshapeInPlace
"Reshape the matrix, in place.

Parameters
----------
newRowDim : int
    The row dimension of the reshaped matrix.
newColDim : int
    The column dimension of the reshaped matrix.

Notes
-----
If the size of the reshaped matrix is smaller than the size of the matrix to be
reshaped, only the :math:`newRowDim\\times newColDim` first elements are kept (in
a column-major storage sense). If the size is greater, the new elements are set
to zero. If the size is unchanged, no copy of data is done.
    
Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> print(M)
[[ 1 2 ]
 [ 3 4 ]
 [ 5 6 ]]
>>> M.reshapeInPlace(1, 6)
>>> print(M)
1x6
[[ 1 3 5 2 4 6 ]]
>>> M.reshapeInPlace(2, 2)
>>> print(M)
[[ 1 5 ]
 [ 3 2 ]]
>>> M.reshapeInPlace(2, 6)
>>> print(M)
2x6
[[ 1 5 0 0 0 0 ]
 [ 3 2 0 0 0 0 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::transpose
"Transpose the matrix.

Returns
-------
MT : :class:`~openturns.Matrix`
    The transposed matrix.

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> print(M)
[[ 1 2 ]
 [ 3 4 ]
 [ 5 6 ]]
>>> print(M.transpose())
[[ 1 3 5 ]
 [ 2 4 6 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeGram
"Compute the associated Gram matrix.

Parameters
----------
transposed : bool
    Tells if matrix is to be transposed or not.
    Default value is True

Returns
-------
MMT : :class:`~openturns.Matrix`
    The Gram matrix.

Notes
-----
When `transposed` is `True`, compute :math:`\\Tr{M} M`.
Otherwise, compute :math:`M \\Tr{M}`.

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> MtM = M.computeGram()
>>> print(MtM)
[[ 35 44 ]
 [ 44 56 ]]
>>> MMt = M.computeGram(False)
>>> print(MMt)
[[  5 11 17 ]
 [ 11 25 39 ]
 [ 17 39 61 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::isEmpty
"Tell if the matrix is empty.

Returns
-------
is_empty : bool
    True if the matrix contains no element.

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[]])
>>> M.isEmpty()
True"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeHadamardProduct
"Compute the Hadamard product matrix.

Parameters
----------
Mat : :class:`~openturns.Matrix`
    The right hand matrix

Returns
-------
C : :class:`~openturns.Matrix`
    The Hadamard product matrix.

Notes
-----

The matrix :math:`\\mat{C} \\in \\Rset^{m \\times n}` resulting from the Hadamard product (
also known as the elementwise product) of the matrices
:math:`\\mat{A} \\in \\Rset^{m \\times n}` and :math:`\\mat{B} \\in \\Rset^{m \\times n}` is:

.. math::

    c_{i,j} = a_{i,j} b_{i,j}

for any :math:`i = 1, \\cdots, m` and :math:`j = 1, \\cdots, n`.

Examples
--------
>>> import openturns as ot
>>> A = ot.Matrix([[1.0, 2.0], [3.0, 4.0]])
>>> B = ot.Matrix([[1.0, 2.0], [3.0, 4.0]])
>>> C = A.computeHadamardProduct(B)
>>> print(C)
[[  1  4 ]
 [  9 16 ]]
>>> print(B.computeHadamardProduct(A))
[[  1  4 ]
 [  9 16 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::computeSumElements
"Compute the sum of the matrix elements.

Returns
-------
sum : a float
    The sum of the elements.

Notes
-----

Compute the sum of elements of the matrix :math:`\\mat{A} \\in \\Rset^{m \\times n}`:

.. math::

    s = \\sum_{i=1}^{m} \\sum_{j=1}^{n} a_{i,j}

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> s = M.computeSumElements()
>>> print(s)
21.0"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::squareElements
"Square the Matrix, ie each element of the matrix is squared.

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> M.squareElements()
>>> print(M)
[[  1  4 ]
 [  9 16 ]
 [ 25 36 ]]"
 
// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::frobeniusNorm
"Frobenius norm accessor.

Returns
-------
norm : float
    The Frobenius norm :math:`||\\mat{A}||_F = \\sqrt{\\sum_{i=1}^m \\sum_{j=1}^n a_{ij}^2}`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::getDiagonal
"Get the k-th diagonal of the matrix.

Parameters
----------
k : int
    The k-th diagonal to extract
    Default value is 0

Returns
-------
D: :class:`~openturns.Matrix`
    The k-th diagonal.

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0, 3.0], [4.0, 5.0, 6.0], [7.0, 8.0, 9.0]])
>>> diag = M.getDiagonal()
>>> print(diag)
[[ 1 ]
 [ 5 ]
 [ 9 ]]
>>> print(M.getDiagonal(1))
[[ 2 ]
 [ 6 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::getDiagonalAsPoint
"Get the k-th diagonal of the matrix.

Parameters
----------
k : int
    The k-th diagonal to extract
    Default value is 0

Returns
-------
pt : :class:`~openturns.Point`
    The k-th digonal.

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0, 3.0], [4.0, 5.0, 6.0], [7.0, 8.0, 9.0]])
>>> pt = M.getDiagonalAsPoint()
>>> print(pt)
[1,5,9]
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Matrix::setDiagonal
"Set the k-th diagonal of the matrix.

Parameters
----------
diag : :class:`~openturns.Point` or :class:`~openturns.Matrix` or a float
    Value(s) used to fill the diagonal of the matrix

k : int
    The k-th diagonal to fill
    Default value is 0

Examples
--------
>>> import openturns as ot
>>> M = ot.Matrix([[1.0, 2.0, 3.0], [4.0, 5.0, 6.0], [7.0, 8.0, 9.0]])
>>> M.setDiagonal([-1, 23, 9])
>>> print(M)
[[ -1  2  3 ]
 [  4 23  6 ]
 [  7  8  9 ]]
>>> M.setDiagonal(1.0)
>>> print(M)
[[ 1 2 3 ]
 [ 4 1 6 ]
 [ 7 8 1 ]]
>>> M.setDiagonal([2, 6, 9])
>>> print(M)
[[ 2 2 3 ]
 [ 4 6 6 ]
 [ 7 8 9 ]]"

// ---------------------------------------------------------------------
