%feature("docstring") OT::Sample
"Sample of real vectors.

Available constructors:
    Sample(array)

    Sample(*size, dim*)

    Sample(*size, point*)

    Sample(*other, first, last*)

Parameters
----------
array : 2-d sequence of float
    The data
size : int, :math:`m > 0`, optional
    The sample size.
    Default creates an empty sample with dimension 1.
dimension : int, :math:`n \\geq 0`, optional
    The real vectors dimension.
    Default creates an empty sample with dimension 1.
point : :class:`~openturns.Point` or flat (1d) array, list or tuple of floats, optional
    The point that will be repeated along the sample.
    Default creates a sample filled with zeros (null vectors).
other : :class:`~openturns.Sample`
    The sample contains points to copy.
first : int, :math:`0 \\leq first < m`
    The index of the first point to copy.
last : int, :math:`first < last \\leq m`, optional
    The index after the last point to copy.

See Also
--------
ImportFromCSVFile, ImportFromTextFile

Examples
--------
Create a Sample

>>> import openturns as ot
>>> import numpy as np
>>> sample = ot.Sample(3, 2)
>>> print(sample)
0 : [ 0 0 ]
1 : [ 0 0 ]
2 : [ 0 0 ]
>>> sample = ot.Sample(3, [1.0, 2.0])
>>> print(sample)
0 : [ 1 2 ]
1 : [ 1 2 ]
2 : [ 1 2 ]

Create a Sample from a (2d) array, list or tuple

>>> import numpy as np
>>> sample = ot.Sample(np.array([(1.0, 2.0), (3.0, 4.0), (5.0, 6.0)]))

and back

>>> z = np.array(sample)

Eventually samples may also be generated from probability distributions or
experiments

>>> random_sample = ot.Normal(2).getSample(10)
>>> experiment = ot.LHSExperiment(ot.Normal(2), 10).generate()

**Translation:** addition or subtraction of a (compatible) sample or a point, or a scalar which is promoted into a point of compatible dimension with equal components

>>> print(sample + sample)
0 : [  2  4 ]
1 : [  6  8 ]
2 : [ 10 12 ]
>>> print(sample - sample)
0 : [ 0 0 ]
1 : [ 0 0 ]
2 : [ 0 0 ]
>>> print(sample - sample[0])
0 : [ 0 0 ]
1 : [ 2 2 ]
2 : [ 4 4 ]
>>> print(sample - sample[0, 0])
0 : [ 0 1 ]
1 : [ 2 3 ]
2 : [ 4 5 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::BuildFromPoint
"Static method for building a sample from a sequence of float.

Parameters
----------
data : 1d array-like
    Data.

Returns
-------
sample : :class:`~openturns.Sample`
    Sample generated from sequence

Examples
--------
>>> import openturns as ot
>>> n = 20
>>> x = ot.Sample.BuildFromPoint(range(n))
>>> data = [2.0, 2.0, 1.0, 1.0, 2.0, 3.0, 1.0, 2.0, 2.0, 1.0]
>>> sample = ot.Sample.BuildFromPoint(data)"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::ImportFromCSVFile
"Static method for building a sample from a CSV file.

Parameters
----------
file_name : str
    Path to CSV file.
separator : str
    Separating string.
    Default uses `csv-file-separator` from the :class:`~openturns.ResourceMap`.

Returns
-------
sample : :class:`~openturns.Sample`
    Sample loaded from the CSV file.

Notes
-----
The file may or may not contain a header line (columns spanned with strings
delimited with quotes). If it does contain such a header line, it will be
used for setting the sample description using :py:meth:`setDescription`.

The implementation follows the RFC 4180: https://tools.ietf.org/html/rfc4180.

See Also
--------
exportToCSVFile

Examples
--------
>>> import openturns as ot

Let's first create a sample CSV file

>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> sample.exportToCSVFile('sample.csv')

And load it back

>>> loaded_sample = ot.Sample.ImportFromCSVFile('sample.csv')
>>> assert sample == loaded_sample"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::ImportFromTextFile
"Static method for building a sample from a text file.

Parameters
----------
file_name : str
    Path to text file.
separator : str
    Separating string.
    Default uses a blank space.
skipped_lines : int
    Number of lines skipped.
    Default is 0.
numSeparator : str
    Decimal separator.
    Default is dot.

Returns
-------
sample : :class:`~openturns.Sample`
    Sample loaded from the text file.

Notes
-----
The file may or may not contain a header line (columns spanned with strings
delimited with quotes). If it does contain such a header line, it will be
used for setting the sample description using :py:meth:`setDescription`.
It can also contain some comments, if a line starts with one of the
characters contained in `Sample-CommentsMarker` from the :class:`~openturns.ResourceMap`.

See Also
--------
exportToCSVFile

Examples
--------
>>> import openturns as ot

Let's first create a sample text file

>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> sample.exportToCSVFile('sample.txt', ' ')

And load it back

>>> loaded_sample = ot.Sample.ImportFromTextFile('sample.txt')
>>> assert sample == loaded_sample"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::add
"Append a sample (in-place).

Parameters
----------
point or sample : sequence or 2-d sequence of float
    The point(s) to append.

Examples
--------
Append an existing sample with a single point.

>>> import openturns as ot
>>> sample = ot.Sample(3, 2)
>>> sample.add([1.0, 2.0])
>>> print(sample)
0 : [ 0 0 ]
1 : [ 0 0 ]
2 : [ 0 0 ]
3 : [ 1 2 ]

Append an existing sample with another sample.

>>> sample.add(ot.Sample(2, [2.0, 1.0]))
>>> print(sample)
0 : [ 0 0 ]
1 : [ 0 0 ]
2 : [ 0 0 ]
3 : [ 1 2 ]
4 : [ 2 1 ]
5 : [ 2 1 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::argsort
"Returns indices of sorted sample.

The algorithm sorts the points in the sample in the lexicographic order. 

Returns
-------
indices : :class:`~openturns.Indices`
    The indices which sorts the sample.
isIncreasing : bool, optional
    If `True`, sort in increasing order.
    If `False`, sort in decreasing order.
    Default is `True`.

Examples
--------
>>> import openturns as ot
>>> sample = ot.Sample(
...     [[-1.0, 1.0, 0.0], [-1.0, 1.0, 1.0], [-1.0, 0.0, 1.0], [-1.0, 0.0, -1.0]]
... )
>>> print(sample)
0 : [ -1  1  0 ]
1 : [ -1  1  1 ]
2 : [ -1  0  1 ]
3 : [ -1  0 -1 ]
>>> indices = sample.argsort()
>>> print(indices)
[3,2,0,1]
>>> print(sample[indices])
    [ v0 v1 v2 ]
0 : [ -1  0 -1 ]
1 : [ -1  0  1 ]
2 : [ -1  1  0 ]
3 : [ -1  1  1 ]
>>> indices = sample.argsort(False)
>>> print(indices)
[1,0,2,3]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeCentralMoment
"Estimate componentwise central moments.

Parameters
----------
k : int
    The central moment's order.

Returns
-------
m : :class:`~openturns.Point`
    Componentwise central moment of order :math:`k` estimated from the sample.

Notes
-----
The central moment of order :math:`k` is estimated as follows:

.. math::

    \\vect{\\widehat{m}}^{(k)}_0 = \\Tr{\\left(\\frac{1}{m}
                                           \\sum_{j=1}^m
                                           \\left(x_i^{(j)} - \\widehat{\\mu}_i\\right)^k,
                                           \\quad i = 1, \\ldots, n\\right)}

where :math:`\\vect{\\widehat{\\mu}}` is the estimator of the mean.

These estimators are the natural (possibly biased) estimators.
For unbiased estimators use the other dedicated methods such as
:py:meth:`computeVariance` for the variance.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeCentralMoment(2))
[0.915126,0.873119]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeCovariance
"Estimate the covariance matrix.

Returns
-------
covariance : :class:`~openturns.CovarianceMatrix`
    Covariance matrix estimated from the sample.

Notes
-----
The covariance matrix is estimated as follows:

.. math::

    \\mat{\\widehat{\\Sigma}} = \\left[\\frac{1}{m - 1}
                                   \\sum_{k=1}^m
                                   \\left(x_i^{(k)} - \\widehat{\\mu}_i\\right)
                                   \\left(x_j^{(k)} - \\widehat{\\mu}_j\\right),
                                   \\quad i, j = 1, \\ldots, n\\right]

where :math:`\\vect{\\widehat{\\mu}}` denotes the estimate of the mean.

This is an unbiased estimator.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeCovariance())
[[ 0.946682  0.0182104 ]
 [ 0.0182104 0.903226  ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeEmpiricalCDF
"Estimate the empirical cumulative distribution function (ECDF).

Parameters
----------
x : sequence of float
    CDF input.
survival : bool, optional
    A flag telling whether this should estimate the empirical cumulative
    distribution function or the empirical survival function.
    Default is `False` and estimates the CDF.

Returns
-------
p : float, :math:`0 \\leq p \\leq 1`
    Empirical CDF or SF value at point x.

Notes
-----
The empirical cumulative distribution function (CDF) is estimated as follows:

.. math::

    \\hat{F}(\\vect{x}) = \\frac{1}{m} \\sum_{j=1}^m
                        \\mathbf{1}_{\\cap_{i=1}^n x_i^{(j)} \\leq x_i}(\\vect{x})

The empirical survival function (SF) is estimated in a similar way:

.. math::

    \\hat{S}(\\vect{x}) = \\frac{1}{m} \\sum_{j=1}^m
                        \\mathbf{1}_{\\cap_{i=1}^n x_i^{(j)} > x_i}(\\vect{x})

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeEmpiricalCDF(sample[0]))
0.1"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeKendallTau
"Estimate the Kendall coefficients matrix.

Returns
-------
tau : :class:`~openturns.CorrelationMatrix`
    Kendall coefficients matrix estimated from the sample.

Notes
-----
This uses an external implementation provided under the Boost Software License
by David Simcha based on the paper by [knight1966]_. It actually switches
between two implementations depending on the sample size:

    - The most basic implementation performing in :math:`O(m^2)` is used when
      the sample size is less than `SampleImplementation-SmallKendallTau`
      from the :class:`~openturns.ResourceMap`.
    - The other more complex implementation performing in :math:`O(m\\log(m))`
      is used for larger samples.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeKendallTau())
[[ 1          0.00689655 ]
 [ 0.00689655 1          ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeKurtosis
"Estimate the componentwise kurtosis (4th order central normalized moment).

Returns
-------
kurtosis : :class:`~openturns.Point`
    Componentwise kurtosis estimated from the sample.

Notes
-----
The componentwise kurtosis are estimated as follows:

.. math::

    \\vect{\\widehat{\\kappa}} = \\Tr{\\left(\\frac{m (m-1) (m+1)}{(m-2) (m-3)}
                                        \\frac{\\sum_{j=1}^m
                                              \\left(x_i^{(j)} - \\widehat{\\mu}_i\\right)^4}
                                             {\\left(\\sum_{j=1}^m
                                                    \\left(x_i^{(j)} - \\widehat{\\mu}_i\\right)^2
                                              \\right)^2}
                                        - 3 \\frac{3 (m-5)}{(m-2) (m-3)},
                                        \\quad i = 1, \\ldots, n\\right)}

where :math:`\\vect{\\widehat{\\mu}}` is the estimate of the mean.

This estimator is unbiased.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeKurtosis())
[3.27647,2.40275]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeMean
"Estimate the mean vector.

Returns
-------
mean : :class:`~openturns.Point`
    Mean vector estimated from the sample.

Notes
-----
The mean is estimated as follows:

.. math::

    \\vect{\\widehat{\\mu}} = \\Tr{\\left(\\frac{1}{m}
                                     \\sum_{j=1}^m x_i^{(j)},
                                     \\quad i = 1, \\ldots, n\\right)}

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeMean())
[-0.0512622,0.136653]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeMedian
"Estimate the componentwise medians (50%-quantiles).

Returns
-------
median : :class:`~openturns.Point`
    Median vector estimated from the sample.

See Also
--------
computeQuantilePerComponent

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeMedian())
[0.221141,0.108703]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeLinearCorrelation
"Estimate the linear (Pearson) correlation matrix.

Returns
-------
rho : :class:`~openturns.CorrelationMatrix`
    Pearson correlation matrix estimated from the sample.

Notes
-----
The Pearson correlation matrix is estimated as follows:

.. math::

    \\mat{\\widehat{\\rho}} = \\left[\\frac{\\widehat{\\Sigma}_{i,j}}
                                      {\\widehat{\\Sigma}_{i,i} \\widehat{\\Sigma}_{j,j}},
                                 \\quad i,j = 1, \\ldots, n\\right]

where :math:`\\mat{\\widehat{\\Sigma}}` denotes the estimate of the covariance.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeLinearCorrelation())
[[ 1         0.0196933 ]
 [ 0.0196933 1         ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeQuantile
"Estimate the quantile of the joint distribution underlying the sample.

Parameters
----------
p : float, :math:`0 \\leq p \\leq 1`, or sequence of float
    Input probability level.

Returns
-------
quantile : :class:`~openturns.Point` or :class:`~openturns.Sample`
    Quantile of the **joint** distribution at probability level :math:`p`,
    estimated from the sample.

Raises
------
NotImplementedYetError : If the dimension is greater than 1.

See Also
--------
computeQuantilePerComponent

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(1).getSample(30)
>>> print(sample.computeQuantile(.2))
[-0.947394]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeQuantilePerComponent
"Estimate the componentwise quantiles.

Parameters
----------
p : float, :math:`0 \\leq p \\leq 1`, or sequence of float
    Input probability level.

Returns
-------
quantile : :class:`~openturns.Point` or :class:`~openturns.Sample`
    Componentwise quantile at probability level :math:`p`, estimated from the
    sample.

Notes
-----
The present implementation interpolates the quantile between the two adjacent
empirical quantiles (:math:`\\widehat{x}_i^-` and :math:`\\widehat{x}_i^+`):

.. math::

    \\widehat{q}_i = \\alpha \\widehat{x}_i^- + (1 - \\alpha) \\widehat{x}_i^+

where :math:`\\alpha = p m - 0.5`.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeQuantilePerComponent(0.2))
[-0.696412,-0.767092]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeRange
"Compute the range per component.

Returns
-------
range : :class:`~openturns.Point`
    Componentwise ranges estimated from the sample.

Notes
-----
The *statistical range* is defined as the deviation between the maximal and
the minimal value of the sample.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeRange())
[4.02827,3.49949]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeRawMoment
"Compute the raw (non-central) moment per component.

Parameters
----------
k : int, :math:`k \\geq 0`
    Componentwise moment's order.

Returns
-------
moments : :class:`~openturns.Point`
    Componentwise moments estimated from the sample.

Notes
-----
The (raw) moment of order :math:`k` is estimated as follows:

.. math::

    \\vect{\\widehat{m}}^{(k)} = \\Tr{\\left(\\frac{1}{m}
                                         \\sum_{j=1}^m {x_i^{(j)}}^k,
                                         \\quad i = 1, \\ldots, n\\right)}

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeRawMoment(2))
[0.917754,0.891793]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeSkewness
"Estimate the componentwise skewness (3rd order central normalized moment).

Returns
-------
skewness : :class:`~openturns.Point`
    Componentwise skewness estimated from the sample.

Notes
-----
The componentwise skewnesses are estimated as follows:

.. math::

    \\vect{\\widehat{\\delta}} = \\Tr{\\left(m \\frac{\\sqrt{m-1}}{m-2}
                                        \\frac{\\sum_{j=1}^m
                                              \\left(x_i^{(j)} - \\widehat{\\mu}_i\\right)^3}
                                             {\\left(\\sum_{j=1}^m
                                                    \\left(x_i^{(j)} - \\widehat{\\mu}_i\\right)^2
                                              \\right)^{3/2}},
                                        \\quad i = 1, \\ldots, n\\right)}

where :math:`\\vect{\\widehat{\\mu}}` is the estimate of the mean.

This is an unbiased estimator.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeSkewness())
[-0.69393,0.231931]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeSpearmanCorrelation
"Estimate the Spearman correlation matrix.

Returns
-------
rho : :class:`~openturns.CorrelationMatrix`
    Spearman correlation matrix estimated from the sample.

Notes
-----
The Spearman correlation matrix is estimated as the Pearson correlation matrix
of the ranks sample (i.e. using `self.rank().computeLinearCorrelation()`).

See Also
--------
rank, computeLinearCorrelation

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeSpearmanCorrelation())
[[  1          -0.00556174 ]
 [ -0.00556174  1          ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeStandardDeviation
"Estimate the componentwise standard deviations.

Returns
-------
standard_deviations : :class:`~openturns.Point`
    Componentwise standard deviation estimated from the sample.

Notes
-----
The componentwise standard deviations are estimated as the square root of the
componentwise variances.

See Also
--------
computeVariance

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeStandardDeviation())
[0.972976,0.950382]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::computeVariance
"Estimate the componentwise variances.

Returns
-------
variances : :class:`~openturns.Point`
    Componentwise variances estimated from the sample.

Notes
-----
The componentwise variances are estimated as follows:

.. math::

    \\vect{\\widehat{\\sigma^2}} = \\Tr{\\left(\\frac{1}{m-1}
                                          \\sum_{j=1}^m
                                          \\left(x_i^{(j)} - \\widehat{\\mu}_i\\right)^2,
                                          \\quad i = 1, \\ldots, n\\right)}

where :math:`\\vect{\\widehat{\\mu}}` is the estimate of the mean.

This estimator is unbiased.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.computeVariance())
[0.946682,0.903226]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::erase
"Erase point(s) at or between index(es) (in-place).

Parameters
----------
f : int, :math:`0 \\leq f < m`
    The index of the first point to erase.
l : int, :math:`f < l \\leq m`, optional
    The index after the last point to erase.
    Default uses `l = f + 1` and only removes `sample[f]`.

Examples
--------
>>> import openturns as ot
>>> sample = ot.Sample([[i] for i in range(5)])
>>> print(sample)
0 : [ 0 ]
1 : [ 1 ]
2 : [ 2 ]
3 : [ 3 ]
4 : [ 4 ]
>>> sample.erase(1, 3)
>>> print(sample)
0 : [ 0 ]
1 : [ 3 ]
2 : [ 4 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::exportToCSVFile
"Dump the sample to a CSV file.

Parameters
----------
file_name : str
    Path to CSV file.
separator : str
    Separating string.
    Default uses `csv-file-separator` from the :class:`~openturns.ResourceMap`.
numSeparator : str, default='.'
    Decimal separator.
precision : int
    Numerical precision
    Default takes `Sample-CSVPrecision` entry from :class:`~openturns.ResourceMap`.
format : str
    Floating-point formatting, one of:

    - scientific: exponent notation
    - fixed: constant number of digits
    - defaultfloat: variable number of digits

    Default takes `Sample-CSVFormat` entry from :class:`~openturns.ResourceMap`.

Notes
-----
This will create a header line with componentwise descriptions (obtained from
:py:meth:`getDescription`) between quotes as column names.
In scientific formatting the number of significant digits is :math:`precision + 1`.

See Also
--------
ImportFromCSVFile, ImportFromTextFile

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> sample.exportToCSVFile('sample.csv', '; ')"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::find
"Get the position of a point in the sample.

Parameters
----------
point : sequence of float
    The wanted point.

Returns
-------
index : int,
    Returns :math:`m` if the point does not belong to the sample.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(30)
>>> print(sample.find(sample[10]))
10
>>> print(sample.find([0.0, 0.0]))
30"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::getDescription
"Accessor to the componentwise description.

Returns
-------
description : :class:`~openturns.Description`
    Description of the sample's components.

See Also
--------
setDescription"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::getDimension
"Accessor to the sample's dimension.

Returns
-------
n : int
    The number of components of the points in the sample."

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::getMarginal
"Accessor to sample marginal(s) (column(s)).

Parameters
----------
indices : int, sequence of int, :math:`0 \\leq i < n` or sequence of str
    The identifiers of the wanted marginal(s).
    When the description contains duplicate labels, the first marginal is picked up.

Returns
-------
sample : :class:`~openturns.Sample`
    A subsample of the present sample with the requested marginal(s).

Notes
-----
The :class:`~openturns.Sample` also implements slicing in its
`__getitem__` method.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(10).getSample(3)
>>> print(sample.getMarginal([1, 4]))
    [ X1        X4        ]
0 : [ -1.26617  -2.18139  ]
1 : [  0.261018 -1.31178  ]
2 : [  0.445785  0.473617 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::getMax
"Accessor to the componentwise maximum values.

Returns
-------
maximum_values : :class:`~openturns.Point`
    Componentwise maximum values."

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::getMin
"Accessor to the componentwise minimum values.

Returns
-------
minimum_values : :class:`~openturns.Point`
    Componentwise minimum values."

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::getSize
"Accessor to the sample size.

Returns
-------
m : int
    The number points in the sample."

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::rank
"Compute the sample (componentwise) ranks.

Parameters
----------
marginal_index : int, :math:`0 \\leq i < n`, optional
    The component whose ranks are wanted.
    Default computes the ranks of all the components.

Notes
-----
The ranks of a 1d sample is a list of indices that sorts the points in the
ascending order. Ties (equal points) are averaged.

Returns
-------
ranks : :class:`~openturns.Sample`
    The requested ranks.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> print(sample.rank())
    [ X0 X1 ]
0 : [ 2  0  ]
1 : [ 1  2  ]
2 : [ 0  1  ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::setDescription
"Accessor to the componentwise description.

Parameters
----------
description : sequence of str
    Description of the sample's components.

See Also
--------
getDescription"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::sort
"Sort the sample.

The ordering is based on the comparison operator of the :class:`~openturns.Point`. 
Hence, the `sort` method orders the points in the sample 
according to lexicographic order. 

Parameters
----------
marginal_index : int, :math:`0 \\leq i < n`, optional
    The index of the component to sort.
    Default sorts the whole sample, i.e. the returned sorted sample has 
    the same dimension as the input sample.
    If `marginal_index` is provided, then the returned sorted sample has dimension 1: 
    the corresponding marginal sample is sorted and returned.

Returns
-------
sorted_sample : :class:`~openturns.Sample`
    The requested sorted sample.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> print(sample.sort())
    [ X0        X1        ]
0 : [ -2.18139   0.350042 ]
1 : [ -0.438266  1.20548  ]
2 : [  0.608202 -1.26617  ]
>>> print(sample.sort(1))
0 : [ -1.26617  ]
1 : [  0.350042 ]
2 : [  1.20548  ]

In the following sample, the first component of all points is equal to -1, 
which creates a tie where the other components must be used to make 
a difference in the comparison. 
The algorithm sorts the points taking into account all the components in the 
points. 
This shows that the algorithm uses lexicographic ordering, since using only 
the first component would leave the sample unchanged.

>>> import openturns as ot
>>> sample = ot.Sample(
...     [[-1.0, 1.0, 1.0], [-1.0, 1.0, 0.0], [-1.0, 0.0, 1.0], [-1.0, 0.0, -1.0]]
... )
>>> print(sample)
0 : [ -1  1  1 ]
1 : [ -1  1  0 ]
2 : [ -1  0  1 ]
3 : [ -1  0 -1 ]
>>> print(sample.sort())
0 : [ -1  0 -1 ]
1 : [ -1  0  1 ]
2 : [ -1  1  0 ]
3 : [ -1  1  1 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::sortInPlace
"Sort the sample in place.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> sample.sortInPlace()
>>> print(sample)
    [ X0        X1        ]
0 : [ -2.18139   0.350042 ]
1 : [ -0.438266  1.20548  ]
2 : [  0.608202 -1.26617  ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::sortAccordingToAComponent
"Sort the sample according to the given component.

Parameters
----------
marginal_index : int, :math:`0 \\leq i < n`
    The component to use for sorting the sample.

Returns
-------
sorted_sample : :class:`~openturns.Sample`
    The sample sorted according to the given component.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> print(sample.sortAccordingToAComponent(0))
    [ X0        X1        ]
0 : [ -2.18139   0.350042 ]
1 : [ -0.438266  1.20548  ]
2 : [  0.608202 -1.26617  ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::sortAccordingToAComponentInPlace
"Sort the sample in place according to the given component.

Parameters
----------
marginal_index : int, :math:`0 \\leq i < n`
    The component to use for sorting the sample.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> sample.sortAccordingToAComponentInPlace(0)
>>> print(sample)
    [ X0        X1        ]
0 : [ -2.18139   0.350042 ]
1 : [ -0.438266  1.20548  ]
2 : [  0.608202 -1.26617  ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::select
"Select points in a sample.

It selects the points at given locations and returns them as a new sample.

Parameters
----------
indices : sequence of int, :math:`0 \\leq i < m`
    The selected indices.

Returns
-------
selected_sample : :class:`~openturns.Sample`
    The selected points as a sample.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> selected_sample = sample.select([1, 0, 1])
>>> print(selected_sample)
    [ X0        X1        ]
0 : [ -0.438266  1.20548  ]
1 : [  0.608202 -1.26617  ]
2 : [ -0.438266  1.20548  ]
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::split
"Trunk the sample.

It splits the sample before the index passed as argument and returns the
remainder as new sample.

Parameters
----------
index : int, :math:`0 \\leq i < m`
    The truncation index.

Returns
-------
remainder_sample : :class:`~openturns.Sample`
    The remainder sample (everything that comes after the truncation index).

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> remainder_sample = sample.split(1)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
>>> print(remainder_sample)
    [ X0        X1        ]
0 : [ -0.438266  1.20548  ]
1 : [ -2.18139   0.350042 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::stack
"Stack (horizontally) the given sample to the current one (in-place).

Parameters
----------
sample : :class:`~openturns.Sample`
    Sample to stack with compatible size.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal(2).getSample(3)
>>> print(sample)
    [ X0        X1        ]
0 : [  0.608202 -1.26617  ]
1 : [ -0.438266  1.20548  ]
2 : [ -2.18139   0.350042 ]
>>> another_sample = ot.Normal(2).getSample(3)
>>> print(another_sample)
    [ X0        X1        ]
0 : [ -0.355007  1.43725  ]
1 : [  0.810668  0.793156 ]
2 : [ -0.470526  0.261018 ]
>>> sample.stack(another_sample)
>>> print(sample)
    [ X0        X1        X0        X1        ]
0 : [  0.608202 -1.26617  -0.355007  1.43725  ]
1 : [ -0.438266  1.20548   0.810668  0.793156 ]
2 : [ -2.18139   0.350042 -0.470526  0.261018 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::asPoint
"Accessor to the internal linear storage for 1D sample.

Returns
-------
values : :class:`~openturns.Point`
    Flat internal representation of the sample.

Notes
-----
Available only for 1D sample.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Normal().getSample(5)
>>> print(sample)
    [ X0        ]
0 : [  0.608202 ]
1 : [ -1.26617  ]
2 : [ -0.438266 ]
3 : [  1.20548  ]
4 : [ -2.18139  ]
>>> print(sample.asPoint())
[0.608202,-1.26617,-0.438266,1.20548,-2.18139]
"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::clear
"Erase all values."

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::sortUnique
"Sort the sample and remove duplicate points.

Returns
-------
unique_sample : :class:`~openturns.Sample`
    The requested sorted sample with duplicate points removed.

Examples
--------
>>> import openturns as ot
>>> sample = ot.Sample([[3, 0, 3], [1, 1, 0], [0, 2, 2], [1, 1, 0]])
>>> print(sample)
0 : [ 3 0 3 ]
1 : [ 1 1 0 ]
2 : [ 0 2 2 ]
3 : [ 1 1 0 ]
>>> print(sample.sortUnique())
0 : [ 0 2 2 ]
1 : [ 1 1 0 ]
2 : [ 3 0 3 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Sample::sortUniqueInPlace
"Sort the sample in place and remove duplicate points.

Examples
--------
>>> import openturns as ot
>>> sample = ot.Sample([[3, 0, 3], [1, 1, 0], [0, 2, 2], [1, 1, 0]])
>>> print(sample)
0 : [ 3 0 3 ]
1 : [ 1 1 0 ]
2 : [ 0 2 2 ]
3 : [ 1 1 0 ]
>>> sample.sortUniqueInPlace()
>>> print(sample)
0 : [ 0 2 2 ]
1 : [ 1 1 0 ]
2 : [ 3 0 3 ]"
