%feature("docstring") OT::QuadraticTaylor
"Second order polynomial response surface by Taylor expansion.

Available constructors:
    QuadraticTaylor(*center, function*)

Parameters
----------
center : sequence of float
    Point :math:`\\vect{x}_0` where the Taylor expansion of the function
    :math:`h` is performed.
function : :class:`~openturns.Function`
    Function :math:`h` to be approximated.

Notes
-----
The approximation of the model response :math:`\\vect{y} = h(\\vect{x})` around a
specific set :math:`\\vect{x}_0 = (x_{0,1},\\dots,x_{0,n_{X}})` of input
parameters may be of interest. One may then substitute :math:`h` for its Taylor
expansion at point :math:`\\vect{x}_0`. Hence :math:`h` is replaced with a first
or second-order polynomial :math:`\\widehat{h}` whose evaluation is inexpensive,
allowing the analyst to apply the uncertainty anaysis methods.

We consider here the second order Taylor expansion around :math:`\\ux=\\vect{x}_0`.

.. math::

    \\vect{y} \\, \\approx \\, \\widehat{h}(\\vect{x}) \\, = \\,
     h(\\vect{x}_0) \\, + \\, \\sum_{i=1}^{n_{X}} \\;
      \\frac{\\partial h}{\\partial x_i}(\\vect{x}_0).\\left(x_i - x_{0,i} \\right) \\, +
     \\, \\frac{1}{2} \\; \\sum_{i,j=1}^{n_X} \\;
      \\frac{\\partial^2 h}{\\partial x_i \\partial x_j}(\\vect{x}_0).\\left(x_i - x_{0,i} \\right).\\left(x_j - x_{0,j} \\right)

Introducing a vector notation, the previous equation rewrites:

.. math::

    \\vect{y} \\, \\approx \\,
     \\vect{y}_0 \\, +
     \\, \\vect{\\vect{L}} \\: \\left(\\vect{x}-\\vect{x}_0\\right) \\, +
     \\, \\frac{1}{2} \\; \\left\\langle \\left\\langle\\vect{\\vect{\\vect{Q}}}\\:,
                                                      \\vect{x}-\\vect{x}_0 \\right\\rangle,
                                                \\:\\vect{x}-\\vect{x}_0 \\right\\rangle

where

- :math:`\\vect{y_0} = (y_{0,1} , \\dots, y_{0,n_Y})^{\\textsf{T}} = h(\\vect{x}_0)`
  is the vector model response evaluated at :math:`\\vect{x}_0` ;
- :math:`\\vect{x}` is the current set of input parameters ;
- :math:`\\vect{\\vect{L}} = \\left( \\frac{\\partial y_{0,j}}{\\partial x_i} \\,,\\, i=1,\\ldots, n_X \\,,\\, j=1,\\ldots, n_Y \\right)`
  is the transposed Jacobian matrix evaluated at :math:`\\vect{x}_0` ;
- :math:`\\vect{\\vect{Q}} = \\left\\{ \\frac{\\partial^2 y_{0,k}}{\\partial x_i \\partial x_j} \\, \\, , \\, \\, i,j=1,\\ldots, n_X \\, \\, , \\, \\, k=1, \\ldots, n_Y \\right\\}`
  is the transposed Hessian matrix.

See also
--------
LinearTaylor, LinearLeastSquares, QuadraticLeastSquares

Examples
--------
>>> import openturns as ot
>>> formulas = ['x1 * sin(x2)', 'cos(x1 + x2)', '(x2 + 1) * exp(x1 - 2 * x2)']
>>> myFunc = ot.SymbolicFunction(['x1', 'x2'], formulas)
>>> myTaylor = ot.QuadraticTaylor([1, 2], myFunc)
>>> myTaylor.run()
>>> responseSurface = myTaylor.getMetaModel()
>>> print(responseSurface([1.2,1.9]))
[1.13655,-0.999155,0.214084]"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::getCenter
"Get the center.

Returns
-------
center : :class:`~openturns.Point`
    Point :math:`\\vect{x}_0` where the Taylor expansion of the function is
    performed."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::getConstant
"Get the constant vector of the approximation.

Returns
-------
constantVector : :class:`~openturns.Point`
    Constant vector of the approximation, equal to :math:`h(x_0)`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::getLinear
"Get the gradient of the function at :math:`\\vect{x}_0`.

Returns
-------
gradient : :class:`~openturns.Matrix`
    Gradient of the function :math:`h` at the point :math:`\\vect{x}_0` (the
    transposition of the jacobian matrix)."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::getQuadratic
"Get the hessian of the function at :math:`\\vect{x}_0`.

Returns
-------
tensor : :class:`~openturns.SymmetricTensor`
    Hessian of the function :math:`h` at the point :math:`\\vect{x}_0`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::getInputFunction
"Get the function.

Returns
-------
function : :class:`~openturns.Function`
    Function :math:`h` to be approximated."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::getMetaModel
"Get an approximation of the function.

Returns
-------
approximation : :class:`~openturns.Function`
    An approximation of the function :math:`h` by a Quadratic Taylor expansion
    at the point :math:`\\vect{x}_0`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticTaylor::run
"Perform the Quadratic Taylor expansion around :math:`\\vect{x}_0`."
