%feature("docstring") OT::MethodOfMomentsFactory
"Estimation by method of moments.

Implements generic estimation by method of moments.

Parameters
----------
distribution : :class:`~openturns.Distribution`
    The distribution defining the parametric model to be adjusted to data.
bounds : :class:`~openturns.Interval`, optional
    Parameter bounds

Notes
-----
The underlying optimization problem seeks to minimize the sum of slacks between the
central moments of the sample and the central moments of the parametric model:

.. math::

    \\Delta = \\argmin_{\\vect{\\theta} \\in \\Rset^K} [ |\\mu_1-m_1|/\\sigma ]^2 + \\sum_{i=2}^K[ (|\\mu_i|^{1/i}-|m_i|^{1/i})/\\sigma ]^2

with :math:`K` the number of parameters of the distribution, :math:`\\mu_i` and
:math:`m_i` the central moments of the sample and the parametric model.

See also
--------
DistributionFactory

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal(0.9, 1.7)
>>> sample = distribution.getSample(10)
>>> factory = ot.MethodOfMomentsFactory(ot.Normal())
>>> inf_distribution = factory.build(sample)

With parameter bounds:

>>> bounds = ot.Interval([0.8, 1.6], [1.0, 1.8])
>>> factory = ot.MethodOfMomentsFactory(ot.Normal(), bounds)
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setOptimizationAlgorithm
"Accessor to the solver.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getOptimizationAlgorithm
"Accessor to the solver.

Returns
-------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setOptimizationBounds
"Accessor to the optimization bounds.

Parameters
----------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getOptimizationBounds
"Accessor to the optimization bounds.

Returns
-------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setKnownParameter
"Accessor to the known parameters.

Parameters
----------
values : sequence of float
    Values of fixed parameters.
indices : sequence of int
    Indices of fixed parameters.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> sample = distribution.getSample(10)
>>> factory = ot.MethodOfMomentsFactory(ot.Beta())
>>> # set (a,b) out of (r, t, a, b)
>>> factory.setKnownParameter([-1.0, 1.0], [2, 3])
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getKnownParameterValues
"Accessor to the known parameters indices.

Returns
-------
values : :class:`~openturns.Point`
    Values of fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getKnownParameterIndices
"Accessor to the known parameters indices.

Returns
-------
indices : :class:`~openturns.Indices`
    Indices of fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::buildFromMoments
"Build from moments.

Parameters
----------
moments : sequence of float
    Consists in the mean followed by consecutive central moments from order 2
    (variance), of total size at least the distribution parameter dimension.

Returns
-------
dist : :class:`~openturns.Distribution`
    Estimated distribution.

Notes
-----
Depending on the parametric model choosed, not all moments define a valid
distribution, so it should only used with empirical moments from the same model.

Examples
--------
>>> import openturns as ot
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> factory = ot.MethodOfMomentsFactory(ot.Beta())
>>> cm = [distribution.getCentralMoment(i + 2)[0] for i in range(3)]
>>> moments = [distribution.getMean()[0]] + cm
>>> inf_distribution = factory.buildFromMoments(moments)"

