%feature("docstring") OT::GeneralizedExtremeValue
"Generalized ExtremeValue distribution.

Available constructors:
    GeneralizedExtremeValue(*mu=0.0, sigma=1.0, xi=0.0*)

    GeneralizedExtremeValue(*distribution*)

Parameters
----------
mu : float
    Position parameter :math:`\\mu`.
sigma : float, :math:`\\sigma > 0`
    Scale parameter :math:`\\sigma >0`.
xi : float
    Shape parameter :math:`\\xi`.
distribution : :class:`~openturns.WeibullMax`, :class:`~openturns.Frechet` or :class:`~openturns.Gumbel`
    The underlying distribution.

Notes
-----
Its cumulative and probability distribution functions are defined as:

.. math::
    :nowrap:

    \\begin{eqnarray*}
        F_X(x) &  = & \\exp(-t(x)), \\quad \\forall x \\in \\cS \\\\
        p_X(x) & = & \\dfrac{1}{\\sigma}t^{1+\\xi} \\exp(-t(x)), \\quad \\forall x \\in \\cS
    \\end{eqnarray*}

where 

.. math::
    :nowrap:

    \\begin{eqnarray*}
        t(x) &  = & \\left(1+\\xi \\dfrac{x-\\mu}{\\sigma}\\right)^{-1/\\xi} \\quad \\text{ if } \\xi \\neq 0\\\\
        t(x) &  = & \\exp \\left(- \\dfrac{x-\\mu}{\\sigma}\\right) \\quad \\text{ if } \\xi= 0
    \\end{eqnarray*}

and 

.. math::
    :nowrap:

    \\begin{eqnarray*}
        \\cS &  = &  [\\mu-\\sigma/\\xi, +\\infty) \\quad \\text{ if } \\xi > 0\\\\
            &  = &  \\Rset \\quad \\text{ if } \\xi = 0\\\\
            &  = &  (-\\infty,\\mu-\\sigma/\\xi] \\quad \\text{ if } \\xi < 0
    \\end{eqnarray*}


Its first moments are:

.. math::
    :nowrap:

    \\begin{eqnarray*}
    \\Expect{X} & = & \\mu+\\dfrac{\\sigma}{\\xi}\\left(\\Gamma(1-\\xi)-1\\right) \\quad \\text{ if } \\xi < 1, \\xi\\neq 0 \\\\
               & = &     \\mu + \\sigma\\gamma_e \\quad \\text{ if } \\xi= 0 \\\\
               & = &    \\infty  \\quad \\text{ if } \\xi \\geq 1
    \\end{eqnarray*}

    \\begin{eqnarray*}
        \\Var{X} & = &  \\dfrac{\\sigma^2}{\\xi^2}\\left(\\Gamma(1-2\\xi)-\\Gamma^2(1-\\xi)\\right) \\quad \\text{ if } \\xi < 1/2, \\xi\\neq 0 \\\\
                & = & \\dfrac{1}{6} \\sigma^2\\pi^2 \\quad \\text{ if } \\xi= 0 \\\\
                & = & \\infty \\quad \\text{ if } \\xi \\geq 1/2
    \\end{eqnarray*}

where :math:`\\gamma_e` is Euler's constant.


When the constructor from a :class:`~openturns.WeibullMax`:math:`(\\beta, \\alpha, \\gamma)` distribution is used,  then it creates :class:`~openturns.GeneralizedExtremeValue`:math:`(\\mu, \\sigma, \\xi)` where:

.. math::
    :nowrap:

    \\begin{eqnarray*}
       \\mu & = & \\gamma - \\beta \\\\
       \\sigma & = & \\dfrac{\\beta}{\\alpha} \\\\
       \\xi & = & -\\dfrac{1}{\\alpha}
    \\end{eqnarray*}



When the constructor from a :class:`~openturns.Frechet`:math:`(\\beta, \\alpha, \\gamma)` distribution is used,  then it creates :class:`~openturns.GeneralizedExtremeValue`:math:`(\\mu, \\sigma, \\xi)` where:

.. math::
    :nowrap:

    \\begin{eqnarray*}
       \\mu & = & \\gamma + \\beta \\\\
       \\sigma & = & \\dfrac{\\beta}{\\alpha} \\\\
       \\xi & = & \\dfrac{1}{\\alpha}
    \\end{eqnarray*}



When the constructor from a :class:`~openturns.Gumbel`:math:`(\\beta, \\gamma)` distribution is used,  then it creates :class:`~openturns.GeneralizedExtremeValue`:math:`(\\mu, \\sigma, \\xi)` where:

.. math::
    :nowrap:

    \\begin{eqnarray*}
       \\mu & = & \\gamma  \\\\
       \\sigma & = & \\beta \\\\
       \\xi & = & 0.0
    \\end{eqnarray*}

Examples
--------
Create a distribution:

>>> import openturns as ot
>>> dist1 = ot.GeneralizedExtremeValue(1.0, 2.0, -0.2)

Create it from a Frechet distribution:

>>> myFrechet = Frechet(1.0, 2.0, 3.0)
>>> dist1 = ot.GeneralizedExtremeValue(myFrechet)

Draw a sample:

>>> sample = dist1.getSample(5)"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::getActualDistribution
"Accessor to the internal distribution.

Returns
-------
distribution : :class:`~openturns.Distribution`
     The actual distribution in charge of the computation (:class:`~openturns.WeibullMax`, :class:`~openturns.Frechet`, :class:`~openturns.Gumbel`)."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::asFrechet
"Temptative conversion to the underlying Frechet distribution.

Returns
-------
distribution : :class:`~openturns.Frechet`
     The underlying Frechet distribution.

Notes
-----
If :math:`\\xi >0` then the :class:`~openturns.GeneralizedExtremeValue`:math:`(\\mu, \\sigma, \\xi)` is actually a :class:`~openturns.Frechet`:math:`(\\beta, \\alpha, \\gamma)` distribution where:

.. math::
    :nowrap:

    \\begin{eqnarray*}
       \\beta & = & \\dfrac{\\sigma}{\\xi} \\\\
       \\alpha & = & \\dfrac{1}{\\xi} \\\\
       \\gamma & = & \\mu - \\dfrac{\\sigma}{\\xi}
    \\end{eqnarray*}
"
// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::asWeibullMax
"Temptative conversion to the underlying WeibullMax distribution.

Returns
-------
distribution : :class:`~openturns.WeibullMax`
     The underlying WeibullMax  distribution.

Notes
-----
If :math:`\\xi <0` then the :class:`~openturns.GeneralizedExtremeValue`:math:`(\\mu, \\sigma, \\xi)` is actually a :class:`~openturns.WeibullMax`:math:`(\\beta, \\alpha, \\gamma)` distribution where:

.. math::
    :nowrap:

    \\begin{eqnarray*}
       \\beta & = & -\\dfrac{\\sigma}{\\xi} \\\\
       \\alpha & = & -\\dfrac{1}{\\xi} \\\\
       \\gamma & = & \\mu - \\dfrac{\\sigma}{\\xi}
    \\end{eqnarray*}
"
// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::asGumbel
"Temptative conversion to the underlying Gumbel distribution.

Returns
-------
distribution : :class:`~openturns.Gumbel`
     The underlying Gumbel distribution.

Notes
-----
If :math:`\\xi =0` then the :class:`~openturns.GeneralizedExtremeValue`:math:`(\\mu, \\sigma, 0.0)` is actually a :class:`~openturns.Gumbel`:math:`(\\beta, \\gamma)` distribution where:

.. math::
    :nowrap:

    \\begin{eqnarray*}
       \\beta & = & \\sigma \\\\
       \\gamma & = & \\mu 
    \\end{eqnarray*}
"
// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::getMu
"Accessor to the distribution's position parameter :math:`\\mu`.

Returns
-------
mu : float
    Position parameter :math:`\\mu`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::getSigma
"Accessor to the distribution's scale parameter :math:`\\sigma`.

Returns
-------
sigma : float
    Scale parameter :math:`\\sigma`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::getXi
"Accessor to the distribution's shape parameter :math:`\\xi`.

Returns
-------
xi : float
    Shape parameter :math:`\\xi`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::setActualDistribution
"Accessor to the internal distribution.

Parameters
----------
distribution : :class:`~openturns.Distribution`
     The actual distribution in charge of the computation (:class:`~openturns.WeibullMax`, :class:`~openturns.Frechet`, :class:`~openturns.Gumbel`)."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::setMu
"Accessor to the distribution's position parameter :math:`\\mu`.

Parameters
----------
mu : float
    Position parameter :math:`\\mu`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::setSigma
"Accessor to the distribution's scale parameter :math:`\\sigma`.

Parameters
----------
sigma : float, :math:`\\sigma > 0`
    Scale parameter :math:`\\sigma`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::setXi
"Accessor to the distribution's shape parameter :math:`\\xi`.

Parameters
----------
xi : float, :math:`\\xi \\in \\Rset`
    Shape parameter :math:`\\xi`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::computeReturnLevel
"Accessor to the return level.

Returns
-------
q : float, quantile
    Return level computed as the quantile of level :math:`1-1/m`."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExtremeValue::drawReturnLevel
"Plot the return level.

Returns
-------
graph : :class:`~openturns.Graph`
    Return level graph.

Notes
-----
The return level plot bounds are defined by the :class:`~openturns.ResourceMap`
entries `GeneralizedExtremeValue-MMin` and `GeneralizedExtremeValue-MMax`"
