%feature("docstring") OT::DickeyFullerTest
"The Dickey-Fuller stationarity test.

Refer to :ref:`dickey_fuller`.

Notes
-----
The Dickey-Fuller test checks the stationarity of a scalar time series using one time series.


Examples
--------
Create an ARMA process and generate a time series:

>>> import openturns as ot
>>> arcoefficients = ot.ARMACoefficients([0.3])
>>> macoefficients = ot.ARMACoefficients(0)
>>> timeGrid = ot.RegularGrid(0.0, 0.1, 10)
>>> whiteNoise = ot.WhiteNoise(ot.Normal(), timeGrid)
>>> myARMA = ot.ARMA(arcoefficients, macoefficients, whiteNoise)

>>> realization = ot.TimeSeries(myARMA.getRealization())
>>> test = ot.DickeyFullerTest(realization)

Test the stationarity of the data without any asumption on the model:

>>> globalRes = test.runStrategy()

Test the stationarity knowing you have a drift and linear trend model:

>>> res1 = test.testUnitRootInDriftAndLinearTrendModel(0.95)

Test the stationarity knowing you have a drift model:

>>> res2 = test.testUnitRootInDriftModel(0.95)
 
Test the stationarity knowing you have an AR1 model:

>>> res3 = test.testUnitRootInAR1Model(0.95)"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::testUnitRootInDriftAndLinearTrendModel
"Test for unit root in model with drift and trend.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestModel1`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::testUnitRootInDriftModel
"Test for unit root in model with drift.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestModel2`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::testUnitRootInAR1Model
"Test for unit root in AR1 model.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestModel3`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::DickeyFullerTest::testUnitRootAndNoLinearTrendInDriftAndLinearTrendModel
"Test for linear trend in model with unit root.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestSousModele1_2`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::testNoUnitRootAndNoLinearTrendInDriftAndLinearTrendModel
"Test for trend in model without unit root.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestSousModele1_1`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::testUnitRootAndNoDriftInDriftModel
"Test for null drift in model with unit root.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestSousModele2_2`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::testNoUnitRootAndNoDriftInDriftModel
"Test for null drift in model without unit root.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the test detailed in :eq:`TestSousModele2_1`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::DickeyFullerTest::runStrategy
"Test the stationarity without any assumption on the model.

Parameters
----------
alpha : float, :math:`0 < \\alpha < 1`
    The first kind risk of the test.

    By default, :math:`\\alpha=0.05`.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Results container of the tests. The strategy if the one described above.
"
