/*
 * Decompiled with CFR 0.152.
 */
package io.github.dsheirer.record.wave;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.sound.sampled.AudioFormat;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WaveWriter
implements AutoCloseable {
    private static final Logger mLog = LoggerFactory.getLogger(WaveWriter.class);
    public static final String RIFF_ID = "RIFF";
    public static final int INITIAL_TOTAL_LENGTH = 4;
    public static final String WAVE_ID = "WAVE";
    public static final String FORMAT_CHUNK_ID = "fmt ";
    public static final int FORMAT_CHUNK_LENGTH = 16;
    public static final short FORMAT_UNCOMPRESSED_PCM = 1;
    public static final String DATA_CHUNK_ID = "data";
    private static final Pattern FILENAME_PATTERN = Pattern.compile("(.*_)(\\d+)(\\.tmp)");
    public static final long MAX_WAVE_SIZE = 0xFFFFFFFEL;
    private AudioFormat mAudioFormat;
    private int mFileRolloverCounter = 1;
    private long mMaxSize;
    private Path mFile;
    private FileChannel mFileChannel;
    private boolean mDataChunkOpen = false;
    private long mDataChunkSizeOffset = 0L;
    private int mDataChunkSize = 0;

    public WaveWriter(AudioFormat format, Path file, long maxSize) throws IOException {
        Validate.isTrue((format != null ? 1 : 0) != 0);
        Validate.isTrue((file != null ? 1 : 0) != 0);
        this.mAudioFormat = format;
        this.mFile = file;
        this.mMaxSize = 0L < maxSize && maxSize <= 0xFFFFFFFEL ? maxSize : 0xFFFFFFFEL;
        this.open();
    }

    public WaveWriter(AudioFormat format, Path file) throws IOException {
        this(format, file, 0L);
    }

    private void open() throws IOException {
        int version;
        for (version = 2; Files.exists(this.mFile, new LinkOption[0]) && version < 20; ++version) {
            this.mFile = Paths.get(this.mFile.toFile().getAbsolutePath().replace(".tmp", "_" + version + ".tmp"), new String[0]);
        }
        if (version >= 20) {
            throw new IOException("Unable to create a unique file name for recording - exceeded 20 versioning attempts");
        }
        this.mFileChannel = FileChannel.open(this.mFile, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
        ByteBuffer header = WaveWriter.getWaveHeader(this.mAudioFormat);
        while (header.hasRemaining()) {
            this.mFileChannel.write(header);
        }
    }

    @Override
    public void close() throws IOException {
        this.close(null);
    }

    public void close(Path path) throws IOException {
        this.mFileChannel.force(true);
        this.mFileChannel.close();
        this.rename(path);
    }

    private void rename(Path path) throws IOException {
        if (this.mFile != null && Files.exists(this.mFile, new LinkOption[0]) && path != null) {
            if (Files.exists(path, new LinkOption[0])) {
                mLog.warn("Duplicate recording file detected - ignoring [" + String.valueOf(path) + "]");
                Files.delete(this.mFile);
            } else {
                Files.move(this.mFile, path, new CopyOption[0]);
            }
        }
    }

    public void writeData(ByteBuffer buffer) throws IOException {
        buffer.position(0);
        this.openDataChunk();
        if (this.mFileChannel.size() + (long)buffer.capacity() < this.mMaxSize) {
            while (buffer.hasRemaining()) {
                this.mDataChunkSize += this.mFileChannel.write(buffer);
            }
            this.updateTotalSize();
            this.updateDataChunkSize();
        } else {
            int remaining = (int)(this.mMaxSize - this.mFileChannel.size());
            remaining -= remaining % this.mAudioFormat.getFrameSize();
            byte[] bytes = buffer.array();
            ByteBuffer current = ByteBuffer.wrap(Arrays.copyOf(bytes, remaining));
            ByteBuffer next = ByteBuffer.wrap(Arrays.copyOfRange(bytes, remaining, bytes.length));
            while (current.hasRemaining()) {
                this.mDataChunkSize += this.mFileChannel.write(current);
            }
            this.updateTotalSize();
            this.updateDataChunkSize();
            this.rollover();
            this.openDataChunk();
            while (next.hasRemaining()) {
                this.mDataChunkSize += this.mFileChannel.write(next);
            }
            this.updateTotalSize();
            this.updateDataChunkSize();
        }
    }

    private void closeDataChunk() {
        this.mDataChunkOpen = false;
    }

    private void openDataChunk() throws IOException {
        if (!this.mDataChunkOpen) {
            if (this.mFileChannel.size() + 32L >= this.mMaxSize) {
                this.rollover();
            }
            ByteBuffer formatChunk = WaveWriter.getFormatChunk(this.mAudioFormat);
            formatChunk.position(0);
            while (formatChunk.hasRemaining()) {
                this.mFileChannel.write(formatChunk);
            }
            ByteBuffer dataHeader = WaveWriter.getDataHeader();
            dataHeader.position(0);
            while (dataHeader.hasRemaining()) {
                this.mFileChannel.write(dataHeader);
            }
            this.mDataChunkSizeOffset = this.mFileChannel.size() - 4L;
            this.mDataChunkSize = 0;
            this.mDataChunkOpen = true;
            this.updateTotalSize();
        }
    }

    public void writeMetadata(ByteBuffer listChunk, ByteBuffer id3Chunk) throws IOException {
        if (this.mFileChannel.size() + (long)listChunk.capacity() >= this.mMaxSize) {
            throw new IOException("Cannot write LIST metadata chunk - insufficient file space remaining");
        }
        this.closeDataChunk();
        listChunk.position(0);
        while (listChunk.hasRemaining()) {
            this.mFileChannel.write(listChunk);
        }
        this.updateTotalSize();
        if (this.mFileChannel.size() + (long)id3Chunk.capacity() >= this.mMaxSize) {
            throw new IOException("Cannot write ID3 metadata chunk - insufficient file space remaining");
        }
        id3Chunk.position(0);
        while (id3Chunk.hasRemaining()) {
            this.mFileChannel.write(id3Chunk);
        }
        this.updateTotalSize();
    }

    private void rollover() throws IOException {
        this.closeDataChunk();
        this.close();
        ++this.mFileRolloverCounter;
        this.updateFileName();
        this.open();
    }

    private void updateTotalSize() throws IOException {
        ByteBuffer buffer = WaveWriter.getUnsignedIntegerBuffer(this.mFileChannel.size() - 8L);
        this.mFileChannel.write(buffer, 4L);
    }

    private void updateDataChunkSize() throws IOException {
        if (!this.mDataChunkOpen) {
            throw new IOException("Can't update data chunk size - data chunk is not currently open");
        }
        ByteBuffer size = WaveWriter.getUnsignedIntegerBuffer(this.mDataChunkSize);
        this.mFileChannel.write(size, this.mDataChunkSizeOffset);
    }

    protected static ByteBuffer getUnsignedIntegerBuffer(long size) {
        ByteBuffer buffer = ByteBuffer.allocate(4);
        buffer.put((byte)(size & 0xFFL));
        buffer.put((byte)Long.rotateRight(size & 0xFF00L, 8));
        buffer.put((byte)Long.rotateRight(size & 0xFF0000L, 16));
        buffer.put((byte)Long.rotateRight(Long.rotateRight(size & 0xFF000000L, 16), 8));
        buffer.position(0);
        return buffer;
    }

    public static String toString(ByteBuffer buffer) {
        byte[] bytes;
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes = buffer.array()) {
            sb.append(String.format("%02X ", b));
            sb.append(" ");
        }
        return sb.toString();
    }

    private void updateFileName() {
        String filename = this.mFile.toString();
        if (this.mFileRolloverCounter == 2) {
            filename = filename.replace(".tmp", "_2.tmp");
        } else {
            Matcher m = FILENAME_PATTERN.matcher(filename);
            if (m.find()) {
                StringBuilder sb = new StringBuilder();
                sb.append(m.group(1));
                sb.append(this.mFileRolloverCounter);
                sb.append(m.group(3));
                filename = sb.toString();
            }
        }
        this.mFile = Paths.get(filename, new String[0]);
    }

    public static ByteBuffer getDataHeader() {
        ByteBuffer header = ByteBuffer.allocate(8);
        header.put(DATA_CHUNK_ID.getBytes());
        header.position(0);
        return header;
    }

    public static ByteBuffer getWaveHeader(AudioFormat format) {
        ByteBuffer header = ByteBuffer.allocate(12).order(ByteOrder.LITTLE_ENDIAN);
        header.put(RIFF_ID.getBytes());
        header.putInt(4);
        header.put(WAVE_ID.getBytes());
        header.position(0);
        return header;
    }

    public static ByteBuffer getFormatChunk(AudioFormat format) {
        ByteBuffer header = ByteBuffer.allocate(24).order(ByteOrder.LITTLE_ENDIAN);
        header.put(FORMAT_CHUNK_ID.getBytes());
        header.putInt(16);
        header.putShort((short)1);
        header.putShort((short)format.getChannels());
        header.putInt((int)format.getSampleRate());
        int frameByteRate = format.getChannels() * format.getSampleSizeInBits() / 8;
        int byteRate = (int)(format.getSampleRate() * (float)frameByteRate);
        header.putInt(byteRate);
        header.putShort((short)frameByteRate);
        header.putShort((short)format.getSampleSizeInBits());
        header.position(0);
        return header;
    }
}

