/*
 * Decompiled with CFR 0.152.
 */
package io.github.dsheirer.dsp.filter.fir.remez;

import io.github.dsheirer.dsp.filter.design.FilterDesignException;
import io.github.dsheirer.dsp.filter.fir.FIRFilterSpecification;
import io.github.dsheirer.dsp.filter.fir.remez.Grid;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.math3.util.FastMath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RemezFIRFilterDesigner {
    private static final Logger mLog = LoggerFactory.getLogger(RemezFIRFilterDesigner.class);
    private static final double CONVERGENCE_THRESHOLD = 1.0E-4;
    public static final int MAXIMUM_ITERATION_COUNT = 40;
    public static final double TWO_PI = Math.PI * 2;
    private static final DecimalFormat DECIMAL_FORMATTER = new DecimalFormat("0.00000000");
    private FIRFilterSpecification mSpecification;
    private Grid mGrid;
    private List<Integer> mExtremalIndices;
    private double[] mD;
    private double[] mGridErrors;
    private double[] mGridFrequencyResponse;
    private double[] mIdealFrequencyResponse;
    private double mDelta;
    private boolean mConverged;

    public RemezFIRFilterDesigner(FIRFilterSpecification specification) {
        this.mSpecification = specification;
        this.design();
    }

    private void design() {
        this.mGrid = new Grid(this.mSpecification);
        this.mExtremalIndices = this.getInitialExtremalIndices();
        int iterationCount = 0;
        try {
            do {
                this.calculateGridFrequencyResponse();
                this.calculateGridError();
                this.findExtremalIndices();
                this.checkConvergence();
            } while (!this.mConverged && ++iterationCount < 40);
        }
        catch (FilterDesignException fde) {
            mLog.error("Filter design error - couldn't find extremal indices at count [" + iterationCount + "] try changing filter order up/down from [" + this.mSpecification.getOrder() + "]");
            this.mConverged = false;
        }
        if (this.mConverged) {
            this.calculateGridFrequencyResponse();
        }
    }

    public boolean isValid() {
        return this.mConverged;
    }

    public double[] getFrequencyResponse() throws FilterDesignException {
        if (!this.mConverged) {
            throw new FilterDesignException("Can't create filter from specification - failed to converge");
        }
        double[] resampled = this.resample();
        return this.correctFrequencyResponse(resampled);
    }

    public float[] getImpulseResponse() throws FilterDesignException {
        return RemezFIRFilterDesigner.convertToFloatArray(this.getImpulseResponseDoubles());
    }

    public double[] getImpulseResponseDoubles() throws FilterDesignException {
        double[] frequencyResponse = this.getFrequencyResponse();
        int length = this.mSpecification.getFilterLength();
        double[] impulseResponse = new double[length];
        switch (this.mSpecification.getFilterType()) {
            case TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL: {
                double M = ((double)length - 1.0) / 2.0;
                for (int n = 0; n < length; ++n) {
                    double accumulator = frequencyResponse[0];
                    double frequency = Math.PI * 2 * ((double)n - M) / (double)length;
                    int k = 1;
                    while ((double)k <= M) {
                        accumulator += 2.0 * frequencyResponse[k] * FastMath.cos((double)(frequency * (double)k));
                        ++k;
                    }
                    impulseResponse[n] = accumulator / (double)length;
                }
                break;
            }
            case TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL: {
                double offset = (double)(length - 1) / 2.0;
                for (int n = 0; n < length; ++n) {
                    double accumulator = frequencyResponse[0];
                    double frequency = Math.PI * 2 * ((double)n - offset) / (double)length;
                    for (int k = 1; k < frequencyResponse.length; ++k) {
                        accumulator += 2.0 * frequencyResponse[k] * FastMath.cos((double)(frequency * (double)k));
                    }
                    impulseResponse[n] = accumulator / (double)length;
                }
                break;
            }
        }
        return impulseResponse;
    }

    private double getFrequencyResponse(double cosineOfFrequency) {
        double numerator = 0.0;
        double denominator = 0.0;
        for (int k = 0; k < this.mExtremalIndices.size() - 1; ++k) {
            double cosineDelta = cosineOfFrequency - this.mGrid.getCosineFrequencyGrid()[this.mExtremalIndices.get(k)];
            if (FastMath.abs((double)cosineDelta) < 1.0E-7) {
                return this.mIdealFrequencyResponse[k];
            }
            double dkOverCosineDelta = this.mD[k] / cosineDelta;
            numerator += dkOverCosineDelta * this.mIdealFrequencyResponse[k];
            denominator += dkOverCosineDelta;
        }
        return numerator / denominator;
    }

    public double getDelta() {
        return this.mDelta;
    }

    private List<Integer> getInitialExtremalIndices() {
        int count = this.mSpecification.getExtremaCount();
        int density = this.mSpecification.getGridDensity();
        ArrayList<Integer> extremalIndices = new ArrayList<Integer>();
        for (int i = 0; i < count; ++i) {
            extremalIndices.add(i * density);
        }
        return extremalIndices;
    }

    private void calculateGridFrequencyResponse() {
        double[] b = this.calculateB();
        this.calculateDelta(b);
        this.calculateC();
        this.calculateD(b);
        this.updateGridFrequencyResponse();
    }

    private void updateGridFrequencyResponse() {
        this.mGridFrequencyResponse = new double[this.mGrid.getSize()];
        double[] gridFrequencyCosines = this.mGrid.getCosineFrequencyGrid();
        for (int i = 0; i < this.mGridFrequencyResponse.length; ++i) {
            this.mGridFrequencyResponse[i] = this.getFrequencyResponse(gridFrequencyCosines[i]);
        }
    }

    private double[] calculateB() {
        int length = this.mExtremalIndices.size();
        double[] b = new double[length];
        for (int k = 0; k < length; ++k) {
            b[k] = 1.0;
            double xk = this.mGrid.getCosineFrequencyGrid()[this.mExtremalIndices.get(k)];
            for (int i = 0; i < length; ++i) {
                if (i == k) continue;
                double xi = this.mGrid.getCosineFrequencyGrid()[this.mExtremalIndices.get(i)];
                double denominator = xk - xi;
                if (FastMath.abs((double)denominator) < 1.0E-5) {
                    denominator = 1.0E-5;
                }
                int n = k;
                b[n] = b[n] * (1.0 / denominator);
            }
        }
        return b;
    }

    private void calculateDelta(double[] b) {
        double numerator = 0.0;
        double denominator = 0.0;
        double sign = 1.0;
        for (int k = 0; k < b.length; ++k) {
            if (k < this.mExtremalIndices.size()) {
                int extremalIndex = this.mExtremalIndices.get(k);
                numerator += b[k] * this.mGrid.getDesiredResponse()[extremalIndex];
                denominator += b[k] * sign / this.mGrid.getWeight()[extremalIndex];
                sign = -sign;
                continue;
            }
            mLog.error("Something went wrong -- the length of b exceeds the set of extremal indices");
        }
        this.mDelta = numerator / denominator;
    }

    private void calculateC() {
        int length = this.mSpecification.getExtremaCount() - 1;
        this.mIdealFrequencyResponse = new double[length];
        double sign = 1.0;
        for (int k = 0; k < length; ++k) {
            if (k >= this.mExtremalIndices.size()) continue;
            int index = this.mExtremalIndices.get(k);
            this.mIdealFrequencyResponse[k] = this.mGrid.getDesiredResponse()[index] - sign * this.mDelta / this.mGrid.getWeight()[index];
            sign = -sign;
        }
    }

    private void calculateD(double[] b) {
        int length = this.mExtremalIndices.size() - 1;
        this.mD = new double[length];
        for (int k = 0; k < length; ++k) {
            this.mD[k] = b[k] * (this.mGrid.getCosineFrequencyGrid()[this.mExtremalIndices.get(k)] - this.mGrid.getCosineFrequencyGrid()[this.mExtremalIndices.get(length)]);
        }
    }

    public void calculateGridError() {
        int length = this.mGridFrequencyResponse.length;
        this.mGridErrors = new double[length];
        for (int i = 0; i < length; ++i) {
            this.mGridErrors[i] = this.mGrid.getWeight()[i] * (this.mGrid.getDesiredResponse()[i] - this.mGridFrequencyResponse[i]);
        }
    }

    private void findExtremalIndices() throws FilterDesignException {
        boolean positiveAxis;
        this.mExtremalIndices.clear();
        if ((this.mGridErrors[0] > 0.0 && this.mGridErrors[0] > this.mGridErrors[1] || this.mGridErrors[0] < 0.0 && this.mGridErrors[0] < this.mGridErrors[1]) && this.isGTEDelta(this.mGridErrors[0])) {
            this.mExtremalIndices.add(0);
        }
        for (int x = 1; x < this.mGridErrors.length - 1; ++x) {
            if (!(this.mGridErrors[x] > 0.0 && this.mGridErrors[x - 1] <= this.mGridErrors[x] && this.mGridErrors[x] > this.mGridErrors[x + 1]) && (!(this.mGridErrors[x] < 0.0) || !(this.mGridErrors[x - 1] >= this.mGridErrors[x]) || !(this.mGridErrors[x] < this.mGridErrors[x + 1])) || !this.isGTEDelta(this.mGridErrors[x])) continue;
            this.mExtremalIndices.add(x);
        }
        int last = this.mGridErrors.length - 1;
        if ((this.mGridErrors[last] > 0.0 && this.mGridErrors[last] > this.mGridErrors[last - 1] || this.mGridErrors[last] < 0.0 && this.mGridErrors[last] < this.mGridErrors[last - 1]) && this.isGTEDelta(this.mGridErrors[last])) {
            this.mExtremalIndices.add(last);
        }
        if (this.mExtremalIndices.size() < this.mSpecification.getExtremaCount()) {
            throw new FilterDesignException("Couldn't find the minimum extremal frequencies in error set");
        }
        ArrayList<Integer> indicesToRemove = new ArrayList<Integer>();
        Iterator<Integer> it = this.mExtremalIndices.iterator();
        Integer current = it.next();
        boolean bl = positiveAxis = this.mGridErrors[current] > 0.0;
        while (it.hasNext()) {
            Integer next = it.next();
            if (!(positiveAxis ^ this.mGridErrors[next] > 0.0)) {
                if (FastMath.abs((double)this.mGridErrors[next]) <= FastMath.abs((double)this.mGridErrors[current])) {
                    it.remove();
                    next = current;
                } else {
                    indicesToRemove.add(current);
                }
            } else {
                positiveAxis = !positiveAxis;
            }
            current = next;
        }
        this.mExtremalIndices.removeAll(indicesToRemove);
        while (this.mExtremalIndices.size() > this.mSpecification.getExtremaCount()) {
            this.mExtremalIndices.remove(this.mExtremalIndices.size() - 1);
        }
        if (this.mExtremalIndices.size() > this.mSpecification.getExtremaCount()) {
            int lastIndex = this.mExtremalIndices.size() - 1;
            if (FastMath.abs((double)this.mGridErrors[this.mExtremalIndices.get(0)]) > FastMath.abs((double)this.mGridErrors[this.mExtremalIndices.get(lastIndex)])) {
                this.mExtremalIndices.remove(lastIndex);
            } else {
                this.mExtremalIndices.remove(0);
            }
        }
        if (this.mExtremalIndices.size() < this.mSpecification.getExtremaCount()) {
            throw new FilterDesignException("Couldn't find the minimum extremal frequencies in error set");
        }
    }

    private boolean isGTEDelta(double value) {
        return FastMath.abs((double)value) - FastMath.abs((double)this.mDelta) > -1.0E-5;
    }

    public void checkConvergence() {
        double maximum = FastMath.abs((double)this.mGridErrors[this.mExtremalIndices.get(0)]);
        for (int i = 1; i < this.mExtremalIndices.size(); ++i) {
            double current = FastMath.abs((double)this.mGridErrors[this.mExtremalIndices.get(i)]);
            if (!(current > maximum)) continue;
            maximum = current;
        }
        double convergence = maximum - FastMath.abs((double)this.mDelta);
        this.mConverged = convergence < 1.0E-4;
    }

    private static float[] convertToFloatArray(double[] samples) {
        float[] converted = new float[samples.length];
        for (int x = 0; x < samples.length; ++x) {
            converted[x] = (float)samples[x];
        }
        return converted;
    }

    private double[] resample() {
        int length = this.mSpecification.getFilterLength();
        if (length % 2 == 0) {
            --length;
        }
        double half = (double)length / 2.0;
        double[] resampled = new double[(int)FastMath.ceil((double)half)];
        for (int x = 0; x < resampled.length; ++x) {
            resampled[x] = this.getFrequencyResponse(FastMath.cos((double)(Math.PI * (double)x / half)));
        }
        return resampled;
    }

    private double[] correctFrequencyResponse(double[] frequencyResponse) {
        double filterLength = this.mSpecification.getFilterLength();
        switch (this.mSpecification.getFilterType()) {
            case TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL: {
                break;
            }
            case TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL: {
                break;
            }
            case TYPE_3_ODD_LENGTH_EVEN_ORDER_ANTI_SYMMETRICAL: {
                int x = 0;
                while (x < frequencyResponse.length) {
                    double frequencyRadians = Math.PI * ((double)x / filterLength);
                    int n = x++;
                    frequencyResponse[n] = frequencyResponse[n] * FastMath.sin((double)(Math.PI * 2 * frequencyRadians));
                }
                break;
            }
            case TYPE_4_EVEN_LENGTH_ODD_ORDER_ANTI_SYMMETRICAL: {
                int x = 0;
                while (x < frequencyResponse.length) {
                    double frequencyRadians = Math.PI * ((double)x / filterLength);
                    int n = x++;
                    frequencyResponse[n] = frequencyResponse[n] * FastMath.sin((double)(Math.PI * frequencyRadians));
                }
                break;
            }
        }
        return frequencyResponse;
    }
}

