/*
 * Decompiled with CFR 0.152.
 */
package io.github.dsheirer.dsp.filter.fir;

import io.github.dsheirer.dsp.filter.fir.remez.FIRLinearPhaseFilterType;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.apache.commons.lang3.Validate;
import org.apache.commons.math3.util.FastMath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FIRFilterSpecification {
    private static final Logger mLog = LoggerFactory.getLogger(FIRFilterSpecification.class);
    private static final double PERFECT_RECONSTRUCTION_GAIN_AT_BAND_EDGE = (double)-6.0206f;
    private static final DecimalFormat DECIMAL_FORMATTER = new DecimalFormat("0.00000");
    private FIRLinearPhaseFilterType mRemezFilterType;
    private int mOrder;
    private int mGridDensity = 16;
    private List<FrequencyBand> mFrequencyBands = new ArrayList<FrequencyBand>();

    protected FIRFilterSpecification(FIRLinearPhaseFilterType type, int order, int gridDensity) {
        this.mRemezFilterType = type;
        this.mOrder = order;
        this.mGridDensity = gridDensity;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Type: ");
        sb.append(this.mRemezFilterType.name());
        sb.append(" Order: ");
        sb.append(this.mOrder);
        sb.append(" Extrema: ");
        sb.append(this.getExtremaCount());
        sb.append(" Grid Density: ");
        sb.append(this.mGridDensity);
        sb.append(" Grid Size: ");
        sb.append(this.getGridSize());
        for (FrequencyBand band : this.mFrequencyBands) {
            sb.append("\n");
            sb.append(band.toString());
        }
        return sb.toString();
    }

    public void addFrequencyBand(FrequencyBand band) {
        this.mFrequencyBands.add(band);
        this.updateGridSize();
    }

    public void addFrequencyBands(Collection<FrequencyBand> bands) {
        this.mFrequencyBands.addAll(bands);
        this.updateGridSize();
    }

    public void clearFrequencyBands() {
        this.mFrequencyBands.clear();
    }

    public List<FrequencyBand> getFrequencyBands() {
        return this.mFrequencyBands;
    }

    public double getTotalBandwidth() {
        double bandwidth = 0.0;
        for (FrequencyBand band : this.mFrequencyBands) {
            bandwidth += band.getBandWidth();
        }
        return bandwidth;
    }

    public static LowPassBuilder lowPassBuilder() {
        return new LowPassBuilder();
    }

    public static HighPassBuilder highPassBuilder() {
        return new HighPassBuilder();
    }

    public static BandPassBuilder bandPassBuilder() {
        return new BandPassBuilder();
    }

    public static ChannelizerBuilder channelizerBuilder() {
        return new ChannelizerBuilder();
    }

    public FIRLinearPhaseFilterType getFilterType() {
        return this.mRemezFilterType;
    }

    public boolean isSymmetrical() {
        return this.mRemezFilterType == FIRLinearPhaseFilterType.TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL || this.mRemezFilterType == FIRLinearPhaseFilterType.TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL;
    }

    public boolean isAntiSymmetrical() {
        return this.mRemezFilterType == FIRLinearPhaseFilterType.TYPE_3_ODD_LENGTH_EVEN_ORDER_ANTI_SYMMETRICAL || this.mRemezFilterType == FIRLinearPhaseFilterType.TYPE_4_EVEN_LENGTH_ODD_ORDER_ANTI_SYMMETRICAL;
    }

    public int getOrder() {
        return this.mOrder;
    }

    public boolean isEvenOrder() {
        return this.mOrder % 2 == 0;
    }

    public boolean isOddOrder() {
        return this.mOrder % 2 == 1;
    }

    public int getFilterLength() {
        return this.mOrder + 1;
    }

    public int getBandCount() {
        return this.mFrequencyBands.size();
    }

    public int getGridDensity() {
        return this.mGridDensity;
    }

    public void setGridDensity(int density) {
        this.mGridDensity = density;
        this.updateGridSize();
    }

    public int getExtremaCount() {
        return this.getHalfFilterOrder() + 2;
    }

    public int getHalfFilterOrder() {
        switch (this.mRemezFilterType) {
            case TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL: {
                return this.mOrder / 2;
            }
            case TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL: {
                return (this.mOrder - 1) / 2;
            }
            case TYPE_3_ODD_LENGTH_EVEN_ORDER_ANTI_SYMMETRICAL: {
                return (this.mOrder - 2) / 2;
            }
            case TYPE_4_EVEN_LENGTH_ODD_ORDER_ANTI_SYMMETRICAL: {
                return (this.mOrder - 1) / 2;
            }
        }
        return 0;
    }

    public double getMaxBandAmplitude() {
        double maxRippleAmplitude = 0.0;
        for (FrequencyBand band : this.mFrequencyBands) {
            double bandRippleAmplitude = band.getRippleAmplitude();
            if (!(bandRippleAmplitude > maxRippleAmplitude)) continue;
            maxRippleAmplitude = bandRippleAmplitude;
        }
        return maxRippleAmplitude;
    }

    public int getNominalGridSize() {
        return (this.getExtremaCount() - 1) * this.getGridDensity() + 1;
    }

    public int getGridSize() {
        int gridSize = 0;
        for (FrequencyBand band : this.mFrequencyBands) {
            gridSize += band.getGridSize();
        }
        return gridSize;
    }

    private void updateGridSize() {
        if (!this.mFrequencyBands.isEmpty()) {
            int gridSize = this.getNominalGridSize();
            double totalBandwidth = this.getTotalBandwidth();
            for (FrequencyBand band : this.mFrequencyBands) {
                band.setGridSize(gridSize, totalBandwidth);
            }
        }
    }

    public double getGridFrequencyInterval() {
        return this.getTotalBandwidth() / (double)(this.getGridSize() - this.mFrequencyBands.size());
    }

    private static int getFilterOrder(double length) {
        return (int)(FastMath.ceil((double)length) - 1.0);
    }

    public static int estimateFilterOrder(double sampleRate, double frequency1, double frequency2, double passBandRipple, double stopBandRipple) {
        double df = FastMath.abs((double)(frequency2 - frequency1)) / sampleRate;
        double ddp = FastMath.log10((double)FastMath.max((double)stopBandRipple, (double)passBandRipple));
        double dds = FastMath.log10((double)FastMath.min((double)stopBandRipple, (double)passBandRipple));
        double a1 = 0.005309;
        double a2 = 0.07114;
        double a3 = -0.4761;
        double a4 = -0.00266;
        double a5 = -0.5941;
        double a6 = -0.4278;
        double b1 = 11.01217;
        double b2 = 0.5124401;
        double t1 = a1 * ddp * ddp;
        double t2 = a2 * ddp;
        double t3 = a4 * ddp * ddp;
        double t4 = a5 * ddp;
        double dinf = (t1 + t2 + a3) * dds + (t3 + t4 + a6);
        double ff = b1 + b2 * (ddp - dds);
        double n = dinf / df - ff * df + 1.0;
        return (int)FastMath.ceil((double)n);
    }

    public static int estimateBandPassOrder(double sampleRate, int passBandStart, int passBandEnd, double passBandRippleDb, double stopBandRippleDb) {
        double df = (double)FastMath.abs((int)(passBandEnd - passBandStart)) / sampleRate;
        double ddp = FastMath.log10((double)passBandRippleDb);
        double dds = FastMath.log10((double)stopBandRippleDb);
        double a1 = 0.01201;
        double a2 = 0.09664;
        double a3 = -0.51325;
        double a4 = 0.00203;
        double a5 = -0.57054;
        double a6 = -0.44314;
        double t1 = a1 * ddp * ddp;
        double t2 = a2 * ddp;
        double t3 = a4 * ddp * ddp;
        double t4 = a5 * ddp;
        double cinf = dds * (t1 + t2 + a3) + t3 + t4 + a6;
        double ginf = (double)-14.6f * FastMath.log10((double)(passBandRippleDb / stopBandRippleDb)) - 16.9;
        double n = cinf / df + ginf * df + 1.0;
        return (int)FastMath.ceil((double)n);
    }

    public static class FrequencyBand {
        private int mGridSize;
        private double mStart;
        private double mEnd;
        private double mAmplitude;
        private double mRippleDB;
        private Double mWeight;

        public FrequencyBand(double start, double end, double amplitude, double ripple) {
            assert (0.0 <= this.mStart);
            assert (this.mStart <= this.mEnd);
            assert (this.mEnd <= 1.0);
            assert (0.0 <= amplitude && amplitude <= 1.0);
            assert (0.0 <= ripple);
            this.mStart = start;
            this.mEnd = end;
            this.mAmplitude = amplitude;
            this.mRippleDB = ripple;
        }

        public FrequencyBand(double sampleRate, double start, double end, double amplitude, double ripple) {
            this(start / sampleRate, end / sampleRate, amplitude, ripple);
        }

        public FrequencyBand(double sampleRate, double start, double end, double amplitude, double ripple, double weight) {
            this(sampleRate, start, end, amplitude, ripple);
            this.mWeight = weight;
        }

        public void setWeight(double weight) {
            this.mWeight = weight;
        }

        public int getGridSize() {
            return this.mGridSize;
        }

        public void setGridSize(int totalGridSize, double totalBandwidth) {
            this.mGridSize = FastMath.max((int)1, (int)((int)FastMath.ceil((double)((double)totalGridSize * (this.getBandWidth() / totalBandwidth)))));
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("Band Start: ");
            sb.append(this.mStart);
            sb.append(" End:");
            sb.append(this.mEnd);
            sb.append(" Amplitude: ");
            sb.append(this.mAmplitude);
            sb.append(" Ripple dB: ");
            sb.append(this.mRippleDB);
            sb.append(" Ripple Amplitude: ");
            sb.append(DECIMAL_FORMATTER.format(this.getRippleAmplitude()));
            sb.append(" Grid Size: ");
            sb.append(this.mGridSize);
            sb.append(" Weight: ");
            sb.append(this.getWeight(this.mRippleDB));
            return sb.toString();
        }

        public double getStart() {
            return this.mStart;
        }

        public double getEnd() {
            return this.mEnd;
        }

        public double getAmplitude() {
            return this.mAmplitude;
        }

        public double getRippleDB() {
            return this.mRippleDB;
        }

        public double getRippleAmplitude() {
            return (FastMath.pow((double)10.0, (double)(this.mRippleDB / 20.0)) - 1.0) / (FastMath.pow((double)10.0, (double)(this.mRippleDB / 20.0)) + 1.0);
        }

        public double getWeight(double maxRippleAmplitude) {
            if (this.mWeight == null) {
                return 1.0 / (this.getRippleAmplitude() / maxRippleAmplitude);
            }
            return this.mWeight;
        }

        public double getBandWidth() {
            return this.mEnd - this.mStart;
        }
    }

    public static class LowPassBuilder {
        private double mSampleRate;
        private int mOrder;
        private Boolean mOddLength;
        private int mGridDensity = 16;
        private double mPassBandEndFrequency;
        private double mStopBandStartFrequency;
        private double mPassBandRipple;
        private double mStopBandRipple;
        private double mPassBandAmplitude = 1.0;
        private double mStopBandAmplitude = 0.0;

        public LowPassBuilder sampleRate(double sampleRateHz) {
            this.mSampleRate = sampleRateHz;
            return this;
        }

        public LowPassBuilder order(int order) {
            this.mOrder = order;
            return this;
        }

        public LowPassBuilder oddLength(boolean oddLength) {
            this.mOddLength = oddLength;
            return this;
        }

        public LowPassBuilder gridDensity(int density) {
            this.mGridDensity = density;
            return this;
        }

        public LowPassBuilder passBandCutoff(double frequencyHz) {
            this.mPassBandEndFrequency = frequencyHz;
            return this;
        }

        public LowPassBuilder stopBandStart(double frequencyHz) {
            this.mStopBandStartFrequency = frequencyHz;
            return this;
        }

        public LowPassBuilder passBandRipple(double rippleDb) {
            this.mPassBandRipple = rippleDb;
            return this;
        }

        public LowPassBuilder stopBandRipple(double rippleDb) {
            this.mStopBandRipple = rippleDb;
            return this;
        }

        public LowPassBuilder passBandAmplitude(double amplitude) {
            this.mPassBandAmplitude = amplitude;
            return this;
        }

        public LowPassBuilder stopBandAmplitude(double amplitude) {
            this.mStopBandAmplitude = amplitude;
            return this;
        }

        public FIRFilterSpecification build() {
            if (this.mOrder < 6) {
                this.mOrder = FIRFilterSpecification.estimateFilterOrder(this.mSampleRate, this.mPassBandEndFrequency, this.mStopBandStartFrequency, this.mPassBandRipple, this.mStopBandRipple);
            }
            FIRLinearPhaseFilterType type = null;
            if (this.mOddLength != null) {
                if (this.mOddLength.booleanValue()) {
                    type = FIRLinearPhaseFilterType.TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL;
                    this.mOrder += this.mOrder % 2;
                } else {
                    type = FIRLinearPhaseFilterType.TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL;
                    this.mOrder += this.mOrder % 2 == 0 ? 1 : 0;
                }
            } else {
                type = this.mOrder % 2 == 0 ? FIRLinearPhaseFilterType.TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL : FIRLinearPhaseFilterType.TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL;
            }
            FIRFilterSpecification spec = new FIRFilterSpecification(type, this.mOrder, this.mGridDensity);
            FrequencyBand passBand = new FrequencyBand(this.mSampleRate, 0.0, this.mPassBandEndFrequency, this.mPassBandAmplitude, this.mPassBandRipple);
            spec.addFrequencyBand(passBand);
            FrequencyBand stopBand = new FrequencyBand(this.mSampleRate, this.mStopBandStartFrequency, (int)(this.mSampleRate / 2.0), this.mStopBandAmplitude, this.mStopBandRipple);
            spec.addFrequencyBand(stopBand);
            return spec;
        }
    }

    public static class HighPassBuilder {
        private double mSampleRate;
        private int mOrder = -1;
        private int mGridDensity = 16;
        private int mPassBandStartFrequency;
        private int mStopBandEndFrequency;
        private double mPassBandRipple;
        private double mStopBandRipple;
        private double mPassBandAmplitude = 1.0;
        private double mStopBandAmplitude = 0.0;

        public HighPassBuilder sampleRate(double sampleRateHz) {
            this.mSampleRate = sampleRateHz;
            return this;
        }

        public HighPassBuilder order(int order) {
            this.mOrder = order;
            return this;
        }

        public HighPassBuilder gridDensity(int density) {
            this.mGridDensity = density;
            return this;
        }

        public HighPassBuilder passBandStart(int frequencyHz) {
            this.mPassBandStartFrequency = frequencyHz;
            return this;
        }

        public HighPassBuilder stopBandCutoff(int frequencyHz) {
            this.mStopBandEndFrequency = frequencyHz;
            return this;
        }

        public HighPassBuilder passBandRipple(double rippleDb) {
            this.mPassBandRipple = rippleDb;
            return this;
        }

        public HighPassBuilder stopBandRipple(double rippleDb) {
            this.mStopBandRipple = rippleDb;
            return this;
        }

        public HighPassBuilder passBandAmplitude(double amplitude) {
            this.mPassBandAmplitude = amplitude;
            return this;
        }

        public HighPassBuilder stopBandAmplitude(double amplitude) {
            this.mStopBandAmplitude = amplitude;
            return this;
        }

        public FIRFilterSpecification build() {
            if (this.mOrder < 6) {
                this.mOrder = FIRFilterSpecification.estimateFilterOrder(this.mSampleRate, this.mStopBandEndFrequency, this.mPassBandStartFrequency, this.mPassBandRipple, this.mStopBandRipple);
            }
            if (this.mOrder % 2 == 1) {
                ++this.mOrder;
            }
            FIRFilterSpecification spec = new FIRFilterSpecification(FIRLinearPhaseFilterType.TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL, this.mOrder, this.mGridDensity);
            FrequencyBand stopBand = new FrequencyBand(this.mSampleRate, 0.0, this.mStopBandEndFrequency, this.mStopBandAmplitude, this.mStopBandRipple);
            spec.addFrequencyBand(stopBand);
            FrequencyBand passBand = new FrequencyBand(this.mSampleRate, this.mPassBandStartFrequency, (int)(this.mSampleRate / 2.0), this.mPassBandAmplitude, this.mPassBandRipple);
            spec.addFrequencyBand(passBand);
            return spec;
        }
    }

    public static class BandPassBuilder {
        private int mOrder;
        private int mGridDensity = 16;
        private double mSampleRate;
        private int mStopFrequency1;
        private int mPassFrequencyBegin;
        private int mPassFrequencyEnd;
        private int mStopFrequency2;
        private double mStopRipple = 0.01;
        private double mPassRipple = 0.01;
        private double mStopAmplitude = 0.0;
        private double mPassAmplitude = 1.0;

        public BandPassBuilder order(int order) {
            this.mOrder = order;
            return this;
        }

        public BandPassBuilder gridDensity(int density) {
            this.mGridDensity = density;
            return this;
        }

        public BandPassBuilder sampleRate(double sampleRate) {
            this.mSampleRate = sampleRate;
            return this;
        }

        public BandPassBuilder stopFrequency1(int stopFrequency) {
            this.mStopFrequency1 = stopFrequency;
            return this;
        }

        public BandPassBuilder stopFrequency2(int stopFrequency) {
            this.mStopFrequency2 = stopFrequency;
            return this;
        }

        public BandPassBuilder passFrequencyBegin(int passFrequency) {
            this.mPassFrequencyBegin = passFrequency;
            return this;
        }

        public BandPassBuilder passFrequencyEnd(int passFrequency) {
            this.mPassFrequencyEnd = passFrequency;
            return this;
        }

        public BandPassBuilder stopRipple(double rippleDb) {
            this.mStopRipple = rippleDb;
            return this;
        }

        public BandPassBuilder passRipple(double rippleDb) {
            this.mPassRipple = rippleDb;
            return this;
        }

        public BandPassBuilder stopAmplitude(double amplitude) {
            this.mStopAmplitude = amplitude;
            return this;
        }

        public BandPassBuilder passAmplitude(double amplitude) {
            this.mPassAmplitude = amplitude;
            return this;
        }

        private void validateFrequencyBands() {
            if (this.mStopFrequency1 >= this.mPassFrequencyBegin) {
                throw new IllegalArgumentException("Stop band 1 ending frequency [" + this.mStopFrequency1 + "] must be less than the pass band start frequency [" + this.mPassFrequencyBegin + "]");
            }
            if (this.mPassFrequencyBegin >= this.mPassFrequencyEnd) {
                throw new IllegalArgumentException("Pass band begin frequency [" + this.mPassFrequencyBegin + "] must be less than the pass band end frequency [" + this.mPassFrequencyEnd + "]");
            }
            if (this.mPassFrequencyEnd >= this.mStopFrequency2) {
                throw new IllegalArgumentException("Pass band end frequency [" + this.mPassFrequencyEnd + "] must be less than stop band 2 beginning frequency [" + this.mStopFrequency2 + "]");
            }
            if ((double)this.mStopFrequency2 >= this.mSampleRate / 2.0) {
                throw new IllegalArgumentException("Stop band 2 beginning frequency [" + this.mStopFrequency2 + "] must be less than half of the sample rate [" + this.mSampleRate / 2.0 + "]");
            }
        }

        public FIRFilterSpecification build() {
            this.validateFrequencyBands();
            FIRLinearPhaseFilterType type = FIRLinearPhaseFilterType.TYPE_1_ODD_LENGTH_EVEN_ORDER_SYMMETRICAL;
            if (this.mOrder < 10) {
                this.mOrder = FIRFilterSpecification.estimateBandPassOrder(this.mSampleRate, this.mPassFrequencyBegin, this.mPassFrequencyEnd, this.mPassRipple, this.mStopRipple);
            }
            if (this.mOrder % 2 == 1) {
                ++this.mOrder;
            }
            FIRFilterSpecification spec = new FIRFilterSpecification(type, this.mOrder, this.mGridDensity);
            spec.addFrequencyBand(new FrequencyBand(this.mSampleRate, 0.0, this.mStopFrequency1, this.mStopAmplitude, this.mStopRipple));
            spec.addFrequencyBand(new FrequencyBand(this.mSampleRate, this.mPassFrequencyBegin, this.mPassFrequencyEnd, this.mPassAmplitude, this.mPassRipple));
            spec.addFrequencyBand(new FrequencyBand(this.mSampleRate, this.mStopFrequency2, (int)(this.mSampleRate / 2.0), this.mStopAmplitude, this.mStopRipple));
            return spec;
        }
    }

    public static class ChannelizerBuilder {
        private int mGridDensity = 16;
        private double mSampleRate;
        private int mChannelCount;
        private int mChannelBandwidth;
        private int mTapsPerChannel;
        private double mStopRipple = 0.001;
        private double mPassRipple = 0.01;
        private double mAlpha = 0.2;

        public ChannelizerBuilder sampleRate(double sampleRate) {
            this.mSampleRate = sampleRate;
            return this;
        }

        public ChannelizerBuilder channels(int channels) {
            Validate.isTrue((channels % 2 == 0 ? 1 : 0) != 0, (String)"Channel count must be a multiple of 2", (Object[])new Object[0]);
            this.mChannelCount = channels;
            return this;
        }

        public ChannelizerBuilder channelBandwidth(int channelBandwidth) {
            this.mChannelBandwidth = channelBandwidth;
            return this;
        }

        public ChannelizerBuilder tapsPerChannel(int tapsPerChannel) {
            this.mTapsPerChannel = tapsPerChannel;
            return this;
        }

        public ChannelizerBuilder gridDensity(int density) {
            this.mGridDensity = density;
            return this;
        }

        public ChannelizerBuilder stopRipple(double rippleDb) {
            this.mStopRipple = rippleDb;
            return this;
        }

        public ChannelizerBuilder passRipple(double rippleDb) {
            this.mPassRipple = rippleDb;
            return this;
        }

        public ChannelizerBuilder alpha(double alpha) {
            Validate.isTrue((0.0 <= alpha && alpha <= 1.0 ? 1 : 0) != 0);
            this.mAlpha = alpha;
            return this;
        }

        public FIRFilterSpecification build() {
            FIRLinearPhaseFilterType type = FIRLinearPhaseFilterType.TYPE_2_EVEN_LENGTH_ODD_ORDER_SYMMETRICAL;
            int passFrequencyEnd = (int)((double)this.mChannelBandwidth * (1.0 - this.mAlpha));
            int stopFrequencyBegin = (int)((double)this.mChannelBandwidth * (1.0 + this.mAlpha));
            int estimatedOrder = FIRFilterSpecification.estimateFilterOrder(this.mSampleRate, passFrequencyEnd, stopFrequencyBegin, this.mPassRipple, this.mStopRipple);
            if (estimatedOrder % 2 == 0) {
                ++estimatedOrder;
            }
            int order = this.mChannelCount * this.mTapsPerChannel - 1;
            mLog.debug("Order:" + order + " Sample Rate:" + this.mSampleRate + " Estimated Order:" + estimatedOrder + " Requested Order:" + order);
            FIRFilterSpecification spec = new FIRFilterSpecification(type, order, this.mGridDensity);
            double bandEdge = (double)this.mChannelBandwidth / this.mSampleRate;
            double passEnd = bandEdge * (1.0 - this.mAlpha);
            double stopStart = bandEdge * (1.0 + this.mAlpha);
            mLog.debug("Pass End:" + passEnd + " Band Edge:" + bandEdge + " Stop Begin:" + stopStart);
            mLog.debug("Pass End:" + (int)(this.mSampleRate * passEnd) + " Band Edge:" + (int)(this.mSampleRate * bandEdge) + " Stop Begin:" + (int)(this.mSampleRate * stopStart));
            FrequencyBand passBand = new FrequencyBand(0.0, passEnd, 1.0, this.mPassRipple);
            FrequencyBand edgeBand = new FrequencyBand(bandEdge, bandEdge, 0.5, this.mPassRipple);
            edgeBand.setWeight(8.0 * edgeBand.getWeight(this.mPassRipple));
            FrequencyBand stopBand = new FrequencyBand(stopStart, 0.5, 0.0, this.mStopRipple);
            spec.addFrequencyBand(passBand);
            spec.addFrequencyBand(edgeBand);
            spec.addFrequencyBand(stopBand);
            mLog.debug(spec.toString());
            return spec;
        }
    }
}

