##############################################################################
#
# Copyright (c) 2006 Zope Foundation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################

import zope.interface


class StaleCache:
    """A ZEO cache is stale and requires verification.
    """

    def __init__(self, storage):
        self.storage = storage


class IClientCache(zope.interface.Interface):
    """Client cache interface.

    Note that caches need to be thread safe.
    """

    def close():
        """Close the cache
        """

    def load(oid):
        """Get current data for object

        Returns data and serial, or None.
        """

    def __len__():
        """Retirn the number of items in the cache.
        """

    def store(oid, start_tid, end_tid, data):
        """Store data for the object

        The start_tid is the transaction that committed this data.

        The end_tid is the tid of the next transaction that modified
        the objects, or None if this is the current version.
        """

    def loadBefore(oid, tid):
        """Load the data for the object last modified before the tid

        Returns the data, and start and end tids.
        """

    def invalidate(oid, tid):
        """Invalidate data for the object

        If ``tid`` is None, forget all knowledge of `oid`.  (``tid``
        can be None only for invalidations generated by startup cache
        verification.)

        If ``tid`` isn't None, and we had current data for ``oid``,
        stop believing we have current data, and mark the data we had
        as being valid only up to `tid`.  In all other cases, do
        nothing.
        """

    def getLastTid():
        """Get the last tid seen by the cache

        This is the cached last tid we've seen from the server.

        This method may be called from multiple threads. (It's assumed
        to be trivial.)
        """

    def setLastTid(tid):
        """Save the last tid sent by the server
        """

    def clear():
        """Clear/empty the cache
        """


class IServeable(zope.interface.Interface):
    """Interface provided by storages that can be served by ZEO
    """

    def tpc_transaction():
        """The current transaction being committed.

        If a storage is participating in a two-phase commit, then
        return the transaction (object) being committed.  Otherwise
        return None.
        """

    def lastInvalidations(size):
        """Get recent transaction invalidations

        This method is optional and is used to get invalidations
        performed by the most recent transactions.

        An iterable of up to size entries must be returned, where each
        entry is a transaction id and a sequence of object-id/empty-string
        pairs describing the objects written by the
        transaction, in chronological order.
        """
