/*
 * Decompiled with CFR 0.152.
 */
package horse.wtf.nzyme.dot11.networks;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import horse.wtf.nzyme.NzymeLeader;
import horse.wtf.nzyme.dot11.Dot11TaggedParameters;
import horse.wtf.nzyme.dot11.frames.Dot11BeaconFrame;
import horse.wtf.nzyme.dot11.frames.Dot11ProbeResponseFrame;
import horse.wtf.nzyme.dot11.networks.BSSID;
import horse.wtf.nzyme.dot11.networks.Channel;
import horse.wtf.nzyme.dot11.networks.SSID;
import horse.wtf.nzyme.dot11.networks.beaconrate.BeaconRateManager;
import horse.wtf.nzyme.dot11.networks.signalstrength.SignalStrengthTable;
import horse.wtf.nzyme.util.MetricNames;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;

public class Networks {
    private static final Logger LOG = LogManager.getLogger(Networks.class);
    private final Map<String, BSSID> bssids;
    private final NzymeLeader nzyme;
    private final BeaconRateManager beaconRateManager;

    public Networks(NzymeLeader nzyme) {
        this.nzyme = nzyme;
        this.bssids = Maps.newConcurrentMap();
        this.beaconRateManager = new BeaconRateManager(nzyme);
        if (!nzyme.getMetrics().getGauges().containsKey(MetricNames.NETWORKS_SIGNAL_STRENGTH_MEASUREMENTS)) {
            nzyme.getMetrics().register(MetricNames.NETWORKS_SIGNAL_STRENGTH_MEASUREMENTS, () -> {
                long result = 0L;
                for (BSSID bssid : this.bssids.values()) {
                    for (SSID ssid : bssid.ssids().values()) {
                        for (Channel channel : ssid.channels().values()) {
                            result += (long)channel.signalStrengthTable().getSize();
                        }
                    }
                }
                return result;
            });
        }
        Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("bssids-cleaner").build()).scheduleAtFixedRate(() -> this.retentionClean(600), 1L, 1L, TimeUnit.MINUTES);
        Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("signalstrengths-cleaner").build()).scheduleAtFixedRate(() -> {
            for (BSSID bssid : this.bssids.values()) {
                for (SSID ssid : bssid.ssids().values()) {
                    for (Channel channel : ssid.channels().values()) {
                        channel.signalStrengthTable().retentionClean((int)TimeUnit.MINUTES.toSeconds(5L));
                    }
                }
            }
        }, 10L, 10L, TimeUnit.SECONDS);
        Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("channel-recent-frames-cleaner-%d").build()).scheduleAtFixedRate(() -> {
            try {
                for (BSSID bssid : this.bssids.values()) {
                    for (SSID ssid : bssid.ssids().values()) {
                        for (Channel channel : ssid.channels().values()) {
                            channel.cycleRecentFrames();
                        }
                    }
                }
            }
            catch (Exception e2) {
                LOG.error("Could not cycle recent channel frames.", (Throwable)e2);
            }
        }, 1L, 1L, TimeUnit.MINUTES);
    }

    public void registerBeaconFrame(Dot11BeaconFrame frame) {
        if (this.nzyme.getIgnoredFingerprints().contains(frame.transmitterFingerprint())) {
            LOG.trace("Not registering ignored fingerprint [{}]", (Object)frame.transmitterFingerprint());
            return;
        }
        if (!Strings.isNullOrEmpty(frame.ssid())) {
            this.register((byte)8, frame.transmitter(), frame.transmitterFingerprint(), frame.taggedParameters(), frame.ssid(), frame.meta().getChannel(), frame.meta().getAntennaSignal());
        }
    }

    public void registerProbeResponseFrame(Dot11ProbeResponseFrame frame) {
        if (this.nzyme.getIgnoredFingerprints().contains(frame.transmitterFingerprint())) {
            LOG.trace("Not registering ignored fingerprint [{}]", (Object)frame.transmitterFingerprint());
            return;
        }
        if (!Strings.isNullOrEmpty(frame.ssid())) {
            this.register((byte)5, frame.transmitter(), frame.transmitterFingerprint(), frame.taggedParameters(), frame.ssid(), frame.meta().getChannel(), frame.meta().getAntennaSignal());
        }
    }

    private void register(byte subtype, String transmitter, String transmitterFingerprint, Dot11TaggedParameters taggedParameters, final String ssidName, int channelNumber, int antennaSignal) {
        BSSID bssid;
        if (this.bssids.containsKey(transmitter)) {
            bssid = this.bssids.get(transmitter);
            if (!bssid.ssids().containsKey(ssidName)) {
                bssid.ssids().put(ssidName, SSID.create(ssidName, bssid.bssid(), this.beaconRateManager));
            }
        } else {
            String oui = this.nzyme.getOUIManager().lookupBSSID(transmitter);
            if (oui == null) {
                oui = "unknown";
            }
            final SSID ssid = SSID.create(ssidName, transmitter, this.beaconRateManager);
            bssid = BSSID.create((Map<String, SSID>)new HashMap<String, SSID>(){
                {
                    this.put(ssidName, ssid);
                }
            }, oui, transmitter);
            this.bssids.put(transmitter, bssid);
        }
        bssid.updateLastSeen();
        bssid.updateIsWPS(taggedParameters.isWPS());
        SSID ssid = bssid.ssids().get(ssidName);
        ssid.updateSecurity(taggedParameters.getSecurityConfiguration());
        if (subtype == 8) {
            ssid.beaconCount.incrementAndGet();
        }
        DateTime now = DateTime.now();
        try {
            if (ssid.channels().containsKey(channelNumber)) {
                Channel channel = ssid.channels().get(channelNumber);
                channel.totalFrames().incrementAndGet();
                channel.totalFramesRecent().incrementAndGet();
                if (transmitterFingerprint != null) {
                    channel.registerFingerprint(transmitterFingerprint);
                }
                channel.signalStrengthTable().recordSignalStrength(SignalStrengthTable.SignalStrength.create(now, antennaSignal));
                ssid.channels().replace(channelNumber, channel);
            } else {
                Channel channel = Channel.create(this.nzyme, channelNumber, bssid.bssid(), ssid.name(), new AtomicLong(1L), new AtomicLong(1L), transmitterFingerprint);
                channel.signalStrengthTable().recordSignalStrength(SignalStrengthTable.SignalStrength.create(now, antennaSignal));
                ssid.channels().put(channelNumber, channel);
            }
        }
        catch (NullPointerException e2) {
            LOG.error(ssid);
            throw e2;
        }
    }

    public Map<String, BSSID> getBSSIDs() {
        return new ImmutableMap.Builder<String, BSSID>().putAll(this.bssids).build();
    }

    public Set<String> getSSIDs() {
        HashSet<String> ssids = Sets.newHashSet();
        for (BSSID bssid : this.bssids.values()) {
            for (String ssid : bssid.ssids().keySet()) {
                if (ssids.contains(ssid)) continue;
                ssids.add(ssid);
            }
        }
        return ((ImmutableSet.Builder)new ImmutableSet.Builder().addAll(ssids)).build();
    }

    @Nullable
    public Channel findChannel(String bssidMac, String ssidName, int channelNumber) {
        for (BSSID bssid : this.bssids.values()) {
            if (!bssid.bssid().equals(bssidMac)) continue;
            for (SSID ssid : bssid.ssids().values()) {
                if (!ssid.name().equals(ssidName)) continue;
                for (Channel channel : ssid.channels().values()) {
                    if (channel.channelNumber() != channelNumber) continue;
                    return channel;
                }
            }
        }
        return null;
    }

    public void retentionClean(int seconds) {
        try {
            for (Map.Entry<String, BSSID> entry : Lists.newArrayList(this.bssids.entrySet())) {
                BSSID bssid = entry.getValue();
                if (!bssid.getLastSeen().isBefore(DateTime.now().minusSeconds(seconds))) continue;
                LOG.debug("Retention cleaning expired BSSID [{}] from internal networks list.", (Object)bssid.bssid());
                this.bssids.remove(entry.getKey());
            }
        }
        catch (Exception e2) {
            LOG.error("Error when trying to clean expired BSSIDs.", (Throwable)e2);
        }
    }

    public void reset() {
        this.bssids.clear();
    }

    public static class NoSuchChannelException
    extends Exception {
    }

    public static class NoSuchNetworkException
    extends Exception {
    }
}

