"use strict";
var path = require('path');
var _ = require('underscore');
var utils = require('./utils');
var LeenaConfiguration = (function () {
    function LeenaConfiguration(jsonOBJ) {
        this.jsonOBJ = jsonOBJ;
        this.errors = [];
    }
    LeenaConfiguration.prototype.parseJSONObject = function (solverToUse) {
        this.initBrowserSync();
        this.initChromeTester();
        this.initSMTSolvers(solverToUse);
        this.initFilesToTest();
        return this.errors;
    };
    LeenaConfiguration.prototype.getContentFile = function () {
        return JSON.stringify(this.jsonOBJ, null, 2);
    };
    LeenaConfiguration.prototype.initBrowserSync = function () {
        var Wobj = this.jsonOBJ.browserSync.watcher || {};
        var WBobj = this.jsonOBJ.browserSync.webServer || {};
        var watcher = {};
        var webServer = {};
        var errorsLength = this.errors.length;
        var prop;
        watcher.server = path.normalize(Wobj.server);
        watcher.port = Wobj.port;
        watcher.uiPort = Wobj.ui.port;
        webServer.server = path.normalize(WBobj.server);
        webServer.port = WBobj.port;
        webServer.uiPort = WBobj.ui.port;
        this.browserSync = {};
        this.browserSync.watcher = {};
        this.browserSync.webServer = {};
        for (prop in watcher) {
            if (watcher.hasOwnProperty(prop)) {
                if (watcher[prop] === undefined) {
                    this.errors.push(this.getErrorMessage(prop, 'browserSync.watcher'));
                }
            }
        }
        if (!utils.pathExists(watcher.server)) {
            this.errors.push(this.getErrorMessage('server', 'browserSync.watcher', 'must be an existing path'));
        }
        for (prop in webServer) {
            if (webServer.hasOwnProperty(prop)) {
                if (webServer[prop] === undefined) {
                    this.errors.push(this.getErrorMessage(prop, 'browserSync.webServer'));
                }
            }
        }
        if (utils.pathExists(webServer.server)) {
            this.errors.push(this.getErrorMessage('server', 'browserSync.webServer', 'must not exist'));
        }
        if (errorsLength === this.errors.length) {
            this.browserSync.watcher = watcher;
            this.browserSync.webServer = webServer;
        }
    };
    LeenaConfiguration.prototype.initChromeTester = function () {
        var DPobj = this.jsonOBJ.chrome.debuggingProtocol || {};
        var TSobj = this.jsonOBJ.chrome.testerServer || {};
        var debuggingProtocol = {};
        var testerServer = {};
        var errorsLength = this.errors.length;
        var prop;
        this.chromeTester = {};
        this.chromeTester.debuggingProtocol = {};
        this.chromeTester.testerServer = {};
        debuggingProtocol.hostname = DPobj.hostname;
        debuggingProtocol.port = DPobj.port;
        testerServer.hostname = TSobj.hostname;
        testerServer.port = TSobj.port;
        for (prop in debuggingProtocol) {
            if (debuggingProtocol.hasOwnProperty(prop)) {
                if (debuggingProtocol[prop] === undefined) {
                    this.errors.push(this.getErrorMessage(prop, 'chrome.debuggingProtocol'));
                }
            }
        }
        if (!utils.isCorrectPort(debuggingProtocol.port)) {
            this.errors.push(this.getErrorMessage('port', 'debuggingProtocol.port', 'has an incorrect value'));
        }
        for (prop in testerServer) {
            if (testerServer.hasOwnProperty(prop)) {
                if (testerServer[prop] === undefined) {
                    this.errors.push(this.getErrorMessage(prop, 'chrome.testerServer'));
                }
            }
        }
        if (!utils.isCorrectPort(testerServer.port)) {
            this.errors.push(this.getErrorMessage('port', 'testerServer.port', 'has an incorrect value'));
        }
        if (errorsLength === this.errors.length) {
            this.chromeTester.debuggingProtocol = debuggingProtocol;
            this.chromeTester.testerServer = testerServer;
        }
    };
    LeenaConfiguration.prototype.initSMTSolvers = function (solverToUse) {
        if (this.jsonOBJ['smt-solvers'] === undefined) {
            this.errors.push(this.getErrorMessage('smt-solvers'));
        }
        else {
            var solvers = this.jsonOBJ['smt-solvers'];
            var solver = solvers[solverToUse];
            if (solver === undefined) {
                this.errors.push(this.getErrorMessage(solverToUse, 'smt-solvers'));
            }
            else if (!utils.fileExists(solver)) {
                this.errors.push('Path of "' + solverToUse + '" does not exist');
            }
            else {
                this.solver = {};
                this.solver.name = solverToUse;
                this.solver.path = solver;
            }
        }
    };
    LeenaConfiguration.prototype.initFilesToTest = function () {
        var appFile = [];
        this.files = [];
        if (this.jsonOBJ.files === undefined) {
            this.errors.push(this.getErrorMessage('files'));
        }
        else {
            appFile = (_.isArray(this.jsonOBJ.files))
                ? this.jsonOBJ.files
                : (_.isString(this.jsonOBJ.files))
                    ? [this.jsonOBJ.files]
                    : undefined;
            if (appFile === undefined && !_.isArray(this.jsonOBJ.files)) {
                this.errors.push(this.getErrorMessage('files', '', 'can be a string or an array of strings'));
            }
            else {
                appFile = _.map(appFile, function (s) {
                    return s.replace(/\s/g, '');
                });
                appFile = _.uniq(appFile);
                if (this.browserSync.watcher.server) {
                    for (var k = 0, lengthF = appFile.length; k < lengthF; k++) {
                        var fileToTest = {};
                        fileToTest.originalPath = path.normalize(([
                            this.browserSync.watcher.server,
                            appFile[k]
                        ].join(path.sep)));
                        fileToTest.tempPath = path.normalize(([
                            this.browserSync.webServer.server,
                            appFile[k]
                        ].join(path.sep)));
                        if (!utils.fileExists(fileToTest.originalPath)) {
                            this.errors.push(this.getErrorMessage('files', '', '- path "' + fileToTest.originalPath + '" does not exist'));
                        }
                        this.files.push(fileToTest);
                    }
                }
            }
        }
    };
    LeenaConfiguration.prototype.getErrorMessage = function (property, rootObject, message) {
        var errorMessage;
        errorMessage = 'Property "' + property + '"';
        errorMessage += (rootObject && rootObject !== '')
            ? ' of node "' + rootObject + '" '
            : ' ';
        errorMessage += (message)
            ? message
            : 'is undefined';
        return errorMessage;
    };
    return LeenaConfiguration;
}());
module.exports = LeenaConfiguration;
