#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["SuspiciousURL", "SuspiciousURLPath"]

from .. import Vulnerability, WebVulnerability
from ... import identity


#------------------------------------------------------------------------------
class SuspiciousURL(WebVulnerability):
    """
    Suspicious URL.

    An URL was found that may contain sensitive information.
    User attention could be required.
    """

    data_subtype = "url"

    DEFAULTS = WebVulnerability.DEFAULTS.copy()
    DEFAULTS["level"]      = "informational"
    DEFAULTS["cvss_base"]  = "2.9"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


#------------------------------------------------------------------------------
class SuspiciousURLPath(SuspiciousURL):
    """
    Suspicious URL Path.

    An URL was found that may contain a path with sensitive information.
    User attention could be required.
    For example:
    - http://www.example.com/**users**/ -> URL with suspicious path.
    - http://www.example.com/index.jsp? **jsessionid** =.... -> suspicious parameter
    - http://www.example.com/index.php?id=1 -> no suspicious info
    """

    data_subtype = "path"


    #--------------------------------------------------------------------------
    def __init__(self, target, substring, **kwargs):
        """
        :param target: URL where the vulnerability was found.
        :type target: Url

        :param substring: Substring within the URL that causes suspicion.
        :type substring: str
        """

        # Substring property.
        self.__substr = substring

        # Call the superclass constructor.
        super(SuspiciousURLPath, self).__init__(target, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    def __str__(self):
        return str(self.url.url).replace\
                (self.substring, "[%s]" % self.substring)


    #--------------------------------------------------------------------------

    @identity
    def substring(self):
        """
        Matched substring within the URL that causes suspicion.

        >>> from golismero.api.data.resource.url import URL
        >>> from golismero.api.data.vulnerability.suspicious.url import SuspiciousURL
        >>> url = Url('http://www.example.com/path/index.jsp?para1=1&jsessionid=901828392012')
        >>> suspicious = SuspiciousURL(url, 'jsessionid')
        >>> suspicious
        <SuspiciousURL url='http://www.example.com/path/index.jsp?para1=1&jsessionid=901828392012'>
        >>> suspicious.substring
        'jsessionid'

        :return: Matched substring within the URL that causes suspicion.
        :rtype: str
        """
        return self.__substr
