#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["SuspiciousCookie"]

from ..cookie import CookieVulnerability
from .. import Vulnerability
from ... import identity


#------------------------------------------------------------------------------
class SuspiciousCookie(CookieVulnerability):
    """
    Suspicious Cookie.

    A cookie was found that may contain sensitive information.
    User attention could be required.
    """

    DEFAULTS = CookieVulnerability.DEFAULTS.copy()
    DEFAULTS["level"]      = "informational"
    DEFAULTS["cvss_base"]  = "4.3"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


    #--------------------------------------------------------------------------
    def __init__(self, target, cookie, value, substring, **kwargs):
        """
        :param target: HTTP response where the vulnerable cookie was found.
        :type target: HTTP_Response

        :param cookie: Name of the vulnerable cookie.
        :type cookie: str

        :param value: Cookie value.
        :type value: str

        :param substring: Substring within the cookie that causes suspicion.
        :type substring: str
        """

        # Validate the data types.
        if type(value) is not str:
            raise TypeError("Expected str, got %r instead" % type(value))
        if type(substring) is not str:
            raise TypeError("Expected str, got %r instead" % type(substring))

        # Save the properties.
        self.__value     = value
        self.__substring = substring

        # Parent constructor.
        super(SuspiciousCookie, self).__init__(
            target, cookie, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    @property
    def value(self):
        """
        :returns: Cookie value.
        :rtype: str
        """
        return self.__value


    #--------------------------------------------------------------------------
    @identity
    def substring(self):
        """
        :returns: Substring within the cookie that causes suspicion.
        :rtype: str
        """
        return self.__substring
