#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["WeakKey"]

from . import SSLVulnerability
from .. import Vulnerability
from ... import identity


#------------------------------------------------------------------------------
class WeakKey(SSLVulnerability):
    """
    Weak SSL/TLS Key.

    An SSL/TLS certificate with a weak cryptographic key was found. This may
    allow a strategically located attacker to snoop on network traffic, or
    perform a Man-In-The-Middle attack against unsuspecting users connecting
    to this host.

    Create a new certificate with stronger cryptographic keys.
    """

    DEFAULTS = SSLVulnerability.DEFAULTS.copy()
    DEFAULTS["level"]     = "high"
    DEFAULTS["cwe"]       = "CWE-327"
    DEFAULTS["cvss_base"] = "4"

    #--------------------------------------------------------------------------
    def __init__(self, target, keys, **kwargs):
        """
        :param target: Domain where the vulnerability was found.
        :type target: Domain

        :param keys: Lengths in bits of the vulnerable keys found.
        :type keys: tuple(int...)
        """

        # Sanitize the arguments.
        if type(keys) in (int, long):
            keys = (keys,)
        else:
            keys = tuple(int(k) for k in keys)

        # Key bitlengths.
        self.__keys = keys

        # Parent constructor.
        super(WeakKey, self).__init__(target, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    @identity
    def keys(self):
        """
        :returns: Lengths in bits of the vulnerable keys found.
        :rtype: tuple(int...)
        """
        return self.__keys
