#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["MaliciousUrl", "MaliciousDomain", "MaliciousIP", "MaliciousASN"]

from . import Malware
from .. import Vulnerability, WebVulnerability
from ...information.asn import ASN
from ...resource.domain import Domain
from ...resource.ip import IP


#------------------------------------------------------------------------------
class MaliciousUrl(WebVulnerability):
    """
    Malicious URL Detected.

    A URL was found that could contain output links to a malicious site or
    malware. This may be the result of a security intrusion, or a successful
    persistent XSS attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    data_subtype = "url"

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "4.3"


#------------------------------------------------------------------------------
class MaliciousDomain(Vulnerability):
    """
    Malicious Domain Detected.

    A domain was found that could contain output links to a malicious site or
    malware. This may be the result of a security intrusion, or a successful
    DNS poisoning attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    data_subtype = "domain"

    TARGET_CLASS = Domain

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "4.3"


#------------------------------------------------------------------------------
class MaliciousIP(Vulnerability):
    """
    Malicious IP Detected.

    An IP was found that could contain output links to a malicious site or
    malware. This may be the result of a security intrusion, or a successful
    persistent XSS attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    data_subtype = "ip"

    TARGET_CLASS = IP

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "5.8"


#------------------------------------------------------------------------------
class MaliciousASN(Vulnerability):
    """
    Malicious ASN Detected.

    An Autonomous System Number (ASN) was found that could contain a malicious
    site or malware. This may be the result of a security intrusion, or a
    successful BGP reconfiguration attack by a nefarious entity.

    You should review your website and ensure that your site was not
    compromised by a security intrusion.
    """

    data_subtype = "asn"

    TARGET_CLASS = ASN

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cvss_base"] = "6.8"
