#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["InsecureMethod"]

from .. import Vulnerability, WebVulnerability
from ... import identity


#------------------------------------------------------------------------------
class InsecureMethod(WebVulnerability):
    """
    Insecure HTTP Method Allowed.

    An insecure HTTP method is allowed by a web server. The typical insecure
    methods are: TRACE, HEAD, DELETE, SEARCH, COPY, MOVE, PROPFIND, PROPPATCH,
    MKCOL, LOCK, UNLOCK and PUT.

    The details on how to disable selected HTTP methods is specific to the
    server software being used. Please consult the documentation of your HTTP
    server software on how to do this.
    """

    DEFAULTS = WebVulnerability.DEFAULTS.copy()
    DEFAULTS["level"]      = "informational"
    DEFAULTS["cwe"]        = "CWE-16"
    DEFAULTS["cvss_base"]  = "2.1"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Test_HTTP_Methods_(OTG-CONFIG-006)",
        "http://security.stackexchange.com/questions/21413/"
                                                "how-to-exploit-http-methods",
    )


    #--------------------------------------------------------------------------
    def __init__(self, target, method, **kwargs):
        """
        :param target: URL where the vulnerability was found.
        :type target: Url

        :param method: Method name.
        :type method: str
        """

        # Name of the server that has this error page.
        self.__method = method

        # Parent constructor.
        super(InsecureMethod, self).__init__(target, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    @identity
    def method(self):
        """
        :returns: Method name.
        :rtype: str
        """
        return self.__method
