#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["DomainDisclosure"]

from .. import Vulnerability
from ...resource.domain import Domain

from ....net.web_utils import split_hostname

from netaddr import IPAddress


#------------------------------------------------------------------------------
class DomainDisclosure(Vulnerability):
    """
    Domain Disclosure.

    A domain was found by brute force or leaked by other means.
    """

    TARGET_CLASS = Domain

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"]      = "informational"
    DEFAULTS["cwe"]        = "CWE-200"
    DEFAULTS["cvss_base"]  = "2.2"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


    # FIXME maybe these properties should be in a DomainVulnerability class.


    #--------------------------------------------------------------------------
    @property
    def hostname(self):
        """
        :return: Domain name.
        :rtype: str
        """
        return self.target.hostname


    #--------------------------------------------------------------------------
    @property
    def root(self):
        """
        :return: Root domain. i.e: www.mysite.com -> mysite.com
        :rtype: str
        """
        _, domain, suffix = split_hostname(self.hostname)
        if suffix:
            return "%s.%s" % (domain, suffix)
        return domain
