/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.blobstore.cache;

import java.io.IOException;
import java.time.Instant;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.action.support.TransportActions;
import org.elasticsearch.blobstore.cache.CachedBlob;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.OriginSettingClient;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.node.NodeClosedException;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.ConnectTransportException;

public class BlobStoreCacheService {
    private static final Logger logger = LogManager.getLogger(BlobStoreCacheService.class);
    public static final int DEFAULT_CACHED_BLOB_SIZE = ByteSizeUnit.KB.toIntBytes(4L);
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final Client client;
    private final String index;

    public BlobStoreCacheService(ClusterService clusterService, ThreadPool threadPool, Client client, String index) {
        this.client = new OriginSettingClient(client, "searchable_snapshots");
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.index = index;
    }

    private void createIndexIfNecessary(final ActionListener<String> listener) {
        if (this.clusterService.state().routingTable().hasIndex(this.index)) {
            listener.onResponse((Object)this.index);
            return;
        }
        try {
            this.client.admin().indices().prepareCreate(this.index).setSettings(BlobStoreCacheService.indexSettings()).addMapping("_doc", BlobStoreCacheService.mappings()).execute((ActionListener)new ActionListener<CreateIndexResponse>(){

                public void onResponse(CreateIndexResponse createIndexResponse) {
                    assert (createIndexResponse.index().equals(BlobStoreCacheService.this.index));
                    listener.onResponse((Object)createIndexResponse.index());
                }

                public void onFailure(Exception e) {
                    if (e instanceof ResourceAlreadyExistsException || ExceptionsHelper.unwrapCause((Throwable)e) instanceof ResourceAlreadyExistsException) {
                        listener.onResponse((Object)BlobStoreCacheService.this.index);
                    } else {
                        listener.onFailure(e);
                    }
                }
            });
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    private static Settings indexSettings() {
        return Settings.builder().put("index.number_of_shards", 1).put("index.auto_expand_replicas", "0-1").put("index.priority", "900").build();
    }

    private static XContentBuilder mappings() throws IOException {
        XContentBuilder builder = XContentFactory.jsonBuilder();
        builder.startObject();
        builder.startObject("_doc");
        builder.field("dynamic", "strict");
        builder.startObject("_meta");
        builder.field("version", (ToXContent)Version.CURRENT);
        builder.endObject();
        builder.startObject("properties");
        builder.startObject("type");
        builder.field("type", "keyword");
        builder.endObject();
        builder.startObject("creation_time");
        builder.field("type", "date");
        builder.field("format", "epoch_millis");
        builder.endObject();
        builder.startObject("version");
        builder.field("type", "integer");
        builder.endObject();
        builder.startObject("repository");
        builder.field("type", "keyword");
        builder.endObject();
        builder.startObject("blob");
        builder.field("type", "object");
        builder.startObject("properties");
        builder.startObject("name");
        builder.field("type", "keyword");
        builder.endObject();
        builder.startObject("path");
        builder.field("type", "keyword");
        builder.endObject();
        builder.endObject();
        builder.endObject();
        builder.startObject("data");
        builder.field("type", "object");
        builder.startObject("properties");
        builder.startObject("content");
        builder.field("type", "binary");
        builder.endObject();
        builder.startObject("length");
        builder.field("type", "long");
        builder.endObject();
        builder.startObject("from");
        builder.field("type", "long");
        builder.endObject();
        builder.startObject("to");
        builder.field("type", "long");
        builder.endObject();
        builder.endObject();
        builder.endObject();
        builder.endObject();
        builder.endObject();
        builder.endObject();
        return builder;
    }

    public CachedBlob get(String repository, String name, String path, long offset) {
        assert (!Thread.currentThread().getName().contains("[system_read]")) : "must not block [" + Thread.currentThread().getName() + "] for a cache read";
        PlainActionFuture future = PlainActionFuture.newFuture();
        this.getAsync(repository, name, path, offset, (ActionListener<CachedBlob>)future);
        try {
            return (CachedBlob)future.actionGet(5L, TimeUnit.SECONDS);
        }
        catch (ElasticsearchTimeoutException e) {
            if (logger.isDebugEnabled()) {
                logger.warn(() -> new ParameterizedMessage("get from cache index timed out after [5s], retrieving from blob store instead [id={}]", (Object)CachedBlob.generateId(repository, name, path, offset)), (Throwable)e);
            } else {
                logger.warn("get from cache index timed out after [5s], retrieving from blob store instead");
            }
            return CachedBlob.CACHE_NOT_READY;
        }
    }

    protected void getAsync(String repository, String name, String path, long offset, final ActionListener<CachedBlob> listener) {
        final GetRequest request = new GetRequest(this.index).id(CachedBlob.generateId(repository, name, path, offset));
        this.client.get(request, (ActionListener)new ActionListener<GetResponse>(){

            public void onResponse(GetResponse response) {
                if (response.isExists()) {
                    logger.debug("cache hit : [{}]", (Object)request.id());
                    assert (!response.isSourceEmpty());
                    CachedBlob cachedBlob = CachedBlob.fromSource(response.getSource());
                    assert (response.getId().equals(cachedBlob.generatedId()));
                    listener.onResponse((Object)cachedBlob);
                } else {
                    logger.debug("cache miss: [{}]", (Object)request.id());
                    listener.onResponse((Object)CachedBlob.CACHE_MISS);
                }
            }

            public void onFailure(Exception e) {
                if (BlobStoreCacheService.isExpectedCacheGetException(e)) {
                    logger.debug(() -> new ParameterizedMessage("failed to retrieve cached blob from system index [{}]", (Object)BlobStoreCacheService.this.index), (Throwable)e);
                } else {
                    logger.warn(() -> new ParameterizedMessage("failed to retrieve cached blob from system index [{}]", (Object)BlobStoreCacheService.this.index), (Throwable)e);
                    assert (false) : e;
                }
                listener.onResponse((Object)CachedBlob.CACHE_NOT_READY);
            }
        });
    }

    private static boolean isExpectedCacheGetException(Exception e) {
        if (TransportActions.isShardNotAvailableException((Throwable)e) || e instanceof ConnectTransportException || e instanceof ClusterBlockException) {
            return true;
        }
        Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
        return cause instanceof NodeClosedException || cause instanceof ConnectTransportException;
    }

    public void putAsync(final String repository, final String name, final String path, final long offset, final BytesReference content, final ActionListener<Void> listener) {
        this.createIndexIfNecessary(new ActionListener<String>(){

            public void onResponse(String s) {
                try {
                    CachedBlob cachedBlob = new CachedBlob(Instant.ofEpochMilli(BlobStoreCacheService.this.threadPool.absoluteTimeInMillis()), Version.CURRENT, repository, name, path, content, offset);
                    final IndexRequest request = new IndexRequest(BlobStoreCacheService.this.index).id(cachedBlob.generatedId());
                    try (XContentBuilder builder = XContentFactory.jsonBuilder();){
                        request.source(cachedBlob.toXContent(builder, ToXContent.EMPTY_PARAMS));
                    }
                    BlobStoreCacheService.this.client.index(request, (ActionListener)new ActionListener<IndexResponse>(){

                        public void onResponse(IndexResponse indexResponse) {
                            logger.trace("cache fill ({}): [{}]", (Object)indexResponse.status(), (Object)request.id());
                            listener.onResponse(null);
                        }

                        public void onFailure(Exception e) {
                            logger.debug((Message)new ParameterizedMessage("failure in cache fill: [{}]", (Object)request.id()), (Throwable)e);
                            listener.onFailure(e);
                        }
                    });
                }
                catch (Exception e) {
                    logger.warn((Message)new ParameterizedMessage("cache fill failure: [{}]", (Object)CachedBlob.generateId(repository, name, path, offset)), (Throwable)e);
                    listener.onFailure(e);
                }
            }

            public void onFailure(Exception e) {
                logger.error(() -> new ParameterizedMessage("failed to create blob cache system index [{}]", (Object)BlobStoreCacheService.this.index), (Throwable)e);
                listener.onFailure(e);
            }
        });
    }
}

