/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.job;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.Strings;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.xpack.ml.MachineLearning;
import org.elasticsearch.xpack.ml.job.NodeLoadDetector;
import org.elasticsearch.xpack.ml.process.MlMemoryTracker;

public class JobNodeSelector {
    public static final PersistentTasksCustomMetadata.Assignment AWAITING_LAZY_ASSIGNMENT = new PersistentTasksCustomMetadata.Assignment(null, "persistent task is awaiting node assignment.");
    private static final Logger logger = LogManager.getLogger(JobNodeSelector.class);
    private final String jobId;
    private final String taskName;
    private final ClusterState clusterState;
    private final MlMemoryTracker memoryTracker;
    private final Function<DiscoveryNode, String> nodeFilter;
    private final NodeLoadDetector nodeLoadDetector;
    private final int maxLazyNodes;

    public JobNodeSelector(ClusterState clusterState, String jobId, String taskName, MlMemoryTracker memoryTracker, int maxLazyNodes, Function<DiscoveryNode, String> nodeFilter) {
        this.jobId = Objects.requireNonNull(jobId);
        this.taskName = Objects.requireNonNull(taskName);
        this.clusterState = Objects.requireNonNull(clusterState);
        this.memoryTracker = Objects.requireNonNull(memoryTracker);
        this.nodeLoadDetector = new NodeLoadDetector(Objects.requireNonNull(memoryTracker));
        this.maxLazyNodes = maxLazyNodes;
        this.nodeFilter = node -> {
            if (MachineLearning.isMlNode(node)) {
                return nodeFilter != null ? (String)nodeFilter.apply((DiscoveryNode)node) : null;
            }
            return "Not opening job [" + jobId + "] on node [" + JobNodeSelector.nodeNameOrId(node) + "], because this node isn't a ml node.";
        };
    }

    public PersistentTasksCustomMetadata.Assignment selectNode(int dynamicMaxOpenJobs, int maxConcurrentJobAllocations, int maxMachineMemoryPercent, boolean isMemoryTrackerRecentlyRefreshed) {
        boolean allNodesHaveDynamicMaxWorkers = this.clusterState.getNodes().getMinNodeVersion().onOrAfter(Version.V_7_2_0);
        boolean allocateByMemory = isMemoryTrackerRecentlyRefreshed;
        if (!isMemoryTrackerRecentlyRefreshed) {
            logger.warn("Falling back to allocating job [{}] by job counts because a memory requirement refresh could not be scheduled", (Object)this.jobId);
        }
        LinkedList<String> reasons = new LinkedList<String>();
        long maxAvailableCount = Long.MIN_VALUE;
        long maxAvailableMemory = Long.MIN_VALUE;
        DiscoveryNode minLoadedNodeByCount = null;
        DiscoveryNode minLoadedNodeByMemory = null;
        for (DiscoveryNode node : this.clusterState.getNodes()) {
            String reason = this.nodeFilter.apply(node);
            if (reason != null) {
                logger.trace(reason);
                reasons.add(reason);
                continue;
            }
            NodeLoadDetector.NodeLoad currentLoad = this.nodeLoadDetector.detectNodeLoad(this.clusterState, allNodesHaveDynamicMaxWorkers, node, dynamicMaxOpenJobs, maxMachineMemoryPercent, allocateByMemory);
            if (currentLoad.getError() != null) {
                reason = "Not opening job [" + this.jobId + "] on node [" + JobNodeSelector.nodeNameAndMlAttributes(node) + "], because [" + currentLoad.getError() + "]";
                logger.trace(reason);
                reasons.add(reason);
                continue;
            }
            allocateByMemory = currentLoad.isUseMemory();
            int maxNumberOfOpenJobs = currentLoad.getMaxJobs();
            if (currentLoad.getNumAllocatingJobs() >= (long)maxConcurrentJobAllocations) {
                reason = "Not opening job [" + this.jobId + "] on node [" + JobNodeSelector.nodeNameAndMlAttributes(node) + "], because node exceeds [" + currentLoad.getNumAllocatingJobs() + "] the maximum number of jobs [" + maxConcurrentJobAllocations + "] in opening state";
                logger.trace(reason);
                reasons.add(reason);
                continue;
            }
            long availableCount = (long)maxNumberOfOpenJobs - currentLoad.getNumAssignedJobs();
            if (availableCount == 0L) {
                reason = "Not opening job [" + this.jobId + "] on node [" + JobNodeSelector.nodeNameAndMlAttributes(node) + "], because this node is full. Number of opened jobs [" + currentLoad.getNumAssignedJobs() + "], " + MachineLearning.MAX_OPEN_JOBS_PER_NODE.getKey() + " [" + maxNumberOfOpenJobs + "]";
                logger.trace(reason);
                reasons.add(reason);
                continue;
            }
            if (maxAvailableCount < availableCount) {
                maxAvailableCount = availableCount;
                minLoadedNodeByCount = node;
            }
            if (!allocateByMemory) continue;
            if (currentLoad.getMaxMlMemory() > 0L) {
                Long estimatedMemoryFootprint = this.memoryTracker.getJobMemoryRequirement(this.taskName, this.jobId);
                if (estimatedMemoryFootprint != null) {
                    if (currentLoad.getNumAssignedJobs() == 0L) {
                        estimatedMemoryFootprint = estimatedMemoryFootprint + MachineLearning.NATIVE_EXECUTABLE_CODE_OVERHEAD.getBytes();
                    }
                    long availableMemory = currentLoad.getMaxMlMemory() - currentLoad.getAssignedJobMemory();
                    if (estimatedMemoryFootprint > availableMemory) {
                        reason = "Not opening job [" + this.jobId + "] on node [" + JobNodeSelector.nodeNameAndMlAttributes(node) + "], because this node has insufficient available memory. Available memory for ML [" + currentLoad.getMaxMlMemory() + "], memory required by existing jobs [" + currentLoad.getAssignedJobMemory() + "], estimated memory required for this job [" + estimatedMemoryFootprint + "]";
                        logger.trace(reason);
                        reasons.add(reason);
                        continue;
                    }
                    if (maxAvailableMemory >= availableMemory) continue;
                    maxAvailableMemory = availableMemory;
                    minLoadedNodeByMemory = node;
                    continue;
                }
                allocateByMemory = false;
                logger.debug(() -> new ParameterizedMessage("Falling back to allocating job [{}] by job counts because its memory requirement was not available", (Object)this.jobId));
                continue;
            }
            allocateByMemory = false;
            logger.debug(() -> new ParameterizedMessage("Falling back to allocating job [{}] by job counts because machine memory was not available for node [{}]", (Object)this.jobId, (Object)JobNodeSelector.nodeNameAndMlAttributes(node)));
        }
        return this.createAssignment(allocateByMemory ? minLoadedNodeByMemory : minLoadedNodeByCount, reasons);
    }

    private PersistentTasksCustomMetadata.Assignment createAssignment(DiscoveryNode minLoadedNode, List<String> reasons) {
        if (minLoadedNode == null) {
            String explanation = String.join((CharSequence)"|", reasons);
            logger.debug("no node selected for job [{}], reasons [{}]", (Object)this.jobId, (Object)explanation);
            return this.considerLazyAssignment(new PersistentTasksCustomMetadata.Assignment(null, explanation));
        }
        logger.debug("selected node [{}] for job [{}]", (Object)minLoadedNode, (Object)this.jobId);
        return new PersistentTasksCustomMetadata.Assignment(minLoadedNode.getId(), "");
    }

    PersistentTasksCustomMetadata.Assignment considerLazyAssignment(PersistentTasksCustomMetadata.Assignment currentAssignment) {
        assert (currentAssignment.getExecutorNode() == null);
        int numMlNodes = 0;
        for (DiscoveryNode node : this.clusterState.getNodes()) {
            if (!MachineLearning.isMlNode(node)) continue;
            ++numMlNodes;
        }
        if (numMlNodes < this.maxLazyNodes) {
            return AWAITING_LAZY_ASSIGNMENT;
        }
        return currentAssignment;
    }

    static String nodeNameOrId(DiscoveryNode node) {
        String nodeNameOrID = node.getName();
        if (Strings.isNullOrEmpty((String)nodeNameOrID)) {
            nodeNameOrID = node.getId();
        }
        return nodeNameOrID;
    }

    public static String nodeNameAndVersion(DiscoveryNode node) {
        String nodeNameOrID = JobNodeSelector.nodeNameOrId(node);
        StringBuilder builder = new StringBuilder("{").append(nodeNameOrID).append('}');
        builder.append('{').append("version=").append(node.getVersion()).append('}');
        return builder.toString();
    }

    static String nodeNameAndMlAttributes(DiscoveryNode node) {
        String nodeNameOrID = JobNodeSelector.nodeNameOrId(node);
        StringBuilder builder = new StringBuilder("{").append(nodeNameOrID).append('}');
        for (Map.Entry entry : node.getAttributes().entrySet()) {
            if (!((String)entry.getKey()).startsWith("ml.") && !((String)entry.getKey()).equals("node.ml")) continue;
            builder.append('{').append(entry).append('}');
        }
        return builder.toString();
    }
}

