/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.filestructurefinder;

import java.time.Instant;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.common.time.DateFormatters;
import org.elasticsearch.index.mapper.DateFieldMapper;
import org.elasticsearch.xpack.core.ml.filestructurefinder.FieldStats;

public class FieldStatsCalculator {
    private long count;
    private SortedMap<String, Integer> countsByStringValue;
    private SortedMap<Double, Integer> countsByNumericValue;
    private DateFormatter dateFormatter;
    private Instant earliestTimestamp;
    private Instant latestTimestamp;
    private String earliestTimeString;
    private String latestTimeString;

    public FieldStatsCalculator(Map<String, String> mapping) {
        switch (mapping.get("type")) {
            case "byte": 
            case "short": 
            case "integer": 
            case "long": 
            case "half_float": 
            case "float": 
            case "double": {
                this.countsByNumericValue = new TreeMap<Double, Integer>();
                break;
            }
            case "date": 
            case "date_nanos": {
                String format = mapping.get("format");
                this.dateFormatter = format == null ? DateFieldMapper.DEFAULT_DATE_TIME_FORMATTER : DateFormatter.forPattern((String)format);
                this.countsByStringValue = new TreeMap<String, Integer>();
                break;
            }
            default: {
                this.countsByStringValue = new TreeMap<String, Integer>();
            }
        }
    }

    public void accept(Collection<String> fieldValues) {
        this.count += (long)fieldValues.size();
        for (String fieldValue : fieldValues) {
            if (this.countsByNumericValue != null) {
                try {
                    this.countsByNumericValue.compute(Double.valueOf(fieldValue), (k, v) -> v == null ? 1 : 1 + v);
                    continue;
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Field with numeric mapping [" + fieldValue + "] could not be parsed as type double", e);
                }
            }
            this.countsByStringValue.compute(fieldValue, (k, v) -> v == null ? 1 : 1 + v);
            if (this.dateFormatter == null) continue;
            Instant parsedTimestamp = DateFormatters.from((TemporalAccessor)this.dateFormatter.parse(fieldValue)).toInstant();
            if (this.earliestTimestamp == null || this.earliestTimestamp.isAfter(parsedTimestamp)) {
                this.earliestTimestamp = parsedTimestamp;
                this.earliestTimeString = fieldValue;
            }
            if (this.latestTimestamp != null && !this.latestTimestamp.isBefore(parsedTimestamp)) continue;
            this.latestTimestamp = parsedTimestamp;
            this.latestTimeString = fieldValue;
        }
    }

    public FieldStats calculate(int numTopHits) {
        if (this.countsByNumericValue != null) {
            if (this.countsByNumericValue.isEmpty()) {
                assert (this.count == 0L);
                return new FieldStats(this.count, 0, Collections.emptyList());
            }
            assert (this.count > 0L);
            return new FieldStats(this.count, this.countsByNumericValue.size(), this.countsByNumericValue.firstKey(), this.countsByNumericValue.lastKey(), this.calculateMean(), this.calculateMedian(), this.findNumericTopHits(numTopHits));
        }
        return new FieldStats(this.count, this.countsByStringValue.size(), this.earliestTimeString, this.latestTimeString, this.findStringTopHits(numTopHits));
    }

    Double calculateMean() {
        assert (this.countsByNumericValue != null);
        if (this.countsByNumericValue.isEmpty()) {
            return null;
        }
        double runningCount = 0.0;
        double runningMean = Double.NaN;
        for (Map.Entry<Double, Integer> entry : this.countsByNumericValue.entrySet()) {
            double entryCount = entry.getValue().intValue();
            double newRunningCount = runningCount + entryCount;
            if (runningCount > 0.0) {
                runningMean = runningMean * (runningCount / newRunningCount) + entry.getKey() * (entryCount / newRunningCount);
            } else if (entryCount > 0.0) {
                runningMean = entry.getKey();
            }
            runningCount = newRunningCount;
        }
        return runningMean;
    }

    Double calculateMedian() {
        assert (this.countsByNumericValue != null);
        if (this.count % 2L == 1L) {
            long targetCount = this.count / 2L + 1L;
            long currentUpperBound = 0L;
            for (Map.Entry<Double, Integer> entry : this.countsByNumericValue.entrySet()) {
                if ((currentUpperBound += (long)entry.getValue().intValue()) < targetCount) continue;
                return entry.getKey();
            }
        } else {
            long target1Count = this.count / 2L;
            long target2Count = target1Count + 1L;
            double target1Value = Double.NaN;
            long prevUpperBound = -1L;
            long currentUpperBound = 0L;
            for (Map.Entry<Double, Integer> entry : this.countsByNumericValue.entrySet()) {
                if ((currentUpperBound += (long)entry.getValue().intValue()) >= target2Count) {
                    if (prevUpperBound < target1Count) {
                        return entry.getKey();
                    }
                    return (target1Value + entry.getKey()) / 2.0;
                }
                if (currentUpperBound >= target1Count) {
                    target1Value = entry.getKey();
                }
                prevUpperBound = currentUpperBound;
            }
        }
        return null;
    }

    List<Map<String, Object>> findNumericTopHits(int numTopHits) {
        assert (this.countsByNumericValue != null);
        return FieldStatsCalculator.findTopHits(numTopHits, this.countsByNumericValue, Comparator.comparing(Map.Entry::getKey), FieldStats::toIntegerIfInteger);
    }

    List<Map<String, Object>> findStringTopHits(int numTopHits) {
        return FieldStatsCalculator.findTopHits(numTopHits, this.countsByStringValue, Comparator.comparing(Map.Entry::getKey), s -> s);
    }

    private static <T> List<Map<String, Object>> findTopHits(int numTopHits, Map<T, Integer> countsByValue, Comparator<Map.Entry<T, Integer>> secondarySort, Function<T, Object> outputMapper) {
        List sortedByCount = countsByValue.entrySet().stream().sorted(Comparator.comparing(Map.Entry::getValue, Comparator.reverseOrder()).thenComparing(secondarySort)).limit(numTopHits).collect(Collectors.toList());
        ArrayList<Map<String, Object>> topHits = new ArrayList<Map<String, Object>>(sortedByCount.size());
        for (Map.Entry entry : sortedByCount) {
            LinkedHashMap<String, Object> topHit = new LinkedHashMap<String, Object>(3);
            topHit.put("value", outputMapper.apply(entry.getKey()));
            topHit.put("count", entry.getValue());
            topHits.add(topHit);
        }
        return topHits;
    }
}

