var async = require('async');
const helpers = require('../../../helpers/azure');

module.exports = {
    title: 'Event Hubs Namespace CMK Encrypted',
    category: 'Event Hubs',
    domain: 'Content Delivery',
    severity: 'Medium',
    description: 'Ensures that Event Hubs premium namespaces are encrypted with CMK.',
    more_info: 'Event Hubs allows you to encrypt data in your namespaces using customer-managed keys (CMK) instead of using platform-managed keys, which are enabled by default. Using CMK encryption offers enhanced security and compliance, allowing centralized management and control of encryption keys through Azure Key Vault.',
    recommended_action: 'Ensure that Event Hubs premium namespaces have CMK encryption enabled.',
    link: 'https://learn.microsoft.com/en-us/azure/event-hubs/configure-customer-managed-key',
    apis: ['eventHub:listEventHub'],
    realtime_triggers: ['microsofteventhub:namespaces:write', 'microsofteventhub:namespaces:delete'],

    run: function(cache, settings, callback) {
        var results = [];
        var source = {};
        var locations = helpers.locations(settings.govcloud);

        async.each(locations.eventHub, function(location, rcb) {
            var eventHubs = helpers.addSource(cache, source,
                ['eventHub', 'listEventHub', location]);

            if (!eventHubs) return rcb();

            if (eventHubs.err || !eventHubs.data) {
                helpers.addResult(results, 3,
                    'Unable to query for Event Hubs namespaces: ' + helpers.addError(eventHubs), location);
                return rcb();
            }

            if (!eventHubs.data.length) {
                helpers.addResult(results, 0, 'No Event Hubs namespaces found', location);
                return rcb();
            }

            for (let eventHub of eventHubs.data){
                if (!eventHub.id) continue;

                if (eventHub.sku && eventHub.sku.tier && eventHub.sku.tier.toLowerCase() !== 'premium') {
                    helpers.addResult(results, 0, 'Event Hubs namespace is not a premium namespace', location, eventHub.id);
                } else if (eventHub.encryption && eventHub.encryption.keySource) {
                    helpers.addResult(results, 0, 'Event Hubs namespace is encrypted using CMK', location, eventHub.id);
                } else {
                    helpers.addResult(results, 2, 'Event Hubs namespace is not encrypted using CMK', location, eventHub.id);
                }
            }

            rcb();
        }, function() {
            callback(null, results, source);
        });
    }
};
