#!/usr/bin/python3
import getopt
import os
import re
import sys

'''
Project generator for a JEB plugin in Java (Eclipse IDE, IntelliJ IDEA).


JEB Decompiler (c) PNF Software
'''


tplEclipseProject = '''<?xml version="1.0" encoding="UTF-8"?>
<projectDescription>
	<name>%s</name>
	<comment></comment>
	<projects></projects>
	<buildSpec>
		<buildCommand>
			<name>org.eclipse.jdt.core.javabuilder</name>
			<arguments>
			</arguments>
		</buildCommand>
	</buildSpec>
	<natures>
		<nature>org.eclipse.jdt.core.javanature</nature>
	</natures>
  %s
</projectDescription>
'''
tplEclipseLinkedResInProject = '''<linkedResources>
		<link>
			<name>scripts</name>
			<type>2</type>
			<location>%s</location>
		</link>
	</linkedResources>'''

tplEclipseClasspath = '''<?xml version="1.0" encoding="UTF-8"?>
<classpath>
	<classpathentry kind="src" path="%s" including="*.java"/>
	<classpathentry kind="con" path="org.eclipse.jdt.launching.JRE_CONTAINER"/>
  %s
	<classpathentry kind="output" path="bin"/>
</classpath>
'''

tplIdeaProjectIml = '''<?xml version="1.0" encoding="UTF-8"?>
<module type="JAVA_MODULE" version="4">
  <component name="NewModuleRootManager" inherit-compiler-output="true">
    <exclude-output />%s
    <content url="file://$MODULE_DIR$">
      <sourceFolder url="file://$MODULE_DIR$/src" isTestSource="false" />
    </content>
    <orderEntry type="inheritedJdk" />
    <orderEntry type="sourceFolder" forTests="false" />
    <orderEntry type="module-library">
      <library>
        <CLASSES>
          <root url="jar://%s!/" />
        </CLASSES>
        <JAVADOC>
          <root url="jar://%s!/reference" />
        </JAVADOC>
        <SOURCES />
      </library>
    </orderEntry>
  </component>
</module>'''

tplIdeaProjectSourceEntry = '''
    <content url="file://%s">
      <sourceFolder url="file://%s" isTestSource="false" />
    </content>'''

tplIdeaMisc = '''<?xml version="1.0" encoding="UTF-8"?>
<project version="4">
  <component name="ProjectRootManager" version="2" languageLevel="JDK_17" default="true" project-jdk-name="17" project-jdk-type="JavaSDK">
    <output url="file://$PROJECT_DIR$/out" />
  </component>
</project>'''

tplIdeaModules = '''<?xml version="1.0" encoding="UTF-8"?>
<project version="4">
  <component name="ProjectModuleManager">
    <modules>
      <module fileurl="file://$PROJECT_DIR$/%s.iml" filepath="$PROJECT_DIR$/%s.iml" />
    </modules>
  </component>
</project>'''


tplSourceFile = '''//?type=%s
import %s;

/**
 * Skeleton code for your JEB plugin.
 */
public class %s extends %s {

    @Override
    %s {
        throw new RuntimeException("Stub");
    }

}
'''


def check_project_name(s):
  return re.fullmatch('[a-zA-Z][a-zA-Z0-9_]*', s)


def err(msg):
  print(msg)
  sys.exit(-1)


def usage():
  print('Project generator for a JEB plugin in Java (Eclipse IDE, IntelliJ IDEA)')
  print('Usage:')
  print('  %s [options] project_name' % os.path.split(sys.argv[0])[-1])
  print('')
  print('Options:')
  print('  -i             : generate an IntelliJ IDEA project instead of an Eclipse IDE project')
  print('  -s             : create a script plugin (the Java source will be located in JEB\'s coreplugins/scripts/ folder)')
  print('  -d foldername  : destination folder for the project files (default: new folder is created)')
  print('  -p plugintype  : can be specified to create a new skeleton class (see the legal plugin types below)')
  #print('  -v             : be extra verbose')
  #print('  -h             : print this help and exit')
  print('')
  print('Supported plugin types (to use with the optional -p flag):')
  print('  gendec-ir      : generic decompiler IR plugin')
  print('  gendec-ast     : generic decompiler AST plugin')
  print('  dexdec-ir      : Dex decompiler IR plugin')
  print('  dexdec-ast     : Dex decompiler AST plugin')
  sys.exit(-1)


if __name__ == '__main__':
  if 'JEB_HOME' not in os.environ:
    err('Set an environment variable JEB_HOME pointing to your JEB folder')
  jebhome = os.environ['JEB_HOME']
  jebcorepath = os.path.join(jebhome, 'bin', 'app', 'jeb.jar')
  if not os.path.isfile(jebcorepath):
    err('Based on your value of JEB_HOME, jeb.jar was expected at this location, but it was not found: %s' % jebcorepath)
  jebdocpath = os.path.join(jebhome, 'doc', 'apidoc.zip')

  try:
    opts, args = getopt.getopt(sys.argv[1:], 'isd:p:vh')
  except getopt.GetoptError as err:
    usage()

  prjname = None
  dstfolder = None
  idetype = 0  # Eclipse(0), IDEA(1)
  link_to_scripts = False
  plugintype = None
  verbose = False
  internal = False  # RESERVED
  for o, a in opts:
    if o == '-i':
      idetype = 1
    elif o == '-s':
      link_to_scripts = True
    elif o == '-d':
      dstfolder = a
    elif o == '-p':
      plugintype = a
    elif o == '-v':
      verbose = True
    elif o == '-h':
      usage()
    else:
      err('Illegal option: %s' % o)

  if len(args) == 0:
    err('Specify a project name. Use -h to see how to use.')
  prjname = args[0]
  if not re.fullmatch('[a-zA-Z][a-zA-Z0-9_]*', prjname):
    err('Invalid project name: %s. Use alphanumeric chars and the underscore; the first character must be alphabetic.' % prjname)
  if not dstfolder:
    dstfolder = prjname

  prj_skeleton_info = None
  if plugintype:
    prjskelinfomap = {
      'gendec-ir' : ('com.pnfsoftware.jeb.core.units.code.asm.decompiler.ir.opt',   'AbstractEOptimizer', 'public int perform()'),
      'gendec-ast': ('com.pnfsoftware.jeb.core.units.code.asm.decompiler.ast.opt',  'AbstractCOptimizer', 'public int perform()'),
      'dexdec-ir' : ('com.pnfsoftware.jeb.core.units.code.android.ir',              'AbstractDOptimizer', 'public int perform()'),
      'dexdec-ast': ('com.pnfsoftware.jeb.core.units.code.java',                    'AbstractJOptimizer', 'public int perform()'),
    }
    prj_skeleton_info = prjskelinfomap.get(plugintype, None)
    if not prj_skeleton_info:
      err('Illegal project type: %s' % plugintype)

  if not os.path.exists(dstfolder):
    os.makedirs(dstfolder)
  elif not os.path.isdir(dstfolder):
    err('The destination path exists but it is not a folder: %s' % dstfolder)
  #elif os.listdir(dstfolder):
  #  err('Will not overwrite contents in a non-empty directory: %s' % dstfolder)

  print('Project name: %s' % prjname)
  print('Destination folder: %s' % dstfolder)

  if link_to_scripts:
    srcname = 'scripts'
    srcfullpath = os.path.join(jebhome, 'coreplugins', 'scripts')
  else:
    srcname = 'src'
    srcfullpath = os.path.join(dstfolder, 'src')
    os.mkdir(srcfullpath)
  print('Source folder: %s' % srcfullpath)

  if idetype == 0:
    extra = ''
    if link_to_scripts:
      extra = tplEclipseLinkedResInProject % srcfullpath.replace('\\', '/')
    _Project = tplEclipseProject % (prjname, extra)
    with open(os.path.join(dstfolder, '.project'), 'w') as f:
      f.write(_Project)
    print('Created: .project')

    jeblibentry = '''<classpathentry kind="lib" path="%s">
      <attributes><attribute name="javadoc_location" value="jar:file:/%s!/reference"/></attributes>
    </classpathentry>''' % (jebcorepath, jebdocpath)
    if internal:
      jeblibentry = '<classpathentry combineaccessrules="false" kind="src" path="/jeb3.core"/>'
    _Classpath = tplEclipseClasspath % (srcname, jeblibentry)
    with open(os.path.join(dstfolder, '.classpath'), 'w') as f:
      f.write(_Classpath)
    print('Created: .classpath')

  else:  # IDEA
    if internal:
      raise Exception('Not implemented for IDEA projects')

    extra = ''
    if link_to_scripts:
      extra = tplIdeaProjectSourceEntry % (srcfullpath.replace('\\', '/'), srcfullpath.replace('\\', '/'))
    _Project = tplIdeaProjectIml % (extra, jebcorepath.replace('\\', '/'), jebdocpath.replace('\\', '/'))
    with open(os.path.join(dstfolder, prjname + '.iml'), 'w') as f:
      f.write(_Project)
    print('Created: %s.iml' % prjname)

    ideadir = os.path.join(dstfolder, '.idea')
    os.mkdir(ideadir)

    _Misc = tplIdeaMisc
    with open(os.path.join(ideadir, 'misc.xml'), 'w') as f:
      f.write(_Misc)
    print('Created: misc.xml')

    _Modules = tplIdeaModules % (prjname, prjname)
    with open(os.path.join(ideadir, 'modules.xml'), 'w') as f:
      f.write(_Modules)
    print('Created: modules.xml')

  if prj_skeleton_info:
    e = prj_skeleton_info
    clname = 'JebPlugin_' + prjname    
    _Src = tplSourceFile % (plugintype, e[0] + '.' + e[1], clname, e[1], e[2])
    skelfilename = clname + '.java'
    path = os.path.join(srcfullpath, skelfilename)
    with open(path, 'w') as f:
      f.write(_Src)
    print('Created: %s' % (path if link_to_scripts else skelfilename))

  print('DONE.')
