#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["Backdoor"]

from . import Malware
from .. import Vulnerability
from ... import identity
from ...resource.domain import Domain
from ...resource.ip import IP


#------------------------------------------------------------------------------
class Backdoor(Malware):
    """
    Backdoor Service Detected.

    A backdoor service was detected. This is an indication of an earlier
    compromise by an unknown agent.
    """

    DEFAULTS = Malware.DEFAULTS.copy()
    DEFAULTS["cwe"]        = ("CWE-508", "CWE-510", "CWE-514")
    DEFAULTS["capec"]      = "CAPEC-448"
    DEFAULTS["cvss_base"]  = "9.5"
    DEFAULTS["references"] = (
        "https://en.wikipedia.org/wiki/Backdoor_(computing)",
    )


    #--------------------------------------------------------------------------
    def __init__(self, target, port, **kwargs):
        """
        :param target: Host where the backdoor service was found.
        :type target: Domain | IP

        :param port: Port where the backdoor service is listening.
        :type port: int

        """

        # Check the host argument.
        if isinstance(target, basestring):
            try:
                host = IP(target)
            except ValueError:
                host = Domain(target)
        elif not isinstance(target, Domain) and not isinstance(target, IP):
            raise TypeError(
                "Expected Domain or IP, got %r instead" % type(target))

        # Check the port argument.
        if not isinstance(port, int):
            raise TypeError("Expected int, got '%s'" % type(port))

        # Save the properties.
        self.__host = host.address if hasattr(host, "address") \
            else host.hostname
        self.__port = port

        # Call the superclass constructor.
        super(Backdoor, self).__init__(**kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    @property
    def host(self):
        return self.__host


    #--------------------------------------------------------------------------
    @identity
    def port(self):
        return self.__port
