"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _Log = _interopRequireDefault(require("./Utilities/Log"));

var _ObjectHelper = _interopRequireDefault(require("./Utilities/ObjectHelper"));

var _Coverage = _interopRequireDefault(require("./Coverage"));

var _Config = _interopRequireDefault(require("./Config"));

var _SymbolicHelper = _interopRequireDefault(require("./SymbolicHelper"));

var _SymbolicObject = require("./Values/SymbolicObject");

var _WrappedValue = require("./Values/WrappedValue");

var _Stats = _interopRequireDefault(require("Stats"));

var _z3javascript = _interopRequireDefault(require("z3javascript"));

var _Helpers = _interopRequireDefault(require("./Models/Helpers"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _typeof(obj) { "@babel/helpers - typeof"; if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") { _typeof = function _typeof(obj) { return typeof obj; }; } else { _typeof = function _typeof(obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; }; } return _typeof(obj); }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function BuildUnaryJumpTable(state) {
  var ctx = state.ctx;
  return {
    "boolean": {
      "+": function _(val_s) {
        return ctx.mkIte(val_s, state.constantSymbol(1), state.constantSymbol(0));
      },
      "-": function _(val_s) {
        return ctx.mkIte(val_s, state.constantSymbol(-1), state.constantSymbol(0));
      },
      "!": function _(val_s) {
        return ctx.mkNot(val_s);
      }
    },
    "number": {
      "!": function _(val_s, val_c) {
        var bool_s = state.asSymbolic(state.toBool(new _WrappedValue.ConcolicValue(val_c, val_s)));
        return bool_s ? ctx.mkNot(bool_s) : undefined;
      },
      "+": function _(val_s) {
        return val_s;
      },
      "-": function _(val_s) {
        return ctx.mkUnaryMinus(val_s);
      }
    },
    "string": {
      "!": function _(val_s, val_c) {
        var bool_s = state.asSymbolic(state.toBool(new _WrappedValue.ConcolicValue(val_c, val_s)));
        return bool_s ? ctx.mkNot(bool_s) : undefined;
      },
      "+": function _(val_s) {
        return ctx.mkStrToInt(val_s);
      },
      "-": function _(val_s) {
        return ctx.mkUnaryMinus(ctx.mkStrToInt(val_s));
      }
    }
  };
}

var SymbolicState = /*#__PURE__*/function () {
  function SymbolicState(input, sandbox) {
    _classCallCheck(this, SymbolicState);

    this.ctx = new _z3javascript["default"].Context();
    this.slv = new _z3javascript["default"].Solver(this.ctx, _Config["default"].incrementalSolverEnabled, [{
      name: "smt.string_solver",
      value: _Config["default"].stringSolver
    }, //				{ name: "timeout", value: Config.maxSolverTime },
    {
      name: "random_seed",
      value: Math.floor(Math.random() * Math.pow(2, 32))
    }, {
      name: "phase_selection",
      value: 5
    }]);
    this.helpers = new _Helpers["default"](this, this.ctx);
    _z3javascript["default"].Query.MAX_REFINEMENTS = _Config["default"].maxRefinements;
    this.input = input;
    this.inputSymbols = {};
    this.pathCondition = [];
    this.stats = new _Stats["default"]();
    this.coverage = new _Coverage["default"](sandbox);
    this.errors = [];
    this._unaryJumpTable = BuildUnaryJumpTable(this);

    this._setupSmtFunctions();
  }
  /** Set up a bunch of SMT functions used by the models **/


  _createClass(SymbolicState, [{
    key: "_setupSmtFunctions",
    value: function _setupSmtFunctions() {
      this.stringRepeat = this.ctx.mkRecFunc(this.ctx.mkStringSymbol("str.repeat"), [this.ctx.mkStringSort(), this.ctx.mkIntSort()], this.ctx.mkStringSort());
      this.slv.fromString("(define-fun-rec str.repeat ((a String) (b Int)) String (if (<= b 0) \"\" (str.++ a (str.repeat a (- b 1)))))");
      this.whiteLeft = this.ctx.mkRecFunc(this.ctx.mkStringSymbol("str.whiteLeft"), [this.ctx.mkStringSort(), this.ctx.mkIntSort()], this.ctx.mkIntSort());
      this.whiteRight = this.ctx.mkRecFunc(this.ctx.mkStringSymbol("str.whiteRight"), [this.ctx.mkStringSort(), this.ctx.mkIntSort()], this.ctx.mkIntSort());
      /** Set up trim methods **/

      this.slv.fromString("(define-fun str.isWhite ((c String)) Bool (= c \" \"))\n" + //TODO: Only handles  
      "(define-fun-rec str.whiteLeft ((s String) (i Int)) Int (if (str.isWhite (str.at s i)) (str.whiteLeft s (+ i 1)) i))\n" + "(define-fun-rec str.whiteRight ((s String) (i Int)) Int (if (str.isWhite (str.at s i)) (str.whiteRight s (- i 1)) i))\n");
    }
  }, {
    key: "pushCondition",
    value: function pushCondition(cnd, binder) {
      this.pathCondition.push({
        ast: cnd,
        binder: binder || false,
        forkIid: this.coverage.last()
      });
    }
  }, {
    key: "conditional",
    value: function conditional(result) {
      var result_c = this.getConcrete(result),
          result_s = this.asSymbolic(result);

      if (result_c === true) {
        this.pushCondition(result_s);
      } else if (result_c === false) {
        this.pushCondition(this.ctx.mkNot(result_s));
      } else {
        _Log["default"].log("WARNING: Symbolic Conditional on non-bool, concretizing");
      }

      return result_c;
    }
    /**
      * Creates a full (up to date) solver instance and then calls toString on it to create an SMT2Lib problem
      * TODO: This is a stop-gag implementation for the work with Ronny - not to be relied upon.
      */

  }, {
    key: "inlineToSMTLib",
    value: function inlineToSMTLib() {
      var _this = this;

      this.slv.push();
      this.pathCondition.forEach(function (pcItem) {
        return _this.slv.assert(pcItem.ast);
      });
      var resultString = this.slv.toString();
      this.slv.pop();
      return resultString;
    }
    /**
       * Returns the final PC as a string (if any symbols exist)
       */

  }, {
    key: "finalPC",
    value: function finalPC() {
      return this.pathCondition.filter(function (x) {
        return x.ast;
      }).map(function (x) {
        return x.ast;
      });
    }
  }, {
    key: "_stringPC",
    value: function _stringPC(pc) {
      return pc.length ? pc.reduce(function (prev, current) {
        var this_line = current.simplify().toPrettyString().replace(/\s+/g, " ").replace(/not /g, "¬");

        if (this_line.startsWith("(¬")) {
          this_line = this_line.substr(1, this_line.length - 2);
        }

        if (this_line == "true" || this_line == "false") {
          return prev;
        } else {
          return prev + (prev.length ? ", " : "") + this_line;
        }
      }, "") : "";
    }
  }, {
    key: "_addInput",
    value: function _addInput(pc, solution, pcIndex, childInputs) {
      solution._bound = pcIndex + 1;
      childInputs.push({
        input: solution,
        pc: this._stringPC(pc),
        forkIid: this.pathCondition[pcIndex].forkIid
      });
    }
  }, {
    key: "_buildPC",
    value: function _buildPC(childInputs, i, inputCallback) {
      var newPC = this.ctx.mkNot(this.pathCondition[i].ast);
      var allChecks = this.pathCondition.slice(0, i).reduce(function (last, next) {
        return last.concat(next.ast.checks);
      }, []).concat(newPC.checks);

      var solution = this._checkSat(newPC, i, allChecks);

      if (solution) {
        this._addInput(newPC, solution, i, childInputs);

        if (inputCallback) {
          inputCallback(childInputs);
        }
      } else {}
    }
  }, {
    key: "_buildAsserts",
    value: function _buildAsserts(i) {
      return this.pathCondition.slice(0, i).map(function (x) {
        return x.ast;
      });
    }
  }, {
    key: "alternatives",
    value: function alternatives(inputCallback) {
      var _this2 = this;

      var childInputs = [];

      if (this.input._bound > this.pathCondition.length) {
        throw "Bound ".concat(this.input._bound, " > ").concat(this.pathCondition.length, ", divergence has occured");
      } //Push all PCs up until bound


      this._buildAsserts(Math.min(this.input._bound, this.pathCondition.length)).forEach(function (x) {
        return _this2.slv.assert(x);
      });

      this.slv.push();

      for (var i = this.input._bound; i < this.pathCondition.length; i++) {
        //TODO: Make checks on expressions smarter
        if (!this.pathCondition[i].binder) {
          this._buildPC(childInputs, i, inputCallback);
        }

        //Push the current thing we're looking at to the solver
        this.slv.assert(this.pathCondition[i].ast);
        this.slv.push();
      }

      this.slv.reset(); //Guarentee inputCallback is called at least once

      inputCallback(childInputs);
    }
  }, {
    key: "_getSort",
    value: function _getSort(concrete) {
      var sort;

      switch (_typeof(concrete)) {
        case "boolean":
          sort = this.ctx.mkBoolSort();
          break;

        case "number":
          sort = this.ctx.mkRealSort();
          break;

        case "string":
          sort = this.ctx.mkStringSort();
          break;

        default:
          _Log["default"].log("Symbolic input variable of type ".concat(typeof val === "undefined" ? "undefined" : _typeof(val), " not yet supported."));

      }

      return sort;
    }
  }, {
    key: "_deepConcrete",
    value: function _deepConcrete(start, _concreteCount) {
      start = this.getConcrete(start);
      /*
      let worklist = [this.getConcrete(start)];
      let seen = [];
      	while (worklist.length) {
      	const arg = worklist.pop();
      	seen.push(arg);
      		for (let i in arg) {
      		if (this.isSymbolic(arg[i])) {
      			arg[i] = this.getConcrete(arg[i]);
      			concreteCount.val += 1;
      		}
      			const seenBefore = !!seen.find(x => x === arg); 
      		if (arg[i] instanceof Object && !seenBefore) {
      			worklist.push(arg[i]); 
      		}
      	}
      }
        */

      return start;
    }
  }, {
    key: "concretizeCall",
    value: function concretizeCall(f, base, args) {
      var report = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : true;
      var numConcretizedProperties = {
        val: 0
      };
      base = this._deepConcrete(base, numConcretizedProperties);
      var n_args = Array(args.length);

      for (var i = 0; i < args.length; i++) {
        n_args[i] = this._deepConcrete(args[i], numConcretizedProperties);
      }

      if (report && numConcretizedProperties.val) {
        this.stats.set("Concretized Function Calls", f.name);
      }

      return {
        base: base,
        args: n_args,
        count: numConcretizedProperties.val
      };
    }
  }, {
    key: "createPureSymbol",
    value: function createPureSymbol(name) {
      this.stats.seen("Pure Symbols");
      var pureType = this.createSymbolicValue(name + "_t", "undefined");
      var res;

      if (this.assertEqual(pureType, this.concolic("string"))) {
        res = this.createSymbolicValue(name, "seed_string");
      } else if (this.assertEqual(pureType, this.concolic("number"))) {
        res = this.createSymbolicValue(name, 0);
      } else if (this.assertEqual(pureType, this.concolic("boolean"))) {
        res = this.createSymbolicValue(name, false);
      } else if (this.assertEqual(pureType, this.concolic("object"))) {
        res = this.createSymbolicValue(name, {});
      } else if (this.assertEqual(pureType, this.concolic("array_number"))) {
        res = this.createSymbolicValue(name, [0]);
      } else if (this.assertEqual(pureType, this.concolic("array_string"))) {
        res = this.createSymbolicValue(name, [""]);
      } else if (this.assertEqual(pureType, this.concolic("array_bool"))) {
        res = this.createSymbolicValue(name, [false]);
      } else if (this.assertEqual(pureType, this.concolic("null"))) {
        res = null;
      } else {
        res = undefined;
      }

      return res;
    }
    /**
        * TODO: Symbol Renaming internalization
        */

  }, {
    key: "createSymbolicValue",
    value: function createSymbolicValue(name, concrete) {
      this.stats.seen("Symbolic Values"); //TODO: Very ugly short circuit

      if (!(concrete instanceof Array) && _typeof(concrete) === "object") {
        return new _SymbolicObject.SymbolicObject(name);
      }

      var symbolic;
      var arrayType;

      if (concrete instanceof Array) {
        this.stats.seen("Symbolic Arrays");
        symbolic = this.ctx.mkArray(name, this._getSort(concrete[0]));
        this.pushCondition(this.ctx.mkGe(symbolic.getLength(), this.ctx.mkIntVal(0)), true);
        arrayType = _typeof(concrete[0]);
      } else {
        this.stats.seen("Symbolic Primitives");

        var sort = this._getSort(concrete);

        var symbol = this.ctx.mkStringSymbol(name);
        symbolic = this.ctx.mkConst(symbol, sort);
      } // Use generated input if available


      if (name in this.input) {
        concrete = this.input[name];
      } else {
        this.input[name] = concrete;
      }

      this.inputSymbols[name] = symbolic;
      return new _WrappedValue.ConcolicValue(concrete, symbolic, arrayType);
    }
  }, {
    key: "getSolution",
    value: function getSolution(model) {
      var solution = {};

      for (var name in this.inputSymbols) {
        var solutionAst = model.eval(this.inputSymbols[name]);
        solution[name] = solutionAst.asConstant(model);
        solutionAst.destroy();
      }

      model.destroy();
      return solution;
    }
  }, {
    key: "_checkSat",
    value: function _checkSat(clause, i, checks) {
      var startTime = new Date().getTime();
      var model = new _z3javascript["default"].Query([clause], checks).getModel(this.slv);
      var endTime = new Date().getTime();
      this.stats.max("Max Queries (Any)", _z3javascript["default"].Query.LAST_ATTEMPTS);

      if (model) {
        this.stats.max("Max Queries (Succesful)", _z3javascript["default"].Query.LAST_ATTEMPTS);
      } else {
        this.stats.seen("Failed Queries");

        if (_z3javascript["default"].Query.LAST_ATTEMPTS == _z3javascript["default"].Query.MAX_REFINEMENTS) {
          this.stats.seen("Failed Queries (Max Refinements)");
        }
      }

      _Log["default"].logQuery(clause.toString(), this.slv.toString(), checks.length, startTime, endTime, model ? model.toString() : undefined, _z3javascript["default"].Query.LAST_ATTEMPTS, _z3javascript["default"].Query.LAST_ATTEMPTS == _z3javascript["default"].Query.MAX_REFINEMENTS);

      return model ? this.getSolution(model) : undefined;
    }
  }, {
    key: "isWrapped",
    value: function isWrapped(val) {
      return val instanceof _WrappedValue.WrappedValue;
    }
  }, {
    key: "isSymbolic",
    value: function isSymbolic(val) {
      return !!_WrappedValue.ConcolicValue.getSymbolic(val);
    }
  }, {
    key: "updateSymbolic",
    value: function updateSymbolic(val, val_s) {
      return _WrappedValue.ConcolicValue.setSymbolic(val, val_s);
    }
  }, {
    key: "getConcrete",
    value: function getConcrete(val) {
      return val instanceof _WrappedValue.WrappedValue ? val.getConcrete() : val;
    }
  }, {
    key: "arrayType",
    value: function arrayType(val) {
      return val instanceof _WrappedValue.WrappedValue ? val.getArrayType() : undefined;
    }
  }, {
    key: "getSymbolic",
    value: function getSymbolic(val) {
      return _WrappedValue.ConcolicValue.getSymbolic(val);
    }
  }, {
    key: "asSymbolic",
    value: function asSymbolic(val) {
      return _WrappedValue.ConcolicValue.getSymbolic(val) || this.constantSymbol(val);
    }
  }, {
    key: "_symbolicBinary",
    value: function _symbolicBinary(op, left_c, left_s, right_c, right_s) {
      this.stats.seen("Symbolic Binary");

      switch (op) {
        case "===":
        case "==":
          return this.ctx.mkEq(left_s, right_s);

        case "!==":
        case "!=":
          return this.ctx.mkNot(this.ctx.mkEq(left_s, right_s));

        case "&&":
          return this.ctx.mkAnd(left_s, right_s);

        case "||":
          return this.ctx.mkOr(left_s, right_s);

        case ">":
          return this.ctx.mkGt(left_s, right_s);

        case ">=":
          return this.ctx.mkGe(left_s, right_s);

        case "<=":
          return this.ctx.mkLe(left_s, right_s);

        case "<":
          return this.ctx.mkLt(left_s, right_s);

        case "<<":
        case "<<<":
          left_s = this.ctx.mkRealToInt(left_s);
          right_s = this.ctx.mkRealToInt(right_s);
          return this.ctx.mkIntToReal(this.ctx.mkMul(left_s, this.ctx.mkPower(this.ctx.mkIntVal(2), right_s)));

        case ">>":
        case ">>>":
          left_s = this.ctx.mkRealToInt(left_s);
          right_s = this.ctx.mkRealToInt(right_s);
          return this.ctx.mkIntToReal(this.ctx.mkDiv(left_s, this.ctx.mkPower(this.ctx.mkIntVal(2), right_s)));

        case "+":
          return typeof left_c === "string" ? this.ctx.mkSeqConcat([left_s, right_s]) : this.ctx.mkAdd(left_s, right_s);

        case "-":
          return this.ctx.mkSub(left_s, right_s);

        case "*":
          return this.ctx.mkMul(left_s, right_s);

        case "/":
          return this.ctx.mkDiv(left_s, right_s);

        case "%":
          return this.ctx.mkMod(left_s, right_s);

        default:
          _Log["default"].log("Symbolic execution does not support operand ".concat(op, ", concretizing."));

          break;
      }

      return undefined;
    }
    /** 
      * Symbolic binary operation, expects two concolic values and an operator
      */

  }, {
    key: "binary",
    value: function binary(op, left, right) {
      if (typeof this.getConcrete(left) === "string") {
        right = this.ToString(right);
      }

      var result_c = _SymbolicHelper["default"].evalBinary(op, this.getConcrete(left), this.getConcrete(right));

      var result_s = this._symbolicBinary(op, this.getConcrete(left), this.asSymbolic(left), this.getConcrete(right), this.asSymbolic(right));

      return _typeof(result_s) !== undefined ? new _WrappedValue.ConcolicValue(result_c, result_s) : result_c;
    }
    /**
      * Symbolic field lookup - currently only has support for symbolic arrays / strings
      */

  }, {
    key: "symbolicField",
    value: function symbolicField(base_c, base_s, field_c, field_s) {
      this.stats.seen("Symbolic Field");

      function canHaveFields() {
        return typeof base_c === "string" || base_c instanceof Array;
      }

      function isRealNumber() {
        return typeof field_c === "number" && Number.isFinite(field_c);
      }

      if (canHaveFields() && isRealNumber()) {
        var withinBounds = this.ctx.mkAnd(this.ctx.mkGt(field_s, this.ctx.mkIntVal(-1)), this.ctx.mkLt(field_s, base_s.getLength()));

        if (this.conditional(new _WrappedValue.ConcolicValue(field_c > -1 && field_c < base_c.length, withinBounds))) {
          return base_s.getField(this.ctx.mkRealToInt(field_s));
        } else {
          return undefined;
        }
      }

      switch (field_c) {
        case "length":
          {
            if (base_s.getLength()) {
              return base_s.getLength();
            } else {
              _Log["default"].log("No length field on symbolic value");
            }

            break;
          }

        default:
          {
            _Log["default"].log("Unsupported symbolic field - concretizing " + base_c + " and field " + field_c);

            break;
          }
      }

      return undefined;
    }
    /**
        * Coerce either a concrete or ConcolicValue to a boolean
        * Concretizes the ConcolicValue if no coercion rule is known
        */

  }, {
    key: "toBool",
    value: function toBool(val) {
      if (this.isSymbolic(val)) {
        var val_type = _typeof(this.getConcrete(val));

        switch (val_type) {
          case "boolean":
            return val;

          case "number":
            return this.binary("!=", val, this.concolic(0));

          case "string":
            return this.binary("!=", val, this.concolic(""));
        }

        _Log["default"].log("WARNING: Concretizing coercion to boolean (toBool) due to unknown type");
      }

      return this.getConcrete(!!val);
    }
    /**
        * Perform a symbolic unary action.
        * Expects an Expr and returns an Expr or undefined if we don't
        * know how to do this op symbolically
        */

  }, {
    key: "_symbolicUnary",
    value: function _symbolicUnary(op, left_c, left_s) {
      this.stats.seen("Symbolic Unary");
      var unaryFn = this._unaryJumpTable[_typeof(left_c)] ? this._unaryJumpTable[_typeof(left_c)][op] : undefined;

      if (unaryFn) {
        return unaryFn(left_s, left_c);
      } else {
        _Log["default"].log("Unsupported symbolic operand: ".concat(op, " on ").concat(left_c, " symbolic ").concat(left_s));

        return undefined;
      }
    }
  }, {
    key: "ToString",
    value: function ToString(symbol) {
      if (typeof this.getConcrete(symbol) !== "string") {
        _Log["default"].log("TODO: Concretizing non string input ".concat(symbol, " reduced to ").concat(this.getConcrete(symbol)));

        return "" + this.getConcrete(symbol);
      }

      return symbol;
    }
    /**
        * Perform a unary op on a ConcolicValue or a concrete value
        * Concretizes the ConcolicValue if we don't know how to do that action symbolically
        */

  }, {
    key: "unary",
    value: function unary(op, left) {
      var result_c = _SymbolicHelper["default"].evalUnary(op, this.getConcrete(left));

      var result_s = this.isSymbolic(left) ? this._symbolicUnary(op, this.getConcrete(left), this.asSymbolic(left)) : undefined;
      return result_s ? new _WrappedValue.ConcolicValue(result_c, result_s) : result_c;
    }
    /**
        * Return a symbol which will always be equal to the constant value val
        * returns undefined if the theory is not supported.
        */

  }, {
    key: "constantSymbol",
    value: function constantSymbol(val) {
      this.stats.seen("Wrapped Constants");

      if (val && _typeof(val) === "object") {
        val = val.valueOf();
      }

      switch (_typeof(val)) {
        case "boolean":
          return val ? this.ctx.mkTrue() : this.ctx.mkFalse();

        case "number":
          return Math.round(val) === val ? this.ctx.mkReal(val, 1) : this.ctx.mkNumeral("" + val, this.ctx.mkRealSort());

        case "string":
          return this.ctx.mkString(val.toString());

        default:
          _Log["default"].log("Symbolic expressions with " + _typeof(val) + " literals not yet supported.");

      }

      return undefined;
    }
    /**
        * If val is a symbolic value then return val otherwise wrap it
        * with a constant symbol inside a ConcolicValue.
        *
        * Used to turn a concrete value into a constant symbol for symbolic ops.
        */

  }, {
    key: "concolic",
    value: function concolic(val) {
      return this.isSymbolic(val) ? val : new _WrappedValue.ConcolicValue(val, this.constantSymbol(val));
    }
    /**
        * Assert left == right on the path condition
        */

  }, {
    key: "assertEqual",
    value: function assertEqual(left, right) {
      var equalityTest = this.binary("==", left, right);
      this.conditional(equalityTest);
      return this.getConcrete(equalityTest);
    }
  }]);

  return SymbolicState;
}();

var _default = SymbolicState;
exports["default"] = _default;