"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const node_1 = require("vscode-languageserver/node");
const yamlSettings_1 = require("../src/yamlSettings");
const serviceSetup_1 = require("./utils/serviceSetup");
const testHelper_1 = require("./utils/testHelper");
const chai_1 = require("chai");
const verifyError_1 = require("./utils/verifyError");
const path = require("path");
describe('Auto Completion Fix Tests', () => {
    let languageSettingsSetup;
    let languageService;
    let languageHandler;
    let yamlSettings;
    before(() => {
        languageSettingsSetup = new serviceSetup_1.ServiceSetup().withCompletion().withSchemaFileMatch({
            uri: 'https://raw.githubusercontent.com/yannh/kubernetes-json-schema/master/v1.22.4-standalone-strict/all.json',
            fileMatch: [testHelper_1.SCHEMA_ID],
        });
        const { languageService: langService, languageHandler: langHandler, yamlSettings: settings } = testHelper_1.setupLanguageService(languageSettingsSetup.languageSettings);
        languageService = langService;
        languageHandler = langHandler;
        yamlSettings = settings;
    });
    /**
     *
     * @param content
     * @param line starts with 0 index
     * @param character starts with 1 index
     * @returns
     */
    function parseSetup(content, line, character) {
        const testTextDocument = testHelper_1.setupSchemaIDTextDocument(content);
        yamlSettings.documents = new yamlSettings_1.TextDocumentTestManager();
        yamlSettings.documents.set(testTextDocument);
        return languageHandler.completionHandler({
            position: node_1.Position.create(line, character),
            textDocument: testTextDocument,
        });
    }
    afterEach(() => {
        languageService.deleteSchema(testHelper_1.SCHEMA_ID);
        languageService.configure(languageSettingsSetup.languageSettings);
    });
    it('should show completion on map under array', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'array',
            items: {
                type: 'object',
                properties: {
                    from: {
                        type: 'object',
                        properties: {
                            foo: {
                                type: 'boolean',
                            },
                        },
                    },
                },
            },
        });
        const content = '- from:\n    ';
        const completion = yield parseSetup(content, 1, 3);
        chai_1.expect(completion.items).lengthOf(1);
        chai_1.expect(completion.items[0]).eql(verifyError_1.createExpectedCompletion('foo', 'foo: ', 1, 3, 1, 3, 10, 2, {
            documentation: '',
        }));
    }));
    it('should show completion on array empty array item', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'array',
            items: {
                type: 'object',
                properties: {
                    from: {
                        type: 'object',
                        properties: {
                            foo: {
                                type: 'boolean',
                            },
                        },
                    },
                },
            },
        });
        const content = '- ';
        const completion = yield parseSetup(content, 0, 2);
        chai_1.expect(completion.items).lengthOf(1);
        chai_1.expect(completion.items[0]).eql(verifyError_1.createExpectedCompletion('from', 'from:\n    ', 0, 2, 0, 2, 10, 2, {
            documentation: '',
        }));
    }));
    it('should show completion items in the middle of map in array', () => __awaiter(void 0, void 0, void 0, function* () {
        const content = `apiVersion: v1
kind: Pod
metadata:
  name: foo
spec:
  containers:
    - name: test
      
      image: alpine
    `;
        const completion = yield parseSetup(content, 7, 6);
        chai_1.expect(completion.items).length.greaterThan(1);
    }));
    it('should show completion on array item on first line', () => __awaiter(void 0, void 0, void 0, function* () {
        const content = '-d';
        const completion = yield parseSetup(content, 0, 1);
        chai_1.expect(completion.items).is.empty;
    }));
    it('should complete without error on map inside array', () => __awaiter(void 0, void 0, void 0, function* () {
        const content = '- foo\n- bar:\n    so';
        const completion = yield parseSetup(content, 2, 6);
        chai_1.expect(completion.items).is.empty;
    }));
    it('should complete  array', () => __awaiter(void 0, void 0, void 0, function* () {
        // eslint-disable-next-line @typescript-eslint/no-var-requires
        const schema = require(path.join(__dirname, './fixtures/test-nested-object-array.json'));
        languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
        const content = `objA:
  - name: nameA1
      
objB:
  size: midle
  name: nameB2  
`;
        const completion = yield parseSetup(content, 2, 4);
        chai_1.expect(completion.items).is.not.empty;
    }));
    it('should complete array item for "oneOf" schema', () => __awaiter(void 0, void 0, void 0, function* () {
        // eslint-disable-next-line @typescript-eslint/no-var-requires
        const schema = require(path.join(__dirname, './fixtures/test-completion-oneOf.json'));
        languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
        const content = `metadata:
  Selector:
    query:
      - 
`;
        const completion = yield parseSetup(content, 3, 8);
        chai_1.expect(completion.items).length(5);
        chai_1.expect(completion.items.map((it) => it.label)).to.have.members(['NOT', 'attribute', 'operation', 'value', 'FUNC_item']);
    }));
    it('Autocomplete with short nextLine - nested object', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'object',
            properties: {
                example: {
                    type: 'object',
                    properties: {
                        sample: {
                            type: 'object',
                            properties: {
                                detail: {
                                    type: 'object',
                                },
                            },
                        },
                    },
                },
                a: {
                    type: 'string',
                    description: 'short prop name because of distance to the cursor',
                },
            },
        });
        const content = 'example:\n  sample:\n    ';
        const completion = yield parseSetup(content + '\na: test', 2, 4);
        chai_1.expect(completion.items.length).equal(1);
        chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('detail', 'detail:\n  ', 2, 4, 2, 4, 10, 2, {
            documentation: '',
        }));
    }));
    it('Autocomplete with a new line inside the object', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'object',
            properties: {
                example: {
                    type: 'object',
                    properties: {
                        sample: {
                            type: 'object',
                            properties: {
                                prop1: {
                                    type: 'string',
                                },
                                prop2: {
                                    type: 'string',
                                },
                            },
                        },
                    },
                },
            },
        });
        const content = 'example:\n  sample:\n    \n    prop2: value2';
        const completion = yield parseSetup(content, 2, 4);
        chai_1.expect(completion.items.length).equal(1);
        chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('prop1', 'prop1: ', 2, 4, 2, 4, 10, 2, {
            documentation: '',
        }));
    }));
    it('Autocomplete on the first array item', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'object',
            properties: {
                examples: {
                    type: 'array',
                    items: {
                        type: 'object',
                        properties: {
                            sample: {
                                type: 'object',
                                properties: {
                                    prop1: {
                                        type: 'string',
                                    },
                                },
                            },
                        },
                    },
                },
            },
        });
        const content = 'examples:\n  \n  - sample:\n      prop1: value1';
        const completion = yield parseSetup(content, 1, 2);
        chai_1.expect(completion.items.length).equal(1);
        chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('- (array item)', '- ', 1, 2, 1, 2, 9, 2, {
            documentation: {
                kind: 'markdown',
                value: 'Create an item of an array\n ```\n- \n```',
            },
        }));
    }));
    it('Autocomplete indent on array when parent is array', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'object',
            properties: {
                examples: {
                    type: 'array',
                    items: {
                        type: 'object',
                        properties: {
                            objectWithArray: {
                                type: 'array',
                                items: {
                                    type: 'string',
                                },
                            },
                        },
                    },
                },
            },
        });
        const content = 'examples:\n  - ';
        const completion = yield parseSetup(content, 1, 4);
        chai_1.expect(completion.items.length).equal(1);
        chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('objectWithArray', 'objectWithArray:\n    - ${1:""}', 1, 4, 1, 4, 10, 2, {
            documentation: '',
        }));
    }));
    it('Autocomplete indent on array object when parent is array', () => __awaiter(void 0, void 0, void 0, function* () {
        languageService.addSchema(testHelper_1.SCHEMA_ID, {
            type: 'object',
            properties: {
                examples: {
                    type: 'array',
                    items: {
                        type: 'object',
                        properties: {
                            objectWithArray: {
                                type: 'array',
                                items: {
                                    type: 'object',
                                    required: ['item', 'item2'],
                                    properties: {
                                        item: { type: 'string' },
                                        item2: { type: 'string' },
                                    },
                                },
                            },
                        },
                    },
                },
            },
        });
        const content = 'examples:\n  - ';
        const completion = yield parseSetup(content, 1, 4);
        chai_1.expect(completion.items.length).equal(1);
        chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('objectWithArray', 'objectWithArray:\n    - item: $1\n      item2: $2', 1, 4, 1, 4, 10, 2, {
            documentation: '',
        }));
    }));
    describe('array indent on different index position', () => {
        const schema = {
            type: 'object',
            properties: {
                objectWithArray: {
                    type: 'array',
                    items: {
                        type: 'object',
                        required: ['item', 'item2'],
                        properties: {
                            item: { type: 'string' },
                            item2: {
                                type: 'object',
                                required: ['prop1', 'prop2'],
                                properties: {
                                    prop1: { type: 'string' },
                                    prop2: { type: 'string' },
                                },
                            },
                        },
                    },
                },
            },
        };
        it('array indent on the first item', () => __awaiter(void 0, void 0, void 0, function* () {
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = 'objectWithArray:\n  - ';
            const completion = yield parseSetup(content, 1, 4);
            chai_1.expect(completion.items.length).equal(3);
            chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('item', 'item: ', 1, 4, 1, 4, 10, 2, {
                documentation: '',
            }));
            chai_1.expect(completion.items[2]).to.be.deep.equal(verifyError_1.createExpectedCompletion('item2', 'item2:\n    prop1: $1\n    prop2: $2', 1, 4, 1, 4, 10, 2, {
                documentation: '',
            }));
        }));
        it('array indent on the second item', () => __awaiter(void 0, void 0, void 0, function* () {
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = 'objectWithArray:\n  - item: first line\n    ';
            const completion = yield parseSetup(content, 2, 4);
            chai_1.expect(completion.items.length).equal(2);
            chai_1.expect(completion.items[0]).to.be.deep.equal(verifyError_1.createExpectedCompletion('item2', 'item2:\n  prop1: $1\n  prop2: $2', 2, 4, 2, 4, 10, 2, {
                documentation: '',
            }));
        }));
    });
    describe('merge properties from anyOf objects', () => {
        it('should merge different simple values', () => __awaiter(void 0, void 0, void 0, function* () {
            const schema = {
                anyOf: [
                    {
                        properties: {
                            simplePropWithSimpleValue: { type: 'string', const: 'const value' },
                        },
                    },
                    {
                        properties: {
                            simplePropWithSimpleValue: { type: 'boolean', default: false },
                        },
                    },
                    {
                        properties: {
                            simplePropWithSimpleValue: { type: 'null', default: null },
                        },
                    },
                    {
                        properties: {
                            simplePropWithSimpleValue: { type: 'string' },
                        },
                    },
                ],
            };
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = '';
            const completion = yield parseSetup(content, 0, 1);
            chai_1.expect(completion.items.length).equal(1);
            chai_1.expect(completion.items[0].insertText).to.be.equal('simplePropWithSimpleValue: ${1|const value,false,null|}');
        }));
        it('should autocomplete as single item with same value', () => __awaiter(void 0, void 0, void 0, function* () {
            const schema = {
                anyOf: [
                    {
                        properties: {
                            simplePropWithSameValue: { type: 'string', const: 'const value 1' },
                            obj1: { properties: { prop1: { type: 'string' } } },
                        },
                    },
                    {
                        properties: {
                            simplePropWithSameValue: { type: 'string', const: 'const value 1' },
                            obj1: { properties: { prop1: { type: 'string' } } },
                        },
                    },
                ],
            };
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = '';
            const completion = yield parseSetup(content, 0, 1);
            chai_1.expect(completion.items.length).equal(2);
            chai_1.expect(completion.items[0].insertText).to.be.equal('simplePropWithSameValue: const value 1');
            chai_1.expect(completion.items[1].insertText).to.be.equal('obj1:\n  ');
        }));
        it('should not merge objects', () => __awaiter(void 0, void 0, void 0, function* () {
            const schema = {
                anyOf: [
                    {
                        properties: {
                            obj1: { properties: { prop1: { type: 'string' } }, required: ['prop1'] },
                        },
                    },
                    {
                        properties: {
                            obj1: { properties: { prop2: { type: 'string', const: 'value' } }, required: ['prop2'] },
                        },
                    },
                ],
            };
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = '';
            const completion = yield parseSetup(content, 0, 1);
            chai_1.expect(completion.items.length).equal(2);
            chai_1.expect(completion.items[0].label).to.be.equal('obj1');
            chai_1.expect(completion.items[0].insertText).to.be.equal('obj1:\n  prop1: ');
            chai_1.expect(completion.items[1].label).to.be.equal('obj1');
            chai_1.expect(completion.items[1].insertText).to.be.equal('obj1:\n  prop2: ${1:value}');
        }));
        it('should suggest when cursor is not on the end of the line', () => __awaiter(void 0, void 0, void 0, function* () {
            const schema = {
                properties: {
                    prop: {
                        const: 'const',
                    },
                },
            };
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = 'prop:   ';
            const completion = yield parseSetup(content, 0, 6);
            chai_1.expect(completion.items.length).equal(1);
            chai_1.expect(completion.items[0].label).to.be.equal('const');
            chai_1.expect(completion.items[0].textEdit).to.be.deep.equal({ newText: 'const', range: node_1.Range.create(0, 6, 0, content.length) });
        }));
        it('should suggest object array when extra space is after cursor', () => __awaiter(void 0, void 0, void 0, function* () {
            const schema = {
                properties: {
                    arrayObj: {
                        type: 'array',
                        items: {
                            type: 'object',
                            properties: {
                                item1: {
                                    type: 'string',
                                },
                                item2: {
                                    type: 'string',
                                },
                            },
                            required: ['item1', 'item2'],
                        },
                    },
                },
            };
            languageService.addSchema(testHelper_1.SCHEMA_ID, schema);
            const content = 'arrayObj:\n  -   ';
            const completion = yield parseSetup(content, 1, 4);
            chai_1.expect(completion.items.length).equal(3);
            chai_1.expect(completion.items[1].textEdit).to.be.deep.equal({
                newText: 'item1: $1\n  item2: $2',
                range: node_1.Range.create(1, 4, 1, 6),
            });
        }));
    });
});
//# sourceMappingURL=autoCompletionFix.test.js.map