# (c) 2018-2022, NetApp, Inc
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

''' unit test template for ONTAP Ansible module '''

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type
import copy
import pytest
import sys

from ansible_collections.netapp.ontap.tests.unit.compat import unittest
from ansible_collections.netapp.ontap.tests.unit.compat.mock import patch, Mock
import ansible_collections.netapp.ontap.plugins.module_utils.netapp as netapp_utils
from ansible_collections.netapp.ontap.tests.unit.plugins.module_utils.ansible_mocks import\
    assert_warning_was_raised, print_warnings, create_module, create_and_apply, expect_and_capture_ansible_exception, patch_ansible
from ansible_collections.netapp.ontap.tests.unit.framework.mock_rest_and_zapi_requests import\
    patch_request_and_invoke, register_responses
from ansible_collections.netapp.ontap.tests.unit.framework.rest_factory import rest_responses
from ansible_collections.netapp.ontap.tests.unit.framework.zapi_factory import build_zapi_response, zapi_responses
from ansible_collections.netapp.ontap.plugins.modules.na_ontap_interface \
    import NetAppOntapInterface as interface_module, netmask_length_to_netmask, netmask_to_netmask_length


if not netapp_utils.HAS_REQUESTS and sys.version_info < (2, 7):
    pytestmark = pytest.mark.skip('Skipping Unit Tests on 2.6 as requests is not be available')


if not netapp_utils.has_netapp_lib():
    pytestmark = pytest.mark.skip('skipping as missing required netapp_lib')


interface_info = {
    'attributes-list': {
        'net-interface-info': {
            'interface-name': 'abc_if',
            'administrative-status': 'up',
            'failover-policy': 'up',
            'firewall-policy': 'up',
            'is-auto-revert': 'true',
            'home-node': 'node',
            'current-node': 'node',
            'home-port': 'e0c',
            'current-port': 'e0c',
            'address': '2.2.2.2',
            'netmask': '1.1.1.1',
            'role': 'data',
            'dns-domain-name': 'test.com',
            'listen-for-dns_query': 'true',
            'is-dns-update-enabled': 'true'
        }
    }
}


ZRR = zapi_responses({
    'interface_info': build_zapi_response(interface_info, 1)
})


DEFAULT_ARGS = {
    'hostname': '10.10.10.10',
    'username': 'admin',
    'password': 'password',
    'home_port': 'e0c',
    'interface_name': 'abc_if',
}


def test_module_fail_when_required_args_missing():
    ''' required arguments are reported as errors '''
    module_args = {
        'vserver': 'vserver',
        'use_rest': 'never'
    }
    error = create_module(interface_module, module_args, fail=True)['msg']
    assert 'missing required arguments:' in error
    assert 'interface_name' in error


def test_create_error_missing_param():
    ''' Test successful create '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
    ])
    module_args = {
        'vserver': 'vserver',
        'home_node': 'node',
        'home_port': 'e0c',
        'use_rest': 'never'
    }
    msg = 'Error: Missing one or more required parameters for creating interface:'
    error = create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    assert msg in error
    assert 'role' in error


def test_successful_create():
    ''' Test successful create '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'net-interface-create', ZRR['success']),
    ])
    module_args = {
        'vserver': 'vserver',
        'home_node': 'node',
        'role': 'data',
        'home_port': 'e0c',
        'use_rest': 'never'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_successful_create_for_NVMe():
    ''' Test successful create for NVMe protocol'''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'net-interface-create', ZRR['success']),
    ])
    module_args = {
        'vserver': 'vserver',
        'home_node': 'node',
        'role': 'data',
        'home_port': 'e0c',
        'protocols': ['fc-nvme'],
        'use_rest': 'never'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_create_idempotency_for_NVMe():
    ''' Test successful create for NVMe protocol'''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
    ])
    module_args = {
        'vserver': 'vserver',
        'home_node': 'node',
        'role': 'data',
        'home_port': 'e0c',
        'protocols': ['fc-nvme'],
        'use_rest': 'never'
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_create_error_for_NVMe():
    ''' Test if create throws an error if required param 'protocols' uses NVMe'''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
    ])
    msg = 'Error: Following parameters for creating interface are not supported for data-protocol fc-nvme:'
    module_args = {
        'vserver': 'vserver',
        'protocols': ['fc-nvme'],
        'address': '1.1.1.1',
        'use_rest': 'never'
    }
    error = create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    assert msg in error
    for option in ('netmask', 'address', 'firewall_policy'):
        assert option in error


def test_create_idempotency():
    ''' Test create idempotency '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
    ])
    module_args = {
        'vserver': 'vserver',
        'use_rest': 'never'
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_successful_delete():
    ''' Test delete existing interface '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),               # offline
        ('ZAPI', 'net-interface-delete', ZRR['success']),
    ])
    module_args = {
        'state': 'absent',
        'vserver': 'vserver',
        'use_rest': 'never'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_delete_idempotency():
    ''' Test delete idempotency '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
    ])
    module_args = {
        'state': 'absent',
        'vserver': 'vserver',
        'use_rest': 'never'
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_successful_modify():
    ''' Test successful modify interface_minutes '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),
    ])
    module_args = {
        'vserver': 'vserver',
        'dns_domain_name': 'test2.com',
        'home_port': 'e0d',
        'is_dns_update_enabled': False,
        'listen_for_dns_query': False,
        'use_rest': 'never'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_modify_idempotency():
    ''' Test modify idempotency '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
    ])
    module_args = {
        'vserver': 'vserver',
        'use_rest': 'never'
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_error_message():
    register_responses([
        # create, missing params
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'cluster-node-get-iter', ZRR['no_records']),

        # create error
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'cluster-node-get-iter', ZRR['no_records']),
        ('ZAPI', 'net-interface-create', ZRR['error']),

        # modify error
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['error']),

        # rename error
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-rename', ZRR['error']),

        # delete error
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),
        ('ZAPI', 'net-interface-delete', ZRR['error']),
    ])
    module_args = {
        'vserver': 'vserver',
        'use_rest': 'never',
    }
    msg = 'Error: Missing one or more required parameters for creating interface:'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    module_args['home_port'] = 'e0d'
    module_args['role'] = 'data'
    msg = 'Error Creating interface abc_if: NetApp API failed. Reason - 12345:'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    module_args['home_port'] = 'new_port'
    msg = 'Error modifying interface abc_if: NetApp API failed. Reason - 12345:'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    module_args['from_name'] = 'old_name'
    msg = 'Error renaming old_name to abc_if: NetApp API failed. Reason - 12345:'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    module_args['state'] = 'absent'
    msg = 'Error deleting interface abc_if: NetApp API failed. Reason - 12345:'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_successful_rename():
    ''' Test successful '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-rename', ZRR['success']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),
    ])
    module_args = {
        'vserver': 'vserver',
        'dns_domain_name': 'test2.com',
        'from_name': 'from_interface_name',
        'home_port': 'new_port',
        'is_dns_update_enabled': False,
        'listen_for_dns_query': False,
        'use_rest': 'never'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_negative_rename_not_found():
    ''' Test from interface not found '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
        ('ZAPI', 'net-interface-get-iter', ZRR['no_records']),
    ])
    msg = 'Error renaming interface abc_if: no interface with from_name from_interface_name.'
    module_args = {
        'vserver': 'vserver',
        'dns_domain_name': 'test2.com',
        'from_name': 'from_interface_name',
        'home_port': 'new_port',
        'is_dns_update_enabled': False,
        'listen_for_dns_query': False,
        'use_rest': 'never'
    }
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_successful_migrate():
    ''' Test successful '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),
        ('ZAPI', 'net-interface-migrate', ZRR['success']),
        ('ZAPI', 'net-interface-migrate', ZRR['success']),
    ])
    module_args = {
        'vserver': 'vserver',
        'dns_domain_name': 'test2.com',
        'current_node': 'new_node',
        'is_dns_update_enabled': False,
        'listen_for_dns_query': False,
        'use_rest': 'never'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_negative_migrate():
    ''' Test successful '''
    register_responses([
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),

        # 2nd try
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),
        ('ZAPI', 'net-interface-migrate', ZRR['error']),

        # 3rd try
        ('ZAPI', 'vserver-get-iter', ZRR['no_records']),
        ('ZAPI', 'ems-autosupport-log', ZRR['success']),
        ('ZAPI', 'net-interface-get-iter', ZRR['interface_info']),
        ('ZAPI', 'net-interface-modify', ZRR['success']),
        ('ZAPI', 'net-interface-migrate', ZRR['success']),
        ('ZAPI', 'net-interface-migrate', ZRR['error']),
    ])
    module_args = {
        'vserver': 'vserver',
        'dns_domain_name': 'test2.com',
        'current_port': 'new_port',
        'is_dns_update_enabled': False,
        'listen_for_dns_query': False,
        'use_rest': 'never'
    }
    msg = 'current_node must be set to migrate'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    module_args['current_node'] = 'new_node'
    msg = 'Error migrating new_node: NetApp API failed. Reason - 12345'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    msg = 'Error migrating new_node: NetApp API failed. Reason - 12345'
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


SRR = rest_responses({
    'one_record_home_node': (200, {'records': [
        {'name': 'node2_abc_if',
         'uuid': '54321',
         'enabled': True,
         'location': {'home_port': {'name': 'e0c'}, 'home_node': {'name': 'node2'}, 'node': {'name': 'node2'}}
         }]}, None),
    'one_record_vserver': (200, {'records': [
        {'name': 'abc_if',
         'uuid': '54321',
         'enabled': True,
         'location': {'home_port': {'name': 'e0c'}, 'home_node': {'name': 'node2'}, 'node': {'name': 'node2'}}
         }]}, None),
    'two_records': (200, {'records': [{'name': 'node2_abc_if'}, {'name': 'node2_abc_if'}]}, None),
    'precluster': (500, None, {'message': 'are available in precluster.'}),
    'cluster_identity': (200, {'location': 'Oz', 'name': 'abc'}, None),
    'nodes': (200, {'records': [
        {'name': 'node2', 'uuid': 'uuid2', 'cluster_interfaces': [{'ip': {'address': '10.10.10.2'}}]}
    ]}, None),
}, False)


def test_rest_create_ip_no_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),      # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                     # get nodes
        ('POST', 'network/ip/interfaces', SRR['success']),          # post
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_create_ip_no_svm_idempotent():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),      # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                             # get nodes
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_create_ip_no_svm_idempotent_localhost():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),      # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                             # get nodes
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'home_node': 'localhost',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_create_ip_with_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),       # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                     # get nodes
        ('POST', 'network/ip/interfaces', SRR['success']),          # post
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'vserver': 'vserver',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_create_ip_with_cluster_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),       # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                     # get nodes
        ('POST', 'network/ip/interfaces', SRR['success']),          # post
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'vserver': 'vserver',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
        'role': 'intercluster'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']
    print_warnings()
    assert_warning_was_raised('Ignoring vserver with REST for non data SVM.')


def test_rest_negative_create_ip():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),       # get IP
        ('GET', 'cluster/nodes', SRR['zero_records']),               # get nodes
    ])
    msg = 'Error: Cannot guess home_node, home_node is required when home_port is present with REST.'
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
    }
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_negative_create_no_ip_address():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),    # get IP
        ('GET', 'network/fc/interfaces', SRR['zero_records']),    # get FC
        ('GET', 'cluster/nodes', SRR['nodes']),     # get nodes
    ])
    msg = 'Error: Missing one or more required parameters for creating interface: interface_type.'
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
    }
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_negative_get_multiple_ip_if():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['two_records']),       # get IP
        ('GET', 'network/fc/interfaces', SRR['zero_records']),      # get FC
        ('GET', 'cluster/nodes', SRR['nodes']),                     # get nodes
    ])
    msg = 'Error: multiple records for: node2_abc_if'
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
    }
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_negative_get_multiple_fc_if():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),  # get IP
        ('GET', 'network/fc/interfaces', SRR['two_records']),   # get FC
        ('GET', 'cluster/nodes', SRR['nodes']),                 # get nodes
    ])
    msg = 'Error: multiple records for: node2_abc_if'
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
    }
    assert msg in create_and_apply(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_modify_idempotent_ip_no_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),  # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                         # get nodes
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_modify_ip_no_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),           # get IP
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),  # get IP
        ('PATCH', 'network/ip/interfaces/54321', SRR['success']),       # patch
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
        'home_node': 'node2',
        'interface_name': 'new_name',
        'from_name': 'abc_if'
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_modify_ip_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['one_record_vserver']),    # get IP
        ('PATCH', 'network/ip/interfaces/54321', SRR['success']),       # patch
    ])
    module_args = {
        'use_rest': 'always',
        'vserver': 'vserver',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
        'home_node': 'node1',
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


@patch('time.sleep')
def test_rest_migrate_ip_no_svm(sleep_mock):
    ''' create cluster '''
    modified = copy.deepcopy(SRR['one_record_home_node'])
    modified[1]['records'][0]['location']['node']['name'] = 'node1'
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),      # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                             # get nodes (for get)
        ('PATCH', 'network/ip/interfaces/54321', SRR['success']),           # patch
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),      # get - no change
        ('PATCH', 'network/ip/interfaces/54321', SRR['success']),           # patch
        ('GET', 'network/ip/interfaces', modified),                         # get
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
        'current_node': 'node1',
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_delete_ip_no_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['one_record_home_node']),  # get IP
        ('GET', 'cluster/nodes', SRR['nodes']),                         # get nodes (for get)
        ('DELETE', 'network/ip/interfaces/54321', SRR['success']),      # delete
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
        'state': 'absent',
    }
    assert create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_rest_delete_idempotent_ip_no_svm():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
        ('GET', 'network/ip/interfaces', SRR['zero_records']),         # get IP
    ])
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'address': '10.12.12.13',
        'netmask': '255.255.192.0',
        'state': 'absent',
    }
    assert not create_and_apply(interface_module, DEFAULT_ARGS, module_args)['changed']


def test_netmask_to_len():
    # note the address has host bits set
    assert netmask_to_netmask_length('10.10.10.10', '255.255.0.0') == '16'


def test_len_to_netmask():
    # note the address has host bits set
    assert netmask_length_to_netmask('10.10.10.10', '16') == '255.255.0.0'


def test_derive_fc_protocol_fcp():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': ['fcp'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    my_obj.derive_fc_data_protocol()
    assert my_obj.parameters['data_protocol'] == 'fcp'


def test_derive_fc_protocol_nvme():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': ['fc-nvme'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    my_obj.derive_fc_data_protocol()
    assert my_obj.parameters['data_protocol'] == 'fc_nvme'


def test_derive_fc_protocol_empty():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': [],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    assert my_obj.derive_fc_data_protocol() is None


def test_negative_derive_fc_protocol_nvme():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': ['fc-nvme', 'fcp'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    msg = "A single protocol entry is expected for FC interface, got ['fc-nvme', 'fcp']."
    assert msg in expect_and_capture_ansible_exception(my_obj.derive_fc_data_protocol, 'fail')['msg']


def test_negative_derive_fc_protocol_nvme_mismatch():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': ['fc-nvme'],
        'data_protocol': 'fcp'
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    msg = "Error: mismatch between configured data_protocol: fcp and data_protocols: ['fc-nvme']"
    assert msg in expect_and_capture_ansible_exception(my_obj.derive_fc_data_protocol, 'fail')['msg']


def test_derive_interface_type_nvme():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': ['fc-nvme'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    my_obj.derive_interface_type()
    assert my_obj.parameters['interface_type'] == 'fc'


def test_derive_interface_type_iscsi():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'protocols': ['iscsi'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    my_obj.derive_interface_type()
    assert my_obj.parameters['interface_type'] == 'ip'


def test_derive_interface_type_cluster():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'role': 'cluster',
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    my_obj.derive_interface_type()
    assert my_obj.parameters['interface_type'] == 'ip'


def test_negative_derive_interface_type_nvme_mismatch():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    msg = "Error: mismatch between configured interface_type: ip and derived interface_type: fc."
    module_args = {
        'use_rest': 'always',
        'protocols': ['fc-nvme'],
        'interface_type': 'ip'
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    assert msg in expect_and_capture_ansible_exception(my_obj.derive_interface_type, 'fail')['msg']


def test_negative_derive_interface_type_unknown():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    msg = "Error: Unexpected value(s) for protocols: ['unexpected']"
    module_args = {
        'use_rest': 'always',
        'protocols': ['unexpected'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    assert msg in expect_and_capture_ansible_exception(my_obj.derive_interface_type, 'fail')['msg']


def test_negative_derive_interface_type_multiple():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    msg = "Error: Incompatible value(s) for protocols: ['fc-nvme', 'cifs']"
    module_args = {
        'use_rest': 'always',
        'protocols': ['fc-nvme', 'cifs'],
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    assert msg in expect_and_capture_ansible_exception(my_obj.derive_interface_type, 'fail')['msg']


def test_derive_block_file_type_fcp():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    block_p, file_p = my_obj.derive_block_file_type(['fcp'])
    assert block_p, not file_p


def test_derive_block_file_type_cifs():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    block_p, file_p = my_obj.derive_block_file_type(['cifs'])
    assert not block_p, file_p


def test_map_failover_policy():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    module_args = {
        'use_rest': 'always',
        'failover_policy': 'local-only',
    }
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    my_obj.map_failover_policy()
    assert my_obj.parameters['failover_scope'] == 'home_node_only'


def test_rest_negative_unsupported_zapi_option_fail():
    ''' create cluster '''
    register_responses([
    ])
    msg = "REST API currently does not support 'is_ipv4_link_local'"
    module_args = {
        'use_rest': 'always',
        'ipspace': 'cluster',
        'is_ipv4_link_local': True,
    }
    assert msg in create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_negative_unsupported_zapi_option_force_zapi_1():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    msg = "missing required argument with ZAPI: vserver"
    module_args = {
        'use_rest': 'auto',
        'ipspace': 'cluster',
        'is_ipv4_link_local': True,
    }
    assert msg in create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


@patch('ansible_collections.netapp.ontap.plugins.module_utils.netapp.has_netapp_lib')
def test_rest_negative_unsupported_zapi_option_force_zapi_2(mock_netapp_lib):
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97']),
    ])
    mock_netapp_lib.return_value = False
    msg = "the python NetApp-Lib module is required"
    module_args = {
        'use_rest': 'auto',
        'ipspace': 'cluster',
        'is_ipv4_link_local': True,
    }
    assert msg in create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_negative_unsupported_rest_version():
    ''' create cluster '''
    register_responses([
        ('GET', 'cluster', SRR['is_rest_96']),
    ])
    msg = "Error: REST requires ONTAP 9.7 or later for interface APIs."
    module_args = {'use_rest': 'always'}
    assert msg == create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']


def test_rest_auto_falls_back_to_zapi_if_ip_9_6():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_96'])
    ])
    module_args = {'use_rest': 'auto'}
    # vserver is a required parameter with ZAPI
    msg = "missing required argument with ZAPI: vserver"
    assert msg in create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    print_warnings
    assert_warning_was_raised('Falling back to ZAPI: REST requires ONTAP 9.7 or later for interface APIs.')


@patch('ansible_collections.netapp.ontap.plugins.modules.na_ontap_interface.HAS_IPADDRESS_LIB', False)
def test_rest_auto_falls_back_to_zapi_if_ip_address_library_is_missing():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97'])
    ])
    module_args = {'use_rest': 'auto'}
    # vserver is a required parameter with ZAPI
    msg = "missing required argument with ZAPI: vserver"
    assert msg in create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    print_warnings
    assert_warning_was_raised('Falling back to ZAPI: the python ipaddress package is required for this module: None')


@patch('ansible_collections.netapp.ontap.plugins.modules.na_ontap_interface.HAS_IPADDRESS_LIB', False)
def test_rest_always_fail_if_ip_address_library_is_missing():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97'])
    ])
    module_args = {'use_rest': 'always'}
    error = create_module(interface_module, DEFAULT_ARGS, module_args, fail=True)['msg']
    assert error == 'Error: the python ipaddress package is required for this module: None'


def test_fix_errors():
    register_responses([
        ('GET', 'cluster', SRR['is_rest_97'])
    ])
    module_args = {'use_rest': 'auto'}
    my_obj = create_module(interface_module, DEFAULT_ARGS, module_args)
    control = {'xx': 11, 'yy': 22}
    # no role in error
    errors = dict(control)
    assert my_obj.fix_errors(None, errors) is None
    assert errors == control
    # role/firewall_policy/protocols/service_policy -> service_policy
    tests = [
        ('data', 'data', ['nfs'], None, 'default-data-files', True),
        ('data', 'data', ['cifs'], None, 'default-data-files', True),
        ('data', 'data', ['iscsi'], None, 'default-data-blocks', True),
        ('data', 'mgmt', ['ignored'], None, 'default-management', True),
        ('data', '', ['nfs'], None, 'default-data-files', True),
        ('data', '', ['cifs'], None, 'default-data-files', True),
        ('data', '', ['iscsi'], None, 'default-data-blocks', True),
        ('data', 'mgmt', ['ignored'], None, 'default-management', True),
        ('intercluster', 'intercluster', ['ignored'], None, 'default-intercluster', True),
        ('intercluster', '', ['ignored'], None, 'default-intercluster', True),
        ('cluster', 'mgmt', ['ignored'], None, 'default-cluster', True),
        ('cluster', '', ['ignored'], None, 'default-cluster', True),
        ('cluster', 'other', ['ignored'], None, 'unchanged', False),
    ]
    for role, firewall_policy, protocols, service_policy, expected_service_policy, fixed in tests:
        my_obj.parameters['protocols'] = protocols
        if service_policy:
            my_obj['service_policy'] = service_policy
        options = {'service_policy': 'unchanged'}
        errors = dict(control)
        errors['role'] = role
        if firewall_policy:
            errors['firewall_policy'] = firewall_policy
        assert my_obj.fix_errors(options, errors) is None
        print('OPTIONS', options)
        assert 'service_policy' in options
        assert options['service_policy'] == expected_service_policy
        assert errors == control or not fixed
        assert fixed or 'role' in errors
